C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE ISOROPIA
C *** THIS SUBROUTINE IS THE MASTER ROUTINE FOR THE ISORROPIA
C     THERMODYNAMIC EQUILIBRIUM AEROSOL MODEL (VERSION 1.1 and above)
C
C ======================== ARGUMENTS / USAGE ===========================
C
C  INPUT:
C  1. [WI] 
C     DOUBLE PRECISION array of length [5].
C     Concentrations, expressed in moles/m3. Depending on the type of
C     problem solved (specified in CNTRL(1)), WI contains either 
C     GAS+AEROSOL or AEROSOL only concentratios.
C     WI(1) - sodium
C     WI(2) - sulfate
C     WI(3) - ammonium
C     WI(4) - nitrate
C     WI(5) - chloride
C
C  2. [RHI] 
C     DOUBLE PRECISION variable.  
C     Ambient relative humidity expressed on a (0,1) scale.
C
C  3. [TEMPI]
C     DOUBLE PRECISION variable. 
C     Ambient temperature expressed in Kelvins. 
C
C  4. [CNTRL]
C     DOUBLE PRECISION array of length [2].
C     Parameters that control the type of problem solved.
C
C     CNTRL(1): Defines the type of problem solved.
C     0 - Forward problem is solved. In this case, array WI contains 
C         GAS and AEROSOL concentrations together.
C     1 - Reverse problem is solved. In this case, array WI contains
C         AEROSOL concentrations only.
C
C     CNTRL(2): Defines the state of the aerosol
C     0 - The aerosol can have both solid+liquid phases (deliquescent)
C     1 - The aerosol is in only liquid state (metastable aerosol)
C
C  OUTPUT:
C  1. [WT] 
C     DOUBLE PRECISION array of length [5].
C     Total concentrations (GAS+AEROSOL) of species, expressed in moles/m3. 
C     If the foreward probelm is solved (CNTRL(1)=0), array WT is 
C     identical to array WI.
C     WT(1) - total sodium
C     WT(2) - total sulfate
C     WT(3) - total ammonium
C     WT(4) - total nitrate
C     WT(5) - total chloride
C
C  2. [GAS]
C     DOUBLE PRECISION array of length [03]. 
C     Gaseous species concentrations, expressed in moles/m3. 
C     GAS(1) - NH3
C     GAS(2) - HNO3
C     GAS(3) - HCl 
C
C  3. [AERLIQ]
C     DOUBLE PRECISION array of length [11]. 
C     Liquid aerosol species concentrations, expressed in moles/m3. 
C     AERLIQ(01) - H+(aq)          
C     AERLIQ(02) - Na+(aq)         
C     AERLIQ(03) - NH4+(aq)
C     AERLIQ(04) - Cl-(aq)         
C     AERLIQ(05) - SO4--(aq)       
C     AERLIQ(06) - HSO4-(aq)       
C     AERLIQ(07) - NO3-(aq)        
C     AERLIQ(08) - H2O             
C     AERLIQ(09) - NH3(aq) (undissociated)
C     AERLIQ(10) - HNCl(aq) (undissociated)
C     AERLIQ(11) - HNO3(aq) (undissociated)
C     AERLIQ(12) - OH-(aq)
C
C  4. [AERSLD]
C     DOUBLE PRECISION array of length [09]. 
C     Solid aerosol species concentrations, expressed in moles/m3. 
C     AERSLD(01) - NaNO3(s)
C     AERSLD(02) - NH4NO3(s)
C     AERSLD(03) - NaCl(s)         
C     AERSLD(04) - NH4Cl(s)
C     AERSLD(05) - Na2SO4(s)       
C     AERSLD(06) - (NH4)2SO4(s)
C     AERSLD(07) - NaHSO4(s)
C     AERSLD(08) - NH4HSO4(s)
C     AERSLD(09) - (NH4)4H(SO4)2(s)
C
C  5. [SCASI]
C     CHARACTER*15 variable.
C     Returns the subcase which the input corresponds to.
C
C  6. [OTHER]
C     DOUBLE PRECISION array of length [6].
C     Returns solution information.
C
C     OTHER(1): Shows if aerosol water exists.
C     0 - Aerosol is WET
C     1 - Aerosol is DRY
C
C     OTHER(2): Aerosol Sulfate ratio, defined as (in moles/m3) :
C               (total ammonia + total Na) / (total sulfate)
C
C     OTHER(3): Sulfate ratio based on aerosol properties that defines 
C               a sulfate poor system:
C               (aerosol ammonia + aerosol Na) / (aerosol sulfate)
C           
C     OTHER(4): Aerosol sodium ratio, defined as (in moles/m3) :
C               (total Na) / (total sulfate)
C      
C     OTHER(5): Ionic strength of the aqueous aerosol (if it exists).
C      
C     OTHER(6): Total number of calls to the activity coefficient 
C               calculation subroutine.
C 
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE ISOROPIA (WI, RHI, TEMPI,  CNTRL, 
     &                     WT, GAS, AERLIQ, AERSLD, SCASI, OTHER)
      INCLUDE 'isrpia.inc'
      PARAMETER (NCTRL=2,NOTHER=6)
      CHARACTER SCASI*15
      DIMENSION WI(NCOMP), WT(NCOMP),   GAS(NGASAQ),  AERSLD(NSLDS), 
     &          AERLIQ(NIONS+NGASAQ+2), CNTRL(NCTRL), OTHER(NOTHER)
C
C *** PROBLEM TYPE (0=FOREWARD, 1=REVERSE) ******************************
C
      IPROB   = NINT(CNTRL(1))
C
C *** AEROSOL STATE (0=SOLID+LIQUID, 1=METASTABLE) **********************
C
      METSTBL = NINT(CNTRL(2))
C
C *** SOLVE FOREWARD PROBLEM ********************************************
C
50    IF (IPROB.EQ.0) THEN
         IF (WI(1)+WI(2)+WI(3)+WI(4)+WI(5) .LE. TINY) THEN ! Everything=0
            CALL INIT1 (WI, RHI, TEMPI)
         ELSE IF (WI(1)+WI(4)+WI(5) .LE. TINY) THEN        ! Na,Cl,NO3=0
            CALL ISRP1F (WI, RHI, TEMPI)
         ELSE IF (WI(1)+WI(5) .LE. TINY) THEN              ! Na,Cl=0
            CALL ISRP2F (WI, RHI, TEMPI)
         ELSE
            CALL ISRP3F (WI, RHI, TEMPI)
         ENDIF
C
C *** SOLVE REVERSE PROBLEM *********************************************
C
      ELSE
         IF (WI(1)+WI(2)+WI(3)+WI(4)+WI(5) .LE. TINY) THEN ! Everything=0
            CALL INIT1 (WI, RHI, TEMPI)
         ELSE IF (WI(1)+WI(4)+WI(5) .LE. TINY) THEN        ! Na,Cl,NO3=0
            CALL ISRP1R (WI, RHI, TEMPI)
         ELSE IF (WI(1)+WI(5) .LE. TINY) THEN              ! Na,Cl=0
            CALL ISRP2R (WI, RHI, TEMPI)
         ELSE
            CALL ISRP3R (WI, RHI, TEMPI)
         ENDIF
      ENDIF
C
C *** ADJUST MASS BALANCE ***********************************************
C
      CALL ADJUST (WI)
ccC
ccC *** IF METASTABLE AND NO WATER - RESOLVE AS NORMAL ********************
ccC
cc      IF (WATER.LE.TINY .AND. METSTBL.EQ.1) THEN
cc         METSTBL = 0
cc         GOTO 50
cc      ENDIF
C
C *** SAVE RESULTS TO ARRAYS (units = mole/m3) ****************************
C
      GAS(1) = GNH3                ! Gaseous aerosol species
      GAS(2) = GHNO3
      GAS(3) = GHCL
C
      DO 10 I=1,NIONS              ! Liquid aerosol species
         AERLIQ(I) = MOLAL(I)
  10  CONTINUE
      DO 20 I=1,NGASAQ
         AERLIQ(NIONS+1+I) = GASAQ(I)
  20  CONTINUE
      AERLIQ(NIONS+1)        = WATER*1.0D3/18.0D0
      AERLIQ(NIONS+NGASAQ+2) = COH
C
      AERSLD(1) = CNANO3           ! Solid aerosol species
      AERSLD(2) = CNH4NO3
      AERSLD(3) = CNACL
      AERSLD(4) = CNH4CL
      AERSLD(5) = CNA2SO4
      AERSLD(6) = CNH42S4
      AERSLD(7) = CNAHSO4
      AERSLD(8) = CNH4HS4
      AERSLD(9) = CLC
C
      IF(WATER.LE.TINY) THEN       ! Dry flag
        OTHER(1) = 1.d0
      ELSE
        OTHER(1) = 0.d0
      ENDIF
C
      OTHER(2) = SULRAT            ! Other stuff
      OTHER(3) = SULRATW
      OTHER(4) = SODRAT
      OTHER(5) = IONIC
      OTHER(6) = ICLACT
C
      SCASI = SCASE
C
      WT(1) = WI(1)                ! Total gas+aerosol phase
      WT(2) = WI(2)
      WT(3) = WI(3) 
      WT(4) = WI(4)
      WT(5) = WI(5)
      IF (IPROB.GT.0 .AND. WATER.GT.TINY) THEN 
         WT(3) = WT(3) + GNH3 
         WT(4) = WT(4) + GHNO3
         WT(5) = WT(5) + GHCL
      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE ISOROPIA ******************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE SETPARM
C *** THIS SUBROUTINE REDEFINES THE SOLUTION PARAMETERS OF ISORROPIA
C
C ======================== ARGUMENTS / USAGE ===========================
C
C *** NOTE: IF NEGATIVE VALUES ARE GIVEN FOR A PARAMETER, IT IS
C     IGNORED AND THE CURRENT VALUE IS USED INSTEAD.
C 
C  INPUT:
C  1. [WFTYPI] 
C     INTEGER variable.
C     Defines the type of weighting algorithm for the solution in Mutual 
C     Deliquescence Regions (MDR's):
C     0 - MDR's are assumed dry. This is equivalent to the approach 
C         used by SEQUILIB.
C     1 - The solution is assumed "half" dry and "half" wet throughout
C         the MDR.
C     2 - The solution is a relative-humidity weighted mean of the
C         dry and wet solutions (as defined in Nenes et al., 1998)
C
C  2. [IACALCI] 
C     INTEGER variable.
C     Method of activity coefficient calculation:
C     0 - Calculate coefficients during runtime
C     1 - Use precalculated tables
C 
C  3. [EPSI] 
C     DOUBLE PRECITION variable.
C     Defines the convergence criterion for all iterative processes
C     in ISORROPIA, except those for activity coefficient calculations
C     (EPSACTI controls that).
C
C  4. [MAXITI]
C     INTEGER variable.
C     Defines the maximum number of iterations for all iterative 
C     processes in ISORROPIA, except for activity coefficient calculations 
C     (NSWEEPI controls that).
C
C  5. [NSWEEPI]
C     INTEGER variable.
C     Defines the maximum number of iterations for activity coefficient 
C     calculations.
C 
C  6. [EPSACTI] 
C     DOUBLE PRECISION variable.
C     Defines the convergence criterion for activity coefficient 
C     calculations.
C 
C  7. [NDIV] 
C     INTEGER variable.
C     Defines the number of subdivisions needed for the initial root
C     tracking for the bisection method. Usually this parameter should 
C     not be altered, but is included for completeness.
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE SETPARM (WFTYPI,  IACALCI, EPSI, MAXITI, NSWEEPI, 
     &                    EPSACTI, NDIVI)
      INCLUDE 'isrpia.inc'
      INTEGER  WFTYPI
C
C *** SETUP SOLUTION PARAMETERS *****************************************
C
      IF (WFTYPI .GE. 0)   WFTYP  = WFTYPI
      IF (IACALCI.GE. 0)   IACALC = IACALCI
      IF (EPSI   .GE.ZERO) EPS    = EPSI
      IF (MAXITI .GT. 0)   MAXIT  = MAXITI
      IF (NSWEEPI.GT. 0)   NSWEEP = NSWEEPI
      IF (EPSACTI.GE.ZERO) EPSACT = EPSACTI
      IF (NDIVI  .GT. 0)   NDIV   = NDIVI
C
C *** END OF SUBROUTINE SETPARM *****************************************
C
      RETURN
      END




C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE GETPARM
C *** THIS SUBROUTINE OBTAINS THE CURRENT VAULES OF THE SOLUTION 
C     PARAMETERS OF ISORROPIA
C
C ======================== ARGUMENTS / USAGE ===========================
C
C *** THE PARAMETERS ARE THOSE OF SUBROUTINE SETPARM
C 
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE GETPARM (WFTYPI,  IACALCI, EPSI, MAXITI, NSWEEPI, 
     &                    EPSACTI, NDIVI)
      INCLUDE 'isrpia.inc'
      INTEGER  WFTYPI
C
C *** GET SOLUTION PARAMETERS *******************************************
C
      WFTYPI  = WFTYP
      IACALCI = IACALC
      EPSI    = EPS
      MAXITI  = MAXIT
      NSWEEPI = NSWEEP
      EPSACTI = EPSACT
      NDIVI   = NDIV
C
C *** END OF SUBROUTINE GETPARM *****************************************
C
      RETURN
      END

C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA BLKISO
C *** THIS SUBROUTINE PROVIDES INITIAL (DEFAULT) VALUES TO PROGRAM
C     PARAMETERS VIA DATA STATEMENTS
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA BLKISO
      INCLUDE 'isrpia.inc'
C
C *** DEFAULT VALUES *************************************************
C
      DATA TEMP/298.0/, R/82.0567D-6/, RH/0.9D0/, EPS/1D-6/, MAXIT/100/,
     &     TINY/1D-20/, GREAT/1D10/, ZERO/0.0D0/, ONE/1.0D0/,NSWEEP/4/, 
     &     TINY2/1D-11/,NDIV/5/
C
      DATA MOLAL/NIONS*0.0D0/, MOLALR/NPAIR*0.0D0/, GAMA/NPAIR*0.1D0/,
     &     GAMOU/NPAIR*1D10/,  GAMIN/NPAIR*1D10/,   CALAIN/.TRUE./,
     &     CALAOU/.TRUE./,     EPSACT/5D-2/,        ICLACT/0/,
     &     IACALC/1/,          WFTYP/2/
C
      DATA ERRSTK/NERRMX*0/,   ERRMSG/NERRMX*' '/,  NOFER/0/, 
     &     STKOFL/.FALSE./ 
C
      DATA IPROB/0/, METSTBL/0/
C
      DATA VERSION /'1.6 (11/10/04)'/
C
C *** OTHER PARAMETERS ***********************************************
C
      DATA SMW/58.5,142.,85.0,132.,80.0,53.5,98.0,98.0,115.,63.0,
     &         36.5,120.,247./
     &     IMW/ 1.0,23.0,18.0,35.5,96.0,97.0,63.0/,
     &     WMW/23.0,98.0,17.0,63.0,36.5/
C
      DATA ZZ/1,2,1,2,1,1,2,1,1,1,1,1,2/, Z /1,1,1,1,2,1,1/
C
C *** ZSR RELATIONSHIP PARAMETERS **************************************
C
C awas= ammonium sulfate
C
      DATA AWAS/33*100.,30,30,30,29.54,28.25,27.06,25.94,
     & 24.89,23.90,22.97,22.10,21.27,20.48,19.73,19.02,18.34,17.69,
     & 17.07,16.48,15.91,15.37,14.85,14.34,13.86,13.39,12.94,12.50,
     & 12.08,11.67,11.27,10.88,10.51,10.14, 9.79, 9.44, 9.10, 8.78,
     &  8.45, 8.14, 7.83, 7.53, 7.23, 6.94, 6.65, 6.36, 6.08, 5.81,
     &  5.53, 5.26, 4.99, 4.72, 4.46, 4.19, 3.92, 3.65, 3.38, 3.11,
     &  2.83, 2.54, 2.25, 1.95, 1.63, 1.31, 0.97, 0.63, 0.30, 0.001/
C
C awsn= sodium nitrate
C
      DATA AWSN/ 9*1.e5,685.59,
     & 451.00,336.46,268.48,223.41,191.28,
     & 167.20,148.46,133.44,121.12,110.83,
     & 102.09,94.57,88.03,82.29,77.20,72.65,68.56,64.87,61.51,58.44,
     & 55.62,53.03,50.63,48.40,46.32,44.39,42.57,40.87,39.27,37.76,
     & 36.33,34.98,33.70,32.48,31.32,30.21,29.16,28.14,27.18,26.25,
     & 25.35,24.50,23.67,22.87,22.11,21.36,20.65,19.95,19.28,18.62,
     & 17.99,17.37,16.77,16.18,15.61,15.05,14.51,13.98,13.45,12.94,
     & 12.44,11.94,11.46,10.98,10.51,10.04, 9.58, 9.12, 8.67, 8.22,
     &  7.77, 7.32, 6.88, 6.43, 5.98, 5.53, 5.07, 4.61, 4.15, 3.69,
     &  3.22, 2.76, 2.31, 1.87, 1.47, 1.10, 0.77, 0.48, 0.23, 0.001/
C
C awsc= sodium chloride
C
      DATA AWSC/
     &  100., 100., 100., 100., 100., 100., 100., 100., 100., 100.,
     &  100., 100., 100., 100., 100., 100., 100., 100., 100.,16.34,
     & 16.28,16.22,16.15,16.09,16.02,15.95,15.88,15.80,15.72,15.64,
     & 15.55,15.45,15.36,15.25,15.14,15.02,14.89,14.75,14.60,14.43,
     & 14.25,14.04,13.81,13.55,13.25,12.92,12.56,12.19,11.82,11.47,
     & 11.13,10.82,10.53,10.26,10.00, 9.76, 9.53, 9.30, 9.09, 8.88,
     &  8.67, 8.48, 8.28, 8.09, 7.90, 7.72, 7.54, 7.36, 7.17, 6.99,
     &  6.81, 6.63, 6.45, 6.27, 6.09, 5.91, 5.72, 5.53, 5.34, 5.14,
     &  4.94, 4.74, 4.53, 4.31, 4.09, 3.86, 3.62, 3.37, 3.12, 2.85,
     &  2.58, 2.30, 2.01, 1.72, 1.44, 1.16, 0.89, 0.64, 0.40, 0.18/
C
C awac= ammonium chloride
C
      DATA AWAC/
     &  100., 100., 100., 100., 100., 100., 100., 100., 100., 100.,
     &  100., 100., 100., 100., 100., 100., 100., 100., 100.,31.45,
     & 31.30,31.14,30.98,30.82,30.65,30.48,30.30,30.11,29.92,29.71,
     & 29.50,29.29,29.06,28.82,28.57,28.30,28.03,27.78,27.78,27.77,
     & 27.77,27.43,27.07,26.67,26.21,25.73,25.18,24.56,23.84,23.01,
     & 22.05,20.97,19.85,18.77,17.78,16.89,16.10,15.39,14.74,14.14,
     & 13.59,13.06,12.56,12.09,11.65,11.22,10.81,10.42,10.03, 9.66,
     &  9.30, 8.94, 8.59, 8.25, 7.92, 7.59, 7.27, 6.95, 6.63, 6.32,
     &  6.01, 5.70, 5.39, 5.08, 4.78, 4.47, 4.17, 3.86, 3.56, 3.25,
     &  2.94, 2.62, 2.30, 1.98, 1.65, 1.32, 0.97, 0.62, 0.26, 0.13/
C
C awss= sodium sulfate
C
      DATA AWSS/34*1.e5,23*14.30,14.21,12.53,11.47,
     & 10.66,10.01, 9.46, 8.99, 8.57, 8.19, 7.85, 7.54, 7.25, 6.98,
     &  6.74, 6.50, 6.29, 6.08, 5.88, 5.70, 5.52, 5.36, 5.20, 5.04,
     &  4.90, 4.75, 4.54, 4.34, 4.14, 3.93, 3.71, 3.49, 3.26, 3.02,
     &  2.76, 2.49, 2.20, 1.89, 1.55, 1.18, 0.82, 0.49, 0.22, 0.001/
C
C awab= ammonium bisulfate
C
      DATA AWAB/356.45,296.51,253.21,220.47,194.85,
     & 174.24,157.31,143.16,131.15,120.82,
     & 111.86,103.99,97.04,90.86,85.31,80.31,75.78,71.66,67.90,64.44,
     &  61.25,58.31,55.58,53.04,50.68,48.47,46.40,44.46,42.63,40.91,
     &  39.29,37.75,36.30,34.92,33.61,32.36,31.18,30.04,28.96,27.93,
     &  26.94,25.99,25.08,24.21,23.37,22.57,21.79,21.05,20.32,19.63,
     &  18.96,18.31,17.68,17.07,16.49,15.92,15.36,14.83,14.31,13.80,
     &  13.31,12.83,12.36,11.91,11.46,11.03,10.61,10.20, 9.80, 9.41,
     &   9.02, 8.64, 8.28, 7.91, 7.56, 7.21, 6.87, 6.54, 6.21, 5.88,
     &   5.56, 5.25, 4.94, 4.63, 4.33, 4.03, 3.73, 3.44, 3.14, 2.85,
     &   2.57, 2.28, 1.99, 1.71, 1.42, 1.14, 0.86, 0.57, 0.29, 0.001/
C
C awsa= sulfuric acid
C
      DATA AWSA/
     & 34.0,33.56,29.22,26.55,24.61,23.11,21.89,20.87,19.99,
     & 19.21,18.51,17.87,17.29,16.76,16.26,15.8,15.37,14.95,14.56,
     & 14.20,13.85,13.53,13.22,12.93,12.66,12.40,12.14,11.90,11.67,
     & 11.44,11.22,11.01,10.8,10.60,10.4,10.2,10.01,9.83,9.65,9.47,
     & 9.3,9.13,8.96,8.81,8.64,8.48,8.33,8.17,8.02,7.87,7.72,7.58,
     & 7.44,7.30,7.16,7.02,6.88,6.75,6.61,6.48,6.35,6.21,6.08,5.95,
     & 5.82,5.69,5.56,5.44,5.31,5.18,5.05,4.92,4.79,4.66,4.53,4.40,
     & 4.27,4.14,4.,3.87,3.73,3.6,3.46,3.31,3.17,3.02,2.87,2.72,
     & 2.56,2.4,2.23,2.05,1.87,1.68,1.48,1.27,1.05,0.807,0.552,0.281/
C
C awlc= (NH4)3H(SO4)2
C
      DATA AWLC/34*1.e5,17.0,16.5,15.94,15.31,14.71,14.14,
     & 13.60,13.08,12.59,12.12,11.68,11.25,10.84,10.44,10.07, 9.71,
     &  9.36, 9.02, 8.70, 8.39, 8.09, 7.80, 7.52, 7.25, 6.99, 6.73,
     &  6.49, 6.25, 6.02, 5.79, 5.57, 5.36, 5.15, 4.95, 4.76, 4.56,
     &  4.38, 4.20, 4.02, 3.84, 3.67, 3.51, 3.34, 3.18, 3.02, 2.87,
     &  2.72, 2.57, 2.42, 2.28, 2.13, 1.99, 1.85, 1.71, 1.57, 1.43,
     &  1.30, 1.16, 1.02, 0.89, 0.75, 0.61, 0.46, 0.32, 0.16, 0.001/
C
C awan= ammonium nitrate
C
      DATA AWAN/31*1.e5,
     &       97.17,92.28,87.66,83.15,78.87,74.84,70.98,67.46,64.11,
     & 60.98,58.07,55.37,52.85,50.43,48.24,46.19,44.26,42.40,40.70,
     & 39.10,37.54,36.10,34.69,33.35,32.11,30.89,29.71,28.58,27.46,
     & 26.42,25.37,24.33,23.89,22.42,21.48,20.56,19.65,18.76,17.91,
     & 17.05,16.23,15.40,14.61,13.82,13.03,12.30,11.55,10.83,10.14,
     &  9.44, 8.79, 8.13, 7.51, 6.91, 6.32, 5.75, 5.18, 4.65, 4.14,
     &  3.65, 3.16, 2.71, 2.26, 1.83, 1.42, 1.03, 0.66, 0.30, 0.001/
C
C awsb= sodium bisulfate
C
      DATA AWSB/173.72,156.88,142.80,130.85,120.57,
     & 111.64,103.80,96.88,90.71,85.18,
     & 80.20,75.69,71.58,67.82,64.37,61.19,58.26,55.53,53.00,50.64,
     & 48.44,46.37,44.44,42.61,40.90,39.27,37.74,36.29,34.91,33.61,
     & 32.36,31.18,30.05,28.97,27.94,26.95,26.00,25.10,24.23,23.39,
     & 22.59,21.81,21.07,20.35,19.65,18.98,18.34,17.71,17.11,16.52,
     & 15.95,15.40,14.87,14.35,13.85,13.36,12.88,12.42,11.97,11.53,
     & 11.10,10.69,10.28, 9.88, 9.49, 9.12, 8.75, 8.38, 8.03, 7.68,
     &  7.34, 7.01, 6.69, 6.37, 6.06, 5.75, 5.45, 5.15, 4.86, 4.58,
     &  4.30, 4.02, 3.76, 3.49, 3.23, 2.98, 2.73, 2.48, 2.24, 2.01,
     &  1.78, 1.56, 1.34, 1.13, 0.92, 0.73, 0.53, 0.35, 0.17, 0.001/
C
C *** END OF BLOCK DATA SUBPROGRAM *************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE INIT1
C *** THIS SUBROUTINE INITIALIZES ALL GLOBAL VARIABLES FOR AMMONIUM     
C     SULFATE AEROSOL SYSTEMS (SUBROUTINE ISRP1)
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE INIT1 (WI, RHI, TEMPI)
      INCLUDE 'isrpia.inc'
      DIMENSION WI(NCOMP)
      REAL      IC,GII,GI0,XX,LN10
      PARAMETER (LN10=2.3025851)
      ZZ(1) = 1 ! bkoo (07/23/03)
      ZZ(2) = 2 ! bkoo (07/23/03)
      ZZ(3) = 1 ! bkoo (07/23/03)
      ZZ(4) = 2 ! bkoo (07/23/03)
      ZZ(5) = 1 ! bkoo (07/23/03)
      ZZ(6) = 1 ! bkoo (07/23/03)
      ZZ(7) = 2 ! bkoo (07/23/03)
      ZZ(8) = 1 ! bkoo (07/23/03)
      ZZ(9) = 1 ! bkoo (07/23/03)
      ZZ(10)= 1 ! bkoo (07/23/03)
      ZZ(11)= 1 ! bkoo (07/23/03)
      ZZ(12)= 1 ! bkoo (07/23/03)
      ZZ(13)= 2 ! bkoo (07/23/03)
      Z(1) = 1  ! bkoo (07/23/03)
      Z(2) = 1  ! bkoo (07/23/03)
      Z(3) = 1  ! bkoo (07/23/03)
      Z(4) = 1  ! bkoo (07/23/03)
      Z(5) = 2  ! bkoo (07/23/03)
      Z(6) = 1  ! bkoo (07/23/03)
      Z(7) = 1  ! bkoo (07/23/03)
C
C *** SAVE INPUT VARIABLES IN COMMON BLOCK ******************************
C
      IF (IPROB.EQ.0) THEN                 ! FORWARD CALCULATION
         DO 10 I=1,NCOMP
            W(I) = MAX(WI(I), TINY)
10       CONTINUE
      ELSE
         DO 15 I=1,NCOMP                   ! REVERSE CALCULATION
            WAER(I) = MAX(WI(I), TINY)
            W(I)    = ZERO
15       CONTINUE
      ENDIF
      RH      = RHI
      TEMP    = TEMPI
C
C *** CALCULATE EQUILIBRIUM CONSTANTS ***********************************
C
      XK1  = 1.015e-2  ! HSO4(aq)         <==> H(aq)     + SO4(aq)
      XK21 = 57.639    ! NH3(g)           <==> NH3(aq)
      XK22 = 1.805e-5  ! NH3(aq)          <==> NH4(aq)   + OH(aq)
      XK7  = 1.817     ! (NH4)2SO4(s)     <==> 2*NH4(aq) + SO4(aq)
      XK12 = 1.382e2   ! NH4HSO4(s)       <==> NH4(aq)   + HSO4(aq)
      XK13 = 29.268    ! (NH4)3H(SO4)2(s) <==> 3*NH4(aq) + HSO4(aq) + SO4(aq)
      XKW  = 1.010e-14 ! H2O              <==> H(aq)     + OH(aq)
C
      IF (INT(TEMP) .NE. 298) THEN   ! FOR T != 298K or 298.15K
         T0  = 298.15
         T0T = T0/TEMP
         COEF= 1.0+LOG(T0T)-T0T
         XK1 = XK1 *EXP(  8.85*(T0T-1.0) + 25.140*COEF)
         XK21= XK21*EXP( 13.79*(T0T-1.0) -  5.393*COEF)
         XK22= XK22*EXP( -1.50*(T0T-1.0) + 26.920*COEF)
         XK7 = XK7 *EXP( -2.65*(T0T-1.0) + 38.570*COEF)
         XK12= XK12*EXP( -2.87*(T0T-1.0) + 15.830*COEF)
         XK13= XK13*EXP( -5.19*(T0T-1.0) + 54.400*COEF)
         XKW = XKW *EXP(-22.52*(T0T-1.0) + 26.920*COEF)
      ENDIF
      XK2 = XK21*XK22       
C
C *** CALCULATE DELIQUESCENCE RELATIVE HUMIDITIES (UNICOMPONENT) ********
C
      DRH2SO4  = 0.0000D0
      DRNH42S4 = 0.7997D0
      DRNH4HS4 = 0.4000D0
      DRLC     = 0.6900D0
      IF (INT(TEMP) .NE. 298) THEN
         T0       = 298.15d0
         TCF      = 1.0/TEMP - 1.0/T0
         DRNH42S4 = DRNH42S4*EXP( 80.*TCF) 
         DRNH4HS4 = DRNH4HS4*EXP(384.*TCF) 
         DRLC     = DRLC    *EXP(186.*TCF) 
      ENDIF
C
C *** CALCULATE MUTUAL DELIQUESCENCE RELATIVE HUMIDITIES ****************
C
      DRMLCAB = 0.3780D0              ! (NH4)3H(SO4)2 & NH4HSO4 
      DRMLCAS = 0.6900D0              ! (NH4)3H(SO4)2 & (NH4)2SO4 
CCC      IF (INT(TEMP) .NE. 298) THEN      ! For the time being.
CCC         T0       = 298.15d0
CCC         TCF      = 1.0/TEMP - 1.0/T0
CCC         DRMLCAB  = DRMLCAB*EXP(507.506*TCF) 
CCC         DRMLCAS  = DRMLCAS*EXP(133.865*TCF) 
CCC      ENDIF
C
C *** LIQUID PHASE ******************************************************
C
      CHNO3  = ZERO
      CHCL   = ZERO
      CH2SO4 = ZERO
      COH    = ZERO
      WATER  = TINY
C
      DO 20 I=1,NPAIR
         MOLALR(I)=ZERO
         GAMA(I)  =0.1
         GAMIN(I) =GREAT
         GAMOU(I) =GREAT
         M0(I)    =1d5
 20   CONTINUE
C
      DO 30 I=1,NPAIR
         GAMA(I) = 0.1d0
 30   CONTINUE
C
      DO 40 I=1,NIONS
         MOLAL(I)=ZERO
40    CONTINUE
      COH = ZERO
C
      DO 50 I=1,NGASAQ
         GASAQ(I)=ZERO
50    CONTINUE
C
C *** SOLID PHASE *******************************************************
C
      CNH42S4= ZERO
      CNH4HS4= ZERO
      CNACL  = ZERO
      CNA2SO4= ZERO
      CNANO3 = ZERO
      CNH4NO3= ZERO
      CNH4CL = ZERO
      CNAHSO4= ZERO
      CLC    = ZERO
C
C *** GAS PHASE *********************************************************
C
      GNH3   = ZERO
      GHNO3  = ZERO
      GHCL   = ZERO
C
C *** CALCULATE ZSR PARAMETERS ******************************************
C
      IRH    = MIN (INT(RH*NZSR+0.5),NZSR)  ! Position in ZSR arrays
      IRH    = MAX (IRH, 1)
C
      M0(01) = AWSC(IRH)      ! NACl
      IF (M0(01) .LT. 100.0) THEN
         IC = M0(01)
         CALL KMTAB(IC,298.0,     GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(01) = M0(01)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(02) = AWSS(IRH)      ! (NA)2SO4
      IF (M0(02) .LT. 100.0) THEN
         IC = 3.0*M0(02)
         CALL KMTAB(IC,298.0,     XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(02) = M0(02)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(03) = AWSN(IRH)      ! NANO3
      IF (M0(03) .LT. 100.0) THEN
         IC = M0(03)
         CALL KMTAB(IC,298.0,     XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(03) = M0(03)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(04) = AWAS(IRH)      ! (NH4)2SO4
      IF (M0(04) .LT. 100.0) THEN
         IC = 3.0*M0(04)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(04) = M0(04)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(05) = AWAN(IRH)      ! NH4NO3
      IF (M0(05) .LT. 100.0) THEN
         IC     = M0(05)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX)
         M0(05) = M0(05)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(06) = AWAC(IRH)      ! NH4CL
      IF (M0(06) .LT. 100.0) THEN
         IC = M0(06)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX)
         M0(06) = M0(06)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(07) = AWSA(IRH)      ! 2H-SO4
      IF (M0(07) .LT. 100.0) THEN
         IC = 3.0*M0(07)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX)
         M0(07) = M0(07)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(08) = AWSA(IRH)      ! H-HSO4
CCC      IF (M0(08) .LT. 100.0) THEN   ! This is redundant because M0(8) is not used
CCC         IC = M0(08)
CCC         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCCCCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX)
CCC         M0(08) = M0(08)*EXP(LN10*(GI0-GII))
CCC      ENDIF
C
      M0(09) = AWAB(IRH)      ! NH4HSO4
      IF (M0(09) .LT. 100.0) THEN
         IC = M0(09)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX)
         M0(09) = M0(09)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(12) = AWSB(IRH)      ! NAHSO4
      IF (M0(12) .LT. 100.0) THEN
         IC = M0(12)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GI0)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GII)
         M0(12) = M0(12)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(13) = AWLC(IRH)      ! (NH4)3H(SO4)2
      IF (M0(13) .LT. 100.0) THEN
         IC     = 4.0*M0(13)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G130   = 0.2*(3.0*GI0+2.0*GII)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G13I   = 0.2*(3.0*GI0+2.0*GII)
         M0(13) = M0(13)*EXP(LN10*SNGL(G130-G13I))
      ENDIF
C
C *** OTHER INITIALIZATIONS *********************************************
C
      ICLACT  = 0
      CALAOU  = .TRUE.
      CALAIN  = .TRUE.
      FRST    = .TRUE.
      SCASE   = '??'
      SULRATW = 2.D0
      NOFER   = 0
      STKOFL  =.FALSE.
      SODRAT  = ZERO
      DO 60 I=1,NERRMX
         ERRSTK(I) =-999
         ERRMSG(I) = 'MESSAGE N/A'
   60 CONTINUE
C
C *** END OF SUBROUTINE INIT1 *******************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE INIT2
C *** THIS SUBROUTINE INITIALIZES ALL GLOBAL VARIABLES FOR AMMONIUM,
C     NITRATE, SULFATE AEROSOL SYSTEMS (SUBROUTINE ISRP2)
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE INIT2 (WI, RHI, TEMPI)
      INCLUDE 'isrpia.inc'
      DIMENSION WI(NCOMP)
      REAL      IC,GII,GI0,XX,LN10
      PARAMETER (LN10=2.3025851)
      ZZ(1) = 1 ! bkoo (07/23/03)
      ZZ(2) = 2 ! bkoo (07/23/03)
      ZZ(3) = 1 ! bkoo (07/23/03)
      ZZ(4) = 2 ! bkoo (07/23/03)
      ZZ(5) = 1 ! bkoo (07/23/03)
      ZZ(6) = 1 ! bkoo (07/23/03)
      ZZ(7) = 2 ! bkoo (07/23/03)
      ZZ(8) = 1 ! bkoo (07/23/03)
      ZZ(9) = 1 ! bkoo (07/23/03)
      ZZ(10)= 1 ! bkoo (07/23/03)
      ZZ(11)= 1 ! bkoo (07/23/03)
      ZZ(12)= 1 ! bkoo (07/23/03)
      ZZ(13)= 2 ! bkoo (07/23/03)
      Z(1) = 1  ! bkoo (07/23/03)
      Z(2) = 1  ! bkoo (07/23/03)
      Z(3) = 1  ! bkoo (07/23/03)
      Z(4) = 1  ! bkoo (07/23/03)
      Z(5) = 2  ! bkoo (07/23/03)
      Z(6) = 1  ! bkoo (07/23/03)
      Z(7) = 1  ! bkoo (07/23/03)
C
C *** SAVE INPUT VARIABLES IN COMMON BLOCK ******************************
C
      IF (IPROB.EQ.0) THEN                 ! FORWARD CALCULATION
         DO 10 I=1,NCOMP
            W(I) = MAX(WI(I), TINY)
10       CONTINUE
      ELSE
         DO 15 I=1,NCOMP                   ! REVERSE CALCULATION
            WAER(I) = MAX(WI(I), TINY)
            W(I)    = ZERO
15       CONTINUE
      ENDIF
      RH      = RHI
      TEMP    = TEMPI
C
C *** CALCULATE EQUILIBRIUM CONSTANTS ***********************************
C
      XK1  = 1.015e-2  ! HSO4(aq)         <==> H(aq)     + SO4(aq)
      XK21 = 57.639    ! NH3(g)           <==> NH3(aq)
      XK22 = 1.805e-5  ! NH3(aq)          <==> NH4(aq)   + OH(aq)
      XK4  = 2.511e6   ! HNO3(g)          <==> H(aq)     + NO3(aq) ! ISORR
CCC      XK4  = 3.638e6   ! HNO3(g)          <==> H(aq)     + NO3(aq) ! SEQUIL
      XK41 = 2.100e5   ! HNO3(g)          <==> HNO3(aq)
      XK7  = 1.817     ! (NH4)2SO4(s)     <==> 2*NH4(aq) + SO4(aq)
      XK10 = 5.746e-17 ! NH4NO3(s)        <==> NH3(g)    + HNO3(g) ! ISORR
CCC      XK10 = 2.985e-17 ! NH4NO3(s)        <==> NH3(g)    + HNO3(g) ! SEQUIL
      XK12 = 1.382e2   ! NH4HSO4(s)       <==> NH4(aq)   + HSO4(aq)
      XK13 = 29.268    ! (NH4)3H(SO4)2(s) <==> 3*NH4(aq) + HSO4(aq) + SO4(aq)
      XKW  = 1.010e-14 ! H2O              <==> H(aq)     + OH(aq)
C
      IF (INT(TEMP) .NE. 298) THEN   ! FOR T != 298K or 298.15K
         T0  = 298.15D0
         T0T = T0/TEMP
         COEF= 1.0+LOG(T0T)-T0T
         XK1 = XK1 *EXP(  8.85*(T0T-1.0) + 25.140*COEF)
         XK21= XK21*EXP( 13.79*(T0T-1.0) -  5.393*COEF)
         XK22= XK22*EXP( -1.50*(T0T-1.0) + 26.920*COEF)
         XK4 = XK4 *EXP( 29.17*(T0T-1.0) + 16.830*COEF) !ISORR
CCC         XK4 = XK4 *EXP( 29.47*(T0T-1.0) + 16.840*COEF) ! SEQUIL
         XK41= XK41*EXP( 29.17*(T0T-1.0) + 16.830*COEF)
         XK7 = XK7 *EXP( -2.65*(T0T-1.0) + 38.570*COEF)
         XK10= XK10*EXP(-74.38*(T0T-1.0) +  6.120*COEF) ! ISORR
CCC         XK10= XK10*EXP(-75.11*(T0T-1.0) + 13.460*COEF) ! SEQUIL
         XK12= XK12*EXP( -2.87*(T0T-1.0) + 15.830*COEF)
         XK13= XK13*EXP( -5.19*(T0T-1.0) + 54.400*COEF)
         XKW = XKW *EXP(-22.52*(T0T-1.0) + 26.920*COEF)
      ENDIF
      XK2  = XK21*XK22       
      XK42 = XK4/XK41
C
C *** CALCULATE DELIQUESCENCE RELATIVE HUMIDITIES (UNICOMPONENT) ********
C
      DRH2SO4  = ZERO
      DRNH42S4 = 0.7997D0
      DRNH4HS4 = 0.4000D0
      DRNH4NO3 = 0.6183D0
      DRLC     = 0.6900D0
      IF (INT(TEMP) .NE. 298) THEN
         T0       = 298.15D0
         TCF      = 1.0/TEMP - 1.0/T0
         DRNH4NO3 = DRNH4NO3*EXP(852.*TCF)
         DRNH42S4 = DRNH42S4*EXP( 80.*TCF)
         DRNH4HS4 = DRNH4HS4*EXP(384.*TCF) 
         DRLC     = DRLC    *EXP(186.*TCF) 
      ENDIF
C
C *** CALCULATE MUTUAL DELIQUESCENCE RELATIVE HUMIDITIES ****************
C
      DRMLCAB = 0.3780D0              ! (NH4)3H(SO4)2 & NH4HSO4 
      DRMLCAS = 0.6900D0              ! (NH4)3H(SO4)2 & (NH4)2SO4 
      DRMASAN = 0.6000D0              ! (NH4)2SO4     & NH4NO3
CCC      IF (INT(TEMP) .NE. 298) THEN    ! For the time being
CCC         T0       = 298.15d0
CCC         TCF      = 1.0/TEMP - 1.0/T0
CCC         DRMLCAB  = DRMLCAB*EXP( 507.506*TCF) 
CCC         DRMLCAS  = DRMLCAS*EXP( 133.865*TCF) 
CCC         DRMASAN  = DRMASAN*EXP(1269.068*TCF)
CCC      ENDIF
C
C *** LIQUID PHASE ******************************************************
C
      CHNO3  = ZERO
      CHCL   = ZERO
      CH2SO4 = ZERO
      COH    = ZERO
      WATER  = TINY
C
      DO 20 I=1,NPAIR
         MOLALR(I)=ZERO
         GAMA(I)  =0.1
         GAMIN(I) =GREAT
         GAMOU(I) =GREAT
         M0(I)    =1d5
 20   CONTINUE
C
      DO 30 I=1,NPAIR
         GAMA(I) = 0.1d0
 30   CONTINUE
C
      DO 40 I=1,NIONS
         MOLAL(I)=ZERO
40    CONTINUE
      COH = ZERO
C
      DO 50 I=1,NGASAQ
         GASAQ(I)=ZERO
50    CONTINUE
C
C *** SOLID PHASE *******************************************************
C
      CNH42S4= ZERO
      CNH4HS4= ZERO
      CNACL  = ZERO
      CNA2SO4= ZERO
      CNANO3 = ZERO
      CNH4NO3= ZERO
      CNH4CL = ZERO
      CNAHSO4= ZERO
      CLC    = ZERO
C
C *** GAS PHASE *********************************************************
C
      GNH3   = ZERO
      GHNO3  = ZERO
      GHCL   = ZERO
C
C *** CALCULATE ZSR PARAMETERS ******************************************
C
      IRH    = MIN (INT(RH*NZSR+0.5),NZSR)  ! Position in ZSR arrays
      IRH    = MAX (IRH, 1)
C
      M0(01) = AWSC(IRH)      ! NACl
      IF (M0(01) .LT. 100.0) THEN
         IC = M0(01)
         CALL KMTAB(IC,298.0,     GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(01) = M0(01)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(02) = AWSS(IRH)      ! (NA)2SO4
      IF (M0(02) .LT. 100.0) THEN
         IC = 3.0*M0(02)
         CALL KMTAB(IC,298.0,     XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(02) = M0(02)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(03) = AWSN(IRH)      ! NANO3
      IF (M0(03) .LT. 100.0) THEN
         IC = M0(03)
         CALL KMTAB(IC,298.0,     XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(03) = M0(03)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(04) = AWAS(IRH)      ! (NH4)2SO4
      IF (M0(04) .LT. 100.0) THEN
         IC = 3.0*M0(04)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(04) = M0(04)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(05) = AWAN(IRH)      ! NH4NO3
      IF (M0(05) .LT. 100.0) THEN
         IC     = M0(05)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX)
         M0(05) = M0(05)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(06) = AWAC(IRH)      ! NH4CL
      IF (M0(06) .LT. 100.0) THEN
         IC = M0(06)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX)
         M0(06) = M0(06)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(07) = AWSA(IRH)      ! 2H-SO4
      IF (M0(07) .LT. 100.0) THEN
         IC = 3.0*M0(07)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX)
         M0(07) = M0(07)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(08) = AWSA(IRH)      ! H-HSO4
CCC      IF (M0(08) .LT. 100.0) THEN   ! This is redundant because M0(8) is not used
CCC         IC = M0(08)
CCC         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCCCCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX)
CCC         M0(08) = M0(08)*EXP(LN10*(GI0-GII))
CCC      ENDIF
C
      M0(09) = AWAB(IRH)      ! NH4HSO4
      IF (M0(09) .LT. 100.0) THEN
         IC = M0(09)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX)
         M0(09) = M0(09)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(12) = AWSB(IRH)      ! NAHSO4
      IF (M0(12) .LT. 100.0) THEN
         IC = M0(12)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GI0)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GII)
         M0(12) = M0(12)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(13) = AWLC(IRH)      ! (NH4)3H(SO4)2
      IF (M0(13) .LT. 100.0) THEN
         IC     = 4.0*M0(13)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G130   = 0.2*(3.0*GI0+2.0*GII)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G13I   = 0.2*(3.0*GI0+2.0*GII)
         M0(13) = M0(13)*EXP(LN10*SNGL(G130-G13I))
      ENDIF
C
C *** OTHER INITIALIZATIONS *********************************************
C
      ICLACT  = 0
      CALAOU  = .TRUE.
      CALAIN  = .TRUE.
      FRST    = .TRUE.
      SCASE   = '??'
      SULRATW = 2.D0
      SODRAT  = ZERO
      NOFER   = 0
      STKOFL  =.FALSE.
      DO 60 I=1,NERRMX
         ERRSTK(I) =-999
         ERRMSG(I) = 'MESSAGE N/A'
   60 CONTINUE
C
C *** END OF SUBROUTINE INIT2 *******************************************
C
      END





C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE INIT3
C *** THIS SUBROUTINE INITIALIZES ALL GLOBAL VARIABLES FOR AMMONIUM,
C     SODIUM, CHLORIDE, NITRATE, SULFATE AEROSOL SYSTEMS (SUBROUTINE 
C     ISRP3)
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE INIT3 (WI, RHI, TEMPI)
      INCLUDE 'isrpia.inc'
      DIMENSION WI(NCOMP)
      REAL      IC,GII,GI0,XX,LN10
      PARAMETER (LN10=2.3025851)
      ZZ(1) = 1 ! bkoo (07/23/03)
      ZZ(2) = 2 ! bkoo (07/23/03)
      ZZ(3) = 1 ! bkoo (07/23/03)
      ZZ(4) = 2 ! bkoo (07/23/03)
      ZZ(5) = 1 ! bkoo (07/23/03)
      ZZ(6) = 1 ! bkoo (07/23/03)
      ZZ(7) = 2 ! bkoo (07/23/03)
      ZZ(8) = 1 ! bkoo (07/23/03)
      ZZ(9) = 1 ! bkoo (07/23/03)
      ZZ(10)= 1 ! bkoo (07/23/03)
      ZZ(11)= 1 ! bkoo (07/23/03)
      ZZ(12)= 1 ! bkoo (07/23/03)
      ZZ(13)= 2 ! bkoo (07/23/03)
      Z(1) = 1  ! bkoo (07/23/03)
      Z(2) = 1  ! bkoo (07/23/03)
      Z(3) = 1  ! bkoo (07/23/03)
      Z(4) = 1  ! bkoo (07/23/03)
      Z(5) = 2  ! bkoo (07/23/03)
      Z(6) = 1  ! bkoo (07/23/03)
      Z(7) = 1  ! bkoo (07/23/03)
C
C *** SAVE INPUT VARIABLES IN COMMON BLOCK ******************************
C
      IF (IPROB.EQ.0) THEN                 ! FORWARD CALCULATION
         DO 10 I=1,NCOMP
            W(I) = MAX(WI(I), TINY)
10       CONTINUE
      ELSE
         DO 15 I=1,NCOMP                   ! REVERSE CALCULATION
            WAER(I) = MAX(WI(I), TINY)
            W(I)    = ZERO
15       CONTINUE
      ENDIF
      RH      = RHI
      TEMP    = TEMPI
C
C *** CALCULATE EQUILIBRIUM CONSTANTS ***********************************
C
      XK1  = 1.015D-2  ! HSO4(aq)         <==> H(aq)     + SO4(aq)
      XK21 = 57.639D0  ! NH3(g)           <==> NH3(aq)
      XK22 = 1.805D-5  ! NH3(aq)          <==> NH4(aq)   + OH(aq)
      XK3  = 1.971D6   ! HCL(g)           <==> H(aq)     + CL(aq)
      XK31 = 2.500e3   ! HCL(g)           <==> HCL(aq)
      XK4  = 2.511e6   ! HNO3(g)          <==> H(aq)     + NO3(aq) ! ISORR
CCC      XK4  = 3.638e6   ! HNO3(g)          <==> H(aq)     + NO3(aq) ! SEQUIL
      XK41 = 2.100e5   ! HNO3(g)          <==> HNO3(aq)
      XK5  = 0.4799D0  ! NA2SO4(s)        <==> 2*NA(aq)  + SO4(aq)
      XK6  = 1.086D-16 ! NH4CL(s)         <==> NH3(g)    + HCL(g)
      XK7  = 1.817D0   ! (NH4)2SO4(s)     <==> 2*NH4(aq) + SO4(aq)
      XK8  = 37.661D0  ! NACL(s)          <==> NA(aq)    + CL(aq)
      XK10 = 5.746D-17 ! NH4NO3(s)        <==> NH3(g)    + HNO3(g) ! ISORR
CCC      XK10 = 2.985e-17 ! NH4NO3(s)        <==> NH3(g)    + HNO3(g) ! SEQUIL
      XK11 = 2.413D4   ! NAHSO4(s)        <==> NA(aq)    + HSO4(aq)
      XK12 = 1.382D2   ! NH4HSO4(s)       <==> NH4(aq)   + HSO4(aq)
      XK13 = 29.268D0  ! (NH4)3H(SO4)2(s) <==> 3*NH4(aq) + HSO4(aq) + SO4(aq)
      XK14 = 22.05D0   ! NH4CL(s)         <==> NH4(aq)   + CL(aq)
      XKW  = 1.010D-14 ! H2O              <==> H(aq)     + OH(aq)
      XK9  = 11.977D0  ! NANO3(s)         <==> NA(aq)    + NO3(aq)
C
      IF (INT(TEMP) .NE. 298) THEN   ! FOR T != 298K or 298.15K
         T0  = 298.15D0
         T0T = T0/TEMP
         COEF= 1.0+LOG(T0T)-T0T
         XK1 = XK1 *EXP(  8.85*(T0T-1.0) + 25.140*COEF)
         XK21= XK21*EXP( 13.79*(T0T-1.0) -  5.393*COEF)
         XK22= XK22*EXP( -1.50*(T0T-1.0) + 26.920*COEF)
         XK3 = XK3 *EXP( 30.20*(T0T-1.0) + 19.910*COEF)
         XK31= XK31*EXP( 30.20*(T0T-1.0) + 19.910*COEF)
         XK4 = XK4 *EXP( 29.17*(T0T-1.0) + 16.830*COEF) !ISORR
CCC         XK4 = XK4 *EXP( 29.47*(T0T-1.0) + 16.840*COEF) ! SEQUIL
         XK41= XK41*EXP( 29.17*(T0T-1.0) + 16.830*COEF)
         XK5 = XK5 *EXP(  0.98*(T0T-1.0) + 39.500*COEF)
         XK6 = XK6 *EXP(-71.00*(T0T-1.0) +  2.400*COEF)
         XK7 = XK7 *EXP( -2.65*(T0T-1.0) + 38.570*COEF)
         XK8 = XK8 *EXP( -1.56*(T0T-1.0) + 16.900*COEF)
         XK9 = XK9 *EXP( -8.22*(T0T-1.0) + 16.010*COEF)
         XK10= XK10*EXP(-74.38*(T0T-1.0) +  6.120*COEF) ! ISORR
CCC         XK10= XK10*EXP(-75.11*(T0T-1.0) + 13.460*COEF) ! SEQUIL
         XK11= XK11*EXP(  0.79*(T0T-1.0) + 14.746*COEF)
         XK12= XK12*EXP( -2.87*(T0T-1.0) + 15.830*COEF)
         XK13= XK13*EXP( -5.19*(T0T-1.0) + 54.400*COEF)
         XK14= XK14*EXP( 24.55*(T0T-1.0) + 16.900*COEF)
         XKW = XKW *EXP(-22.52*(T0T-1.0) + 26.920*COEF)
      ENDIF
      XK2  = XK21*XK22       
      XK42 = XK4/XK41
      XK32 = XK3/XK31
C
C *** CALCULATE DELIQUESCENCE RELATIVE HUMIDITIES (UNICOMPONENT) ********
C
      DRH2SO4  = ZERO
      DRNH42S4 = 0.7997D0
      DRNH4HS4 = 0.4000D0
      DRLC     = 0.6900D0
      DRNACL   = 0.7528D0
      DRNANO3  = 0.7379D0
      DRNH4CL  = 0.7710D0
      DRNH4NO3 = 0.6183D0
      DRNA2SO4 = 0.9300D0
      DRNAHSO4 = 0.5200D0
      IF (INT(TEMP) .NE. 298) THEN
         T0       = 298.15D0
         TCF      = 1.0/TEMP - 1.0/T0
         DRNACL   = DRNACL  *EXP( 25.*TCF)
         DRNANO3  = DRNANO3 *EXP(304.*TCF)
         DRNA2SO4 = DRNA2SO4*EXP( 80.*TCF)
         DRNH4NO3 = DRNH4NO3*EXP(852.*TCF)
         DRNH42S4 = DRNH42S4*EXP( 80.*TCF)
         DRNH4HS4 = DRNH4HS4*EXP(384.*TCF) 
         DRLC     = DRLC    *EXP(186.*TCF)
         DRNH4CL  = DRNH4Cl *EXP(239.*TCF)
         DRNAHSO4 = DRNAHSO4*EXP(-45.*TCF) 
      ENDIF
C
C *** CALCULATE MUTUAL DELIQUESCENCE RELATIVE HUMIDITIES ****************
C
      DRMLCAB = 0.378D0    ! (NH4)3H(SO4)2 & NH4HSO4 
      DRMLCAS = 0.690D0    ! (NH4)3H(SO4)2 & (NH4)2SO4 
      DRMASAN = 0.600D0    ! (NH4)2SO4     & NH4NO3
      DRMG1   = 0.460D0    ! (NH4)2SO4, NH4NO3, NA2SO4, NH4CL
      DRMG2   = 0.691D0    ! (NH4)2SO4, NA2SO4, NH4CL
      DRMG3   = 0.697D0    ! (NH4)2SO4, NA2SO4
      DRMH1   = 0.240D0    ! NA2SO4, NANO3, NACL, NH4NO3, NH4CL
      DRMH2   = 0.596D0    ! NA2SO4, NANO3, NACL, NH4CL
      DRMI1   = 0.240D0    ! LC, NAHSO4, NH4HSO4, NA2SO4, (NH4)2SO4
      DRMI2   = 0.363D0    ! LC, NAHSO4, NA2SO4, (NH4)2SO4  - NO DATA -
      DRMI3   = 0.610D0    ! LC, NA2SO4, (NH4)2SO4 
      DRMQ1   = 0.494D0    ! (NH4)2SO4, NH4NO3, NA2SO4
      DRMR1   = 0.663D0    ! NA2SO4, NANO3, NACL
      DRMR2   = 0.735D0    ! NA2SO4, NACL
      DRMR3   = 0.673D0    ! NANO3, NACL
      DRMR4   = 0.694D0    ! NA2SO4, NACL, NH4CL
      DRMR5   = 0.731D0    ! NA2SO4, NH4CL
      DRMR6   = 0.596D0    ! NA2SO4, NANO3, NH4CL
      DRMR7   = 0.380D0    ! NA2SO4, NANO3, NACL, NH4NO3
      DRMR8   = 0.380D0    ! NA2SO4, NACL, NH4NO3
      DRMR9   = 0.494D0    ! NA2SO4, NH4NO3
      DRMR10  = 0.476D0    ! NA2SO4, NANO3, NH4NO3
      DRMR11  = 0.340D0    ! NA2SO4, NACL, NH4NO3, NH4CL
      DRMR12  = 0.460D0    ! NA2SO4, NH4NO3, NH4CL
      DRMR13  = 0.438D0    ! NA2SO4, NANO3, NH4NO3, NH4CL
CCC      IF (INT(TEMP) .NE. 298) THEN
CCC         T0       = 298.15d0
CCC         TCF      = 1.0/TEMP - 1.0/T0
CCC         DRMLCAB  = DRMLCAB*EXP( 507.506*TCF) 
CCC         DRMLCAS  = DRMLCAS*EXP( 133.865*TCF) 
CCC         DRMASAN  = DRMASAN*EXP(1269.068*TCF)
CCC         DRMG1    = DRMG1  *EXP( 572.207*TCF)
CCC         DRMG2    = DRMG2  *EXP(  58.166*TCF)
CCC         DRMG3    = DRMG3  *EXP(  22.253*TCF)
CCC         DRMH1    = DRMH1  *EXP(2116.542*TCF)
CCC         DRMH2    = DRMH2  *EXP( 650.549*TCF)
CCC         DRMI1    = DRMI1  *EXP( 565.743*TCF)
CCC         DRMI2    = DRMI2  *EXP(  91.745*TCF)
CCC         DRMI3    = DRMI3  *EXP( 161.272*TCF)
CCC         DRMQ1    = DRMQ1  *EXP(1616.621*TCF)
CCC         DRMR1    = DRMR1  *EXP( 292.564*TCF)
CCC         DRMR2    = DRMR2  *EXP(  14.587*TCF)
CCC         DRMR3    = DRMR3  *EXP( 307.907*TCF)
CCC         DRMR4    = DRMR4  *EXP(  97.605*TCF)
CCC         DRMR5    = DRMR5  *EXP(  98.523*TCF)
CCC         DRMR6    = DRMR6  *EXP( 465.500*TCF)
CCC         DRMR7    = DRMR7  *EXP( 324.425*TCF)
CCC         DRMR8    = DRMR8  *EXP(2660.184*TCF)
CCC         DRMR9    = DRMR9  *EXP(1617.178*TCF)
CCC         DRMR10   = DRMR10 *EXP(1745.226*TCF)
CCC         DRMR11   = DRMR11 *EXP(3691.328*TCF)
CCC         DRMR12   = DRMR12 *EXP(1836.842*TCF)
CCC         DRMR13   = DRMR13 *EXP(1967.938*TCF)
CCC      ENDIF
C
C *** LIQUID PHASE ******************************************************
C
      CHNO3  = ZERO
      CHCL   = ZERO
      CH2SO4 = ZERO
      COH    = ZERO
      WATER  = TINY
C
      DO 20 I=1,NPAIR
         MOLALR(I)=ZERO
         GAMA(I)  =0.1
         GAMIN(I) =GREAT
         GAMOU(I) =GREAT
         M0(I)    =1d5
 20   CONTINUE
C
      DO 30 I=1,NPAIR
         GAMA(I) = 0.1d0
 30   CONTINUE
C
      DO 40 I=1,NIONS
         MOLAL(I)=ZERO
40    CONTINUE
      COH = ZERO
C
      DO 50 I=1,NGASAQ
         GASAQ(I)=ZERO
50    CONTINUE
C
C *** SOLID PHASE *******************************************************
C
      CNH42S4= ZERO
      CNH4HS4= ZERO
      CNACL  = ZERO
      CNA2SO4= ZERO
      CNANO3 = ZERO
      CNH4NO3= ZERO
      CNH4CL = ZERO
      CNAHSO4= ZERO
      CLC    = ZERO
C
C *** GAS PHASE *********************************************************
C
      GNH3   = ZERO
      GHNO3  = ZERO
      GHCL   = ZERO
C
C *** CALCULATE ZSR PARAMETERS ******************************************
C
      IRH    = MIN (INT(RH*NZSR+0.5),NZSR)  ! Position in ZSR arrays
      IRH    = MAX (IRH, 1)
C
      M0(01) = AWSC(IRH)      ! NACl
      IF (M0(01) .LT. 100.0) THEN
         IC = M0(01)
         CALL KMTAB(IC,298.0,     GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(01) = M0(01)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(02) = AWSS(IRH)      ! (NA)2SO4
      IF (M0(02) .LT. 100.0) THEN
         IC = 3.0*M0(02)
         CALL KMTAB(IC,298.0,     XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(02) = M0(02)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(03) = AWSN(IRH)      ! NANO3
      IF (M0(03) .LT. 100.0) THEN
         IC = M0(03)
         CALL KMTAB(IC,298.0,     XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(03) = M0(03)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(04) = AWAS(IRH)      ! (NH4)2SO4
      IF (M0(04) .LT. 100.0) THEN
         IC = 3.0*M0(04)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX,XX)
         M0(04) = M0(04)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(05) = AWAN(IRH)      ! NH4NO3
      IF (M0(05) .LT. 100.0) THEN
         IC     = M0(05)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX,XX)
         M0(05) = M0(05)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(06) = AWAC(IRH)      ! NH4CL
      IF (M0(06) .LT. 100.0) THEN
         IC = M0(06)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX,XX)
         M0(06) = M0(06)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(07) = AWSA(IRH)      ! 2H-SO4
      IF (M0(07) .LT. 100.0) THEN
         IC = 3.0*M0(07)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX,XX)
         M0(07) = M0(07)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(08) = AWSA(IRH)      ! H-HSO4
CCC      IF (M0(08) .LT. 100.0) THEN   ! This is redundant because M0(8) is not used
CCC         IC = M0(08)
CCC         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX,XX)
CCCCCC         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX,XX)
CCC         M0(08) = M0(08)*EXP(LN10*(GI0-GII))
CCC      ENDIF
C
      M0(09) = AWAB(IRH)      ! NH4HSO4
      IF (M0(09) .LT. 100.0) THEN
         IC = M0(09)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,GI0,XX,XX,XX)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,GII,XX,XX,XX)
         M0(09) = M0(09)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(12) = AWSB(IRH)      ! NAHSO4
      IF (M0(12) .LT. 100.0) THEN
         IC = M0(12)
         CALL KMTAB(IC,298.0,     XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GI0)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,XX,GII)
         M0(12) = M0(12)*EXP(LN10*(GI0-GII))
      ENDIF
C
      M0(13) = AWLC(IRH)      ! (NH4)3H(SO4)2
      IF (M0(13) .LT. 100.0) THEN
         IC     = 4.0*M0(13)
         CALL KMTAB(IC,298.0,     XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G130   = 0.2*(3.0*GI0+2.0*GII)
         CALL KMTAB(IC,SNGL(TEMP),XX,XX,XX,GI0,XX,XX,XX,XX,GII,XX,XX,XX)
         G13I   = 0.2*(3.0*GI0+2.0*GII)
         M0(13) = M0(13)*EXP(LN10*SNGL(G130-G13I))
      ENDIF
C
C *** OTHER INITIALIZATIONS *********************************************
C
      ICLACT  = 0
      CALAOU  = .TRUE.
      CALAIN  = .TRUE.
      FRST    = .TRUE.
      SCASE   = '??'
      SULRATW = 2.D0
      NOFER   = 0
      STKOFL  =.FALSE.
      DO 60 I=1,NERRMX
         ERRSTK(I) =-999
         ERRMSG(I) = 'MESSAGE N/A'
   60 CONTINUE
C
C *** END OF SUBROUTINE INIT3 *******************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** FUNCTION GETASR
C *** CALCULATES THE LIMITING NH4+/SO4 RATIO OF A SULFATE POOR SYSTEM
C     (i.e. SULFATE RATIO = 2.0) FOR GIVEN SO4 LEVEL AND RH
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      DOUBLE PRECISION FUNCTION GETASR (SO4I, RHI)
      PARAMETER (NSO4S=14, NRHS=20, NASRD=NSO4S*NRHS)
      COMMON /ASRC/ ASRAT(NASRD), ASSO4(NSO4S)
      DOUBLE PRECISION SO4I, RHI
CCC
CCC *** SOLVE USING FULL COMPUTATIONS, NOT LOOK-UP TABLES **************
CCC
CCC         W(2) = WAER(2)
CCC         W(3) = WAER(2)*2.0001D0
CCC         CALL CALCA2
CCC         SULRATW = MOLAL(3)/WAER(2)
CCC         CALL INIT1 (WI, RHI, TEMPI)   ! Re-initialize COMMON BLOCK
C
C *** CALCULATE INDICES ************************************************
C
      RAT    = SO4I/1.E-9    
      A1     = INT(ALOG10(RAT))                   ! Magnitude of RAT
      IA1    = INT(RAT/2.5/10.0**A1)
C
      INDS   = 4.0*A1 + MIN(IA1,4)
      INDS   = MIN(MAX(0, INDS), NSO4S-1) + 1     ! SO4 component of IPOS
C
      INDR   = INT(99.0-RHI*100.0) + 1
      INDR   = MIN(MAX(1, INDR), NRHS)            ! RH component of IPOS
C
C *** GET VALUE AND RETURN *********************************************
C
      INDSL  = INDS
      INDSH  = MIN(INDSL+1, NSO4S)
      IPOSL  = (INDSL-1)*NRHS + INDR              ! Low position in array
      IPOSH  = (INDSH-1)*NRHS + INDR              ! High position in array
C
      WF     = (SO4I-ASSO4(INDSL))/(ASSO4(INDSH)-ASSO4(INDSL) + 1e-7)
      WF     = MIN(MAX(WF, 0.0), 1.0)
C
      GETASR = WF*ASRAT(IPOSH) + (1.0-WF)*ASRAT(IPOSL)
C
C *** END OF FUNCTION GETASR *******************************************
C
      RETURN
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA AERSR
C *** CONTAINS DATA FOR AEROSOL SULFATE RATIO ARRAY NEEDED IN FUNCTION 
C     GETASR
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA AERSR
      PARAMETER (NSO4S=14, NRHS=20, NASRD=NSO4S*NRHS)
      COMMON /ASRC/ ASRAT(NASRD), ASSO4(NSO4S)
C
      DATA ASSO4/1.0E-9, 2.5E-9, 5.0E-9, 7.5E-9, 1.0E-8,
     &           2.5E-8, 5.0E-8, 7.5E-8, 1.0E-7, 2.5E-7, 
     &           5.0E-7, 7.5E-7, 1.0E-6, 5.0E-6/
C
      DATA (ASRAT(I), I=1,100)/
     & 1.020464, 0.9998130, 0.9960167, 0.9984423, 1.004004,
     & 1.010885,  1.018356,  1.026726,  1.034268, 1.043846,
     & 1.052933,  1.062230,  1.062213,  1.080050, 1.088350,
     & 1.096603,  1.104289,  1.111745,  1.094662, 1.121594,
     & 1.268909,  1.242444,  1.233815,  1.232088, 1.234020,
     & 1.238068,  1.243455,  1.250636,  1.258734, 1.267543,
     & 1.276948,  1.286642,  1.293337,  1.305592, 1.314726,
     & 1.323463,  1.333258,  1.343604,  1.344793, 1.355571,
     & 1.431463,  1.405204,  1.395791,  1.393190, 1.394403,
     & 1.398107,  1.403811,  1.411744,  1.420560, 1.429990,
     & 1.439742,  1.449507,  1.458986,  1.468403, 1.477394,
     & 1.487373,  1.495385,  1.503854,  1.512281, 1.520394,
     & 1.514464,  1.489699,  1.480686,  1.478187, 1.479446,
     & 1.483310,  1.489316,  1.497517,  1.506501, 1.515816,
     & 1.524724,  1.533950,  1.542758,  1.551730, 1.559587,
     & 1.568343,  1.575610,  1.583140,  1.590440, 1.596481,
     & 1.567743,  1.544426,  1.535928,  1.533645, 1.535016,
     & 1.539003,  1.545124,  1.553283,  1.561886, 1.570530,
     & 1.579234,  1.587813,  1.595956,  1.603901, 1.611349,
     & 1.618833,  1.625819,  1.632543,  1.639032, 1.645276/

      DATA (ASRAT(I), I=101,200)/
     & 1.707390,  1.689553,  1.683198,  1.681810, 1.683490,
     & 1.687477,  1.693148,  1.700084,  1.706917, 1.713507,
     & 1.719952,  1.726190,  1.731985,  1.737544, 1.742673,
     & 1.747756,  1.752431,  1.756890,  1.761141, 1.765190,
     & 1.785657,  1.771851,  1.767063,  1.766229, 1.767901,
     & 1.771455,  1.776223,  1.781769,  1.787065, 1.792081,
     & 1.796922,  1.801561,  1.805832,  1.809896, 1.813622,
     & 1.817292,  1.820651,  1.823841,  1.826871, 1.829745,
     & 1.822215,  1.810497,  1.806496,  1.805898, 1.807480,
     & 1.810684,  1.814860,  1.819613,  1.824093, 1.828306,
     & 1.832352,  1.836209,  1.839748,  1.843105, 1.846175,
     & 1.849192,  1.851948,  1.854574,  1.857038, 1.859387,
     & 1.844588,  1.834208,  1.830701,  1.830233, 1.831727,
     & 1.834665,  1.838429,  1.842658,  1.846615, 1.850321,
     & 1.853869,  1.857243,  1.860332,  1.863257, 1.865928,
     & 1.868550,  1.870942,  1.873208,  1.875355, 1.877389,
     & 1.899556,  1.892637,  1.890367,  1.890165, 1.891317,
     & 1.893436,  1.896036,  1.898872,  1.901485, 1.903908,
     & 1.906212,  1.908391,  1.910375,  1.912248, 1.913952,
     & 1.915621,  1.917140,  1.918576,  1.919934, 1.921220/

      DATA (ASRAT(I), I=201,280)/
     & 1.928264,  1.923245,  1.921625,  1.921523, 1.922421,
     & 1.924016,  1.925931,  1.927991,  1.929875, 1.931614,
     & 1.933262,  1.934816,  1.936229,  1.937560, 1.938769,
     & 1.939951,  1.941026,  1.942042,  1.943003, 1.943911,
     & 1.941205,  1.937060,  1.935734,  1.935666, 1.936430,
     & 1.937769,  1.939359,  1.941061,  1.942612, 1.944041,
     & 1.945393,  1.946666,  1.947823,  1.948911, 1.949900,
     & 1.950866,  1.951744,  1.952574,  1.953358, 1.954099,
     & 1.948985,  1.945372,  1.944221,  1.944171, 1.944850,
     & 1.946027,  1.947419,  1.948902,  1.950251, 1.951494,
     & 1.952668,  1.953773,  1.954776,  1.955719, 1.956576,
     & 1.957413,  1.958174,  1.958892,  1.959571, 1.960213,
     & 1.977193,  1.975540,  1.975023,  1.975015, 1.975346,
     & 1.975903,  1.976547,  1.977225,  1.977838, 1.978401,
     & 1.978930,  1.979428,  1.979879,  1.980302, 1.980686,
     & 1.981060,  1.981401,  1.981722,  1.982025, 1.982312/
C
C *** END OF BLOCK DATA AERSR ******************************************
C
       END
      
       
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCHA
C *** CALCULATES CHLORIDES SPECIATION
C
C     HYDROCHLORIC ACID IN THE LIQUID PHASE IS ASSUMED A MINOR SPECIES,  
C     AND DOES NOT SIGNIFICANTLY PERTURB THE HSO4-SO4 EQUILIBRIUM. THE 
C     HYDROCHLORIC ACID DISSOLVED IS CALCULATED FROM THE 
C     HCL(G) <-> (H+) + (CL-) 
C     EQUILIBRIUM, USING THE (H+) FROM THE SULFATES.
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCHA
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION KAPA
CC      CHARACTER ERRINF*40
C
C *** CALCULATE HCL DISSOLUTION *****************************************
C
      X    = W(5) 
      DELT = 0.0d0
      IF (WATER.GT.TINY) THEN
         KAPA = MOLAL(1)
         ALFA = XK3*R*TEMP*(WATER/GAMA(11))**2.0
         DIAK = SQRT( (KAPA+ALFA)**2.0 + 4.0*ALFA*X)
         DELT = 0.5*(-(KAPA+ALFA) + DIAK)
CC         IF (DELT/KAPA.GT.0.1d0) THEN
CC            WRITE (ERRINF,'(1PE10.3)') DELT/KAPA*100.0
CC            CALL PUSHERR (0033, ERRINF)    
CC         ENDIF
      ENDIF
C
C *** CALCULATE HCL SPECIATION IN THE GAS PHASE *************************
C
      GHCL     = MAX(X-DELT, 0.0d0)  ! GAS HCL
C
C *** CALCULATE HCL SPECIATION IN THE LIQUID PHASE **********************
C
      MOLAL(4) = DELT                ! CL-
      MOLAL(1) = MOLAL(1) + DELT     ! H+ 
C 
      RETURN
C
C *** END OF SUBROUTINE CALCHA ******************************************
C
      END





C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCHAP
C *** CALCULATES CHLORIDES SPECIATION
C
C     HYDROCHLORIC ACID IN THE LIQUID PHASE IS ASSUMED A MINOR SPECIES, 
C     THAT DOES NOT SIGNIFICANTLY PERTURB THE HSO4-SO4 EQUILIBRIUM. 
C     THE HYDROCHLORIC ACID DISSOLVED IS CALCULATED FROM THE 
C     HCL(G) -> HCL(AQ)   AND  HCL(AQ) ->  (H+) + (CL-) 
C     EQUILIBRIA, USING (H+) FROM THE SULFATES.
C
C     THIS IS THE VERSION USED BY THE INVERSE PROBLEM SOVER
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCHAP
      INCLUDE 'isrpia.inc'
C
C *** IS THERE A LIQUID PHASE? ******************************************
C
      IF (WATER.LE.TINY) RETURN
C
C *** CALCULATE HCL SPECIATION IN THE GAS PHASE *************************
C
      CALL CALCCLAQ (MOLAL(4), MOLAL(1), DELT)
      ALFA     = XK3*R*TEMP*(WATER/GAMA(11))**2.0
      GASAQ(3) = DELT
      MOLAL(1) = MOLAL(1) - DELT
      MOLAL(4) = MOLAL(4) - DELT
      GHCL     = MOLAL(1)*MOLAL(4)/ALFA
C 
      RETURN
C
C *** END OF SUBROUTINE CALCHAP *****************************************
C
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNA
C *** CALCULATES NITRATES SPECIATION
C
C     NITRIC ACID IN THE LIQUID PHASE IS ASSUMED A MINOR SPECIES, THAT 
C     DOES NOT SIGNIFICANTLY PERTURB THE HSO4-SO4 EQUILIBRIUM. THE NITRIC
C     ACID DISSOLVED IS CALCULATED FROM THE HNO3(G) -> (H+) + (NO3-) 
C     EQUILIBRIUM, USING THE (H+) FROM THE SULFATES.
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNA
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION KAPA
CC      CHARACTER ERRINF*40
C
C *** CALCULATE HNO3 DISSOLUTION ****************************************
C
      X    = W(4) 
      DELT = 0.0d0
      IF (WATER.GT.TINY) THEN
         KAPA = MOLAL(1)
         ALFA = XK4*R*TEMP*(WATER/GAMA(10))**2.0
         DIAK = SQRT( (KAPA+ALFA)**2.0 + 4.0*ALFA*X)
         DELT = 0.5*(-(KAPA+ALFA) + DIAK)
CC         IF (DELT/KAPA.GT.0.1d0) THEN
CC            WRITE (ERRINF,'(1PE10.3)') DELT/KAPA*100.0
CC            CALL PUSHERR (0019, ERRINF)    ! WARNING ERROR: NO SOLUTION
CC         ENDIF
      ENDIF
C
C *** CALCULATE HNO3 SPECIATION IN THE GAS PHASE ************************
C
      GHNO3    = MAX(X-DELT, 0.0d0)  ! GAS HNO3
C
C *** CALCULATE HNO3 SPECIATION IN THE LIQUID PHASE *********************
C
      MOLAL(7) = DELT                ! NO3-
      MOLAL(1) = MOLAL(1) + DELT     ! H+ 
C 
      RETURN
C
C *** END OF SUBROUTINE CALCNA ******************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNAP
C *** CALCULATES NITRATES SPECIATION
C
C     NITRIC ACID IN THE LIQUID PHASE IS ASSUMED A MINOR SPECIES, THAT 
C     DOES NOT SIGNIFICANTLY PERTURB THE HSO4-SO4 EQUILIBRIUM. THE NITRIC
C     ACID DISSOLVED IS CALCULATED FROM THE HNO3(G) -> HNO3(AQ) AND
C     HNO3(AQ) -> (H+) + (CL-) EQUILIBRIA, USING (H+) FROM THE SULFATES.
C
C     THIS IS THE VERSION USED BY THE INVERSE PROBLEM SOVER
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNAP
      INCLUDE 'isrpia.inc'
C
C *** IS THERE A LIQUID PHASE? ******************************************
C
      IF (WATER.LE.TINY) RETURN
C
C *** CALCULATE HNO3 SPECIATION IN THE GAS PHASE ************************
C
      CALL CALCNIAQ (MOLAL(7), MOLAL(1), DELT)
      ALFA     = XK4*R*TEMP*(WATER/GAMA(10))**2.0
      GASAQ(3) = DELT
      MOLAL(1) = MOLAL(1) - DELT
      MOLAL(7) = MOLAL(7) - DELT
      GHNO3    = MOLAL(1)*MOLAL(7)/ALFA
C 
      RETURN
C
C *** END OF SUBROUTINE CALCNAP *****************************************
C
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNH3
C *** CALCULATES AMMONIA IN GAS PHASE
C
C     AMMONIA IN THE GAS PHASE IS ASSUMED A MINOR SPECIES, THAT 
C     DOES NOT SIGNIFICANTLY PERTURB THE AEROSOL EQUILIBRIUM. 
C     AMMONIA GAS IS CALCULATED FROM THE NH3(g) + (H+)(l) <==> (NH4+)(l)
C     EQUILIBRIUM, USING (H+), (NH4+) FROM THE AEROSOL SOLUTION.
C
C     THIS IS THE VERSION USED BY THE DIRECT PROBLEM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNH3
      INCLUDE 'isrpia.inc'
C
C *** IS THERE A LIQUID PHASE? ******************************************
C
      IF (WATER.LE.TINY) RETURN
C
C *** CALCULATE NH3 SUBLIMATION *****************************************
C
      A1   = (XK2/XKW)*R*TEMP*(GAMA(10)/GAMA(5))**2.0
      CHI1 = MOLAL(3)
      CHI2 = MOLAL(1)
C
      BB   =(CHI2 + ONE/A1)          ! a=1; b!=1; c!=1 
      CC   =-CHI1/A1             
      DIAK = SQRT(BB*BB - 4.D0*CC)   ! Always > 0
      PSI  = 0.5*(-BB + DIAK)        ! One positive root
      PSI  = MAX(TINY, MIN(PSI,CHI1))! Constrict in acceptible range
C
C *** CALCULATE NH3 SPECIATION IN THE GAS PHASE *************************
C
      GNH3     = PSI                 ! GAS HNO3
C
C *** CALCULATE NH3 AFFECT IN THE LIQUID PHASE **************************
C
      MOLAL(3) = CHI1 - PSI          ! NH4+
      MOLAL(1) = CHI2 + PSI          ! H+ 
C 
      RETURN
C
C *** END OF SUBROUTINE CALCNH3 *****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNH3P
C *** CALCULATES AMMONIA IN GAS PHASE
C
C     AMMONIA GAS IS CALCULATED FROM THE NH3(g) + (H+)(l) <==> (NH4+)(l)
C     EQUILIBRIUM, USING (H+), (NH4+) FROM THE AEROSOL SOLUTION.
C
C     THIS IS THE VERSION USED BY THE INVERSE PROBLEM SOLVER
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNH3P
      INCLUDE 'isrpia.inc'
C
C *** IS THERE A LIQUID PHASE? ******************************************
C
      IF (WATER.LE.TINY) RETURN
C
C *** CALCULATE NH3 GAS PHASE CONCENTRATION *****************************
C
      A1   = (XK2/XKW)*R*TEMP*(GAMA(10)/GAMA(5))**2.0
      GNH3 = MOLAL(3)/MOLAL(1)/A1
C 
      RETURN
C
C *** END OF SUBROUTINE CALCNH3P ****************************************
C
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNHA
C
C     THIS SUBROUTINE CALCULATES THE DISSOLUTION OF HCL, HNO3 AT
C     THE PRESENCE OF (H,SO4). HCL, HNO3 ARE CONSIDERED MINOR SPECIES,
C     THAT DO NOT SIGNIFICANTLY AFFECT THE EQUILIBRIUM POINT.
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNHA
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION M1, M2, M3
      CHARACTER ERRINF*40     
C
C *** SPECIAL CASE; WATER=ZERO ******************************************
C
      IF (WATER.LE.TINY) THEN
         GOTO 55
C
C *** SPECIAL CASE; HCL=HNO3=ZERO ***************************************
C
      ELSEIF (W(5).LE.TINY .AND. W(4).LE.TINY) THEN
         GOTO 60
C
C *** SPECIAL CASE; HCL=ZERO ********************************************
C
      ELSE IF (W(5).LE.TINY) THEN
         CALL CALCNA              ! CALL HNO3 DISSOLUTION ROUTINE
         GOTO 60
C
C *** SPECIAL CASE; HNO3=ZERO *******************************************
C
      ELSE IF (W(4).LE.TINY) THEN
         CALL CALCHA              ! CALL HCL DISSOLUTION ROUTINE
         GOTO 60
      ENDIF
C
C *** CALCULATE EQUILIBRIUM CONSTANTS ***********************************
C
      A3 = XK4*R*TEMP*(WATER/GAMA(10))**2.0   ! HNO3
      A4 = XK3*R*TEMP*(WATER/GAMA(11))**2.0   ! HCL
C
C *** CALCULATE CUBIC EQUATION COEFFICIENTS *****************************
C
      DELCL = ZERO
      DELNO = ZERO
C
      OMEGA = MOLAL(1)       ! H+
      CHI3  = W(4)           ! HNO3
      CHI4  = W(5)           ! HCL
C
      C1    = A3*CHI3
      C2    = A4*CHI4
      C3    = A3 - A4
C
      M1    = (C1 + C2 + (OMEGA+A4)*C3)/C3
      M2    = ((OMEGA+A4)*C2 - A4*C3*CHI4)/C3
      M3    =-A4*C2*CHI4/C3
C
C *** CALCULATE ROOTS ***************************************************
C
      CALL POLY3 (M1, M2, M3, DELCL, ISLV) ! HCL DISSOLUTION
      IF (ISLV.NE.0) THEN
         DELCL = TINY       ! TINY AMOUNTS OF HCL ASSUMED WHEN NO ROOT 
         WRITE (ERRINF,'(1PE7.1)') TINY
         CALL PUSHERR (0022, ERRINF)    ! WARNING ERROR: NO SOLUTION
      ENDIF
      DELCL = MIN(DELCL, CHI4)
C
      DELNO = C1*DELCL/(C2 + C3*DELCL)  
      DELNO = MIN(DELNO, CHI3)
C
      IF (DELCL.LT.ZERO .OR. DELNO.LT.ZERO .OR.
     &   DELCL.GT.CHI4 .OR. DELNO.GT.CHI3       ) THEN
         DELCL = TINY  ! TINY AMOUNTS OF HCL ASSUMED WHEN NO ROOT 
         DELNO = TINY
         WRITE (ERRINF,'(1PE7.1)') TINY
         CALL PUSHERR (0022, ERRINF)    ! WARNING ERROR: NO SOLUTION
      ENDIF
CCC
CCC *** COMPARE DELTA TO TOTAL H+ ; ESTIMATE EFFECT TO HSO4 ***************
CCC
CC      IF ((DELCL+DELNO)/MOLAL(1).GT.0.1d0) THEN
CC         WRITE (ERRINF,'(1PE10.3)') (DELCL+DELNO)/MOLAL(1)*100.0
CC         CALL PUSHERR (0021, ERRINF)   
CC      ENDIF
C
C *** EFFECT ON LIQUID PHASE ********************************************
C
50    MOLAL(1) = MOLAL(1) + (DELNO+DELCL)  ! H+   CHANGE
      MOLAL(4) = MOLAL(4) + DELCL          ! CL-  CHANGE
      MOLAL(7) = MOLAL(7) + DELNO          ! NO3- CHANGE
C
C *** EFFECT ON GAS PHASE ***********************************************
C
55    GHCL     = MAX(W(5) - MOLAL(4), TINY)
      GHNO3    = MAX(W(4) - MOLAL(7), TINY)
C
60    RETURN
C
C *** END OF SUBROUTINE CALCNHA *****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNHP
C
C     THIS SUBROUTINE CALCULATES THE GAS PHASE NITRIC AND HYDROCHLORIC
C     ACID. CONCENTRATIONS ARE CALCULATED FROM THE DISSOLUTION 
C     EQUILIBRIA, USING (H+), (Cl-), (NO3-) IN THE AEROSOL PHASE.
C
C     THIS IS THE VERSION USED BY THE INVERSE PROBLEM SOLVER
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNHP
      INCLUDE 'isrpia.inc'
C
C *** IS THERE A LIQUID PHASE? ******************************************
C
      IF (WATER.LE.TINY) RETURN
C
C *** CALCULATE EQUILIBRIUM CONSTANTS ***********************************
C
      A3       = XK3*R*TEMP*(WATER/GAMA(11))**2.0
      A4       = XK4*R*TEMP*(WATER/GAMA(10))**2.0
      MOLAL(1) = MOLAL(1) + WAER(4) + WAER(5) 
C
C *** CALCULATE CONCENTRATIONS ******************************************
C *** ASSUME THAT 'DELT' FROM HNO3 >> 'DELT' FROM HCL
C
      CALL CALCNIAQ (WAER(4), MOLAL(1)+MOLAL(7)+MOLAL(4), DELT)
      MOLAL(1) = MOLAL(1) - DELT 
      MOLAL(7) = WAER(4)  - DELT
      GASAQ(3) = DELT
C
      CALL CALCCLAQ (WAER(5), MOLAL(1)+MOLAL(7)+MOLAL(4), DELT)
      MOLAL(1) = MOLAL(1) - DELT
      MOLAL(4) = WAER(5)  - DELT
      GASAQ(2) = DELT
C
      GHNO3    = MOLAL(1)*MOLAL(7)/A4
      GHCL     = MOLAL(1)*MOLAL(4)/A3
C
      RETURN
C
C *** END OF SUBROUTINE CALCNHP *****************************************
C
      END

C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCAMAQ
C *** THIS SUBROUTINE CALCULATES THE NH3(aq) GENERATED FROM (H,NH4+).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCAMAQ (NH4I, OHI, DELT)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION NH4I
CC      CHARACTER ERRINF*40
C
C *** EQUILIBRIUM CONSTANTS
C
      A22  = XK22/XKW/WATER*(GAMA(8)/GAMA(9))**2. ! GAMA(NH3) ASSUMED 1
      AKW  = XKW *RH*WATER*WATER
C
C *** FIND ROOT
C
      OM1  = NH4I          
      OM2  = OHI
      BB   =-(OM1+OM2+A22*AKW)
      CC   = OM1*OM2
      DD   = SQRT(BB*BB-4.D0*CC)

      DEL1 = 0.5D0*(-BB - DD)
      DEL2 = 0.5D0*(-BB + DD)
C
C *** GET APPROPRIATE ROOT.
C
      IF (DEL1.LT.ZERO) THEN                 
         IF (DEL2.GT.NH4I .OR. DEL2.GT.OHI) THEN
            DELT = ZERO
         ELSE
            DELT = DEL2
         ENDIF
      ELSE
         DELT = DEL1
      ENDIF
CC
CC *** COMPARE DELTA TO TOTAL NH4+ ; ESTIMATE EFFECT *********************
CC
CC      IF (DELTA/HYD.GT.0.1d0) THEN
CC         WRITE (ERRINF,'(1PE10.3)') DELTA/HYD*100.0
CC         CALL PUSHERR (0020, ERRINF)
CC      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE CALCAMAQ ****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCAMAQ2
C
C     THIS SUBROUTINE CALCULATES THE NH3(aq) GENERATED FROM (H,NH4+).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCAMAQ2 (GGNH3, NH4I, OHI, NH3AQ)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION NH4I, NH3AQ
C
C *** EQUILIBRIUM CONSTANTS
C
      A22  = XK22/XKW/WATER*(GAMA(8)/GAMA(9))**2. ! GAMA(NH3) ASSUMED 1
      AKW  = XKW *RH*WATER*WATER
C
C *** FIND ROOT
C
      ALF1 = NH4I - GGNH3
      ALF2 = GGNH3
      BB   = ALF1 + A22*AKW
      CC   =-A22*AKW*ALF2
      DEL  = 0.5D0*(-BB + SQRT(BB*BB-4.D0*CC))
C
C *** ADJUST CONCENTRATIONS
C
      NH4I  = ALF1 + DEL
      OHI   = DEL
      IF (OHI.LE.TINY) OHI = SQRT(AKW)   ! If solution is neutral.
      NH3AQ = ALF2 - DEL 
C
      RETURN
C
C *** END OF SUBROUTINE CALCAMAQ2 ****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCCLAQ
C
C     THIS SUBROUTINE CALCULATES THE HCL(aq) GENERATED FROM (H+,CL-).
C
C *** COPYRIGHT 1996-98, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCCLAQ (CLI, HI, DELT)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION CLI
C
C *** EQUILIBRIUM CONSTANTS
C
      A32  = XK32*WATER/(GAMA(11))**2. ! GAMA(HCL) ASSUMED 1
C
C *** FIND ROOT
C
      OM1  = CLI          
      OM2  = HI
      BB   =-(OM1+OM2+A32)
      CC   = OM1*OM2
      DD   = SQRT(BB*BB-4.D0*CC)

      DEL1 = 0.5D0*(-BB - DD)
      DEL2 = 0.5D0*(-BB + DD)
C
C *** GET APPROPRIATE ROOT.
C
      IF (DEL1.LT.ZERO) THEN                 
         IF (DEL2.LT.ZERO .OR. DEL2.GT.CLI .OR. DEL2.GT.HI) THEN
            DELT = ZERO
         ELSE
            DELT = DEL2
         ENDIF
      ELSE
         DELT = DEL1
      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE CALCCLAQ ****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCCLAQ2
C
C     THIS SUBROUTINE CALCULATES THE HCL(aq) GENERATED FROM (H+,CL-).
C
C *** COPYRIGHT 1996-98, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCCLAQ2 (GGCL, CLI, HI, CLAQ)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION CLI
C
C *** EQUILIBRIUM CONSTANTS
C
      A32  = XK32*WATER/(GAMA(11))**2. ! GAMA(HCL) ASSUMED 1
      AKW  = XKW *RH*WATER*WATER
C
C *** FIND ROOT
C
      ALF1  = CLI - GGCL
      ALF2  = GGCL
      COEF  = (ALF1+A32)
      DEL1  = 0.5*(-COEF + SQRT(COEF*COEF+4.D0*A32*ALF2))
C
C *** CORRECT CONCENTRATIONS
C
      CLI  = ALF1 + DEL1
      HI   = DEL1
      IF (HI.LE.TINY) HI = SQRT(AKW)   ! If solution is neutral.
      CLAQ = ALF2 - DEL1
C
      RETURN
C
C *** END OF SUBROUTINE CALCCLAQ2 ****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNIAQ
C
C     THIS SUBROUTINE CALCULATES THE HNO3(aq) GENERATED FROM (H,NO3-).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNIAQ (NO3I, HI, DELT)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION NO3I
C
C *** EQUILIBRIUM CONSTANTS
C
      A42  = XK42*WATER/(GAMA(10))**2. ! GAMA(HNO3) ASSUMED 1
C
C *** FIND ROOT
C
      OM1  = NO3I          
      OM2  = HI
      BB   =-(OM1+OM2+A42)
      CC   = OM1*OM2
      DD   = SQRT(BB*BB-4.D0*CC)

      DEL1 = 0.5D0*(-BB - DD)
      DEL2 = 0.5D0*(-BB + DD)
C
C *** GET APPROPRIATE ROOT.
C
      IF (DEL1.LT.ZERO .OR. DEL1.GT.HI .OR. DEL1.GT.NO3I) THEN
         DELT = ZERO
      ELSE
         DELT = DEL1
         RETURN
      ENDIF
C
      IF (DEL2.LT.ZERO .OR. DEL2.GT.NO3I .OR. DEL2.GT.HI) THEN
         DELT = ZERO
      ELSE
         DELT = DEL2
      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE CALCNIAQ ****************************************
C
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCNIAQ2
C
C     THIS SUBROUTINE CALCULATES THE UNDISSOCIATED HNO3(aq)
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCNIAQ2 (GGNO3, NO3I, HI, NO3AQ)
      INCLUDE 'isrpia.inc'
      DOUBLE PRECISION NO3I, NO3AQ
C
C *** EQUILIBRIUM CONSTANTS
C
      A42  = XK42*WATER/(GAMA(10))**2. ! GAMA(HNO3) ASSUMED 1
      AKW  = XKW *RH*WATER*WATER
C
C *** FIND ROOT
C
      ALF1  = NO3I - GGNO3
      ALF2  = GGNO3
      ALF3  = HI
C
      BB    = ALF3 + ALF1 + A42
      CC    = ALF3*ALF1 - A42*ALF2
      DEL1  = 0.5*(-BB + SQRT(BB*BB-4.D0*CC))
C
C *** CORRECT CONCENTRATIONS
C
      NO3I  = ALF1 + DEL1
      HI    = ALF3 + DEL1
      IF (HI.LE.TINY) HI = SQRT(AKW)   ! If solution is neutral.
      NO3AQ = ALF2 - DEL1
C
      RETURN
C
C *** END OF SUBROUTINE CALCNIAQ2 ****************************************
C
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCMR
C *** THIS SUBROUTINE CALCULATES:
C     1. ION PAIR CONCENTRATIONS (FROM [MOLAR] ARRAY)
C     2. WATER CONTENT OF LIQUID AEROSOL PHASE (FROM ZSR CORRELATION)
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCMR
      INCLUDE 'isrpia.inc'
      COMMON /SOLUT/ CHI1, CHI2, CHI3, CHI4, CHI5, CHI6, CHI7, CHI8,
     &               PSI1, PSI2, PSI3, PSI4, PSI5, PSI6, PSI7, PSI8,
     &               A1,   A2,   A3,   A4,   A5,   A6,   A7,   A8
c$omp threadprivate(/SOLUT/)
      CHARACTER SC*1
C
C *** CALCULATE ION PAIR CONCENTRATIONS ACCORDING TO SPECIFIC CASE ****
C 
      SC =SCASE(1:1)                   ! SULRAT & SODRAT case
C
C *** NH4-SO4 SYSTEM ; SULFATE POOR CASE
C
      IF (SC.EQ.'A') THEN      
         MOLALR(4) = MOLAL(5)+MOLAL(6) ! (NH4)2SO4 - CORRECT FOR SO4 TO HSO4
C
C *** NH4-SO4 SYSTEM ; SULFATE RICH CASE ; NO FREE ACID
C
      ELSE IF (SC.EQ.'B') THEN
         SO4I  = MOLAL(5)-MOLAL(1)     ! CORRECT FOR HSO4 DISSOCIATION 
         HSO4I = MOLAL(6)+MOLAL(1)              
         IF (SO4I.LT.HSO4I) THEN                
            MOLALR(13) = SO4I                   ! [LC] = [SO4]       
            MOLALR(9)  = MAX(HSO4I-SO4I, ZERO)  ! NH4HSO4
         ELSE                                   
            MOLALR(13) = HSO4I                  ! [LC] = [HSO4]
            MOLALR(4)  = MAX(SO4I-HSO4I, ZERO)  ! (NH4)2SO4
         ENDIF
C
C *** NH4-SO4 SYSTEM ; SULFATE RICH CASE ; FREE ACID 
C
      ELSE IF (SC.EQ.'C') THEN
         MOLALR(4) = MOLAL(3)                     ! NH4HSO4
         MOLALR(7) = MAX(W(2)-W(3), ZERO)         ! H2SO4
C
C *** NH4-SO4-NO3 SYSTEM ; SULFATE POOR CASE
C
      ELSE IF (SC.EQ.'D') THEN      
         MOLALR(4) = MOLAL(5) + MOLAL(6)          ! (NH4)2SO4
         AML5      = MOLAL(3)-2.D0*MOLALR(4)      ! "free" NH4
         MOLALR(5) = MAX(MIN(AML5,MOLAL(7)), ZERO)! NH4NO3 = MIN("free", NO3)
C
C *** NH4-SO4-NO3 SYSTEM ; SULFATE RICH CASE ; NO FREE ACID
C
      ELSE IF (SC.EQ.'E') THEN      
         SO4I  = MAX(MOLAL(5)-MOLAL(1),ZERO)      ! FROM HSO4 DISSOCIATION 
         HSO4I = MOLAL(6)+MOLAL(1)              
         IF (SO4I.LT.HSO4I) THEN                
            MOLALR(13) = SO4I                     ! [LC] = [SO4] 
            MOLALR(9)  = MAX(HSO4I-SO4I, ZERO)    ! NH4HSO4
         ELSE                                   
            MOLALR(13) = HSO4I                    ! [LC] = [HSO4]
            MOLALR(4)  = MAX(SO4I-HSO4I, ZERO)    ! (NH4)2SO4
         ENDIF
C
C *** NH4-SO4-NO3 SYSTEM ; SULFATE RICH CASE ; FREE ACID
C
      ELSE IF (SC.EQ.'F') THEN      
         MOLALR(4) = MOLAL(3)                              ! NH4HSO4
         MOLALR(7) = MAX(MOLAL(5)+MOLAL(6)-MOLAL(3),ZERO)  ! H2SO4
C
C *** NA-NH4-SO4-NO3-CL SYSTEM ; SULFATE POOR ; SODIUM POOR CASE
C
      ELSE IF (SC.EQ.'G') THEN      
         MOLALR(2) = 0.5*MOLAL(2)                          ! NA2SO4
         TOTS4     = MOLAL(5)+MOLAL(6)                     ! Total SO4
         MOLALR(4) = MAX(TOTS4 - MOLALR(2), ZERO)          ! (NH4)2SO4
         FRNH4     = MAX(MOLAL(3) - 2.D0*MOLALR(4), ZERO)
         MOLALR(5) = MIN(MOLAL(7),FRNH4)                   ! NH4NO3
         FRNH4     = MAX(FRNH4 - MOLALR(5), ZERO)
         MOLALR(6) = MIN(MOLAL(4), FRNH4)                  ! NH4CL
C
C *** NA-NH4-SO4-NO3-CL SYSTEM ; SULFATE POOR ; SODIUM RICH CASE
C *** RETREIVE DISSOLVED SALTS DIRECTLY FROM COMMON BLOCK /SOLUT/
C
      ELSE IF (SC.EQ.'H') THEN      
         MOLALR(1) = PSI7                                  ! NACL 
         MOLALR(2) = PSI1                                  ! NA2SO4
         MOLALR(3) = PSI8                                  ! NANO3
         MOLALR(4) = ZERO                                  ! (NH4)2SO4
         FRNO3     = MAX(MOLAL(7) - MOLALR(3), ZERO)       ! "FREE" NO3
         FRCL      = MAX(MOLAL(4) - MOLALR(1), ZERO)       ! "FREE" CL
         MOLALR(5) = MIN(MOLAL(3),FRNO3)                   ! NH4NO3
         FRNH4     = MAX(MOLAL(3) - MOLALR(5), ZERO)       ! "FREE" NH3
         MOLALR(6) = MIN(FRCL, FRNH4)                      ! NH4CL
C
C *** NA-NH4-SO4-NO3-CL SYSTEM ; SULFATE RICH CASE ; NO FREE ACID
C *** RETREIVE DISSOLVED SALTS DIRECTLY FROM COMMON BLOCK /SOLUT/
C
      ELSE IF (SC.EQ.'I') THEN      
         MOLALR(04) = PSI5                                 ! (NH4)2SO4
         MOLALR(02) = PSI4                                 ! NA2SO4
         MOLALR(09) = PSI1                                 ! NH4HSO4
         MOLALR(12) = PSI3                                 ! NAHSO4
         MOLALR(13) = PSI2                                 ! LC
C
C *** NA-NH4-SO4-NO3-CL SYSTEM ; SULFATE RICH CASE ; FREE ACID
C
      ELSE IF (SC.EQ.'J') THEN      
         MOLALR(09) = MOLAL(3)                             ! NH4HSO4
         MOLALR(12) = MOLAL(2)                             ! NAHSO4
         MOLALR(07) = MOLAL(5)+MOLAL(6)-MOLAL(3)-MOLAL(2)  ! H2SO4
         MOLALR(07) = MAX(MOLALR(07),ZERO)
C
C ======= REVERSE PROBLEMS ===========================================
C
C *** NH4-SO4-NO3 SYSTEM ; SULFATE POOR CASE
C
      ELSE IF (SC.EQ.'N') THEN      
         MOLALR(4) = MOLAL(5) + MOLAL(6)          ! (NH4)2SO4
         AML5      = WAER(3)-2.D0*MOLALR(4)       ! "free" NH4
         MOLALR(5) = MAX(MIN(AML5,WAER(4)), ZERO) ! NH4NO3 = MIN("free", NO3)
C
C *** NH4-SO4-NO3-NA-CL SYSTEM ; SULFATE POOR, SODIUM POOR CASE
C
      ELSE IF (SC.EQ.'Q') THEN      
         MOLALR(2) = PSI1                                  ! NA2SO4
         MOLALR(4) = PSI6                                  ! (NH4)2SO4
         MOLALR(5) = PSI5                                  ! NH4NO3
         MOLALR(6) = PSI4                                  ! NH4CL
C
C *** NH4-SO4-NO3-NA-CL SYSTEM ; SULFATE POOR, SODIUM RICH CASE
C
      ELSE IF (SC.EQ.'R') THEN      
         MOLALR(1) = PSI3                                  ! NACL 
         MOLALR(2) = PSI1                                  ! NA2SO4
         MOLALR(3) = PSI2                                  ! NANO3
         MOLALR(4) = ZERO                                  ! (NH4)2SO4
         MOLALR(5) = PSI5                                  ! NH4NO3
         MOLALR(6) = PSI4                                  ! NH4CL
C
C *** UNKNOWN CASE
C
      ELSE
         CALL PUSHERR (1001, ' ') ! FATAL ERROR: CASE NOT SUPPORTED 
      ENDIF
C
C *** CALCULATE WATER CONTENT ; ZSR CORRELATION ***********************
C
      WATER = ZERO
      DO 10 I=1,NPAIR
         WATER = WATER + MOLALR(I)/M0(I)
10    CONTINUE
      WATER = MAX(WATER, TINY)
C
      RETURN
C
C *** END OF SUBROUTINE CALCMR ******************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCMDRH
C
C     THIS IS THE CASE WHERE THE RELATIVE HUMIDITY IS IN THE MUTUAL
C     DRH REGION. THE SOLUTION IS ASSUMED TO BE THE SUM OF TWO WEIGHTED
C     SOLUTIONS ; THE 'DRY' SOLUTION (SUBROUTINE DRYCASE) AND THE
C     'SATURATED LIQUID' SOLUTION (SUBROUTINE LIQCASE).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCMDRH (RHI, RHDRY, RHLIQ, DRYCASE, LIQCASE)
      INCLUDE 'isrpia.inc'
      EXTERNAL DRYCASE, LIQCASE
C
C *** FIND WEIGHT FACTOR **********************************************
C
      IF (WFTYP.EQ.0) THEN
         WF = ONE
      ELSEIF (WFTYP.EQ.1) THEN
         WF = 0.5D0
      ELSE
         WF = (RHLIQ-RHI)/(RHLIQ-RHDRY)
      ENDIF
      ONEMWF  = ONE - WF
C
C *** FIND FIRST SECTION ; DRY ONE ************************************
C
      CALL DRYCASE
      IF (ABS(ONEMWF).LE.1D-5) GOTO 200  ! DRY AEROSOL
C
      CNH42SO = CNH42S4                  ! FIRST (DRY) SOLUTION
      CNH4HSO = CNH4HS4
      CLCO    = CLC 
      CNH4N3O = CNH4NO3
      CNH4CLO = CNH4CL
      CNA2SO  = CNA2SO4
      CNAHSO  = CNAHSO4
      CNANO   = CNANO3
      CNACLO  = CNACL
      GNH3O   = GNH3
      GHNO3O  = GHNO3
      GHCLO   = GHCL
C
C *** FIND SECOND SECTION ; DRY & LIQUID ******************************
C
      CNH42S4 = ZERO
      CNH4HS4 = ZERO
      CLC     = ZERO
      CNH4NO3 = ZERO
      CNH4CL  = ZERO
      CNA2SO4 = ZERO
      CNAHSO4 = ZERO
      CNANO3  = ZERO
      CNACL   = ZERO
      GNH3    = ZERO
      GHNO3   = ZERO
      GHCL    = ZERO
      CALL LIQCASE                   ! SECOND (LIQUID) SOLUTION
C
C *** ADJUST THINGS FOR THE CASE THAT THE LIQUID SUB PREDICTS DRY AEROSOL
C
      IF (WATER.LE.TINY) THEN
         DO 100 I=1,NIONS
            MOLAL(I)= ZERO           ! Aqueous phase
  100    CONTINUE
         WATER   = ZERO
C
         CNH42S4 = CNH42SO           ! Solid phase
         CNA2SO4 = CNA2SO
         CNAHSO4 = CNAHSO
         CNH4HS4 = CNH4HSO
         CLC     = CLCO
         CNH4NO3 = CNH4N3O
         CNANO3  = CNANO
         CNACL   = CNACLO                                                  
         CNH4CL  = CNH4CLO 
C
         GNH3    = GNH3O             ! Gas phase
         GHNO3   = GHNO3O
         GHCL    = GHCLO
C
         GOTO 200
      ENDIF
C
C *** FIND SALT DISSOLUTIONS BETWEEN DRY & LIQUID SOLUTIONS.
C
      DAMSUL  = CNH42SO - CNH42S4
      DSOSUL  = CNA2SO  - CNA2SO4
      DAMBIS  = CNH4HSO - CNH4HS4
      DSOBIS  = CNAHSO  - CNAHSO4
      DLC     = CLCO    - CLC
      DAMNIT  = CNH4N3O - CNH4NO3
      DAMCHL  = CNH4CLO - CNH4CL
      DSONIT  = CNANO   - CNANO3
      DSOCHL  = CNACLO  - CNACL
C
C *** FIND GAS DISSOLUTIONS BETWEEN DRY & LIQUID SOLUTIONS.
C
      DAMG    = GNH3O   - GNH3 
      DHAG    = GHCLO   - GHCL
      DNAG    = GHNO3O  - GHNO3
C
C *** FIND SOLUTION AT MDRH BY WEIGHTING DRY & LIQUID SOLUTIONS.
C
C     LIQUID
C
      MOLAL(1)= ONEMWF*MOLAL(1)                                 ! H+
      MOLAL(2)= ONEMWF*(2.D0*DSOSUL + DSOBIS + DSONIT + DSOCHL) ! NA+
      MOLAL(3)= ONEMWF*(2.D0*DAMSUL + DAMG   + DAMBIS + DAMCHL +
     &                  3.D0*DLC    + DAMNIT )                  ! NH4+
      MOLAL(4)= ONEMWF*(     DAMCHL + DSOCHL + DHAG)            ! CL-
      MOLAL(5)= ONEMWF*(     DAMSUL + DSOSUL + DLC - MOLAL(6))  ! SO4-- ! VB 17 Sept 2001
      MOLAL(6)= ONEMWF*(   MOLAL(6) + DSOBIS + DAMBIS + DLC)    ! HSO4-
      MOLAL(7)= ONEMWF*(     DAMNIT + DSONIT + DNAG)            ! NO3-
      WATER   = ONEMWF*WATER
C
C     SOLID
C
      CNH42S4 = WF*CNH42SO + ONEMWF*CNH42S4
      CNA2SO4 = WF*CNA2SO  + ONEMWF*CNA2SO4
      CNAHSO4 = WF*CNAHSO  + ONEMWF*CNAHSO4
      CNH4HS4 = WF*CNH4HSO + ONEMWF*CNH4HS4
      CLC     = WF*CLCO    + ONEMWF*CLC
      CNH4NO3 = WF*CNH4N3O + ONEMWF*CNH4NO3
      CNANO3  = WF*CNANO   + ONEMWF*CNANO3
      CNACL   = WF*CNACLO  + ONEMWF*CNACL
      CNH4CL  = WF*CNH4CLO + ONEMWF*CNH4CL
C
C     GAS
C
      GNH3    = WF*GNH3O   + ONEMWF*GNH3
      GHNO3   = WF*GHNO3O  + ONEMWF*GHNO3
      GHCL    = WF*GHCLO   + ONEMWF*GHCL
C
C *** RETURN POINT
C
200   RETURN
C
C *** END OF SUBROUTINE CALCMDRH ****************************************
C
      END






C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCMDRP
C
C     THIS IS THE CASE WHERE THE RELATIVE HUMIDITY IS IN THE MUTUAL
C     DRH REGION. THE SOLUTION IS ASSUMED TO BE THE SUM OF TWO WEIGHTED
C     SOLUTIONS ; THE 'DRY' SOLUTION (SUBROUTINE DRYCASE) AND THE
C     'SATURATED LIQUID' SOLUTION (SUBROUTINE LIQCASE).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCMDRP (RHI, RHDRY, RHLIQ, DRYCASE, LIQCASE)
      INCLUDE 'isrpia.inc'
      EXTERNAL DRYCASE, LIQCASE
C
C *** FIND WEIGHT FACTOR **********************************************
C
      IF (WFTYP.EQ.0) THEN
         WF = ONE
      ELSEIF (WFTYP.EQ.1) THEN
         WF = 0.5D0
      ELSE
         WF = (RHLIQ-RHI)/(RHLIQ-RHDRY)
      ENDIF
      ONEMWF  = ONE - WF
C
C *** FIND FIRST SECTION ; DRY ONE ************************************
C
      CALL DRYCASE
      IF (ABS(ONEMWF).LE.1D-5) GOTO 200  ! DRY AEROSOL
C
      CNH42SO = CNH42S4              ! FIRST (DRY) SOLUTION
      CNH4HSO = CNH4HS4
      CLCO    = CLC 
      CNH4N3O = CNH4NO3
      CNH4CLO = CNH4CL
      CNA2SO  = CNA2SO4
      CNAHSO  = CNAHSO4
      CNANO   = CNANO3
      CNACLO  = CNACL
C
C *** FIND SECOND SECTION ; DRY & LIQUID ******************************
C
      CNH42S4 = ZERO
      CNH4HS4 = ZERO
      CLC     = ZERO
      CNH4NO3 = ZERO
      CNH4CL  = ZERO
      CNA2SO4 = ZERO
      CNAHSO4 = ZERO
      CNANO3  = ZERO
      CNACL   = ZERO
      GNH3    = ZERO
      GHNO3   = ZERO
      GHCL    = ZERO
      CALL LIQCASE                   ! SECOND (LIQUID) SOLUTION
C
C *** ADJUST THINGS FOR THE CASE THAT THE LIQUID SUB PREDICTS DRY AEROSOL
C
      IF (WATER.LE.TINY) THEN
         WATER = ZERO
         DO 100 I=1,NIONS
            MOLAL(I)= ZERO
 100     CONTINUE
         CALL DRYCASE
         GOTO 200
      ENDIF
C
C *** FIND SALT DISSOLUTIONS BETWEEN DRY & LIQUID SOLUTIONS.
C
      DAMBIS  = CNH4HSO - CNH4HS4
      DSOBIS  = CNAHSO  - CNAHSO4
      DLC     = CLCO    - CLC
C
C *** FIND SOLUTION AT MDRH BY WEIGHTING DRY & LIQUID SOLUTIONS.
C
C *** SOLID
C
      CNH42S4 = WF*CNH42SO + ONEMWF*CNH42S4
      CNA2SO4 = WF*CNA2SO  + ONEMWF*CNA2SO4
      CNAHSO4 = WF*CNAHSO  + ONEMWF*CNAHSO4
      CNH4HS4 = WF*CNH4HSO + ONEMWF*CNH4HS4
      CLC     = WF*CLCO    + ONEMWF*CLC
      CNH4NO3 = WF*CNH4N3O + ONEMWF*CNH4NO3
      CNANO3  = WF*CNANO   + ONEMWF*CNANO3
      CNACL   = WF*CNACLO  + ONEMWF*CNACL
      CNH4CL  = WF*CNH4CLO + ONEMWF*CNH4CL
C
C *** LIQUID
C
      WATER   = ONEMWF*WATER
C
      MOLAL(2)= WAER(1) - 2.D0*CNA2SO4 - CNAHSO4 - CNANO3 -     
     &                         CNACL                            ! NA+
      MOLAL(3)= WAER(3) - 2.D0*CNH42S4 - CNH4HS4 - CNH4CL - 
     &                    3.D0*CLC     - CNH4NO3                ! NH4+
      MOLAL(4)= WAER(5) - CNACL - CNH4CL                        ! CL-
      MOLAL(7)= WAER(4) - CNANO3 - CNH4NO3                      ! NO3-
      MOLAL(6)= ONEMWF*(MOLAL(6) + DSOBIS + DAMBIS + DLC)       ! HSO4-
      MOLAL(5)= WAER(2) - MOLAL(6) - CLC - CNH42S4 - CNA2SO4    ! SO4--
C
      A8      = XK1*WATER/GAMA(7)*(GAMA(8)/GAMA(7))**2.
      IF (MOLAL(5).LE.TINY) THEN
         HIEQ = SQRT(XKW *RH*WATER*WATER)  ! Neutral solution
      ELSE
         HIEQ = A8*MOLAL(6)/MOLAL(5)          
      ENDIF
      HIEN    = MOLAL(4) + MOLAL(7) + MOLAL(6) + 2.D0*MOLAL(5) -
     &          MOLAL(2) - MOLAL(3)
      MOLAL(1)= MAX (HIEQ, HIEN)                                ! H+
C
C *** GAS (ACTIVITY COEFS FROM LIQUID SOLUTION)
C
      A2      = (XK2/XKW)*R*TEMP*(GAMA(10)/GAMA(5))**2. ! NH3  <==> NH4+
      A3      = XK4 *R*TEMP*(WATER/GAMA(10))**2.        ! HNO3 <==> NO3-
      A4      = XK3 *R*TEMP*(WATER/GAMA(11))**2.        ! HCL  <==> CL-
C
      GNH3    = MOLAL(3)/MAX(MOLAL(1),TINY)/A2
      GHNO3   = MOLAL(1)*MOLAL(7)/A3
      GHCL    = MOLAL(1)*MOLAL(4)/A4
C
200   RETURN
C
C *** END OF SUBROUTINE CALCMDRP ****************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCHS4
C *** THIS SUBROUTINE CALCULATES THE HSO4 GENERATED FROM (H,SO4).
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCHS4 (HI, SO4I, HSO4I, DELTA)
      INCLUDE 'isrpia.inc'
CC      CHARACTER ERRINF*40
C
C *** IF TOO LITTLE WATER, DONT SOLVE
C
      IF (WATER.LE.1d1*TINY) THEN
         DELTA = ZERO 
         RETURN
      ENDIF
C
C *** CALCULATE HSO4 SPECIATION *****************************************
C
      A8 = XK1*WATER/GAMA(7)*(GAMA(8)/GAMA(7))**2.
C
      BB =-(HI + SO4I + A8)
      CC = HI*SO4I - HSO4I*A8
      DD = BB*BB - 4.D0*CC
C
      IF (DD.GE.ZERO) THEN
         SQDD   = SQRT(DD)
         DELTA1 = 0.5*(-BB + SQDD)
         DELTA2 = 0.5*(-BB - SQDD)
         IF (HSO4I.LE.TINY) THEN
            DELTA = DELTA2
         ELSEIF( HI*SO4I .GE. A8*HSO4I ) THEN
            DELTA = DELTA2
         ELSEIF( HI*SO4I .LT. A8*HSO4I ) THEN
            DELTA = DELTA1
         ELSE
            DELTA = ZERO
         ENDIF
      ELSE
         DELTA  = ZERO
      ENDIF
CCC
CCC *** COMPARE DELTA TO TOTAL H+ ; ESTIMATE EFFECT OF HSO4 ***************
CCC
CC      HYD = MAX(HI, MOLAL(1))
CC      IF (HYD.GT.TINY) THEN
CC         IF (DELTA/HYD.GT.0.1d0) THEN
CC            WRITE (ERRINF,'(1PE10.3)') DELTA/HYD*100.0
CC            CALL PUSHERR (0020, ERRINF)
CC         ENDIF
CC      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE CALCHS4 *****************************************
C
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCPH
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCPH (GG, HI, OHI)
      INCLUDE 'isrpia.inc'
C
      AKW  = XKW *RH*WATER*WATER
      CN   = SQRT(AKW)
C
C *** GG = (negative charge) - (positive charge)
C
      IF (GG.GT.TINY) THEN                        ! H+ in excess
         BB =-GG
         CC =-AKW
         DD = BB*BB - 4.D0*CC
         HI = MAX(0.5D0*(-BB + SQRT(DD)),CN)
         OHI= AKW/HI
      ELSE                                        ! OH- in excess
         BB = GG
         CC =-AKW
         DD = BB*BB - 4.D0*CC
         OHI= MAX(0.5D0*(-BB + SQRT(DD)),CN)
         HI = AKW/OHI
      ENDIF
C
      RETURN
C
C *** END OF SUBROUTINE CALCPH ******************************************
C
      END

C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE CALCACT
C *** CALCULATES MULTI-COMPONENET ACTIVITY COEFFICIENTS FROM BROMLEYS
C     METHOD. THE BINARY ACTIVITY COEFFICIENTS ARE CALCULATED BY 
C     KUSIK-MEISNER RELATION (SUBROUTINE KMTAB or SUBROUTINE KMFUL). 
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE CALCACT
      INCLUDE 'isrpia.inc'
C
      REAL EX10, URF
      REAL G0(3,4),ZPL,ZMI,AGAMA,SION,H,CH,F1(3),F2(4)
      DOUBLE PRECISION MPL, XIJ, YJI
      PARAMETER (URF=0.5)
C
      G(I,J)= (F1(I)/Z(I) + F2(J)/Z(J+3)) / (Z(I)+Z(J+3)) - H
C
C *** SAVE ACTIVITIES IN OLD ARRAY *************************************
C
      IF (FRST) THEN               ! Outer loop
         DO 10 I=1,NPAIR
            GAMOU(I) = GAMA(I)
10       CONTINUE
      ENDIF
C
      DO 20 I=1,NPAIR              ! Inner loop
         GAMIN(I) = GAMA(I)
20    CONTINUE
C
C *** CALCULATE IONIC ACTIVITY OF SOLUTION *****************************
C
      IONIC=0.0
      DO 30 I=1,NIONS
         IONIC=IONIC + MOLAL(I)*Z(I)*Z(I)
30    CONTINUE
      IONIC = MAX(MIN(0.5*IONIC/WATER,20.d0), TINY)
C
C *** CALCULATE BINARY ACTIVITY COEFFICIENTS ***************************
C
C  G0(1,1)=G11;G0(1,2)=G07;G0(1,3)=G08;G0(1,4)=G10;G0(2,1)=G01;G0(2,2)=G02
C  G0(2,3)=G12;G0(2,4)=G03;G0(3,1)=G06;G0(3,2)=G04;G0(3,3)=G09;G0(3,4)=G05
C
      IF (IACALC.EQ.0) THEN              ! K.M.; FULL
         CALL KMFUL (IONIC, SNGL(TEMP),G0(2,1),G0(2,2),G0(2,4),
     &               G0(3,2),G0(3,4),G0(3,1),G0(1,2),G0(1,3),G0(3,3),
     &               G0(1,4),G0(1,1),G0(2,3))
      ELSE                               ! K.M.; TABULATED
         CALL KMTAB (IONIC, SNGL(TEMP),G0(2,1),G0(2,2),G0(2,4),
     &               G0(3,2),G0(3,4),G0(3,1),G0(1,2),G0(1,3),G0(3,3),
     &               G0(1,4),G0(1,1),G0(2,3))
      ENDIF
C
C *** CALCULATE MULTICOMPONENT ACTIVITY COEFFICIENTS *******************
C
      AGAMA = 0.511*(298.0/TEMP)**1.5    ! Debye Huckel const. at T
      SION  = SQRT(IONIC)
      H     = AGAMA*SION/(1+SION)
C
      DO 100 I=1,3
         F1(I)=0.0
         F2(I)=0.0
100   CONTINUE
      F2(4)=0.0
C
      DO 110 I=1,3
         ZPL = Z(I)
         MPL = MOLAL(I)/WATER
         DO 110 J=1,4
            ZMI   = Z(J+3)
            CH    = 0.25*(ZPL+ZMI)*(ZPL+ZMI)/IONIC
            XIJ   = CH*MPL
            YJI   = CH*MOLAL(J+3)/WATER
            F1(I) = F1(I) + SNGL(YJI*(G0(I,J) + ZPL*ZMI*H))
            F2(J) = F2(J) + SNGL(XIJ*(G0(I,J) + ZPL*ZMI*H))
110   CONTINUE
C
C *** LOG10 OF ACTIVITY COEFFICIENTS ***********************************
C
      GAMA(01) = G(2,1)*ZZ(01)                     ! NACL
      GAMA(02) = G(2,2)*ZZ(02)                     ! NA2SO4
      GAMA(03) = G(2,4)*ZZ(03)                     ! NANO3
      GAMA(04) = G(3,2)*ZZ(04)                     ! (NH4)2SO4
      GAMA(05) = G(3,4)*ZZ(05)                     ! NH4NO3
      GAMA(06) = G(3,1)*ZZ(06)                     ! NH4CL
      GAMA(07) = G(1,2)*ZZ(07)                     ! 2H-SO4
      GAMA(08) = G(1,3)*ZZ(08)                     ! H-HSO4
      GAMA(09) = G(3,3)*ZZ(09)                     ! NH4HSO4
      GAMA(10) = G(1,4)*ZZ(10)                     ! HNO3
      GAMA(11) = G(1,1)*ZZ(11)                     ! HCL
      GAMA(12) = G(2,3)*ZZ(12)                     ! NAHSO4
      GAMA(13) = 0.20*(3.0*GAMA(04)+2.0*GAMA(09))  ! LC ; SCAPE
CC      GAMA(13) = 0.50*(GAMA(04)+GAMA(09))          ! LC ; SEQUILIB
CC      GAMA(13) = 0.25*(3.0*GAMA(04)+GAMA(07))      ! LC ; AIM
C
C *** CONVERT LOG (GAMA) COEFFICIENTS TO GAMA **************************
C
      DO 200 I=1,NPAIR
CC         GAMA(I)=MAX(-5.0d0, MIN(GAMA(I),5.0d0) ) ! F77 LIBRARY ROUTINE
CC         GAMA(I)=10.0**GAMA(I)
         GAMA(I)=EX10(SNGL(GAMA(I)), 5.0)    ! CUTOFF SET TO [-5,5]
         GAMA(I) = GAMIN(I)*(1.0-URF) + URF*GAMA(I)  ! Under-relax GAMA's
  200 CONTINUE
C
C *** SETUP ACTIVITY CALCULATION FLAGS *********************************
C
C OUTER CALCULATION LOOP ; ONLY IF FRST=.TRUE.
C
      IF (FRST) THEN          
         ERROU = ZERO                    ! CONVERGENCE CRITERION
         DO 210 I=1,NPAIR
            ERROU=MAX(ERROU, ABS((GAMOU(I)-GAMA(I))/GAMOU(I)))
210      CONTINUE
         CALAOU = ERROU .GE. EPSACT      ! SETUP FLAGS
         FRST   =.FALSE.
      ENDIF
C
C INNER CALCULATION LOOP ; ALWAYS
C
      ERRIN = ZERO                       ! CONVERGENCE CRITERION
      DO 220 I=1,NPAIR
         ERRIN = MAX (ERRIN, ABS((GAMIN(I)-GAMA(I))/GAMIN(I)))
220   CONTINUE
      CALAIN = ERRIN .GE. EPSACT
C
      ICLACT = ICLACT + 1                ! Increment ACTIVITY call counter
C
C *** END OF SUBROUTINE ACTIVITY ****************************************
C
      RETURN
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE RSTGAM
C *** RESETS ACTIVITY COEFFICIENT ARRAYS TO DEFAULT VALUE OF 0.1
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE RSTGAM
      INCLUDE 'isrpia.inc'
C
      DO 10 I=1, NPAIR
         GAMA(I) = 0.1
10    CONTINUE
C
C *** END OF SUBROUTINE RSTGAM ******************************************
C
      RETURN
      END      
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KMFUL
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KMFUL (IONIC,TEMP,G01,G02,G03,G04,G05,G06,G07,G08,G09,
     &                  G10,G11,G12)
      REAL Ionic, TEMP
      DATA Z01,Z02,Z03,Z04,Z05,Z06,Z07,Z08,Z10,Z11
     &    /1,  2,  1,  2,  1,  1,  2,  1,  1,  1/
C
      SION = SQRT(IONIC)
C
C *** Coefficients at 25 oC
C
      CALL MKBI(2.230, IONIC, SION, Z01, G01)
      CALL MKBI(-0.19, IONIC, SION, Z02, G02)
      CALL MKBI(-0.39, IONIC, SION, Z03, G03)
      CALL MKBI(-0.25, IONIC, SION, Z04, G04)
      CALL MKBI(-1.15, IONIC, SION, Z05, G05)
      CALL MKBI(0.820, IONIC, SION, Z06, G06)
      CALL MKBI(-.100, IONIC, SION, Z07, G07)
      CALL MKBI(8.000, IONIC, SION, Z08, G08)
      CALL MKBI(2.600, IONIC, SION, Z10, G10)
      CALL MKBI(6.000, IONIC, SION, Z11, G11)
C
C *** Correct for T other than 298 K
C
      TI  = TEMP-273.0
      IF (ABS(TI) .GT. 1.0) THEN
         CF1 = 1.125-0.005*TI
         CF2 = (0.125-0.005*TI)*(0.039*IONIC**0.92-0.41*SION/(1.+SION))
         G01 = CF1*G01 - CF2*Z01
         G02 = CF1*G02 - CF2*Z02
         G03 = CF1*G03 - CF2*Z03
         G04 = CF1*G04 - CF2*Z04
         G05 = CF1*G05 - CF2*Z05
         G06 = CF1*G06 - CF2*Z06
         G07 = CF1*G07 - CF2*Z07
         G08 = CF1*G08 - CF2*Z08
         G10 = CF1*G10 - CF2*Z10
         G11 = CF1*G11 - CF2*Z11
      ENDIF
C
      G09 = G06 + G08 - G11
      G12 = G01 + G08 - G11
C
C *** Return point ; End of subroutine
C
      RETURN
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE MKBI
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE MKBI(Q,IONIC,SION,ZIP,BI)
C
      REAL IONIC
C
      B=.75-.065*Q
      C= 1.0
      IF (IONIC.LT.6.0) C=1.+.055*Q*EXP(-.023*IONIC*IONIC*IONIC)
      XX=-0.5107*SION/(1.+C*SION)
      BI=(1.+B*(1.+.1*IONIC)**Q-B)
      BI=ZIP*ALOG10(BI) + ZIP*XX
C
      RETURN
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KMTAB
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IONIC' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KMTAB (IN,TEMP,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                          G11,G12)
      REAL IN, Temp
C
C *** Find temperature range
C
      IND = NINT((TEMP-198.0)/25.0) + 1
      IND = MIN(MAX(IND,1),6)
C
C *** Call appropriate routine
C
      IF (IND.EQ.1) THEN
         CALL KM198 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ELSEIF (IND.EQ.2) THEN
         CALL KM223 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ELSEIF (IND.EQ.3) THEN
         CALL KM248 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ELSEIF (IND.EQ.4) THEN
         CALL KM273 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ELSEIF (IND.EQ.5) THEN
         CALL KM298 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ELSE
         CALL KM323 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,G11,G12)
      ENDIF
C
C *** Return point; End of subroutine
C
      RETURN
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KM198
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 198K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM198 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC198/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KM223
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 223K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM223 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC223/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END



C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KM248
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 248K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM248 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC248/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END
 

C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KMTAB273
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 273K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM273 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC273/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN 
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END
 

C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KM298
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 298K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM298 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC298/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END
 

C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE KM323
C *** CALCULATES BINARY ACTIVITY COEFFICIENTS BY KUSIK-MEISSNER METHOD. 
C     THE COMPUTATIONS HAVE BEEN PERFORMED AND THE RESULTS ARE STORED IN
C     LOOKUP TABLES. THE IONIC ACTIVITY 'IN' IS INPUT, AND THE ARRAY
C     'BINARR' IS RETURNED WITH THE BINARY COEFFICIENTS. 
C
C     TEMPERATURE IS 323K
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE KM323 (IN,G01,G02,G03,G04,G05,G06,G07,G08,G09,G10,
     &                     G11,G12)
C
C *** Common block definition
C
      COMMON /KMC323/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
      REAL IN 
C
C *** Find position in arrays for bincoef
C
      IF (IN.LE. 0.300000E+02) THEN
         ipos = NINT( 0.200000E+02*IN) + 1
      ELSE
         ipos =   600+NINT( 0.200000E+01*IN- 0.300000E+02)
      ENDIF
      ipos = min(ipos,  741)
C
C *** Assign values to return array
C
      G01 = BNC01M(ipos)
      G02 = BNC02M(ipos)
      G03 = BNC03M(ipos)
      G04 = BNC04M(ipos)
      G05 = BNC05M(ipos)
      G06 = BNC06M(ipos)
      G07 = BNC07M(ipos)
      G08 = BNC08M(ipos)
      G09 = BNC09M(ipos)
      G10 = BNC10M(ipos)
      G11 = BNC11M(ipos)
      G12 = BNC12M(ipos)
C
C *** Return point ; End of subroutine
C
      RETURN
      END
 

 
C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF198A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA KMCF198A
C
C *** Common block definition
C
      COMMON /KMC198/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.49640E-01,-0.10303E+00,-0.12698E+00,-0.14245E+00,-0.15358E+00,
     &-0.16202E+00,-0.16860E+00,-0.17383E+00,-0.17802E+00,-0.18139E+00,
     &-0.18410E+00,-0.18626E+00,-0.18796E+00,-0.18927E+00,-0.19024E+00,
     &-0.19093E+00,-0.19135E+00,-0.19156E+00,-0.19156E+00,-0.19139E+00,
     &-0.19106E+00,-0.19059E+00,-0.18999E+00,-0.18928E+00,-0.18846E+00,
     &-0.18756E+00,-0.18657E+00,-0.18550E+00,-0.18436E+00,-0.18316E+00,
     &-0.18191E+00,-0.18060E+00,-0.17925E+00,-0.17786E+00,-0.17643E+00,
     &-0.17496E+00,-0.17347E+00,-0.17194E+00,-0.17040E+00,-0.16883E+00,
     &-0.16724E+00,-0.16563E+00,-0.16401E+00,-0.16238E+00,-0.16073E+00,
     &-0.15908E+00,-0.15741E+00,-0.15574E+00,-0.15406E+00,-0.15237E+00,
     &-0.15068E+00,-0.14899E+00,-0.14729E+00,-0.14558E+00,-0.14388E+00,
     &-0.14217E+00,-0.14046E+00,-0.13874E+00,-0.13702E+00,-0.13530E+00,
     &-0.13357E+00,-0.13184E+00,-0.13011E+00,-0.12837E+00,-0.12663E+00,
     &-0.12488E+00,-0.12312E+00,-0.12136E+00,-0.11959E+00,-0.11781E+00,
     &-0.11603E+00,-0.11423E+00,-0.11243E+00,-0.11061E+00,-0.10878E+00,
     &-0.10694E+00,-0.10509E+00,-0.10322E+00,-0.10134E+00,-0.99450E-01,
     &-0.97542E-01,-0.95618E-01,-0.93679E-01,-0.91723E-01,-0.89752E-01,
     &-0.87764E-01,-0.85759E-01,-0.83737E-01,-0.81699E-01,-0.79643E-01,
     &-0.77570E-01,-0.75481E-01,-0.73375E-01,-0.71252E-01,-0.69113E-01,
     &-0.66957E-01,-0.64786E-01,-0.62600E-01,-0.60399E-01,-0.58183E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.55953E-01,-0.53709E-01,-0.51453E-01,-0.49184E-01,-0.46903E-01,
     &-0.44610E-01,-0.42307E-01,-0.39994E-01,-0.37670E-01,-0.35338E-01,
     &-0.32997E-01,-0.30649E-01,-0.28293E-01,-0.25931E-01,-0.23562E-01,
     &-0.21188E-01,-0.18808E-01,-0.16424E-01,-0.14036E-01,-0.11644E-01,
     &-0.92495E-02,-0.68520E-02,-0.44521E-02,-0.20502E-02, 0.35305E-03,
     & 0.27576E-02, 0.51630E-02, 0.75689E-02, 0.99750E-02, 0.12381E-01,
     & 0.14787E-01, 0.17192E-01, 0.19597E-01, 0.22000E-01, 0.24402E-01,
     & 0.26803E-01, 0.29203E-01, 0.31600E-01, 0.33996E-01, 0.36390E-01,
     & 0.38782E-01, 0.41172E-01, 0.43559E-01, 0.45944E-01, 0.48326E-01,
     & 0.50706E-01, 0.53083E-01, 0.55458E-01, 0.57830E-01, 0.60198E-01,
     & 0.62564E-01, 0.64927E-01, 0.67287E-01, 0.69644E-01, 0.71998E-01,
     & 0.74349E-01, 0.76696E-01, 0.79041E-01, 0.81382E-01, 0.83719E-01,
     & 0.86054E-01, 0.88385E-01, 0.90713E-01, 0.93037E-01, 0.95358E-01,
     & 0.97676E-01, 0.99990E-01, 0.10230E+00, 0.10461E+00, 0.10691E+00,
     & 0.10921E+00, 0.11151E+00, 0.11380E+00, 0.11609E+00, 0.11837E+00,
     & 0.12066E+00, 0.12293E+00, 0.12521E+00, 0.12748E+00, 0.12975E+00,
     & 0.13201E+00, 0.13427E+00, 0.13653E+00, 0.13878E+00, 0.14103E+00,
     & 0.14327E+00, 0.14551E+00, 0.14775E+00, 0.14998E+00, 0.15221E+00,
     & 0.15444E+00, 0.15666E+00, 0.15888E+00, 0.16109E+00, 0.16330E+00,
     & 0.16551E+00, 0.16771E+00, 0.16991E+00, 0.17211E+00, 0.17430E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.17649E+00, 0.17867E+00, 0.18085E+00, 0.18302E+00, 0.18520E+00,
     & 0.18736E+00, 0.18953E+00, 0.19169E+00, 0.19384E+00, 0.19600E+00,
     & 0.19814E+00, 0.20029E+00, 0.20243E+00, 0.20457E+00, 0.20670E+00,
     & 0.20883E+00, 0.21095E+00, 0.21307E+00, 0.21519E+00, 0.21730E+00,
     & 0.21941E+00, 0.22151E+00, 0.22362E+00, 0.22571E+00, 0.22781E+00,
     & 0.22990E+00, 0.23198E+00, 0.23406E+00, 0.23614E+00, 0.23821E+00,
     & 0.24028E+00, 0.24235E+00, 0.24441E+00, 0.24647E+00, 0.24852E+00,
     & 0.25057E+00, 0.25262E+00, 0.25466E+00, 0.25670E+00, 0.25873E+00,
     & 0.26076E+00, 0.26279E+00, 0.26481E+00, 0.26683E+00, 0.26885E+00,
     & 0.27086E+00, 0.27286E+00, 0.27487E+00, 0.27687E+00, 0.27886E+00,
     & 0.28085E+00, 0.28284E+00, 0.28483E+00, 0.28681E+00, 0.28878E+00,
     & 0.29075E+00, 0.29272E+00, 0.29469E+00, 0.29665E+00, 0.29861E+00,
     & 0.30056E+00, 0.30251E+00, 0.30445E+00, 0.30640E+00, 0.30833E+00,
     & 0.31027E+00, 0.31220E+00, 0.31412E+00, 0.31605E+00, 0.31797E+00,
     & 0.31988E+00, 0.32179E+00, 0.32370E+00, 0.32561E+00, 0.32751E+00,
     & 0.32940E+00, 0.33129E+00, 0.33318E+00, 0.33507E+00, 0.33695E+00,
     & 0.33883E+00, 0.34070E+00, 0.34257E+00, 0.34444E+00, 0.34630E+00,
     & 0.34816E+00, 0.35002E+00, 0.35187E+00, 0.35372E+00, 0.35556E+00,
     & 0.35741E+00, 0.35924E+00, 0.36108E+00, 0.36291E+00, 0.36473E+00,
     & 0.36656E+00, 0.36838E+00, 0.37019E+00, 0.37200E+00, 0.37381E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.37562E+00, 0.37742E+00, 0.37922E+00, 0.38101E+00, 0.38280E+00,
     & 0.38459E+00, 0.38637E+00, 0.38815E+00, 0.38993E+00, 0.39170E+00,
     & 0.39347E+00, 0.39524E+00, 0.39700E+00, 0.39876E+00, 0.40052E+00,
     & 0.40227E+00, 0.40402E+00, 0.40576E+00, 0.40751E+00, 0.40924E+00,
     & 0.41098E+00, 0.41271E+00, 0.41444E+00, 0.41616E+00, 0.41788E+00,
     & 0.41960E+00, 0.42132E+00, 0.42303E+00, 0.42474E+00, 0.42644E+00,
     & 0.42814E+00, 0.42984E+00, 0.43153E+00, 0.43322E+00, 0.43491E+00,
     & 0.43659E+00, 0.43828E+00, 0.43995E+00, 0.44163E+00, 0.44330E+00,
     & 0.44497E+00, 0.44663E+00, 0.44829E+00, 0.44995E+00, 0.45160E+00,
     & 0.45325E+00, 0.45490E+00, 0.45655E+00, 0.45819E+00, 0.45983E+00,
     & 0.46146E+00, 0.46309E+00, 0.46472E+00, 0.46635E+00, 0.46797E+00,
     & 0.46959E+00, 0.47120E+00, 0.47282E+00, 0.47442E+00, 0.47603E+00,
     & 0.47763E+00, 0.47923E+00, 0.48083E+00, 0.48242E+00, 0.48401E+00,
     & 0.48560E+00, 0.48719E+00, 0.48877E+00, 0.49035E+00, 0.49192E+00,
     & 0.49349E+00, 0.49506E+00, 0.49663E+00, 0.49819E+00, 0.49975E+00,
     & 0.50130E+00, 0.50286E+00, 0.50441E+00, 0.50596E+00, 0.50750E+00,
     & 0.50904E+00, 0.51058E+00, 0.51212E+00, 0.51365E+00, 0.51518E+00,
     & 0.51670E+00, 0.51823E+00, 0.51975E+00, 0.52126E+00, 0.52278E+00,
     & 0.52429E+00, 0.52580E+00, 0.52730E+00, 0.52881E+00, 0.53031E+00,
     & 0.53180E+00, 0.53330E+00, 0.53479E+00, 0.53628E+00, 0.53776E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.53924E+00, 0.54072E+00, 0.54220E+00, 0.54367E+00, 0.54514E+00,
     & 0.54661E+00, 0.54808E+00, 0.54954E+00, 0.55100E+00, 0.55246E+00,
     & 0.55391E+00, 0.55536E+00, 0.55681E+00, 0.55825E+00, 0.55970E+00,
     & 0.56114E+00, 0.56257E+00, 0.56401E+00, 0.56544E+00, 0.56687E+00,
     & 0.56829E+00, 0.56972E+00, 0.57114E+00, 0.57256E+00, 0.57397E+00,
     & 0.57538E+00, 0.57679E+00, 0.57820E+00, 0.57960E+00, 0.58101E+00,
     & 0.58240E+00, 0.58380E+00, 0.58519E+00, 0.58659E+00, 0.58797E+00,
     & 0.58936E+00, 0.59074E+00, 0.59212E+00, 0.59350E+00, 0.59487E+00,
     & 0.59625E+00, 0.59762E+00, 0.59898E+00, 0.60035E+00, 0.60171E+00,
     & 0.60307E+00, 0.60442E+00, 0.60578E+00, 0.60713E+00, 0.60848E+00,
     & 0.60982E+00, 0.61117E+00, 0.61251E+00, 0.61385E+00, 0.61518E+00,
     & 0.61652E+00, 0.61785E+00, 0.61918E+00, 0.62050E+00, 0.62183E+00,
     & 0.62315E+00, 0.62447E+00, 0.62578E+00, 0.62709E+00, 0.62841E+00,
     & 0.62971E+00, 0.63102E+00, 0.63232E+00, 0.63362E+00, 0.63492E+00,
     & 0.63622E+00, 0.63751E+00, 0.63880E+00, 0.64009E+00, 0.64138E+00,
     & 0.64266E+00, 0.64394E+00, 0.64522E+00, 0.64650E+00, 0.64777E+00,
     & 0.64905E+00, 0.65032E+00, 0.65158E+00, 0.65285E+00, 0.65411E+00,
     & 0.65537E+00, 0.65663E+00, 0.65788E+00, 0.65913E+00, 0.66039E+00,
     & 0.66163E+00, 0.66288E+00, 0.66412E+00, 0.66536E+00, 0.66660E+00,
     & 0.66784E+00, 0.66907E+00, 0.67031E+00, 0.67153E+00, 0.67276E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.67399E+00, 0.67521E+00, 0.67643E+00, 0.67765E+00, 0.67886E+00,
     & 0.68008E+00, 0.68129E+00, 0.68250E+00, 0.68371E+00, 0.68491E+00,
     & 0.68611E+00, 0.68731E+00, 0.68851E+00, 0.68971E+00, 0.69090E+00,
     & 0.69209E+00, 0.69328E+00, 0.69447E+00, 0.69565E+00, 0.69683E+00,
     & 0.69801E+00, 0.69919E+00, 0.70037E+00, 0.70154E+00, 0.70271E+00,
     & 0.70388E+00, 0.70505E+00, 0.70621E+00, 0.70738E+00, 0.70854E+00,
     & 0.70970E+00, 0.71085E+00, 0.71201E+00, 0.71316E+00, 0.71431E+00,
     & 0.71546E+00, 0.71660E+00, 0.71775E+00, 0.71889E+00, 0.72003E+00,
     & 0.72117E+00, 0.72230E+00, 0.72343E+00, 0.72456E+00, 0.72569E+00,
     & 0.72682E+00, 0.72795E+00, 0.72907E+00, 0.73019E+00, 0.73131E+00,
     & 0.73242E+00, 0.73354E+00, 0.73465E+00, 0.73576E+00, 0.73687E+00,
     & 0.73798E+00, 0.73908E+00, 0.74019E+00, 0.74129E+00, 0.74238E+00,
     & 0.74348E+00, 0.74457E+00, 0.74567E+00, 0.74676E+00, 0.74785E+00,
     & 0.74893E+00, 0.75002E+00, 0.75110E+00, 0.75218E+00, 0.75326E+00,
     & 0.75434E+00, 0.75541E+00, 0.75648E+00, 0.75755E+00, 0.75862E+00,
     & 0.75969E+00, 0.76075E+00, 0.76182E+00, 0.76288E+00, 0.76394E+00,
     & 0.76500E+00, 0.76605E+00, 0.76710E+00, 0.76816E+00, 0.76921E+00,
     & 0.77025E+00, 0.77130E+00, 0.77234E+00, 0.77339E+00, 0.77443E+00,
     & 0.77546E+00, 0.77650E+00, 0.77754E+00, 0.77857E+00, 0.77960E+00,
     & 0.78063E+00, 0.78165E+00, 0.78268E+00, 0.78370E+00, 0.78473E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.79565E+00, 0.80557E+00, 0.81531E+00, 0.82489E+00, 0.83429E+00,
     & 0.84353E+00, 0.85261E+00, 0.86153E+00, 0.87029E+00, 0.87890E+00,
     & 0.88737E+00, 0.89568E+00, 0.90385E+00, 0.91188E+00, 0.91978E+00,
     & 0.92754E+00, 0.93516E+00, 0.94265E+00, 0.95002E+00, 0.95726E+00,
     & 0.96438E+00, 0.97138E+00, 0.97825E+00, 0.98501E+00, 0.99166E+00,
     & 0.99819E+00, 0.10046E+01, 0.10109E+01, 0.10171E+01, 0.10232E+01,
     & 0.10292E+01, 0.10351E+01, 0.10409E+01, 0.10466E+01, 0.10522E+01,
     & 0.10577E+01, 0.10632E+01, 0.10685E+01, 0.10737E+01, 0.10788E+01,
     & 0.10839E+01, 0.10889E+01, 0.10938E+01, 0.10986E+01, 0.11033E+01,
     & 0.11079E+01, 0.11125E+01, 0.11169E+01, 0.11213E+01, 0.11257E+01,
     & 0.11299E+01, 0.11341E+01, 0.11382E+01, 0.11422E+01, 0.11462E+01,
     & 0.11501E+01, 0.11539E+01, 0.11577E+01, 0.11613E+01, 0.11650E+01,
     & 0.11685E+01, 0.11720E+01, 0.11754E+01, 0.11788E+01, 0.11821E+01,
     & 0.11853E+01, 0.11885E+01, 0.11917E+01, 0.11947E+01, 0.11977E+01,
     & 0.12007E+01, 0.12036E+01, 0.12064E+01, 0.12092E+01, 0.12119E+01,
     & 0.12146E+01, 0.12172E+01, 0.12198E+01, 0.12223E+01, 0.12248E+01,
     & 0.12272E+01, 0.12296E+01, 0.12319E+01, 0.12342E+01, 0.12364E+01,
     & 0.12386E+01, 0.12407E+01, 0.12428E+01, 0.12449E+01, 0.12469E+01,
     & 0.12488E+01, 0.12507E+01, 0.12526E+01, 0.12544E+01, 0.12562E+01,
     & 0.12579E+01, 0.12596E+01, 0.12613E+01, 0.12629E+01, 0.12645E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.12660E+01, 0.12675E+01, 0.12690E+01, 0.12704E+01, 0.12718E+01,
     & 0.12731E+01, 0.12744E+01, 0.12757E+01, 0.12769E+01, 0.12781E+01,
     & 0.12793E+01, 0.12804E+01, 0.12815E+01, 0.12825E+01, 0.12836E+01,
     & 0.12846E+01, 0.12855E+01, 0.12864E+01, 0.12873E+01, 0.12882E+01,
     & 0.12890E+01, 0.12898E+01, 0.12906E+01, 0.12913E+01, 0.12920E+01,
     & 0.12927E+01, 0.12933E+01, 0.12939E+01, 0.12945E+01, 0.12951E+01,
     & 0.12956E+01, 0.12961E+01, 0.12966E+01, 0.12970E+01, 0.12974E+01,
     & 0.12978E+01, 0.12981E+01, 0.12985E+01, 0.12988E+01, 0.12991E+01,
     & 0.12993E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.10289E+00,-0.22546E+00,-0.28765E+00,-0.33213E+00,-0.36741E+00,
     &-0.39691E+00,-0.42240E+00,-0.44495E+00,-0.46522E+00,-0.48367E+00,
     &-0.50064E+00,-0.51637E+00,-0.53106E+00,-0.54484E+00,-0.55785E+00,
     &-0.57017E+00,-0.58189E+00,-0.59306E+00,-0.60376E+00,-0.61401E+00,
     &-0.62387E+00,-0.63337E+00,-0.64254E+00,-0.65140E+00,-0.65998E+00,
     &-0.66830E+00,-0.67639E+00,-0.68424E+00,-0.69189E+00,-0.69934E+00,
     &-0.70661E+00,-0.71370E+00,-0.72063E+00,-0.72740E+00,-0.73403E+00,
     &-0.74052E+00,-0.74688E+00,-0.75312E+00,-0.75923E+00,-0.76524E+00,
     &-0.77113E+00,-0.77693E+00,-0.78262E+00,-0.78822E+00,-0.79373E+00,
     &-0.79915E+00,-0.80450E+00,-0.80976E+00,-0.81494E+00,-0.82005E+00,
     &-0.82509E+00,-0.83006E+00,-0.83497E+00,-0.83981E+00,-0.84459E+00,
     &-0.84931E+00,-0.85398E+00,-0.85859E+00,-0.86315E+00,-0.86766E+00,
     &-0.87211E+00,-0.87652E+00,-0.88089E+00,-0.88521E+00,-0.88948E+00,
     &-0.89372E+00,-0.89791E+00,-0.90207E+00,-0.90619E+00,-0.91027E+00,
     &-0.91432E+00,-0.91833E+00,-0.92231E+00,-0.92626E+00,-0.93018E+00,
     &-0.93406E+00,-0.93792E+00,-0.94175E+00,-0.94556E+00,-0.94933E+00,
     &-0.95308E+00,-0.95681E+00,-0.96051E+00,-0.96419E+00,-0.96785E+00,
     &-0.97149E+00,-0.97510E+00,-0.97869E+00,-0.98226E+00,-0.98582E+00,
     &-0.98935E+00,-0.99286E+00,-0.99636E+00,-0.99984E+00,-0.10033E+01,
     &-0.10067E+01,-0.10102E+01,-0.10136E+01,-0.10170E+01,-0.10203E+01/

      DATA (BNC02M(I),I=101,200)/
     &-0.10237E+01,-0.10271E+01,-0.10304E+01,-0.10337E+01,-0.10370E+01,
     &-0.10403E+01,-0.10436E+01,-0.10468E+01,-0.10501E+01,-0.10533E+01,
     &-0.10565E+01,-0.10597E+01,-0.10629E+01,-0.10661E+01,-0.10692E+01,
     &-0.10724E+01,-0.10755E+01,-0.10787E+01,-0.10818E+01,-0.10849E+01,
     &-0.10880E+01,-0.10911E+01,-0.10941E+01,-0.10972E+01,-0.11002E+01,
     &-0.11033E+01,-0.11063E+01,-0.11093E+01,-0.11123E+01,-0.11153E+01,
     &-0.11182E+01,-0.11212E+01,-0.11242E+01,-0.11271E+01,-0.11301E+01,
     &-0.11330E+01,-0.11359E+01,-0.11388E+01,-0.11417E+01,-0.11446E+01,
     &-0.11475E+01,-0.11503E+01,-0.11532E+01,-0.11560E+01,-0.11589E+01,
     &-0.11617E+01,-0.11645E+01,-0.11673E+01,-0.11701E+01,-0.11729E+01,
     &-0.11757E+01,-0.11785E+01,-0.11813E+01,-0.11840E+01,-0.11868E+01,
     &-0.11895E+01,-0.11923E+01,-0.11950E+01,-0.11977E+01,-0.12004E+01,
     &-0.12032E+01,-0.12059E+01,-0.12085E+01,-0.12112E+01,-0.12139E+01,
     &-0.12166E+01,-0.12192E+01,-0.12219E+01,-0.12246E+01,-0.12272E+01,
     &-0.12298E+01,-0.12325E+01,-0.12351E+01,-0.12377E+01,-0.12403E+01,
     &-0.12429E+01,-0.12455E+01,-0.12481E+01,-0.12507E+01,-0.12533E+01,
     &-0.12558E+01,-0.12584E+01,-0.12610E+01,-0.12635E+01,-0.12661E+01,
     &-0.12686E+01,-0.12712E+01,-0.12737E+01,-0.12762E+01,-0.12787E+01,
     &-0.12813E+01,-0.12838E+01,-0.12863E+01,-0.12888E+01,-0.12913E+01,
     &-0.12937E+01,-0.12962E+01,-0.12987E+01,-0.13012E+01,-0.13036E+01/

      DATA (BNC02M(I),I=201,300)/
     &-0.13061E+01,-0.13086E+01,-0.13110E+01,-0.13135E+01,-0.13159E+01,
     &-0.13183E+01,-0.13208E+01,-0.13232E+01,-0.13256E+01,-0.13280E+01,
     &-0.13305E+01,-0.13329E+01,-0.13353E+01,-0.13377E+01,-0.13401E+01,
     &-0.13425E+01,-0.13449E+01,-0.13472E+01,-0.13496E+01,-0.13520E+01,
     &-0.13544E+01,-0.13567E+01,-0.13591E+01,-0.13614E+01,-0.13638E+01,
     &-0.13661E+01,-0.13685E+01,-0.13708E+01,-0.13732E+01,-0.13755E+01,
     &-0.13778E+01,-0.13802E+01,-0.13825E+01,-0.13848E+01,-0.13871E+01,
     &-0.13894E+01,-0.13917E+01,-0.13940E+01,-0.13963E+01,-0.13986E+01,
     &-0.14009E+01,-0.14032E+01,-0.14055E+01,-0.14078E+01,-0.14100E+01,
     &-0.14123E+01,-0.14146E+01,-0.14168E+01,-0.14191E+01,-0.14214E+01,
     &-0.14236E+01,-0.14259E+01,-0.14281E+01,-0.14304E+01,-0.14326E+01,
     &-0.14349E+01,-0.14371E+01,-0.14393E+01,-0.14415E+01,-0.14438E+01,
     &-0.14460E+01,-0.14482E+01,-0.14504E+01,-0.14526E+01,-0.14549E+01,
     &-0.14571E+01,-0.14593E+01,-0.14615E+01,-0.14637E+01,-0.14659E+01,
     &-0.14681E+01,-0.14702E+01,-0.14724E+01,-0.14746E+01,-0.14768E+01,
     &-0.14790E+01,-0.14812E+01,-0.14833E+01,-0.14855E+01,-0.14877E+01,
     &-0.14898E+01,-0.14920E+01,-0.14941E+01,-0.14963E+01,-0.14985E+01,
     &-0.15006E+01,-0.15028E+01,-0.15049E+01,-0.15070E+01,-0.15092E+01,
     &-0.15113E+01,-0.15134E+01,-0.15156E+01,-0.15177E+01,-0.15198E+01,
     &-0.15220E+01,-0.15241E+01,-0.15262E+01,-0.15283E+01,-0.15304E+01/

      DATA (BNC02M(I),I=301,400)/
     &-0.15325E+01,-0.15347E+01,-0.15368E+01,-0.15389E+01,-0.15410E+01,
     &-0.15431E+01,-0.15452E+01,-0.15473E+01,-0.15493E+01,-0.15514E+01,
     &-0.15535E+01,-0.15556E+01,-0.15577E+01,-0.15598E+01,-0.15618E+01,
     &-0.15639E+01,-0.15660E+01,-0.15681E+01,-0.15701E+01,-0.15722E+01,
     &-0.15743E+01,-0.15763E+01,-0.15784E+01,-0.15805E+01,-0.15825E+01,
     &-0.15846E+01,-0.15866E+01,-0.15887E+01,-0.15907E+01,-0.15928E+01,
     &-0.15948E+01,-0.15968E+01,-0.15989E+01,-0.16009E+01,-0.16030E+01,
     &-0.16050E+01,-0.16070E+01,-0.16090E+01,-0.16111E+01,-0.16131E+01,
     &-0.16151E+01,-0.16171E+01,-0.16192E+01,-0.16212E+01,-0.16232E+01,
     &-0.16252E+01,-0.16272E+01,-0.16292E+01,-0.16312E+01,-0.16332E+01,
     &-0.16352E+01,-0.16372E+01,-0.16392E+01,-0.16412E+01,-0.16432E+01,
     &-0.16452E+01,-0.16472E+01,-0.16492E+01,-0.16512E+01,-0.16532E+01,
     &-0.16552E+01,-0.16572E+01,-0.16591E+01,-0.16611E+01,-0.16631E+01,
     &-0.16651E+01,-0.16671E+01,-0.16690E+01,-0.16710E+01,-0.16730E+01,
     &-0.16749E+01,-0.16769E+01,-0.16789E+01,-0.16808E+01,-0.16828E+01,
     &-0.16847E+01,-0.16867E+01,-0.16887E+01,-0.16906E+01,-0.16926E+01,
     &-0.16945E+01,-0.16965E+01,-0.16984E+01,-0.17004E+01,-0.17023E+01,
     &-0.17042E+01,-0.17062E+01,-0.17081E+01,-0.17101E+01,-0.17120E+01,
     &-0.17139E+01,-0.17159E+01,-0.17178E+01,-0.17197E+01,-0.17217E+01,
     &-0.17236E+01,-0.17255E+01,-0.17274E+01,-0.17294E+01,-0.17313E+01/

      DATA (BNC02M(I),I=401,500)/
     &-0.17332E+01,-0.17351E+01,-0.17370E+01,-0.17390E+01,-0.17409E+01,
     &-0.17428E+01,-0.17447E+01,-0.17466E+01,-0.17485E+01,-0.17504E+01,
     &-0.17523E+01,-0.17542E+01,-0.17561E+01,-0.17580E+01,-0.17599E+01,
     &-0.17618E+01,-0.17637E+01,-0.17656E+01,-0.17675E+01,-0.17694E+01,
     &-0.17713E+01,-0.17732E+01,-0.17751E+01,-0.17770E+01,-0.17788E+01,
     &-0.17807E+01,-0.17826E+01,-0.17845E+01,-0.17864E+01,-0.17883E+01,
     &-0.17901E+01,-0.17920E+01,-0.17939E+01,-0.17958E+01,-0.17976E+01,
     &-0.17995E+01,-0.18014E+01,-0.18032E+01,-0.18051E+01,-0.18070E+01,
     &-0.18088E+01,-0.18107E+01,-0.18126E+01,-0.18144E+01,-0.18163E+01,
     &-0.18181E+01,-0.18200E+01,-0.18218E+01,-0.18237E+01,-0.18256E+01,
     &-0.18274E+01,-0.18293E+01,-0.18311E+01,-0.18330E+01,-0.18348E+01,
     &-0.18367E+01,-0.18385E+01,-0.18403E+01,-0.18422E+01,-0.18440E+01,
     &-0.18459E+01,-0.18477E+01,-0.18495E+01,-0.18514E+01,-0.18532E+01,
     &-0.18550E+01,-0.18569E+01,-0.18587E+01,-0.18605E+01,-0.18624E+01,
     &-0.18642E+01,-0.18660E+01,-0.18679E+01,-0.18697E+01,-0.18715E+01,
     &-0.18733E+01,-0.18751E+01,-0.18770E+01,-0.18788E+01,-0.18806E+01,
     &-0.18824E+01,-0.18842E+01,-0.18861E+01,-0.18879E+01,-0.18897E+01,
     &-0.18915E+01,-0.18933E+01,-0.18951E+01,-0.18969E+01,-0.18987E+01,
     &-0.19005E+01,-0.19023E+01,-0.19041E+01,-0.19060E+01,-0.19078E+01,
     &-0.19096E+01,-0.19114E+01,-0.19132E+01,-0.19150E+01,-0.19167E+01/

      DATA (BNC02M(I),I=501,600)/
     &-0.19185E+01,-0.19203E+01,-0.19221E+01,-0.19239E+01,-0.19257E+01,
     &-0.19275E+01,-0.19293E+01,-0.19311E+01,-0.19329E+01,-0.19347E+01,
     &-0.19365E+01,-0.19382E+01,-0.19400E+01,-0.19418E+01,-0.19436E+01,
     &-0.19454E+01,-0.19472E+01,-0.19489E+01,-0.19507E+01,-0.19525E+01,
     &-0.19543E+01,-0.19560E+01,-0.19578E+01,-0.19596E+01,-0.19614E+01,
     &-0.19631E+01,-0.19649E+01,-0.19667E+01,-0.19684E+01,-0.19702E+01,
     &-0.19720E+01,-0.19737E+01,-0.19755E+01,-0.19773E+01,-0.19790E+01,
     &-0.19808E+01,-0.19826E+01,-0.19843E+01,-0.19861E+01,-0.19878E+01,
     &-0.19896E+01,-0.19914E+01,-0.19931E+01,-0.19949E+01,-0.19966E+01,
     &-0.19984E+01,-0.20001E+01,-0.20019E+01,-0.20036E+01,-0.20054E+01,
     &-0.20071E+01,-0.20089E+01,-0.20106E+01,-0.20124E+01,-0.20141E+01,
     &-0.20159E+01,-0.20176E+01,-0.20193E+01,-0.20211E+01,-0.20228E+01,
     &-0.20246E+01,-0.20263E+01,-0.20281E+01,-0.20298E+01,-0.20315E+01,
     &-0.20333E+01,-0.20350E+01,-0.20367E+01,-0.20385E+01,-0.20402E+01,
     &-0.20419E+01,-0.20437E+01,-0.20454E+01,-0.20471E+01,-0.20489E+01,
     &-0.20506E+01,-0.20523E+01,-0.20540E+01,-0.20558E+01,-0.20575E+01,
     &-0.20592E+01,-0.20609E+01,-0.20627E+01,-0.20644E+01,-0.20661E+01,
     &-0.20678E+01,-0.20695E+01,-0.20713E+01,-0.20730E+01,-0.20747E+01,
     &-0.20764E+01,-0.20781E+01,-0.20798E+01,-0.20816E+01,-0.20833E+01,
     &-0.20850E+01,-0.20867E+01,-0.20884E+01,-0.20901E+01,-0.20918E+01/

      DATA (BNC02M(I),I=601,700)/
     &-0.21103E+01,-0.21272E+01,-0.21441E+01,-0.21610E+01,-0.21778E+01,
     &-0.21945E+01,-0.22111E+01,-0.22277E+01,-0.22442E+01,-0.22607E+01,
     &-0.22771E+01,-0.22934E+01,-0.23097E+01,-0.23260E+01,-0.23422E+01,
     &-0.23583E+01,-0.23744E+01,-0.23904E+01,-0.24064E+01,-0.24224E+01,
     &-0.24383E+01,-0.24541E+01,-0.24699E+01,-0.24857E+01,-0.25014E+01,
     &-0.25171E+01,-0.25327E+01,-0.25483E+01,-0.25639E+01,-0.25794E+01,
     &-0.25949E+01,-0.26104E+01,-0.26258E+01,-0.26412E+01,-0.26565E+01,
     &-0.26718E+01,-0.26871E+01,-0.27023E+01,-0.27175E+01,-0.27327E+01,
     &-0.27479E+01,-0.27630E+01,-0.27781E+01,-0.27931E+01,-0.28082E+01,
     &-0.28232E+01,-0.28381E+01,-0.28531E+01,-0.28680E+01,-0.28829E+01,
     &-0.28977E+01,-0.29125E+01,-0.29274E+01,-0.29421E+01,-0.29569E+01,
     &-0.29716E+01,-0.29863E+01,-0.30010E+01,-0.30157E+01,-0.30303E+01,
     &-0.30449E+01,-0.30595E+01,-0.30740E+01,-0.30886E+01,-0.31031E+01,
     &-0.31176E+01,-0.31321E+01,-0.31465E+01,-0.31610E+01,-0.31754E+01,
     &-0.31898E+01,-0.32042E+01,-0.32185E+01,-0.32328E+01,-0.32472E+01,
     &-0.32614E+01,-0.32757E+01,-0.32900E+01,-0.33042E+01,-0.33184E+01,
     &-0.33326E+01,-0.33468E+01,-0.33610E+01,-0.33751E+01,-0.33893E+01,
     &-0.34034E+01,-0.34175E+01,-0.34316E+01,-0.34456E+01,-0.34597E+01,
     &-0.34737E+01,-0.34877E+01,-0.35017E+01,-0.35157E+01,-0.35297E+01,
     &-0.35436E+01,-0.35576E+01,-0.35715E+01,-0.35854E+01,-0.35993E+01/

      DATA (BNC02M(I),I=701,741)/
     &-0.36132E+01,-0.36270E+01,-0.36409E+01,-0.36547E+01,-0.36685E+01,
     &-0.36823E+01,-0.36961E+01,-0.37099E+01,-0.37237E+01,-0.37374E+01,
     &-0.37512E+01,-0.37649E+01,-0.37786E+01,-0.37923E+01,-0.38060E+01,
     &-0.38197E+01,-0.38333E+01,-0.38470E+01,-0.38606E+01,-0.38743E+01,
     &-0.38879E+01,-0.39015E+01,-0.39151E+01,-0.39286E+01,-0.39422E+01,
     &-0.39558E+01,-0.39693E+01,-0.39828E+01,-0.39964E+01,-0.40099E+01,
     &-0.40234E+01,-0.40369E+01,-0.40503E+01,-0.40638E+01,-0.40773E+01,
     &-0.40907E+01,-0.41041E+01,-0.41176E+01,-0.41310E+01,-0.41444E+01,
     &-0.41578E+01
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.51619E-01,-0.11367E+00,-0.14547E+00,-0.16839E+00,-0.18667E+00,
     &-0.20205E+00,-0.21542E+00,-0.22729E+00,-0.23802E+00,-0.24783E+00,
     &-0.25689E+00,-0.26532E+00,-0.27322E+00,-0.28066E+00,-0.28771E+00,
     &-0.29440E+00,-0.30079E+00,-0.30691E+00,-0.31277E+00,-0.31841E+00,
     &-0.32385E+00,-0.32910E+00,-0.33418E+00,-0.33911E+00,-0.34388E+00,
     &-0.34853E+00,-0.35304E+00,-0.35744E+00,-0.36173E+00,-0.36592E+00,
     &-0.37001E+00,-0.37401E+00,-0.37792E+00,-0.38175E+00,-0.38550E+00,
     &-0.38918E+00,-0.39279E+00,-0.39633E+00,-0.39981E+00,-0.40323E+00,
     &-0.40658E+00,-0.40989E+00,-0.41313E+00,-0.41633E+00,-0.41948E+00,
     &-0.42258E+00,-0.42563E+00,-0.42864E+00,-0.43161E+00,-0.43454E+00,
     &-0.43743E+00,-0.44028E+00,-0.44309E+00,-0.44587E+00,-0.44861E+00,
     &-0.45133E+00,-0.45401E+00,-0.45666E+00,-0.45928E+00,-0.46187E+00,
     &-0.46443E+00,-0.46697E+00,-0.46948E+00,-0.47197E+00,-0.47443E+00,
     &-0.47688E+00,-0.47929E+00,-0.48169E+00,-0.48407E+00,-0.48643E+00,
     &-0.48877E+00,-0.49109E+00,-0.49339E+00,-0.49567E+00,-0.49794E+00,
     &-0.50020E+00,-0.50243E+00,-0.50466E+00,-0.50687E+00,-0.50906E+00,
     &-0.51124E+00,-0.51341E+00,-0.51557E+00,-0.51771E+00,-0.51985E+00,
     &-0.52197E+00,-0.52408E+00,-0.52618E+00,-0.52827E+00,-0.53035E+00,
     &-0.53242E+00,-0.53448E+00,-0.53654E+00,-0.53858E+00,-0.54062E+00,
     &-0.54264E+00,-0.54466E+00,-0.54667E+00,-0.54867E+00,-0.55066E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.55265E+00,-0.55463E+00,-0.55660E+00,-0.55856E+00,-0.56051E+00,
     &-0.56246E+00,-0.56440E+00,-0.56633E+00,-0.56826E+00,-0.57018E+00,
     &-0.57209E+00,-0.57399E+00,-0.57589E+00,-0.57778E+00,-0.57966E+00,
     &-0.58154E+00,-0.58341E+00,-0.58527E+00,-0.58713E+00,-0.58898E+00,
     &-0.59082E+00,-0.59265E+00,-0.59448E+00,-0.59631E+00,-0.59812E+00,
     &-0.59993E+00,-0.60174E+00,-0.60353E+00,-0.60532E+00,-0.60711E+00,
     &-0.60888E+00,-0.61066E+00,-0.61242E+00,-0.61418E+00,-0.61593E+00,
     &-0.61768E+00,-0.61942E+00,-0.62116E+00,-0.62289E+00,-0.62461E+00,
     &-0.62633E+00,-0.62805E+00,-0.62975E+00,-0.63146E+00,-0.63315E+00,
     &-0.63484E+00,-0.63653E+00,-0.63821E+00,-0.63988E+00,-0.64155E+00,
     &-0.64322E+00,-0.64488E+00,-0.64653E+00,-0.64818E+00,-0.64983E+00,
     &-0.65147E+00,-0.65310E+00,-0.65473E+00,-0.65636E+00,-0.65798E+00,
     &-0.65960E+00,-0.66121E+00,-0.66282E+00,-0.66442E+00,-0.66602E+00,
     &-0.66761E+00,-0.66920E+00,-0.67079E+00,-0.67237E+00,-0.67395E+00,
     &-0.67552E+00,-0.67709E+00,-0.67865E+00,-0.68021E+00,-0.68177E+00,
     &-0.68332E+00,-0.68487E+00,-0.68641E+00,-0.68795E+00,-0.68949E+00,
     &-0.69102E+00,-0.69255E+00,-0.69408E+00,-0.69560E+00,-0.69712E+00,
     &-0.69863E+00,-0.70014E+00,-0.70165E+00,-0.70316E+00,-0.70466E+00,
     &-0.70615E+00,-0.70765E+00,-0.70914E+00,-0.71062E+00,-0.71211E+00,
     &-0.71358E+00,-0.71506E+00,-0.71653E+00,-0.71800E+00,-0.71947E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.72093E+00,-0.72239E+00,-0.72385E+00,-0.72531E+00,-0.72676E+00,
     &-0.72820E+00,-0.72965E+00,-0.73109E+00,-0.73253E+00,-0.73397E+00,
     &-0.73540E+00,-0.73683E+00,-0.73825E+00,-0.73968E+00,-0.74110E+00,
     &-0.74252E+00,-0.74393E+00,-0.74535E+00,-0.74676E+00,-0.74816E+00,
     &-0.74957E+00,-0.75097E+00,-0.75237E+00,-0.75376E+00,-0.75516E+00,
     &-0.75655E+00,-0.75794E+00,-0.75932E+00,-0.76071E+00,-0.76209E+00,
     &-0.76346E+00,-0.76484E+00,-0.76621E+00,-0.76758E+00,-0.76895E+00,
     &-0.77032E+00,-0.77168E+00,-0.77304E+00,-0.77440E+00,-0.77575E+00,
     &-0.77711E+00,-0.77846E+00,-0.77981E+00,-0.78115E+00,-0.78250E+00,
     &-0.78384E+00,-0.78518E+00,-0.78652E+00,-0.78785E+00,-0.78918E+00,
     &-0.79051E+00,-0.79184E+00,-0.79317E+00,-0.79449E+00,-0.79581E+00,
     &-0.79713E+00,-0.79845E+00,-0.79977E+00,-0.80108E+00,-0.80239E+00,
     &-0.80370E+00,-0.80500E+00,-0.80631E+00,-0.80761E+00,-0.80891E+00,
     &-0.81021E+00,-0.81151E+00,-0.81280E+00,-0.81410E+00,-0.81539E+00,
     &-0.81667E+00,-0.81796E+00,-0.81925E+00,-0.82053E+00,-0.82181E+00,
     &-0.82309E+00,-0.82437E+00,-0.82564E+00,-0.82691E+00,-0.82819E+00,
     &-0.82946E+00,-0.83072E+00,-0.83199E+00,-0.83325E+00,-0.83452E+00,
     &-0.83578E+00,-0.83703E+00,-0.83829E+00,-0.83955E+00,-0.84080E+00,
     &-0.84205E+00,-0.84330E+00,-0.84455E+00,-0.84580E+00,-0.84704E+00,
     &-0.84828E+00,-0.84952E+00,-0.85076E+00,-0.85200E+00,-0.85324E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.85447E+00,-0.85571E+00,-0.85694E+00,-0.85817E+00,-0.85939E+00,
     &-0.86062E+00,-0.86185E+00,-0.86307E+00,-0.86429E+00,-0.86551E+00,
     &-0.86673E+00,-0.86795E+00,-0.86916E+00,-0.87038E+00,-0.87159E+00,
     &-0.87280E+00,-0.87401E+00,-0.87522E+00,-0.87642E+00,-0.87763E+00,
     &-0.87883E+00,-0.88003E+00,-0.88123E+00,-0.88243E+00,-0.88363E+00,
     &-0.88482E+00,-0.88602E+00,-0.88721E+00,-0.88840E+00,-0.88959E+00,
     &-0.89078E+00,-0.89197E+00,-0.89315E+00,-0.89434E+00,-0.89552E+00,
     &-0.89670E+00,-0.89788E+00,-0.89906E+00,-0.90024E+00,-0.90142E+00,
     &-0.90259E+00,-0.90376E+00,-0.90494E+00,-0.90611E+00,-0.90728E+00,
     &-0.90845E+00,-0.90961E+00,-0.91078E+00,-0.91194E+00,-0.91310E+00,
     &-0.91427E+00,-0.91543E+00,-0.91659E+00,-0.91774E+00,-0.91890E+00,
     &-0.92006E+00,-0.92121E+00,-0.92236E+00,-0.92351E+00,-0.92466E+00,
     &-0.92581E+00,-0.92696E+00,-0.92811E+00,-0.92925E+00,-0.93040E+00,
     &-0.93154E+00,-0.93268E+00,-0.93382E+00,-0.93496E+00,-0.93610E+00,
     &-0.93724E+00,-0.93837E+00,-0.93951E+00,-0.94064E+00,-0.94177E+00,
     &-0.94290E+00,-0.94404E+00,-0.94516E+00,-0.94629E+00,-0.94742E+00,
     &-0.94854E+00,-0.94967E+00,-0.95079E+00,-0.95191E+00,-0.95303E+00,
     &-0.95416E+00,-0.95527E+00,-0.95639E+00,-0.95751E+00,-0.95862E+00,
     &-0.95974E+00,-0.96085E+00,-0.96196E+00,-0.96308E+00,-0.96419E+00,
     &-0.96530E+00,-0.96640E+00,-0.96751E+00,-0.96862E+00,-0.96972E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.97083E+00,-0.97193E+00,-0.97303E+00,-0.97413E+00,-0.97523E+00,
     &-0.97633E+00,-0.97743E+00,-0.97852E+00,-0.97962E+00,-0.98072E+00,
     &-0.98181E+00,-0.98290E+00,-0.98399E+00,-0.98508E+00,-0.98617E+00,
     &-0.98726E+00,-0.98835E+00,-0.98944E+00,-0.99052E+00,-0.99161E+00,
     &-0.99269E+00,-0.99378E+00,-0.99486E+00,-0.99594E+00,-0.99702E+00,
     &-0.99810E+00,-0.99918E+00,-0.10003E+01,-0.10013E+01,-0.10024E+01,
     &-0.10035E+01,-0.10046E+01,-0.10056E+01,-0.10067E+01,-0.10078E+01,
     &-0.10088E+01,-0.10099E+01,-0.10110E+01,-0.10120E+01,-0.10131E+01,
     &-0.10142E+01,-0.10152E+01,-0.10163E+01,-0.10174E+01,-0.10184E+01,
     &-0.10195E+01,-0.10206E+01,-0.10216E+01,-0.10227E+01,-0.10237E+01,
     &-0.10248E+01,-0.10258E+01,-0.10269E+01,-0.10280E+01,-0.10290E+01,
     &-0.10301E+01,-0.10311E+01,-0.10322E+01,-0.10332E+01,-0.10343E+01,
     &-0.10353E+01,-0.10364E+01,-0.10374E+01,-0.10385E+01,-0.10395E+01,
     &-0.10405E+01,-0.10416E+01,-0.10426E+01,-0.10437E+01,-0.10447E+01,
     &-0.10458E+01,-0.10468E+01,-0.10478E+01,-0.10489E+01,-0.10499E+01,
     &-0.10509E+01,-0.10520E+01,-0.10530E+01,-0.10541E+01,-0.10551E+01,
     &-0.10561E+01,-0.10572E+01,-0.10582E+01,-0.10592E+01,-0.10602E+01,
     &-0.10613E+01,-0.10623E+01,-0.10633E+01,-0.10644E+01,-0.10654E+01,
     &-0.10664E+01,-0.10674E+01,-0.10685E+01,-0.10695E+01,-0.10705E+01,
     &-0.10715E+01,-0.10725E+01,-0.10736E+01,-0.10746E+01,-0.10756E+01/

      DATA (BNC03M(I),I=501,600)/
     &-0.10766E+01,-0.10776E+01,-0.10787E+01,-0.10797E+01,-0.10807E+01,
     &-0.10817E+01,-0.10827E+01,-0.10837E+01,-0.10847E+01,-0.10857E+01,
     &-0.10868E+01,-0.10878E+01,-0.10888E+01,-0.10898E+01,-0.10908E+01,
     &-0.10918E+01,-0.10928E+01,-0.10938E+01,-0.10948E+01,-0.10958E+01,
     &-0.10968E+01,-0.10978E+01,-0.10988E+01,-0.10998E+01,-0.11008E+01,
     &-0.11018E+01,-0.11028E+01,-0.11038E+01,-0.11048E+01,-0.11058E+01,
     &-0.11068E+01,-0.11078E+01,-0.11088E+01,-0.11098E+01,-0.11108E+01,
     &-0.11118E+01,-0.11128E+01,-0.11138E+01,-0.11148E+01,-0.11158E+01,
     &-0.11168E+01,-0.11178E+01,-0.11188E+01,-0.11197E+01,-0.11207E+01,
     &-0.11217E+01,-0.11227E+01,-0.11237E+01,-0.11247E+01,-0.11257E+01,
     &-0.11267E+01,-0.11276E+01,-0.11286E+01,-0.11296E+01,-0.11306E+01,
     &-0.11316E+01,-0.11326E+01,-0.11335E+01,-0.11345E+01,-0.11355E+01,
     &-0.11365E+01,-0.11374E+01,-0.11384E+01,-0.11394E+01,-0.11404E+01,
     &-0.11414E+01,-0.11423E+01,-0.11433E+01,-0.11443E+01,-0.11453E+01,
     &-0.11462E+01,-0.11472E+01,-0.11482E+01,-0.11491E+01,-0.11501E+01,
     &-0.11511E+01,-0.11521E+01,-0.11530E+01,-0.11540E+01,-0.11550E+01,
     &-0.11559E+01,-0.11569E+01,-0.11579E+01,-0.11588E+01,-0.11598E+01,
     &-0.11608E+01,-0.11617E+01,-0.11627E+01,-0.11636E+01,-0.11646E+01,
     &-0.11656E+01,-0.11665E+01,-0.11675E+01,-0.11685E+01,-0.11694E+01,
     &-0.11704E+01,-0.11713E+01,-0.11723E+01,-0.11732E+01,-0.11742E+01/

      DATA (BNC03M(I),I=601,700)/
     &-0.11845E+01,-0.11940E+01,-0.12034E+01,-0.12128E+01,-0.12222E+01,
     &-0.12315E+01,-0.12407E+01,-0.12499E+01,-0.12591E+01,-0.12682E+01,
     &-0.12773E+01,-0.12863E+01,-0.12953E+01,-0.13043E+01,-0.13132E+01,
     &-0.13221E+01,-0.13310E+01,-0.13398E+01,-0.13486E+01,-0.13573E+01,
     &-0.13660E+01,-0.13747E+01,-0.13834E+01,-0.13920E+01,-0.14006E+01,
     &-0.14092E+01,-0.14177E+01,-0.14262E+01,-0.14347E+01,-0.14432E+01,
     &-0.14516E+01,-0.14600E+01,-0.14684E+01,-0.14768E+01,-0.14851E+01,
     &-0.14934E+01,-0.15017E+01,-0.15099E+01,-0.15182E+01,-0.15264E+01,
     &-0.15346E+01,-0.15427E+01,-0.15509E+01,-0.15590E+01,-0.15671E+01,
     &-0.15752E+01,-0.15833E+01,-0.15913E+01,-0.15993E+01,-0.16073E+01,
     &-0.16153E+01,-0.16233E+01,-0.16313E+01,-0.16392E+01,-0.16471E+01,
     &-0.16550E+01,-0.16629E+01,-0.16707E+01,-0.16786E+01,-0.16864E+01,
     &-0.16942E+01,-0.17020E+01,-0.17098E+01,-0.17176E+01,-0.17253E+01,
     &-0.17331E+01,-0.17408E+01,-0.17485E+01,-0.17562E+01,-0.17639E+01,
     &-0.17715E+01,-0.17792E+01,-0.17868E+01,-0.17944E+01,-0.18021E+01,
     &-0.18096E+01,-0.18172E+01,-0.18248E+01,-0.18324E+01,-0.18399E+01,
     &-0.18474E+01,-0.18550E+01,-0.18625E+01,-0.18700E+01,-0.18774E+01,
     &-0.18849E+01,-0.18924E+01,-0.18998E+01,-0.19073E+01,-0.19147E+01,
     &-0.19221E+01,-0.19295E+01,-0.19369E+01,-0.19443E+01,-0.19516E+01,
     &-0.19590E+01,-0.19664E+01,-0.19737E+01,-0.19810E+01,-0.19883E+01/

      DATA (BNC03M(I),I=701,741)/
     &-0.19957E+01,-0.20030E+01,-0.20102E+01,-0.20175E+01,-0.20248E+01,
     &-0.20321E+01,-0.20393E+01,-0.20466E+01,-0.20538E+01,-0.20610E+01,
     &-0.20682E+01,-0.20754E+01,-0.20826E+01,-0.20898E+01,-0.20970E+01,
     &-0.21042E+01,-0.21113E+01,-0.21185E+01,-0.21256E+01,-0.21328E+01,
     &-0.21399E+01,-0.21470E+01,-0.21542E+01,-0.21613E+01,-0.21684E+01,
     &-0.21754E+01,-0.21825E+01,-0.21896E+01,-0.21967E+01,-0.22037E+01,
     &-0.22108E+01,-0.22178E+01,-0.22249E+01,-0.22319E+01,-0.22389E+01,
     &-0.22459E+01,-0.22530E+01,-0.22600E+01,-0.22670E+01,-0.22739E+01,
     &-0.22809E+01
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.10300E+00,-0.22602E+00,-0.28863E+00,-0.33351E+00,-0.36917E+00,
     &-0.39905E+00,-0.42491E+00,-0.44782E+00,-0.46843E+00,-0.48723E+00,
     &-0.50454E+00,-0.52061E+00,-0.53563E+00,-0.54974E+00,-0.56307E+00,
     &-0.57571E+00,-0.58774E+00,-0.59923E+00,-0.61023E+00,-0.62079E+00,
     &-0.63096E+00,-0.64075E+00,-0.65021E+00,-0.65937E+00,-0.66824E+00,
     &-0.67685E+00,-0.68522E+00,-0.69335E+00,-0.70128E+00,-0.70900E+00,
     &-0.71654E+00,-0.72390E+00,-0.73110E+00,-0.73814E+00,-0.74503E+00,
     &-0.75178E+00,-0.75839E+00,-0.76488E+00,-0.77125E+00,-0.77750E+00,
     &-0.78364E+00,-0.78967E+00,-0.79561E+00,-0.80144E+00,-0.80719E+00,
     &-0.81284E+00,-0.81841E+00,-0.82390E+00,-0.82931E+00,-0.83464E+00,
     &-0.83990E+00,-0.84509E+00,-0.85021E+00,-0.85527E+00,-0.86026E+00,
     &-0.86519E+00,-0.87007E+00,-0.87488E+00,-0.87964E+00,-0.88435E+00,
     &-0.88901E+00,-0.89362E+00,-0.89818E+00,-0.90270E+00,-0.90717E+00,
     &-0.91160E+00,-0.91599E+00,-0.92033E+00,-0.92464E+00,-0.92891E+00,
     &-0.93315E+00,-0.93735E+00,-0.94152E+00,-0.94565E+00,-0.94975E+00,
     &-0.95383E+00,-0.95787E+00,-0.96188E+00,-0.96587E+00,-0.96983E+00,
     &-0.97377E+00,-0.97768E+00,-0.98156E+00,-0.98543E+00,-0.98927E+00,
     &-0.99308E+00,-0.99688E+00,-0.10007E+01,-0.10044E+01,-0.10081E+01,
     &-0.10119E+01,-0.10156E+01,-0.10192E+01,-0.10229E+01,-0.10265E+01,
     &-0.10302E+01,-0.10338E+01,-0.10374E+01,-0.10409E+01,-0.10445E+01/

      DATA (BNC04M(I),I=101,200)/
     &-0.10480E+01,-0.10516E+01,-0.10551E+01,-0.10586E+01,-0.10621E+01,
     &-0.10655E+01,-0.10690E+01,-0.10724E+01,-0.10759E+01,-0.10793E+01,
     &-0.10827E+01,-0.10861E+01,-0.10894E+01,-0.10928E+01,-0.10961E+01,
     &-0.10995E+01,-0.11028E+01,-0.11061E+01,-0.11094E+01,-0.11127E+01,
     &-0.11159E+01,-0.11192E+01,-0.11224E+01,-0.11257E+01,-0.11289E+01,
     &-0.11321E+01,-0.11353E+01,-0.11385E+01,-0.11417E+01,-0.11448E+01,
     &-0.11480E+01,-0.11511E+01,-0.11543E+01,-0.11574E+01,-0.11605E+01,
     &-0.11636E+01,-0.11667E+01,-0.11697E+01,-0.11728E+01,-0.11759E+01,
     &-0.11789E+01,-0.11819E+01,-0.11850E+01,-0.11880E+01,-0.11910E+01,
     &-0.11940E+01,-0.11970E+01,-0.12000E+01,-0.12029E+01,-0.12059E+01,
     &-0.12088E+01,-0.12118E+01,-0.12147E+01,-0.12176E+01,-0.12206E+01,
     &-0.12235E+01,-0.12264E+01,-0.12293E+01,-0.12321E+01,-0.12350E+01,
     &-0.12379E+01,-0.12407E+01,-0.12436E+01,-0.12464E+01,-0.12493E+01,
     &-0.12521E+01,-0.12549E+01,-0.12577E+01,-0.12605E+01,-0.12633E+01,
     &-0.12661E+01,-0.12689E+01,-0.12717E+01,-0.12745E+01,-0.12772E+01,
     &-0.12800E+01,-0.12827E+01,-0.12855E+01,-0.12882E+01,-0.12909E+01,
     &-0.12936E+01,-0.12964E+01,-0.12991E+01,-0.13018E+01,-0.13045E+01,
     &-0.13072E+01,-0.13098E+01,-0.13125E+01,-0.13152E+01,-0.13179E+01,
     &-0.13205E+01,-0.13232E+01,-0.13258E+01,-0.13285E+01,-0.13311E+01,
     &-0.13337E+01,-0.13364E+01,-0.13390E+01,-0.13416E+01,-0.13442E+01/

      DATA (BNC04M(I),I=201,300)/
     &-0.13468E+01,-0.13494E+01,-0.13520E+01,-0.13546E+01,-0.13572E+01,
     &-0.13597E+01,-0.13623E+01,-0.13649E+01,-0.13674E+01,-0.13700E+01,
     &-0.13726E+01,-0.13751E+01,-0.13776E+01,-0.13802E+01,-0.13827E+01,
     &-0.13852E+01,-0.13878E+01,-0.13903E+01,-0.13928E+01,-0.13953E+01,
     &-0.13978E+01,-0.14003E+01,-0.14028E+01,-0.14053E+01,-0.14078E+01,
     &-0.14102E+01,-0.14127E+01,-0.14152E+01,-0.14177E+01,-0.14201E+01,
     &-0.14226E+01,-0.14250E+01,-0.14275E+01,-0.14299E+01,-0.14324E+01,
     &-0.14348E+01,-0.14373E+01,-0.14397E+01,-0.14421E+01,-0.14445E+01,
     &-0.14469E+01,-0.14494E+01,-0.14518E+01,-0.14542E+01,-0.14566E+01,
     &-0.14590E+01,-0.14614E+01,-0.14638E+01,-0.14662E+01,-0.14685E+01,
     &-0.14709E+01,-0.14733E+01,-0.14757E+01,-0.14780E+01,-0.14804E+01,
     &-0.14828E+01,-0.14851E+01,-0.14875E+01,-0.14898E+01,-0.14922E+01,
     &-0.14945E+01,-0.14969E+01,-0.14992E+01,-0.15015E+01,-0.15039E+01,
     &-0.15062E+01,-0.15085E+01,-0.15108E+01,-0.15131E+01,-0.15155E+01,
     &-0.15178E+01,-0.15201E+01,-0.15224E+01,-0.15247E+01,-0.15270E+01,
     &-0.15293E+01,-0.15316E+01,-0.15338E+01,-0.15361E+01,-0.15384E+01,
     &-0.15407E+01,-0.15430E+01,-0.15452E+01,-0.15475E+01,-0.15498E+01,
     &-0.15520E+01,-0.15543E+01,-0.15566E+01,-0.15588E+01,-0.15611E+01,
     &-0.15633E+01,-0.15656E+01,-0.15678E+01,-0.15700E+01,-0.15723E+01,
     &-0.15745E+01,-0.15767E+01,-0.15790E+01,-0.15812E+01,-0.15834E+01/

      DATA (BNC04M(I),I=301,400)/
     &-0.15857E+01,-0.15879E+01,-0.15901E+01,-0.15923E+01,-0.15945E+01,
     &-0.15967E+01,-0.15989E+01,-0.16011E+01,-0.16033E+01,-0.16055E+01,
     &-0.16077E+01,-0.16099E+01,-0.16121E+01,-0.16143E+01,-0.16165E+01,
     &-0.16187E+01,-0.16208E+01,-0.16230E+01,-0.16252E+01,-0.16274E+01,
     &-0.16295E+01,-0.16317E+01,-0.16339E+01,-0.16360E+01,-0.16382E+01,
     &-0.16403E+01,-0.16425E+01,-0.16446E+01,-0.16468E+01,-0.16489E+01,
     &-0.16511E+01,-0.16532E+01,-0.16554E+01,-0.16575E+01,-0.16597E+01,
     &-0.16618E+01,-0.16639E+01,-0.16661E+01,-0.16682E+01,-0.16703E+01,
     &-0.16724E+01,-0.16746E+01,-0.16767E+01,-0.16788E+01,-0.16809E+01,
     &-0.16830E+01,-0.16851E+01,-0.16872E+01,-0.16893E+01,-0.16914E+01,
     &-0.16935E+01,-0.16956E+01,-0.16977E+01,-0.16998E+01,-0.17019E+01,
     &-0.17040E+01,-0.17061E+01,-0.17082E+01,-0.17103E+01,-0.17124E+01,
     &-0.17145E+01,-0.17165E+01,-0.17186E+01,-0.17207E+01,-0.17228E+01,
     &-0.17248E+01,-0.17269E+01,-0.17290E+01,-0.17310E+01,-0.17331E+01,
     &-0.17352E+01,-0.17372E+01,-0.17393E+01,-0.17413E+01,-0.17434E+01,
     &-0.17455E+01,-0.17475E+01,-0.17496E+01,-0.17516E+01,-0.17536E+01,
     &-0.17557E+01,-0.17577E+01,-0.17598E+01,-0.17618E+01,-0.17638E+01,
     &-0.17659E+01,-0.17679E+01,-0.17699E+01,-0.17720E+01,-0.17740E+01,
     &-0.17760E+01,-0.17781E+01,-0.17801E+01,-0.17821E+01,-0.17841E+01,
     &-0.17861E+01,-0.17881E+01,-0.17902E+01,-0.17922E+01,-0.17942E+01/

      DATA (BNC04M(I),I=401,500)/
     &-0.17962E+01,-0.17982E+01,-0.18002E+01,-0.18022E+01,-0.18042E+01,
     &-0.18062E+01,-0.18082E+01,-0.18102E+01,-0.18122E+01,-0.18142E+01,
     &-0.18162E+01,-0.18182E+01,-0.18202E+01,-0.18222E+01,-0.18241E+01,
     &-0.18261E+01,-0.18281E+01,-0.18301E+01,-0.18321E+01,-0.18341E+01,
     &-0.18360E+01,-0.18380E+01,-0.18400E+01,-0.18420E+01,-0.18439E+01,
     &-0.18459E+01,-0.18479E+01,-0.18498E+01,-0.18518E+01,-0.18538E+01,
     &-0.18557E+01,-0.18577E+01,-0.18596E+01,-0.18616E+01,-0.18636E+01,
     &-0.18655E+01,-0.18675E+01,-0.18694E+01,-0.18714E+01,-0.18733E+01,
     &-0.18753E+01,-0.18772E+01,-0.18792E+01,-0.18811E+01,-0.18830E+01,
     &-0.18850E+01,-0.18869E+01,-0.18889E+01,-0.18908E+01,-0.18927E+01,
     &-0.18947E+01,-0.18966E+01,-0.18985E+01,-0.19005E+01,-0.19024E+01,
     &-0.19043E+01,-0.19062E+01,-0.19082E+01,-0.19101E+01,-0.19120E+01,
     &-0.19139E+01,-0.19158E+01,-0.19178E+01,-0.19197E+01,-0.19216E+01,
     &-0.19235E+01,-0.19254E+01,-0.19273E+01,-0.19292E+01,-0.19311E+01,
     &-0.19331E+01,-0.19350E+01,-0.19369E+01,-0.19388E+01,-0.19407E+01,
     &-0.19426E+01,-0.19445E+01,-0.19464E+01,-0.19483E+01,-0.19502E+01,
     &-0.19521E+01,-0.19539E+01,-0.19558E+01,-0.19577E+01,-0.19596E+01,
     &-0.19615E+01,-0.19634E+01,-0.19653E+01,-0.19672E+01,-0.19691E+01,
     &-0.19709E+01,-0.19728E+01,-0.19747E+01,-0.19766E+01,-0.19785E+01,
     &-0.19803E+01,-0.19822E+01,-0.19841E+01,-0.19860E+01,-0.19878E+01/

      DATA (BNC04M(I),I=501,600)/
     &-0.19897E+01,-0.19916E+01,-0.19934E+01,-0.19953E+01,-0.19972E+01,
     &-0.19990E+01,-0.20009E+01,-0.20028E+01,-0.20046E+01,-0.20065E+01,
     &-0.20084E+01,-0.20102E+01,-0.20121E+01,-0.20139E+01,-0.20158E+01,
     &-0.20176E+01,-0.20195E+01,-0.20213E+01,-0.20232E+01,-0.20250E+01,
     &-0.20269E+01,-0.20287E+01,-0.20306E+01,-0.20324E+01,-0.20343E+01,
     &-0.20361E+01,-0.20380E+01,-0.20398E+01,-0.20416E+01,-0.20435E+01,
     &-0.20453E+01,-0.20472E+01,-0.20490E+01,-0.20508E+01,-0.20527E+01,
     &-0.20545E+01,-0.20563E+01,-0.20582E+01,-0.20600E+01,-0.20618E+01,
     &-0.20637E+01,-0.20655E+01,-0.20673E+01,-0.20691E+01,-0.20710E+01,
     &-0.20728E+01,-0.20746E+01,-0.20764E+01,-0.20782E+01,-0.20801E+01,
     &-0.20819E+01,-0.20837E+01,-0.20855E+01,-0.20873E+01,-0.20891E+01,
     &-0.20910E+01,-0.20928E+01,-0.20946E+01,-0.20964E+01,-0.20982E+01,
     &-0.21000E+01,-0.21018E+01,-0.21036E+01,-0.21054E+01,-0.21072E+01,
     &-0.21090E+01,-0.21109E+01,-0.21127E+01,-0.21145E+01,-0.21163E+01,
     &-0.21181E+01,-0.21199E+01,-0.21217E+01,-0.21234E+01,-0.21252E+01,
     &-0.21270E+01,-0.21288E+01,-0.21306E+01,-0.21324E+01,-0.21342E+01,
     &-0.21360E+01,-0.21378E+01,-0.21396E+01,-0.21414E+01,-0.21432E+01,
     &-0.21449E+01,-0.21467E+01,-0.21485E+01,-0.21503E+01,-0.21521E+01,
     &-0.21539E+01,-0.21556E+01,-0.21574E+01,-0.21592E+01,-0.21610E+01,
     &-0.21628E+01,-0.21645E+01,-0.21663E+01,-0.21681E+01,-0.21699E+01/

      DATA (BNC04M(I),I=601,700)/
     &-0.21890E+01,-0.22066E+01,-0.22241E+01,-0.22416E+01,-0.22589E+01,
     &-0.22762E+01,-0.22935E+01,-0.23106E+01,-0.23277E+01,-0.23448E+01,
     &-0.23617E+01,-0.23786E+01,-0.23955E+01,-0.24122E+01,-0.24290E+01,
     &-0.24456E+01,-0.24622E+01,-0.24788E+01,-0.24953E+01,-0.25117E+01,
     &-0.25281E+01,-0.25445E+01,-0.25608E+01,-0.25770E+01,-0.25932E+01,
     &-0.26094E+01,-0.26255E+01,-0.26415E+01,-0.26575E+01,-0.26735E+01,
     &-0.26894E+01,-0.27053E+01,-0.27212E+01,-0.27370E+01,-0.27528E+01,
     &-0.27685E+01,-0.27842E+01,-0.27998E+01,-0.28155E+01,-0.28310E+01,
     &-0.28466E+01,-0.28621E+01,-0.28776E+01,-0.28930E+01,-0.29084E+01,
     &-0.29238E+01,-0.29392E+01,-0.29545E+01,-0.29698E+01,-0.29850E+01,
     &-0.30002E+01,-0.30154E+01,-0.30306E+01,-0.30457E+01,-0.30608E+01,
     &-0.30759E+01,-0.30910E+01,-0.31060E+01,-0.31210E+01,-0.31360E+01,
     &-0.31509E+01,-0.31658E+01,-0.31807E+01,-0.31956E+01,-0.32104E+01,
     &-0.32252E+01,-0.32400E+01,-0.32548E+01,-0.32696E+01,-0.32843E+01,
     &-0.32990E+01,-0.33137E+01,-0.33283E+01,-0.33430E+01,-0.33576E+01,
     &-0.33722E+01,-0.33867E+01,-0.34013E+01,-0.34158E+01,-0.34303E+01,
     &-0.34448E+01,-0.34593E+01,-0.34737E+01,-0.34882E+01,-0.35026E+01,
     &-0.35170E+01,-0.35313E+01,-0.35457E+01,-0.35600E+01,-0.35743E+01,
     &-0.35886E+01,-0.36029E+01,-0.36172E+01,-0.36314E+01,-0.36456E+01,
     &-0.36599E+01,-0.36741E+01,-0.36882E+01,-0.37024E+01,-0.37165E+01/

      DATA (BNC04M(I),I=701,741)/
     &-0.37307E+01,-0.37448E+01,-0.37589E+01,-0.37730E+01,-0.37870E+01,
     &-0.38011E+01,-0.38151E+01,-0.38291E+01,-0.38431E+01,-0.38571E+01,
     &-0.38711E+01,-0.38850E+01,-0.38990E+01,-0.39129E+01,-0.39268E+01,
     &-0.39407E+01,-0.39546E+01,-0.39685E+01,-0.39824E+01,-0.39962E+01,
     &-0.40100E+01,-0.40239E+01,-0.40377E+01,-0.40515E+01,-0.40653E+01,
     &-0.40790E+01,-0.40928E+01,-0.41065E+01,-0.41203E+01,-0.41340E+01,
     &-0.41477E+01,-0.41614E+01,-0.41751E+01,-0.41887E+01,-0.42024E+01,
     &-0.42160E+01,-0.42297E+01,-0.42433E+01,-0.42569E+01,-0.42705E+01,
     &-0.42841E+01
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.52307E-01,-0.11745E+00,-0.15210E+00,-0.17773E+00,-0.19864E+00,
     &-0.21658E+00,-0.23244E+00,-0.24676E+00,-0.25988E+00,-0.27205E+00,
     &-0.28343E+00,-0.29415E+00,-0.30431E+00,-0.31398E+00,-0.32322E+00,
     &-0.33209E+00,-0.34063E+00,-0.34886E+00,-0.35683E+00,-0.36454E+00,
     &-0.37203E+00,-0.37931E+00,-0.38640E+00,-0.39330E+00,-0.40004E+00,
     &-0.40662E+00,-0.41306E+00,-0.41936E+00,-0.42552E+00,-0.43157E+00,
     &-0.43749E+00,-0.44330E+00,-0.44901E+00,-0.45461E+00,-0.46012E+00,
     &-0.46553E+00,-0.47085E+00,-0.47609E+00,-0.48124E+00,-0.48631E+00,
     &-0.49130E+00,-0.49622E+00,-0.50107E+00,-0.50584E+00,-0.51055E+00,
     &-0.51519E+00,-0.51977E+00,-0.52429E+00,-0.52875E+00,-0.53315E+00,
     &-0.53750E+00,-0.54179E+00,-0.54603E+00,-0.55022E+00,-0.55436E+00,
     &-0.55846E+00,-0.56250E+00,-0.56651E+00,-0.57047E+00,-0.57439E+00,
     &-0.57828E+00,-0.58212E+00,-0.58593E+00,-0.58970E+00,-0.59344E+00,
     &-0.59715E+00,-0.60083E+00,-0.60448E+00,-0.60810E+00,-0.61169E+00,
     &-0.61526E+00,-0.61880E+00,-0.62232E+00,-0.62582E+00,-0.62930E+00,
     &-0.63275E+00,-0.63619E+00,-0.63960E+00,-0.64300E+00,-0.64639E+00,
     &-0.64976E+00,-0.65311E+00,-0.65645E+00,-0.65977E+00,-0.66308E+00,
     &-0.66638E+00,-0.66966E+00,-0.67294E+00,-0.67620E+00,-0.67945E+00,
     &-0.68269E+00,-0.68592E+00,-0.68914E+00,-0.69236E+00,-0.69556E+00,
     &-0.69875E+00,-0.70193E+00,-0.70511E+00,-0.70827E+00,-0.71143E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.71457E+00,-0.71771E+00,-0.72084E+00,-0.72396E+00,-0.72707E+00,
     &-0.73017E+00,-0.73326E+00,-0.73635E+00,-0.73942E+00,-0.74248E+00,
     &-0.74554E+00,-0.74859E+00,-0.75162E+00,-0.75465E+00,-0.75766E+00,
     &-0.76067E+00,-0.76367E+00,-0.76665E+00,-0.76963E+00,-0.77260E+00,
     &-0.77555E+00,-0.77850E+00,-0.78143E+00,-0.78436E+00,-0.78727E+00,
     &-0.79018E+00,-0.79307E+00,-0.79596E+00,-0.79883E+00,-0.80169E+00,
     &-0.80455E+00,-0.80739E+00,-0.81022E+00,-0.81304E+00,-0.81586E+00,
     &-0.81866E+00,-0.82145E+00,-0.82423E+00,-0.82700E+00,-0.82976E+00,
     &-0.83252E+00,-0.83526E+00,-0.83799E+00,-0.84071E+00,-0.84343E+00,
     &-0.84613E+00,-0.84882E+00,-0.85151E+00,-0.85418E+00,-0.85685E+00,
     &-0.85950E+00,-0.86215E+00,-0.86479E+00,-0.86742E+00,-0.87004E+00,
     &-0.87265E+00,-0.87525E+00,-0.87785E+00,-0.88043E+00,-0.88301E+00,
     &-0.88558E+00,-0.88814E+00,-0.89069E+00,-0.89323E+00,-0.89577E+00,
     &-0.89829E+00,-0.90081E+00,-0.90332E+00,-0.90583E+00,-0.90832E+00,
     &-0.91081E+00,-0.91329E+00,-0.91576E+00,-0.91822E+00,-0.92068E+00,
     &-0.92313E+00,-0.92557E+00,-0.92800E+00,-0.93043E+00,-0.93284E+00,
     &-0.93526E+00,-0.93766E+00,-0.94006E+00,-0.94245E+00,-0.94483E+00,
     &-0.94721E+00,-0.94958E+00,-0.95194E+00,-0.95429E+00,-0.95664E+00,
     &-0.95898E+00,-0.96132E+00,-0.96365E+00,-0.96597E+00,-0.96828E+00,
     &-0.97059E+00,-0.97290E+00,-0.97519E+00,-0.97748E+00,-0.97976E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.98204E+00,-0.98431E+00,-0.98657E+00,-0.98883E+00,-0.99108E+00,
     &-0.99333E+00,-0.99557E+00,-0.99780E+00,-0.10000E+01,-0.10023E+01,
     &-0.10045E+01,-0.10067E+01,-0.10089E+01,-0.10111E+01,-0.10133E+01,
     &-0.10155E+01,-0.10176E+01,-0.10198E+01,-0.10220E+01,-0.10242E+01,
     &-0.10263E+01,-0.10285E+01,-0.10306E+01,-0.10328E+01,-0.10349E+01,
     &-0.10370E+01,-0.10392E+01,-0.10413E+01,-0.10434E+01,-0.10455E+01,
     &-0.10476E+01,-0.10497E+01,-0.10518E+01,-0.10539E+01,-0.10560E+01,
     &-0.10581E+01,-0.10601E+01,-0.10622E+01,-0.10643E+01,-0.10663E+01,
     &-0.10684E+01,-0.10704E+01,-0.10725E+01,-0.10745E+01,-0.10766E+01,
     &-0.10786E+01,-0.10806E+01,-0.10826E+01,-0.10847E+01,-0.10867E+01,
     &-0.10887E+01,-0.10907E+01,-0.10927E+01,-0.10947E+01,-0.10967E+01,
     &-0.10987E+01,-0.11006E+01,-0.11026E+01,-0.11046E+01,-0.11066E+01,
     &-0.11085E+01,-0.11105E+01,-0.11124E+01,-0.11144E+01,-0.11163E+01,
     &-0.11183E+01,-0.11202E+01,-0.11221E+01,-0.11241E+01,-0.11260E+01,
     &-0.11279E+01,-0.11298E+01,-0.11317E+01,-0.11337E+01,-0.11356E+01,
     &-0.11375E+01,-0.11394E+01,-0.11413E+01,-0.11431E+01,-0.11450E+01,
     &-0.11469E+01,-0.11488E+01,-0.11507E+01,-0.11525E+01,-0.11544E+01,
     &-0.11563E+01,-0.11581E+01,-0.11600E+01,-0.11618E+01,-0.11637E+01,
     &-0.11655E+01,-0.11673E+01,-0.11692E+01,-0.11710E+01,-0.11728E+01,
     &-0.11747E+01,-0.11765E+01,-0.11783E+01,-0.11801E+01,-0.11819E+01/

      DATA (BNC05M(I),I=301,400)/
     &-0.11837E+01,-0.11855E+01,-0.11873E+01,-0.11891E+01,-0.11909E+01,
     &-0.11927E+01,-0.11945E+01,-0.11963E+01,-0.11981E+01,-0.11998E+01,
     &-0.12016E+01,-0.12034E+01,-0.12051E+01,-0.12069E+01,-0.12087E+01,
     &-0.12104E+01,-0.12122E+01,-0.12139E+01,-0.12157E+01,-0.12174E+01,
     &-0.12192E+01,-0.12209E+01,-0.12226E+01,-0.12244E+01,-0.12261E+01,
     &-0.12278E+01,-0.12295E+01,-0.12312E+01,-0.12330E+01,-0.12347E+01,
     &-0.12364E+01,-0.12381E+01,-0.12398E+01,-0.12415E+01,-0.12432E+01,
     &-0.12449E+01,-0.12466E+01,-0.12482E+01,-0.12499E+01,-0.12516E+01,
     &-0.12533E+01,-0.12550E+01,-0.12566E+01,-0.12583E+01,-0.12600E+01,
     &-0.12616E+01,-0.12633E+01,-0.12650E+01,-0.12666E+01,-0.12683E+01,
     &-0.12699E+01,-0.12716E+01,-0.12732E+01,-0.12748E+01,-0.12765E+01,
     &-0.12781E+01,-0.12798E+01,-0.12814E+01,-0.12830E+01,-0.12846E+01,
     &-0.12863E+01,-0.12879E+01,-0.12895E+01,-0.12911E+01,-0.12927E+01,
     &-0.12943E+01,-0.12959E+01,-0.12975E+01,-0.12991E+01,-0.13007E+01,
     &-0.13023E+01,-0.13039E+01,-0.13055E+01,-0.13071E+01,-0.13087E+01,
     &-0.13103E+01,-0.13119E+01,-0.13134E+01,-0.13150E+01,-0.13166E+01,
     &-0.13182E+01,-0.13197E+01,-0.13213E+01,-0.13229E+01,-0.13244E+01,
     &-0.13260E+01,-0.13275E+01,-0.13291E+01,-0.13306E+01,-0.13322E+01,
     &-0.13337E+01,-0.13353E+01,-0.13368E+01,-0.13384E+01,-0.13399E+01,
     &-0.13414E+01,-0.13430E+01,-0.13445E+01,-0.13460E+01,-0.13476E+01/

      DATA (BNC05M(I),I=401,500)/
     &-0.13491E+01,-0.13506E+01,-0.13521E+01,-0.13536E+01,-0.13552E+01,
     &-0.13567E+01,-0.13582E+01,-0.13597E+01,-0.13612E+01,-0.13627E+01,
     &-0.13642E+01,-0.13657E+01,-0.13672E+01,-0.13687E+01,-0.13702E+01,
     &-0.13717E+01,-0.13732E+01,-0.13747E+01,-0.13761E+01,-0.13776E+01,
     &-0.13791E+01,-0.13806E+01,-0.13821E+01,-0.13835E+01,-0.13850E+01,
     &-0.13865E+01,-0.13879E+01,-0.13894E+01,-0.13909E+01,-0.13923E+01,
     &-0.13938E+01,-0.13953E+01,-0.13967E+01,-0.13982E+01,-0.13996E+01,
     &-0.14011E+01,-0.14025E+01,-0.14040E+01,-0.14054E+01,-0.14068E+01,
     &-0.14083E+01,-0.14097E+01,-0.14112E+01,-0.14126E+01,-0.14140E+01,
     &-0.14155E+01,-0.14169E+01,-0.14183E+01,-0.14197E+01,-0.14212E+01,
     &-0.14226E+01,-0.14240E+01,-0.14254E+01,-0.14268E+01,-0.14283E+01,
     &-0.14297E+01,-0.14311E+01,-0.14325E+01,-0.14339E+01,-0.14353E+01,
     &-0.14367E+01,-0.14381E+01,-0.14395E+01,-0.14409E+01,-0.14423E+01,
     &-0.14437E+01,-0.14451E+01,-0.14465E+01,-0.14479E+01,-0.14492E+01,
     &-0.14506E+01,-0.14520E+01,-0.14534E+01,-0.14548E+01,-0.14562E+01,
     &-0.14575E+01,-0.14589E+01,-0.14603E+01,-0.14616E+01,-0.14630E+01,
     &-0.14644E+01,-0.14658E+01,-0.14671E+01,-0.14685E+01,-0.14698E+01,
     &-0.14712E+01,-0.14726E+01,-0.14739E+01,-0.14753E+01,-0.14766E+01,
     &-0.14780E+01,-0.14793E+01,-0.14807E+01,-0.14820E+01,-0.14834E+01,
     &-0.14847E+01,-0.14861E+01,-0.14874E+01,-0.14887E+01,-0.14901E+01/

      DATA (BNC05M(I),I=501,600)/
     &-0.14914E+01,-0.14927E+01,-0.14941E+01,-0.14954E+01,-0.14967E+01,
     &-0.14981E+01,-0.14994E+01,-0.15007E+01,-0.15020E+01,-0.15034E+01,
     &-0.15047E+01,-0.15060E+01,-0.15073E+01,-0.15086E+01,-0.15099E+01,
     &-0.15113E+01,-0.15126E+01,-0.15139E+01,-0.15152E+01,-0.15165E+01,
     &-0.15178E+01,-0.15191E+01,-0.15204E+01,-0.15217E+01,-0.15230E+01,
     &-0.15243E+01,-0.15256E+01,-0.15269E+01,-0.15282E+01,-0.15295E+01,
     &-0.15308E+01,-0.15321E+01,-0.15334E+01,-0.15346E+01,-0.15359E+01,
     &-0.15372E+01,-0.15385E+01,-0.15398E+01,-0.15411E+01,-0.15423E+01,
     &-0.15436E+01,-0.15449E+01,-0.15462E+01,-0.15474E+01,-0.15487E+01,
     &-0.15500E+01,-0.15512E+01,-0.15525E+01,-0.15538E+01,-0.15550E+01,
     &-0.15563E+01,-0.15576E+01,-0.15588E+01,-0.15601E+01,-0.15613E+01,
     &-0.15626E+01,-0.15639E+01,-0.15651E+01,-0.15664E+01,-0.15676E+01,
     &-0.15689E+01,-0.15701E+01,-0.15714E+01,-0.15726E+01,-0.15739E+01,
     &-0.15751E+01,-0.15764E+01,-0.15776E+01,-0.15788E+01,-0.15801E+01,
     &-0.15813E+01,-0.15825E+01,-0.15838E+01,-0.15850E+01,-0.15862E+01,
     &-0.15875E+01,-0.15887E+01,-0.15899E+01,-0.15912E+01,-0.15924E+01,
     &-0.15936E+01,-0.15948E+01,-0.15961E+01,-0.15973E+01,-0.15985E+01,
     &-0.15997E+01,-0.16010E+01,-0.16022E+01,-0.16034E+01,-0.16046E+01,
     &-0.16058E+01,-0.16070E+01,-0.16082E+01,-0.16094E+01,-0.16107E+01,
     &-0.16119E+01,-0.16131E+01,-0.16143E+01,-0.16155E+01,-0.16167E+01/

      DATA (BNC05M(I),I=601,700)/
     &-0.16296E+01,-0.16415E+01,-0.16532E+01,-0.16649E+01,-0.16764E+01,
     &-0.16878E+01,-0.16992E+01,-0.17104E+01,-0.17216E+01,-0.17327E+01,
     &-0.17437E+01,-0.17546E+01,-0.17654E+01,-0.17761E+01,-0.17868E+01,
     &-0.17974E+01,-0.18079E+01,-0.18184E+01,-0.18287E+01,-0.18391E+01,
     &-0.18493E+01,-0.18595E+01,-0.18696E+01,-0.18796E+01,-0.18896E+01,
     &-0.18996E+01,-0.19094E+01,-0.19193E+01,-0.19290E+01,-0.19387E+01,
     &-0.19484E+01,-0.19580E+01,-0.19675E+01,-0.19770E+01,-0.19865E+01,
     &-0.19959E+01,-0.20053E+01,-0.20146E+01,-0.20238E+01,-0.20331E+01,
     &-0.20423E+01,-0.20514E+01,-0.20605E+01,-0.20695E+01,-0.20786E+01,
     &-0.20875E+01,-0.20965E+01,-0.21054E+01,-0.21143E+01,-0.21231E+01,
     &-0.21319E+01,-0.21407E+01,-0.21494E+01,-0.21581E+01,-0.21667E+01,
     &-0.21754E+01,-0.21840E+01,-0.21925E+01,-0.22011E+01,-0.22096E+01,
     &-0.22180E+01,-0.22265E+01,-0.22349E+01,-0.22433E+01,-0.22517E+01,
     &-0.22600E+01,-0.22683E+01,-0.22766E+01,-0.22848E+01,-0.22931E+01,
     &-0.23013E+01,-0.23095E+01,-0.23176E+01,-0.23257E+01,-0.23338E+01,
     &-0.23419E+01,-0.23500E+01,-0.23580E+01,-0.23661E+01,-0.23741E+01,
     &-0.23820E+01,-0.23900E+01,-0.23979E+01,-0.24058E+01,-0.24137E+01,
     &-0.24216E+01,-0.24294E+01,-0.24373E+01,-0.24451E+01,-0.24529E+01,
     &-0.24607E+01,-0.24684E+01,-0.24762E+01,-0.24839E+01,-0.24916E+01,
     &-0.24993E+01,-0.25070E+01,-0.25146E+01,-0.25223E+01,-0.25299E+01/

      DATA (BNC05M(I),I=701,741)/
     &-0.25375E+01,-0.25451E+01,-0.25526E+01,-0.25602E+01,-0.25677E+01,
     &-0.25753E+01,-0.25828E+01,-0.25903E+01,-0.25978E+01,-0.26052E+01,
     &-0.26127E+01,-0.26201E+01,-0.26276E+01,-0.26350E+01,-0.26424E+01,
     &-0.26498E+01,-0.26571E+01,-0.26645E+01,-0.26718E+01,-0.26792E+01,
     &-0.26865E+01,-0.26938E+01,-0.27011E+01,-0.27084E+01,-0.27157E+01,
     &-0.27229E+01,-0.27302E+01,-0.27374E+01,-0.27446E+01,-0.27519E+01,
     &-0.27591E+01,-0.27663E+01,-0.27734E+01,-0.27806E+01,-0.27878E+01,
     &-0.27949E+01,-0.28021E+01,-0.28092E+01,-0.28163E+01,-0.28234E+01,
     &-0.28305E+01
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.50630E-01,-0.10830E+00,-0.13610E+00,-0.15521E+00,-0.16983E+00,
     &-0.18165E+00,-0.19155E+00,-0.20002E+00,-0.20740E+00,-0.21392E+00,
     &-0.21974E+00,-0.22498E+00,-0.22973E+00,-0.23406E+00,-0.23803E+00,
     &-0.24168E+00,-0.24506E+00,-0.24819E+00,-0.25111E+00,-0.25383E+00,
     &-0.25638E+00,-0.25876E+00,-0.26101E+00,-0.26312E+00,-0.26512E+00,
     &-0.26700E+00,-0.26879E+00,-0.27049E+00,-0.27210E+00,-0.27363E+00,
     &-0.27509E+00,-0.27649E+00,-0.27782E+00,-0.27910E+00,-0.28032E+00,
     &-0.28149E+00,-0.28262E+00,-0.28371E+00,-0.28475E+00,-0.28576E+00,
     &-0.28673E+00,-0.28767E+00,-0.28858E+00,-0.28946E+00,-0.29032E+00,
     &-0.29115E+00,-0.29195E+00,-0.29273E+00,-0.29350E+00,-0.29424E+00,
     &-0.29496E+00,-0.29566E+00,-0.29635E+00,-0.29702E+00,-0.29767E+00,
     &-0.29831E+00,-0.29894E+00,-0.29955E+00,-0.30014E+00,-0.30073E+00,
     &-0.30130E+00,-0.30185E+00,-0.30240E+00,-0.30293E+00,-0.30345E+00,
     &-0.30396E+00,-0.30445E+00,-0.30494E+00,-0.30541E+00,-0.30587E+00,
     &-0.30632E+00,-0.30675E+00,-0.30718E+00,-0.30759E+00,-0.30799E+00,
     &-0.30838E+00,-0.30876E+00,-0.30912E+00,-0.30948E+00,-0.30982E+00,
     &-0.31015E+00,-0.31046E+00,-0.31077E+00,-0.31106E+00,-0.31134E+00,
     &-0.31161E+00,-0.31187E+00,-0.31212E+00,-0.31235E+00,-0.31257E+00,
     &-0.31279E+00,-0.31299E+00,-0.31318E+00,-0.31336E+00,-0.31352E+00,
     &-0.31368E+00,-0.31383E+00,-0.31397E+00,-0.31410E+00,-0.31422E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.31433E+00,-0.31443E+00,-0.31452E+00,-0.31461E+00,-0.31469E+00,
     &-0.31475E+00,-0.31482E+00,-0.31487E+00,-0.31492E+00,-0.31496E+00,
     &-0.31499E+00,-0.31502E+00,-0.31505E+00,-0.31506E+00,-0.31508E+00,
     &-0.31508E+00,-0.31509E+00,-0.31509E+00,-0.31508E+00,-0.31507E+00,
     &-0.31506E+00,-0.31504E+00,-0.31502E+00,-0.31500E+00,-0.31497E+00,
     &-0.31494E+00,-0.31491E+00,-0.31487E+00,-0.31483E+00,-0.31479E+00,
     &-0.31475E+00,-0.31471E+00,-0.31466E+00,-0.31461E+00,-0.31456E+00,
     &-0.31451E+00,-0.31446E+00,-0.31441E+00,-0.31435E+00,-0.31430E+00,
     &-0.31424E+00,-0.31418E+00,-0.31412E+00,-0.31406E+00,-0.31400E+00,
     &-0.31393E+00,-0.31387E+00,-0.31381E+00,-0.31374E+00,-0.31367E+00,
     &-0.31361E+00,-0.31354E+00,-0.31347E+00,-0.31340E+00,-0.31334E+00,
     &-0.31327E+00,-0.31320E+00,-0.31313E+00,-0.31305E+00,-0.31298E+00,
     &-0.31291E+00,-0.31284E+00,-0.31277E+00,-0.31270E+00,-0.31262E+00,
     &-0.31255E+00,-0.31248E+00,-0.31240E+00,-0.31233E+00,-0.31226E+00,
     &-0.31218E+00,-0.31211E+00,-0.31204E+00,-0.31196E+00,-0.31189E+00,
     &-0.31182E+00,-0.31174E+00,-0.31167E+00,-0.31159E+00,-0.31152E+00,
     &-0.31145E+00,-0.31137E+00,-0.31130E+00,-0.31123E+00,-0.31115E+00,
     &-0.31108E+00,-0.31101E+00,-0.31094E+00,-0.31086E+00,-0.31079E+00,
     &-0.31072E+00,-0.31065E+00,-0.31058E+00,-0.31050E+00,-0.31043E+00,
     &-0.31036E+00,-0.31029E+00,-0.31022E+00,-0.31015E+00,-0.31008E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.31001E+00,-0.30994E+00,-0.30988E+00,-0.30981E+00,-0.30974E+00,
     &-0.30967E+00,-0.30960E+00,-0.30954E+00,-0.30947E+00,-0.30941E+00,
     &-0.30934E+00,-0.30928E+00,-0.30921E+00,-0.30915E+00,-0.30908E+00,
     &-0.30902E+00,-0.30896E+00,-0.30889E+00,-0.30883E+00,-0.30877E+00,
     &-0.30871E+00,-0.30865E+00,-0.30859E+00,-0.30853E+00,-0.30847E+00,
     &-0.30841E+00,-0.30835E+00,-0.30830E+00,-0.30824E+00,-0.30818E+00,
     &-0.30813E+00,-0.30807E+00,-0.30802E+00,-0.30796E+00,-0.30791E+00,
     &-0.30786E+00,-0.30780E+00,-0.30775E+00,-0.30770E+00,-0.30765E+00,
     &-0.30760E+00,-0.30755E+00,-0.30750E+00,-0.30745E+00,-0.30740E+00,
     &-0.30736E+00,-0.30731E+00,-0.30726E+00,-0.30722E+00,-0.30717E+00,
     &-0.30713E+00,-0.30708E+00,-0.30704E+00,-0.30700E+00,-0.30696E+00,
     &-0.30692E+00,-0.30687E+00,-0.30683E+00,-0.30680E+00,-0.30676E+00,
     &-0.30672E+00,-0.30668E+00,-0.30664E+00,-0.30661E+00,-0.30657E+00,
     &-0.30654E+00,-0.30650E+00,-0.30647E+00,-0.30644E+00,-0.30640E+00,
     &-0.30637E+00,-0.30634E+00,-0.30631E+00,-0.30628E+00,-0.30625E+00,
     &-0.30622E+00,-0.30620E+00,-0.30617E+00,-0.30614E+00,-0.30612E+00,
     &-0.30609E+00,-0.30607E+00,-0.30604E+00,-0.30602E+00,-0.30600E+00,
     &-0.30598E+00,-0.30595E+00,-0.30593E+00,-0.30591E+00,-0.30590E+00,
     &-0.30588E+00,-0.30586E+00,-0.30584E+00,-0.30583E+00,-0.30581E+00,
     &-0.30579E+00,-0.30578E+00,-0.30577E+00,-0.30575E+00,-0.30574E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.30573E+00,-0.30572E+00,-0.30571E+00,-0.30570E+00,-0.30569E+00,
     &-0.30568E+00,-0.30567E+00,-0.30567E+00,-0.30566E+00,-0.30565E+00,
     &-0.30565E+00,-0.30564E+00,-0.30564E+00,-0.30564E+00,-0.30563E+00,
     &-0.30563E+00,-0.30563E+00,-0.30563E+00,-0.30563E+00,-0.30563E+00,
     &-0.30564E+00,-0.30564E+00,-0.30564E+00,-0.30564E+00,-0.30565E+00,
     &-0.30565E+00,-0.30566E+00,-0.30567E+00,-0.30567E+00,-0.30568E+00,
     &-0.30569E+00,-0.30570E+00,-0.30571E+00,-0.30572E+00,-0.30573E+00,
     &-0.30574E+00,-0.30575E+00,-0.30577E+00,-0.30578E+00,-0.30580E+00,
     &-0.30581E+00,-0.30583E+00,-0.30584E+00,-0.30586E+00,-0.30588E+00,
     &-0.30590E+00,-0.30592E+00,-0.30594E+00,-0.30596E+00,-0.30598E+00,
     &-0.30600E+00,-0.30602E+00,-0.30605E+00,-0.30607E+00,-0.30609E+00,
     &-0.30612E+00,-0.30614E+00,-0.30617E+00,-0.30620E+00,-0.30623E+00,
     &-0.30625E+00,-0.30628E+00,-0.30631E+00,-0.30634E+00,-0.30637E+00,
     &-0.30641E+00,-0.30644E+00,-0.30647E+00,-0.30650E+00,-0.30654E+00,
     &-0.30657E+00,-0.30661E+00,-0.30665E+00,-0.30668E+00,-0.30672E+00,
     &-0.30676E+00,-0.30680E+00,-0.30684E+00,-0.30688E+00,-0.30692E+00,
     &-0.30696E+00,-0.30700E+00,-0.30704E+00,-0.30709E+00,-0.30713E+00,
     &-0.30717E+00,-0.30722E+00,-0.30726E+00,-0.30731E+00,-0.30736E+00,
     &-0.30741E+00,-0.30745E+00,-0.30750E+00,-0.30755E+00,-0.30760E+00,
     &-0.30765E+00,-0.30770E+00,-0.30776E+00,-0.30781E+00,-0.30786E+00/

      DATA (BNC06M(I),I=401,500)/
     &-0.30792E+00,-0.30797E+00,-0.30803E+00,-0.30808E+00,-0.30814E+00,
     &-0.30820E+00,-0.30825E+00,-0.30831E+00,-0.30837E+00,-0.30843E+00,
     &-0.30849E+00,-0.30855E+00,-0.30861E+00,-0.30867E+00,-0.30874E+00,
     &-0.30880E+00,-0.30886E+00,-0.30893E+00,-0.30899E+00,-0.30906E+00,
     &-0.30912E+00,-0.30919E+00,-0.30926E+00,-0.30932E+00,-0.30939E+00,
     &-0.30946E+00,-0.30953E+00,-0.30960E+00,-0.30967E+00,-0.30974E+00,
     &-0.30982E+00,-0.30989E+00,-0.30996E+00,-0.31004E+00,-0.31011E+00,
     &-0.31019E+00,-0.31026E+00,-0.31034E+00,-0.31041E+00,-0.31049E+00,
     &-0.31057E+00,-0.31065E+00,-0.31073E+00,-0.31081E+00,-0.31089E+00,
     &-0.31097E+00,-0.31105E+00,-0.31113E+00,-0.31121E+00,-0.31130E+00,
     &-0.31138E+00,-0.31146E+00,-0.31155E+00,-0.31163E+00,-0.31172E+00,
     &-0.31181E+00,-0.31189E+00,-0.31198E+00,-0.31207E+00,-0.31216E+00,
     &-0.31225E+00,-0.31234E+00,-0.31243E+00,-0.31252E+00,-0.31261E+00,
     &-0.31270E+00,-0.31279E+00,-0.31289E+00,-0.31298E+00,-0.31308E+00,
     &-0.31317E+00,-0.31327E+00,-0.31336E+00,-0.31346E+00,-0.31356E+00,
     &-0.31365E+00,-0.31375E+00,-0.31385E+00,-0.31395E+00,-0.31405E+00,
     &-0.31415E+00,-0.31425E+00,-0.31435E+00,-0.31445E+00,-0.31456E+00,
     &-0.31466E+00,-0.31476E+00,-0.31487E+00,-0.31497E+00,-0.31508E+00,
     &-0.31518E+00,-0.31529E+00,-0.31539E+00,-0.31550E+00,-0.31561E+00,
     &-0.31572E+00,-0.31583E+00,-0.31594E+00,-0.31605E+00,-0.31616E+00/

      DATA (BNC06M(I),I=501,600)/
     &-0.31627E+00,-0.31638E+00,-0.31649E+00,-0.31660E+00,-0.31672E+00,
     &-0.31683E+00,-0.31694E+00,-0.31706E+00,-0.31717E+00,-0.31729E+00,
     &-0.31740E+00,-0.31752E+00,-0.31764E+00,-0.31776E+00,-0.31787E+00,
     &-0.31799E+00,-0.31811E+00,-0.31823E+00,-0.31835E+00,-0.31847E+00,
     &-0.31859E+00,-0.31871E+00,-0.31884E+00,-0.31896E+00,-0.31908E+00,
     &-0.31921E+00,-0.31933E+00,-0.31945E+00,-0.31958E+00,-0.31971E+00,
     &-0.31983E+00,-0.31996E+00,-0.32009E+00,-0.32021E+00,-0.32034E+00,
     &-0.32047E+00,-0.32060E+00,-0.32073E+00,-0.32086E+00,-0.32099E+00,
     &-0.32112E+00,-0.32125E+00,-0.32138E+00,-0.32151E+00,-0.32165E+00,
     &-0.32178E+00,-0.32191E+00,-0.32205E+00,-0.32218E+00,-0.32232E+00,
     &-0.32246E+00,-0.32259E+00,-0.32273E+00,-0.32287E+00,-0.32300E+00,
     &-0.32314E+00,-0.32328E+00,-0.32342E+00,-0.32356E+00,-0.32370E+00,
     &-0.32384E+00,-0.32398E+00,-0.32412E+00,-0.32426E+00,-0.32440E+00,
     &-0.32455E+00,-0.32469E+00,-0.32483E+00,-0.32498E+00,-0.32512E+00,
     &-0.32527E+00,-0.32541E+00,-0.32556E+00,-0.32571E+00,-0.32585E+00,
     &-0.32600E+00,-0.32615E+00,-0.32630E+00,-0.32644E+00,-0.32659E+00,
     &-0.32674E+00,-0.32689E+00,-0.32704E+00,-0.32719E+00,-0.32735E+00,
     &-0.32750E+00,-0.32765E+00,-0.32780E+00,-0.32796E+00,-0.32811E+00,
     &-0.32826E+00,-0.32842E+00,-0.32857E+00,-0.32873E+00,-0.32888E+00,
     &-0.32904E+00,-0.32920E+00,-0.32935E+00,-0.32951E+00,-0.32967E+00/

      DATA (BNC06M(I),I=601,700)/
     &-0.33140E+00,-0.33305E+00,-0.33474E+00,-0.33647E+00,-0.33824E+00,
     &-0.34005E+00,-0.34190E+00,-0.34379E+00,-0.34571E+00,-0.34767E+00,
     &-0.34966E+00,-0.35169E+00,-0.35376E+00,-0.35586E+00,-0.35799E+00,
     &-0.36016E+00,-0.36235E+00,-0.36458E+00,-0.36684E+00,-0.36913E+00,
     &-0.37146E+00,-0.37381E+00,-0.37619E+00,-0.37859E+00,-0.38103E+00,
     &-0.38349E+00,-0.38598E+00,-0.38850E+00,-0.39105E+00,-0.39361E+00,
     &-0.39621E+00,-0.39883E+00,-0.40147E+00,-0.40414E+00,-0.40684E+00,
     &-0.40955E+00,-0.41229E+00,-0.41506E+00,-0.41784E+00,-0.42065E+00,
     &-0.42348E+00,-0.42633E+00,-0.42920E+00,-0.43209E+00,-0.43500E+00,
     &-0.43793E+00,-0.44089E+00,-0.44386E+00,-0.44685E+00,-0.44986E+00,
     &-0.45289E+00,-0.45594E+00,-0.45901E+00,-0.46209E+00,-0.46519E+00,
     &-0.46831E+00,-0.47145E+00,-0.47460E+00,-0.47777E+00,-0.48096E+00,
     &-0.48416E+00,-0.48738E+00,-0.49062E+00,-0.49387E+00,-0.49713E+00,
     &-0.50042E+00,-0.50371E+00,-0.50702E+00,-0.51035E+00,-0.51369E+00,
     &-0.51705E+00,-0.52042E+00,-0.52380E+00,-0.52720E+00,-0.53061E+00,
     &-0.53403E+00,-0.53747E+00,-0.54092E+00,-0.54438E+00,-0.54786E+00,
     &-0.55135E+00,-0.55485E+00,-0.55836E+00,-0.56189E+00,-0.56543E+00,
     &-0.56898E+00,-0.57254E+00,-0.57611E+00,-0.57970E+00,-0.58329E+00,
     &-0.58690E+00,-0.59052E+00,-0.59415E+00,-0.59779E+00,-0.60144E+00,
     &-0.60510E+00,-0.60877E+00,-0.61245E+00,-0.61614E+00,-0.61985E+00/

      DATA (BNC06M(I),I=701,741)/
     &-0.62356E+00,-0.62728E+00,-0.63101E+00,-0.63475E+00,-0.63851E+00,
     &-0.64227E+00,-0.64604E+00,-0.64982E+00,-0.65360E+00,-0.65740E+00,
     &-0.66121E+00,-0.66502E+00,-0.66885E+00,-0.67268E+00,-0.67652E+00,
     &-0.68037E+00,-0.68423E+00,-0.68809E+00,-0.69197E+00,-0.69585E+00,
     &-0.69974E+00,-0.70364E+00,-0.70755E+00,-0.71146E+00,-0.71539E+00,
     &-0.71932E+00,-0.72325E+00,-0.72720E+00,-0.73115E+00,-0.73511E+00,
     &-0.73908E+00,-0.74305E+00,-0.74703E+00,-0.75102E+00,-0.75502E+00,
     &-0.75902E+00,-0.76303E+00,-0.76705E+00,-0.77107E+00,-0.77510E+00,
     &-0.77914E+00
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.10274E+00,-0.22462E+00,-0.28619E+00,-0.33008E+00,-0.36478E+00,
     &-0.39372E+00,-0.41868E+00,-0.44069E+00,-0.46044E+00,-0.47838E+00,
     &-0.49484E+00,-0.51007E+00,-0.52427E+00,-0.53757E+00,-0.55009E+00,
     &-0.56194E+00,-0.57319E+00,-0.58390E+00,-0.59414E+00,-0.60394E+00,
     &-0.61335E+00,-0.62240E+00,-0.63113E+00,-0.63956E+00,-0.64771E+00,
     &-0.65561E+00,-0.66327E+00,-0.67070E+00,-0.67794E+00,-0.68498E+00,
     &-0.69184E+00,-0.69853E+00,-0.70506E+00,-0.71145E+00,-0.71769E+00,
     &-0.72379E+00,-0.72977E+00,-0.73563E+00,-0.74138E+00,-0.74701E+00,
     &-0.75254E+00,-0.75798E+00,-0.76331E+00,-0.76856E+00,-0.77372E+00,
     &-0.77880E+00,-0.78380E+00,-0.78872E+00,-0.79357E+00,-0.79835E+00,
     &-0.80306E+00,-0.80771E+00,-0.81229E+00,-0.81682E+00,-0.82128E+00,
     &-0.82569E+00,-0.83005E+00,-0.83435E+00,-0.83860E+00,-0.84281E+00,
     &-0.84697E+00,-0.85108E+00,-0.85515E+00,-0.85917E+00,-0.86316E+00,
     &-0.86710E+00,-0.87101E+00,-0.87488E+00,-0.87871E+00,-0.88251E+00,
     &-0.88628E+00,-0.89001E+00,-0.89371E+00,-0.89738E+00,-0.90102E+00,
     &-0.90463E+00,-0.90821E+00,-0.91176E+00,-0.91529E+00,-0.91879E+00,
     &-0.92227E+00,-0.92572E+00,-0.92915E+00,-0.93256E+00,-0.93594E+00,
     &-0.93930E+00,-0.94264E+00,-0.94596E+00,-0.94926E+00,-0.95253E+00,
     &-0.95579E+00,-0.95903E+00,-0.96225E+00,-0.96546E+00,-0.96864E+00,
     &-0.97181E+00,-0.97496E+00,-0.97810E+00,-0.98121E+00,-0.98432E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.98740E+00,-0.99047E+00,-0.99353E+00,-0.99657E+00,-0.99960E+00,
     &-0.10026E+01,-0.10056E+01,-0.10086E+01,-0.10116E+01,-0.10145E+01,
     &-0.10175E+01,-0.10204E+01,-0.10233E+01,-0.10262E+01,-0.10291E+01,
     &-0.10320E+01,-0.10349E+01,-0.10377E+01,-0.10406E+01,-0.10434E+01,
     &-0.10462E+01,-0.10490E+01,-0.10518E+01,-0.10546E+01,-0.10574E+01,
     &-0.10601E+01,-0.10629E+01,-0.10656E+01,-0.10684E+01,-0.10711E+01,
     &-0.10738E+01,-0.10765E+01,-0.10792E+01,-0.10819E+01,-0.10846E+01,
     &-0.10872E+01,-0.10899E+01,-0.10926E+01,-0.10952E+01,-0.10978E+01,
     &-0.11004E+01,-0.11031E+01,-0.11057E+01,-0.11083E+01,-0.11108E+01,
     &-0.11134E+01,-0.11160E+01,-0.11186E+01,-0.11211E+01,-0.11237E+01,
     &-0.11262E+01,-0.11287E+01,-0.11313E+01,-0.11338E+01,-0.11363E+01,
     &-0.11388E+01,-0.11413E+01,-0.11438E+01,-0.11463E+01,-0.11487E+01,
     &-0.11512E+01,-0.11537E+01,-0.11561E+01,-0.11586E+01,-0.11610E+01,
     &-0.11634E+01,-0.11659E+01,-0.11683E+01,-0.11707E+01,-0.11731E+01,
     &-0.11755E+01,-0.11779E+01,-0.11803E+01,-0.11827E+01,-0.11851E+01,
     &-0.11874E+01,-0.11898E+01,-0.11922E+01,-0.11945E+01,-0.11969E+01,
     &-0.11992E+01,-0.12016E+01,-0.12039E+01,-0.12062E+01,-0.12086E+01,
     &-0.12109E+01,-0.12132E+01,-0.12155E+01,-0.12178E+01,-0.12201E+01,
     &-0.12224E+01,-0.12247E+01,-0.12270E+01,-0.12293E+01,-0.12315E+01,
     &-0.12338E+01,-0.12361E+01,-0.12383E+01,-0.12406E+01,-0.12428E+01/

      DATA (BNC07M(I),I=201,300)/
     &-0.12451E+01,-0.12473E+01,-0.12496E+01,-0.12518E+01,-0.12540E+01,
     &-0.12563E+01,-0.12585E+01,-0.12607E+01,-0.12629E+01,-0.12651E+01,
     &-0.12673E+01,-0.12695E+01,-0.12717E+01,-0.12739E+01,-0.12761E+01,
     &-0.12783E+01,-0.12805E+01,-0.12827E+01,-0.12848E+01,-0.12870E+01,
     &-0.12892E+01,-0.12913E+01,-0.12935E+01,-0.12956E+01,-0.12978E+01,
     &-0.12999E+01,-0.13021E+01,-0.13042E+01,-0.13064E+01,-0.13085E+01,
     &-0.13106E+01,-0.13128E+01,-0.13149E+01,-0.13170E+01,-0.13191E+01,
     &-0.13212E+01,-0.13233E+01,-0.13254E+01,-0.13275E+01,-0.13296E+01,
     &-0.13317E+01,-0.13338E+01,-0.13359E+01,-0.13380E+01,-0.13401E+01,
     &-0.13422E+01,-0.13443E+01,-0.13463E+01,-0.13484E+01,-0.13505E+01,
     &-0.13526E+01,-0.13546E+01,-0.13567E+01,-0.13587E+01,-0.13608E+01,
     &-0.13628E+01,-0.13649E+01,-0.13669E+01,-0.13690E+01,-0.13710E+01,
     &-0.13731E+01,-0.13751E+01,-0.13771E+01,-0.13792E+01,-0.13812E+01,
     &-0.13832E+01,-0.13852E+01,-0.13873E+01,-0.13893E+01,-0.13913E+01,
     &-0.13933E+01,-0.13953E+01,-0.13973E+01,-0.13993E+01,-0.14013E+01,
     &-0.14033E+01,-0.14053E+01,-0.14073E+01,-0.14093E+01,-0.14113E+01,
     &-0.14133E+01,-0.14153E+01,-0.14173E+01,-0.14192E+01,-0.14212E+01,
     &-0.14232E+01,-0.14252E+01,-0.14271E+01,-0.14291E+01,-0.14311E+01,
     &-0.14330E+01,-0.14350E+01,-0.14370E+01,-0.14389E+01,-0.14409E+01,
     &-0.14428E+01,-0.14448E+01,-0.14467E+01,-0.14487E+01,-0.14506E+01/

      DATA (BNC07M(I),I=301,400)/
     &-0.14526E+01,-0.14545E+01,-0.14564E+01,-0.14584E+01,-0.14603E+01,
     &-0.14622E+01,-0.14642E+01,-0.14661E+01,-0.14680E+01,-0.14700E+01,
     &-0.14719E+01,-0.14738E+01,-0.14757E+01,-0.14776E+01,-0.14795E+01,
     &-0.14815E+01,-0.14834E+01,-0.14853E+01,-0.14872E+01,-0.14891E+01,
     &-0.14910E+01,-0.14929E+01,-0.14948E+01,-0.14967E+01,-0.14986E+01,
     &-0.15005E+01,-0.15024E+01,-0.15043E+01,-0.15062E+01,-0.15080E+01,
     &-0.15099E+01,-0.15118E+01,-0.15137E+01,-0.15156E+01,-0.15174E+01,
     &-0.15193E+01,-0.15212E+01,-0.15231E+01,-0.15249E+01,-0.15268E+01,
     &-0.15287E+01,-0.15305E+01,-0.15324E+01,-0.15343E+01,-0.15361E+01,
     &-0.15380E+01,-0.15399E+01,-0.15417E+01,-0.15436E+01,-0.15454E+01,
     &-0.15473E+01,-0.15491E+01,-0.15510E+01,-0.15528E+01,-0.15547E+01,
     &-0.15565E+01,-0.15583E+01,-0.15602E+01,-0.15620E+01,-0.15639E+01,
     &-0.15657E+01,-0.15675E+01,-0.15694E+01,-0.15712E+01,-0.15730E+01,
     &-0.15749E+01,-0.15767E+01,-0.15785E+01,-0.15803E+01,-0.15822E+01,
     &-0.15840E+01,-0.15858E+01,-0.15876E+01,-0.15894E+01,-0.15913E+01,
     &-0.15931E+01,-0.15949E+01,-0.15967E+01,-0.15985E+01,-0.16003E+01,
     &-0.16021E+01,-0.16039E+01,-0.16057E+01,-0.16075E+01,-0.16093E+01,
     &-0.16111E+01,-0.16129E+01,-0.16147E+01,-0.16165E+01,-0.16183E+01,
     &-0.16201E+01,-0.16219E+01,-0.16237E+01,-0.16255E+01,-0.16273E+01,
     &-0.16291E+01,-0.16309E+01,-0.16326E+01,-0.16344E+01,-0.16362E+01/

      DATA (BNC07M(I),I=401,500)/
     &-0.16380E+01,-0.16398E+01,-0.16415E+01,-0.16433E+01,-0.16451E+01,
     &-0.16469E+01,-0.16486E+01,-0.16504E+01,-0.16522E+01,-0.16540E+01,
     &-0.16557E+01,-0.16575E+01,-0.16593E+01,-0.16610E+01,-0.16628E+01,
     &-0.16646E+01,-0.16663E+01,-0.16681E+01,-0.16698E+01,-0.16716E+01,
     &-0.16734E+01,-0.16751E+01,-0.16769E+01,-0.16786E+01,-0.16804E+01,
     &-0.16821E+01,-0.16839E+01,-0.16856E+01,-0.16874E+01,-0.16891E+01,
     &-0.16909E+01,-0.16926E+01,-0.16944E+01,-0.16961E+01,-0.16978E+01,
     &-0.16996E+01,-0.17013E+01,-0.17031E+01,-0.17048E+01,-0.17065E+01,
     &-0.17083E+01,-0.17100E+01,-0.17117E+01,-0.17135E+01,-0.17152E+01,
     &-0.17169E+01,-0.17187E+01,-0.17204E+01,-0.17221E+01,-0.17238E+01,
     &-0.17256E+01,-0.17273E+01,-0.17290E+01,-0.17307E+01,-0.17325E+01,
     &-0.17342E+01,-0.17359E+01,-0.17376E+01,-0.17393E+01,-0.17411E+01,
     &-0.17428E+01,-0.17445E+01,-0.17462E+01,-0.17479E+01,-0.17496E+01,
     &-0.17513E+01,-0.17530E+01,-0.17547E+01,-0.17565E+01,-0.17582E+01,
     &-0.17599E+01,-0.17616E+01,-0.17633E+01,-0.17650E+01,-0.17667E+01,
     &-0.17684E+01,-0.17701E+01,-0.17718E+01,-0.17735E+01,-0.17752E+01,
     &-0.17769E+01,-0.17786E+01,-0.17803E+01,-0.17820E+01,-0.17836E+01,
     &-0.17853E+01,-0.17870E+01,-0.17887E+01,-0.17904E+01,-0.17921E+01,
     &-0.17938E+01,-0.17955E+01,-0.17972E+01,-0.17988E+01,-0.18005E+01,
     &-0.18022E+01,-0.18039E+01,-0.18056E+01,-0.18073E+01,-0.18089E+01/

      DATA (BNC07M(I),I=501,600)/
     &-0.18106E+01,-0.18123E+01,-0.18140E+01,-0.18157E+01,-0.18173E+01,
     &-0.18190E+01,-0.18207E+01,-0.18223E+01,-0.18240E+01,-0.18257E+01,
     &-0.18274E+01,-0.18290E+01,-0.18307E+01,-0.18324E+01,-0.18340E+01,
     &-0.18357E+01,-0.18374E+01,-0.18390E+01,-0.18407E+01,-0.18424E+01,
     &-0.18440E+01,-0.18457E+01,-0.18474E+01,-0.18490E+01,-0.18507E+01,
     &-0.18523E+01,-0.18540E+01,-0.18556E+01,-0.18573E+01,-0.18590E+01,
     &-0.18606E+01,-0.18623E+01,-0.18639E+01,-0.18656E+01,-0.18672E+01,
     &-0.18689E+01,-0.18705E+01,-0.18722E+01,-0.18738E+01,-0.18755E+01,
     &-0.18771E+01,-0.18788E+01,-0.18804E+01,-0.18821E+01,-0.18837E+01,
     &-0.18854E+01,-0.18870E+01,-0.18886E+01,-0.18903E+01,-0.18919E+01,
     &-0.18936E+01,-0.18952E+01,-0.18968E+01,-0.18985E+01,-0.19001E+01,
     &-0.19018E+01,-0.19034E+01,-0.19050E+01,-0.19067E+01,-0.19083E+01,
     &-0.19099E+01,-0.19116E+01,-0.19132E+01,-0.19148E+01,-0.19165E+01,
     &-0.19181E+01,-0.19197E+01,-0.19214E+01,-0.19230E+01,-0.19246E+01,
     &-0.19262E+01,-0.19279E+01,-0.19295E+01,-0.19311E+01,-0.19327E+01,
     &-0.19344E+01,-0.19360E+01,-0.19376E+01,-0.19392E+01,-0.19408E+01,
     &-0.19425E+01,-0.19441E+01,-0.19457E+01,-0.19473E+01,-0.19489E+01,
     &-0.19506E+01,-0.19522E+01,-0.19538E+01,-0.19554E+01,-0.19570E+01,
     &-0.19586E+01,-0.19602E+01,-0.19619E+01,-0.19635E+01,-0.19651E+01,
     &-0.19667E+01,-0.19683E+01,-0.19699E+01,-0.19715E+01,-0.19731E+01/

      DATA (BNC07M(I),I=601,700)/
     &-0.19905E+01,-0.20065E+01,-0.20224E+01,-0.20383E+01,-0.20541E+01,
     &-0.20698E+01,-0.20856E+01,-0.21012E+01,-0.21168E+01,-0.21324E+01,
     &-0.21479E+01,-0.21634E+01,-0.21788E+01,-0.21942E+01,-0.22096E+01,
     &-0.22249E+01,-0.22401E+01,-0.22554E+01,-0.22705E+01,-0.22857E+01,
     &-0.23008E+01,-0.23159E+01,-0.23309E+01,-0.23459E+01,-0.23609E+01,
     &-0.23758E+01,-0.23907E+01,-0.24056E+01,-0.24204E+01,-0.24352E+01,
     &-0.24500E+01,-0.24647E+01,-0.24795E+01,-0.24941E+01,-0.25088E+01,
     &-0.25234E+01,-0.25380E+01,-0.25526E+01,-0.25672E+01,-0.25817E+01,
     &-0.25962E+01,-0.26107E+01,-0.26251E+01,-0.26395E+01,-0.26539E+01,
     &-0.26683E+01,-0.26827E+01,-0.26970E+01,-0.27113E+01,-0.27256E+01,
     &-0.27399E+01,-0.27541E+01,-0.27683E+01,-0.27825E+01,-0.27967E+01,
     &-0.28109E+01,-0.28250E+01,-0.28391E+01,-0.28532E+01,-0.28673E+01,
     &-0.28814E+01,-0.28954E+01,-0.29094E+01,-0.29235E+01,-0.29374E+01,
     &-0.29514E+01,-0.29654E+01,-0.29793E+01,-0.29932E+01,-0.30071E+01,
     &-0.30210E+01,-0.30349E+01,-0.30487E+01,-0.30626E+01,-0.30764E+01,
     &-0.30902E+01,-0.31040E+01,-0.31178E+01,-0.31315E+01,-0.31453E+01,
     &-0.31590E+01,-0.31727E+01,-0.31864E+01,-0.32001E+01,-0.32138E+01,
     &-0.32274E+01,-0.32411E+01,-0.32547E+01,-0.32683E+01,-0.32819E+01,
     &-0.32955E+01,-0.33091E+01,-0.33226E+01,-0.33362E+01,-0.33497E+01,
     &-0.33632E+01,-0.33767E+01,-0.33902E+01,-0.34037E+01,-0.34172E+01/

      DATA (BNC07M(I),I=701,741)/
     &-0.34307E+01,-0.34441E+01,-0.34576E+01,-0.34710E+01,-0.34844E+01,
     &-0.34978E+01,-0.35112E+01,-0.35246E+01,-0.35380E+01,-0.35513E+01,
     &-0.35647E+01,-0.35780E+01,-0.35913E+01,-0.36046E+01,-0.36179E+01,
     &-0.36312E+01,-0.36445E+01,-0.36578E+01,-0.36711E+01,-0.36843E+01,
     &-0.36976E+01,-0.37108E+01,-0.37240E+01,-0.37372E+01,-0.37504E+01,
     &-0.37636E+01,-0.37768E+01,-0.37900E+01,-0.38032E+01,-0.38163E+01,
     &-0.38295E+01,-0.38426E+01,-0.38557E+01,-0.38689E+01,-0.38820E+01,
     &-0.38951E+01,-0.39082E+01,-0.39213E+01,-0.39343E+01,-0.39474E+01,
     &-0.39605E+01
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.47406E-01,-0.92542E-01,-0.10989E+00,-0.11949E+00,-0.12516E+00,
     &-0.12837E+00,-0.12988E+00,-0.13012E+00,-0.12936E+00,-0.12778E+00,
     &-0.12553E+00,-0.12268E+00,-0.11932E+00,-0.11550E+00,-0.11127E+00,
     &-0.10666E+00,-0.10171E+00,-0.96432E-01,-0.90858E-01,-0.85004E-01,
     &-0.78885E-01,-0.72517E-01,-0.65912E-01,-0.59082E-01,-0.52037E-01,
     &-0.44788E-01,-0.37343E-01,-0.29710E-01,-0.21898E-01,-0.13914E-01,
     &-0.57643E-02, 0.25435E-02, 0.11003E-01, 0.19609E-01, 0.28355E-01,
     & 0.37235E-01, 0.46244E-01, 0.55377E-01, 0.64628E-01, 0.73993E-01,
     & 0.83468E-01, 0.93047E-01, 0.10273E+00, 0.11250E+00, 0.12237E+00,
     & 0.13232E+00, 0.14236E+00, 0.15248E+00, 0.16268E+00, 0.17295E+00,
     & 0.18330E+00, 0.19371E+00, 0.20419E+00, 0.21473E+00, 0.22534E+00,
     & 0.23600E+00, 0.24672E+00, 0.25750E+00, 0.26834E+00, 0.27922E+00,
     & 0.29016E+00, 0.30116E+00, 0.31220E+00, 0.32330E+00, 0.33445E+00,
     & 0.34564E+00, 0.35689E+00, 0.36820E+00, 0.37955E+00, 0.39095E+00,
     & 0.40241E+00, 0.41392E+00, 0.42549E+00, 0.43711E+00, 0.44879E+00,
     & 0.46052E+00, 0.47232E+00, 0.48417E+00, 0.49608E+00, 0.50805E+00,
     & 0.52009E+00, 0.53219E+00, 0.54435E+00, 0.55657E+00, 0.56886E+00,
     & 0.58121E+00, 0.59362E+00, 0.60610E+00, 0.61865E+00, 0.63125E+00,
     & 0.64392E+00, 0.65665E+00, 0.66945E+00, 0.68230E+00, 0.69522E+00,
     & 0.70819E+00, 0.72121E+00, 0.73429E+00, 0.74743E+00, 0.76061E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.77385E+00, 0.78713E+00, 0.80045E+00, 0.81382E+00, 0.82722E+00,
     & 0.84066E+00, 0.85414E+00, 0.86765E+00, 0.88119E+00, 0.89476E+00,
     & 0.90835E+00, 0.92196E+00, 0.93559E+00, 0.94924E+00, 0.96290E+00,
     & 0.97658E+00, 0.99027E+00, 0.10040E+01, 0.10177E+01, 0.10314E+01,
     & 0.10451E+01, 0.10588E+01, 0.10725E+01, 0.10862E+01, 0.10999E+01,
     & 0.11136E+01, 0.11273E+01, 0.11409E+01, 0.11546E+01, 0.11682E+01,
     & 0.11819E+01, 0.11955E+01, 0.12091E+01, 0.12227E+01, 0.12363E+01,
     & 0.12498E+01, 0.12633E+01, 0.12769E+01, 0.12904E+01, 0.13038E+01,
     & 0.13173E+01, 0.13307E+01, 0.13441E+01, 0.13575E+01, 0.13709E+01,
     & 0.13842E+01, 0.13976E+01, 0.14108E+01, 0.14241E+01, 0.14374E+01,
     & 0.14506E+01, 0.14638E+01, 0.14769E+01, 0.14901E+01, 0.15032E+01,
     & 0.15163E+01, 0.15294E+01, 0.15424E+01, 0.15554E+01, 0.15684E+01,
     & 0.15814E+01, 0.15943E+01, 0.16072E+01, 0.16201E+01, 0.16329E+01,
     & 0.16458E+01, 0.16586E+01, 0.16713E+01, 0.16841E+01, 0.16968E+01,
     & 0.17095E+01, 0.17221E+01, 0.17348E+01, 0.17474E+01, 0.17599E+01,
     & 0.17725E+01, 0.17850E+01, 0.17975E+01, 0.18100E+01, 0.18224E+01,
     & 0.18348E+01, 0.18472E+01, 0.18596E+01, 0.18719E+01, 0.18842E+01,
     & 0.18965E+01, 0.19087E+01, 0.19210E+01, 0.19332E+01, 0.19453E+01,
     & 0.19575E+01, 0.19696E+01, 0.19817E+01, 0.19937E+01, 0.20057E+01,
     & 0.20177E+01, 0.20297E+01, 0.20417E+01, 0.20536E+01, 0.20655E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.20773E+01, 0.20892E+01, 0.21010E+01, 0.21128E+01, 0.21245E+01,
     & 0.21363E+01, 0.21480E+01, 0.21597E+01, 0.21713E+01, 0.21829E+01,
     & 0.21945E+01, 0.22061E+01, 0.22177E+01, 0.22292E+01, 0.22407E+01,
     & 0.22521E+01, 0.22636E+01, 0.22750E+01, 0.22864E+01, 0.22978E+01,
     & 0.23091E+01, 0.23204E+01, 0.23317E+01, 0.23430E+01, 0.23542E+01,
     & 0.23654E+01, 0.23766E+01, 0.23878E+01, 0.23989E+01, 0.24100E+01,
     & 0.24211E+01, 0.24322E+01, 0.24432E+01, 0.24542E+01, 0.24652E+01,
     & 0.24762E+01, 0.24871E+01, 0.24980E+01, 0.25089E+01, 0.25198E+01,
     & 0.25306E+01, 0.25414E+01, 0.25522E+01, 0.25630E+01, 0.25737E+01,
     & 0.25845E+01, 0.25952E+01, 0.26058E+01, 0.26165E+01, 0.26271E+01,
     & 0.26377E+01, 0.26483E+01, 0.26589E+01, 0.26694E+01, 0.26799E+01,
     & 0.26904E+01, 0.27009E+01, 0.27113E+01, 0.27217E+01, 0.27321E+01,
     & 0.27425E+01, 0.27528E+01, 0.27632E+01, 0.27735E+01, 0.27838E+01,
     & 0.27940E+01, 0.28043E+01, 0.28145E+01, 0.28247E+01, 0.28348E+01,
     & 0.28450E+01, 0.28551E+01, 0.28652E+01, 0.28753E+01, 0.28854E+01,
     & 0.28954E+01, 0.29055E+01, 0.29155E+01, 0.29254E+01, 0.29354E+01,
     & 0.29453E+01, 0.29552E+01, 0.29651E+01, 0.29750E+01, 0.29849E+01,
     & 0.29947E+01, 0.30045E+01, 0.30143E+01, 0.30241E+01, 0.30338E+01,
     & 0.30436E+01, 0.30533E+01, 0.30630E+01, 0.30726E+01, 0.30823E+01,
     & 0.30919E+01, 0.31015E+01, 0.31111E+01, 0.31207E+01, 0.31302E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.31398E+01, 0.31493E+01, 0.31588E+01, 0.31682E+01, 0.31777E+01,
     & 0.31871E+01, 0.31966E+01, 0.32059E+01, 0.32153E+01, 0.32247E+01,
     & 0.32340E+01, 0.32433E+01, 0.32526E+01, 0.32619E+01, 0.32712E+01,
     & 0.32804E+01, 0.32897E+01, 0.32989E+01, 0.33081E+01, 0.33172E+01,
     & 0.33264E+01, 0.33355E+01, 0.33446E+01, 0.33537E+01, 0.33628E+01,
     & 0.33719E+01, 0.33809E+01, 0.33899E+01, 0.33989E+01, 0.34079E+01,
     & 0.34169E+01, 0.34259E+01, 0.34348E+01, 0.34437E+01, 0.34526E+01,
     & 0.34615E+01, 0.34704E+01, 0.34792E+01, 0.34881E+01, 0.34969E+01,
     & 0.35057E+01, 0.35145E+01, 0.35232E+01, 0.35320E+01, 0.35407E+01,
     & 0.35494E+01, 0.35581E+01, 0.35668E+01, 0.35755E+01, 0.35841E+01,
     & 0.35928E+01, 0.36014E+01, 0.36100E+01, 0.36186E+01, 0.36271E+01,
     & 0.36357E+01, 0.36442E+01, 0.36527E+01, 0.36612E+01, 0.36697E+01,
     & 0.36782E+01, 0.36866E+01, 0.36951E+01, 0.37035E+01, 0.37119E+01,
     & 0.37203E+01, 0.37287E+01, 0.37370E+01, 0.37454E+01, 0.37537E+01,
     & 0.37620E+01, 0.37703E+01, 0.37786E+01, 0.37869E+01, 0.37951E+01,
     & 0.38034E+01, 0.38116E+01, 0.38198E+01, 0.38280E+01, 0.38362E+01,
     & 0.38443E+01, 0.38525E+01, 0.38606E+01, 0.38687E+01, 0.38768E+01,
     & 0.38849E+01, 0.38930E+01, 0.39011E+01, 0.39091E+01, 0.39171E+01,
     & 0.39252E+01, 0.39332E+01, 0.39411E+01, 0.39491E+01, 0.39571E+01,
     & 0.39650E+01, 0.39730E+01, 0.39809E+01, 0.39888E+01, 0.39967E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.40045E+01, 0.40124E+01, 0.40202E+01, 0.40281E+01, 0.40359E+01,
     & 0.40437E+01, 0.40515E+01, 0.40593E+01, 0.40670E+01, 0.40748E+01,
     & 0.40825E+01, 0.40903E+01, 0.40980E+01, 0.41057E+01, 0.41133E+01,
     & 0.41210E+01, 0.41287E+01, 0.41363E+01, 0.41440E+01, 0.41516E+01,
     & 0.41592E+01, 0.41668E+01, 0.41743E+01, 0.41819E+01, 0.41895E+01,
     & 0.41970E+01, 0.42045E+01, 0.42120E+01, 0.42195E+01, 0.42270E+01,
     & 0.42345E+01, 0.42420E+01, 0.42494E+01, 0.42569E+01, 0.42643E+01,
     & 0.42717E+01, 0.42791E+01, 0.42865E+01, 0.42939E+01, 0.43012E+01,
     & 0.43086E+01, 0.43159E+01, 0.43232E+01, 0.43306E+01, 0.43379E+01,
     & 0.43452E+01, 0.43524E+01, 0.43597E+01, 0.43670E+01, 0.43742E+01,
     & 0.43814E+01, 0.43886E+01, 0.43959E+01, 0.44030E+01, 0.44102E+01,
     & 0.44174E+01, 0.44246E+01, 0.44317E+01, 0.44388E+01, 0.44460E+01,
     & 0.44531E+01, 0.44602E+01, 0.44673E+01, 0.44743E+01, 0.44814E+01,
     & 0.44885E+01, 0.44955E+01, 0.45025E+01, 0.45096E+01, 0.45166E+01,
     & 0.45236E+01, 0.45306E+01, 0.45375E+01, 0.45445E+01, 0.45515E+01,
     & 0.45584E+01, 0.45653E+01, 0.45723E+01, 0.45792E+01, 0.45861E+01,
     & 0.45930E+01, 0.45998E+01, 0.46067E+01, 0.46135E+01, 0.46204E+01,
     & 0.46272E+01, 0.46341E+01, 0.46409E+01, 0.46477E+01, 0.46545E+01,
     & 0.46612E+01, 0.46680E+01, 0.46748E+01, 0.46815E+01, 0.46883E+01,
     & 0.46950E+01, 0.47017E+01, 0.47084E+01, 0.47151E+01, 0.47218E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.47285E+01, 0.47351E+01, 0.47418E+01, 0.47484E+01, 0.47551E+01,
     & 0.47617E+01, 0.47683E+01, 0.47749E+01, 0.47815E+01, 0.47881E+01,
     & 0.47947E+01, 0.48012E+01, 0.48078E+01, 0.48143E+01, 0.48209E+01,
     & 0.48274E+01, 0.48339E+01, 0.48404E+01, 0.48469E+01, 0.48534E+01,
     & 0.48598E+01, 0.48663E+01, 0.48728E+01, 0.48792E+01, 0.48857E+01,
     & 0.48921E+01, 0.48985E+01, 0.49049E+01, 0.49113E+01, 0.49177E+01,
     & 0.49241E+01, 0.49304E+01, 0.49368E+01, 0.49431E+01, 0.49495E+01,
     & 0.49558E+01, 0.49621E+01, 0.49685E+01, 0.49748E+01, 0.49811E+01,
     & 0.49873E+01, 0.49936E+01, 0.49999E+01, 0.50061E+01, 0.50124E+01,
     & 0.50186E+01, 0.50249E+01, 0.50311E+01, 0.50373E+01, 0.50435E+01,
     & 0.50497E+01, 0.50559E+01, 0.50621E+01, 0.50682E+01, 0.50744E+01,
     & 0.50805E+01, 0.50867E+01, 0.50928E+01, 0.50989E+01, 0.51050E+01,
     & 0.51111E+01, 0.51172E+01, 0.51233E+01, 0.51294E+01, 0.51355E+01,
     & 0.51415E+01, 0.51476E+01, 0.51536E+01, 0.51597E+01, 0.51657E+01,
     & 0.51717E+01, 0.51777E+01, 0.51837E+01, 0.51897E+01, 0.51957E+01,
     & 0.52017E+01, 0.52076E+01, 0.52136E+01, 0.52196E+01, 0.52255E+01,
     & 0.52314E+01, 0.52374E+01, 0.52433E+01, 0.52492E+01, 0.52551E+01,
     & 0.52610E+01, 0.52669E+01, 0.52727E+01, 0.52786E+01, 0.52845E+01,
     & 0.52903E+01, 0.52962E+01, 0.53020E+01, 0.53078E+01, 0.53137E+01,
     & 0.53195E+01, 0.53253E+01, 0.53311E+01, 0.53369E+01, 0.53426E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.54046E+01, 0.54611E+01, 0.55169E+01, 0.55719E+01, 0.56263E+01,
     & 0.56799E+01, 0.57328E+01, 0.57851E+01, 0.58367E+01, 0.58877E+01,
     & 0.59380E+01, 0.59878E+01, 0.60369E+01, 0.60855E+01, 0.61334E+01,
     & 0.61809E+01, 0.62277E+01, 0.62740E+01, 0.63198E+01, 0.63651E+01,
     & 0.64099E+01, 0.64542E+01, 0.64979E+01, 0.65413E+01, 0.65841E+01,
     & 0.66265E+01, 0.66684E+01, 0.67099E+01, 0.67509E+01, 0.67916E+01,
     & 0.68318E+01, 0.68715E+01, 0.69109E+01, 0.69499E+01, 0.69885E+01,
     & 0.70267E+01, 0.70646E+01, 0.71020E+01, 0.71391E+01, 0.71759E+01,
     & 0.72123E+01, 0.72483E+01, 0.72840E+01, 0.73194E+01, 0.73544E+01,
     & 0.73891E+01, 0.74235E+01, 0.74576E+01, 0.74914E+01, 0.75248E+01,
     & 0.75580E+01, 0.75909E+01, 0.76234E+01, 0.76557E+01, 0.76877E+01,
     & 0.77195E+01, 0.77509E+01, 0.77821E+01, 0.78130E+01, 0.78437E+01,
     & 0.78741E+01, 0.79042E+01, 0.79341E+01, 0.79637E+01, 0.79931E+01,
     & 0.80223E+01, 0.80512E+01, 0.80799E+01, 0.81083E+01, 0.81365E+01,
     & 0.81645E+01, 0.81923E+01, 0.82199E+01, 0.82472E+01, 0.82743E+01,
     & 0.83012E+01, 0.83279E+01, 0.83544E+01, 0.83807E+01, 0.84068E+01,
     & 0.84327E+01, 0.84584E+01, 0.84839E+01, 0.85093E+01, 0.85344E+01,
     & 0.85593E+01, 0.85841E+01, 0.86087E+01, 0.86331E+01, 0.86573E+01,
     & 0.86813E+01, 0.87052E+01, 0.87289E+01, 0.87524E+01, 0.87758E+01,
     & 0.87990E+01, 0.88220E+01, 0.88449E+01, 0.88676E+01, 0.88902E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.89126E+01, 0.89348E+01, 0.89569E+01, 0.89789E+01, 0.90006E+01,
     & 0.90223E+01, 0.90438E+01, 0.90651E+01, 0.90863E+01, 0.91074E+01,
     & 0.91283E+01, 0.91491E+01, 0.91698E+01, 0.91903E+01, 0.92107E+01,
     & 0.92309E+01, 0.92510E+01, 0.92710E+01, 0.92909E+01, 0.93106E+01,
     & 0.93302E+01, 0.93497E+01, 0.93690E+01, 0.93883E+01, 0.94074E+01,
     & 0.94264E+01, 0.94452E+01, 0.94640E+01, 0.94826E+01, 0.95011E+01,
     & 0.95195E+01, 0.95378E+01, 0.95560E+01, 0.95741E+01, 0.95920E+01,
     & 0.96099E+01, 0.96276E+01, 0.96452E+01, 0.96628E+01, 0.96802E+01,
     & 0.96975E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.50187E-01,-0.10669E+00,-0.13390E+00,-0.15267E+00,-0.16708E+00,
     &-0.17876E+00,-0.18855E+00,-0.19694E+00,-0.20423E+00,-0.21065E+00,
     &-0.21635E+00,-0.22143E+00,-0.22598E+00,-0.23007E+00,-0.23375E+00,
     &-0.23707E+00,-0.24005E+00,-0.24274E+00,-0.24514E+00,-0.24730E+00,
     &-0.24921E+00,-0.25090E+00,-0.25238E+00,-0.25366E+00,-0.25476E+00,
     &-0.25568E+00,-0.25644E+00,-0.25703E+00,-0.25747E+00,-0.25776E+00,
     &-0.25791E+00,-0.25793E+00,-0.25781E+00,-0.25758E+00,-0.25722E+00,
     &-0.25674E+00,-0.25615E+00,-0.25546E+00,-0.25466E+00,-0.25376E+00,
     &-0.25276E+00,-0.25167E+00,-0.25049E+00,-0.24922E+00,-0.24787E+00,
     &-0.24644E+00,-0.24492E+00,-0.24333E+00,-0.24167E+00,-0.23994E+00,
     &-0.23813E+00,-0.23626E+00,-0.23433E+00,-0.23233E+00,-0.23027E+00,
     &-0.22816E+00,-0.22599E+00,-0.22376E+00,-0.22148E+00,-0.21915E+00,
     &-0.21677E+00,-0.21434E+00,-0.21186E+00,-0.20934E+00,-0.20677E+00,
     &-0.20415E+00,-0.20150E+00,-0.19880E+00,-0.19606E+00,-0.19328E+00,
     &-0.19046E+00,-0.18760E+00,-0.18470E+00,-0.18176E+00,-0.17878E+00,
     &-0.17577E+00,-0.17272E+00,-0.16963E+00,-0.16651E+00,-0.16334E+00,
     &-0.16015E+00,-0.15692E+00,-0.15365E+00,-0.15035E+00,-0.14701E+00,
     &-0.14364E+00,-0.14024E+00,-0.13681E+00,-0.13334E+00,-0.12984E+00,
     &-0.12631E+00,-0.12274E+00,-0.11915E+00,-0.11553E+00,-0.11188E+00,
     &-0.10820E+00,-0.10450E+00,-0.10077E+00,-0.97011E-01,-0.93232E-01/

      DATA (BNC09M(I),I=101,200)/
     &-0.89429E-01,-0.85605E-01,-0.81759E-01,-0.77893E-01,-0.74009E-01,
     &-0.70107E-01,-0.66188E-01,-0.62254E-01,-0.58306E-01,-0.54343E-01,
     &-0.50369E-01,-0.46384E-01,-0.42388E-01,-0.38383E-01,-0.34370E-01,
     &-0.30350E-01,-0.26323E-01,-0.22291E-01,-0.18255E-01,-0.14216E-01,
     &-0.10173E-01,-0.61287E-02,-0.20831E-02, 0.19630E-02, 0.60090E-02,
     & 0.10054E-01, 0.14098E-01, 0.18140E-01, 0.22179E-01, 0.26215E-01,
     & 0.30247E-01, 0.34276E-01, 0.38300E-01, 0.42319E-01, 0.46333E-01,
     & 0.50341E-01, 0.54344E-01, 0.58340E-01, 0.62330E-01, 0.66313E-01,
     & 0.70289E-01, 0.74258E-01, 0.78219E-01, 0.82173E-01, 0.86120E-01,
     & 0.90058E-01, 0.93988E-01, 0.97910E-01, 0.10182E+00, 0.10573E+00,
     & 0.10962E+00, 0.11351E+00, 0.11739E+00, 0.12126E+00, 0.12512E+00,
     & 0.12897E+00, 0.13282E+00, 0.13665E+00, 0.14048E+00, 0.14429E+00,
     & 0.14810E+00, 0.15189E+00, 0.15568E+00, 0.15946E+00, 0.16323E+00,
     & 0.16698E+00, 0.17073E+00, 0.17447E+00, 0.17820E+00, 0.18192E+00,
     & 0.18563E+00, 0.18934E+00, 0.19303E+00, 0.19671E+00, 0.20038E+00,
     & 0.20404E+00, 0.20770E+00, 0.21134E+00, 0.21497E+00, 0.21860E+00,
     & 0.22221E+00, 0.22582E+00, 0.22941E+00, 0.23300E+00, 0.23658E+00,
     & 0.24014E+00, 0.24370E+00, 0.24725E+00, 0.25078E+00, 0.25431E+00,
     & 0.25783E+00, 0.26134E+00, 0.26484E+00, 0.26833E+00, 0.27181E+00,
     & 0.27528E+00, 0.27875E+00, 0.28220E+00, 0.28564E+00, 0.28908E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.29250E+00, 0.29592E+00, 0.29932E+00, 0.30272E+00, 0.30611E+00,
     & 0.30949E+00, 0.31286E+00, 0.31622E+00, 0.31957E+00, 0.32291E+00,
     & 0.32625E+00, 0.32957E+00, 0.33289E+00, 0.33620E+00, 0.33949E+00,
     & 0.34278E+00, 0.34606E+00, 0.34934E+00, 0.35260E+00, 0.35585E+00,
     & 0.35910E+00, 0.36234E+00, 0.36557E+00, 0.36879E+00, 0.37200E+00,
     & 0.37520E+00, 0.37839E+00, 0.38158E+00, 0.38476E+00, 0.38793E+00,
     & 0.39109E+00, 0.39424E+00, 0.39738E+00, 0.40052E+00, 0.40365E+00,
     & 0.40677E+00, 0.40988E+00, 0.41298E+00, 0.41608E+00, 0.41916E+00,
     & 0.42224E+00, 0.42531E+00, 0.42838E+00, 0.43143E+00, 0.43448E+00,
     & 0.43752E+00, 0.44055E+00, 0.44357E+00, 0.44659E+00, 0.44960E+00,
     & 0.45260E+00, 0.45559E+00, 0.45857E+00, 0.46155E+00, 0.46452E+00,
     & 0.46748E+00, 0.47044E+00, 0.47339E+00, 0.47633E+00, 0.47926E+00,
     & 0.48218E+00, 0.48510E+00, 0.48801E+00, 0.49092E+00, 0.49381E+00,
     & 0.49670E+00, 0.49958E+00, 0.50246E+00, 0.50532E+00, 0.50818E+00,
     & 0.51104E+00, 0.51388E+00, 0.51672E+00, 0.51955E+00, 0.52238E+00,
     & 0.52519E+00, 0.52800E+00, 0.53081E+00, 0.53361E+00, 0.53640E+00,
     & 0.53918E+00, 0.54196E+00, 0.54473E+00, 0.54749E+00, 0.55025E+00,
     & 0.55300E+00, 0.55574E+00, 0.55848E+00, 0.56121E+00, 0.56393E+00,
     & 0.56665E+00, 0.56936E+00, 0.57206E+00, 0.57476E+00, 0.57745E+00,
     & 0.58013E+00, 0.58281E+00, 0.58548E+00, 0.58815E+00, 0.59081E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.59346E+00, 0.59611E+00, 0.59875E+00, 0.60138E+00, 0.60401E+00,
     & 0.60663E+00, 0.60925E+00, 0.61186E+00, 0.61446E+00, 0.61706E+00,
     & 0.61965E+00, 0.62223E+00, 0.62481E+00, 0.62739E+00, 0.62996E+00,
     & 0.63252E+00, 0.63507E+00, 0.63762E+00, 0.64017E+00, 0.64270E+00,
     & 0.64524E+00, 0.64776E+00, 0.65028E+00, 0.65280E+00, 0.65531E+00,
     & 0.65781E+00, 0.66031E+00, 0.66280E+00, 0.66529E+00, 0.66777E+00,
     & 0.67024E+00, 0.67271E+00, 0.67518E+00, 0.67764E+00, 0.68009E+00,
     & 0.68254E+00, 0.68498E+00, 0.68742E+00, 0.68985E+00, 0.69227E+00,
     & 0.69470E+00, 0.69711E+00, 0.69952E+00, 0.70192E+00, 0.70432E+00,
     & 0.70672E+00, 0.70911E+00, 0.71149E+00, 0.71387E+00, 0.71624E+00,
     & 0.71861E+00, 0.72097E+00, 0.72333E+00, 0.72568E+00, 0.72803E+00,
     & 0.73037E+00, 0.73271E+00, 0.73504E+00, 0.73737E+00, 0.73969E+00,
     & 0.74201E+00, 0.74432E+00, 0.74662E+00, 0.74893E+00, 0.75122E+00,
     & 0.75352E+00, 0.75580E+00, 0.75809E+00, 0.76036E+00, 0.76263E+00,
     & 0.76490E+00, 0.76717E+00, 0.76942E+00, 0.77168E+00, 0.77393E+00,
     & 0.77617E+00, 0.77841E+00, 0.78064E+00, 0.78287E+00, 0.78510E+00,
     & 0.78732E+00, 0.78953E+00, 0.79174E+00, 0.79395E+00, 0.79615E+00,
     & 0.79835E+00, 0.80054E+00, 0.80273E+00, 0.80491E+00, 0.80709E+00,
     & 0.80927E+00, 0.81144E+00, 0.81360E+00, 0.81576E+00, 0.81792E+00,
     & 0.82007E+00, 0.82222E+00, 0.82436E+00, 0.82650E+00, 0.82864E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.83077E+00, 0.83289E+00, 0.83501E+00, 0.83713E+00, 0.83924E+00,
     & 0.84135E+00, 0.84346E+00, 0.84556E+00, 0.84765E+00, 0.84974E+00,
     & 0.85183E+00, 0.85391E+00, 0.85599E+00, 0.85807E+00, 0.86014E+00,
     & 0.86220E+00, 0.86427E+00, 0.86632E+00, 0.86838E+00, 0.87043E+00,
     & 0.87247E+00, 0.87451E+00, 0.87655E+00, 0.87859E+00, 0.88061E+00,
     & 0.88264E+00, 0.88466E+00, 0.88668E+00, 0.88869E+00, 0.89070E+00,
     & 0.89271E+00, 0.89471E+00, 0.89671E+00, 0.89870E+00, 0.90069E+00,
     & 0.90268E+00, 0.90466E+00, 0.90664E+00, 0.90861E+00, 0.91058E+00,
     & 0.91255E+00, 0.91451E+00, 0.91647E+00, 0.91842E+00, 0.92037E+00,
     & 0.92232E+00, 0.92426E+00, 0.92620E+00, 0.92814E+00, 0.93007E+00,
     & 0.93200E+00, 0.93393E+00, 0.93585E+00, 0.93777E+00, 0.93968E+00,
     & 0.94159E+00, 0.94350E+00, 0.94540E+00, 0.94730E+00, 0.94919E+00,
     & 0.95108E+00, 0.95297E+00, 0.95486E+00, 0.95674E+00, 0.95862E+00,
     & 0.96049E+00, 0.96236E+00, 0.96423E+00, 0.96609E+00, 0.96795E+00,
     & 0.96981E+00, 0.97166E+00, 0.97351E+00, 0.97536E+00, 0.97720E+00,
     & 0.97904E+00, 0.98087E+00, 0.98270E+00, 0.98453E+00, 0.98636E+00,
     & 0.98818E+00, 0.99000E+00, 0.99181E+00, 0.99362E+00, 0.99543E+00,
     & 0.99724E+00, 0.99904E+00, 0.10008E+01, 0.10026E+01, 0.10044E+01,
     & 0.10062E+01, 0.10080E+01, 0.10098E+01, 0.10116E+01, 0.10133E+01,
     & 0.10151E+01, 0.10169E+01, 0.10186E+01, 0.10204E+01, 0.10222E+01/

      DATA (BNC09M(I),I=501,600)/
     & 0.10239E+01, 0.10257E+01, 0.10274E+01, 0.10292E+01, 0.10309E+01,
     & 0.10326E+01, 0.10344E+01, 0.10361E+01, 0.10378E+01, 0.10396E+01,
     & 0.10413E+01, 0.10430E+01, 0.10447E+01, 0.10464E+01, 0.10482E+01,
     & 0.10499E+01, 0.10516E+01, 0.10533E+01, 0.10550E+01, 0.10567E+01,
     & 0.10584E+01, 0.10600E+01, 0.10617E+01, 0.10634E+01, 0.10651E+01,
     & 0.10668E+01, 0.10684E+01, 0.10701E+01, 0.10718E+01, 0.10735E+01,
     & 0.10751E+01, 0.10768E+01, 0.10784E+01, 0.10801E+01, 0.10817E+01,
     & 0.10834E+01, 0.10850E+01, 0.10867E+01, 0.10883E+01, 0.10899E+01,
     & 0.10916E+01, 0.10932E+01, 0.10948E+01, 0.10965E+01, 0.10981E+01,
     & 0.10997E+01, 0.11013E+01, 0.11029E+01, 0.11045E+01, 0.11062E+01,
     & 0.11078E+01, 0.11094E+01, 0.11110E+01, 0.11126E+01, 0.11142E+01,
     & 0.11157E+01, 0.11173E+01, 0.11189E+01, 0.11205E+01, 0.11221E+01,
     & 0.11237E+01, 0.11252E+01, 0.11268E+01, 0.11284E+01, 0.11299E+01,
     & 0.11315E+01, 0.11331E+01, 0.11346E+01, 0.11362E+01, 0.11377E+01,
     & 0.11393E+01, 0.11408E+01, 0.11424E+01, 0.11439E+01, 0.11454E+01,
     & 0.11470E+01, 0.11485E+01, 0.11501E+01, 0.11516E+01, 0.11531E+01,
     & 0.11546E+01, 0.11562E+01, 0.11577E+01, 0.11592E+01, 0.11607E+01,
     & 0.11622E+01, 0.11637E+01, 0.11652E+01, 0.11667E+01, 0.11682E+01,
     & 0.11697E+01, 0.11712E+01, 0.11727E+01, 0.11742E+01, 0.11757E+01,
     & 0.11772E+01, 0.11787E+01, 0.11801E+01, 0.11816E+01, 0.11831E+01/

      DATA (BNC09M(I),I=601,700)/
     & 0.11989E+01, 0.12132E+01, 0.12273E+01, 0.12412E+01, 0.12549E+01,
     & 0.12683E+01, 0.12816E+01, 0.12946E+01, 0.13074E+01, 0.13200E+01,
     & 0.13324E+01, 0.13446E+01, 0.13566E+01, 0.13685E+01, 0.13801E+01,
     & 0.13916E+01, 0.14029E+01, 0.14141E+01, 0.14250E+01, 0.14358E+01,
     & 0.14465E+01, 0.14570E+01, 0.14673E+01, 0.14775E+01, 0.14875E+01,
     & 0.14974E+01, 0.15072E+01, 0.15168E+01, 0.15262E+01, 0.15356E+01,
     & 0.15448E+01, 0.15538E+01, 0.15628E+01, 0.15716E+01, 0.15803E+01,
     & 0.15888E+01, 0.15973E+01, 0.16056E+01, 0.16138E+01, 0.16219E+01,
     & 0.16299E+01, 0.16378E+01, 0.16455E+01, 0.16532E+01, 0.16608E+01,
     & 0.16682E+01, 0.16756E+01, 0.16828E+01, 0.16900E+01, 0.16970E+01,
     & 0.17040E+01, 0.17108E+01, 0.17176E+01, 0.17243E+01, 0.17308E+01,
     & 0.17373E+01, 0.17438E+01, 0.17501E+01, 0.17563E+01, 0.17625E+01,
     & 0.17685E+01, 0.17745E+01, 0.17804E+01, 0.17863E+01, 0.17920E+01,
     & 0.17977E+01, 0.18033E+01, 0.18088E+01, 0.18142E+01, 0.18196E+01,
     & 0.18249E+01, 0.18302E+01, 0.18353E+01, 0.18404E+01, 0.18454E+01,
     & 0.18504E+01, 0.18553E+01, 0.18601E+01, 0.18649E+01, 0.18695E+01,
     & 0.18742E+01, 0.18787E+01, 0.18832E+01, 0.18877E+01, 0.18921E+01,
     & 0.18964E+01, 0.19007E+01, 0.19049E+01, 0.19090E+01, 0.19131E+01,
     & 0.19171E+01, 0.19211E+01, 0.19250E+01, 0.19289E+01, 0.19327E+01,
     & 0.19365E+01, 0.19402E+01, 0.19439E+01, 0.19475E+01, 0.19510E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19545E+01, 0.19580E+01, 0.19614E+01, 0.19647E+01, 0.19681E+01,
     & 0.19713E+01, 0.19745E+01, 0.19777E+01, 0.19808E+01, 0.19839E+01,
     & 0.19869E+01, 0.19899E+01, 0.19929E+01, 0.19958E+01, 0.19986E+01,
     & 0.20014E+01, 0.20042E+01, 0.20069E+01, 0.20096E+01, 0.20123E+01,
     & 0.20149E+01, 0.20175E+01, 0.20200E+01, 0.20225E+01, 0.20249E+01,
     & 0.20273E+01, 0.20297E+01, 0.20320E+01, 0.20343E+01, 0.20366E+01,
     & 0.20388E+01, 0.20410E+01, 0.20432E+01, 0.20453E+01, 0.20473E+01,
     & 0.20494E+01, 0.20514E+01, 0.20534E+01, 0.20553E+01, 0.20572E+01,
     & 0.20591E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.49409E-01,-0.10182E+00,-0.12490E+00,-0.13957E+00,-0.14992E+00,
     &-0.15759E+00,-0.16343E+00,-0.16794E+00,-0.17141E+00,-0.17407E+00,
     &-0.17608E+00,-0.17754E+00,-0.17855E+00,-0.17917E+00,-0.17946E+00,
     &-0.17946E+00,-0.17921E+00,-0.17873E+00,-0.17806E+00,-0.17722E+00,
     &-0.17621E+00,-0.17507E+00,-0.17380E+00,-0.17242E+00,-0.17094E+00,
     &-0.16937E+00,-0.16771E+00,-0.16598E+00,-0.16418E+00,-0.16232E+00,
     &-0.16041E+00,-0.15845E+00,-0.15644E+00,-0.15439E+00,-0.15230E+00,
     &-0.15019E+00,-0.14804E+00,-0.14587E+00,-0.14368E+00,-0.14146E+00,
     &-0.13923E+00,-0.13699E+00,-0.13473E+00,-0.13245E+00,-0.13017E+00,
     &-0.12788E+00,-0.12559E+00,-0.12329E+00,-0.12098E+00,-0.11867E+00,
     &-0.11635E+00,-0.11404E+00,-0.11172E+00,-0.10939E+00,-0.10707E+00,
     &-0.10475E+00,-0.10242E+00,-0.10009E+00,-0.97766E-01,-0.95435E-01,
     &-0.93102E-01,-0.90766E-01,-0.88427E-01,-0.86084E-01,-0.83737E-01,
     &-0.81385E-01,-0.79027E-01,-0.76663E-01,-0.74291E-01,-0.71912E-01,
     &-0.69524E-01,-0.67126E-01,-0.64718E-01,-0.62299E-01,-0.59868E-01,
     &-0.57424E-01,-0.54967E-01,-0.52495E-01,-0.50009E-01,-0.47506E-01,
     &-0.44988E-01,-0.42453E-01,-0.39901E-01,-0.37331E-01,-0.34744E-01,
     &-0.32138E-01,-0.29513E-01,-0.26869E-01,-0.24207E-01,-0.21526E-01,
     &-0.18825E-01,-0.16106E-01,-0.13368E-01,-0.10611E-01,-0.78361E-02,
     &-0.50430E-02,-0.22323E-02, 0.59576E-03, 0.34406E-02, 0.63019E-02/

      DATA (BNC10M(I),I=101,200)/
     & 0.91788E-02, 0.12071E-01, 0.14978E-01, 0.17899E-01, 0.20833E-01,
     & 0.23781E-01, 0.26740E-01, 0.29711E-01, 0.32692E-01, 0.35683E-01,
     & 0.38684E-01, 0.41694E-01, 0.44711E-01, 0.47736E-01, 0.50768E-01,
     & 0.53805E-01, 0.56848E-01, 0.59896E-01, 0.62949E-01, 0.66005E-01,
     & 0.69064E-01, 0.72126E-01, 0.75190E-01, 0.78257E-01, 0.81324E-01,
     & 0.84392E-01, 0.87461E-01, 0.90530E-01, 0.93599E-01, 0.96668E-01,
     & 0.99735E-01, 0.10280E+00, 0.10587E+00, 0.10893E+00, 0.11199E+00,
     & 0.11505E+00, 0.11811E+00, 0.12117E+00, 0.12422E+00, 0.12727E+00,
     & 0.13031E+00, 0.13336E+00, 0.13640E+00, 0.13944E+00, 0.14247E+00,
     & 0.14550E+00, 0.14853E+00, 0.15155E+00, 0.15457E+00, 0.15759E+00,
     & 0.16060E+00, 0.16361E+00, 0.16662E+00, 0.16962E+00, 0.17262E+00,
     & 0.17561E+00, 0.17860E+00, 0.18158E+00, 0.18456E+00, 0.18754E+00,
     & 0.19051E+00, 0.19348E+00, 0.19644E+00, 0.19940E+00, 0.20235E+00,
     & 0.20530E+00, 0.20825E+00, 0.21119E+00, 0.21413E+00, 0.21706E+00,
     & 0.21998E+00, 0.22291E+00, 0.22583E+00, 0.22874E+00, 0.23165E+00,
     & 0.23455E+00, 0.23745E+00, 0.24035E+00, 0.24324E+00, 0.24612E+00,
     & 0.24900E+00, 0.25188E+00, 0.25475E+00, 0.25762E+00, 0.26048E+00,
     & 0.26334E+00, 0.26619E+00, 0.26903E+00, 0.27188E+00, 0.27471E+00,
     & 0.27755E+00, 0.28037E+00, 0.28320E+00, 0.28602E+00, 0.28883E+00,
     & 0.29164E+00, 0.29444E+00, 0.29724E+00, 0.30003E+00, 0.30282E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.30560E+00, 0.30838E+00, 0.31116E+00, 0.31393E+00, 0.31669E+00,
     & 0.31945E+00, 0.32220E+00, 0.32495E+00, 0.32769E+00, 0.33043E+00,
     & 0.33317E+00, 0.33590E+00, 0.33862E+00, 0.34134E+00, 0.34405E+00,
     & 0.34676E+00, 0.34947E+00, 0.35217E+00, 0.35486E+00, 0.35755E+00,
     & 0.36024E+00, 0.36291E+00, 0.36559E+00, 0.36826E+00, 0.37092E+00,
     & 0.37358E+00, 0.37624E+00, 0.37889E+00, 0.38153E+00, 0.38417E+00,
     & 0.38681E+00, 0.38944E+00, 0.39206E+00, 0.39468E+00, 0.39730E+00,
     & 0.39991E+00, 0.40251E+00, 0.40511E+00, 0.40771E+00, 0.41030E+00,
     & 0.41288E+00, 0.41546E+00, 0.41804E+00, 0.42061E+00, 0.42317E+00,
     & 0.42574E+00, 0.42829E+00, 0.43084E+00, 0.43339E+00, 0.43593E+00,
     & 0.43847E+00, 0.44100E+00, 0.44353E+00, 0.44605E+00, 0.44856E+00,
     & 0.45108E+00, 0.45358E+00, 0.45609E+00, 0.45858E+00, 0.46108E+00,
     & 0.46357E+00, 0.46605E+00, 0.46853E+00, 0.47100E+00, 0.47347E+00,
     & 0.47594E+00, 0.47840E+00, 0.48085E+00, 0.48330E+00, 0.48575E+00,
     & 0.48819E+00, 0.49062E+00, 0.49305E+00, 0.49548E+00, 0.49790E+00,
     & 0.50032E+00, 0.50273E+00, 0.50514E+00, 0.50754E+00, 0.50994E+00,
     & 0.51233E+00, 0.51472E+00, 0.51711E+00, 0.51949E+00, 0.52186E+00,
     & 0.52423E+00, 0.52660E+00, 0.52896E+00, 0.53132E+00, 0.53367E+00,
     & 0.53602E+00, 0.53836E+00, 0.54070E+00, 0.54303E+00, 0.54536E+00,
     & 0.54769E+00, 0.55001E+00, 0.55232E+00, 0.55464E+00, 0.55694E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.55925E+00, 0.56154E+00, 0.56384E+00, 0.56613E+00, 0.56841E+00,
     & 0.57069E+00, 0.57297E+00, 0.57524E+00, 0.57751E+00, 0.57977E+00,
     & 0.58203E+00, 0.58428E+00, 0.58653E+00, 0.58878E+00, 0.59102E+00,
     & 0.59325E+00, 0.59549E+00, 0.59771E+00, 0.59994E+00, 0.60216E+00,
     & 0.60437E+00, 0.60658E+00, 0.60879E+00, 0.61099E+00, 0.61319E+00,
     & 0.61538E+00, 0.61757E+00, 0.61976E+00, 0.62194E+00, 0.62411E+00,
     & 0.62629E+00, 0.62845E+00, 0.63062E+00, 0.63278E+00, 0.63493E+00,
     & 0.63709E+00, 0.63923E+00, 0.64138E+00, 0.64351E+00, 0.64565E+00,
     & 0.64778E+00, 0.64991E+00, 0.65203E+00, 0.65415E+00, 0.65626E+00,
     & 0.65837E+00, 0.66048E+00, 0.66258E+00, 0.66468E+00, 0.66678E+00,
     & 0.66887E+00, 0.67095E+00, 0.67304E+00, 0.67511E+00, 0.67719E+00,
     & 0.67926E+00, 0.68132E+00, 0.68339E+00, 0.68545E+00, 0.68750E+00,
     & 0.68955E+00, 0.69160E+00, 0.69364E+00, 0.69568E+00, 0.69771E+00,
     & 0.69975E+00, 0.70177E+00, 0.70380E+00, 0.70582E+00, 0.70783E+00,
     & 0.70984E+00, 0.71185E+00, 0.71386E+00, 0.71586E+00, 0.71785E+00,
     & 0.71985E+00, 0.72184E+00, 0.72382E+00, 0.72580E+00, 0.72778E+00,
     & 0.72975E+00, 0.73172E+00, 0.73369E+00, 0.73565E+00, 0.73761E+00,
     & 0.73957E+00, 0.74152E+00, 0.74347E+00, 0.74541E+00, 0.74735E+00,
     & 0.74929E+00, 0.75122E+00, 0.75315E+00, 0.75508E+00, 0.75700E+00,
     & 0.75892E+00, 0.76084E+00, 0.76275E+00, 0.76466E+00, 0.76656E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.76846E+00, 0.77036E+00, 0.77225E+00, 0.77414E+00, 0.77603E+00,
     & 0.77791E+00, 0.77979E+00, 0.78167E+00, 0.78354E+00, 0.78541E+00,
     & 0.78728E+00, 0.78914E+00, 0.79100E+00, 0.79285E+00, 0.79471E+00,
     & 0.79656E+00, 0.79840E+00, 0.80024E+00, 0.80208E+00, 0.80391E+00,
     & 0.80575E+00, 0.80757E+00, 0.80940E+00, 0.81122E+00, 0.81304E+00,
     & 0.81485E+00, 0.81666E+00, 0.81847E+00, 0.82028E+00, 0.82208E+00,
     & 0.82387E+00, 0.82567E+00, 0.82746E+00, 0.82925E+00, 0.83103E+00,
     & 0.83281E+00, 0.83459E+00, 0.83637E+00, 0.83814E+00, 0.83991E+00,
     & 0.84167E+00, 0.84343E+00, 0.84519E+00, 0.84695E+00, 0.84870E+00,
     & 0.85045E+00, 0.85219E+00, 0.85394E+00, 0.85568E+00, 0.85741E+00,
     & 0.85914E+00, 0.86087E+00, 0.86260E+00, 0.86432E+00, 0.86604E+00,
     & 0.86776E+00, 0.86948E+00, 0.87119E+00, 0.87290E+00, 0.87460E+00,
     & 0.87630E+00, 0.87800E+00, 0.87970E+00, 0.88139E+00, 0.88308E+00,
     & 0.88476E+00, 0.88645E+00, 0.88813E+00, 0.88980E+00, 0.89148E+00,
     & 0.89315E+00, 0.89482E+00, 0.89648E+00, 0.89815E+00, 0.89980E+00,
     & 0.90146E+00, 0.90311E+00, 0.90476E+00, 0.90641E+00, 0.90806E+00,
     & 0.90970E+00, 0.91134E+00, 0.91297E+00, 0.91460E+00, 0.91623E+00,
     & 0.91786E+00, 0.91948E+00, 0.92110E+00, 0.92272E+00, 0.92434E+00,
     & 0.92595E+00, 0.92756E+00, 0.92917E+00, 0.93077E+00, 0.93237E+00,
     & 0.93397E+00, 0.93556E+00, 0.93716E+00, 0.93875E+00, 0.94033E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.94192E+00, 0.94350E+00, 0.94508E+00, 0.94665E+00, 0.94822E+00,
     & 0.94979E+00, 0.95136E+00, 0.95292E+00, 0.95449E+00, 0.95604E+00,
     & 0.95760E+00, 0.95915E+00, 0.96070E+00, 0.96225E+00, 0.96380E+00,
     & 0.96534E+00, 0.96688E+00, 0.96842E+00, 0.96995E+00, 0.97148E+00,
     & 0.97301E+00, 0.97454E+00, 0.97606E+00, 0.97758E+00, 0.97910E+00,
     & 0.98062E+00, 0.98213E+00, 0.98364E+00, 0.98515E+00, 0.98665E+00,
     & 0.98815E+00, 0.98965E+00, 0.99115E+00, 0.99265E+00, 0.99414E+00,
     & 0.99563E+00, 0.99711E+00, 0.99860E+00, 0.10001E+01, 0.10016E+01,
     & 0.10030E+01, 0.10045E+01, 0.10060E+01, 0.10074E+01, 0.10089E+01,
     & 0.10104E+01, 0.10118E+01, 0.10133E+01, 0.10148E+01, 0.10162E+01,
     & 0.10177E+01, 0.10191E+01, 0.10206E+01, 0.10220E+01, 0.10234E+01,
     & 0.10249E+01, 0.10263E+01, 0.10277E+01, 0.10292E+01, 0.10306E+01,
     & 0.10320E+01, 0.10335E+01, 0.10349E+01, 0.10363E+01, 0.10377E+01,
     & 0.10391E+01, 0.10405E+01, 0.10420E+01, 0.10434E+01, 0.10448E+01,
     & 0.10462E+01, 0.10476E+01, 0.10490E+01, 0.10504E+01, 0.10518E+01,
     & 0.10531E+01, 0.10545E+01, 0.10559E+01, 0.10573E+01, 0.10587E+01,
     & 0.10601E+01, 0.10614E+01, 0.10628E+01, 0.10642E+01, 0.10655E+01,
     & 0.10669E+01, 0.10683E+01, 0.10696E+01, 0.10710E+01, 0.10724E+01,
     & 0.10737E+01, 0.10751E+01, 0.10764E+01, 0.10778E+01, 0.10791E+01,
     & 0.10805E+01, 0.10818E+01, 0.10831E+01, 0.10845E+01, 0.10858E+01/

      DATA (BNC10M(I),I=601,700)/
     & 0.11001E+01, 0.11131E+01, 0.11259E+01, 0.11384E+01, 0.11508E+01,
     & 0.11629E+01, 0.11749E+01, 0.11867E+01, 0.11983E+01, 0.12097E+01,
     & 0.12209E+01, 0.12319E+01, 0.12428E+01, 0.12535E+01, 0.12640E+01,
     & 0.12744E+01, 0.12846E+01, 0.12946E+01, 0.13045E+01, 0.13142E+01,
     & 0.13238E+01, 0.13333E+01, 0.13426E+01, 0.13517E+01, 0.13607E+01,
     & 0.13696E+01, 0.13784E+01, 0.13870E+01, 0.13955E+01, 0.14038E+01,
     & 0.14121E+01, 0.14202E+01, 0.14282E+01, 0.14361E+01, 0.14438E+01,
     & 0.14515E+01, 0.14590E+01, 0.14664E+01, 0.14738E+01, 0.14810E+01,
     & 0.14881E+01, 0.14951E+01, 0.15020E+01, 0.15088E+01, 0.15155E+01,
     & 0.15221E+01, 0.15286E+01, 0.15350E+01, 0.15413E+01, 0.15475E+01,
     & 0.15536E+01, 0.15597E+01, 0.15656E+01, 0.15715E+01, 0.15773E+01,
     & 0.15830E+01, 0.15886E+01, 0.15942E+01, 0.15996E+01, 0.16050E+01,
     & 0.16103E+01, 0.16155E+01, 0.16207E+01, 0.16258E+01, 0.16308E+01,
     & 0.16357E+01, 0.16406E+01, 0.16453E+01, 0.16501E+01, 0.16547E+01,
     & 0.16593E+01, 0.16638E+01, 0.16682E+01, 0.16726E+01, 0.16769E+01,
     & 0.16812E+01, 0.16854E+01, 0.16895E+01, 0.16936E+01, 0.16976E+01,
     & 0.17015E+01, 0.17054E+01, 0.17092E+01, 0.17130E+01, 0.17167E+01,
     & 0.17204E+01, 0.17240E+01, 0.17275E+01, 0.17310E+01, 0.17345E+01,
     & 0.17379E+01, 0.17412E+01, 0.17445E+01, 0.17477E+01, 0.17509E+01,
     & 0.17540E+01, 0.17571E+01, 0.17601E+01, 0.17631E+01, 0.17660E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17689E+01, 0.17718E+01, 0.17746E+01, 0.17773E+01, 0.17800E+01,
     & 0.17827E+01, 0.17853E+01, 0.17879E+01, 0.17904E+01, 0.17929E+01,
     & 0.17953E+01, 0.17977E+01, 0.18001E+01, 0.18024E+01, 0.18047E+01,
     & 0.18069E+01, 0.18091E+01, 0.18113E+01, 0.18134E+01, 0.18155E+01,
     & 0.18175E+01, 0.18195E+01, 0.18215E+01, 0.18234E+01, 0.18253E+01,
     & 0.18271E+01, 0.18290E+01, 0.18307E+01, 0.18325E+01, 0.18342E+01,
     & 0.18359E+01, 0.18375E+01, 0.18391E+01, 0.18407E+01, 0.18423E+01,
     & 0.18438E+01, 0.18453E+01, 0.18467E+01, 0.18481E+01, 0.18495E+01,
     & 0.18509E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.47849E-01,-0.94150E-01,-0.11209E+00,-0.12203E+00,-0.12791E+00,
     &-0.13126E+00,-0.13288E+00,-0.13320E+00,-0.13253E+00,-0.13106E+00,
     &-0.12892E+00,-0.12624E+00,-0.12307E+00,-0.11949E+00,-0.11555E+00,
     &-0.11128E+00,-0.10671E+00,-0.10189E+00,-0.96822E-01,-0.91537E-01,
     &-0.86052E-01,-0.80381E-01,-0.74539E-01,-0.68539E-01,-0.62391E-01,
     &-0.56107E-01,-0.49696E-01,-0.43167E-01,-0.36526E-01,-0.29783E-01,
     &-0.22944E-01,-0.16015E-01,-0.90018E-02,-0.19110E-02, 0.52525E-02,
     & 0.12484E-01, 0.19778E-01, 0.27131E-01, 0.34538E-01, 0.41996E-01,
     & 0.49501E-01, 0.57049E-01, 0.64637E-01, 0.72262E-01, 0.79922E-01,
     & 0.87613E-01, 0.95334E-01, 0.10308E+00, 0.11085E+00, 0.11865E+00,
     & 0.12647E+00, 0.13431E+00, 0.14217E+00, 0.15004E+00, 0.15794E+00,
     & 0.16585E+00, 0.17377E+00, 0.18172E+00, 0.18967E+00, 0.19765E+00,
     & 0.20564E+00, 0.21364E+00, 0.22167E+00, 0.22971E+00, 0.23776E+00,
     & 0.24584E+00, 0.25394E+00, 0.26206E+00, 0.27020E+00, 0.27836E+00,
     & 0.28655E+00, 0.29476E+00, 0.30301E+00, 0.31128E+00, 0.31958E+00,
     & 0.32791E+00, 0.33628E+00, 0.34468E+00, 0.35311E+00, 0.36158E+00,
     & 0.37009E+00, 0.37864E+00, 0.38723E+00, 0.39586E+00, 0.40453E+00,
     & 0.41324E+00, 0.42200E+00, 0.43079E+00, 0.43963E+00, 0.44852E+00,
     & 0.45744E+00, 0.46641E+00, 0.47542E+00, 0.48448E+00, 0.49357E+00,
     & 0.50270E+00, 0.51188E+00, 0.52109E+00, 0.53034E+00, 0.53963E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.54895E+00, 0.55830E+00, 0.56769E+00, 0.57710E+00, 0.58655E+00,
     & 0.59602E+00, 0.60551E+00, 0.61503E+00, 0.62458E+00, 0.63414E+00,
     & 0.64372E+00, 0.65332E+00, 0.66293E+00, 0.67256E+00, 0.68220E+00,
     & 0.69184E+00, 0.70150E+00, 0.71117E+00, 0.72084E+00, 0.73051E+00,
     & 0.74019E+00, 0.74987E+00, 0.75955E+00, 0.76923E+00, 0.77890E+00,
     & 0.78858E+00, 0.79825E+00, 0.80791E+00, 0.81757E+00, 0.82722E+00,
     & 0.83687E+00, 0.84651E+00, 0.85613E+00, 0.86575E+00, 0.87536E+00,
     & 0.88496E+00, 0.89454E+00, 0.90412E+00, 0.91368E+00, 0.92323E+00,
     & 0.93277E+00, 0.94229E+00, 0.95180E+00, 0.96129E+00, 0.97078E+00,
     & 0.98024E+00, 0.98969E+00, 0.99913E+00, 0.10086E+01, 0.10180E+01,
     & 0.10273E+01, 0.10367E+01, 0.10461E+01, 0.10554E+01, 0.10647E+01,
     & 0.10741E+01, 0.10833E+01, 0.10926E+01, 0.11019E+01, 0.11111E+01,
     & 0.11203E+01, 0.11296E+01, 0.11387E+01, 0.11479E+01, 0.11571E+01,
     & 0.11662E+01, 0.11753E+01, 0.11844E+01, 0.11935E+01, 0.12026E+01,
     & 0.12116E+01, 0.12207E+01, 0.12297E+01, 0.12387E+01, 0.12477E+01,
     & 0.12566E+01, 0.12656E+01, 0.12745E+01, 0.12834E+01, 0.12923E+01,
     & 0.13012E+01, 0.13100E+01, 0.13189E+01, 0.13277E+01, 0.13365E+01,
     & 0.13453E+01, 0.13540E+01, 0.13628E+01, 0.13715E+01, 0.13802E+01,
     & 0.13889E+01, 0.13976E+01, 0.14062E+01, 0.14149E+01, 0.14235E+01,
     & 0.14321E+01, 0.14407E+01, 0.14492E+01, 0.14578E+01, 0.14663E+01/

      DATA (BNC11M(I),I=201,300)/
     & 0.14748E+01, 0.14833E+01, 0.14918E+01, 0.15003E+01, 0.15087E+01,
     & 0.15171E+01, 0.15255E+01, 0.15339E+01, 0.15423E+01, 0.15506E+01,
     & 0.15589E+01, 0.15673E+01, 0.15756E+01, 0.15838E+01, 0.15921E+01,
     & 0.16003E+01, 0.16086E+01, 0.16168E+01, 0.16250E+01, 0.16331E+01,
     & 0.16413E+01, 0.16494E+01, 0.16576E+01, 0.16657E+01, 0.16737E+01,
     & 0.16818E+01, 0.16899E+01, 0.16979E+01, 0.17059E+01, 0.17139E+01,
     & 0.17219E+01, 0.17299E+01, 0.17378E+01, 0.17457E+01, 0.17536E+01,
     & 0.17615E+01, 0.17694E+01, 0.17773E+01, 0.17851E+01, 0.17930E+01,
     & 0.18008E+01, 0.18086E+01, 0.18164E+01, 0.18241E+01, 0.18319E+01,
     & 0.18396E+01, 0.18473E+01, 0.18550E+01, 0.18627E+01, 0.18703E+01,
     & 0.18780E+01, 0.18856E+01, 0.18932E+01, 0.19008E+01, 0.19084E+01,
     & 0.19160E+01, 0.19235E+01, 0.19311E+01, 0.19386E+01, 0.19461E+01,
     & 0.19536E+01, 0.19611E+01, 0.19685E+01, 0.19759E+01, 0.19834E+01,
     & 0.19908E+01, 0.19982E+01, 0.20055E+01, 0.20129E+01, 0.20203E+01,
     & 0.20276E+01, 0.20349E+01, 0.20422E+01, 0.20495E+01, 0.20568E+01,
     & 0.20640E+01, 0.20713E+01, 0.20785E+01, 0.20857E+01, 0.20929E+01,
     & 0.21001E+01, 0.21072E+01, 0.21144E+01, 0.21215E+01, 0.21286E+01,
     & 0.21357E+01, 0.21428E+01, 0.21499E+01, 0.21570E+01, 0.21640E+01,
     & 0.21710E+01, 0.21781E+01, 0.21851E+01, 0.21921E+01, 0.21990E+01,
     & 0.22060E+01, 0.22129E+01, 0.22199E+01, 0.22268E+01, 0.22337E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.22406E+01, 0.22475E+01, 0.22543E+01, 0.22612E+01, 0.22680E+01,
     & 0.22748E+01, 0.22816E+01, 0.22884E+01, 0.22952E+01, 0.23020E+01,
     & 0.23087E+01, 0.23155E+01, 0.23222E+01, 0.23289E+01, 0.23356E+01,
     & 0.23423E+01, 0.23490E+01, 0.23556E+01, 0.23623E+01, 0.23689E+01,
     & 0.23755E+01, 0.23821E+01, 0.23887E+01, 0.23953E+01, 0.24019E+01,
     & 0.24084E+01, 0.24149E+01, 0.24215E+01, 0.24280E+01, 0.24345E+01,
     & 0.24410E+01, 0.24475E+01, 0.24539E+01, 0.24604E+01, 0.24668E+01,
     & 0.24732E+01, 0.24796E+01, 0.24860E+01, 0.24924E+01, 0.24988E+01,
     & 0.25052E+01, 0.25115E+01, 0.25179E+01, 0.25242E+01, 0.25305E+01,
     & 0.25368E+01, 0.25431E+01, 0.25494E+01, 0.25557E+01, 0.25619E+01,
     & 0.25681E+01, 0.25744E+01, 0.25806E+01, 0.25868E+01, 0.25930E+01,
     & 0.25992E+01, 0.26054E+01, 0.26115E+01, 0.26177E+01, 0.26238E+01,
     & 0.26299E+01, 0.26360E+01, 0.26421E+01, 0.26482E+01, 0.26543E+01,
     & 0.26604E+01, 0.26664E+01, 0.26725E+01, 0.26785E+01, 0.26845E+01,
     & 0.26906E+01, 0.26966E+01, 0.27025E+01, 0.27085E+01, 0.27145E+01,
     & 0.27204E+01, 0.27264E+01, 0.27323E+01, 0.27382E+01, 0.27442E+01,
     & 0.27501E+01, 0.27560E+01, 0.27618E+01, 0.27677E+01, 0.27736E+01,
     & 0.27794E+01, 0.27852E+01, 0.27911E+01, 0.27969E+01, 0.28027E+01,
     & 0.28085E+01, 0.28143E+01, 0.28200E+01, 0.28258E+01, 0.28316E+01,
     & 0.28373E+01, 0.28430E+01, 0.28488E+01, 0.28545E+01, 0.28602E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.28659E+01, 0.28715E+01, 0.28772E+01, 0.28829E+01, 0.28885E+01,
     & 0.28942E+01, 0.28998E+01, 0.29054E+01, 0.29110E+01, 0.29166E+01,
     & 0.29222E+01, 0.29278E+01, 0.29334E+01, 0.29389E+01, 0.29445E+01,
     & 0.29500E+01, 0.29555E+01, 0.29611E+01, 0.29666E+01, 0.29721E+01,
     & 0.29776E+01, 0.29831E+01, 0.29885E+01, 0.29940E+01, 0.29995E+01,
     & 0.30049E+01, 0.30103E+01, 0.30158E+01, 0.30212E+01, 0.30266E+01,
     & 0.30320E+01, 0.30374E+01, 0.30428E+01, 0.30481E+01, 0.30535E+01,
     & 0.30588E+01, 0.30642E+01, 0.30695E+01, 0.30748E+01, 0.30802E+01,
     & 0.30855E+01, 0.30908E+01, 0.30961E+01, 0.31013E+01, 0.31066E+01,
     & 0.31119E+01, 0.31171E+01, 0.31224E+01, 0.31276E+01, 0.31328E+01,
     & 0.31380E+01, 0.31433E+01, 0.31485E+01, 0.31536E+01, 0.31588E+01,
     & 0.31640E+01, 0.31692E+01, 0.31743E+01, 0.31795E+01, 0.31846E+01,
     & 0.31897E+01, 0.31949E+01, 0.32000E+01, 0.32051E+01, 0.32102E+01,
     & 0.32153E+01, 0.32204E+01, 0.32254E+01, 0.32305E+01, 0.32356E+01,
     & 0.32406E+01, 0.32456E+01, 0.32507E+01, 0.32557E+01, 0.32607E+01,
     & 0.32657E+01, 0.32707E+01, 0.32757E+01, 0.32807E+01, 0.32857E+01,
     & 0.32906E+01, 0.32956E+01, 0.33005E+01, 0.33055E+01, 0.33104E+01,
     & 0.33153E+01, 0.33202E+01, 0.33252E+01, 0.33301E+01, 0.33350E+01,
     & 0.33398E+01, 0.33447E+01, 0.33496E+01, 0.33545E+01, 0.33593E+01,
     & 0.33642E+01, 0.33690E+01, 0.33738E+01, 0.33787E+01, 0.33835E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.33883E+01, 0.33931E+01, 0.33979E+01, 0.34027E+01, 0.34074E+01,
     & 0.34122E+01, 0.34170E+01, 0.34217E+01, 0.34265E+01, 0.34312E+01,
     & 0.34360E+01, 0.34407E+01, 0.34454E+01, 0.34501E+01, 0.34548E+01,
     & 0.34595E+01, 0.34642E+01, 0.34689E+01, 0.34736E+01, 0.34782E+01,
     & 0.34829E+01, 0.34876E+01, 0.34922E+01, 0.34968E+01, 0.35015E+01,
     & 0.35061E+01, 0.35107E+01, 0.35153E+01, 0.35199E+01, 0.35245E+01,
     & 0.35291E+01, 0.35337E+01, 0.35383E+01, 0.35428E+01, 0.35474E+01,
     & 0.35520E+01, 0.35565E+01, 0.35611E+01, 0.35656E+01, 0.35701E+01,
     & 0.35746E+01, 0.35792E+01, 0.35837E+01, 0.35882E+01, 0.35927E+01,
     & 0.35971E+01, 0.36016E+01, 0.36061E+01, 0.36106E+01, 0.36150E+01,
     & 0.36195E+01, 0.36239E+01, 0.36284E+01, 0.36328E+01, 0.36372E+01,
     & 0.36416E+01, 0.36461E+01, 0.36505E+01, 0.36549E+01, 0.36593E+01,
     & 0.36636E+01, 0.36680E+01, 0.36724E+01, 0.36768E+01, 0.36811E+01,
     & 0.36855E+01, 0.36898E+01, 0.36942E+01, 0.36985E+01, 0.37028E+01,
     & 0.37072E+01, 0.37115E+01, 0.37158E+01, 0.37201E+01, 0.37244E+01,
     & 0.37287E+01, 0.37330E+01, 0.37373E+01, 0.37415E+01, 0.37458E+01,
     & 0.37501E+01, 0.37543E+01, 0.37586E+01, 0.37628E+01, 0.37670E+01,
     & 0.37713E+01, 0.37755E+01, 0.37797E+01, 0.37839E+01, 0.37881E+01,
     & 0.37923E+01, 0.37965E+01, 0.38007E+01, 0.38049E+01, 0.38091E+01,
     & 0.38133E+01, 0.38174E+01, 0.38216E+01, 0.38257E+01, 0.38299E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.38743E+01, 0.39148E+01, 0.39548E+01, 0.39942E+01, 0.40331E+01,
     & 0.40715E+01, 0.41094E+01, 0.41467E+01, 0.41836E+01, 0.42200E+01,
     & 0.42560E+01, 0.42915E+01, 0.43265E+01, 0.43611E+01, 0.43953E+01,
     & 0.44291E+01, 0.44624E+01, 0.44954E+01, 0.45280E+01, 0.45601E+01,
     & 0.45919E+01, 0.46234E+01, 0.46544E+01, 0.46852E+01, 0.47155E+01,
     & 0.47456E+01, 0.47752E+01, 0.48046E+01, 0.48336E+01, 0.48624E+01,
     & 0.48908E+01, 0.49189E+01, 0.49467E+01, 0.49742E+01, 0.50014E+01,
     & 0.50283E+01, 0.50550E+01, 0.50814E+01, 0.51075E+01, 0.51333E+01,
     & 0.51589E+01, 0.51842E+01, 0.52093E+01, 0.52341E+01, 0.52587E+01,
     & 0.52830E+01, 0.53071E+01, 0.53309E+01, 0.53546E+01, 0.53780E+01,
     & 0.54011E+01, 0.54241E+01, 0.54468E+01, 0.54694E+01, 0.54917E+01,
     & 0.55138E+01, 0.55357E+01, 0.55574E+01, 0.55789E+01, 0.56002E+01,
     & 0.56214E+01, 0.56423E+01, 0.56630E+01, 0.56836E+01, 0.57040E+01,
     & 0.57242E+01, 0.57442E+01, 0.57640E+01, 0.57837E+01, 0.58032E+01,
     & 0.58226E+01, 0.58417E+01, 0.58608E+01, 0.58796E+01, 0.58983E+01,
     & 0.59168E+01, 0.59352E+01, 0.59534E+01, 0.59715E+01, 0.59894E+01,
     & 0.60072E+01, 0.60248E+01, 0.60423E+01, 0.60597E+01, 0.60769E+01,
     & 0.60940E+01, 0.61109E+01, 0.61277E+01, 0.61444E+01, 0.61609E+01,
     & 0.61773E+01, 0.61936E+01, 0.62097E+01, 0.62257E+01, 0.62416E+01,
     & 0.62574E+01, 0.62731E+01, 0.62886E+01, 0.63040E+01, 0.63193E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.63345E+01, 0.63496E+01, 0.63645E+01, 0.63794E+01, 0.63941E+01,
     & 0.64087E+01, 0.64232E+01, 0.64376E+01, 0.64519E+01, 0.64661E+01,
     & 0.64802E+01, 0.64942E+01, 0.65081E+01, 0.65218E+01, 0.65355E+01,
     & 0.65491E+01, 0.65626E+01, 0.65760E+01, 0.65893E+01, 0.66025E+01,
     & 0.66156E+01, 0.66286E+01, 0.66415E+01, 0.66543E+01, 0.66671E+01,
     & 0.66797E+01, 0.66923E+01, 0.67047E+01, 0.67171E+01, 0.67294E+01,
     & 0.67416E+01, 0.67538E+01, 0.67658E+01, 0.67778E+01, 0.67897E+01,
     & 0.68015E+01, 0.68132E+01, 0.68248E+01, 0.68364E+01, 0.68479E+01,
     & 0.68593E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.49197E-01,-0.10142E+00,-0.12478E+00,-0.13991E+00,-0.15083E+00,
     &-0.15912E+00,-0.16560E+00,-0.17074E+00,-0.17485E+00,-0.17812E+00,
     &-0.18070E+00,-0.18270E+00,-0.18421E+00,-0.18528E+00,-0.18597E+00,
     &-0.18631E+00,-0.18635E+00,-0.18610E+00,-0.18560E+00,-0.18485E+00,
     &-0.18389E+00,-0.18272E+00,-0.18136E+00,-0.17982E+00,-0.17811E+00,
     &-0.17624E+00,-0.17421E+00,-0.17204E+00,-0.16973E+00,-0.16730E+00,
     &-0.16473E+00,-0.16205E+00,-0.15925E+00,-0.15634E+00,-0.15333E+00,
     &-0.15021E+00,-0.14700E+00,-0.14370E+00,-0.14031E+00,-0.13683E+00,
     &-0.13327E+00,-0.12964E+00,-0.12592E+00,-0.12214E+00,-0.11829E+00,
     &-0.11437E+00,-0.11038E+00,-0.10634E+00,-0.10223E+00,-0.98072E-01,
     &-0.93855E-01,-0.89586E-01,-0.85265E-01,-0.80895E-01,-0.76477E-01,
     &-0.72014E-01,-0.67505E-01,-0.62952E-01,-0.58357E-01,-0.53721E-01,
     &-0.49044E-01,-0.44327E-01,-0.39571E-01,-0.34777E-01,-0.29944E-01,
     &-0.25074E-01,-0.20167E-01,-0.15222E-01,-0.10240E-01,-0.52212E-02,
     &-0.16537E-03, 0.49275E-02, 0.10057E-01, 0.15225E-01, 0.20429E-01,
     & 0.25672E-01, 0.30952E-01, 0.36270E-01, 0.41626E-01, 0.47021E-01,
     & 0.52455E-01, 0.57927E-01, 0.63438E-01, 0.68988E-01, 0.74576E-01,
     & 0.80203E-01, 0.85868E-01, 0.91572E-01, 0.97314E-01, 0.10309E+00,
     & 0.10891E+00, 0.11476E+00, 0.12065E+00, 0.12657E+00, 0.13253E+00,
     & 0.13852E+00, 0.14455E+00, 0.15060E+00, 0.15669E+00, 0.16280E+00/

      DATA (BNC12M(I),I=101,200)/
     & 0.16895E+00, 0.17512E+00, 0.18131E+00, 0.18753E+00, 0.19377E+00,
     & 0.20004E+00, 0.20632E+00, 0.21262E+00, 0.21894E+00, 0.22528E+00,
     & 0.23163E+00, 0.23799E+00, 0.24437E+00, 0.25075E+00, 0.25715E+00,
     & 0.26355E+00, 0.26996E+00, 0.27637E+00, 0.28279E+00, 0.28921E+00,
     & 0.29563E+00, 0.30206E+00, 0.30848E+00, 0.31491E+00, 0.32133E+00,
     & 0.32775E+00, 0.33417E+00, 0.34058E+00, 0.34699E+00, 0.35339E+00,
     & 0.35979E+00, 0.36618E+00, 0.37256E+00, 0.37893E+00, 0.38530E+00,
     & 0.39166E+00, 0.39801E+00, 0.40435E+00, 0.41068E+00, 0.41700E+00,
     & 0.42331E+00, 0.42961E+00, 0.43590E+00, 0.44218E+00, 0.44844E+00,
     & 0.45470E+00, 0.46094E+00, 0.46717E+00, 0.47339E+00, 0.47960E+00,
     & 0.48580E+00, 0.49198E+00, 0.49815E+00, 0.50431E+00, 0.51046E+00,
     & 0.51659E+00, 0.52271E+00, 0.52882E+00, 0.53491E+00, 0.54099E+00,
     & 0.54706E+00, 0.55312E+00, 0.55916E+00, 0.56519E+00, 0.57121E+00,
     & 0.57721E+00, 0.58320E+00, 0.58918E+00, 0.59514E+00, 0.60109E+00,
     & 0.60703E+00, 0.61295E+00, 0.61886E+00, 0.62476E+00, 0.63065E+00,
     & 0.63652E+00, 0.64237E+00, 0.64822E+00, 0.65405E+00, 0.65987E+00,
     & 0.66567E+00, 0.67146E+00, 0.67724E+00, 0.68301E+00, 0.68876E+00,
     & 0.69450E+00, 0.70022E+00, 0.70593E+00, 0.71163E+00, 0.71732E+00,
     & 0.72299E+00, 0.72865E+00, 0.73430E+00, 0.73993E+00, 0.74555E+00,
     & 0.75116E+00, 0.75675E+00, 0.76233E+00, 0.76790E+00, 0.77346E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.77900E+00, 0.78453E+00, 0.79005E+00, 0.79555E+00, 0.80105E+00,
     & 0.80653E+00, 0.81199E+00, 0.81745E+00, 0.82289E+00, 0.82832E+00,
     & 0.83373E+00, 0.83914E+00, 0.84453E+00, 0.84991E+00, 0.85528E+00,
     & 0.86063E+00, 0.86597E+00, 0.87130E+00, 0.87662E+00, 0.88193E+00,
     & 0.88722E+00, 0.89250E+00, 0.89777E+00, 0.90303E+00, 0.90827E+00,
     & 0.91351E+00, 0.91873E+00, 0.92394E+00, 0.92914E+00, 0.93432E+00,
     & 0.93950E+00, 0.94466E+00, 0.94981E+00, 0.95495E+00, 0.96008E+00,
     & 0.96519E+00, 0.97030E+00, 0.97539E+00, 0.98047E+00, 0.98554E+00,
     & 0.99060E+00, 0.99565E+00, 0.10007E+01, 0.10057E+01, 0.10107E+01,
     & 0.10157E+01, 0.10207E+01, 0.10257E+01, 0.10307E+01, 0.10356E+01,
     & 0.10406E+01, 0.10455E+01, 0.10504E+01, 0.10554E+01, 0.10603E+01,
     & 0.10652E+01, 0.10700E+01, 0.10749E+01, 0.10798E+01, 0.10846E+01,
     & 0.10895E+01, 0.10943E+01, 0.10991E+01, 0.11039E+01, 0.11087E+01,
     & 0.11135E+01, 0.11183E+01, 0.11230E+01, 0.11278E+01, 0.11326E+01,
     & 0.11373E+01, 0.11420E+01, 0.11467E+01, 0.11514E+01, 0.11561E+01,
     & 0.11608E+01, 0.11655E+01, 0.11702E+01, 0.11748E+01, 0.11795E+01,
     & 0.11841E+01, 0.11887E+01, 0.11933E+01, 0.11980E+01, 0.12025E+01,
     & 0.12071E+01, 0.12117E+01, 0.12163E+01, 0.12208E+01, 0.12254E+01,
     & 0.12299E+01, 0.12345E+01, 0.12390E+01, 0.12435E+01, 0.12480E+01,
     & 0.12525E+01, 0.12570E+01, 0.12614E+01, 0.12659E+01, 0.12704E+01/

      DATA (BNC12M(I),I=301,400)/
     & 0.12748E+01, 0.12792E+01, 0.12837E+01, 0.12881E+01, 0.12925E+01,
     & 0.12969E+01, 0.13013E+01, 0.13057E+01, 0.13101E+01, 0.13144E+01,
     & 0.13188E+01, 0.13231E+01, 0.13275E+01, 0.13318E+01, 0.13361E+01,
     & 0.13404E+01, 0.13447E+01, 0.13490E+01, 0.13533E+01, 0.13576E+01,
     & 0.13619E+01, 0.13661E+01, 0.13704E+01, 0.13746E+01, 0.13788E+01,
     & 0.13831E+01, 0.13873E+01, 0.13915E+01, 0.13957E+01, 0.13999E+01,
     & 0.14041E+01, 0.14083E+01, 0.14124E+01, 0.14166E+01, 0.14207E+01,
     & 0.14249E+01, 0.14290E+01, 0.14331E+01, 0.14373E+01, 0.14414E+01,
     & 0.14455E+01, 0.14496E+01, 0.14537E+01, 0.14577E+01, 0.14618E+01,
     & 0.14659E+01, 0.14699E+01, 0.14740E+01, 0.14780E+01, 0.14820E+01,
     & 0.14861E+01, 0.14901E+01, 0.14941E+01, 0.14981E+01, 0.15021E+01,
     & 0.15061E+01, 0.15101E+01, 0.15140E+01, 0.15180E+01, 0.15219E+01,
     & 0.15259E+01, 0.15298E+01, 0.15338E+01, 0.15377E+01, 0.15416E+01,
     & 0.15455E+01, 0.15494E+01, 0.15533E+01, 0.15572E+01, 0.15611E+01,
     & 0.15650E+01, 0.15688E+01, 0.15727E+01, 0.15765E+01, 0.15804E+01,
     & 0.15842E+01, 0.15881E+01, 0.15919E+01, 0.15957E+01, 0.15995E+01,
     & 0.16033E+01, 0.16071E+01, 0.16109E+01, 0.16147E+01, 0.16185E+01,
     & 0.16222E+01, 0.16260E+01, 0.16297E+01, 0.16335E+01, 0.16372E+01,
     & 0.16410E+01, 0.16447E+01, 0.16484E+01, 0.16521E+01, 0.16558E+01,
     & 0.16595E+01, 0.16632E+01, 0.16669E+01, 0.16706E+01, 0.16743E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.16779E+01, 0.16816E+01, 0.16852E+01, 0.16889E+01, 0.16925E+01,
     & 0.16962E+01, 0.16998E+01, 0.17034E+01, 0.17070E+01, 0.17106E+01,
     & 0.17142E+01, 0.17178E+01, 0.17214E+01, 0.17250E+01, 0.17286E+01,
     & 0.17321E+01, 0.17357E+01, 0.17393E+01, 0.17428E+01, 0.17464E+01,
     & 0.17499E+01, 0.17534E+01, 0.17569E+01, 0.17605E+01, 0.17640E+01,
     & 0.17675E+01, 0.17710E+01, 0.17745E+01, 0.17780E+01, 0.17815E+01,
     & 0.17849E+01, 0.17884E+01, 0.17919E+01, 0.17953E+01, 0.17988E+01,
     & 0.18022E+01, 0.18057E+01, 0.18091E+01, 0.18125E+01, 0.18159E+01,
     & 0.18194E+01, 0.18228E+01, 0.18262E+01, 0.18296E+01, 0.18330E+01,
     & 0.18364E+01, 0.18397E+01, 0.18431E+01, 0.18465E+01, 0.18498E+01,
     & 0.18532E+01, 0.18566E+01, 0.18599E+01, 0.18632E+01, 0.18666E+01,
     & 0.18699E+01, 0.18732E+01, 0.18766E+01, 0.18799E+01, 0.18832E+01,
     & 0.18865E+01, 0.18898E+01, 0.18931E+01, 0.18964E+01, 0.18996E+01,
     & 0.19029E+01, 0.19062E+01, 0.19094E+01, 0.19127E+01, 0.19160E+01,
     & 0.19192E+01, 0.19224E+01, 0.19257E+01, 0.19289E+01, 0.19321E+01,
     & 0.19354E+01, 0.19386E+01, 0.19418E+01, 0.19450E+01, 0.19482E+01,
     & 0.19514E+01, 0.19546E+01, 0.19577E+01, 0.19609E+01, 0.19641E+01,
     & 0.19673E+01, 0.19704E+01, 0.19736E+01, 0.19767E+01, 0.19799E+01,
     & 0.19830E+01, 0.19862E+01, 0.19893E+01, 0.19924E+01, 0.19955E+01,
     & 0.19987E+01, 0.20018E+01, 0.20049E+01, 0.20080E+01, 0.20111E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.20142E+01, 0.20173E+01, 0.20203E+01, 0.20234E+01, 0.20265E+01,
     & 0.20295E+01, 0.20326E+01, 0.20357E+01, 0.20387E+01, 0.20418E+01,
     & 0.20448E+01, 0.20478E+01, 0.20509E+01, 0.20539E+01, 0.20569E+01,
     & 0.20599E+01, 0.20630E+01, 0.20660E+01, 0.20690E+01, 0.20720E+01,
     & 0.20750E+01, 0.20780E+01, 0.20809E+01, 0.20839E+01, 0.20869E+01,
     & 0.20899E+01, 0.20928E+01, 0.20958E+01, 0.20987E+01, 0.21017E+01,
     & 0.21046E+01, 0.21076E+01, 0.21105E+01, 0.21135E+01, 0.21164E+01,
     & 0.21193E+01, 0.21222E+01, 0.21251E+01, 0.21281E+01, 0.21310E+01,
     & 0.21339E+01, 0.21368E+01, 0.21397E+01, 0.21425E+01, 0.21454E+01,
     & 0.21483E+01, 0.21512E+01, 0.21541E+01, 0.21569E+01, 0.21598E+01,
     & 0.21626E+01, 0.21655E+01, 0.21683E+01, 0.21712E+01, 0.21740E+01,
     & 0.21769E+01, 0.21797E+01, 0.21825E+01, 0.21853E+01, 0.21882E+01,
     & 0.21910E+01, 0.21938E+01, 0.21966E+01, 0.21994E+01, 0.22022E+01,
     & 0.22050E+01, 0.22078E+01, 0.22106E+01, 0.22133E+01, 0.22161E+01,
     & 0.22189E+01, 0.22216E+01, 0.22244E+01, 0.22272E+01, 0.22299E+01,
     & 0.22327E+01, 0.22354E+01, 0.22382E+01, 0.22409E+01, 0.22436E+01,
     & 0.22464E+01, 0.22491E+01, 0.22518E+01, 0.22545E+01, 0.22572E+01,
     & 0.22600E+01, 0.22627E+01, 0.22654E+01, 0.22681E+01, 0.22708E+01,
     & 0.22734E+01, 0.22761E+01, 0.22788E+01, 0.22815E+01, 0.22842E+01,
     & 0.22868E+01, 0.22895E+01, 0.22922E+01, 0.22948E+01, 0.22975E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.23259E+01, 0.23518E+01, 0.23774E+01, 0.24026E+01, 0.24274E+01,
     & 0.24519E+01, 0.24761E+01, 0.24999E+01, 0.25234E+01, 0.25466E+01,
     & 0.25694E+01, 0.25920E+01, 0.26142E+01, 0.26362E+01, 0.26579E+01,
     & 0.26793E+01, 0.27004E+01, 0.27213E+01, 0.27419E+01, 0.27622E+01,
     & 0.27823E+01, 0.28022E+01, 0.28218E+01, 0.28411E+01, 0.28602E+01,
     & 0.28791E+01, 0.28978E+01, 0.29162E+01, 0.29344E+01, 0.29524E+01,
     & 0.29702E+01, 0.29878E+01, 0.30052E+01, 0.30224E+01, 0.30393E+01,
     & 0.30561E+01, 0.30727E+01, 0.30891E+01, 0.31054E+01, 0.31214E+01,
     & 0.31373E+01, 0.31530E+01, 0.31685E+01, 0.31838E+01, 0.31990E+01,
     & 0.32140E+01, 0.32289E+01, 0.32436E+01, 0.32581E+01, 0.32725E+01,
     & 0.32868E+01, 0.33008E+01, 0.33148E+01, 0.33286E+01, 0.33422E+01,
     & 0.33557E+01, 0.33691E+01, 0.33823E+01, 0.33954E+01, 0.34084E+01,
     & 0.34212E+01, 0.34339E+01, 0.34465E+01, 0.34589E+01, 0.34712E+01,
     & 0.34834E+01, 0.34955E+01, 0.35075E+01, 0.35193E+01, 0.35310E+01,
     & 0.35426E+01, 0.35541E+01, 0.35655E+01, 0.35768E+01, 0.35880E+01,
     & 0.35990E+01, 0.36100E+01, 0.36208E+01, 0.36316E+01, 0.36422E+01,
     & 0.36527E+01, 0.36632E+01, 0.36735E+01, 0.36838E+01, 0.36939E+01,
     & 0.37040E+01, 0.37139E+01, 0.37238E+01, 0.37336E+01, 0.37432E+01,
     & 0.37528E+01, 0.37623E+01, 0.37718E+01, 0.37811E+01, 0.37903E+01,
     & 0.37995E+01, 0.38086E+01, 0.38176E+01, 0.38265E+01, 0.38353E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.38441E+01, 0.38528E+01, 0.38614E+01, 0.38699E+01, 0.38783E+01,
     & 0.38867E+01, 0.38950E+01, 0.39032E+01, 0.39113E+01, 0.39194E+01,
     & 0.39274E+01, 0.39353E+01, 0.39432E+01, 0.39510E+01, 0.39587E+01,
     & 0.39664E+01, 0.39740E+01, 0.39815E+01, 0.39889E+01, 0.39963E+01,
     & 0.40037E+01, 0.40109E+01, 0.40181E+01, 0.40253E+01, 0.40323E+01,
     & 0.40393E+01, 0.40463E+01, 0.40532E+01, 0.40600E+01, 0.40668E+01,
     & 0.40735E+01, 0.40802E+01, 0.40867E+01, 0.40933E+01, 0.40998E+01,
     & 0.41062E+01, 0.41126E+01, 0.41189E+01, 0.41252E+01, 0.41314E+01,
     & 0.41375E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.81873E-01,-0.17829E+00,-0.22674E+00,-0.26117E+00,-0.28833E+00,
     &-0.31093E+00,-0.33037E+00,-0.34746E+00,-0.36275E+00,-0.37660E+00,
     &-0.38926E+00,-0.40094E+00,-0.41177E+00,-0.42187E+00,-0.43134E+00,
     &-0.44025E+00,-0.44867E+00,-0.45663E+00,-0.46420E+00,-0.47139E+00,
     &-0.47826E+00,-0.48481E+00,-0.49108E+00,-0.49709E+00,-0.50285E+00,
     &-0.50838E+00,-0.51370E+00,-0.51882E+00,-0.52375E+00,-0.52851E+00,
     &-0.53309E+00,-0.53751E+00,-0.54179E+00,-0.54591E+00,-0.54990E+00,
     &-0.55376E+00,-0.55750E+00,-0.56111E+00,-0.56461E+00,-0.56800E+00,
     &-0.57129E+00,-0.57447E+00,-0.57756E+00,-0.58056E+00,-0.58346E+00,
     &-0.58628E+00,-0.58902E+00,-0.59167E+00,-0.59425E+00,-0.59676E+00,
     &-0.59919E+00,-0.60156E+00,-0.60386E+00,-0.60609E+00,-0.60827E+00,
     &-0.61038E+00,-0.61243E+00,-0.61443E+00,-0.61638E+00,-0.61827E+00,
     &-0.62011E+00,-0.62191E+00,-0.62365E+00,-0.62535E+00,-0.62701E+00,
     &-0.62862E+00,-0.63019E+00,-0.63172E+00,-0.63321E+00,-0.63466E+00,
     &-0.63607E+00,-0.63745E+00,-0.63879E+00,-0.64009E+00,-0.64136E+00,
     &-0.64260E+00,-0.64381E+00,-0.64498E+00,-0.64613E+00,-0.64724E+00,
     &-0.64832E+00,-0.64937E+00,-0.65040E+00,-0.65140E+00,-0.65237E+00,
     &-0.65331E+00,-0.65422E+00,-0.65512E+00,-0.65598E+00,-0.65682E+00,
     &-0.65764E+00,-0.65843E+00,-0.65920E+00,-0.65995E+00,-0.66068E+00,
     &-0.66138E+00,-0.66206E+00,-0.66273E+00,-0.66337E+00,-0.66399E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.66460E+00,-0.66519E+00,-0.66576E+00,-0.66631E+00,-0.66685E+00,
     &-0.66737E+00,-0.66787E+00,-0.66836E+00,-0.66884E+00,-0.66930E+00,
     &-0.66975E+00,-0.67019E+00,-0.67062E+00,-0.67103E+00,-0.67143E+00,
     &-0.67182E+00,-0.67220E+00,-0.67258E+00,-0.67294E+00,-0.67329E+00,
     &-0.67364E+00,-0.67397E+00,-0.67430E+00,-0.67462E+00,-0.67494E+00,
     &-0.67524E+00,-0.67555E+00,-0.67584E+00,-0.67613E+00,-0.67641E+00,
     &-0.67669E+00,-0.67696E+00,-0.67723E+00,-0.67750E+00,-0.67775E+00,
     &-0.67801E+00,-0.67826E+00,-0.67851E+00,-0.67875E+00,-0.67899E+00,
     &-0.67923E+00,-0.67946E+00,-0.67969E+00,-0.67992E+00,-0.68015E+00,
     &-0.68037E+00,-0.68059E+00,-0.68081E+00,-0.68103E+00,-0.68124E+00,
     &-0.68145E+00,-0.68166E+00,-0.68187E+00,-0.68208E+00,-0.68228E+00,
     &-0.68249E+00,-0.68269E+00,-0.68289E+00,-0.68309E+00,-0.68329E+00,
     &-0.68349E+00,-0.68368E+00,-0.68388E+00,-0.68408E+00,-0.68427E+00,
     &-0.68446E+00,-0.68465E+00,-0.68485E+00,-0.68504E+00,-0.68523E+00,
     &-0.68542E+00,-0.68561E+00,-0.68580E+00,-0.68599E+00,-0.68618E+00,
     &-0.68636E+00,-0.68655E+00,-0.68674E+00,-0.68693E+00,-0.68712E+00,
     &-0.68730E+00,-0.68749E+00,-0.68768E+00,-0.68787E+00,-0.68805E+00,
     &-0.68824E+00,-0.68843E+00,-0.68862E+00,-0.68880E+00,-0.68899E+00,
     &-0.68918E+00,-0.68937E+00,-0.68956E+00,-0.68975E+00,-0.68994E+00,
     &-0.69013E+00,-0.69032E+00,-0.69051E+00,-0.69070E+00,-0.69089E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.69108E+00,-0.69128E+00,-0.69147E+00,-0.69166E+00,-0.69186E+00,
     &-0.69205E+00,-0.69225E+00,-0.69244E+00,-0.69264E+00,-0.69283E+00,
     &-0.69303E+00,-0.69323E+00,-0.69343E+00,-0.69363E+00,-0.69383E+00,
     &-0.69403E+00,-0.69423E+00,-0.69443E+00,-0.69463E+00,-0.69484E+00,
     &-0.69504E+00,-0.69524E+00,-0.69545E+00,-0.69566E+00,-0.69586E+00,
     &-0.69607E+00,-0.69628E+00,-0.69649E+00,-0.69670E+00,-0.69691E+00,
     &-0.69712E+00,-0.69733E+00,-0.69754E+00,-0.69776E+00,-0.69797E+00,
     &-0.69818E+00,-0.69840E+00,-0.69862E+00,-0.69883E+00,-0.69905E+00,
     &-0.69927E+00,-0.69949E+00,-0.69971E+00,-0.69993E+00,-0.70016E+00,
     &-0.70038E+00,-0.70060E+00,-0.70083E+00,-0.70106E+00,-0.70128E+00,
     &-0.70151E+00,-0.70174E+00,-0.70197E+00,-0.70220E+00,-0.70243E+00,
     &-0.70266E+00,-0.70289E+00,-0.70313E+00,-0.70336E+00,-0.70360E+00,
     &-0.70383E+00,-0.70407E+00,-0.70431E+00,-0.70455E+00,-0.70479E+00,
     &-0.70503E+00,-0.70527E+00,-0.70551E+00,-0.70575E+00,-0.70600E+00,
     &-0.70624E+00,-0.70649E+00,-0.70674E+00,-0.70698E+00,-0.70723E+00,
     &-0.70748E+00,-0.70773E+00,-0.70798E+00,-0.70824E+00,-0.70849E+00,
     &-0.70874E+00,-0.70900E+00,-0.70925E+00,-0.70951E+00,-0.70977E+00,
     &-0.71002E+00,-0.71028E+00,-0.71054E+00,-0.71080E+00,-0.71107E+00,
     &-0.71133E+00,-0.71159E+00,-0.71186E+00,-0.71212E+00,-0.71239E+00,
     &-0.71266E+00,-0.71292E+00,-0.71319E+00,-0.71346E+00,-0.71373E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.71401E+00,-0.71428E+00,-0.71455E+00,-0.71483E+00,-0.71510E+00,
     &-0.71538E+00,-0.71565E+00,-0.71593E+00,-0.71621E+00,-0.71649E+00,
     &-0.71677E+00,-0.71705E+00,-0.71733E+00,-0.71762E+00,-0.71790E+00,
     &-0.71818E+00,-0.71847E+00,-0.71876E+00,-0.71904E+00,-0.71933E+00,
     &-0.71962E+00,-0.71991E+00,-0.72020E+00,-0.72049E+00,-0.72079E+00,
     &-0.72108E+00,-0.72137E+00,-0.72167E+00,-0.72196E+00,-0.72226E+00,
     &-0.72256E+00,-0.72286E+00,-0.72316E+00,-0.72346E+00,-0.72376E+00,
     &-0.72406E+00,-0.72436E+00,-0.72466E+00,-0.72497E+00,-0.72527E+00,
     &-0.72558E+00,-0.72589E+00,-0.72619E+00,-0.72650E+00,-0.72681E+00,
     &-0.72712E+00,-0.72743E+00,-0.72774E+00,-0.72806E+00,-0.72837E+00,
     &-0.72868E+00,-0.72900E+00,-0.72932E+00,-0.72963E+00,-0.72995E+00,
     &-0.73027E+00,-0.73059E+00,-0.73091E+00,-0.73123E+00,-0.73155E+00,
     &-0.73187E+00,-0.73220E+00,-0.73252E+00,-0.73284E+00,-0.73317E+00,
     &-0.73350E+00,-0.73382E+00,-0.73415E+00,-0.73448E+00,-0.73481E+00,
     &-0.73514E+00,-0.73547E+00,-0.73580E+00,-0.73614E+00,-0.73647E+00,
     &-0.73680E+00,-0.73714E+00,-0.73747E+00,-0.73781E+00,-0.73815E+00,
     &-0.73849E+00,-0.73882E+00,-0.73916E+00,-0.73951E+00,-0.73985E+00,
     &-0.74019E+00,-0.74053E+00,-0.74087E+00,-0.74122E+00,-0.74156E+00,
     &-0.74191E+00,-0.74226E+00,-0.74260E+00,-0.74295E+00,-0.74330E+00,
     &-0.74365E+00,-0.74400E+00,-0.74435E+00,-0.74470E+00,-0.74506E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.74541E+00,-0.74576E+00,-0.74612E+00,-0.74647E+00,-0.74683E+00,
     &-0.74719E+00,-0.74755E+00,-0.74790E+00,-0.74826E+00,-0.74862E+00,
     &-0.74898E+00,-0.74935E+00,-0.74971E+00,-0.75007E+00,-0.75043E+00,
     &-0.75080E+00,-0.75116E+00,-0.75153E+00,-0.75190E+00,-0.75226E+00,
     &-0.75263E+00,-0.75300E+00,-0.75337E+00,-0.75374E+00,-0.75411E+00,
     &-0.75448E+00,-0.75485E+00,-0.75523E+00,-0.75560E+00,-0.75597E+00,
     &-0.75635E+00,-0.75673E+00,-0.75710E+00,-0.75748E+00,-0.75786E+00,
     &-0.75824E+00,-0.75861E+00,-0.75899E+00,-0.75937E+00,-0.75976E+00,
     &-0.76014E+00,-0.76052E+00,-0.76090E+00,-0.76129E+00,-0.76167E+00,
     &-0.76206E+00,-0.76244E+00,-0.76283E+00,-0.76322E+00,-0.76361E+00,
     &-0.76399E+00,-0.76438E+00,-0.76477E+00,-0.76517E+00,-0.76556E+00,
     &-0.76595E+00,-0.76634E+00,-0.76673E+00,-0.76713E+00,-0.76752E+00,
     &-0.76792E+00,-0.76831E+00,-0.76871E+00,-0.76911E+00,-0.76951E+00,
     &-0.76991E+00,-0.77030E+00,-0.77070E+00,-0.77111E+00,-0.77151E+00,
     &-0.77191E+00,-0.77231E+00,-0.77271E+00,-0.77312E+00,-0.77352E+00,
     &-0.77393E+00,-0.77433E+00,-0.77474E+00,-0.77515E+00,-0.77555E+00,
     &-0.77596E+00,-0.77637E+00,-0.77678E+00,-0.77719E+00,-0.77760E+00,
     &-0.77801E+00,-0.77842E+00,-0.77884E+00,-0.77925E+00,-0.77966E+00,
     &-0.78008E+00,-0.78049E+00,-0.78091E+00,-0.78133E+00,-0.78174E+00,
     &-0.78216E+00,-0.78258E+00,-0.78300E+00,-0.78342E+00,-0.78384E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.78426E+00,-0.78468E+00,-0.78510E+00,-0.78552E+00,-0.78594E+00,
     &-0.78637E+00,-0.78679E+00,-0.78722E+00,-0.78764E+00,-0.78807E+00,
     &-0.78849E+00,-0.78892E+00,-0.78935E+00,-0.78978E+00,-0.79021E+00,
     &-0.79064E+00,-0.79107E+00,-0.79150E+00,-0.79193E+00,-0.79236E+00,
     &-0.79279E+00,-0.79323E+00,-0.79366E+00,-0.79409E+00,-0.79453E+00,
     &-0.79496E+00,-0.79540E+00,-0.79583E+00,-0.79627E+00,-0.79671E+00,
     &-0.79715E+00,-0.79759E+00,-0.79803E+00,-0.79846E+00,-0.79891E+00,
     &-0.79935E+00,-0.79979E+00,-0.80023E+00,-0.80067E+00,-0.80112E+00,
     &-0.80156E+00,-0.80200E+00,-0.80245E+00,-0.80289E+00,-0.80334E+00,
     &-0.80379E+00,-0.80423E+00,-0.80468E+00,-0.80513E+00,-0.80558E+00,
     &-0.80603E+00,-0.80648E+00,-0.80693E+00,-0.80738E+00,-0.80783E+00,
     &-0.80828E+00,-0.80873E+00,-0.80919E+00,-0.80964E+00,-0.81009E+00,
     &-0.81055E+00,-0.81100E+00,-0.81146E+00,-0.81191E+00,-0.81237E+00,
     &-0.81283E+00,-0.81329E+00,-0.81374E+00,-0.81420E+00,-0.81466E+00,
     &-0.81512E+00,-0.81558E+00,-0.81604E+00,-0.81650E+00,-0.81697E+00,
     &-0.81743E+00,-0.81789E+00,-0.81836E+00,-0.81882E+00,-0.81928E+00,
     &-0.81975E+00,-0.82021E+00,-0.82068E+00,-0.82115E+00,-0.82161E+00,
     &-0.82208E+00,-0.82255E+00,-0.82302E+00,-0.82349E+00,-0.82396E+00,
     &-0.82443E+00,-0.82490E+00,-0.82537E+00,-0.82584E+00,-0.82631E+00,
     &-0.82678E+00,-0.82726E+00,-0.82773E+00,-0.82820E+00,-0.82868E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.83384E+00,-0.83866E+00,-0.84353E+00,-0.84845E+00,-0.85341E+00,
     &-0.85841E+00,-0.86346E+00,-0.86856E+00,-0.87369E+00,-0.87886E+00,
     &-0.88408E+00,-0.88933E+00,-0.89463E+00,-0.89996E+00,-0.90532E+00,
     &-0.91073E+00,-0.91617E+00,-0.92164E+00,-0.92715E+00,-0.93270E+00,
     &-0.93828E+00,-0.94389E+00,-0.94953E+00,-0.95520E+00,-0.96091E+00,
     &-0.96664E+00,-0.97241E+00,-0.97820E+00,-0.98402E+00,-0.98987E+00,
     &-0.99575E+00,-0.10017E+01,-0.10076E+01,-0.10136E+01,-0.10195E+01,
     &-0.10256E+01,-0.10316E+01,-0.10377E+01,-0.10437E+01,-0.10499E+01,
     &-0.10560E+01,-0.10621E+01,-0.10683E+01,-0.10745E+01,-0.10808E+01,
     &-0.10870E+01,-0.10933E+01,-0.10996E+01,-0.11059E+01,-0.11122E+01,
     &-0.11186E+01,-0.11249E+01,-0.11313E+01,-0.11377E+01,-0.11442E+01,
     &-0.11506E+01,-0.11571E+01,-0.11636E+01,-0.11701E+01,-0.11766E+01,
     &-0.11831E+01,-0.11897E+01,-0.11963E+01,-0.12028E+01,-0.12095E+01,
     &-0.12161E+01,-0.12227E+01,-0.12294E+01,-0.12360E+01,-0.12427E+01,
     &-0.12494E+01,-0.12561E+01,-0.12629E+01,-0.12696E+01,-0.12764E+01,
     &-0.12831E+01,-0.12899E+01,-0.12967E+01,-0.13035E+01,-0.13104E+01,
     &-0.13172E+01,-0.13241E+01,-0.13309E+01,-0.13378E+01,-0.13447E+01,
     &-0.13516E+01,-0.13585E+01,-0.13655E+01,-0.13724E+01,-0.13794E+01,
     &-0.13863E+01,-0.13933E+01,-0.14003E+01,-0.14073E+01,-0.14143E+01,
     &-0.14213E+01,-0.14284E+01,-0.14354E+01,-0.14424E+01,-0.14495E+01/

      DATA (BNC13M(I),I=701,741)/
     &-0.14566E+01,-0.14637E+01,-0.14708E+01,-0.14779E+01,-0.14850E+01,
     &-0.14921E+01,-0.14992E+01,-0.15064E+01,-0.15135E+01,-0.15207E+01,
     &-0.15279E+01,-0.15351E+01,-0.15422E+01,-0.15494E+01,-0.15567E+01,
     &-0.15639E+01,-0.15711E+01,-0.15783E+01,-0.15856E+01,-0.15928E+01,
     &-0.16001E+01,-0.16073E+01,-0.16146E+01,-0.16219E+01,-0.16292E+01,
     &-0.16365E+01,-0.16438E+01,-0.16511E+01,-0.16584E+01,-0.16657E+01,
     &-0.16731E+01,-0.16804E+01,-0.16878E+01,-0.16951E+01,-0.17025E+01,
     &-0.17099E+01,-0.17172E+01,-0.17246E+01,-0.17320E+01,-0.17394E+01,
     &-0.17468E+01
     & /
      END


 
 
 
C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF223A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA KMCF223A
C
C *** Common block definition
C
      COMMON /KMC223/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.48562E-01,-0.10092E+00,-0.12448E+00,-0.13974E+00,-0.15075E+00,
     &-0.15911E+00,-0.16566E+00,-0.17088E+00,-0.17509E+00,-0.17849E+00,
     &-0.18124E+00,-0.18345E+00,-0.18522E+00,-0.18659E+00,-0.18764E+00,
     &-0.18840E+00,-0.18891E+00,-0.18920E+00,-0.18930E+00,-0.18922E+00,
     &-0.18898E+00,-0.18861E+00,-0.18811E+00,-0.18750E+00,-0.18678E+00,
     &-0.18598E+00,-0.18509E+00,-0.18412E+00,-0.18309E+00,-0.18199E+00,
     &-0.18084E+00,-0.17964E+00,-0.17839E+00,-0.17709E+00,-0.17576E+00,
     &-0.17440E+00,-0.17300E+00,-0.17158E+00,-0.17013E+00,-0.16865E+00,
     &-0.16716E+00,-0.16565E+00,-0.16412E+00,-0.16258E+00,-0.16102E+00,
     &-0.15946E+00,-0.15788E+00,-0.15629E+00,-0.15470E+00,-0.15310E+00,
     &-0.15149E+00,-0.14988E+00,-0.14826E+00,-0.14664E+00,-0.14501E+00,
     &-0.14338E+00,-0.14174E+00,-0.14010E+00,-0.13846E+00,-0.13681E+00,
     &-0.13516E+00,-0.13350E+00,-0.13184E+00,-0.13017E+00,-0.12850E+00,
     &-0.12682E+00,-0.12514E+00,-0.12344E+00,-0.12174E+00,-0.12003E+00,
     &-0.11832E+00,-0.11659E+00,-0.11485E+00,-0.11311E+00,-0.11135E+00,
     &-0.10958E+00,-0.10780E+00,-0.10600E+00,-0.10420E+00,-0.10237E+00,
     &-0.10054E+00,-0.98688E-01,-0.96823E-01,-0.94943E-01,-0.93048E-01,
     &-0.91136E-01,-0.89209E-01,-0.87266E-01,-0.85307E-01,-0.83332E-01,
     &-0.81341E-01,-0.79334E-01,-0.77312E-01,-0.75274E-01,-0.73220E-01,
     &-0.71151E-01,-0.69068E-01,-0.66970E-01,-0.64858E-01,-0.62732E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.60593E-01,-0.58441E-01,-0.56277E-01,-0.54101E-01,-0.51914E-01,
     &-0.49716E-01,-0.47508E-01,-0.45291E-01,-0.43064E-01,-0.40828E-01,
     &-0.38585E-01,-0.36334E-01,-0.34076E-01,-0.31812E-01,-0.29542E-01,
     &-0.27266E-01,-0.24986E-01,-0.22701E-01,-0.20412E-01,-0.18120E-01,
     &-0.15825E-01,-0.13526E-01,-0.11226E-01,-0.89235E-02,-0.66195E-02,
     &-0.43141E-02,-0.20079E-02, 0.29912E-03, 0.26064E-02, 0.49139E-02,
     & 0.72212E-02, 0.95283E-02, 0.11835E-01, 0.14140E-01, 0.16445E-01,
     & 0.18749E-01, 0.21052E-01, 0.23353E-01, 0.25652E-01, 0.27950E-01,
     & 0.30246E-01, 0.32540E-01, 0.34833E-01, 0.37123E-01, 0.39411E-01,
     & 0.41697E-01, 0.43980E-01, 0.46261E-01, 0.48540E-01, 0.50816E-01,
     & 0.53090E-01, 0.55361E-01, 0.57630E-01, 0.59895E-01, 0.62158E-01,
     & 0.64419E-01, 0.66676E-01, 0.68931E-01, 0.71182E-01, 0.73431E-01,
     & 0.75677E-01, 0.77920E-01, 0.80160E-01, 0.82397E-01, 0.84631E-01,
     & 0.86861E-01, 0.89089E-01, 0.91314E-01, 0.93535E-01, 0.95754E-01,
     & 0.97969E-01, 0.10018E+00, 0.10239E+00, 0.10459E+00, 0.10680E+00,
     & 0.10900E+00, 0.11119E+00, 0.11338E+00, 0.11557E+00, 0.11776E+00,
     & 0.11994E+00, 0.12212E+00, 0.12430E+00, 0.12647E+00, 0.12864E+00,
     & 0.13080E+00, 0.13297E+00, 0.13512E+00, 0.13728E+00, 0.13943E+00,
     & 0.14158E+00, 0.14373E+00, 0.14587E+00, 0.14801E+00, 0.15014E+00,
     & 0.15227E+00, 0.15440E+00, 0.15652E+00, 0.15864E+00, 0.16076E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.16288E+00, 0.16499E+00, 0.16709E+00, 0.16920E+00, 0.17129E+00,
     & 0.17339E+00, 0.17548E+00, 0.17757E+00, 0.17966E+00, 0.18174E+00,
     & 0.18382E+00, 0.18589E+00, 0.18796E+00, 0.19003E+00, 0.19210E+00,
     & 0.19416E+00, 0.19621E+00, 0.19827E+00, 0.20032E+00, 0.20236E+00,
     & 0.20441E+00, 0.20644E+00, 0.20848E+00, 0.21051E+00, 0.21254E+00,
     & 0.21457E+00, 0.21659E+00, 0.21860E+00, 0.22062E+00, 0.22263E+00,
     & 0.22464E+00, 0.22664E+00, 0.22864E+00, 0.23064E+00, 0.23263E+00,
     & 0.23462E+00, 0.23660E+00, 0.23859E+00, 0.24056E+00, 0.24254E+00,
     & 0.24451E+00, 0.24648E+00, 0.24844E+00, 0.25040E+00, 0.25236E+00,
     & 0.25431E+00, 0.25627E+00, 0.25821E+00, 0.26015E+00, 0.26209E+00,
     & 0.26403E+00, 0.26596E+00, 0.26789E+00, 0.26982E+00, 0.27174E+00,
     & 0.27366E+00, 0.27557E+00, 0.27748E+00, 0.27939E+00, 0.28130E+00,
     & 0.28320E+00, 0.28510E+00, 0.28699E+00, 0.28888E+00, 0.29077E+00,
     & 0.29265E+00, 0.29453E+00, 0.29641E+00, 0.29828E+00, 0.30015E+00,
     & 0.30202E+00, 0.30388E+00, 0.30574E+00, 0.30760E+00, 0.30945E+00,
     & 0.31130E+00, 0.31314E+00, 0.31499E+00, 0.31683E+00, 0.31866E+00,
     & 0.32049E+00, 0.32232E+00, 0.32415E+00, 0.32597E+00, 0.32779E+00,
     & 0.32961E+00, 0.33142E+00, 0.33323E+00, 0.33503E+00, 0.33683E+00,
     & 0.33863E+00, 0.34043E+00, 0.34222E+00, 0.34401E+00, 0.34579E+00,
     & 0.34758E+00, 0.34936E+00, 0.35113E+00, 0.35290E+00, 0.35467E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.35644E+00, 0.35820E+00, 0.35996E+00, 0.36172E+00, 0.36347E+00,
     & 0.36522E+00, 0.36696E+00, 0.36871E+00, 0.37045E+00, 0.37218E+00,
     & 0.37392E+00, 0.37565E+00, 0.37737E+00, 0.37910E+00, 0.38082E+00,
     & 0.38254E+00, 0.38425E+00, 0.38596E+00, 0.38767E+00, 0.38937E+00,
     & 0.39108E+00, 0.39277E+00, 0.39447E+00, 0.39616E+00, 0.39785E+00,
     & 0.39954E+00, 0.40122E+00, 0.40290E+00, 0.40458E+00, 0.40625E+00,
     & 0.40792E+00, 0.40959E+00, 0.41125E+00, 0.41291E+00, 0.41457E+00,
     & 0.41623E+00, 0.41788E+00, 0.41953E+00, 0.42117E+00, 0.42281E+00,
     & 0.42445E+00, 0.42609E+00, 0.42773E+00, 0.42936E+00, 0.43098E+00,
     & 0.43261E+00, 0.43423E+00, 0.43585E+00, 0.43747E+00, 0.43908E+00,
     & 0.44069E+00, 0.44230E+00, 0.44390E+00, 0.44550E+00, 0.44710E+00,
     & 0.44869E+00, 0.45029E+00, 0.45188E+00, 0.45346E+00, 0.45505E+00,
     & 0.45663E+00, 0.45820E+00, 0.45978E+00, 0.46135E+00, 0.46292E+00,
     & 0.46449E+00, 0.46605E+00, 0.46761E+00, 0.46917E+00, 0.47072E+00,
     & 0.47228E+00, 0.47383E+00, 0.47537E+00, 0.47692E+00, 0.47846E+00,
     & 0.47999E+00, 0.48153E+00, 0.48306E+00, 0.48459E+00, 0.48612E+00,
     & 0.48764E+00, 0.48916E+00, 0.49068E+00, 0.49220E+00, 0.49371E+00,
     & 0.49522E+00, 0.49673E+00, 0.49824E+00, 0.49974E+00, 0.50124E+00,
     & 0.50273E+00, 0.50423E+00, 0.50572E+00, 0.50721E+00, 0.50869E+00,
     & 0.51018E+00, 0.51166E+00, 0.51314E+00, 0.51461E+00, 0.51608E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.51755E+00, 0.51902E+00, 0.52049E+00, 0.52195E+00, 0.52341E+00,
     & 0.52487E+00, 0.52632E+00, 0.52777E+00, 0.52922E+00, 0.53067E+00,
     & 0.53211E+00, 0.53355E+00, 0.53499E+00, 0.53643E+00, 0.53786E+00,
     & 0.53929E+00, 0.54072E+00, 0.54215E+00, 0.54357E+00, 0.54499E+00,
     & 0.54641E+00, 0.54782E+00, 0.54924E+00, 0.55065E+00, 0.55206E+00,
     & 0.55346E+00, 0.55487E+00, 0.55627E+00, 0.55767E+00, 0.55906E+00,
     & 0.56046E+00, 0.56185E+00, 0.56323E+00, 0.56462E+00, 0.56600E+00,
     & 0.56739E+00, 0.56876E+00, 0.57014E+00, 0.57151E+00, 0.57289E+00,
     & 0.57425E+00, 0.57562E+00, 0.57699E+00, 0.57835E+00, 0.57971E+00,
     & 0.58106E+00, 0.58242E+00, 0.58377E+00, 0.58512E+00, 0.58647E+00,
     & 0.58781E+00, 0.58916E+00, 0.59050E+00, 0.59183E+00, 0.59317E+00,
     & 0.59450E+00, 0.59583E+00, 0.59716E+00, 0.59849E+00, 0.59981E+00,
     & 0.60114E+00, 0.60245E+00, 0.60377E+00, 0.60509E+00, 0.60640E+00,
     & 0.60771E+00, 0.60902E+00, 0.61032E+00, 0.61163E+00, 0.61293E+00,
     & 0.61423E+00, 0.61552E+00, 0.61682E+00, 0.61811E+00, 0.61940E+00,
     & 0.62069E+00, 0.62198E+00, 0.62326E+00, 0.62454E+00, 0.62582E+00,
     & 0.62710E+00, 0.62837E+00, 0.62964E+00, 0.63091E+00, 0.63218E+00,
     & 0.63345E+00, 0.63471E+00, 0.63597E+00, 0.63723E+00, 0.63849E+00,
     & 0.63974E+00, 0.64100E+00, 0.64225E+00, 0.64350E+00, 0.64474E+00,
     & 0.64599E+00, 0.64723E+00, 0.64847E+00, 0.64971E+00, 0.65094E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.65218E+00, 0.65341E+00, 0.65464E+00, 0.65586E+00, 0.65709E+00,
     & 0.65831E+00, 0.65953E+00, 0.66075E+00, 0.66197E+00, 0.66318E+00,
     & 0.66440E+00, 0.66561E+00, 0.66682E+00, 0.66802E+00, 0.66923E+00,
     & 0.67043E+00, 0.67163E+00, 0.67283E+00, 0.67403E+00, 0.67522E+00,
     & 0.67641E+00, 0.67760E+00, 0.67879E+00, 0.67998E+00, 0.68116E+00,
     & 0.68235E+00, 0.68353E+00, 0.68471E+00, 0.68588E+00, 0.68706E+00,
     & 0.68823E+00, 0.68940E+00, 0.69057E+00, 0.69174E+00, 0.69290E+00,
     & 0.69406E+00, 0.69523E+00, 0.69638E+00, 0.69754E+00, 0.69870E+00,
     & 0.69985E+00, 0.70100E+00, 0.70215E+00, 0.70330E+00, 0.70444E+00,
     & 0.70559E+00, 0.70673E+00, 0.70787E+00, 0.70901E+00, 0.71014E+00,
     & 0.71128E+00, 0.71241E+00, 0.71354E+00, 0.71467E+00, 0.71579E+00,
     & 0.71692E+00, 0.71804E+00, 0.71916E+00, 0.72028E+00, 0.72140E+00,
     & 0.72252E+00, 0.72363E+00, 0.72474E+00, 0.72585E+00, 0.72696E+00,
     & 0.72807E+00, 0.72917E+00, 0.73027E+00, 0.73137E+00, 0.73247E+00,
     & 0.73357E+00, 0.73467E+00, 0.73576E+00, 0.73685E+00, 0.73794E+00,
     & 0.73903E+00, 0.74012E+00, 0.74120E+00, 0.74228E+00, 0.74337E+00,
     & 0.74444E+00, 0.74552E+00, 0.74660E+00, 0.74767E+00, 0.74875E+00,
     & 0.74982E+00, 0.75088E+00, 0.75195E+00, 0.75302E+00, 0.75408E+00,
     & 0.75514E+00, 0.75620E+00, 0.75726E+00, 0.75832E+00, 0.75937E+00,
     & 0.76043E+00, 0.76148E+00, 0.76253E+00, 0.76358E+00, 0.76462E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.77583E+00, 0.78602E+00, 0.79605E+00, 0.80593E+00, 0.81565E+00,
     & 0.82522E+00, 0.83463E+00, 0.84391E+00, 0.85304E+00, 0.86203E+00,
     & 0.87088E+00, 0.87959E+00, 0.88818E+00, 0.89663E+00, 0.90496E+00,
     & 0.91316E+00, 0.92124E+00, 0.92920E+00, 0.93704E+00, 0.94477E+00,
     & 0.95238E+00, 0.95988E+00, 0.96727E+00, 0.97455E+00, 0.98173E+00,
     & 0.98880E+00, 0.99577E+00, 0.10026E+01, 0.10094E+01, 0.10161E+01,
     & 0.10227E+01, 0.10292E+01, 0.10355E+01, 0.10418E+01, 0.10481E+01,
     & 0.10542E+01, 0.10602E+01, 0.10662E+01, 0.10721E+01, 0.10778E+01,
     & 0.10835E+01, 0.10892E+01, 0.10947E+01, 0.11002E+01, 0.11056E+01,
     & 0.11109E+01, 0.11162E+01, 0.11213E+01, 0.11264E+01, 0.11315E+01,
     & 0.11364E+01, 0.11413E+01, 0.11462E+01, 0.11509E+01, 0.11556E+01,
     & 0.11603E+01, 0.11648E+01, 0.11693E+01, 0.11738E+01, 0.11782E+01,
     & 0.11825E+01, 0.11867E+01, 0.11909E+01, 0.11951E+01, 0.11992E+01,
     & 0.12032E+01, 0.12072E+01, 0.12111E+01, 0.12150E+01, 0.12188E+01,
     & 0.12226E+01, 0.12263E+01, 0.12299E+01, 0.12335E+01, 0.12371E+01,
     & 0.12406E+01, 0.12441E+01, 0.12475E+01, 0.12508E+01, 0.12542E+01,
     & 0.12574E+01, 0.12607E+01, 0.12638E+01, 0.12670E+01, 0.12701E+01,
     & 0.12731E+01, 0.12761E+01, 0.12791E+01, 0.12820E+01, 0.12849E+01,
     & 0.12877E+01, 0.12905E+01, 0.12933E+01, 0.12960E+01, 0.12987E+01,
     & 0.13013E+01, 0.13039E+01, 0.13065E+01, 0.13090E+01, 0.13115E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.13139E+01, 0.13163E+01, 0.13187E+01, 0.13211E+01, 0.13234E+01,
     & 0.13256E+01, 0.13279E+01, 0.13301E+01, 0.13323E+01, 0.13344E+01,
     & 0.13365E+01, 0.13386E+01, 0.13406E+01, 0.13426E+01, 0.13446E+01,
     & 0.13465E+01, 0.13485E+01, 0.13503E+01, 0.13522E+01, 0.13540E+01,
     & 0.13558E+01, 0.13576E+01, 0.13593E+01, 0.13610E+01, 0.13627E+01,
     & 0.13643E+01, 0.13660E+01, 0.13675E+01, 0.13691E+01, 0.13706E+01,
     & 0.13722E+01, 0.13736E+01, 0.13751E+01, 0.13765E+01, 0.13779E+01,
     & 0.13793E+01, 0.13807E+01, 0.13820E+01, 0.13833E+01, 0.13846E+01,
     & 0.13858E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.10044E+00,-0.21963E+00,-0.27984E+00,-0.32277E+00,-0.35672E+00,
     &-0.38502E+00,-0.40943E+00,-0.43096E+00,-0.45026E+00,-0.46780E+00,
     &-0.48389E+00,-0.49877E+00,-0.51264E+00,-0.52563E+00,-0.53787E+00,
     &-0.54943E+00,-0.56041E+00,-0.57086E+00,-0.58084E+00,-0.59040E+00,
     &-0.59957E+00,-0.60840E+00,-0.61689E+00,-0.62510E+00,-0.63303E+00,
     &-0.64071E+00,-0.64816E+00,-0.65538E+00,-0.66241E+00,-0.66924E+00,
     &-0.67590E+00,-0.68239E+00,-0.68872E+00,-0.69490E+00,-0.70094E+00,
     &-0.70684E+00,-0.71262E+00,-0.71828E+00,-0.72382E+00,-0.72926E+00,
     &-0.73459E+00,-0.73982E+00,-0.74495E+00,-0.75000E+00,-0.75496E+00,
     &-0.75983E+00,-0.76463E+00,-0.76935E+00,-0.77399E+00,-0.77856E+00,
     &-0.78307E+00,-0.78751E+00,-0.79188E+00,-0.79620E+00,-0.80045E+00,
     &-0.80465E+00,-0.80880E+00,-0.81289E+00,-0.81693E+00,-0.82093E+00,
     &-0.82487E+00,-0.82877E+00,-0.83263E+00,-0.83644E+00,-0.84021E+00,
     &-0.84394E+00,-0.84763E+00,-0.85129E+00,-0.85491E+00,-0.85849E+00,
     &-0.86204E+00,-0.86556E+00,-0.86905E+00,-0.87250E+00,-0.87593E+00,
     &-0.87932E+00,-0.88269E+00,-0.88604E+00,-0.88935E+00,-0.89264E+00,
     &-0.89591E+00,-0.89915E+00,-0.90237E+00,-0.90557E+00,-0.90875E+00,
     &-0.91190E+00,-0.91503E+00,-0.91815E+00,-0.92124E+00,-0.92432E+00,
     &-0.92737E+00,-0.93041E+00,-0.93343E+00,-0.93644E+00,-0.93942E+00,
     &-0.94239E+00,-0.94534E+00,-0.94828E+00,-0.95120E+00,-0.95411E+00/

      DATA (BNC02M(I),I=101,200)/
     &-0.95700E+00,-0.95988E+00,-0.96274E+00,-0.96559E+00,-0.96842E+00,
     &-0.97124E+00,-0.97405E+00,-0.97684E+00,-0.97962E+00,-0.98239E+00,
     &-0.98514E+00,-0.98788E+00,-0.99061E+00,-0.99333E+00,-0.99603E+00,
     &-0.99872E+00,-0.10014E+01,-0.10041E+01,-0.10067E+01,-0.10094E+01,
     &-0.10120E+01,-0.10146E+01,-0.10172E+01,-0.10198E+01,-0.10224E+01,
     &-0.10250E+01,-0.10276E+01,-0.10301E+01,-0.10327E+01,-0.10352E+01,
     &-0.10377E+01,-0.10402E+01,-0.10427E+01,-0.10452E+01,-0.10477E+01,
     &-0.10502E+01,-0.10527E+01,-0.10551E+01,-0.10576E+01,-0.10600E+01,
     &-0.10624E+01,-0.10649E+01,-0.10673E+01,-0.10697E+01,-0.10721E+01,
     &-0.10745E+01,-0.10768E+01,-0.10792E+01,-0.10816E+01,-0.10839E+01,
     &-0.10863E+01,-0.10886E+01,-0.10909E+01,-0.10933E+01,-0.10956E+01,
     &-0.10979E+01,-0.11002E+01,-0.11025E+01,-0.11048E+01,-0.11070E+01,
     &-0.11093E+01,-0.11116E+01,-0.11138E+01,-0.11161E+01,-0.11183E+01,
     &-0.11206E+01,-0.11228E+01,-0.11250E+01,-0.11272E+01,-0.11294E+01,
     &-0.11316E+01,-0.11338E+01,-0.11360E+01,-0.11382E+01,-0.11404E+01,
     &-0.11426E+01,-0.11447E+01,-0.11469E+01,-0.11490E+01,-0.11512E+01,
     &-0.11533E+01,-0.11555E+01,-0.11576E+01,-0.11597E+01,-0.11618E+01,
     &-0.11640E+01,-0.11661E+01,-0.11682E+01,-0.11703E+01,-0.11724E+01,
     &-0.11745E+01,-0.11765E+01,-0.11786E+01,-0.11807E+01,-0.11828E+01,
     &-0.11848E+01,-0.11869E+01,-0.11889E+01,-0.11910E+01,-0.11930E+01/

      DATA (BNC02M(I),I=201,300)/
     &-0.11951E+01,-0.11971E+01,-0.11991E+01,-0.12012E+01,-0.12032E+01,
     &-0.12052E+01,-0.12072E+01,-0.12092E+01,-0.12112E+01,-0.12132E+01,
     &-0.12152E+01,-0.12172E+01,-0.12192E+01,-0.12212E+01,-0.12232E+01,
     &-0.12251E+01,-0.12271E+01,-0.12291E+01,-0.12310E+01,-0.12330E+01,
     &-0.12349E+01,-0.12369E+01,-0.12388E+01,-0.12408E+01,-0.12427E+01,
     &-0.12446E+01,-0.12466E+01,-0.12485E+01,-0.12504E+01,-0.12523E+01,
     &-0.12543E+01,-0.12562E+01,-0.12581E+01,-0.12600E+01,-0.12619E+01,
     &-0.12638E+01,-0.12657E+01,-0.12676E+01,-0.12694E+01,-0.12713E+01,
     &-0.12732E+01,-0.12751E+01,-0.12770E+01,-0.12788E+01,-0.12807E+01,
     &-0.12826E+01,-0.12844E+01,-0.12863E+01,-0.12881E+01,-0.12900E+01,
     &-0.12918E+01,-0.12937E+01,-0.12955E+01,-0.12973E+01,-0.12992E+01,
     &-0.13010E+01,-0.13028E+01,-0.13047E+01,-0.13065E+01,-0.13083E+01,
     &-0.13101E+01,-0.13119E+01,-0.13138E+01,-0.13156E+01,-0.13174E+01,
     &-0.13192E+01,-0.13210E+01,-0.13228E+01,-0.13246E+01,-0.13263E+01,
     &-0.13281E+01,-0.13299E+01,-0.13317E+01,-0.13335E+01,-0.13353E+01,
     &-0.13370E+01,-0.13388E+01,-0.13406E+01,-0.13423E+01,-0.13441E+01,
     &-0.13459E+01,-0.13476E+01,-0.13494E+01,-0.13511E+01,-0.13529E+01,
     &-0.13546E+01,-0.13564E+01,-0.13581E+01,-0.13599E+01,-0.13616E+01,
     &-0.13633E+01,-0.13651E+01,-0.13668E+01,-0.13685E+01,-0.13703E+01,
     &-0.13720E+01,-0.13737E+01,-0.13754E+01,-0.13772E+01,-0.13789E+01/

      DATA (BNC02M(I),I=301,400)/
     &-0.13806E+01,-0.13823E+01,-0.13840E+01,-0.13857E+01,-0.13874E+01,
     &-0.13891E+01,-0.13908E+01,-0.13925E+01,-0.13942E+01,-0.13959E+01,
     &-0.13976E+01,-0.13993E+01,-0.14010E+01,-0.14027E+01,-0.14043E+01,
     &-0.14060E+01,-0.14077E+01,-0.14094E+01,-0.14111E+01,-0.14127E+01,
     &-0.14144E+01,-0.14161E+01,-0.14177E+01,-0.14194E+01,-0.14211E+01,
     &-0.14227E+01,-0.14244E+01,-0.14260E+01,-0.14277E+01,-0.14293E+01,
     &-0.14310E+01,-0.14326E+01,-0.14343E+01,-0.14359E+01,-0.14376E+01,
     &-0.14392E+01,-0.14409E+01,-0.14425E+01,-0.14441E+01,-0.14458E+01,
     &-0.14474E+01,-0.14490E+01,-0.14506E+01,-0.14523E+01,-0.14539E+01,
     &-0.14555E+01,-0.14571E+01,-0.14588E+01,-0.14604E+01,-0.14620E+01,
     &-0.14636E+01,-0.14652E+01,-0.14668E+01,-0.14684E+01,-0.14700E+01,
     &-0.14717E+01,-0.14733E+01,-0.14749E+01,-0.14765E+01,-0.14781E+01,
     &-0.14797E+01,-0.14813E+01,-0.14828E+01,-0.14844E+01,-0.14860E+01,
     &-0.14876E+01,-0.14892E+01,-0.14908E+01,-0.14924E+01,-0.14940E+01,
     &-0.14955E+01,-0.14971E+01,-0.14987E+01,-0.15003E+01,-0.15018E+01,
     &-0.15034E+01,-0.15050E+01,-0.15066E+01,-0.15081E+01,-0.15097E+01,
     &-0.15113E+01,-0.15128E+01,-0.15144E+01,-0.15160E+01,-0.15175E+01,
     &-0.15191E+01,-0.15206E+01,-0.15222E+01,-0.15237E+01,-0.15253E+01,
     &-0.15268E+01,-0.15284E+01,-0.15299E+01,-0.15315E+01,-0.15330E+01,
     &-0.15346E+01,-0.15361E+01,-0.15377E+01,-0.15392E+01,-0.15407E+01/

      DATA (BNC02M(I),I=401,500)/
     &-0.15423E+01,-0.15438E+01,-0.15453E+01,-0.15469E+01,-0.15484E+01,
     &-0.15499E+01,-0.15515E+01,-0.15530E+01,-0.15545E+01,-0.15561E+01,
     &-0.15576E+01,-0.15591E+01,-0.15606E+01,-0.15621E+01,-0.15637E+01,
     &-0.15652E+01,-0.15667E+01,-0.15682E+01,-0.15697E+01,-0.15712E+01,
     &-0.15727E+01,-0.15743E+01,-0.15758E+01,-0.15773E+01,-0.15788E+01,
     &-0.15803E+01,-0.15818E+01,-0.15833E+01,-0.15848E+01,-0.15863E+01,
     &-0.15878E+01,-0.15893E+01,-0.15908E+01,-0.15923E+01,-0.15938E+01,
     &-0.15953E+01,-0.15968E+01,-0.15982E+01,-0.15997E+01,-0.16012E+01,
     &-0.16027E+01,-0.16042E+01,-0.16057E+01,-0.16072E+01,-0.16086E+01,
     &-0.16101E+01,-0.16116E+01,-0.16131E+01,-0.16146E+01,-0.16160E+01,
     &-0.16175E+01,-0.16190E+01,-0.16205E+01,-0.16219E+01,-0.16234E+01,
     &-0.16249E+01,-0.16263E+01,-0.16278E+01,-0.16293E+01,-0.16307E+01,
     &-0.16322E+01,-0.16337E+01,-0.16351E+01,-0.16366E+01,-0.16381E+01,
     &-0.16395E+01,-0.16410E+01,-0.16424E+01,-0.16439E+01,-0.16453E+01,
     &-0.16468E+01,-0.16482E+01,-0.16497E+01,-0.16512E+01,-0.16526E+01,
     &-0.16541E+01,-0.16555E+01,-0.16569E+01,-0.16584E+01,-0.16598E+01,
     &-0.16613E+01,-0.16627E+01,-0.16642E+01,-0.16656E+01,-0.16670E+01,
     &-0.16685E+01,-0.16699E+01,-0.16714E+01,-0.16728E+01,-0.16742E+01,
     &-0.16757E+01,-0.16771E+01,-0.16785E+01,-0.16800E+01,-0.16814E+01,
     &-0.16828E+01,-0.16843E+01,-0.16857E+01,-0.16871E+01,-0.16885E+01/

      DATA (BNC02M(I),I=501,600)/
     &-0.16900E+01,-0.16914E+01,-0.16928E+01,-0.16942E+01,-0.16957E+01,
     &-0.16971E+01,-0.16985E+01,-0.16999E+01,-0.17013E+01,-0.17027E+01,
     &-0.17042E+01,-0.17056E+01,-0.17070E+01,-0.17084E+01,-0.17098E+01,
     &-0.17112E+01,-0.17126E+01,-0.17140E+01,-0.17155E+01,-0.17169E+01,
     &-0.17183E+01,-0.17197E+01,-0.17211E+01,-0.17225E+01,-0.17239E+01,
     &-0.17253E+01,-0.17267E+01,-0.17281E+01,-0.17295E+01,-0.17309E+01,
     &-0.17323E+01,-0.17337E+01,-0.17351E+01,-0.17365E+01,-0.17379E+01,
     &-0.17393E+01,-0.17407E+01,-0.17421E+01,-0.17435E+01,-0.17448E+01,
     &-0.17462E+01,-0.17476E+01,-0.17490E+01,-0.17504E+01,-0.17518E+01,
     &-0.17532E+01,-0.17546E+01,-0.17559E+01,-0.17573E+01,-0.17587E+01,
     &-0.17601E+01,-0.17615E+01,-0.17629E+01,-0.17642E+01,-0.17656E+01,
     &-0.17670E+01,-0.17684E+01,-0.17698E+01,-0.17711E+01,-0.17725E+01,
     &-0.17739E+01,-0.17753E+01,-0.17766E+01,-0.17780E+01,-0.17794E+01,
     &-0.17807E+01,-0.17821E+01,-0.17835E+01,-0.17848E+01,-0.17862E+01,
     &-0.17876E+01,-0.17889E+01,-0.17903E+01,-0.17917E+01,-0.17930E+01,
     &-0.17944E+01,-0.17958E+01,-0.17971E+01,-0.17985E+01,-0.17999E+01,
     &-0.18012E+01,-0.18026E+01,-0.18039E+01,-0.18053E+01,-0.18067E+01,
     &-0.18080E+01,-0.18094E+01,-0.18107E+01,-0.18121E+01,-0.18134E+01,
     &-0.18148E+01,-0.18161E+01,-0.18175E+01,-0.18188E+01,-0.18202E+01,
     &-0.18215E+01,-0.18229E+01,-0.18242E+01,-0.18256E+01,-0.18269E+01/

      DATA (BNC02M(I),I=601,700)/
     &-0.18414E+01,-0.18548E+01,-0.18681E+01,-0.18813E+01,-0.18945E+01,
     &-0.19076E+01,-0.19207E+01,-0.19337E+01,-0.19467E+01,-0.19596E+01,
     &-0.19724E+01,-0.19852E+01,-0.19980E+01,-0.20107E+01,-0.20233E+01,
     &-0.20359E+01,-0.20485E+01,-0.20610E+01,-0.20735E+01,-0.20859E+01,
     &-0.20983E+01,-0.21107E+01,-0.21230E+01,-0.21353E+01,-0.21475E+01,
     &-0.21598E+01,-0.21719E+01,-0.21841E+01,-0.21962E+01,-0.22082E+01,
     &-0.22203E+01,-0.22323E+01,-0.22442E+01,-0.22562E+01,-0.22681E+01,
     &-0.22800E+01,-0.22918E+01,-0.23036E+01,-0.23154E+01,-0.23272E+01,
     &-0.23389E+01,-0.23506E+01,-0.23623E+01,-0.23740E+01,-0.23856E+01,
     &-0.23972E+01,-0.24088E+01,-0.24204E+01,-0.24319E+01,-0.24434E+01,
     &-0.24549E+01,-0.24663E+01,-0.24778E+01,-0.24892E+01,-0.25006E+01,
     &-0.25119E+01,-0.25233E+01,-0.25346E+01,-0.25459E+01,-0.25572E+01,
     &-0.25685E+01,-0.25797E+01,-0.25910E+01,-0.26022E+01,-0.26134E+01,
     &-0.26245E+01,-0.26357E+01,-0.26468E+01,-0.26579E+01,-0.26690E+01,
     &-0.26801E+01,-0.26912E+01,-0.27022E+01,-0.27132E+01,-0.27242E+01,
     &-0.27352E+01,-0.27462E+01,-0.27572E+01,-0.27681E+01,-0.27790E+01,
     &-0.27900E+01,-0.28009E+01,-0.28117E+01,-0.28226E+01,-0.28335E+01,
     &-0.28443E+01,-0.28551E+01,-0.28659E+01,-0.28767E+01,-0.28875E+01,
     &-0.28983E+01,-0.29090E+01,-0.29198E+01,-0.29305E+01,-0.29412E+01,
     &-0.29519E+01,-0.29626E+01,-0.29733E+01,-0.29839E+01,-0.29946E+01/

      DATA (BNC02M(I),I=701,741)/
     &-0.30052E+01,-0.30158E+01,-0.30264E+01,-0.30370E+01,-0.30476E+01,
     &-0.30582E+01,-0.30688E+01,-0.30793E+01,-0.30899E+01,-0.31004E+01,
     &-0.31109E+01,-0.31214E+01,-0.31319E+01,-0.31424E+01,-0.31529E+01,
     &-0.31633E+01,-0.31738E+01,-0.31842E+01,-0.31946E+01,-0.32051E+01,
     &-0.32155E+01,-0.32259E+01,-0.32363E+01,-0.32466E+01,-0.32570E+01,
     &-0.32674E+01,-0.32777E+01,-0.32881E+01,-0.32984E+01,-0.33087E+01,
     &-0.33190E+01,-0.33293E+01,-0.33396E+01,-0.33499E+01,-0.33602E+01,
     &-0.33705E+01,-0.33807E+01,-0.33910E+01,-0.34012E+01,-0.34114E+01,
     &-0.34217E+01
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.50377E-01,-0.11068E+00,-0.14143E+00,-0.16351E+00,-0.18108E+00,
     &-0.19581E+00,-0.20858E+00,-0.21990E+00,-0.23009E+00,-0.23939E+00,
     &-0.24796E+00,-0.25592E+00,-0.26337E+00,-0.27037E+00,-0.27698E+00,
     &-0.28326E+00,-0.28923E+00,-0.29494E+00,-0.30041E+00,-0.30566E+00,
     &-0.31071E+00,-0.31558E+00,-0.32029E+00,-0.32484E+00,-0.32925E+00,
     &-0.33353E+00,-0.33769E+00,-0.34174E+00,-0.34568E+00,-0.34952E+00,
     &-0.35327E+00,-0.35692E+00,-0.36050E+00,-0.36399E+00,-0.36741E+00,
     &-0.37076E+00,-0.37405E+00,-0.37726E+00,-0.38042E+00,-0.38352E+00,
     &-0.38656E+00,-0.38955E+00,-0.39248E+00,-0.39537E+00,-0.39821E+00,
     &-0.40100E+00,-0.40375E+00,-0.40646E+00,-0.40912E+00,-0.41175E+00,
     &-0.41434E+00,-0.41690E+00,-0.41942E+00,-0.42190E+00,-0.42435E+00,
     &-0.42677E+00,-0.42917E+00,-0.43153E+00,-0.43386E+00,-0.43617E+00,
     &-0.43845E+00,-0.44070E+00,-0.44293E+00,-0.44514E+00,-0.44732E+00,
     &-0.44949E+00,-0.45163E+00,-0.45375E+00,-0.45585E+00,-0.45793E+00,
     &-0.45999E+00,-0.46204E+00,-0.46407E+00,-0.46608E+00,-0.46808E+00,
     &-0.47006E+00,-0.47203E+00,-0.47398E+00,-0.47592E+00,-0.47785E+00,
     &-0.47976E+00,-0.48167E+00,-0.48356E+00,-0.48543E+00,-0.48730E+00,
     &-0.48916E+00,-0.49100E+00,-0.49284E+00,-0.49467E+00,-0.49648E+00,
     &-0.49829E+00,-0.50009E+00,-0.50188E+00,-0.50366E+00,-0.50543E+00,
     &-0.50719E+00,-0.50895E+00,-0.51070E+00,-0.51244E+00,-0.51417E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.51590E+00,-0.51761E+00,-0.51932E+00,-0.52103E+00,-0.52272E+00,
     &-0.52441E+00,-0.52609E+00,-0.52777E+00,-0.52944E+00,-0.53110E+00,
     &-0.53275E+00,-0.53440E+00,-0.53604E+00,-0.53767E+00,-0.53930E+00,
     &-0.54092E+00,-0.54254E+00,-0.54414E+00,-0.54575E+00,-0.54734E+00,
     &-0.54893E+00,-0.55051E+00,-0.55209E+00,-0.55366E+00,-0.55522E+00,
     &-0.55678E+00,-0.55833E+00,-0.55988E+00,-0.56142E+00,-0.56295E+00,
     &-0.56448E+00,-0.56600E+00,-0.56752E+00,-0.56903E+00,-0.57053E+00,
     &-0.57203E+00,-0.57352E+00,-0.57501E+00,-0.57649E+00,-0.57797E+00,
     &-0.57944E+00,-0.58091E+00,-0.58237E+00,-0.58383E+00,-0.58528E+00,
     &-0.58672E+00,-0.58816E+00,-0.58960E+00,-0.59103E+00,-0.59246E+00,
     &-0.59388E+00,-0.59529E+00,-0.59671E+00,-0.59811E+00,-0.59952E+00,
     &-0.60091E+00,-0.60231E+00,-0.60370E+00,-0.60508E+00,-0.60646E+00,
     &-0.60784E+00,-0.60921E+00,-0.61058E+00,-0.61194E+00,-0.61330E+00,
     &-0.61465E+00,-0.61600E+00,-0.61735E+00,-0.61869E+00,-0.62003E+00,
     &-0.62137E+00,-0.62270E+00,-0.62402E+00,-0.62535E+00,-0.62667E+00,
     &-0.62798E+00,-0.62930E+00,-0.63060E+00,-0.63191E+00,-0.63321E+00,
     &-0.63451E+00,-0.63580E+00,-0.63709E+00,-0.63838E+00,-0.63966E+00,
     &-0.64094E+00,-0.64222E+00,-0.64349E+00,-0.64476E+00,-0.64603E+00,
     &-0.64730E+00,-0.64856E+00,-0.64981E+00,-0.65107E+00,-0.65232E+00,
     &-0.65357E+00,-0.65481E+00,-0.65605E+00,-0.65729E+00,-0.65853E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.65976E+00,-0.66099E+00,-0.66222E+00,-0.66344E+00,-0.66466E+00,
     &-0.66588E+00,-0.66710E+00,-0.66831E+00,-0.66952E+00,-0.67072E+00,
     &-0.67193E+00,-0.67313E+00,-0.67433E+00,-0.67552E+00,-0.67672E+00,
     &-0.67791E+00,-0.67910E+00,-0.68028E+00,-0.68147E+00,-0.68265E+00,
     &-0.68382E+00,-0.68500E+00,-0.68617E+00,-0.68734E+00,-0.68851E+00,
     &-0.68967E+00,-0.69084E+00,-0.69200E+00,-0.69316E+00,-0.69431E+00,
     &-0.69546E+00,-0.69662E+00,-0.69776E+00,-0.69891E+00,-0.70005E+00,
     &-0.70120E+00,-0.70234E+00,-0.70347E+00,-0.70461E+00,-0.70574E+00,
     &-0.70687E+00,-0.70800E+00,-0.70912E+00,-0.71025E+00,-0.71137E+00,
     &-0.71249E+00,-0.71361E+00,-0.71472E+00,-0.71584E+00,-0.71695E+00,
     &-0.71806E+00,-0.71916E+00,-0.72027E+00,-0.72137E+00,-0.72247E+00,
     &-0.72357E+00,-0.72467E+00,-0.72576E+00,-0.72686E+00,-0.72795E+00,
     &-0.72904E+00,-0.73012E+00,-0.73121E+00,-0.73229E+00,-0.73337E+00,
     &-0.73445E+00,-0.73553E+00,-0.73661E+00,-0.73768E+00,-0.73875E+00,
     &-0.73983E+00,-0.74089E+00,-0.74196E+00,-0.74303E+00,-0.74409E+00,
     &-0.74515E+00,-0.74621E+00,-0.74727E+00,-0.74832E+00,-0.74938E+00,
     &-0.75043E+00,-0.75148E+00,-0.75253E+00,-0.75358E+00,-0.75463E+00,
     &-0.75567E+00,-0.75671E+00,-0.75776E+00,-0.75880E+00,-0.75983E+00,
     &-0.76087E+00,-0.76190E+00,-0.76294E+00,-0.76397E+00,-0.76500E+00,
     &-0.76603E+00,-0.76705E+00,-0.76808E+00,-0.76910E+00,-0.77012E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.77115E+00,-0.77216E+00,-0.77318E+00,-0.77420E+00,-0.77521E+00,
     &-0.77623E+00,-0.77724E+00,-0.77825E+00,-0.77926E+00,-0.78026E+00,
     &-0.78127E+00,-0.78227E+00,-0.78328E+00,-0.78428E+00,-0.78528E+00,
     &-0.78628E+00,-0.78727E+00,-0.78827E+00,-0.78926E+00,-0.79026E+00,
     &-0.79125E+00,-0.79224E+00,-0.79323E+00,-0.79422E+00,-0.79520E+00,
     &-0.79619E+00,-0.79717E+00,-0.79815E+00,-0.79913E+00,-0.80011E+00,
     &-0.80109E+00,-0.80207E+00,-0.80304E+00,-0.80402E+00,-0.80499E+00,
     &-0.80596E+00,-0.80694E+00,-0.80790E+00,-0.80887E+00,-0.80984E+00,
     &-0.81081E+00,-0.81177E+00,-0.81273E+00,-0.81369E+00,-0.81466E+00,
     &-0.81562E+00,-0.81657E+00,-0.81753E+00,-0.81849E+00,-0.81944E+00,
     &-0.82039E+00,-0.82135E+00,-0.82230E+00,-0.82325E+00,-0.82420E+00,
     &-0.82514E+00,-0.82609E+00,-0.82704E+00,-0.82798E+00,-0.82892E+00,
     &-0.82987E+00,-0.83081E+00,-0.83175E+00,-0.83269E+00,-0.83362E+00,
     &-0.83456E+00,-0.83549E+00,-0.83643E+00,-0.83736E+00,-0.83829E+00,
     &-0.83923E+00,-0.84016E+00,-0.84108E+00,-0.84201E+00,-0.84294E+00,
     &-0.84386E+00,-0.84479E+00,-0.84571E+00,-0.84663E+00,-0.84756E+00,
     &-0.84848E+00,-0.84940E+00,-0.85031E+00,-0.85123E+00,-0.85215E+00,
     &-0.85306E+00,-0.85398E+00,-0.85489E+00,-0.85580E+00,-0.85672E+00,
     &-0.85763E+00,-0.85853E+00,-0.85944E+00,-0.86035E+00,-0.86126E+00,
     &-0.86216E+00,-0.86307E+00,-0.86397E+00,-0.86487E+00,-0.86578E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.86668E+00,-0.86758E+00,-0.86847E+00,-0.86937E+00,-0.87027E+00,
     &-0.87117E+00,-0.87206E+00,-0.87295E+00,-0.87385E+00,-0.87474E+00,
     &-0.87563E+00,-0.87652E+00,-0.87741E+00,-0.87830E+00,-0.87919E+00,
     &-0.88008E+00,-0.88096E+00,-0.88185E+00,-0.88273E+00,-0.88361E+00,
     &-0.88450E+00,-0.88538E+00,-0.88626E+00,-0.88714E+00,-0.88802E+00,
     &-0.88890E+00,-0.88977E+00,-0.89065E+00,-0.89153E+00,-0.89240E+00,
     &-0.89328E+00,-0.89415E+00,-0.89502E+00,-0.89589E+00,-0.89676E+00,
     &-0.89763E+00,-0.89850E+00,-0.89937E+00,-0.90024E+00,-0.90110E+00,
     &-0.90197E+00,-0.90283E+00,-0.90370E+00,-0.90456E+00,-0.90542E+00,
     &-0.90629E+00,-0.90715E+00,-0.90801E+00,-0.90887E+00,-0.90973E+00,
     &-0.91058E+00,-0.91144E+00,-0.91230E+00,-0.91315E+00,-0.91401E+00,
     &-0.91486E+00,-0.91572E+00,-0.91657E+00,-0.91742E+00,-0.91827E+00,
     &-0.91912E+00,-0.91997E+00,-0.92082E+00,-0.92167E+00,-0.92251E+00,
     &-0.92336E+00,-0.92421E+00,-0.92505E+00,-0.92590E+00,-0.92674E+00,
     &-0.92758E+00,-0.92842E+00,-0.92927E+00,-0.93011E+00,-0.93095E+00,
     &-0.93179E+00,-0.93262E+00,-0.93346E+00,-0.93430E+00,-0.93514E+00,
     &-0.93597E+00,-0.93681E+00,-0.93764E+00,-0.93848E+00,-0.93931E+00,
     &-0.94014E+00,-0.94097E+00,-0.94180E+00,-0.94263E+00,-0.94346E+00,
     &-0.94429E+00,-0.94512E+00,-0.94595E+00,-0.94678E+00,-0.94760E+00,
     &-0.94843E+00,-0.94925E+00,-0.95008E+00,-0.95090E+00,-0.95172E+00/

      DATA (BNC03M(I),I=501,600)/
     &-0.95255E+00,-0.95337E+00,-0.95419E+00,-0.95501E+00,-0.95583E+00,
     &-0.95665E+00,-0.95747E+00,-0.95828E+00,-0.95910E+00,-0.95992E+00,
     &-0.96073E+00,-0.96155E+00,-0.96236E+00,-0.96318E+00,-0.96399E+00,
     &-0.96480E+00,-0.96562E+00,-0.96643E+00,-0.96724E+00,-0.96805E+00,
     &-0.96886E+00,-0.96967E+00,-0.97048E+00,-0.97128E+00,-0.97209E+00,
     &-0.97290E+00,-0.97371E+00,-0.97451E+00,-0.97532E+00,-0.97612E+00,
     &-0.97692E+00,-0.97773E+00,-0.97853E+00,-0.97933E+00,-0.98013E+00,
     &-0.98093E+00,-0.98173E+00,-0.98253E+00,-0.98333E+00,-0.98413E+00,
     &-0.98493E+00,-0.98573E+00,-0.98652E+00,-0.98732E+00,-0.98812E+00,
     &-0.98891E+00,-0.98971E+00,-0.99050E+00,-0.99129E+00,-0.99209E+00,
     &-0.99288E+00,-0.99367E+00,-0.99446E+00,-0.99525E+00,-0.99604E+00,
     &-0.99683E+00,-0.99762E+00,-0.99841E+00,-0.99920E+00,-0.99999E+00,
     &-0.10008E+01,-0.10016E+01,-0.10023E+01,-0.10031E+01,-0.10039E+01,
     &-0.10047E+01,-0.10055E+01,-0.10063E+01,-0.10070E+01,-0.10078E+01,
     &-0.10086E+01,-0.10094E+01,-0.10102E+01,-0.10110E+01,-0.10117E+01,
     &-0.10125E+01,-0.10133E+01,-0.10141E+01,-0.10148E+01,-0.10156E+01,
     &-0.10164E+01,-0.10172E+01,-0.10180E+01,-0.10187E+01,-0.10195E+01,
     &-0.10203E+01,-0.10210E+01,-0.10218E+01,-0.10226E+01,-0.10234E+01,
     &-0.10241E+01,-0.10249E+01,-0.10257E+01,-0.10264E+01,-0.10272E+01,
     &-0.10280E+01,-0.10288E+01,-0.10295E+01,-0.10303E+01,-0.10311E+01/

      DATA (BNC03M(I),I=601,700)/
     &-0.10393E+01,-0.10469E+01,-0.10545E+01,-0.10620E+01,-0.10694E+01,
     &-0.10769E+01,-0.10842E+01,-0.10916E+01,-0.10989E+01,-0.11062E+01,
     &-0.11134E+01,-0.11206E+01,-0.11277E+01,-0.11349E+01,-0.11420E+01,
     &-0.11490E+01,-0.11560E+01,-0.11630E+01,-0.11700E+01,-0.11769E+01,
     &-0.11838E+01,-0.11907E+01,-0.11976E+01,-0.12044E+01,-0.12112E+01,
     &-0.12180E+01,-0.12247E+01,-0.12314E+01,-0.12381E+01,-0.12448E+01,
     &-0.12514E+01,-0.12581E+01,-0.12647E+01,-0.12713E+01,-0.12778E+01,
     &-0.12843E+01,-0.12909E+01,-0.12974E+01,-0.13038E+01,-0.13103E+01,
     &-0.13167E+01,-0.13231E+01,-0.13295E+01,-0.13359E+01,-0.13423E+01,
     &-0.13486E+01,-0.13549E+01,-0.13612E+01,-0.13675E+01,-0.13738E+01,
     &-0.13800E+01,-0.13863E+01,-0.13925E+01,-0.13987E+01,-0.14049E+01,
     &-0.14111E+01,-0.14172E+01,-0.14234E+01,-0.14295E+01,-0.14356E+01,
     &-0.14417E+01,-0.14478E+01,-0.14539E+01,-0.14599E+01,-0.14660E+01,
     &-0.14720E+01,-0.14780E+01,-0.14840E+01,-0.14900E+01,-0.14960E+01,
     &-0.15020E+01,-0.15079E+01,-0.15139E+01,-0.15198E+01,-0.15257E+01,
     &-0.15316E+01,-0.15375E+01,-0.15434E+01,-0.15493E+01,-0.15551E+01,
     &-0.15610E+01,-0.15668E+01,-0.15727E+01,-0.15785E+01,-0.15843E+01,
     &-0.15901E+01,-0.15959E+01,-0.16017E+01,-0.16074E+01,-0.16132E+01,
     &-0.16189E+01,-0.16247E+01,-0.16304E+01,-0.16361E+01,-0.16418E+01,
     &-0.16475E+01,-0.16532E+01,-0.16589E+01,-0.16646E+01,-0.16703E+01/

      DATA (BNC03M(I),I=701,741)/
     &-0.16759E+01,-0.16816E+01,-0.16872E+01,-0.16928E+01,-0.16985E+01,
     &-0.17041E+01,-0.17097E+01,-0.17153E+01,-0.17209E+01,-0.17265E+01,
     &-0.17320E+01,-0.17376E+01,-0.17432E+01,-0.17487E+01,-0.17543E+01,
     &-0.17598E+01,-0.17653E+01,-0.17709E+01,-0.17764E+01,-0.17819E+01,
     &-0.17874E+01,-0.17929E+01,-0.17984E+01,-0.18039E+01,-0.18093E+01,
     &-0.18148E+01,-0.18203E+01,-0.18257E+01,-0.18312E+01,-0.18366E+01,
     &-0.18420E+01,-0.18475E+01,-0.18529E+01,-0.18583E+01,-0.18637E+01,
     &-0.18691E+01,-0.18745E+01,-0.18799E+01,-0.18853E+01,-0.18907E+01,
     &-0.18960E+01
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.10053E+00,-0.22014E+00,-0.28074E+00,-0.32404E+00,-0.35833E+00,
     &-0.38699E+00,-0.41173E+00,-0.43358E+00,-0.45321E+00,-0.47106E+00,
     &-0.48747E+00,-0.50266E+00,-0.51683E+00,-0.53012E+00,-0.54265E+00,
     &-0.55451E+00,-0.56578E+00,-0.57652E+00,-0.58678E+00,-0.59662E+00,
     &-0.60607E+00,-0.61516E+00,-0.62393E+00,-0.63241E+00,-0.64060E+00,
     &-0.64855E+00,-0.65625E+00,-0.66374E+00,-0.67102E+00,-0.67810E+00,
     &-0.68501E+00,-0.69174E+00,-0.69832E+00,-0.70474E+00,-0.71102E+00,
     &-0.71716E+00,-0.72317E+00,-0.72906E+00,-0.73483E+00,-0.74049E+00,
     &-0.74605E+00,-0.75150E+00,-0.75686E+00,-0.76212E+00,-0.76729E+00,
     &-0.77238E+00,-0.77738E+00,-0.78231E+00,-0.78716E+00,-0.79194E+00,
     &-0.79664E+00,-0.80128E+00,-0.80586E+00,-0.81037E+00,-0.81482E+00,
     &-0.81921E+00,-0.82355E+00,-0.82783E+00,-0.83205E+00,-0.83623E+00,
     &-0.84036E+00,-0.84444E+00,-0.84848E+00,-0.85247E+00,-0.85642E+00,
     &-0.86033E+00,-0.86420E+00,-0.86803E+00,-0.87182E+00,-0.87558E+00,
     &-0.87930E+00,-0.88299E+00,-0.88665E+00,-0.89028E+00,-0.89387E+00,
     &-0.89744E+00,-0.90098E+00,-0.90449E+00,-0.90798E+00,-0.91144E+00,
     &-0.91487E+00,-0.91828E+00,-0.92167E+00,-0.92503E+00,-0.92838E+00,
     &-0.93170E+00,-0.93500E+00,-0.93828E+00,-0.94154E+00,-0.94479E+00,
     &-0.94801E+00,-0.95122E+00,-0.95440E+00,-0.95757E+00,-0.96073E+00,
     &-0.96387E+00,-0.96699E+00,-0.97009E+00,-0.97318E+00,-0.97626E+00/

      DATA (BNC04M(I),I=101,200)/
     &-0.97931E+00,-0.98236E+00,-0.98539E+00,-0.98840E+00,-0.99141E+00,
     &-0.99439E+00,-0.99737E+00,-0.10003E+01,-0.10033E+01,-0.10062E+01,
     &-0.10091E+01,-0.10120E+01,-0.10149E+01,-0.10178E+01,-0.10207E+01,
     &-0.10235E+01,-0.10264E+01,-0.10292E+01,-0.10320E+01,-0.10348E+01,
     &-0.10376E+01,-0.10404E+01,-0.10432E+01,-0.10460E+01,-0.10487E+01,
     &-0.10515E+01,-0.10542E+01,-0.10569E+01,-0.10596E+01,-0.10623E+01,
     &-0.10650E+01,-0.10677E+01,-0.10703E+01,-0.10730E+01,-0.10756E+01,
     &-0.10783E+01,-0.10809E+01,-0.10835E+01,-0.10861E+01,-0.10887E+01,
     &-0.10913E+01,-0.10938E+01,-0.10964E+01,-0.10990E+01,-0.11015E+01,
     &-0.11041E+01,-0.11066E+01,-0.11091E+01,-0.11116E+01,-0.11141E+01,
     &-0.11166E+01,-0.11191E+01,-0.11216E+01,-0.11241E+01,-0.11265E+01,
     &-0.11290E+01,-0.11314E+01,-0.11339E+01,-0.11363E+01,-0.11387E+01,
     &-0.11411E+01,-0.11435E+01,-0.11459E+01,-0.11483E+01,-0.11507E+01,
     &-0.11531E+01,-0.11555E+01,-0.11578E+01,-0.11602E+01,-0.11625E+01,
     &-0.11649E+01,-0.11672E+01,-0.11696E+01,-0.11719E+01,-0.11742E+01,
     &-0.11765E+01,-0.11788E+01,-0.11811E+01,-0.11834E+01,-0.11857E+01,
     &-0.11880E+01,-0.11903E+01,-0.11925E+01,-0.11948E+01,-0.11970E+01,
     &-0.11993E+01,-0.12015E+01,-0.12038E+01,-0.12060E+01,-0.12082E+01,
     &-0.12105E+01,-0.12127E+01,-0.12149E+01,-0.12171E+01,-0.12193E+01,
     &-0.12215E+01,-0.12237E+01,-0.12259E+01,-0.12280E+01,-0.12302E+01/

      DATA (BNC04M(I),I=201,300)/
     &-0.12324E+01,-0.12345E+01,-0.12367E+01,-0.12389E+01,-0.12410E+01,
     &-0.12432E+01,-0.12453E+01,-0.12474E+01,-0.12496E+01,-0.12517E+01,
     &-0.12538E+01,-0.12559E+01,-0.12580E+01,-0.12601E+01,-0.12622E+01,
     &-0.12643E+01,-0.12664E+01,-0.12685E+01,-0.12706E+01,-0.12727E+01,
     &-0.12748E+01,-0.12768E+01,-0.12789E+01,-0.12810E+01,-0.12830E+01,
     &-0.12851E+01,-0.12871E+01,-0.12892E+01,-0.12912E+01,-0.12932E+01,
     &-0.12953E+01,-0.12973E+01,-0.12993E+01,-0.13014E+01,-0.13034E+01,
     &-0.13054E+01,-0.13074E+01,-0.13094E+01,-0.13114E+01,-0.13134E+01,
     &-0.13154E+01,-0.13174E+01,-0.13194E+01,-0.13214E+01,-0.13234E+01,
     &-0.13253E+01,-0.13273E+01,-0.13293E+01,-0.13313E+01,-0.13332E+01,
     &-0.13352E+01,-0.13371E+01,-0.13391E+01,-0.13410E+01,-0.13430E+01,
     &-0.13449E+01,-0.13469E+01,-0.13488E+01,-0.13507E+01,-0.13527E+01,
     &-0.13546E+01,-0.13565E+01,-0.13584E+01,-0.13604E+01,-0.13623E+01,
     &-0.13642E+01,-0.13661E+01,-0.13680E+01,-0.13699E+01,-0.13718E+01,
     &-0.13737E+01,-0.13756E+01,-0.13775E+01,-0.13794E+01,-0.13813E+01,
     &-0.13831E+01,-0.13850E+01,-0.13869E+01,-0.13888E+01,-0.13906E+01,
     &-0.13925E+01,-0.13944E+01,-0.13962E+01,-0.13981E+01,-0.13999E+01,
     &-0.14018E+01,-0.14036E+01,-0.14055E+01,-0.14073E+01,-0.14092E+01,
     &-0.14110E+01,-0.14129E+01,-0.14147E+01,-0.14165E+01,-0.14183E+01,
     &-0.14202E+01,-0.14220E+01,-0.14238E+01,-0.14256E+01,-0.14275E+01/

      DATA (BNC04M(I),I=301,400)/
     &-0.14293E+01,-0.14311E+01,-0.14329E+01,-0.14347E+01,-0.14365E+01,
     &-0.14383E+01,-0.14401E+01,-0.14419E+01,-0.14437E+01,-0.14455E+01,
     &-0.14473E+01,-0.14491E+01,-0.14508E+01,-0.14526E+01,-0.14544E+01,
     &-0.14562E+01,-0.14580E+01,-0.14597E+01,-0.14615E+01,-0.14633E+01,
     &-0.14650E+01,-0.14668E+01,-0.14686E+01,-0.14703E+01,-0.14721E+01,
     &-0.14738E+01,-0.14756E+01,-0.14773E+01,-0.14791E+01,-0.14808E+01,
     &-0.14826E+01,-0.14843E+01,-0.14861E+01,-0.14878E+01,-0.14895E+01,
     &-0.14913E+01,-0.14930E+01,-0.14947E+01,-0.14965E+01,-0.14982E+01,
     &-0.14999E+01,-0.15016E+01,-0.15034E+01,-0.15051E+01,-0.15068E+01,
     &-0.15085E+01,-0.15102E+01,-0.15119E+01,-0.15136E+01,-0.15153E+01,
     &-0.15171E+01,-0.15188E+01,-0.15205E+01,-0.15222E+01,-0.15239E+01,
     &-0.15255E+01,-0.15272E+01,-0.15289E+01,-0.15306E+01,-0.15323E+01,
     &-0.15340E+01,-0.15357E+01,-0.15374E+01,-0.15390E+01,-0.15407E+01,
     &-0.15424E+01,-0.15441E+01,-0.15457E+01,-0.15474E+01,-0.15491E+01,
     &-0.15508E+01,-0.15524E+01,-0.15541E+01,-0.15557E+01,-0.15574E+01,
     &-0.15591E+01,-0.15607E+01,-0.15624E+01,-0.15640E+01,-0.15657E+01,
     &-0.15673E+01,-0.15690E+01,-0.15706E+01,-0.15723E+01,-0.15739E+01,
     &-0.15756E+01,-0.15772E+01,-0.15788E+01,-0.15805E+01,-0.15821E+01,
     &-0.15838E+01,-0.15854E+01,-0.15870E+01,-0.15887E+01,-0.15903E+01,
     &-0.15919E+01,-0.15935E+01,-0.15952E+01,-0.15968E+01,-0.15984E+01/

      DATA (BNC04M(I),I=401,500)/
     &-0.16000E+01,-0.16016E+01,-0.16032E+01,-0.16049E+01,-0.16065E+01,
     &-0.16081E+01,-0.16097E+01,-0.16113E+01,-0.16129E+01,-0.16145E+01,
     &-0.16161E+01,-0.16177E+01,-0.16193E+01,-0.16209E+01,-0.16225E+01,
     &-0.16241E+01,-0.16257E+01,-0.16273E+01,-0.16289E+01,-0.16305E+01,
     &-0.16321E+01,-0.16337E+01,-0.16353E+01,-0.16368E+01,-0.16384E+01,
     &-0.16400E+01,-0.16416E+01,-0.16432E+01,-0.16448E+01,-0.16463E+01,
     &-0.16479E+01,-0.16495E+01,-0.16511E+01,-0.16526E+01,-0.16542E+01,
     &-0.16558E+01,-0.16573E+01,-0.16589E+01,-0.16605E+01,-0.16620E+01,
     &-0.16636E+01,-0.16652E+01,-0.16667E+01,-0.16683E+01,-0.16698E+01,
     &-0.16714E+01,-0.16729E+01,-0.16745E+01,-0.16761E+01,-0.16776E+01,
     &-0.16792E+01,-0.16807E+01,-0.16823E+01,-0.16838E+01,-0.16853E+01,
     &-0.16869E+01,-0.16884E+01,-0.16900E+01,-0.16915E+01,-0.16931E+01,
     &-0.16946E+01,-0.16961E+01,-0.16977E+01,-0.16992E+01,-0.17007E+01,
     &-0.17023E+01,-0.17038E+01,-0.17053E+01,-0.17069E+01,-0.17084E+01,
     &-0.17099E+01,-0.17114E+01,-0.17130E+01,-0.17145E+01,-0.17160E+01,
     &-0.17175E+01,-0.17190E+01,-0.17206E+01,-0.17221E+01,-0.17236E+01,
     &-0.17251E+01,-0.17266E+01,-0.17281E+01,-0.17297E+01,-0.17312E+01,
     &-0.17327E+01,-0.17342E+01,-0.17357E+01,-0.17372E+01,-0.17387E+01,
     &-0.17402E+01,-0.17417E+01,-0.17432E+01,-0.17447E+01,-0.17462E+01,
     &-0.17477E+01,-0.17492E+01,-0.17507E+01,-0.17522E+01,-0.17537E+01/

      DATA (BNC04M(I),I=501,600)/
     &-0.17552E+01,-0.17567E+01,-0.17582E+01,-0.17597E+01,-0.17612E+01,
     &-0.17626E+01,-0.17641E+01,-0.17656E+01,-0.17671E+01,-0.17686E+01,
     &-0.17701E+01,-0.17715E+01,-0.17730E+01,-0.17745E+01,-0.17760E+01,
     &-0.17775E+01,-0.17789E+01,-0.17804E+01,-0.17819E+01,-0.17834E+01,
     &-0.17848E+01,-0.17863E+01,-0.17878E+01,-0.17893E+01,-0.17907E+01,
     &-0.17922E+01,-0.17937E+01,-0.17951E+01,-0.17966E+01,-0.17981E+01,
     &-0.17995E+01,-0.18010E+01,-0.18025E+01,-0.18039E+01,-0.18054E+01,
     &-0.18068E+01,-0.18083E+01,-0.18098E+01,-0.18112E+01,-0.18127E+01,
     &-0.18141E+01,-0.18156E+01,-0.18170E+01,-0.18185E+01,-0.18199E+01,
     &-0.18214E+01,-0.18228E+01,-0.18243E+01,-0.18257E+01,-0.18272E+01,
     &-0.18286E+01,-0.18301E+01,-0.18315E+01,-0.18330E+01,-0.18344E+01,
     &-0.18358E+01,-0.18373E+01,-0.18387E+01,-0.18402E+01,-0.18416E+01,
     &-0.18430E+01,-0.18445E+01,-0.18459E+01,-0.18473E+01,-0.18488E+01,
     &-0.18502E+01,-0.18516E+01,-0.18531E+01,-0.18545E+01,-0.18559E+01,
     &-0.18574E+01,-0.18588E+01,-0.18602E+01,-0.18616E+01,-0.18631E+01,
     &-0.18645E+01,-0.18659E+01,-0.18673E+01,-0.18688E+01,-0.18702E+01,
     &-0.18716E+01,-0.18730E+01,-0.18744E+01,-0.18759E+01,-0.18773E+01,
     &-0.18787E+01,-0.18801E+01,-0.18815E+01,-0.18829E+01,-0.18844E+01,
     &-0.18858E+01,-0.18872E+01,-0.18886E+01,-0.18900E+01,-0.18914E+01,
     &-0.18928E+01,-0.18942E+01,-0.18956E+01,-0.18970E+01,-0.18984E+01/

      DATA (BNC04M(I),I=601,700)/
     &-0.19136E+01,-0.19275E+01,-0.19414E+01,-0.19552E+01,-0.19689E+01,
     &-0.19826E+01,-0.19962E+01,-0.20097E+01,-0.20232E+01,-0.20366E+01,
     &-0.20500E+01,-0.20633E+01,-0.20766E+01,-0.20897E+01,-0.21029E+01,
     &-0.21160E+01,-0.21290E+01,-0.21420E+01,-0.21550E+01,-0.21679E+01,
     &-0.21807E+01,-0.21935E+01,-0.22063E+01,-0.22190E+01,-0.22317E+01,
     &-0.22443E+01,-0.22569E+01,-0.22695E+01,-0.22820E+01,-0.22945E+01,
     &-0.23069E+01,-0.23193E+01,-0.23317E+01,-0.23440E+01,-0.23563E+01,
     &-0.23686E+01,-0.23808E+01,-0.23930E+01,-0.24052E+01,-0.24173E+01,
     &-0.24294E+01,-0.24415E+01,-0.24535E+01,-0.24655E+01,-0.24775E+01,
     &-0.24895E+01,-0.25014E+01,-0.25133E+01,-0.25252E+01,-0.25370E+01,
     &-0.25488E+01,-0.25606E+01,-0.25724E+01,-0.25841E+01,-0.25959E+01,
     &-0.26076E+01,-0.26192E+01,-0.26309E+01,-0.26425E+01,-0.26541E+01,
     &-0.26657E+01,-0.26772E+01,-0.26887E+01,-0.27002E+01,-0.27117E+01,
     &-0.27232E+01,-0.27346E+01,-0.27461E+01,-0.27575E+01,-0.27688E+01,
     &-0.27802E+01,-0.27915E+01,-0.28029E+01,-0.28142E+01,-0.28255E+01,
     &-0.28367E+01,-0.28480E+01,-0.28592E+01,-0.28704E+01,-0.28816E+01,
     &-0.28928E+01,-0.29039E+01,-0.29151E+01,-0.29262E+01,-0.29373E+01,
     &-0.29484E+01,-0.29595E+01,-0.29705E+01,-0.29816E+01,-0.29926E+01,
     &-0.30036E+01,-0.30146E+01,-0.30256E+01,-0.30366E+01,-0.30475E+01,
     &-0.30585E+01,-0.30694E+01,-0.30803E+01,-0.30912E+01,-0.31021E+01/

      DATA (BNC04M(I),I=701,741)/
     &-0.31129E+01,-0.31238E+01,-0.31346E+01,-0.31454E+01,-0.31562E+01,
     &-0.31670E+01,-0.31778E+01,-0.31886E+01,-0.31993E+01,-0.32101E+01,
     &-0.32208E+01,-0.32315E+01,-0.32423E+01,-0.32529E+01,-0.32636E+01,
     &-0.32743E+01,-0.32850E+01,-0.32956E+01,-0.33062E+01,-0.33169E+01,
     &-0.33275E+01,-0.33381E+01,-0.33487E+01,-0.33592E+01,-0.33698E+01,
     &-0.33804E+01,-0.33909E+01,-0.34015E+01,-0.34120E+01,-0.34225E+01,
     &-0.34330E+01,-0.34435E+01,-0.34540E+01,-0.34644E+01,-0.34749E+01,
     &-0.34854E+01,-0.34958E+01,-0.35062E+01,-0.35166E+01,-0.35271E+01,
     &-0.35375E+01
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.51007E-01,-0.11414E+00,-0.14751E+00,-0.17208E+00,-0.19205E+00,
     &-0.20913E+00,-0.22418E+00,-0.23774E+00,-0.25013E+00,-0.26159E+00,
     &-0.27229E+00,-0.28235E+00,-0.29187E+00,-0.30091E+00,-0.30954E+00,
     &-0.31781E+00,-0.32575E+00,-0.33340E+00,-0.34079E+00,-0.34794E+00,
     &-0.35487E+00,-0.36160E+00,-0.36815E+00,-0.37452E+00,-0.38073E+00,
     &-0.38679E+00,-0.39271E+00,-0.39849E+00,-0.40415E+00,-0.40969E+00,
     &-0.41512E+00,-0.42044E+00,-0.42566E+00,-0.43078E+00,-0.43581E+00,
     &-0.44075E+00,-0.44560E+00,-0.45037E+00,-0.45506E+00,-0.45968E+00,
     &-0.46422E+00,-0.46869E+00,-0.47309E+00,-0.47742E+00,-0.48169E+00,
     &-0.48590E+00,-0.49005E+00,-0.49414E+00,-0.49817E+00,-0.50215E+00,
     &-0.50607E+00,-0.50995E+00,-0.51378E+00,-0.51755E+00,-0.52129E+00,
     &-0.52498E+00,-0.52862E+00,-0.53223E+00,-0.53579E+00,-0.53931E+00,
     &-0.54280E+00,-0.54626E+00,-0.54968E+00,-0.55306E+00,-0.55642E+00,
     &-0.55974E+00,-0.56303E+00,-0.56630E+00,-0.56954E+00,-0.57276E+00,
     &-0.57595E+00,-0.57911E+00,-0.58226E+00,-0.58538E+00,-0.58849E+00,
     &-0.59157E+00,-0.59464E+00,-0.59769E+00,-0.60072E+00,-0.60373E+00,
     &-0.60673E+00,-0.60972E+00,-0.61269E+00,-0.61565E+00,-0.61860E+00,
     &-0.62153E+00,-0.62445E+00,-0.62737E+00,-0.63027E+00,-0.63316E+00,
     &-0.63604E+00,-0.63891E+00,-0.64177E+00,-0.64462E+00,-0.64746E+00,
     &-0.65029E+00,-0.65312E+00,-0.65593E+00,-0.65874E+00,-0.66154E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.66433E+00,-0.66711E+00,-0.66988E+00,-0.67265E+00,-0.67540E+00,
     &-0.67815E+00,-0.68089E+00,-0.68361E+00,-0.68633E+00,-0.68905E+00,
     &-0.69175E+00,-0.69444E+00,-0.69713E+00,-0.69980E+00,-0.70247E+00,
     &-0.70513E+00,-0.70777E+00,-0.71041E+00,-0.71304E+00,-0.71566E+00,
     &-0.71827E+00,-0.72087E+00,-0.72346E+00,-0.72604E+00,-0.72861E+00,
     &-0.73117E+00,-0.73372E+00,-0.73627E+00,-0.73880E+00,-0.74132E+00,
     &-0.74383E+00,-0.74634E+00,-0.74883E+00,-0.75132E+00,-0.75379E+00,
     &-0.75626E+00,-0.75871E+00,-0.76116E+00,-0.76360E+00,-0.76603E+00,
     &-0.76844E+00,-0.77085E+00,-0.77325E+00,-0.77565E+00,-0.77803E+00,
     &-0.78040E+00,-0.78277E+00,-0.78512E+00,-0.78747E+00,-0.78981E+00,
     &-0.79214E+00,-0.79446E+00,-0.79677E+00,-0.79908E+00,-0.80138E+00,
     &-0.80366E+00,-0.80594E+00,-0.80822E+00,-0.81048E+00,-0.81274E+00,
     &-0.81498E+00,-0.81723E+00,-0.81946E+00,-0.82168E+00,-0.82390E+00,
     &-0.82611E+00,-0.82831E+00,-0.83051E+00,-0.83269E+00,-0.83487E+00,
     &-0.83705E+00,-0.83921E+00,-0.84137E+00,-0.84352E+00,-0.84567E+00,
     &-0.84780E+00,-0.84993E+00,-0.85206E+00,-0.85417E+00,-0.85628E+00,
     &-0.85839E+00,-0.86048E+00,-0.86257E+00,-0.86466E+00,-0.86673E+00,
     &-0.86880E+00,-0.87087E+00,-0.87292E+00,-0.87497E+00,-0.87702E+00,
     &-0.87906E+00,-0.88109E+00,-0.88312E+00,-0.88514E+00,-0.88715E+00,
     &-0.88916E+00,-0.89116E+00,-0.89315E+00,-0.89514E+00,-0.89713E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.89911E+00,-0.90108E+00,-0.90305E+00,-0.90501E+00,-0.90696E+00,
     &-0.90891E+00,-0.91086E+00,-0.91280E+00,-0.91473E+00,-0.91666E+00,
     &-0.91858E+00,-0.92050E+00,-0.92241E+00,-0.92431E+00,-0.92621E+00,
     &-0.92811E+00,-0.93000E+00,-0.93188E+00,-0.93376E+00,-0.93564E+00,
     &-0.93751E+00,-0.93937E+00,-0.94123E+00,-0.94309E+00,-0.94494E+00,
     &-0.94678E+00,-0.94862E+00,-0.95046E+00,-0.95229E+00,-0.95411E+00,
     &-0.95593E+00,-0.95775E+00,-0.95956E+00,-0.96136E+00,-0.96317E+00,
     &-0.96496E+00,-0.96676E+00,-0.96854E+00,-0.97033E+00,-0.97210E+00,
     &-0.97388E+00,-0.97565E+00,-0.97741E+00,-0.97917E+00,-0.98093E+00,
     &-0.98268E+00,-0.98443E+00,-0.98617E+00,-0.98791E+00,-0.98965E+00,
     &-0.99138E+00,-0.99310E+00,-0.99483E+00,-0.99654E+00,-0.99826E+00,
     &-0.99997E+00,-0.10017E+01,-0.10034E+01,-0.10051E+01,-0.10068E+01,
     &-0.10085E+01,-0.10101E+01,-0.10118E+01,-0.10135E+01,-0.10152E+01,
     &-0.10168E+01,-0.10185E+01,-0.10202E+01,-0.10218E+01,-0.10235E+01,
     &-0.10251E+01,-0.10268E+01,-0.10284E+01,-0.10301E+01,-0.10317E+01,
     &-0.10333E+01,-0.10350E+01,-0.10366E+01,-0.10382E+01,-0.10398E+01,
     &-0.10414E+01,-0.10430E+01,-0.10446E+01,-0.10462E+01,-0.10478E+01,
     &-0.10494E+01,-0.10510E+01,-0.10526E+01,-0.10542E+01,-0.10558E+01,
     &-0.10574E+01,-0.10589E+01,-0.10605E+01,-0.10621E+01,-0.10636E+01,
     &-0.10652E+01,-0.10668E+01,-0.10683E+01,-0.10699E+01,-0.10714E+01/

      DATA (BNC05M(I),I=301,400)/
     &-0.10730E+01,-0.10745E+01,-0.10760E+01,-0.10776E+01,-0.10791E+01,
     &-0.10806E+01,-0.10822E+01,-0.10837E+01,-0.10852E+01,-0.10867E+01,
     &-0.10882E+01,-0.10898E+01,-0.10913E+01,-0.10928E+01,-0.10943E+01,
     &-0.10958E+01,-0.10973E+01,-0.10988E+01,-0.11002E+01,-0.11017E+01,
     &-0.11032E+01,-0.11047E+01,-0.11062E+01,-0.11076E+01,-0.11091E+01,
     &-0.11106E+01,-0.11121E+01,-0.11135E+01,-0.11150E+01,-0.11164E+01,
     &-0.11179E+01,-0.11193E+01,-0.11208E+01,-0.11222E+01,-0.11237E+01,
     &-0.11251E+01,-0.11266E+01,-0.11280E+01,-0.11294E+01,-0.11309E+01,
     &-0.11323E+01,-0.11337E+01,-0.11351E+01,-0.11365E+01,-0.11380E+01,
     &-0.11394E+01,-0.11408E+01,-0.11422E+01,-0.11436E+01,-0.11450E+01,
     &-0.11464E+01,-0.11478E+01,-0.11492E+01,-0.11506E+01,-0.11520E+01,
     &-0.11534E+01,-0.11548E+01,-0.11561E+01,-0.11575E+01,-0.11589E+01,
     &-0.11603E+01,-0.11617E+01,-0.11630E+01,-0.11644E+01,-0.11658E+01,
     &-0.11671E+01,-0.11685E+01,-0.11698E+01,-0.11712E+01,-0.11726E+01,
     &-0.11739E+01,-0.11753E+01,-0.11766E+01,-0.11779E+01,-0.11793E+01,
     &-0.11806E+01,-0.11820E+01,-0.11833E+01,-0.11846E+01,-0.11860E+01,
     &-0.11873E+01,-0.11886E+01,-0.11899E+01,-0.11913E+01,-0.11926E+01,
     &-0.11939E+01,-0.11952E+01,-0.11965E+01,-0.11978E+01,-0.11992E+01,
     &-0.12005E+01,-0.12018E+01,-0.12031E+01,-0.12044E+01,-0.12057E+01,
     &-0.12070E+01,-0.12083E+01,-0.12096E+01,-0.12108E+01,-0.12121E+01/

      DATA (BNC05M(I),I=401,500)/
     &-0.12134E+01,-0.12147E+01,-0.12160E+01,-0.12173E+01,-0.12185E+01,
     &-0.12198E+01,-0.12211E+01,-0.12224E+01,-0.12236E+01,-0.12249E+01,
     &-0.12262E+01,-0.12274E+01,-0.12287E+01,-0.12299E+01,-0.12312E+01,
     &-0.12325E+01,-0.12337E+01,-0.12350E+01,-0.12362E+01,-0.12375E+01,
     &-0.12387E+01,-0.12400E+01,-0.12412E+01,-0.12424E+01,-0.12437E+01,
     &-0.12449E+01,-0.12461E+01,-0.12474E+01,-0.12486E+01,-0.12498E+01,
     &-0.12511E+01,-0.12523E+01,-0.12535E+01,-0.12547E+01,-0.12560E+01,
     &-0.12572E+01,-0.12584E+01,-0.12596E+01,-0.12608E+01,-0.12620E+01,
     &-0.12632E+01,-0.12644E+01,-0.12656E+01,-0.12669E+01,-0.12681E+01,
     &-0.12693E+01,-0.12705E+01,-0.12717E+01,-0.12728E+01,-0.12740E+01,
     &-0.12752E+01,-0.12764E+01,-0.12776E+01,-0.12788E+01,-0.12800E+01,
     &-0.12812E+01,-0.12823E+01,-0.12835E+01,-0.12847E+01,-0.12859E+01,
     &-0.12871E+01,-0.12882E+01,-0.12894E+01,-0.12906E+01,-0.12917E+01,
     &-0.12929E+01,-0.12941E+01,-0.12952E+01,-0.12964E+01,-0.12976E+01,
     &-0.12987E+01,-0.12999E+01,-0.13010E+01,-0.13022E+01,-0.13033E+01,
     &-0.13045E+01,-0.13056E+01,-0.13068E+01,-0.13079E+01,-0.13091E+01,
     &-0.13102E+01,-0.13114E+01,-0.13125E+01,-0.13136E+01,-0.13148E+01,
     &-0.13159E+01,-0.13170E+01,-0.13182E+01,-0.13193E+01,-0.13204E+01,
     &-0.13216E+01,-0.13227E+01,-0.13238E+01,-0.13249E+01,-0.13261E+01,
     &-0.13272E+01,-0.13283E+01,-0.13294E+01,-0.13305E+01,-0.13317E+01/

      DATA (BNC05M(I),I=501,600)/
     &-0.13328E+01,-0.13339E+01,-0.13350E+01,-0.13361E+01,-0.13372E+01,
     &-0.13383E+01,-0.13394E+01,-0.13405E+01,-0.13416E+01,-0.13427E+01,
     &-0.13438E+01,-0.13449E+01,-0.13460E+01,-0.13471E+01,-0.13482E+01,
     &-0.13493E+01,-0.13504E+01,-0.13515E+01,-0.13526E+01,-0.13537E+01,
     &-0.13547E+01,-0.13558E+01,-0.13569E+01,-0.13580E+01,-0.13591E+01,
     &-0.13602E+01,-0.13612E+01,-0.13623E+01,-0.13634E+01,-0.13645E+01,
     &-0.13655E+01,-0.13666E+01,-0.13677E+01,-0.13687E+01,-0.13698E+01,
     &-0.13709E+01,-0.13719E+01,-0.13730E+01,-0.13741E+01,-0.13751E+01,
     &-0.13762E+01,-0.13773E+01,-0.13783E+01,-0.13794E+01,-0.13804E+01,
     &-0.13815E+01,-0.13825E+01,-0.13836E+01,-0.13846E+01,-0.13857E+01,
     &-0.13867E+01,-0.13878E+01,-0.13888E+01,-0.13899E+01,-0.13909E+01,
     &-0.13920E+01,-0.13930E+01,-0.13940E+01,-0.13951E+01,-0.13961E+01,
     &-0.13971E+01,-0.13982E+01,-0.13992E+01,-0.14002E+01,-0.14013E+01,
     &-0.14023E+01,-0.14033E+01,-0.14044E+01,-0.14054E+01,-0.14064E+01,
     &-0.14074E+01,-0.14085E+01,-0.14095E+01,-0.14105E+01,-0.14115E+01,
     &-0.14125E+01,-0.14136E+01,-0.14146E+01,-0.14156E+01,-0.14166E+01,
     &-0.14176E+01,-0.14186E+01,-0.14196E+01,-0.14207E+01,-0.14217E+01,
     &-0.14227E+01,-0.14237E+01,-0.14247E+01,-0.14257E+01,-0.14267E+01,
     &-0.14277E+01,-0.14287E+01,-0.14297E+01,-0.14307E+01,-0.14317E+01,
     &-0.14327E+01,-0.14337E+01,-0.14347E+01,-0.14357E+01,-0.14367E+01/

      DATA (BNC05M(I),I=601,700)/
     &-0.14473E+01,-0.14571E+01,-0.14668E+01,-0.14763E+01,-0.14858E+01,
     &-0.14952E+01,-0.15045E+01,-0.15137E+01,-0.15229E+01,-0.15319E+01,
     &-0.15409E+01,-0.15498E+01,-0.15586E+01,-0.15674E+01,-0.15761E+01,
     &-0.15847E+01,-0.15933E+01,-0.16017E+01,-0.16102E+01,-0.16185E+01,
     &-0.16268E+01,-0.16351E+01,-0.16433E+01,-0.16514E+01,-0.16595E+01,
     &-0.16675E+01,-0.16754E+01,-0.16834E+01,-0.16912E+01,-0.16990E+01,
     &-0.17068E+01,-0.17145E+01,-0.17222E+01,-0.17298E+01,-0.17374E+01,
     &-0.17450E+01,-0.17525E+01,-0.17599E+01,-0.17674E+01,-0.17747E+01,
     &-0.17821E+01,-0.17894E+01,-0.17967E+01,-0.18039E+01,-0.18111E+01,
     &-0.18182E+01,-0.18254E+01,-0.18325E+01,-0.18395E+01,-0.18466E+01,
     &-0.18535E+01,-0.18605E+01,-0.18674E+01,-0.18743E+01,-0.18812E+01,
     &-0.18881E+01,-0.18949E+01,-0.19017E+01,-0.19084E+01,-0.19152E+01,
     &-0.19219E+01,-0.19285E+01,-0.19352E+01,-0.19418E+01,-0.19484E+01,
     &-0.19550E+01,-0.19616E+01,-0.19681E+01,-0.19746E+01,-0.19811E+01,
     &-0.19876E+01,-0.19940E+01,-0.20004E+01,-0.20068E+01,-0.20132E+01,
     &-0.20196E+01,-0.20259E+01,-0.20322E+01,-0.20385E+01,-0.20448E+01,
     &-0.20510E+01,-0.20573E+01,-0.20635E+01,-0.20697E+01,-0.20759E+01,
     &-0.20821E+01,-0.20882E+01,-0.20943E+01,-0.21005E+01,-0.21066E+01,
     &-0.21126E+01,-0.21187E+01,-0.21247E+01,-0.21308E+01,-0.21368E+01,
     &-0.21428E+01,-0.21488E+01,-0.21548E+01,-0.21607E+01,-0.21667E+01/

      DATA (BNC05M(I),I=701,741)/
     &-0.21726E+01,-0.21785E+01,-0.21844E+01,-0.21903E+01,-0.21962E+01,
     &-0.22020E+01,-0.22079E+01,-0.22137E+01,-0.22195E+01,-0.22253E+01,
     &-0.22311E+01,-0.22369E+01,-0.22427E+01,-0.22484E+01,-0.22542E+01,
     &-0.22599E+01,-0.22656E+01,-0.22714E+01,-0.22771E+01,-0.22827E+01,
     &-0.22884E+01,-0.22941E+01,-0.22997E+01,-0.23054E+01,-0.23110E+01,
     &-0.23167E+01,-0.23223E+01,-0.23279E+01,-0.23335E+01,-0.23390E+01,
     &-0.23446E+01,-0.23502E+01,-0.23557E+01,-0.23613E+01,-0.23668E+01,
     &-0.23723E+01,-0.23779E+01,-0.23834E+01,-0.23889E+01,-0.23944E+01,
     &-0.23998E+01
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.49470E-01,-0.10575E+00,-0.13284E+00,-0.15144E+00,-0.16564E+00,
     &-0.17711E+00,-0.18670E+00,-0.19489E+00,-0.20203E+00,-0.20831E+00,
     &-0.21392E+00,-0.21895E+00,-0.22351E+00,-0.22765E+00,-0.23144E+00,
     &-0.23493E+00,-0.23814E+00,-0.24112E+00,-0.24388E+00,-0.24646E+00,
     &-0.24886E+00,-0.25110E+00,-0.25321E+00,-0.25519E+00,-0.25705E+00,
     &-0.25880E+00,-0.26046E+00,-0.26203E+00,-0.26351E+00,-0.26492E+00,
     &-0.26626E+00,-0.26753E+00,-0.26874E+00,-0.26989E+00,-0.27100E+00,
     &-0.27205E+00,-0.27306E+00,-0.27402E+00,-0.27495E+00,-0.27584E+00,
     &-0.27669E+00,-0.27752E+00,-0.27831E+00,-0.27907E+00,-0.27981E+00,
     &-0.28052E+00,-0.28121E+00,-0.28187E+00,-0.28252E+00,-0.28314E+00,
     &-0.28375E+00,-0.28433E+00,-0.28490E+00,-0.28545E+00,-0.28599E+00,
     &-0.28651E+00,-0.28702E+00,-0.28751E+00,-0.28799E+00,-0.28845E+00,
     &-0.28891E+00,-0.28935E+00,-0.28977E+00,-0.29019E+00,-0.29059E+00,
     &-0.29098E+00,-0.29136E+00,-0.29172E+00,-0.29208E+00,-0.29242E+00,
     &-0.29275E+00,-0.29307E+00,-0.29338E+00,-0.29367E+00,-0.29396E+00,
     &-0.29423E+00,-0.29449E+00,-0.29474E+00,-0.29498E+00,-0.29521E+00,
     &-0.29543E+00,-0.29563E+00,-0.29582E+00,-0.29600E+00,-0.29617E+00,
     &-0.29633E+00,-0.29648E+00,-0.29661E+00,-0.29674E+00,-0.29685E+00,
     &-0.29696E+00,-0.29705E+00,-0.29713E+00,-0.29720E+00,-0.29726E+00,
     &-0.29732E+00,-0.29736E+00,-0.29739E+00,-0.29742E+00,-0.29743E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.29744E+00,-0.29744E+00,-0.29743E+00,-0.29741E+00,-0.29738E+00,
     &-0.29735E+00,-0.29731E+00,-0.29726E+00,-0.29721E+00,-0.29715E+00,
     &-0.29708E+00,-0.29701E+00,-0.29693E+00,-0.29685E+00,-0.29676E+00,
     &-0.29667E+00,-0.29658E+00,-0.29647E+00,-0.29637E+00,-0.29626E+00,
     &-0.29615E+00,-0.29603E+00,-0.29591E+00,-0.29579E+00,-0.29566E+00,
     &-0.29554E+00,-0.29540E+00,-0.29527E+00,-0.29513E+00,-0.29500E+00,
     &-0.29486E+00,-0.29471E+00,-0.29457E+00,-0.29442E+00,-0.29427E+00,
     &-0.29413E+00,-0.29397E+00,-0.29382E+00,-0.29367E+00,-0.29351E+00,
     &-0.29336E+00,-0.29320E+00,-0.29304E+00,-0.29288E+00,-0.29272E+00,
     &-0.29256E+00,-0.29239E+00,-0.29223E+00,-0.29207E+00,-0.29190E+00,
     &-0.29173E+00,-0.29157E+00,-0.29140E+00,-0.29123E+00,-0.29106E+00,
     &-0.29089E+00,-0.29072E+00,-0.29055E+00,-0.29038E+00,-0.29021E+00,
     &-0.29004E+00,-0.28987E+00,-0.28970E+00,-0.28953E+00,-0.28935E+00,
     &-0.28918E+00,-0.28901E+00,-0.28883E+00,-0.28866E+00,-0.28848E+00,
     &-0.28831E+00,-0.28814E+00,-0.28796E+00,-0.28779E+00,-0.28761E+00,
     &-0.28744E+00,-0.28726E+00,-0.28709E+00,-0.28691E+00,-0.28674E+00,
     &-0.28656E+00,-0.28639E+00,-0.28621E+00,-0.28604E+00,-0.28586E+00,
     &-0.28569E+00,-0.28551E+00,-0.28534E+00,-0.28516E+00,-0.28499E+00,
     &-0.28481E+00,-0.28464E+00,-0.28447E+00,-0.28429E+00,-0.28412E+00,
     &-0.28395E+00,-0.28377E+00,-0.28360E+00,-0.28343E+00,-0.28325E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.28308E+00,-0.28291E+00,-0.28274E+00,-0.28257E+00,-0.28240E+00,
     &-0.28223E+00,-0.28205E+00,-0.28188E+00,-0.28171E+00,-0.28155E+00,
     &-0.28138E+00,-0.28121E+00,-0.28104E+00,-0.28087E+00,-0.28070E+00,
     &-0.28053E+00,-0.28037E+00,-0.28020E+00,-0.28003E+00,-0.27987E+00,
     &-0.27970E+00,-0.27954E+00,-0.27937E+00,-0.27921E+00,-0.27905E+00,
     &-0.27888E+00,-0.27872E+00,-0.27856E+00,-0.27839E+00,-0.27823E+00,
     &-0.27807E+00,-0.27791E+00,-0.27775E+00,-0.27759E+00,-0.27743E+00,
     &-0.27727E+00,-0.27712E+00,-0.27696E+00,-0.27680E+00,-0.27664E+00,
     &-0.27649E+00,-0.27633E+00,-0.27618E+00,-0.27602E+00,-0.27587E+00,
     &-0.27571E+00,-0.27556E+00,-0.27541E+00,-0.27526E+00,-0.27510E+00,
     &-0.27495E+00,-0.27480E+00,-0.27465E+00,-0.27450E+00,-0.27435E+00,
     &-0.27421E+00,-0.27406E+00,-0.27391E+00,-0.27376E+00,-0.27362E+00,
     &-0.27347E+00,-0.27333E+00,-0.27318E+00,-0.27304E+00,-0.27290E+00,
     &-0.27275E+00,-0.27261E+00,-0.27247E+00,-0.27233E+00,-0.27219E+00,
     &-0.27205E+00,-0.27191E+00,-0.27177E+00,-0.27163E+00,-0.27149E+00,
     &-0.27136E+00,-0.27122E+00,-0.27109E+00,-0.27095E+00,-0.27082E+00,
     &-0.27068E+00,-0.27055E+00,-0.27042E+00,-0.27028E+00,-0.27015E+00,
     &-0.27002E+00,-0.26989E+00,-0.26976E+00,-0.26963E+00,-0.26950E+00,
     &-0.26938E+00,-0.26925E+00,-0.26912E+00,-0.26900E+00,-0.26887E+00,
     &-0.26875E+00,-0.26862E+00,-0.26850E+00,-0.26837E+00,-0.26825E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.26813E+00,-0.26801E+00,-0.26789E+00,-0.26777E+00,-0.26765E+00,
     &-0.26753E+00,-0.26741E+00,-0.26729E+00,-0.26718E+00,-0.26706E+00,
     &-0.26694E+00,-0.26683E+00,-0.26672E+00,-0.26660E+00,-0.26649E+00,
     &-0.26638E+00,-0.26626E+00,-0.26615E+00,-0.26604E+00,-0.26593E+00,
     &-0.26582E+00,-0.26571E+00,-0.26560E+00,-0.26550E+00,-0.26539E+00,
     &-0.26528E+00,-0.26518E+00,-0.26507E+00,-0.26497E+00,-0.26486E+00,
     &-0.26476E+00,-0.26466E+00,-0.26455E+00,-0.26445E+00,-0.26435E+00,
     &-0.26425E+00,-0.26415E+00,-0.26405E+00,-0.26395E+00,-0.26385E+00,
     &-0.26376E+00,-0.26366E+00,-0.26356E+00,-0.26347E+00,-0.26337E+00,
     &-0.26328E+00,-0.26319E+00,-0.26309E+00,-0.26300E+00,-0.26291E+00,
     &-0.26282E+00,-0.26273E+00,-0.26264E+00,-0.26255E+00,-0.26246E+00,
     &-0.26237E+00,-0.26228E+00,-0.26219E+00,-0.26211E+00,-0.26202E+00,
     &-0.26194E+00,-0.26185E+00,-0.26177E+00,-0.26168E+00,-0.26160E+00,
     &-0.26152E+00,-0.26144E+00,-0.26136E+00,-0.26128E+00,-0.26120E+00,
     &-0.26112E+00,-0.26104E+00,-0.26096E+00,-0.26088E+00,-0.26081E+00,
     &-0.26073E+00,-0.26065E+00,-0.26058E+00,-0.26050E+00,-0.26043E+00,
     &-0.26036E+00,-0.26028E+00,-0.26021E+00,-0.26014E+00,-0.26007E+00,
     &-0.26000E+00,-0.25993E+00,-0.25986E+00,-0.25979E+00,-0.25972E+00,
     &-0.25965E+00,-0.25959E+00,-0.25952E+00,-0.25945E+00,-0.25939E+00,
     &-0.25932E+00,-0.25926E+00,-0.25920E+00,-0.25913E+00,-0.25907E+00/

      DATA (BNC06M(I),I=401,500)/
     &-0.25901E+00,-0.25895E+00,-0.25889E+00,-0.25883E+00,-0.25877E+00,
     &-0.25871E+00,-0.25865E+00,-0.25859E+00,-0.25853E+00,-0.25848E+00,
     &-0.25842E+00,-0.25837E+00,-0.25831E+00,-0.25826E+00,-0.25820E+00,
     &-0.25815E+00,-0.25810E+00,-0.25804E+00,-0.25799E+00,-0.25794E+00,
     &-0.25789E+00,-0.25784E+00,-0.25779E+00,-0.25774E+00,-0.25769E+00,
     &-0.25765E+00,-0.25760E+00,-0.25755E+00,-0.25751E+00,-0.25746E+00,
     &-0.25741E+00,-0.25737E+00,-0.25733E+00,-0.25728E+00,-0.25724E+00,
     &-0.25720E+00,-0.25715E+00,-0.25711E+00,-0.25707E+00,-0.25703E+00,
     &-0.25699E+00,-0.25695E+00,-0.25692E+00,-0.25688E+00,-0.25684E+00,
     &-0.25680E+00,-0.25677E+00,-0.25673E+00,-0.25669E+00,-0.25666E+00,
     &-0.25662E+00,-0.25659E+00,-0.25656E+00,-0.25652E+00,-0.25649E+00,
     &-0.25646E+00,-0.25643E+00,-0.25640E+00,-0.25637E+00,-0.25634E+00,
     &-0.25631E+00,-0.25628E+00,-0.25625E+00,-0.25623E+00,-0.25620E+00,
     &-0.25617E+00,-0.25615E+00,-0.25612E+00,-0.25609E+00,-0.25607E+00,
     &-0.25605E+00,-0.25602E+00,-0.25600E+00,-0.25598E+00,-0.25596E+00,
     &-0.25593E+00,-0.25591E+00,-0.25589E+00,-0.25587E+00,-0.25585E+00,
     &-0.25583E+00,-0.25582E+00,-0.25580E+00,-0.25578E+00,-0.25576E+00,
     &-0.25575E+00,-0.25573E+00,-0.25571E+00,-0.25570E+00,-0.25569E+00,
     &-0.25567E+00,-0.25566E+00,-0.25564E+00,-0.25563E+00,-0.25562E+00,
     &-0.25561E+00,-0.25560E+00,-0.25559E+00,-0.25558E+00,-0.25557E+00/

      DATA (BNC06M(I),I=501,600)/
     &-0.25556E+00,-0.25555E+00,-0.25554E+00,-0.25553E+00,-0.25553E+00,
     &-0.25552E+00,-0.25551E+00,-0.25551E+00,-0.25550E+00,-0.25550E+00,
     &-0.25549E+00,-0.25549E+00,-0.25549E+00,-0.25548E+00,-0.25548E+00,
     &-0.25548E+00,-0.25548E+00,-0.25548E+00,-0.25547E+00,-0.25547E+00,
     &-0.25547E+00,-0.25548E+00,-0.25548E+00,-0.25548E+00,-0.25548E+00,
     &-0.25548E+00,-0.25549E+00,-0.25549E+00,-0.25549E+00,-0.25550E+00,
     &-0.25550E+00,-0.25551E+00,-0.25551E+00,-0.25552E+00,-0.25553E+00,
     &-0.25553E+00,-0.25554E+00,-0.25555E+00,-0.25556E+00,-0.25557E+00,
     &-0.25558E+00,-0.25559E+00,-0.25560E+00,-0.25561E+00,-0.25562E+00,
     &-0.25563E+00,-0.25564E+00,-0.25566E+00,-0.25567E+00,-0.25568E+00,
     &-0.25570E+00,-0.25571E+00,-0.25573E+00,-0.25574E+00,-0.25576E+00,
     &-0.25577E+00,-0.25579E+00,-0.25581E+00,-0.25582E+00,-0.25584E+00,
     &-0.25586E+00,-0.25588E+00,-0.25590E+00,-0.25592E+00,-0.25594E+00,
     &-0.25596E+00,-0.25598E+00,-0.25600E+00,-0.25602E+00,-0.25604E+00,
     &-0.25607E+00,-0.25609E+00,-0.25611E+00,-0.25614E+00,-0.25616E+00,
     &-0.25619E+00,-0.25621E+00,-0.25624E+00,-0.25626E+00,-0.25629E+00,
     &-0.25632E+00,-0.25634E+00,-0.25637E+00,-0.25640E+00,-0.25643E+00,
     &-0.25646E+00,-0.25649E+00,-0.25651E+00,-0.25654E+00,-0.25658E+00,
     &-0.25661E+00,-0.25664E+00,-0.25667E+00,-0.25670E+00,-0.25673E+00,
     &-0.25677E+00,-0.25680E+00,-0.25683E+00,-0.25687E+00,-0.25690E+00/

      DATA (BNC06M(I),I=601,700)/
     &-0.25730E+00,-0.25771E+00,-0.25816E+00,-0.25865E+00,-0.25917E+00,
     &-0.25973E+00,-0.26033E+00,-0.26096E+00,-0.26163E+00,-0.26233E+00,
     &-0.26307E+00,-0.26383E+00,-0.26463E+00,-0.26547E+00,-0.26633E+00,
     &-0.26722E+00,-0.26815E+00,-0.26910E+00,-0.27009E+00,-0.27110E+00,
     &-0.27214E+00,-0.27320E+00,-0.27430E+00,-0.27542E+00,-0.27657E+00,
     &-0.27774E+00,-0.27894E+00,-0.28017E+00,-0.28142E+00,-0.28269E+00,
     &-0.28399E+00,-0.28531E+00,-0.28666E+00,-0.28803E+00,-0.28942E+00,
     &-0.29083E+00,-0.29226E+00,-0.29372E+00,-0.29520E+00,-0.29669E+00,
     &-0.29821E+00,-0.29975E+00,-0.30131E+00,-0.30289E+00,-0.30449E+00,
     &-0.30610E+00,-0.30774E+00,-0.30939E+00,-0.31107E+00,-0.31276E+00,
     &-0.31447E+00,-0.31619E+00,-0.31794E+00,-0.31970E+00,-0.32147E+00,
     &-0.32327E+00,-0.32508E+00,-0.32690E+00,-0.32875E+00,-0.33060E+00,
     &-0.33248E+00,-0.33437E+00,-0.33627E+00,-0.33819E+00,-0.34012E+00,
     &-0.34207E+00,-0.34403E+00,-0.34601E+00,-0.34800E+00,-0.35001E+00,
     &-0.35203E+00,-0.35406E+00,-0.35610E+00,-0.35816E+00,-0.36023E+00,
     &-0.36232E+00,-0.36441E+00,-0.36652E+00,-0.36864E+00,-0.37078E+00,
     &-0.37292E+00,-0.37508E+00,-0.37725E+00,-0.37943E+00,-0.38162E+00,
     &-0.38383E+00,-0.38604E+00,-0.38827E+00,-0.39051E+00,-0.39275E+00,
     &-0.39501E+00,-0.39728E+00,-0.39956E+00,-0.40185E+00,-0.40415E+00,
     &-0.40646E+00,-0.40879E+00,-0.41112E+00,-0.41346E+00,-0.41581E+00/

      DATA (BNC06M(I),I=701,741)/
     &-0.41817E+00,-0.42054E+00,-0.42292E+00,-0.42530E+00,-0.42770E+00,
     &-0.43011E+00,-0.43252E+00,-0.43495E+00,-0.43738E+00,-0.43982E+00,
     &-0.44227E+00,-0.44473E+00,-0.44720E+00,-0.44967E+00,-0.45216E+00,
     &-0.45465E+00,-0.45715E+00,-0.45966E+00,-0.46218E+00,-0.46470E+00,
     &-0.46723E+00,-0.46977E+00,-0.47232E+00,-0.47487E+00,-0.47744E+00,
     &-0.48001E+00,-0.48258E+00,-0.48517E+00,-0.48776E+00,-0.49036E+00,
     &-0.49296E+00,-0.49558E+00,-0.49820E+00,-0.50082E+00,-0.50346E+00,
     &-0.50610E+00,-0.50874E+00,-0.51140E+00,-0.51406E+00,-0.51672E+00,
     &-0.51940E+00
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.10030E+00,-0.21886E+00,-0.27851E+00,-0.32089E+00,-0.35431E+00,
     &-0.38211E+00,-0.40602E+00,-0.42706E+00,-0.44588E+00,-0.46295E+00,
     &-0.47857E+00,-0.49300E+00,-0.50642E+00,-0.51897E+00,-0.53076E+00,
     &-0.54189E+00,-0.55244E+00,-0.56247E+00,-0.57203E+00,-0.58117E+00,
     &-0.58993E+00,-0.59834E+00,-0.60644E+00,-0.61425E+00,-0.62178E+00,
     &-0.62907E+00,-0.63613E+00,-0.64298E+00,-0.64962E+00,-0.65608E+00,
     &-0.66236E+00,-0.66848E+00,-0.67445E+00,-0.68027E+00,-0.68595E+00,
     &-0.69151E+00,-0.69694E+00,-0.70225E+00,-0.70745E+00,-0.71255E+00,
     &-0.71755E+00,-0.72245E+00,-0.72726E+00,-0.73198E+00,-0.73662E+00,
     &-0.74117E+00,-0.74566E+00,-0.75006E+00,-0.75440E+00,-0.75867E+00,
     &-0.76287E+00,-0.76702E+00,-0.77110E+00,-0.77512E+00,-0.77909E+00,
     &-0.78300E+00,-0.78686E+00,-0.79067E+00,-0.79443E+00,-0.79815E+00,
     &-0.80182E+00,-0.80545E+00,-0.80903E+00,-0.81257E+00,-0.81608E+00,
     &-0.81954E+00,-0.82297E+00,-0.82637E+00,-0.82972E+00,-0.83305E+00,
     &-0.83634E+00,-0.83960E+00,-0.84283E+00,-0.84603E+00,-0.84920E+00,
     &-0.85234E+00,-0.85546E+00,-0.85855E+00,-0.86161E+00,-0.86465E+00,
     &-0.86767E+00,-0.87066E+00,-0.87362E+00,-0.87657E+00,-0.87949E+00,
     &-0.88240E+00,-0.88528E+00,-0.88814E+00,-0.89099E+00,-0.89381E+00,
     &-0.89661E+00,-0.89940E+00,-0.90217E+00,-0.90492E+00,-0.90766E+00,
     &-0.91037E+00,-0.91308E+00,-0.91576E+00,-0.91843E+00,-0.92109E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.92372E+00,-0.92635E+00,-0.92896E+00,-0.93155E+00,-0.93414E+00,
     &-0.93670E+00,-0.93926E+00,-0.94180E+00,-0.94433E+00,-0.94684E+00,
     &-0.94935E+00,-0.95184E+00,-0.95431E+00,-0.95678E+00,-0.95923E+00,
     &-0.96168E+00,-0.96411E+00,-0.96653E+00,-0.96893E+00,-0.97133E+00,
     &-0.97372E+00,-0.97609E+00,-0.97845E+00,-0.98081E+00,-0.98315E+00,
     &-0.98549E+00,-0.98781E+00,-0.99012E+00,-0.99243E+00,-0.99472E+00,
     &-0.99700E+00,-0.99928E+00,-0.10015E+01,-0.10038E+01,-0.10060E+01,
     &-0.10083E+01,-0.10105E+01,-0.10127E+01,-0.10150E+01,-0.10172E+01,
     &-0.10194E+01,-0.10215E+01,-0.10237E+01,-0.10259E+01,-0.10281E+01,
     &-0.10302E+01,-0.10324E+01,-0.10345E+01,-0.10366E+01,-0.10388E+01,
     &-0.10409E+01,-0.10430E+01,-0.10451E+01,-0.10472E+01,-0.10493E+01,
     &-0.10514E+01,-0.10534E+01,-0.10555E+01,-0.10576E+01,-0.10596E+01,
     &-0.10617E+01,-0.10637E+01,-0.10658E+01,-0.10678E+01,-0.10698E+01,
     &-0.10718E+01,-0.10738E+01,-0.10759E+01,-0.10779E+01,-0.10798E+01,
     &-0.10818E+01,-0.10838E+01,-0.10858E+01,-0.10878E+01,-0.10897E+01,
     &-0.10917E+01,-0.10937E+01,-0.10956E+01,-0.10976E+01,-0.10995E+01,
     &-0.11014E+01,-0.11034E+01,-0.11053E+01,-0.11072E+01,-0.11091E+01,
     &-0.11110E+01,-0.11129E+01,-0.11148E+01,-0.11167E+01,-0.11186E+01,
     &-0.11205E+01,-0.11224E+01,-0.11243E+01,-0.11261E+01,-0.11280E+01,
     &-0.11299E+01,-0.11317E+01,-0.11336E+01,-0.11354E+01,-0.11373E+01/

      DATA (BNC07M(I),I=201,300)/
     &-0.11391E+01,-0.11410E+01,-0.11428E+01,-0.11446E+01,-0.11465E+01,
     &-0.11483E+01,-0.11501E+01,-0.11519E+01,-0.11537E+01,-0.11555E+01,
     &-0.11573E+01,-0.11591E+01,-0.11609E+01,-0.11627E+01,-0.11645E+01,
     &-0.11663E+01,-0.11681E+01,-0.11699E+01,-0.11716E+01,-0.11734E+01,
     &-0.11752E+01,-0.11769E+01,-0.11787E+01,-0.11804E+01,-0.11822E+01,
     &-0.11840E+01,-0.11857E+01,-0.11874E+01,-0.11892E+01,-0.11909E+01,
     &-0.11926E+01,-0.11944E+01,-0.11961E+01,-0.11978E+01,-0.11996E+01,
     &-0.12013E+01,-0.12030E+01,-0.12047E+01,-0.12064E+01,-0.12081E+01,
     &-0.12098E+01,-0.12115E+01,-0.12132E+01,-0.12149E+01,-0.12166E+01,
     &-0.12183E+01,-0.12200E+01,-0.12216E+01,-0.12233E+01,-0.12250E+01,
     &-0.12267E+01,-0.12283E+01,-0.12300E+01,-0.12317E+01,-0.12333E+01,
     &-0.12350E+01,-0.12367E+01,-0.12383E+01,-0.12400E+01,-0.12416E+01,
     &-0.12433E+01,-0.12449E+01,-0.12466E+01,-0.12482E+01,-0.12498E+01,
     &-0.12515E+01,-0.12531E+01,-0.12547E+01,-0.12564E+01,-0.12580E+01,
     &-0.12596E+01,-0.12612E+01,-0.12628E+01,-0.12645E+01,-0.12661E+01,
     &-0.12677E+01,-0.12693E+01,-0.12709E+01,-0.12725E+01,-0.12741E+01,
     &-0.12757E+01,-0.12773E+01,-0.12789E+01,-0.12805E+01,-0.12821E+01,
     &-0.12837E+01,-0.12853E+01,-0.12869E+01,-0.12884E+01,-0.12900E+01,
     &-0.12916E+01,-0.12932E+01,-0.12947E+01,-0.12963E+01,-0.12979E+01,
     &-0.12995E+01,-0.13010E+01,-0.13026E+01,-0.13042E+01,-0.13057E+01/

      DATA (BNC07M(I),I=301,400)/
     &-0.13073E+01,-0.13088E+01,-0.13104E+01,-0.13119E+01,-0.13135E+01,
     &-0.13150E+01,-0.13166E+01,-0.13181E+01,-0.13197E+01,-0.13212E+01,
     &-0.13228E+01,-0.13243E+01,-0.13258E+01,-0.13274E+01,-0.13289E+01,
     &-0.13304E+01,-0.13320E+01,-0.13335E+01,-0.13350E+01,-0.13365E+01,
     &-0.13381E+01,-0.13396E+01,-0.13411E+01,-0.13426E+01,-0.13441E+01,
     &-0.13456E+01,-0.13472E+01,-0.13487E+01,-0.13502E+01,-0.13517E+01,
     &-0.13532E+01,-0.13547E+01,-0.13562E+01,-0.13577E+01,-0.13592E+01,
     &-0.13607E+01,-0.13622E+01,-0.13637E+01,-0.13652E+01,-0.13667E+01,
     &-0.13682E+01,-0.13696E+01,-0.13711E+01,-0.13726E+01,-0.13741E+01,
     &-0.13756E+01,-0.13771E+01,-0.13785E+01,-0.13800E+01,-0.13815E+01,
     &-0.13830E+01,-0.13844E+01,-0.13859E+01,-0.13874E+01,-0.13888E+01,
     &-0.13903E+01,-0.13918E+01,-0.13932E+01,-0.13947E+01,-0.13962E+01,
     &-0.13976E+01,-0.13991E+01,-0.14005E+01,-0.14020E+01,-0.14035E+01,
     &-0.14049E+01,-0.14064E+01,-0.14078E+01,-0.14093E+01,-0.14107E+01,
     &-0.14122E+01,-0.14136E+01,-0.14151E+01,-0.14165E+01,-0.14179E+01,
     &-0.14194E+01,-0.14208E+01,-0.14223E+01,-0.14237E+01,-0.14251E+01,
     &-0.14266E+01,-0.14280E+01,-0.14294E+01,-0.14309E+01,-0.14323E+01,
     &-0.14337E+01,-0.14351E+01,-0.14366E+01,-0.14380E+01,-0.14394E+01,
     &-0.14408E+01,-0.14423E+01,-0.14437E+01,-0.14451E+01,-0.14465E+01,
     &-0.14479E+01,-0.14493E+01,-0.14508E+01,-0.14522E+01,-0.14536E+01/

      DATA (BNC07M(I),I=401,500)/
     &-0.14550E+01,-0.14564E+01,-0.14578E+01,-0.14592E+01,-0.14606E+01,
     &-0.14620E+01,-0.14634E+01,-0.14648E+01,-0.14662E+01,-0.14676E+01,
     &-0.14690E+01,-0.14704E+01,-0.14718E+01,-0.14732E+01,-0.14746E+01,
     &-0.14760E+01,-0.14774E+01,-0.14788E+01,-0.14802E+01,-0.14816E+01,
     &-0.14830E+01,-0.14844E+01,-0.14857E+01,-0.14871E+01,-0.14885E+01,
     &-0.14899E+01,-0.14913E+01,-0.14927E+01,-0.14940E+01,-0.14954E+01,
     &-0.14968E+01,-0.14982E+01,-0.14995E+01,-0.15009E+01,-0.15023E+01,
     &-0.15037E+01,-0.15050E+01,-0.15064E+01,-0.15078E+01,-0.15092E+01,
     &-0.15105E+01,-0.15119E+01,-0.15133E+01,-0.15146E+01,-0.15160E+01,
     &-0.15174E+01,-0.15187E+01,-0.15201E+01,-0.15214E+01,-0.15228E+01,
     &-0.15242E+01,-0.15255E+01,-0.15269E+01,-0.15282E+01,-0.15296E+01,
     &-0.15309E+01,-0.15323E+01,-0.15336E+01,-0.15350E+01,-0.15364E+01,
     &-0.15377E+01,-0.15391E+01,-0.15404E+01,-0.15417E+01,-0.15431E+01,
     &-0.15444E+01,-0.15458E+01,-0.15471E+01,-0.15485E+01,-0.15498E+01,
     &-0.15512E+01,-0.15525E+01,-0.15538E+01,-0.15552E+01,-0.15565E+01,
     &-0.15579E+01,-0.15592E+01,-0.15605E+01,-0.15619E+01,-0.15632E+01,
     &-0.15645E+01,-0.15659E+01,-0.15672E+01,-0.15685E+01,-0.15699E+01,
     &-0.15712E+01,-0.15725E+01,-0.15738E+01,-0.15752E+01,-0.15765E+01,
     &-0.15778E+01,-0.15791E+01,-0.15805E+01,-0.15818E+01,-0.15831E+01,
     &-0.15844E+01,-0.15858E+01,-0.15871E+01,-0.15884E+01,-0.15897E+01/

      DATA (BNC07M(I),I=501,600)/
     &-0.15910E+01,-0.15923E+01,-0.15937E+01,-0.15950E+01,-0.15963E+01,
     &-0.15976E+01,-0.15989E+01,-0.16002E+01,-0.16015E+01,-0.16029E+01,
     &-0.16042E+01,-0.16055E+01,-0.16068E+01,-0.16081E+01,-0.16094E+01,
     &-0.16107E+01,-0.16120E+01,-0.16133E+01,-0.16146E+01,-0.16159E+01,
     &-0.16172E+01,-0.16185E+01,-0.16198E+01,-0.16211E+01,-0.16224E+01,
     &-0.16237E+01,-0.16250E+01,-0.16263E+01,-0.16276E+01,-0.16289E+01,
     &-0.16302E+01,-0.16315E+01,-0.16328E+01,-0.16341E+01,-0.16354E+01,
     &-0.16367E+01,-0.16380E+01,-0.16393E+01,-0.16406E+01,-0.16418E+01,
     &-0.16431E+01,-0.16444E+01,-0.16457E+01,-0.16470E+01,-0.16483E+01,
     &-0.16496E+01,-0.16509E+01,-0.16521E+01,-0.16534E+01,-0.16547E+01,
     &-0.16560E+01,-0.16573E+01,-0.16586E+01,-0.16598E+01,-0.16611E+01,
     &-0.16624E+01,-0.16637E+01,-0.16650E+01,-0.16662E+01,-0.16675E+01,
     &-0.16688E+01,-0.16701E+01,-0.16713E+01,-0.16726E+01,-0.16739E+01,
     &-0.16752E+01,-0.16764E+01,-0.16777E+01,-0.16790E+01,-0.16803E+01,
     &-0.16815E+01,-0.16828E+01,-0.16841E+01,-0.16853E+01,-0.16866E+01,
     &-0.16879E+01,-0.16891E+01,-0.16904E+01,-0.16917E+01,-0.16929E+01,
     &-0.16942E+01,-0.16955E+01,-0.16967E+01,-0.16980E+01,-0.16992E+01,
     &-0.17005E+01,-0.17018E+01,-0.17030E+01,-0.17043E+01,-0.17056E+01,
     &-0.17068E+01,-0.17081E+01,-0.17093E+01,-0.17106E+01,-0.17118E+01,
     &-0.17131E+01,-0.17144E+01,-0.17156E+01,-0.17169E+01,-0.17181E+01/

      DATA (BNC07M(I),I=601,700)/
     &-0.17316E+01,-0.17441E+01,-0.17565E+01,-0.17688E+01,-0.17811E+01,
     &-0.17934E+01,-0.18056E+01,-0.18178E+01,-0.18299E+01,-0.18420E+01,
     &-0.18540E+01,-0.18660E+01,-0.18780E+01,-0.18899E+01,-0.19018E+01,
     &-0.19136E+01,-0.19254E+01,-0.19372E+01,-0.19489E+01,-0.19607E+01,
     &-0.19723E+01,-0.19840E+01,-0.19956E+01,-0.20072E+01,-0.20187E+01,
     &-0.20302E+01,-0.20417E+01,-0.20532E+01,-0.20646E+01,-0.20760E+01,
     &-0.20874E+01,-0.20988E+01,-0.21101E+01,-0.21214E+01,-0.21327E+01,
     &-0.21440E+01,-0.21552E+01,-0.21664E+01,-0.21776E+01,-0.21888E+01,
     &-0.21999E+01,-0.22110E+01,-0.22221E+01,-0.22332E+01,-0.22443E+01,
     &-0.22553E+01,-0.22663E+01,-0.22773E+01,-0.22883E+01,-0.22992E+01,
     &-0.23102E+01,-0.23211E+01,-0.23320E+01,-0.23429E+01,-0.23537E+01,
     &-0.23646E+01,-0.23754E+01,-0.23862E+01,-0.23970E+01,-0.24078E+01,
     &-0.24186E+01,-0.24293E+01,-0.24401E+01,-0.24508E+01,-0.24615E+01,
     &-0.24722E+01,-0.24828E+01,-0.24935E+01,-0.25041E+01,-0.25148E+01,
     &-0.25254E+01,-0.25360E+01,-0.25466E+01,-0.25571E+01,-0.25677E+01,
     &-0.25782E+01,-0.25888E+01,-0.25993E+01,-0.26098E+01,-0.26203E+01,
     &-0.26308E+01,-0.26412E+01,-0.26517E+01,-0.26621E+01,-0.26726E+01,
     &-0.26830E+01,-0.26934E+01,-0.27038E+01,-0.27142E+01,-0.27245E+01,
     &-0.27349E+01,-0.27453E+01,-0.27556E+01,-0.27659E+01,-0.27762E+01,
     &-0.27866E+01,-0.27968E+01,-0.28071E+01,-0.28174E+01,-0.28277E+01/

      DATA (BNC07M(I),I=701,741)/
     &-0.28379E+01,-0.28482E+01,-0.28584E+01,-0.28686E+01,-0.28788E+01,
     &-0.28891E+01,-0.28992E+01,-0.29094E+01,-0.29196E+01,-0.29298E+01,
     &-0.29399E+01,-0.29501E+01,-0.29602E+01,-0.29704E+01,-0.29805E+01,
     &-0.29906E+01,-0.30007E+01,-0.30108E+01,-0.30209E+01,-0.30310E+01,
     &-0.30410E+01,-0.30511E+01,-0.30612E+01,-0.30712E+01,-0.30812E+01,
     &-0.30913E+01,-0.31013E+01,-0.31113E+01,-0.31213E+01,-0.31313E+01,
     &-0.31413E+01,-0.31513E+01,-0.31613E+01,-0.31712E+01,-0.31812E+01,
     &-0.31911E+01,-0.32011E+01,-0.32110E+01,-0.32210E+01,-0.32309E+01,
     &-0.32408E+01
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.46515E-01,-0.91308E-01,-0.10881E+00,-0.11870E+00,-0.12469E+00,
     &-0.12827E+00,-0.13017E+00,-0.13082E+00,-0.13049E+00,-0.12935E+00,
     &-0.12755E+00,-0.12518E+00,-0.12230E+00,-0.11897E+00,-0.11525E+00,
     &-0.11116E+00,-0.10674E+00,-0.10201E+00,-0.96987E-01,-0.91699E-01,
     &-0.86157E-01,-0.80377E-01,-0.74371E-01,-0.68150E-01,-0.61725E-01,
     &-0.55105E-01,-0.48299E-01,-0.41316E-01,-0.34162E-01,-0.26845E-01,
     &-0.19373E-01,-0.11750E-01,-0.39845E-02, 0.39191E-02, 0.11955E-01,
     & 0.20117E-01, 0.28401E-01, 0.36802E-01, 0.45314E-01, 0.53933E-01,
     & 0.62655E-01, 0.71475E-01, 0.80389E-01, 0.89394E-01, 0.98485E-01,
     & 0.10766E+00, 0.11691E+00, 0.12624E+00, 0.13564E+00, 0.14512E+00,
     & 0.15465E+00, 0.16426E+00, 0.17392E+00, 0.18365E+00, 0.19344E+00,
     & 0.20328E+00, 0.21317E+00, 0.22312E+00, 0.23312E+00, 0.24317E+00,
     & 0.25327E+00, 0.26342E+00, 0.27361E+00, 0.28386E+00, 0.29415E+00,
     & 0.30449E+00, 0.31488E+00, 0.32532E+00, 0.33580E+00, 0.34633E+00,
     & 0.35692E+00, 0.36755E+00, 0.37823E+00, 0.38897E+00, 0.39976E+00,
     & 0.41060E+00, 0.42149E+00, 0.43244E+00, 0.44344E+00, 0.45450E+00,
     & 0.46562E+00, 0.47680E+00, 0.48803E+00, 0.49932E+00, 0.51068E+00,
     & 0.52209E+00, 0.53356E+00, 0.54509E+00, 0.55667E+00, 0.56832E+00,
     & 0.58003E+00, 0.59179E+00, 0.60361E+00, 0.61548E+00, 0.62741E+00,
     & 0.63940E+00, 0.65143E+00, 0.66352E+00, 0.67565E+00, 0.68783E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.70006E+00, 0.71232E+00, 0.72463E+00, 0.73698E+00, 0.74937E+00,
     & 0.76178E+00, 0.77424E+00, 0.78672E+00, 0.79922E+00, 0.81176E+00,
     & 0.82431E+00, 0.83689E+00, 0.84948E+00, 0.86209E+00, 0.87472E+00,
     & 0.88735E+00, 0.90000E+00, 0.91265E+00, 0.92531E+00, 0.93797E+00,
     & 0.95064E+00, 0.96330E+00, 0.97596E+00, 0.98862E+00, 0.10013E+01,
     & 0.10139E+01, 0.10266E+01, 0.10392E+01, 0.10518E+01, 0.10644E+01,
     & 0.10770E+01, 0.10896E+01, 0.11022E+01, 0.11148E+01, 0.11273E+01,
     & 0.11398E+01, 0.11524E+01, 0.11648E+01, 0.11773E+01, 0.11898E+01,
     & 0.12022E+01, 0.12146E+01, 0.12270E+01, 0.12394E+01, 0.12518E+01,
     & 0.12641E+01, 0.12764E+01, 0.12887E+01, 0.13010E+01, 0.13132E+01,
     & 0.13254E+01, 0.13376E+01, 0.13498E+01, 0.13620E+01, 0.13741E+01,
     & 0.13862E+01, 0.13983E+01, 0.14103E+01, 0.14224E+01, 0.14344E+01,
     & 0.14464E+01, 0.14583E+01, 0.14703E+01, 0.14822E+01, 0.14941E+01,
     & 0.15059E+01, 0.15178E+01, 0.15296E+01, 0.15414E+01, 0.15531E+01,
     & 0.15649E+01, 0.15766E+01, 0.15883E+01, 0.15999E+01, 0.16116E+01,
     & 0.16232E+01, 0.16348E+01, 0.16463E+01, 0.16579E+01, 0.16694E+01,
     & 0.16809E+01, 0.16923E+01, 0.17038E+01, 0.17152E+01, 0.17266E+01,
     & 0.17379E+01, 0.17493E+01, 0.17606E+01, 0.17719E+01, 0.17831E+01,
     & 0.17943E+01, 0.18056E+01, 0.18167E+01, 0.18279E+01, 0.18390E+01,
     & 0.18501E+01, 0.18612E+01, 0.18723E+01, 0.18833E+01, 0.18943E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.19053E+01, 0.19163E+01, 0.19272E+01, 0.19381E+01, 0.19490E+01,
     & 0.19599E+01, 0.19707E+01, 0.19815E+01, 0.19923E+01, 0.20031E+01,
     & 0.20138E+01, 0.20246E+01, 0.20353E+01, 0.20459E+01, 0.20566E+01,
     & 0.20672E+01, 0.20778E+01, 0.20884E+01, 0.20989E+01, 0.21095E+01,
     & 0.21200E+01, 0.21304E+01, 0.21409E+01, 0.21513E+01, 0.21617E+01,
     & 0.21721E+01, 0.21825E+01, 0.21928E+01, 0.22032E+01, 0.22134E+01,
     & 0.22237E+01, 0.22340E+01, 0.22442E+01, 0.22544E+01, 0.22646E+01,
     & 0.22747E+01, 0.22849E+01, 0.22950E+01, 0.23051E+01, 0.23152E+01,
     & 0.23252E+01, 0.23352E+01, 0.23452E+01, 0.23552E+01, 0.23652E+01,
     & 0.23751E+01, 0.23850E+01, 0.23949E+01, 0.24048E+01, 0.24147E+01,
     & 0.24245E+01, 0.24343E+01, 0.24441E+01, 0.24539E+01, 0.24636E+01,
     & 0.24733E+01, 0.24830E+01, 0.24927E+01, 0.25024E+01, 0.25120E+01,
     & 0.25216E+01, 0.25312E+01, 0.25408E+01, 0.25504E+01, 0.25599E+01,
     & 0.25694E+01, 0.25789E+01, 0.25884E+01, 0.25979E+01, 0.26073E+01,
     & 0.26167E+01, 0.26261E+01, 0.26355E+01, 0.26448E+01, 0.26542E+01,
     & 0.26635E+01, 0.26728E+01, 0.26821E+01, 0.26913E+01, 0.27006E+01,
     & 0.27098E+01, 0.27190E+01, 0.27282E+01, 0.27373E+01, 0.27465E+01,
     & 0.27556E+01, 0.27647E+01, 0.27738E+01, 0.27829E+01, 0.27919E+01,
     & 0.28009E+01, 0.28100E+01, 0.28190E+01, 0.28279E+01, 0.28369E+01,
     & 0.28458E+01, 0.28547E+01, 0.28636E+01, 0.28725E+01, 0.28814E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.28902E+01, 0.28991E+01, 0.29079E+01, 0.29167E+01, 0.29255E+01,
     & 0.29342E+01, 0.29430E+01, 0.29517E+01, 0.29604E+01, 0.29691E+01,
     & 0.29778E+01, 0.29864E+01, 0.29950E+01, 0.30037E+01, 0.30123E+01,
     & 0.30209E+01, 0.30294E+01, 0.30380E+01, 0.30465E+01, 0.30550E+01,
     & 0.30635E+01, 0.30720E+01, 0.30805E+01, 0.30889E+01, 0.30974E+01,
     & 0.31058E+01, 0.31142E+01, 0.31226E+01, 0.31309E+01, 0.31393E+01,
     & 0.31476E+01, 0.31559E+01, 0.31643E+01, 0.31725E+01, 0.31808E+01,
     & 0.31891E+01, 0.31973E+01, 0.32055E+01, 0.32137E+01, 0.32219E+01,
     & 0.32301E+01, 0.32383E+01, 0.32464E+01, 0.32546E+01, 0.32627E+01,
     & 0.32708E+01, 0.32789E+01, 0.32869E+01, 0.32950E+01, 0.33030E+01,
     & 0.33110E+01, 0.33191E+01, 0.33270E+01, 0.33350E+01, 0.33430E+01,
     & 0.33509E+01, 0.33589E+01, 0.33668E+01, 0.33747E+01, 0.33826E+01,
     & 0.33905E+01, 0.33983E+01, 0.34062E+01, 0.34140E+01, 0.34218E+01,
     & 0.34296E+01, 0.34374E+01, 0.34452E+01, 0.34530E+01, 0.34607E+01,
     & 0.34684E+01, 0.34762E+01, 0.34839E+01, 0.34915E+01, 0.34992E+01,
     & 0.35069E+01, 0.35145E+01, 0.35222E+01, 0.35298E+01, 0.35374E+01,
     & 0.35450E+01, 0.35526E+01, 0.35601E+01, 0.35677E+01, 0.35752E+01,
     & 0.35828E+01, 0.35903E+01, 0.35978E+01, 0.36053E+01, 0.36127E+01,
     & 0.36202E+01, 0.36276E+01, 0.36351E+01, 0.36425E+01, 0.36499E+01,
     & 0.36573E+01, 0.36647E+01, 0.36720E+01, 0.36794E+01, 0.36867E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.36941E+01, 0.37014E+01, 0.37087E+01, 0.37160E+01, 0.37233E+01,
     & 0.37305E+01, 0.37378E+01, 0.37450E+01, 0.37523E+01, 0.37595E+01,
     & 0.37667E+01, 0.37739E+01, 0.37811E+01, 0.37882E+01, 0.37954E+01,
     & 0.38025E+01, 0.38096E+01, 0.38168E+01, 0.38239E+01, 0.38310E+01,
     & 0.38380E+01, 0.38451E+01, 0.38522E+01, 0.38592E+01, 0.38663E+01,
     & 0.38733E+01, 0.38803E+01, 0.38873E+01, 0.38943E+01, 0.39013E+01,
     & 0.39082E+01, 0.39152E+01, 0.39221E+01, 0.39290E+01, 0.39360E+01,
     & 0.39429E+01, 0.39498E+01, 0.39566E+01, 0.39635E+01, 0.39704E+01,
     & 0.39772E+01, 0.39841E+01, 0.39909E+01, 0.39977E+01, 0.40045E+01,
     & 0.40113E+01, 0.40181E+01, 0.40249E+01, 0.40316E+01, 0.40384E+01,
     & 0.40451E+01, 0.40518E+01, 0.40586E+01, 0.40653E+01, 0.40720E+01,
     & 0.40786E+01, 0.40853E+01, 0.40920E+01, 0.40986E+01, 0.41053E+01,
     & 0.41119E+01, 0.41185E+01, 0.41251E+01, 0.41317E+01, 0.41383E+01,
     & 0.41449E+01, 0.41515E+01, 0.41580E+01, 0.41646E+01, 0.41711E+01,
     & 0.41776E+01, 0.41842E+01, 0.41907E+01, 0.41972E+01, 0.42036E+01,
     & 0.42101E+01, 0.42166E+01, 0.42230E+01, 0.42295E+01, 0.42359E+01,
     & 0.42423E+01, 0.42488E+01, 0.42552E+01, 0.42616E+01, 0.42679E+01,
     & 0.42743E+01, 0.42807E+01, 0.42870E+01, 0.42934E+01, 0.42997E+01,
     & 0.43060E+01, 0.43124E+01, 0.43187E+01, 0.43250E+01, 0.43313E+01,
     & 0.43375E+01, 0.43438E+01, 0.43501E+01, 0.43563E+01, 0.43625E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.43688E+01, 0.43750E+01, 0.43812E+01, 0.43874E+01, 0.43936E+01,
     & 0.43998E+01, 0.44060E+01, 0.44121E+01, 0.44183E+01, 0.44244E+01,
     & 0.44306E+01, 0.44367E+01, 0.44428E+01, 0.44489E+01, 0.44550E+01,
     & 0.44611E+01, 0.44672E+01, 0.44733E+01, 0.44793E+01, 0.44854E+01,
     & 0.44914E+01, 0.44975E+01, 0.45035E+01, 0.45095E+01, 0.45155E+01,
     & 0.45215E+01, 0.45275E+01, 0.45335E+01, 0.45395E+01, 0.45454E+01,
     & 0.45514E+01, 0.45574E+01, 0.45633E+01, 0.45692E+01, 0.45752E+01,
     & 0.45811E+01, 0.45870E+01, 0.45929E+01, 0.45988E+01, 0.46046E+01,
     & 0.46105E+01, 0.46164E+01, 0.46222E+01, 0.46281E+01, 0.46339E+01,
     & 0.46397E+01, 0.46456E+01, 0.46514E+01, 0.46572E+01, 0.46630E+01,
     & 0.46688E+01, 0.46745E+01, 0.46803E+01, 0.46861E+01, 0.46918E+01,
     & 0.46976E+01, 0.47033E+01, 0.47091E+01, 0.47148E+01, 0.47205E+01,
     & 0.47262E+01, 0.47319E+01, 0.47376E+01, 0.47433E+01, 0.47490E+01,
     & 0.47546E+01, 0.47603E+01, 0.47659E+01, 0.47716E+01, 0.47772E+01,
     & 0.47828E+01, 0.47885E+01, 0.47941E+01, 0.47997E+01, 0.48053E+01,
     & 0.48109E+01, 0.48164E+01, 0.48220E+01, 0.48276E+01, 0.48331E+01,
     & 0.48387E+01, 0.48442E+01, 0.48498E+01, 0.48553E+01, 0.48608E+01,
     & 0.48663E+01, 0.48718E+01, 0.48773E+01, 0.48828E+01, 0.48883E+01,
     & 0.48938E+01, 0.48992E+01, 0.49047E+01, 0.49101E+01, 0.49156E+01,
     & 0.49210E+01, 0.49265E+01, 0.49319E+01, 0.49373E+01, 0.49427E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.50007E+01, 0.50536E+01, 0.51058E+01, 0.51574E+01, 0.52083E+01,
     & 0.52585E+01, 0.53082E+01, 0.53572E+01, 0.54056E+01, 0.54534E+01,
     & 0.55007E+01, 0.55473E+01, 0.55935E+01, 0.56391E+01, 0.56841E+01,
     & 0.57287E+01, 0.57727E+01, 0.58163E+01, 0.58594E+01, 0.59020E+01,
     & 0.59441E+01, 0.59858E+01, 0.60270E+01, 0.60678E+01, 0.61081E+01,
     & 0.61481E+01, 0.61876E+01, 0.62267E+01, 0.62654E+01, 0.63037E+01,
     & 0.63416E+01, 0.63792E+01, 0.64164E+01, 0.64532E+01, 0.64897E+01,
     & 0.65258E+01, 0.65615E+01, 0.65969E+01, 0.66320E+01, 0.66668E+01,
     & 0.67012E+01, 0.67353E+01, 0.67691E+01, 0.68026E+01, 0.68358E+01,
     & 0.68687E+01, 0.69013E+01, 0.69336E+01, 0.69656E+01, 0.69974E+01,
     & 0.70288E+01, 0.70600E+01, 0.70910E+01, 0.71216E+01, 0.71520E+01,
     & 0.71822E+01, 0.72121E+01, 0.72417E+01, 0.72711E+01, 0.73003E+01,
     & 0.73292E+01, 0.73579E+01, 0.73864E+01, 0.74146E+01, 0.74426E+01,
     & 0.74704E+01, 0.74980E+01, 0.75253E+01, 0.75524E+01, 0.75794E+01,
     & 0.76061E+01, 0.76326E+01, 0.76589E+01, 0.76850E+01, 0.77110E+01,
     & 0.77367E+01, 0.77622E+01, 0.77876E+01, 0.78127E+01, 0.78377E+01,
     & 0.78625E+01, 0.78871E+01, 0.79115E+01, 0.79358E+01, 0.79599E+01,
     & 0.79838E+01, 0.80075E+01, 0.80311E+01, 0.80545E+01, 0.80778E+01,
     & 0.81009E+01, 0.81238E+01, 0.81466E+01, 0.81692E+01, 0.81916E+01,
     & 0.82140E+01, 0.82361E+01, 0.82581E+01, 0.82800E+01, 0.83017E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.83233E+01, 0.83447E+01, 0.83660E+01, 0.83872E+01, 0.84082E+01,
     & 0.84291E+01, 0.84498E+01, 0.84704E+01, 0.84909E+01, 0.85113E+01,
     & 0.85315E+01, 0.85516E+01, 0.85715E+01, 0.85914E+01, 0.86111E+01,
     & 0.86307E+01, 0.86502E+01, 0.86695E+01, 0.86888E+01, 0.87079E+01,
     & 0.87269E+01, 0.87458E+01, 0.87645E+01, 0.87832E+01, 0.88018E+01,
     & 0.88202E+01, 0.88385E+01, 0.88567E+01, 0.88749E+01, 0.88929E+01,
     & 0.89108E+01, 0.89286E+01, 0.89463E+01, 0.89638E+01, 0.89813E+01,
     & 0.89987E+01, 0.90160E+01, 0.90332E+01, 0.90503E+01, 0.90673E+01,
     & 0.90842E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.49064E-01,-0.10428E+00,-0.13083E+00,-0.14911E+00,-0.16312E+00,
     &-0.17446E+00,-0.18395E+00,-0.19207E+00,-0.19912E+00,-0.20532E+00,
     &-0.21080E+00,-0.21569E+00,-0.22007E+00,-0.22399E+00,-0.22752E+00,
     &-0.23070E+00,-0.23356E+00,-0.23612E+00,-0.23842E+00,-0.24047E+00,
     &-0.24229E+00,-0.24389E+00,-0.24530E+00,-0.24652E+00,-0.24756E+00,
     &-0.24843E+00,-0.24914E+00,-0.24969E+00,-0.25010E+00,-0.25037E+00,
     &-0.25051E+00,-0.25052E+00,-0.25040E+00,-0.25017E+00,-0.24982E+00,
     &-0.24936E+00,-0.24880E+00,-0.24813E+00,-0.24737E+00,-0.24651E+00,
     &-0.24556E+00,-0.24452E+00,-0.24339E+00,-0.24219E+00,-0.24090E+00,
     &-0.23954E+00,-0.23810E+00,-0.23659E+00,-0.23501E+00,-0.23337E+00,
     &-0.23166E+00,-0.22988E+00,-0.22805E+00,-0.22616E+00,-0.22421E+00,
     &-0.22220E+00,-0.22015E+00,-0.21804E+00,-0.21588E+00,-0.21368E+00,
     &-0.21142E+00,-0.20912E+00,-0.20678E+00,-0.20439E+00,-0.20196E+00,
     &-0.19949E+00,-0.19698E+00,-0.19443E+00,-0.19184E+00,-0.18921E+00,
     &-0.18654E+00,-0.18384E+00,-0.18110E+00,-0.17833E+00,-0.17552E+00,
     &-0.17267E+00,-0.16979E+00,-0.16688E+00,-0.16393E+00,-0.16094E+00,
     &-0.15793E+00,-0.15488E+00,-0.15180E+00,-0.14868E+00,-0.14554E+00,
     &-0.14236E+00,-0.13915E+00,-0.13591E+00,-0.13264E+00,-0.12934E+00,
     &-0.12602E+00,-0.12266E+00,-0.11927E+00,-0.11586E+00,-0.11242E+00,
     &-0.10896E+00,-0.10547E+00,-0.10196E+00,-0.98419E-01,-0.94860E-01/

      DATA (BNC09M(I),I=101,200)/
     &-0.91280E-01,-0.87679E-01,-0.84058E-01,-0.80419E-01,-0.76762E-01,
     &-0.73089E-01,-0.69400E-01,-0.65696E-01,-0.61979E-01,-0.58250E-01,
     &-0.54509E-01,-0.50757E-01,-0.46996E-01,-0.43227E-01,-0.39449E-01,
     &-0.35665E-01,-0.31875E-01,-0.28080E-01,-0.24280E-01,-0.20477E-01,
     &-0.16671E-01,-0.12863E-01,-0.90543E-02,-0.52447E-02,-0.14349E-02,
     & 0.23745E-02, 0.61825E-02, 0.99888E-02, 0.13793E-01, 0.17595E-01,
     & 0.21393E-01, 0.25188E-01, 0.28979E-01, 0.32766E-01, 0.36548E-01,
     & 0.40325E-01, 0.44097E-01, 0.47864E-01, 0.51625E-01, 0.55379E-01,
     & 0.59128E-01, 0.62870E-01, 0.66605E-01, 0.70333E-01, 0.74055E-01,
     & 0.77769E-01, 0.81476E-01, 0.85176E-01, 0.88868E-01, 0.92552E-01,
     & 0.96229E-01, 0.99897E-01, 0.10356E+00, 0.10721E+00, 0.11086E+00,
     & 0.11449E+00, 0.11812E+00, 0.12174E+00, 0.12535E+00, 0.12895E+00,
     & 0.13255E+00, 0.13614E+00, 0.13971E+00, 0.14328E+00, 0.14684E+00,
     & 0.15039E+00, 0.15394E+00, 0.15747E+00, 0.16100E+00, 0.16452E+00,
     & 0.16802E+00, 0.17152E+00, 0.17501E+00, 0.17850E+00, 0.18197E+00,
     & 0.18544E+00, 0.18889E+00, 0.19234E+00, 0.19578E+00, 0.19921E+00,
     & 0.20263E+00, 0.20604E+00, 0.20944E+00, 0.21284E+00, 0.21622E+00,
     & 0.21960E+00, 0.22297E+00, 0.22633E+00, 0.22968E+00, 0.23302E+00,
     & 0.23636E+00, 0.23968E+00, 0.24300E+00, 0.24631E+00, 0.24961E+00,
     & 0.25290E+00, 0.25618E+00, 0.25945E+00, 0.26272E+00, 0.26597E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.26922E+00, 0.27246E+00, 0.27569E+00, 0.27892E+00, 0.28213E+00,
     & 0.28534E+00, 0.28854E+00, 0.29173E+00, 0.29491E+00, 0.29808E+00,
     & 0.30125E+00, 0.30440E+00, 0.30755E+00, 0.31069E+00, 0.31383E+00,
     & 0.31695E+00, 0.32007E+00, 0.32318E+00, 0.32628E+00, 0.32937E+00,
     & 0.33245E+00, 0.33553E+00, 0.33860E+00, 0.34166E+00, 0.34472E+00,
     & 0.34776E+00, 0.35080E+00, 0.35383E+00, 0.35685E+00, 0.35987E+00,
     & 0.36287E+00, 0.36587E+00, 0.36887E+00, 0.37185E+00, 0.37483E+00,
     & 0.37780E+00, 0.38076E+00, 0.38371E+00, 0.38666E+00, 0.38960E+00,
     & 0.39253E+00, 0.39546E+00, 0.39838E+00, 0.40129E+00, 0.40419E+00,
     & 0.40709E+00, 0.40998E+00, 0.41286E+00, 0.41573E+00, 0.41860E+00,
     & 0.42146E+00, 0.42432E+00, 0.42716E+00, 0.43000E+00, 0.43283E+00,
     & 0.43566E+00, 0.43848E+00, 0.44129E+00, 0.44410E+00, 0.44690E+00,
     & 0.44969E+00, 0.45247E+00, 0.45525E+00, 0.45802E+00, 0.46079E+00,
     & 0.46355E+00, 0.46630E+00, 0.46904E+00, 0.47178E+00, 0.47452E+00,
     & 0.47724E+00, 0.47996E+00, 0.48267E+00, 0.48538E+00, 0.48808E+00,
     & 0.49077E+00, 0.49346E+00, 0.49614E+00, 0.49882E+00, 0.50149E+00,
     & 0.50415E+00, 0.50681E+00, 0.50946E+00, 0.51210E+00, 0.51474E+00,
     & 0.51737E+00, 0.51999E+00, 0.52261E+00, 0.52523E+00, 0.52783E+00,
     & 0.53044E+00, 0.53303E+00, 0.53562E+00, 0.53820E+00, 0.54078E+00,
     & 0.54335E+00, 0.54592E+00, 0.54848E+00, 0.55104E+00, 0.55358E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.55613E+00, 0.55866E+00, 0.56120E+00, 0.56372E+00, 0.56624E+00,
     & 0.56876E+00, 0.57126E+00, 0.57377E+00, 0.57627E+00, 0.57876E+00,
     & 0.58124E+00, 0.58373E+00, 0.58620E+00, 0.58867E+00, 0.59114E+00,
     & 0.59360E+00, 0.59605E+00, 0.59850E+00, 0.60094E+00, 0.60338E+00,
     & 0.60581E+00, 0.60824E+00, 0.61066E+00, 0.61308E+00, 0.61549E+00,
     & 0.61789E+00, 0.62030E+00, 0.62269E+00, 0.62508E+00, 0.62747E+00,
     & 0.62985E+00, 0.63222E+00, 0.63459E+00, 0.63696E+00, 0.63932E+00,
     & 0.64167E+00, 0.64402E+00, 0.64637E+00, 0.64871E+00, 0.65104E+00,
     & 0.65337E+00, 0.65570E+00, 0.65802E+00, 0.66033E+00, 0.66265E+00,
     & 0.66495E+00, 0.66725E+00, 0.66955E+00, 0.67184E+00, 0.67413E+00,
     & 0.67641E+00, 0.67868E+00, 0.68096E+00, 0.68322E+00, 0.68549E+00,
     & 0.68775E+00, 0.69000E+00, 0.69225E+00, 0.69449E+00, 0.69673E+00,
     & 0.69897E+00, 0.70120E+00, 0.70342E+00, 0.70565E+00, 0.70786E+00,
     & 0.71008E+00, 0.71228E+00, 0.71449E+00, 0.71669E+00, 0.71888E+00,
     & 0.72107E+00, 0.72326E+00, 0.72544E+00, 0.72761E+00, 0.72979E+00,
     & 0.73195E+00, 0.73412E+00, 0.73628E+00, 0.73843E+00, 0.74058E+00,
     & 0.74273E+00, 0.74487E+00, 0.74701E+00, 0.74914E+00, 0.75127E+00,
     & 0.75340E+00, 0.75552E+00, 0.75764E+00, 0.75975E+00, 0.76186E+00,
     & 0.76396E+00, 0.76606E+00, 0.76816E+00, 0.77025E+00, 0.77234E+00,
     & 0.77442E+00, 0.77650E+00, 0.77858E+00, 0.78065E+00, 0.78272E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.78478E+00, 0.78684E+00, 0.78890E+00, 0.79095E+00, 0.79300E+00,
     & 0.79504E+00, 0.79708E+00, 0.79912E+00, 0.80115E+00, 0.80318E+00,
     & 0.80520E+00, 0.80723E+00, 0.80924E+00, 0.81126E+00, 0.81326E+00,
     & 0.81527E+00, 0.81727E+00, 0.81927E+00, 0.82126E+00, 0.82325E+00,
     & 0.82524E+00, 0.82722E+00, 0.82920E+00, 0.83118E+00, 0.83315E+00,
     & 0.83512E+00, 0.83708E+00, 0.83904E+00, 0.84100E+00, 0.84295E+00,
     & 0.84490E+00, 0.84684E+00, 0.84879E+00, 0.85073E+00, 0.85266E+00,
     & 0.85459E+00, 0.85652E+00, 0.85844E+00, 0.86037E+00, 0.86228E+00,
     & 0.86420E+00, 0.86611E+00, 0.86801E+00, 0.86992E+00, 0.87182E+00,
     & 0.87371E+00, 0.87560E+00, 0.87749E+00, 0.87938E+00, 0.88126E+00,
     & 0.88314E+00, 0.88502E+00, 0.88689E+00, 0.88876E+00, 0.89062E+00,
     & 0.89249E+00, 0.89434E+00, 0.89620E+00, 0.89805E+00, 0.89990E+00,
     & 0.90174E+00, 0.90359E+00, 0.90543E+00, 0.90726E+00, 0.90909E+00,
     & 0.91092E+00, 0.91275E+00, 0.91457E+00, 0.91639E+00, 0.91820E+00,
     & 0.92002E+00, 0.92183E+00, 0.92363E+00, 0.92544E+00, 0.92724E+00,
     & 0.92903E+00, 0.93083E+00, 0.93262E+00, 0.93440E+00, 0.93619E+00,
     & 0.93797E+00, 0.93975E+00, 0.94152E+00, 0.94329E+00, 0.94506E+00,
     & 0.94682E+00, 0.94859E+00, 0.95035E+00, 0.95210E+00, 0.95386E+00,
     & 0.95561E+00, 0.95735E+00, 0.95910E+00, 0.96084E+00, 0.96258E+00,
     & 0.96431E+00, 0.96604E+00, 0.96777E+00, 0.96950E+00, 0.97122E+00/

      DATA (BNC09M(I),I=501,600)/
     & 0.97294E+00, 0.97466E+00, 0.97637E+00, 0.97808E+00, 0.97979E+00,
     & 0.98150E+00, 0.98320E+00, 0.98490E+00, 0.98659E+00, 0.98829E+00,
     & 0.98998E+00, 0.99167E+00, 0.99335E+00, 0.99503E+00, 0.99671E+00,
     & 0.99839E+00, 0.10001E+01, 0.10017E+01, 0.10034E+01, 0.10051E+01,
     & 0.10067E+01, 0.10084E+01, 0.10100E+01, 0.10117E+01, 0.10134E+01,
     & 0.10150E+01, 0.10166E+01, 0.10183E+01, 0.10199E+01, 0.10216E+01,
     & 0.10232E+01, 0.10248E+01, 0.10265E+01, 0.10281E+01, 0.10297E+01,
     & 0.10313E+01, 0.10330E+01, 0.10346E+01, 0.10362E+01, 0.10378E+01,
     & 0.10394E+01, 0.10410E+01, 0.10426E+01, 0.10442E+01, 0.10458E+01,
     & 0.10474E+01, 0.10490E+01, 0.10506E+01, 0.10522E+01, 0.10538E+01,
     & 0.10553E+01, 0.10569E+01, 0.10585E+01, 0.10601E+01, 0.10616E+01,
     & 0.10632E+01, 0.10648E+01, 0.10663E+01, 0.10679E+01, 0.10695E+01,
     & 0.10710E+01, 0.10726E+01, 0.10741E+01, 0.10757E+01, 0.10772E+01,
     & 0.10788E+01, 0.10803E+01, 0.10818E+01, 0.10834E+01, 0.10849E+01,
     & 0.10864E+01, 0.10880E+01, 0.10895E+01, 0.10910E+01, 0.10925E+01,
     & 0.10941E+01, 0.10956E+01, 0.10971E+01, 0.10986E+01, 0.11001E+01,
     & 0.11016E+01, 0.11031E+01, 0.11046E+01, 0.11061E+01, 0.11076E+01,
     & 0.11091E+01, 0.11106E+01, 0.11121E+01, 0.11136E+01, 0.11151E+01,
     & 0.11165E+01, 0.11180E+01, 0.11195E+01, 0.11210E+01, 0.11225E+01,
     & 0.11239E+01, 0.11254E+01, 0.11269E+01, 0.11283E+01, 0.11298E+01/

      DATA (BNC09M(I),I=601,700)/
     & 0.11454E+01, 0.11597E+01, 0.11737E+01, 0.11876E+01, 0.12012E+01,
     & 0.12146E+01, 0.12278E+01, 0.12409E+01, 0.12537E+01, 0.12663E+01,
     & 0.12788E+01, 0.12911E+01, 0.13032E+01, 0.13152E+01, 0.13270E+01,
     & 0.13386E+01, 0.13500E+01, 0.13613E+01, 0.13725E+01, 0.13835E+01,
     & 0.13943E+01, 0.14050E+01, 0.14156E+01, 0.14260E+01, 0.14363E+01,
     & 0.14464E+01, 0.14564E+01, 0.14663E+01, 0.14761E+01, 0.14857E+01,
     & 0.14952E+01, 0.15046E+01, 0.15139E+01, 0.15231E+01, 0.15321E+01,
     & 0.15410E+01, 0.15499E+01, 0.15586E+01, 0.15672E+01, 0.15757E+01,
     & 0.15841E+01, 0.15923E+01, 0.16005E+01, 0.16086E+01, 0.16166E+01,
     & 0.16245E+01, 0.16323E+01, 0.16400E+01, 0.16477E+01, 0.16552E+01,
     & 0.16626E+01, 0.16700E+01, 0.16773E+01, 0.16845E+01, 0.16916E+01,
     & 0.16986E+01, 0.17055E+01, 0.17124E+01, 0.17192E+01, 0.17259E+01,
     & 0.17325E+01, 0.17390E+01, 0.17455E+01, 0.17519E+01, 0.17583E+01,
     & 0.17645E+01, 0.17707E+01, 0.17768E+01, 0.17829E+01, 0.17889E+01,
     & 0.17948E+01, 0.18006E+01, 0.18064E+01, 0.18121E+01, 0.18178E+01,
     & 0.18234E+01, 0.18289E+01, 0.18344E+01, 0.18398E+01, 0.18452E+01,
     & 0.18505E+01, 0.18557E+01, 0.18609E+01, 0.18660E+01, 0.18711E+01,
     & 0.18761E+01, 0.18811E+01, 0.18860E+01, 0.18908E+01, 0.18956E+01,
     & 0.19004E+01, 0.19050E+01, 0.19097E+01, 0.19143E+01, 0.19188E+01,
     & 0.19233E+01, 0.19278E+01, 0.19322E+01, 0.19365E+01, 0.19408E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19451E+01, 0.19493E+01, 0.19535E+01, 0.19576E+01, 0.19616E+01,
     & 0.19657E+01, 0.19697E+01, 0.19736E+01, 0.19775E+01, 0.19814E+01,
     & 0.19852E+01, 0.19890E+01, 0.19927E+01, 0.19964E+01, 0.20001E+01,
     & 0.20037E+01, 0.20073E+01, 0.20108E+01, 0.20143E+01, 0.20178E+01,
     & 0.20212E+01, 0.20246E+01, 0.20279E+01, 0.20312E+01, 0.20345E+01,
     & 0.20378E+01, 0.20410E+01, 0.20441E+01, 0.20473E+01, 0.20504E+01,
     & 0.20534E+01, 0.20565E+01, 0.20595E+01, 0.20624E+01, 0.20654E+01,
     & 0.20683E+01, 0.20711E+01, 0.20740E+01, 0.20768E+01, 0.20796E+01,
     & 0.20823E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.48350E-01,-0.99817E-01,-0.12258E+00,-0.13710E+00,-0.14739E+00,
     &-0.15506E+00,-0.16093E+00,-0.16548E+00,-0.16903E+00,-0.17178E+00,
     &-0.17389E+00,-0.17547E+00,-0.17659E+00,-0.17734E+00,-0.17776E+00,
     &-0.17790E+00,-0.17778E+00,-0.17745E+00,-0.17692E+00,-0.17623E+00,
     &-0.17537E+00,-0.17438E+00,-0.17327E+00,-0.17205E+00,-0.17072E+00,
     &-0.16930E+00,-0.16780E+00,-0.16623E+00,-0.16459E+00,-0.16289E+00,
     &-0.16113E+00,-0.15932E+00,-0.15747E+00,-0.15558E+00,-0.15365E+00,
     &-0.15169E+00,-0.14969E+00,-0.14768E+00,-0.14563E+00,-0.14357E+00,
     &-0.14149E+00,-0.13939E+00,-0.13727E+00,-0.13515E+00,-0.13301E+00,
     &-0.13086E+00,-0.12871E+00,-0.12655E+00,-0.12438E+00,-0.12220E+00,
     &-0.12002E+00,-0.11784E+00,-0.11565E+00,-0.11346E+00,-0.11127E+00,
     &-0.10908E+00,-0.10688E+00,-0.10468E+00,-0.10248E+00,-0.10027E+00,
     &-0.98060E-01,-0.95847E-01,-0.93631E-01,-0.91410E-01,-0.89183E-01,
     &-0.86952E-01,-0.84714E-01,-0.82470E-01,-0.80218E-01,-0.77958E-01,
     &-0.75689E-01,-0.73411E-01,-0.71123E-01,-0.68823E-01,-0.66512E-01,
     &-0.64189E-01,-0.61853E-01,-0.59502E-01,-0.57138E-01,-0.54759E-01,
     &-0.52364E-01,-0.49954E-01,-0.47527E-01,-0.45084E-01,-0.42623E-01,
     &-0.40146E-01,-0.37650E-01,-0.35137E-01,-0.32607E-01,-0.30058E-01,
     &-0.27492E-01,-0.24907E-01,-0.22306E-01,-0.19686E-01,-0.17050E-01,
     &-0.14396E-01,-0.11726E-01,-0.90402E-02,-0.63382E-02,-0.36209E-02/

      DATA (BNC10M(I),I=101,200)/
     &-0.88885E-03, 0.18576E-02, 0.46179E-02, 0.73912E-02, 0.10177E-01,
     & 0.12975E-01, 0.15785E-01, 0.18605E-01, 0.21435E-01, 0.24275E-01,
     & 0.27123E-01, 0.29980E-01, 0.32844E-01, 0.35716E-01, 0.38593E-01,
     & 0.41477E-01, 0.44366E-01, 0.47259E-01, 0.50157E-01, 0.53058E-01,
     & 0.55963E-01, 0.58870E-01, 0.61780E-01, 0.64691E-01, 0.67604E-01,
     & 0.70518E-01, 0.73432E-01, 0.76347E-01, 0.79262E-01, 0.82177E-01,
     & 0.85091E-01, 0.88004E-01, 0.90916E-01, 0.93827E-01, 0.96736E-01,
     & 0.99644E-01, 0.10255E+00, 0.10545E+00, 0.10836E+00, 0.11125E+00,
     & 0.11415E+00, 0.11705E+00, 0.11994E+00, 0.12283E+00, 0.12571E+00,
     & 0.12859E+00, 0.13147E+00, 0.13435E+00, 0.13722E+00, 0.14009E+00,
     & 0.14296E+00, 0.14582E+00, 0.14868E+00, 0.15154E+00, 0.15439E+00,
     & 0.15724E+00, 0.16008E+00, 0.16293E+00, 0.16576E+00, 0.16860E+00,
     & 0.17143E+00, 0.17425E+00, 0.17708E+00, 0.17989E+00, 0.18271E+00,
     & 0.18552E+00, 0.18833E+00, 0.19113E+00, 0.19393E+00, 0.19672E+00,
     & 0.19951E+00, 0.20230E+00, 0.20508E+00, 0.20786E+00, 0.21063E+00,
     & 0.21340E+00, 0.21617E+00, 0.21893E+00, 0.22168E+00, 0.22444E+00,
     & 0.22718E+00, 0.22993E+00, 0.23267E+00, 0.23540E+00, 0.23813E+00,
     & 0.24086E+00, 0.24358E+00, 0.24630E+00, 0.24901E+00, 0.25172E+00,
     & 0.25443E+00, 0.25713E+00, 0.25983E+00, 0.26252E+00, 0.26520E+00,
     & 0.26789E+00, 0.27056E+00, 0.27324E+00, 0.27591E+00, 0.27857E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.28123E+00, 0.28389E+00, 0.28654E+00, 0.28919E+00, 0.29183E+00,
     & 0.29447E+00, 0.29710E+00, 0.29973E+00, 0.30235E+00, 0.30497E+00,
     & 0.30759E+00, 0.31020E+00, 0.31281E+00, 0.31541E+00, 0.31801E+00,
     & 0.32060E+00, 0.32319E+00, 0.32577E+00, 0.32835E+00, 0.33093E+00,
     & 0.33350E+00, 0.33606E+00, 0.33862E+00, 0.34118E+00, 0.34373E+00,
     & 0.34628E+00, 0.34882E+00, 0.35136E+00, 0.35390E+00, 0.35642E+00,
     & 0.35895E+00, 0.36147E+00, 0.36399E+00, 0.36650E+00, 0.36901E+00,
     & 0.37151E+00, 0.37401E+00, 0.37650E+00, 0.37899E+00, 0.38147E+00,
     & 0.38395E+00, 0.38643E+00, 0.38890E+00, 0.39137E+00, 0.39383E+00,
     & 0.39629E+00, 0.39874E+00, 0.40119E+00, 0.40363E+00, 0.40607E+00,
     & 0.40851E+00, 0.41094E+00, 0.41337E+00, 0.41579E+00, 0.41821E+00,
     & 0.42062E+00, 0.42303E+00, 0.42543E+00, 0.42783E+00, 0.43023E+00,
     & 0.43262E+00, 0.43501E+00, 0.43739E+00, 0.43977E+00, 0.44214E+00,
     & 0.44451E+00, 0.44688E+00, 0.44924E+00, 0.45160E+00, 0.45395E+00,
     & 0.45630E+00, 0.45864E+00, 0.46098E+00, 0.46332E+00, 0.46565E+00,
     & 0.46797E+00, 0.47029E+00, 0.47261E+00, 0.47493E+00, 0.47724E+00,
     & 0.47954E+00, 0.48184E+00, 0.48414E+00, 0.48643E+00, 0.48872E+00,
     & 0.49100E+00, 0.49328E+00, 0.49556E+00, 0.49783E+00, 0.50010E+00,
     & 0.50236E+00, 0.50462E+00, 0.50687E+00, 0.50912E+00, 0.51137E+00,
     & 0.51361E+00, 0.51585E+00, 0.51809E+00, 0.52032E+00, 0.52254E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.52476E+00, 0.52698E+00, 0.52919E+00, 0.53140E+00, 0.53361E+00,
     & 0.53581E+00, 0.53801E+00, 0.54020E+00, 0.54239E+00, 0.54458E+00,
     & 0.54676E+00, 0.54894E+00, 0.55111E+00, 0.55328E+00, 0.55544E+00,
     & 0.55760E+00, 0.55976E+00, 0.56191E+00, 0.56406E+00, 0.56621E+00,
     & 0.56835E+00, 0.57049E+00, 0.57262E+00, 0.57475E+00, 0.57688E+00,
     & 0.57900E+00, 0.58112E+00, 0.58323E+00, 0.58534E+00, 0.58745E+00,
     & 0.58955E+00, 0.59165E+00, 0.59375E+00, 0.59584E+00, 0.59792E+00,
     & 0.60001E+00, 0.60209E+00, 0.60416E+00, 0.60624E+00, 0.60830E+00,
     & 0.61037E+00, 0.61243E+00, 0.61449E+00, 0.61654E+00, 0.61859E+00,
     & 0.62064E+00, 0.62268E+00, 0.62472E+00, 0.62675E+00, 0.62878E+00,
     & 0.63081E+00, 0.63283E+00, 0.63485E+00, 0.63687E+00, 0.63888E+00,
     & 0.64089E+00, 0.64290E+00, 0.64490E+00, 0.64690E+00, 0.64889E+00,
     & 0.65088E+00, 0.65287E+00, 0.65486E+00, 0.65684E+00, 0.65881E+00,
     & 0.66079E+00, 0.66276E+00, 0.66472E+00, 0.66668E+00, 0.66864E+00,
     & 0.67060E+00, 0.67255E+00, 0.67450E+00, 0.67644E+00, 0.67839E+00,
     & 0.68032E+00, 0.68226E+00, 0.68419E+00, 0.68612E+00, 0.68804E+00,
     & 0.68996E+00, 0.69188E+00, 0.69379E+00, 0.69571E+00, 0.69761E+00,
     & 0.69952E+00, 0.70142E+00, 0.70331E+00, 0.70521E+00, 0.70710E+00,
     & 0.70898E+00, 0.71087E+00, 0.71275E+00, 0.71463E+00, 0.71650E+00,
     & 0.71837E+00, 0.72024E+00, 0.72210E+00, 0.72396E+00, 0.72582E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.72767E+00, 0.72952E+00, 0.73137E+00, 0.73321E+00, 0.73505E+00,
     & 0.73689E+00, 0.73873E+00, 0.74056E+00, 0.74239E+00, 0.74421E+00,
     & 0.74603E+00, 0.74785E+00, 0.74967E+00, 0.75148E+00, 0.75329E+00,
     & 0.75509E+00, 0.75689E+00, 0.75869E+00, 0.76049E+00, 0.76228E+00,
     & 0.76407E+00, 0.76586E+00, 0.76764E+00, 0.76942E+00, 0.77120E+00,
     & 0.77298E+00, 0.77475E+00, 0.77652E+00, 0.77828E+00, 0.78004E+00,
     & 0.78180E+00, 0.78356E+00, 0.78531E+00, 0.78706E+00, 0.78881E+00,
     & 0.79055E+00, 0.79229E+00, 0.79403E+00, 0.79577E+00, 0.79750E+00,
     & 0.79923E+00, 0.80095E+00, 0.80268E+00, 0.80440E+00, 0.80611E+00,
     & 0.80783E+00, 0.80954E+00, 0.81125E+00, 0.81295E+00, 0.81466E+00,
     & 0.81636E+00, 0.81805E+00, 0.81975E+00, 0.82144E+00, 0.82313E+00,
     & 0.82481E+00, 0.82649E+00, 0.82817E+00, 0.82985E+00, 0.83152E+00,
     & 0.83319E+00, 0.83486E+00, 0.83653E+00, 0.83819E+00, 0.83985E+00,
     & 0.84151E+00, 0.84316E+00, 0.84481E+00, 0.84646E+00, 0.84810E+00,
     & 0.84975E+00, 0.85139E+00, 0.85303E+00, 0.85466E+00, 0.85629E+00,
     & 0.85792E+00, 0.85955E+00, 0.86117E+00, 0.86279E+00, 0.86441E+00,
     & 0.86603E+00, 0.86764E+00, 0.86925E+00, 0.87086E+00, 0.87246E+00,
     & 0.87406E+00, 0.87566E+00, 0.87726E+00, 0.87885E+00, 0.88044E+00,
     & 0.88203E+00, 0.88362E+00, 0.88520E+00, 0.88678E+00, 0.88836E+00,
     & 0.88994E+00, 0.89151E+00, 0.89308E+00, 0.89465E+00, 0.89621E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.89778E+00, 0.89934E+00, 0.90089E+00, 0.90245E+00, 0.90400E+00,
     & 0.90555E+00, 0.90710E+00, 0.90864E+00, 0.91019E+00, 0.91173E+00,
     & 0.91326E+00, 0.91480E+00, 0.91633E+00, 0.91786E+00, 0.91939E+00,
     & 0.92091E+00, 0.92243E+00, 0.92395E+00, 0.92547E+00, 0.92698E+00,
     & 0.92850E+00, 0.93001E+00, 0.93151E+00, 0.93302E+00, 0.93452E+00,
     & 0.93602E+00, 0.93752E+00, 0.93901E+00, 0.94051E+00, 0.94200E+00,
     & 0.94348E+00, 0.94497E+00, 0.94645E+00, 0.94793E+00, 0.94941E+00,
     & 0.95089E+00, 0.95236E+00, 0.95383E+00, 0.95530E+00, 0.95677E+00,
     & 0.95823E+00, 0.95969E+00, 0.96115E+00, 0.96261E+00, 0.96406E+00,
     & 0.96552E+00, 0.96697E+00, 0.96841E+00, 0.96986E+00, 0.97130E+00,
     & 0.97274E+00, 0.97418E+00, 0.97562E+00, 0.97705E+00, 0.97848E+00,
     & 0.97991E+00, 0.98134E+00, 0.98276E+00, 0.98419E+00, 0.98561E+00,
     & 0.98702E+00, 0.98844E+00, 0.98985E+00, 0.99126E+00, 0.99267E+00,
     & 0.99408E+00, 0.99548E+00, 0.99689E+00, 0.99829E+00, 0.99969E+00,
     & 0.10011E+01, 0.10025E+01, 0.10039E+01, 0.10053E+01, 0.10066E+01,
     & 0.10080E+01, 0.10094E+01, 0.10108E+01, 0.10122E+01, 0.10135E+01,
     & 0.10149E+01, 0.10163E+01, 0.10177E+01, 0.10190E+01, 0.10204E+01,
     & 0.10218E+01, 0.10231E+01, 0.10245E+01, 0.10258E+01, 0.10272E+01,
     & 0.10285E+01, 0.10299E+01, 0.10312E+01, 0.10326E+01, 0.10339E+01,
     & 0.10353E+01, 0.10366E+01, 0.10380E+01, 0.10393E+01, 0.10406E+01/

      DATA (BNC10M(I),I=601,700)/
     & 0.10549E+01, 0.10679E+01, 0.10807E+01, 0.10933E+01, 0.11058E+01,
     & 0.11180E+01, 0.11301E+01, 0.11420E+01, 0.11537E+01, 0.11652E+01,
     & 0.11766E+01, 0.11878E+01, 0.11989E+01, 0.12098E+01, 0.12205E+01,
     & 0.12311E+01, 0.12415E+01, 0.12518E+01, 0.12620E+01, 0.12720E+01,
     & 0.12819E+01, 0.12916E+01, 0.13012E+01, 0.13107E+01, 0.13201E+01,
     & 0.13293E+01, 0.13384E+01, 0.13474E+01, 0.13562E+01, 0.13650E+01,
     & 0.13736E+01, 0.13821E+01, 0.13905E+01, 0.13988E+01, 0.14070E+01,
     & 0.14151E+01, 0.14231E+01, 0.14310E+01, 0.14388E+01, 0.14464E+01,
     & 0.14540E+01, 0.14615E+01, 0.14689E+01, 0.14762E+01, 0.14834E+01,
     & 0.14906E+01, 0.14976E+01, 0.15045E+01, 0.15114E+01, 0.15182E+01,
     & 0.15249E+01, 0.15315E+01, 0.15380E+01, 0.15444E+01, 0.15508E+01,
     & 0.15571E+01, 0.15633E+01, 0.15695E+01, 0.15755E+01, 0.15815E+01,
     & 0.15875E+01, 0.15933E+01, 0.15991E+01, 0.16048E+01, 0.16105E+01,
     & 0.16160E+01, 0.16215E+01, 0.16270E+01, 0.16324E+01, 0.16377E+01,
     & 0.16429E+01, 0.16481E+01, 0.16533E+01, 0.16583E+01, 0.16634E+01,
     & 0.16683E+01, 0.16732E+01, 0.16780E+01, 0.16828E+01, 0.16875E+01,
     & 0.16922E+01, 0.16968E+01, 0.17014E+01, 0.17059E+01, 0.17104E+01,
     & 0.17148E+01, 0.17191E+01, 0.17234E+01, 0.17277E+01, 0.17319E+01,
     & 0.17360E+01, 0.17401E+01, 0.17442E+01, 0.17482E+01, 0.17521E+01,
     & 0.17560E+01, 0.17599E+01, 0.17637E+01, 0.17675E+01, 0.17712E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17749E+01, 0.17786E+01, 0.17822E+01, 0.17858E+01, 0.17893E+01,
     & 0.17927E+01, 0.17962E+01, 0.17996E+01, 0.18029E+01, 0.18063E+01,
     & 0.18095E+01, 0.18128E+01, 0.18160E+01, 0.18191E+01, 0.18223E+01,
     & 0.18253E+01, 0.18284E+01, 0.18314E+01, 0.18344E+01, 0.18373E+01,
     & 0.18402E+01, 0.18431E+01, 0.18459E+01, 0.18487E+01, 0.18515E+01,
     & 0.18542E+01, 0.18569E+01, 0.18596E+01, 0.18623E+01, 0.18649E+01,
     & 0.18674E+01, 0.18700E+01, 0.18725E+01, 0.18749E+01, 0.18774E+01,
     & 0.18798E+01, 0.18822E+01, 0.18845E+01, 0.18869E+01, 0.18892E+01,
     & 0.18914E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.46921E-01,-0.92783E-01,-0.11083E+00,-0.12102E+00,-0.12722E+00,
     &-0.13092E+00,-0.13292E+00,-0.13364E+00,-0.13339E+00,-0.13235E+00,
     &-0.13067E+00,-0.12843E+00,-0.12574E+00,-0.12263E+00,-0.11917E+00,
     &-0.11539E+00,-0.11133E+00,-0.10701E+00,-0.10245E+00,-0.97688E-01,
     &-0.92727E-01,-0.87586E-01,-0.82279E-01,-0.76818E-01,-0.71216E-01,
     &-0.65481E-01,-0.59623E-01,-0.53651E-01,-0.47571E-01,-0.41393E-01,
     &-0.35121E-01,-0.28762E-01,-0.22322E-01,-0.15807E-01,-0.92221E-02,
     &-0.25712E-02, 0.41408E-02, 0.10910E-01, 0.17731E-01, 0.24602E-01,
     & 0.31518E-01, 0.38477E-01, 0.45474E-01, 0.52508E-01, 0.59576E-01,
     & 0.66675E-01, 0.73803E-01, 0.80957E-01, 0.88137E-01, 0.95339E-01,
     & 0.10256E+00, 0.10981E+00, 0.11707E+00, 0.12435E+00, 0.13165E+00,
     & 0.13897E+00, 0.14630E+00, 0.15365E+00, 0.16101E+00, 0.16839E+00,
     & 0.17578E+00, 0.18319E+00, 0.19062E+00, 0.19806E+00, 0.20553E+00,
     & 0.21300E+00, 0.22050E+00, 0.22802E+00, 0.23556E+00, 0.24313E+00,
     & 0.25071E+00, 0.25832E+00, 0.26596E+00, 0.27362E+00, 0.28131E+00,
     & 0.28903E+00, 0.29679E+00, 0.30457E+00, 0.31239E+00, 0.32024E+00,
     & 0.32813E+00, 0.33605E+00, 0.34401E+00, 0.35201E+00, 0.36004E+00,
     & 0.36812E+00, 0.37623E+00, 0.38439E+00, 0.39258E+00, 0.40081E+00,
     & 0.40909E+00, 0.41740E+00, 0.42575E+00, 0.43414E+00, 0.44257E+00,
     & 0.45104E+00, 0.45954E+00, 0.46808E+00, 0.47665E+00, 0.48526E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.49390E+00, 0.50257E+00, 0.51127E+00, 0.51999E+00, 0.52875E+00,
     & 0.53752E+00, 0.54633E+00, 0.55515E+00, 0.56400E+00, 0.57286E+00,
     & 0.58174E+00, 0.59064E+00, 0.59955E+00, 0.60847E+00, 0.61740E+00,
     & 0.62635E+00, 0.63530E+00, 0.64426E+00, 0.65322E+00, 0.66219E+00,
     & 0.67116E+00, 0.68013E+00, 0.68911E+00, 0.69808E+00, 0.70705E+00,
     & 0.71602E+00, 0.72499E+00, 0.73395E+00, 0.74290E+00, 0.75185E+00,
     & 0.76080E+00, 0.76973E+00, 0.77866E+00, 0.78758E+00, 0.79649E+00,
     & 0.80539E+00, 0.81428E+00, 0.82316E+00, 0.83203E+00, 0.84089E+00,
     & 0.84973E+00, 0.85857E+00, 0.86739E+00, 0.87619E+00, 0.88499E+00,
     & 0.89377E+00, 0.90254E+00, 0.91129E+00, 0.92003E+00, 0.92876E+00,
     & 0.93747E+00, 0.94617E+00, 0.95486E+00, 0.96352E+00, 0.97218E+00,
     & 0.98082E+00, 0.98944E+00, 0.99805E+00, 0.10066E+01, 0.10152E+01,
     & 0.10238E+01, 0.10323E+01, 0.10409E+01, 0.10494E+01, 0.10579E+01,
     & 0.10664E+01, 0.10748E+01, 0.10833E+01, 0.10917E+01, 0.11001E+01,
     & 0.11085E+01, 0.11169E+01, 0.11253E+01, 0.11337E+01, 0.11420E+01,
     & 0.11503E+01, 0.11586E+01, 0.11669E+01, 0.11752E+01, 0.11834E+01,
     & 0.11917E+01, 0.11999E+01, 0.12081E+01, 0.12163E+01, 0.12245E+01,
     & 0.12326E+01, 0.12408E+01, 0.12489E+01, 0.12570E+01, 0.12651E+01,
     & 0.12732E+01, 0.12812E+01, 0.12893E+01, 0.12973E+01, 0.13053E+01,
     & 0.13133E+01, 0.13213E+01, 0.13292E+01, 0.13372E+01, 0.13451E+01/

      DATA (BNC11M(I),I=201,300)/
     & 0.13530E+01, 0.13609E+01, 0.13688E+01, 0.13767E+01, 0.13845E+01,
     & 0.13923E+01, 0.14001E+01, 0.14079E+01, 0.14157E+01, 0.14235E+01,
     & 0.14312E+01, 0.14390E+01, 0.14467E+01, 0.14544E+01, 0.14620E+01,
     & 0.14697E+01, 0.14774E+01, 0.14850E+01, 0.14926E+01, 0.15002E+01,
     & 0.15078E+01, 0.15154E+01, 0.15229E+01, 0.15305E+01, 0.15380E+01,
     & 0.15455E+01, 0.15530E+01, 0.15604E+01, 0.15679E+01, 0.15753E+01,
     & 0.15828E+01, 0.15902E+01, 0.15976E+01, 0.16050E+01, 0.16123E+01,
     & 0.16197E+01, 0.16270E+01, 0.16343E+01, 0.16416E+01, 0.16489E+01,
     & 0.16562E+01, 0.16634E+01, 0.16707E+01, 0.16779E+01, 0.16851E+01,
     & 0.16923E+01, 0.16995E+01, 0.17067E+01, 0.17138E+01, 0.17210E+01,
     & 0.17281E+01, 0.17352E+01, 0.17423E+01, 0.17493E+01, 0.17564E+01,
     & 0.17635E+01, 0.17705E+01, 0.17775E+01, 0.17845E+01, 0.17915E+01,
     & 0.17985E+01, 0.18054E+01, 0.18124E+01, 0.18193E+01, 0.18262E+01,
     & 0.18331E+01, 0.18400E+01, 0.18469E+01, 0.18537E+01, 0.18606E+01,
     & 0.18674E+01, 0.18742E+01, 0.18810E+01, 0.18878E+01, 0.18946E+01,
     & 0.19014E+01, 0.19081E+01, 0.19148E+01, 0.19216E+01, 0.19283E+01,
     & 0.19350E+01, 0.19416E+01, 0.19483E+01, 0.19550E+01, 0.19616E+01,
     & 0.19682E+01, 0.19748E+01, 0.19814E+01, 0.19880E+01, 0.19946E+01,
     & 0.20011E+01, 0.20077E+01, 0.20142E+01, 0.20207E+01, 0.20272E+01,
     & 0.20337E+01, 0.20402E+01, 0.20467E+01, 0.20531E+01, 0.20596E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.20660E+01, 0.20724E+01, 0.20788E+01, 0.20852E+01, 0.20916E+01,
     & 0.20979E+01, 0.21043E+01, 0.21106E+01, 0.21169E+01, 0.21233E+01,
     & 0.21296E+01, 0.21358E+01, 0.21421E+01, 0.21484E+01, 0.21546E+01,
     & 0.21609E+01, 0.21671E+01, 0.21733E+01, 0.21795E+01, 0.21857E+01,
     & 0.21919E+01, 0.21981E+01, 0.22042E+01, 0.22104E+01, 0.22165E+01,
     & 0.22226E+01, 0.22287E+01, 0.22348E+01, 0.22409E+01, 0.22470E+01,
     & 0.22530E+01, 0.22591E+01, 0.22651E+01, 0.22711E+01, 0.22771E+01,
     & 0.22831E+01, 0.22891E+01, 0.22951E+01, 0.23011E+01, 0.23070E+01,
     & 0.23130E+01, 0.23189E+01, 0.23248E+01, 0.23308E+01, 0.23367E+01,
     & 0.23425E+01, 0.23484E+01, 0.23543E+01, 0.23601E+01, 0.23660E+01,
     & 0.23718E+01, 0.23776E+01, 0.23835E+01, 0.23893E+01, 0.23950E+01,
     & 0.24008E+01, 0.24066E+01, 0.24124E+01, 0.24181E+01, 0.24238E+01,
     & 0.24296E+01, 0.24353E+01, 0.24410E+01, 0.24467E+01, 0.24524E+01,
     & 0.24580E+01, 0.24637E+01, 0.24694E+01, 0.24750E+01, 0.24806E+01,
     & 0.24862E+01, 0.24919E+01, 0.24975E+01, 0.25031E+01, 0.25086E+01,
     & 0.25142E+01, 0.25198E+01, 0.25253E+01, 0.25309E+01, 0.25364E+01,
     & 0.25419E+01, 0.25474E+01, 0.25529E+01, 0.25584E+01, 0.25639E+01,
     & 0.25694E+01, 0.25748E+01, 0.25803E+01, 0.25857E+01, 0.25912E+01,
     & 0.25966E+01, 0.26020E+01, 0.26074E+01, 0.26128E+01, 0.26182E+01,
     & 0.26235E+01, 0.26289E+01, 0.26343E+01, 0.26396E+01, 0.26450E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.26503E+01, 0.26556E+01, 0.26609E+01, 0.26662E+01, 0.26715E+01,
     & 0.26768E+01, 0.26821E+01, 0.26873E+01, 0.26926E+01, 0.26978E+01,
     & 0.27031E+01, 0.27083E+01, 0.27135E+01, 0.27187E+01, 0.27239E+01,
     & 0.27291E+01, 0.27343E+01, 0.27395E+01, 0.27446E+01, 0.27498E+01,
     & 0.27549E+01, 0.27601E+01, 0.27652E+01, 0.27703E+01, 0.27754E+01,
     & 0.27805E+01, 0.27856E+01, 0.27907E+01, 0.27958E+01, 0.28008E+01,
     & 0.28059E+01, 0.28110E+01, 0.28160E+01, 0.28210E+01, 0.28261E+01,
     & 0.28311E+01, 0.28361E+01, 0.28411E+01, 0.28461E+01, 0.28511E+01,
     & 0.28560E+01, 0.28610E+01, 0.28660E+01, 0.28709E+01, 0.28759E+01,
     & 0.28808E+01, 0.28857E+01, 0.28906E+01, 0.28956E+01, 0.29005E+01,
     & 0.29053E+01, 0.29102E+01, 0.29151E+01, 0.29200E+01, 0.29248E+01,
     & 0.29297E+01, 0.29345E+01, 0.29394E+01, 0.29442E+01, 0.29490E+01,
     & 0.29539E+01, 0.29587E+01, 0.29635E+01, 0.29683E+01, 0.29730E+01,
     & 0.29778E+01, 0.29826E+01, 0.29873E+01, 0.29921E+01, 0.29968E+01,
     & 0.30016E+01, 0.30063E+01, 0.30110E+01, 0.30157E+01, 0.30205E+01,
     & 0.30252E+01, 0.30298E+01, 0.30345E+01, 0.30392E+01, 0.30439E+01,
     & 0.30485E+01, 0.30532E+01, 0.30578E+01, 0.30625E+01, 0.30671E+01,
     & 0.30717E+01, 0.30764E+01, 0.30810E+01, 0.30856E+01, 0.30902E+01,
     & 0.30948E+01, 0.30994E+01, 0.31039E+01, 0.31085E+01, 0.31131E+01,
     & 0.31176E+01, 0.31222E+01, 0.31267E+01, 0.31312E+01, 0.31358E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.31403E+01, 0.31448E+01, 0.31493E+01, 0.31538E+01, 0.31583E+01,
     & 0.31628E+01, 0.31672E+01, 0.31717E+01, 0.31762E+01, 0.31806E+01,
     & 0.31851E+01, 0.31895E+01, 0.31940E+01, 0.31984E+01, 0.32028E+01,
     & 0.32072E+01, 0.32117E+01, 0.32161E+01, 0.32205E+01, 0.32248E+01,
     & 0.32292E+01, 0.32336E+01, 0.32380E+01, 0.32423E+01, 0.32467E+01,
     & 0.32510E+01, 0.32554E+01, 0.32597E+01, 0.32641E+01, 0.32684E+01,
     & 0.32727E+01, 0.32770E+01, 0.32813E+01, 0.32856E+01, 0.32899E+01,
     & 0.32942E+01, 0.32985E+01, 0.33028E+01, 0.33070E+01, 0.33113E+01,
     & 0.33155E+01, 0.33198E+01, 0.33240E+01, 0.33283E+01, 0.33325E+01,
     & 0.33367E+01, 0.33409E+01, 0.33451E+01, 0.33493E+01, 0.33535E+01,
     & 0.33577E+01, 0.33619E+01, 0.33661E+01, 0.33703E+01, 0.33744E+01,
     & 0.33786E+01, 0.33828E+01, 0.33869E+01, 0.33911E+01, 0.33952E+01,
     & 0.33993E+01, 0.34035E+01, 0.34076E+01, 0.34117E+01, 0.34158E+01,
     & 0.34199E+01, 0.34240E+01, 0.34281E+01, 0.34322E+01, 0.34363E+01,
     & 0.34403E+01, 0.34444E+01, 0.34485E+01, 0.34525E+01, 0.34566E+01,
     & 0.34606E+01, 0.34647E+01, 0.34687E+01, 0.34727E+01, 0.34767E+01,
     & 0.34808E+01, 0.34848E+01, 0.34888E+01, 0.34928E+01, 0.34968E+01,
     & 0.35008E+01, 0.35047E+01, 0.35087E+01, 0.35127E+01, 0.35167E+01,
     & 0.35206E+01, 0.35246E+01, 0.35285E+01, 0.35325E+01, 0.35364E+01,
     & 0.35403E+01, 0.35443E+01, 0.35482E+01, 0.35521E+01, 0.35560E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.35979E+01, 0.36362E+01, 0.36739E+01, 0.37112E+01, 0.37479E+01,
     & 0.37842E+01, 0.38200E+01, 0.38554E+01, 0.38903E+01, 0.39247E+01,
     & 0.39588E+01, 0.39924E+01, 0.40256E+01, 0.40584E+01, 0.40909E+01,
     & 0.41229E+01, 0.41546E+01, 0.41859E+01, 0.42168E+01, 0.42474E+01,
     & 0.42776E+01, 0.43075E+01, 0.43371E+01, 0.43664E+01, 0.43953E+01,
     & 0.44239E+01, 0.44522E+01, 0.44802E+01, 0.45079E+01, 0.45353E+01,
     & 0.45624E+01, 0.45893E+01, 0.46158E+01, 0.46421E+01, 0.46681E+01,
     & 0.46939E+01, 0.47194E+01, 0.47447E+01, 0.47697E+01, 0.47944E+01,
     & 0.48190E+01, 0.48432E+01, 0.48673E+01, 0.48911E+01, 0.49147E+01,
     & 0.49381E+01, 0.49612E+01, 0.49842E+01, 0.50069E+01, 0.50294E+01,
     & 0.50517E+01, 0.50738E+01, 0.50958E+01, 0.51175E+01, 0.51390E+01,
     & 0.51603E+01, 0.51815E+01, 0.52024E+01, 0.52232E+01, 0.52438E+01,
     & 0.52643E+01, 0.52845E+01, 0.53046E+01, 0.53245E+01, 0.53442E+01,
     & 0.53638E+01, 0.53832E+01, 0.54025E+01, 0.54216E+01, 0.54405E+01,
     & 0.54593E+01, 0.54779E+01, 0.54964E+01, 0.55147E+01, 0.55329E+01,
     & 0.55510E+01, 0.55689E+01, 0.55866E+01, 0.56043E+01, 0.56217E+01,
     & 0.56391E+01, 0.56563E+01, 0.56734E+01, 0.56903E+01, 0.57072E+01,
     & 0.57239E+01, 0.57404E+01, 0.57569E+01, 0.57732E+01, 0.57894E+01,
     & 0.58055E+01, 0.58215E+01, 0.58373E+01, 0.58530E+01, 0.58687E+01,
     & 0.58842E+01, 0.58996E+01, 0.59148E+01, 0.59300E+01, 0.59451E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.59600E+01, 0.59749E+01, 0.59896E+01, 0.60043E+01, 0.60188E+01,
     & 0.60333E+01, 0.60476E+01, 0.60619E+01, 0.60760E+01, 0.60901E+01,
     & 0.61040E+01, 0.61179E+01, 0.61316E+01, 0.61453E+01, 0.61589E+01,
     & 0.61724E+01, 0.61858E+01, 0.61991E+01, 0.62123E+01, 0.62254E+01,
     & 0.62385E+01, 0.62514E+01, 0.62643E+01, 0.62771E+01, 0.62898E+01,
     & 0.63024E+01, 0.63150E+01, 0.63274E+01, 0.63398E+01, 0.63521E+01,
     & 0.63644E+01, 0.63765E+01, 0.63886E+01, 0.64006E+01, 0.64125E+01,
     & 0.64243E+01, 0.64361E+01, 0.64478E+01, 0.64594E+01, 0.64710E+01,
     & 0.64825E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.48156E-01,-0.99449E-01,-0.12247E+00,-0.13741E+00,-0.14823E+00,
     &-0.15646E+00,-0.16292E+00,-0.16806E+00,-0.17218E+00,-0.17549E+00,
     &-0.17813E+00,-0.18020E+00,-0.18178E+00,-0.18294E+00,-0.18372E+00,
     &-0.18417E+00,-0.18432E+00,-0.18420E+00,-0.18383E+00,-0.18323E+00,
     &-0.18241E+00,-0.18140E+00,-0.18020E+00,-0.17883E+00,-0.17729E+00,
     &-0.17560E+00,-0.17376E+00,-0.17179E+00,-0.16968E+00,-0.16745E+00,
     &-0.16509E+00,-0.16262E+00,-0.16005E+00,-0.15737E+00,-0.15459E+00,
     &-0.15171E+00,-0.14874E+00,-0.14568E+00,-0.14254E+00,-0.13932E+00,
     &-0.13602E+00,-0.13265E+00,-0.12921E+00,-0.12569E+00,-0.12212E+00,
     &-0.11847E+00,-0.11477E+00,-0.11101E+00,-0.10719E+00,-0.10332E+00,
     &-0.99401E-01,-0.95428E-01,-0.91408E-01,-0.87340E-01,-0.83227E-01,
     &-0.79071E-01,-0.74872E-01,-0.70632E-01,-0.66351E-01,-0.62032E-01,
     &-0.57674E-01,-0.53279E-01,-0.48846E-01,-0.44378E-01,-0.39874E-01,
     &-0.35334E-01,-0.30759E-01,-0.26149E-01,-0.21504E-01,-0.16825E-01,
     &-0.12111E-01,-0.73619E-02,-0.25784E-02, 0.22401E-02, 0.70935E-02,
     & 0.11982E-01, 0.16906E-01, 0.21866E-01, 0.26861E-01, 0.31892E-01,
     & 0.36958E-01, 0.42061E-01, 0.47200E-01, 0.52375E-01, 0.57586E-01,
     & 0.62833E-01, 0.68116E-01, 0.73434E-01, 0.78787E-01, 0.84176E-01,
     & 0.89598E-01, 0.95055E-01, 0.10054E+00, 0.10607E+00, 0.11162E+00,
     & 0.11721E+00, 0.12282E+00, 0.12847E+00, 0.13414E+00, 0.13984E+00/

      DATA (BNC12M(I),I=101,200)/
     & 0.14556E+00, 0.15132E+00, 0.15709E+00, 0.16289E+00, 0.16871E+00,
     & 0.17454E+00, 0.18040E+00, 0.18627E+00, 0.19216E+00, 0.19807E+00,
     & 0.20399E+00, 0.20992E+00, 0.21586E+00, 0.22181E+00, 0.22777E+00,
     & 0.23374E+00, 0.23971E+00, 0.24569E+00, 0.25168E+00, 0.25766E+00,
     & 0.26365E+00, 0.26964E+00, 0.27563E+00, 0.28162E+00, 0.28761E+00,
     & 0.29360E+00, 0.29958E+00, 0.30556E+00, 0.31153E+00, 0.31750E+00,
     & 0.32347E+00, 0.32943E+00, 0.33538E+00, 0.34133E+00, 0.34727E+00,
     & 0.35320E+00, 0.35912E+00, 0.36504E+00, 0.37094E+00, 0.37684E+00,
     & 0.38273E+00, 0.38861E+00, 0.39448E+00, 0.40034E+00, 0.40618E+00,
     & 0.41202E+00, 0.41785E+00, 0.42367E+00, 0.42947E+00, 0.43527E+00,
     & 0.44105E+00, 0.44683E+00, 0.45259E+00, 0.45834E+00, 0.46408E+00,
     & 0.46981E+00, 0.47552E+00, 0.48123E+00, 0.48692E+00, 0.49260E+00,
     & 0.49827E+00, 0.50393E+00, 0.50957E+00, 0.51520E+00, 0.52083E+00,
     & 0.52644E+00, 0.53203E+00, 0.53762E+00, 0.54319E+00, 0.54875E+00,
     & 0.55430E+00, 0.55984E+00, 0.56537E+00, 0.57088E+00, 0.57638E+00,
     & 0.58187E+00, 0.58734E+00, 0.59281E+00, 0.59826E+00, 0.60370E+00,
     & 0.60913E+00, 0.61455E+00, 0.61995E+00, 0.62534E+00, 0.63072E+00,
     & 0.63609E+00, 0.64145E+00, 0.64679E+00, 0.65212E+00, 0.65744E+00,
     & 0.66275E+00, 0.66805E+00, 0.67333E+00, 0.67860E+00, 0.68387E+00,
     & 0.68911E+00, 0.69435E+00, 0.69958E+00, 0.70479E+00, 0.70999E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.71518E+00, 0.72036E+00, 0.72553E+00, 0.73068E+00, 0.73582E+00,
     & 0.74096E+00, 0.74608E+00, 0.75118E+00, 0.75628E+00, 0.76137E+00,
     & 0.76644E+00, 0.77150E+00, 0.77656E+00, 0.78160E+00, 0.78663E+00,
     & 0.79164E+00, 0.79665E+00, 0.80164E+00, 0.80663E+00, 0.81160E+00,
     & 0.81656E+00, 0.82152E+00, 0.82646E+00, 0.83138E+00, 0.83630E+00,
     & 0.84121E+00, 0.84611E+00, 0.85099E+00, 0.85587E+00, 0.86073E+00,
     & 0.86558E+00, 0.87042E+00, 0.87526E+00, 0.88008E+00, 0.88489E+00,
     & 0.88969E+00, 0.89448E+00, 0.89926E+00, 0.90403E+00, 0.90878E+00,
     & 0.91353E+00, 0.91827E+00, 0.92300E+00, 0.92771E+00, 0.93242E+00,
     & 0.93711E+00, 0.94180E+00, 0.94648E+00, 0.95114E+00, 0.95580E+00,
     & 0.96044E+00, 0.96508E+00, 0.96971E+00, 0.97432E+00, 0.97893E+00,
     & 0.98352E+00, 0.98811E+00, 0.99269E+00, 0.99725E+00, 0.10018E+01,
     & 0.10064E+01, 0.10109E+01, 0.10154E+01, 0.10199E+01, 0.10245E+01,
     & 0.10290E+01, 0.10334E+01, 0.10379E+01, 0.10424E+01, 0.10469E+01,
     & 0.10513E+01, 0.10558E+01, 0.10602E+01, 0.10646E+01, 0.10690E+01,
     & 0.10734E+01, 0.10778E+01, 0.10822E+01, 0.10866E+01, 0.10910E+01,
     & 0.10953E+01, 0.10997E+01, 0.11040E+01, 0.11084E+01, 0.11127E+01,
     & 0.11170E+01, 0.11213E+01, 0.11256E+01, 0.11299E+01, 0.11342E+01,
     & 0.11384E+01, 0.11427E+01, 0.11470E+01, 0.11512E+01, 0.11554E+01,
     & 0.11597E+01, 0.11639E+01, 0.11681E+01, 0.11723E+01, 0.11765E+01/

      DATA (BNC12M(I),I=301,400)/
     & 0.11807E+01, 0.11849E+01, 0.11890E+01, 0.11932E+01, 0.11974E+01,
     & 0.12015E+01, 0.12056E+01, 0.12098E+01, 0.12139E+01, 0.12180E+01,
     & 0.12221E+01, 0.12262E+01, 0.12303E+01, 0.12344E+01, 0.12384E+01,
     & 0.12425E+01, 0.12466E+01, 0.12506E+01, 0.12547E+01, 0.12587E+01,
     & 0.12627E+01, 0.12667E+01, 0.12707E+01, 0.12747E+01, 0.12787E+01,
     & 0.12827E+01, 0.12867E+01, 0.12907E+01, 0.12946E+01, 0.12986E+01,
     & 0.13025E+01, 0.13065E+01, 0.13104E+01, 0.13143E+01, 0.13182E+01,
     & 0.13221E+01, 0.13261E+01, 0.13299E+01, 0.13338E+01, 0.13377E+01,
     & 0.13416E+01, 0.13455E+01, 0.13493E+01, 0.13532E+01, 0.13570E+01,
     & 0.13608E+01, 0.13647E+01, 0.13685E+01, 0.13723E+01, 0.13761E+01,
     & 0.13799E+01, 0.13837E+01, 0.13875E+01, 0.13913E+01, 0.13950E+01,
     & 0.13988E+01, 0.14026E+01, 0.14063E+01, 0.14101E+01, 0.14138E+01,
     & 0.14175E+01, 0.14213E+01, 0.14250E+01, 0.14287E+01, 0.14324E+01,
     & 0.14361E+01, 0.14398E+01, 0.14435E+01, 0.14471E+01, 0.14508E+01,
     & 0.14545E+01, 0.14581E+01, 0.14618E+01, 0.14654E+01, 0.14690E+01,
     & 0.14727E+01, 0.14763E+01, 0.14799E+01, 0.14835E+01, 0.14871E+01,
     & 0.14907E+01, 0.14943E+01, 0.14979E+01, 0.15015E+01, 0.15051E+01,
     & 0.15086E+01, 0.15122E+01, 0.15157E+01, 0.15193E+01, 0.15228E+01,
     & 0.15264E+01, 0.15299E+01, 0.15334E+01, 0.15369E+01, 0.15404E+01,
     & 0.15439E+01, 0.15474E+01, 0.15509E+01, 0.15544E+01, 0.15579E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.15613E+01, 0.15648E+01, 0.15683E+01, 0.15717E+01, 0.15752E+01,
     & 0.15786E+01, 0.15821E+01, 0.15855E+01, 0.15889E+01, 0.15923E+01,
     & 0.15957E+01, 0.15991E+01, 0.16025E+01, 0.16059E+01, 0.16093E+01,
     & 0.16127E+01, 0.16161E+01, 0.16195E+01, 0.16228E+01, 0.16262E+01,
     & 0.16295E+01, 0.16329E+01, 0.16362E+01, 0.16396E+01, 0.16429E+01,
     & 0.16462E+01, 0.16495E+01, 0.16529E+01, 0.16562E+01, 0.16595E+01,
     & 0.16628E+01, 0.16661E+01, 0.16693E+01, 0.16726E+01, 0.16759E+01,
     & 0.16792E+01, 0.16824E+01, 0.16857E+01, 0.16890E+01, 0.16922E+01,
     & 0.16954E+01, 0.16987E+01, 0.17019E+01, 0.17051E+01, 0.17084E+01,
     & 0.17116E+01, 0.17148E+01, 0.17180E+01, 0.17212E+01, 0.17244E+01,
     & 0.17276E+01, 0.17308E+01, 0.17339E+01, 0.17371E+01, 0.17403E+01,
     & 0.17434E+01, 0.17466E+01, 0.17498E+01, 0.17529E+01, 0.17561E+01,
     & 0.17592E+01, 0.17623E+01, 0.17655E+01, 0.17686E+01, 0.17717E+01,
     & 0.17748E+01, 0.17779E+01, 0.17810E+01, 0.17841E+01, 0.17872E+01,
     & 0.17903E+01, 0.17934E+01, 0.17965E+01, 0.17995E+01, 0.18026E+01,
     & 0.18057E+01, 0.18087E+01, 0.18118E+01, 0.18148E+01, 0.18179E+01,
     & 0.18209E+01, 0.18239E+01, 0.18270E+01, 0.18300E+01, 0.18330E+01,
     & 0.18360E+01, 0.18390E+01, 0.18420E+01, 0.18450E+01, 0.18480E+01,
     & 0.18510E+01, 0.18540E+01, 0.18570E+01, 0.18600E+01, 0.18629E+01,
     & 0.18659E+01, 0.18689E+01, 0.18718E+01, 0.18748E+01, 0.18777E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.18807E+01, 0.18836E+01, 0.18865E+01, 0.18895E+01, 0.18924E+01,
     & 0.18953E+01, 0.18982E+01, 0.19012E+01, 0.19041E+01, 0.19070E+01,
     & 0.19099E+01, 0.19128E+01, 0.19157E+01, 0.19185E+01, 0.19214E+01,
     & 0.19243E+01, 0.19272E+01, 0.19300E+01, 0.19329E+01, 0.19358E+01,
     & 0.19386E+01, 0.19415E+01, 0.19443E+01, 0.19472E+01, 0.19500E+01,
     & 0.19528E+01, 0.19557E+01, 0.19585E+01, 0.19613E+01, 0.19641E+01,
     & 0.19669E+01, 0.19697E+01, 0.19725E+01, 0.19753E+01, 0.19781E+01,
     & 0.19809E+01, 0.19837E+01, 0.19865E+01, 0.19893E+01, 0.19921E+01,
     & 0.19948E+01, 0.19976E+01, 0.20004E+01, 0.20031E+01, 0.20059E+01,
     & 0.20086E+01, 0.20114E+01, 0.20141E+01, 0.20168E+01, 0.20196E+01,
     & 0.20223E+01, 0.20250E+01, 0.20278E+01, 0.20305E+01, 0.20332E+01,
     & 0.20359E+01, 0.20386E+01, 0.20413E+01, 0.20440E+01, 0.20467E+01,
     & 0.20494E+01, 0.20521E+01, 0.20548E+01, 0.20574E+01, 0.20601E+01,
     & 0.20628E+01, 0.20654E+01, 0.20681E+01, 0.20708E+01, 0.20734E+01,
     & 0.20761E+01, 0.20787E+01, 0.20814E+01, 0.20840E+01, 0.20866E+01,
     & 0.20893E+01, 0.20919E+01, 0.20945E+01, 0.20971E+01, 0.20998E+01,
     & 0.21024E+01, 0.21050E+01, 0.21076E+01, 0.21102E+01, 0.21128E+01,
     & 0.21154E+01, 0.21180E+01, 0.21206E+01, 0.21231E+01, 0.21257E+01,
     & 0.21283E+01, 0.21309E+01, 0.21334E+01, 0.21360E+01, 0.21386E+01,
     & 0.21411E+01, 0.21437E+01, 0.21462E+01, 0.21488E+01, 0.21513E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.21786E+01, 0.22034E+01, 0.22280E+01, 0.22522E+01, 0.22760E+01,
     & 0.22996E+01, 0.23228E+01, 0.23457E+01, 0.23684E+01, 0.23907E+01,
     & 0.24128E+01, 0.24345E+01, 0.24560E+01, 0.24773E+01, 0.24982E+01,
     & 0.25190E+01, 0.25394E+01, 0.25596E+01, 0.25796E+01, 0.25993E+01,
     & 0.26188E+01, 0.26381E+01, 0.26571E+01, 0.26760E+01, 0.26946E+01,
     & 0.27130E+01, 0.27312E+01, 0.27491E+01, 0.27669E+01, 0.27845E+01,
     & 0.28019E+01, 0.28191E+01, 0.28361E+01, 0.28529E+01, 0.28696E+01,
     & 0.28860E+01, 0.29023E+01, 0.29185E+01, 0.29344E+01, 0.29502E+01,
     & 0.29658E+01, 0.29813E+01, 0.29966E+01, 0.30117E+01, 0.30267E+01,
     & 0.30415E+01, 0.30562E+01, 0.30708E+01, 0.30852E+01, 0.30994E+01,
     & 0.31136E+01, 0.31275E+01, 0.31414E+01, 0.31551E+01, 0.31687E+01,
     & 0.31821E+01, 0.31954E+01, 0.32086E+01, 0.32217E+01, 0.32346E+01,
     & 0.32474E+01, 0.32601E+01, 0.32727E+01, 0.32852E+01, 0.32976E+01,
     & 0.33098E+01, 0.33219E+01, 0.33339E+01, 0.33459E+01, 0.33577E+01,
     & 0.33694E+01, 0.33810E+01, 0.33924E+01, 0.34038E+01, 0.34151E+01,
     & 0.34263E+01, 0.34374E+01, 0.34484E+01, 0.34593E+01, 0.34701E+01,
     & 0.34808E+01, 0.34914E+01, 0.35020E+01, 0.35124E+01, 0.35228E+01,
     & 0.35330E+01, 0.35432E+01, 0.35533E+01, 0.35633E+01, 0.35732E+01,
     & 0.35831E+01, 0.35928E+01, 0.36025E+01, 0.36121E+01, 0.36216E+01,
     & 0.36311E+01, 0.36405E+01, 0.36497E+01, 0.36590E+01, 0.36681E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.36772E+01, 0.36862E+01, 0.36951E+01, 0.37039E+01, 0.37127E+01,
     & 0.37214E+01, 0.37301E+01, 0.37386E+01, 0.37472E+01, 0.37556E+01,
     & 0.37640E+01, 0.37723E+01, 0.37805E+01, 0.37887E+01, 0.37968E+01,
     & 0.38049E+01, 0.38129E+01, 0.38208E+01, 0.38287E+01, 0.38365E+01,
     & 0.38442E+01, 0.38519E+01, 0.38595E+01, 0.38671E+01, 0.38746E+01,
     & 0.38821E+01, 0.38895E+01, 0.38968E+01, 0.39041E+01, 0.39114E+01,
     & 0.39186E+01, 0.39257E+01, 0.39328E+01, 0.39398E+01, 0.39468E+01,
     & 0.39537E+01, 0.39606E+01, 0.39674E+01, 0.39742E+01, 0.39809E+01,
     & 0.39875E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.79945E-01,-0.17380E+00,-0.22078E+00,-0.25407E+00,-0.28025E+00,
     &-0.30198E+00,-0.32062E+00,-0.33698E+00,-0.35158E+00,-0.36476E+00,
     &-0.37680E+00,-0.38787E+00,-0.39813E+00,-0.40767E+00,-0.41660E+00,
     &-0.42499E+00,-0.43289E+00,-0.44036E+00,-0.44744E+00,-0.45416E+00,
     &-0.46056E+00,-0.46665E+00,-0.47248E+00,-0.47805E+00,-0.48338E+00,
     &-0.48850E+00,-0.49341E+00,-0.49812E+00,-0.50265E+00,-0.50701E+00,
     &-0.51121E+00,-0.51525E+00,-0.51915E+00,-0.52291E+00,-0.52654E+00,
     &-0.53004E+00,-0.53342E+00,-0.53669E+00,-0.53985E+00,-0.54290E+00,
     &-0.54585E+00,-0.54871E+00,-0.55147E+00,-0.55415E+00,-0.55674E+00,
     &-0.55924E+00,-0.56167E+00,-0.56402E+00,-0.56630E+00,-0.56851E+00,
     &-0.57065E+00,-0.57272E+00,-0.57473E+00,-0.57668E+00,-0.57857E+00,
     &-0.58041E+00,-0.58219E+00,-0.58391E+00,-0.58559E+00,-0.58721E+00,
     &-0.58879E+00,-0.59031E+00,-0.59180E+00,-0.59324E+00,-0.59464E+00,
     &-0.59599E+00,-0.59731E+00,-0.59859E+00,-0.59983E+00,-0.60103E+00,
     &-0.60220E+00,-0.60333E+00,-0.60443E+00,-0.60550E+00,-0.60653E+00,
     &-0.60753E+00,-0.60850E+00,-0.60945E+00,-0.61036E+00,-0.61124E+00,
     &-0.61209E+00,-0.61292E+00,-0.61372E+00,-0.61449E+00,-0.61524E+00,
     &-0.61596E+00,-0.61666E+00,-0.61733E+00,-0.61798E+00,-0.61861E+00,
     &-0.61921E+00,-0.61979E+00,-0.62035E+00,-0.62089E+00,-0.62141E+00,
     &-0.62190E+00,-0.62238E+00,-0.62284E+00,-0.62328E+00,-0.62370E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.62410E+00,-0.62449E+00,-0.62486E+00,-0.62521E+00,-0.62555E+00,
     &-0.62587E+00,-0.62618E+00,-0.62648E+00,-0.62676E+00,-0.62702E+00,
     &-0.62728E+00,-0.62752E+00,-0.62776E+00,-0.62798E+00,-0.62819E+00,
     &-0.62839E+00,-0.62858E+00,-0.62876E+00,-0.62893E+00,-0.62910E+00,
     &-0.62925E+00,-0.62940E+00,-0.62954E+00,-0.62967E+00,-0.62980E+00,
     &-0.62992E+00,-0.63003E+00,-0.63014E+00,-0.63024E+00,-0.63034E+00,
     &-0.63043E+00,-0.63052E+00,-0.63060E+00,-0.63068E+00,-0.63075E+00,
     &-0.63082E+00,-0.63089E+00,-0.63095E+00,-0.63101E+00,-0.63106E+00,
     &-0.63111E+00,-0.63116E+00,-0.63121E+00,-0.63125E+00,-0.63129E+00,
     &-0.63133E+00,-0.63136E+00,-0.63140E+00,-0.63143E+00,-0.63146E+00,
     &-0.63148E+00,-0.63151E+00,-0.63153E+00,-0.63155E+00,-0.63157E+00,
     &-0.63159E+00,-0.63161E+00,-0.63162E+00,-0.63164E+00,-0.63165E+00,
     &-0.63166E+00,-0.63167E+00,-0.63168E+00,-0.63169E+00,-0.63170E+00,
     &-0.63170E+00,-0.63171E+00,-0.63171E+00,-0.63172E+00,-0.63172E+00,
     &-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,
     &-0.63174E+00,-0.63174E+00,-0.63174E+00,-0.63174E+00,-0.63174E+00,
     &-0.63174E+00,-0.63174E+00,-0.63174E+00,-0.63174E+00,-0.63174E+00,
     &-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,
     &-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,-0.63173E+00,
     &-0.63173E+00,-0.63173E+00,-0.63174E+00,-0.63174E+00,-0.63174E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.63174E+00,-0.63174E+00,-0.63175E+00,-0.63175E+00,-0.63175E+00,
     &-0.63176E+00,-0.63176E+00,-0.63176E+00,-0.63177E+00,-0.63178E+00,
     &-0.63178E+00,-0.63179E+00,-0.63180E+00,-0.63180E+00,-0.63181E+00,
     &-0.63182E+00,-0.63183E+00,-0.63184E+00,-0.63185E+00,-0.63186E+00,
     &-0.63187E+00,-0.63188E+00,-0.63190E+00,-0.63191E+00,-0.63192E+00,
     &-0.63194E+00,-0.63195E+00,-0.63197E+00,-0.63199E+00,-0.63200E+00,
     &-0.63202E+00,-0.63204E+00,-0.63206E+00,-0.63208E+00,-0.63210E+00,
     &-0.63212E+00,-0.63214E+00,-0.63216E+00,-0.63219E+00,-0.63221E+00,
     &-0.63223E+00,-0.63226E+00,-0.63229E+00,-0.63231E+00,-0.63234E+00,
     &-0.63237E+00,-0.63240E+00,-0.63243E+00,-0.63246E+00,-0.63249E+00,
     &-0.63252E+00,-0.63255E+00,-0.63259E+00,-0.63262E+00,-0.63266E+00,
     &-0.63269E+00,-0.63273E+00,-0.63277E+00,-0.63280E+00,-0.63284E+00,
     &-0.63288E+00,-0.63292E+00,-0.63296E+00,-0.63301E+00,-0.63305E+00,
     &-0.63309E+00,-0.63314E+00,-0.63318E+00,-0.63323E+00,-0.63327E+00,
     &-0.63332E+00,-0.63337E+00,-0.63342E+00,-0.63347E+00,-0.63352E+00,
     &-0.63357E+00,-0.63362E+00,-0.63367E+00,-0.63373E+00,-0.63378E+00,
     &-0.63384E+00,-0.63389E+00,-0.63395E+00,-0.63401E+00,-0.63407E+00,
     &-0.63413E+00,-0.63419E+00,-0.63425E+00,-0.63431E+00,-0.63437E+00,
     &-0.63443E+00,-0.63450E+00,-0.63456E+00,-0.63463E+00,-0.63470E+00,
     &-0.63476E+00,-0.63483E+00,-0.63490E+00,-0.63497E+00,-0.63504E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.63511E+00,-0.63518E+00,-0.63526E+00,-0.63533E+00,-0.63540E+00,
     &-0.63548E+00,-0.63556E+00,-0.63563E+00,-0.63571E+00,-0.63579E+00,
     &-0.63587E+00,-0.63595E+00,-0.63603E+00,-0.63611E+00,-0.63619E+00,
     &-0.63627E+00,-0.63636E+00,-0.63644E+00,-0.63653E+00,-0.63661E+00,
     &-0.63670E+00,-0.63679E+00,-0.63688E+00,-0.63697E+00,-0.63706E+00,
     &-0.63715E+00,-0.63724E+00,-0.63733E+00,-0.63743E+00,-0.63752E+00,
     &-0.63761E+00,-0.63771E+00,-0.63781E+00,-0.63790E+00,-0.63800E+00,
     &-0.63810E+00,-0.63820E+00,-0.63830E+00,-0.63840E+00,-0.63850E+00,
     &-0.63860E+00,-0.63871E+00,-0.63881E+00,-0.63892E+00,-0.63902E+00,
     &-0.63913E+00,-0.63923E+00,-0.63934E+00,-0.63945E+00,-0.63956E+00,
     &-0.63967E+00,-0.63978E+00,-0.63989E+00,-0.64000E+00,-0.64012E+00,
     &-0.64023E+00,-0.64034E+00,-0.64046E+00,-0.64058E+00,-0.64069E+00,
     &-0.64081E+00,-0.64093E+00,-0.64105E+00,-0.64117E+00,-0.64129E+00,
     &-0.64141E+00,-0.64153E+00,-0.64165E+00,-0.64177E+00,-0.64190E+00,
     &-0.64202E+00,-0.64215E+00,-0.64228E+00,-0.64240E+00,-0.64253E+00,
     &-0.64266E+00,-0.64279E+00,-0.64292E+00,-0.64305E+00,-0.64318E+00,
     &-0.64331E+00,-0.64344E+00,-0.64358E+00,-0.64371E+00,-0.64385E+00,
     &-0.64398E+00,-0.64412E+00,-0.64425E+00,-0.64439E+00,-0.64453E+00,
     &-0.64467E+00,-0.64481E+00,-0.64495E+00,-0.64509E+00,-0.64523E+00,
     &-0.64537E+00,-0.64552E+00,-0.64566E+00,-0.64581E+00,-0.64595E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.64610E+00,-0.64624E+00,-0.64639E+00,-0.64654E+00,-0.64669E+00,
     &-0.64684E+00,-0.64699E+00,-0.64714E+00,-0.64729E+00,-0.64744E+00,
     &-0.64759E+00,-0.64775E+00,-0.64790E+00,-0.64805E+00,-0.64821E+00,
     &-0.64837E+00,-0.64852E+00,-0.64868E+00,-0.64884E+00,-0.64900E+00,
     &-0.64916E+00,-0.64932E+00,-0.64948E+00,-0.64964E+00,-0.64980E+00,
     &-0.64996E+00,-0.65013E+00,-0.65029E+00,-0.65045E+00,-0.65062E+00,
     &-0.65079E+00,-0.65095E+00,-0.65112E+00,-0.65129E+00,-0.65146E+00,
     &-0.65163E+00,-0.65180E+00,-0.65197E+00,-0.65214E+00,-0.65231E+00,
     &-0.65248E+00,-0.65265E+00,-0.65283E+00,-0.65300E+00,-0.65318E+00,
     &-0.65335E+00,-0.65353E+00,-0.65370E+00,-0.65388E+00,-0.65406E+00,
     &-0.65424E+00,-0.65442E+00,-0.65460E+00,-0.65478E+00,-0.65496E+00,
     &-0.65514E+00,-0.65532E+00,-0.65551E+00,-0.65569E+00,-0.65587E+00,
     &-0.65606E+00,-0.65624E+00,-0.65643E+00,-0.65662E+00,-0.65680E+00,
     &-0.65699E+00,-0.65718E+00,-0.65737E+00,-0.65756E+00,-0.65775E+00,
     &-0.65794E+00,-0.65813E+00,-0.65832E+00,-0.65852E+00,-0.65871E+00,
     &-0.65890E+00,-0.65910E+00,-0.65929E+00,-0.65949E+00,-0.65968E+00,
     &-0.65988E+00,-0.66008E+00,-0.66028E+00,-0.66047E+00,-0.66067E+00,
     &-0.66087E+00,-0.66107E+00,-0.66127E+00,-0.66147E+00,-0.66168E+00,
     &-0.66188E+00,-0.66208E+00,-0.66229E+00,-0.66249E+00,-0.66269E+00,
     &-0.66290E+00,-0.66311E+00,-0.66331E+00,-0.66352E+00,-0.66373E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.66394E+00,-0.66414E+00,-0.66435E+00,-0.66456E+00,-0.66477E+00,
     &-0.66499E+00,-0.66520E+00,-0.66541E+00,-0.66562E+00,-0.66583E+00,
     &-0.66605E+00,-0.66626E+00,-0.66648E+00,-0.66669E+00,-0.66691E+00,
     &-0.66713E+00,-0.66734E+00,-0.66756E+00,-0.66778E+00,-0.66800E+00,
     &-0.66822E+00,-0.66844E+00,-0.66866E+00,-0.66888E+00,-0.66910E+00,
     &-0.66932E+00,-0.66954E+00,-0.66977E+00,-0.66999E+00,-0.67021E+00,
     &-0.67044E+00,-0.67066E+00,-0.67089E+00,-0.67111E+00,-0.67134E+00,
     &-0.67157E+00,-0.67180E+00,-0.67202E+00,-0.67225E+00,-0.67248E+00,
     &-0.67271E+00,-0.67294E+00,-0.67317E+00,-0.67340E+00,-0.67364E+00,
     &-0.67387E+00,-0.67410E+00,-0.67434E+00,-0.67457E+00,-0.67480E+00,
     &-0.67504E+00,-0.67527E+00,-0.67551E+00,-0.67575E+00,-0.67598E+00,
     &-0.67622E+00,-0.67646E+00,-0.67670E+00,-0.67694E+00,-0.67718E+00,
     &-0.67742E+00,-0.67766E+00,-0.67790E+00,-0.67814E+00,-0.67838E+00,
     &-0.67862E+00,-0.67887E+00,-0.67911E+00,-0.67935E+00,-0.67960E+00,
     &-0.67984E+00,-0.68009E+00,-0.68033E+00,-0.68058E+00,-0.68083E+00,
     &-0.68107E+00,-0.68132E+00,-0.68157E+00,-0.68182E+00,-0.68207E+00,
     &-0.68232E+00,-0.68257E+00,-0.68282E+00,-0.68307E+00,-0.68332E+00,
     &-0.68357E+00,-0.68383E+00,-0.68408E+00,-0.68433E+00,-0.68459E+00,
     &-0.68484E+00,-0.68510E+00,-0.68535E+00,-0.68561E+00,-0.68586E+00,
     &-0.68612E+00,-0.68638E+00,-0.68664E+00,-0.68689E+00,-0.68715E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.68998E+00,-0.69264E+00,-0.69534E+00,-0.69809E+00,-0.70088E+00,
     &-0.70371E+00,-0.70659E+00,-0.70950E+00,-0.71245E+00,-0.71544E+00,
     &-0.71847E+00,-0.72154E+00,-0.72464E+00,-0.72778E+00,-0.73095E+00,
     &-0.73416E+00,-0.73741E+00,-0.74068E+00,-0.74399E+00,-0.74733E+00,
     &-0.75070E+00,-0.75411E+00,-0.75754E+00,-0.76101E+00,-0.76450E+00,
     &-0.76802E+00,-0.77157E+00,-0.77515E+00,-0.77876E+00,-0.78239E+00,
     &-0.78605E+00,-0.78974E+00,-0.79345E+00,-0.79719E+00,-0.80095E+00,
     &-0.80474E+00,-0.80855E+00,-0.81239E+00,-0.81625E+00,-0.82013E+00,
     &-0.82403E+00,-0.82796E+00,-0.83191E+00,-0.83588E+00,-0.83987E+00,
     &-0.84388E+00,-0.84792E+00,-0.85197E+00,-0.85604E+00,-0.86014E+00,
     &-0.86425E+00,-0.86838E+00,-0.87253E+00,-0.87670E+00,-0.88089E+00,
     &-0.88510E+00,-0.88932E+00,-0.89357E+00,-0.89782E+00,-0.90210E+00,
     &-0.90639E+00,-0.91070E+00,-0.91503E+00,-0.91937E+00,-0.92373E+00,
     &-0.92811E+00,-0.93250E+00,-0.93690E+00,-0.94132E+00,-0.94576E+00,
     &-0.95021E+00,-0.95467E+00,-0.95915E+00,-0.96364E+00,-0.96815E+00,
     &-0.97267E+00,-0.97721E+00,-0.98176E+00,-0.98632E+00,-0.99089E+00,
     &-0.99548E+00,-0.10001E+01,-0.10047E+01,-0.10093E+01,-0.10140E+01,
     &-0.10186E+01,-0.10233E+01,-0.10279E+01,-0.10326E+01,-0.10373E+01,
     &-0.10420E+01,-0.10468E+01,-0.10515E+01,-0.10562E+01,-0.10610E+01,
     &-0.10657E+01,-0.10705E+01,-0.10753E+01,-0.10801E+01,-0.10849E+01/

      DATA (BNC13M(I),I=701,741)/
     &-0.10897E+01,-0.10945E+01,-0.10994E+01,-0.11042E+01,-0.11091E+01,
     &-0.11139E+01,-0.11188E+01,-0.11237E+01,-0.11286E+01,-0.11335E+01,
     &-0.11384E+01,-0.11433E+01,-0.11483E+01,-0.11532E+01,-0.11582E+01,
     &-0.11631E+01,-0.11681E+01,-0.11730E+01,-0.11780E+01,-0.11830E+01,
     &-0.11880E+01,-0.11930E+01,-0.11980E+01,-0.12031E+01,-0.12081E+01,
     &-0.12131E+01,-0.12182E+01,-0.12232E+01,-0.12283E+01,-0.12333E+01,
     &-0.12384E+01,-0.12435E+01,-0.12486E+01,-0.12537E+01,-0.12588E+01,
     &-0.12639E+01,-0.12690E+01,-0.12741E+01,-0.12793E+01,-0.12844E+01,
     &-0.12896E+01
     & /
      END

 
 
C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF248A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C 
      BLOCK DATA KMCF248A
C
C *** Common block definition
C
      COMMON /KMC248/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.47484E-01,-0.98816E-01,-0.12198E+00,-0.13703E+00,-0.14791E+00,
     &-0.15621E+00,-0.16272E+00,-0.16794E+00,-0.17216E+00,-0.17559E+00,
     &-0.17838E+00,-0.18065E+00,-0.18247E+00,-0.18392E+00,-0.18504E+00,
     &-0.18588E+00,-0.18647E+00,-0.18685E+00,-0.18703E+00,-0.18705E+00,
     &-0.18691E+00,-0.18663E+00,-0.18623E+00,-0.18571E+00,-0.18510E+00,
     &-0.18440E+00,-0.18361E+00,-0.18275E+00,-0.18181E+00,-0.18082E+00,
     &-0.17977E+00,-0.17867E+00,-0.17752E+00,-0.17633E+00,-0.17510E+00,
     &-0.17383E+00,-0.17253E+00,-0.17121E+00,-0.16985E+00,-0.16848E+00,
     &-0.16708E+00,-0.16566E+00,-0.16423E+00,-0.16278E+00,-0.16131E+00,
     &-0.15984E+00,-0.15835E+00,-0.15685E+00,-0.15534E+00,-0.15383E+00,
     &-0.15230E+00,-0.15077E+00,-0.14924E+00,-0.14769E+00,-0.14614E+00,
     &-0.14459E+00,-0.14303E+00,-0.14147E+00,-0.13990E+00,-0.13832E+00,
     &-0.13674E+00,-0.13516E+00,-0.13357E+00,-0.13197E+00,-0.13037E+00,
     &-0.12876E+00,-0.12715E+00,-0.12553E+00,-0.12389E+00,-0.12226E+00,
     &-0.12061E+00,-0.11895E+00,-0.11728E+00,-0.11561E+00,-0.11392E+00,
     &-0.11222E+00,-0.11051E+00,-0.10878E+00,-0.10705E+00,-0.10530E+00,
     &-0.10354E+00,-0.10176E+00,-0.99968E-01,-0.98163E-01,-0.96343E-01,
     &-0.94509E-01,-0.92660E-01,-0.90795E-01,-0.88916E-01,-0.87022E-01,
     &-0.85112E-01,-0.83188E-01,-0.81249E-01,-0.79295E-01,-0.77327E-01,
     &-0.75345E-01,-0.73349E-01,-0.71339E-01,-0.69317E-01,-0.67281E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.65233E-01,-0.63173E-01,-0.61101E-01,-0.59019E-01,-0.56926E-01,
     &-0.54823E-01,-0.52710E-01,-0.50588E-01,-0.48457E-01,-0.46319E-01,
     &-0.44173E-01,-0.42019E-01,-0.39859E-01,-0.37694E-01,-0.35522E-01,
     &-0.33345E-01,-0.31164E-01,-0.28978E-01,-0.26789E-01,-0.24596E-01,
     &-0.22400E-01,-0.20201E-01,-0.18000E-01,-0.15797E-01,-0.13592E-01,
     &-0.11386E-01,-0.91787E-02,-0.69707E-02,-0.47622E-02,-0.25533E-02,
     &-0.34438E-03, 0.18645E-02, 0.40730E-02, 0.62810E-02, 0.84884E-02,
     & 0.10695E-01, 0.12900E-01, 0.15105E-01, 0.17308E-01, 0.19510E-01,
     & 0.21710E-01, 0.23909E-01, 0.26107E-01, 0.28302E-01, 0.30496E-01,
     & 0.32687E-01, 0.34877E-01, 0.37065E-01, 0.39251E-01, 0.41434E-01,
     & 0.43616E-01, 0.45795E-01, 0.47972E-01, 0.50147E-01, 0.52319E-01,
     & 0.54489E-01, 0.56656E-01, 0.58821E-01, 0.60983E-01, 0.63143E-01,
     & 0.65300E-01, 0.67455E-01, 0.69607E-01, 0.71757E-01, 0.73903E-01,
     & 0.76047E-01, 0.78189E-01, 0.80327E-01, 0.82463E-01, 0.84596E-01,
     & 0.86727E-01, 0.88854E-01, 0.90979E-01, 0.93101E-01, 0.95220E-01,
     & 0.97336E-01, 0.99449E-01, 0.10156E+00, 0.10367E+00, 0.10577E+00,
     & 0.10787E+00, 0.10997E+00, 0.11207E+00, 0.11416E+00, 0.11625E+00,
     & 0.11833E+00, 0.12042E+00, 0.12250E+00, 0.12458E+00, 0.12665E+00,
     & 0.12872E+00, 0.13079E+00, 0.13286E+00, 0.13492E+00, 0.13698E+00,
     & 0.13903E+00, 0.14108E+00, 0.14313E+00, 0.14518E+00, 0.14722E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.14926E+00, 0.15130E+00, 0.15334E+00, 0.15537E+00, 0.15739E+00,
     & 0.15942E+00, 0.16144E+00, 0.16346E+00, 0.16547E+00, 0.16748E+00,
     & 0.16949E+00, 0.17150E+00, 0.17350E+00, 0.17550E+00, 0.17749E+00,
     & 0.17949E+00, 0.18148E+00, 0.18346E+00, 0.18545E+00, 0.18743E+00,
     & 0.18940E+00, 0.19138E+00, 0.19335E+00, 0.19531E+00, 0.19728E+00,
     & 0.19924E+00, 0.20119E+00, 0.20315E+00, 0.20510E+00, 0.20705E+00,
     & 0.20899E+00, 0.21093E+00, 0.21287E+00, 0.21480E+00, 0.21674E+00,
     & 0.21866E+00, 0.22059E+00, 0.22251E+00, 0.22443E+00, 0.22635E+00,
     & 0.22826E+00, 0.23017E+00, 0.23207E+00, 0.23398E+00, 0.23588E+00,
     & 0.23777E+00, 0.23967E+00, 0.24156E+00, 0.24344E+00, 0.24533E+00,
     & 0.24721E+00, 0.24908E+00, 0.25096E+00, 0.25283E+00, 0.25470E+00,
     & 0.25656E+00, 0.25842E+00, 0.26028E+00, 0.26214E+00, 0.26399E+00,
     & 0.26584E+00, 0.26768E+00, 0.26953E+00, 0.27137E+00, 0.27320E+00,
     & 0.27504E+00, 0.27687E+00, 0.27869E+00, 0.28052E+00, 0.28234E+00,
     & 0.28415E+00, 0.28597E+00, 0.28778E+00, 0.28959E+00, 0.29139E+00,
     & 0.29320E+00, 0.29499E+00, 0.29679E+00, 0.29858E+00, 0.30037E+00,
     & 0.30216E+00, 0.30394E+00, 0.30572E+00, 0.30750E+00, 0.30928E+00,
     & 0.31105E+00, 0.31282E+00, 0.31458E+00, 0.31634E+00, 0.31810E+00,
     & 0.31986E+00, 0.32161E+00, 0.32336E+00, 0.32511E+00, 0.32686E+00,
     & 0.32860E+00, 0.33033E+00, 0.33207E+00, 0.33380E+00, 0.33553E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.33726E+00, 0.33898E+00, 0.34070E+00, 0.34242E+00, 0.34413E+00,
     & 0.34585E+00, 0.34755E+00, 0.34926E+00, 0.35096E+00, 0.35266E+00,
     & 0.35436E+00, 0.35605E+00, 0.35775E+00, 0.35943E+00, 0.36112E+00,
     & 0.36280E+00, 0.36448E+00, 0.36616E+00, 0.36783E+00, 0.36950E+00,
     & 0.37117E+00, 0.37284E+00, 0.37450E+00, 0.37616E+00, 0.37782E+00,
     & 0.37947E+00, 0.38112E+00, 0.38277E+00, 0.38442E+00, 0.38606E+00,
     & 0.38770E+00, 0.38934E+00, 0.39097E+00, 0.39260E+00, 0.39423E+00,
     & 0.39586E+00, 0.39748E+00, 0.39910E+00, 0.40072E+00, 0.40233E+00,
     & 0.40394E+00, 0.40555E+00, 0.40716E+00, 0.40876E+00, 0.41037E+00,
     & 0.41196E+00, 0.41356E+00, 0.41515E+00, 0.41674E+00, 0.41833E+00,
     & 0.41992E+00, 0.42150E+00, 0.42308E+00, 0.42466E+00, 0.42623E+00,
     & 0.42780E+00, 0.42937E+00, 0.43094E+00, 0.43250E+00, 0.43406E+00,
     & 0.43562E+00, 0.43718E+00, 0.43873E+00, 0.44028E+00, 0.44183E+00,
     & 0.44337E+00, 0.44492E+00, 0.44646E+00, 0.44799E+00, 0.44953E+00,
     & 0.45106E+00, 0.45259E+00, 0.45412E+00, 0.45564E+00, 0.45717E+00,
     & 0.45868E+00, 0.46020E+00, 0.46172E+00, 0.46323E+00, 0.46474E+00,
     & 0.46625E+00, 0.46775E+00, 0.46925E+00, 0.47075E+00, 0.47225E+00,
     & 0.47374E+00, 0.47523E+00, 0.47672E+00, 0.47821E+00, 0.47970E+00,
     & 0.48118E+00, 0.48266E+00, 0.48414E+00, 0.48561E+00, 0.48708E+00,
     & 0.48855E+00, 0.49002E+00, 0.49148E+00, 0.49295E+00, 0.49441E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.49587E+00, 0.49732E+00, 0.49877E+00, 0.50022E+00, 0.50167E+00,
     & 0.50312E+00, 0.50456E+00, 0.50600E+00, 0.50744E+00, 0.50888E+00,
     & 0.51031E+00, 0.51174E+00, 0.51317E+00, 0.51460E+00, 0.51602E+00,
     & 0.51745E+00, 0.51887E+00, 0.52028E+00, 0.52170E+00, 0.52311E+00,
     & 0.52452E+00, 0.52593E+00, 0.52734E+00, 0.52874E+00, 0.53014E+00,
     & 0.53154E+00, 0.53294E+00, 0.53433E+00, 0.53573E+00, 0.53712E+00,
     & 0.53851E+00, 0.53989E+00, 0.54127E+00, 0.54266E+00, 0.54403E+00,
     & 0.54541E+00, 0.54679E+00, 0.54816E+00, 0.54953E+00, 0.55090E+00,
     & 0.55226E+00, 0.55363E+00, 0.55499E+00, 0.55635E+00, 0.55770E+00,
     & 0.55906E+00, 0.56041E+00, 0.56176E+00, 0.56311E+00, 0.56446E+00,
     & 0.56580E+00, 0.56714E+00, 0.56848E+00, 0.56982E+00, 0.57115E+00,
     & 0.57249E+00, 0.57382E+00, 0.57515E+00, 0.57648E+00, 0.57780E+00,
     & 0.57912E+00, 0.58044E+00, 0.58176E+00, 0.58308E+00, 0.58439E+00,
     & 0.58571E+00, 0.58702E+00, 0.58832E+00, 0.58963E+00, 0.59093E+00,
     & 0.59224E+00, 0.59354E+00, 0.59483E+00, 0.59613E+00, 0.59742E+00,
     & 0.59872E+00, 0.60001E+00, 0.60129E+00, 0.60258E+00, 0.60386E+00,
     & 0.60514E+00, 0.60642E+00, 0.60770E+00, 0.60898E+00, 0.61025E+00,
     & 0.61152E+00, 0.61279E+00, 0.61406E+00, 0.61533E+00, 0.61659E+00,
     & 0.61785E+00, 0.61911E+00, 0.62037E+00, 0.62163E+00, 0.62288E+00,
     & 0.62413E+00, 0.62538E+00, 0.62663E+00, 0.62788E+00, 0.62912E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.63036E+00, 0.63160E+00, 0.63284E+00, 0.63408E+00, 0.63531E+00,
     & 0.63655E+00, 0.63778E+00, 0.63901E+00, 0.64023E+00, 0.64146E+00,
     & 0.64268E+00, 0.64390E+00, 0.64512E+00, 0.64634E+00, 0.64756E+00,
     & 0.64877E+00, 0.64998E+00, 0.65120E+00, 0.65240E+00, 0.65361E+00,
     & 0.65482E+00, 0.65602E+00, 0.65722E+00, 0.65842E+00, 0.65962E+00,
     & 0.66081E+00, 0.66201E+00, 0.66320E+00, 0.66439E+00, 0.66558E+00,
     & 0.66676E+00, 0.66795E+00, 0.66913E+00, 0.67031E+00, 0.67149E+00,
     & 0.67267E+00, 0.67385E+00, 0.67502E+00, 0.67619E+00, 0.67736E+00,
     & 0.67853E+00, 0.67970E+00, 0.68087E+00, 0.68203E+00, 0.68319E+00,
     & 0.68435E+00, 0.68551E+00, 0.68667E+00, 0.68782E+00, 0.68898E+00,
     & 0.69013E+00, 0.69128E+00, 0.69243E+00, 0.69357E+00, 0.69472E+00,
     & 0.69586E+00, 0.69700E+00, 0.69814E+00, 0.69928E+00, 0.70042E+00,
     & 0.70155E+00, 0.70268E+00, 0.70381E+00, 0.70494E+00, 0.70607E+00,
     & 0.70720E+00, 0.70832E+00, 0.70945E+00, 0.71057E+00, 0.71169E+00,
     & 0.71280E+00, 0.71392E+00, 0.71504E+00, 0.71615E+00, 0.71726E+00,
     & 0.71837E+00, 0.71948E+00, 0.72058E+00, 0.72169E+00, 0.72279E+00,
     & 0.72389E+00, 0.72499E+00, 0.72609E+00, 0.72719E+00, 0.72828E+00,
     & 0.72938E+00, 0.73047E+00, 0.73156E+00, 0.73265E+00, 0.73374E+00,
     & 0.73482E+00, 0.73591E+00, 0.73699E+00, 0.73807E+00, 0.73915E+00,
     & 0.74023E+00, 0.74130E+00, 0.74238E+00, 0.74345E+00, 0.74452E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.75600E+00, 0.76647E+00, 0.77680E+00, 0.78697E+00, 0.79701E+00,
     & 0.80690E+00, 0.81666E+00, 0.82629E+00, 0.83578E+00, 0.84515E+00,
     & 0.85439E+00, 0.86351E+00, 0.87250E+00, 0.88138E+00, 0.89014E+00,
     & 0.89879E+00, 0.90732E+00, 0.91574E+00, 0.92406E+00, 0.93227E+00,
     & 0.94038E+00, 0.94838E+00, 0.95628E+00, 0.96409E+00, 0.97180E+00,
     & 0.97941E+00, 0.98693E+00, 0.99435E+00, 0.10017E+01, 0.10089E+01,
     & 0.10161E+01, 0.10232E+01, 0.10302E+01, 0.10371E+01, 0.10439E+01,
     & 0.10506E+01, 0.10573E+01, 0.10639E+01, 0.10704E+01, 0.10768E+01,
     & 0.10832E+01, 0.10895E+01, 0.10957E+01, 0.11018E+01, 0.11079E+01,
     & 0.11139E+01, 0.11199E+01, 0.11257E+01, 0.11315E+01, 0.11373E+01,
     & 0.11430E+01, 0.11486E+01, 0.11541E+01, 0.11596E+01, 0.11651E+01,
     & 0.11704E+01, 0.11757E+01, 0.11810E+01, 0.11862E+01, 0.11913E+01,
     & 0.11964E+01, 0.12015E+01, 0.12064E+01, 0.12114E+01, 0.12162E+01,
     & 0.12211E+01, 0.12258E+01, 0.12306E+01, 0.12352E+01, 0.12399E+01,
     & 0.12444E+01, 0.12490E+01, 0.12534E+01, 0.12579E+01, 0.12623E+01,
     & 0.12666E+01, 0.12709E+01, 0.12751E+01, 0.12794E+01, 0.12835E+01,
     & 0.12876E+01, 0.12917E+01, 0.12958E+01, 0.12998E+01, 0.13037E+01,
     & 0.13076E+01, 0.13115E+01, 0.13153E+01, 0.13191E+01, 0.13229E+01,
     & 0.13266E+01, 0.13303E+01, 0.13340E+01, 0.13376E+01, 0.13411E+01,
     & 0.13447E+01, 0.13482E+01, 0.13517E+01, 0.13551E+01, 0.13585E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.13619E+01, 0.13652E+01, 0.13685E+01, 0.13718E+01, 0.13750E+01,
     & 0.13782E+01, 0.13814E+01, 0.13845E+01, 0.13876E+01, 0.13907E+01,
     & 0.13937E+01, 0.13967E+01, 0.13997E+01, 0.14027E+01, 0.14056E+01,
     & 0.14085E+01, 0.14114E+01, 0.14142E+01, 0.14170E+01, 0.14198E+01,
     & 0.14226E+01, 0.14253E+01, 0.14280E+01, 0.14307E+01, 0.14333E+01,
     & 0.14360E+01, 0.14386E+01, 0.14411E+01, 0.14437E+01, 0.14462E+01,
     & 0.14487E+01, 0.14512E+01, 0.14536E+01, 0.14561E+01, 0.14585E+01,
     & 0.14608E+01, 0.14632E+01, 0.14655E+01, 0.14678E+01, 0.14701E+01,
     & 0.14724E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.97981E-01,-0.21379E+00,-0.27204E+00,-0.31341E+00,-0.34603E+00,
     &-0.37314E+00,-0.39646E+00,-0.41696E+00,-0.43531E+00,-0.45193E+00,
     &-0.46714E+00,-0.48118E+00,-0.49423E+00,-0.50643E+00,-0.51788E+00,
     &-0.52870E+00,-0.53893E+00,-0.54866E+00,-0.55793E+00,-0.56679E+00,
     &-0.57528E+00,-0.58342E+00,-0.59125E+00,-0.59880E+00,-0.60608E+00,
     &-0.61312E+00,-0.61993E+00,-0.62653E+00,-0.63293E+00,-0.63915E+00,
     &-0.64519E+00,-0.65108E+00,-0.65681E+00,-0.66239E+00,-0.66784E+00,
     &-0.67316E+00,-0.67836E+00,-0.68344E+00,-0.68841E+00,-0.69328E+00,
     &-0.69804E+00,-0.70271E+00,-0.70729E+00,-0.71178E+00,-0.71618E+00,
     &-0.72051E+00,-0.72476E+00,-0.72893E+00,-0.73304E+00,-0.73707E+00,
     &-0.74104E+00,-0.74495E+00,-0.74880E+00,-0.75259E+00,-0.75632E+00,
     &-0.75999E+00,-0.76362E+00,-0.76719E+00,-0.77072E+00,-0.77420E+00,
     &-0.77763E+00,-0.78102E+00,-0.78436E+00,-0.78767E+00,-0.79094E+00,
     &-0.79416E+00,-0.79735E+00,-0.80051E+00,-0.80363E+00,-0.80671E+00,
     &-0.80977E+00,-0.81279E+00,-0.81578E+00,-0.81875E+00,-0.82168E+00,
     &-0.82459E+00,-0.82747E+00,-0.83032E+00,-0.83315E+00,-0.83596E+00,
     &-0.83874E+00,-0.84150E+00,-0.84423E+00,-0.84695E+00,-0.84964E+00,
     &-0.85232E+00,-0.85497E+00,-0.85760E+00,-0.86022E+00,-0.86282E+00,
     &-0.86540E+00,-0.86796E+00,-0.87051E+00,-0.87303E+00,-0.87555E+00,
     &-0.87804E+00,-0.88052E+00,-0.88299E+00,-0.88544E+00,-0.88788E+00/

      DATA (BNC02M(I),I=101,200)/
     &-0.89030E+00,-0.89271E+00,-0.89510E+00,-0.89748E+00,-0.89985E+00,
     &-0.90220E+00,-0.90454E+00,-0.90687E+00,-0.90918E+00,-0.91148E+00,
     &-0.91377E+00,-0.91605E+00,-0.91832E+00,-0.92057E+00,-0.92281E+00,
     &-0.92504E+00,-0.92726E+00,-0.92947E+00,-0.93167E+00,-0.93385E+00,
     &-0.93603E+00,-0.93819E+00,-0.94034E+00,-0.94249E+00,-0.94462E+00,
     &-0.94674E+00,-0.94886E+00,-0.95096E+00,-0.95305E+00,-0.95513E+00,
     &-0.95721E+00,-0.95927E+00,-0.96132E+00,-0.96337E+00,-0.96540E+00,
     &-0.96743E+00,-0.96945E+00,-0.97146E+00,-0.97346E+00,-0.97545E+00,
     &-0.97743E+00,-0.97941E+00,-0.98137E+00,-0.98333E+00,-0.98528E+00,
     &-0.98722E+00,-0.98915E+00,-0.99108E+00,-0.99300E+00,-0.99491E+00,
     &-0.99681E+00,-0.99871E+00,-0.10006E+01,-0.10025E+01,-0.10044E+01,
     &-0.10062E+01,-0.10081E+01,-0.10099E+01,-0.10118E+01,-0.10136E+01,
     &-0.10155E+01,-0.10173E+01,-0.10191E+01,-0.10209E+01,-0.10227E+01,
     &-0.10245E+01,-0.10263E+01,-0.10281E+01,-0.10299E+01,-0.10317E+01,
     &-0.10334E+01,-0.10352E+01,-0.10369E+01,-0.10387E+01,-0.10404E+01,
     &-0.10422E+01,-0.10439E+01,-0.10457E+01,-0.10474E+01,-0.10491E+01,
     &-0.10508E+01,-0.10525E+01,-0.10542E+01,-0.10559E+01,-0.10576E+01,
     &-0.10593E+01,-0.10610E+01,-0.10627E+01,-0.10643E+01,-0.10660E+01,
     &-0.10677E+01,-0.10693E+01,-0.10710E+01,-0.10726E+01,-0.10743E+01,
     &-0.10759E+01,-0.10775E+01,-0.10792E+01,-0.10808E+01,-0.10824E+01/

      DATA (BNC02M(I),I=201,300)/
     &-0.10840E+01,-0.10857E+01,-0.10873E+01,-0.10889E+01,-0.10905E+01,
     &-0.10921E+01,-0.10937E+01,-0.10952E+01,-0.10968E+01,-0.10984E+01,
     &-0.11000E+01,-0.11015E+01,-0.11031E+01,-0.11047E+01,-0.11062E+01,
     &-0.11078E+01,-0.11093E+01,-0.11109E+01,-0.11124E+01,-0.11140E+01,
     &-0.11155E+01,-0.11170E+01,-0.11186E+01,-0.11201E+01,-0.11216E+01,
     &-0.11231E+01,-0.11247E+01,-0.11262E+01,-0.11277E+01,-0.11292E+01,
     &-0.11307E+01,-0.11322E+01,-0.11337E+01,-0.11352E+01,-0.11366E+01,
     &-0.11381E+01,-0.11396E+01,-0.11411E+01,-0.11426E+01,-0.11440E+01,
     &-0.11455E+01,-0.11470E+01,-0.11484E+01,-0.11499E+01,-0.11514E+01,
     &-0.11528E+01,-0.11543E+01,-0.11557E+01,-0.11571E+01,-0.11586E+01,
     &-0.11600E+01,-0.11615E+01,-0.11629E+01,-0.11643E+01,-0.11658E+01,
     &-0.11672E+01,-0.11686E+01,-0.11700E+01,-0.11714E+01,-0.11728E+01,
     &-0.11743E+01,-0.11757E+01,-0.11771E+01,-0.11785E+01,-0.11799E+01,
     &-0.11813E+01,-0.11827E+01,-0.11841E+01,-0.11854E+01,-0.11868E+01,
     &-0.11882E+01,-0.11896E+01,-0.11910E+01,-0.11923E+01,-0.11937E+01,
     &-0.11951E+01,-0.11965E+01,-0.11978E+01,-0.11992E+01,-0.12006E+01,
     &-0.12019E+01,-0.12033E+01,-0.12046E+01,-0.12060E+01,-0.12073E+01,
     &-0.12087E+01,-0.12100E+01,-0.12114E+01,-0.12127E+01,-0.12140E+01,
     &-0.12154E+01,-0.12167E+01,-0.12181E+01,-0.12194E+01,-0.12207E+01,
     &-0.12220E+01,-0.12234E+01,-0.12247E+01,-0.12260E+01,-0.12273E+01/

      DATA (BNC02M(I),I=301,400)/
     &-0.12286E+01,-0.12299E+01,-0.12313E+01,-0.12326E+01,-0.12339E+01,
     &-0.12352E+01,-0.12365E+01,-0.12378E+01,-0.12391E+01,-0.12404E+01,
     &-0.12417E+01,-0.12430E+01,-0.12443E+01,-0.12455E+01,-0.12468E+01,
     &-0.12481E+01,-0.12494E+01,-0.12507E+01,-0.12520E+01,-0.12532E+01,
     &-0.12545E+01,-0.12558E+01,-0.12571E+01,-0.12583E+01,-0.12596E+01,
     &-0.12609E+01,-0.12621E+01,-0.12634E+01,-0.12647E+01,-0.12659E+01,
     &-0.12672E+01,-0.12684E+01,-0.12697E+01,-0.12709E+01,-0.12722E+01,
     &-0.12734E+01,-0.12747E+01,-0.12759E+01,-0.12772E+01,-0.12784E+01,
     &-0.12797E+01,-0.12809E+01,-0.12821E+01,-0.12834E+01,-0.12846E+01,
     &-0.12858E+01,-0.12871E+01,-0.12883E+01,-0.12895E+01,-0.12907E+01,
     &-0.12920E+01,-0.12932E+01,-0.12944E+01,-0.12956E+01,-0.12969E+01,
     &-0.12981E+01,-0.12993E+01,-0.13005E+01,-0.13017E+01,-0.13029E+01,
     &-0.13041E+01,-0.13053E+01,-0.13065E+01,-0.13078E+01,-0.13090E+01,
     &-0.13102E+01,-0.13114E+01,-0.13126E+01,-0.13138E+01,-0.13149E+01,
     &-0.13161E+01,-0.13173E+01,-0.13185E+01,-0.13197E+01,-0.13209E+01,
     &-0.13221E+01,-0.13233E+01,-0.13245E+01,-0.13256E+01,-0.13268E+01,
     &-0.13280E+01,-0.13292E+01,-0.13304E+01,-0.13315E+01,-0.13327E+01,
     &-0.13339E+01,-0.13351E+01,-0.13362E+01,-0.13374E+01,-0.13386E+01,
     &-0.13397E+01,-0.13409E+01,-0.13421E+01,-0.13432E+01,-0.13444E+01,
     &-0.13456E+01,-0.13467E+01,-0.13479E+01,-0.13490E+01,-0.13502E+01/

      DATA (BNC02M(I),I=401,500)/
     &-0.13513E+01,-0.13525E+01,-0.13537E+01,-0.13548E+01,-0.13560E+01,
     &-0.13571E+01,-0.13582E+01,-0.13594E+01,-0.13605E+01,-0.13617E+01,
     &-0.13628E+01,-0.13640E+01,-0.13651E+01,-0.13662E+01,-0.13674E+01,
     &-0.13685E+01,-0.13697E+01,-0.13708E+01,-0.13719E+01,-0.13731E+01,
     &-0.13742E+01,-0.13753E+01,-0.13764E+01,-0.13776E+01,-0.13787E+01,
     &-0.13798E+01,-0.13810E+01,-0.13821E+01,-0.13832E+01,-0.13843E+01,
     &-0.13854E+01,-0.13866E+01,-0.13877E+01,-0.13888E+01,-0.13899E+01,
     &-0.13910E+01,-0.13921E+01,-0.13932E+01,-0.13944E+01,-0.13955E+01,
     &-0.13966E+01,-0.13977E+01,-0.13988E+01,-0.13999E+01,-0.14010E+01,
     &-0.14021E+01,-0.14032E+01,-0.14043E+01,-0.14054E+01,-0.14065E+01,
     &-0.14076E+01,-0.14087E+01,-0.14098E+01,-0.14109E+01,-0.14120E+01,
     &-0.14131E+01,-0.14142E+01,-0.14153E+01,-0.14164E+01,-0.14175E+01,
     &-0.14186E+01,-0.14196E+01,-0.14207E+01,-0.14218E+01,-0.14229E+01,
     &-0.14240E+01,-0.14251E+01,-0.14261E+01,-0.14272E+01,-0.14283E+01,
     &-0.14294E+01,-0.14305E+01,-0.14315E+01,-0.14326E+01,-0.14337E+01,
     &-0.14348E+01,-0.14359E+01,-0.14369E+01,-0.14380E+01,-0.14391E+01,
     &-0.14401E+01,-0.14412E+01,-0.14423E+01,-0.14433E+01,-0.14444E+01,
     &-0.14455E+01,-0.14465E+01,-0.14476E+01,-0.14487E+01,-0.14497E+01,
     &-0.14508E+01,-0.14519E+01,-0.14529E+01,-0.14540E+01,-0.14550E+01,
     &-0.14561E+01,-0.14572E+01,-0.14582E+01,-0.14593E+01,-0.14603E+01/

      DATA (BNC02M(I),I=501,600)/
     &-0.14614E+01,-0.14624E+01,-0.14635E+01,-0.14645E+01,-0.14656E+01,
     &-0.14666E+01,-0.14677E+01,-0.14687E+01,-0.14698E+01,-0.14708E+01,
     &-0.14719E+01,-0.14729E+01,-0.14740E+01,-0.14750E+01,-0.14760E+01,
     &-0.14771E+01,-0.14781E+01,-0.14792E+01,-0.14802E+01,-0.14812E+01,
     &-0.14823E+01,-0.14833E+01,-0.14843E+01,-0.14854E+01,-0.14864E+01,
     &-0.14875E+01,-0.14885E+01,-0.14895E+01,-0.14905E+01,-0.14916E+01,
     &-0.14926E+01,-0.14936E+01,-0.14947E+01,-0.14957E+01,-0.14967E+01,
     &-0.14978E+01,-0.14988E+01,-0.14998E+01,-0.15008E+01,-0.15018E+01,
     &-0.15029E+01,-0.15039E+01,-0.15049E+01,-0.15059E+01,-0.15070E+01,
     &-0.15080E+01,-0.15090E+01,-0.15100E+01,-0.15110E+01,-0.15120E+01,
     &-0.15131E+01,-0.15141E+01,-0.15151E+01,-0.15161E+01,-0.15171E+01,
     &-0.15181E+01,-0.15191E+01,-0.15202E+01,-0.15212E+01,-0.15222E+01,
     &-0.15232E+01,-0.15242E+01,-0.15252E+01,-0.15262E+01,-0.15272E+01,
     &-0.15282E+01,-0.15292E+01,-0.15302E+01,-0.15312E+01,-0.15322E+01,
     &-0.15332E+01,-0.15342E+01,-0.15352E+01,-0.15362E+01,-0.15372E+01,
     &-0.15382E+01,-0.15392E+01,-0.15402E+01,-0.15412E+01,-0.15422E+01,
     &-0.15432E+01,-0.15442E+01,-0.15452E+01,-0.15462E+01,-0.15472E+01,
     &-0.15482E+01,-0.15492E+01,-0.15502E+01,-0.15512E+01,-0.15522E+01,
     &-0.15531E+01,-0.15541E+01,-0.15551E+01,-0.15561E+01,-0.15571E+01,
     &-0.15581E+01,-0.15591E+01,-0.15601E+01,-0.15610E+01,-0.15620E+01/

      DATA (BNC02M(I),I=601,700)/
     &-0.15726E+01,-0.15824E+01,-0.15921E+01,-0.16017E+01,-0.16113E+01,
     &-0.16208E+01,-0.16303E+01,-0.16397E+01,-0.16491E+01,-0.16584E+01,
     &-0.16677E+01,-0.16770E+01,-0.16862E+01,-0.16954E+01,-0.17045E+01,
     &-0.17136E+01,-0.17226E+01,-0.17316E+01,-0.17406E+01,-0.17495E+01,
     &-0.17584E+01,-0.17673E+01,-0.17761E+01,-0.17849E+01,-0.17937E+01,
     &-0.18024E+01,-0.18111E+01,-0.18198E+01,-0.18284E+01,-0.18370E+01,
     &-0.18456E+01,-0.18542E+01,-0.18627E+01,-0.18712E+01,-0.18797E+01,
     &-0.18881E+01,-0.18966E+01,-0.19050E+01,-0.19133E+01,-0.19217E+01,
     &-0.19300E+01,-0.19383E+01,-0.19466E+01,-0.19548E+01,-0.19631E+01,
     &-0.19713E+01,-0.19795E+01,-0.19876E+01,-0.19958E+01,-0.20039E+01,
     &-0.20120E+01,-0.20201E+01,-0.20282E+01,-0.20362E+01,-0.20443E+01,
     &-0.20523E+01,-0.20603E+01,-0.20682E+01,-0.20762E+01,-0.20841E+01,
     &-0.20921E+01,-0.21000E+01,-0.21079E+01,-0.21157E+01,-0.21236E+01,
     &-0.21314E+01,-0.21393E+01,-0.21471E+01,-0.21549E+01,-0.21626E+01,
     &-0.21704E+01,-0.21782E+01,-0.21859E+01,-0.21936E+01,-0.22013E+01,
     &-0.22090E+01,-0.22167E+01,-0.22244E+01,-0.22320E+01,-0.22396E+01,
     &-0.22473E+01,-0.22549E+01,-0.22625E+01,-0.22701E+01,-0.22776E+01,
     &-0.22852E+01,-0.22928E+01,-0.23003E+01,-0.23078E+01,-0.23153E+01,
     &-0.23228E+01,-0.23303E+01,-0.23378E+01,-0.23453E+01,-0.23527E+01,
     &-0.23602E+01,-0.23676E+01,-0.23750E+01,-0.23825E+01,-0.23899E+01/

      DATA (BNC02M(I),I=701,741)/
     &-0.23973E+01,-0.24046E+01,-0.24120E+01,-0.24194E+01,-0.24267E+01,
     &-0.24341E+01,-0.24414E+01,-0.24487E+01,-0.24560E+01,-0.24633E+01,
     &-0.24706E+01,-0.24779E+01,-0.24852E+01,-0.24925E+01,-0.24997E+01,
     &-0.25070E+01,-0.25142E+01,-0.25214E+01,-0.25287E+01,-0.25359E+01,
     &-0.25431E+01,-0.25503E+01,-0.25575E+01,-0.25647E+01,-0.25718E+01,
     &-0.25790E+01,-0.25862E+01,-0.25933E+01,-0.26004E+01,-0.26076E+01,
     &-0.26147E+01,-0.26218E+01,-0.26289E+01,-0.26360E+01,-0.26431E+01,
     &-0.26502E+01,-0.26573E+01,-0.26644E+01,-0.26714E+01,-0.26785E+01,
     &-0.26856E+01
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.49134E-01,-0.10768E+00,-0.13739E+00,-0.15864E+00,-0.17549E+00,
     &-0.18957E+00,-0.20174E+00,-0.21250E+00,-0.22216E+00,-0.23096E+00,
     &-0.23904E+00,-0.24653E+00,-0.25352E+00,-0.26008E+00,-0.26626E+00,
     &-0.27211E+00,-0.27767E+00,-0.28298E+00,-0.28804E+00,-0.29290E+00,
     &-0.29757E+00,-0.30206E+00,-0.30639E+00,-0.31057E+00,-0.31462E+00,
     &-0.31854E+00,-0.32234E+00,-0.32603E+00,-0.32962E+00,-0.33312E+00,
     &-0.33652E+00,-0.33984E+00,-0.34308E+00,-0.34624E+00,-0.34933E+00,
     &-0.35235E+00,-0.35530E+00,-0.35820E+00,-0.36103E+00,-0.36381E+00,
     &-0.36653E+00,-0.36921E+00,-0.37183E+00,-0.37441E+00,-0.37694E+00,
     &-0.37942E+00,-0.38187E+00,-0.38427E+00,-0.38664E+00,-0.38897E+00,
     &-0.39126E+00,-0.39352E+00,-0.39574E+00,-0.39793E+00,-0.40009E+00,
     &-0.40222E+00,-0.40432E+00,-0.40640E+00,-0.40844E+00,-0.41047E+00,
     &-0.41246E+00,-0.41443E+00,-0.41638E+00,-0.41831E+00,-0.42021E+00,
     &-0.42209E+00,-0.42396E+00,-0.42580E+00,-0.42763E+00,-0.42943E+00,
     &-0.43122E+00,-0.43300E+00,-0.43475E+00,-0.43649E+00,-0.43822E+00,
     &-0.43993E+00,-0.44163E+00,-0.44331E+00,-0.44498E+00,-0.44664E+00,
     &-0.44829E+00,-0.44992E+00,-0.45154E+00,-0.45315E+00,-0.45476E+00,
     &-0.45635E+00,-0.45793E+00,-0.45950E+00,-0.46106E+00,-0.46261E+00,
     &-0.46416E+00,-0.46569E+00,-0.46722E+00,-0.46874E+00,-0.47025E+00,
     &-0.47175E+00,-0.47324E+00,-0.47473E+00,-0.47621E+00,-0.47768E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.47915E+00,-0.48060E+00,-0.48205E+00,-0.48350E+00,-0.48493E+00,
     &-0.48636E+00,-0.48779E+00,-0.48920E+00,-0.49061E+00,-0.49202E+00,
     &-0.49342E+00,-0.49481E+00,-0.49619E+00,-0.49757E+00,-0.49894E+00,
     &-0.50031E+00,-0.50166E+00,-0.50302E+00,-0.50436E+00,-0.50571E+00,
     &-0.50704E+00,-0.50837E+00,-0.50969E+00,-0.51101E+00,-0.51232E+00,
     &-0.51363E+00,-0.51493E+00,-0.51622E+00,-0.51751E+00,-0.51879E+00,
     &-0.52007E+00,-0.52134E+00,-0.52261E+00,-0.52387E+00,-0.52513E+00,
     &-0.52638E+00,-0.52762E+00,-0.52886E+00,-0.53010E+00,-0.53133E+00,
     &-0.53255E+00,-0.53377E+00,-0.53499E+00,-0.53620E+00,-0.53740E+00,
     &-0.53860E+00,-0.53980E+00,-0.54099E+00,-0.54218E+00,-0.54336E+00,
     &-0.54454E+00,-0.54571E+00,-0.54688E+00,-0.54804E+00,-0.54920E+00,
     &-0.55036E+00,-0.55151E+00,-0.55266E+00,-0.55380E+00,-0.55494E+00,
     &-0.55608E+00,-0.55721E+00,-0.55833E+00,-0.55946E+00,-0.56058E+00,
     &-0.56169E+00,-0.56280E+00,-0.56391E+00,-0.56502E+00,-0.56612E+00,
     &-0.56721E+00,-0.56831E+00,-0.56940E+00,-0.57048E+00,-0.57157E+00,
     &-0.57265E+00,-0.57372E+00,-0.57479E+00,-0.57586E+00,-0.57693E+00,
     &-0.57799E+00,-0.57905E+00,-0.58011E+00,-0.58116E+00,-0.58221E+00,
     &-0.58325E+00,-0.58430E+00,-0.58534E+00,-0.58637E+00,-0.58741E+00,
     &-0.58844E+00,-0.58947E+00,-0.59049E+00,-0.59151E+00,-0.59253E+00,
     &-0.59355E+00,-0.59456E+00,-0.59557E+00,-0.59658E+00,-0.59758E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.59859E+00,-0.59959E+00,-0.60058E+00,-0.60158E+00,-0.60257E+00,
     &-0.60356E+00,-0.60454E+00,-0.60553E+00,-0.60651E+00,-0.60748E+00,
     &-0.60846E+00,-0.60943E+00,-0.61040E+00,-0.61137E+00,-0.61234E+00,
     &-0.61330E+00,-0.61426E+00,-0.61522E+00,-0.61617E+00,-0.61713E+00,
     &-0.61808E+00,-0.61903E+00,-0.61997E+00,-0.62092E+00,-0.62186E+00,
     &-0.62280E+00,-0.62374E+00,-0.62467E+00,-0.62561E+00,-0.62654E+00,
     &-0.62746E+00,-0.62839E+00,-0.62932E+00,-0.63024E+00,-0.63116E+00,
     &-0.63208E+00,-0.63299E+00,-0.63390E+00,-0.63482E+00,-0.63573E+00,
     &-0.63663E+00,-0.63754E+00,-0.63844E+00,-0.63934E+00,-0.64024E+00,
     &-0.64114E+00,-0.64204E+00,-0.64293E+00,-0.64382E+00,-0.64471E+00,
     &-0.64560E+00,-0.64649E+00,-0.64737E+00,-0.64825E+00,-0.64913E+00,
     &-0.65001E+00,-0.65089E+00,-0.65176E+00,-0.65264E+00,-0.65351E+00,
     &-0.65438E+00,-0.65524E+00,-0.65611E+00,-0.65697E+00,-0.65784E+00,
     &-0.65870E+00,-0.65956E+00,-0.66041E+00,-0.66127E+00,-0.66212E+00,
     &-0.66298E+00,-0.66383E+00,-0.66468E+00,-0.66552E+00,-0.66637E+00,
     &-0.66721E+00,-0.66806E+00,-0.66890E+00,-0.66974E+00,-0.67057E+00,
     &-0.67141E+00,-0.67224E+00,-0.67308E+00,-0.67391E+00,-0.67474E+00,
     &-0.67557E+00,-0.67639E+00,-0.67722E+00,-0.67804E+00,-0.67887E+00,
     &-0.67969E+00,-0.68051E+00,-0.68133E+00,-0.68214E+00,-0.68296E+00,
     &-0.68377E+00,-0.68458E+00,-0.68539E+00,-0.68620E+00,-0.68701E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.68782E+00,-0.68862E+00,-0.68943E+00,-0.69023E+00,-0.69103E+00,
     &-0.69183E+00,-0.69263E+00,-0.69343E+00,-0.69422E+00,-0.69502E+00,
     &-0.69581E+00,-0.69660E+00,-0.69739E+00,-0.69818E+00,-0.69897E+00,
     &-0.69976E+00,-0.70054E+00,-0.70132E+00,-0.70211E+00,-0.70289E+00,
     &-0.70367E+00,-0.70445E+00,-0.70523E+00,-0.70600E+00,-0.70678E+00,
     &-0.70755E+00,-0.70832E+00,-0.70910E+00,-0.70987E+00,-0.71064E+00,
     &-0.71140E+00,-0.71217E+00,-0.71294E+00,-0.71370E+00,-0.71446E+00,
     &-0.71523E+00,-0.71599E+00,-0.71675E+00,-0.71751E+00,-0.71826E+00,
     &-0.71902E+00,-0.71977E+00,-0.72053E+00,-0.72128E+00,-0.72203E+00,
     &-0.72279E+00,-0.72353E+00,-0.72428E+00,-0.72503E+00,-0.72578E+00,
     &-0.72652E+00,-0.72727E+00,-0.72801E+00,-0.72875E+00,-0.72949E+00,
     &-0.73023E+00,-0.73097E+00,-0.73171E+00,-0.73245E+00,-0.73318E+00,
     &-0.73392E+00,-0.73465E+00,-0.73539E+00,-0.73612E+00,-0.73685E+00,
     &-0.73758E+00,-0.73831E+00,-0.73904E+00,-0.73976E+00,-0.74049E+00,
     &-0.74121E+00,-0.74194E+00,-0.74266E+00,-0.74338E+00,-0.74410E+00,
     &-0.74482E+00,-0.74554E+00,-0.74626E+00,-0.74698E+00,-0.74769E+00,
     &-0.74841E+00,-0.74912E+00,-0.74984E+00,-0.75055E+00,-0.75126E+00,
     &-0.75197E+00,-0.75268E+00,-0.75339E+00,-0.75410E+00,-0.75481E+00,
     &-0.75551E+00,-0.75622E+00,-0.75692E+00,-0.75763E+00,-0.75833E+00,
     &-0.75903E+00,-0.75973E+00,-0.76043E+00,-0.76113E+00,-0.76183E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.76253E+00,-0.76322E+00,-0.76392E+00,-0.76461E+00,-0.76531E+00,
     &-0.76600E+00,-0.76669E+00,-0.76738E+00,-0.76808E+00,-0.76877E+00,
     &-0.76945E+00,-0.77014E+00,-0.77083E+00,-0.77152E+00,-0.77220E+00,
     &-0.77289E+00,-0.77357E+00,-0.77425E+00,-0.77494E+00,-0.77562E+00,
     &-0.77630E+00,-0.77698E+00,-0.77766E+00,-0.77834E+00,-0.77902E+00,
     &-0.77969E+00,-0.78037E+00,-0.78105E+00,-0.78172E+00,-0.78239E+00,
     &-0.78307E+00,-0.78374E+00,-0.78441E+00,-0.78508E+00,-0.78575E+00,
     &-0.78642E+00,-0.78709E+00,-0.78776E+00,-0.78843E+00,-0.78909E+00,
     &-0.78976E+00,-0.79042E+00,-0.79109E+00,-0.79175E+00,-0.79242E+00,
     &-0.79308E+00,-0.79374E+00,-0.79440E+00,-0.79506E+00,-0.79572E+00,
     &-0.79638E+00,-0.79704E+00,-0.79769E+00,-0.79835E+00,-0.79901E+00,
     &-0.79966E+00,-0.80032E+00,-0.80097E+00,-0.80162E+00,-0.80228E+00,
     &-0.80293E+00,-0.80358E+00,-0.80423E+00,-0.80488E+00,-0.80553E+00,
     &-0.80618E+00,-0.80682E+00,-0.80747E+00,-0.80812E+00,-0.80876E+00,
     &-0.80941E+00,-0.81005E+00,-0.81070E+00,-0.81134E+00,-0.81198E+00,
     &-0.81263E+00,-0.81327E+00,-0.81391E+00,-0.81455E+00,-0.81519E+00,
     &-0.81583E+00,-0.81646E+00,-0.81710E+00,-0.81774E+00,-0.81837E+00,
     &-0.81901E+00,-0.81965E+00,-0.82028E+00,-0.82091E+00,-0.82155E+00,
     &-0.82218E+00,-0.82281E+00,-0.82344E+00,-0.82407E+00,-0.82470E+00,
     &-0.82533E+00,-0.82596E+00,-0.82659E+00,-0.82722E+00,-0.82785E+00/

      DATA (BNC03M(I),I=501,600)/
     &-0.82847E+00,-0.82910E+00,-0.82973E+00,-0.83035E+00,-0.83097E+00,
     &-0.83160E+00,-0.83222E+00,-0.83284E+00,-0.83347E+00,-0.83409E+00,
     &-0.83471E+00,-0.83533E+00,-0.83595E+00,-0.83657E+00,-0.83719E+00,
     &-0.83781E+00,-0.83842E+00,-0.83904E+00,-0.83966E+00,-0.84027E+00,
     &-0.84089E+00,-0.84150E+00,-0.84212E+00,-0.84273E+00,-0.84334E+00,
     &-0.84396E+00,-0.84457E+00,-0.84518E+00,-0.84579E+00,-0.84640E+00,
     &-0.84701E+00,-0.84762E+00,-0.84823E+00,-0.84884E+00,-0.84945E+00,
     &-0.85005E+00,-0.85066E+00,-0.85127E+00,-0.85187E+00,-0.85248E+00,
     &-0.85308E+00,-0.85369E+00,-0.85429E+00,-0.85489E+00,-0.85550E+00,
     &-0.85610E+00,-0.85670E+00,-0.85730E+00,-0.85790E+00,-0.85850E+00,
     &-0.85910E+00,-0.85970E+00,-0.86030E+00,-0.86090E+00,-0.86150E+00,
     &-0.86210E+00,-0.86269E+00,-0.86329E+00,-0.86388E+00,-0.86448E+00,
     &-0.86507E+00,-0.86567E+00,-0.86626E+00,-0.86686E+00,-0.86745E+00,
     &-0.86804E+00,-0.86863E+00,-0.86923E+00,-0.86982E+00,-0.87041E+00,
     &-0.87100E+00,-0.87159E+00,-0.87218E+00,-0.87277E+00,-0.87335E+00,
     &-0.87394E+00,-0.87453E+00,-0.87512E+00,-0.87570E+00,-0.87629E+00,
     &-0.87687E+00,-0.87746E+00,-0.87804E+00,-0.87863E+00,-0.87921E+00,
     &-0.87980E+00,-0.88038E+00,-0.88096E+00,-0.88154E+00,-0.88212E+00,
     &-0.88271E+00,-0.88329E+00,-0.88387E+00,-0.88445E+00,-0.88503E+00,
     &-0.88560E+00,-0.88618E+00,-0.88676E+00,-0.88734E+00,-0.88792E+00/

      DATA (BNC03M(I),I=601,700)/
     &-0.89413E+00,-0.89983E+00,-0.90549E+00,-0.91112E+00,-0.91671E+00,
     &-0.92226E+00,-0.92777E+00,-0.93325E+00,-0.93869E+00,-0.94410E+00,
     &-0.94948E+00,-0.95483E+00,-0.96015E+00,-0.96543E+00,-0.97069E+00,
     &-0.97592E+00,-0.98112E+00,-0.98629E+00,-0.99143E+00,-0.99655E+00,
     &-0.10016E+01,-0.10067E+01,-0.10118E+01,-0.10168E+01,-0.10218E+01,
     &-0.10267E+01,-0.10317E+01,-0.10366E+01,-0.10415E+01,-0.10464E+01,
     &-0.10513E+01,-0.10561E+01,-0.10609E+01,-0.10658E+01,-0.10705E+01,
     &-0.10753E+01,-0.10801E+01,-0.10848E+01,-0.10895E+01,-0.10942E+01,
     &-0.10989E+01,-0.11035E+01,-0.11082E+01,-0.11128E+01,-0.11174E+01,
     &-0.11220E+01,-0.11266E+01,-0.11311E+01,-0.11357E+01,-0.11402E+01,
     &-0.11447E+01,-0.11493E+01,-0.11537E+01,-0.11582E+01,-0.11627E+01,
     &-0.11671E+01,-0.11716E+01,-0.11760E+01,-0.11804E+01,-0.11848E+01,
     &-0.11892E+01,-0.11936E+01,-0.11979E+01,-0.12023E+01,-0.12066E+01,
     &-0.12109E+01,-0.12153E+01,-0.12196E+01,-0.12239E+01,-0.12281E+01,
     &-0.12324E+01,-0.12367E+01,-0.12409E+01,-0.12452E+01,-0.12494E+01,
     &-0.12536E+01,-0.12578E+01,-0.12620E+01,-0.12662E+01,-0.12704E+01,
     &-0.12746E+01,-0.12787E+01,-0.12829E+01,-0.12870E+01,-0.12912E+01,
     &-0.12953E+01,-0.12994E+01,-0.13035E+01,-0.13076E+01,-0.13117E+01,
     &-0.13158E+01,-0.13199E+01,-0.13239E+01,-0.13280E+01,-0.13320E+01,
     &-0.13361E+01,-0.13401E+01,-0.13441E+01,-0.13482E+01,-0.13522E+01/

      DATA (BNC03M(I),I=701,741)/
     &-0.13562E+01,-0.13602E+01,-0.13642E+01,-0.13682E+01,-0.13721E+01,
     &-0.13761E+01,-0.13801E+01,-0.13840E+01,-0.13880E+01,-0.13919E+01,
     &-0.13959E+01,-0.13998E+01,-0.14037E+01,-0.14076E+01,-0.14115E+01,
     &-0.14154E+01,-0.14193E+01,-0.14232E+01,-0.14271E+01,-0.14310E+01,
     &-0.14349E+01,-0.14387E+01,-0.14426E+01,-0.14464E+01,-0.14503E+01,
     &-0.14541E+01,-0.14580E+01,-0.14618E+01,-0.14656E+01,-0.14695E+01,
     &-0.14733E+01,-0.14771E+01,-0.14809E+01,-0.14847E+01,-0.14885E+01,
     &-0.14923E+01,-0.14961E+01,-0.14998E+01,-0.15036E+01,-0.15074E+01,
     &-0.15112E+01
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.98067E-01,-0.21426E+00,-0.27286E+00,-0.31457E+00,-0.34750E+00,
     &-0.37493E+00,-0.39855E+00,-0.41935E+00,-0.43799E+00,-0.45490E+00,
     &-0.47039E+00,-0.48471E+00,-0.49804E+00,-0.51051E+00,-0.52223E+00,
     &-0.53331E+00,-0.54381E+00,-0.55380E+00,-0.56333E+00,-0.57244E+00,
     &-0.58118E+00,-0.58957E+00,-0.59765E+00,-0.60544E+00,-0.61296E+00,
     &-0.62024E+00,-0.62729E+00,-0.63412E+00,-0.64075E+00,-0.64720E+00,
     &-0.65347E+00,-0.65958E+00,-0.66553E+00,-0.67134E+00,-0.67700E+00,
     &-0.68254E+00,-0.68795E+00,-0.69324E+00,-0.69842E+00,-0.70349E+00,
     &-0.70846E+00,-0.71333E+00,-0.71811E+00,-0.72280E+00,-0.72740E+00,
     &-0.73192E+00,-0.73636E+00,-0.74072E+00,-0.74501E+00,-0.74923E+00,
     &-0.75339E+00,-0.75748E+00,-0.76150E+00,-0.76547E+00,-0.76938E+00,
     &-0.77323E+00,-0.77702E+00,-0.78077E+00,-0.78446E+00,-0.78811E+00,
     &-0.79171E+00,-0.79527E+00,-0.79878E+00,-0.80225E+00,-0.80567E+00,
     &-0.80906E+00,-0.81241E+00,-0.81573E+00,-0.81900E+00,-0.82225E+00,
     &-0.82546E+00,-0.82864E+00,-0.83179E+00,-0.83491E+00,-0.83799E+00,
     &-0.84106E+00,-0.84409E+00,-0.84710E+00,-0.85008E+00,-0.85304E+00,
     &-0.85597E+00,-0.85889E+00,-0.86178E+00,-0.86464E+00,-0.86749E+00,
     &-0.87032E+00,-0.87312E+00,-0.87591E+00,-0.87868E+00,-0.88143E+00,
     &-0.88416E+00,-0.88687E+00,-0.88957E+00,-0.89225E+00,-0.89492E+00,
     &-0.89757E+00,-0.90020E+00,-0.90282E+00,-0.90542E+00,-0.90801E+00/

      DATA (BNC04M(I),I=101,200)/
     &-0.91058E+00,-0.91314E+00,-0.91569E+00,-0.91822E+00,-0.92074E+00,
     &-0.92325E+00,-0.92574E+00,-0.92822E+00,-0.93068E+00,-0.93314E+00,
     &-0.93558E+00,-0.93801E+00,-0.94043E+00,-0.94283E+00,-0.94522E+00,
     &-0.94760E+00,-0.94997E+00,-0.95233E+00,-0.95468E+00,-0.95701E+00,
     &-0.95934E+00,-0.96165E+00,-0.96395E+00,-0.96624E+00,-0.96852E+00,
     &-0.97079E+00,-0.97305E+00,-0.97530E+00,-0.97754E+00,-0.97977E+00,
     &-0.98199E+00,-0.98419E+00,-0.98639E+00,-0.98858E+00,-0.99076E+00,
     &-0.99293E+00,-0.99509E+00,-0.99724E+00,-0.99938E+00,-0.10015E+01,
     &-0.10036E+01,-0.10058E+01,-0.10079E+01,-0.10100E+01,-0.10120E+01,
     &-0.10141E+01,-0.10162E+01,-0.10183E+01,-0.10203E+01,-0.10224E+01,
     &-0.10244E+01,-0.10264E+01,-0.10285E+01,-0.10305E+01,-0.10325E+01,
     &-0.10345E+01,-0.10365E+01,-0.10385E+01,-0.10405E+01,-0.10424E+01,
     &-0.10444E+01,-0.10463E+01,-0.10483E+01,-0.10502E+01,-0.10522E+01,
     &-0.10541E+01,-0.10560E+01,-0.10580E+01,-0.10599E+01,-0.10618E+01,
     &-0.10637E+01,-0.10656E+01,-0.10674E+01,-0.10693E+01,-0.10712E+01,
     &-0.10731E+01,-0.10749E+01,-0.10768E+01,-0.10786E+01,-0.10805E+01,
     &-0.10823E+01,-0.10841E+01,-0.10860E+01,-0.10878E+01,-0.10896E+01,
     &-0.10914E+01,-0.10932E+01,-0.10950E+01,-0.10968E+01,-0.10986E+01,
     &-0.11004E+01,-0.11022E+01,-0.11039E+01,-0.11057E+01,-0.11075E+01,
     &-0.11092E+01,-0.11110E+01,-0.11127E+01,-0.11145E+01,-0.11162E+01/

      DATA (BNC04M(I),I=201,300)/
     &-0.11180E+01,-0.11197E+01,-0.11214E+01,-0.11231E+01,-0.11248E+01,
     &-0.11266E+01,-0.11283E+01,-0.11300E+01,-0.11317E+01,-0.11334E+01,
     &-0.11350E+01,-0.11367E+01,-0.11384E+01,-0.11401E+01,-0.11418E+01,
     &-0.11434E+01,-0.11451E+01,-0.11468E+01,-0.11484E+01,-0.11501E+01,
     &-0.11517E+01,-0.11534E+01,-0.11550E+01,-0.11566E+01,-0.11583E+01,
     &-0.11599E+01,-0.11615E+01,-0.11631E+01,-0.11648E+01,-0.11664E+01,
     &-0.11680E+01,-0.11696E+01,-0.11712E+01,-0.11728E+01,-0.11744E+01,
     &-0.11760E+01,-0.11776E+01,-0.11791E+01,-0.11807E+01,-0.11823E+01,
     &-0.11839E+01,-0.11854E+01,-0.11870E+01,-0.11886E+01,-0.11901E+01,
     &-0.11917E+01,-0.11933E+01,-0.11948E+01,-0.11964E+01,-0.11979E+01,
     &-0.11994E+01,-0.12010E+01,-0.12025E+01,-0.12040E+01,-0.12056E+01,
     &-0.12071E+01,-0.12086E+01,-0.12101E+01,-0.12117E+01,-0.12132E+01,
     &-0.12147E+01,-0.12162E+01,-0.12177E+01,-0.12192E+01,-0.12207E+01,
     &-0.12222E+01,-0.12237E+01,-0.12252E+01,-0.12267E+01,-0.12281E+01,
     &-0.12296E+01,-0.12311E+01,-0.12326E+01,-0.12341E+01,-0.12355E+01,
     &-0.12370E+01,-0.12385E+01,-0.12399E+01,-0.12414E+01,-0.12428E+01,
     &-0.12443E+01,-0.12458E+01,-0.12472E+01,-0.12487E+01,-0.12501E+01,
     &-0.12515E+01,-0.12530E+01,-0.12544E+01,-0.12559E+01,-0.12573E+01,
     &-0.12587E+01,-0.12601E+01,-0.12616E+01,-0.12630E+01,-0.12644E+01,
     &-0.12658E+01,-0.12672E+01,-0.12687E+01,-0.12701E+01,-0.12715E+01/

      DATA (BNC04M(I),I=301,400)/
     &-0.12729E+01,-0.12743E+01,-0.12757E+01,-0.12771E+01,-0.12785E+01,
     &-0.12799E+01,-0.12813E+01,-0.12827E+01,-0.12841E+01,-0.12854E+01,
     &-0.12868E+01,-0.12882E+01,-0.12896E+01,-0.12910E+01,-0.12923E+01,
     &-0.12937E+01,-0.12951E+01,-0.12965E+01,-0.12978E+01,-0.12992E+01,
     &-0.13006E+01,-0.13019E+01,-0.13033E+01,-0.13046E+01,-0.13060E+01,
     &-0.13073E+01,-0.13087E+01,-0.13100E+01,-0.13114E+01,-0.13127E+01,
     &-0.13141E+01,-0.13154E+01,-0.13168E+01,-0.13181E+01,-0.13194E+01,
     &-0.13208E+01,-0.13221E+01,-0.13234E+01,-0.13248E+01,-0.13261E+01,
     &-0.13274E+01,-0.13287E+01,-0.13301E+01,-0.13314E+01,-0.13327E+01,
     &-0.13340E+01,-0.13353E+01,-0.13366E+01,-0.13379E+01,-0.13393E+01,
     &-0.13406E+01,-0.13419E+01,-0.13432E+01,-0.13445E+01,-0.13458E+01,
     &-0.13471E+01,-0.13484E+01,-0.13497E+01,-0.13510E+01,-0.13522E+01,
     &-0.13535E+01,-0.13548E+01,-0.13561E+01,-0.13574E+01,-0.13587E+01,
     &-0.13600E+01,-0.13612E+01,-0.13625E+01,-0.13638E+01,-0.13651E+01,
     &-0.13663E+01,-0.13676E+01,-0.13689E+01,-0.13701E+01,-0.13714E+01,
     &-0.13727E+01,-0.13739E+01,-0.13752E+01,-0.13765E+01,-0.13777E+01,
     &-0.13790E+01,-0.13802E+01,-0.13815E+01,-0.13828E+01,-0.13840E+01,
     &-0.13853E+01,-0.13865E+01,-0.13878E+01,-0.13890E+01,-0.13902E+01,
     &-0.13915E+01,-0.13927E+01,-0.13940E+01,-0.13952E+01,-0.13964E+01,
     &-0.13977E+01,-0.13989E+01,-0.14001E+01,-0.14014E+01,-0.14026E+01/

      DATA (BNC04M(I),I=401,500)/
     &-0.14038E+01,-0.14051E+01,-0.14063E+01,-0.14075E+01,-0.14087E+01,
     &-0.14100E+01,-0.14112E+01,-0.14124E+01,-0.14136E+01,-0.14148E+01,
     &-0.14161E+01,-0.14173E+01,-0.14185E+01,-0.14197E+01,-0.14209E+01,
     &-0.14221E+01,-0.14233E+01,-0.14245E+01,-0.14257E+01,-0.14269E+01,
     &-0.14281E+01,-0.14293E+01,-0.14305E+01,-0.14317E+01,-0.14329E+01,
     &-0.14341E+01,-0.14353E+01,-0.14365E+01,-0.14377E+01,-0.14389E+01,
     &-0.14401E+01,-0.14413E+01,-0.14425E+01,-0.14437E+01,-0.14448E+01,
     &-0.14460E+01,-0.14472E+01,-0.14484E+01,-0.14496E+01,-0.14508E+01,
     &-0.14519E+01,-0.14531E+01,-0.14543E+01,-0.14555E+01,-0.14566E+01,
     &-0.14578E+01,-0.14590E+01,-0.14602E+01,-0.14613E+01,-0.14625E+01,
     &-0.14637E+01,-0.14648E+01,-0.14660E+01,-0.14672E+01,-0.14683E+01,
     &-0.14695E+01,-0.14706E+01,-0.14718E+01,-0.14730E+01,-0.14741E+01,
     &-0.14753E+01,-0.14764E+01,-0.14776E+01,-0.14787E+01,-0.14799E+01,
     &-0.14810E+01,-0.14822E+01,-0.14833E+01,-0.14845E+01,-0.14856E+01,
     &-0.14868E+01,-0.14879E+01,-0.14891E+01,-0.14902E+01,-0.14913E+01,
     &-0.14925E+01,-0.14936E+01,-0.14948E+01,-0.14959E+01,-0.14970E+01,
     &-0.14982E+01,-0.14993E+01,-0.15004E+01,-0.15016E+01,-0.15027E+01,
     &-0.15038E+01,-0.15050E+01,-0.15061E+01,-0.15072E+01,-0.15083E+01,
     &-0.15095E+01,-0.15106E+01,-0.15117E+01,-0.15128E+01,-0.15140E+01,
     &-0.15151E+01,-0.15162E+01,-0.15173E+01,-0.15184E+01,-0.15196E+01/

      DATA (BNC04M(I),I=501,600)/
     &-0.15207E+01,-0.15218E+01,-0.15229E+01,-0.15240E+01,-0.15251E+01,
     &-0.15262E+01,-0.15273E+01,-0.15285E+01,-0.15296E+01,-0.15307E+01,
     &-0.15318E+01,-0.15329E+01,-0.15340E+01,-0.15351E+01,-0.15362E+01,
     &-0.15373E+01,-0.15384E+01,-0.15395E+01,-0.15406E+01,-0.15417E+01,
     &-0.15428E+01,-0.15439E+01,-0.15450E+01,-0.15461E+01,-0.15472E+01,
     &-0.15483E+01,-0.15494E+01,-0.15505E+01,-0.15516E+01,-0.15526E+01,
     &-0.15537E+01,-0.15548E+01,-0.15559E+01,-0.15570E+01,-0.15581E+01,
     &-0.15592E+01,-0.15603E+01,-0.15613E+01,-0.15624E+01,-0.15635E+01,
     &-0.15646E+01,-0.15657E+01,-0.15667E+01,-0.15678E+01,-0.15689E+01,
     &-0.15700E+01,-0.15711E+01,-0.15721E+01,-0.15732E+01,-0.15743E+01,
     &-0.15754E+01,-0.15764E+01,-0.15775E+01,-0.15786E+01,-0.15796E+01,
     &-0.15807E+01,-0.15818E+01,-0.15828E+01,-0.15839E+01,-0.15850E+01,
     &-0.15860E+01,-0.15871E+01,-0.15882E+01,-0.15892E+01,-0.15903E+01,
     &-0.15914E+01,-0.15924E+01,-0.15935E+01,-0.15945E+01,-0.15956E+01,
     &-0.15967E+01,-0.15977E+01,-0.15988E+01,-0.15998E+01,-0.16009E+01,
     &-0.16019E+01,-0.16030E+01,-0.16041E+01,-0.16051E+01,-0.16062E+01,
     &-0.16072E+01,-0.16083E+01,-0.16093E+01,-0.16104E+01,-0.16114E+01,
     &-0.16124E+01,-0.16135E+01,-0.16145E+01,-0.16156E+01,-0.16166E+01,
     &-0.16177E+01,-0.16187E+01,-0.16198E+01,-0.16208E+01,-0.16218E+01,
     &-0.16229E+01,-0.16239E+01,-0.16250E+01,-0.16260E+01,-0.16270E+01/

      DATA (BNC04M(I),I=601,700)/
     &-0.16382E+01,-0.16485E+01,-0.16587E+01,-0.16688E+01,-0.16789E+01,
     &-0.16890E+01,-0.16989E+01,-0.17088E+01,-0.17187E+01,-0.17285E+01,
     &-0.17383E+01,-0.17480E+01,-0.17576E+01,-0.17673E+01,-0.17768E+01,
     &-0.17863E+01,-0.17958E+01,-0.18053E+01,-0.18146E+01,-0.18240E+01,
     &-0.18333E+01,-0.18426E+01,-0.18518E+01,-0.18610E+01,-0.18702E+01,
     &-0.18793E+01,-0.18884E+01,-0.18974E+01,-0.19065E+01,-0.19154E+01,
     &-0.19244E+01,-0.19333E+01,-0.19422E+01,-0.19511E+01,-0.19599E+01,
     &-0.19687E+01,-0.19775E+01,-0.19862E+01,-0.19949E+01,-0.20036E+01,
     &-0.20123E+01,-0.20209E+01,-0.20295E+01,-0.20381E+01,-0.20466E+01,
     &-0.20552E+01,-0.20637E+01,-0.20722E+01,-0.20806E+01,-0.20890E+01,
     &-0.20975E+01,-0.21058E+01,-0.21142E+01,-0.21226E+01,-0.21309E+01,
     &-0.21392E+01,-0.21475E+01,-0.21557E+01,-0.21640E+01,-0.21722E+01,
     &-0.21804E+01,-0.21886E+01,-0.21968E+01,-0.22049E+01,-0.22130E+01,
     &-0.22211E+01,-0.22292E+01,-0.22373E+01,-0.22454E+01,-0.22534E+01,
     &-0.22614E+01,-0.22694E+01,-0.22774E+01,-0.22854E+01,-0.22933E+01,
     &-0.23013E+01,-0.23092E+01,-0.23171E+01,-0.23250E+01,-0.23329E+01,
     &-0.23408E+01,-0.23486E+01,-0.23564E+01,-0.23643E+01,-0.23721E+01,
     &-0.23799E+01,-0.23876E+01,-0.23954E+01,-0.24032E+01,-0.24109E+01,
     &-0.24186E+01,-0.24263E+01,-0.24340E+01,-0.24417E+01,-0.24494E+01,
     &-0.24570E+01,-0.24647E+01,-0.24723E+01,-0.24800E+01,-0.24876E+01/

      DATA (BNC04M(I),I=701,741)/
     &-0.24952E+01,-0.25028E+01,-0.25103E+01,-0.25179E+01,-0.25255E+01,
     &-0.25330E+01,-0.25405E+01,-0.25481E+01,-0.25556E+01,-0.25631E+01,
     &-0.25706E+01,-0.25780E+01,-0.25855E+01,-0.25930E+01,-0.26004E+01,
     &-0.26079E+01,-0.26153E+01,-0.26227E+01,-0.26301E+01,-0.26375E+01,
     &-0.26449E+01,-0.26523E+01,-0.26597E+01,-0.26670E+01,-0.26744E+01,
     &-0.26817E+01,-0.26891E+01,-0.26964E+01,-0.27037E+01,-0.27110E+01,
     &-0.27183E+01,-0.27256E+01,-0.27329E+01,-0.27401E+01,-0.27474E+01,
     &-0.27547E+01,-0.27619E+01,-0.27692E+01,-0.27764E+01,-0.27836E+01,
     &-0.27908E+01
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.49707E-01,-0.11083E+00,-0.14292E+00,-0.16643E+00,-0.18546E+00,
     &-0.20168E+00,-0.21592E+00,-0.22872E+00,-0.24038E+00,-0.25114E+00,
     &-0.26116E+00,-0.27056E+00,-0.27943E+00,-0.28784E+00,-0.29586E+00,
     &-0.30352E+00,-0.31087E+00,-0.31794E+00,-0.32476E+00,-0.33134E+00,
     &-0.33772E+00,-0.34390E+00,-0.34990E+00,-0.35574E+00,-0.36142E+00,
     &-0.36695E+00,-0.37235E+00,-0.37763E+00,-0.38278E+00,-0.38782E+00,
     &-0.39275E+00,-0.39758E+00,-0.40232E+00,-0.40696E+00,-0.41151E+00,
     &-0.41597E+00,-0.42035E+00,-0.42466E+00,-0.42889E+00,-0.43305E+00,
     &-0.43713E+00,-0.44115E+00,-0.44511E+00,-0.44900E+00,-0.45283E+00,
     &-0.45660E+00,-0.46032E+00,-0.46398E+00,-0.46759E+00,-0.47114E+00,
     &-0.47465E+00,-0.47811E+00,-0.48152E+00,-0.48489E+00,-0.48821E+00,
     &-0.49150E+00,-0.49474E+00,-0.49794E+00,-0.50111E+00,-0.50424E+00,
     &-0.50733E+00,-0.51039E+00,-0.51342E+00,-0.51642E+00,-0.51939E+00,
     &-0.52233E+00,-0.52524E+00,-0.52812E+00,-0.53098E+00,-0.53382E+00,
     &-0.53663E+00,-0.53943E+00,-0.54220E+00,-0.54495E+00,-0.54768E+00,
     &-0.55039E+00,-0.55309E+00,-0.55577E+00,-0.55843E+00,-0.56108E+00,
     &-0.56371E+00,-0.56633E+00,-0.56894E+00,-0.57153E+00,-0.57412E+00,
     &-0.57669E+00,-0.57925E+00,-0.58180E+00,-0.58433E+00,-0.58686E+00,
     &-0.58938E+00,-0.59189E+00,-0.59439E+00,-0.59688E+00,-0.59936E+00,
     &-0.60184E+00,-0.60430E+00,-0.60676E+00,-0.60921E+00,-0.61165E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.61408E+00,-0.61651E+00,-0.61892E+00,-0.62133E+00,-0.62373E+00,
     &-0.62612E+00,-0.62851E+00,-0.63088E+00,-0.63325E+00,-0.63561E+00,
     &-0.63796E+00,-0.64030E+00,-0.64263E+00,-0.64496E+00,-0.64727E+00,
     &-0.64958E+00,-0.65188E+00,-0.65417E+00,-0.65645E+00,-0.65872E+00,
     &-0.66098E+00,-0.66324E+00,-0.66548E+00,-0.66772E+00,-0.66995E+00,
     &-0.67217E+00,-0.67438E+00,-0.67658E+00,-0.67877E+00,-0.68095E+00,
     &-0.68312E+00,-0.68529E+00,-0.68744E+00,-0.68959E+00,-0.69173E+00,
     &-0.69386E+00,-0.69598E+00,-0.69809E+00,-0.70019E+00,-0.70229E+00,
     &-0.70437E+00,-0.70645E+00,-0.70852E+00,-0.71058E+00,-0.71263E+00,
     &-0.71468E+00,-0.71671E+00,-0.71874E+00,-0.72076E+00,-0.72277E+00,
     &-0.72477E+00,-0.72677E+00,-0.72876E+00,-0.73074E+00,-0.73271E+00,
     &-0.73468E+00,-0.73664E+00,-0.73859E+00,-0.74053E+00,-0.74246E+00,
     &-0.74439E+00,-0.74631E+00,-0.74823E+00,-0.75013E+00,-0.75203E+00,
     &-0.75393E+00,-0.75581E+00,-0.75769E+00,-0.75956E+00,-0.76143E+00,
     &-0.76329E+00,-0.76514E+00,-0.76698E+00,-0.76882E+00,-0.77066E+00,
     &-0.77248E+00,-0.77430E+00,-0.77611E+00,-0.77792E+00,-0.77972E+00,
     &-0.78152E+00,-0.78331E+00,-0.78509E+00,-0.78686E+00,-0.78863E+00,
     &-0.79040E+00,-0.79216E+00,-0.79391E+00,-0.79566E+00,-0.79740E+00,
     &-0.79913E+00,-0.80086E+00,-0.80258E+00,-0.80430E+00,-0.80601E+00,
     &-0.80772E+00,-0.80942E+00,-0.81112E+00,-0.81281E+00,-0.81449E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.81617E+00,-0.81785E+00,-0.81952E+00,-0.82118E+00,-0.82284E+00,
     &-0.82449E+00,-0.82614E+00,-0.82779E+00,-0.82942E+00,-0.83106E+00,
     &-0.83269E+00,-0.83431E+00,-0.83593E+00,-0.83754E+00,-0.83915E+00,
     &-0.84075E+00,-0.84235E+00,-0.84395E+00,-0.84554E+00,-0.84712E+00,
     &-0.84870E+00,-0.85028E+00,-0.85185E+00,-0.85341E+00,-0.85498E+00,
     &-0.85653E+00,-0.85809E+00,-0.85963E+00,-0.86118E+00,-0.86272E+00,
     &-0.86425E+00,-0.86578E+00,-0.86731E+00,-0.86883E+00,-0.87035E+00,
     &-0.87186E+00,-0.87337E+00,-0.87488E+00,-0.87638E+00,-0.87788E+00,
     &-0.87937E+00,-0.88086E+00,-0.88234E+00,-0.88382E+00,-0.88530E+00,
     &-0.88677E+00,-0.88824E+00,-0.88970E+00,-0.89116E+00,-0.89262E+00,
     &-0.89407E+00,-0.89552E+00,-0.89697E+00,-0.89841E+00,-0.89985E+00,
     &-0.90128E+00,-0.90271E+00,-0.90414E+00,-0.90556E+00,-0.90698E+00,
     &-0.90839E+00,-0.90980E+00,-0.91121E+00,-0.91261E+00,-0.91402E+00,
     &-0.91541E+00,-0.91681E+00,-0.91820E+00,-0.91958E+00,-0.92096E+00,
     &-0.92234E+00,-0.92372E+00,-0.92509E+00,-0.92646E+00,-0.92783E+00,
     &-0.92919E+00,-0.93055E+00,-0.93190E+00,-0.93326E+00,-0.93461E+00,
     &-0.93595E+00,-0.93729E+00,-0.93863E+00,-0.93997E+00,-0.94130E+00,
     &-0.94263E+00,-0.94396E+00,-0.94528E+00,-0.94660E+00,-0.94792E+00,
     &-0.94923E+00,-0.95054E+00,-0.95185E+00,-0.95316E+00,-0.95446E+00,
     &-0.95576E+00,-0.95705E+00,-0.95834E+00,-0.95963E+00,-0.96092E+00/

      DATA (BNC05M(I),I=301,400)/
     &-0.96220E+00,-0.96348E+00,-0.96476E+00,-0.96604E+00,-0.96731E+00,
     &-0.96858E+00,-0.96984E+00,-0.97110E+00,-0.97237E+00,-0.97362E+00,
     &-0.97488E+00,-0.97613E+00,-0.97738E+00,-0.97862E+00,-0.97987E+00,
     &-0.98111E+00,-0.98235E+00,-0.98358E+00,-0.98481E+00,-0.98604E+00,
     &-0.98727E+00,-0.98850E+00,-0.98972E+00,-0.99094E+00,-0.99215E+00,
     &-0.99337E+00,-0.99458E+00,-0.99579E+00,-0.99699E+00,-0.99820E+00,
     &-0.99940E+00,-0.10006E+01,-0.10018E+01,-0.10030E+01,-0.10042E+01,
     &-0.10054E+01,-0.10066E+01,-0.10077E+01,-0.10089E+01,-0.10101E+01,
     &-0.10113E+01,-0.10124E+01,-0.10136E+01,-0.10148E+01,-0.10159E+01,
     &-0.10171E+01,-0.10183E+01,-0.10194E+01,-0.10206E+01,-0.10217E+01,
     &-0.10229E+01,-0.10240E+01,-0.10252E+01,-0.10263E+01,-0.10275E+01,
     &-0.10286E+01,-0.10298E+01,-0.10309E+01,-0.10320E+01,-0.10332E+01,
     &-0.10343E+01,-0.10354E+01,-0.10365E+01,-0.10377E+01,-0.10388E+01,
     &-0.10399E+01,-0.10410E+01,-0.10421E+01,-0.10433E+01,-0.10444E+01,
     &-0.10455E+01,-0.10466E+01,-0.10477E+01,-0.10488E+01,-0.10499E+01,
     &-0.10510E+01,-0.10521E+01,-0.10532E+01,-0.10543E+01,-0.10553E+01,
     &-0.10564E+01,-0.10575E+01,-0.10586E+01,-0.10597E+01,-0.10608E+01,
     &-0.10618E+01,-0.10629E+01,-0.10640E+01,-0.10650E+01,-0.10661E+01,
     &-0.10672E+01,-0.10682E+01,-0.10693E+01,-0.10704E+01,-0.10714E+01,
     &-0.10725E+01,-0.10735E+01,-0.10746E+01,-0.10756E+01,-0.10767E+01/

      DATA (BNC05M(I),I=401,500)/
     &-0.10777E+01,-0.10788E+01,-0.10798E+01,-0.10809E+01,-0.10819E+01,
     &-0.10830E+01,-0.10840E+01,-0.10850E+01,-0.10861E+01,-0.10871E+01,
     &-0.10881E+01,-0.10891E+01,-0.10902E+01,-0.10912E+01,-0.10922E+01,
     &-0.10932E+01,-0.10943E+01,-0.10953E+01,-0.10963E+01,-0.10973E+01,
     &-0.10983E+01,-0.10993E+01,-0.11003E+01,-0.11013E+01,-0.11023E+01,
     &-0.11033E+01,-0.11043E+01,-0.11053E+01,-0.11063E+01,-0.11073E+01,
     &-0.11083E+01,-0.11093E+01,-0.11103E+01,-0.11113E+01,-0.11123E+01,
     &-0.11133E+01,-0.11143E+01,-0.11152E+01,-0.11162E+01,-0.11172E+01,
     &-0.11182E+01,-0.11192E+01,-0.11201E+01,-0.11211E+01,-0.11221E+01,
     &-0.11230E+01,-0.11240E+01,-0.11250E+01,-0.11260E+01,-0.11269E+01,
     &-0.11279E+01,-0.11288E+01,-0.11298E+01,-0.11308E+01,-0.11317E+01,
     &-0.11327E+01,-0.11336E+01,-0.11346E+01,-0.11355E+01,-0.11365E+01,
     &-0.11374E+01,-0.11384E+01,-0.11393E+01,-0.11402E+01,-0.11412E+01,
     &-0.11421E+01,-0.11431E+01,-0.11440E+01,-0.11449E+01,-0.11459E+01,
     &-0.11468E+01,-0.11477E+01,-0.11487E+01,-0.11496E+01,-0.11505E+01,
     &-0.11514E+01,-0.11524E+01,-0.11533E+01,-0.11542E+01,-0.11551E+01,
     &-0.11560E+01,-0.11570E+01,-0.11579E+01,-0.11588E+01,-0.11597E+01,
     &-0.11606E+01,-0.11615E+01,-0.11624E+01,-0.11633E+01,-0.11643E+01,
     &-0.11652E+01,-0.11661E+01,-0.11670E+01,-0.11679E+01,-0.11688E+01,
     &-0.11697E+01,-0.11706E+01,-0.11715E+01,-0.11723E+01,-0.11732E+01/

      DATA (BNC05M(I),I=501,600)/
     &-0.11741E+01,-0.11750E+01,-0.11759E+01,-0.11768E+01,-0.11777E+01,
     &-0.11786E+01,-0.11795E+01,-0.11803E+01,-0.11812E+01,-0.11821E+01,
     &-0.11830E+01,-0.11839E+01,-0.11847E+01,-0.11856E+01,-0.11865E+01,
     &-0.11874E+01,-0.11882E+01,-0.11891E+01,-0.11900E+01,-0.11908E+01,
     &-0.11917E+01,-0.11926E+01,-0.11934E+01,-0.11943E+01,-0.11952E+01,
     &-0.11960E+01,-0.11969E+01,-0.11977E+01,-0.11986E+01,-0.11994E+01,
     &-0.12003E+01,-0.12011E+01,-0.12020E+01,-0.12029E+01,-0.12037E+01,
     &-0.12046E+01,-0.12054E+01,-0.12062E+01,-0.12071E+01,-0.12079E+01,
     &-0.12088E+01,-0.12096E+01,-0.12105E+01,-0.12113E+01,-0.12121E+01,
     &-0.12130E+01,-0.12138E+01,-0.12147E+01,-0.12155E+01,-0.12163E+01,
     &-0.12171E+01,-0.12180E+01,-0.12188E+01,-0.12196E+01,-0.12205E+01,
     &-0.12213E+01,-0.12221E+01,-0.12229E+01,-0.12238E+01,-0.12246E+01,
     &-0.12254E+01,-0.12262E+01,-0.12271E+01,-0.12279E+01,-0.12287E+01,
     &-0.12295E+01,-0.12303E+01,-0.12311E+01,-0.12319E+01,-0.12328E+01,
     &-0.12336E+01,-0.12344E+01,-0.12352E+01,-0.12360E+01,-0.12368E+01,
     &-0.12376E+01,-0.12384E+01,-0.12392E+01,-0.12400E+01,-0.12408E+01,
     &-0.12416E+01,-0.12424E+01,-0.12432E+01,-0.12440E+01,-0.12448E+01,
     &-0.12456E+01,-0.12464E+01,-0.12472E+01,-0.12480E+01,-0.12488E+01,
     &-0.12496E+01,-0.12504E+01,-0.12512E+01,-0.12519E+01,-0.12527E+01,
     &-0.12535E+01,-0.12543E+01,-0.12551E+01,-0.12559E+01,-0.12567E+01/

      DATA (BNC05M(I),I=601,700)/
     &-0.12650E+01,-0.12727E+01,-0.12803E+01,-0.12878E+01,-0.12952E+01,
     &-0.13026E+01,-0.13098E+01,-0.13170E+01,-0.13241E+01,-0.13312E+01,
     &-0.13381E+01,-0.13450E+01,-0.13519E+01,-0.13586E+01,-0.13654E+01,
     &-0.13720E+01,-0.13786E+01,-0.13851E+01,-0.13916E+01,-0.13980E+01,
     &-0.14044E+01,-0.14107E+01,-0.14169E+01,-0.14231E+01,-0.14293E+01,
     &-0.14354E+01,-0.14415E+01,-0.14475E+01,-0.14534E+01,-0.14594E+01,
     &-0.14652E+01,-0.14711E+01,-0.14769E+01,-0.14827E+01,-0.14884E+01,
     &-0.14941E+01,-0.14997E+01,-0.15053E+01,-0.15109E+01,-0.15164E+01,
     &-0.15219E+01,-0.15274E+01,-0.15328E+01,-0.15382E+01,-0.15436E+01,
     &-0.15490E+01,-0.15543E+01,-0.15595E+01,-0.15648E+01,-0.15700E+01,
     &-0.15752E+01,-0.15804E+01,-0.15855E+01,-0.15906E+01,-0.15957E+01,
     &-0.16008E+01,-0.16058E+01,-0.16108E+01,-0.16158E+01,-0.16208E+01,
     &-0.16257E+01,-0.16306E+01,-0.16355E+01,-0.16404E+01,-0.16452E+01,
     &-0.16500E+01,-0.16548E+01,-0.16596E+01,-0.16644E+01,-0.16691E+01,
     &-0.16739E+01,-0.16786E+01,-0.16832E+01,-0.16879E+01,-0.16926E+01,
     &-0.16972E+01,-0.17018E+01,-0.17064E+01,-0.17110E+01,-0.17155E+01,
     &-0.17201E+01,-0.17246E+01,-0.17291E+01,-0.17336E+01,-0.17381E+01,
     &-0.17425E+01,-0.17470E+01,-0.17514E+01,-0.17558E+01,-0.17602E+01,
     &-0.17646E+01,-0.17690E+01,-0.17733E+01,-0.17777E+01,-0.17820E+01,
     &-0.17863E+01,-0.17906E+01,-0.17949E+01,-0.17992E+01,-0.18035E+01/

      DATA (BNC05M(I),I=701,741)/
     &-0.18077E+01,-0.18119E+01,-0.18162E+01,-0.18204E+01,-0.18246E+01,
     &-0.18288E+01,-0.18330E+01,-0.18371E+01,-0.18413E+01,-0.18454E+01,
     &-0.18496E+01,-0.18537E+01,-0.18578E+01,-0.18619E+01,-0.18660E+01,
     &-0.18701E+01,-0.18742E+01,-0.18782E+01,-0.18823E+01,-0.18863E+01,
     &-0.18903E+01,-0.18944E+01,-0.18984E+01,-0.19024E+01,-0.19064E+01,
     &-0.19104E+01,-0.19144E+01,-0.19183E+01,-0.19223E+01,-0.19262E+01,
     &-0.19302E+01,-0.19341E+01,-0.19380E+01,-0.19419E+01,-0.19459E+01,
     &-0.19498E+01,-0.19537E+01,-0.19575E+01,-0.19614E+01,-0.19653E+01,
     &-0.19691E+01
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.48310E-01,-0.10321E+00,-0.12958E+00,-0.14766E+00,-0.16145E+00,
     &-0.17257E+00,-0.18185E+00,-0.18977E+00,-0.19665E+00,-0.20270E+00,
     &-0.20809E+00,-0.21292E+00,-0.21728E+00,-0.22124E+00,-0.22486E+00,
     &-0.22818E+00,-0.23123E+00,-0.23405E+00,-0.23666E+00,-0.23908E+00,
     &-0.24134E+00,-0.24344E+00,-0.24541E+00,-0.24725E+00,-0.24898E+00,
     &-0.25060E+00,-0.25213E+00,-0.25357E+00,-0.25493E+00,-0.25621E+00,
     &-0.25742E+00,-0.25857E+00,-0.25966E+00,-0.26069E+00,-0.26167E+00,
     &-0.26261E+00,-0.26350E+00,-0.26434E+00,-0.26515E+00,-0.26592E+00,
     &-0.26666E+00,-0.26736E+00,-0.26804E+00,-0.26868E+00,-0.26930E+00,
     &-0.26990E+00,-0.27047E+00,-0.27101E+00,-0.27154E+00,-0.27205E+00,
     &-0.27253E+00,-0.27300E+00,-0.27345E+00,-0.27389E+00,-0.27431E+00,
     &-0.27471E+00,-0.27510E+00,-0.27547E+00,-0.27583E+00,-0.27618E+00,
     &-0.27651E+00,-0.27684E+00,-0.27714E+00,-0.27744E+00,-0.27772E+00,
     &-0.27800E+00,-0.27826E+00,-0.27851E+00,-0.27874E+00,-0.27897E+00,
     &-0.27918E+00,-0.27939E+00,-0.27958E+00,-0.27976E+00,-0.27993E+00,
     &-0.28008E+00,-0.28023E+00,-0.28037E+00,-0.28049E+00,-0.28060E+00,
     &-0.28070E+00,-0.28080E+00,-0.28088E+00,-0.28094E+00,-0.28100E+00,
     &-0.28105E+00,-0.28108E+00,-0.28111E+00,-0.28113E+00,-0.28113E+00,
     &-0.28112E+00,-0.28111E+00,-0.28108E+00,-0.28105E+00,-0.28100E+00,
     &-0.28095E+00,-0.28089E+00,-0.28081E+00,-0.28073E+00,-0.28064E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.28055E+00,-0.28044E+00,-0.28033E+00,-0.28021E+00,-0.28008E+00,
     &-0.27994E+00,-0.27980E+00,-0.27965E+00,-0.27950E+00,-0.27934E+00,
     &-0.27917E+00,-0.27900E+00,-0.27882E+00,-0.27864E+00,-0.27845E+00,
     &-0.27826E+00,-0.27806E+00,-0.27786E+00,-0.27766E+00,-0.27745E+00,
     &-0.27724E+00,-0.27702E+00,-0.27681E+00,-0.27658E+00,-0.27636E+00,
     &-0.27613E+00,-0.27590E+00,-0.27567E+00,-0.27544E+00,-0.27520E+00,
     &-0.27496E+00,-0.27472E+00,-0.27448E+00,-0.27423E+00,-0.27398E+00,
     &-0.27374E+00,-0.27349E+00,-0.27323E+00,-0.27298E+00,-0.27273E+00,
     &-0.27247E+00,-0.27222E+00,-0.27196E+00,-0.27170E+00,-0.27144E+00,
     &-0.27118E+00,-0.27092E+00,-0.27065E+00,-0.27039E+00,-0.27013E+00,
     &-0.26986E+00,-0.26959E+00,-0.26933E+00,-0.26906E+00,-0.26879E+00,
     &-0.26852E+00,-0.26825E+00,-0.26798E+00,-0.26771E+00,-0.26744E+00,
     &-0.26717E+00,-0.26690E+00,-0.26663E+00,-0.26635E+00,-0.26608E+00,
     &-0.26581E+00,-0.26553E+00,-0.26526E+00,-0.26499E+00,-0.26471E+00,
     &-0.26444E+00,-0.26416E+00,-0.26389E+00,-0.26361E+00,-0.26333E+00,
     &-0.26306E+00,-0.26278E+00,-0.26251E+00,-0.26223E+00,-0.26195E+00,
     &-0.26168E+00,-0.26140E+00,-0.26112E+00,-0.26085E+00,-0.26057E+00,
     &-0.26029E+00,-0.26002E+00,-0.25974E+00,-0.25946E+00,-0.25919E+00,
     &-0.25891E+00,-0.25863E+00,-0.25836E+00,-0.25808E+00,-0.25781E+00,
     &-0.25753E+00,-0.25725E+00,-0.25698E+00,-0.25670E+00,-0.25643E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.25615E+00,-0.25588E+00,-0.25560E+00,-0.25533E+00,-0.25505E+00,
     &-0.25478E+00,-0.25450E+00,-0.25423E+00,-0.25396E+00,-0.25368E+00,
     &-0.25341E+00,-0.25314E+00,-0.25287E+00,-0.25259E+00,-0.25232E+00,
     &-0.25205E+00,-0.25178E+00,-0.25151E+00,-0.25124E+00,-0.25097E+00,
     &-0.25070E+00,-0.25043E+00,-0.25016E+00,-0.24989E+00,-0.24962E+00,
     &-0.24935E+00,-0.24908E+00,-0.24882E+00,-0.24855E+00,-0.24828E+00,
     &-0.24802E+00,-0.24775E+00,-0.24749E+00,-0.24722E+00,-0.24696E+00,
     &-0.24669E+00,-0.24643E+00,-0.24616E+00,-0.24590E+00,-0.24564E+00,
     &-0.24538E+00,-0.24511E+00,-0.24485E+00,-0.24459E+00,-0.24433E+00,
     &-0.24407E+00,-0.24381E+00,-0.24355E+00,-0.24329E+00,-0.24304E+00,
     &-0.24278E+00,-0.24252E+00,-0.24226E+00,-0.24201E+00,-0.24175E+00,
     &-0.24150E+00,-0.24124E+00,-0.24099E+00,-0.24073E+00,-0.24048E+00,
     &-0.24023E+00,-0.23997E+00,-0.23972E+00,-0.23947E+00,-0.23922E+00,
     &-0.23897E+00,-0.23872E+00,-0.23847E+00,-0.23822E+00,-0.23797E+00,
     &-0.23772E+00,-0.23748E+00,-0.23723E+00,-0.23698E+00,-0.23674E+00,
     &-0.23649E+00,-0.23625E+00,-0.23600E+00,-0.23576E+00,-0.23552E+00,
     &-0.23527E+00,-0.23503E+00,-0.23479E+00,-0.23455E+00,-0.23431E+00,
     &-0.23407E+00,-0.23383E+00,-0.23359E+00,-0.23335E+00,-0.23311E+00,
     &-0.23288E+00,-0.23264E+00,-0.23240E+00,-0.23217E+00,-0.23193E+00,
     &-0.23170E+00,-0.23146E+00,-0.23123E+00,-0.23100E+00,-0.23076E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.23053E+00,-0.23030E+00,-0.23007E+00,-0.22984E+00,-0.22961E+00,
     &-0.22938E+00,-0.22915E+00,-0.22892E+00,-0.22869E+00,-0.22847E+00,
     &-0.22824E+00,-0.22802E+00,-0.22779E+00,-0.22757E+00,-0.22734E+00,
     &-0.22712E+00,-0.22689E+00,-0.22667E+00,-0.22645E+00,-0.22623E+00,
     &-0.22601E+00,-0.22579E+00,-0.22557E+00,-0.22535E+00,-0.22513E+00,
     &-0.22491E+00,-0.22469E+00,-0.22448E+00,-0.22426E+00,-0.22404E+00,
     &-0.22383E+00,-0.22361E+00,-0.22340E+00,-0.22318E+00,-0.22297E+00,
     &-0.22276E+00,-0.22255E+00,-0.22233E+00,-0.22212E+00,-0.22191E+00,
     &-0.22170E+00,-0.22149E+00,-0.22129E+00,-0.22108E+00,-0.22087E+00,
     &-0.22066E+00,-0.22046E+00,-0.22025E+00,-0.22004E+00,-0.21984E+00,
     &-0.21963E+00,-0.21943E+00,-0.21923E+00,-0.21902E+00,-0.21882E+00,
     &-0.21862E+00,-0.21842E+00,-0.21822E+00,-0.21802E+00,-0.21782E+00,
     &-0.21762E+00,-0.21742E+00,-0.21722E+00,-0.21703E+00,-0.21683E+00,
     &-0.21663E+00,-0.21644E+00,-0.21624E+00,-0.21605E+00,-0.21585E+00,
     &-0.21566E+00,-0.21547E+00,-0.21527E+00,-0.21508E+00,-0.21489E+00,
     &-0.21470E+00,-0.21451E+00,-0.21432E+00,-0.21413E+00,-0.21394E+00,
     &-0.21375E+00,-0.21357E+00,-0.21338E+00,-0.21319E+00,-0.21301E+00,
     &-0.21282E+00,-0.21264E+00,-0.21245E+00,-0.21227E+00,-0.21208E+00,
     &-0.21190E+00,-0.21172E+00,-0.21154E+00,-0.21136E+00,-0.21118E+00,
     &-0.21100E+00,-0.21082E+00,-0.21064E+00,-0.21046E+00,-0.21028E+00/

      DATA (BNC06M(I),I=401,500)/
     &-0.21010E+00,-0.20992E+00,-0.20975E+00,-0.20957E+00,-0.20940E+00,
     &-0.20922E+00,-0.20905E+00,-0.20887E+00,-0.20870E+00,-0.20853E+00,
     &-0.20835E+00,-0.20818E+00,-0.20801E+00,-0.20784E+00,-0.20767E+00,
     &-0.20750E+00,-0.20733E+00,-0.20716E+00,-0.20699E+00,-0.20683E+00,
     &-0.20666E+00,-0.20649E+00,-0.20632E+00,-0.20616E+00,-0.20599E+00,
     &-0.20583E+00,-0.20566E+00,-0.20550E+00,-0.20534E+00,-0.20517E+00,
     &-0.20501E+00,-0.20485E+00,-0.20469E+00,-0.20453E+00,-0.20437E+00,
     &-0.20421E+00,-0.20405E+00,-0.20389E+00,-0.20373E+00,-0.20357E+00,
     &-0.20342E+00,-0.20326E+00,-0.20310E+00,-0.20295E+00,-0.20279E+00,
     &-0.20264E+00,-0.20248E+00,-0.20233E+00,-0.20218E+00,-0.20202E+00,
     &-0.20187E+00,-0.20172E+00,-0.20157E+00,-0.20142E+00,-0.20127E+00,
     &-0.20112E+00,-0.20097E+00,-0.20082E+00,-0.20067E+00,-0.20052E+00,
     &-0.20037E+00,-0.20022E+00,-0.20008E+00,-0.19993E+00,-0.19979E+00,
     &-0.19964E+00,-0.19950E+00,-0.19935E+00,-0.19921E+00,-0.19906E+00,
     &-0.19892E+00,-0.19878E+00,-0.19864E+00,-0.19850E+00,-0.19835E+00,
     &-0.19821E+00,-0.19807E+00,-0.19793E+00,-0.19780E+00,-0.19766E+00,
     &-0.19752E+00,-0.19738E+00,-0.19724E+00,-0.19711E+00,-0.19697E+00,
     &-0.19683E+00,-0.19670E+00,-0.19656E+00,-0.19643E+00,-0.19629E+00,
     &-0.19616E+00,-0.19603E+00,-0.19589E+00,-0.19576E+00,-0.19563E+00,
     &-0.19550E+00,-0.19537E+00,-0.19524E+00,-0.19511E+00,-0.19498E+00/

      DATA (BNC06M(I),I=501,600)/
     &-0.19485E+00,-0.19472E+00,-0.19459E+00,-0.19446E+00,-0.19434E+00,
     &-0.19421E+00,-0.19408E+00,-0.19396E+00,-0.19383E+00,-0.19371E+00,
     &-0.19358E+00,-0.19346E+00,-0.19333E+00,-0.19321E+00,-0.19309E+00,
     &-0.19296E+00,-0.19284E+00,-0.19272E+00,-0.19260E+00,-0.19248E+00,
     &-0.19236E+00,-0.19224E+00,-0.19212E+00,-0.19200E+00,-0.19188E+00,
     &-0.19176E+00,-0.19164E+00,-0.19153E+00,-0.19141E+00,-0.19129E+00,
     &-0.19118E+00,-0.19106E+00,-0.19094E+00,-0.19083E+00,-0.19071E+00,
     &-0.19060E+00,-0.19049E+00,-0.19037E+00,-0.19026E+00,-0.19015E+00,
     &-0.19004E+00,-0.18992E+00,-0.18981E+00,-0.18970E+00,-0.18959E+00,
     &-0.18948E+00,-0.18937E+00,-0.18926E+00,-0.18916E+00,-0.18905E+00,
     &-0.18894E+00,-0.18883E+00,-0.18872E+00,-0.18862E+00,-0.18851E+00,
     &-0.18841E+00,-0.18830E+00,-0.18819E+00,-0.18809E+00,-0.18799E+00,
     &-0.18788E+00,-0.18778E+00,-0.18768E+00,-0.18757E+00,-0.18747E+00,
     &-0.18737E+00,-0.18727E+00,-0.18717E+00,-0.18707E+00,-0.18697E+00,
     &-0.18687E+00,-0.18677E+00,-0.18667E+00,-0.18657E+00,-0.18647E+00,
     &-0.18637E+00,-0.18627E+00,-0.18618E+00,-0.18608E+00,-0.18598E+00,
     &-0.18589E+00,-0.18579E+00,-0.18570E+00,-0.18560E+00,-0.18551E+00,
     &-0.18541E+00,-0.18532E+00,-0.18523E+00,-0.18513E+00,-0.18504E+00,
     &-0.18495E+00,-0.18486E+00,-0.18477E+00,-0.18468E+00,-0.18458E+00,
     &-0.18449E+00,-0.18440E+00,-0.18431E+00,-0.18423E+00,-0.18414E+00/

      DATA (BNC06M(I),I=601,700)/
     &-0.18320E+00,-0.18237E+00,-0.18158E+00,-0.18083E+00,-0.18010E+00,
     &-0.17942E+00,-0.17876E+00,-0.17814E+00,-0.17755E+00,-0.17700E+00,
     &-0.17647E+00,-0.17597E+00,-0.17551E+00,-0.17507E+00,-0.17467E+00,
     &-0.17429E+00,-0.17394E+00,-0.17362E+00,-0.17333E+00,-0.17306E+00,
     &-0.17282E+00,-0.17260E+00,-0.17241E+00,-0.17225E+00,-0.17211E+00,
     &-0.17200E+00,-0.17190E+00,-0.17184E+00,-0.17179E+00,-0.17177E+00,
     &-0.17177E+00,-0.17180E+00,-0.17184E+00,-0.17191E+00,-0.17200E+00,
     &-0.17210E+00,-0.17223E+00,-0.17238E+00,-0.17255E+00,-0.17274E+00,
     &-0.17295E+00,-0.17318E+00,-0.17342E+00,-0.17369E+00,-0.17397E+00,
     &-0.17427E+00,-0.17459E+00,-0.17493E+00,-0.17528E+00,-0.17565E+00,
     &-0.17604E+00,-0.17644E+00,-0.17687E+00,-0.17730E+00,-0.17775E+00,
     &-0.17822E+00,-0.17871E+00,-0.17921E+00,-0.17972E+00,-0.18025E+00,
     &-0.18079E+00,-0.18135E+00,-0.18193E+00,-0.18251E+00,-0.18311E+00,
     &-0.18373E+00,-0.18436E+00,-0.18500E+00,-0.18565E+00,-0.18632E+00,
     &-0.18700E+00,-0.18770E+00,-0.18840E+00,-0.18912E+00,-0.18985E+00,
     &-0.19060E+00,-0.19135E+00,-0.19212E+00,-0.19290E+00,-0.19369E+00,
     &-0.19450E+00,-0.19531E+00,-0.19614E+00,-0.19697E+00,-0.19782E+00,
     &-0.19868E+00,-0.19955E+00,-0.20043E+00,-0.20132E+00,-0.20222E+00,
     &-0.20313E+00,-0.20405E+00,-0.20498E+00,-0.20592E+00,-0.20687E+00,
     &-0.20783E+00,-0.20880E+00,-0.20978E+00,-0.21077E+00,-0.21177E+00/

      DATA (BNC06M(I),I=701,741)/
     &-0.21278E+00,-0.21379E+00,-0.21482E+00,-0.21585E+00,-0.21690E+00,
     &-0.21795E+00,-0.21901E+00,-0.22008E+00,-0.22116E+00,-0.22224E+00,
     &-0.22334E+00,-0.22444E+00,-0.22555E+00,-0.22667E+00,-0.22780E+00,
     &-0.22893E+00,-0.23007E+00,-0.23122E+00,-0.23238E+00,-0.23355E+00,
     &-0.23472E+00,-0.23590E+00,-0.23709E+00,-0.23828E+00,-0.23949E+00,
     &-0.24070E+00,-0.24191E+00,-0.24314E+00,-0.24437E+00,-0.24561E+00,
     &-0.24685E+00,-0.24810E+00,-0.24936E+00,-0.25062E+00,-0.25189E+00,
     &-0.25317E+00,-0.25446E+00,-0.25575E+00,-0.25704E+00,-0.25835E+00,
     &-0.25966E+00
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.97854E-01,-0.21310E+00,-0.27082E+00,-0.31170E+00,-0.34384E+00,
     &-0.37049E+00,-0.39335E+00,-0.41342E+00,-0.43132E+00,-0.44752E+00,
     &-0.46231E+00,-0.47593E+00,-0.48857E+00,-0.50036E+00,-0.51142E+00,
     &-0.52184E+00,-0.53169E+00,-0.54103E+00,-0.54992E+00,-0.55840E+00,
     &-0.56651E+00,-0.57428E+00,-0.58175E+00,-0.58893E+00,-0.59585E+00,
     &-0.60254E+00,-0.60899E+00,-0.61525E+00,-0.62130E+00,-0.62718E+00,
     &-0.63289E+00,-0.63844E+00,-0.64384E+00,-0.64910E+00,-0.65422E+00,
     &-0.65922E+00,-0.66410E+00,-0.66887E+00,-0.67353E+00,-0.67809E+00,
     &-0.68255E+00,-0.68692E+00,-0.69120E+00,-0.69539E+00,-0.69951E+00,
     &-0.70355E+00,-0.70751E+00,-0.71141E+00,-0.71523E+00,-0.71899E+00,
     &-0.72269E+00,-0.72632E+00,-0.72990E+00,-0.73342E+00,-0.73689E+00,
     &-0.74031E+00,-0.74367E+00,-0.74699E+00,-0.75026E+00,-0.75349E+00,
     &-0.75667E+00,-0.75981E+00,-0.76291E+00,-0.76597E+00,-0.76900E+00,
     &-0.77198E+00,-0.77493E+00,-0.77785E+00,-0.78073E+00,-0.78358E+00,
     &-0.78640E+00,-0.78919E+00,-0.79195E+00,-0.79468E+00,-0.79738E+00,
     &-0.80006E+00,-0.80271E+00,-0.80533E+00,-0.80793E+00,-0.81051E+00,
     &-0.81306E+00,-0.81559E+00,-0.81810E+00,-0.82058E+00,-0.82305E+00,
     &-0.82550E+00,-0.82792E+00,-0.83033E+00,-0.83271E+00,-0.83508E+00,
     &-0.83744E+00,-0.83977E+00,-0.84209E+00,-0.84439E+00,-0.84667E+00,
     &-0.84894E+00,-0.85119E+00,-0.85343E+00,-0.85565E+00,-0.85785E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.86005E+00,-0.86222E+00,-0.86439E+00,-0.86654E+00,-0.86868E+00,
     &-0.87080E+00,-0.87291E+00,-0.87501E+00,-0.87710E+00,-0.87917E+00,
     &-0.88123E+00,-0.88328E+00,-0.88532E+00,-0.88735E+00,-0.88936E+00,
     &-0.89137E+00,-0.89336E+00,-0.89534E+00,-0.89731E+00,-0.89927E+00,
     &-0.90122E+00,-0.90316E+00,-0.90509E+00,-0.90701E+00,-0.90892E+00,
     &-0.91082E+00,-0.91271E+00,-0.91460E+00,-0.91647E+00,-0.91833E+00,
     &-0.92019E+00,-0.92203E+00,-0.92387E+00,-0.92569E+00,-0.92751E+00,
     &-0.92932E+00,-0.93113E+00,-0.93292E+00,-0.93471E+00,-0.93648E+00,
     &-0.93825E+00,-0.94002E+00,-0.94177E+00,-0.94352E+00,-0.94526E+00,
     &-0.94699E+00,-0.94872E+00,-0.95043E+00,-0.95214E+00,-0.95385E+00,
     &-0.95554E+00,-0.95723E+00,-0.95892E+00,-0.96060E+00,-0.96227E+00,
     &-0.96393E+00,-0.96559E+00,-0.96724E+00,-0.96888E+00,-0.97052E+00,
     &-0.97215E+00,-0.97378E+00,-0.97540E+00,-0.97701E+00,-0.97862E+00,
     &-0.98023E+00,-0.98182E+00,-0.98342E+00,-0.98500E+00,-0.98658E+00,
     &-0.98816E+00,-0.98973E+00,-0.99129E+00,-0.99285E+00,-0.99441E+00,
     &-0.99596E+00,-0.99750E+00,-0.99904E+00,-0.10006E+01,-0.10021E+01,
     &-0.10036E+01,-0.10052E+01,-0.10067E+01,-0.10082E+01,-0.10097E+01,
     &-0.10112E+01,-0.10127E+01,-0.10142E+01,-0.10157E+01,-0.10171E+01,
     &-0.10186E+01,-0.10201E+01,-0.10216E+01,-0.10230E+01,-0.10245E+01,
     &-0.10260E+01,-0.10274E+01,-0.10289E+01,-0.10303E+01,-0.10317E+01/

      DATA (BNC07M(I),I=201,300)/
     &-0.10332E+01,-0.10346E+01,-0.10361E+01,-0.10375E+01,-0.10389E+01,
     &-0.10403E+01,-0.10417E+01,-0.10431E+01,-0.10446E+01,-0.10460E+01,
     &-0.10474E+01,-0.10488E+01,-0.10502E+01,-0.10515E+01,-0.10529E+01,
     &-0.10543E+01,-0.10557E+01,-0.10571E+01,-0.10584E+01,-0.10598E+01,
     &-0.10612E+01,-0.10625E+01,-0.10639E+01,-0.10653E+01,-0.10666E+01,
     &-0.10680E+01,-0.10693E+01,-0.10707E+01,-0.10720E+01,-0.10733E+01,
     &-0.10747E+01,-0.10760E+01,-0.10773E+01,-0.10787E+01,-0.10800E+01,
     &-0.10813E+01,-0.10826E+01,-0.10839E+01,-0.10853E+01,-0.10866E+01,
     &-0.10879E+01,-0.10892E+01,-0.10905E+01,-0.10918E+01,-0.10931E+01,
     &-0.10944E+01,-0.10957E+01,-0.10970E+01,-0.10982E+01,-0.10995E+01,
     &-0.11008E+01,-0.11021E+01,-0.11034E+01,-0.11046E+01,-0.11059E+01,
     &-0.11072E+01,-0.11084E+01,-0.11097E+01,-0.11110E+01,-0.11122E+01,
     &-0.11135E+01,-0.11147E+01,-0.11160E+01,-0.11172E+01,-0.11185E+01,
     &-0.11197E+01,-0.11210E+01,-0.11222E+01,-0.11234E+01,-0.11247E+01,
     &-0.11259E+01,-0.11271E+01,-0.11284E+01,-0.11296E+01,-0.11308E+01,
     &-0.11321E+01,-0.11333E+01,-0.11345E+01,-0.11357E+01,-0.11369E+01,
     &-0.11381E+01,-0.11393E+01,-0.11406E+01,-0.11418E+01,-0.11430E+01,
     &-0.11442E+01,-0.11454E+01,-0.11466E+01,-0.11478E+01,-0.11490E+01,
     &-0.11502E+01,-0.11513E+01,-0.11525E+01,-0.11537E+01,-0.11549E+01,
     &-0.11561E+01,-0.11573E+01,-0.11585E+01,-0.11596E+01,-0.11608E+01/

      DATA (BNC07M(I),I=301,400)/
     &-0.11620E+01,-0.11632E+01,-0.11643E+01,-0.11655E+01,-0.11667E+01,
     &-0.11678E+01,-0.11690E+01,-0.11702E+01,-0.11713E+01,-0.11725E+01,
     &-0.11736E+01,-0.11748E+01,-0.11759E+01,-0.11771E+01,-0.11782E+01,
     &-0.11794E+01,-0.11805E+01,-0.11817E+01,-0.11828E+01,-0.11840E+01,
     &-0.11851E+01,-0.11863E+01,-0.11874E+01,-0.11885E+01,-0.11897E+01,
     &-0.11908E+01,-0.11919E+01,-0.11931E+01,-0.11942E+01,-0.11953E+01,
     &-0.11964E+01,-0.11976E+01,-0.11987E+01,-0.11998E+01,-0.12009E+01,
     &-0.12021E+01,-0.12032E+01,-0.12043E+01,-0.12054E+01,-0.12065E+01,
     &-0.12076E+01,-0.12087E+01,-0.12098E+01,-0.12109E+01,-0.12121E+01,
     &-0.12132E+01,-0.12143E+01,-0.12154E+01,-0.12165E+01,-0.12176E+01,
     &-0.12187E+01,-0.12198E+01,-0.12209E+01,-0.12219E+01,-0.12230E+01,
     &-0.12241E+01,-0.12252E+01,-0.12263E+01,-0.12274E+01,-0.12285E+01,
     &-0.12296E+01,-0.12306E+01,-0.12317E+01,-0.12328E+01,-0.12339E+01,
     &-0.12350E+01,-0.12360E+01,-0.12371E+01,-0.12382E+01,-0.12393E+01,
     &-0.12403E+01,-0.12414E+01,-0.12425E+01,-0.12436E+01,-0.12446E+01,
     &-0.12457E+01,-0.12468E+01,-0.12478E+01,-0.12489E+01,-0.12499E+01,
     &-0.12510E+01,-0.12521E+01,-0.12531E+01,-0.12542E+01,-0.12552E+01,
     &-0.12563E+01,-0.12574E+01,-0.12584E+01,-0.12595E+01,-0.12605E+01,
     &-0.12616E+01,-0.12626E+01,-0.12637E+01,-0.12647E+01,-0.12657E+01,
     &-0.12668E+01,-0.12678E+01,-0.12689E+01,-0.12699E+01,-0.12710E+01/

      DATA (BNC07M(I),I=401,500)/
     &-0.12720E+01,-0.12730E+01,-0.12741E+01,-0.12751E+01,-0.12761E+01,
     &-0.12772E+01,-0.12782E+01,-0.12792E+01,-0.12803E+01,-0.12813E+01,
     &-0.12823E+01,-0.12834E+01,-0.12844E+01,-0.12854E+01,-0.12864E+01,
     &-0.12875E+01,-0.12885E+01,-0.12895E+01,-0.12905E+01,-0.12916E+01,
     &-0.12926E+01,-0.12936E+01,-0.12946E+01,-0.12956E+01,-0.12966E+01,
     &-0.12977E+01,-0.12987E+01,-0.12997E+01,-0.13007E+01,-0.13017E+01,
     &-0.13027E+01,-0.13037E+01,-0.13047E+01,-0.13057E+01,-0.13068E+01,
     &-0.13078E+01,-0.13088E+01,-0.13098E+01,-0.13108E+01,-0.13118E+01,
     &-0.13128E+01,-0.13138E+01,-0.13148E+01,-0.13158E+01,-0.13168E+01,
     &-0.13178E+01,-0.13188E+01,-0.13198E+01,-0.13208E+01,-0.13218E+01,
     &-0.13227E+01,-0.13237E+01,-0.13247E+01,-0.13257E+01,-0.13267E+01,
     &-0.13277E+01,-0.13287E+01,-0.13297E+01,-0.13307E+01,-0.13317E+01,
     &-0.13326E+01,-0.13336E+01,-0.13346E+01,-0.13356E+01,-0.13366E+01,
     &-0.13376E+01,-0.13385E+01,-0.13395E+01,-0.13405E+01,-0.13415E+01,
     &-0.13424E+01,-0.13434E+01,-0.13444E+01,-0.13454E+01,-0.13463E+01,
     &-0.13473E+01,-0.13483E+01,-0.13493E+01,-0.13502E+01,-0.13512E+01,
     &-0.13522E+01,-0.13532E+01,-0.13541E+01,-0.13551E+01,-0.13561E+01,
     &-0.13570E+01,-0.13580E+01,-0.13590E+01,-0.13599E+01,-0.13609E+01,
     &-0.13618E+01,-0.13628E+01,-0.13638E+01,-0.13647E+01,-0.13657E+01,
     &-0.13666E+01,-0.13676E+01,-0.13686E+01,-0.13695E+01,-0.13705E+01/

      DATA (BNC07M(I),I=501,600)/
     &-0.13714E+01,-0.13724E+01,-0.13733E+01,-0.13743E+01,-0.13753E+01,
     &-0.13762E+01,-0.13772E+01,-0.13781E+01,-0.13791E+01,-0.13800E+01,
     &-0.13810E+01,-0.13819E+01,-0.13829E+01,-0.13838E+01,-0.13847E+01,
     &-0.13857E+01,-0.13866E+01,-0.13876E+01,-0.13885E+01,-0.13895E+01,
     &-0.13904E+01,-0.13914E+01,-0.13923E+01,-0.13932E+01,-0.13942E+01,
     &-0.13951E+01,-0.13961E+01,-0.13970E+01,-0.13979E+01,-0.13989E+01,
     &-0.13998E+01,-0.14007E+01,-0.14017E+01,-0.14026E+01,-0.14036E+01,
     &-0.14045E+01,-0.14054E+01,-0.14064E+01,-0.14073E+01,-0.14082E+01,
     &-0.14091E+01,-0.14101E+01,-0.14110E+01,-0.14119E+01,-0.14129E+01,
     &-0.14138E+01,-0.14147E+01,-0.14156E+01,-0.14166E+01,-0.14175E+01,
     &-0.14184E+01,-0.14193E+01,-0.14203E+01,-0.14212E+01,-0.14221E+01,
     &-0.14230E+01,-0.14240E+01,-0.14249E+01,-0.14258E+01,-0.14267E+01,
     &-0.14276E+01,-0.14286E+01,-0.14295E+01,-0.14304E+01,-0.14313E+01,
     &-0.14322E+01,-0.14332E+01,-0.14341E+01,-0.14350E+01,-0.14359E+01,
     &-0.14368E+01,-0.14377E+01,-0.14386E+01,-0.14396E+01,-0.14405E+01,
     &-0.14414E+01,-0.14423E+01,-0.14432E+01,-0.14441E+01,-0.14450E+01,
     &-0.14459E+01,-0.14468E+01,-0.14477E+01,-0.14487E+01,-0.14496E+01,
     &-0.14505E+01,-0.14514E+01,-0.14523E+01,-0.14532E+01,-0.14541E+01,
     &-0.14550E+01,-0.14559E+01,-0.14568E+01,-0.14577E+01,-0.14586E+01,
     &-0.14595E+01,-0.14604E+01,-0.14613E+01,-0.14622E+01,-0.14631E+01/

      DATA (BNC07M(I),I=601,700)/
     &-0.14728E+01,-0.14817E+01,-0.14906E+01,-0.14994E+01,-0.15082E+01,
     &-0.15170E+01,-0.15257E+01,-0.15343E+01,-0.15430E+01,-0.15515E+01,
     &-0.15601E+01,-0.15686E+01,-0.15771E+01,-0.15856E+01,-0.15940E+01,
     &-0.16024E+01,-0.16107E+01,-0.16191E+01,-0.16274E+01,-0.16356E+01,
     &-0.16439E+01,-0.16521E+01,-0.16603E+01,-0.16684E+01,-0.16766E+01,
     &-0.16847E+01,-0.16928E+01,-0.17008E+01,-0.17089E+01,-0.17169E+01,
     &-0.17249E+01,-0.17328E+01,-0.17408E+01,-0.17487E+01,-0.17566E+01,
     &-0.17645E+01,-0.17723E+01,-0.17802E+01,-0.17880E+01,-0.17958E+01,
     &-0.18036E+01,-0.18114E+01,-0.18191E+01,-0.18268E+01,-0.18346E+01,
     &-0.18423E+01,-0.18499E+01,-0.18576E+01,-0.18652E+01,-0.18729E+01,
     &-0.18805E+01,-0.18881E+01,-0.18957E+01,-0.19032E+01,-0.19108E+01,
     &-0.19183E+01,-0.19258E+01,-0.19334E+01,-0.19409E+01,-0.19483E+01,
     &-0.19558E+01,-0.19633E+01,-0.19707E+01,-0.19781E+01,-0.19855E+01,
     &-0.19929E+01,-0.20003E+01,-0.20077E+01,-0.20151E+01,-0.20224E+01,
     &-0.20298E+01,-0.20371E+01,-0.20444E+01,-0.20517E+01,-0.20590E+01,
     &-0.20663E+01,-0.20736E+01,-0.20808E+01,-0.20881E+01,-0.20953E+01,
     &-0.21026E+01,-0.21098E+01,-0.21170E+01,-0.21242E+01,-0.21314E+01,
     &-0.21386E+01,-0.21457E+01,-0.21529E+01,-0.21601E+01,-0.21672E+01,
     &-0.21743E+01,-0.21815E+01,-0.21886E+01,-0.21957E+01,-0.22028E+01,
     &-0.22099E+01,-0.22169E+01,-0.22240E+01,-0.22311E+01,-0.22381E+01/

      DATA (BNC07M(I),I=701,741)/
     &-0.22452E+01,-0.22522E+01,-0.22593E+01,-0.22663E+01,-0.22733E+01,
     &-0.22803E+01,-0.22873E+01,-0.22943E+01,-0.23013E+01,-0.23082E+01,
     &-0.23152E+01,-0.23222E+01,-0.23291E+01,-0.23361E+01,-0.23430E+01,
     &-0.23499E+01,-0.23569E+01,-0.23638E+01,-0.23707E+01,-0.23776E+01,
     &-0.23845E+01,-0.23914E+01,-0.23983E+01,-0.24052E+01,-0.24120E+01,
     &-0.24189E+01,-0.24258E+01,-0.24326E+01,-0.24395E+01,-0.24463E+01,
     &-0.24531E+01,-0.24600E+01,-0.24668E+01,-0.24736E+01,-0.24804E+01,
     &-0.24872E+01,-0.24940E+01,-0.25008E+01,-0.25076E+01,-0.25144E+01,
     &-0.25211E+01
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.45623E-01,-0.90075E-01,-0.10774E+00,-0.11790E+00,-0.12423E+00,
     &-0.12817E+00,-0.13046E+00,-0.13152E+00,-0.13161E+00,-0.13092E+00,
     &-0.12958E+00,-0.12767E+00,-0.12527E+00,-0.12245E+00,-0.11923E+00,
     &-0.11566E+00,-0.11177E+00,-0.10758E+00,-0.10312E+00,-0.98394E-01,
     &-0.93430E-01,-0.88237E-01,-0.82830E-01,-0.77218E-01,-0.71412E-01,
     &-0.65422E-01,-0.59256E-01,-0.52921E-01,-0.46426E-01,-0.39777E-01,
     &-0.32981E-01,-0.26044E-01,-0.18972E-01,-0.11771E-01,-0.44450E-02,
     & 0.29997E-02, 0.10558E-01, 0.18226E-01, 0.25999E-01, 0.33872E-01,
     & 0.41842E-01, 0.49903E-01, 0.58053E-01, 0.66287E-01, 0.74602E-01,
     & 0.82995E-01, 0.91462E-01, 0.10000E+00, 0.10861E+00, 0.11728E+00,
     & 0.12601E+00, 0.13481E+00, 0.14366E+00, 0.15257E+00, 0.16153E+00,
     & 0.17055E+00, 0.17962E+00, 0.18873E+00, 0.19790E+00, 0.20711E+00,
     & 0.21637E+00, 0.22567E+00, 0.23502E+00, 0.24442E+00, 0.25386E+00,
     & 0.26334E+00, 0.27287E+00, 0.28244E+00, 0.29205E+00, 0.30172E+00,
     & 0.31142E+00, 0.32118E+00, 0.33098E+00, 0.34083E+00, 0.35072E+00,
     & 0.36067E+00, 0.37066E+00, 0.38071E+00, 0.39081E+00, 0.40096E+00,
     & 0.41116E+00, 0.42141E+00, 0.43172E+00, 0.44208E+00, 0.45250E+00,
     & 0.46297E+00, 0.47349E+00, 0.48407E+00, 0.49470E+00, 0.50539E+00,
     & 0.51613E+00, 0.52692E+00, 0.53777E+00, 0.54867E+00, 0.55961E+00,
     & 0.57061E+00, 0.58165E+00, 0.59274E+00, 0.60387E+00, 0.61505E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.62627E+00, 0.63752E+00, 0.64882E+00, 0.66015E+00, 0.67151E+00,
     & 0.68291E+00, 0.69433E+00, 0.70578E+00, 0.71726E+00, 0.72876E+00,
     & 0.74028E+00, 0.75182E+00, 0.76338E+00, 0.77495E+00, 0.78653E+00,
     & 0.79813E+00, 0.80973E+00, 0.82134E+00, 0.83296E+00, 0.84458E+00,
     & 0.85620E+00, 0.86782E+00, 0.87945E+00, 0.89107E+00, 0.90268E+00,
     & 0.91429E+00, 0.92589E+00, 0.93749E+00, 0.94908E+00, 0.96066E+00,
     & 0.97222E+00, 0.98378E+00, 0.99532E+00, 0.10069E+01, 0.10184E+01,
     & 0.10299E+01, 0.10414E+01, 0.10528E+01, 0.10643E+01, 0.10757E+01,
     & 0.10871E+01, 0.10985E+01, 0.11099E+01, 0.11213E+01, 0.11326E+01,
     & 0.11440E+01, 0.11553E+01, 0.11666E+01, 0.11778E+01, 0.11891E+01,
     & 0.12003E+01, 0.12115E+01, 0.12227E+01, 0.12338E+01, 0.12450E+01,
     & 0.12561E+01, 0.12672E+01, 0.12783E+01, 0.12893E+01, 0.13004E+01,
     & 0.13114E+01, 0.13224E+01, 0.13333E+01, 0.13443E+01, 0.13552E+01,
     & 0.13661E+01, 0.13770E+01, 0.13878E+01, 0.13987E+01, 0.14095E+01,
     & 0.14203E+01, 0.14310E+01, 0.14418E+01, 0.14525E+01, 0.14632E+01,
     & 0.14739E+01, 0.14845E+01, 0.14951E+01, 0.15058E+01, 0.15163E+01,
     & 0.15269E+01, 0.15374E+01, 0.15479E+01, 0.15584E+01, 0.15689E+01,
     & 0.15794E+01, 0.15898E+01, 0.16002E+01, 0.16105E+01, 0.16209E+01,
     & 0.16312E+01, 0.16415E+01, 0.16518E+01, 0.16621E+01, 0.16723E+01,
     & 0.16826E+01, 0.16928E+01, 0.17029E+01, 0.17131E+01, 0.17232E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.17333E+01, 0.17434E+01, 0.17535E+01, 0.17635E+01, 0.17735E+01,
     & 0.17835E+01, 0.17935E+01, 0.18034E+01, 0.18134E+01, 0.18233E+01,
     & 0.18332E+01, 0.18430E+01, 0.18529E+01, 0.18627E+01, 0.18725E+01,
     & 0.18823E+01, 0.18920E+01, 0.19017E+01, 0.19115E+01, 0.19211E+01,
     & 0.19308E+01, 0.19405E+01, 0.19501E+01, 0.19597E+01, 0.19693E+01,
     & 0.19788E+01, 0.19884E+01, 0.19979E+01, 0.20074E+01, 0.20169E+01,
     & 0.20263E+01, 0.20358E+01, 0.20452E+01, 0.20546E+01, 0.20640E+01,
     & 0.20733E+01, 0.20827E+01, 0.20920E+01, 0.21013E+01, 0.21105E+01,
     & 0.21198E+01, 0.21290E+01, 0.21383E+01, 0.21474E+01, 0.21566E+01,
     & 0.21658E+01, 0.21749E+01, 0.21840E+01, 0.21931E+01, 0.22022E+01,
     & 0.22113E+01, 0.22203E+01, 0.22293E+01, 0.22383E+01, 0.22473E+01,
     & 0.22563E+01, 0.22652E+01, 0.22741E+01, 0.22830E+01, 0.22919E+01,
     & 0.23008E+01, 0.23096E+01, 0.23185E+01, 0.23273E+01, 0.23361E+01,
     & 0.23448E+01, 0.23536E+01, 0.23623E+01, 0.23710E+01, 0.23797E+01,
     & 0.23884E+01, 0.23971E+01, 0.24057E+01, 0.24144E+01, 0.24230E+01,
     & 0.24316E+01, 0.24401E+01, 0.24487E+01, 0.24572E+01, 0.24657E+01,
     & 0.24742E+01, 0.24827E+01, 0.24912E+01, 0.24997E+01, 0.25081E+01,
     & 0.25165E+01, 0.25249E+01, 0.25333E+01, 0.25416E+01, 0.25500E+01,
     & 0.25583E+01, 0.25666E+01, 0.25749E+01, 0.25832E+01, 0.25915E+01,
     & 0.25997E+01, 0.26080E+01, 0.26162E+01, 0.26244E+01, 0.26326E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.26407E+01, 0.26489E+01, 0.26570E+01, 0.26651E+01, 0.26732E+01,
     & 0.26813E+01, 0.26894E+01, 0.26974E+01, 0.27055E+01, 0.27135E+01,
     & 0.27215E+01, 0.27295E+01, 0.27374E+01, 0.27454E+01, 0.27533E+01,
     & 0.27613E+01, 0.27692E+01, 0.27771E+01, 0.27850E+01, 0.27928E+01,
     & 0.28007E+01, 0.28085E+01, 0.28163E+01, 0.28241E+01, 0.28319E+01,
     & 0.28397E+01, 0.28475E+01, 0.28552E+01, 0.28629E+01, 0.28706E+01,
     & 0.28783E+01, 0.28860E+01, 0.28937E+01, 0.29014E+01, 0.29090E+01,
     & 0.29166E+01, 0.29242E+01, 0.29318E+01, 0.29394E+01, 0.29470E+01,
     & 0.29545E+01, 0.29621E+01, 0.29696E+01, 0.29771E+01, 0.29846E+01,
     & 0.29921E+01, 0.29996E+01, 0.30070E+01, 0.30145E+01, 0.30219E+01,
     & 0.30293E+01, 0.30367E+01, 0.30441E+01, 0.30515E+01, 0.30589E+01,
     & 0.30662E+01, 0.30735E+01, 0.30809E+01, 0.30882E+01, 0.30955E+01,
     & 0.31027E+01, 0.31100E+01, 0.31173E+01, 0.31245E+01, 0.31317E+01,
     & 0.31390E+01, 0.31462E+01, 0.31534E+01, 0.31605E+01, 0.31677E+01,
     & 0.31748E+01, 0.31820E+01, 0.31891E+01, 0.31962E+01, 0.32033E+01,
     & 0.32104E+01, 0.32175E+01, 0.32245E+01, 0.32316E+01, 0.32386E+01,
     & 0.32457E+01, 0.32527E+01, 0.32597E+01, 0.32667E+01, 0.32736E+01,
     & 0.32806E+01, 0.32875E+01, 0.32945E+01, 0.33014E+01, 0.33083E+01,
     & 0.33152E+01, 0.33221E+01, 0.33290E+01, 0.33359E+01, 0.33427E+01,
     & 0.33496E+01, 0.33564E+01, 0.33632E+01, 0.33700E+01, 0.33768E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.33836E+01, 0.33904E+01, 0.33971E+01, 0.34039E+01, 0.34106E+01,
     & 0.34174E+01, 0.34241E+01, 0.34308E+01, 0.34375E+01, 0.34442E+01,
     & 0.34508E+01, 0.34575E+01, 0.34641E+01, 0.34708E+01, 0.34774E+01,
     & 0.34840E+01, 0.34906E+01, 0.34972E+01, 0.35038E+01, 0.35104E+01,
     & 0.35169E+01, 0.35235E+01, 0.35300E+01, 0.35365E+01, 0.35431E+01,
     & 0.35496E+01, 0.35561E+01, 0.35625E+01, 0.35690E+01, 0.35755E+01,
     & 0.35819E+01, 0.35884E+01, 0.35948E+01, 0.36012E+01, 0.36076E+01,
     & 0.36140E+01, 0.36204E+01, 0.36268E+01, 0.36332E+01, 0.36395E+01,
     & 0.36459E+01, 0.36522E+01, 0.36585E+01, 0.36649E+01, 0.36712E+01,
     & 0.36775E+01, 0.36838E+01, 0.36900E+01, 0.36963E+01, 0.37026E+01,
     & 0.37088E+01, 0.37150E+01, 0.37213E+01, 0.37275E+01, 0.37337E+01,
     & 0.37399E+01, 0.37461E+01, 0.37523E+01, 0.37584E+01, 0.37646E+01,
     & 0.37707E+01, 0.37769E+01, 0.37830E+01, 0.37891E+01, 0.37952E+01,
     & 0.38013E+01, 0.38074E+01, 0.38135E+01, 0.38196E+01, 0.38256E+01,
     & 0.38317E+01, 0.38377E+01, 0.38438E+01, 0.38498E+01, 0.38558E+01,
     & 0.38618E+01, 0.38678E+01, 0.38738E+01, 0.38798E+01, 0.38858E+01,
     & 0.38917E+01, 0.38977E+01, 0.39036E+01, 0.39096E+01, 0.39155E+01,
     & 0.39214E+01, 0.39273E+01, 0.39332E+01, 0.39391E+01, 0.39450E+01,
     & 0.39509E+01, 0.39567E+01, 0.39626E+01, 0.39684E+01, 0.39743E+01,
     & 0.39801E+01, 0.39859E+01, 0.39917E+01, 0.39975E+01, 0.40033E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.40091E+01, 0.40149E+01, 0.40206E+01, 0.40264E+01, 0.40321E+01,
     & 0.40379E+01, 0.40436E+01, 0.40493E+01, 0.40551E+01, 0.40608E+01,
     & 0.40665E+01, 0.40722E+01, 0.40778E+01, 0.40835E+01, 0.40892E+01,
     & 0.40948E+01, 0.41005E+01, 0.41061E+01, 0.41118E+01, 0.41174E+01,
     & 0.41230E+01, 0.41286E+01, 0.41342E+01, 0.41398E+01, 0.41454E+01,
     & 0.41510E+01, 0.41565E+01, 0.41621E+01, 0.41677E+01, 0.41732E+01,
     & 0.41787E+01, 0.41843E+01, 0.41898E+01, 0.41953E+01, 0.42008E+01,
     & 0.42063E+01, 0.42118E+01, 0.42173E+01, 0.42228E+01, 0.42282E+01,
     & 0.42337E+01, 0.42391E+01, 0.42446E+01, 0.42500E+01, 0.42554E+01,
     & 0.42609E+01, 0.42663E+01, 0.42717E+01, 0.42771E+01, 0.42825E+01,
     & 0.42878E+01, 0.42932E+01, 0.42986E+01, 0.43040E+01, 0.43093E+01,
     & 0.43146E+01, 0.43200E+01, 0.43253E+01, 0.43306E+01, 0.43360E+01,
     & 0.43413E+01, 0.43466E+01, 0.43519E+01, 0.43571E+01, 0.43624E+01,
     & 0.43677E+01, 0.43730E+01, 0.43782E+01, 0.43835E+01, 0.43887E+01,
     & 0.43940E+01, 0.43992E+01, 0.44044E+01, 0.44096E+01, 0.44148E+01,
     & 0.44200E+01, 0.44252E+01, 0.44304E+01, 0.44356E+01, 0.44408E+01,
     & 0.44459E+01, 0.44511E+01, 0.44562E+01, 0.44614E+01, 0.44665E+01,
     & 0.44717E+01, 0.44768E+01, 0.44819E+01, 0.44870E+01, 0.44921E+01,
     & 0.44972E+01, 0.45023E+01, 0.45074E+01, 0.45125E+01, 0.45175E+01,
     & 0.45226E+01, 0.45277E+01, 0.45327E+01, 0.45377E+01, 0.45428E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.45968E+01, 0.46461E+01, 0.46948E+01, 0.47428E+01, 0.47903E+01,
     & 0.48372E+01, 0.48835E+01, 0.49293E+01, 0.49745E+01, 0.50191E+01,
     & 0.50633E+01, 0.51069E+01, 0.51500E+01, 0.51927E+01, 0.52349E+01,
     & 0.52765E+01, 0.53178E+01, 0.53586E+01, 0.53989E+01, 0.54388E+01,
     & 0.54783E+01, 0.55174E+01, 0.55560E+01, 0.55943E+01, 0.56322E+01,
     & 0.56696E+01, 0.57068E+01, 0.57435E+01, 0.57799E+01, 0.58159E+01,
     & 0.58515E+01, 0.58869E+01, 0.59218E+01, 0.59565E+01, 0.59908E+01,
     & 0.60248E+01, 0.60585E+01, 0.60919E+01, 0.61249E+01, 0.61577E+01,
     & 0.61902E+01, 0.62224E+01, 0.62543E+01, 0.62859E+01, 0.63172E+01,
     & 0.63483E+01, 0.63791E+01, 0.64096E+01, 0.64399E+01, 0.64699E+01,
     & 0.64997E+01, 0.65292E+01, 0.65585E+01, 0.65875E+01, 0.66163E+01,
     & 0.66449E+01, 0.66733E+01, 0.67014E+01, 0.67293E+01, 0.67569E+01,
     & 0.67844E+01, 0.68116E+01, 0.68387E+01, 0.68655E+01, 0.68921E+01,
     & 0.69185E+01, 0.69447E+01, 0.69708E+01, 0.69966E+01, 0.70222E+01,
     & 0.70477E+01, 0.70729E+01, 0.70980E+01, 0.71229E+01, 0.71476E+01,
     & 0.71721E+01, 0.71965E+01, 0.72207E+01, 0.72447E+01, 0.72686E+01,
     & 0.72922E+01, 0.73158E+01, 0.73391E+01, 0.73623E+01, 0.73854E+01,
     & 0.74082E+01, 0.74310E+01, 0.74536E+01, 0.74760E+01, 0.74983E+01,
     & 0.75204E+01, 0.75424E+01, 0.75642E+01, 0.75859E+01, 0.76075E+01,
     & 0.76289E+01, 0.76502E+01, 0.76714E+01, 0.76924E+01, 0.77133E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.77340E+01, 0.77546E+01, 0.77751E+01, 0.77955E+01, 0.78157E+01,
     & 0.78358E+01, 0.78558E+01, 0.78757E+01, 0.78955E+01, 0.79151E+01,
     & 0.79346E+01, 0.79540E+01, 0.79733E+01, 0.79925E+01, 0.80115E+01,
     & 0.80305E+01, 0.80493E+01, 0.80680E+01, 0.80867E+01, 0.81052E+01,
     & 0.81236E+01, 0.81419E+01, 0.81601E+01, 0.81782E+01, 0.81961E+01,
     & 0.82140E+01, 0.82318E+01, 0.82495E+01, 0.82671E+01, 0.82846E+01,
     & 0.83020E+01, 0.83193E+01, 0.83365E+01, 0.83536E+01, 0.83706E+01,
     & 0.83876E+01, 0.84044E+01, 0.84212E+01, 0.84378E+01, 0.84544E+01,
     & 0.84709E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.47941E-01,-0.10187E+00,-0.12775E+00,-0.14554E+00,-0.15916E+00,
     &-0.17016E+00,-0.17935E+00,-0.18720E+00,-0.19401E+00,-0.19998E+00,
     &-0.20526E+00,-0.20996E+00,-0.21416E+00,-0.21792E+00,-0.22130E+00,
     &-0.22433E+00,-0.22706E+00,-0.22950E+00,-0.23169E+00,-0.23364E+00,
     &-0.23537E+00,-0.23689E+00,-0.23822E+00,-0.23937E+00,-0.24035E+00,
     &-0.24117E+00,-0.24183E+00,-0.24235E+00,-0.24273E+00,-0.24298E+00,
     &-0.24310E+00,-0.24310E+00,-0.24299E+00,-0.24276E+00,-0.24242E+00,
     &-0.24198E+00,-0.24144E+00,-0.24080E+00,-0.24008E+00,-0.23926E+00,
     &-0.23835E+00,-0.23736E+00,-0.23630E+00,-0.23515E+00,-0.23393E+00,
     &-0.23264E+00,-0.23128E+00,-0.22985E+00,-0.22835E+00,-0.22680E+00,
     &-0.22518E+00,-0.22350E+00,-0.22177E+00,-0.21998E+00,-0.21814E+00,
     &-0.21625E+00,-0.21431E+00,-0.21232E+00,-0.21028E+00,-0.20820E+00,
     &-0.20607E+00,-0.20391E+00,-0.20170E+00,-0.19945E+00,-0.19716E+00,
     &-0.19483E+00,-0.19246E+00,-0.19006E+00,-0.18762E+00,-0.18514E+00,
     &-0.18263E+00,-0.18009E+00,-0.17751E+00,-0.17490E+00,-0.17225E+00,
     &-0.16957E+00,-0.16686E+00,-0.16412E+00,-0.16135E+00,-0.15854E+00,
     &-0.15571E+00,-0.15284E+00,-0.14994E+00,-0.14702E+00,-0.14406E+00,
     &-0.14108E+00,-0.13806E+00,-0.13502E+00,-0.13195E+00,-0.12885E+00,
     &-0.12573E+00,-0.12257E+00,-0.11940E+00,-0.11619E+00,-0.11297E+00,
     &-0.10972E+00,-0.10644E+00,-0.10315E+00,-0.99827E-01,-0.96489E-01/

      DATA (BNC09M(I),I=101,200)/
     &-0.93130E-01,-0.89752E-01,-0.86357E-01,-0.82944E-01,-0.79515E-01,
     &-0.76070E-01,-0.72611E-01,-0.69138E-01,-0.65653E-01,-0.62156E-01,
     &-0.58649E-01,-0.55131E-01,-0.51605E-01,-0.48070E-01,-0.44529E-01,
     &-0.40980E-01,-0.37426E-01,-0.33868E-01,-0.30305E-01,-0.26738E-01,
     &-0.23169E-01,-0.19598E-01,-0.16026E-01,-0.12452E-01,-0.88787E-02,
     &-0.53053E-02,-0.17329E-02, 0.18381E-02, 0.54075E-02, 0.89747E-02,
     & 0.12539E-01, 0.16101E-01, 0.19659E-01, 0.23213E-01, 0.26764E-01,
     & 0.30310E-01, 0.33851E-01, 0.37388E-01, 0.40919E-01, 0.44446E-01,
     & 0.47966E-01, 0.51481E-01, 0.54990E-01, 0.58493E-01, 0.61990E-01,
     & 0.65481E-01, 0.68964E-01, 0.72442E-01, 0.75912E-01, 0.79376E-01,
     & 0.82833E-01, 0.86282E-01, 0.89725E-01, 0.93161E-01, 0.96589E-01,
     & 0.10001E+00, 0.10342E+00, 0.10683E+00, 0.11023E+00, 0.11362E+00,
     & 0.11700E+00, 0.12038E+00, 0.12375E+00, 0.12711E+00, 0.13046E+00,
     & 0.13381E+00, 0.13714E+00, 0.14047E+00, 0.14379E+00, 0.14711E+00,
     & 0.15041E+00, 0.15371E+00, 0.15700E+00, 0.16028E+00, 0.16356E+00,
     & 0.16683E+00, 0.17008E+00, 0.17334E+00, 0.17658E+00, 0.17981E+00,
     & 0.18304E+00, 0.18626E+00, 0.18947E+00, 0.19268E+00, 0.19587E+00,
     & 0.19906E+00, 0.20224E+00, 0.20541E+00, 0.20858E+00, 0.21173E+00,
     & 0.21488E+00, 0.21802E+00, 0.22116E+00, 0.22428E+00, 0.22740E+00,
     & 0.23051E+00, 0.23361E+00, 0.23671E+00, 0.23979E+00, 0.24287E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.24594E+00, 0.24901E+00, 0.25206E+00, 0.25511E+00, 0.25815E+00,
     & 0.26119E+00, 0.26422E+00, 0.26723E+00, 0.27025E+00, 0.27325E+00,
     & 0.27625E+00, 0.27924E+00, 0.28222E+00, 0.28519E+00, 0.28816E+00,
     & 0.29112E+00, 0.29407E+00, 0.29702E+00, 0.29996E+00, 0.30289E+00,
     & 0.30581E+00, 0.30873E+00, 0.31164E+00, 0.31454E+00, 0.31743E+00,
     & 0.32032E+00, 0.32320E+00, 0.32608E+00, 0.32895E+00, 0.33181E+00,
     & 0.33466E+00, 0.33751E+00, 0.34035E+00, 0.34318E+00, 0.34601E+00,
     & 0.34883E+00, 0.35164E+00, 0.35445E+00, 0.35725E+00, 0.36004E+00,
     & 0.36282E+00, 0.36560E+00, 0.36838E+00, 0.37114E+00, 0.37390E+00,
     & 0.37666E+00, 0.37940E+00, 0.38214E+00, 0.38488E+00, 0.38760E+00,
     & 0.39033E+00, 0.39304E+00, 0.39575E+00, 0.39845E+00, 0.40115E+00,
     & 0.40384E+00, 0.40652E+00, 0.40920E+00, 0.41187E+00, 0.41453E+00,
     & 0.41719E+00, 0.41985E+00, 0.42249E+00, 0.42513E+00, 0.42777E+00,
     & 0.43040E+00, 0.43302E+00, 0.43563E+00, 0.43825E+00, 0.44085E+00,
     & 0.44345E+00, 0.44604E+00, 0.44863E+00, 0.45121E+00, 0.45379E+00,
     & 0.45636E+00, 0.45892E+00, 0.46148E+00, 0.46403E+00, 0.46658E+00,
     & 0.46912E+00, 0.47165E+00, 0.47419E+00, 0.47671E+00, 0.47923E+00,
     & 0.48174E+00, 0.48425E+00, 0.48675E+00, 0.48925E+00, 0.49174E+00,
     & 0.49423E+00, 0.49671E+00, 0.49918E+00, 0.50165E+00, 0.50412E+00,
     & 0.50658E+00, 0.50903E+00, 0.51148E+00, 0.51392E+00, 0.51636E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.51879E+00, 0.52122E+00, 0.52364E+00, 0.52606E+00, 0.52847E+00,
     & 0.53088E+00, 0.53328E+00, 0.53568E+00, 0.53807E+00, 0.54046E+00,
     & 0.54284E+00, 0.54522E+00, 0.54759E+00, 0.54996E+00, 0.55232E+00,
     & 0.55467E+00, 0.55703E+00, 0.55937E+00, 0.56172E+00, 0.56405E+00,
     & 0.56639E+00, 0.56871E+00, 0.57104E+00, 0.57336E+00, 0.57567E+00,
     & 0.57798E+00, 0.58028E+00, 0.58258E+00, 0.58488E+00, 0.58717E+00,
     & 0.58945E+00, 0.59173E+00, 0.59401E+00, 0.59628E+00, 0.59855E+00,
     & 0.60081E+00, 0.60307E+00, 0.60532E+00, 0.60757E+00, 0.60981E+00,
     & 0.61205E+00, 0.61429E+00, 0.61652E+00, 0.61874E+00, 0.62097E+00,
     & 0.62318E+00, 0.62540E+00, 0.62761E+00, 0.62981E+00, 0.63201E+00,
     & 0.63421E+00, 0.63640E+00, 0.63859E+00, 0.64077E+00, 0.64295E+00,
     & 0.64512E+00, 0.64729E+00, 0.64946E+00, 0.65162E+00, 0.65378E+00,
     & 0.65593E+00, 0.65808E+00, 0.66022E+00, 0.66236E+00, 0.66450E+00,
     & 0.66663E+00, 0.66876E+00, 0.67089E+00, 0.67301E+00, 0.67512E+00,
     & 0.67724E+00, 0.67935E+00, 0.68145E+00, 0.68355E+00, 0.68565E+00,
     & 0.68774E+00, 0.68983E+00, 0.69191E+00, 0.69399E+00, 0.69607E+00,
     & 0.69814E+00, 0.70021E+00, 0.70228E+00, 0.70434E+00, 0.70639E+00,
     & 0.70845E+00, 0.71050E+00, 0.71254E+00, 0.71459E+00, 0.71662E+00,
     & 0.71866E+00, 0.72069E+00, 0.72272E+00, 0.72474E+00, 0.72676E+00,
     & 0.72878E+00, 0.73079E+00, 0.73280E+00, 0.73480E+00, 0.73680E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.73880E+00, 0.74079E+00, 0.74279E+00, 0.74477E+00, 0.74675E+00,
     & 0.74873E+00, 0.75071E+00, 0.75268E+00, 0.75465E+00, 0.75662E+00,
     & 0.75858E+00, 0.76054E+00, 0.76249E+00, 0.76444E+00, 0.76639E+00,
     & 0.76833E+00, 0.77028E+00, 0.77221E+00, 0.77415E+00, 0.77608E+00,
     & 0.77801E+00, 0.77993E+00, 0.78185E+00, 0.78377E+00, 0.78568E+00,
     & 0.78759E+00, 0.78950E+00, 0.79140E+00, 0.79330E+00, 0.79520E+00,
     & 0.79709E+00, 0.79898E+00, 0.80087E+00, 0.80275E+00, 0.80463E+00,
     & 0.80651E+00, 0.80838E+00, 0.81025E+00, 0.81212E+00, 0.81399E+00,
     & 0.81585E+00, 0.81770E+00, 0.81956E+00, 0.82141E+00, 0.82326E+00,
     & 0.82510E+00, 0.82694E+00, 0.82878E+00, 0.83062E+00, 0.83245E+00,
     & 0.83428E+00, 0.83611E+00, 0.83793E+00, 0.83975E+00, 0.84157E+00,
     & 0.84338E+00, 0.84519E+00, 0.84700E+00, 0.84880E+00, 0.85061E+00,
     & 0.85240E+00, 0.85420E+00, 0.85599E+00, 0.85778E+00, 0.85957E+00,
     & 0.86135E+00, 0.86313E+00, 0.86491E+00, 0.86669E+00, 0.86846E+00,
     & 0.87023E+00, 0.87199E+00, 0.87376E+00, 0.87552E+00, 0.87727E+00,
     & 0.87903E+00, 0.88078E+00, 0.88253E+00, 0.88427E+00, 0.88602E+00,
     & 0.88776E+00, 0.88949E+00, 0.89123E+00, 0.89296E+00, 0.89469E+00,
     & 0.89641E+00, 0.89814E+00, 0.89986E+00, 0.90157E+00, 0.90329E+00,
     & 0.90500E+00, 0.90671E+00, 0.90842E+00, 0.91012E+00, 0.91182E+00,
     & 0.91352E+00, 0.91521E+00, 0.91691E+00, 0.91860E+00, 0.92028E+00/

      DATA (BNC09M(I),I=501,600)/
     & 0.92197E+00, 0.92365E+00, 0.92533E+00, 0.92700E+00, 0.92868E+00,
     & 0.93035E+00, 0.93202E+00, 0.93368E+00, 0.93535E+00, 0.93701E+00,
     & 0.93867E+00, 0.94032E+00, 0.94197E+00, 0.94362E+00, 0.94527E+00,
     & 0.94692E+00, 0.94856E+00, 0.95020E+00, 0.95184E+00, 0.95347E+00,
     & 0.95510E+00, 0.95673E+00, 0.95836E+00, 0.95998E+00, 0.96160E+00,
     & 0.96322E+00, 0.96484E+00, 0.96646E+00, 0.96807E+00, 0.96968E+00,
     & 0.97128E+00, 0.97289E+00, 0.97449E+00, 0.97609E+00, 0.97769E+00,
     & 0.97928E+00, 0.98087E+00, 0.98246E+00, 0.98405E+00, 0.98563E+00,
     & 0.98722E+00, 0.98880E+00, 0.99037E+00, 0.99195E+00, 0.99352E+00,
     & 0.99509E+00, 0.99666E+00, 0.99822E+00, 0.99979E+00, 0.10013E+01,
     & 0.10029E+01, 0.10045E+01, 0.10060E+01, 0.10076E+01, 0.10091E+01,
     & 0.10107E+01, 0.10122E+01, 0.10137E+01, 0.10153E+01, 0.10168E+01,
     & 0.10184E+01, 0.10199E+01, 0.10214E+01, 0.10230E+01, 0.10245E+01,
     & 0.10260E+01, 0.10275E+01, 0.10290E+01, 0.10306E+01, 0.10321E+01,
     & 0.10336E+01, 0.10351E+01, 0.10366E+01, 0.10381E+01, 0.10396E+01,
     & 0.10411E+01, 0.10426E+01, 0.10441E+01, 0.10456E+01, 0.10471E+01,
     & 0.10486E+01, 0.10501E+01, 0.10516E+01, 0.10530E+01, 0.10545E+01,
     & 0.10560E+01, 0.10575E+01, 0.10590E+01, 0.10604E+01, 0.10619E+01,
     & 0.10634E+01, 0.10648E+01, 0.10663E+01, 0.10678E+01, 0.10692E+01,
     & 0.10707E+01, 0.10721E+01, 0.10736E+01, 0.10750E+01, 0.10765E+01/

      DATA (BNC09M(I),I=601,700)/
     & 0.10920E+01, 0.11062E+01, 0.11202E+01, 0.11339E+01, 0.11475E+01,
     & 0.11609E+01, 0.11741E+01, 0.11872E+01, 0.12000E+01, 0.12127E+01,
     & 0.12252E+01, 0.12376E+01, 0.12498E+01, 0.12619E+01, 0.12738E+01,
     & 0.12855E+01, 0.12971E+01, 0.13086E+01, 0.13199E+01, 0.13311E+01,
     & 0.13421E+01, 0.13531E+01, 0.13638E+01, 0.13745E+01, 0.13850E+01,
     & 0.13954E+01, 0.14057E+01, 0.14159E+01, 0.14259E+01, 0.14359E+01,
     & 0.14457E+01, 0.14554E+01, 0.14650E+01, 0.14745E+01, 0.14839E+01,
     & 0.14932E+01, 0.15024E+01, 0.15115E+01, 0.15205E+01, 0.15294E+01,
     & 0.15382E+01, 0.15469E+01, 0.15555E+01, 0.15641E+01, 0.15725E+01,
     & 0.15808E+01, 0.15891E+01, 0.15973E+01, 0.16054E+01, 0.16134E+01,
     & 0.16213E+01, 0.16292E+01, 0.16370E+01, 0.16447E+01, 0.16523E+01,
     & 0.16598E+01, 0.16673E+01, 0.16747E+01, 0.16820E+01, 0.16893E+01,
     & 0.16965E+01, 0.17036E+01, 0.17106E+01, 0.17176E+01, 0.17245E+01,
     & 0.17314E+01, 0.17381E+01, 0.17449E+01, 0.17515E+01, 0.17581E+01,
     & 0.17646E+01, 0.17711E+01, 0.17775E+01, 0.17839E+01, 0.17902E+01,
     & 0.17964E+01, 0.18026E+01, 0.18087E+01, 0.18148E+01, 0.18208E+01,
     & 0.18268E+01, 0.18327E+01, 0.18385E+01, 0.18443E+01, 0.18501E+01,
     & 0.18558E+01, 0.18614E+01, 0.18671E+01, 0.18726E+01, 0.18781E+01,
     & 0.18836E+01, 0.18890E+01, 0.18943E+01, 0.18997E+01, 0.19049E+01,
     & 0.19102E+01, 0.19153E+01, 0.19205E+01, 0.19256E+01, 0.19306E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19356E+01, 0.19406E+01, 0.19455E+01, 0.19504E+01, 0.19552E+01,
     & 0.19600E+01, 0.19648E+01, 0.19695E+01, 0.19742E+01, 0.19788E+01,
     & 0.19835E+01, 0.19880E+01, 0.19925E+01, 0.19970E+01, 0.20015E+01,
     & 0.20059E+01, 0.20103E+01, 0.20146E+01, 0.20190E+01, 0.20232E+01,
     & 0.20275E+01, 0.20317E+01, 0.20359E+01, 0.20400E+01, 0.20441E+01,
     & 0.20482E+01, 0.20522E+01, 0.20562E+01, 0.20602E+01, 0.20641E+01,
     & 0.20681E+01, 0.20719E+01, 0.20758E+01, 0.20796E+01, 0.20834E+01,
     & 0.20872E+01, 0.20909E+01, 0.20946E+01, 0.20983E+01, 0.21019E+01,
     & 0.21055E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.47292E-01,-0.97810E-01,-0.12025E+00,-0.13463E+00,-0.14486E+00,
     &-0.15252E+00,-0.15842E+00,-0.16303E+00,-0.16665E+00,-0.16950E+00,
     &-0.17170E+00,-0.17339E+00,-0.17464E+00,-0.17551E+00,-0.17606E+00,
     &-0.17633E+00,-0.17636E+00,-0.17617E+00,-0.17579E+00,-0.17524E+00,
     &-0.17454E+00,-0.17370E+00,-0.17274E+00,-0.17167E+00,-0.17050E+00,
     &-0.16924E+00,-0.16790E+00,-0.16648E+00,-0.16500E+00,-0.16345E+00,
     &-0.16185E+00,-0.16020E+00,-0.15851E+00,-0.15677E+00,-0.15499E+00,
     &-0.15319E+00,-0.15135E+00,-0.14948E+00,-0.14759E+00,-0.14568E+00,
     &-0.14374E+00,-0.14179E+00,-0.13982E+00,-0.13784E+00,-0.13585E+00,
     &-0.13384E+00,-0.13183E+00,-0.12981E+00,-0.12778E+00,-0.12574E+00,
     &-0.12369E+00,-0.12165E+00,-0.11959E+00,-0.11753E+00,-0.11547E+00,
     &-0.11341E+00,-0.11134E+00,-0.10926E+00,-0.10719E+00,-0.10510E+00,
     &-0.10302E+00,-0.10093E+00,-0.98835E-01,-0.96735E-01,-0.94630E-01,
     &-0.92519E-01,-0.90401E-01,-0.88277E-01,-0.86145E-01,-0.84004E-01,
     &-0.81855E-01,-0.79696E-01,-0.77527E-01,-0.75348E-01,-0.73157E-01,
     &-0.70954E-01,-0.68738E-01,-0.66510E-01,-0.64268E-01,-0.62011E-01,
     &-0.59741E-01,-0.57455E-01,-0.55153E-01,-0.52836E-01,-0.50503E-01,
     &-0.48154E-01,-0.45788E-01,-0.43406E-01,-0.41006E-01,-0.38591E-01,
     &-0.36158E-01,-0.33709E-01,-0.31243E-01,-0.28761E-01,-0.26264E-01,
     &-0.23750E-01,-0.21221E-01,-0.18676E-01,-0.16117E-01,-0.13544E-01/

      DATA (BNC10M(I),I=101,200)/
     &-0.10957E-01,-0.83559E-02,-0.57424E-02,-0.31166E-02,-0.47900E-03,
     & 0.21698E-02, 0.48293E-02, 0.74989E-02, 0.10178E-01, 0.12866E-01,
     & 0.15562E-01, 0.18266E-01, 0.20977E-01, 0.23695E-01, 0.26419E-01,
     & 0.29149E-01, 0.31883E-01, 0.34622E-01, 0.37365E-01, 0.40112E-01,
     & 0.42862E-01, 0.45614E-01, 0.48369E-01, 0.51126E-01, 0.53884E-01,
     & 0.56643E-01, 0.59403E-01, 0.62164E-01, 0.64925E-01, 0.67685E-01,
     & 0.70446E-01, 0.73206E-01, 0.75965E-01, 0.78723E-01, 0.81480E-01,
     & 0.84236E-01, 0.86990E-01, 0.89742E-01, 0.92493E-01, 0.95241E-01,
     & 0.97988E-01, 0.10073E+00, 0.10347E+00, 0.10621E+00, 0.10895E+00,
     & 0.11169E+00, 0.11442E+00, 0.11715E+00, 0.11987E+00, 0.12260E+00,
     & 0.12532E+00, 0.12803E+00, 0.13075E+00, 0.13346E+00, 0.13617E+00,
     & 0.13887E+00, 0.14157E+00, 0.14427E+00, 0.14697E+00, 0.14966E+00,
     & 0.15235E+00, 0.15503E+00, 0.15771E+00, 0.16039E+00, 0.16307E+00,
     & 0.16574E+00, 0.16840E+00, 0.17107E+00, 0.17373E+00, 0.17639E+00,
     & 0.17904E+00, 0.18169E+00, 0.18433E+00, 0.18698E+00, 0.18961E+00,
     & 0.19225E+00, 0.19488E+00, 0.19751E+00, 0.20013E+00, 0.20275E+00,
     & 0.20537E+00, 0.20798E+00, 0.21059E+00, 0.21319E+00, 0.21579E+00,
     & 0.21839E+00, 0.22098E+00, 0.22357E+00, 0.22615E+00, 0.22873E+00,
     & 0.23131E+00, 0.23388E+00, 0.23645E+00, 0.23902E+00, 0.24158E+00,
     & 0.24414E+00, 0.24669E+00, 0.24924E+00, 0.25178E+00, 0.25433E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.25686E+00, 0.25940E+00, 0.26192E+00, 0.26445E+00, 0.26697E+00,
     & 0.26949E+00, 0.27200E+00, 0.27451E+00, 0.27701E+00, 0.27952E+00,
     & 0.28201E+00, 0.28450E+00, 0.28699E+00, 0.28948E+00, 0.29196E+00,
     & 0.29444E+00, 0.29691E+00, 0.29938E+00, 0.30184E+00, 0.30430E+00,
     & 0.30676E+00, 0.30921E+00, 0.31166E+00, 0.31410E+00, 0.31654E+00,
     & 0.31898E+00, 0.32141E+00, 0.32384E+00, 0.32626E+00, 0.32868E+00,
     & 0.33109E+00, 0.33351E+00, 0.33591E+00, 0.33832E+00, 0.34071E+00,
     & 0.34311E+00, 0.34550E+00, 0.34789E+00, 0.35027E+00, 0.35265E+00,
     & 0.35502E+00, 0.35740E+00, 0.35976E+00, 0.36213E+00, 0.36448E+00,
     & 0.36684E+00, 0.36919E+00, 0.37154E+00, 0.37388E+00, 0.37622E+00,
     & 0.37855E+00, 0.38088E+00, 0.38321E+00, 0.38553E+00, 0.38785E+00,
     & 0.39016E+00, 0.39247E+00, 0.39478E+00, 0.39708E+00, 0.39938E+00,
     & 0.40168E+00, 0.40397E+00, 0.40626E+00, 0.40854E+00, 0.41082E+00,
     & 0.41309E+00, 0.41536E+00, 0.41763E+00, 0.41989E+00, 0.42215E+00,
     & 0.42441E+00, 0.42666E+00, 0.42891E+00, 0.43115E+00, 0.43339E+00,
     & 0.43563E+00, 0.43786E+00, 0.44009E+00, 0.44231E+00, 0.44453E+00,
     & 0.44675E+00, 0.44896E+00, 0.45117E+00, 0.45337E+00, 0.45558E+00,
     & 0.45777E+00, 0.45997E+00, 0.46216E+00, 0.46434E+00, 0.46652E+00,
     & 0.46870E+00, 0.47088E+00, 0.47305E+00, 0.47522E+00, 0.47738E+00,
     & 0.47954E+00, 0.48169E+00, 0.48385E+00, 0.48600E+00, 0.48814E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.49028E+00, 0.49242E+00, 0.49455E+00, 0.49668E+00, 0.49881E+00,
     & 0.50093E+00, 0.50305E+00, 0.50516E+00, 0.50728E+00, 0.50938E+00,
     & 0.51149E+00, 0.51359E+00, 0.51569E+00, 0.51778E+00, 0.51987E+00,
     & 0.52196E+00, 0.52404E+00, 0.52612E+00, 0.52819E+00, 0.53026E+00,
     & 0.53233E+00, 0.53440E+00, 0.53646E+00, 0.53851E+00, 0.54057E+00,
     & 0.54262E+00, 0.54467E+00, 0.54671E+00, 0.54875E+00, 0.55079E+00,
     & 0.55282E+00, 0.55485E+00, 0.55687E+00, 0.55890E+00, 0.56092E+00,
     & 0.56293E+00, 0.56494E+00, 0.56695E+00, 0.56896E+00, 0.57096E+00,
     & 0.57296E+00, 0.57495E+00, 0.57694E+00, 0.57893E+00, 0.58092E+00,
     & 0.58290E+00, 0.58488E+00, 0.58685E+00, 0.58882E+00, 0.59079E+00,
     & 0.59275E+00, 0.59472E+00, 0.59667E+00, 0.59863E+00, 0.60058E+00,
     & 0.60253E+00, 0.60447E+00, 0.60641E+00, 0.60835E+00, 0.61029E+00,
     & 0.61222E+00, 0.61415E+00, 0.61607E+00, 0.61799E+00, 0.61991E+00,
     & 0.62183E+00, 0.62374E+00, 0.62565E+00, 0.62755E+00, 0.62945E+00,
     & 0.63135E+00, 0.63325E+00, 0.63514E+00, 0.63703E+00, 0.63892E+00,
     & 0.64080E+00, 0.64268E+00, 0.64456E+00, 0.64643E+00, 0.64830E+00,
     & 0.65017E+00, 0.65204E+00, 0.65390E+00, 0.65576E+00, 0.65761E+00,
     & 0.65946E+00, 0.66131E+00, 0.66316E+00, 0.66500E+00, 0.66684E+00,
     & 0.66868E+00, 0.67051E+00, 0.67234E+00, 0.67417E+00, 0.67600E+00,
     & 0.67782E+00, 0.67964E+00, 0.68145E+00, 0.68326E+00, 0.68507E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.68688E+00, 0.68868E+00, 0.69049E+00, 0.69228E+00, 0.69408E+00,
     & 0.69587E+00, 0.69766E+00, 0.69944E+00, 0.70123E+00, 0.70301E+00,
     & 0.70479E+00, 0.70656E+00, 0.70833E+00, 0.71010E+00, 0.71187E+00,
     & 0.71363E+00, 0.71539E+00, 0.71715E+00, 0.71890E+00, 0.72065E+00,
     & 0.72240E+00, 0.72415E+00, 0.72589E+00, 0.72763E+00, 0.72937E+00,
     & 0.73110E+00, 0.73283E+00, 0.73456E+00, 0.73629E+00, 0.73801E+00,
     & 0.73973E+00, 0.74145E+00, 0.74316E+00, 0.74487E+00, 0.74658E+00,
     & 0.74829E+00, 0.75000E+00, 0.75170E+00, 0.75339E+00, 0.75509E+00,
     & 0.75678E+00, 0.75847E+00, 0.76016E+00, 0.76185E+00, 0.76353E+00,
     & 0.76521E+00, 0.76689E+00, 0.76856E+00, 0.77023E+00, 0.77190E+00,
     & 0.77357E+00, 0.77523E+00, 0.77689E+00, 0.77855E+00, 0.78021E+00,
     & 0.78186E+00, 0.78351E+00, 0.78516E+00, 0.78680E+00, 0.78844E+00,
     & 0.79009E+00, 0.79172E+00, 0.79336E+00, 0.79499E+00, 0.79662E+00,
     & 0.79825E+00, 0.79987E+00, 0.80149E+00, 0.80311E+00, 0.80473E+00,
     & 0.80635E+00, 0.80796E+00, 0.80957E+00, 0.81117E+00, 0.81278E+00,
     & 0.81438E+00, 0.81598E+00, 0.81758E+00, 0.81917E+00, 0.82076E+00,
     & 0.82235E+00, 0.82394E+00, 0.82553E+00, 0.82711E+00, 0.82869E+00,
     & 0.83027E+00, 0.83184E+00, 0.83341E+00, 0.83498E+00, 0.83655E+00,
     & 0.83812E+00, 0.83968E+00, 0.84124E+00, 0.84280E+00, 0.84435E+00,
     & 0.84591E+00, 0.84746E+00, 0.84901E+00, 0.85055E+00, 0.85210E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.85364E+00, 0.85518E+00, 0.85671E+00, 0.85825E+00, 0.85978E+00,
     & 0.86131E+00, 0.86284E+00, 0.86436E+00, 0.86589E+00, 0.86741E+00,
     & 0.86892E+00, 0.87044E+00, 0.87195E+00, 0.87346E+00, 0.87497E+00,
     & 0.87648E+00, 0.87799E+00, 0.87949E+00, 0.88099E+00, 0.88249E+00,
     & 0.88398E+00, 0.88547E+00, 0.88696E+00, 0.88845E+00, 0.88994E+00,
     & 0.89142E+00, 0.89291E+00, 0.89439E+00, 0.89586E+00, 0.89734E+00,
     & 0.89881E+00, 0.90028E+00, 0.90175E+00, 0.90322E+00, 0.90468E+00,
     & 0.90615E+00, 0.90761E+00, 0.90906E+00, 0.91052E+00, 0.91197E+00,
     & 0.91343E+00, 0.91488E+00, 0.91632E+00, 0.91777E+00, 0.91921E+00,
     & 0.92065E+00, 0.92209E+00, 0.92353E+00, 0.92496E+00, 0.92639E+00,
     & 0.92782E+00, 0.92925E+00, 0.93068E+00, 0.93210E+00, 0.93352E+00,
     & 0.93494E+00, 0.93636E+00, 0.93778E+00, 0.93919E+00, 0.94060E+00,
     & 0.94201E+00, 0.94342E+00, 0.94483E+00, 0.94623E+00, 0.94763E+00,
     & 0.94903E+00, 0.95043E+00, 0.95182E+00, 0.95322E+00, 0.95461E+00,
     & 0.95600E+00, 0.95738E+00, 0.95877E+00, 0.96015E+00, 0.96153E+00,
     & 0.96291E+00, 0.96429E+00, 0.96566E+00, 0.96704E+00, 0.96841E+00,
     & 0.96978E+00, 0.97115E+00, 0.97251E+00, 0.97388E+00, 0.97524E+00,
     & 0.97660E+00, 0.97795E+00, 0.97931E+00, 0.98066E+00, 0.98202E+00,
     & 0.98337E+00, 0.98471E+00, 0.98606E+00, 0.98740E+00, 0.98875E+00,
     & 0.99009E+00, 0.99143E+00, 0.99276E+00, 0.99410E+00, 0.99543E+00/

      DATA (BNC10M(I),I=601,700)/
     & 0.10097E+01, 0.10227E+01, 0.10356E+01, 0.10483E+01, 0.10608E+01,
     & 0.10731E+01, 0.10852E+01, 0.10972E+01, 0.11091E+01, 0.11208E+01,
     & 0.11323E+01, 0.11437E+01, 0.11549E+01, 0.11660E+01, 0.11770E+01,
     & 0.11878E+01, 0.11985E+01, 0.12090E+01, 0.12195E+01, 0.12298E+01,
     & 0.12399E+01, 0.12500E+01, 0.12599E+01, 0.12697E+01, 0.12794E+01,
     & 0.12889E+01, 0.12984E+01, 0.13077E+01, 0.13170E+01, 0.13261E+01,
     & 0.13351E+01, 0.13441E+01, 0.13529E+01, 0.13616E+01, 0.13702E+01,
     & 0.13788E+01, 0.13872E+01, 0.13955E+01, 0.14038E+01, 0.14119E+01,
     & 0.14200E+01, 0.14280E+01, 0.14359E+01, 0.14437E+01, 0.14514E+01,
     & 0.14590E+01, 0.14666E+01, 0.14741E+01, 0.14815E+01, 0.14888E+01,
     & 0.14961E+01, 0.15032E+01, 0.15103E+01, 0.15174E+01, 0.15243E+01,
     & 0.15312E+01, 0.15380E+01, 0.15448E+01, 0.15515E+01, 0.15581E+01,
     & 0.15646E+01, 0.15711E+01, 0.15775E+01, 0.15839E+01, 0.15901E+01,
     & 0.15964E+01, 0.16025E+01, 0.16086E+01, 0.16147E+01, 0.16207E+01,
     & 0.16266E+01, 0.16325E+01, 0.16383E+01, 0.16441E+01, 0.16498E+01,
     & 0.16554E+01, 0.16610E+01, 0.16666E+01, 0.16721E+01, 0.16775E+01,
     & 0.16829E+01, 0.16882E+01, 0.16935E+01, 0.16988E+01, 0.17040E+01,
     & 0.17091E+01, 0.17142E+01, 0.17193E+01, 0.17243E+01, 0.17292E+01,
     & 0.17342E+01, 0.17390E+01, 0.17439E+01, 0.17486E+01, 0.17534E+01,
     & 0.17581E+01, 0.17627E+01, 0.17674E+01, 0.17719E+01, 0.17765E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17810E+01, 0.17854E+01, 0.17898E+01, 0.17942E+01, 0.17985E+01,
     & 0.18028E+01, 0.18071E+01, 0.18113E+01, 0.18155E+01, 0.18196E+01,
     & 0.18238E+01, 0.18278E+01, 0.18319E+01, 0.18359E+01, 0.18399E+01,
     & 0.18438E+01, 0.18477E+01, 0.18516E+01, 0.18554E+01, 0.18592E+01,
     & 0.18630E+01, 0.18667E+01, 0.18704E+01, 0.18741E+01, 0.18777E+01,
     & 0.18814E+01, 0.18849E+01, 0.18885E+01, 0.18920E+01, 0.18955E+01,
     & 0.18990E+01, 0.19024E+01, 0.19058E+01, 0.19092E+01, 0.19125E+01,
     & 0.19158E+01, 0.19191E+01, 0.19224E+01, 0.19256E+01, 0.19288E+01,
     & 0.19320E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.45992E-01,-0.91415E-01,-0.10957E+00,-0.12001E+00,-0.12652E+00,
     &-0.13058E+00,-0.13296E+00,-0.13409E+00,-0.13425E+00,-0.13365E+00,
     &-0.13241E+00,-0.13063E+00,-0.12840E+00,-0.12577E+00,-0.12279E+00,
     &-0.11951E+00,-0.11594E+00,-0.11213E+00,-0.10809E+00,-0.10384E+00,
     &-0.99402E-01,-0.94790E-01,-0.90019E-01,-0.85098E-01,-0.80040E-01,
     &-0.74855E-01,-0.69550E-01,-0.64135E-01,-0.58616E-01,-0.53002E-01,
     &-0.47297E-01,-0.41509E-01,-0.35643E-01,-0.29704E-01,-0.23697E-01,
     &-0.17626E-01,-0.11497E-01,-0.53118E-02, 0.92416E-03, 0.72078E-02,
     & 0.13536E-01, 0.19905E-01, 0.26312E-01, 0.32755E-01, 0.39231E-01,
     & 0.45737E-01, 0.52272E-01, 0.58833E-01, 0.65418E-01, 0.72027E-01,
     & 0.78657E-01, 0.85307E-01, 0.91976E-01, 0.98664E-01, 0.10537E+00,
     & 0.11209E+00, 0.11883E+00, 0.12558E+00, 0.13235E+00, 0.13913E+00,
     & 0.14593E+00, 0.15274E+00, 0.15958E+00, 0.16642E+00, 0.17329E+00,
     & 0.18017E+00, 0.18707E+00, 0.19399E+00, 0.20093E+00, 0.20789E+00,
     & 0.21487E+00, 0.22188E+00, 0.22891E+00, 0.23597E+00, 0.24305E+00,
     & 0.25016E+00, 0.25730E+00, 0.26447E+00, 0.27166E+00, 0.27890E+00,
     & 0.28616E+00, 0.29346E+00, 0.30079E+00, 0.30815E+00, 0.31556E+00,
     & 0.32299E+00, 0.33047E+00, 0.33798E+00, 0.34553E+00, 0.35311E+00,
     & 0.36073E+00, 0.36839E+00, 0.37608E+00, 0.38381E+00, 0.39158E+00,
     & 0.39937E+00, 0.40721E+00, 0.41507E+00, 0.42297E+00, 0.43089E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.43885E+00, 0.44683E+00, 0.45485E+00, 0.46288E+00, 0.47095E+00,
     & 0.47903E+00, 0.48714E+00, 0.49527E+00, 0.50341E+00, 0.51158E+00,
     & 0.51976E+00, 0.52795E+00, 0.53616E+00, 0.54438E+00, 0.55261E+00,
     & 0.56085E+00, 0.56909E+00, 0.57735E+00, 0.58561E+00, 0.59387E+00,
     & 0.60213E+00, 0.61040E+00, 0.61867E+00, 0.62693E+00, 0.63520E+00,
     & 0.64346E+00, 0.65172E+00, 0.65998E+00, 0.66823E+00, 0.67648E+00,
     & 0.68472E+00, 0.69296E+00, 0.70119E+00, 0.70941E+00, 0.71762E+00,
     & 0.72582E+00, 0.73402E+00, 0.74220E+00, 0.75038E+00, 0.75854E+00,
     & 0.76670E+00, 0.77484E+00, 0.78297E+00, 0.79109E+00, 0.79920E+00,
     & 0.80730E+00, 0.81539E+00, 0.82346E+00, 0.83152E+00, 0.83957E+00,
     & 0.84760E+00, 0.85562E+00, 0.86363E+00, 0.87163E+00, 0.87961E+00,
     & 0.88758E+00, 0.89553E+00, 0.90347E+00, 0.91140E+00, 0.91931E+00,
     & 0.92721E+00, 0.93510E+00, 0.94297E+00, 0.95083E+00, 0.95867E+00,
     & 0.96650E+00, 0.97431E+00, 0.98211E+00, 0.98990E+00, 0.99767E+00,
     & 0.10054E+01, 0.10132E+01, 0.10209E+01, 0.10286E+01, 0.10363E+01,
     & 0.10440E+01, 0.10517E+01, 0.10593E+01, 0.10669E+01, 0.10746E+01,
     & 0.10822E+01, 0.10898E+01, 0.10974E+01, 0.11049E+01, 0.11125E+01,
     & 0.11200E+01, 0.11275E+01, 0.11350E+01, 0.11425E+01, 0.11500E+01,
     & 0.11574E+01, 0.11649E+01, 0.11723E+01, 0.11797E+01, 0.11871E+01,
     & 0.11945E+01, 0.12019E+01, 0.12092E+01, 0.12166E+01, 0.12239E+01/

      DATA (BNC11M(I),I=201,300)/
     & 0.12312E+01, 0.12385E+01, 0.12458E+01, 0.12531E+01, 0.12603E+01,
     & 0.12675E+01, 0.12748E+01, 0.12820E+01, 0.12892E+01, 0.12963E+01,
     & 0.13035E+01, 0.13106E+01, 0.13178E+01, 0.13249E+01, 0.13320E+01,
     & 0.13391E+01, 0.13462E+01, 0.13532E+01, 0.13603E+01, 0.13673E+01,
     & 0.13743E+01, 0.13813E+01, 0.13883E+01, 0.13953E+01, 0.14022E+01,
     & 0.14092E+01, 0.14161E+01, 0.14230E+01, 0.14299E+01, 0.14368E+01,
     & 0.14437E+01, 0.14505E+01, 0.14574E+01, 0.14642E+01, 0.14710E+01,
     & 0.14778E+01, 0.14846E+01, 0.14914E+01, 0.14981E+01, 0.15049E+01,
     & 0.15116E+01, 0.15183E+01, 0.15250E+01, 0.15317E+01, 0.15384E+01,
     & 0.15450E+01, 0.15517E+01, 0.15583E+01, 0.15650E+01, 0.15716E+01,
     & 0.15782E+01, 0.15847E+01, 0.15913E+01, 0.15979E+01, 0.16044E+01,
     & 0.16109E+01, 0.16174E+01, 0.16239E+01, 0.16304E+01, 0.16369E+01,
     & 0.16434E+01, 0.16498E+01, 0.16562E+01, 0.16627E+01, 0.16691E+01,
     & 0.16755E+01, 0.16818E+01, 0.16882E+01, 0.16946E+01, 0.17009E+01,
     & 0.17072E+01, 0.17136E+01, 0.17199E+01, 0.17262E+01, 0.17324E+01,
     & 0.17387E+01, 0.17450E+01, 0.17512E+01, 0.17574E+01, 0.17636E+01,
     & 0.17699E+01, 0.17760E+01, 0.17822E+01, 0.17884E+01, 0.17945E+01,
     & 0.18007E+01, 0.18068E+01, 0.18129E+01, 0.18190E+01, 0.18251E+01,
     & 0.18312E+01, 0.18373E+01, 0.18434E+01, 0.18494E+01, 0.18554E+01,
     & 0.18615E+01, 0.18675E+01, 0.18735E+01, 0.18795E+01, 0.18854E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.18914E+01, 0.18973E+01, 0.19033E+01, 0.19092E+01, 0.19151E+01,
     & 0.19210E+01, 0.19269E+01, 0.19328E+01, 0.19387E+01, 0.19446E+01,
     & 0.19504E+01, 0.19562E+01, 0.19621E+01, 0.19679E+01, 0.19737E+01,
     & 0.19795E+01, 0.19853E+01, 0.19910E+01, 0.19968E+01, 0.20025E+01,
     & 0.20083E+01, 0.20140E+01, 0.20197E+01, 0.20254E+01, 0.20311E+01,
     & 0.20368E+01, 0.20425E+01, 0.20481E+01, 0.20538E+01, 0.20594E+01,
     & 0.20651E+01, 0.20707E+01, 0.20763E+01, 0.20819E+01, 0.20875E+01,
     & 0.20931E+01, 0.20986E+01, 0.21042E+01, 0.21097E+01, 0.21153E+01,
     & 0.21208E+01, 0.21263E+01, 0.21318E+01, 0.21373E+01, 0.21428E+01,
     & 0.21483E+01, 0.21537E+01, 0.21592E+01, 0.21646E+01, 0.21701E+01,
     & 0.21755E+01, 0.21809E+01, 0.21863E+01, 0.21917E+01, 0.21971E+01,
     & 0.22025E+01, 0.22078E+01, 0.22132E+01, 0.22185E+01, 0.22239E+01,
     & 0.22292E+01, 0.22345E+01, 0.22398E+01, 0.22451E+01, 0.22504E+01,
     & 0.22557E+01, 0.22610E+01, 0.22662E+01, 0.22715E+01, 0.22767E+01,
     & 0.22819E+01, 0.22872E+01, 0.22924E+01, 0.22976E+01, 0.23028E+01,
     & 0.23080E+01, 0.23131E+01, 0.23183E+01, 0.23235E+01, 0.23286E+01,
     & 0.23338E+01, 0.23389E+01, 0.23440E+01, 0.23491E+01, 0.23542E+01,
     & 0.23593E+01, 0.23644E+01, 0.23695E+01, 0.23746E+01, 0.23796E+01,
     & 0.23847E+01, 0.23897E+01, 0.23947E+01, 0.23998E+01, 0.24048E+01,
     & 0.24098E+01, 0.24148E+01, 0.24198E+01, 0.24248E+01, 0.24297E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.24347E+01, 0.24397E+01, 0.24446E+01, 0.24496E+01, 0.24545E+01,
     & 0.24594E+01, 0.24643E+01, 0.24692E+01, 0.24741E+01, 0.24790E+01,
     & 0.24839E+01, 0.24888E+01, 0.24936E+01, 0.24985E+01, 0.25033E+01,
     & 0.25082E+01, 0.25130E+01, 0.25178E+01, 0.25227E+01, 0.25275E+01,
     & 0.25323E+01, 0.25371E+01, 0.25418E+01, 0.25466E+01, 0.25514E+01,
     & 0.25561E+01, 0.25609E+01, 0.25656E+01, 0.25704E+01, 0.25751E+01,
     & 0.25798E+01, 0.25845E+01, 0.25892E+01, 0.25939E+01, 0.25986E+01,
     & 0.26033E+01, 0.26080E+01, 0.26127E+01, 0.26173E+01, 0.26220E+01,
     & 0.26266E+01, 0.26313E+01, 0.26359E+01, 0.26405E+01, 0.26451E+01,
     & 0.26497E+01, 0.26543E+01, 0.26589E+01, 0.26635E+01, 0.26681E+01,
     & 0.26727E+01, 0.26772E+01, 0.26818E+01, 0.26863E+01, 0.26909E+01,
     & 0.26954E+01, 0.26999E+01, 0.27044E+01, 0.27090E+01, 0.27135E+01,
     & 0.27180E+01, 0.27224E+01, 0.27269E+01, 0.27314E+01, 0.27359E+01,
     & 0.27403E+01, 0.27448E+01, 0.27492E+01, 0.27537E+01, 0.27581E+01,
     & 0.27626E+01, 0.27670E+01, 0.27714E+01, 0.27758E+01, 0.27802E+01,
     & 0.27846E+01, 0.27890E+01, 0.27934E+01, 0.27977E+01, 0.28021E+01,
     & 0.28065E+01, 0.28108E+01, 0.28152E+01, 0.28195E+01, 0.28238E+01,
     & 0.28282E+01, 0.28325E+01, 0.28368E+01, 0.28411E+01, 0.28454E+01,
     & 0.28497E+01, 0.28540E+01, 0.28583E+01, 0.28625E+01, 0.28668E+01,
     & 0.28711E+01, 0.28753E+01, 0.28796E+01, 0.28838E+01, 0.28880E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.28923E+01, 0.28965E+01, 0.29007E+01, 0.29049E+01, 0.29091E+01,
     & 0.29133E+01, 0.29175E+01, 0.29217E+01, 0.29259E+01, 0.29301E+01,
     & 0.29342E+01, 0.29384E+01, 0.29425E+01, 0.29467E+01, 0.29508E+01,
     & 0.29550E+01, 0.29591E+01, 0.29632E+01, 0.29673E+01, 0.29714E+01,
     & 0.29756E+01, 0.29797E+01, 0.29837E+01, 0.29878E+01, 0.29919E+01,
     & 0.29960E+01, 0.30001E+01, 0.30041E+01, 0.30082E+01, 0.30122E+01,
     & 0.30163E+01, 0.30203E+01, 0.30244E+01, 0.30284E+01, 0.30324E+01,
     & 0.30364E+01, 0.30404E+01, 0.30444E+01, 0.30484E+01, 0.30524E+01,
     & 0.30564E+01, 0.30604E+01, 0.30644E+01, 0.30684E+01, 0.30723E+01,
     & 0.30763E+01, 0.30802E+01, 0.30842E+01, 0.30881E+01, 0.30921E+01,
     & 0.30960E+01, 0.30999E+01, 0.31039E+01, 0.31078E+01, 0.31117E+01,
     & 0.31156E+01, 0.31195E+01, 0.31234E+01, 0.31273E+01, 0.31311E+01,
     & 0.31350E+01, 0.31389E+01, 0.31428E+01, 0.31466E+01, 0.31505E+01,
     & 0.31543E+01, 0.31582E+01, 0.31620E+01, 0.31658E+01, 0.31697E+01,
     & 0.31735E+01, 0.31773E+01, 0.31811E+01, 0.31849E+01, 0.31887E+01,
     & 0.31925E+01, 0.31963E+01, 0.32001E+01, 0.32039E+01, 0.32077E+01,
     & 0.32115E+01, 0.32152E+01, 0.32190E+01, 0.32227E+01, 0.32265E+01,
     & 0.32302E+01, 0.32340E+01, 0.32377E+01, 0.32414E+01, 0.32452E+01,
     & 0.32489E+01, 0.32526E+01, 0.32563E+01, 0.32600E+01, 0.32637E+01,
     & 0.32674E+01, 0.32711E+01, 0.32748E+01, 0.32785E+01, 0.32822E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.33216E+01, 0.33575E+01, 0.33930E+01, 0.34281E+01, 0.34627E+01,
     & 0.34969E+01, 0.35306E+01, 0.35640E+01, 0.35969E+01, 0.36294E+01,
     & 0.36616E+01, 0.36933E+01, 0.37247E+01, 0.37557E+01, 0.37864E+01,
     & 0.38167E+01, 0.38467E+01, 0.38764E+01, 0.39057E+01, 0.39347E+01,
     & 0.39633E+01, 0.39917E+01, 0.40198E+01, 0.40475E+01, 0.40750E+01,
     & 0.41022E+01, 0.41291E+01, 0.41558E+01, 0.41821E+01, 0.42082E+01,
     & 0.42341E+01, 0.42596E+01, 0.42850E+01, 0.43100E+01, 0.43349E+01,
     & 0.43595E+01, 0.43838E+01, 0.44080E+01, 0.44319E+01, 0.44556E+01,
     & 0.44790E+01, 0.45023E+01, 0.45253E+01, 0.45481E+01, 0.45707E+01,
     & 0.45932E+01, 0.46154E+01, 0.46374E+01, 0.46592E+01, 0.46809E+01,
     & 0.47023E+01, 0.47236E+01, 0.47447E+01, 0.47656E+01, 0.47863E+01,
     & 0.48069E+01, 0.48273E+01, 0.48475E+01, 0.48675E+01, 0.48874E+01,
     & 0.49071E+01, 0.49267E+01, 0.49461E+01, 0.49654E+01, 0.49845E+01,
     & 0.50034E+01, 0.50222E+01, 0.50409E+01, 0.50594E+01, 0.50778E+01,
     & 0.50960E+01, 0.51141E+01, 0.51321E+01, 0.51499E+01, 0.51676E+01,
     & 0.51851E+01, 0.52025E+01, 0.52198E+01, 0.52370E+01, 0.52541E+01,
     & 0.52710E+01, 0.52878E+01, 0.53044E+01, 0.53210E+01, 0.53374E+01,
     & 0.53538E+01, 0.53700E+01, 0.53861E+01, 0.54021E+01, 0.54179E+01,
     & 0.54337E+01, 0.54494E+01, 0.54649E+01, 0.54804E+01, 0.54957E+01,
     & 0.55109E+01, 0.55261E+01, 0.55411E+01, 0.55560E+01, 0.55709E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.55856E+01, 0.56002E+01, 0.56148E+01, 0.56292E+01, 0.56436E+01,
     & 0.56579E+01, 0.56720E+01, 0.56861E+01, 0.57001E+01, 0.57140E+01,
     & 0.57278E+01, 0.57416E+01, 0.57552E+01, 0.57688E+01, 0.57822E+01,
     & 0.57956E+01, 0.58089E+01, 0.58222E+01, 0.58353E+01, 0.58484E+01,
     & 0.58614E+01, 0.58743E+01, 0.58871E+01, 0.58999E+01, 0.59126E+01,
     & 0.59252E+01, 0.59377E+01, 0.59501E+01, 0.59625E+01, 0.59748E+01,
     & 0.59871E+01, 0.59993E+01, 0.60114E+01, 0.60234E+01, 0.60353E+01,
     & 0.60472E+01, 0.60591E+01, 0.60708E+01, 0.60825E+01, 0.60941E+01,
     & 0.61057E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.47116E-01,-0.97475E-01,-0.12015E+00,-0.13491E+00,-0.14562E+00,
     &-0.15380E+00,-0.16023E+00,-0.16537E+00,-0.16952E+00,-0.17287E+00,
     &-0.17555E+00,-0.17769E+00,-0.17935E+00,-0.18059E+00,-0.18148E+00,
     &-0.18204E+00,-0.18230E+00,-0.18230E+00,-0.18206E+00,-0.18160E+00,
     &-0.18093E+00,-0.18008E+00,-0.17904E+00,-0.17783E+00,-0.17647E+00,
     &-0.17496E+00,-0.17331E+00,-0.17153E+00,-0.16962E+00,-0.16760E+00,
     &-0.16545E+00,-0.16320E+00,-0.16085E+00,-0.15839E+00,-0.15584E+00,
     &-0.15321E+00,-0.15048E+00,-0.14767E+00,-0.14478E+00,-0.14181E+00,
     &-0.13877E+00,-0.13567E+00,-0.13249E+00,-0.12925E+00,-0.12594E+00,
     &-0.12258E+00,-0.11916E+00,-0.11568E+00,-0.11216E+00,-0.10858E+00,
     &-0.10495E+00,-0.10127E+00,-0.97550E-01,-0.93785E-01,-0.89977E-01,
     &-0.86128E-01,-0.82239E-01,-0.78311E-01,-0.74345E-01,-0.70342E-01,
     &-0.66304E-01,-0.62230E-01,-0.58122E-01,-0.53979E-01,-0.49803E-01,
     &-0.45593E-01,-0.41351E-01,-0.37076E-01,-0.32768E-01,-0.28428E-01,
     &-0.24056E-01,-0.19651E-01,-0.15214E-01,-0.10744E-01,-0.62423E-02,
     &-0.17073E-02, 0.28604E-02, 0.74611E-02, 0.12095E-01, 0.16762E-01,
     & 0.21462E-01, 0.26196E-01, 0.30963E-01, 0.35763E-01, 0.40597E-01,
     & 0.45463E-01, 0.50363E-01, 0.55296E-01, 0.60261E-01, 0.65258E-01,
     & 0.70287E-01, 0.75348E-01, 0.80439E-01, 0.85560E-01, 0.90710E-01,
     & 0.95889E-01, 0.10110E+00, 0.10633E+00, 0.11159E+00, 0.11687E+00/

      DATA (BNC12M(I),I=101,200)/
     & 0.12218E+00, 0.12752E+00, 0.13287E+00, 0.13824E+00, 0.14364E+00,
     & 0.14905E+00, 0.15448E+00, 0.15993E+00, 0.16539E+00, 0.17086E+00,
     & 0.17635E+00, 0.18185E+00, 0.18736E+00, 0.19287E+00, 0.19840E+00,
     & 0.20393E+00, 0.20947E+00, 0.21502E+00, 0.22057E+00, 0.22612E+00,
     & 0.23167E+00, 0.23722E+00, 0.24278E+00, 0.24834E+00, 0.25389E+00,
     & 0.25944E+00, 0.26499E+00, 0.27054E+00, 0.27608E+00, 0.28162E+00,
     & 0.28715E+00, 0.29268E+00, 0.29821E+00, 0.30373E+00, 0.30924E+00,
     & 0.31474E+00, 0.32024E+00, 0.32573E+00, 0.33121E+00, 0.33668E+00,
     & 0.34215E+00, 0.34761E+00, 0.35306E+00, 0.35850E+00, 0.36393E+00,
     & 0.36935E+00, 0.37476E+00, 0.38016E+00, 0.38555E+00, 0.39094E+00,
     & 0.39631E+00, 0.40167E+00, 0.40703E+00, 0.41237E+00, 0.41770E+00,
     & 0.42302E+00, 0.42833E+00, 0.43363E+00, 0.43893E+00, 0.44420E+00,
     & 0.44947E+00, 0.45473E+00, 0.45998E+00, 0.46522E+00, 0.47044E+00,
     & 0.47566E+00, 0.48087E+00, 0.48606E+00, 0.49124E+00, 0.49641E+00,
     & 0.50158E+00, 0.50673E+00, 0.51187E+00, 0.51699E+00, 0.52211E+00,
     & 0.52722E+00, 0.53232E+00, 0.53740E+00, 0.54247E+00, 0.54754E+00,
     & 0.55259E+00, 0.55763E+00, 0.56266E+00, 0.56768E+00, 0.57269E+00,
     & 0.57769E+00, 0.58267E+00, 0.58765E+00, 0.59262E+00, 0.59757E+00,
     & 0.60251E+00, 0.60745E+00, 0.61237E+00, 0.61728E+00, 0.62218E+00,
     & 0.62707E+00, 0.63195E+00, 0.63682E+00, 0.64168E+00, 0.64652E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.65136E+00, 0.65619E+00, 0.66100E+00, 0.66581E+00, 0.67060E+00,
     & 0.67539E+00, 0.68016E+00, 0.68492E+00, 0.68968E+00, 0.69442E+00,
     & 0.69915E+00, 0.70387E+00, 0.70858E+00, 0.71328E+00, 0.71798E+00,
     & 0.72266E+00, 0.72733E+00, 0.73199E+00, 0.73664E+00, 0.74128E+00,
     & 0.74591E+00, 0.75053E+00, 0.75514E+00, 0.75974E+00, 0.76433E+00,
     & 0.76891E+00, 0.77348E+00, 0.77804E+00, 0.78260E+00, 0.78714E+00,
     & 0.79167E+00, 0.79619E+00, 0.80070E+00, 0.80521E+00, 0.80970E+00,
     & 0.81418E+00, 0.81866E+00, 0.82312E+00, 0.82758E+00, 0.83202E+00,
     & 0.83646E+00, 0.84089E+00, 0.84530E+00, 0.84971E+00, 0.85411E+00,
     & 0.85850E+00, 0.86288E+00, 0.86725E+00, 0.87161E+00, 0.87597E+00,
     & 0.88031E+00, 0.88465E+00, 0.88897E+00, 0.89329E+00, 0.89760E+00,
     & 0.90189E+00, 0.90619E+00, 0.91047E+00, 0.91474E+00, 0.91900E+00,
     & 0.92326E+00, 0.92750E+00, 0.93174E+00, 0.93597E+00, 0.94019E+00,
     & 0.94440E+00, 0.94860E+00, 0.95280E+00, 0.95698E+00, 0.96116E+00,
     & 0.96533E+00, 0.96949E+00, 0.97364E+00, 0.97778E+00, 0.98192E+00,
     & 0.98604E+00, 0.99016E+00, 0.99427E+00, 0.99837E+00, 0.10025E+01,
     & 0.10066E+01, 0.10106E+01, 0.10147E+01, 0.10188E+01, 0.10228E+01,
     & 0.10269E+01, 0.10309E+01, 0.10349E+01, 0.10389E+01, 0.10430E+01,
     & 0.10470E+01, 0.10510E+01, 0.10549E+01, 0.10589E+01, 0.10629E+01,
     & 0.10669E+01, 0.10708E+01, 0.10748E+01, 0.10787E+01, 0.10827E+01/

      DATA (BNC12M(I),I=301,400)/
     & 0.10866E+01, 0.10905E+01, 0.10944E+01, 0.10983E+01, 0.11022E+01,
     & 0.11061E+01, 0.11100E+01, 0.11139E+01, 0.11177E+01, 0.11216E+01,
     & 0.11254E+01, 0.11293E+01, 0.11331E+01, 0.11370E+01, 0.11408E+01,
     & 0.11446E+01, 0.11484E+01, 0.11522E+01, 0.11560E+01, 0.11598E+01,
     & 0.11636E+01, 0.11673E+01, 0.11711E+01, 0.11749E+01, 0.11786E+01,
     & 0.11824E+01, 0.11861E+01, 0.11898E+01, 0.11936E+01, 0.11973E+01,
     & 0.12010E+01, 0.12047E+01, 0.12084E+01, 0.12121E+01, 0.12157E+01,
     & 0.12194E+01, 0.12231E+01, 0.12268E+01, 0.12304E+01, 0.12341E+01,
     & 0.12377E+01, 0.12413E+01, 0.12450E+01, 0.12486E+01, 0.12522E+01,
     & 0.12558E+01, 0.12594E+01, 0.12630E+01, 0.12666E+01, 0.12702E+01,
     & 0.12738E+01, 0.12773E+01, 0.12809E+01, 0.12844E+01, 0.12880E+01,
     & 0.12915E+01, 0.12951E+01, 0.12986E+01, 0.13021E+01, 0.13057E+01,
     & 0.13092E+01, 0.13127E+01, 0.13162E+01, 0.13197E+01, 0.13232E+01,
     & 0.13266E+01, 0.13301E+01, 0.13336E+01, 0.13370E+01, 0.13405E+01,
     & 0.13440E+01, 0.13474E+01, 0.13508E+01, 0.13543E+01, 0.13577E+01,
     & 0.13611E+01, 0.13645E+01, 0.13679E+01, 0.13714E+01, 0.13747E+01,
     & 0.13781E+01, 0.13815E+01, 0.13849E+01, 0.13883E+01, 0.13916E+01,
     & 0.13950E+01, 0.13984E+01, 0.14017E+01, 0.14051E+01, 0.14084E+01,
     & 0.14117E+01, 0.14151E+01, 0.14184E+01, 0.14217E+01, 0.14250E+01,
     & 0.14283E+01, 0.14316E+01, 0.14349E+01, 0.14382E+01, 0.14415E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.14448E+01, 0.14480E+01, 0.14513E+01, 0.14546E+01, 0.14578E+01,
     & 0.14611E+01, 0.14643E+01, 0.14676E+01, 0.14708E+01, 0.14740E+01,
     & 0.14772E+01, 0.14805E+01, 0.14837E+01, 0.14869E+01, 0.14901E+01,
     & 0.14933E+01, 0.14965E+01, 0.14997E+01, 0.15028E+01, 0.15060E+01,
     & 0.15092E+01, 0.15124E+01, 0.15155E+01, 0.15187E+01, 0.15218E+01,
     & 0.15250E+01, 0.15281E+01, 0.15312E+01, 0.15344E+01, 0.15375E+01,
     & 0.15406E+01, 0.15437E+01, 0.15468E+01, 0.15499E+01, 0.15530E+01,
     & 0.15561E+01, 0.15592E+01, 0.15623E+01, 0.15654E+01, 0.15685E+01,
     & 0.15715E+01, 0.15746E+01, 0.15776E+01, 0.15807E+01, 0.15838E+01,
     & 0.15868E+01, 0.15898E+01, 0.15929E+01, 0.15959E+01, 0.15989E+01,
     & 0.16020E+01, 0.16050E+01, 0.16080E+01, 0.16110E+01, 0.16140E+01,
     & 0.16170E+01, 0.16200E+01, 0.16230E+01, 0.16259E+01, 0.16289E+01,
     & 0.16319E+01, 0.16349E+01, 0.16378E+01, 0.16408E+01, 0.16437E+01,
     & 0.16467E+01, 0.16496E+01, 0.16526E+01, 0.16555E+01, 0.16585E+01,
     & 0.16614E+01, 0.16643E+01, 0.16672E+01, 0.16701E+01, 0.16731E+01,
     & 0.16760E+01, 0.16789E+01, 0.16818E+01, 0.16846E+01, 0.16875E+01,
     & 0.16904E+01, 0.16933E+01, 0.16962E+01, 0.16990E+01, 0.17019E+01,
     & 0.17048E+01, 0.17076E+01, 0.17105E+01, 0.17133E+01, 0.17162E+01,
     & 0.17190E+01, 0.17218E+01, 0.17247E+01, 0.17275E+01, 0.17303E+01,
     & 0.17331E+01, 0.17360E+01, 0.17388E+01, 0.17416E+01, 0.17444E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.17472E+01, 0.17500E+01, 0.17528E+01, 0.17555E+01, 0.17583E+01,
     & 0.17611E+01, 0.17639E+01, 0.17666E+01, 0.17694E+01, 0.17722E+01,
     & 0.17749E+01, 0.17777E+01, 0.17804E+01, 0.17832E+01, 0.17859E+01,
     & 0.17887E+01, 0.17914E+01, 0.17941E+01, 0.17968E+01, 0.17996E+01,
     & 0.18023E+01, 0.18050E+01, 0.18077E+01, 0.18104E+01, 0.18131E+01,
     & 0.18158E+01, 0.18185E+01, 0.18212E+01, 0.18239E+01, 0.18265E+01,
     & 0.18292E+01, 0.18319E+01, 0.18346E+01, 0.18372E+01, 0.18399E+01,
     & 0.18426E+01, 0.18452E+01, 0.18479E+01, 0.18505E+01, 0.18531E+01,
     & 0.18558E+01, 0.18584E+01, 0.18611E+01, 0.18637E+01, 0.18663E+01,
     & 0.18689E+01, 0.18715E+01, 0.18742E+01, 0.18768E+01, 0.18794E+01,
     & 0.18820E+01, 0.18846E+01, 0.18872E+01, 0.18898E+01, 0.18923E+01,
     & 0.18949E+01, 0.18975E+01, 0.19001E+01, 0.19027E+01, 0.19052E+01,
     & 0.19078E+01, 0.19104E+01, 0.19129E+01, 0.19155E+01, 0.19180E+01,
     & 0.19206E+01, 0.19231E+01, 0.19257E+01, 0.19282E+01, 0.19307E+01,
     & 0.19333E+01, 0.19358E+01, 0.19383E+01, 0.19408E+01, 0.19433E+01,
     & 0.19459E+01, 0.19484E+01, 0.19509E+01, 0.19534E+01, 0.19559E+01,
     & 0.19584E+01, 0.19609E+01, 0.19634E+01, 0.19658E+01, 0.19683E+01,
     & 0.19708E+01, 0.19733E+01, 0.19757E+01, 0.19782E+01, 0.19807E+01,
     & 0.19831E+01, 0.19856E+01, 0.19881E+01, 0.19905E+01, 0.19930E+01,
     & 0.19954E+01, 0.19978E+01, 0.20003E+01, 0.20027E+01, 0.20052E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.20312E+01, 0.20550E+01, 0.20785E+01, 0.21017E+01, 0.21246E+01,
     & 0.21472E+01, 0.21695E+01, 0.21916E+01, 0.22134E+01, 0.22349E+01,
     & 0.22561E+01, 0.22771E+01, 0.22978E+01, 0.23183E+01, 0.23386E+01,
     & 0.23586E+01, 0.23784E+01, 0.23980E+01, 0.24173E+01, 0.24364E+01,
     & 0.24553E+01, 0.24740E+01, 0.24925E+01, 0.25108E+01, 0.25289E+01,
     & 0.25468E+01, 0.25646E+01, 0.25821E+01, 0.25994E+01, 0.26166E+01,
     & 0.26336E+01, 0.26504E+01, 0.26670E+01, 0.26835E+01, 0.26998E+01,
     & 0.27160E+01, 0.27320E+01, 0.27478E+01, 0.27635E+01, 0.27790E+01,
     & 0.27944E+01, 0.28096E+01, 0.28247E+01, 0.28396E+01, 0.28544E+01,
     & 0.28690E+01, 0.28836E+01, 0.28980E+01, 0.29122E+01, 0.29263E+01,
     & 0.29403E+01, 0.29542E+01, 0.29680E+01, 0.29816E+01, 0.29951E+01,
     & 0.30085E+01, 0.30217E+01, 0.30349E+01, 0.30479E+01, 0.30609E+01,
     & 0.30737E+01, 0.30864E+01, 0.30990E+01, 0.31115E+01, 0.31239E+01,
     & 0.31362E+01, 0.31483E+01, 0.31604E+01, 0.31724E+01, 0.31843E+01,
     & 0.31961E+01, 0.32078E+01, 0.32194E+01, 0.32309E+01, 0.32423E+01,
     & 0.32536E+01, 0.32648E+01, 0.32760E+01, 0.32870E+01, 0.32980E+01,
     & 0.33089E+01, 0.33197E+01, 0.33304E+01, 0.33411E+01, 0.33516E+01,
     & 0.33621E+01, 0.33725E+01, 0.33828E+01, 0.33931E+01, 0.34032E+01,
     & 0.34133E+01, 0.34233E+01, 0.34333E+01, 0.34431E+01, 0.34529E+01,
     & 0.34627E+01, 0.34723E+01, 0.34819E+01, 0.34914E+01, 0.35009E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.35103E+01, 0.35196E+01, 0.35288E+01, 0.35380E+01, 0.35471E+01,
     & 0.35562E+01, 0.35652E+01, 0.35741E+01, 0.35830E+01, 0.35918E+01,
     & 0.36005E+01, 0.36092E+01, 0.36178E+01, 0.36264E+01, 0.36349E+01,
     & 0.36434E+01, 0.36518E+01, 0.36601E+01, 0.36684E+01, 0.36766E+01,
     & 0.36848E+01, 0.36929E+01, 0.37010E+01, 0.37090E+01, 0.37169E+01,
     & 0.37248E+01, 0.37327E+01, 0.37405E+01, 0.37483E+01, 0.37560E+01,
     & 0.37636E+01, 0.37712E+01, 0.37788E+01, 0.37863E+01, 0.37938E+01,
     & 0.38012E+01, 0.38085E+01, 0.38159E+01, 0.38231E+01, 0.38304E+01,
     & 0.38376E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.78016E-01,-0.16930E+00,-0.21481E+00,-0.24696E+00,-0.27216E+00,
     &-0.29302E+00,-0.31087E+00,-0.32649E+00,-0.34040E+00,-0.35293E+00,
     &-0.36434E+00,-0.37481E+00,-0.38448E+00,-0.39347E+00,-0.40186E+00,
     &-0.40972E+00,-0.41711E+00,-0.42408E+00,-0.43067E+00,-0.43692E+00,
     &-0.44285E+00,-0.44850E+00,-0.45388E+00,-0.45901E+00,-0.46392E+00,
     &-0.46861E+00,-0.47311E+00,-0.47741E+00,-0.48155E+00,-0.48551E+00,
     &-0.48933E+00,-0.49299E+00,-0.49651E+00,-0.49991E+00,-0.50317E+00,
     &-0.50632E+00,-0.50935E+00,-0.51227E+00,-0.51508E+00,-0.51780E+00,
     &-0.52042E+00,-0.52294E+00,-0.52538E+00,-0.52774E+00,-0.53001E+00,
     &-0.53221E+00,-0.53432E+00,-0.53637E+00,-0.53835E+00,-0.54026E+00,
     &-0.54210E+00,-0.54389E+00,-0.54561E+00,-0.54727E+00,-0.54888E+00,
     &-0.55044E+00,-0.55194E+00,-0.55339E+00,-0.55479E+00,-0.55615E+00,
     &-0.55746E+00,-0.55872E+00,-0.55994E+00,-0.56113E+00,-0.56227E+00,
     &-0.56337E+00,-0.56443E+00,-0.56546E+00,-0.56645E+00,-0.56741E+00,
     &-0.56833E+00,-0.56922E+00,-0.57008E+00,-0.57090E+00,-0.57170E+00,
     &-0.57246E+00,-0.57320E+00,-0.57391E+00,-0.57459E+00,-0.57524E+00,
     &-0.57587E+00,-0.57647E+00,-0.57704E+00,-0.57759E+00,-0.57812E+00,
     &-0.57862E+00,-0.57910E+00,-0.57955E+00,-0.57999E+00,-0.58040E+00,
     &-0.58078E+00,-0.58115E+00,-0.58150E+00,-0.58183E+00,-0.58214E+00,
     &-0.58243E+00,-0.58270E+00,-0.58295E+00,-0.58318E+00,-0.58340E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.58360E+00,-0.58379E+00,-0.58396E+00,-0.58411E+00,-0.58425E+00,
     &-0.58438E+00,-0.58449E+00,-0.58459E+00,-0.58467E+00,-0.58475E+00,
     &-0.58481E+00,-0.58486E+00,-0.58490E+00,-0.58493E+00,-0.58495E+00,
     &-0.58495E+00,-0.58495E+00,-0.58495E+00,-0.58493E+00,-0.58490E+00,
     &-0.58487E+00,-0.58483E+00,-0.58478E+00,-0.58473E+00,-0.58467E+00,
     &-0.58460E+00,-0.58452E+00,-0.58445E+00,-0.58436E+00,-0.58427E+00,
     &-0.58418E+00,-0.58408E+00,-0.58397E+00,-0.58386E+00,-0.58375E+00,
     &-0.58363E+00,-0.58351E+00,-0.58339E+00,-0.58326E+00,-0.58313E+00,
     &-0.58300E+00,-0.58286E+00,-0.58272E+00,-0.58258E+00,-0.58243E+00,
     &-0.58228E+00,-0.58213E+00,-0.58198E+00,-0.58183E+00,-0.58167E+00,
     &-0.58151E+00,-0.58135E+00,-0.58119E+00,-0.58102E+00,-0.58086E+00,
     &-0.58069E+00,-0.58052E+00,-0.58035E+00,-0.58018E+00,-0.58001E+00,
     &-0.57983E+00,-0.57966E+00,-0.57948E+00,-0.57930E+00,-0.57912E+00,
     &-0.57894E+00,-0.57876E+00,-0.57858E+00,-0.57840E+00,-0.57822E+00,
     &-0.57803E+00,-0.57785E+00,-0.57766E+00,-0.57748E+00,-0.57729E+00,
     &-0.57711E+00,-0.57692E+00,-0.57673E+00,-0.57655E+00,-0.57636E+00,
     &-0.57617E+00,-0.57598E+00,-0.57579E+00,-0.57561E+00,-0.57542E+00,
     &-0.57523E+00,-0.57504E+00,-0.57485E+00,-0.57466E+00,-0.57447E+00,
     &-0.57428E+00,-0.57409E+00,-0.57390E+00,-0.57372E+00,-0.57353E+00,
     &-0.57334E+00,-0.57315E+00,-0.57296E+00,-0.57277E+00,-0.57258E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.57240E+00,-0.57221E+00,-0.57202E+00,-0.57183E+00,-0.57165E+00,
     &-0.57146E+00,-0.57127E+00,-0.57109E+00,-0.57090E+00,-0.57072E+00,
     &-0.57053E+00,-0.57035E+00,-0.57016E+00,-0.56998E+00,-0.56980E+00,
     &-0.56961E+00,-0.56943E+00,-0.56925E+00,-0.56907E+00,-0.56888E+00,
     &-0.56870E+00,-0.56852E+00,-0.56834E+00,-0.56816E+00,-0.56798E+00,
     &-0.56781E+00,-0.56763E+00,-0.56745E+00,-0.56727E+00,-0.56710E+00,
     &-0.56692E+00,-0.56675E+00,-0.56657E+00,-0.56640E+00,-0.56623E+00,
     &-0.56605E+00,-0.56588E+00,-0.56571E+00,-0.56554E+00,-0.56537E+00,
     &-0.56520E+00,-0.56503E+00,-0.56486E+00,-0.56469E+00,-0.56452E+00,
     &-0.56436E+00,-0.56419E+00,-0.56402E+00,-0.56386E+00,-0.56370E+00,
     &-0.56353E+00,-0.56337E+00,-0.56321E+00,-0.56304E+00,-0.56288E+00,
     &-0.56272E+00,-0.56256E+00,-0.56240E+00,-0.56225E+00,-0.56209E+00,
     &-0.56193E+00,-0.56178E+00,-0.56162E+00,-0.56146E+00,-0.56131E+00,
     &-0.56116E+00,-0.56100E+00,-0.56085E+00,-0.56070E+00,-0.56055E+00,
     &-0.56040E+00,-0.56025E+00,-0.56010E+00,-0.55995E+00,-0.55980E+00,
     &-0.55966E+00,-0.55951E+00,-0.55937E+00,-0.55922E+00,-0.55908E+00,
     &-0.55893E+00,-0.55879E+00,-0.55865E+00,-0.55851E+00,-0.55837E+00,
     &-0.55823E+00,-0.55809E+00,-0.55795E+00,-0.55781E+00,-0.55768E+00,
     &-0.55754E+00,-0.55740E+00,-0.55727E+00,-0.55714E+00,-0.55700E+00,
     &-0.55687E+00,-0.55674E+00,-0.55661E+00,-0.55648E+00,-0.55635E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.55622E+00,-0.55609E+00,-0.55596E+00,-0.55583E+00,-0.55571E+00,
     &-0.55558E+00,-0.55546E+00,-0.55533E+00,-0.55521E+00,-0.55509E+00,
     &-0.55496E+00,-0.55484E+00,-0.55472E+00,-0.55460E+00,-0.55448E+00,
     &-0.55436E+00,-0.55425E+00,-0.55413E+00,-0.55401E+00,-0.55390E+00,
     &-0.55378E+00,-0.55367E+00,-0.55355E+00,-0.55344E+00,-0.55333E+00,
     &-0.55322E+00,-0.55311E+00,-0.55300E+00,-0.55289E+00,-0.55278E+00,
     &-0.55267E+00,-0.55256E+00,-0.55246E+00,-0.55235E+00,-0.55224E+00,
     &-0.55214E+00,-0.55204E+00,-0.55193E+00,-0.55183E+00,-0.55173E+00,
     &-0.55163E+00,-0.55153E+00,-0.55143E+00,-0.55133E+00,-0.55123E+00,
     &-0.55113E+00,-0.55103E+00,-0.55094E+00,-0.55084E+00,-0.55075E+00,
     &-0.55065E+00,-0.55056E+00,-0.55047E+00,-0.55037E+00,-0.55028E+00,
     &-0.55019E+00,-0.55010E+00,-0.55001E+00,-0.54992E+00,-0.54983E+00,
     &-0.54975E+00,-0.54966E+00,-0.54957E+00,-0.54949E+00,-0.54940E+00,
     &-0.54932E+00,-0.54924E+00,-0.54915E+00,-0.54907E+00,-0.54899E+00,
     &-0.54891E+00,-0.54883E+00,-0.54875E+00,-0.54867E+00,-0.54859E+00,
     &-0.54851E+00,-0.54844E+00,-0.54836E+00,-0.54828E+00,-0.54821E+00,
     &-0.54813E+00,-0.54806E+00,-0.54799E+00,-0.54792E+00,-0.54784E+00,
     &-0.54777E+00,-0.54770E+00,-0.54763E+00,-0.54756E+00,-0.54749E+00,
     &-0.54743E+00,-0.54736E+00,-0.54729E+00,-0.54723E+00,-0.54716E+00,
     &-0.54710E+00,-0.54703E+00,-0.54697E+00,-0.54691E+00,-0.54684E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.54678E+00,-0.54672E+00,-0.54666E+00,-0.54660E+00,-0.54654E+00,
     &-0.54648E+00,-0.54643E+00,-0.54637E+00,-0.54631E+00,-0.54626E+00,
     &-0.54620E+00,-0.54615E+00,-0.54609E+00,-0.54604E+00,-0.54599E+00,
     &-0.54593E+00,-0.54588E+00,-0.54583E+00,-0.54578E+00,-0.54573E+00,
     &-0.54568E+00,-0.54563E+00,-0.54558E+00,-0.54554E+00,-0.54549E+00,
     &-0.54544E+00,-0.54540E+00,-0.54535E+00,-0.54531E+00,-0.54527E+00,
     &-0.54522E+00,-0.54518E+00,-0.54514E+00,-0.54510E+00,-0.54506E+00,
     &-0.54502E+00,-0.54498E+00,-0.54494E+00,-0.54490E+00,-0.54486E+00,
     &-0.54482E+00,-0.54479E+00,-0.54475E+00,-0.54471E+00,-0.54468E+00,
     &-0.54465E+00,-0.54461E+00,-0.54458E+00,-0.54455E+00,-0.54451E+00,
     &-0.54448E+00,-0.54445E+00,-0.54442E+00,-0.54439E+00,-0.54436E+00,
     &-0.54433E+00,-0.54430E+00,-0.54428E+00,-0.54425E+00,-0.54422E+00,
     &-0.54420E+00,-0.54417E+00,-0.54415E+00,-0.54412E+00,-0.54410E+00,
     &-0.54408E+00,-0.54405E+00,-0.54403E+00,-0.54401E+00,-0.54399E+00,
     &-0.54397E+00,-0.54395E+00,-0.54393E+00,-0.54391E+00,-0.54389E+00,
     &-0.54388E+00,-0.54386E+00,-0.54384E+00,-0.54383E+00,-0.54381E+00,
     &-0.54380E+00,-0.54378E+00,-0.54377E+00,-0.54376E+00,-0.54374E+00,
     &-0.54373E+00,-0.54372E+00,-0.54371E+00,-0.54370E+00,-0.54369E+00,
     &-0.54368E+00,-0.54367E+00,-0.54366E+00,-0.54365E+00,-0.54365E+00,
     &-0.54364E+00,-0.54363E+00,-0.54363E+00,-0.54362E+00,-0.54362E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.54361E+00,-0.54361E+00,-0.54361E+00,-0.54361E+00,-0.54360E+00,
     &-0.54360E+00,-0.54360E+00,-0.54360E+00,-0.54360E+00,-0.54360E+00,
     &-0.54360E+00,-0.54360E+00,-0.54361E+00,-0.54361E+00,-0.54361E+00,
     &-0.54361E+00,-0.54362E+00,-0.54362E+00,-0.54363E+00,-0.54363E+00,
     &-0.54364E+00,-0.54365E+00,-0.54365E+00,-0.54366E+00,-0.54367E+00,
     &-0.54368E+00,-0.54369E+00,-0.54370E+00,-0.54371E+00,-0.54372E+00,
     &-0.54373E+00,-0.54374E+00,-0.54375E+00,-0.54376E+00,-0.54378E+00,
     &-0.54379E+00,-0.54380E+00,-0.54382E+00,-0.54383E+00,-0.54385E+00,
     &-0.54387E+00,-0.54388E+00,-0.54390E+00,-0.54392E+00,-0.54393E+00,
     &-0.54395E+00,-0.54397E+00,-0.54399E+00,-0.54401E+00,-0.54403E+00,
     &-0.54405E+00,-0.54407E+00,-0.54409E+00,-0.54412E+00,-0.54414E+00,
     &-0.54416E+00,-0.54418E+00,-0.54421E+00,-0.54423E+00,-0.54426E+00,
     &-0.54428E+00,-0.54431E+00,-0.54434E+00,-0.54436E+00,-0.54439E+00,
     &-0.54442E+00,-0.54445E+00,-0.54447E+00,-0.54450E+00,-0.54453E+00,
     &-0.54456E+00,-0.54459E+00,-0.54462E+00,-0.54466E+00,-0.54469E+00,
     &-0.54472E+00,-0.54475E+00,-0.54479E+00,-0.54482E+00,-0.54485E+00,
     &-0.54489E+00,-0.54492E+00,-0.54496E+00,-0.54499E+00,-0.54503E+00,
     &-0.54507E+00,-0.54510E+00,-0.54514E+00,-0.54518E+00,-0.54522E+00,
     &-0.54526E+00,-0.54530E+00,-0.54534E+00,-0.54538E+00,-0.54542E+00,
     &-0.54546E+00,-0.54550E+00,-0.54554E+00,-0.54558E+00,-0.54563E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.54612E+00,-0.54661E+00,-0.54715E+00,-0.54773E+00,-0.54835E+00,
     &-0.54901E+00,-0.54971E+00,-0.55044E+00,-0.55121E+00,-0.55202E+00,
     &-0.55287E+00,-0.55374E+00,-0.55466E+00,-0.55560E+00,-0.55658E+00,
     &-0.55760E+00,-0.55864E+00,-0.55972E+00,-0.56083E+00,-0.56196E+00,
     &-0.56313E+00,-0.56433E+00,-0.56556E+00,-0.56681E+00,-0.56809E+00,
     &-0.56940E+00,-0.57074E+00,-0.57210E+00,-0.57349E+00,-0.57491E+00,
     &-0.57635E+00,-0.57782E+00,-0.57931E+00,-0.58082E+00,-0.58236E+00,
     &-0.58392E+00,-0.58551E+00,-0.58712E+00,-0.58875E+00,-0.59040E+00,
     &-0.59207E+00,-0.59377E+00,-0.59549E+00,-0.59722E+00,-0.59898E+00,
     &-0.60076E+00,-0.60256E+00,-0.60438E+00,-0.60621E+00,-0.60807E+00,
     &-0.60994E+00,-0.61183E+00,-0.61375E+00,-0.61567E+00,-0.61762E+00,
     &-0.61959E+00,-0.62157E+00,-0.62357E+00,-0.62558E+00,-0.62761E+00,
     &-0.62966E+00,-0.63173E+00,-0.63381E+00,-0.63590E+00,-0.63801E+00,
     &-0.64014E+00,-0.64228E+00,-0.64444E+00,-0.64661E+00,-0.64879E+00,
     &-0.65099E+00,-0.65321E+00,-0.65544E+00,-0.65768E+00,-0.65993E+00,
     &-0.66220E+00,-0.66448E+00,-0.66678E+00,-0.66909E+00,-0.67141E+00,
     &-0.67374E+00,-0.67609E+00,-0.67845E+00,-0.68082E+00,-0.68320E+00,
     &-0.68559E+00,-0.68800E+00,-0.69042E+00,-0.69285E+00,-0.69529E+00,
     &-0.69774E+00,-0.70021E+00,-0.70268E+00,-0.70516E+00,-0.70766E+00,
     &-0.71017E+00,-0.71268E+00,-0.71521E+00,-0.71775E+00,-0.72030E+00/

      DATA (BNC13M(I),I=701,741)/
     &-0.72285E+00,-0.72542E+00,-0.72800E+00,-0.73059E+00,-0.73318E+00,
     &-0.73579E+00,-0.73840E+00,-0.74103E+00,-0.74366E+00,-0.74631E+00,
     &-0.74896E+00,-0.75162E+00,-0.75429E+00,-0.75697E+00,-0.75965E+00,
     &-0.76235E+00,-0.76505E+00,-0.76777E+00,-0.77049E+00,-0.77322E+00,
     &-0.77595E+00,-0.77870E+00,-0.78145E+00,-0.78421E+00,-0.78698E+00,
     &-0.78976E+00,-0.79254E+00,-0.79534E+00,-0.79814E+00,-0.80094E+00,
     &-0.80376E+00,-0.80658E+00,-0.80941E+00,-0.81224E+00,-0.81509E+00,
     &-0.81794E+00,-0.82079E+00,-0.82366E+00,-0.82653E+00,-0.82940E+00,
     &-0.83229E+00
     & /
      END


 
 
C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF273A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C 
      BLOCK DATA KMCF273A
C
C *** Common block definition
C
      COMMON /KMC273/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.46407E-01,-0.96708E-01,-0.11948E+00,-0.13432E+00,-0.14508E+00,
     &-0.15330E+00,-0.15979E+00,-0.16500E+00,-0.16923E+00,-0.17269E+00,
     &-0.17553E+00,-0.17785E+00,-0.17973E+00,-0.18124E+00,-0.18244E+00,
     &-0.18336E+00,-0.18404E+00,-0.18450E+00,-0.18477E+00,-0.18488E+00,
     &-0.18483E+00,-0.18465E+00,-0.18435E+00,-0.18393E+00,-0.18342E+00,
     &-0.18282E+00,-0.18213E+00,-0.18137E+00,-0.18054E+00,-0.17965E+00,
     &-0.17870E+00,-0.17770E+00,-0.17665E+00,-0.17556E+00,-0.17443E+00,
     &-0.17327E+00,-0.17207E+00,-0.17084E+00,-0.16958E+00,-0.16830E+00,
     &-0.16700E+00,-0.16568E+00,-0.16434E+00,-0.16298E+00,-0.16161E+00,
     &-0.16022E+00,-0.15882E+00,-0.15741E+00,-0.15599E+00,-0.15455E+00,
     &-0.15311E+00,-0.15166E+00,-0.15021E+00,-0.14875E+00,-0.14728E+00,
     &-0.14580E+00,-0.14432E+00,-0.14283E+00,-0.14134E+00,-0.13984E+00,
     &-0.13833E+00,-0.13682E+00,-0.13530E+00,-0.13378E+00,-0.13224E+00,
     &-0.13071E+00,-0.12916E+00,-0.12761E+00,-0.12605E+00,-0.12448E+00,
     &-0.12290E+00,-0.12131E+00,-0.11971E+00,-0.11810E+00,-0.11649E+00,
     &-0.11486E+00,-0.11322E+00,-0.11156E+00,-0.10990E+00,-0.10822E+00,
     &-0.10653E+00,-0.10483E+00,-0.10311E+00,-0.10138E+00,-0.99639E-01,
     &-0.97882E-01,-0.96110E-01,-0.94324E-01,-0.92525E-01,-0.90711E-01,
     &-0.88883E-01,-0.87042E-01,-0.85186E-01,-0.83317E-01,-0.81435E-01,
     &-0.79539E-01,-0.77630E-01,-0.75709E-01,-0.73776E-01,-0.71830E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.69873E-01,-0.67905E-01,-0.65926E-01,-0.63937E-01,-0.61937E-01,
     &-0.59929E-01,-0.57911E-01,-0.55885E-01,-0.53851E-01,-0.51809E-01,
     &-0.49760E-01,-0.47705E-01,-0.45643E-01,-0.43575E-01,-0.41502E-01,
     &-0.39424E-01,-0.37342E-01,-0.35255E-01,-0.33165E-01,-0.31071E-01,
     &-0.28975E-01,-0.26875E-01,-0.24774E-01,-0.22670E-01,-0.20565E-01,
     &-0.18458E-01,-0.16350E-01,-0.14240E-01,-0.12131E-01,-0.10020E-01,
     &-0.79100E-02,-0.57993E-02,-0.36888E-02,-0.15785E-02, 0.53139E-03,
     & 0.26407E-02, 0.47494E-02, 0.68572E-02, 0.89641E-02, 0.11070E-01,
     & 0.13175E-01, 0.15278E-01, 0.17380E-01, 0.19481E-01, 0.21580E-01,
     & 0.23678E-01, 0.25774E-01, 0.27869E-01, 0.29962E-01, 0.32052E-01,
     & 0.34142E-01, 0.36229E-01, 0.38314E-01, 0.40398E-01, 0.42479E-01,
     & 0.44558E-01, 0.46636E-01, 0.48711E-01, 0.50784E-01, 0.52855E-01,
     & 0.54924E-01, 0.56990E-01, 0.59054E-01, 0.61116E-01, 0.63176E-01,
     & 0.65233E-01, 0.67288E-01, 0.69341E-01, 0.71391E-01, 0.73439E-01,
     & 0.75485E-01, 0.77528E-01, 0.79569E-01, 0.81607E-01, 0.83642E-01,
     & 0.85675E-01, 0.87706E-01, 0.89734E-01, 0.91760E-01, 0.93783E-01,
     & 0.95803E-01, 0.97821E-01, 0.99836E-01, 0.10185E+00, 0.10386E+00,
     & 0.10587E+00, 0.10787E+00, 0.10987E+00, 0.11187E+00, 0.11387E+00,
     & 0.11586E+00, 0.11785E+00, 0.11984E+00, 0.12183E+00, 0.12381E+00,
     & 0.12579E+00, 0.12777E+00, 0.12974E+00, 0.13172E+00, 0.13369E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.13565E+00, 0.13762E+00, 0.13958E+00, 0.14154E+00, 0.14349E+00,
     & 0.14544E+00, 0.14739E+00, 0.14934E+00, 0.15129E+00, 0.15323E+00,
     & 0.15517E+00, 0.15710E+00, 0.15904E+00, 0.16097E+00, 0.16289E+00,
     & 0.16482E+00, 0.16674E+00, 0.16866E+00, 0.17057E+00, 0.17249E+00,
     & 0.17440E+00, 0.17631E+00, 0.17821E+00, 0.18011E+00, 0.18201E+00,
     & 0.18391E+00, 0.18580E+00, 0.18769E+00, 0.18958E+00, 0.19146E+00,
     & 0.19334E+00, 0.19522E+00, 0.19710E+00, 0.19897E+00, 0.20084E+00,
     & 0.20271E+00, 0.20458E+00, 0.20644E+00, 0.20830E+00, 0.21015E+00,
     & 0.21201E+00, 0.21386E+00, 0.21570E+00, 0.21755E+00, 0.21939E+00,
     & 0.22123E+00, 0.22307E+00, 0.22490E+00, 0.22673E+00, 0.22856E+00,
     & 0.23038E+00, 0.23221E+00, 0.23403E+00, 0.23584E+00, 0.23766E+00,
     & 0.23947E+00, 0.24127E+00, 0.24308E+00, 0.24488E+00, 0.24668E+00,
     & 0.24848E+00, 0.25027E+00, 0.25206E+00, 0.25385E+00, 0.25564E+00,
     & 0.25742E+00, 0.25920E+00, 0.26098E+00, 0.26275E+00, 0.26452E+00,
     & 0.26629E+00, 0.26806E+00, 0.26982E+00, 0.27158E+00, 0.27334E+00,
     & 0.27509E+00, 0.27684E+00, 0.27859E+00, 0.28034E+00, 0.28208E+00,
     & 0.28383E+00, 0.28556E+00, 0.28730E+00, 0.28903E+00, 0.29076E+00,
     & 0.29249E+00, 0.29422E+00, 0.29594E+00, 0.29766E+00, 0.29937E+00,
     & 0.30109E+00, 0.30280E+00, 0.30451E+00, 0.30621E+00, 0.30792E+00,
     & 0.30962E+00, 0.31131E+00, 0.31301E+00, 0.31470E+00, 0.31639E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.31808E+00, 0.31976E+00, 0.32144E+00, 0.32312E+00, 0.32480E+00,
     & 0.32647E+00, 0.32815E+00, 0.32981E+00, 0.33148E+00, 0.33314E+00,
     & 0.33480E+00, 0.33646E+00, 0.33812E+00, 0.33977E+00, 0.34142E+00,
     & 0.34307E+00, 0.34471E+00, 0.34636E+00, 0.34800E+00, 0.34963E+00,
     & 0.35127E+00, 0.35290E+00, 0.35453E+00, 0.35616E+00, 0.35778E+00,
     & 0.35940E+00, 0.36102E+00, 0.36264E+00, 0.36426E+00, 0.36587E+00,
     & 0.36748E+00, 0.36908E+00, 0.37069E+00, 0.37229E+00, 0.37389E+00,
     & 0.37549E+00, 0.37708E+00, 0.37867E+00, 0.38026E+00, 0.38185E+00,
     & 0.38343E+00, 0.38502E+00, 0.38659E+00, 0.38817E+00, 0.38975E+00,
     & 0.39132E+00, 0.39289E+00, 0.39446E+00, 0.39602E+00, 0.39758E+00,
     & 0.39914E+00, 0.40070E+00, 0.40226E+00, 0.40381E+00, 0.40536E+00,
     & 0.40691E+00, 0.40845E+00, 0.41000E+00, 0.41154E+00, 0.41308E+00,
     & 0.41461E+00, 0.41615E+00, 0.41768E+00, 0.41921E+00, 0.42073E+00,
     & 0.42226E+00, 0.42378E+00, 0.42530E+00, 0.42682E+00, 0.42833E+00,
     & 0.42985E+00, 0.43136E+00, 0.43286E+00, 0.43437E+00, 0.43587E+00,
     & 0.43737E+00, 0.43887E+00, 0.44037E+00, 0.44187E+00, 0.44336E+00,
     & 0.44485E+00, 0.44633E+00, 0.44782E+00, 0.44930E+00, 0.45078E+00,
     & 0.45226E+00, 0.45374E+00, 0.45521E+00, 0.45669E+00, 0.45815E+00,
     & 0.45962E+00, 0.46109E+00, 0.46255E+00, 0.46401E+00, 0.46547E+00,
     & 0.46693E+00, 0.46838E+00, 0.46983E+00, 0.47128E+00, 0.47273E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.47418E+00, 0.47562E+00, 0.47706E+00, 0.47850E+00, 0.47994E+00,
     & 0.48137E+00, 0.48280E+00, 0.48423E+00, 0.48566E+00, 0.48709E+00,
     & 0.48851E+00, 0.48993E+00, 0.49135E+00, 0.49277E+00, 0.49419E+00,
     & 0.49560E+00, 0.49701E+00, 0.49842E+00, 0.49983E+00, 0.50123E+00,
     & 0.50264E+00, 0.50404E+00, 0.50544E+00, 0.50683E+00, 0.50823E+00,
     & 0.50962E+00, 0.51101E+00, 0.51240E+00, 0.51379E+00, 0.51517E+00,
     & 0.51656E+00, 0.51794E+00, 0.51931E+00, 0.52069E+00, 0.52207E+00,
     & 0.52344E+00, 0.52481E+00, 0.52618E+00, 0.52754E+00, 0.52891E+00,
     & 0.53027E+00, 0.53163E+00, 0.53299E+00, 0.53435E+00, 0.53570E+00,
     & 0.53705E+00, 0.53840E+00, 0.53975E+00, 0.54110E+00, 0.54245E+00,
     & 0.54379E+00, 0.54513E+00, 0.54647E+00, 0.54781E+00, 0.54914E+00,
     & 0.55047E+00, 0.55180E+00, 0.55313E+00, 0.55446E+00, 0.55579E+00,
     & 0.55711E+00, 0.55843E+00, 0.55975E+00, 0.56107E+00, 0.56239E+00,
     & 0.56370E+00, 0.56501E+00, 0.56632E+00, 0.56763E+00, 0.56894E+00,
     & 0.57025E+00, 0.57155E+00, 0.57285E+00, 0.57415E+00, 0.57545E+00,
     & 0.57674E+00, 0.57804E+00, 0.57933E+00, 0.58062E+00, 0.58191E+00,
     & 0.58319E+00, 0.58448E+00, 0.58576E+00, 0.58704E+00, 0.58832E+00,
     & 0.58960E+00, 0.59088E+00, 0.59215E+00, 0.59342E+00, 0.59469E+00,
     & 0.59596E+00, 0.59723E+00, 0.59849E+00, 0.59976E+00, 0.60102E+00,
     & 0.60228E+00, 0.60354E+00, 0.60479E+00, 0.60605E+00, 0.60730E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.60855E+00, 0.60980E+00, 0.61105E+00, 0.61230E+00, 0.61354E+00,
     & 0.61478E+00, 0.61602E+00, 0.61726E+00, 0.61850E+00, 0.61974E+00,
     & 0.62097E+00, 0.62220E+00, 0.62343E+00, 0.62466E+00, 0.62589E+00,
     & 0.62711E+00, 0.62834E+00, 0.62956E+00, 0.63078E+00, 0.63200E+00,
     & 0.63322E+00, 0.63443E+00, 0.63565E+00, 0.63686E+00, 0.63807E+00,
     & 0.63928E+00, 0.64049E+00, 0.64169E+00, 0.64290E+00, 0.64410E+00,
     & 0.64530E+00, 0.64650E+00, 0.64770E+00, 0.64889E+00, 0.65009E+00,
     & 0.65128E+00, 0.65247E+00, 0.65366E+00, 0.65485E+00, 0.65603E+00,
     & 0.65722E+00, 0.65840E+00, 0.65958E+00, 0.66076E+00, 0.66194E+00,
     & 0.66312E+00, 0.66429E+00, 0.66547E+00, 0.66664E+00, 0.66781E+00,
     & 0.66898E+00, 0.67015E+00, 0.67131E+00, 0.67248E+00, 0.67364E+00,
     & 0.67480E+00, 0.67596E+00, 0.67712E+00, 0.67828E+00, 0.67943E+00,
     & 0.68058E+00, 0.68174E+00, 0.68289E+00, 0.68404E+00, 0.68518E+00,
     & 0.68633E+00, 0.68748E+00, 0.68862E+00, 0.68976E+00, 0.69090E+00,
     & 0.69204E+00, 0.69318E+00, 0.69431E+00, 0.69545E+00, 0.69658E+00,
     & 0.69771E+00, 0.69884E+00, 0.69997E+00, 0.70109E+00, 0.70222E+00,
     & 0.70334E+00, 0.70447E+00, 0.70559E+00, 0.70671E+00, 0.70782E+00,
     & 0.70894E+00, 0.71006E+00, 0.71117E+00, 0.71228E+00, 0.71339E+00,
     & 0.71450E+00, 0.71561E+00, 0.71672E+00, 0.71782E+00, 0.71893E+00,
     & 0.72003E+00, 0.72113E+00, 0.72223E+00, 0.72333E+00, 0.72442E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.73618E+00, 0.74693E+00, 0.75754E+00, 0.76801E+00, 0.77836E+00,
     & 0.78859E+00, 0.79869E+00, 0.80867E+00, 0.81853E+00, 0.82827E+00,
     & 0.83790E+00, 0.84742E+00, 0.85683E+00, 0.86613E+00, 0.87532E+00,
     & 0.88441E+00, 0.89340E+00, 0.90229E+00, 0.91108E+00, 0.91978E+00,
     & 0.92838E+00, 0.93688E+00, 0.94530E+00, 0.95363E+00, 0.96187E+00,
     & 0.97002E+00, 0.97808E+00, 0.98607E+00, 0.99397E+00, 0.10018E+01,
     & 0.10095E+01, 0.10172E+01, 0.10248E+01, 0.10323E+01, 0.10397E+01,
     & 0.10471E+01, 0.10544E+01, 0.10616E+01, 0.10688E+01, 0.10758E+01,
     & 0.10828E+01, 0.10898E+01, 0.10967E+01, 0.11035E+01, 0.11102E+01,
     & 0.11169E+01, 0.11236E+01, 0.11301E+01, 0.11366E+01, 0.11431E+01,
     & 0.11495E+01, 0.11558E+01, 0.11621E+01, 0.11683E+01, 0.11745E+01,
     & 0.11806E+01, 0.11867E+01, 0.11927E+01, 0.11986E+01, 0.12045E+01,
     & 0.12104E+01, 0.12162E+01, 0.12220E+01, 0.12277E+01, 0.12333E+01,
     & 0.12389E+01, 0.12445E+01, 0.12500E+01, 0.12555E+01, 0.12609E+01,
     & 0.12663E+01, 0.12717E+01, 0.12770E+01, 0.12822E+01, 0.12874E+01,
     & 0.12926E+01, 0.12977E+01, 0.13028E+01, 0.13079E+01, 0.13129E+01,
     & 0.13179E+01, 0.13228E+01, 0.13277E+01, 0.13325E+01, 0.13374E+01,
     & 0.13421E+01, 0.13469E+01, 0.13516E+01, 0.13563E+01, 0.13609E+01,
     & 0.13655E+01, 0.13701E+01, 0.13746E+01, 0.13792E+01, 0.13836E+01,
     & 0.13881E+01, 0.13925E+01, 0.13968E+01, 0.14012E+01, 0.14055E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.14098E+01, 0.14140E+01, 0.14183E+01, 0.14224E+01, 0.14266E+01,
     & 0.14307E+01, 0.14348E+01, 0.14389E+01, 0.14430E+01, 0.14470E+01,
     & 0.14510E+01, 0.14549E+01, 0.14589E+01, 0.14628E+01, 0.14666E+01,
     & 0.14705E+01, 0.14743E+01, 0.14781E+01, 0.14819E+01, 0.14856E+01,
     & 0.14894E+01, 0.14931E+01, 0.14967E+01, 0.15004E+01, 0.15040E+01,
     & 0.15076E+01, 0.15112E+01, 0.15147E+01, 0.15183E+01, 0.15218E+01,
     & 0.15253E+01, 0.15287E+01, 0.15322E+01, 0.15356E+01, 0.15390E+01,
     & 0.15424E+01, 0.15457E+01, 0.15490E+01, 0.15524E+01, 0.15556E+01,
     & 0.15589E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.95525E-01,-0.20796E+00,-0.26423E+00,-0.30406E+00,-0.33534E+00,
     &-0.36126E+00,-0.38348E+00,-0.40297E+00,-0.42035E+00,-0.43605E+00,
     &-0.45039E+00,-0.46358E+00,-0.47581E+00,-0.48722E+00,-0.49790E+00,
     &-0.50796E+00,-0.51746E+00,-0.52646E+00,-0.53502E+00,-0.54318E+00,
     &-0.55098E+00,-0.55845E+00,-0.56561E+00,-0.57250E+00,-0.57913E+00,
     &-0.58552E+00,-0.59170E+00,-0.59767E+00,-0.60345E+00,-0.60905E+00,
     &-0.61449E+00,-0.61976E+00,-0.62489E+00,-0.62989E+00,-0.63474E+00,
     &-0.63948E+00,-0.64410E+00,-0.64860E+00,-0.65300E+00,-0.65729E+00,
     &-0.66149E+00,-0.66560E+00,-0.66962E+00,-0.67356E+00,-0.67741E+00,
     &-0.68119E+00,-0.68489E+00,-0.68852E+00,-0.69209E+00,-0.69559E+00,
     &-0.69902E+00,-0.70240E+00,-0.70571E+00,-0.70897E+00,-0.71218E+00,
     &-0.71534E+00,-0.71844E+00,-0.72149E+00,-0.72450E+00,-0.72747E+00,
     &-0.73039E+00,-0.73327E+00,-0.73610E+00,-0.73890E+00,-0.74166E+00,
     &-0.74438E+00,-0.74707E+00,-0.74973E+00,-0.75235E+00,-0.75493E+00,
     &-0.75749E+00,-0.76002E+00,-0.76252E+00,-0.76499E+00,-0.76743E+00,
     &-0.76985E+00,-0.77224E+00,-0.77461E+00,-0.77695E+00,-0.77927E+00,
     &-0.78156E+00,-0.78384E+00,-0.78609E+00,-0.78833E+00,-0.79054E+00,
     &-0.79273E+00,-0.79491E+00,-0.79706E+00,-0.79920E+00,-0.80132E+00,
     &-0.80342E+00,-0.80551E+00,-0.80758E+00,-0.80963E+00,-0.81167E+00,
     &-0.81370E+00,-0.81570E+00,-0.81770E+00,-0.81968E+00,-0.82164E+00/

      DATA (BNC02M(I),I=101,200)/
     &-0.82360E+00,-0.82553E+00,-0.82746E+00,-0.82937E+00,-0.83127E+00,
     &-0.83316E+00,-0.83503E+00,-0.83689E+00,-0.83874E+00,-0.84058E+00,
     &-0.84241E+00,-0.84422E+00,-0.84602E+00,-0.84782E+00,-0.84960E+00,
     &-0.85137E+00,-0.85312E+00,-0.85487E+00,-0.85661E+00,-0.85834E+00,
     &-0.86005E+00,-0.86176E+00,-0.86346E+00,-0.86514E+00,-0.86682E+00,
     &-0.86849E+00,-0.87015E+00,-0.87180E+00,-0.87343E+00,-0.87506E+00,
     &-0.87668E+00,-0.87830E+00,-0.87990E+00,-0.88149E+00,-0.88308E+00,
     &-0.88466E+00,-0.88622E+00,-0.88779E+00,-0.88934E+00,-0.89088E+00,
     &-0.89242E+00,-0.89395E+00,-0.89547E+00,-0.89698E+00,-0.89848E+00,
     &-0.89998E+00,-0.90147E+00,-0.90295E+00,-0.90443E+00,-0.90590E+00,
     &-0.90736E+00,-0.90881E+00,-0.91026E+00,-0.91170E+00,-0.91313E+00,
     &-0.91456E+00,-0.91598E+00,-0.91740E+00,-0.91880E+00,-0.92020E+00,
     &-0.92160E+00,-0.92299E+00,-0.92437E+00,-0.92575E+00,-0.92712E+00,
     &-0.92848E+00,-0.92984E+00,-0.93120E+00,-0.93254E+00,-0.93389E+00,
     &-0.93522E+00,-0.93655E+00,-0.93788E+00,-0.93920E+00,-0.94052E+00,
     &-0.94182E+00,-0.94313E+00,-0.94443E+00,-0.94572E+00,-0.94701E+00,
     &-0.94830E+00,-0.94958E+00,-0.95085E+00,-0.95212E+00,-0.95338E+00,
     &-0.95464E+00,-0.95590E+00,-0.95715E+00,-0.95840E+00,-0.95964E+00,
     &-0.96087E+00,-0.96211E+00,-0.96333E+00,-0.96456E+00,-0.96578E+00,
     &-0.96699E+00,-0.96820E+00,-0.96941E+00,-0.97061E+00,-0.97181E+00/

      DATA (BNC02M(I),I=201,300)/
     &-0.97300E+00,-0.97419E+00,-0.97538E+00,-0.97656E+00,-0.97774E+00,
     &-0.97892E+00,-0.98009E+00,-0.98125E+00,-0.98242E+00,-0.98358E+00,
     &-0.98473E+00,-0.98588E+00,-0.98703E+00,-0.98817E+00,-0.98932E+00,
     &-0.99045E+00,-0.99159E+00,-0.99272E+00,-0.99384E+00,-0.99497E+00,
     &-0.99609E+00,-0.99720E+00,-0.99832E+00,-0.99943E+00,-0.10005E+01,
     &-0.10016E+01,-0.10027E+01,-0.10038E+01,-0.10049E+01,-0.10060E+01,
     &-0.10071E+01,-0.10082E+01,-0.10093E+01,-0.10103E+01,-0.10114E+01,
     &-0.10125E+01,-0.10136E+01,-0.10146E+01,-0.10157E+01,-0.10168E+01,
     &-0.10178E+01,-0.10189E+01,-0.10199E+01,-0.10210E+01,-0.10220E+01,
     &-0.10231E+01,-0.10241E+01,-0.10251E+01,-0.10262E+01,-0.10272E+01,
     &-0.10282E+01,-0.10293E+01,-0.10303E+01,-0.10313E+01,-0.10323E+01,
     &-0.10333E+01,-0.10344E+01,-0.10354E+01,-0.10364E+01,-0.10374E+01,
     &-0.10384E+01,-0.10394E+01,-0.10404E+01,-0.10414E+01,-0.10424E+01,
     &-0.10434E+01,-0.10444E+01,-0.10453E+01,-0.10463E+01,-0.10473E+01,
     &-0.10483E+01,-0.10493E+01,-0.10502E+01,-0.10512E+01,-0.10522E+01,
     &-0.10532E+01,-0.10541E+01,-0.10551E+01,-0.10560E+01,-0.10570E+01,
     &-0.10580E+01,-0.10589E+01,-0.10599E+01,-0.10608E+01,-0.10618E+01,
     &-0.10627E+01,-0.10637E+01,-0.10646E+01,-0.10655E+01,-0.10665E+01,
     &-0.10674E+01,-0.10684E+01,-0.10693E+01,-0.10702E+01,-0.10711E+01,
     &-0.10721E+01,-0.10730E+01,-0.10739E+01,-0.10748E+01,-0.10758E+01/

      DATA (BNC02M(I),I=301,400)/
     &-0.10767E+01,-0.10776E+01,-0.10785E+01,-0.10794E+01,-0.10803E+01,
     &-0.10812E+01,-0.10821E+01,-0.10830E+01,-0.10839E+01,-0.10848E+01,
     &-0.10857E+01,-0.10866E+01,-0.10875E+01,-0.10884E+01,-0.10893E+01,
     &-0.10902E+01,-0.10911E+01,-0.10920E+01,-0.10929E+01,-0.10938E+01,
     &-0.10946E+01,-0.10955E+01,-0.10964E+01,-0.10973E+01,-0.10981E+01,
     &-0.10990E+01,-0.10999E+01,-0.11008E+01,-0.11016E+01,-0.11025E+01,
     &-0.11034E+01,-0.11042E+01,-0.11051E+01,-0.11059E+01,-0.11068E+01,
     &-0.11077E+01,-0.11085E+01,-0.11094E+01,-0.11102E+01,-0.11111E+01,
     &-0.11119E+01,-0.11128E+01,-0.11136E+01,-0.11145E+01,-0.11153E+01,
     &-0.11161E+01,-0.11170E+01,-0.11178E+01,-0.11187E+01,-0.11195E+01,
     &-0.11203E+01,-0.11212E+01,-0.11220E+01,-0.11228E+01,-0.11237E+01,
     &-0.11245E+01,-0.11253E+01,-0.11261E+01,-0.11270E+01,-0.11278E+01,
     &-0.11286E+01,-0.11294E+01,-0.11302E+01,-0.11311E+01,-0.11319E+01,
     &-0.11327E+01,-0.11335E+01,-0.11343E+01,-0.11351E+01,-0.11359E+01,
     &-0.11367E+01,-0.11376E+01,-0.11384E+01,-0.11392E+01,-0.11400E+01,
     &-0.11408E+01,-0.11416E+01,-0.11424E+01,-0.11432E+01,-0.11440E+01,
     &-0.11448E+01,-0.11456E+01,-0.11463E+01,-0.11471E+01,-0.11479E+01,
     &-0.11487E+01,-0.11495E+01,-0.11503E+01,-0.11511E+01,-0.11519E+01,
     &-0.11526E+01,-0.11534E+01,-0.11542E+01,-0.11550E+01,-0.11558E+01,
     &-0.11565E+01,-0.11573E+01,-0.11581E+01,-0.11589E+01,-0.11596E+01/

      DATA (BNC02M(I),I=401,500)/
     &-0.11604E+01,-0.11612E+01,-0.11620E+01,-0.11627E+01,-0.11635E+01,
     &-0.11643E+01,-0.11650E+01,-0.11658E+01,-0.11666E+01,-0.11673E+01,
     &-0.11681E+01,-0.11688E+01,-0.11696E+01,-0.11704E+01,-0.11711E+01,
     &-0.11719E+01,-0.11726E+01,-0.11734E+01,-0.11741E+01,-0.11749E+01,
     &-0.11756E+01,-0.11764E+01,-0.11771E+01,-0.11779E+01,-0.11786E+01,
     &-0.11794E+01,-0.11801E+01,-0.11809E+01,-0.11816E+01,-0.11824E+01,
     &-0.11831E+01,-0.11838E+01,-0.11846E+01,-0.11853E+01,-0.11860E+01,
     &-0.11868E+01,-0.11875E+01,-0.11883E+01,-0.11890E+01,-0.11897E+01,
     &-0.11905E+01,-0.11912E+01,-0.11919E+01,-0.11926E+01,-0.11934E+01,
     &-0.11941E+01,-0.11948E+01,-0.11955E+01,-0.11963E+01,-0.11970E+01,
     &-0.11977E+01,-0.11984E+01,-0.11992E+01,-0.11999E+01,-0.12006E+01,
     &-0.12013E+01,-0.12020E+01,-0.12028E+01,-0.12035E+01,-0.12042E+01,
     &-0.12049E+01,-0.12056E+01,-0.12063E+01,-0.12070E+01,-0.12077E+01,
     &-0.12085E+01,-0.12092E+01,-0.12099E+01,-0.12106E+01,-0.12113E+01,
     &-0.12120E+01,-0.12127E+01,-0.12134E+01,-0.12141E+01,-0.12148E+01,
     &-0.12155E+01,-0.12162E+01,-0.12169E+01,-0.12176E+01,-0.12183E+01,
     &-0.12190E+01,-0.12197E+01,-0.12204E+01,-0.12211E+01,-0.12218E+01,
     &-0.12225E+01,-0.12232E+01,-0.12239E+01,-0.12246E+01,-0.12252E+01,
     &-0.12259E+01,-0.12266E+01,-0.12273E+01,-0.12280E+01,-0.12287E+01,
     &-0.12294E+01,-0.12301E+01,-0.12307E+01,-0.12314E+01,-0.12321E+01/

      DATA (BNC02M(I),I=501,600)/
     &-0.12328E+01,-0.12335E+01,-0.12341E+01,-0.12348E+01,-0.12355E+01,
     &-0.12362E+01,-0.12369E+01,-0.12375E+01,-0.12382E+01,-0.12389E+01,
     &-0.12396E+01,-0.12402E+01,-0.12409E+01,-0.12416E+01,-0.12423E+01,
     &-0.12429E+01,-0.12436E+01,-0.12443E+01,-0.12449E+01,-0.12456E+01,
     &-0.12463E+01,-0.12470E+01,-0.12476E+01,-0.12483E+01,-0.12489E+01,
     &-0.12496E+01,-0.12503E+01,-0.12509E+01,-0.12516E+01,-0.12523E+01,
     &-0.12529E+01,-0.12536E+01,-0.12542E+01,-0.12549E+01,-0.12556E+01,
     &-0.12562E+01,-0.12569E+01,-0.12575E+01,-0.12582E+01,-0.12589E+01,
     &-0.12595E+01,-0.12602E+01,-0.12608E+01,-0.12615E+01,-0.12621E+01,
     &-0.12628E+01,-0.12634E+01,-0.12641E+01,-0.12647E+01,-0.12654E+01,
     &-0.12660E+01,-0.12667E+01,-0.12673E+01,-0.12680E+01,-0.12686E+01,
     &-0.12693E+01,-0.12699E+01,-0.12706E+01,-0.12712E+01,-0.12718E+01,
     &-0.12725E+01,-0.12731E+01,-0.12738E+01,-0.12744E+01,-0.12750E+01,
     &-0.12757E+01,-0.12763E+01,-0.12770E+01,-0.12776E+01,-0.12782E+01,
     &-0.12789E+01,-0.12795E+01,-0.12802E+01,-0.12808E+01,-0.12814E+01,
     &-0.12821E+01,-0.12827E+01,-0.12833E+01,-0.12840E+01,-0.12846E+01,
     &-0.12852E+01,-0.12858E+01,-0.12865E+01,-0.12871E+01,-0.12877E+01,
     &-0.12884E+01,-0.12890E+01,-0.12896E+01,-0.12903E+01,-0.12909E+01,
     &-0.12915E+01,-0.12921E+01,-0.12928E+01,-0.12934E+01,-0.12940E+01,
     &-0.12946E+01,-0.12952E+01,-0.12959E+01,-0.12965E+01,-0.12971E+01/

      DATA (BNC02M(I),I=601,700)/
     &-0.13038E+01,-0.13099E+01,-0.13160E+01,-0.13220E+01,-0.13280E+01,
     &-0.13340E+01,-0.13399E+01,-0.13457E+01,-0.13515E+01,-0.13573E+01,
     &-0.13631E+01,-0.13688E+01,-0.13744E+01,-0.13801E+01,-0.13856E+01,
     &-0.13912E+01,-0.13967E+01,-0.14022E+01,-0.14077E+01,-0.14131E+01,
     &-0.14185E+01,-0.14239E+01,-0.14292E+01,-0.14345E+01,-0.14398E+01,
     &-0.14451E+01,-0.14503E+01,-0.14555E+01,-0.14607E+01,-0.14659E+01,
     &-0.14710E+01,-0.14761E+01,-0.14812E+01,-0.14862E+01,-0.14913E+01,
     &-0.14963E+01,-0.15013E+01,-0.15063E+01,-0.15112E+01,-0.15161E+01,
     &-0.15211E+01,-0.15260E+01,-0.15308E+01,-0.15357E+01,-0.15405E+01,
     &-0.15453E+01,-0.15501E+01,-0.15549E+01,-0.15597E+01,-0.15644E+01,
     &-0.15692E+01,-0.15739E+01,-0.15786E+01,-0.15833E+01,-0.15880E+01,
     &-0.15926E+01,-0.15972E+01,-0.16019E+01,-0.16065E+01,-0.16111E+01,
     &-0.16157E+01,-0.16202E+01,-0.16248E+01,-0.16293E+01,-0.16338E+01,
     &-0.16384E+01,-0.16429E+01,-0.16473E+01,-0.16518E+01,-0.16563E+01,
     &-0.16607E+01,-0.16652E+01,-0.16696E+01,-0.16740E+01,-0.16784E+01,
     &-0.16828E+01,-0.16872E+01,-0.16915E+01,-0.16959E+01,-0.17003E+01,
     &-0.17046E+01,-0.17089E+01,-0.17132E+01,-0.17175E+01,-0.17218E+01,
     &-0.17261E+01,-0.17304E+01,-0.17347E+01,-0.17389E+01,-0.17432E+01,
     &-0.17474E+01,-0.17516E+01,-0.17559E+01,-0.17601E+01,-0.17643E+01,
     &-0.17685E+01,-0.17726E+01,-0.17768E+01,-0.17810E+01,-0.17851E+01/

      DATA (BNC02M(I),I=701,741)/
     &-0.17893E+01,-0.17934E+01,-0.17976E+01,-0.18017E+01,-0.18058E+01,
     &-0.18099E+01,-0.18140E+01,-0.18181E+01,-0.18222E+01,-0.18263E+01,
     &-0.18304E+01,-0.18344E+01,-0.18385E+01,-0.18425E+01,-0.18466E+01,
     &-0.18506E+01,-0.18547E+01,-0.18587E+01,-0.18627E+01,-0.18667E+01,
     &-0.18707E+01,-0.18747E+01,-0.18787E+01,-0.18827E+01,-0.18866E+01,
     &-0.18906E+01,-0.18946E+01,-0.18985E+01,-0.19025E+01,-0.19064E+01,
     &-0.19104E+01,-0.19143E+01,-0.19182E+01,-0.19222E+01,-0.19261E+01,
     &-0.19300E+01,-0.19339E+01,-0.19378E+01,-0.19417E+01,-0.19456E+01,
     &-0.19495E+01
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.47892E-01,-0.10469E+00,-0.13335E+00,-0.15377E+00,-0.16990E+00,
     &-0.18333E+00,-0.19490E+00,-0.20510E+00,-0.21423E+00,-0.22252E+00,
     &-0.23012E+00,-0.23714E+00,-0.24367E+00,-0.24979E+00,-0.25554E+00,
     &-0.26097E+00,-0.26612E+00,-0.27101E+00,-0.27568E+00,-0.28015E+00,
     &-0.28443E+00,-0.28854E+00,-0.29249E+00,-0.29630E+00,-0.29998E+00,
     &-0.30354E+00,-0.30699E+00,-0.31033E+00,-0.31357E+00,-0.31671E+00,
     &-0.31977E+00,-0.32275E+00,-0.32565E+00,-0.32848E+00,-0.33124E+00,
     &-0.33393E+00,-0.33656E+00,-0.33913E+00,-0.34164E+00,-0.34410E+00,
     &-0.34651E+00,-0.34887E+00,-0.35118E+00,-0.35344E+00,-0.35567E+00,
     &-0.35785E+00,-0.35999E+00,-0.36209E+00,-0.36415E+00,-0.36618E+00,
     &-0.36817E+00,-0.37013E+00,-0.37206E+00,-0.37396E+00,-0.37583E+00,
     &-0.37767E+00,-0.37948E+00,-0.38127E+00,-0.38303E+00,-0.38476E+00,
     &-0.38648E+00,-0.38816E+00,-0.38983E+00,-0.39148E+00,-0.39310E+00,
     &-0.39470E+00,-0.39629E+00,-0.39786E+00,-0.39941E+00,-0.40094E+00,
     &-0.40245E+00,-0.40395E+00,-0.40543E+00,-0.40690E+00,-0.40836E+00,
     &-0.40980E+00,-0.41122E+00,-0.41264E+00,-0.41404E+00,-0.41543E+00,
     &-0.41681E+00,-0.41817E+00,-0.41953E+00,-0.42087E+00,-0.42221E+00,
     &-0.42354E+00,-0.42485E+00,-0.42616E+00,-0.42745E+00,-0.42874E+00,
     &-0.43002E+00,-0.43129E+00,-0.43256E+00,-0.43381E+00,-0.43506E+00,
     &-0.43630E+00,-0.43753E+00,-0.43876E+00,-0.43998E+00,-0.44119E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.44239E+00,-0.44359E+00,-0.44478E+00,-0.44597E+00,-0.44715E+00,
     &-0.44832E+00,-0.44948E+00,-0.45064E+00,-0.45179E+00,-0.45294E+00,
     &-0.45408E+00,-0.45521E+00,-0.45634E+00,-0.45746E+00,-0.45858E+00,
     &-0.45969E+00,-0.46079E+00,-0.46189E+00,-0.46298E+00,-0.46407E+00,
     &-0.46515E+00,-0.46623E+00,-0.46730E+00,-0.46836E+00,-0.46942E+00,
     &-0.47047E+00,-0.47152E+00,-0.47257E+00,-0.47360E+00,-0.47464E+00,
     &-0.47566E+00,-0.47669E+00,-0.47770E+00,-0.47871E+00,-0.47972E+00,
     &-0.48072E+00,-0.48172E+00,-0.48271E+00,-0.48370E+00,-0.48468E+00,
     &-0.48566E+00,-0.48663E+00,-0.48760E+00,-0.48857E+00,-0.48953E+00,
     &-0.49048E+00,-0.49143E+00,-0.49238E+00,-0.49332E+00,-0.49426E+00,
     &-0.49520E+00,-0.49613E+00,-0.49705E+00,-0.49797E+00,-0.49889E+00,
     &-0.49980E+00,-0.50071E+00,-0.50162E+00,-0.50252E+00,-0.50342E+00,
     &-0.50432E+00,-0.50521E+00,-0.50609E+00,-0.50698E+00,-0.50786E+00,
     &-0.50873E+00,-0.50961E+00,-0.51047E+00,-0.51134E+00,-0.51220E+00,
     &-0.51306E+00,-0.51392E+00,-0.51477E+00,-0.51562E+00,-0.51647E+00,
     &-0.51731E+00,-0.51815E+00,-0.51898E+00,-0.51982E+00,-0.52065E+00,
     &-0.52147E+00,-0.52230E+00,-0.52312E+00,-0.52394E+00,-0.52475E+00,
     &-0.52556E+00,-0.52637E+00,-0.52718E+00,-0.52798E+00,-0.52878E+00,
     &-0.52958E+00,-0.53038E+00,-0.53117E+00,-0.53196E+00,-0.53275E+00,
     &-0.53353E+00,-0.53431E+00,-0.53509E+00,-0.53587E+00,-0.53664E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.53741E+00,-0.53818E+00,-0.53895E+00,-0.53971E+00,-0.54047E+00,
     &-0.54123E+00,-0.54199E+00,-0.54274E+00,-0.54349E+00,-0.54424E+00,
     &-0.54499E+00,-0.54573E+00,-0.54648E+00,-0.54722E+00,-0.54796E+00,
     &-0.54869E+00,-0.54942E+00,-0.55016E+00,-0.55088E+00,-0.55161E+00,
     &-0.55234E+00,-0.55306E+00,-0.55378E+00,-0.55450E+00,-0.55521E+00,
     &-0.55593E+00,-0.55664E+00,-0.55735E+00,-0.55806E+00,-0.55876E+00,
     &-0.55947E+00,-0.56017E+00,-0.56087E+00,-0.56156E+00,-0.56226E+00,
     &-0.56295E+00,-0.56365E+00,-0.56434E+00,-0.56503E+00,-0.56571E+00,
     &-0.56640E+00,-0.56708E+00,-0.56776E+00,-0.56844E+00,-0.56912E+00,
     &-0.56979E+00,-0.57046E+00,-0.57114E+00,-0.57181E+00,-0.57248E+00,
     &-0.57314E+00,-0.57381E+00,-0.57447E+00,-0.57513E+00,-0.57579E+00,
     &-0.57645E+00,-0.57711E+00,-0.57776E+00,-0.57841E+00,-0.57907E+00,
     &-0.57971E+00,-0.58036E+00,-0.58101E+00,-0.58165E+00,-0.58230E+00,
     &-0.58294E+00,-0.58358E+00,-0.58422E+00,-0.58486E+00,-0.58549E+00,
     &-0.58613E+00,-0.58676E+00,-0.58739E+00,-0.58802E+00,-0.58865E+00,
     &-0.58928E+00,-0.58990E+00,-0.59052E+00,-0.59115E+00,-0.59177E+00,
     &-0.59239E+00,-0.59301E+00,-0.59362E+00,-0.59424E+00,-0.59485E+00,
     &-0.59546E+00,-0.59607E+00,-0.59668E+00,-0.59729E+00,-0.59790E+00,
     &-0.59851E+00,-0.59911E+00,-0.59971E+00,-0.60031E+00,-0.60092E+00,
     &-0.60151E+00,-0.60211E+00,-0.60271E+00,-0.60330E+00,-0.60390E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.60449E+00,-0.60508E+00,-0.60567E+00,-0.60626E+00,-0.60685E+00,
     &-0.60744E+00,-0.60802E+00,-0.60860E+00,-0.60919E+00,-0.60977E+00,
     &-0.61035E+00,-0.61093E+00,-0.61151E+00,-0.61208E+00,-0.61266E+00,
     &-0.61323E+00,-0.61381E+00,-0.61438E+00,-0.61495E+00,-0.61552E+00,
     &-0.61609E+00,-0.61666E+00,-0.61722E+00,-0.61779E+00,-0.61835E+00,
     &-0.61892E+00,-0.61948E+00,-0.62004E+00,-0.62060E+00,-0.62116E+00,
     &-0.62171E+00,-0.62227E+00,-0.62283E+00,-0.62338E+00,-0.62393E+00,
     &-0.62449E+00,-0.62504E+00,-0.62559E+00,-0.62614E+00,-0.62669E+00,
     &-0.62723E+00,-0.62778E+00,-0.62833E+00,-0.62887E+00,-0.62941E+00,
     &-0.62996E+00,-0.63050E+00,-0.63104E+00,-0.63158E+00,-0.63211E+00,
     &-0.63265E+00,-0.63319E+00,-0.63372E+00,-0.63426E+00,-0.63479E+00,
     &-0.63532E+00,-0.63586E+00,-0.63639E+00,-0.63692E+00,-0.63744E+00,
     &-0.63797E+00,-0.63850E+00,-0.63903E+00,-0.63955E+00,-0.64007E+00,
     &-0.64060E+00,-0.64112E+00,-0.64164E+00,-0.64216E+00,-0.64268E+00,
     &-0.64320E+00,-0.64372E+00,-0.64424E+00,-0.64475E+00,-0.64527E+00,
     &-0.64578E+00,-0.64630E+00,-0.64681E+00,-0.64732E+00,-0.64783E+00,
     &-0.64834E+00,-0.64885E+00,-0.64936E+00,-0.64987E+00,-0.65037E+00,
     &-0.65088E+00,-0.65139E+00,-0.65189E+00,-0.65239E+00,-0.65290E+00,
     &-0.65340E+00,-0.65390E+00,-0.65440E+00,-0.65490E+00,-0.65540E+00,
     &-0.65590E+00,-0.65639E+00,-0.65689E+00,-0.65739E+00,-0.65788E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.65838E+00,-0.65887E+00,-0.65936E+00,-0.65985E+00,-0.66035E+00,
     &-0.66084E+00,-0.66133E+00,-0.66181E+00,-0.66230E+00,-0.66279E+00,
     &-0.66328E+00,-0.66376E+00,-0.66425E+00,-0.66473E+00,-0.66522E+00,
     &-0.66570E+00,-0.66618E+00,-0.66666E+00,-0.66714E+00,-0.66762E+00,
     &-0.66810E+00,-0.66858E+00,-0.66906E+00,-0.66954E+00,-0.67001E+00,
     &-0.67049E+00,-0.67097E+00,-0.67144E+00,-0.67191E+00,-0.67239E+00,
     &-0.67286E+00,-0.67333E+00,-0.67380E+00,-0.67427E+00,-0.67474E+00,
     &-0.67521E+00,-0.67568E+00,-0.67615E+00,-0.67662E+00,-0.67708E+00,
     &-0.67755E+00,-0.67801E+00,-0.67848E+00,-0.67894E+00,-0.67941E+00,
     &-0.67987E+00,-0.68033E+00,-0.68079E+00,-0.68125E+00,-0.68171E+00,
     &-0.68217E+00,-0.68263E+00,-0.68309E+00,-0.68355E+00,-0.68400E+00,
     &-0.68446E+00,-0.68492E+00,-0.68537E+00,-0.68583E+00,-0.68628E+00,
     &-0.68673E+00,-0.68719E+00,-0.68764E+00,-0.68809E+00,-0.68854E+00,
     &-0.68899E+00,-0.68944E+00,-0.68989E+00,-0.69034E+00,-0.69079E+00,
     &-0.69124E+00,-0.69168E+00,-0.69213E+00,-0.69257E+00,-0.69302E+00,
     &-0.69346E+00,-0.69391E+00,-0.69435E+00,-0.69480E+00,-0.69524E+00,
     &-0.69568E+00,-0.69612E+00,-0.69656E+00,-0.69700E+00,-0.69744E+00,
     &-0.69788E+00,-0.69832E+00,-0.69876E+00,-0.69919E+00,-0.69963E+00,
     &-0.70007E+00,-0.70050E+00,-0.70094E+00,-0.70137E+00,-0.70181E+00,
     &-0.70224E+00,-0.70267E+00,-0.70311E+00,-0.70354E+00,-0.70397E+00/

      DATA (BNC03M(I),I=501,600)/
     &-0.70440E+00,-0.70483E+00,-0.70526E+00,-0.70569E+00,-0.70612E+00,
     &-0.70655E+00,-0.70698E+00,-0.70740E+00,-0.70783E+00,-0.70826E+00,
     &-0.70868E+00,-0.70911E+00,-0.70953E+00,-0.70996E+00,-0.71038E+00,
     &-0.71081E+00,-0.71123E+00,-0.71165E+00,-0.71207E+00,-0.71250E+00,
     &-0.71292E+00,-0.71334E+00,-0.71376E+00,-0.71418E+00,-0.71460E+00,
     &-0.71501E+00,-0.71543E+00,-0.71585E+00,-0.71627E+00,-0.71668E+00,
     &-0.71710E+00,-0.71752E+00,-0.71793E+00,-0.71835E+00,-0.71876E+00,
     &-0.71917E+00,-0.71959E+00,-0.72000E+00,-0.72041E+00,-0.72083E+00,
     &-0.72124E+00,-0.72165E+00,-0.72206E+00,-0.72247E+00,-0.72288E+00,
     &-0.72329E+00,-0.72370E+00,-0.72411E+00,-0.72451E+00,-0.72492E+00,
     &-0.72533E+00,-0.72574E+00,-0.72614E+00,-0.72655E+00,-0.72695E+00,
     &-0.72736E+00,-0.72776E+00,-0.72817E+00,-0.72857E+00,-0.72897E+00,
     &-0.72938E+00,-0.72978E+00,-0.73018E+00,-0.73058E+00,-0.73098E+00,
     &-0.73139E+00,-0.73179E+00,-0.73219E+00,-0.73259E+00,-0.73298E+00,
     &-0.73338E+00,-0.73378E+00,-0.73418E+00,-0.73458E+00,-0.73497E+00,
     &-0.73537E+00,-0.73577E+00,-0.73616E+00,-0.73656E+00,-0.73695E+00,
     &-0.73735E+00,-0.73774E+00,-0.73814E+00,-0.73853E+00,-0.73892E+00,
     &-0.73932E+00,-0.73971E+00,-0.74010E+00,-0.74049E+00,-0.74088E+00,
     &-0.74127E+00,-0.74166E+00,-0.74205E+00,-0.74244E+00,-0.74283E+00,
     &-0.74322E+00,-0.74361E+00,-0.74400E+00,-0.74439E+00,-0.74477E+00/

      DATA (BNC03M(I),I=601,700)/
     &-0.74893E+00,-0.75275E+00,-0.75652E+00,-0.76027E+00,-0.76398E+00,
     &-0.76765E+00,-0.77130E+00,-0.77492E+00,-0.77850E+00,-0.78206E+00,
     &-0.78558E+00,-0.78908E+00,-0.79256E+00,-0.79600E+00,-0.79942E+00,
     &-0.80282E+00,-0.80619E+00,-0.80954E+00,-0.81286E+00,-0.81617E+00,
     &-0.81944E+00,-0.82270E+00,-0.82594E+00,-0.82915E+00,-0.83235E+00,
     &-0.83552E+00,-0.83868E+00,-0.84181E+00,-0.84493E+00,-0.84803E+00,
     &-0.85111E+00,-0.85417E+00,-0.85722E+00,-0.86025E+00,-0.86326E+00,
     &-0.86626E+00,-0.86924E+00,-0.87221E+00,-0.87515E+00,-0.87809E+00,
     &-0.88101E+00,-0.88391E+00,-0.88681E+00,-0.88968E+00,-0.89255E+00,
     &-0.89540E+00,-0.89823E+00,-0.90106E+00,-0.90387E+00,-0.90666E+00,
     &-0.90945E+00,-0.91222E+00,-0.91499E+00,-0.91773E+00,-0.92047E+00,
     &-0.92320E+00,-0.92592E+00,-0.92862E+00,-0.93131E+00,-0.93400E+00,
     &-0.93667E+00,-0.93933E+00,-0.94198E+00,-0.94462E+00,-0.94726E+00,
     &-0.94988E+00,-0.95249E+00,-0.95509E+00,-0.95769E+00,-0.96027E+00,
     &-0.96285E+00,-0.96541E+00,-0.96797E+00,-0.97052E+00,-0.97306E+00,
     &-0.97559E+00,-0.97812E+00,-0.98063E+00,-0.98314E+00,-0.98564E+00,
     &-0.98813E+00,-0.99062E+00,-0.99309E+00,-0.99556E+00,-0.99802E+00,
     &-0.10005E+01,-0.10029E+01,-0.10054E+01,-0.10078E+01,-0.10102E+01,
     &-0.10126E+01,-0.10150E+01,-0.10175E+01,-0.10198E+01,-0.10222E+01,
     &-0.10246E+01,-0.10270E+01,-0.10294E+01,-0.10317E+01,-0.10341E+01/

      DATA (BNC03M(I),I=701,741)/
     &-0.10365E+01,-0.10388E+01,-0.10411E+01,-0.10435E+01,-0.10458E+01,
     &-0.10481E+01,-0.10505E+01,-0.10528E+01,-0.10551E+01,-0.10574E+01,
     &-0.10597E+01,-0.10620E+01,-0.10642E+01,-0.10665E+01,-0.10688E+01,
     &-0.10711E+01,-0.10733E+01,-0.10756E+01,-0.10778E+01,-0.10801E+01,
     &-0.10823E+01,-0.10846E+01,-0.10868E+01,-0.10890E+01,-0.10913E+01,
     &-0.10935E+01,-0.10957E+01,-0.10979E+01,-0.11001E+01,-0.11023E+01,
     &-0.11045E+01,-0.11067E+01,-0.11089E+01,-0.11111E+01,-0.11133E+01,
     &-0.11154E+01,-0.11176E+01,-0.11198E+01,-0.11219E+01,-0.11241E+01,
     &-0.11263E+01
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.95602E-01,-0.20838E+00,-0.26497E+00,-0.30509E+00,-0.33666E+00,
     &-0.36287E+00,-0.38536E+00,-0.40512E+00,-0.42276E+00,-0.43873E+00,
     &-0.45332E+00,-0.46676E+00,-0.47924E+00,-0.49089E+00,-0.50182E+00,
     &-0.51211E+00,-0.52185E+00,-0.53109E+00,-0.53988E+00,-0.54827E+00,
     &-0.55629E+00,-0.56398E+00,-0.57137E+00,-0.57848E+00,-0.58532E+00,
     &-0.59193E+00,-0.59832E+00,-0.60450E+00,-0.61049E+00,-0.61630E+00,
     &-0.62194E+00,-0.62742E+00,-0.63275E+00,-0.63794E+00,-0.64299E+00,
     &-0.64792E+00,-0.65273E+00,-0.65742E+00,-0.66201E+00,-0.66649E+00,
     &-0.67087E+00,-0.67516E+00,-0.67936E+00,-0.68347E+00,-0.68750E+00,
     &-0.69145E+00,-0.69533E+00,-0.69913E+00,-0.70286E+00,-0.70653E+00,
     &-0.71013E+00,-0.71367E+00,-0.71715E+00,-0.72057E+00,-0.72393E+00,
     &-0.72724E+00,-0.73050E+00,-0.73371E+00,-0.73687E+00,-0.73999E+00,
     &-0.74306E+00,-0.74609E+00,-0.74907E+00,-0.75202E+00,-0.75492E+00,
     &-0.75779E+00,-0.76063E+00,-0.76342E+00,-0.76619E+00,-0.76892E+00,
     &-0.77162E+00,-0.77428E+00,-0.77692E+00,-0.77953E+00,-0.78212E+00,
     &-0.78467E+00,-0.78720E+00,-0.78971E+00,-0.79219E+00,-0.79464E+00,
     &-0.79708E+00,-0.79949E+00,-0.80188E+00,-0.80425E+00,-0.80660E+00,
     &-0.80893E+00,-0.81124E+00,-0.81353E+00,-0.81581E+00,-0.81807E+00,
     &-0.82031E+00,-0.82253E+00,-0.82474E+00,-0.82693E+00,-0.82910E+00,
     &-0.83127E+00,-0.83341E+00,-0.83554E+00,-0.83766E+00,-0.83976E+00/

      DATA (BNC04M(I),I=101,200)/
     &-0.84185E+00,-0.84393E+00,-0.84599E+00,-0.84804E+00,-0.85007E+00,
     &-0.85210E+00,-0.85411E+00,-0.85611E+00,-0.85809E+00,-0.86007E+00,
     &-0.86203E+00,-0.86398E+00,-0.86592E+00,-0.86785E+00,-0.86977E+00,
     &-0.87167E+00,-0.87357E+00,-0.87545E+00,-0.87732E+00,-0.87918E+00,
     &-0.88103E+00,-0.88287E+00,-0.88470E+00,-0.88652E+00,-0.88833E+00,
     &-0.89013E+00,-0.89192E+00,-0.89370E+00,-0.89547E+00,-0.89724E+00,
     &-0.89899E+00,-0.90073E+00,-0.90246E+00,-0.90419E+00,-0.90590E+00,
     &-0.90761E+00,-0.90930E+00,-0.91099E+00,-0.91267E+00,-0.91434E+00,
     &-0.91601E+00,-0.91766E+00,-0.91931E+00,-0.92094E+00,-0.92257E+00,
     &-0.92420E+00,-0.92581E+00,-0.92742E+00,-0.92902E+00,-0.93061E+00,
     &-0.93219E+00,-0.93377E+00,-0.93534E+00,-0.93690E+00,-0.93846E+00,
     &-0.94000E+00,-0.94155E+00,-0.94308E+00,-0.94461E+00,-0.94613E+00,
     &-0.94764E+00,-0.94915E+00,-0.95065E+00,-0.95215E+00,-0.95363E+00,
     &-0.95512E+00,-0.95659E+00,-0.95806E+00,-0.95953E+00,-0.96099E+00,
     &-0.96244E+00,-0.96388E+00,-0.96532E+00,-0.96676E+00,-0.96819E+00,
     &-0.96961E+00,-0.97103E+00,-0.97244E+00,-0.97385E+00,-0.97525E+00,
     &-0.97665E+00,-0.97804E+00,-0.97942E+00,-0.98080E+00,-0.98218E+00,
     &-0.98355E+00,-0.98491E+00,-0.98627E+00,-0.98763E+00,-0.98898E+00,
     &-0.99033E+00,-0.99167E+00,-0.99300E+00,-0.99434E+00,-0.99566E+00,
     &-0.99699E+00,-0.99830E+00,-0.99962E+00,-0.10009E+01,-0.10022E+01/

      DATA (BNC04M(I),I=201,300)/
     &-0.10035E+01,-0.10048E+01,-0.10061E+01,-0.10074E+01,-0.10087E+01,
     &-0.10100E+01,-0.10112E+01,-0.10125E+01,-0.10138E+01,-0.10150E+01,
     &-0.10163E+01,-0.10176E+01,-0.10188E+01,-0.10200E+01,-0.10213E+01,
     &-0.10225E+01,-0.10238E+01,-0.10250E+01,-0.10262E+01,-0.10274E+01,
     &-0.10287E+01,-0.10299E+01,-0.10311E+01,-0.10323E+01,-0.10335E+01,
     &-0.10347E+01,-0.10359E+01,-0.10371E+01,-0.10383E+01,-0.10395E+01,
     &-0.10407E+01,-0.10419E+01,-0.10430E+01,-0.10442E+01,-0.10454E+01,
     &-0.10465E+01,-0.10477E+01,-0.10489E+01,-0.10500E+01,-0.10512E+01,
     &-0.10523E+01,-0.10535E+01,-0.10546E+01,-0.10558E+01,-0.10569E+01,
     &-0.10581E+01,-0.10592E+01,-0.10603E+01,-0.10615E+01,-0.10626E+01,
     &-0.10637E+01,-0.10648E+01,-0.10659E+01,-0.10670E+01,-0.10682E+01,
     &-0.10693E+01,-0.10704E+01,-0.10715E+01,-0.10726E+01,-0.10737E+01,
     &-0.10748E+01,-0.10759E+01,-0.10769E+01,-0.10780E+01,-0.10791E+01,
     &-0.10802E+01,-0.10813E+01,-0.10824E+01,-0.10834E+01,-0.10845E+01,
     &-0.10856E+01,-0.10866E+01,-0.10877E+01,-0.10888E+01,-0.10898E+01,
     &-0.10909E+01,-0.10919E+01,-0.10930E+01,-0.10940E+01,-0.10951E+01,
     &-0.10961E+01,-0.10972E+01,-0.10982E+01,-0.10992E+01,-0.11003E+01,
     &-0.11013E+01,-0.11023E+01,-0.11034E+01,-0.11044E+01,-0.11054E+01,
     &-0.11064E+01,-0.11074E+01,-0.11085E+01,-0.11095E+01,-0.11105E+01,
     &-0.11115E+01,-0.11125E+01,-0.11135E+01,-0.11145E+01,-0.11155E+01/

      DATA (BNC04M(I),I=301,400)/
     &-0.11165E+01,-0.11175E+01,-0.11185E+01,-0.11195E+01,-0.11205E+01,
     &-0.11215E+01,-0.11225E+01,-0.11234E+01,-0.11244E+01,-0.11254E+01,
     &-0.11264E+01,-0.11274E+01,-0.11283E+01,-0.11293E+01,-0.11303E+01,
     &-0.11313E+01,-0.11322E+01,-0.11332E+01,-0.11342E+01,-0.11351E+01,
     &-0.11361E+01,-0.11370E+01,-0.11380E+01,-0.11389E+01,-0.11399E+01,
     &-0.11408E+01,-0.11418E+01,-0.11427E+01,-0.11437E+01,-0.11446E+01,
     &-0.11456E+01,-0.11465E+01,-0.11475E+01,-0.11484E+01,-0.11493E+01,
     &-0.11503E+01,-0.11512E+01,-0.11521E+01,-0.11531E+01,-0.11540E+01,
     &-0.11549E+01,-0.11558E+01,-0.11567E+01,-0.11577E+01,-0.11586E+01,
     &-0.11595E+01,-0.11604E+01,-0.11613E+01,-0.11622E+01,-0.11632E+01,
     &-0.11641E+01,-0.11650E+01,-0.11659E+01,-0.11668E+01,-0.11677E+01,
     &-0.11686E+01,-0.11695E+01,-0.11704E+01,-0.11713E+01,-0.11722E+01,
     &-0.11731E+01,-0.11740E+01,-0.11749E+01,-0.11757E+01,-0.11766E+01,
     &-0.11775E+01,-0.11784E+01,-0.11793E+01,-0.11802E+01,-0.11810E+01,
     &-0.11819E+01,-0.11828E+01,-0.11837E+01,-0.11846E+01,-0.11854E+01,
     &-0.11863E+01,-0.11872E+01,-0.11880E+01,-0.11889E+01,-0.11898E+01,
     &-0.11906E+01,-0.11915E+01,-0.11924E+01,-0.11932E+01,-0.11941E+01,
     &-0.11949E+01,-0.11958E+01,-0.11967E+01,-0.11975E+01,-0.11984E+01,
     &-0.11992E+01,-0.12001E+01,-0.12009E+01,-0.12018E+01,-0.12026E+01,
     &-0.12035E+01,-0.12043E+01,-0.12051E+01,-0.12060E+01,-0.12068E+01/

      DATA (BNC04M(I),I=401,500)/
     &-0.12077E+01,-0.12085E+01,-0.12093E+01,-0.12102E+01,-0.12110E+01,
     &-0.12118E+01,-0.12127E+01,-0.12135E+01,-0.12143E+01,-0.12152E+01,
     &-0.12160E+01,-0.12168E+01,-0.12176E+01,-0.12185E+01,-0.12193E+01,
     &-0.12201E+01,-0.12209E+01,-0.12217E+01,-0.12226E+01,-0.12234E+01,
     &-0.12242E+01,-0.12250E+01,-0.12258E+01,-0.12266E+01,-0.12274E+01,
     &-0.12283E+01,-0.12291E+01,-0.12299E+01,-0.12307E+01,-0.12315E+01,
     &-0.12323E+01,-0.12331E+01,-0.12339E+01,-0.12347E+01,-0.12355E+01,
     &-0.12363E+01,-0.12371E+01,-0.12379E+01,-0.12387E+01,-0.12395E+01,
     &-0.12403E+01,-0.12411E+01,-0.12419E+01,-0.12426E+01,-0.12434E+01,
     &-0.12442E+01,-0.12450E+01,-0.12458E+01,-0.12466E+01,-0.12474E+01,
     &-0.12482E+01,-0.12489E+01,-0.12497E+01,-0.12505E+01,-0.12513E+01,
     &-0.12521E+01,-0.12528E+01,-0.12536E+01,-0.12544E+01,-0.12552E+01,
     &-0.12559E+01,-0.12567E+01,-0.12575E+01,-0.12583E+01,-0.12590E+01,
     &-0.12598E+01,-0.12606E+01,-0.12613E+01,-0.12621E+01,-0.12629E+01,
     &-0.12636E+01,-0.12644E+01,-0.12652E+01,-0.12659E+01,-0.12667E+01,
     &-0.12674E+01,-0.12682E+01,-0.12690E+01,-0.12697E+01,-0.12705E+01,
     &-0.12712E+01,-0.12720E+01,-0.12727E+01,-0.12735E+01,-0.12742E+01,
     &-0.12750E+01,-0.12757E+01,-0.12765E+01,-0.12772E+01,-0.12780E+01,
     &-0.12787E+01,-0.12795E+01,-0.12802E+01,-0.12810E+01,-0.12817E+01,
     &-0.12825E+01,-0.12832E+01,-0.12839E+01,-0.12847E+01,-0.12854E+01/

      DATA (BNC04M(I),I=501,600)/
     &-0.12862E+01,-0.12869E+01,-0.12876E+01,-0.12884E+01,-0.12891E+01,
     &-0.12898E+01,-0.12906E+01,-0.12913E+01,-0.12920E+01,-0.12928E+01,
     &-0.12935E+01,-0.12942E+01,-0.12950E+01,-0.12957E+01,-0.12964E+01,
     &-0.12971E+01,-0.12979E+01,-0.12986E+01,-0.12993E+01,-0.13000E+01,
     &-0.13008E+01,-0.13015E+01,-0.13022E+01,-0.13029E+01,-0.13036E+01,
     &-0.13044E+01,-0.13051E+01,-0.13058E+01,-0.13065E+01,-0.13072E+01,
     &-0.13079E+01,-0.13087E+01,-0.13094E+01,-0.13101E+01,-0.13108E+01,
     &-0.13115E+01,-0.13122E+01,-0.13129E+01,-0.13136E+01,-0.13143E+01,
     &-0.13151E+01,-0.13158E+01,-0.13165E+01,-0.13172E+01,-0.13179E+01,
     &-0.13186E+01,-0.13193E+01,-0.13200E+01,-0.13207E+01,-0.13214E+01,
     &-0.13221E+01,-0.13228E+01,-0.13235E+01,-0.13242E+01,-0.13249E+01,
     &-0.13256E+01,-0.13263E+01,-0.13270E+01,-0.13277E+01,-0.13284E+01,
     &-0.13291E+01,-0.13298E+01,-0.13304E+01,-0.13311E+01,-0.13318E+01,
     &-0.13325E+01,-0.13332E+01,-0.13339E+01,-0.13346E+01,-0.13353E+01,
     &-0.13360E+01,-0.13367E+01,-0.13373E+01,-0.13380E+01,-0.13387E+01,
     &-0.13394E+01,-0.13401E+01,-0.13408E+01,-0.13414E+01,-0.13421E+01,
     &-0.13428E+01,-0.13435E+01,-0.13442E+01,-0.13448E+01,-0.13455E+01,
     &-0.13462E+01,-0.13469E+01,-0.13476E+01,-0.13482E+01,-0.13489E+01,
     &-0.13496E+01,-0.13503E+01,-0.13509E+01,-0.13516E+01,-0.13523E+01,
     &-0.13530E+01,-0.13536E+01,-0.13543E+01,-0.13550E+01,-0.13556E+01/

      DATA (BNC04M(I),I=601,700)/
     &-0.13628E+01,-0.13694E+01,-0.13760E+01,-0.13825E+01,-0.13889E+01,
     &-0.13953E+01,-0.14016E+01,-0.14079E+01,-0.14142E+01,-0.14204E+01,
     &-0.14265E+01,-0.14327E+01,-0.14387E+01,-0.14448E+01,-0.14507E+01,
     &-0.14567E+01,-0.14626E+01,-0.14685E+01,-0.14743E+01,-0.14801E+01,
     &-0.14859E+01,-0.14916E+01,-0.14973E+01,-0.15030E+01,-0.15087E+01,
     &-0.15143E+01,-0.15198E+01,-0.15254E+01,-0.15309E+01,-0.15364E+01,
     &-0.15419E+01,-0.15473E+01,-0.15527E+01,-0.15581E+01,-0.15635E+01,
     &-0.15688E+01,-0.15741E+01,-0.15794E+01,-0.15846E+01,-0.15899E+01,
     &-0.15951E+01,-0.16003E+01,-0.16055E+01,-0.16106E+01,-0.16157E+01,
     &-0.16208E+01,-0.16259E+01,-0.16310E+01,-0.16360E+01,-0.16411E+01,
     &-0.16461E+01,-0.16511E+01,-0.16560E+01,-0.16610E+01,-0.16659E+01,
     &-0.16708E+01,-0.16757E+01,-0.16806E+01,-0.16855E+01,-0.16903E+01,
     &-0.16952E+01,-0.17000E+01,-0.17048E+01,-0.17096E+01,-0.17143E+01,
     &-0.17191E+01,-0.17238E+01,-0.17285E+01,-0.17333E+01,-0.17379E+01,
     &-0.17426E+01,-0.17473E+01,-0.17520E+01,-0.17566E+01,-0.17612E+01,
     &-0.17658E+01,-0.17704E+01,-0.17750E+01,-0.17796E+01,-0.17842E+01,
     &-0.17887E+01,-0.17933E+01,-0.17978E+01,-0.18023E+01,-0.18068E+01,
     &-0.18113E+01,-0.18158E+01,-0.18203E+01,-0.18247E+01,-0.18292E+01,
     &-0.18336E+01,-0.18380E+01,-0.18425E+01,-0.18469E+01,-0.18513E+01,
     &-0.18556E+01,-0.18600E+01,-0.18644E+01,-0.18687E+01,-0.18731E+01/

      DATA (BNC04M(I),I=701,741)/
     &-0.18774E+01,-0.18818E+01,-0.18861E+01,-0.18904E+01,-0.18947E+01,
     &-0.18990E+01,-0.19033E+01,-0.19075E+01,-0.19118E+01,-0.19161E+01,
     &-0.19203E+01,-0.19245E+01,-0.19288E+01,-0.19330E+01,-0.19372E+01,
     &-0.19414E+01,-0.19456E+01,-0.19498E+01,-0.19540E+01,-0.19582E+01,
     &-0.19623E+01,-0.19665E+01,-0.19707E+01,-0.19748E+01,-0.19789E+01,
     &-0.19831E+01,-0.19872E+01,-0.19913E+01,-0.19954E+01,-0.19995E+01,
     &-0.20036E+01,-0.20077E+01,-0.20118E+01,-0.20159E+01,-0.20199E+01,
     &-0.20240E+01,-0.20280E+01,-0.20321E+01,-0.20361E+01,-0.20402E+01,
     &-0.20442E+01
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.48407E-01,-0.10752E+00,-0.13832E+00,-0.16078E+00,-0.17887E+00,
     &-0.19423E+00,-0.20767E+00,-0.21970E+00,-0.23063E+00,-0.24069E+00,
     &-0.25003E+00,-0.25876E+00,-0.26699E+00,-0.27478E+00,-0.28218E+00,
     &-0.28924E+00,-0.29599E+00,-0.30248E+00,-0.30872E+00,-0.31474E+00,
     &-0.32056E+00,-0.32619E+00,-0.33165E+00,-0.33695E+00,-0.34210E+00,
     &-0.34712E+00,-0.35200E+00,-0.35676E+00,-0.36141E+00,-0.36595E+00,
     &-0.37039E+00,-0.37472E+00,-0.37897E+00,-0.38313E+00,-0.38720E+00,
     &-0.39119E+00,-0.39511E+00,-0.39895E+00,-0.40271E+00,-0.40642E+00,
     &-0.41005E+00,-0.41362E+00,-0.41713E+00,-0.42058E+00,-0.42397E+00,
     &-0.42731E+00,-0.43059E+00,-0.43382E+00,-0.43700E+00,-0.44014E+00,
     &-0.44323E+00,-0.44627E+00,-0.44927E+00,-0.45222E+00,-0.45514E+00,
     &-0.45802E+00,-0.46085E+00,-0.46366E+00,-0.46642E+00,-0.46916E+00,
     &-0.47186E+00,-0.47453E+00,-0.47717E+00,-0.47978E+00,-0.48236E+00,
     &-0.48491E+00,-0.48744E+00,-0.48995E+00,-0.49243E+00,-0.49489E+00,
     &-0.49732E+00,-0.49974E+00,-0.50213E+00,-0.50451E+00,-0.50687E+00,
     &-0.50921E+00,-0.51154E+00,-0.51385E+00,-0.51614E+00,-0.51842E+00,
     &-0.52069E+00,-0.52295E+00,-0.52519E+00,-0.52742E+00,-0.52963E+00,
     &-0.53184E+00,-0.53404E+00,-0.53622E+00,-0.53840E+00,-0.54057E+00,
     &-0.54273E+00,-0.54487E+00,-0.54701E+00,-0.54915E+00,-0.55127E+00,
     &-0.55338E+00,-0.55549E+00,-0.55759E+00,-0.55968E+00,-0.56176E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.56384E+00,-0.56591E+00,-0.56797E+00,-0.57002E+00,-0.57206E+00,
     &-0.57410E+00,-0.57613E+00,-0.57815E+00,-0.58016E+00,-0.58217E+00,
     &-0.58417E+00,-0.58616E+00,-0.58814E+00,-0.59011E+00,-0.59208E+00,
     &-0.59404E+00,-0.59599E+00,-0.59793E+00,-0.59986E+00,-0.60178E+00,
     &-0.60370E+00,-0.60561E+00,-0.60751E+00,-0.60940E+00,-0.61128E+00,
     &-0.61316E+00,-0.61503E+00,-0.61688E+00,-0.61873E+00,-0.62058E+00,
     &-0.62241E+00,-0.62424E+00,-0.62605E+00,-0.62786E+00,-0.62966E+00,
     &-0.63146E+00,-0.63324E+00,-0.63502E+00,-0.63679E+00,-0.63855E+00,
     &-0.64030E+00,-0.64204E+00,-0.64378E+00,-0.64551E+00,-0.64723E+00,
     &-0.64895E+00,-0.65066E+00,-0.65235E+00,-0.65405E+00,-0.65573E+00,
     &-0.65741E+00,-0.65908E+00,-0.66074E+00,-0.66240E+00,-0.66405E+00,
     &-0.66569E+00,-0.66733E+00,-0.66895E+00,-0.67058E+00,-0.67219E+00,
     &-0.67380E+00,-0.67540E+00,-0.67700E+00,-0.67859E+00,-0.68017E+00,
     &-0.68174E+00,-0.68331E+00,-0.68488E+00,-0.68643E+00,-0.68798E+00,
     &-0.68953E+00,-0.69107E+00,-0.69260E+00,-0.69413E+00,-0.69565E+00,
     &-0.69716E+00,-0.69867E+00,-0.70017E+00,-0.70167E+00,-0.70316E+00,
     &-0.70465E+00,-0.70613E+00,-0.70760E+00,-0.70907E+00,-0.71054E+00,
     &-0.71199E+00,-0.71345E+00,-0.71489E+00,-0.71634E+00,-0.71777E+00,
     &-0.71920E+00,-0.72063E+00,-0.72205E+00,-0.72347E+00,-0.72488E+00,
     &-0.72629E+00,-0.72769E+00,-0.72908E+00,-0.73047E+00,-0.73186E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.73324E+00,-0.73462E+00,-0.73599E+00,-0.73736E+00,-0.73872E+00,
     &-0.74008E+00,-0.74143E+00,-0.74278E+00,-0.74412E+00,-0.74546E+00,
     &-0.74679E+00,-0.74812E+00,-0.74945E+00,-0.75077E+00,-0.75209E+00,
     &-0.75340E+00,-0.75471E+00,-0.75601E+00,-0.75731E+00,-0.75861E+00,
     &-0.75990E+00,-0.76118E+00,-0.76246E+00,-0.76374E+00,-0.76502E+00,
     &-0.76629E+00,-0.76755E+00,-0.76881E+00,-0.77007E+00,-0.77133E+00,
     &-0.77257E+00,-0.77382E+00,-0.77506E+00,-0.77630E+00,-0.77753E+00,
     &-0.77876E+00,-0.77999E+00,-0.78121E+00,-0.78243E+00,-0.78365E+00,
     &-0.78486E+00,-0.78607E+00,-0.78727E+00,-0.78847E+00,-0.78967E+00,
     &-0.79086E+00,-0.79205E+00,-0.79323E+00,-0.79441E+00,-0.79559E+00,
     &-0.79677E+00,-0.79794E+00,-0.79911E+00,-0.80027E+00,-0.80143E+00,
     &-0.80259E+00,-0.80375E+00,-0.80490E+00,-0.80604E+00,-0.80719E+00,
     &-0.80833E+00,-0.80947E+00,-0.81060E+00,-0.81173E+00,-0.81286E+00,
     &-0.81398E+00,-0.81510E+00,-0.81622E+00,-0.81734E+00,-0.81845E+00,
     &-0.81956E+00,-0.82066E+00,-0.82177E+00,-0.82287E+00,-0.82396E+00,
     &-0.82505E+00,-0.82614E+00,-0.82723E+00,-0.82832E+00,-0.82940E+00,
     &-0.83048E+00,-0.83155E+00,-0.83262E+00,-0.83369E+00,-0.83476E+00,
     &-0.83582E+00,-0.83688E+00,-0.83794E+00,-0.83899E+00,-0.84005E+00,
     &-0.84110E+00,-0.84214E+00,-0.84319E+00,-0.84423E+00,-0.84527E+00,
     &-0.84630E+00,-0.84733E+00,-0.84836E+00,-0.84939E+00,-0.85041E+00/

      DATA (BNC05M(I),I=301,400)/
     &-0.85144E+00,-0.85246E+00,-0.85347E+00,-0.85449E+00,-0.85550E+00,
     &-0.85651E+00,-0.85751E+00,-0.85852E+00,-0.85952E+00,-0.86051E+00,
     &-0.86151E+00,-0.86250E+00,-0.86349E+00,-0.86448E+00,-0.86547E+00,
     &-0.86645E+00,-0.86743E+00,-0.86841E+00,-0.86939E+00,-0.87036E+00,
     &-0.87133E+00,-0.87230E+00,-0.87327E+00,-0.87423E+00,-0.87519E+00,
     &-0.87615E+00,-0.87711E+00,-0.87806E+00,-0.87901E+00,-0.87996E+00,
     &-0.88091E+00,-0.88185E+00,-0.88280E+00,-0.88374E+00,-0.88468E+00,
     &-0.88561E+00,-0.88655E+00,-0.88748E+00,-0.88841E+00,-0.88933E+00,
     &-0.89026E+00,-0.89118E+00,-0.89210E+00,-0.89302E+00,-0.89394E+00,
     &-0.89485E+00,-0.89576E+00,-0.89667E+00,-0.89758E+00,-0.89849E+00,
     &-0.89939E+00,-0.90029E+00,-0.90119E+00,-0.90209E+00,-0.90298E+00,
     &-0.90387E+00,-0.90477E+00,-0.90565E+00,-0.90654E+00,-0.90743E+00,
     &-0.90831E+00,-0.90919E+00,-0.91007E+00,-0.91095E+00,-0.91182E+00,
     &-0.91269E+00,-0.91357E+00,-0.91443E+00,-0.91530E+00,-0.91617E+00,
     &-0.91703E+00,-0.91789E+00,-0.91875E+00,-0.91961E+00,-0.92046E+00,
     &-0.92132E+00,-0.92217E+00,-0.92302E+00,-0.92387E+00,-0.92471E+00,
     &-0.92556E+00,-0.92640E+00,-0.92724E+00,-0.92808E+00,-0.92892E+00,
     &-0.92975E+00,-0.93059E+00,-0.93142E+00,-0.93225E+00,-0.93308E+00,
     &-0.93390E+00,-0.93473E+00,-0.93555E+00,-0.93637E+00,-0.93719E+00,
     &-0.93801E+00,-0.93883E+00,-0.93964E+00,-0.94045E+00,-0.94126E+00/

      DATA (BNC05M(I),I=401,500)/
     &-0.94207E+00,-0.94288E+00,-0.94369E+00,-0.94449E+00,-0.94529E+00,
     &-0.94609E+00,-0.94689E+00,-0.94769E+00,-0.94849E+00,-0.94928E+00,
     &-0.95007E+00,-0.95087E+00,-0.95166E+00,-0.95244E+00,-0.95323E+00,
     &-0.95401E+00,-0.95480E+00,-0.95558E+00,-0.95636E+00,-0.95714E+00,
     &-0.95791E+00,-0.95869E+00,-0.95946E+00,-0.96023E+00,-0.96101E+00,
     &-0.96177E+00,-0.96254E+00,-0.96331E+00,-0.96407E+00,-0.96484E+00,
     &-0.96560E+00,-0.96636E+00,-0.96712E+00,-0.96787E+00,-0.96863E+00,
     &-0.96938E+00,-0.97014E+00,-0.97089E+00,-0.97164E+00,-0.97238E+00,
     &-0.97313E+00,-0.97388E+00,-0.97462E+00,-0.97536E+00,-0.97610E+00,
     &-0.97684E+00,-0.97758E+00,-0.97832E+00,-0.97905E+00,-0.97979E+00,
     &-0.98052E+00,-0.98125E+00,-0.98198E+00,-0.98271E+00,-0.98344E+00,
     &-0.98416E+00,-0.98489E+00,-0.98561E+00,-0.98633E+00,-0.98705E+00,
     &-0.98777E+00,-0.98849E+00,-0.98921E+00,-0.98992E+00,-0.99064E+00,
     &-0.99135E+00,-0.99206E+00,-0.99277E+00,-0.99348E+00,-0.99419E+00,
     &-0.99489E+00,-0.99560E+00,-0.99630E+00,-0.99700E+00,-0.99770E+00,
     &-0.99840E+00,-0.99910E+00,-0.99980E+00,-0.10005E+01,-0.10012E+01,
     &-0.10019E+01,-0.10026E+01,-0.10033E+01,-0.10040E+01,-0.10046E+01,
     &-0.10053E+01,-0.10060E+01,-0.10067E+01,-0.10074E+01,-0.10081E+01,
     &-0.10087E+01,-0.10094E+01,-0.10101E+01,-0.10108E+01,-0.10115E+01,
     &-0.10121E+01,-0.10128E+01,-0.10135E+01,-0.10142E+01,-0.10148E+01/

      DATA (BNC05M(I),I=501,600)/
     &-0.10155E+01,-0.10162E+01,-0.10168E+01,-0.10175E+01,-0.10182E+01,
     &-0.10188E+01,-0.10195E+01,-0.10201E+01,-0.10208E+01,-0.10215E+01,
     &-0.10221E+01,-0.10228E+01,-0.10234E+01,-0.10241E+01,-0.10247E+01,
     &-0.10254E+01,-0.10260E+01,-0.10267E+01,-0.10273E+01,-0.10280E+01,
     &-0.10286E+01,-0.10293E+01,-0.10299E+01,-0.10306E+01,-0.10312E+01,
     &-0.10319E+01,-0.10325E+01,-0.10331E+01,-0.10338E+01,-0.10344E+01,
     &-0.10351E+01,-0.10357E+01,-0.10363E+01,-0.10370E+01,-0.10376E+01,
     &-0.10382E+01,-0.10389E+01,-0.10395E+01,-0.10401E+01,-0.10407E+01,
     &-0.10414E+01,-0.10420E+01,-0.10426E+01,-0.10432E+01,-0.10439E+01,
     &-0.10445E+01,-0.10451E+01,-0.10457E+01,-0.10463E+01,-0.10470E+01,
     &-0.10476E+01,-0.10482E+01,-0.10488E+01,-0.10494E+01,-0.10500E+01,
     &-0.10506E+01,-0.10512E+01,-0.10519E+01,-0.10525E+01,-0.10531E+01,
     &-0.10537E+01,-0.10543E+01,-0.10549E+01,-0.10555E+01,-0.10561E+01,
     &-0.10567E+01,-0.10573E+01,-0.10579E+01,-0.10585E+01,-0.10591E+01,
     &-0.10597E+01,-0.10603E+01,-0.10609E+01,-0.10615E+01,-0.10621E+01,
     &-0.10627E+01,-0.10633E+01,-0.10639E+01,-0.10644E+01,-0.10650E+01,
     &-0.10656E+01,-0.10662E+01,-0.10668E+01,-0.10674E+01,-0.10680E+01,
     &-0.10685E+01,-0.10691E+01,-0.10697E+01,-0.10703E+01,-0.10709E+01,
     &-0.10715E+01,-0.10720E+01,-0.10726E+01,-0.10732E+01,-0.10738E+01,
     &-0.10743E+01,-0.10749E+01,-0.10755E+01,-0.10761E+01,-0.10766E+01/

      DATA (BNC05M(I),I=601,700)/
     &-0.10828E+01,-0.10883E+01,-0.10939E+01,-0.10993E+01,-0.11047E+01,
     &-0.11099E+01,-0.11152E+01,-0.11203E+01,-0.11254E+01,-0.11304E+01,
     &-0.11354E+01,-0.11403E+01,-0.11451E+01,-0.11499E+01,-0.11546E+01,
     &-0.11593E+01,-0.11639E+01,-0.11685E+01,-0.11730E+01,-0.11775E+01,
     &-0.11819E+01,-0.11863E+01,-0.11906E+01,-0.11949E+01,-0.11991E+01,
     &-0.12033E+01,-0.12075E+01,-0.12116E+01,-0.12156E+01,-0.12197E+01,
     &-0.12237E+01,-0.12276E+01,-0.12316E+01,-0.12355E+01,-0.12393E+01,
     &-0.12431E+01,-0.12469E+01,-0.12507E+01,-0.12544E+01,-0.12581E+01,
     &-0.12618E+01,-0.12654E+01,-0.12690E+01,-0.12726E+01,-0.12761E+01,
     &-0.12797E+01,-0.12831E+01,-0.12866E+01,-0.12901E+01,-0.12935E+01,
     &-0.12969E+01,-0.13002E+01,-0.13036E+01,-0.13069E+01,-0.13102E+01,
     &-0.13135E+01,-0.13167E+01,-0.13200E+01,-0.13232E+01,-0.13264E+01,
     &-0.13295E+01,-0.13327E+01,-0.13358E+01,-0.13389E+01,-0.13420E+01,
     &-0.13451E+01,-0.13481E+01,-0.13512E+01,-0.13542E+01,-0.13572E+01,
     &-0.13601E+01,-0.13631E+01,-0.13661E+01,-0.13690E+01,-0.13719E+01,
     &-0.13748E+01,-0.13777E+01,-0.13806E+01,-0.13834E+01,-0.13863E+01,
     &-0.13891E+01,-0.13919E+01,-0.13947E+01,-0.13975E+01,-0.14002E+01,
     &-0.14030E+01,-0.14057E+01,-0.14085E+01,-0.14112E+01,-0.14139E+01,
     &-0.14166E+01,-0.14192E+01,-0.14219E+01,-0.14246E+01,-0.14272E+01,
     &-0.14298E+01,-0.14325E+01,-0.14351E+01,-0.14377E+01,-0.14402E+01/

      DATA (BNC05M(I),I=701,741)/
     &-0.14428E+01,-0.14454E+01,-0.14479E+01,-0.14505E+01,-0.14530E+01,
     &-0.14555E+01,-0.14581E+01,-0.14606E+01,-0.14631E+01,-0.14655E+01,
     &-0.14680E+01,-0.14705E+01,-0.14729E+01,-0.14754E+01,-0.14778E+01,
     &-0.14803E+01,-0.14827E+01,-0.14851E+01,-0.14875E+01,-0.14899E+01,
     &-0.14923E+01,-0.14947E+01,-0.14970E+01,-0.14994E+01,-0.15017E+01,
     &-0.15041E+01,-0.15064E+01,-0.15088E+01,-0.15111E+01,-0.15134E+01,
     &-0.15157E+01,-0.15180E+01,-0.15203E+01,-0.15226E+01,-0.15249E+01,
     &-0.15272E+01,-0.15294E+01,-0.15317E+01,-0.15340E+01,-0.15362E+01,
     &-0.15385E+01
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.47150E-01,-0.10066E+00,-0.12632E+00,-0.14389E+00,-0.15726E+00,
     &-0.16803E+00,-0.17700E+00,-0.18464E+00,-0.19127E+00,-0.19709E+00,
     &-0.20226E+00,-0.20689E+00,-0.21106E+00,-0.21484E+00,-0.21828E+00,
     &-0.22143E+00,-0.22432E+00,-0.22698E+00,-0.22943E+00,-0.23171E+00,
     &-0.23382E+00,-0.23578E+00,-0.23761E+00,-0.23932E+00,-0.24091E+00,
     &-0.24240E+00,-0.24380E+00,-0.24511E+00,-0.24634E+00,-0.24750E+00,
     &-0.24859E+00,-0.24961E+00,-0.25058E+00,-0.25149E+00,-0.25235E+00,
     &-0.25316E+00,-0.25393E+00,-0.25466E+00,-0.25535E+00,-0.25600E+00,
     &-0.25662E+00,-0.25721E+00,-0.25776E+00,-0.25829E+00,-0.25879E+00,
     &-0.25927E+00,-0.25972E+00,-0.26015E+00,-0.26056E+00,-0.26095E+00,
     &-0.26132E+00,-0.26167E+00,-0.26201E+00,-0.26232E+00,-0.26262E+00,
     &-0.26291E+00,-0.26318E+00,-0.26344E+00,-0.26368E+00,-0.26391E+00,
     &-0.26412E+00,-0.26433E+00,-0.26452E+00,-0.26470E+00,-0.26486E+00,
     &-0.26502E+00,-0.26516E+00,-0.26529E+00,-0.26541E+00,-0.26552E+00,
     &-0.26562E+00,-0.26570E+00,-0.26578E+00,-0.26584E+00,-0.26589E+00,
     &-0.26594E+00,-0.26597E+00,-0.26599E+00,-0.26600E+00,-0.26600E+00,
     &-0.26598E+00,-0.26596E+00,-0.26593E+00,-0.26589E+00,-0.26583E+00,
     &-0.26577E+00,-0.26569E+00,-0.26561E+00,-0.26551E+00,-0.26541E+00,
     &-0.26529E+00,-0.26517E+00,-0.26504E+00,-0.26490E+00,-0.26474E+00,
     &-0.26458E+00,-0.26441E+00,-0.26424E+00,-0.26405E+00,-0.26386E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.26365E+00,-0.26345E+00,-0.26323E+00,-0.26300E+00,-0.26277E+00,
     &-0.26254E+00,-0.26229E+00,-0.26204E+00,-0.26179E+00,-0.26153E+00,
     &-0.26126E+00,-0.26099E+00,-0.26071E+00,-0.26043E+00,-0.26014E+00,
     &-0.25985E+00,-0.25955E+00,-0.25925E+00,-0.25895E+00,-0.25864E+00,
     &-0.25833E+00,-0.25802E+00,-0.25770E+00,-0.25738E+00,-0.25706E+00,
     &-0.25673E+00,-0.25640E+00,-0.25607E+00,-0.25574E+00,-0.25540E+00,
     &-0.25506E+00,-0.25472E+00,-0.25438E+00,-0.25404E+00,-0.25369E+00,
     &-0.25335E+00,-0.25300E+00,-0.25265E+00,-0.25230E+00,-0.25194E+00,
     &-0.25159E+00,-0.25123E+00,-0.25088E+00,-0.25052E+00,-0.25016E+00,
     &-0.24980E+00,-0.24944E+00,-0.24908E+00,-0.24872E+00,-0.24835E+00,
     &-0.24799E+00,-0.24762E+00,-0.24726E+00,-0.24689E+00,-0.24652E+00,
     &-0.24615E+00,-0.24578E+00,-0.24541E+00,-0.24504E+00,-0.24467E+00,
     &-0.24430E+00,-0.24393E+00,-0.24356E+00,-0.24318E+00,-0.24281E+00,
     &-0.24244E+00,-0.24206E+00,-0.24169E+00,-0.24131E+00,-0.24094E+00,
     &-0.24056E+00,-0.24019E+00,-0.23981E+00,-0.23943E+00,-0.23906E+00,
     &-0.23868E+00,-0.23830E+00,-0.23792E+00,-0.23755E+00,-0.23717E+00,
     &-0.23679E+00,-0.23641E+00,-0.23603E+00,-0.23566E+00,-0.23528E+00,
     &-0.23490E+00,-0.23452E+00,-0.23414E+00,-0.23376E+00,-0.23339E+00,
     &-0.23301E+00,-0.23263E+00,-0.23225E+00,-0.23187E+00,-0.23149E+00,
     &-0.23111E+00,-0.23073E+00,-0.23036E+00,-0.22998E+00,-0.22960E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.22922E+00,-0.22884E+00,-0.22847E+00,-0.22809E+00,-0.22771E+00,
     &-0.22733E+00,-0.22695E+00,-0.22658E+00,-0.22620E+00,-0.22582E+00,
     &-0.22545E+00,-0.22507E+00,-0.22469E+00,-0.22432E+00,-0.22394E+00,
     &-0.22357E+00,-0.22319E+00,-0.22282E+00,-0.22244E+00,-0.22207E+00,
     &-0.22169E+00,-0.22132E+00,-0.22094E+00,-0.22057E+00,-0.22020E+00,
     &-0.21982E+00,-0.21945E+00,-0.21908E+00,-0.21871E+00,-0.21833E+00,
     &-0.21796E+00,-0.21759E+00,-0.21722E+00,-0.21685E+00,-0.21648E+00,
     &-0.21611E+00,-0.21574E+00,-0.21537E+00,-0.21500E+00,-0.21463E+00,
     &-0.21426E+00,-0.21390E+00,-0.21353E+00,-0.21316E+00,-0.21280E+00,
     &-0.21243E+00,-0.21206E+00,-0.21170E+00,-0.21133E+00,-0.21097E+00,
     &-0.21060E+00,-0.21024E+00,-0.20987E+00,-0.20951E+00,-0.20915E+00,
     &-0.20879E+00,-0.20842E+00,-0.20806E+00,-0.20770E+00,-0.20734E+00,
     &-0.20698E+00,-0.20662E+00,-0.20626E+00,-0.20590E+00,-0.20554E+00,
     &-0.20518E+00,-0.20483E+00,-0.20447E+00,-0.20411E+00,-0.20376E+00,
     &-0.20340E+00,-0.20304E+00,-0.20269E+00,-0.20233E+00,-0.20198E+00,
     &-0.20163E+00,-0.20127E+00,-0.20092E+00,-0.20057E+00,-0.20022E+00,
     &-0.19986E+00,-0.19951E+00,-0.19916E+00,-0.19881E+00,-0.19846E+00,
     &-0.19811E+00,-0.19776E+00,-0.19742E+00,-0.19707E+00,-0.19672E+00,
     &-0.19637E+00,-0.19603E+00,-0.19568E+00,-0.19534E+00,-0.19499E+00,
     &-0.19465E+00,-0.19430E+00,-0.19396E+00,-0.19362E+00,-0.19327E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.19293E+00,-0.19259E+00,-0.19225E+00,-0.19191E+00,-0.19157E+00,
     &-0.19123E+00,-0.19089E+00,-0.19055E+00,-0.19021E+00,-0.18988E+00,
     &-0.18954E+00,-0.18920E+00,-0.18887E+00,-0.18853E+00,-0.18819E+00,
     &-0.18786E+00,-0.18752E+00,-0.18719E+00,-0.18686E+00,-0.18652E+00,
     &-0.18619E+00,-0.18586E+00,-0.18553E+00,-0.18520E+00,-0.18487E+00,
     &-0.18454E+00,-0.18421E+00,-0.18388E+00,-0.18355E+00,-0.18322E+00,
     &-0.18290E+00,-0.18257E+00,-0.18224E+00,-0.18192E+00,-0.18159E+00,
     &-0.18127E+00,-0.18094E+00,-0.18062E+00,-0.18029E+00,-0.17997E+00,
     &-0.17965E+00,-0.17933E+00,-0.17901E+00,-0.17868E+00,-0.17836E+00,
     &-0.17804E+00,-0.17772E+00,-0.17741E+00,-0.17709E+00,-0.17677E+00,
     &-0.17645E+00,-0.17613E+00,-0.17582E+00,-0.17550E+00,-0.17519E+00,
     &-0.17487E+00,-0.17456E+00,-0.17424E+00,-0.17393E+00,-0.17362E+00,
     &-0.17330E+00,-0.17299E+00,-0.17268E+00,-0.17237E+00,-0.17206E+00,
     &-0.17175E+00,-0.17144E+00,-0.17113E+00,-0.17082E+00,-0.17051E+00,
     &-0.17020E+00,-0.16990E+00,-0.16959E+00,-0.16928E+00,-0.16898E+00,
     &-0.16867E+00,-0.16837E+00,-0.16806E+00,-0.16776E+00,-0.16746E+00,
     &-0.16715E+00,-0.16685E+00,-0.16655E+00,-0.16625E+00,-0.16595E+00,
     &-0.16565E+00,-0.16535E+00,-0.16505E+00,-0.16475E+00,-0.16445E+00,
     &-0.16415E+00,-0.16385E+00,-0.16355E+00,-0.16326E+00,-0.16296E+00,
     &-0.16267E+00,-0.16237E+00,-0.16208E+00,-0.16178E+00,-0.16149E+00/

      DATA (BNC06M(I),I=401,500)/
     &-0.16119E+00,-0.16090E+00,-0.16061E+00,-0.16032E+00,-0.16003E+00,
     &-0.15973E+00,-0.15944E+00,-0.15915E+00,-0.15886E+00,-0.15858E+00,
     &-0.15829E+00,-0.15800E+00,-0.15771E+00,-0.15742E+00,-0.15714E+00,
     &-0.15685E+00,-0.15656E+00,-0.15628E+00,-0.15599E+00,-0.15571E+00,
     &-0.15543E+00,-0.15514E+00,-0.15486E+00,-0.15458E+00,-0.15429E+00,
     &-0.15401E+00,-0.15373E+00,-0.15345E+00,-0.15317E+00,-0.15289E+00,
     &-0.15261E+00,-0.15233E+00,-0.15205E+00,-0.15177E+00,-0.15150E+00,
     &-0.15122E+00,-0.15094E+00,-0.15067E+00,-0.15039E+00,-0.15012E+00,
     &-0.14984E+00,-0.14957E+00,-0.14929E+00,-0.14902E+00,-0.14875E+00,
     &-0.14847E+00,-0.14820E+00,-0.14793E+00,-0.14766E+00,-0.14739E+00,
     &-0.14712E+00,-0.14685E+00,-0.14658E+00,-0.14631E+00,-0.14604E+00,
     &-0.14577E+00,-0.14550E+00,-0.14523E+00,-0.14497E+00,-0.14470E+00,
     &-0.14443E+00,-0.14417E+00,-0.14390E+00,-0.14364E+00,-0.14337E+00,
     &-0.14311E+00,-0.14285E+00,-0.14258E+00,-0.14232E+00,-0.14206E+00,
     &-0.14180E+00,-0.14154E+00,-0.14127E+00,-0.14101E+00,-0.14075E+00,
     &-0.14049E+00,-0.14024E+00,-0.13998E+00,-0.13972E+00,-0.13946E+00,
     &-0.13920E+00,-0.13895E+00,-0.13869E+00,-0.13843E+00,-0.13818E+00,
     &-0.13792E+00,-0.13767E+00,-0.13741E+00,-0.13716E+00,-0.13690E+00,
     &-0.13665E+00,-0.13640E+00,-0.13614E+00,-0.13589E+00,-0.13564E+00,
     &-0.13539E+00,-0.13514E+00,-0.13489E+00,-0.13464E+00,-0.13439E+00/

      DATA (BNC06M(I),I=501,600)/
     &-0.13414E+00,-0.13389E+00,-0.13364E+00,-0.13339E+00,-0.13315E+00,
     &-0.13290E+00,-0.13265E+00,-0.13241E+00,-0.13216E+00,-0.13191E+00,
     &-0.13167E+00,-0.13142E+00,-0.13118E+00,-0.13094E+00,-0.13069E+00,
     &-0.13045E+00,-0.13021E+00,-0.12996E+00,-0.12972E+00,-0.12948E+00,
     &-0.12924E+00,-0.12900E+00,-0.12876E+00,-0.12852E+00,-0.12828E+00,
     &-0.12804E+00,-0.12780E+00,-0.12756E+00,-0.12732E+00,-0.12708E+00,
     &-0.12685E+00,-0.12661E+00,-0.12637E+00,-0.12614E+00,-0.12590E+00,
     &-0.12567E+00,-0.12543E+00,-0.12520E+00,-0.12496E+00,-0.12473E+00,
     &-0.12449E+00,-0.12426E+00,-0.12403E+00,-0.12380E+00,-0.12356E+00,
     &-0.12333E+00,-0.12310E+00,-0.12287E+00,-0.12264E+00,-0.12241E+00,
     &-0.12218E+00,-0.12195E+00,-0.12172E+00,-0.12149E+00,-0.12127E+00,
     &-0.12104E+00,-0.12081E+00,-0.12058E+00,-0.12036E+00,-0.12013E+00,
     &-0.11990E+00,-0.11968E+00,-0.11945E+00,-0.11923E+00,-0.11900E+00,
     &-0.11878E+00,-0.11856E+00,-0.11833E+00,-0.11811E+00,-0.11789E+00,
     &-0.11766E+00,-0.11744E+00,-0.11722E+00,-0.11700E+00,-0.11678E+00,
     &-0.11656E+00,-0.11634E+00,-0.11612E+00,-0.11590E+00,-0.11568E+00,
     &-0.11546E+00,-0.11524E+00,-0.11502E+00,-0.11481E+00,-0.11459E+00,
     &-0.11437E+00,-0.11416E+00,-0.11394E+00,-0.11372E+00,-0.11351E+00,
     &-0.11329E+00,-0.11308E+00,-0.11286E+00,-0.11265E+00,-0.11244E+00,
     &-0.11222E+00,-0.11201E+00,-0.11180E+00,-0.11158E+00,-0.11137E+00/

      DATA (BNC06M(I),I=601,700)/
     &-0.10910E+00,-0.10704E+00,-0.10500E+00,-0.10300E+00,-0.10104E+00,
     &-0.99099E-01,-0.97194E-01,-0.95319E-01,-0.93474E-01,-0.91659E-01,
     &-0.89872E-01,-0.88114E-01,-0.86385E-01,-0.84682E-01,-0.83007E-01,
     &-0.81358E-01,-0.79736E-01,-0.78139E-01,-0.76568E-01,-0.75022E-01,
     &-0.73500E-01,-0.72002E-01,-0.70529E-01,-0.69079E-01,-0.67651E-01,
     &-0.66247E-01,-0.64865E-01,-0.63505E-01,-0.62167E-01,-0.60850E-01,
     &-0.59554E-01,-0.58278E-01,-0.57024E-01,-0.55789E-01,-0.54574E-01,
     &-0.53379E-01,-0.52203E-01,-0.51046E-01,-0.49908E-01,-0.48788E-01,
     &-0.47686E-01,-0.46602E-01,-0.45536E-01,-0.44487E-01,-0.43456E-01,
     &-0.42441E-01,-0.41444E-01,-0.40462E-01,-0.39497E-01,-0.38548E-01,
     &-0.37615E-01,-0.36697E-01,-0.35795E-01,-0.34908E-01,-0.34037E-01,
     &-0.33180E-01,-0.32337E-01,-0.31509E-01,-0.30696E-01,-0.29896E-01,
     &-0.29110E-01,-0.28338E-01,-0.27580E-01,-0.26835E-01,-0.26103E-01,
     &-0.25384E-01,-0.24679E-01,-0.23985E-01,-0.23305E-01,-0.22637E-01,
     &-0.21981E-01,-0.21338E-01,-0.20706E-01,-0.20086E-01,-0.19478E-01,
     &-0.18882E-01,-0.18297E-01,-0.17724E-01,-0.17161E-01,-0.16610E-01,
     &-0.16070E-01,-0.15540E-01,-0.15021E-01,-0.14513E-01,-0.14015E-01,
     &-0.13528E-01,-0.13050E-01,-0.12583E-01,-0.12126E-01,-0.11679E-01,
     &-0.11242E-01,-0.10814E-01,-0.10396E-01,-0.99873E-02,-0.95881E-02,
     &-0.91981E-02,-0.88174E-02,-0.84457E-02,-0.80831E-02,-0.77293E-02/

      DATA (BNC06M(I),I=701,741)/
     &-0.73844E-02,-0.70483E-02,-0.67208E-02,-0.64018E-02,-0.60913E-02,
     &-0.57891E-02,-0.54953E-02,-0.52097E-02,-0.49322E-02,-0.46628E-02,
     &-0.44013E-02,-0.41478E-02,-0.39020E-02,-0.36640E-02,-0.34338E-02,
     &-0.32110E-02,-0.29958E-02,-0.27881E-02,-0.25878E-02,-0.23947E-02,
     &-0.22090E-02,-0.20303E-02,-0.18588E-02,-0.16944E-02,-0.15370E-02,
     &-0.13864E-02,-0.12427E-02,-0.11059E-02,-0.97561E-03,-0.85213E-03,
     &-0.73525E-03,-0.62493E-03,-0.52107E-03,-0.42366E-03,-0.33261E-03,
     &-0.24793E-03,-0.16947E-03,-0.97264E-04,-0.31222E-04, 0.28782E-04,
     & 0.82683E-04
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.95410E-01,-0.20734E+00,-0.26314E+00,-0.30252E+00,-0.33337E+00,
     &-0.35888E+00,-0.38069E+00,-0.39978E+00,-0.41677E+00,-0.43209E+00,
     &-0.44604E+00,-0.45886E+00,-0.47072E+00,-0.48176E+00,-0.49209E+00,
     &-0.50179E+00,-0.51093E+00,-0.51959E+00,-0.52781E+00,-0.53563E+00,
     &-0.54309E+00,-0.55022E+00,-0.55706E+00,-0.56362E+00,-0.56993E+00,
     &-0.57600E+00,-0.58186E+00,-0.58752E+00,-0.59299E+00,-0.59828E+00,
     &-0.60341E+00,-0.60839E+00,-0.61322E+00,-0.61792E+00,-0.62249E+00,
     &-0.62693E+00,-0.63126E+00,-0.63549E+00,-0.63961E+00,-0.64363E+00,
     &-0.64755E+00,-0.65139E+00,-0.65514E+00,-0.65881E+00,-0.66240E+00,
     &-0.66592E+00,-0.66937E+00,-0.67275E+00,-0.67606E+00,-0.67931E+00,
     &-0.68250E+00,-0.68563E+00,-0.68871E+00,-0.69173E+00,-0.69470E+00,
     &-0.69762E+00,-0.70049E+00,-0.70331E+00,-0.70609E+00,-0.70883E+00,
     &-0.71153E+00,-0.71418E+00,-0.71680E+00,-0.71938E+00,-0.72192E+00,
     &-0.72442E+00,-0.72690E+00,-0.72933E+00,-0.73174E+00,-0.73412E+00,
     &-0.73646E+00,-0.73878E+00,-0.74107E+00,-0.74333E+00,-0.74556E+00,
     &-0.74777E+00,-0.74996E+00,-0.75212E+00,-0.75425E+00,-0.75636E+00,
     &-0.75845E+00,-0.76052E+00,-0.76257E+00,-0.76460E+00,-0.76661E+00,
     &-0.76859E+00,-0.77056E+00,-0.77251E+00,-0.77444E+00,-0.77636E+00,
     &-0.77826E+00,-0.78014E+00,-0.78200E+00,-0.78385E+00,-0.78568E+00,
     &-0.78750E+00,-0.78930E+00,-0.79109E+00,-0.79286E+00,-0.79462E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.79637E+00,-0.79810E+00,-0.79982E+00,-0.80152E+00,-0.80322E+00,
     &-0.80490E+00,-0.80657E+00,-0.80822E+00,-0.80987E+00,-0.81150E+00,
     &-0.81312E+00,-0.81473E+00,-0.81633E+00,-0.81791E+00,-0.81949E+00,
     &-0.82106E+00,-0.82261E+00,-0.82416E+00,-0.82569E+00,-0.82722E+00,
     &-0.82873E+00,-0.83024E+00,-0.83173E+00,-0.83322E+00,-0.83469E+00,
     &-0.83616E+00,-0.83762E+00,-0.83907E+00,-0.84051E+00,-0.84194E+00,
     &-0.84337E+00,-0.84478E+00,-0.84619E+00,-0.84759E+00,-0.84898E+00,
     &-0.85036E+00,-0.85174E+00,-0.85310E+00,-0.85446E+00,-0.85581E+00,
     &-0.85716E+00,-0.85850E+00,-0.85983E+00,-0.86115E+00,-0.86246E+00,
     &-0.86377E+00,-0.86507E+00,-0.86637E+00,-0.86766E+00,-0.86894E+00,
     &-0.87022E+00,-0.87148E+00,-0.87275E+00,-0.87400E+00,-0.87525E+00,
     &-0.87650E+00,-0.87774E+00,-0.87897E+00,-0.88020E+00,-0.88142E+00,
     &-0.88263E+00,-0.88384E+00,-0.88504E+00,-0.88624E+00,-0.88744E+00,
     &-0.88862E+00,-0.88980E+00,-0.89098E+00,-0.89215E+00,-0.89332E+00,
     &-0.89448E+00,-0.89564E+00,-0.89679E+00,-0.89794E+00,-0.89908E+00,
     &-0.90022E+00,-0.90135E+00,-0.90248E+00,-0.90360E+00,-0.90472E+00,
     &-0.90583E+00,-0.90694E+00,-0.90805E+00,-0.90915E+00,-0.91024E+00,
     &-0.91134E+00,-0.91242E+00,-0.91351E+00,-0.91459E+00,-0.91566E+00,
     &-0.91673E+00,-0.91780E+00,-0.91887E+00,-0.91993E+00,-0.92098E+00,
     &-0.92203E+00,-0.92308E+00,-0.92413E+00,-0.92517E+00,-0.92620E+00/

      DATA (BNC07M(I),I=201,300)/
     &-0.92724E+00,-0.92827E+00,-0.92929E+00,-0.93032E+00,-0.93134E+00,
     &-0.93235E+00,-0.93336E+00,-0.93437E+00,-0.93538E+00,-0.93638E+00,
     &-0.93738E+00,-0.93837E+00,-0.93937E+00,-0.94035E+00,-0.94134E+00,
     &-0.94232E+00,-0.94330E+00,-0.94428E+00,-0.94525E+00,-0.94622E+00,
     &-0.94719E+00,-0.94815E+00,-0.94912E+00,-0.95007E+00,-0.95103E+00,
     &-0.95198E+00,-0.95293E+00,-0.95388E+00,-0.95482E+00,-0.95576E+00,
     &-0.95670E+00,-0.95764E+00,-0.95857E+00,-0.95950E+00,-0.96043E+00,
     &-0.96135E+00,-0.96228E+00,-0.96320E+00,-0.96411E+00,-0.96503E+00,
     &-0.96594E+00,-0.96685E+00,-0.96776E+00,-0.96866E+00,-0.96956E+00,
     &-0.97046E+00,-0.97136E+00,-0.97226E+00,-0.97315E+00,-0.97404E+00,
     &-0.97493E+00,-0.97581E+00,-0.97670E+00,-0.97758E+00,-0.97845E+00,
     &-0.97933E+00,-0.98021E+00,-0.98108E+00,-0.98195E+00,-0.98281E+00,
     &-0.98368E+00,-0.98454E+00,-0.98540E+00,-0.98626E+00,-0.98712E+00,
     &-0.98798E+00,-0.98883E+00,-0.98968E+00,-0.99053E+00,-0.99137E+00,
     &-0.99222E+00,-0.99306E+00,-0.99390E+00,-0.99474E+00,-0.99558E+00,
     &-0.99641E+00,-0.99725E+00,-0.99808E+00,-0.99891E+00,-0.99973E+00,
     &-0.10006E+01,-0.10014E+01,-0.10022E+01,-0.10030E+01,-0.10038E+01,
     &-0.10047E+01,-0.10055E+01,-0.10063E+01,-0.10071E+01,-0.10079E+01,
     &-0.10087E+01,-0.10095E+01,-0.10103E+01,-0.10111E+01,-0.10119E+01,
     &-0.10127E+01,-0.10135E+01,-0.10143E+01,-0.10151E+01,-0.10159E+01/

      DATA (BNC07M(I),I=301,400)/
     &-0.10167E+01,-0.10175E+01,-0.10183E+01,-0.10191E+01,-0.10198E+01,
     &-0.10206E+01,-0.10214E+01,-0.10222E+01,-0.10230E+01,-0.10237E+01,
     &-0.10245E+01,-0.10253E+01,-0.10261E+01,-0.10268E+01,-0.10276E+01,
     &-0.10284E+01,-0.10291E+01,-0.10299E+01,-0.10307E+01,-0.10314E+01,
     &-0.10322E+01,-0.10329E+01,-0.10337E+01,-0.10344E+01,-0.10352E+01,
     &-0.10360E+01,-0.10367E+01,-0.10375E+01,-0.10382E+01,-0.10390E+01,
     &-0.10397E+01,-0.10404E+01,-0.10412E+01,-0.10419E+01,-0.10427E+01,
     &-0.10434E+01,-0.10442E+01,-0.10449E+01,-0.10456E+01,-0.10464E+01,
     &-0.10471E+01,-0.10478E+01,-0.10486E+01,-0.10493E+01,-0.10500E+01,
     &-0.10507E+01,-0.10515E+01,-0.10522E+01,-0.10529E+01,-0.10536E+01,
     &-0.10544E+01,-0.10551E+01,-0.10558E+01,-0.10565E+01,-0.10572E+01,
     &-0.10579E+01,-0.10587E+01,-0.10594E+01,-0.10601E+01,-0.10608E+01,
     &-0.10615E+01,-0.10622E+01,-0.10629E+01,-0.10636E+01,-0.10643E+01,
     &-0.10650E+01,-0.10657E+01,-0.10664E+01,-0.10671E+01,-0.10678E+01,
     &-0.10685E+01,-0.10692E+01,-0.10699E+01,-0.10706E+01,-0.10713E+01,
     &-0.10720E+01,-0.10727E+01,-0.10734E+01,-0.10741E+01,-0.10748E+01,
     &-0.10755E+01,-0.10761E+01,-0.10768E+01,-0.10775E+01,-0.10782E+01,
     &-0.10789E+01,-0.10796E+01,-0.10802E+01,-0.10809E+01,-0.10816E+01,
     &-0.10823E+01,-0.10830E+01,-0.10836E+01,-0.10843E+01,-0.10850E+01,
     &-0.10857E+01,-0.10863E+01,-0.10870E+01,-0.10877E+01,-0.10883E+01/

      DATA (BNC07M(I),I=401,500)/
     &-0.10890E+01,-0.10897E+01,-0.10903E+01,-0.10910E+01,-0.10917E+01,
     &-0.10923E+01,-0.10930E+01,-0.10937E+01,-0.10943E+01,-0.10950E+01,
     &-0.10956E+01,-0.10963E+01,-0.10970E+01,-0.10976E+01,-0.10983E+01,
     &-0.10989E+01,-0.10996E+01,-0.11002E+01,-0.11009E+01,-0.11015E+01,
     &-0.11022E+01,-0.11028E+01,-0.11035E+01,-0.11041E+01,-0.11048E+01,
     &-0.11054E+01,-0.11061E+01,-0.11067E+01,-0.11074E+01,-0.11080E+01,
     &-0.11086E+01,-0.11093E+01,-0.11099E+01,-0.11106E+01,-0.11112E+01,
     &-0.11118E+01,-0.11125E+01,-0.11131E+01,-0.11138E+01,-0.11144E+01,
     &-0.11150E+01,-0.11157E+01,-0.11163E+01,-0.11169E+01,-0.11176E+01,
     &-0.11182E+01,-0.11188E+01,-0.11195E+01,-0.11201E+01,-0.11207E+01,
     &-0.11213E+01,-0.11220E+01,-0.11226E+01,-0.11232E+01,-0.11238E+01,
     &-0.11245E+01,-0.11251E+01,-0.11257E+01,-0.11263E+01,-0.11270E+01,
     &-0.11276E+01,-0.11282E+01,-0.11288E+01,-0.11294E+01,-0.11300E+01,
     &-0.11307E+01,-0.11313E+01,-0.11319E+01,-0.11325E+01,-0.11331E+01,
     &-0.11337E+01,-0.11344E+01,-0.11350E+01,-0.11356E+01,-0.11362E+01,
     &-0.11368E+01,-0.11374E+01,-0.11380E+01,-0.11386E+01,-0.11392E+01,
     &-0.11398E+01,-0.11404E+01,-0.11410E+01,-0.11417E+01,-0.11423E+01,
     &-0.11429E+01,-0.11435E+01,-0.11441E+01,-0.11447E+01,-0.11453E+01,
     &-0.11459E+01,-0.11465E+01,-0.11471E+01,-0.11477E+01,-0.11483E+01,
     &-0.11489E+01,-0.11495E+01,-0.11501E+01,-0.11507E+01,-0.11512E+01/

      DATA (BNC07M(I),I=501,600)/
     &-0.11518E+01,-0.11524E+01,-0.11530E+01,-0.11536E+01,-0.11542E+01,
     &-0.11548E+01,-0.11554E+01,-0.11560E+01,-0.11566E+01,-0.11572E+01,
     &-0.11578E+01,-0.11583E+01,-0.11589E+01,-0.11595E+01,-0.11601E+01,
     &-0.11607E+01,-0.11613E+01,-0.11619E+01,-0.11624E+01,-0.11630E+01,
     &-0.11636E+01,-0.11642E+01,-0.11648E+01,-0.11654E+01,-0.11659E+01,
     &-0.11665E+01,-0.11671E+01,-0.11677E+01,-0.11683E+01,-0.11688E+01,
     &-0.11694E+01,-0.11700E+01,-0.11706E+01,-0.11711E+01,-0.11717E+01,
     &-0.11723E+01,-0.11729E+01,-0.11734E+01,-0.11740E+01,-0.11746E+01,
     &-0.11752E+01,-0.11757E+01,-0.11763E+01,-0.11769E+01,-0.11774E+01,
     &-0.11780E+01,-0.11786E+01,-0.11792E+01,-0.11797E+01,-0.11803E+01,
     &-0.11809E+01,-0.11814E+01,-0.11820E+01,-0.11826E+01,-0.11831E+01,
     &-0.11837E+01,-0.11842E+01,-0.11848E+01,-0.11854E+01,-0.11859E+01,
     &-0.11865E+01,-0.11871E+01,-0.11876E+01,-0.11882E+01,-0.11887E+01,
     &-0.11893E+01,-0.11899E+01,-0.11904E+01,-0.11910E+01,-0.11915E+01,
     &-0.11921E+01,-0.11927E+01,-0.11932E+01,-0.11938E+01,-0.11943E+01,
     &-0.11949E+01,-0.11954E+01,-0.11960E+01,-0.11965E+01,-0.11971E+01,
     &-0.11977E+01,-0.11982E+01,-0.11988E+01,-0.11993E+01,-0.11999E+01,
     &-0.12004E+01,-0.12010E+01,-0.12015E+01,-0.12021E+01,-0.12026E+01,
     &-0.12032E+01,-0.12037E+01,-0.12043E+01,-0.12048E+01,-0.12054E+01,
     &-0.12059E+01,-0.12065E+01,-0.12070E+01,-0.12075E+01,-0.12081E+01/

      DATA (BNC07M(I),I=601,700)/
     &-0.12140E+01,-0.12193E+01,-0.12247E+01,-0.12300E+01,-0.12353E+01,
     &-0.12405E+01,-0.12457E+01,-0.12509E+01,-0.12560E+01,-0.12611E+01,
     &-0.12662E+01,-0.12712E+01,-0.12763E+01,-0.12812E+01,-0.12862E+01,
     &-0.12911E+01,-0.12960E+01,-0.13009E+01,-0.13058E+01,-0.13106E+01,
     &-0.13154E+01,-0.13202E+01,-0.13250E+01,-0.13297E+01,-0.13344E+01,
     &-0.13391E+01,-0.13438E+01,-0.13484E+01,-0.13531E+01,-0.13577E+01,
     &-0.13623E+01,-0.13669E+01,-0.13714E+01,-0.13760E+01,-0.13805E+01,
     &-0.13850E+01,-0.13895E+01,-0.13940E+01,-0.13984E+01,-0.14029E+01,
     &-0.14073E+01,-0.14117E+01,-0.14161E+01,-0.14205E+01,-0.14249E+01,
     &-0.14292E+01,-0.14336E+01,-0.14379E+01,-0.14422E+01,-0.14465E+01,
     &-0.14508E+01,-0.14551E+01,-0.14593E+01,-0.14636E+01,-0.14678E+01,
     &-0.14720E+01,-0.14763E+01,-0.14805E+01,-0.14847E+01,-0.14888E+01,
     &-0.14930E+01,-0.14972E+01,-0.15013E+01,-0.15055E+01,-0.15096E+01,
     &-0.15137E+01,-0.15178E+01,-0.15219E+01,-0.15260E+01,-0.15301E+01,
     &-0.15341E+01,-0.15382E+01,-0.15423E+01,-0.15463E+01,-0.15503E+01,
     &-0.15544E+01,-0.15584E+01,-0.15624E+01,-0.15664E+01,-0.15704E+01,
     &-0.15744E+01,-0.15783E+01,-0.15823E+01,-0.15863E+01,-0.15902E+01,
     &-0.15941E+01,-0.15981E+01,-0.16020E+01,-0.16059E+01,-0.16098E+01,
     &-0.16138E+01,-0.16176E+01,-0.16215E+01,-0.16254E+01,-0.16293E+01,
     &-0.16332E+01,-0.16370E+01,-0.16409E+01,-0.16448E+01,-0.16486E+01/

      DATA (BNC07M(I),I=701,741)/
     &-0.16524E+01,-0.16563E+01,-0.16601E+01,-0.16639E+01,-0.16677E+01,
     &-0.16715E+01,-0.16753E+01,-0.16791E+01,-0.16829E+01,-0.16867E+01,
     &-0.16905E+01,-0.16943E+01,-0.16980E+01,-0.17018E+01,-0.17055E+01,
     &-0.17093E+01,-0.17130E+01,-0.17168E+01,-0.17205E+01,-0.17243E+01,
     &-0.17280E+01,-0.17317E+01,-0.17354E+01,-0.17391E+01,-0.17428E+01,
     &-0.17465E+01,-0.17502E+01,-0.17539E+01,-0.17576E+01,-0.17613E+01,
     &-0.17650E+01,-0.17686E+01,-0.17723E+01,-0.17760E+01,-0.17796E+01,
     &-0.17833E+01,-0.17869E+01,-0.17906E+01,-0.17942E+01,-0.17978E+01,
     &-0.18015E+01
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.44732E-01,-0.88841E-01,-0.10667E+00,-0.11710E+00,-0.12376E+00,
     &-0.12807E+00,-0.13075E+00,-0.13222E+00,-0.13274E+00,-0.13249E+00,
     &-0.13160E+00,-0.13016E+00,-0.12825E+00,-0.12592E+00,-0.12321E+00,
     &-0.12016E+00,-0.11680E+00,-0.11316E+00,-0.10925E+00,-0.10509E+00,
     &-0.10070E+00,-0.96097E-01,-0.91288E-01,-0.86286E-01,-0.81100E-01,
     &-0.75739E-01,-0.70212E-01,-0.64527E-01,-0.58690E-01,-0.52709E-01,
     &-0.46590E-01,-0.40338E-01,-0.33960E-01,-0.27460E-01,-0.20845E-01,
     &-0.14118E-01,-0.72843E-02,-0.34872E-03, 0.66847E-02, 0.13812E-01,
     & 0.21029E-01, 0.28331E-01, 0.35716E-01, 0.43180E-01, 0.50719E-01,
     & 0.58331E-01, 0.66012E-01, 0.73759E-01, 0.81569E-01, 0.89441E-01,
     & 0.97371E-01, 0.10536E+00, 0.11340E+00, 0.12149E+00, 0.12963E+00,
     & 0.13783E+00, 0.14607E+00, 0.15435E+00, 0.16268E+00, 0.17106E+00,
     & 0.17947E+00, 0.18793E+00, 0.19643E+00, 0.20498E+00, 0.21356E+00,
     & 0.22219E+00, 0.23085E+00, 0.23956E+00, 0.24831E+00, 0.25710E+00,
     & 0.26593E+00, 0.27481E+00, 0.28372E+00, 0.29269E+00, 0.30169E+00,
     & 0.31074E+00, 0.31984E+00, 0.32898E+00, 0.33817E+00, 0.34741E+00,
     & 0.35669E+00, 0.36602E+00, 0.37541E+00, 0.38484E+00, 0.39432E+00,
     & 0.40385E+00, 0.41343E+00, 0.42306E+00, 0.43273E+00, 0.44246E+00,
     & 0.45224E+00, 0.46206E+00, 0.47193E+00, 0.48185E+00, 0.49181E+00,
     & 0.50182E+00, 0.51187E+00, 0.52196E+00, 0.53209E+00, 0.54227E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.55248E+00, 0.56272E+00, 0.57300E+00, 0.58331E+00, 0.59366E+00,
     & 0.60403E+00, 0.61442E+00, 0.62485E+00, 0.63529E+00, 0.64576E+00,
     & 0.65625E+00, 0.66675E+00, 0.67727E+00, 0.68780E+00, 0.69835E+00,
     & 0.70890E+00, 0.71946E+00, 0.73003E+00, 0.74061E+00, 0.75119E+00,
     & 0.76177E+00, 0.77235E+00, 0.78293E+00, 0.79351E+00, 0.80408E+00,
     & 0.81465E+00, 0.82522E+00, 0.83577E+00, 0.84633E+00, 0.85687E+00,
     & 0.86740E+00, 0.87792E+00, 0.88844E+00, 0.89894E+00, 0.90942E+00,
     & 0.91990E+00, 0.93036E+00, 0.94081E+00, 0.95124E+00, 0.96166E+00,
     & 0.97206E+00, 0.98244E+00, 0.99281E+00, 0.10032E+01, 0.10135E+01,
     & 0.10238E+01, 0.10341E+01, 0.10444E+01, 0.10547E+01, 0.10649E+01,
     & 0.10752E+01, 0.10854E+01, 0.10956E+01, 0.11057E+01, 0.11159E+01,
     & 0.11260E+01, 0.11361E+01, 0.11462E+01, 0.11563E+01, 0.11664E+01,
     & 0.11764E+01, 0.11864E+01, 0.11964E+01, 0.12064E+01, 0.12164E+01,
     & 0.12263E+01, 0.12362E+01, 0.12461E+01, 0.12560E+01, 0.12658E+01,
     & 0.12757E+01, 0.12855E+01, 0.12953E+01, 0.13051E+01, 0.13148E+01,
     & 0.13246E+01, 0.13343E+01, 0.13440E+01, 0.13536E+01, 0.13633E+01,
     & 0.13729E+01, 0.13825E+01, 0.13921E+01, 0.14017E+01, 0.14112E+01,
     & 0.14208E+01, 0.14303E+01, 0.14398E+01, 0.14492E+01, 0.14587E+01,
     & 0.14681E+01, 0.14775E+01, 0.14869E+01, 0.14963E+01, 0.15056E+01,
     & 0.15150E+01, 0.15243E+01, 0.15336E+01, 0.15428E+01, 0.15521E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.15613E+01, 0.15705E+01, 0.15797E+01, 0.15889E+01, 0.15980E+01,
     & 0.16071E+01, 0.16162E+01, 0.16253E+01, 0.16344E+01, 0.16434E+01,
     & 0.16525E+01, 0.16615E+01, 0.16705E+01, 0.16794E+01, 0.16884E+01,
     & 0.16973E+01, 0.17062E+01, 0.17151E+01, 0.17240E+01, 0.17328E+01,
     & 0.17417E+01, 0.17505E+01, 0.17593E+01, 0.17681E+01, 0.17768E+01,
     & 0.17855E+01, 0.17943E+01, 0.18030E+01, 0.18117E+01, 0.18203E+01,
     & 0.18290E+01, 0.18376E+01, 0.18462E+01, 0.18548E+01, 0.18634E+01,
     & 0.18719E+01, 0.18804E+01, 0.18890E+01, 0.18975E+01, 0.19059E+01,
     & 0.19144E+01, 0.19228E+01, 0.19313E+01, 0.19397E+01, 0.19481E+01,
     & 0.19564E+01, 0.19648E+01, 0.19731E+01, 0.19814E+01, 0.19897E+01,
     & 0.19980E+01, 0.20063E+01, 0.20145E+01, 0.20228E+01, 0.20310E+01,
     & 0.20392E+01, 0.20474E+01, 0.20555E+01, 0.20637E+01, 0.20718E+01,
     & 0.20799E+01, 0.20880E+01, 0.20961E+01, 0.21042E+01, 0.21122E+01,
     & 0.21202E+01, 0.21282E+01, 0.21362E+01, 0.21442E+01, 0.21522E+01,
     & 0.21601E+01, 0.21681E+01, 0.21760E+01, 0.21839E+01, 0.21917E+01,
     & 0.21996E+01, 0.22075E+01, 0.22153E+01, 0.22231E+01, 0.22309E+01,
     & 0.22387E+01, 0.22465E+01, 0.22542E+01, 0.22620E+01, 0.22697E+01,
     & 0.22774E+01, 0.22851E+01, 0.22928E+01, 0.23004E+01, 0.23081E+01,
     & 0.23157E+01, 0.23233E+01, 0.23309E+01, 0.23385E+01, 0.23461E+01,
     & 0.23536E+01, 0.23612E+01, 0.23687E+01, 0.23762E+01, 0.23837E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.23912E+01, 0.23987E+01, 0.24061E+01, 0.24135E+01, 0.24210E+01,
     & 0.24284E+01, 0.24358E+01, 0.24432E+01, 0.24505E+01, 0.24579E+01,
     & 0.24652E+01, 0.24725E+01, 0.24798E+01, 0.24871E+01, 0.24944E+01,
     & 0.25017E+01, 0.25089E+01, 0.25162E+01, 0.25234E+01, 0.25306E+01,
     & 0.25378E+01, 0.25450E+01, 0.25522E+01, 0.25593E+01, 0.25665E+01,
     & 0.25736E+01, 0.25807E+01, 0.25878E+01, 0.25949E+01, 0.26020E+01,
     & 0.26091E+01, 0.26161E+01, 0.26231E+01, 0.26302E+01, 0.26372E+01,
     & 0.26442E+01, 0.26512E+01, 0.26581E+01, 0.26651E+01, 0.26720E+01,
     & 0.26790E+01, 0.26859E+01, 0.26928E+01, 0.26997E+01, 0.27066E+01,
     & 0.27135E+01, 0.27203E+01, 0.27272E+01, 0.27340E+01, 0.27408E+01,
     & 0.27476E+01, 0.27544E+01, 0.27612E+01, 0.27680E+01, 0.27747E+01,
     & 0.27815E+01, 0.27882E+01, 0.27949E+01, 0.28016E+01, 0.28083E+01,
     & 0.28150E+01, 0.28217E+01, 0.28284E+01, 0.28350E+01, 0.28417E+01,
     & 0.28483E+01, 0.28549E+01, 0.28615E+01, 0.28681E+01, 0.28747E+01,
     & 0.28812E+01, 0.28878E+01, 0.28944E+01, 0.29009E+01, 0.29074E+01,
     & 0.29139E+01, 0.29204E+01, 0.29269E+01, 0.29334E+01, 0.29399E+01,
     & 0.29463E+01, 0.29528E+01, 0.29592E+01, 0.29656E+01, 0.29720E+01,
     & 0.29784E+01, 0.29848E+01, 0.29912E+01, 0.29976E+01, 0.30039E+01,
     & 0.30103E+01, 0.30166E+01, 0.30229E+01, 0.30292E+01, 0.30355E+01,
     & 0.30418E+01, 0.30481E+01, 0.30544E+01, 0.30607E+01, 0.30669E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.30731E+01, 0.30794E+01, 0.30856E+01, 0.30918E+01, 0.30980E+01,
     & 0.31042E+01, 0.31104E+01, 0.31165E+01, 0.31227E+01, 0.31288E+01,
     & 0.31350E+01, 0.31411E+01, 0.31472E+01, 0.31533E+01, 0.31594E+01,
     & 0.31655E+01, 0.31716E+01, 0.31777E+01, 0.31837E+01, 0.31898E+01,
     & 0.31958E+01, 0.32018E+01, 0.32078E+01, 0.32139E+01, 0.32198E+01,
     & 0.32258E+01, 0.32318E+01, 0.32378E+01, 0.32437E+01, 0.32497E+01,
     & 0.32556E+01, 0.32616E+01, 0.32675E+01, 0.32734E+01, 0.32793E+01,
     & 0.32852E+01, 0.32911E+01, 0.32970E+01, 0.33028E+01, 0.33087E+01,
     & 0.33145E+01, 0.33204E+01, 0.33262E+01, 0.33320E+01, 0.33378E+01,
     & 0.33436E+01, 0.33494E+01, 0.33552E+01, 0.33610E+01, 0.33667E+01,
     & 0.33725E+01, 0.33782E+01, 0.33840E+01, 0.33897E+01, 0.33954E+01,
     & 0.34011E+01, 0.34068E+01, 0.34125E+01, 0.34182E+01, 0.34239E+01,
     & 0.34296E+01, 0.34352E+01, 0.34409E+01, 0.34465E+01, 0.34521E+01,
     & 0.34578E+01, 0.34634E+01, 0.34690E+01, 0.34746E+01, 0.34802E+01,
     & 0.34858E+01, 0.34913E+01, 0.34969E+01, 0.35025E+01, 0.35080E+01,
     & 0.35135E+01, 0.35191E+01, 0.35246E+01, 0.35301E+01, 0.35356E+01,
     & 0.35411E+01, 0.35466E+01, 0.35521E+01, 0.35576E+01, 0.35630E+01,
     & 0.35685E+01, 0.35739E+01, 0.35794E+01, 0.35848E+01, 0.35902E+01,
     & 0.35957E+01, 0.36011E+01, 0.36065E+01, 0.36119E+01, 0.36173E+01,
     & 0.36226E+01, 0.36280E+01, 0.36334E+01, 0.36387E+01, 0.36441E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.36494E+01, 0.36547E+01, 0.36601E+01, 0.36654E+01, 0.36707E+01,
     & 0.36760E+01, 0.36813E+01, 0.36866E+01, 0.36918E+01, 0.36971E+01,
     & 0.37024E+01, 0.37076E+01, 0.37129E+01, 0.37181E+01, 0.37234E+01,
     & 0.37286E+01, 0.37338E+01, 0.37390E+01, 0.37442E+01, 0.37494E+01,
     & 0.37546E+01, 0.37598E+01, 0.37649E+01, 0.37701E+01, 0.37753E+01,
     & 0.37804E+01, 0.37856E+01, 0.37907E+01, 0.37958E+01, 0.38010E+01,
     & 0.38061E+01, 0.38112E+01, 0.38163E+01, 0.38214E+01, 0.38265E+01,
     & 0.38316E+01, 0.38366E+01, 0.38417E+01, 0.38468E+01, 0.38518E+01,
     & 0.38568E+01, 0.38619E+01, 0.38669E+01, 0.38719E+01, 0.38770E+01,
     & 0.38820E+01, 0.38870E+01, 0.38920E+01, 0.38970E+01, 0.39020E+01,
     & 0.39069E+01, 0.39119E+01, 0.39169E+01, 0.39218E+01, 0.39268E+01,
     & 0.39317E+01, 0.39366E+01, 0.39416E+01, 0.39465E+01, 0.39514E+01,
     & 0.39563E+01, 0.39612E+01, 0.39661E+01, 0.39710E+01, 0.39759E+01,
     & 0.39808E+01, 0.39857E+01, 0.39905E+01, 0.39954E+01, 0.40002E+01,
     & 0.40051E+01, 0.40099E+01, 0.40147E+01, 0.40196E+01, 0.40244E+01,
     & 0.40292E+01, 0.40340E+01, 0.40388E+01, 0.40436E+01, 0.40484E+01,
     & 0.40532E+01, 0.40579E+01, 0.40627E+01, 0.40675E+01, 0.40722E+01,
     & 0.40770E+01, 0.40817E+01, 0.40865E+01, 0.40912E+01, 0.40959E+01,
     & 0.41006E+01, 0.41054E+01, 0.41101E+01, 0.41148E+01, 0.41195E+01,
     & 0.41242E+01, 0.41288E+01, 0.41335E+01, 0.41382E+01, 0.41429E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.41929E+01, 0.42386E+01, 0.42837E+01, 0.43283E+01, 0.43723E+01,
     & 0.44159E+01, 0.44588E+01, 0.45013E+01, 0.45433E+01, 0.45849E+01,
     & 0.46259E+01, 0.46665E+01, 0.47066E+01, 0.47463E+01, 0.47856E+01,
     & 0.48244E+01, 0.48628E+01, 0.49008E+01, 0.49384E+01, 0.49757E+01,
     & 0.50125E+01, 0.50490E+01, 0.50851E+01, 0.51208E+01, 0.51562E+01,
     & 0.51912E+01, 0.52259E+01, 0.52603E+01, 0.52943E+01, 0.53280E+01,
     & 0.53614E+01, 0.53945E+01, 0.54273E+01, 0.54598E+01, 0.54919E+01,
     & 0.55238E+01, 0.55554E+01, 0.55868E+01, 0.56178E+01, 0.56486E+01,
     & 0.56791E+01, 0.57094E+01, 0.57394E+01, 0.57691E+01, 0.57986E+01,
     & 0.58279E+01, 0.58569E+01, 0.58856E+01, 0.59142E+01, 0.59425E+01,
     & 0.59705E+01, 0.59984E+01, 0.60260E+01, 0.60534E+01, 0.60807E+01,
     & 0.61076E+01, 0.61344E+01, 0.61610E+01, 0.61874E+01, 0.62136E+01,
     & 0.62396E+01, 0.62653E+01, 0.62909E+01, 0.63164E+01, 0.63416E+01,
     & 0.63666E+01, 0.63915E+01, 0.64162E+01, 0.64407E+01, 0.64650E+01,
     & 0.64892E+01, 0.65132E+01, 0.65370E+01, 0.65607E+01, 0.65842E+01,
     & 0.66076E+01, 0.66308E+01, 0.66538E+01, 0.66767E+01, 0.66994E+01,
     & 0.67220E+01, 0.67444E+01, 0.67667E+01, 0.67889E+01, 0.68108E+01,
     & 0.68327E+01, 0.68544E+01, 0.68760E+01, 0.68974E+01, 0.69188E+01,
     & 0.69399E+01, 0.69610E+01, 0.69819E+01, 0.70027E+01, 0.70233E+01,
     & 0.70439E+01, 0.70643E+01, 0.70846E+01, 0.71047E+01, 0.71248E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.71447E+01, 0.71645E+01, 0.71842E+01, 0.72038E+01, 0.72233E+01,
     & 0.72426E+01, 0.72619E+01, 0.72810E+01, 0.73000E+01, 0.73189E+01,
     & 0.73378E+01, 0.73565E+01, 0.73751E+01, 0.73936E+01, 0.74120E+01,
     & 0.74303E+01, 0.74484E+01, 0.74665E+01, 0.74845E+01, 0.75024E+01,
     & 0.75203E+01, 0.75380E+01, 0.75556E+01, 0.75731E+01, 0.75905E+01,
     & 0.76079E+01, 0.76251E+01, 0.76423E+01, 0.76593E+01, 0.76763E+01,
     & 0.76932E+01, 0.77100E+01, 0.77268E+01, 0.77434E+01, 0.77600E+01,
     & 0.77764E+01, 0.77928E+01, 0.78091E+01, 0.78253E+01, 0.78415E+01,
     & 0.78576E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.46818E-01,-0.99454E-01,-0.12468E+00,-0.14198E+00,-0.15520E+00,
     &-0.16586E+00,-0.17475E+00,-0.18233E+00,-0.18889E+00,-0.19464E+00,
     &-0.19972E+00,-0.20422E+00,-0.20825E+00,-0.21184E+00,-0.21507E+00,
     &-0.21797E+00,-0.22056E+00,-0.22288E+00,-0.22496E+00,-0.22681E+00,
     &-0.22844E+00,-0.22988E+00,-0.23114E+00,-0.23222E+00,-0.23314E+00,
     &-0.23391E+00,-0.23453E+00,-0.23502E+00,-0.23537E+00,-0.23559E+00,
     &-0.23570E+00,-0.23569E+00,-0.23557E+00,-0.23535E+00,-0.23502E+00,
     &-0.23460E+00,-0.23408E+00,-0.23348E+00,-0.23278E+00,-0.23200E+00,
     &-0.23115E+00,-0.23021E+00,-0.22920E+00,-0.22811E+00,-0.22696E+00,
     &-0.22574E+00,-0.22445E+00,-0.22310E+00,-0.22169E+00,-0.22023E+00,
     &-0.21870E+00,-0.21712E+00,-0.21549E+00,-0.21381E+00,-0.21207E+00,
     &-0.21029E+00,-0.20847E+00,-0.20660E+00,-0.20468E+00,-0.20272E+00,
     &-0.20073E+00,-0.19869E+00,-0.19661E+00,-0.19450E+00,-0.19235E+00,
     &-0.19016E+00,-0.18794E+00,-0.18569E+00,-0.18340E+00,-0.18107E+00,
     &-0.17872E+00,-0.17633E+00,-0.17392E+00,-0.17147E+00,-0.16899E+00,
     &-0.16648E+00,-0.16394E+00,-0.16137E+00,-0.15877E+00,-0.15614E+00,
     &-0.15349E+00,-0.15080E+00,-0.14809E+00,-0.14535E+00,-0.14259E+00,
     &-0.13979E+00,-0.13697E+00,-0.13413E+00,-0.13125E+00,-0.12836E+00,
     &-0.12544E+00,-0.12249E+00,-0.11952E+00,-0.11653E+00,-0.11351E+00,
     &-0.11047E+00,-0.10741E+00,-0.10433E+00,-0.10124E+00,-0.98117E-01/

      DATA (BNC09M(I),I=101,200)/
     &-0.94980E-01,-0.91826E-01,-0.88655E-01,-0.85469E-01,-0.82267E-01,
     &-0.79051E-01,-0.75822E-01,-0.72581E-01,-0.69327E-01,-0.66063E-01,
     &-0.62789E-01,-0.59505E-01,-0.56214E-01,-0.52914E-01,-0.49608E-01,
     &-0.46296E-01,-0.42978E-01,-0.39656E-01,-0.36329E-01,-0.33000E-01,
     &-0.29667E-01,-0.26333E-01,-0.22997E-01,-0.19660E-01,-0.16323E-01,
     &-0.12985E-01,-0.96483E-02,-0.63125E-02,-0.29780E-02, 0.35480E-03,
     & 0.36852E-02, 0.70134E-02, 0.10339E-01, 0.13661E-01, 0.16979E-01,
     & 0.20294E-01, 0.23605E-01, 0.26912E-01, 0.30214E-01, 0.33512E-01,
     & 0.36805E-01, 0.40093E-01, 0.43376E-01, 0.46653E-01, 0.49925E-01,
     & 0.53192E-01, 0.56453E-01, 0.59708E-01, 0.62957E-01, 0.66200E-01,
     & 0.69437E-01, 0.72668E-01, 0.75892E-01, 0.79111E-01, 0.82322E-01,
     & 0.85528E-01, 0.88727E-01, 0.91919E-01, 0.95104E-01, 0.98283E-01,
     & 0.10146E+00, 0.10462E+00, 0.10778E+00, 0.11093E+00, 0.11408E+00,
     & 0.11722E+00, 0.12035E+00, 0.12347E+00, 0.12659E+00, 0.12970E+00,
     & 0.13280E+00, 0.13590E+00, 0.13899E+00, 0.14207E+00, 0.14515E+00,
     & 0.14822E+00, 0.15128E+00, 0.15433E+00, 0.15738E+00, 0.16042E+00,
     & 0.16345E+00, 0.16648E+00, 0.16950E+00, 0.17251E+00, 0.17552E+00,
     & 0.17852E+00, 0.18151E+00, 0.18449E+00, 0.18747E+00, 0.19044E+00,
     & 0.19341E+00, 0.19636E+00, 0.19931E+00, 0.20226E+00, 0.20519E+00,
     & 0.20812E+00, 0.21104E+00, 0.21396E+00, 0.21687E+00, 0.21977E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.22266E+00, 0.22555E+00, 0.22843E+00, 0.23131E+00, 0.23418E+00,
     & 0.23704E+00, 0.23989E+00, 0.24274E+00, 0.24558E+00, 0.24842E+00,
     & 0.25124E+00, 0.25407E+00, 0.25688E+00, 0.25969E+00, 0.26249E+00,
     & 0.26529E+00, 0.26807E+00, 0.27086E+00, 0.27363E+00, 0.27640E+00,
     & 0.27917E+00, 0.28192E+00, 0.28467E+00, 0.28742E+00, 0.29015E+00,
     & 0.29288E+00, 0.29561E+00, 0.29833E+00, 0.30104E+00, 0.30375E+00,
     & 0.30645E+00, 0.30914E+00, 0.31183E+00, 0.31451E+00, 0.31719E+00,
     & 0.31986E+00, 0.32252E+00, 0.32518E+00, 0.32783E+00, 0.33048E+00,
     & 0.33312E+00, 0.33575E+00, 0.33838E+00, 0.34100E+00, 0.34362E+00,
     & 0.34623E+00, 0.34883E+00, 0.35143E+00, 0.35402E+00, 0.35661E+00,
     & 0.35919E+00, 0.36177E+00, 0.36434E+00, 0.36690E+00, 0.36946E+00,
     & 0.37201E+00, 0.37456E+00, 0.37710E+00, 0.37964E+00, 0.38217E+00,
     & 0.38470E+00, 0.38722E+00, 0.38973E+00, 0.39224E+00, 0.39474E+00,
     & 0.39724E+00, 0.39974E+00, 0.40222E+00, 0.40471E+00, 0.40718E+00,
     & 0.40966E+00, 0.41212E+00, 0.41458E+00, 0.41704E+00, 0.41949E+00,
     & 0.42194E+00, 0.42438E+00, 0.42681E+00, 0.42924E+00, 0.43167E+00,
     & 0.43409E+00, 0.43650E+00, 0.43891E+00, 0.44132E+00, 0.44372E+00,
     & 0.44611E+00, 0.44851E+00, 0.45089E+00, 0.45327E+00, 0.45565E+00,
     & 0.45802E+00, 0.46038E+00, 0.46274E+00, 0.46510E+00, 0.46745E+00,
     & 0.46980E+00, 0.47214E+00, 0.47448E+00, 0.47681E+00, 0.47914E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.48146E+00, 0.48378E+00, 0.48609E+00, 0.48840E+00, 0.49070E+00,
     & 0.49300E+00, 0.49530E+00, 0.49759E+00, 0.49988E+00, 0.50216E+00,
     & 0.50444E+00, 0.50671E+00, 0.50898E+00, 0.51124E+00, 0.51350E+00,
     & 0.51575E+00, 0.51800E+00, 0.52025E+00, 0.52249E+00, 0.52473E+00,
     & 0.52696E+00, 0.52919E+00, 0.53141E+00, 0.53363E+00, 0.53585E+00,
     & 0.53806E+00, 0.54027E+00, 0.54247E+00, 0.54467E+00, 0.54686E+00,
     & 0.54905E+00, 0.55124E+00, 0.55342E+00, 0.55560E+00, 0.55777E+00,
     & 0.55994E+00, 0.56211E+00, 0.56427E+00, 0.56643E+00, 0.56858E+00,
     & 0.57073E+00, 0.57288E+00, 0.57502E+00, 0.57715E+00, 0.57929E+00,
     & 0.58142E+00, 0.58354E+00, 0.58566E+00, 0.58778E+00, 0.58990E+00,
     & 0.59201E+00, 0.59411E+00, 0.59621E+00, 0.59831E+00, 0.60041E+00,
     & 0.60250E+00, 0.60458E+00, 0.60667E+00, 0.60874E+00, 0.61082E+00,
     & 0.61289E+00, 0.61496E+00, 0.61702E+00, 0.61908E+00, 0.62114E+00,
     & 0.62319E+00, 0.62524E+00, 0.62729E+00, 0.62933E+00, 0.63137E+00,
     & 0.63340E+00, 0.63544E+00, 0.63746E+00, 0.63949E+00, 0.64151E+00,
     & 0.64352E+00, 0.64554E+00, 0.64755E+00, 0.64955E+00, 0.65156E+00,
     & 0.65355E+00, 0.65555E+00, 0.65754E+00, 0.65953E+00, 0.66152E+00,
     & 0.66350E+00, 0.66548E+00, 0.66745E+00, 0.66942E+00, 0.67139E+00,
     & 0.67335E+00, 0.67532E+00, 0.67727E+00, 0.67923E+00, 0.68118E+00,
     & 0.68313E+00, 0.68507E+00, 0.68701E+00, 0.68895E+00, 0.69089E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.69282E+00, 0.69475E+00, 0.69667E+00, 0.69859E+00, 0.70051E+00,
     & 0.70243E+00, 0.70434E+00, 0.70625E+00, 0.70815E+00, 0.71005E+00,
     & 0.71195E+00, 0.71385E+00, 0.71574E+00, 0.71763E+00, 0.71952E+00,
     & 0.72140E+00, 0.72328E+00, 0.72516E+00, 0.72703E+00, 0.72890E+00,
     & 0.73077E+00, 0.73264E+00, 0.73450E+00, 0.73636E+00, 0.73821E+00,
     & 0.74006E+00, 0.74191E+00, 0.74376E+00, 0.74560E+00, 0.74745E+00,
     & 0.74928E+00, 0.75112E+00, 0.75295E+00, 0.75478E+00, 0.75660E+00,
     & 0.75843E+00, 0.76025E+00, 0.76206E+00, 0.76388E+00, 0.76569E+00,
     & 0.76750E+00, 0.76930E+00, 0.77110E+00, 0.77290E+00, 0.77470E+00,
     & 0.77649E+00, 0.77828E+00, 0.78007E+00, 0.78186E+00, 0.78364E+00,
     & 0.78542E+00, 0.78720E+00, 0.78897E+00, 0.79074E+00, 0.79251E+00,
     & 0.79428E+00, 0.79604E+00, 0.79780E+00, 0.79956E+00, 0.80131E+00,
     & 0.80306E+00, 0.80481E+00, 0.80656E+00, 0.80830E+00, 0.81005E+00,
     & 0.81178E+00, 0.81352E+00, 0.81525E+00, 0.81698E+00, 0.81871E+00,
     & 0.82044E+00, 0.82216E+00, 0.82388E+00, 0.82560E+00, 0.82731E+00,
     & 0.82902E+00, 0.83073E+00, 0.83244E+00, 0.83414E+00, 0.83585E+00,
     & 0.83754E+00, 0.83924E+00, 0.84093E+00, 0.84263E+00, 0.84431E+00,
     & 0.84600E+00, 0.84768E+00, 0.84937E+00, 0.85104E+00, 0.85272E+00,
     & 0.85439E+00, 0.85607E+00, 0.85773E+00, 0.85940E+00, 0.86106E+00,
     & 0.86273E+00, 0.86438E+00, 0.86604E+00, 0.86769E+00, 0.86935E+00/

      DATA (BNC09M(I),I=501,600)/
     & 0.87100E+00, 0.87264E+00, 0.87429E+00, 0.87593E+00, 0.87757E+00,
     & 0.87920E+00, 0.88084E+00, 0.88247E+00, 0.88410E+00, 0.88573E+00,
     & 0.88735E+00, 0.88898E+00, 0.89060E+00, 0.89221E+00, 0.89383E+00,
     & 0.89544E+00, 0.89705E+00, 0.89866E+00, 0.90027E+00, 0.90187E+00,
     & 0.90347E+00, 0.90507E+00, 0.90667E+00, 0.90827E+00, 0.90986E+00,
     & 0.91145E+00, 0.91304E+00, 0.91462E+00, 0.91621E+00, 0.91779E+00,
     & 0.91937E+00, 0.92094E+00, 0.92252E+00, 0.92409E+00, 0.92566E+00,
     & 0.92723E+00, 0.92879E+00, 0.93036E+00, 0.93192E+00, 0.93347E+00,
     & 0.93503E+00, 0.93659E+00, 0.93814E+00, 0.93969E+00, 0.94124E+00,
     & 0.94278E+00, 0.94433E+00, 0.94587E+00, 0.94741E+00, 0.94894E+00,
     & 0.95048E+00, 0.95201E+00, 0.95354E+00, 0.95507E+00, 0.95660E+00,
     & 0.95812E+00, 0.95965E+00, 0.96117E+00, 0.96269E+00, 0.96420E+00,
     & 0.96572E+00, 0.96723E+00, 0.96874E+00, 0.97025E+00, 0.97175E+00,
     & 0.97325E+00, 0.97476E+00, 0.97626E+00, 0.97775E+00, 0.97925E+00,
     & 0.98074E+00, 0.98223E+00, 0.98372E+00, 0.98521E+00, 0.98670E+00,
     & 0.98818E+00, 0.98966E+00, 0.99114E+00, 0.99262E+00, 0.99410E+00,
     & 0.99557E+00, 0.99704E+00, 0.99851E+00, 0.99998E+00, 0.10014E+01,
     & 0.10029E+01, 0.10044E+01, 0.10058E+01, 0.10073E+01, 0.10087E+01,
     & 0.10102E+01, 0.10116E+01, 0.10131E+01, 0.10145E+01, 0.10160E+01,
     & 0.10174E+01, 0.10189E+01, 0.10203E+01, 0.10218E+01, 0.10232E+01/

      DATA (BNC09M(I),I=601,700)/
     & 0.10386E+01, 0.10527E+01, 0.10666E+01, 0.10803E+01, 0.10938E+01,
     & 0.11072E+01, 0.11204E+01, 0.11334E+01, 0.11463E+01, 0.11591E+01,
     & 0.11717E+01, 0.11841E+01, 0.11964E+01, 0.12086E+01, 0.12206E+01,
     & 0.12325E+01, 0.12442E+01, 0.12558E+01, 0.12673E+01, 0.12787E+01,
     & 0.12900E+01, 0.13011E+01, 0.13121E+01, 0.13230E+01, 0.13338E+01,
     & 0.13444E+01, 0.13550E+01, 0.13655E+01, 0.13758E+01, 0.13860E+01,
     & 0.13962E+01, 0.14062E+01, 0.14162E+01, 0.14260E+01, 0.14358E+01,
     & 0.14454E+01, 0.14550E+01, 0.14645E+01, 0.14738E+01, 0.14831E+01,
     & 0.14924E+01, 0.15015E+01, 0.15105E+01, 0.15195E+01, 0.15284E+01,
     & 0.15372E+01, 0.15459E+01, 0.15545E+01, 0.15631E+01, 0.15716E+01,
     & 0.15800E+01, 0.15884E+01, 0.15966E+01, 0.16049E+01, 0.16130E+01,
     & 0.16211E+01, 0.16291E+01, 0.16370E+01, 0.16449E+01, 0.16527E+01,
     & 0.16604E+01, 0.16681E+01, 0.16757E+01, 0.16833E+01, 0.16908E+01,
     & 0.16982E+01, 0.17056E+01, 0.17129E+01, 0.17201E+01, 0.17273E+01,
     & 0.17345E+01, 0.17416E+01, 0.17486E+01, 0.17556E+01, 0.17626E+01,
     & 0.17694E+01, 0.17763E+01, 0.17830E+01, 0.17898E+01, 0.17964E+01,
     & 0.18031E+01, 0.18097E+01, 0.18162E+01, 0.18227E+01, 0.18291E+01,
     & 0.18355E+01, 0.18418E+01, 0.18481E+01, 0.18544E+01, 0.18606E+01,
     & 0.18668E+01, 0.18729E+01, 0.18790E+01, 0.18850E+01, 0.18910E+01,
     & 0.18970E+01, 0.19029E+01, 0.19088E+01, 0.19146E+01, 0.19204E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19262E+01, 0.19319E+01, 0.19376E+01, 0.19432E+01, 0.19488E+01,
     & 0.19544E+01, 0.19599E+01, 0.19654E+01, 0.19709E+01, 0.19763E+01,
     & 0.19817E+01, 0.19871E+01, 0.19924E+01, 0.19977E+01, 0.20029E+01,
     & 0.20082E+01, 0.20133E+01, 0.20185E+01, 0.20236E+01, 0.20287E+01,
     & 0.20338E+01, 0.20388E+01, 0.20438E+01, 0.20488E+01, 0.20537E+01,
     & 0.20586E+01, 0.20635E+01, 0.20683E+01, 0.20731E+01, 0.20779E+01,
     & 0.20827E+01, 0.20874E+01, 0.20921E+01, 0.20968E+01, 0.21014E+01,
     & 0.21061E+01, 0.21107E+01, 0.21152E+01, 0.21198E+01, 0.21243E+01,
     & 0.21288E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.46234E-01,-0.95802E-01,-0.11793E+00,-0.13215E+00,-0.14233E+00,
     &-0.14999E+00,-0.15592E+00,-0.16058E+00,-0.16428E+00,-0.16721E+00,
     &-0.16952E+00,-0.17131E+00,-0.17268E+00,-0.17367E+00,-0.17435E+00,
     &-0.17476E+00,-0.17493E+00,-0.17488E+00,-0.17465E+00,-0.17425E+00,
     &-0.17370E+00,-0.17301E+00,-0.17221E+00,-0.17129E+00,-0.17028E+00,
     &-0.16917E+00,-0.16799E+00,-0.16673E+00,-0.16540E+00,-0.16402E+00,
     &-0.16257E+00,-0.16108E+00,-0.15954E+00,-0.15796E+00,-0.15634E+00,
     &-0.15468E+00,-0.15300E+00,-0.15128E+00,-0.14954E+00,-0.14778E+00,
     &-0.14600E+00,-0.14419E+00,-0.14237E+00,-0.14054E+00,-0.13869E+00,
     &-0.13682E+00,-0.13495E+00,-0.13307E+00,-0.13117E+00,-0.12927E+00,
     &-0.12737E+00,-0.12545E+00,-0.12353E+00,-0.12160E+00,-0.11967E+00,
     &-0.11774E+00,-0.11579E+00,-0.11385E+00,-0.11190E+00,-0.10994E+00,
     &-0.10798E+00,-0.10601E+00,-0.10404E+00,-0.10206E+00,-0.10008E+00,
     &-0.98086E-01,-0.96089E-01,-0.94084E-01,-0.92071E-01,-0.90050E-01,
     &-0.88020E-01,-0.85981E-01,-0.83932E-01,-0.81872E-01,-0.79801E-01,
     &-0.77719E-01,-0.75624E-01,-0.73517E-01,-0.71397E-01,-0.69264E-01,
     &-0.67117E-01,-0.64955E-01,-0.62779E-01,-0.60589E-01,-0.58383E-01,
     &-0.56162E-01,-0.53926E-01,-0.51674E-01,-0.49406E-01,-0.47123E-01,
     &-0.44825E-01,-0.42510E-01,-0.40181E-01,-0.37837E-01,-0.35477E-01,
     &-0.33103E-01,-0.30715E-01,-0.28312E-01,-0.25896E-01,-0.23466E-01/

      DATA (BNC10M(I),I=101,200)/
     &-0.21024E-01,-0.18569E-01,-0.16103E-01,-0.13624E-01,-0.11135E-01,
     &-0.86356E-02,-0.61260E-02,-0.36070E-02,-0.10791E-02, 0.14571E-02,
     & 0.40011E-02, 0.65524E-02, 0.91106E-02, 0.11675E-01, 0.14245E-01,
     & 0.16820E-01, 0.19401E-01, 0.21985E-01, 0.24574E-01, 0.27166E-01,
     & 0.29761E-01, 0.32358E-01, 0.34958E-01, 0.37560E-01, 0.40164E-01,
     & 0.42768E-01, 0.45374E-01, 0.47981E-01, 0.50587E-01, 0.53194E-01,
     & 0.55801E-01, 0.58408E-01, 0.61014E-01, 0.63620E-01, 0.66224E-01,
     & 0.68828E-01, 0.71430E-01, 0.74031E-01, 0.76630E-01, 0.79228E-01,
     & 0.81824E-01, 0.84419E-01, 0.87011E-01, 0.89602E-01, 0.92190E-01,
     & 0.94776E-01, 0.97360E-01, 0.99941E-01, 0.10252E+00, 0.10510E+00,
     & 0.10767E+00, 0.11024E+00, 0.11281E+00, 0.11538E+00, 0.11794E+00,
     & 0.12050E+00, 0.12306E+00, 0.12562E+00, 0.12817E+00, 0.13072E+00,
     & 0.13327E+00, 0.13581E+00, 0.13835E+00, 0.14089E+00, 0.14342E+00,
     & 0.14595E+00, 0.14848E+00, 0.15101E+00, 0.15353E+00, 0.15605E+00,
     & 0.15857E+00, 0.16108E+00, 0.16359E+00, 0.16609E+00, 0.16860E+00,
     & 0.17110E+00, 0.17359E+00, 0.17609E+00, 0.17858E+00, 0.18106E+00,
     & 0.18355E+00, 0.18603E+00, 0.18850E+00, 0.19098E+00, 0.19345E+00,
     & 0.19591E+00, 0.19838E+00, 0.20084E+00, 0.20329E+00, 0.20574E+00,
     & 0.20819E+00, 0.21064E+00, 0.21308E+00, 0.21552E+00, 0.21795E+00,
     & 0.22039E+00, 0.22281E+00, 0.22524E+00, 0.22766E+00, 0.23008E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.23249E+00, 0.23490E+00, 0.23731E+00, 0.23971E+00, 0.24211E+00,
     & 0.24451E+00, 0.24690E+00, 0.24929E+00, 0.25167E+00, 0.25406E+00,
     & 0.25643E+00, 0.25881E+00, 0.26118E+00, 0.26355E+00, 0.26591E+00,
     & 0.26827E+00, 0.27063E+00, 0.27298E+00, 0.27533E+00, 0.27768E+00,
     & 0.28002E+00, 0.28236E+00, 0.28469E+00, 0.28702E+00, 0.28935E+00,
     & 0.29167E+00, 0.29399E+00, 0.29631E+00, 0.29862E+00, 0.30093E+00,
     & 0.30324E+00, 0.30554E+00, 0.30784E+00, 0.31013E+00, 0.31242E+00,
     & 0.31471E+00, 0.31700E+00, 0.31928E+00, 0.32155E+00, 0.32383E+00,
     & 0.32610E+00, 0.32836E+00, 0.33062E+00, 0.33288E+00, 0.33514E+00,
     & 0.33739E+00, 0.33964E+00, 0.34188E+00, 0.34412E+00, 0.34636E+00,
     & 0.34859E+00, 0.35082E+00, 0.35305E+00, 0.35527E+00, 0.35749E+00,
     & 0.35971E+00, 0.36192E+00, 0.36413E+00, 0.36633E+00, 0.36854E+00,
     & 0.37073E+00, 0.37293E+00, 0.37512E+00, 0.37731E+00, 0.37949E+00,
     & 0.38167E+00, 0.38385E+00, 0.38602E+00, 0.38819E+00, 0.39036E+00,
     & 0.39252E+00, 0.39468E+00, 0.39683E+00, 0.39899E+00, 0.40113E+00,
     & 0.40328E+00, 0.40542E+00, 0.40756E+00, 0.40970E+00, 0.41183E+00,
     & 0.41395E+00, 0.41608E+00, 0.41820E+00, 0.42032E+00, 0.42243E+00,
     & 0.42454E+00, 0.42665E+00, 0.42875E+00, 0.43085E+00, 0.43295E+00,
     & 0.43505E+00, 0.43714E+00, 0.43922E+00, 0.44131E+00, 0.44339E+00,
     & 0.44546E+00, 0.44754E+00, 0.44961E+00, 0.45167E+00, 0.45374E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.45580E+00, 0.45786E+00, 0.45991E+00, 0.46196E+00, 0.46401E+00,
     & 0.46605E+00, 0.46809E+00, 0.47013E+00, 0.47216E+00, 0.47419E+00,
     & 0.47622E+00, 0.47824E+00, 0.48026E+00, 0.48228E+00, 0.48429E+00,
     & 0.48631E+00, 0.48831E+00, 0.49032E+00, 0.49232E+00, 0.49432E+00,
     & 0.49631E+00, 0.49830E+00, 0.50029E+00, 0.50228E+00, 0.50426E+00,
     & 0.50624E+00, 0.50821E+00, 0.51019E+00, 0.51216E+00, 0.51412E+00,
     & 0.51609E+00, 0.51805E+00, 0.52000E+00, 0.52196E+00, 0.52391E+00,
     & 0.52585E+00, 0.52780E+00, 0.52974E+00, 0.53168E+00, 0.53361E+00,
     & 0.53554E+00, 0.53747E+00, 0.53940E+00, 0.54132E+00, 0.54324E+00,
     & 0.54516E+00, 0.54707E+00, 0.54898E+00, 0.55089E+00, 0.55280E+00,
     & 0.55470E+00, 0.55660E+00, 0.55849E+00, 0.56038E+00, 0.56227E+00,
     & 0.56416E+00, 0.56605E+00, 0.56793E+00, 0.56980E+00, 0.57168E+00,
     & 0.57355E+00, 0.57542E+00, 0.57729E+00, 0.57915E+00, 0.58101E+00,
     & 0.58287E+00, 0.58472E+00, 0.58657E+00, 0.58842E+00, 0.59027E+00,
     & 0.59211E+00, 0.59395E+00, 0.59579E+00, 0.59762E+00, 0.59945E+00,
     & 0.60128E+00, 0.60311E+00, 0.60493E+00, 0.60675E+00, 0.60857E+00,
     & 0.61038E+00, 0.61219E+00, 0.61400E+00, 0.61581E+00, 0.61761E+00,
     & 0.61941E+00, 0.62121E+00, 0.62300E+00, 0.62480E+00, 0.62659E+00,
     & 0.62837E+00, 0.63016E+00, 0.63194E+00, 0.63372E+00, 0.63549E+00,
     & 0.63727E+00, 0.63904E+00, 0.64080E+00, 0.64257E+00, 0.64433E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.64609E+00, 0.64785E+00, 0.64960E+00, 0.65135E+00, 0.65310E+00,
     & 0.65485E+00, 0.65659E+00, 0.65833E+00, 0.66007E+00, 0.66181E+00,
     & 0.66354E+00, 0.66527E+00, 0.66700E+00, 0.66872E+00, 0.67044E+00,
     & 0.67216E+00, 0.67388E+00, 0.67560E+00, 0.67731E+00, 0.67902E+00,
     & 0.68073E+00, 0.68243E+00, 0.68413E+00, 0.68583E+00, 0.68753E+00,
     & 0.68922E+00, 0.69092E+00, 0.69260E+00, 0.69429E+00, 0.69598E+00,
     & 0.69766E+00, 0.69934E+00, 0.70101E+00, 0.70269E+00, 0.70436E+00,
     & 0.70603E+00, 0.70770E+00, 0.70936E+00, 0.71102E+00, 0.71268E+00,
     & 0.71434E+00, 0.71599E+00, 0.71765E+00, 0.71930E+00, 0.72094E+00,
     & 0.72259E+00, 0.72423E+00, 0.72587E+00, 0.72751E+00, 0.72914E+00,
     & 0.73078E+00, 0.73241E+00, 0.73404E+00, 0.73566E+00, 0.73729E+00,
     & 0.73891E+00, 0.74053E+00, 0.74214E+00, 0.74376E+00, 0.74537E+00,
     & 0.74698E+00, 0.74858E+00, 0.75019E+00, 0.75179E+00, 0.75339E+00,
     & 0.75499E+00, 0.75658E+00, 0.75818E+00, 0.75977E+00, 0.76136E+00,
     & 0.76294E+00, 0.76453E+00, 0.76611E+00, 0.76769E+00, 0.76927E+00,
     & 0.77084E+00, 0.77241E+00, 0.77398E+00, 0.77555E+00, 0.77712E+00,
     & 0.77868E+00, 0.78024E+00, 0.78180E+00, 0.78336E+00, 0.78492E+00,
     & 0.78647E+00, 0.78802E+00, 0.78957E+00, 0.79111E+00, 0.79266E+00,
     & 0.79420E+00, 0.79574E+00, 0.79728E+00, 0.79881E+00, 0.80034E+00,
     & 0.80188E+00, 0.80340E+00, 0.80493E+00, 0.80646E+00, 0.80798E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.80950E+00, 0.81102E+00, 0.81253E+00, 0.81405E+00, 0.81556E+00,
     & 0.81707E+00, 0.81858E+00, 0.82008E+00, 0.82158E+00, 0.82309E+00,
     & 0.82459E+00, 0.82608E+00, 0.82758E+00, 0.82907E+00, 0.83056E+00,
     & 0.83205E+00, 0.83354E+00, 0.83502E+00, 0.83651E+00, 0.83799E+00,
     & 0.83947E+00, 0.84094E+00, 0.84242E+00, 0.84389E+00, 0.84536E+00,
     & 0.84683E+00, 0.84830E+00, 0.84976E+00, 0.85122E+00, 0.85268E+00,
     & 0.85414E+00, 0.85560E+00, 0.85705E+00, 0.85851E+00, 0.85996E+00,
     & 0.86141E+00, 0.86285E+00, 0.86430E+00, 0.86574E+00, 0.86718E+00,
     & 0.86862E+00, 0.87006E+00, 0.87149E+00, 0.87293E+00, 0.87436E+00,
     & 0.87579E+00, 0.87721E+00, 0.87864E+00, 0.88006E+00, 0.88149E+00,
     & 0.88291E+00, 0.88432E+00, 0.88574E+00, 0.88715E+00, 0.88857E+00,
     & 0.88998E+00, 0.89139E+00, 0.89279E+00, 0.89420E+00, 0.89560E+00,
     & 0.89700E+00, 0.89840E+00, 0.89980E+00, 0.90119E+00, 0.90259E+00,
     & 0.90398E+00, 0.90537E+00, 0.90676E+00, 0.90814E+00, 0.90953E+00,
     & 0.91091E+00, 0.91229E+00, 0.91367E+00, 0.91505E+00, 0.91642E+00,
     & 0.91780E+00, 0.91917E+00, 0.92054E+00, 0.92191E+00, 0.92327E+00,
     & 0.92464E+00, 0.92600E+00, 0.92736E+00, 0.92872E+00, 0.93008E+00,
     & 0.93144E+00, 0.93279E+00, 0.93414E+00, 0.93549E+00, 0.93684E+00,
     & 0.93819E+00, 0.93954E+00, 0.94088E+00, 0.94222E+00, 0.94356E+00,
     & 0.94490E+00, 0.94624E+00, 0.94757E+00, 0.94891E+00, 0.95024E+00/

      DATA (BNC10M(I),I=601,700)/
     & 0.96452E+00, 0.97756E+00, 0.99045E+00, 0.10032E+01, 0.10157E+01,
     & 0.10282E+01, 0.10404E+01, 0.10525E+01, 0.10645E+01, 0.10763E+01,
     & 0.10880E+01, 0.10996E+01, 0.11110E+01, 0.11223E+01, 0.11335E+01,
     & 0.11445E+01, 0.11555E+01, 0.11663E+01, 0.11769E+01, 0.11875E+01,
     & 0.11980E+01, 0.12083E+01, 0.12185E+01, 0.12287E+01, 0.12387E+01,
     & 0.12486E+01, 0.12584E+01, 0.12681E+01, 0.12777E+01, 0.12872E+01,
     & 0.12967E+01, 0.13060E+01, 0.13152E+01, 0.13244E+01, 0.13334E+01,
     & 0.13424E+01, 0.13513E+01, 0.13601E+01, 0.13688E+01, 0.13774E+01,
     & 0.13860E+01, 0.13944E+01, 0.14028E+01, 0.14111E+01, 0.14194E+01,
     & 0.14275E+01, 0.14356E+01, 0.14437E+01, 0.14516E+01, 0.14595E+01,
     & 0.14673E+01, 0.14750E+01, 0.14827E+01, 0.14903E+01, 0.14978E+01,
     & 0.15053E+01, 0.15127E+01, 0.15201E+01, 0.15274E+01, 0.15346E+01,
     & 0.15418E+01, 0.15489E+01, 0.15559E+01, 0.15629E+01, 0.15698E+01,
     & 0.15767E+01, 0.15835E+01, 0.15903E+01, 0.15970E+01, 0.16037E+01,
     & 0.16103E+01, 0.16168E+01, 0.16233E+01, 0.16298E+01, 0.16362E+01,
     & 0.16425E+01, 0.16488E+01, 0.16551E+01, 0.16613E+01, 0.16675E+01,
     & 0.16736E+01, 0.16797E+01, 0.16857E+01, 0.16917E+01, 0.16976E+01,
     & 0.17035E+01, 0.17093E+01, 0.17152E+01, 0.17209E+01, 0.17266E+01,
     & 0.17323E+01, 0.17380E+01, 0.17436E+01, 0.17491E+01, 0.17546E+01,
     & 0.17601E+01, 0.17656E+01, 0.17710E+01, 0.17763E+01, 0.17817E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17870E+01, 0.17922E+01, 0.17975E+01, 0.18026E+01, 0.18078E+01,
     & 0.18129E+01, 0.18180E+01, 0.18230E+01, 0.18281E+01, 0.18330E+01,
     & 0.18380E+01, 0.18429E+01, 0.18478E+01, 0.18526E+01, 0.18575E+01,
     & 0.18622E+01, 0.18670E+01, 0.18717E+01, 0.18764E+01, 0.18811E+01,
     & 0.18857E+01, 0.18903E+01, 0.18949E+01, 0.18994E+01, 0.19040E+01,
     & 0.19085E+01, 0.19129E+01, 0.19174E+01, 0.19218E+01, 0.19261E+01,
     & 0.19305E+01, 0.19348E+01, 0.19391E+01, 0.19434E+01, 0.19476E+01,
     & 0.19519E+01, 0.19560E+01, 0.19602E+01, 0.19644E+01, 0.19685E+01,
     & 0.19726E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.45064E-01,-0.90047E-01,-0.10831E+00,-0.11900E+00,-0.12582E+00,
     &-0.13024E+00,-0.13300E+00,-0.13453E+00,-0.13511E+00,-0.13494E+00,
     &-0.13415E+00,-0.13283E+00,-0.13107E+00,-0.12891E+00,-0.12642E+00,
     &-0.12362E+00,-0.12056E+00,-0.11725E+00,-0.11372E+00,-0.10999E+00,
     &-0.10608E+00,-0.10200E+00,-0.97758E-01,-0.93378E-01,-0.88865E-01,
     &-0.84228E-01,-0.79477E-01,-0.74619E-01,-0.69661E-01,-0.64611E-01,
     &-0.59474E-01,-0.54257E-01,-0.48964E-01,-0.43600E-01,-0.38171E-01,
     &-0.32681E-01,-0.27134E-01,-0.21533E-01,-0.15883E-01,-0.10186E-01,
     &-0.44467E-02, 0.13329E-02, 0.71497E-02, 0.13001E-01, 0.18885E-01,
     & 0.24799E-01, 0.30740E-01, 0.36708E-01, 0.42700E-01, 0.48715E-01,
     & 0.54751E-01, 0.60806E-01, 0.66881E-01, 0.72974E-01, 0.79084E-01,
     & 0.85210E-01, 0.91353E-01, 0.97511E-01, 0.10368E+00, 0.10987E+00,
     & 0.11608E+00, 0.12230E+00, 0.12853E+00, 0.13478E+00, 0.14105E+00,
     & 0.14733E+00, 0.15364E+00, 0.15996E+00, 0.16629E+00, 0.17265E+00,
     & 0.17903E+00, 0.18544E+00, 0.19186E+00, 0.19831E+00, 0.20478E+00,
     & 0.21128E+00, 0.21781E+00, 0.22436E+00, 0.23094E+00, 0.23755E+00,
     & 0.24419E+00, 0.25087E+00, 0.25757E+00, 0.26430E+00, 0.27107E+00,
     & 0.27787E+00, 0.28471E+00, 0.29157E+00, 0.29847E+00, 0.30541E+00,
     & 0.31238E+00, 0.31938E+00, 0.32641E+00, 0.33348E+00, 0.34058E+00,
     & 0.34771E+00, 0.35487E+00, 0.36206E+00, 0.36928E+00, 0.37653E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.38380E+00, 0.39110E+00, 0.39843E+00, 0.40578E+00, 0.41315E+00,
     & 0.42054E+00, 0.42795E+00, 0.43539E+00, 0.44283E+00, 0.45030E+00,
     & 0.45778E+00, 0.46527E+00, 0.47278E+00, 0.48029E+00, 0.48782E+00,
     & 0.49535E+00, 0.50289E+00, 0.51044E+00, 0.51799E+00, 0.52555E+00,
     & 0.53310E+00, 0.54066E+00, 0.54823E+00, 0.55579E+00, 0.56335E+00,
     & 0.57091E+00, 0.57846E+00, 0.58602E+00, 0.59357E+00, 0.60111E+00,
     & 0.60865E+00, 0.61619E+00, 0.62371E+00, 0.63124E+00, 0.63875E+00,
     & 0.64626E+00, 0.65376E+00, 0.66125E+00, 0.66873E+00, 0.67620E+00,
     & 0.68366E+00, 0.69112E+00, 0.69856E+00, 0.70599E+00, 0.71342E+00,
     & 0.72083E+00, 0.72823E+00, 0.73562E+00, 0.74300E+00, 0.75037E+00,
     & 0.75773E+00, 0.76507E+00, 0.77241E+00, 0.77973E+00, 0.78704E+00,
     & 0.79434E+00, 0.80162E+00, 0.80890E+00, 0.81616E+00, 0.82341E+00,
     & 0.83065E+00, 0.83787E+00, 0.84508E+00, 0.85228E+00, 0.85947E+00,
     & 0.86664E+00, 0.87380E+00, 0.88095E+00, 0.88808E+00, 0.89520E+00,
     & 0.90231E+00, 0.90941E+00, 0.91649E+00, 0.92356E+00, 0.93062E+00,
     & 0.93766E+00, 0.94469E+00, 0.95171E+00, 0.95871E+00, 0.96570E+00,
     & 0.97268E+00, 0.97964E+00, 0.98659E+00, 0.99353E+00, 0.10005E+01,
     & 0.10074E+01, 0.10143E+01, 0.10211E+01, 0.10280E+01, 0.10349E+01,
     & 0.10417E+01, 0.10485E+01, 0.10554E+01, 0.10622E+01, 0.10690E+01,
     & 0.10757E+01, 0.10825E+01, 0.10892E+01, 0.10960E+01, 0.11027E+01/

      DATA (BNC11M(I),I=201,300)/
     & 0.11094E+01, 0.11161E+01, 0.11228E+01, 0.11295E+01, 0.11361E+01,
     & 0.11428E+01, 0.11494E+01, 0.11560E+01, 0.11626E+01, 0.11692E+01,
     & 0.11758E+01, 0.11823E+01, 0.11889E+01, 0.11954E+01, 0.12019E+01,
     & 0.12085E+01, 0.12150E+01, 0.12214E+01, 0.12279E+01, 0.12344E+01,
     & 0.12408E+01, 0.12472E+01, 0.12537E+01, 0.12601E+01, 0.12665E+01,
     & 0.12728E+01, 0.12792E+01, 0.12856E+01, 0.12919E+01, 0.12982E+01,
     & 0.13045E+01, 0.13109E+01, 0.13171E+01, 0.13234E+01, 0.13297E+01,
     & 0.13359E+01, 0.13422E+01, 0.13484E+01, 0.13546E+01, 0.13608E+01,
     & 0.13670E+01, 0.13732E+01, 0.13794E+01, 0.13855E+01, 0.13917E+01,
     & 0.13978E+01, 0.14039E+01, 0.14100E+01, 0.14161E+01, 0.14222E+01,
     & 0.14282E+01, 0.14343E+01, 0.14403E+01, 0.14464E+01, 0.14524E+01,
     & 0.14584E+01, 0.14644E+01, 0.14704E+01, 0.14763E+01, 0.14823E+01,
     & 0.14882E+01, 0.14942E+01, 0.15001E+01, 0.15060E+01, 0.15119E+01,
     & 0.15178E+01, 0.15237E+01, 0.15295E+01, 0.15354E+01, 0.15412E+01,
     & 0.15471E+01, 0.15529E+01, 0.15587E+01, 0.15645E+01, 0.15703E+01,
     & 0.15761E+01, 0.15818E+01, 0.15876E+01, 0.15933E+01, 0.15990E+01,
     & 0.16047E+01, 0.16105E+01, 0.16162E+01, 0.16218E+01, 0.16275E+01,
     & 0.16332E+01, 0.16388E+01, 0.16445E+01, 0.16501E+01, 0.16557E+01,
     & 0.16613E+01, 0.16669E+01, 0.16725E+01, 0.16781E+01, 0.16836E+01,
     & 0.16892E+01, 0.16947E+01, 0.17003E+01, 0.17058E+01, 0.17113E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.17168E+01, 0.17223E+01, 0.17278E+01, 0.17332E+01, 0.17387E+01,
     & 0.17441E+01, 0.17496E+01, 0.17550E+01, 0.17604E+01, 0.17658E+01,
     & 0.17712E+01, 0.17766E+01, 0.17820E+01, 0.17874E+01, 0.17927E+01,
     & 0.17981E+01, 0.18034E+01, 0.18087E+01, 0.18141E+01, 0.18194E+01,
     & 0.18247E+01, 0.18300E+01, 0.18352E+01, 0.18405E+01, 0.18458E+01,
     & 0.18510E+01, 0.18562E+01, 0.18615E+01, 0.18667E+01, 0.18719E+01,
     & 0.18771E+01, 0.18823E+01, 0.18875E+01, 0.18927E+01, 0.18978E+01,
     & 0.19030E+01, 0.19081E+01, 0.19132E+01, 0.19184E+01, 0.19235E+01,
     & 0.19286E+01, 0.19337E+01, 0.19388E+01, 0.19439E+01, 0.19489E+01,
     & 0.19540E+01, 0.19590E+01, 0.19641E+01, 0.19691E+01, 0.19741E+01,
     & 0.19792E+01, 0.19842E+01, 0.19892E+01, 0.19942E+01, 0.19991E+01,
     & 0.20041E+01, 0.20091E+01, 0.20140E+01, 0.20190E+01, 0.20239E+01,
     & 0.20288E+01, 0.20338E+01, 0.20387E+01, 0.20436E+01, 0.20485E+01,
     & 0.20533E+01, 0.20582E+01, 0.20631E+01, 0.20679E+01, 0.20728E+01,
     & 0.20776E+01, 0.20825E+01, 0.20873E+01, 0.20921E+01, 0.20969E+01,
     & 0.21017E+01, 0.21065E+01, 0.21113E+01, 0.21161E+01, 0.21209E+01,
     & 0.21256E+01, 0.21304E+01, 0.21351E+01, 0.21398E+01, 0.21446E+01,
     & 0.21493E+01, 0.21540E+01, 0.21587E+01, 0.21634E+01, 0.21681E+01,
     & 0.21728E+01, 0.21774E+01, 0.21821E+01, 0.21868E+01, 0.21914E+01,
     & 0.21960E+01, 0.22007E+01, 0.22053E+01, 0.22099E+01, 0.22145E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.22191E+01, 0.22237E+01, 0.22283E+01, 0.22329E+01, 0.22375E+01,
     & 0.22420E+01, 0.22466E+01, 0.22511E+01, 0.22557E+01, 0.22602E+01,
     & 0.22647E+01, 0.22693E+01, 0.22738E+01, 0.22783E+01, 0.22828E+01,
     & 0.22873E+01, 0.22917E+01, 0.22962E+01, 0.23007E+01, 0.23051E+01,
     & 0.23096E+01, 0.23140E+01, 0.23185E+01, 0.23229E+01, 0.23273E+01,
     & 0.23318E+01, 0.23362E+01, 0.23406E+01, 0.23450E+01, 0.23494E+01,
     & 0.23537E+01, 0.23581E+01, 0.23625E+01, 0.23668E+01, 0.23712E+01,
     & 0.23756E+01, 0.23799E+01, 0.23842E+01, 0.23886E+01, 0.23929E+01,
     & 0.23972E+01, 0.24015E+01, 0.24058E+01, 0.24101E+01, 0.24144E+01,
     & 0.24187E+01, 0.24229E+01, 0.24272E+01, 0.24315E+01, 0.24357E+01,
     & 0.24400E+01, 0.24442E+01, 0.24484E+01, 0.24527E+01, 0.24569E+01,
     & 0.24611E+01, 0.24653E+01, 0.24695E+01, 0.24737E+01, 0.24779E+01,
     & 0.24821E+01, 0.24862E+01, 0.24904E+01, 0.24946E+01, 0.24987E+01,
     & 0.25029E+01, 0.25070E+01, 0.25112E+01, 0.25153E+01, 0.25194E+01,
     & 0.25235E+01, 0.25276E+01, 0.25317E+01, 0.25358E+01, 0.25399E+01,
     & 0.25440E+01, 0.25481E+01, 0.25522E+01, 0.25563E+01, 0.25603E+01,
     & 0.25644E+01, 0.25684E+01, 0.25725E+01, 0.25765E+01, 0.25805E+01,
     & 0.25846E+01, 0.25886E+01, 0.25926E+01, 0.25966E+01, 0.26006E+01,
     & 0.26046E+01, 0.26086E+01, 0.26126E+01, 0.26166E+01, 0.26205E+01,
     & 0.26245E+01, 0.26285E+01, 0.26324E+01, 0.26364E+01, 0.26403E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.26443E+01, 0.26482E+01, 0.26521E+01, 0.26561E+01, 0.26600E+01,
     & 0.26639E+01, 0.26678E+01, 0.26717E+01, 0.26756E+01, 0.26795E+01,
     & 0.26834E+01, 0.26872E+01, 0.26911E+01, 0.26950E+01, 0.26988E+01,
     & 0.27027E+01, 0.27065E+01, 0.27104E+01, 0.27142E+01, 0.27181E+01,
     & 0.27219E+01, 0.27257E+01, 0.27295E+01, 0.27333E+01, 0.27371E+01,
     & 0.27409E+01, 0.27447E+01, 0.27485E+01, 0.27523E+01, 0.27561E+01,
     & 0.27599E+01, 0.27636E+01, 0.27674E+01, 0.27712E+01, 0.27749E+01,
     & 0.27787E+01, 0.27824E+01, 0.27861E+01, 0.27899E+01, 0.27936E+01,
     & 0.27973E+01, 0.28010E+01, 0.28048E+01, 0.28085E+01, 0.28122E+01,
     & 0.28159E+01, 0.28195E+01, 0.28232E+01, 0.28269E+01, 0.28306E+01,
     & 0.28343E+01, 0.28379E+01, 0.28416E+01, 0.28453E+01, 0.28489E+01,
     & 0.28525E+01, 0.28562E+01, 0.28598E+01, 0.28635E+01, 0.28671E+01,
     & 0.28707E+01, 0.28743E+01, 0.28779E+01, 0.28815E+01, 0.28851E+01,
     & 0.28887E+01, 0.28923E+01, 0.28959E+01, 0.28995E+01, 0.29031E+01,
     & 0.29067E+01, 0.29102E+01, 0.29138E+01, 0.29174E+01, 0.29209E+01,
     & 0.29245E+01, 0.29280E+01, 0.29315E+01, 0.29351E+01, 0.29386E+01,
     & 0.29421E+01, 0.29457E+01, 0.29492E+01, 0.29527E+01, 0.29562E+01,
     & 0.29597E+01, 0.29632E+01, 0.29667E+01, 0.29702E+01, 0.29737E+01,
     & 0.29772E+01, 0.29806E+01, 0.29841E+01, 0.29876E+01, 0.29910E+01,
     & 0.29945E+01, 0.29980E+01, 0.30014E+01, 0.30048E+01, 0.30083E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.30452E+01, 0.30789E+01, 0.31122E+01, 0.31450E+01, 0.31775E+01,
     & 0.32096E+01, 0.32413E+01, 0.32726E+01, 0.33035E+01, 0.33341E+01,
     & 0.33644E+01, 0.33943E+01, 0.34238E+01, 0.34531E+01, 0.34820E+01,
     & 0.35106E+01, 0.35389E+01, 0.35668E+01, 0.35945E+01, 0.36219E+01,
     & 0.36490E+01, 0.36759E+01, 0.37024E+01, 0.37287E+01, 0.37548E+01,
     & 0.37805E+01, 0.38061E+01, 0.38313E+01, 0.38564E+01, 0.38811E+01,
     & 0.39057E+01, 0.39300E+01, 0.39541E+01, 0.39780E+01, 0.40016E+01,
     & 0.40250E+01, 0.40483E+01, 0.40713E+01, 0.40941E+01, 0.41167E+01,
     & 0.41391E+01, 0.41613E+01, 0.41833E+01, 0.42051E+01, 0.42268E+01,
     & 0.42482E+01, 0.42695E+01, 0.42906E+01, 0.43116E+01, 0.43323E+01,
     & 0.43529E+01, 0.43733E+01, 0.43936E+01, 0.44137E+01, 0.44336E+01,
     & 0.44534E+01, 0.44730E+01, 0.44925E+01, 0.45118E+01, 0.45310E+01,
     & 0.45500E+01, 0.45689E+01, 0.45877E+01, 0.46063E+01, 0.46247E+01,
     & 0.46431E+01, 0.46613E+01, 0.46793E+01, 0.46973E+01, 0.47151E+01,
     & 0.47327E+01, 0.47503E+01, 0.47677E+01, 0.47850E+01, 0.48022E+01,
     & 0.48193E+01, 0.48362E+01, 0.48530E+01, 0.48698E+01, 0.48864E+01,
     & 0.49029E+01, 0.49192E+01, 0.49355E+01, 0.49517E+01, 0.49677E+01,
     & 0.49837E+01, 0.49995E+01, 0.50153E+01, 0.50309E+01, 0.50465E+01,
     & 0.50619E+01, 0.50773E+01, 0.50925E+01, 0.51077E+01, 0.51227E+01,
     & 0.51377E+01, 0.51526E+01, 0.51673E+01, 0.51820E+01, 0.51966E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.52112E+01, 0.52256E+01, 0.52399E+01, 0.52542E+01, 0.52683E+01,
     & 0.52824E+01, 0.52964E+01, 0.53104E+01, 0.53242E+01, 0.53380E+01,
     & 0.53516E+01, 0.53652E+01, 0.53788E+01, 0.53922E+01, 0.54056E+01,
     & 0.54189E+01, 0.54321E+01, 0.54453E+01, 0.54583E+01, 0.54713E+01,
     & 0.54843E+01, 0.54971E+01, 0.55099E+01, 0.55226E+01, 0.55353E+01,
     & 0.55479E+01, 0.55604E+01, 0.55729E+01, 0.55852E+01, 0.55976E+01,
     & 0.56098E+01, 0.56220E+01, 0.56341E+01, 0.56462E+01, 0.56582E+01,
     & 0.56701E+01, 0.56820E+01, 0.56938E+01, 0.57056E+01, 0.57173E+01,
     & 0.57289E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.46075E-01,-0.95502E-01,-0.11783E+00,-0.13242E+00,-0.14301E+00,
     &-0.15113E+00,-0.15754E+00,-0.16269E+00,-0.16685E+00,-0.17024E+00,
     &-0.17298E+00,-0.17518E+00,-0.17692E+00,-0.17825E+00,-0.17923E+00,
     &-0.17990E+00,-0.18028E+00,-0.18041E+00,-0.18030E+00,-0.17998E+00,
     &-0.17946E+00,-0.17875E+00,-0.17788E+00,-0.17684E+00,-0.17565E+00,
     &-0.17433E+00,-0.17286E+00,-0.17128E+00,-0.16957E+00,-0.16775E+00,
     &-0.16581E+00,-0.16378E+00,-0.16165E+00,-0.15942E+00,-0.15710E+00,
     &-0.15470E+00,-0.15222E+00,-0.14965E+00,-0.14702E+00,-0.14431E+00,
     &-0.14153E+00,-0.13868E+00,-0.13577E+00,-0.13280E+00,-0.12977E+00,
     &-0.12669E+00,-0.12355E+00,-0.12036E+00,-0.11712E+00,-0.11383E+00,
     &-0.11049E+00,-0.10711E+00,-0.10369E+00,-0.10023E+00,-0.96726E-01,
     &-0.93185E-01,-0.89606E-01,-0.85990E-01,-0.82339E-01,-0.78653E-01,
     &-0.74934E-01,-0.71181E-01,-0.67397E-01,-0.63580E-01,-0.59732E-01,
     &-0.55853E-01,-0.51943E-01,-0.48003E-01,-0.44033E-01,-0.40032E-01,
     &-0.36001E-01,-0.31941E-01,-0.27850E-01,-0.23729E-01,-0.19578E-01,
     &-0.15397E-01,-0.11185E-01,-0.69434E-02,-0.26710E-02, 0.16320E-02,
     & 0.59656E-02, 0.10330E-01, 0.14725E-01, 0.19151E-01, 0.23607E-01,
     & 0.28093E-01, 0.32610E-01, 0.37158E-01, 0.41734E-01, 0.46341E-01,
     & 0.50976E-01, 0.55640E-01, 0.60333E-01, 0.65052E-01, 0.69799E-01,
     & 0.74572E-01, 0.79370E-01, 0.84193E-01, 0.89040E-01, 0.93910E-01/

      DATA (BNC12M(I),I=101,200)/
     & 0.98802E-01, 0.10371E+00, 0.10865E+00, 0.11360E+00, 0.11857E+00,
     & 0.12356E+00, 0.12856E+00, 0.13358E+00, 0.13861E+00, 0.14365E+00,
     & 0.14871E+00, 0.15378E+00, 0.15885E+00, 0.16394E+00, 0.16903E+00,
     & 0.17413E+00, 0.17923E+00, 0.18434E+00, 0.18945E+00, 0.19457E+00,
     & 0.19969E+00, 0.20481E+00, 0.20993E+00, 0.21505E+00, 0.22017E+00,
     & 0.22529E+00, 0.23040E+00, 0.23552E+00, 0.24063E+00, 0.24574E+00,
     & 0.25084E+00, 0.25594E+00, 0.26103E+00, 0.26612E+00, 0.27120E+00,
     & 0.27628E+00, 0.28135E+00, 0.28642E+00, 0.29148E+00, 0.29653E+00,
     & 0.30157E+00, 0.30661E+00, 0.31163E+00, 0.31665E+00, 0.32167E+00,
     & 0.32667E+00, 0.33167E+00, 0.33666E+00, 0.34163E+00, 0.34660E+00,
     & 0.35157E+00, 0.35652E+00, 0.36146E+00, 0.36640E+00, 0.37132E+00,
     & 0.37624E+00, 0.38115E+00, 0.38604E+00, 0.39093E+00, 0.39581E+00,
     & 0.40068E+00, 0.40554E+00, 0.41039E+00, 0.41523E+00, 0.42006E+00,
     & 0.42488E+00, 0.42970E+00, 0.43450E+00, 0.43929E+00, 0.44408E+00,
     & 0.44885E+00, 0.45361E+00, 0.45837E+00, 0.46311E+00, 0.46785E+00,
     & 0.47257E+00, 0.47729E+00, 0.48199E+00, 0.48669E+00, 0.49137E+00,
     & 0.49605E+00, 0.50072E+00, 0.50537E+00, 0.51002E+00, 0.51466E+00,
     & 0.51928E+00, 0.52390E+00, 0.52851E+00, 0.53311E+00, 0.53770E+00,
     & 0.54228E+00, 0.54684E+00, 0.55140E+00, 0.55595E+00, 0.56050E+00,
     & 0.56503E+00, 0.56955E+00, 0.57406E+00, 0.57856E+00, 0.58306E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.58754E+00, 0.59201E+00, 0.59648E+00, 0.60093E+00, 0.60538E+00,
     & 0.60982E+00, 0.61424E+00, 0.61866E+00, 0.62307E+00, 0.62747E+00,
     & 0.63186E+00, 0.63624E+00, 0.64061E+00, 0.64497E+00, 0.64933E+00,
     & 0.65367E+00, 0.65801E+00, 0.66233E+00, 0.66665E+00, 0.67096E+00,
     & 0.67525E+00, 0.67954E+00, 0.68383E+00, 0.68810E+00, 0.69236E+00,
     & 0.69662E+00, 0.70086E+00, 0.70510E+00, 0.70933E+00, 0.71354E+00,
     & 0.71776E+00, 0.72196E+00, 0.72615E+00, 0.73033E+00, 0.73451E+00,
     & 0.73868E+00, 0.74284E+00, 0.74699E+00, 0.75113E+00, 0.75526E+00,
     & 0.75939E+00, 0.76350E+00, 0.76761E+00, 0.77171E+00, 0.77580E+00,
     & 0.77989E+00, 0.78396E+00, 0.78803E+00, 0.79208E+00, 0.79614E+00,
     & 0.80018E+00, 0.80421E+00, 0.80824E+00, 0.81225E+00, 0.81626E+00,
     & 0.82027E+00, 0.82426E+00, 0.82825E+00, 0.83222E+00, 0.83619E+00,
     & 0.84015E+00, 0.84411E+00, 0.84805E+00, 0.85199E+00, 0.85592E+00,
     & 0.85985E+00, 0.86376E+00, 0.86767E+00, 0.87157E+00, 0.87546E+00,
     & 0.87935E+00, 0.88322E+00, 0.88709E+00, 0.89095E+00, 0.89481E+00,
     & 0.89866E+00, 0.90250E+00, 0.90633E+00, 0.91015E+00, 0.91397E+00,
     & 0.91778E+00, 0.92158E+00, 0.92538E+00, 0.92916E+00, 0.93295E+00,
     & 0.93672E+00, 0.94049E+00, 0.94424E+00, 0.94800E+00, 0.95174E+00,
     & 0.95548E+00, 0.95921E+00, 0.96293E+00, 0.96665E+00, 0.97036E+00,
     & 0.97406E+00, 0.97776E+00, 0.98145E+00, 0.98513E+00, 0.98880E+00/

      DATA (BNC12M(I),I=301,400)/
     & 0.99247E+00, 0.99613E+00, 0.99979E+00, 0.10034E+01, 0.10071E+01,
     & 0.10107E+01, 0.10143E+01, 0.10180E+01, 0.10216E+01, 0.10252E+01,
     & 0.10288E+01, 0.10324E+01, 0.10360E+01, 0.10395E+01, 0.10431E+01,
     & 0.10467E+01, 0.10502E+01, 0.10538E+01, 0.10573E+01, 0.10609E+01,
     & 0.10644E+01, 0.10680E+01, 0.10715E+01, 0.10750E+01, 0.10785E+01,
     & 0.10820E+01, 0.10855E+01, 0.10890E+01, 0.10925E+01, 0.10960E+01,
     & 0.10994E+01, 0.11029E+01, 0.11064E+01, 0.11098E+01, 0.11133E+01,
     & 0.11167E+01, 0.11201E+01, 0.11236E+01, 0.11270E+01, 0.11304E+01,
     & 0.11338E+01, 0.11372E+01, 0.11406E+01, 0.11440E+01, 0.11474E+01,
     & 0.11508E+01, 0.11542E+01, 0.11575E+01, 0.11609E+01, 0.11642E+01,
     & 0.11676E+01, 0.11709E+01, 0.11743E+01, 0.11776E+01, 0.11810E+01,
     & 0.11843E+01, 0.11876E+01, 0.11909E+01, 0.11942E+01, 0.11975E+01,
     & 0.12008E+01, 0.12041E+01, 0.12074E+01, 0.12107E+01, 0.12139E+01,
     & 0.12172E+01, 0.12205E+01, 0.12237E+01, 0.12270E+01, 0.12302E+01,
     & 0.12335E+01, 0.12367E+01, 0.12399E+01, 0.12431E+01, 0.12464E+01,
     & 0.12496E+01, 0.12528E+01, 0.12560E+01, 0.12592E+01, 0.12624E+01,
     & 0.12656E+01, 0.12687E+01, 0.12719E+01, 0.12751E+01, 0.12782E+01,
     & 0.12814E+01, 0.12846E+01, 0.12877E+01, 0.12909E+01, 0.12940E+01,
     & 0.12971E+01, 0.13003E+01, 0.13034E+01, 0.13065E+01, 0.13096E+01,
     & 0.13127E+01, 0.13158E+01, 0.13189E+01, 0.13220E+01, 0.13251E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.13282E+01, 0.13313E+01, 0.13343E+01, 0.13374E+01, 0.13405E+01,
     & 0.13435E+01, 0.13466E+01, 0.13496E+01, 0.13527E+01, 0.13557E+01,
     & 0.13588E+01, 0.13618E+01, 0.13648E+01, 0.13678E+01, 0.13708E+01,
     & 0.13739E+01, 0.13769E+01, 0.13799E+01, 0.13829E+01, 0.13858E+01,
     & 0.13888E+01, 0.13918E+01, 0.13948E+01, 0.13978E+01, 0.14007E+01,
     & 0.14037E+01, 0.14067E+01, 0.14096E+01, 0.14126E+01, 0.14155E+01,
     & 0.14184E+01, 0.14214E+01, 0.14243E+01, 0.14272E+01, 0.14302E+01,
     & 0.14331E+01, 0.14360E+01, 0.14389E+01, 0.14418E+01, 0.14447E+01,
     & 0.14476E+01, 0.14505E+01, 0.14534E+01, 0.14563E+01, 0.14591E+01,
     & 0.14620E+01, 0.14649E+01, 0.14678E+01, 0.14706E+01, 0.14735E+01,
     & 0.14763E+01, 0.14792E+01, 0.14820E+01, 0.14849E+01, 0.14877E+01,
     & 0.14905E+01, 0.14933E+01, 0.14962E+01, 0.14990E+01, 0.15018E+01,
     & 0.15046E+01, 0.15074E+01, 0.15102E+01, 0.15130E+01, 0.15158E+01,
     & 0.15186E+01, 0.15214E+01, 0.15242E+01, 0.15269E+01, 0.15297E+01,
     & 0.15325E+01, 0.15352E+01, 0.15380E+01, 0.15408E+01, 0.15435E+01,
     & 0.15463E+01, 0.15490E+01, 0.15517E+01, 0.15545E+01, 0.15572E+01,
     & 0.15599E+01, 0.15627E+01, 0.15654E+01, 0.15681E+01, 0.15708E+01,
     & 0.15735E+01, 0.15762E+01, 0.15789E+01, 0.15816E+01, 0.15843E+01,
     & 0.15870E+01, 0.15897E+01, 0.15924E+01, 0.15951E+01, 0.15977E+01,
     & 0.16004E+01, 0.16031E+01, 0.16057E+01, 0.16084E+01, 0.16110E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.16137E+01, 0.16163E+01, 0.16190E+01, 0.16216E+01, 0.16243E+01,
     & 0.16269E+01, 0.16295E+01, 0.16321E+01, 0.16348E+01, 0.16374E+01,
     & 0.16400E+01, 0.16426E+01, 0.16452E+01, 0.16478E+01, 0.16504E+01,
     & 0.16530E+01, 0.16556E+01, 0.16582E+01, 0.16608E+01, 0.16634E+01,
     & 0.16659E+01, 0.16685E+01, 0.16711E+01, 0.16736E+01, 0.16762E+01,
     & 0.16788E+01, 0.16813E+01, 0.16839E+01, 0.16864E+01, 0.16890E+01,
     & 0.16915E+01, 0.16940E+01, 0.16966E+01, 0.16991E+01, 0.17016E+01,
     & 0.17042E+01, 0.17067E+01, 0.17092E+01, 0.17117E+01, 0.17142E+01,
     & 0.17167E+01, 0.17192E+01, 0.17218E+01, 0.17242E+01, 0.17267E+01,
     & 0.17292E+01, 0.17317E+01, 0.17342E+01, 0.17367E+01, 0.17392E+01,
     & 0.17416E+01, 0.17441E+01, 0.17466E+01, 0.17490E+01, 0.17515E+01,
     & 0.17540E+01, 0.17564E+01, 0.17589E+01, 0.17613E+01, 0.17638E+01,
     & 0.17662E+01, 0.17686E+01, 0.17711E+01, 0.17735E+01, 0.17759E+01,
     & 0.17784E+01, 0.17808E+01, 0.17832E+01, 0.17856E+01, 0.17880E+01,
     & 0.17904E+01, 0.17929E+01, 0.17953E+01, 0.17977E+01, 0.18001E+01,
     & 0.18024E+01, 0.18048E+01, 0.18072E+01, 0.18096E+01, 0.18120E+01,
     & 0.18144E+01, 0.18167E+01, 0.18191E+01, 0.18215E+01, 0.18239E+01,
     & 0.18262E+01, 0.18286E+01, 0.18309E+01, 0.18333E+01, 0.18356E+01,
     & 0.18380E+01, 0.18403E+01, 0.18427E+01, 0.18450E+01, 0.18474E+01,
     & 0.18497E+01, 0.18520E+01, 0.18543E+01, 0.18567E+01, 0.18590E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.18839E+01, 0.19066E+01, 0.19291E+01, 0.19513E+01, 0.19732E+01,
     & 0.19949E+01, 0.20163E+01, 0.20374E+01, 0.20583E+01, 0.20790E+01,
     & 0.20994E+01, 0.21196E+01, 0.21396E+01, 0.21594E+01, 0.21789E+01,
     & 0.21982E+01, 0.22174E+01, 0.22363E+01, 0.22550E+01, 0.22735E+01,
     & 0.22918E+01, 0.23100E+01, 0.23279E+01, 0.23457E+01, 0.23633E+01,
     & 0.23807E+01, 0.23979E+01, 0.24150E+01, 0.24319E+01, 0.24487E+01,
     & 0.24653E+01, 0.24817E+01, 0.24980E+01, 0.25141E+01, 0.25301E+01,
     & 0.25459E+01, 0.25616E+01, 0.25771E+01, 0.25925E+01, 0.26078E+01,
     & 0.26229E+01, 0.26379E+01, 0.26527E+01, 0.26675E+01, 0.26821E+01,
     & 0.26965E+01, 0.27109E+01, 0.27251E+01, 0.27393E+01, 0.27532E+01,
     & 0.27671E+01, 0.27809E+01, 0.27945E+01, 0.28081E+01, 0.28215E+01,
     & 0.28349E+01, 0.28481E+01, 0.28612E+01, 0.28742E+01, 0.28871E+01,
     & 0.28999E+01, 0.29126E+01, 0.29252E+01, 0.29378E+01, 0.29502E+01,
     & 0.29625E+01, 0.29747E+01, 0.29869E+01, 0.29989E+01, 0.30109E+01,
     & 0.30228E+01, 0.30346E+01, 0.30463E+01, 0.30579E+01, 0.30694E+01,
     & 0.30809E+01, 0.30923E+01, 0.31036E+01, 0.31148E+01, 0.31259E+01,
     & 0.31370E+01, 0.31480E+01, 0.31589E+01, 0.31697E+01, 0.31805E+01,
     & 0.31912E+01, 0.32018E+01, 0.32123E+01, 0.32228E+01, 0.32332E+01,
     & 0.32436E+01, 0.32538E+01, 0.32640E+01, 0.32742E+01, 0.32842E+01,
     & 0.32943E+01, 0.33042E+01, 0.33141E+01, 0.33239E+01, 0.33336E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.33433E+01, 0.33530E+01, 0.33626E+01, 0.33721E+01, 0.33815E+01,
     & 0.33909E+01, 0.34003E+01, 0.34095E+01, 0.34188E+01, 0.34279E+01,
     & 0.34371E+01, 0.34461E+01, 0.34551E+01, 0.34641E+01, 0.34730E+01,
     & 0.34819E+01, 0.34906E+01, 0.34994E+01, 0.35081E+01, 0.35167E+01,
     & 0.35253E+01, 0.35339E+01, 0.35424E+01, 0.35508E+01, 0.35592E+01,
     & 0.35676E+01, 0.35759E+01, 0.35842E+01, 0.35924E+01, 0.36006E+01,
     & 0.36087E+01, 0.36168E+01, 0.36248E+01, 0.36328E+01, 0.36408E+01,
     & 0.36487E+01, 0.36565E+01, 0.36644E+01, 0.36721E+01, 0.36799E+01,
     & 0.36876E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.76088E-01,-0.16481E+00,-0.20885E+00,-0.23985E+00,-0.26408E+00,
     &-0.28407E+00,-0.30112E+00,-0.31600E+00,-0.32922E+00,-0.34109E+00,
     &-0.35188E+00,-0.36175E+00,-0.37084E+00,-0.37927E+00,-0.38712E+00,
     &-0.39445E+00,-0.40133E+00,-0.40781E+00,-0.41391E+00,-0.41968E+00,
     &-0.42515E+00,-0.43034E+00,-0.43528E+00,-0.43998E+00,-0.44445E+00,
     &-0.44873E+00,-0.45281E+00,-0.45671E+00,-0.46044E+00,-0.46402E+00,
     &-0.46744E+00,-0.47073E+00,-0.47388E+00,-0.47690E+00,-0.47980E+00,
     &-0.48259E+00,-0.48527E+00,-0.48784E+00,-0.49032E+00,-0.49270E+00,
     &-0.49498E+00,-0.49718E+00,-0.49930E+00,-0.50133E+00,-0.50329E+00,
     &-0.50517E+00,-0.50698E+00,-0.50872E+00,-0.51040E+00,-0.51201E+00,
     &-0.51356E+00,-0.51505E+00,-0.51648E+00,-0.51786E+00,-0.51919E+00,
     &-0.52046E+00,-0.52169E+00,-0.52287E+00,-0.52400E+00,-0.52508E+00,
     &-0.52613E+00,-0.52713E+00,-0.52809E+00,-0.52901E+00,-0.52989E+00,
     &-0.53074E+00,-0.53155E+00,-0.53233E+00,-0.53307E+00,-0.53378E+00,
     &-0.53446E+00,-0.53510E+00,-0.53572E+00,-0.53631E+00,-0.53686E+00,
     &-0.53739E+00,-0.53790E+00,-0.53837E+00,-0.53882E+00,-0.53924E+00,
     &-0.53964E+00,-0.54002E+00,-0.54036E+00,-0.54069E+00,-0.54099E+00,
     &-0.54128E+00,-0.54153E+00,-0.54177E+00,-0.54199E+00,-0.54218E+00,
     &-0.54236E+00,-0.54251E+00,-0.54265E+00,-0.54277E+00,-0.54287E+00,
     &-0.54295E+00,-0.54301E+00,-0.54306E+00,-0.54309E+00,-0.54310E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.54310E+00,-0.54309E+00,-0.54306E+00,-0.54301E+00,-0.54295E+00,
     &-0.54288E+00,-0.54279E+00,-0.54270E+00,-0.54259E+00,-0.54247E+00,
     &-0.54233E+00,-0.54219E+00,-0.54204E+00,-0.54188E+00,-0.54170E+00,
     &-0.54152E+00,-0.54133E+00,-0.54113E+00,-0.54092E+00,-0.54071E+00,
     &-0.54049E+00,-0.54026E+00,-0.54002E+00,-0.53978E+00,-0.53953E+00,
     &-0.53927E+00,-0.53901E+00,-0.53875E+00,-0.53848E+00,-0.53820E+00,
     &-0.53792E+00,-0.53763E+00,-0.53734E+00,-0.53705E+00,-0.53675E+00,
     &-0.53645E+00,-0.53614E+00,-0.53583E+00,-0.53552E+00,-0.53520E+00,
     &-0.53488E+00,-0.53456E+00,-0.53423E+00,-0.53391E+00,-0.53357E+00,
     &-0.53324E+00,-0.53291E+00,-0.53257E+00,-0.53223E+00,-0.53188E+00,
     &-0.53154E+00,-0.53119E+00,-0.53085E+00,-0.53050E+00,-0.53014E+00,
     &-0.52979E+00,-0.52944E+00,-0.52908E+00,-0.52872E+00,-0.52836E+00,
     &-0.52800E+00,-0.52764E+00,-0.52728E+00,-0.52691E+00,-0.52655E+00,
     &-0.52618E+00,-0.52582E+00,-0.52545E+00,-0.52508E+00,-0.52471E+00,
     &-0.52434E+00,-0.52397E+00,-0.52360E+00,-0.52323E+00,-0.52285E+00,
     &-0.52248E+00,-0.52211E+00,-0.52173E+00,-0.52136E+00,-0.52098E+00,
     &-0.52061E+00,-0.52023E+00,-0.51985E+00,-0.51948E+00,-0.51910E+00,
     &-0.51872E+00,-0.51834E+00,-0.51797E+00,-0.51759E+00,-0.51721E+00,
     &-0.51683E+00,-0.51646E+00,-0.51608E+00,-0.51570E+00,-0.51532E+00,
     &-0.51494E+00,-0.51456E+00,-0.51419E+00,-0.51381E+00,-0.51343E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.51305E+00,-0.51267E+00,-0.51230E+00,-0.51192E+00,-0.51154E+00,
     &-0.51116E+00,-0.51079E+00,-0.51041E+00,-0.51003E+00,-0.50966E+00,
     &-0.50928E+00,-0.50891E+00,-0.50853E+00,-0.50815E+00,-0.50778E+00,
     &-0.50740E+00,-0.50703E+00,-0.50666E+00,-0.50628E+00,-0.50591E+00,
     &-0.50554E+00,-0.50516E+00,-0.50479E+00,-0.50442E+00,-0.50405E+00,
     &-0.50368E+00,-0.50330E+00,-0.50293E+00,-0.50256E+00,-0.50219E+00,
     &-0.50183E+00,-0.50146E+00,-0.50109E+00,-0.50072E+00,-0.50035E+00,
     &-0.49999E+00,-0.49962E+00,-0.49925E+00,-0.49889E+00,-0.49852E+00,
     &-0.49816E+00,-0.49780E+00,-0.49743E+00,-0.49707E+00,-0.49671E+00,
     &-0.49634E+00,-0.49598E+00,-0.49562E+00,-0.49526E+00,-0.49490E+00,
     &-0.49454E+00,-0.49418E+00,-0.49383E+00,-0.49347E+00,-0.49311E+00,
     &-0.49275E+00,-0.49240E+00,-0.49204E+00,-0.49169E+00,-0.49133E+00,
     &-0.49098E+00,-0.49063E+00,-0.49028E+00,-0.48992E+00,-0.48957E+00,
     &-0.48922E+00,-0.48887E+00,-0.48852E+00,-0.48817E+00,-0.48782E+00,
     &-0.48748E+00,-0.48713E+00,-0.48678E+00,-0.48644E+00,-0.48609E+00,
     &-0.48575E+00,-0.48540E+00,-0.48506E+00,-0.48471E+00,-0.48437E+00,
     &-0.48403E+00,-0.48369E+00,-0.48335E+00,-0.48301E+00,-0.48267E+00,
     &-0.48233E+00,-0.48199E+00,-0.48165E+00,-0.48132E+00,-0.48098E+00,
     &-0.48065E+00,-0.48031E+00,-0.47998E+00,-0.47964E+00,-0.47931E+00,
     &-0.47898E+00,-0.47864E+00,-0.47831E+00,-0.47798E+00,-0.47765E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.47732E+00,-0.47699E+00,-0.47666E+00,-0.47634E+00,-0.47601E+00,
     &-0.47568E+00,-0.47536E+00,-0.47503E+00,-0.47471E+00,-0.47438E+00,
     &-0.47406E+00,-0.47374E+00,-0.47342E+00,-0.47309E+00,-0.47277E+00,
     &-0.47245E+00,-0.47213E+00,-0.47181E+00,-0.47150E+00,-0.47118E+00,
     &-0.47086E+00,-0.47055E+00,-0.47023E+00,-0.46991E+00,-0.46960E+00,
     &-0.46929E+00,-0.46897E+00,-0.46866E+00,-0.46835E+00,-0.46804E+00,
     &-0.46772E+00,-0.46741E+00,-0.46710E+00,-0.46680E+00,-0.46649E+00,
     &-0.46618E+00,-0.46587E+00,-0.46557E+00,-0.46526E+00,-0.46495E+00,
     &-0.46465E+00,-0.46435E+00,-0.46404E+00,-0.46374E+00,-0.46344E+00,
     &-0.46314E+00,-0.46283E+00,-0.46253E+00,-0.46223E+00,-0.46194E+00,
     &-0.46164E+00,-0.46134E+00,-0.46104E+00,-0.46074E+00,-0.46045E+00,
     &-0.46015E+00,-0.45986E+00,-0.45956E+00,-0.45927E+00,-0.45898E+00,
     &-0.45868E+00,-0.45839E+00,-0.45810E+00,-0.45781E+00,-0.45752E+00,
     &-0.45723E+00,-0.45694E+00,-0.45665E+00,-0.45637E+00,-0.45608E+00,
     &-0.45579E+00,-0.45551E+00,-0.45522E+00,-0.45494E+00,-0.45465E+00,
     &-0.45437E+00,-0.45409E+00,-0.45380E+00,-0.45352E+00,-0.45324E+00,
     &-0.45296E+00,-0.45268E+00,-0.45240E+00,-0.45212E+00,-0.45184E+00,
     &-0.45156E+00,-0.45129E+00,-0.45101E+00,-0.45074E+00,-0.45046E+00,
     &-0.45019E+00,-0.44991E+00,-0.44964E+00,-0.44936E+00,-0.44909E+00,
     &-0.44882E+00,-0.44855E+00,-0.44828E+00,-0.44801E+00,-0.44774E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.44747E+00,-0.44720E+00,-0.44693E+00,-0.44667E+00,-0.44640E+00,
     &-0.44613E+00,-0.44587E+00,-0.44560E+00,-0.44534E+00,-0.44507E+00,
     &-0.44481E+00,-0.44455E+00,-0.44428E+00,-0.44402E+00,-0.44376E+00,
     &-0.44350E+00,-0.44324E+00,-0.44298E+00,-0.44272E+00,-0.44246E+00,
     &-0.44221E+00,-0.44195E+00,-0.44169E+00,-0.44144E+00,-0.44118E+00,
     &-0.44093E+00,-0.44067E+00,-0.44042E+00,-0.44016E+00,-0.43991E+00,
     &-0.43966E+00,-0.43941E+00,-0.43916E+00,-0.43891E+00,-0.43866E+00,
     &-0.43841E+00,-0.43816E+00,-0.43791E+00,-0.43766E+00,-0.43741E+00,
     &-0.43717E+00,-0.43692E+00,-0.43667E+00,-0.43643E+00,-0.43618E+00,
     &-0.43594E+00,-0.43570E+00,-0.43545E+00,-0.43521E+00,-0.43497E+00,
     &-0.43473E+00,-0.43448E+00,-0.43424E+00,-0.43400E+00,-0.43376E+00,
     &-0.43353E+00,-0.43329E+00,-0.43305E+00,-0.43281E+00,-0.43257E+00,
     &-0.43234E+00,-0.43210E+00,-0.43187E+00,-0.43163E+00,-0.43140E+00,
     &-0.43116E+00,-0.43093E+00,-0.43070E+00,-0.43046E+00,-0.43023E+00,
     &-0.43000E+00,-0.42977E+00,-0.42954E+00,-0.42931E+00,-0.42908E+00,
     &-0.42885E+00,-0.42862E+00,-0.42840E+00,-0.42817E+00,-0.42794E+00,
     &-0.42772E+00,-0.42749E+00,-0.42727E+00,-0.42704E+00,-0.42682E+00,
     &-0.42659E+00,-0.42637E+00,-0.42615E+00,-0.42592E+00,-0.42570E+00,
     &-0.42548E+00,-0.42526E+00,-0.42504E+00,-0.42482E+00,-0.42460E+00,
     &-0.42438E+00,-0.42416E+00,-0.42394E+00,-0.42373E+00,-0.42351E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.42329E+00,-0.42308E+00,-0.42286E+00,-0.42265E+00,-0.42243E+00,
     &-0.42222E+00,-0.42200E+00,-0.42179E+00,-0.42158E+00,-0.42137E+00,
     &-0.42116E+00,-0.42094E+00,-0.42073E+00,-0.42052E+00,-0.42031E+00,
     &-0.42010E+00,-0.41989E+00,-0.41969E+00,-0.41948E+00,-0.41927E+00,
     &-0.41906E+00,-0.41886E+00,-0.41865E+00,-0.41844E+00,-0.41824E+00,
     &-0.41804E+00,-0.41783E+00,-0.41763E+00,-0.41742E+00,-0.41722E+00,
     &-0.41702E+00,-0.41682E+00,-0.41661E+00,-0.41641E+00,-0.41621E+00,
     &-0.41601E+00,-0.41581E+00,-0.41561E+00,-0.41541E+00,-0.41522E+00,
     &-0.41502E+00,-0.41482E+00,-0.41462E+00,-0.41443E+00,-0.41423E+00,
     &-0.41404E+00,-0.41384E+00,-0.41364E+00,-0.41345E+00,-0.41326E+00,
     &-0.41306E+00,-0.41287E+00,-0.41268E+00,-0.41249E+00,-0.41229E+00,
     &-0.41210E+00,-0.41191E+00,-0.41172E+00,-0.41153E+00,-0.41134E+00,
     &-0.41115E+00,-0.41096E+00,-0.41077E+00,-0.41059E+00,-0.41040E+00,
     &-0.41021E+00,-0.41003E+00,-0.40984E+00,-0.40965E+00,-0.40947E+00,
     &-0.40928E+00,-0.40910E+00,-0.40891E+00,-0.40873E+00,-0.40855E+00,
     &-0.40836E+00,-0.40818E+00,-0.40800E+00,-0.40782E+00,-0.40764E+00,
     &-0.40746E+00,-0.40728E+00,-0.40710E+00,-0.40692E+00,-0.40674E+00,
     &-0.40656E+00,-0.40638E+00,-0.40620E+00,-0.40603E+00,-0.40585E+00,
     &-0.40567E+00,-0.40550E+00,-0.40532E+00,-0.40514E+00,-0.40497E+00,
     &-0.40480E+00,-0.40462E+00,-0.40445E+00,-0.40427E+00,-0.40410E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.40226E+00,-0.40059E+00,-0.39896E+00,-0.39738E+00,-0.39582E+00,
     &-0.39431E+00,-0.39283E+00,-0.39138E+00,-0.38998E+00,-0.38860E+00,
     &-0.38726E+00,-0.38595E+00,-0.38467E+00,-0.38343E+00,-0.38221E+00,
     &-0.38103E+00,-0.37988E+00,-0.37876E+00,-0.37766E+00,-0.37660E+00,
     &-0.37556E+00,-0.37455E+00,-0.37357E+00,-0.37261E+00,-0.37169E+00,
     &-0.37078E+00,-0.36991E+00,-0.36906E+00,-0.36823E+00,-0.36743E+00,
     &-0.36665E+00,-0.36590E+00,-0.36516E+00,-0.36446E+00,-0.36377E+00,
     &-0.36311E+00,-0.36247E+00,-0.36185E+00,-0.36125E+00,-0.36067E+00,
     &-0.36012E+00,-0.35958E+00,-0.35907E+00,-0.35857E+00,-0.35809E+00,
     &-0.35764E+00,-0.35720E+00,-0.35678E+00,-0.35638E+00,-0.35600E+00,
     &-0.35563E+00,-0.35529E+00,-0.35496E+00,-0.35465E+00,-0.35435E+00,
     &-0.35407E+00,-0.35381E+00,-0.35357E+00,-0.35334E+00,-0.35313E+00,
     &-0.35293E+00,-0.35275E+00,-0.35258E+00,-0.35243E+00,-0.35229E+00,
     &-0.35217E+00,-0.35206E+00,-0.35197E+00,-0.35189E+00,-0.35183E+00,
     &-0.35178E+00,-0.35174E+00,-0.35172E+00,-0.35171E+00,-0.35171E+00,
     &-0.35173E+00,-0.35176E+00,-0.35180E+00,-0.35186E+00,-0.35193E+00,
     &-0.35201E+00,-0.35210E+00,-0.35220E+00,-0.35232E+00,-0.35245E+00,
     &-0.35258E+00,-0.35274E+00,-0.35290E+00,-0.35307E+00,-0.35326E+00,
     &-0.35345E+00,-0.35366E+00,-0.35387E+00,-0.35410E+00,-0.35434E+00,
     &-0.35459E+00,-0.35485E+00,-0.35512E+00,-0.35540E+00,-0.35569E+00/

      DATA (BNC13M(I),I=701,741)/
     &-0.35599E+00,-0.35629E+00,-0.35661E+00,-0.35694E+00,-0.35728E+00,
     &-0.35763E+00,-0.35798E+00,-0.35835E+00,-0.35872E+00,-0.35911E+00,
     &-0.35950E+00,-0.35990E+00,-0.36031E+00,-0.36073E+00,-0.36116E+00,
     &-0.36159E+00,-0.36204E+00,-0.36249E+00,-0.36295E+00,-0.36342E+00,
     &-0.36390E+00,-0.36438E+00,-0.36487E+00,-0.36537E+00,-0.36588E+00,
     &-0.36640E+00,-0.36692E+00,-0.36746E+00,-0.36799E+00,-0.36854E+00,
     &-0.36909E+00,-0.36966E+00,-0.37022E+00,-0.37080E+00,-0.37138E+00,
     &-0.37197E+00,-0.37257E+00,-0.37317E+00,-0.37378E+00,-0.37440E+00,
     &-0.37502E+00
     & /
      END


 
C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF298A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA KMCF298A
C
C *** Common block definition
C
      COMMON /KMC298/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.45329E-01,-0.94600E-01,-0.11698E+00,-0.13161E+00,-0.14224E+00,
     &-0.15040E+00,-0.15685E+00,-0.16205E+00,-0.16630E+00,-0.16979E+00,
     &-0.17267E+00,-0.17504E+00,-0.17699E+00,-0.17857E+00,-0.17984E+00,
     &-0.18084E+00,-0.18160E+00,-0.18215E+00,-0.18251E+00,-0.18271E+00,
     &-0.18276E+00,-0.18267E+00,-0.18247E+00,-0.18215E+00,-0.18174E+00,
     &-0.18124E+00,-0.18065E+00,-0.17999E+00,-0.17926E+00,-0.17847E+00,
     &-0.17763E+00,-0.17673E+00,-0.17578E+00,-0.17480E+00,-0.17377E+00,
     &-0.17270E+00,-0.17160E+00,-0.17047E+00,-0.16931E+00,-0.16813E+00,
     &-0.16692E+00,-0.16569E+00,-0.16445E+00,-0.16318E+00,-0.16190E+00,
     &-0.16060E+00,-0.15929E+00,-0.15796E+00,-0.15663E+00,-0.15528E+00,
     &-0.15392E+00,-0.15256E+00,-0.15118E+00,-0.14980E+00,-0.14841E+00,
     &-0.14701E+00,-0.14561E+00,-0.14419E+00,-0.14277E+00,-0.14135E+00,
     &-0.13992E+00,-0.13848E+00,-0.13703E+00,-0.13558E+00,-0.13412E+00,
     &-0.13265E+00,-0.13117E+00,-0.12969E+00,-0.12820E+00,-0.12670E+00,
     &-0.12519E+00,-0.12367E+00,-0.12214E+00,-0.12060E+00,-0.11905E+00,
     &-0.11750E+00,-0.11593E+00,-0.11434E+00,-0.11275E+00,-0.11115E+00,
     &-0.10953E+00,-0.10790E+00,-0.10626E+00,-0.10460E+00,-0.10293E+00,
     &-0.10125E+00,-0.99560E-01,-0.97854E-01,-0.96134E-01,-0.94401E-01,
     &-0.92654E-01,-0.90895E-01,-0.89123E-01,-0.87339E-01,-0.85542E-01,
     &-0.83733E-01,-0.81912E-01,-0.80079E-01,-0.78234E-01,-0.76379E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.74513E-01,-0.72637E-01,-0.70750E-01,-0.68854E-01,-0.66949E-01,
     &-0.65035E-01,-0.63113E-01,-0.61182E-01,-0.59244E-01,-0.57299E-01,
     &-0.55348E-01,-0.53390E-01,-0.51426E-01,-0.49456E-01,-0.47482E-01,
     &-0.45503E-01,-0.43520E-01,-0.41532E-01,-0.39541E-01,-0.37547E-01,
     &-0.35550E-01,-0.33550E-01,-0.31547E-01,-0.29543E-01,-0.27537E-01,
     &-0.25529E-01,-0.23520E-01,-0.21510E-01,-0.19499E-01,-0.17488E-01,
     &-0.15476E-01,-0.13463E-01,-0.11451E-01,-0.94381E-02,-0.74256E-02,
     &-0.54134E-02,-0.34017E-02,-0.13906E-02, 0.62006E-03, 0.26299E-02,
     & 0.46389E-02, 0.66470E-02, 0.86541E-02, 0.10660E-01, 0.12665E-01,
     & 0.14669E-01, 0.16671E-01, 0.18672E-01, 0.20672E-01, 0.22671E-01,
     & 0.24667E-01, 0.26663E-01, 0.28657E-01, 0.30649E-01, 0.32639E-01,
     & 0.34628E-01, 0.36616E-01, 0.38601E-01, 0.40585E-01, 0.42567E-01,
     & 0.44547E-01, 0.46525E-01, 0.48502E-01, 0.50476E-01, 0.52449E-01,
     & 0.54419E-01, 0.56388E-01, 0.58355E-01, 0.60320E-01, 0.62282E-01,
     & 0.64243E-01, 0.66202E-01, 0.68158E-01, 0.70113E-01, 0.72065E-01,
     & 0.74015E-01, 0.75963E-01, 0.77909E-01, 0.79853E-01, 0.81794E-01,
     & 0.83734E-01, 0.85671E-01, 0.87606E-01, 0.89539E-01, 0.91469E-01,
     & 0.93397E-01, 0.95323E-01, 0.97247E-01, 0.99168E-01, 0.10109E+00,
     & 0.10300E+00, 0.10492E+00, 0.10683E+00, 0.10874E+00, 0.11065E+00,
     & 0.11255E+00, 0.11446E+00, 0.11636E+00, 0.11825E+00, 0.12015E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.12204E+00, 0.12393E+00, 0.12582E+00, 0.12771E+00, 0.12959E+00,
     & 0.13147E+00, 0.13335E+00, 0.13523E+00, 0.13710E+00, 0.13897E+00,
     & 0.14084E+00, 0.14271E+00, 0.14457E+00, 0.14643E+00, 0.14829E+00,
     & 0.15015E+00, 0.15200E+00, 0.15385E+00, 0.15570E+00, 0.15755E+00,
     & 0.15939E+00, 0.16124E+00, 0.16307E+00, 0.16491E+00, 0.16675E+00,
     & 0.16858E+00, 0.17041E+00, 0.17223E+00, 0.17406E+00, 0.17588E+00,
     & 0.17770E+00, 0.17952E+00, 0.18133E+00, 0.18314E+00, 0.18495E+00,
     & 0.18676E+00, 0.18856E+00, 0.19036E+00, 0.19216E+00, 0.19396E+00,
     & 0.19575E+00, 0.19755E+00, 0.19934E+00, 0.20112E+00, 0.20291E+00,
     & 0.20469E+00, 0.20647E+00, 0.20824E+00, 0.21002E+00, 0.21179E+00,
     & 0.21356E+00, 0.21533E+00, 0.21709E+00, 0.21885E+00, 0.22061E+00,
     & 0.22237E+00, 0.22412E+00, 0.22588E+00, 0.22763E+00, 0.22937E+00,
     & 0.23112E+00, 0.23286E+00, 0.23460E+00, 0.23634E+00, 0.23807E+00,
     & 0.23980E+00, 0.24153E+00, 0.24326E+00, 0.24498E+00, 0.24671E+00,
     & 0.24843E+00, 0.25014E+00, 0.25186E+00, 0.25357E+00, 0.25528E+00,
     & 0.25699E+00, 0.25869E+00, 0.26040E+00, 0.26210E+00, 0.26380E+00,
     & 0.26549E+00, 0.26718E+00, 0.26888E+00, 0.27056E+00, 0.27225E+00,
     & 0.27393E+00, 0.27561E+00, 0.27729E+00, 0.27897E+00, 0.28064E+00,
     & 0.28231E+00, 0.28398E+00, 0.28565E+00, 0.28731E+00, 0.28898E+00,
     & 0.29064E+00, 0.29229E+00, 0.29395E+00, 0.29560E+00, 0.29725E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.29890E+00, 0.30054E+00, 0.30219E+00, 0.30383E+00, 0.30547E+00,
     & 0.30710E+00, 0.30874E+00, 0.31037E+00, 0.31200E+00, 0.31362E+00,
     & 0.31525E+00, 0.31687E+00, 0.31849E+00, 0.32011E+00, 0.32172E+00,
     & 0.32333E+00, 0.32495E+00, 0.32655E+00, 0.32816E+00, 0.32976E+00,
     & 0.33136E+00, 0.33296E+00, 0.33456E+00, 0.33616E+00, 0.33775E+00,
     & 0.33934E+00, 0.34093E+00, 0.34251E+00, 0.34410E+00, 0.34568E+00,
     & 0.34726E+00, 0.34883E+00, 0.35041E+00, 0.35198E+00, 0.35355E+00,
     & 0.35512E+00, 0.35668E+00, 0.35825E+00, 0.35981E+00, 0.36137E+00,
     & 0.36292E+00, 0.36448E+00, 0.36603E+00, 0.36758E+00, 0.36913E+00,
     & 0.37067E+00, 0.37222E+00, 0.37376E+00, 0.37530E+00, 0.37684E+00,
     & 0.37837E+00, 0.37990E+00, 0.38144E+00, 0.38296E+00, 0.38449E+00,
     & 0.38602E+00, 0.38754E+00, 0.38906E+00, 0.39058E+00, 0.39209E+00,
     & 0.39361E+00, 0.39512E+00, 0.39663E+00, 0.39814E+00, 0.39964E+00,
     & 0.40114E+00, 0.40265E+00, 0.40414E+00, 0.40564E+00, 0.40714E+00,
     & 0.40863E+00, 0.41012E+00, 0.41161E+00, 0.41310E+00, 0.41458E+00,
     & 0.41606E+00, 0.41755E+00, 0.41902E+00, 0.42050E+00, 0.42198E+00,
     & 0.42345E+00, 0.42492E+00, 0.42639E+00, 0.42785E+00, 0.42932E+00,
     & 0.43078E+00, 0.43224E+00, 0.43370E+00, 0.43516E+00, 0.43661E+00,
     & 0.43807E+00, 0.43952E+00, 0.44097E+00, 0.44241E+00, 0.44386E+00,
     & 0.44530E+00, 0.44674E+00, 0.44818E+00, 0.44962E+00, 0.45105E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.45249E+00, 0.45392E+00, 0.45535E+00, 0.45677E+00, 0.45820E+00,
     & 0.45962E+00, 0.46105E+00, 0.46247E+00, 0.46388E+00, 0.46530E+00,
     & 0.46671E+00, 0.46813E+00, 0.46954E+00, 0.47094E+00, 0.47235E+00,
     & 0.47376E+00, 0.47516E+00, 0.47656E+00, 0.47796E+00, 0.47936E+00,
     & 0.48075E+00, 0.48215E+00, 0.48354E+00, 0.48493E+00, 0.48632E+00,
     & 0.48770E+00, 0.48909E+00, 0.49047E+00, 0.49185E+00, 0.49323E+00,
     & 0.49461E+00, 0.49598E+00, 0.49735E+00, 0.49873E+00, 0.50010E+00,
     & 0.50146E+00, 0.50283E+00, 0.50420E+00, 0.50556E+00, 0.50692E+00,
     & 0.50828E+00, 0.50964E+00, 0.51099E+00, 0.51235E+00, 0.51370E+00,
     & 0.51505E+00, 0.51640E+00, 0.51774E+00, 0.51909E+00, 0.52043E+00,
     & 0.52178E+00, 0.52312E+00, 0.52445E+00, 0.52579E+00, 0.52713E+00,
     & 0.52846E+00, 0.52979E+00, 0.53112E+00, 0.53245E+00, 0.53377E+00,
     & 0.53510E+00, 0.53642E+00, 0.53774E+00, 0.53906E+00, 0.54038E+00,
     & 0.54170E+00, 0.54301E+00, 0.54433E+00, 0.54564E+00, 0.54695E+00,
     & 0.54825E+00, 0.54956E+00, 0.55086E+00, 0.55217E+00, 0.55347E+00,
     & 0.55477E+00, 0.55607E+00, 0.55736E+00, 0.55866E+00, 0.55995E+00,
     & 0.56124E+00, 0.56253E+00, 0.56382E+00, 0.56511E+00, 0.56639E+00,
     & 0.56768E+00, 0.56896E+00, 0.57024E+00, 0.57152E+00, 0.57280E+00,
     & 0.57407E+00, 0.57534E+00, 0.57662E+00, 0.57789E+00, 0.57916E+00,
     & 0.58043E+00, 0.58169E+00, 0.58296E+00, 0.58422E+00, 0.58548E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.58674E+00, 0.58800E+00, 0.58926E+00, 0.59051E+00, 0.59176E+00,
     & 0.59302E+00, 0.59427E+00, 0.59552E+00, 0.59676E+00, 0.59801E+00,
     & 0.59925E+00, 0.60050E+00, 0.60174E+00, 0.60298E+00, 0.60422E+00,
     & 0.60545E+00, 0.60669E+00, 0.60792E+00, 0.60916E+00, 0.61039E+00,
     & 0.61162E+00, 0.61285E+00, 0.61407E+00, 0.61530E+00, 0.61652E+00,
     & 0.61774E+00, 0.61896E+00, 0.62018E+00, 0.62140E+00, 0.62262E+00,
     & 0.62383E+00, 0.62505E+00, 0.62626E+00, 0.62747E+00, 0.62868E+00,
     & 0.62989E+00, 0.63109E+00, 0.63230E+00, 0.63350E+00, 0.63470E+00,
     & 0.63590E+00, 0.63710E+00, 0.63830E+00, 0.63950E+00, 0.64069E+00,
     & 0.64188E+00, 0.64308E+00, 0.64427E+00, 0.64546E+00, 0.64664E+00,
     & 0.64783E+00, 0.64902E+00, 0.65020E+00, 0.65138E+00, 0.65256E+00,
     & 0.65374E+00, 0.65492E+00, 0.65610E+00, 0.65727E+00, 0.65845E+00,
     & 0.65962E+00, 0.66079E+00, 0.66196E+00, 0.66313E+00, 0.66430E+00,
     & 0.66546E+00, 0.66663E+00, 0.66779E+00, 0.66895E+00, 0.67011E+00,
     & 0.67127E+00, 0.67243E+00, 0.67359E+00, 0.67474E+00, 0.67590E+00,
     & 0.67705E+00, 0.67820E+00, 0.67935E+00, 0.68050E+00, 0.68165E+00,
     & 0.68279E+00, 0.68394E+00, 0.68508E+00, 0.68622E+00, 0.68736E+00,
     & 0.68850E+00, 0.68964E+00, 0.69078E+00, 0.69191E+00, 0.69305E+00,
     & 0.69418E+00, 0.69531E+00, 0.69644E+00, 0.69757E+00, 0.69870E+00,
     & 0.69983E+00, 0.70095E+00, 0.70208E+00, 0.70320E+00, 0.70432E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.71636E+00, 0.72738E+00, 0.73828E+00, 0.74906E+00, 0.75972E+00,
     & 0.77027E+00, 0.78071E+00, 0.79105E+00, 0.80127E+00, 0.81139E+00,
     & 0.82141E+00, 0.83133E+00, 0.84115E+00, 0.85087E+00, 0.86050E+00,
     & 0.87004E+00, 0.87948E+00, 0.88883E+00, 0.89810E+00, 0.90728E+00,
     & 0.91638E+00, 0.92539E+00, 0.93432E+00, 0.94316E+00, 0.95193E+00,
     & 0.96063E+00, 0.96924E+00, 0.97778E+00, 0.98625E+00, 0.99464E+00,
     & 0.10030E+01, 0.10112E+01, 0.10194E+01, 0.10275E+01, 0.10356E+01,
     & 0.10435E+01, 0.10515E+01, 0.10593E+01, 0.10671E+01, 0.10748E+01,
     & 0.10825E+01, 0.10901E+01, 0.10977E+01, 0.11051E+01, 0.11126E+01,
     & 0.11200E+01, 0.11273E+01, 0.11345E+01, 0.11418E+01, 0.11489E+01,
     & 0.11560E+01, 0.11631E+01, 0.11701E+01, 0.11770E+01, 0.11839E+01,
     & 0.11908E+01, 0.11976E+01, 0.12044E+01, 0.12111E+01, 0.12177E+01,
     & 0.12244E+01, 0.12309E+01, 0.12375E+01, 0.12440E+01, 0.12504E+01,
     & 0.12568E+01, 0.12632E+01, 0.12695E+01, 0.12758E+01, 0.12820E+01,
     & 0.12882E+01, 0.12943E+01, 0.13005E+01, 0.13065E+01, 0.13126E+01,
     & 0.13186E+01, 0.13246E+01, 0.13305E+01, 0.13364E+01, 0.13422E+01,
     & 0.13481E+01, 0.13539E+01, 0.13596E+01, 0.13653E+01, 0.13710E+01,
     & 0.13767E+01, 0.13823E+01, 0.13879E+01, 0.13934E+01, 0.13990E+01,
     & 0.14044E+01, 0.14099E+01, 0.14153E+01, 0.14207E+01, 0.14261E+01,
     & 0.14314E+01, 0.14368E+01, 0.14420E+01, 0.14473E+01, 0.14525E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.14577E+01, 0.14629E+01, 0.14680E+01, 0.14731E+01, 0.14782E+01,
     & 0.14833E+01, 0.14883E+01, 0.14933E+01, 0.14983E+01, 0.15033E+01,
     & 0.15082E+01, 0.15131E+01, 0.15180E+01, 0.15228E+01, 0.15277E+01,
     & 0.15325E+01, 0.15372E+01, 0.15420E+01, 0.15467E+01, 0.15515E+01,
     & 0.15561E+01, 0.15608E+01, 0.15655E+01, 0.15701E+01, 0.15747E+01,
     & 0.15793E+01, 0.15838E+01, 0.15884E+01, 0.15929E+01, 0.15974E+01,
     & 0.16018E+01, 0.16063E+01, 0.16107E+01, 0.16151E+01, 0.16195E+01,
     & 0.16239E+01, 0.16282E+01, 0.16326E+01, 0.16369E+01, 0.16412E+01,
     & 0.16454E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.93069E-01,-0.20213E+00,-0.25643E+00,-0.29470E+00,-0.32465E+00,
     &-0.34938E+00,-0.37051E+00,-0.38898E+00,-0.40540E+00,-0.42018E+00,
     &-0.43364E+00,-0.44599E+00,-0.45740E+00,-0.46801E+00,-0.47792E+00,
     &-0.48722E+00,-0.49598E+00,-0.50426E+00,-0.51211E+00,-0.51957E+00,
     &-0.52668E+00,-0.53347E+00,-0.53997E+00,-0.54620E+00,-0.55218E+00,
     &-0.55793E+00,-0.56347E+00,-0.56881E+00,-0.57397E+00,-0.57896E+00,
     &-0.58378E+00,-0.58845E+00,-0.59298E+00,-0.59738E+00,-0.60165E+00,
     &-0.60580E+00,-0.60983E+00,-0.61376E+00,-0.61759E+00,-0.62131E+00,
     &-0.62495E+00,-0.62849E+00,-0.63195E+00,-0.63533E+00,-0.63864E+00,
     &-0.64187E+00,-0.64502E+00,-0.64811E+00,-0.65114E+00,-0.65410E+00,
     &-0.65700E+00,-0.65984E+00,-0.66263E+00,-0.66536E+00,-0.66804E+00,
     &-0.67068E+00,-0.67326E+00,-0.67580E+00,-0.67829E+00,-0.68074E+00,
     &-0.68314E+00,-0.68551E+00,-0.68784E+00,-0.69013E+00,-0.69239E+00,
     &-0.69461E+00,-0.69679E+00,-0.69894E+00,-0.70107E+00,-0.70316E+00,
     &-0.70522E+00,-0.70725E+00,-0.70925E+00,-0.71123E+00,-0.71318E+00,
     &-0.71511E+00,-0.71701E+00,-0.71889E+00,-0.72075E+00,-0.72258E+00,
     &-0.72439E+00,-0.72618E+00,-0.72795E+00,-0.72970E+00,-0.73143E+00,
     &-0.73315E+00,-0.73484E+00,-0.73652E+00,-0.73818E+00,-0.73982E+00,
     &-0.74145E+00,-0.74306E+00,-0.74465E+00,-0.74623E+00,-0.74780E+00,
     &-0.74935E+00,-0.75088E+00,-0.75241E+00,-0.75392E+00,-0.75541E+00/

      DATA (BNC02M(I),I=101,200)/
     &-0.75689E+00,-0.75836E+00,-0.75982E+00,-0.76126E+00,-0.76269E+00,
     &-0.76411E+00,-0.76552E+00,-0.76692E+00,-0.76830E+00,-0.76968E+00,
     &-0.77104E+00,-0.77239E+00,-0.77373E+00,-0.77506E+00,-0.77638E+00,
     &-0.77769E+00,-0.77899E+00,-0.78028E+00,-0.78155E+00,-0.78282E+00,
     &-0.78408E+00,-0.78533E+00,-0.78657E+00,-0.78780E+00,-0.78902E+00,
     &-0.79024E+00,-0.79144E+00,-0.79263E+00,-0.79382E+00,-0.79500E+00,
     &-0.79616E+00,-0.79732E+00,-0.79848E+00,-0.79962E+00,-0.80076E+00,
     &-0.80188E+00,-0.80300E+00,-0.80411E+00,-0.80522E+00,-0.80631E+00,
     &-0.80740E+00,-0.80848E+00,-0.80956E+00,-0.81063E+00,-0.81169E+00,
     &-0.81274E+00,-0.81379E+00,-0.81482E+00,-0.81586E+00,-0.81688E+00,
     &-0.81790E+00,-0.81891E+00,-0.81992E+00,-0.82092E+00,-0.82191E+00,
     &-0.82290E+00,-0.82388E+00,-0.82486E+00,-0.82583E+00,-0.82679E+00,
     &-0.82775E+00,-0.82870E+00,-0.82965E+00,-0.83059E+00,-0.83152E+00,
     &-0.83245E+00,-0.83338E+00,-0.83429E+00,-0.83521E+00,-0.83612E+00,
     &-0.83702E+00,-0.83792E+00,-0.83881E+00,-0.83970E+00,-0.84058E+00,
     &-0.84146E+00,-0.84233E+00,-0.84320E+00,-0.84407E+00,-0.84492E+00,
     &-0.84578E+00,-0.84663E+00,-0.84747E+00,-0.84832E+00,-0.84915E+00,
     &-0.84998E+00,-0.85081E+00,-0.85164E+00,-0.85245E+00,-0.85327E+00,
     &-0.85408E+00,-0.85489E+00,-0.85569E+00,-0.85649E+00,-0.85728E+00,
     &-0.85807E+00,-0.85886E+00,-0.85964E+00,-0.86042E+00,-0.86120E+00/

      DATA (BNC02M(I),I=201,300)/
     &-0.86197E+00,-0.86274E+00,-0.86350E+00,-0.86426E+00,-0.86502E+00,
     &-0.86577E+00,-0.86652E+00,-0.86727E+00,-0.86801E+00,-0.86875E+00,
     &-0.86949E+00,-0.87022E+00,-0.87095E+00,-0.87167E+00,-0.87240E+00,
     &-0.87312E+00,-0.87383E+00,-0.87454E+00,-0.87525E+00,-0.87596E+00,
     &-0.87666E+00,-0.87736E+00,-0.87806E+00,-0.87875E+00,-0.87945E+00,
     &-0.88013E+00,-0.88082E+00,-0.88150E+00,-0.88218E+00,-0.88286E+00,
     &-0.88353E+00,-0.88420E+00,-0.88487E+00,-0.88553E+00,-0.88620E+00,
     &-0.88686E+00,-0.88751E+00,-0.88817E+00,-0.88882E+00,-0.88947E+00,
     &-0.89011E+00,-0.89076E+00,-0.89140E+00,-0.89203E+00,-0.89267E+00,
     &-0.89330E+00,-0.89393E+00,-0.89456E+00,-0.89519E+00,-0.89581E+00,
     &-0.89643E+00,-0.89705E+00,-0.89767E+00,-0.89828E+00,-0.89889E+00,
     &-0.89950E+00,-0.90011E+00,-0.90071E+00,-0.90131E+00,-0.90191E+00,
     &-0.90251E+00,-0.90310E+00,-0.90370E+00,-0.90429E+00,-0.90488E+00,
     &-0.90546E+00,-0.90605E+00,-0.90663E+00,-0.90721E+00,-0.90779E+00,
     &-0.90836E+00,-0.90894E+00,-0.90951E+00,-0.91008E+00,-0.91064E+00,
     &-0.91121E+00,-0.91177E+00,-0.91233E+00,-0.91289E+00,-0.91345E+00,
     &-0.91401E+00,-0.91456E+00,-0.91511E+00,-0.91566E+00,-0.91621E+00,
     &-0.91675E+00,-0.91730E+00,-0.91784E+00,-0.91838E+00,-0.91892E+00,
     &-0.91946E+00,-0.91999E+00,-0.92052E+00,-0.92105E+00,-0.92158E+00,
     &-0.92211E+00,-0.92264E+00,-0.92316E+00,-0.92368E+00,-0.92420E+00/

      DATA (BNC02M(I),I=301,400)/
     &-0.92472E+00,-0.92524E+00,-0.92576E+00,-0.92627E+00,-0.92678E+00,
     &-0.92729E+00,-0.92780E+00,-0.92831E+00,-0.92881E+00,-0.92932E+00,
     &-0.92982E+00,-0.93032E+00,-0.93082E+00,-0.93132E+00,-0.93181E+00,
     &-0.93231E+00,-0.93280E+00,-0.93329E+00,-0.93378E+00,-0.93427E+00,
     &-0.93476E+00,-0.93524E+00,-0.93573E+00,-0.93621E+00,-0.93669E+00,
     &-0.93717E+00,-0.93765E+00,-0.93812E+00,-0.93860E+00,-0.93907E+00,
     &-0.93954E+00,-0.94002E+00,-0.94048E+00,-0.94095E+00,-0.94142E+00,
     &-0.94188E+00,-0.94235E+00,-0.94281E+00,-0.94327E+00,-0.94373E+00,
     &-0.94419E+00,-0.94465E+00,-0.94510E+00,-0.94556E+00,-0.94601E+00,
     &-0.94646E+00,-0.94691E+00,-0.94736E+00,-0.94781E+00,-0.94826E+00,
     &-0.94870E+00,-0.94915E+00,-0.94959E+00,-0.95003E+00,-0.95047E+00,
     &-0.95091E+00,-0.95135E+00,-0.95178E+00,-0.95222E+00,-0.95265E+00,
     &-0.95309E+00,-0.95352E+00,-0.95395E+00,-0.95438E+00,-0.95481E+00,
     &-0.95523E+00,-0.95566E+00,-0.95608E+00,-0.95651E+00,-0.95693E+00,
     &-0.95735E+00,-0.95777E+00,-0.95819E+00,-0.95861E+00,-0.95903E+00,
     &-0.95944E+00,-0.95986E+00,-0.96027E+00,-0.96068E+00,-0.96109E+00,
     &-0.96150E+00,-0.96191E+00,-0.96232E+00,-0.96273E+00,-0.96313E+00,
     &-0.96354E+00,-0.96394E+00,-0.96435E+00,-0.96475E+00,-0.96515E+00,
     &-0.96555E+00,-0.96595E+00,-0.96634E+00,-0.96674E+00,-0.96714E+00,
     &-0.96753E+00,-0.96793E+00,-0.96832E+00,-0.96871E+00,-0.96910E+00/

      DATA (BNC02M(I),I=401,500)/
     &-0.96949E+00,-0.96988E+00,-0.97027E+00,-0.97065E+00,-0.97104E+00,
     &-0.97142E+00,-0.97181E+00,-0.97219E+00,-0.97257E+00,-0.97295E+00,
     &-0.97333E+00,-0.97371E+00,-0.97409E+00,-0.97447E+00,-0.97484E+00,
     &-0.97522E+00,-0.97560E+00,-0.97597E+00,-0.97634E+00,-0.97671E+00,
     &-0.97708E+00,-0.97745E+00,-0.97782E+00,-0.97819E+00,-0.97856E+00,
     &-0.97893E+00,-0.97929E+00,-0.97966E+00,-0.98002E+00,-0.98038E+00,
     &-0.98075E+00,-0.98111E+00,-0.98147E+00,-0.98183E+00,-0.98219E+00,
     &-0.98255E+00,-0.98290E+00,-0.98326E+00,-0.98362E+00,-0.98397E+00,
     &-0.98432E+00,-0.98468E+00,-0.98503E+00,-0.98538E+00,-0.98573E+00,
     &-0.98608E+00,-0.98643E+00,-0.98678E+00,-0.98713E+00,-0.98747E+00,
     &-0.98782E+00,-0.98817E+00,-0.98851E+00,-0.98885E+00,-0.98920E+00,
     &-0.98954E+00,-0.98988E+00,-0.99022E+00,-0.99056E+00,-0.99090E+00,
     &-0.99124E+00,-0.99158E+00,-0.99191E+00,-0.99225E+00,-0.99259E+00,
     &-0.99292E+00,-0.99325E+00,-0.99359E+00,-0.99392E+00,-0.99425E+00,
     &-0.99458E+00,-0.99491E+00,-0.99524E+00,-0.99557E+00,-0.99590E+00,
     &-0.99623E+00,-0.99656E+00,-0.99688E+00,-0.99721E+00,-0.99753E+00,
     &-0.99786E+00,-0.99818E+00,-0.99850E+00,-0.99883E+00,-0.99915E+00,
     &-0.99947E+00,-0.99979E+00,-0.10001E+01,-0.10004E+01,-0.10007E+01,
     &-0.10011E+01,-0.10014E+01,-0.10017E+01,-0.10020E+01,-0.10023E+01,
     &-0.10026E+01,-0.10030E+01,-0.10033E+01,-0.10036E+01,-0.10039E+01/

      DATA (BNC02M(I),I=501,600)/
     &-0.10042E+01,-0.10045E+01,-0.10048E+01,-0.10051E+01,-0.10054E+01,
     &-0.10057E+01,-0.10061E+01,-0.10064E+01,-0.10067E+01,-0.10070E+01,
     &-0.10073E+01,-0.10076E+01,-0.10079E+01,-0.10082E+01,-0.10085E+01,
     &-0.10088E+01,-0.10091E+01,-0.10094E+01,-0.10097E+01,-0.10100E+01,
     &-0.10103E+01,-0.10106E+01,-0.10109E+01,-0.10112E+01,-0.10115E+01,
     &-0.10118E+01,-0.10121E+01,-0.10124E+01,-0.10127E+01,-0.10130E+01,
     &-0.10132E+01,-0.10135E+01,-0.10138E+01,-0.10141E+01,-0.10144E+01,
     &-0.10147E+01,-0.10150E+01,-0.10153E+01,-0.10156E+01,-0.10159E+01,
     &-0.10161E+01,-0.10164E+01,-0.10167E+01,-0.10170E+01,-0.10173E+01,
     &-0.10176E+01,-0.10179E+01,-0.10181E+01,-0.10184E+01,-0.10187E+01,
     &-0.10190E+01,-0.10193E+01,-0.10196E+01,-0.10198E+01,-0.10201E+01,
     &-0.10204E+01,-0.10207E+01,-0.10210E+01,-0.10212E+01,-0.10215E+01,
     &-0.10218E+01,-0.10221E+01,-0.10223E+01,-0.10226E+01,-0.10229E+01,
     &-0.10232E+01,-0.10234E+01,-0.10237E+01,-0.10240E+01,-0.10243E+01,
     &-0.10245E+01,-0.10248E+01,-0.10251E+01,-0.10253E+01,-0.10256E+01,
     &-0.10259E+01,-0.10261E+01,-0.10264E+01,-0.10267E+01,-0.10270E+01,
     &-0.10272E+01,-0.10275E+01,-0.10278E+01,-0.10280E+01,-0.10283E+01,
     &-0.10285E+01,-0.10288E+01,-0.10291E+01,-0.10293E+01,-0.10296E+01,
     &-0.10299E+01,-0.10301E+01,-0.10304E+01,-0.10307E+01,-0.10309E+01,
     &-0.10312E+01,-0.10314E+01,-0.10317E+01,-0.10319E+01,-0.10322E+01/

      DATA (BNC02M(I),I=601,700)/
     &-0.10350E+01,-0.10375E+01,-0.10400E+01,-0.10424E+01,-0.10448E+01,
     &-0.10471E+01,-0.10495E+01,-0.10517E+01,-0.10540E+01,-0.10562E+01,
     &-0.10584E+01,-0.10605E+01,-0.10627E+01,-0.10647E+01,-0.10668E+01,
     &-0.10688E+01,-0.10708E+01,-0.10728E+01,-0.10748E+01,-0.10767E+01,
     &-0.10786E+01,-0.10805E+01,-0.10823E+01,-0.10841E+01,-0.10860E+01,
     &-0.10877E+01,-0.10895E+01,-0.10912E+01,-0.10930E+01,-0.10947E+01,
     &-0.10963E+01,-0.10980E+01,-0.10996E+01,-0.11013E+01,-0.11029E+01,
     &-0.11044E+01,-0.11060E+01,-0.11076E+01,-0.11091E+01,-0.11106E+01,
     &-0.11121E+01,-0.11136E+01,-0.11151E+01,-0.11165E+01,-0.11180E+01,
     &-0.11194E+01,-0.11208E+01,-0.11222E+01,-0.11236E+01,-0.11250E+01,
     &-0.11263E+01,-0.11277E+01,-0.11290E+01,-0.11303E+01,-0.11316E+01,
     &-0.11329E+01,-0.11342E+01,-0.11355E+01,-0.11368E+01,-0.11380E+01,
     &-0.11392E+01,-0.11405E+01,-0.11417E+01,-0.11429E+01,-0.11441E+01,
     &-0.11453E+01,-0.11464E+01,-0.11476E+01,-0.11488E+01,-0.11499E+01,
     &-0.11510E+01,-0.11522E+01,-0.11533E+01,-0.11544E+01,-0.11555E+01,
     &-0.11566E+01,-0.11577E+01,-0.11587E+01,-0.11598E+01,-0.11609E+01,
     &-0.11619E+01,-0.11630E+01,-0.11640E+01,-0.11650E+01,-0.11660E+01,
     &-0.11670E+01,-0.11680E+01,-0.11690E+01,-0.11700E+01,-0.11710E+01,
     &-0.11720E+01,-0.11729E+01,-0.11739E+01,-0.11749E+01,-0.11758E+01,
     &-0.11767E+01,-0.11777E+01,-0.11786E+01,-0.11795E+01,-0.11804E+01/

      DATA (BNC02M(I),I=701,741)/
     &-0.11813E+01,-0.11822E+01,-0.11831E+01,-0.11840E+01,-0.11849E+01,
     &-0.11858E+01,-0.11867E+01,-0.11875E+01,-0.11884E+01,-0.11893E+01,
     &-0.11901E+01,-0.11909E+01,-0.11918E+01,-0.11926E+01,-0.11934E+01,
     &-0.11943E+01,-0.11951E+01,-0.11959E+01,-0.11967E+01,-0.11975E+01,
     &-0.11983E+01,-0.11991E+01,-0.11999E+01,-0.12007E+01,-0.12015E+01,
     &-0.12022E+01,-0.12030E+01,-0.12038E+01,-0.12045E+01,-0.12053E+01,
     &-0.12060E+01,-0.12068E+01,-0.12075E+01,-0.12083E+01,-0.12090E+01,
     &-0.12097E+01,-0.12105E+01,-0.12112E+01,-0.12119E+01,-0.12126E+01,
     &-0.12133E+01
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.46649E-01,-0.10169E+00,-0.12931E+00,-0.14890E+00,-0.16430E+00,
     &-0.17709E+00,-0.18807E+00,-0.19770E+00,-0.20631E+00,-0.21409E+00,
     &-0.22120E+00,-0.22775E+00,-0.23383E+00,-0.23950E+00,-0.24481E+00,
     &-0.24982E+00,-0.25456E+00,-0.25905E+00,-0.26332E+00,-0.26739E+00,
     &-0.27128E+00,-0.27501E+00,-0.27859E+00,-0.28204E+00,-0.28535E+00,
     &-0.28855E+00,-0.29164E+00,-0.29462E+00,-0.29751E+00,-0.30031E+00,
     &-0.30303E+00,-0.30567E+00,-0.30823E+00,-0.31072E+00,-0.31315E+00,
     &-0.31551E+00,-0.31782E+00,-0.32006E+00,-0.32225E+00,-0.32439E+00,
     &-0.32649E+00,-0.32853E+00,-0.33053E+00,-0.33248E+00,-0.33439E+00,
     &-0.33627E+00,-0.33810E+00,-0.33990E+00,-0.34166E+00,-0.34339E+00,
     &-0.34509E+00,-0.34675E+00,-0.34839E+00,-0.34999E+00,-0.35157E+00,
     &-0.35312E+00,-0.35464E+00,-0.35614E+00,-0.35761E+00,-0.35906E+00,
     &-0.36049E+00,-0.36190E+00,-0.36328E+00,-0.36464E+00,-0.36599E+00,
     &-0.36731E+00,-0.36862E+00,-0.36991E+00,-0.37118E+00,-0.37244E+00,
     &-0.37368E+00,-0.37491E+00,-0.37612E+00,-0.37731E+00,-0.37849E+00,
     &-0.37966E+00,-0.38082E+00,-0.38197E+00,-0.38310E+00,-0.38422E+00,
     &-0.38533E+00,-0.38643E+00,-0.38752E+00,-0.38860E+00,-0.38966E+00,
     &-0.39072E+00,-0.39177E+00,-0.39282E+00,-0.39385E+00,-0.39487E+00,
     &-0.39589E+00,-0.39690E+00,-0.39790E+00,-0.39889E+00,-0.39988E+00,
     &-0.40086E+00,-0.40183E+00,-0.40279E+00,-0.40375E+00,-0.40470E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.40564E+00,-0.40658E+00,-0.40751E+00,-0.40844E+00,-0.40936E+00,
     &-0.41027E+00,-0.41117E+00,-0.41208E+00,-0.41297E+00,-0.41386E+00,
     &-0.41474E+00,-0.41562E+00,-0.41649E+00,-0.41736E+00,-0.41822E+00,
     &-0.41907E+00,-0.41992E+00,-0.42076E+00,-0.42160E+00,-0.42243E+00,
     &-0.42326E+00,-0.42408E+00,-0.42490E+00,-0.42571E+00,-0.42652E+00,
     &-0.42732E+00,-0.42812E+00,-0.42891E+00,-0.42970E+00,-0.43048E+00,
     &-0.43126E+00,-0.43203E+00,-0.43280E+00,-0.43356E+00,-0.43432E+00,
     &-0.43507E+00,-0.43582E+00,-0.43656E+00,-0.43730E+00,-0.43804E+00,
     &-0.43877E+00,-0.43950E+00,-0.44022E+00,-0.44094E+00,-0.44165E+00,
     &-0.44236E+00,-0.44307E+00,-0.44377E+00,-0.44447E+00,-0.44516E+00,
     &-0.44585E+00,-0.44654E+00,-0.44722E+00,-0.44790E+00,-0.44858E+00,
     &-0.44925E+00,-0.44992E+00,-0.45058E+00,-0.45124E+00,-0.45190E+00,
     &-0.45255E+00,-0.45320E+00,-0.45385E+00,-0.45450E+00,-0.45514E+00,
     &-0.45577E+00,-0.45641E+00,-0.45704E+00,-0.45766E+00,-0.45829E+00,
     &-0.45891E+00,-0.45953E+00,-0.46014E+00,-0.46075E+00,-0.46136E+00,
     &-0.46197E+00,-0.46257E+00,-0.46317E+00,-0.46377E+00,-0.46436E+00,
     &-0.46496E+00,-0.46555E+00,-0.46613E+00,-0.46671E+00,-0.46730E+00,
     &-0.46787E+00,-0.46845E+00,-0.46902E+00,-0.46959E+00,-0.47016E+00,
     &-0.47072E+00,-0.47129E+00,-0.47185E+00,-0.47240E+00,-0.47296E+00,
     &-0.47351E+00,-0.47406E+00,-0.47461E+00,-0.47515E+00,-0.47570E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.47624E+00,-0.47678E+00,-0.47731E+00,-0.47785E+00,-0.47838E+00,
     &-0.47891E+00,-0.47943E+00,-0.47996E+00,-0.48048E+00,-0.48100E+00,
     &-0.48152E+00,-0.48204E+00,-0.48255E+00,-0.48306E+00,-0.48357E+00,
     &-0.48408E+00,-0.48459E+00,-0.48509E+00,-0.48559E+00,-0.48609E+00,
     &-0.48659E+00,-0.48709E+00,-0.48758E+00,-0.48807E+00,-0.48856E+00,
     &-0.48905E+00,-0.48954E+00,-0.49002E+00,-0.49051E+00,-0.49099E+00,
     &-0.49147E+00,-0.49194E+00,-0.49242E+00,-0.49289E+00,-0.49336E+00,
     &-0.49383E+00,-0.49430E+00,-0.49477E+00,-0.49523E+00,-0.49570E+00,
     &-0.49616E+00,-0.49662E+00,-0.49708E+00,-0.49753E+00,-0.49799E+00,
     &-0.49844E+00,-0.49889E+00,-0.49934E+00,-0.49979E+00,-0.50024E+00,
     &-0.50068E+00,-0.50113E+00,-0.50157E+00,-0.50201E+00,-0.50245E+00,
     &-0.50289E+00,-0.50332E+00,-0.50376E+00,-0.50419E+00,-0.50462E+00,
     &-0.50505E+00,-0.50548E+00,-0.50591E+00,-0.50634E+00,-0.50676E+00,
     &-0.50718E+00,-0.50760E+00,-0.50803E+00,-0.50844E+00,-0.50886E+00,
     &-0.50928E+00,-0.50969E+00,-0.51011E+00,-0.51052E+00,-0.51093E+00,
     &-0.51134E+00,-0.51175E+00,-0.51215E+00,-0.51256E+00,-0.51296E+00,
     &-0.51336E+00,-0.51377E+00,-0.51417E+00,-0.51457E+00,-0.51496E+00,
     &-0.51536E+00,-0.51576E+00,-0.51615E+00,-0.51654E+00,-0.51693E+00,
     &-0.51732E+00,-0.51771E+00,-0.51810E+00,-0.51849E+00,-0.51887E+00,
     &-0.51926E+00,-0.51964E+00,-0.52002E+00,-0.52040E+00,-0.52078E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.52116E+00,-0.52154E+00,-0.52192E+00,-0.52229E+00,-0.52267E+00,
     &-0.52304E+00,-0.52341E+00,-0.52378E+00,-0.52415E+00,-0.52452E+00,
     &-0.52489E+00,-0.52526E+00,-0.52562E+00,-0.52599E+00,-0.52635E+00,
     &-0.52671E+00,-0.52707E+00,-0.52743E+00,-0.52779E+00,-0.52815E+00,
     &-0.52851E+00,-0.52886E+00,-0.52922E+00,-0.52957E+00,-0.52993E+00,
     &-0.53028E+00,-0.53063E+00,-0.53098E+00,-0.53133E+00,-0.53168E+00,
     &-0.53203E+00,-0.53237E+00,-0.53272E+00,-0.53306E+00,-0.53341E+00,
     &-0.53375E+00,-0.53409E+00,-0.53443E+00,-0.53477E+00,-0.53511E+00,
     &-0.53545E+00,-0.53579E+00,-0.53612E+00,-0.53646E+00,-0.53679E+00,
     &-0.53713E+00,-0.53746E+00,-0.53779E+00,-0.53812E+00,-0.53845E+00,
     &-0.53878E+00,-0.53911E+00,-0.53944E+00,-0.53976E+00,-0.54009E+00,
     &-0.54041E+00,-0.54074E+00,-0.54106E+00,-0.54138E+00,-0.54170E+00,
     &-0.54203E+00,-0.54235E+00,-0.54266E+00,-0.54298E+00,-0.54330E+00,
     &-0.54362E+00,-0.54393E+00,-0.54425E+00,-0.54456E+00,-0.54488E+00,
     &-0.54519E+00,-0.54550E+00,-0.54581E+00,-0.54612E+00,-0.54643E+00,
     &-0.54674E+00,-0.54705E+00,-0.54736E+00,-0.54766E+00,-0.54797E+00,
     &-0.54827E+00,-0.54858E+00,-0.54888E+00,-0.54919E+00,-0.54949E+00,
     &-0.54979E+00,-0.55009E+00,-0.55039E+00,-0.55069E+00,-0.55099E+00,
     &-0.55129E+00,-0.55158E+00,-0.55188E+00,-0.55218E+00,-0.55247E+00,
     &-0.55277E+00,-0.55306E+00,-0.55335E+00,-0.55364E+00,-0.55394E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.55423E+00,-0.55452E+00,-0.55481E+00,-0.55510E+00,-0.55538E+00,
     &-0.55567E+00,-0.55596E+00,-0.55624E+00,-0.55653E+00,-0.55681E+00,
     &-0.55710E+00,-0.55738E+00,-0.55767E+00,-0.55795E+00,-0.55823E+00,
     &-0.55851E+00,-0.55879E+00,-0.55907E+00,-0.55935E+00,-0.55963E+00,
     &-0.55991E+00,-0.56018E+00,-0.56046E+00,-0.56074E+00,-0.56101E+00,
     &-0.56129E+00,-0.56156E+00,-0.56184E+00,-0.56211E+00,-0.56238E+00,
     &-0.56265E+00,-0.56292E+00,-0.56319E+00,-0.56346E+00,-0.56373E+00,
     &-0.56400E+00,-0.56427E+00,-0.56454E+00,-0.56481E+00,-0.56507E+00,
     &-0.56534E+00,-0.56560E+00,-0.56587E+00,-0.56613E+00,-0.56640E+00,
     &-0.56666E+00,-0.56692E+00,-0.56718E+00,-0.56745E+00,-0.56771E+00,
     &-0.56797E+00,-0.56823E+00,-0.56849E+00,-0.56875E+00,-0.56900E+00,
     &-0.56926E+00,-0.56952E+00,-0.56977E+00,-0.57003E+00,-0.57029E+00,
     &-0.57054E+00,-0.57080E+00,-0.57105E+00,-0.57130E+00,-0.57156E+00,
     &-0.57181E+00,-0.57206E+00,-0.57231E+00,-0.57256E+00,-0.57281E+00,
     &-0.57306E+00,-0.57331E+00,-0.57356E+00,-0.57381E+00,-0.57406E+00,
     &-0.57430E+00,-0.57455E+00,-0.57480E+00,-0.57504E+00,-0.57529E+00,
     &-0.57553E+00,-0.57578E+00,-0.57602E+00,-0.57626E+00,-0.57651E+00,
     &-0.57675E+00,-0.57699E+00,-0.57723E+00,-0.57747E+00,-0.57771E+00,
     &-0.57795E+00,-0.57819E+00,-0.57843E+00,-0.57867E+00,-0.57891E+00,
     &-0.57915E+00,-0.57939E+00,-0.57962E+00,-0.57986E+00,-0.58009E+00/

      DATA (BNC03M(I),I=501,600)/
     &-0.58033E+00,-0.58056E+00,-0.58080E+00,-0.58103E+00,-0.58127E+00,
     &-0.58150E+00,-0.58173E+00,-0.58196E+00,-0.58220E+00,-0.58243E+00,
     &-0.58266E+00,-0.58289E+00,-0.58312E+00,-0.58335E+00,-0.58358E+00,
     &-0.58381E+00,-0.58404E+00,-0.58426E+00,-0.58449E+00,-0.58472E+00,
     &-0.58494E+00,-0.58517E+00,-0.58540E+00,-0.58562E+00,-0.58585E+00,
     &-0.58607E+00,-0.58630E+00,-0.58652E+00,-0.58674E+00,-0.58697E+00,
     &-0.58719E+00,-0.58741E+00,-0.58763E+00,-0.58785E+00,-0.58807E+00,
     &-0.58829E+00,-0.58851E+00,-0.58873E+00,-0.58895E+00,-0.58917E+00,
     &-0.58939E+00,-0.58961E+00,-0.58983E+00,-0.59004E+00,-0.59026E+00,
     &-0.59048E+00,-0.59069E+00,-0.59091E+00,-0.59113E+00,-0.59134E+00,
     &-0.59155E+00,-0.59177E+00,-0.59198E+00,-0.59220E+00,-0.59241E+00,
     &-0.59262E+00,-0.59283E+00,-0.59305E+00,-0.59326E+00,-0.59347E+00,
     &-0.59368E+00,-0.59389E+00,-0.59410E+00,-0.59431E+00,-0.59452E+00,
     &-0.59473E+00,-0.59494E+00,-0.59515E+00,-0.59535E+00,-0.59556E+00,
     &-0.59577E+00,-0.59598E+00,-0.59618E+00,-0.59639E+00,-0.59659E+00,
     &-0.59680E+00,-0.59701E+00,-0.59721E+00,-0.59741E+00,-0.59762E+00,
     &-0.59782E+00,-0.59803E+00,-0.59823E+00,-0.59843E+00,-0.59863E+00,
     &-0.59884E+00,-0.59904E+00,-0.59924E+00,-0.59944E+00,-0.59964E+00,
     &-0.59984E+00,-0.60004E+00,-0.60024E+00,-0.60044E+00,-0.60064E+00,
     &-0.60084E+00,-0.60104E+00,-0.60123E+00,-0.60143E+00,-0.60163E+00/

      DATA (BNC03M(I),I=601,700)/
     &-0.60374E+00,-0.60566E+00,-0.60755E+00,-0.60942E+00,-0.61125E+00,
     &-0.61305E+00,-0.61483E+00,-0.61658E+00,-0.61831E+00,-0.62001E+00,
     &-0.62169E+00,-0.62334E+00,-0.62497E+00,-0.62658E+00,-0.62816E+00,
     &-0.62973E+00,-0.63127E+00,-0.63279E+00,-0.63429E+00,-0.63578E+00,
     &-0.63724E+00,-0.63869E+00,-0.64012E+00,-0.64153E+00,-0.64292E+00,
     &-0.64430E+00,-0.64566E+00,-0.64700E+00,-0.64833E+00,-0.64964E+00,
     &-0.65094E+00,-0.65222E+00,-0.65349E+00,-0.65475E+00,-0.65599E+00,
     &-0.65721E+00,-0.65843E+00,-0.65963E+00,-0.66082E+00,-0.66199E+00,
     &-0.66315E+00,-0.66431E+00,-0.66544E+00,-0.66657E+00,-0.66769E+00,
     &-0.66879E+00,-0.66989E+00,-0.67097E+00,-0.67204E+00,-0.67310E+00,
     &-0.67415E+00,-0.67520E+00,-0.67623E+00,-0.67725E+00,-0.67826E+00,
     &-0.67927E+00,-0.68026E+00,-0.68124E+00,-0.68222E+00,-0.68319E+00,
     &-0.68415E+00,-0.68510E+00,-0.68604E+00,-0.68697E+00,-0.68790E+00,
     &-0.68881E+00,-0.68972E+00,-0.69062E+00,-0.69152E+00,-0.69241E+00,
     &-0.69329E+00,-0.69416E+00,-0.69502E+00,-0.69588E+00,-0.69673E+00,
     &-0.69758E+00,-0.69841E+00,-0.69924E+00,-0.70007E+00,-0.70089E+00,
     &-0.70170E+00,-0.70250E+00,-0.70330E+00,-0.70410E+00,-0.70488E+00,
     &-0.70566E+00,-0.70644E+00,-0.70721E+00,-0.70797E+00,-0.70873E+00,
     &-0.70948E+00,-0.71023E+00,-0.71097E+00,-0.71171E+00,-0.71244E+00,
     &-0.71317E+00,-0.71389E+00,-0.71460E+00,-0.71531E+00,-0.71602E+00/

      DATA (BNC03M(I),I=701,741)/
     &-0.71672E+00,-0.71742E+00,-0.71811E+00,-0.71880E+00,-0.71948E+00,
     &-0.72016E+00,-0.72083E+00,-0.72150E+00,-0.72216E+00,-0.72282E+00,
     &-0.72348E+00,-0.72413E+00,-0.72478E+00,-0.72542E+00,-0.72606E+00,
     &-0.72670E+00,-0.72733E+00,-0.72795E+00,-0.72858E+00,-0.72920E+00,
     &-0.72981E+00,-0.73042E+00,-0.73103E+00,-0.73163E+00,-0.73223E+00,
     &-0.73283E+00,-0.73342E+00,-0.73401E+00,-0.73460E+00,-0.73518E+00,
     &-0.73576E+00,-0.73634E+00,-0.73691E+00,-0.73748E+00,-0.73805E+00,
     &-0.73861E+00,-0.73917E+00,-0.73972E+00,-0.74028E+00,-0.74083E+00,
     &-0.74137E+00
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.93137E-01,-0.20251E+00,-0.25708E+00,-0.29562E+00,-0.32582E+00,
     &-0.35081E+00,-0.37218E+00,-0.39089E+00,-0.40754E+00,-0.42256E+00,
     &-0.43624E+00,-0.44882E+00,-0.46045E+00,-0.47127E+00,-0.48140E+00,
     &-0.49091E+00,-0.49988E+00,-0.50837E+00,-0.51643E+00,-0.52409E+00,
     &-0.53140E+00,-0.53839E+00,-0.54509E+00,-0.55151E+00,-0.55769E+00,
     &-0.56363E+00,-0.56936E+00,-0.57489E+00,-0.58023E+00,-0.58540E+00,
     &-0.59041E+00,-0.59526E+00,-0.59997E+00,-0.60454E+00,-0.60898E+00,
     &-0.61330E+00,-0.61751E+00,-0.62160E+00,-0.62559E+00,-0.62949E+00,
     &-0.63328E+00,-0.63699E+00,-0.64061E+00,-0.64415E+00,-0.64761E+00,
     &-0.65099E+00,-0.65430E+00,-0.65754E+00,-0.66072E+00,-0.66383E+00,
     &-0.66687E+00,-0.66986E+00,-0.67279E+00,-0.67567E+00,-0.67849E+00,
     &-0.68126E+00,-0.68398E+00,-0.68666E+00,-0.68928E+00,-0.69187E+00,
     &-0.69441E+00,-0.69691E+00,-0.69937E+00,-0.70179E+00,-0.70418E+00,
     &-0.70653E+00,-0.70884E+00,-0.71112E+00,-0.71337E+00,-0.71558E+00,
     &-0.71777E+00,-0.71993E+00,-0.72206E+00,-0.72416E+00,-0.72624E+00,
     &-0.72829E+00,-0.73031E+00,-0.73231E+00,-0.73429E+00,-0.73625E+00,
     &-0.73818E+00,-0.74009E+00,-0.74199E+00,-0.74386E+00,-0.74571E+00,
     &-0.74755E+00,-0.74936E+00,-0.75116E+00,-0.75294E+00,-0.75471E+00,
     &-0.75646E+00,-0.75819E+00,-0.75990E+00,-0.76161E+00,-0.76329E+00,
     &-0.76497E+00,-0.76662E+00,-0.76827E+00,-0.76990E+00,-0.77152E+00/

      DATA (BNC04M(I),I=101,200)/
     &-0.77312E+00,-0.77471E+00,-0.77629E+00,-0.77786E+00,-0.77941E+00,
     &-0.78095E+00,-0.78248E+00,-0.78400E+00,-0.78550E+00,-0.78700E+00,
     &-0.78848E+00,-0.78996E+00,-0.79142E+00,-0.79287E+00,-0.79431E+00,
     &-0.79574E+00,-0.79716E+00,-0.79856E+00,-0.79996E+00,-0.80135E+00,
     &-0.80273E+00,-0.80410E+00,-0.80546E+00,-0.80681E+00,-0.80815E+00,
     &-0.80948E+00,-0.81080E+00,-0.81211E+00,-0.81341E+00,-0.81470E+00,
     &-0.81599E+00,-0.81726E+00,-0.81853E+00,-0.81979E+00,-0.82104E+00,
     &-0.82228E+00,-0.82352E+00,-0.82474E+00,-0.82596E+00,-0.82717E+00,
     &-0.82837E+00,-0.82956E+00,-0.83075E+00,-0.83193E+00,-0.83310E+00,
     &-0.83426E+00,-0.83542E+00,-0.83657E+00,-0.83771E+00,-0.83885E+00,
     &-0.83998E+00,-0.84110E+00,-0.84221E+00,-0.84332E+00,-0.84442E+00,
     &-0.84552E+00,-0.84661E+00,-0.84769E+00,-0.84876E+00,-0.84983E+00,
     &-0.85090E+00,-0.85195E+00,-0.85301E+00,-0.85405E+00,-0.85509E+00,
     &-0.85612E+00,-0.85715E+00,-0.85818E+00,-0.85919E+00,-0.86020E+00,
     &-0.86121E+00,-0.86221E+00,-0.86320E+00,-0.86419E+00,-0.86518E+00,
     &-0.86616E+00,-0.86713E+00,-0.86810E+00,-0.86907E+00,-0.87002E+00,
     &-0.87098E+00,-0.87193E+00,-0.87287E+00,-0.87381E+00,-0.87475E+00,
     &-0.87568E+00,-0.87660E+00,-0.87752E+00,-0.87844E+00,-0.87935E+00,
     &-0.88026E+00,-0.88116E+00,-0.88206E+00,-0.88296E+00,-0.88385E+00,
     &-0.88473E+00,-0.88562E+00,-0.88649E+00,-0.88737E+00,-0.88824E+00/

      DATA (BNC04M(I),I=201,300)/
     &-0.88910E+00,-0.88997E+00,-0.89082E+00,-0.89168E+00,-0.89253E+00,
     &-0.89337E+00,-0.89422E+00,-0.89505E+00,-0.89589E+00,-0.89672E+00,
     &-0.89755E+00,-0.89837E+00,-0.89919E+00,-0.90001E+00,-0.90082E+00,
     &-0.90163E+00,-0.90244E+00,-0.90324E+00,-0.90404E+00,-0.90483E+00,
     &-0.90563E+00,-0.90641E+00,-0.90720E+00,-0.90798E+00,-0.90876E+00,
     &-0.90954E+00,-0.91031E+00,-0.91108E+00,-0.91185E+00,-0.91261E+00,
     &-0.91337E+00,-0.91413E+00,-0.91488E+00,-0.91563E+00,-0.91638E+00,
     &-0.91713E+00,-0.91787E+00,-0.91861E+00,-0.91934E+00,-0.92008E+00,
     &-0.92081E+00,-0.92153E+00,-0.92226E+00,-0.92298E+00,-0.92370E+00,
     &-0.92442E+00,-0.92513E+00,-0.92584E+00,-0.92655E+00,-0.92726E+00,
     &-0.92796E+00,-0.92866E+00,-0.92936E+00,-0.93005E+00,-0.93075E+00,
     &-0.93144E+00,-0.93212E+00,-0.93281E+00,-0.93349E+00,-0.93417E+00,
     &-0.93485E+00,-0.93553E+00,-0.93620E+00,-0.93687E+00,-0.93754E+00,
     &-0.93820E+00,-0.93887E+00,-0.93953E+00,-0.94019E+00,-0.94084E+00,
     &-0.94150E+00,-0.94215E+00,-0.94280E+00,-0.94345E+00,-0.94409E+00,
     &-0.94473E+00,-0.94537E+00,-0.94601E+00,-0.94665E+00,-0.94728E+00,
     &-0.94792E+00,-0.94855E+00,-0.94917E+00,-0.94980E+00,-0.95042E+00,
     &-0.95104E+00,-0.95166E+00,-0.95228E+00,-0.95290E+00,-0.95351E+00,
     &-0.95412E+00,-0.95473E+00,-0.95534E+00,-0.95594E+00,-0.95655E+00,
     &-0.95715E+00,-0.95775E+00,-0.95835E+00,-0.95894E+00,-0.95954E+00/

      DATA (BNC04M(I),I=301,400)/
     &-0.96013E+00,-0.96072E+00,-0.96131E+00,-0.96189E+00,-0.96248E+00,
     &-0.96306E+00,-0.96364E+00,-0.96422E+00,-0.96480E+00,-0.96538E+00,
     &-0.96595E+00,-0.96652E+00,-0.96709E+00,-0.96766E+00,-0.96823E+00,
     &-0.96879E+00,-0.96936E+00,-0.96992E+00,-0.97048E+00,-0.97104E+00,
     &-0.97159E+00,-0.97215E+00,-0.97270E+00,-0.97325E+00,-0.97380E+00,
     &-0.97435E+00,-0.97490E+00,-0.97545E+00,-0.97599E+00,-0.97653E+00,
     &-0.97707E+00,-0.97761E+00,-0.97815E+00,-0.97869E+00,-0.97922E+00,
     &-0.97975E+00,-0.98028E+00,-0.98081E+00,-0.98134E+00,-0.98187E+00,
     &-0.98240E+00,-0.98292E+00,-0.98344E+00,-0.98396E+00,-0.98448E+00,
     &-0.98500E+00,-0.98552E+00,-0.98603E+00,-0.98655E+00,-0.98706E+00,
     &-0.98757E+00,-0.98808E+00,-0.98859E+00,-0.98909E+00,-0.98960E+00,
     &-0.99010E+00,-0.99061E+00,-0.99111E+00,-0.99161E+00,-0.99211E+00,
     &-0.99261E+00,-0.99310E+00,-0.99360E+00,-0.99409E+00,-0.99458E+00,
     &-0.99507E+00,-0.99556E+00,-0.99605E+00,-0.99654E+00,-0.99702E+00,
     &-0.99751E+00,-0.99799E+00,-0.99847E+00,-0.99895E+00,-0.99943E+00,
     &-0.99991E+00,-0.10004E+01,-0.10009E+01,-0.10013E+01,-0.10018E+01,
     &-0.10023E+01,-0.10028E+01,-0.10032E+01,-0.10037E+01,-0.10042E+01,
     &-0.10046E+01,-0.10051E+01,-0.10056E+01,-0.10060E+01,-0.10065E+01,
     &-0.10069E+01,-0.10074E+01,-0.10079E+01,-0.10083E+01,-0.10088E+01,
     &-0.10092E+01,-0.10097E+01,-0.10101E+01,-0.10106E+01,-0.10110E+01/

      DATA (BNC04M(I),I=401,500)/
     &-0.10115E+01,-0.10119E+01,-0.10124E+01,-0.10128E+01,-0.10133E+01,
     &-0.10137E+01,-0.10142E+01,-0.10146E+01,-0.10150E+01,-0.10155E+01,
     &-0.10159E+01,-0.10164E+01,-0.10168E+01,-0.10172E+01,-0.10177E+01,
     &-0.10181E+01,-0.10185E+01,-0.10190E+01,-0.10194E+01,-0.10198E+01,
     &-0.10202E+01,-0.10207E+01,-0.10211E+01,-0.10215E+01,-0.10219E+01,
     &-0.10224E+01,-0.10228E+01,-0.10232E+01,-0.10236E+01,-0.10241E+01,
     &-0.10245E+01,-0.10249E+01,-0.10253E+01,-0.10257E+01,-0.10261E+01,
     &-0.10266E+01,-0.10270E+01,-0.10274E+01,-0.10278E+01,-0.10282E+01,
     &-0.10286E+01,-0.10290E+01,-0.10294E+01,-0.10298E+01,-0.10302E+01,
     &-0.10306E+01,-0.10310E+01,-0.10315E+01,-0.10319E+01,-0.10323E+01,
     &-0.10327E+01,-0.10331E+01,-0.10335E+01,-0.10339E+01,-0.10342E+01,
     &-0.10346E+01,-0.10350E+01,-0.10354E+01,-0.10358E+01,-0.10362E+01,
     &-0.10366E+01,-0.10370E+01,-0.10374E+01,-0.10378E+01,-0.10382E+01,
     &-0.10386E+01,-0.10389E+01,-0.10393E+01,-0.10397E+01,-0.10401E+01,
     &-0.10405E+01,-0.10409E+01,-0.10413E+01,-0.10416E+01,-0.10420E+01,
     &-0.10424E+01,-0.10428E+01,-0.10431E+01,-0.10435E+01,-0.10439E+01,
     &-0.10443E+01,-0.10447E+01,-0.10450E+01,-0.10454E+01,-0.10458E+01,
     &-0.10461E+01,-0.10465E+01,-0.10469E+01,-0.10473E+01,-0.10476E+01,
     &-0.10480E+01,-0.10484E+01,-0.10487E+01,-0.10491E+01,-0.10495E+01,
     &-0.10498E+01,-0.10502E+01,-0.10506E+01,-0.10509E+01,-0.10513E+01/

      DATA (BNC04M(I),I=501,600)/
     &-0.10516E+01,-0.10520E+01,-0.10524E+01,-0.10527E+01,-0.10531E+01,
     &-0.10534E+01,-0.10538E+01,-0.10541E+01,-0.10545E+01,-0.10549E+01,
     &-0.10552E+01,-0.10556E+01,-0.10559E+01,-0.10563E+01,-0.10566E+01,
     &-0.10570E+01,-0.10573E+01,-0.10577E+01,-0.10580E+01,-0.10584E+01,
     &-0.10587E+01,-0.10591E+01,-0.10594E+01,-0.10597E+01,-0.10601E+01,
     &-0.10604E+01,-0.10608E+01,-0.10611E+01,-0.10615E+01,-0.10618E+01,
     &-0.10621E+01,-0.10625E+01,-0.10628E+01,-0.10632E+01,-0.10635E+01,
     &-0.10638E+01,-0.10642E+01,-0.10645E+01,-0.10648E+01,-0.10652E+01,
     &-0.10655E+01,-0.10659E+01,-0.10662E+01,-0.10665E+01,-0.10668E+01,
     &-0.10672E+01,-0.10675E+01,-0.10678E+01,-0.10682E+01,-0.10685E+01,
     &-0.10688E+01,-0.10692E+01,-0.10695E+01,-0.10698E+01,-0.10701E+01,
     &-0.10705E+01,-0.10708E+01,-0.10711E+01,-0.10714E+01,-0.10718E+01,
     &-0.10721E+01,-0.10724E+01,-0.10727E+01,-0.10730E+01,-0.10734E+01,
     &-0.10737E+01,-0.10740E+01,-0.10743E+01,-0.10746E+01,-0.10750E+01,
     &-0.10753E+01,-0.10756E+01,-0.10759E+01,-0.10762E+01,-0.10765E+01,
     &-0.10768E+01,-0.10772E+01,-0.10775E+01,-0.10778E+01,-0.10781E+01,
     &-0.10784E+01,-0.10787E+01,-0.10790E+01,-0.10793E+01,-0.10796E+01,
     &-0.10800E+01,-0.10803E+01,-0.10806E+01,-0.10809E+01,-0.10812E+01,
     &-0.10815E+01,-0.10818E+01,-0.10821E+01,-0.10824E+01,-0.10827E+01,
     &-0.10830E+01,-0.10833E+01,-0.10836E+01,-0.10839E+01,-0.10842E+01/

      DATA (BNC04M(I),I=601,700)/
     &-0.10874E+01,-0.10904E+01,-0.10933E+01,-0.10961E+01,-0.10989E+01,
     &-0.11017E+01,-0.11044E+01,-0.11070E+01,-0.11097E+01,-0.11123E+01,
     &-0.11148E+01,-0.11173E+01,-0.11198E+01,-0.11223E+01,-0.11247E+01,
     &-0.11271E+01,-0.11294E+01,-0.11317E+01,-0.11340E+01,-0.11363E+01,
     &-0.11385E+01,-0.11407E+01,-0.11429E+01,-0.11450E+01,-0.11471E+01,
     &-0.11492E+01,-0.11513E+01,-0.11534E+01,-0.11554E+01,-0.11574E+01,
     &-0.11593E+01,-0.11613E+01,-0.11632E+01,-0.11651E+01,-0.11670E+01,
     &-0.11689E+01,-0.11707E+01,-0.11726E+01,-0.11744E+01,-0.11762E+01,
     &-0.11779E+01,-0.11797E+01,-0.11814E+01,-0.11831E+01,-0.11848E+01,
     &-0.11865E+01,-0.11882E+01,-0.11898E+01,-0.11915E+01,-0.11931E+01,
     &-0.11947E+01,-0.11963E+01,-0.11978E+01,-0.11994E+01,-0.12009E+01,
     &-0.12025E+01,-0.12040E+01,-0.12055E+01,-0.12070E+01,-0.12084E+01,
     &-0.12099E+01,-0.12114E+01,-0.12128E+01,-0.12142E+01,-0.12156E+01,
     &-0.12170E+01,-0.12184E+01,-0.12198E+01,-0.12212E+01,-0.12225E+01,
     &-0.12238E+01,-0.12252E+01,-0.12265E+01,-0.12278E+01,-0.12291E+01,
     &-0.12304E+01,-0.12317E+01,-0.12329E+01,-0.12342E+01,-0.12355E+01,
     &-0.12367E+01,-0.12379E+01,-0.12391E+01,-0.12404E+01,-0.12416E+01,
     &-0.12428E+01,-0.12439E+01,-0.12451E+01,-0.12463E+01,-0.12474E+01,
     &-0.12486E+01,-0.12497E+01,-0.12509E+01,-0.12520E+01,-0.12531E+01,
     &-0.12542E+01,-0.12553E+01,-0.12564E+01,-0.12575E+01,-0.12586E+01/

      DATA (BNC04M(I),I=701,741)/
     &-0.12597E+01,-0.12607E+01,-0.12618E+01,-0.12629E+01,-0.12639E+01,
     &-0.12649E+01,-0.12660E+01,-0.12670E+01,-0.12680E+01,-0.12690E+01,
     &-0.12700E+01,-0.12710E+01,-0.12720E+01,-0.12730E+01,-0.12740E+01,
     &-0.12750E+01,-0.12760E+01,-0.12769E+01,-0.12779E+01,-0.12788E+01,
     &-0.12798E+01,-0.12807E+01,-0.12816E+01,-0.12826E+01,-0.12835E+01,
     &-0.12844E+01,-0.12853E+01,-0.12862E+01,-0.12871E+01,-0.12880E+01,
     &-0.12889E+01,-0.12898E+01,-0.12907E+01,-0.12916E+01,-0.12924E+01,
     &-0.12933E+01,-0.12942E+01,-0.12950E+01,-0.12959E+01,-0.12967E+01,
     &-0.12976E+01
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.47107E-01,-0.10421E+00,-0.13373E+00,-0.15513E+00,-0.17228E+00,
     &-0.18677E+00,-0.19941E+00,-0.21068E+00,-0.22088E+00,-0.23023E+00,
     &-0.23889E+00,-0.24697E+00,-0.25455E+00,-0.26171E+00,-0.26849E+00,
     &-0.27495E+00,-0.28111E+00,-0.28702E+00,-0.29269E+00,-0.29814E+00,
     &-0.30340E+00,-0.30849E+00,-0.31340E+00,-0.31817E+00,-0.32279E+00,
     &-0.32728E+00,-0.33165E+00,-0.33590E+00,-0.34004E+00,-0.34408E+00,
     &-0.34802E+00,-0.35186E+00,-0.35562E+00,-0.35930E+00,-0.36289E+00,
     &-0.36641E+00,-0.36986E+00,-0.37323E+00,-0.37654E+00,-0.37978E+00,
     &-0.38296E+00,-0.38609E+00,-0.38915E+00,-0.39216E+00,-0.39511E+00,
     &-0.39801E+00,-0.40086E+00,-0.40367E+00,-0.40642E+00,-0.40913E+00,
     &-0.41180E+00,-0.41443E+00,-0.41701E+00,-0.41956E+00,-0.42207E+00,
     &-0.42454E+00,-0.42697E+00,-0.42937E+00,-0.43174E+00,-0.43408E+00,
     &-0.43639E+00,-0.43866E+00,-0.44091E+00,-0.44313E+00,-0.44533E+00,
     &-0.44750E+00,-0.44965E+00,-0.45177E+00,-0.45387E+00,-0.45595E+00,
     &-0.45801E+00,-0.46005E+00,-0.46207E+00,-0.46408E+00,-0.46606E+00,
     &-0.46803E+00,-0.46999E+00,-0.47193E+00,-0.47386E+00,-0.47577E+00,
     &-0.47767E+00,-0.47956E+00,-0.48143E+00,-0.48330E+00,-0.48515E+00,
     &-0.48700E+00,-0.48883E+00,-0.49065E+00,-0.49247E+00,-0.49427E+00,
     &-0.49607E+00,-0.49786E+00,-0.49964E+00,-0.50141E+00,-0.50317E+00,
     &-0.50493E+00,-0.50668E+00,-0.50842E+00,-0.51015E+00,-0.51188E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.51359E+00,-0.51530E+00,-0.51701E+00,-0.51870E+00,-0.52039E+00,
     &-0.52208E+00,-0.52375E+00,-0.52542E+00,-0.52708E+00,-0.52873E+00,
     &-0.53038E+00,-0.53201E+00,-0.53364E+00,-0.53527E+00,-0.53688E+00,
     &-0.53849E+00,-0.54009E+00,-0.54168E+00,-0.54327E+00,-0.54485E+00,
     &-0.54642E+00,-0.54798E+00,-0.54953E+00,-0.55108E+00,-0.55262E+00,
     &-0.55415E+00,-0.55568E+00,-0.55719E+00,-0.55870E+00,-0.56020E+00,
     &-0.56170E+00,-0.56318E+00,-0.56466E+00,-0.56613E+00,-0.56760E+00,
     &-0.56905E+00,-0.57050E+00,-0.57195E+00,-0.57338E+00,-0.57481E+00,
     &-0.57623E+00,-0.57764E+00,-0.57905E+00,-0.58044E+00,-0.58184E+00,
     &-0.58322E+00,-0.58460E+00,-0.58597E+00,-0.58734E+00,-0.58869E+00,
     &-0.59004E+00,-0.59139E+00,-0.59273E+00,-0.59406E+00,-0.59539E+00,
     &-0.59670E+00,-0.59802E+00,-0.59932E+00,-0.60062E+00,-0.60192E+00,
     &-0.60321E+00,-0.60449E+00,-0.60577E+00,-0.60704E+00,-0.60830E+00,
     &-0.60956E+00,-0.61081E+00,-0.61206E+00,-0.61330E+00,-0.61454E+00,
     &-0.61577E+00,-0.61699E+00,-0.61821E+00,-0.61943E+00,-0.62064E+00,
     &-0.62184E+00,-0.62304E+00,-0.62423E+00,-0.62542E+00,-0.62660E+00,
     &-0.62778E+00,-0.62895E+00,-0.63012E+00,-0.63128E+00,-0.63244E+00,
     &-0.63359E+00,-0.63474E+00,-0.63588E+00,-0.63702E+00,-0.63815E+00,
     &-0.63928E+00,-0.64040E+00,-0.64152E+00,-0.64264E+00,-0.64375E+00,
     &-0.64485E+00,-0.64595E+00,-0.64705E+00,-0.64814E+00,-0.64923E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.65031E+00,-0.65139E+00,-0.65246E+00,-0.65353E+00,-0.65460E+00,
     &-0.65566E+00,-0.65672E+00,-0.65777E+00,-0.65882E+00,-0.65986E+00,
     &-0.66090E+00,-0.66194E+00,-0.66297E+00,-0.66400E+00,-0.66502E+00,
     &-0.66605E+00,-0.66706E+00,-0.66807E+00,-0.66908E+00,-0.67009E+00,
     &-0.67109E+00,-0.67209E+00,-0.67308E+00,-0.67407E+00,-0.67506E+00,
     &-0.67604E+00,-0.67702E+00,-0.67799E+00,-0.67896E+00,-0.67993E+00,
     &-0.68090E+00,-0.68186E+00,-0.68281E+00,-0.68377E+00,-0.68472E+00,
     &-0.68566E+00,-0.68661E+00,-0.68755E+00,-0.68848E+00,-0.68942E+00,
     &-0.69035E+00,-0.69127E+00,-0.69220E+00,-0.69312E+00,-0.69403E+00,
     &-0.69495E+00,-0.69586E+00,-0.69676E+00,-0.69767E+00,-0.69857E+00,
     &-0.69946E+00,-0.70036E+00,-0.70125E+00,-0.70214E+00,-0.70302E+00,
     &-0.70390E+00,-0.70478E+00,-0.70566E+00,-0.70653E+00,-0.70740E+00,
     &-0.70827E+00,-0.70913E+00,-0.70999E+00,-0.71085E+00,-0.71170E+00,
     &-0.71256E+00,-0.71340E+00,-0.71425E+00,-0.71509E+00,-0.71593E+00,
     &-0.71677E+00,-0.71761E+00,-0.71844E+00,-0.71927E+00,-0.72010E+00,
     &-0.72092E+00,-0.72174E+00,-0.72256E+00,-0.72337E+00,-0.72419E+00,
     &-0.72500E+00,-0.72581E+00,-0.72661E+00,-0.72741E+00,-0.72821E+00,
     &-0.72901E+00,-0.72981E+00,-0.73060E+00,-0.73139E+00,-0.73218E+00,
     &-0.73296E+00,-0.73374E+00,-0.73452E+00,-0.73530E+00,-0.73607E+00,
     &-0.73685E+00,-0.73762E+00,-0.73838E+00,-0.73915E+00,-0.73991E+00/

      DATA (BNC05M(I),I=301,400)/
     &-0.74067E+00,-0.74143E+00,-0.74218E+00,-0.74294E+00,-0.74369E+00,
     &-0.74444E+00,-0.74518E+00,-0.74593E+00,-0.74667E+00,-0.74741E+00,
     &-0.74814E+00,-0.74888E+00,-0.74961E+00,-0.75034E+00,-0.75107E+00,
     &-0.75179E+00,-0.75252E+00,-0.75324E+00,-0.75396E+00,-0.75468E+00,
     &-0.75539E+00,-0.75610E+00,-0.75681E+00,-0.75752E+00,-0.75823E+00,
     &-0.75893E+00,-0.75963E+00,-0.76033E+00,-0.76103E+00,-0.76173E+00,
     &-0.76242E+00,-0.76311E+00,-0.76380E+00,-0.76449E+00,-0.76518E+00,
     &-0.76586E+00,-0.76654E+00,-0.76722E+00,-0.76790E+00,-0.76857E+00,
     &-0.76925E+00,-0.76992E+00,-0.77059E+00,-0.77126E+00,-0.77192E+00,
     &-0.77259E+00,-0.77325E+00,-0.77391E+00,-0.77457E+00,-0.77523E+00,
     &-0.77588E+00,-0.77653E+00,-0.77718E+00,-0.77783E+00,-0.77848E+00,
     &-0.77913E+00,-0.77977E+00,-0.78041E+00,-0.78105E+00,-0.78169E+00,
     &-0.78233E+00,-0.78296E+00,-0.78359E+00,-0.78422E+00,-0.78485E+00,
     &-0.78548E+00,-0.78611E+00,-0.78673E+00,-0.78735E+00,-0.78797E+00,
     &-0.78859E+00,-0.78921E+00,-0.78983E+00,-0.79044E+00,-0.79105E+00,
     &-0.79166E+00,-0.79227E+00,-0.79288E+00,-0.79348E+00,-0.79409E+00,
     &-0.79469E+00,-0.79529E+00,-0.79589E+00,-0.79649E+00,-0.79708E+00,
     &-0.79768E+00,-0.79827E+00,-0.79886E+00,-0.79945E+00,-0.80004E+00,
     &-0.80062E+00,-0.80121E+00,-0.80179E+00,-0.80237E+00,-0.80295E+00,
     &-0.80353E+00,-0.80411E+00,-0.80468E+00,-0.80526E+00,-0.80583E+00/

      DATA (BNC05M(I),I=401,500)/
     &-0.80640E+00,-0.80697E+00,-0.80754E+00,-0.80811E+00,-0.80867E+00,
     &-0.80923E+00,-0.80980E+00,-0.81036E+00,-0.81092E+00,-0.81147E+00,
     &-0.81203E+00,-0.81259E+00,-0.81314E+00,-0.81369E+00,-0.81424E+00,
     &-0.81479E+00,-0.81534E+00,-0.81589E+00,-0.81643E+00,-0.81697E+00,
     &-0.81752E+00,-0.81806E+00,-0.81860E+00,-0.81913E+00,-0.81967E+00,
     &-0.82021E+00,-0.82074E+00,-0.82127E+00,-0.82180E+00,-0.82233E+00,
     &-0.82286E+00,-0.82339E+00,-0.82392E+00,-0.82444E+00,-0.82496E+00,
     &-0.82549E+00,-0.82601E+00,-0.82653E+00,-0.82705E+00,-0.82756E+00,
     &-0.82808E+00,-0.82859E+00,-0.82911E+00,-0.82962E+00,-0.83013E+00,
     &-0.83064E+00,-0.83115E+00,-0.83165E+00,-0.83216E+00,-0.83266E+00,
     &-0.83317E+00,-0.83367E+00,-0.83417E+00,-0.83467E+00,-0.83517E+00,
     &-0.83566E+00,-0.83616E+00,-0.83665E+00,-0.83715E+00,-0.83764E+00,
     &-0.83813E+00,-0.83862E+00,-0.83911E+00,-0.83960E+00,-0.84008E+00,
     &-0.84057E+00,-0.84105E+00,-0.84154E+00,-0.84202E+00,-0.84250E+00,
     &-0.84298E+00,-0.84346E+00,-0.84393E+00,-0.84441E+00,-0.84489E+00,
     &-0.84536E+00,-0.84583E+00,-0.84630E+00,-0.84677E+00,-0.84724E+00,
     &-0.84771E+00,-0.84818E+00,-0.84865E+00,-0.84911E+00,-0.84957E+00,
     &-0.85004E+00,-0.85050E+00,-0.85096E+00,-0.85142E+00,-0.85188E+00,
     &-0.85234E+00,-0.85279E+00,-0.85325E+00,-0.85370E+00,-0.85416E+00,
     &-0.85461E+00,-0.85506E+00,-0.85551E+00,-0.85596E+00,-0.85641E+00/

      DATA (BNC05M(I),I=501,600)/
     &-0.85686E+00,-0.85730E+00,-0.85775E+00,-0.85819E+00,-0.85863E+00,
     &-0.85908E+00,-0.85952E+00,-0.85996E+00,-0.86040E+00,-0.86084E+00,
     &-0.86127E+00,-0.86171E+00,-0.86214E+00,-0.86258E+00,-0.86301E+00,
     &-0.86344E+00,-0.86388E+00,-0.86431E+00,-0.86474E+00,-0.86516E+00,
     &-0.86559E+00,-0.86602E+00,-0.86644E+00,-0.86687E+00,-0.86729E+00,
     &-0.86772E+00,-0.86814E+00,-0.86856E+00,-0.86898E+00,-0.86940E+00,
     &-0.86982E+00,-0.87023E+00,-0.87065E+00,-0.87106E+00,-0.87148E+00,
     &-0.87189E+00,-0.87231E+00,-0.87272E+00,-0.87313E+00,-0.87354E+00,
     &-0.87395E+00,-0.87436E+00,-0.87476E+00,-0.87517E+00,-0.87558E+00,
     &-0.87598E+00,-0.87638E+00,-0.87679E+00,-0.87719E+00,-0.87759E+00,
     &-0.87799E+00,-0.87839E+00,-0.87879E+00,-0.87919E+00,-0.87958E+00,
     &-0.87998E+00,-0.88038E+00,-0.88077E+00,-0.88116E+00,-0.88156E+00,
     &-0.88195E+00,-0.88234E+00,-0.88273E+00,-0.88312E+00,-0.88351E+00,
     &-0.88390E+00,-0.88428E+00,-0.88467E+00,-0.88506E+00,-0.88544E+00,
     &-0.88582E+00,-0.88621E+00,-0.88659E+00,-0.88697E+00,-0.88735E+00,
     &-0.88773E+00,-0.88811E+00,-0.88849E+00,-0.88887E+00,-0.88924E+00,
     &-0.88962E+00,-0.88999E+00,-0.89037E+00,-0.89074E+00,-0.89112E+00,
     &-0.89149E+00,-0.89186E+00,-0.89223E+00,-0.89260E+00,-0.89297E+00,
     &-0.89334E+00,-0.89370E+00,-0.89407E+00,-0.89444E+00,-0.89480E+00,
     &-0.89517E+00,-0.89553E+00,-0.89589E+00,-0.89626E+00,-0.89662E+00/

      DATA (BNC05M(I),I=601,700)/
     &-0.90048E+00,-0.90398E+00,-0.90741E+00,-0.91077E+00,-0.91407E+00,
     &-0.91731E+00,-0.92048E+00,-0.92360E+00,-0.92666E+00,-0.92966E+00,
     &-0.93261E+00,-0.93551E+00,-0.93835E+00,-0.94115E+00,-0.94389E+00,
     &-0.94659E+00,-0.94924E+00,-0.95185E+00,-0.95441E+00,-0.95693E+00,
     &-0.95941E+00,-0.96185E+00,-0.96425E+00,-0.96661E+00,-0.96893E+00,
     &-0.97121E+00,-0.97346E+00,-0.97568E+00,-0.97786E+00,-0.98000E+00,
     &-0.98212E+00,-0.98420E+00,-0.98625E+00,-0.98827E+00,-0.99026E+00,
     &-0.99222E+00,-0.99415E+00,-0.99605E+00,-0.99793E+00,-0.99978E+00,
     &-0.10016E+01,-0.10034E+01,-0.10052E+01,-0.10069E+01,-0.10087E+01,
     &-0.10104E+01,-0.10120E+01,-0.10137E+01,-0.10153E+01,-0.10169E+01,
     &-0.10185E+01,-0.10201E+01,-0.10216E+01,-0.10232E+01,-0.10247E+01,
     &-0.10262E+01,-0.10276E+01,-0.10291E+01,-0.10305E+01,-0.10319E+01,
     &-0.10333E+01,-0.10347E+01,-0.10361E+01,-0.10374E+01,-0.10388E+01,
     &-0.10401E+01,-0.10414E+01,-0.10427E+01,-0.10439E+01,-0.10452E+01,
     &-0.10464E+01,-0.10477E+01,-0.10489E+01,-0.10501E+01,-0.10513E+01,
     &-0.10524E+01,-0.10536E+01,-0.10547E+01,-0.10559E+01,-0.10570E+01,
     &-0.10581E+01,-0.10592E+01,-0.10603E+01,-0.10613E+01,-0.10624E+01,
     &-0.10635E+01,-0.10645E+01,-0.10655E+01,-0.10665E+01,-0.10675E+01,
     &-0.10685E+01,-0.10695E+01,-0.10705E+01,-0.10715E+01,-0.10724E+01,
     &-0.10734E+01,-0.10743E+01,-0.10752E+01,-0.10761E+01,-0.10770E+01/

      DATA (BNC05M(I),I=701,741)/
     &-0.10779E+01,-0.10788E+01,-0.10797E+01,-0.10806E+01,-0.10814E+01,
     &-0.10823E+01,-0.10831E+01,-0.10840E+01,-0.10848E+01,-0.10856E+01,
     &-0.10865E+01,-0.10873E+01,-0.10881E+01,-0.10889E+01,-0.10896E+01,
     &-0.10904E+01,-0.10912E+01,-0.10920E+01,-0.10927E+01,-0.10935E+01,
     &-0.10942E+01,-0.10949E+01,-0.10957E+01,-0.10964E+01,-0.10971E+01,
     &-0.10978E+01,-0.10985E+01,-0.10992E+01,-0.10999E+01,-0.11006E+01,
     &-0.11013E+01,-0.11020E+01,-0.11026E+01,-0.11033E+01,-0.11039E+01,
     &-0.11046E+01,-0.11052E+01,-0.11059E+01,-0.11065E+01,-0.11071E+01,
     &-0.11078E+01
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.45990E-01,-0.98113E-01,-0.12306E+00,-0.14011E+00,-0.15308E+00,
     &-0.16349E+00,-0.17215E+00,-0.17952E+00,-0.18589E+00,-0.19149E+00,
     &-0.19644E+00,-0.20086E+00,-0.20484E+00,-0.20843E+00,-0.21170E+00,
     &-0.21467E+00,-0.21740E+00,-0.21990E+00,-0.22221E+00,-0.22433E+00,
     &-0.22630E+00,-0.22812E+00,-0.22981E+00,-0.23138E+00,-0.23284E+00,
     &-0.23420E+00,-0.23547E+00,-0.23665E+00,-0.23775E+00,-0.23879E+00,
     &-0.23975E+00,-0.24065E+00,-0.24150E+00,-0.24229E+00,-0.24303E+00,
     &-0.24372E+00,-0.24437E+00,-0.24498E+00,-0.24555E+00,-0.24608E+00,
     &-0.24658E+00,-0.24705E+00,-0.24749E+00,-0.24790E+00,-0.24829E+00,
     &-0.24865E+00,-0.24898E+00,-0.24929E+00,-0.24959E+00,-0.24986E+00,
     &-0.25011E+00,-0.25034E+00,-0.25056E+00,-0.25076E+00,-0.25094E+00,
     &-0.25111E+00,-0.25126E+00,-0.25140E+00,-0.25152E+00,-0.25163E+00,
     &-0.25173E+00,-0.25182E+00,-0.25189E+00,-0.25195E+00,-0.25200E+00,
     &-0.25204E+00,-0.25206E+00,-0.25207E+00,-0.25208E+00,-0.25207E+00,
     &-0.25205E+00,-0.25202E+00,-0.25198E+00,-0.25192E+00,-0.25186E+00,
     &-0.25179E+00,-0.25170E+00,-0.25161E+00,-0.25150E+00,-0.25139E+00,
     &-0.25126E+00,-0.25113E+00,-0.25098E+00,-0.25083E+00,-0.25066E+00,
     &-0.25049E+00,-0.25030E+00,-0.25011E+00,-0.24990E+00,-0.24969E+00,
     &-0.24946E+00,-0.24923E+00,-0.24899E+00,-0.24874E+00,-0.24848E+00,
     &-0.24822E+00,-0.24794E+00,-0.24766E+00,-0.24737E+00,-0.24707E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.24676E+00,-0.24645E+00,-0.24613E+00,-0.24580E+00,-0.24547E+00,
     &-0.24513E+00,-0.24479E+00,-0.24443E+00,-0.24408E+00,-0.24371E+00,
     &-0.24335E+00,-0.24297E+00,-0.24259E+00,-0.24221E+00,-0.24183E+00,
     &-0.24143E+00,-0.24104E+00,-0.24064E+00,-0.24024E+00,-0.23983E+00,
     &-0.23942E+00,-0.23901E+00,-0.23859E+00,-0.23817E+00,-0.23775E+00,
     &-0.23733E+00,-0.23690E+00,-0.23647E+00,-0.23604E+00,-0.23560E+00,
     &-0.23517E+00,-0.23473E+00,-0.23429E+00,-0.23385E+00,-0.23340E+00,
     &-0.23296E+00,-0.23251E+00,-0.23206E+00,-0.23161E+00,-0.23116E+00,
     &-0.23071E+00,-0.23025E+00,-0.22980E+00,-0.22934E+00,-0.22888E+00,
     &-0.22842E+00,-0.22796E+00,-0.22750E+00,-0.22704E+00,-0.22658E+00,
     &-0.22611E+00,-0.22565E+00,-0.22518E+00,-0.22472E+00,-0.22425E+00,
     &-0.22378E+00,-0.22331E+00,-0.22284E+00,-0.22237E+00,-0.22190E+00,
     &-0.22143E+00,-0.22096E+00,-0.22049E+00,-0.22001E+00,-0.21954E+00,
     &-0.21906E+00,-0.21859E+00,-0.21812E+00,-0.21764E+00,-0.21716E+00,
     &-0.21669E+00,-0.21621E+00,-0.21573E+00,-0.21526E+00,-0.21478E+00,
     &-0.21430E+00,-0.21382E+00,-0.21334E+00,-0.21286E+00,-0.21238E+00,
     &-0.21191E+00,-0.21143E+00,-0.21095E+00,-0.21047E+00,-0.20999E+00,
     &-0.20951E+00,-0.20903E+00,-0.20854E+00,-0.20806E+00,-0.20758E+00,
     &-0.20710E+00,-0.20662E+00,-0.20614E+00,-0.20566E+00,-0.20518E+00,
     &-0.20470E+00,-0.20422E+00,-0.20373E+00,-0.20325E+00,-0.20277E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.20229E+00,-0.20181E+00,-0.20133E+00,-0.20085E+00,-0.20037E+00,
     &-0.19989E+00,-0.19941E+00,-0.19892E+00,-0.19844E+00,-0.19796E+00,
     &-0.19748E+00,-0.19700E+00,-0.19652E+00,-0.19604E+00,-0.19556E+00,
     &-0.19508E+00,-0.19460E+00,-0.19412E+00,-0.19364E+00,-0.19316E+00,
     &-0.19269E+00,-0.19221E+00,-0.19173E+00,-0.19125E+00,-0.19077E+00,
     &-0.19029E+00,-0.18982E+00,-0.18934E+00,-0.18886E+00,-0.18838E+00,
     &-0.18791E+00,-0.18743E+00,-0.18695E+00,-0.18648E+00,-0.18600E+00,
     &-0.18553E+00,-0.18505E+00,-0.18458E+00,-0.18410E+00,-0.18363E+00,
     &-0.18315E+00,-0.18268E+00,-0.18221E+00,-0.18173E+00,-0.18126E+00,
     &-0.18079E+00,-0.18031E+00,-0.17984E+00,-0.17937E+00,-0.17890E+00,
     &-0.17843E+00,-0.17796E+00,-0.17749E+00,-0.17702E+00,-0.17655E+00,
     &-0.17608E+00,-0.17561E+00,-0.17514E+00,-0.17467E+00,-0.17420E+00,
     &-0.17373E+00,-0.17327E+00,-0.17280E+00,-0.17233E+00,-0.17187E+00,
     &-0.17140E+00,-0.17093E+00,-0.17047E+00,-0.17000E+00,-0.16954E+00,
     &-0.16908E+00,-0.16861E+00,-0.16815E+00,-0.16769E+00,-0.16722E+00,
     &-0.16676E+00,-0.16630E+00,-0.16584E+00,-0.16538E+00,-0.16492E+00,
     &-0.16446E+00,-0.16400E+00,-0.16354E+00,-0.16308E+00,-0.16262E+00,
     &-0.16216E+00,-0.16170E+00,-0.16124E+00,-0.16079E+00,-0.16033E+00,
     &-0.15987E+00,-0.15942E+00,-0.15896E+00,-0.15851E+00,-0.15805E+00,
     &-0.15760E+00,-0.15714E+00,-0.15669E+00,-0.15624E+00,-0.15579E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.15533E+00,-0.15488E+00,-0.15443E+00,-0.15398E+00,-0.15353E+00,
     &-0.15308E+00,-0.15263E+00,-0.15218E+00,-0.15173E+00,-0.15128E+00,
     &-0.15083E+00,-0.15039E+00,-0.14994E+00,-0.14949E+00,-0.14905E+00,
     &-0.14860E+00,-0.14816E+00,-0.14771E+00,-0.14727E+00,-0.14682E+00,
     &-0.14638E+00,-0.14594E+00,-0.14549E+00,-0.14505E+00,-0.14461E+00,
     &-0.14417E+00,-0.14373E+00,-0.14328E+00,-0.14284E+00,-0.14240E+00,
     &-0.14197E+00,-0.14153E+00,-0.14109E+00,-0.14065E+00,-0.14021E+00,
     &-0.13977E+00,-0.13934E+00,-0.13890E+00,-0.13847E+00,-0.13803E+00,
     &-0.13760E+00,-0.13716E+00,-0.13673E+00,-0.13629E+00,-0.13586E+00,
     &-0.13543E+00,-0.13499E+00,-0.13456E+00,-0.13413E+00,-0.13370E+00,
     &-0.13327E+00,-0.13284E+00,-0.13241E+00,-0.13198E+00,-0.13155E+00,
     &-0.13112E+00,-0.13069E+00,-0.13027E+00,-0.12984E+00,-0.12941E+00,
     &-0.12899E+00,-0.12856E+00,-0.12813E+00,-0.12771E+00,-0.12729E+00,
     &-0.12686E+00,-0.12644E+00,-0.12601E+00,-0.12559E+00,-0.12517E+00,
     &-0.12475E+00,-0.12433E+00,-0.12390E+00,-0.12348E+00,-0.12306E+00,
     &-0.12264E+00,-0.12222E+00,-0.12181E+00,-0.12139E+00,-0.12097E+00,
     &-0.12055E+00,-0.12013E+00,-0.11972E+00,-0.11930E+00,-0.11888E+00,
     &-0.11847E+00,-0.11805E+00,-0.11764E+00,-0.11722E+00,-0.11681E+00,
     &-0.11640E+00,-0.11598E+00,-0.11557E+00,-0.11516E+00,-0.11475E+00,
     &-0.11434E+00,-0.11393E+00,-0.11352E+00,-0.11311E+00,-0.11270E+00/

      DATA (BNC06M(I),I=401,500)/
     &-0.11229E+00,-0.11188E+00,-0.11147E+00,-0.11106E+00,-0.11066E+00,
     &-0.11025E+00,-0.10984E+00,-0.10944E+00,-0.10903E+00,-0.10862E+00,
     &-0.10822E+00,-0.10781E+00,-0.10741E+00,-0.10701E+00,-0.10660E+00,
     &-0.10620E+00,-0.10580E+00,-0.10540E+00,-0.10499E+00,-0.10459E+00,
     &-0.10419E+00,-0.10379E+00,-0.10339E+00,-0.10299E+00,-0.10259E+00,
     &-0.10220E+00,-0.10180E+00,-0.10140E+00,-0.10100E+00,-0.10060E+00,
     &-0.10021E+00,-0.99812E-01,-0.99416E-01,-0.99021E-01,-0.98626E-01,
     &-0.98231E-01,-0.97837E-01,-0.97443E-01,-0.97050E-01,-0.96657E-01,
     &-0.96264E-01,-0.95872E-01,-0.95480E-01,-0.95089E-01,-0.94698E-01,
     &-0.94308E-01,-0.93917E-01,-0.93528E-01,-0.93138E-01,-0.92749E-01,
     &-0.92361E-01,-0.91973E-01,-0.91585E-01,-0.91197E-01,-0.90810E-01,
     &-0.90424E-01,-0.90038E-01,-0.89652E-01,-0.89267E-01,-0.88882E-01,
     &-0.88497E-01,-0.88113E-01,-0.87729E-01,-0.87346E-01,-0.86963E-01,
     &-0.86580E-01,-0.86198E-01,-0.85816E-01,-0.85434E-01,-0.85053E-01,
     &-0.84673E-01,-0.84292E-01,-0.83912E-01,-0.83533E-01,-0.83154E-01,
     &-0.82775E-01,-0.82397E-01,-0.82019E-01,-0.81641E-01,-0.81264E-01,
     &-0.80887E-01,-0.80510E-01,-0.80134E-01,-0.79759E-01,-0.79383E-01,
     &-0.79008E-01,-0.78634E-01,-0.78260E-01,-0.77886E-01,-0.77512E-01,
     &-0.77139E-01,-0.76767E-01,-0.76394E-01,-0.76022E-01,-0.75651E-01,
     &-0.75280E-01,-0.74909E-01,-0.74538E-01,-0.74168E-01,-0.73799E-01/

      DATA (BNC06M(I),I=501,600)/
     &-0.73429E-01,-0.73060E-01,-0.72692E-01,-0.72324E-01,-0.71956E-01,
     &-0.71588E-01,-0.71221E-01,-0.70854E-01,-0.70488E-01,-0.70122E-01,
     &-0.69756E-01,-0.69391E-01,-0.69026E-01,-0.68661E-01,-0.68297E-01,
     &-0.67933E-01,-0.67570E-01,-0.67207E-01,-0.66844E-01,-0.66482E-01,
     &-0.66120E-01,-0.65758E-01,-0.65397E-01,-0.65036E-01,-0.64675E-01,
     &-0.64315E-01,-0.63955E-01,-0.63595E-01,-0.63236E-01,-0.62877E-01,
     &-0.62519E-01,-0.62161E-01,-0.61803E-01,-0.61446E-01,-0.61089E-01,
     &-0.60732E-01,-0.60375E-01,-0.60019E-01,-0.59664E-01,-0.59308E-01,
     &-0.58954E-01,-0.58599E-01,-0.58245E-01,-0.57891E-01,-0.57537E-01,
     &-0.57184E-01,-0.56831E-01,-0.56478E-01,-0.56126E-01,-0.55774E-01,
     &-0.55423E-01,-0.55072E-01,-0.54721E-01,-0.54370E-01,-0.54020E-01,
     &-0.53670E-01,-0.53321E-01,-0.52972E-01,-0.52623E-01,-0.52274E-01,
     &-0.51926E-01,-0.51578E-01,-0.51231E-01,-0.50884E-01,-0.50537E-01,
     &-0.50191E-01,-0.49844E-01,-0.49499E-01,-0.49153E-01,-0.48808E-01,
     &-0.48463E-01,-0.48119E-01,-0.47775E-01,-0.47431E-01,-0.47087E-01,
     &-0.46744E-01,-0.46401E-01,-0.46059E-01,-0.45717E-01,-0.45375E-01,
     &-0.45033E-01,-0.44692E-01,-0.44351E-01,-0.44011E-01,-0.43670E-01,
     &-0.43331E-01,-0.42991E-01,-0.42652E-01,-0.42313E-01,-0.41974E-01,
     &-0.41636E-01,-0.41298E-01,-0.40960E-01,-0.40623E-01,-0.40286E-01,
     &-0.39949E-01,-0.39613E-01,-0.39277E-01,-0.38941E-01,-0.38605E-01/

      DATA (BNC06M(I),I=601,700)/
     &-0.35001E-01,-0.31698E-01,-0.28424E-01,-0.25181E-01,-0.21967E-01,
     &-0.18782E-01,-0.15626E-01,-0.12497E-01,-0.93962E-02,-0.63224E-02,
     &-0.32753E-02,-0.25445E-03, 0.27406E-02, 0.57102E-02, 0.86548E-02,
     & 0.11575E-01, 0.14470E-01, 0.17342E-01, 0.20191E-01, 0.23016E-01,
     & 0.25819E-01, 0.28599E-01, 0.31357E-01, 0.34093E-01, 0.36808E-01,
     & 0.39502E-01, 0.42175E-01, 0.44827E-01, 0.47460E-01, 0.50072E-01,
     & 0.52665E-01, 0.55239E-01, 0.57793E-01, 0.60329E-01, 0.62847E-01,
     & 0.65346E-01, 0.67827E-01, 0.70291E-01, 0.72737E-01, 0.75166E-01,
     & 0.77577E-01, 0.79972E-01, 0.82351E-01, 0.84713E-01, 0.87059E-01,
     & 0.89390E-01, 0.91704E-01, 0.94003E-01, 0.96287E-01, 0.98556E-01,
     & 0.10081E+00, 0.10305E+00, 0.10527E+00, 0.10749E+00, 0.10968E+00,
     & 0.11186E+00, 0.11403E+00, 0.11619E+00, 0.11833E+00, 0.12046E+00,
     & 0.12257E+00, 0.12468E+00, 0.12677E+00, 0.12884E+00, 0.13091E+00,
     & 0.13296E+00, 0.13500E+00, 0.13703E+00, 0.13904E+00, 0.14105E+00,
     & 0.14304E+00, 0.14502E+00, 0.14699E+00, 0.14895E+00, 0.15090E+00,
     & 0.15283E+00, 0.15476E+00, 0.15668E+00, 0.15858E+00, 0.16047E+00,
     & 0.16236E+00, 0.16423E+00, 0.16609E+00, 0.16795E+00, 0.16979E+00,
     & 0.17162E+00, 0.17345E+00, 0.17526E+00, 0.17706E+00, 0.17886E+00,
     & 0.18064E+00, 0.18242E+00, 0.18419E+00, 0.18595E+00, 0.18769E+00,
     & 0.18944E+00, 0.19117E+00, 0.19289E+00, 0.19460E+00, 0.19631E+00/

      DATA (BNC06M(I),I=701,741)/
     & 0.19801E+00, 0.19970E+00, 0.20138E+00, 0.20305E+00, 0.20471E+00,
     & 0.20637E+00, 0.20802E+00, 0.20966E+00, 0.21129E+00, 0.21292E+00,
     & 0.21453E+00, 0.21614E+00, 0.21775E+00, 0.21934E+00, 0.22093E+00,
     & 0.22251E+00, 0.22408E+00, 0.22565E+00, 0.22721E+00, 0.22876E+00,
     & 0.23030E+00, 0.23184E+00, 0.23337E+00, 0.23490E+00, 0.23641E+00,
     & 0.23792E+00, 0.23943E+00, 0.24093E+00, 0.24242E+00, 0.24390E+00,
     & 0.24538E+00, 0.24685E+00, 0.24832E+00, 0.24978E+00, 0.25123E+00,
     & 0.25268E+00, 0.25412E+00, 0.25555E+00, 0.25698E+00, 0.25840E+00,
     & 0.25982E+00
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.92967E-01,-0.20158E+00,-0.25546E+00,-0.29333E+00,-0.32290E+00,
     &-0.34726E+00,-0.36803E+00,-0.38614E+00,-0.40221E+00,-0.41665E+00,
     &-0.42977E+00,-0.44179E+00,-0.45288E+00,-0.46316E+00,-0.47275E+00,
     &-0.48173E+00,-0.49018E+00,-0.49815E+00,-0.50570E+00,-0.51286E+00,
     &-0.51967E+00,-0.52616E+00,-0.53237E+00,-0.53830E+00,-0.54400E+00,
     &-0.54947E+00,-0.55472E+00,-0.55979E+00,-0.56467E+00,-0.56938E+00,
     &-0.57394E+00,-0.57834E+00,-0.58261E+00,-0.58674E+00,-0.59075E+00,
     &-0.59465E+00,-0.59843E+00,-0.60211E+00,-0.60568E+00,-0.60916E+00,
     &-0.61255E+00,-0.61586E+00,-0.61908E+00,-0.62223E+00,-0.62530E+00,
     &-0.62830E+00,-0.63123E+00,-0.63409E+00,-0.63689E+00,-0.63963E+00,
     &-0.64231E+00,-0.64494E+00,-0.64751E+00,-0.65003E+00,-0.65250E+00,
     &-0.65493E+00,-0.65730E+00,-0.65964E+00,-0.66192E+00,-0.66417E+00,
     &-0.66638E+00,-0.66855E+00,-0.67068E+00,-0.67278E+00,-0.67484E+00,
     &-0.67686E+00,-0.67886E+00,-0.68082E+00,-0.68275E+00,-0.68465E+00,
     &-0.68652E+00,-0.68837E+00,-0.69019E+00,-0.69198E+00,-0.69375E+00,
     &-0.69549E+00,-0.69720E+00,-0.69890E+00,-0.70057E+00,-0.70222E+00,
     &-0.70385E+00,-0.70546E+00,-0.70704E+00,-0.70861E+00,-0.71016E+00,
     &-0.71169E+00,-0.71320E+00,-0.71470E+00,-0.71617E+00,-0.71763E+00,
     &-0.71908E+00,-0.72050E+00,-0.72192E+00,-0.72331E+00,-0.72470E+00,
     &-0.72606E+00,-0.72742E+00,-0.72875E+00,-0.73008E+00,-0.73139E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.73269E+00,-0.73398E+00,-0.73525E+00,-0.73651E+00,-0.73776E+00,
     &-0.73899E+00,-0.74022E+00,-0.74143E+00,-0.74263E+00,-0.74383E+00,
     &-0.74501E+00,-0.74617E+00,-0.74733E+00,-0.74848E+00,-0.74962E+00,
     &-0.75075E+00,-0.75186E+00,-0.75297E+00,-0.75407E+00,-0.75516E+00,
     &-0.75624E+00,-0.75731E+00,-0.75837E+00,-0.75942E+00,-0.76047E+00,
     &-0.76150E+00,-0.76253E+00,-0.76354E+00,-0.76455E+00,-0.76556E+00,
     &-0.76655E+00,-0.76753E+00,-0.76851E+00,-0.76948E+00,-0.77044E+00,
     &-0.77140E+00,-0.77235E+00,-0.77329E+00,-0.77422E+00,-0.77514E+00,
     &-0.77606E+00,-0.77697E+00,-0.77788E+00,-0.77878E+00,-0.77967E+00,
     &-0.78055E+00,-0.78143E+00,-0.78231E+00,-0.78317E+00,-0.78403E+00,
     &-0.78489E+00,-0.78573E+00,-0.78658E+00,-0.78741E+00,-0.78824E+00,
     &-0.78907E+00,-0.78989E+00,-0.79070E+00,-0.79151E+00,-0.79231E+00,
     &-0.79311E+00,-0.79390E+00,-0.79469E+00,-0.79547E+00,-0.79625E+00,
     &-0.79702E+00,-0.79779E+00,-0.79855E+00,-0.79930E+00,-0.80006E+00,
     &-0.80080E+00,-0.80155E+00,-0.80229E+00,-0.80302E+00,-0.80375E+00,
     &-0.80447E+00,-0.80519E+00,-0.80591E+00,-0.80662E+00,-0.80733E+00,
     &-0.80803E+00,-0.80873E+00,-0.80943E+00,-0.81012E+00,-0.81081E+00,
     &-0.81149E+00,-0.81217E+00,-0.81284E+00,-0.81351E+00,-0.81418E+00,
     &-0.81485E+00,-0.81551E+00,-0.81616E+00,-0.81682E+00,-0.81747E+00,
     &-0.81811E+00,-0.81875E+00,-0.81939E+00,-0.82003E+00,-0.82066E+00/

      DATA (BNC07M(I),I=201,300)/
     &-0.82129E+00,-0.82191E+00,-0.82254E+00,-0.82315E+00,-0.82377E+00,
     &-0.82438E+00,-0.82499E+00,-0.82560E+00,-0.82620E+00,-0.82680E+00,
     &-0.82740E+00,-0.82799E+00,-0.82858E+00,-0.82917E+00,-0.82975E+00,
     &-0.83033E+00,-0.83091E+00,-0.83149E+00,-0.83206E+00,-0.83263E+00,
     &-0.83320E+00,-0.83376E+00,-0.83433E+00,-0.83489E+00,-0.83544E+00,
     &-0.83600E+00,-0.83655E+00,-0.83710E+00,-0.83764E+00,-0.83819E+00,
     &-0.83873E+00,-0.83927E+00,-0.83980E+00,-0.84034E+00,-0.84087E+00,
     &-0.84140E+00,-0.84192E+00,-0.84245E+00,-0.84297E+00,-0.84349E+00,
     &-0.84401E+00,-0.84452E+00,-0.84503E+00,-0.84554E+00,-0.84605E+00,
     &-0.84656E+00,-0.84706E+00,-0.84756E+00,-0.84806E+00,-0.84856E+00,
     &-0.84905E+00,-0.84954E+00,-0.85003E+00,-0.85052E+00,-0.85101E+00,
     &-0.85149E+00,-0.85198E+00,-0.85246E+00,-0.85294E+00,-0.85341E+00,
     &-0.85389E+00,-0.85436E+00,-0.85483E+00,-0.85530E+00,-0.85576E+00,
     &-0.85623E+00,-0.85669E+00,-0.85715E+00,-0.85761E+00,-0.85807E+00,
     &-0.85852E+00,-0.85898E+00,-0.85943E+00,-0.85988E+00,-0.86033E+00,
     &-0.86077E+00,-0.86122E+00,-0.86166E+00,-0.86210E+00,-0.86254E+00,
     &-0.86298E+00,-0.86342E+00,-0.86385E+00,-0.86429E+00,-0.86472E+00,
     &-0.86515E+00,-0.86557E+00,-0.86600E+00,-0.86643E+00,-0.86685E+00,
     &-0.86727E+00,-0.86769E+00,-0.86811E+00,-0.86853E+00,-0.86894E+00,
     &-0.86936E+00,-0.86977E+00,-0.87018E+00,-0.87059E+00,-0.87100E+00/

      DATA (BNC07M(I),I=301,400)/
     &-0.87140E+00,-0.87181E+00,-0.87221E+00,-0.87261E+00,-0.87301E+00,
     &-0.87341E+00,-0.87381E+00,-0.87421E+00,-0.87460E+00,-0.87500E+00,
     &-0.87539E+00,-0.87578E+00,-0.87617E+00,-0.87656E+00,-0.87695E+00,
     &-0.87733E+00,-0.87772E+00,-0.87810E+00,-0.87848E+00,-0.87886E+00,
     &-0.87924E+00,-0.87962E+00,-0.87999E+00,-0.88037E+00,-0.88074E+00,
     &-0.88112E+00,-0.88149E+00,-0.88186E+00,-0.88223E+00,-0.88259E+00,
     &-0.88296E+00,-0.88333E+00,-0.88369E+00,-0.88405E+00,-0.88442E+00,
     &-0.88478E+00,-0.88514E+00,-0.88549E+00,-0.88585E+00,-0.88621E+00,
     &-0.88656E+00,-0.88692E+00,-0.88727E+00,-0.88762E+00,-0.88797E+00,
     &-0.88832E+00,-0.88867E+00,-0.88902E+00,-0.88936E+00,-0.88971E+00,
     &-0.89005E+00,-0.89039E+00,-0.89074E+00,-0.89108E+00,-0.89142E+00,
     &-0.89176E+00,-0.89209E+00,-0.89243E+00,-0.89277E+00,-0.89310E+00,
     &-0.89343E+00,-0.89377E+00,-0.89410E+00,-0.89443E+00,-0.89476E+00,
     &-0.89509E+00,-0.89541E+00,-0.89574E+00,-0.89607E+00,-0.89639E+00,
     &-0.89671E+00,-0.89704E+00,-0.89736E+00,-0.89768E+00,-0.89800E+00,
     &-0.89832E+00,-0.89864E+00,-0.89896E+00,-0.89927E+00,-0.89959E+00,
     &-0.89990E+00,-0.90022E+00,-0.90053E+00,-0.90084E+00,-0.90115E+00,
     &-0.90146E+00,-0.90177E+00,-0.90208E+00,-0.90239E+00,-0.90269E+00,
     &-0.90300E+00,-0.90330E+00,-0.90361E+00,-0.90391E+00,-0.90421E+00,
     &-0.90451E+00,-0.90481E+00,-0.90511E+00,-0.90541E+00,-0.90571E+00/

      DATA (BNC07M(I),I=401,500)/
     &-0.90601E+00,-0.90631E+00,-0.90660E+00,-0.90690E+00,-0.90719E+00,
     &-0.90748E+00,-0.90778E+00,-0.90807E+00,-0.90836E+00,-0.90865E+00,
     &-0.90894E+00,-0.90923E+00,-0.90952E+00,-0.90980E+00,-0.91009E+00,
     &-0.91038E+00,-0.91066E+00,-0.91095E+00,-0.91123E+00,-0.91151E+00,
     &-0.91179E+00,-0.91207E+00,-0.91236E+00,-0.91263E+00,-0.91291E+00,
     &-0.91319E+00,-0.91347E+00,-0.91375E+00,-0.91402E+00,-0.91430E+00,
     &-0.91457E+00,-0.91485E+00,-0.91512E+00,-0.91539E+00,-0.91567E+00,
     &-0.91594E+00,-0.91621E+00,-0.91648E+00,-0.91675E+00,-0.91702E+00,
     &-0.91728E+00,-0.91755E+00,-0.91782E+00,-0.91808E+00,-0.91835E+00,
     &-0.91861E+00,-0.91888E+00,-0.91914E+00,-0.91940E+00,-0.91967E+00,
     &-0.91993E+00,-0.92019E+00,-0.92045E+00,-0.92071E+00,-0.92097E+00,
     &-0.92123E+00,-0.92148E+00,-0.92174E+00,-0.92200E+00,-0.92225E+00,
     &-0.92251E+00,-0.92276E+00,-0.92302E+00,-0.92327E+00,-0.92352E+00,
     &-0.92378E+00,-0.92403E+00,-0.92428E+00,-0.92453E+00,-0.92478E+00,
     &-0.92503E+00,-0.92528E+00,-0.92552E+00,-0.92577E+00,-0.92602E+00,
     &-0.92627E+00,-0.92651E+00,-0.92676E+00,-0.92700E+00,-0.92725E+00,
     &-0.92749E+00,-0.92773E+00,-0.92798E+00,-0.92822E+00,-0.92846E+00,
     &-0.92870E+00,-0.92894E+00,-0.92918E+00,-0.92942E+00,-0.92966E+00,
     &-0.92990E+00,-0.93013E+00,-0.93037E+00,-0.93061E+00,-0.93084E+00,
     &-0.93108E+00,-0.93131E+00,-0.93155E+00,-0.93178E+00,-0.93202E+00/

      DATA (BNC07M(I),I=501,600)/
     &-0.93225E+00,-0.93248E+00,-0.93271E+00,-0.93294E+00,-0.93317E+00,
     &-0.93341E+00,-0.93363E+00,-0.93386E+00,-0.93409E+00,-0.93432E+00,
     &-0.93455E+00,-0.93478E+00,-0.93500E+00,-0.93523E+00,-0.93546E+00,
     &-0.93568E+00,-0.93591E+00,-0.93613E+00,-0.93636E+00,-0.93658E+00,
     &-0.93680E+00,-0.93702E+00,-0.93725E+00,-0.93747E+00,-0.93769E+00,
     &-0.93791E+00,-0.93813E+00,-0.93835E+00,-0.93857E+00,-0.93879E+00,
     &-0.93901E+00,-0.93923E+00,-0.93944E+00,-0.93966E+00,-0.93988E+00,
     &-0.94009E+00,-0.94031E+00,-0.94052E+00,-0.94074E+00,-0.94095E+00,
     &-0.94117E+00,-0.94138E+00,-0.94159E+00,-0.94181E+00,-0.94202E+00,
     &-0.94223E+00,-0.94244E+00,-0.94265E+00,-0.94286E+00,-0.94307E+00,
     &-0.94328E+00,-0.94349E+00,-0.94370E+00,-0.94391E+00,-0.94412E+00,
     &-0.94433E+00,-0.94453E+00,-0.94474E+00,-0.94495E+00,-0.94515E+00,
     &-0.94536E+00,-0.94556E+00,-0.94577E+00,-0.94597E+00,-0.94618E+00,
     &-0.94638E+00,-0.94658E+00,-0.94679E+00,-0.94699E+00,-0.94719E+00,
     &-0.94739E+00,-0.94759E+00,-0.94779E+00,-0.94799E+00,-0.94819E+00,
     &-0.94839E+00,-0.94859E+00,-0.94879E+00,-0.94899E+00,-0.94919E+00,
     &-0.94939E+00,-0.94958E+00,-0.94978E+00,-0.94998E+00,-0.95017E+00,
     &-0.95037E+00,-0.95057E+00,-0.95076E+00,-0.95096E+00,-0.95115E+00,
     &-0.95134E+00,-0.95154E+00,-0.95173E+00,-0.95192E+00,-0.95212E+00,
     &-0.95231E+00,-0.95250E+00,-0.95269E+00,-0.95288E+00,-0.95307E+00/

      DATA (BNC07M(I),I=601,700)/
     &-0.95511E+00,-0.95697E+00,-0.95879E+00,-0.96057E+00,-0.96233E+00,
     &-0.96406E+00,-0.96576E+00,-0.96743E+00,-0.96907E+00,-0.97069E+00,
     &-0.97228E+00,-0.97385E+00,-0.97539E+00,-0.97691E+00,-0.97841E+00,
     &-0.97989E+00,-0.98134E+00,-0.98277E+00,-0.98418E+00,-0.98558E+00,
     &-0.98695E+00,-0.98830E+00,-0.98964E+00,-0.99096E+00,-0.99226E+00,
     &-0.99354E+00,-0.99481E+00,-0.99606E+00,-0.99730E+00,-0.99852E+00,
     &-0.99972E+00,-0.10009E+01,-0.10021E+01,-0.10032E+01,-0.10044E+01,
     &-0.10055E+01,-0.10067E+01,-0.10078E+01,-0.10089E+01,-0.10099E+01,
     &-0.10110E+01,-0.10121E+01,-0.10131E+01,-0.10142E+01,-0.10152E+01,
     &-0.10162E+01,-0.10172E+01,-0.10182E+01,-0.10192E+01,-0.10201E+01,
     &-0.10211E+01,-0.10221E+01,-0.10230E+01,-0.10239E+01,-0.10249E+01,
     &-0.10258E+01,-0.10267E+01,-0.10276E+01,-0.10285E+01,-0.10293E+01,
     &-0.10302E+01,-0.10311E+01,-0.10319E+01,-0.10328E+01,-0.10336E+01,
     &-0.10345E+01,-0.10353E+01,-0.10361E+01,-0.10369E+01,-0.10377E+01,
     &-0.10385E+01,-0.10393E+01,-0.10401E+01,-0.10409E+01,-0.10417E+01,
     &-0.10424E+01,-0.10432E+01,-0.10439E+01,-0.10447E+01,-0.10454E+01,
     &-0.10461E+01,-0.10469E+01,-0.10476E+01,-0.10483E+01,-0.10490E+01,
     &-0.10497E+01,-0.10504E+01,-0.10511E+01,-0.10518E+01,-0.10525E+01,
     &-0.10532E+01,-0.10538E+01,-0.10545E+01,-0.10552E+01,-0.10558E+01,
     &-0.10565E+01,-0.10571E+01,-0.10578E+01,-0.10584E+01,-0.10591E+01/

      DATA (BNC07M(I),I=701,741)/
     &-0.10597E+01,-0.10603E+01,-0.10609E+01,-0.10616E+01,-0.10622E+01,
     &-0.10628E+01,-0.10634E+01,-0.10640E+01,-0.10646E+01,-0.10652E+01,
     &-0.10658E+01,-0.10663E+01,-0.10669E+01,-0.10675E+01,-0.10681E+01,
     &-0.10687E+01,-0.10692E+01,-0.10698E+01,-0.10703E+01,-0.10709E+01,
     &-0.10714E+01,-0.10720E+01,-0.10725E+01,-0.10731E+01,-0.10736E+01,
     &-0.10742E+01,-0.10747E+01,-0.10752E+01,-0.10757E+01,-0.10763E+01,
     &-0.10768E+01,-0.10773E+01,-0.10778E+01,-0.10783E+01,-0.10788E+01,
     &-0.10793E+01,-0.10798E+01,-0.10803E+01,-0.10808E+01,-0.10813E+01,
     &-0.10818E+01
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.43840E-01,-0.87607E-01,-0.10559E+00,-0.11630E+00,-0.12329E+00,
     &-0.12797E+00,-0.13104E+00,-0.13292E+00,-0.13386E+00,-0.13406E+00,
     &-0.13363E+00,-0.13266E+00,-0.13123E+00,-0.12939E+00,-0.12719E+00,
     &-0.12466E+00,-0.12183E+00,-0.11873E+00,-0.11538E+00,-0.11178E+00,
     &-0.10797E+00,-0.10396E+00,-0.99747E-01,-0.95354E-01,-0.90787E-01,
     &-0.86056E-01,-0.81168E-01,-0.76132E-01,-0.70954E-01,-0.65641E-01,
     &-0.60198E-01,-0.54632E-01,-0.48948E-01,-0.43150E-01,-0.37245E-01,
     &-0.31235E-01,-0.25127E-01,-0.18924E-01,-0.12630E-01,-0.62488E-02,
     & 0.21565E-03, 0.67597E-02, 0.13380E-01, 0.20073E-01, 0.26837E-01,
     & 0.33667E-01, 0.40561E-01, 0.47517E-01, 0.54532E-01, 0.61603E-01,
     & 0.68729E-01, 0.75906E-01, 0.83134E-01, 0.90410E-01, 0.97733E-01,
     & 0.10510E+00, 0.11251E+00, 0.11997E+00, 0.12746E+00, 0.13500E+00,
     & 0.14258E+00, 0.15019E+00, 0.15784E+00, 0.16554E+00, 0.17327E+00,
     & 0.18103E+00, 0.18884E+00, 0.19668E+00, 0.20456E+00, 0.21248E+00,
     & 0.22044E+00, 0.22843E+00, 0.23647E+00, 0.24454E+00, 0.25266E+00,
     & 0.26081E+00, 0.26901E+00, 0.27725E+00, 0.28553E+00, 0.29386E+00,
     & 0.30223E+00, 0.31064E+00, 0.31909E+00, 0.32759E+00, 0.33614E+00,
     & 0.34473E+00, 0.35336E+00, 0.36204E+00, 0.37076E+00, 0.37953E+00,
     & 0.38834E+00, 0.39720E+00, 0.40609E+00, 0.41503E+00, 0.42401E+00,
     & 0.43303E+00, 0.44209E+00, 0.45118E+00, 0.46032E+00, 0.46948E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.47869E+00, 0.48792E+00, 0.49718E+00, 0.50648E+00, 0.51580E+00,
     & 0.52515E+00, 0.53452E+00, 0.54391E+00, 0.55333E+00, 0.56276E+00,
     & 0.57221E+00, 0.58168E+00, 0.59116E+00, 0.60066E+00, 0.61016E+00,
     & 0.61968E+00, 0.62920E+00, 0.63872E+00, 0.64826E+00, 0.65779E+00,
     & 0.66733E+00, 0.67687E+00, 0.68641E+00, 0.69595E+00, 0.70548E+00,
     & 0.71501E+00, 0.72454E+00, 0.73406E+00, 0.74357E+00, 0.75308E+00,
     & 0.76258E+00, 0.77207E+00, 0.78155E+00, 0.79102E+00, 0.80048E+00,
     & 0.80993E+00, 0.81936E+00, 0.82879E+00, 0.83820E+00, 0.84760E+00,
     & 0.85698E+00, 0.86635E+00, 0.87571E+00, 0.88505E+00, 0.89438E+00,
     & 0.90369E+00, 0.91298E+00, 0.92226E+00, 0.93153E+00, 0.94078E+00,
     & 0.95001E+00, 0.95923E+00, 0.96843E+00, 0.97761E+00, 0.98678E+00,
     & 0.99593E+00, 0.10051E+01, 0.10142E+01, 0.10233E+01, 0.10324E+01,
     & 0.10414E+01, 0.10505E+01, 0.10595E+01, 0.10685E+01, 0.10775E+01,
     & 0.10865E+01, 0.10954E+01, 0.11044E+01, 0.11133E+01, 0.11222E+01,
     & 0.11311E+01, 0.11399E+01, 0.11488E+01, 0.11576E+01, 0.11664E+01,
     & 0.11752E+01, 0.11840E+01, 0.11928E+01, 0.12015E+01, 0.12102E+01,
     & 0.12190E+01, 0.12276E+01, 0.12363E+01, 0.12450E+01, 0.12536E+01,
     & 0.12622E+01, 0.12708E+01, 0.12794E+01, 0.12879E+01, 0.12965E+01,
     & 0.13050E+01, 0.13135E+01, 0.13220E+01, 0.13305E+01, 0.13389E+01,
     & 0.13474E+01, 0.13558E+01, 0.13642E+01, 0.13726E+01, 0.13809E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.13893E+01, 0.13976E+01, 0.14059E+01, 0.14142E+01, 0.14225E+01,
     & 0.14307E+01, 0.14390E+01, 0.14472E+01, 0.14554E+01, 0.14636E+01,
     & 0.14718E+01, 0.14799E+01, 0.14881E+01, 0.14962E+01, 0.15043E+01,
     & 0.15124E+01, 0.15204E+01, 0.15285E+01, 0.15365E+01, 0.15445E+01,
     & 0.15525E+01, 0.15605E+01, 0.15685E+01, 0.15764E+01, 0.15843E+01,
     & 0.15923E+01, 0.16002E+01, 0.16080E+01, 0.16159E+01, 0.16237E+01,
     & 0.16316E+01, 0.16394E+01, 0.16472E+01, 0.16550E+01, 0.16627E+01,
     & 0.16705E+01, 0.16782E+01, 0.16859E+01, 0.16936E+01, 0.17013E+01,
     & 0.17090E+01, 0.17166E+01, 0.17243E+01, 0.17319E+01, 0.17395E+01,
     & 0.17471E+01, 0.17547E+01, 0.17622E+01, 0.17698E+01, 0.17773E+01,
     & 0.17848E+01, 0.17923E+01, 0.17998E+01, 0.18072E+01, 0.18147E+01,
     & 0.18221E+01, 0.18295E+01, 0.18369E+01, 0.18443E+01, 0.18517E+01,
     & 0.18591E+01, 0.18664E+01, 0.18737E+01, 0.18811E+01, 0.18884E+01,
     & 0.18956E+01, 0.19029E+01, 0.19102E+01, 0.19174E+01, 0.19246E+01,
     & 0.19318E+01, 0.19390E+01, 0.19462E+01, 0.19534E+01, 0.19605E+01,
     & 0.19677E+01, 0.19748E+01, 0.19819E+01, 0.19890E+01, 0.19961E+01,
     & 0.20032E+01, 0.20102E+01, 0.20173E+01, 0.20243E+01, 0.20313E+01,
     & 0.20383E+01, 0.20453E+01, 0.20523E+01, 0.20592E+01, 0.20662E+01,
     & 0.20731E+01, 0.20800E+01, 0.20869E+01, 0.20938E+01, 0.21007E+01,
     & 0.21075E+01, 0.21144E+01, 0.21212E+01, 0.21281E+01, 0.21349E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.21417E+01, 0.21485E+01, 0.21552E+01, 0.21620E+01, 0.21687E+01,
     & 0.21755E+01, 0.21822E+01, 0.21889E+01, 0.21956E+01, 0.22023E+01,
     & 0.22089E+01, 0.22156E+01, 0.22222E+01, 0.22289E+01, 0.22355E+01,
     & 0.22421E+01, 0.22487E+01, 0.22553E+01, 0.22619E+01, 0.22684E+01,
     & 0.22750E+01, 0.22815E+01, 0.22880E+01, 0.22945E+01, 0.23010E+01,
     & 0.23075E+01, 0.23140E+01, 0.23205E+01, 0.23269E+01, 0.23333E+01,
     & 0.23398E+01, 0.23462E+01, 0.23526E+01, 0.23590E+01, 0.23654E+01,
     & 0.23717E+01, 0.23781E+01, 0.23844E+01, 0.23908E+01, 0.23971E+01,
     & 0.24034E+01, 0.24097E+01, 0.24160E+01, 0.24223E+01, 0.24285E+01,
     & 0.24348E+01, 0.24410E+01, 0.24473E+01, 0.24535E+01, 0.24597E+01,
     & 0.24659E+01, 0.24721E+01, 0.24783E+01, 0.24844E+01, 0.24906E+01,
     & 0.24967E+01, 0.25029E+01, 0.25090E+01, 0.25151E+01, 0.25212E+01,
     & 0.25273E+01, 0.25334E+01, 0.25395E+01, 0.25455E+01, 0.25516E+01,
     & 0.25576E+01, 0.25636E+01, 0.25697E+01, 0.25757E+01, 0.25817E+01,
     & 0.25877E+01, 0.25936E+01, 0.25996E+01, 0.26056E+01, 0.26115E+01,
     & 0.26174E+01, 0.26234E+01, 0.26293E+01, 0.26352E+01, 0.26411E+01,
     & 0.26470E+01, 0.26529E+01, 0.26587E+01, 0.26646E+01, 0.26704E+01,
     & 0.26763E+01, 0.26821E+01, 0.26879E+01, 0.26937E+01, 0.26995E+01,
     & 0.27053E+01, 0.27111E+01, 0.27169E+01, 0.27226E+01, 0.27284E+01,
     & 0.27341E+01, 0.27398E+01, 0.27456E+01, 0.27513E+01, 0.27570E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.27627E+01, 0.27684E+01, 0.27740E+01, 0.27797E+01, 0.27854E+01,
     & 0.27910E+01, 0.27967E+01, 0.28023E+01, 0.28079E+01, 0.28135E+01,
     & 0.28191E+01, 0.28247E+01, 0.28303E+01, 0.28359E+01, 0.28415E+01,
     & 0.28470E+01, 0.28526E+01, 0.28581E+01, 0.28636E+01, 0.28692E+01,
     & 0.28747E+01, 0.28802E+01, 0.28857E+01, 0.28912E+01, 0.28966E+01,
     & 0.29021E+01, 0.29076E+01, 0.29130E+01, 0.29185E+01, 0.29239E+01,
     & 0.29293E+01, 0.29348E+01, 0.29402E+01, 0.29456E+01, 0.29510E+01,
     & 0.29564E+01, 0.29617E+01, 0.29671E+01, 0.29725E+01, 0.29778E+01,
     & 0.29832E+01, 0.29885E+01, 0.29938E+01, 0.29992E+01, 0.30045E+01,
     & 0.30098E+01, 0.30151E+01, 0.30204E+01, 0.30256E+01, 0.30309E+01,
     & 0.30362E+01, 0.30414E+01, 0.30467E+01, 0.30519E+01, 0.30572E+01,
     & 0.30624E+01, 0.30676E+01, 0.30728E+01, 0.30780E+01, 0.30832E+01,
     & 0.30884E+01, 0.30936E+01, 0.30987E+01, 0.31039E+01, 0.31091E+01,
     & 0.31142E+01, 0.31193E+01, 0.31245E+01, 0.31296E+01, 0.31347E+01,
     & 0.31398E+01, 0.31449E+01, 0.31500E+01, 0.31551E+01, 0.31602E+01,
     & 0.31653E+01, 0.31703E+01, 0.31754E+01, 0.31804E+01, 0.31855E+01,
     & 0.31905E+01, 0.31955E+01, 0.32006E+01, 0.32056E+01, 0.32106E+01,
     & 0.32156E+01, 0.32206E+01, 0.32256E+01, 0.32305E+01, 0.32355E+01,
     & 0.32405E+01, 0.32454E+01, 0.32504E+01, 0.32553E+01, 0.32603E+01,
     & 0.32652E+01, 0.32701E+01, 0.32750E+01, 0.32799E+01, 0.32848E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.32897E+01, 0.32946E+01, 0.32995E+01, 0.33044E+01, 0.33092E+01,
     & 0.33141E+01, 0.33189E+01, 0.33238E+01, 0.33286E+01, 0.33335E+01,
     & 0.33383E+01, 0.33431E+01, 0.33479E+01, 0.33527E+01, 0.33575E+01,
     & 0.33623E+01, 0.33671E+01, 0.33719E+01, 0.33766E+01, 0.33814E+01,
     & 0.33862E+01, 0.33909E+01, 0.33957E+01, 0.34004E+01, 0.34051E+01,
     & 0.34099E+01, 0.34146E+01, 0.34193E+01, 0.34240E+01, 0.34287E+01,
     & 0.34334E+01, 0.34381E+01, 0.34428E+01, 0.34475E+01, 0.34521E+01,
     & 0.34568E+01, 0.34615E+01, 0.34661E+01, 0.34707E+01, 0.34754E+01,
     & 0.34800E+01, 0.34846E+01, 0.34893E+01, 0.34939E+01, 0.34985E+01,
     & 0.35031E+01, 0.35077E+01, 0.35123E+01, 0.35169E+01, 0.35214E+01,
     & 0.35260E+01, 0.35306E+01, 0.35351E+01, 0.35397E+01, 0.35442E+01,
     & 0.35488E+01, 0.35533E+01, 0.35578E+01, 0.35624E+01, 0.35669E+01,
     & 0.35714E+01, 0.35759E+01, 0.35804E+01, 0.35849E+01, 0.35894E+01,
     & 0.35939E+01, 0.35983E+01, 0.36028E+01, 0.36073E+01, 0.36117E+01,
     & 0.36162E+01, 0.36206E+01, 0.36251E+01, 0.36295E+01, 0.36340E+01,
     & 0.36384E+01, 0.36428E+01, 0.36472E+01, 0.36516E+01, 0.36560E+01,
     & 0.36604E+01, 0.36648E+01, 0.36692E+01, 0.36736E+01, 0.36780E+01,
     & 0.36823E+01, 0.36867E+01, 0.36910E+01, 0.36954E+01, 0.36997E+01,
     & 0.37041E+01, 0.37084E+01, 0.37128E+01, 0.37171E+01, 0.37214E+01,
     & 0.37257E+01, 0.37300E+01, 0.37343E+01, 0.37386E+01, 0.37429E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.37890E+01, 0.38311E+01, 0.38727E+01, 0.39138E+01, 0.39544E+01,
     & 0.39945E+01, 0.40342E+01, 0.40734E+01, 0.41122E+01, 0.41506E+01,
     & 0.41885E+01, 0.42261E+01, 0.42632E+01, 0.42999E+01, 0.43363E+01,
     & 0.43722E+01, 0.44079E+01, 0.44431E+01, 0.44780E+01, 0.45125E+01,
     & 0.45467E+01, 0.45806E+01, 0.46141E+01, 0.46473E+01, 0.46802E+01,
     & 0.47128E+01, 0.47451E+01, 0.47771E+01, 0.48088E+01, 0.48402E+01,
     & 0.48713E+01, 0.49022E+01, 0.49327E+01, 0.49630E+01, 0.49931E+01,
     & 0.50229E+01, 0.50524E+01, 0.50817E+01, 0.51107E+01, 0.51395E+01,
     & 0.51681E+01, 0.51964E+01, 0.52245E+01, 0.52524E+01, 0.52800E+01,
     & 0.53074E+01, 0.53346E+01, 0.53616E+01, 0.53884E+01, 0.54150E+01,
     & 0.54414E+01, 0.54676E+01, 0.54936E+01, 0.55194E+01, 0.55450E+01,
     & 0.55704E+01, 0.55956E+01, 0.56206E+01, 0.56455E+01, 0.56702E+01,
     & 0.56947E+01, 0.57191E+01, 0.57432E+01, 0.57672E+01, 0.57911E+01,
     & 0.58148E+01, 0.58383E+01, 0.58616E+01, 0.58848E+01, 0.59079E+01,
     & 0.59308E+01, 0.59535E+01, 0.59761E+01, 0.59986E+01, 0.60209E+01,
     & 0.60430E+01, 0.60650E+01, 0.60869E+01, 0.61087E+01, 0.61303E+01,
     & 0.61517E+01, 0.61731E+01, 0.61943E+01, 0.62154E+01, 0.62363E+01,
     & 0.62572E+01, 0.62779E+01, 0.62984E+01, 0.63189E+01, 0.63392E+01,
     & 0.63595E+01, 0.63796E+01, 0.63996E+01, 0.64194E+01, 0.64392E+01,
     & 0.64588E+01, 0.64784E+01, 0.64978E+01, 0.65171E+01, 0.65363E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.65554E+01, 0.65744E+01, 0.65933E+01, 0.66121E+01, 0.66308E+01,
     & 0.66494E+01, 0.66679E+01, 0.66863E+01, 0.67046E+01, 0.67228E+01,
     & 0.67409E+01, 0.67589E+01, 0.67768E+01, 0.67947E+01, 0.68124E+01,
     & 0.68300E+01, 0.68476E+01, 0.68651E+01, 0.68824E+01, 0.68997E+01,
     & 0.69169E+01, 0.69341E+01, 0.69511E+01, 0.69680E+01, 0.69849E+01,
     & 0.70017E+01, 0.70184E+01, 0.70350E+01, 0.70516E+01, 0.70681E+01,
     & 0.70845E+01, 0.71008E+01, 0.71170E+01, 0.71332E+01, 0.71493E+01,
     & 0.71653E+01, 0.71812E+01, 0.71971E+01, 0.72129E+01, 0.72286E+01,
     & 0.72442E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.45695E-01,-0.97041E-01,-0.12160E+00,-0.13842E+00,-0.15124E+00,
     &-0.16156E+00,-0.17015E+00,-0.17746E+00,-0.18378E+00,-0.18930E+00,
     &-0.19417E+00,-0.19849E+00,-0.20233E+00,-0.20577E+00,-0.20884E+00,
     &-0.21160E+00,-0.21406E+00,-0.21627E+00,-0.21823E+00,-0.21998E+00,
     &-0.22152E+00,-0.22288E+00,-0.22406E+00,-0.22508E+00,-0.22594E+00,
     &-0.22665E+00,-0.22723E+00,-0.22768E+00,-0.22800E+00,-0.22821E+00,
     &-0.22830E+00,-0.22828E+00,-0.22816E+00,-0.22794E+00,-0.22763E+00,
     &-0.22722E+00,-0.22673E+00,-0.22615E+00,-0.22549E+00,-0.22475E+00,
     &-0.22394E+00,-0.22305E+00,-0.22210E+00,-0.22108E+00,-0.21999E+00,
     &-0.21884E+00,-0.21763E+00,-0.21636E+00,-0.21503E+00,-0.21366E+00,
     &-0.21222E+00,-0.21074E+00,-0.20921E+00,-0.20763E+00,-0.20601E+00,
     &-0.20434E+00,-0.20263E+00,-0.20087E+00,-0.19908E+00,-0.19725E+00,
     &-0.19538E+00,-0.19347E+00,-0.19153E+00,-0.18955E+00,-0.18754E+00,
     &-0.18550E+00,-0.18342E+00,-0.18132E+00,-0.17918E+00,-0.17701E+00,
     &-0.17481E+00,-0.17258E+00,-0.17032E+00,-0.16804E+00,-0.16572E+00,
     &-0.16338E+00,-0.16101E+00,-0.15861E+00,-0.15619E+00,-0.15374E+00,
     &-0.15127E+00,-0.14877E+00,-0.14624E+00,-0.14369E+00,-0.14111E+00,
     &-0.13851E+00,-0.13588E+00,-0.13323E+00,-0.13056E+00,-0.12786E+00,
     &-0.12514E+00,-0.12240E+00,-0.11964E+00,-0.11686E+00,-0.11405E+00,
     &-0.11123E+00,-0.10839E+00,-0.10552E+00,-0.10264E+00,-0.99745E-01/

      DATA (BNC09M(I),I=101,200)/
     &-0.96831E-01,-0.93900E-01,-0.90954E-01,-0.87994E-01,-0.85020E-01,
     &-0.82033E-01,-0.79033E-01,-0.76023E-01,-0.73001E-01,-0.69969E-01,
     &-0.66929E-01,-0.63879E-01,-0.60822E-01,-0.57758E-01,-0.54687E-01,
     &-0.51611E-01,-0.48530E-01,-0.45444E-01,-0.42354E-01,-0.39261E-01,
     &-0.36165E-01,-0.33068E-01,-0.29968E-01,-0.26868E-01,-0.23766E-01,
     &-0.20665E-01,-0.17564E-01,-0.14463E-01,-0.11364E-01,-0.82653E-02,
     &-0.51688E-02,-0.20741E-02, 0.10182E-02, 0.41078E-02, 0.71947E-02,
     & 0.10278E-01, 0.13359E-01, 0.16436E-01, 0.19509E-01, 0.22578E-01,
     & 0.25643E-01, 0.28705E-01, 0.31761E-01, 0.34813E-01, 0.37861E-01,
     & 0.40903E-01, 0.43941E-01, 0.46974E-01, 0.50001E-01, 0.53024E-01,
     & 0.56041E-01, 0.59053E-01, 0.62059E-01, 0.65060E-01, 0.68056E-01,
     & 0.71045E-01, 0.74030E-01, 0.77008E-01, 0.79981E-01, 0.82948E-01,
     & 0.85909E-01, 0.88864E-01, 0.91813E-01, 0.94757E-01, 0.97694E-01,
     & 0.10063E+00, 0.10355E+00, 0.10647E+00, 0.10938E+00, 0.11229E+00,
     & 0.11519E+00, 0.11809E+00, 0.12098E+00, 0.12386E+00, 0.12674E+00,
     & 0.12961E+00, 0.13247E+00, 0.13533E+00, 0.13818E+00, 0.14103E+00,
     & 0.14387E+00, 0.14670E+00, 0.14953E+00, 0.15235E+00, 0.15517E+00,
     & 0.15798E+00, 0.16078E+00, 0.16358E+00, 0.16637E+00, 0.16915E+00,
     & 0.17193E+00, 0.17470E+00, 0.17747E+00, 0.18023E+00, 0.18299E+00,
     & 0.18573E+00, 0.18848E+00, 0.19121E+00, 0.19394E+00, 0.19667E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.19939E+00, 0.20210E+00, 0.20480E+00, 0.20751E+00, 0.21020E+00,
     & 0.21289E+00, 0.21557E+00, 0.21825E+00, 0.22092E+00, 0.22358E+00,
     & 0.22624E+00, 0.22890E+00, 0.23154E+00, 0.23419E+00, 0.23682E+00,
     & 0.23945E+00, 0.24208E+00, 0.24470E+00, 0.24731E+00, 0.24992E+00,
     & 0.25252E+00, 0.25512E+00, 0.25771E+00, 0.26029E+00, 0.26287E+00,
     & 0.26545E+00, 0.26802E+00, 0.27058E+00, 0.27314E+00, 0.27569E+00,
     & 0.27824E+00, 0.28078E+00, 0.28331E+00, 0.28584E+00, 0.28837E+00,
     & 0.29089E+00, 0.29340E+00, 0.29591E+00, 0.29841E+00, 0.30091E+00,
     & 0.30341E+00, 0.30589E+00, 0.30838E+00, 0.31086E+00, 0.31333E+00,
     & 0.31580E+00, 0.31826E+00, 0.32071E+00, 0.32317E+00, 0.32561E+00,
     & 0.32806E+00, 0.33049E+00, 0.33292E+00, 0.33535E+00, 0.33777E+00,
     & 0.34019E+00, 0.34260E+00, 0.34501E+00, 0.34741E+00, 0.34981E+00,
     & 0.35220E+00, 0.35459E+00, 0.35697E+00, 0.35935E+00, 0.36172E+00,
     & 0.36409E+00, 0.36645E+00, 0.36881E+00, 0.37117E+00, 0.37352E+00,
     & 0.37586E+00, 0.37820E+00, 0.38054E+00, 0.38287E+00, 0.38520E+00,
     & 0.38752E+00, 0.38983E+00, 0.39215E+00, 0.39446E+00, 0.39676E+00,
     & 0.39906E+00, 0.40135E+00, 0.40364E+00, 0.40593E+00, 0.40821E+00,
     & 0.41049E+00, 0.41276E+00, 0.41503E+00, 0.41729E+00, 0.41955E+00,
     & 0.42181E+00, 0.42406E+00, 0.42631E+00, 0.42855E+00, 0.43079E+00,
     & 0.43302E+00, 0.43525E+00, 0.43747E+00, 0.43970E+00, 0.44191E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.44413E+00, 0.44633E+00, 0.44854E+00, 0.45074E+00, 0.45294E+00,
     & 0.45513E+00, 0.45732E+00, 0.45950E+00, 0.46168E+00, 0.46386E+00,
     & 0.46603E+00, 0.46820E+00, 0.47036E+00, 0.47252E+00, 0.47468E+00,
     & 0.47683E+00, 0.47898E+00, 0.48113E+00, 0.48327E+00, 0.48540E+00,
     & 0.48754E+00, 0.48967E+00, 0.49179E+00, 0.49391E+00, 0.49603E+00,
     & 0.49814E+00, 0.50025E+00, 0.50236E+00, 0.50446E+00, 0.50656E+00,
     & 0.50866E+00, 0.51075E+00, 0.51284E+00, 0.51492E+00, 0.51700E+00,
     & 0.51908E+00, 0.52115E+00, 0.52322E+00, 0.52529E+00, 0.52735E+00,
     & 0.52941E+00, 0.53146E+00, 0.53352E+00, 0.53556E+00, 0.53761E+00,
     & 0.53965E+00, 0.54169E+00, 0.54372E+00, 0.54575E+00, 0.54778E+00,
     & 0.54980E+00, 0.55182E+00, 0.55384E+00, 0.55585E+00, 0.55786E+00,
     & 0.55987E+00, 0.56187E+00, 0.56387E+00, 0.56587E+00, 0.56786E+00,
     & 0.56985E+00, 0.57184E+00, 0.57382E+00, 0.57580E+00, 0.57778E+00,
     & 0.57975E+00, 0.58172E+00, 0.58369E+00, 0.58565E+00, 0.58761E+00,
     & 0.58957E+00, 0.59152E+00, 0.59348E+00, 0.59542E+00, 0.59737E+00,
     & 0.59931E+00, 0.60125E+00, 0.60318E+00, 0.60511E+00, 0.60704E+00,
     & 0.60897E+00, 0.61089E+00, 0.61281E+00, 0.61472E+00, 0.61664E+00,
     & 0.61855E+00, 0.62045E+00, 0.62236E+00, 0.62426E+00, 0.62616E+00,
     & 0.62805E+00, 0.62994E+00, 0.63183E+00, 0.63372E+00, 0.63560E+00,
     & 0.63748E+00, 0.63936E+00, 0.64123E+00, 0.64310E+00, 0.64497E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.64683E+00, 0.64870E+00, 0.65056E+00, 0.65241E+00, 0.65427E+00,
     & 0.65612E+00, 0.65796E+00, 0.65981E+00, 0.66165E+00, 0.66349E+00,
     & 0.66533E+00, 0.66716E+00, 0.66899E+00, 0.67082E+00, 0.67264E+00,
     & 0.67447E+00, 0.67629E+00, 0.67810E+00, 0.67992E+00, 0.68173E+00,
     & 0.68354E+00, 0.68534E+00, 0.68715E+00, 0.68895E+00, 0.69074E+00,
     & 0.69254E+00, 0.69433E+00, 0.69612E+00, 0.69791E+00, 0.69969E+00,
     & 0.70147E+00, 0.70325E+00, 0.70503E+00, 0.70680E+00, 0.70857E+00,
     & 0.71034E+00, 0.71211E+00, 0.71387E+00, 0.71563E+00, 0.71739E+00,
     & 0.71915E+00, 0.72090E+00, 0.72265E+00, 0.72440E+00, 0.72614E+00,
     & 0.72788E+00, 0.72962E+00, 0.73136E+00, 0.73310E+00, 0.73483E+00,
     & 0.73656E+00, 0.73829E+00, 0.74001E+00, 0.74174E+00, 0.74346E+00,
     & 0.74517E+00, 0.74689E+00, 0.74860E+00, 0.75031E+00, 0.75202E+00,
     & 0.75372E+00, 0.75543E+00, 0.75713E+00, 0.75883E+00, 0.76052E+00,
     & 0.76222E+00, 0.76391E+00, 0.76559E+00, 0.76728E+00, 0.76896E+00,
     & 0.77065E+00, 0.77233E+00, 0.77400E+00, 0.77568E+00, 0.77735E+00,
     & 0.77902E+00, 0.78069E+00, 0.78235E+00, 0.78401E+00, 0.78567E+00,
     & 0.78733E+00, 0.78899E+00, 0.79064E+00, 0.79229E+00, 0.79394E+00,
     & 0.79559E+00, 0.79723E+00, 0.79888E+00, 0.80052E+00, 0.80215E+00,
     & 0.80379E+00, 0.80542E+00, 0.80705E+00, 0.80868E+00, 0.81031E+00,
     & 0.81193E+00, 0.81356E+00, 0.81518E+00, 0.81679E+00, 0.81841E+00/

      DATA (BNC09M(I),I=501,600)/
     & 0.82002E+00, 0.82163E+00, 0.82324E+00, 0.82485E+00, 0.82646E+00,
     & 0.82806E+00, 0.82966E+00, 0.83126E+00, 0.83285E+00, 0.83445E+00,
     & 0.83604E+00, 0.83763E+00, 0.83922E+00, 0.84080E+00, 0.84239E+00,
     & 0.84397E+00, 0.84555E+00, 0.84713E+00, 0.84870E+00, 0.85028E+00,
     & 0.85185E+00, 0.85342E+00, 0.85498E+00, 0.85655E+00, 0.85811E+00,
     & 0.85967E+00, 0.86123E+00, 0.86279E+00, 0.86434E+00, 0.86590E+00,
     & 0.86745E+00, 0.86900E+00, 0.87054E+00, 0.87209E+00, 0.87363E+00,
     & 0.87517E+00, 0.87671E+00, 0.87825E+00, 0.87978E+00, 0.88132E+00,
     & 0.88285E+00, 0.88438E+00, 0.88590E+00, 0.88743E+00, 0.88895E+00,
     & 0.89047E+00, 0.89199E+00, 0.89351E+00, 0.89503E+00, 0.89654E+00,
     & 0.89805E+00, 0.89956E+00, 0.90107E+00, 0.90258E+00, 0.90408E+00,
     & 0.90558E+00, 0.90709E+00, 0.90858E+00, 0.91008E+00, 0.91158E+00,
     & 0.91307E+00, 0.91456E+00, 0.91605E+00, 0.91754E+00, 0.91902E+00,
     & 0.92051E+00, 0.92199E+00, 0.92347E+00, 0.92495E+00, 0.92642E+00,
     & 0.92790E+00, 0.92937E+00, 0.93084E+00, 0.93231E+00, 0.93378E+00,
     & 0.93525E+00, 0.93671E+00, 0.93817E+00, 0.93963E+00, 0.94109E+00,
     & 0.94255E+00, 0.94400E+00, 0.94546E+00, 0.94691E+00, 0.94836E+00,
     & 0.94981E+00, 0.95125E+00, 0.95270E+00, 0.95414E+00, 0.95558E+00,
     & 0.95702E+00, 0.95846E+00, 0.95990E+00, 0.96133E+00, 0.96276E+00,
     & 0.96419E+00, 0.96562E+00, 0.96705E+00, 0.96848E+00, 0.96990E+00/

      DATA (BNC09M(I),I=601,700)/
     & 0.98518E+00, 0.99915E+00, 0.10130E+01, 0.10266E+01, 0.10401E+01,
     & 0.10535E+01, 0.10667E+01, 0.10797E+01, 0.10927E+01, 0.11054E+01,
     & 0.11181E+01, 0.11306E+01, 0.11430E+01, 0.11553E+01, 0.11674E+01,
     & 0.11794E+01, 0.11913E+01, 0.12031E+01, 0.12148E+01, 0.12263E+01,
     & 0.12378E+01, 0.12491E+01, 0.12604E+01, 0.12715E+01, 0.12825E+01,
     & 0.12934E+01, 0.13043E+01, 0.13150E+01, 0.13257E+01, 0.13362E+01,
     & 0.13467E+01, 0.13570E+01, 0.13673E+01, 0.13775E+01, 0.13876E+01,
     & 0.13976E+01, 0.14075E+01, 0.14174E+01, 0.14272E+01, 0.14369E+01,
     & 0.14465E+01, 0.14560E+01, 0.14655E+01, 0.14749E+01, 0.14842E+01,
     & 0.14935E+01, 0.15027E+01, 0.15118E+01, 0.15208E+01, 0.15298E+01,
     & 0.15387E+01, 0.15476E+01, 0.15563E+01, 0.15651E+01, 0.15737E+01,
     & 0.15823E+01, 0.15908E+01, 0.15993E+01, 0.16077E+01, 0.16161E+01,
     & 0.16244E+01, 0.16326E+01, 0.16408E+01, 0.16489E+01, 0.16570E+01,
     & 0.16650E+01, 0.16730E+01, 0.16809E+01, 0.16888E+01, 0.16966E+01,
     & 0.17044E+01, 0.17121E+01, 0.17197E+01, 0.17274E+01, 0.17349E+01,
     & 0.17424E+01, 0.17499E+01, 0.17574E+01, 0.17647E+01, 0.17721E+01,
     & 0.17794E+01, 0.17866E+01, 0.17938E+01, 0.18010E+01, 0.18081E+01,
     & 0.18152E+01, 0.18222E+01, 0.18292E+01, 0.18362E+01, 0.18431E+01,
     & 0.18500E+01, 0.18568E+01, 0.18636E+01, 0.18704E+01, 0.18771E+01,
     & 0.18838E+01, 0.18905E+01, 0.18971E+01, 0.19037E+01, 0.19102E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19167E+01, 0.19232E+01, 0.19296E+01, 0.19360E+01, 0.19424E+01,
     & 0.19488E+01, 0.19551E+01, 0.19613E+01, 0.19676E+01, 0.19738E+01,
     & 0.19800E+01, 0.19861E+01, 0.19922E+01, 0.19983E+01, 0.20044E+01,
     & 0.20104E+01, 0.20164E+01, 0.20223E+01, 0.20283E+01, 0.20342E+01,
     & 0.20401E+01, 0.20459E+01, 0.20517E+01, 0.20575E+01, 0.20633E+01,
     & 0.20690E+01, 0.20747E+01, 0.20804E+01, 0.20861E+01, 0.20917E+01,
     & 0.20973E+01, 0.21029E+01, 0.21084E+01, 0.21140E+01, 0.21195E+01,
     & 0.21249E+01, 0.21304E+01, 0.21358E+01, 0.21412E+01, 0.21466E+01,
     & 0.21520E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.45175E-01,-0.93795E-01,-0.11560E+00,-0.12968E+00,-0.13980E+00,
     &-0.14745E+00,-0.15341E+00,-0.15813E+00,-0.16190E+00,-0.16492E+00,
     &-0.16733E+00,-0.16923E+00,-0.17072E+00,-0.17184E+00,-0.17265E+00,
     &-0.17319E+00,-0.17350E+00,-0.17360E+00,-0.17351E+00,-0.17326E+00,
     &-0.17286E+00,-0.17233E+00,-0.17167E+00,-0.17091E+00,-0.17006E+00,
     &-0.16911E+00,-0.16808E+00,-0.16698E+00,-0.16581E+00,-0.16458E+00,
     &-0.16330E+00,-0.16196E+00,-0.16057E+00,-0.15915E+00,-0.15768E+00,
     &-0.15618E+00,-0.15465E+00,-0.15309E+00,-0.15150E+00,-0.14989E+00,
     &-0.14825E+00,-0.14660E+00,-0.14492E+00,-0.14323E+00,-0.14152E+00,
     &-0.13980E+00,-0.13807E+00,-0.13633E+00,-0.13457E+00,-0.13281E+00,
     &-0.13104E+00,-0.12926E+00,-0.12747E+00,-0.12567E+00,-0.12387E+00,
     &-0.12206E+00,-0.12025E+00,-0.11843E+00,-0.11660E+00,-0.11477E+00,
     &-0.11294E+00,-0.11109E+00,-0.10924E+00,-0.10739E+00,-0.10552E+00,
     &-0.10365E+00,-0.10178E+00,-0.99891E-01,-0.97998E-01,-0.96096E-01,
     &-0.94186E-01,-0.92266E-01,-0.90336E-01,-0.88396E-01,-0.86445E-01,
     &-0.84484E-01,-0.82510E-01,-0.80525E-01,-0.78527E-01,-0.76516E-01,
     &-0.74493E-01,-0.72456E-01,-0.70405E-01,-0.68341E-01,-0.66263E-01,
     &-0.64170E-01,-0.62063E-01,-0.59942E-01,-0.57806E-01,-0.55656E-01,
     &-0.53491E-01,-0.51312E-01,-0.49119E-01,-0.46912E-01,-0.44691E-01,
     &-0.42457E-01,-0.40209E-01,-0.37948E-01,-0.35675E-01,-0.33389E-01/

      DATA (BNC10M(I),I=101,200)/
     &-0.31092E-01,-0.28783E-01,-0.26463E-01,-0.24132E-01,-0.21791E-01,
     &-0.19441E-01,-0.17081E-01,-0.14713E-01,-0.12336E-01,-0.99517E-02,
     &-0.75599E-02,-0.51613E-02,-0.27563E-02,-0.34542E-03, 0.20709E-02,
     & 0.44922E-02, 0.69181E-02, 0.93482E-02, 0.11782E-01, 0.14219E-01,
     & 0.16659E-01, 0.19102E-01, 0.21548E-01, 0.23995E-01, 0.26444E-01,
     & 0.28894E-01, 0.31345E-01, 0.33797E-01, 0.36250E-01, 0.38703E-01,
     & 0.41157E-01, 0.43610E-01, 0.46063E-01, 0.48516E-01, 0.50968E-01,
     & 0.53419E-01, 0.55870E-01, 0.58319E-01, 0.60768E-01, 0.63215E-01,
     & 0.65661E-01, 0.68105E-01, 0.70548E-01, 0.72990E-01, 0.75429E-01,
     & 0.77867E-01, 0.80303E-01, 0.82737E-01, 0.85169E-01, 0.87599E-01,
     & 0.90027E-01, 0.92453E-01, 0.94877E-01, 0.97298E-01, 0.99718E-01,
     & 0.10213E+00, 0.10455E+00, 0.10696E+00, 0.10937E+00, 0.11178E+00,
     & 0.11418E+00, 0.11659E+00, 0.11899E+00, 0.12138E+00, 0.12378E+00,
     & 0.12617E+00, 0.12856E+00, 0.13095E+00, 0.13333E+00, 0.13571E+00,
     & 0.13809E+00, 0.14047E+00, 0.14284E+00, 0.14521E+00, 0.14758E+00,
     & 0.14995E+00, 0.15231E+00, 0.15467E+00, 0.15702E+00, 0.15938E+00,
     & 0.16173E+00, 0.16408E+00, 0.16642E+00, 0.16876E+00, 0.17110E+00,
     & 0.17344E+00, 0.17577E+00, 0.17810E+00, 0.18043E+00, 0.18275E+00,
     & 0.18508E+00, 0.18739E+00, 0.18971E+00, 0.19202E+00, 0.19433E+00,
     & 0.19664E+00, 0.19894E+00, 0.20124E+00, 0.20354E+00, 0.20583E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.20812E+00, 0.21041E+00, 0.21269E+00, 0.21497E+00, 0.21725E+00,
     & 0.21953E+00, 0.22180E+00, 0.22407E+00, 0.22633E+00, 0.22860E+00,
     & 0.23086E+00, 0.23311E+00, 0.23537E+00, 0.23762E+00, 0.23986E+00,
     & 0.24211E+00, 0.24435E+00, 0.24658E+00, 0.24882E+00, 0.25105E+00,
     & 0.25328E+00, 0.25550E+00, 0.25772E+00, 0.25994E+00, 0.26216E+00,
     & 0.26437E+00, 0.26658E+00, 0.26878E+00, 0.27099E+00, 0.27319E+00,
     & 0.27538E+00, 0.27757E+00, 0.27976E+00, 0.28195E+00, 0.28413E+00,
     & 0.28631E+00, 0.28849E+00, 0.29066E+00, 0.29284E+00, 0.29500E+00,
     & 0.29717E+00, 0.29933E+00, 0.30149E+00, 0.30364E+00, 0.30579E+00,
     & 0.30794E+00, 0.31009E+00, 0.31223E+00, 0.31437E+00, 0.31650E+00,
     & 0.31864E+00, 0.32077E+00, 0.32289E+00, 0.32501E+00, 0.32713E+00,
     & 0.32925E+00, 0.33137E+00, 0.33348E+00, 0.33558E+00, 0.33769E+00,
     & 0.33979E+00, 0.34189E+00, 0.34398E+00, 0.34607E+00, 0.34816E+00,
     & 0.35025E+00, 0.35233E+00, 0.35441E+00, 0.35649E+00, 0.35856E+00,
     & 0.36063E+00, 0.36270E+00, 0.36476E+00, 0.36682E+00, 0.36888E+00,
     & 0.37093E+00, 0.37299E+00, 0.37503E+00, 0.37708E+00, 0.37912E+00,
     & 0.38116E+00, 0.38320E+00, 0.38523E+00, 0.38726E+00, 0.38929E+00,
     & 0.39131E+00, 0.39333E+00, 0.39535E+00, 0.39737E+00, 0.39938E+00,
     & 0.40139E+00, 0.40339E+00, 0.40540E+00, 0.40740E+00, 0.40940E+00,
     & 0.41139E+00, 0.41338E+00, 0.41537E+00, 0.41735E+00, 0.41934E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.42132E+00, 0.42329E+00, 0.42527E+00, 0.42724E+00, 0.42920E+00,
     & 0.43117E+00, 0.43313E+00, 0.43509E+00, 0.43705E+00, 0.43900E+00,
     & 0.44095E+00, 0.44290E+00, 0.44484E+00, 0.44678E+00, 0.44872E+00,
     & 0.45066E+00, 0.45259E+00, 0.45452E+00, 0.45645E+00, 0.45837E+00,
     & 0.46029E+00, 0.46221E+00, 0.46413E+00, 0.46604E+00, 0.46795E+00,
     & 0.46986E+00, 0.47176E+00, 0.47366E+00, 0.47556E+00, 0.47746E+00,
     & 0.47935E+00, 0.48124E+00, 0.48313E+00, 0.48502E+00, 0.48690E+00,
     & 0.48878E+00, 0.49065E+00, 0.49253E+00, 0.49440E+00, 0.49627E+00,
     & 0.49813E+00, 0.50000E+00, 0.50186E+00, 0.50371E+00, 0.50557E+00,
     & 0.50742E+00, 0.50927E+00, 0.51112E+00, 0.51296E+00, 0.51480E+00,
     & 0.51664E+00, 0.51848E+00, 0.52031E+00, 0.52214E+00, 0.52397E+00,
     & 0.52580E+00, 0.52762E+00, 0.52944E+00, 0.53126E+00, 0.53307E+00,
     & 0.53488E+00, 0.53669E+00, 0.53850E+00, 0.54031E+00, 0.54211E+00,
     & 0.54391E+00, 0.54570E+00, 0.54750E+00, 0.54929E+00, 0.55108E+00,
     & 0.55286E+00, 0.55465E+00, 0.55643E+00, 0.55821E+00, 0.55999E+00,
     & 0.56176E+00, 0.56353E+00, 0.56530E+00, 0.56707E+00, 0.56883E+00,
     & 0.57059E+00, 0.57235E+00, 0.57411E+00, 0.57586E+00, 0.57761E+00,
     & 0.57936E+00, 0.58111E+00, 0.58285E+00, 0.58459E+00, 0.58633E+00,
     & 0.58807E+00, 0.58980E+00, 0.59153E+00, 0.59326E+00, 0.59499E+00,
     & 0.59671E+00, 0.59843E+00, 0.60015E+00, 0.60187E+00, 0.60359E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.60530E+00, 0.60701E+00, 0.60872E+00, 0.61042E+00, 0.61212E+00,
     & 0.61383E+00, 0.61552E+00, 0.61722E+00, 0.61891E+00, 0.62060E+00,
     & 0.62229E+00, 0.62398E+00, 0.62566E+00, 0.62734E+00, 0.62902E+00,
     & 0.63070E+00, 0.63238E+00, 0.63405E+00, 0.63572E+00, 0.63739E+00,
     & 0.63905E+00, 0.64072E+00, 0.64238E+00, 0.64404E+00, 0.64569E+00,
     & 0.64735E+00, 0.64900E+00, 0.65065E+00, 0.65230E+00, 0.65394E+00,
     & 0.65559E+00, 0.65723E+00, 0.65887E+00, 0.66050E+00, 0.66214E+00,
     & 0.66377E+00, 0.66540E+00, 0.66703E+00, 0.66865E+00, 0.67027E+00,
     & 0.67190E+00, 0.67351E+00, 0.67513E+00, 0.67675E+00, 0.67836E+00,
     & 0.67997E+00, 0.68158E+00, 0.68318E+00, 0.68479E+00, 0.68639E+00,
     & 0.68799E+00, 0.68959E+00, 0.69118E+00, 0.69277E+00, 0.69437E+00,
     & 0.69595E+00, 0.69754E+00, 0.69913E+00, 0.70071E+00, 0.70229E+00,
     & 0.70387E+00, 0.70545E+00, 0.70702E+00, 0.70859E+00, 0.71016E+00,
     & 0.71173E+00, 0.71330E+00, 0.71486E+00, 0.71642E+00, 0.71798E+00,
     & 0.71954E+00, 0.72110E+00, 0.72265E+00, 0.72420E+00, 0.72575E+00,
     & 0.72730E+00, 0.72885E+00, 0.73039E+00, 0.73193E+00, 0.73347E+00,
     & 0.73501E+00, 0.73655E+00, 0.73808E+00, 0.73961E+00, 0.74114E+00,
     & 0.74267E+00, 0.74420E+00, 0.74572E+00, 0.74724E+00, 0.74876E+00,
     & 0.75028E+00, 0.75180E+00, 0.75331E+00, 0.75483E+00, 0.75634E+00,
     & 0.75784E+00, 0.75935E+00, 0.76086E+00, 0.76236E+00, 0.76386E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.76536E+00, 0.76686E+00, 0.76835E+00, 0.76985E+00, 0.77134E+00,
     & 0.77283E+00, 0.77431E+00, 0.77580E+00, 0.77728E+00, 0.77877E+00,
     & 0.78025E+00, 0.78173E+00, 0.78320E+00, 0.78468E+00, 0.78615E+00,
     & 0.78762E+00, 0.78909E+00, 0.79056E+00, 0.79202E+00, 0.79349E+00,
     & 0.79495E+00, 0.79641E+00, 0.79787E+00, 0.79933E+00, 0.80078E+00,
     & 0.80223E+00, 0.80368E+00, 0.80513E+00, 0.80658E+00, 0.80803E+00,
     & 0.80947E+00, 0.81091E+00, 0.81235E+00, 0.81379E+00, 0.81523E+00,
     & 0.81667E+00, 0.81810E+00, 0.81953E+00, 0.82096E+00, 0.82239E+00,
     & 0.82382E+00, 0.82524E+00, 0.82666E+00, 0.82809E+00, 0.82951E+00,
     & 0.83092E+00, 0.83234E+00, 0.83375E+00, 0.83517E+00, 0.83658E+00,
     & 0.83799E+00, 0.83940E+00, 0.84080E+00, 0.84221E+00, 0.84361E+00,
     & 0.84501E+00, 0.84641E+00, 0.84781E+00, 0.84920E+00, 0.85060E+00,
     & 0.85199E+00, 0.85338E+00, 0.85477E+00, 0.85616E+00, 0.85754E+00,
     & 0.85893E+00, 0.86031E+00, 0.86169E+00, 0.86307E+00, 0.86445E+00,
     & 0.86583E+00, 0.86720E+00, 0.86857E+00, 0.86995E+00, 0.87131E+00,
     & 0.87268E+00, 0.87405E+00, 0.87541E+00, 0.87678E+00, 0.87814E+00,
     & 0.87950E+00, 0.88086E+00, 0.88222E+00, 0.88357E+00, 0.88493E+00,
     & 0.88628E+00, 0.88763E+00, 0.88898E+00, 0.89033E+00, 0.89167E+00,
     & 0.89302E+00, 0.89436E+00, 0.89570E+00, 0.89704E+00, 0.89838E+00,
     & 0.89972E+00, 0.90105E+00, 0.90238E+00, 0.90372E+00, 0.90505E+00/

      DATA (BNC10M(I),I=601,700)/
     & 0.91933E+00, 0.93239E+00, 0.94531E+00, 0.95808E+00, 0.97072E+00,
     & 0.98322E+00, 0.99558E+00, 0.10078E+01, 0.10199E+01, 0.10319E+01,
     & 0.10438E+01, 0.10555E+01, 0.10671E+01, 0.10786E+01, 0.10900E+01,
     & 0.11013E+01, 0.11124E+01, 0.11235E+01, 0.11344E+01, 0.11453E+01,
     & 0.11560E+01, 0.11667E+01, 0.11772E+01, 0.11876E+01, 0.11980E+01,
     & 0.12082E+01, 0.12184E+01, 0.12285E+01, 0.12385E+01, 0.12484E+01,
     & 0.12582E+01, 0.12679E+01, 0.12776E+01, 0.12871E+01, 0.12966E+01,
     & 0.13060E+01, 0.13154E+01, 0.13246E+01, 0.13338E+01, 0.13429E+01,
     & 0.13519E+01, 0.13609E+01, 0.13698E+01, 0.13786E+01, 0.13874E+01,
     & 0.13960E+01, 0.14047E+01, 0.14132E+01, 0.14217E+01, 0.14301E+01,
     & 0.14385E+01, 0.14468E+01, 0.14550E+01, 0.14632E+01, 0.14713E+01,
     & 0.14794E+01, 0.14874E+01, 0.14954E+01, 0.15033E+01, 0.15111E+01,
     & 0.15189E+01, 0.15266E+01, 0.15343E+01, 0.15419E+01, 0.15495E+01,
     & 0.15570E+01, 0.15645E+01, 0.15719E+01, 0.15793E+01, 0.15866E+01,
     & 0.15939E+01, 0.16012E+01, 0.16084E+01, 0.16155E+01, 0.16226E+01,
     & 0.16297E+01, 0.16367E+01, 0.16436E+01, 0.16506E+01, 0.16574E+01,
     & 0.16643E+01, 0.16711E+01, 0.16778E+01, 0.16845E+01, 0.16912E+01,
     & 0.16979E+01, 0.17045E+01, 0.17110E+01, 0.17175E+01, 0.17240E+01,
     & 0.17305E+01, 0.17369E+01, 0.17433E+01, 0.17496E+01, 0.17559E+01,
     & 0.17622E+01, 0.17684E+01, 0.17746E+01, 0.17808E+01, 0.17869E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17930E+01, 0.17991E+01, 0.18051E+01, 0.18111E+01, 0.18171E+01,
     & 0.18230E+01, 0.18289E+01, 0.18348E+01, 0.18406E+01, 0.18464E+01,
     & 0.18522E+01, 0.18580E+01, 0.18637E+01, 0.18694E+01, 0.18751E+01,
     & 0.18807E+01, 0.18863E+01, 0.18919E+01, 0.18974E+01, 0.19030E+01,
     & 0.19085E+01, 0.19139E+01, 0.19194E+01, 0.19248E+01, 0.19302E+01,
     & 0.19356E+01, 0.19409E+01, 0.19462E+01, 0.19515E+01, 0.19568E+01,
     & 0.19620E+01, 0.19672E+01, 0.19724E+01, 0.19776E+01, 0.19828E+01,
     & 0.19879E+01, 0.19930E+01, 0.19981E+01, 0.20031E+01, 0.20081E+01,
     & 0.20131E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.44136E-01,-0.88680E-01,-0.10706E+00,-0.11800E+00,-0.12513E+00,
     &-0.12990E+00,-0.13304E+00,-0.13497E+00,-0.13598E+00,-0.13624E+00,
     &-0.13589E+00,-0.13503E+00,-0.13373E+00,-0.13205E+00,-0.13004E+00,
     &-0.12774E+00,-0.12517E+00,-0.12237E+00,-0.11935E+00,-0.11614E+00,
     &-0.11275E+00,-0.10920E+00,-0.10550E+00,-0.10166E+00,-0.97690E-01,
     &-0.93602E-01,-0.89404E-01,-0.85103E-01,-0.80706E-01,-0.76221E-01,
     &-0.71651E-01,-0.67004E-01,-0.62284E-01,-0.57497E-01,-0.52646E-01,
     &-0.47736E-01,-0.42771E-01,-0.37754E-01,-0.32690E-01,-0.27580E-01,
     &-0.22429E-01,-0.17239E-01,-0.12013E-01,-0.67523E-02,-0.14605E-02,
     & 0.38607E-02, 0.92093E-02, 0.14584E-01, 0.19982E-01, 0.25402E-01,
     & 0.30844E-01, 0.36306E-01, 0.41786E-01, 0.47284E-01, 0.52800E-01,
     & 0.58332E-01, 0.63879E-01, 0.69443E-01, 0.75021E-01, 0.80615E-01,
     & 0.86223E-01, 0.91847E-01, 0.97486E-01, 0.10314E+00, 0.10881E+00,
     & 0.11450E+00, 0.12020E+00, 0.12592E+00, 0.13166E+00, 0.13742E+00,
     & 0.14320E+00, 0.14899E+00, 0.15481E+00, 0.16065E+00, 0.16652E+00,
     & 0.17241E+00, 0.17832E+00, 0.18426E+00, 0.19022E+00, 0.19621E+00,
     & 0.20223E+00, 0.20827E+00, 0.21435E+00, 0.22045E+00, 0.22659E+00,
     & 0.23275E+00, 0.23894E+00, 0.24517E+00, 0.25142E+00, 0.25771E+00,
     & 0.26402E+00, 0.27037E+00, 0.27674E+00, 0.28315E+00, 0.28958E+00,
     & 0.29604E+00, 0.30253E+00, 0.30905E+00, 0.31559E+00, 0.32216E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.32875E+00, 0.33537E+00, 0.34201E+00, 0.34867E+00, 0.35535E+00,
     & 0.36205E+00, 0.36877E+00, 0.37550E+00, 0.38225E+00, 0.38902E+00,
     & 0.39580E+00, 0.40259E+00, 0.40939E+00, 0.41620E+00, 0.42302E+00,
     & 0.42985E+00, 0.43669E+00, 0.44353E+00, 0.45037E+00, 0.45722E+00,
     & 0.46408E+00, 0.47093E+00, 0.47779E+00, 0.48464E+00, 0.49150E+00,
     & 0.49835E+00, 0.50520E+00, 0.51205E+00, 0.51890E+00, 0.52574E+00,
     & 0.53258E+00, 0.53941E+00, 0.54624E+00, 0.55306E+00, 0.55988E+00,
     & 0.56669E+00, 0.57349E+00, 0.58029E+00, 0.58708E+00, 0.59386E+00,
     & 0.60063E+00, 0.60739E+00, 0.61415E+00, 0.62089E+00, 0.62763E+00,
     & 0.63436E+00, 0.64108E+00, 0.64779E+00, 0.65449E+00, 0.66118E+00,
     & 0.66786E+00, 0.67453E+00, 0.68118E+00, 0.68783E+00, 0.69447E+00,
     & 0.70110E+00, 0.70772E+00, 0.71432E+00, 0.72092E+00, 0.72750E+00,
     & 0.73408E+00, 0.74064E+00, 0.74719E+00, 0.75373E+00, 0.76026E+00,
     & 0.76678E+00, 0.77329E+00, 0.77978E+00, 0.78627E+00, 0.79274E+00,
     & 0.79920E+00, 0.80565E+00, 0.81209E+00, 0.81852E+00, 0.82493E+00,
     & 0.83134E+00, 0.83773E+00, 0.84411E+00, 0.85048E+00, 0.85683E+00,
     & 0.86318E+00, 0.86951E+00, 0.87583E+00, 0.88214E+00, 0.88844E+00,
     & 0.89473E+00, 0.90100E+00, 0.90726E+00, 0.91352E+00, 0.91975E+00,
     & 0.92598E+00, 0.93220E+00, 0.93840E+00, 0.94459E+00, 0.95077E+00,
     & 0.95694E+00, 0.96310E+00, 0.96924E+00, 0.97537E+00, 0.98149E+00/

      DATA (BNC11M(I),I=201,300)/
     & 0.98760E+00, 0.99370E+00, 0.99978E+00, 0.10059E+01, 0.10119E+01,
     & 0.10180E+01, 0.10240E+01, 0.10300E+01, 0.10360E+01, 0.10421E+01,
     & 0.10480E+01, 0.10540E+01, 0.10600E+01, 0.10659E+01, 0.10719E+01,
     & 0.10778E+01, 0.10837E+01, 0.10897E+01, 0.10956E+01, 0.11014E+01,
     & 0.11073E+01, 0.11132E+01, 0.11190E+01, 0.11249E+01, 0.11307E+01,
     & 0.11365E+01, 0.11423E+01, 0.11481E+01, 0.11539E+01, 0.11597E+01,
     & 0.11654E+01, 0.11712E+01, 0.11769E+01, 0.11827E+01, 0.11884E+01,
     & 0.11941E+01, 0.11998E+01, 0.12054E+01, 0.12111E+01, 0.12168E+01,
     & 0.12224E+01, 0.12281E+01, 0.12337E+01, 0.12393E+01, 0.12449E+01,
     & 0.12505E+01, 0.12561E+01, 0.12617E+01, 0.12672E+01, 0.12728E+01,
     & 0.12783E+01, 0.12838E+01, 0.12894E+01, 0.12949E+01, 0.13004E+01,
     & 0.13059E+01, 0.13113E+01, 0.13168E+01, 0.13223E+01, 0.13277E+01,
     & 0.13331E+01, 0.13386E+01, 0.13440E+01, 0.13494E+01, 0.13548E+01,
     & 0.13601E+01, 0.13655E+01, 0.13709E+01, 0.13762E+01, 0.13816E+01,
     & 0.13869E+01, 0.13922E+01, 0.13975E+01, 0.14028E+01, 0.14081E+01,
     & 0.14134E+01, 0.14187E+01, 0.14239E+01, 0.14292E+01, 0.14344E+01,
     & 0.14396E+01, 0.14449E+01, 0.14501E+01, 0.14553E+01, 0.14605E+01,
     & 0.14656E+01, 0.14708E+01, 0.14760E+01, 0.14811E+01, 0.14863E+01,
     & 0.14914E+01, 0.14965E+01, 0.15016E+01, 0.15067E+01, 0.15118E+01,
     & 0.15169E+01, 0.15220E+01, 0.15271E+01, 0.15321E+01, 0.15372E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.15422E+01, 0.15472E+01, 0.15523E+01, 0.15573E+01, 0.15623E+01,
     & 0.15673E+01, 0.15722E+01, 0.15772E+01, 0.15822E+01, 0.15871E+01,
     & 0.15921E+01, 0.15970E+01, 0.16019E+01, 0.16069E+01, 0.16118E+01,
     & 0.16167E+01, 0.16216E+01, 0.16265E+01, 0.16313E+01, 0.16362E+01,
     & 0.16410E+01, 0.16459E+01, 0.16507E+01, 0.16556E+01, 0.16604E+01,
     & 0.16652E+01, 0.16700E+01, 0.16748E+01, 0.16796E+01, 0.16844E+01,
     & 0.16891E+01, 0.16939E+01, 0.16987E+01, 0.17034E+01, 0.17081E+01,
     & 0.17129E+01, 0.17176E+01, 0.17223E+01, 0.17270E+01, 0.17317E+01,
     & 0.17364E+01, 0.17411E+01, 0.17457E+01, 0.17504E+01, 0.17551E+01,
     & 0.17597E+01, 0.17644E+01, 0.17690E+01, 0.17736E+01, 0.17782E+01,
     & 0.17828E+01, 0.17874E+01, 0.17920E+01, 0.17966E+01, 0.18012E+01,
     & 0.18057E+01, 0.18103E+01, 0.18149E+01, 0.18194E+01, 0.18239E+01,
     & 0.18285E+01, 0.18330E+01, 0.18375E+01, 0.18420E+01, 0.18465E+01,
     & 0.18510E+01, 0.18555E+01, 0.18600E+01, 0.18644E+01, 0.18689E+01,
     & 0.18733E+01, 0.18778E+01, 0.18822E+01, 0.18867E+01, 0.18911E+01,
     & 0.18955E+01, 0.18999E+01, 0.19043E+01, 0.19087E+01, 0.19131E+01,
     & 0.19175E+01, 0.19218E+01, 0.19262E+01, 0.19306E+01, 0.19349E+01,
     & 0.19393E+01, 0.19436E+01, 0.19479E+01, 0.19522E+01, 0.19566E+01,
     & 0.19609E+01, 0.19652E+01, 0.19695E+01, 0.19737E+01, 0.19780E+01,
     & 0.19823E+01, 0.19866E+01, 0.19908E+01, 0.19951E+01, 0.19993E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.20036E+01, 0.20078E+01, 0.20120E+01, 0.20162E+01, 0.20204E+01,
     & 0.20247E+01, 0.20289E+01, 0.20330E+01, 0.20372E+01, 0.20414E+01,
     & 0.20456E+01, 0.20497E+01, 0.20539E+01, 0.20581E+01, 0.20622E+01,
     & 0.20663E+01, 0.20705E+01, 0.20746E+01, 0.20787E+01, 0.20828E+01,
     & 0.20869E+01, 0.20910E+01, 0.20951E+01, 0.20992E+01, 0.21033E+01,
     & 0.21074E+01, 0.21114E+01, 0.21155E+01, 0.21196E+01, 0.21236E+01,
     & 0.21277E+01, 0.21317E+01, 0.21357E+01, 0.21398E+01, 0.21438E+01,
     & 0.21478E+01, 0.21518E+01, 0.21558E+01, 0.21598E+01, 0.21638E+01,
     & 0.21678E+01, 0.21717E+01, 0.21757E+01, 0.21797E+01, 0.21836E+01,
     & 0.21876E+01, 0.21915E+01, 0.21955E+01, 0.21994E+01, 0.22033E+01,
     & 0.22073E+01, 0.22112E+01, 0.22151E+01, 0.22190E+01, 0.22229E+01,
     & 0.22268E+01, 0.22307E+01, 0.22346E+01, 0.22384E+01, 0.22423E+01,
     & 0.22462E+01, 0.22500E+01, 0.22539E+01, 0.22577E+01, 0.22616E+01,
     & 0.22654E+01, 0.22692E+01, 0.22731E+01, 0.22769E+01, 0.22807E+01,
     & 0.22845E+01, 0.22883E+01, 0.22921E+01, 0.22959E+01, 0.22997E+01,
     & 0.23035E+01, 0.23072E+01, 0.23110E+01, 0.23148E+01, 0.23185E+01,
     & 0.23223E+01, 0.23260E+01, 0.23298E+01, 0.23335E+01, 0.23373E+01,
     & 0.23410E+01, 0.23447E+01, 0.23484E+01, 0.23521E+01, 0.23558E+01,
     & 0.23595E+01, 0.23632E+01, 0.23669E+01, 0.23706E+01, 0.23743E+01,
     & 0.23780E+01, 0.23816E+01, 0.23853E+01, 0.23890E+01, 0.23926E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.23963E+01, 0.23999E+01, 0.24036E+01, 0.24072E+01, 0.24108E+01,
     & 0.24144E+01, 0.24181E+01, 0.24217E+01, 0.24253E+01, 0.24289E+01,
     & 0.24325E+01, 0.24361E+01, 0.24397E+01, 0.24433E+01, 0.24468E+01,
     & 0.24504E+01, 0.24540E+01, 0.24575E+01, 0.24611E+01, 0.24647E+01,
     & 0.24682E+01, 0.24718E+01, 0.24753E+01, 0.24788E+01, 0.24824E+01,
     & 0.24859E+01, 0.24894E+01, 0.24929E+01, 0.24964E+01, 0.24999E+01,
     & 0.25034E+01, 0.25069E+01, 0.25104E+01, 0.25139E+01, 0.25174E+01,
     & 0.25209E+01, 0.25244E+01, 0.25278E+01, 0.25313E+01, 0.25348E+01,
     & 0.25382E+01, 0.25417E+01, 0.25451E+01, 0.25486E+01, 0.25520E+01,
     & 0.25554E+01, 0.25589E+01, 0.25623E+01, 0.25657E+01, 0.25691E+01,
     & 0.25725E+01, 0.25759E+01, 0.25793E+01, 0.25827E+01, 0.25861E+01,
     & 0.25895E+01, 0.25929E+01, 0.25963E+01, 0.25997E+01, 0.26030E+01,
     & 0.26064E+01, 0.26098E+01, 0.26131E+01, 0.26165E+01, 0.26198E+01,
     & 0.26232E+01, 0.26265E+01, 0.26298E+01, 0.26332E+01, 0.26365E+01,
     & 0.26398E+01, 0.26431E+01, 0.26465E+01, 0.26498E+01, 0.26531E+01,
     & 0.26564E+01, 0.26597E+01, 0.26630E+01, 0.26663E+01, 0.26696E+01,
     & 0.26728E+01, 0.26761E+01, 0.26794E+01, 0.26827E+01, 0.26859E+01,
     & 0.26892E+01, 0.26924E+01, 0.26957E+01, 0.26989E+01, 0.27022E+01,
     & 0.27054E+01, 0.27087E+01, 0.27119E+01, 0.27151E+01, 0.27184E+01,
     & 0.27216E+01, 0.27248E+01, 0.27280E+01, 0.27312E+01, 0.27344E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.27688E+01, 0.28002E+01, 0.28313E+01, 0.28620E+01, 0.28923E+01,
     & 0.29223E+01, 0.29519E+01, 0.29812E+01, 0.30102E+01, 0.30388E+01,
     & 0.30672E+01, 0.30952E+01, 0.31229E+01, 0.31504E+01, 0.31775E+01,
     & 0.32044E+01, 0.32310E+01, 0.32573E+01, 0.32834E+01, 0.33092E+01,
     & 0.33347E+01, 0.33601E+01, 0.33851E+01, 0.34099E+01, 0.34345E+01,
     & 0.34589E+01, 0.34830E+01, 0.35069E+01, 0.35306E+01, 0.35541E+01,
     & 0.35773E+01, 0.36004E+01, 0.36232E+01, 0.36459E+01, 0.36683E+01,
     & 0.36906E+01, 0.37127E+01, 0.37346E+01, 0.37563E+01, 0.37778E+01,
     & 0.37992E+01, 0.38203E+01, 0.38413E+01, 0.38622E+01, 0.38828E+01,
     & 0.39033E+01, 0.39237E+01, 0.39439E+01, 0.39639E+01, 0.39838E+01,
     & 0.40035E+01, 0.40231E+01, 0.40425E+01, 0.40618E+01, 0.40809E+01,
     & 0.40999E+01, 0.41188E+01, 0.41375E+01, 0.41561E+01, 0.41746E+01,
     & 0.41929E+01, 0.42111E+01, 0.42292E+01, 0.42472E+01, 0.42650E+01,
     & 0.42827E+01, 0.43003E+01, 0.43177E+01, 0.43351E+01, 0.43523E+01,
     & 0.43695E+01, 0.43865E+01, 0.44034E+01, 0.44202E+01, 0.44368E+01,
     & 0.44534E+01, 0.44699E+01, 0.44862E+01, 0.45025E+01, 0.45187E+01,
     & 0.45347E+01, 0.45507E+01, 0.45666E+01, 0.45823E+01, 0.45980E+01,
     & 0.46136E+01, 0.46291E+01, 0.46445E+01, 0.46598E+01, 0.46750E+01,
     & 0.46901E+01, 0.47051E+01, 0.47201E+01, 0.47350E+01, 0.47497E+01,
     & 0.47644E+01, 0.47791E+01, 0.47936E+01, 0.48080E+01, 0.48224E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.48367E+01, 0.48509E+01, 0.48651E+01, 0.48791E+01, 0.48931E+01,
     & 0.49070E+01, 0.49208E+01, 0.49346E+01, 0.49483E+01, 0.49619E+01,
     & 0.49755E+01, 0.49889E+01, 0.50023E+01, 0.50157E+01, 0.50290E+01,
     & 0.50422E+01, 0.50553E+01, 0.50684E+01, 0.50814E+01, 0.50943E+01,
     & 0.51072E+01, 0.51200E+01, 0.51327E+01, 0.51454E+01, 0.51580E+01,
     & 0.51706E+01, 0.51831E+01, 0.51956E+01, 0.52079E+01, 0.52203E+01,
     & 0.52325E+01, 0.52447E+01, 0.52569E+01, 0.52690E+01, 0.52810E+01,
     & 0.52930E+01, 0.53049E+01, 0.53168E+01, 0.53286E+01, 0.53404E+01,
     & 0.53521E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.45034E-01,-0.93528E-01,-0.11552E+00,-0.12992E+00,-0.14041E+00,
     &-0.14847E+00,-0.15486E+00,-0.16000E+00,-0.16419E+00,-0.16761E+00,
     &-0.17041E+00,-0.17267E+00,-0.17449E+00,-0.17591E+00,-0.17699E+00,
     &-0.17776E+00,-0.17826E+00,-0.17851E+00,-0.17853E+00,-0.17835E+00,
     &-0.17798E+00,-0.17743E+00,-0.17671E+00,-0.17585E+00,-0.17484E+00,
     &-0.17369E+00,-0.17241E+00,-0.17102E+00,-0.16951E+00,-0.16790E+00,
     &-0.16618E+00,-0.16436E+00,-0.16245E+00,-0.16045E+00,-0.15836E+00,
     &-0.15620E+00,-0.15396E+00,-0.15164E+00,-0.14925E+00,-0.14680E+00,
     &-0.14428E+00,-0.14170E+00,-0.13905E+00,-0.13635E+00,-0.13360E+00,
     &-0.13079E+00,-0.12794E+00,-0.12503E+00,-0.12208E+00,-0.11908E+00,
     &-0.11604E+00,-0.11296E+00,-0.10984E+00,-0.10667E+00,-0.10348E+00,
     &-0.10024E+00,-0.96972E-01,-0.93669E-01,-0.90333E-01,-0.86964E-01,
     &-0.83564E-01,-0.80133E-01,-0.76672E-01,-0.73181E-01,-0.69661E-01,
     &-0.66113E-01,-0.62536E-01,-0.58930E-01,-0.55297E-01,-0.51636E-01,
     &-0.47947E-01,-0.44230E-01,-0.40486E-01,-0.36714E-01,-0.32914E-01,
     &-0.29086E-01,-0.25231E-01,-0.21348E-01,-0.17437E-01,-0.13498E-01,
     &-0.95308E-02,-0.55358E-02,-0.15127E-02, 0.25383E-02, 0.66171E-02,
     & 0.10724E-01, 0.14858E-01, 0.19019E-01, 0.23208E-01, 0.27423E-01,
     & 0.31665E-01, 0.35933E-01, 0.40227E-01, 0.44545E-01, 0.48888E-01,
     & 0.53254E-01, 0.57644E-01, 0.62056E-01, 0.66490E-01, 0.70945E-01/

      DATA (BNC12M(I),I=101,200)/
     & 0.75420E-01, 0.79914E-01, 0.84426E-01, 0.88956E-01, 0.93502E-01,
     & 0.98063E-01, 0.10264E+00, 0.10723E+00, 0.11183E+00, 0.11644E+00,
     & 0.12107E+00, 0.12570E+00, 0.13035E+00, 0.13500E+00, 0.13966E+00,
     & 0.14432E+00, 0.14899E+00, 0.15366E+00, 0.15834E+00, 0.16302E+00,
     & 0.16771E+00, 0.17239E+00, 0.17708E+00, 0.18176E+00, 0.18645E+00,
     & 0.19113E+00, 0.19582E+00, 0.20050E+00, 0.20518E+00, 0.20985E+00,
     & 0.21452E+00, 0.21919E+00, 0.22386E+00, 0.22852E+00, 0.23317E+00,
     & 0.23782E+00, 0.24247E+00, 0.24711E+00, 0.25174E+00, 0.25637E+00,
     & 0.26099E+00, 0.26561E+00, 0.27021E+00, 0.27481E+00, 0.27941E+00,
     & 0.28400E+00, 0.28858E+00, 0.29315E+00, 0.29771E+00, 0.30227E+00,
     & 0.30682E+00, 0.31136E+00, 0.31590E+00, 0.32043E+00, 0.32494E+00,
     & 0.32946E+00, 0.33396E+00, 0.33845E+00, 0.34294E+00, 0.34742E+00,
     & 0.35189E+00, 0.35635E+00, 0.36080E+00, 0.36525E+00, 0.36968E+00,
     & 0.37411E+00, 0.37853E+00, 0.38294E+00, 0.38734E+00, 0.39174E+00,
     & 0.39612E+00, 0.40050E+00, 0.40487E+00, 0.40923E+00, 0.41358E+00,
     & 0.41792E+00, 0.42226E+00, 0.42658E+00, 0.43090E+00, 0.43521E+00,
     & 0.43951E+00, 0.44380E+00, 0.44808E+00, 0.45236E+00, 0.45662E+00,
     & 0.46088E+00, 0.46513E+00, 0.46937E+00, 0.47360E+00, 0.47782E+00,
     & 0.48204E+00, 0.48624E+00, 0.49044E+00, 0.49463E+00, 0.49881E+00,
     & 0.50298E+00, 0.50715E+00, 0.51130E+00, 0.51545E+00, 0.51959E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.52372E+00, 0.52784E+00, 0.53195E+00, 0.53606E+00, 0.54016E+00,
     & 0.54425E+00, 0.54833E+00, 0.55240E+00, 0.55646E+00, 0.56052E+00,
     & 0.56457E+00, 0.56861E+00, 0.57264E+00, 0.57666E+00, 0.58068E+00,
     & 0.58468E+00, 0.58868E+00, 0.59267E+00, 0.59666E+00, 0.60063E+00,
     & 0.60460E+00, 0.60856E+00, 0.61251E+00, 0.61645E+00, 0.62039E+00,
     & 0.62432E+00, 0.62824E+00, 0.63215E+00, 0.63606E+00, 0.63995E+00,
     & 0.64384E+00, 0.64772E+00, 0.65160E+00, 0.65546E+00, 0.65932E+00,
     & 0.66317E+00, 0.66702E+00, 0.67085E+00, 0.67468E+00, 0.67850E+00,
     & 0.68231E+00, 0.68612E+00, 0.68992E+00, 0.69371E+00, 0.69749E+00,
     & 0.70127E+00, 0.70504E+00, 0.70880E+00, 0.71256E+00, 0.71630E+00,
     & 0.72004E+00, 0.72378E+00, 0.72750E+00, 0.73122E+00, 0.73493E+00,
     & 0.73864E+00, 0.74233E+00, 0.74602E+00, 0.74971E+00, 0.75338E+00,
     & 0.75705E+00, 0.76071E+00, 0.76437E+00, 0.76802E+00, 0.77166E+00,
     & 0.77529E+00, 0.77892E+00, 0.78254E+00, 0.78616E+00, 0.78976E+00,
     & 0.79337E+00, 0.79696E+00, 0.80055E+00, 0.80413E+00, 0.80770E+00,
     & 0.81127E+00, 0.81483E+00, 0.81838E+00, 0.82193E+00, 0.82547E+00,
     & 0.82901E+00, 0.83253E+00, 0.83605E+00, 0.83957E+00, 0.84308E+00,
     & 0.84658E+00, 0.85008E+00, 0.85357E+00, 0.85705E+00, 0.86053E+00,
     & 0.86400E+00, 0.86746E+00, 0.87092E+00, 0.87437E+00, 0.87781E+00,
     & 0.88125E+00, 0.88469E+00, 0.88811E+00, 0.89153E+00, 0.89495E+00/

      DATA (BNC12M(I),I=301,400)/
     & 0.89836E+00, 0.90176E+00, 0.90516E+00, 0.90855E+00, 0.91193E+00,
     & 0.91531E+00, 0.91868E+00, 0.92205E+00, 0.92541E+00, 0.92876E+00,
     & 0.93211E+00, 0.93545E+00, 0.93879E+00, 0.94212E+00, 0.94545E+00,
     & 0.94877E+00, 0.95208E+00, 0.95539E+00, 0.95869E+00, 0.96199E+00,
     & 0.96528E+00, 0.96856E+00, 0.97184E+00, 0.97512E+00, 0.97839E+00,
     & 0.98165E+00, 0.98491E+00, 0.98816E+00, 0.99140E+00, 0.99464E+00,
     & 0.99788E+00, 0.10011E+01, 0.10043E+01, 0.10075E+01, 0.10108E+01,
     & 0.10140E+01, 0.10172E+01, 0.10204E+01, 0.10236E+01, 0.10267E+01,
     & 0.10299E+01, 0.10331E+01, 0.10363E+01, 0.10394E+01, 0.10426E+01,
     & 0.10458E+01, 0.10489E+01, 0.10520E+01, 0.10552E+01, 0.10583E+01,
     & 0.10614E+01, 0.10646E+01, 0.10677E+01, 0.10708E+01, 0.10739E+01,
     & 0.10770E+01, 0.10801E+01, 0.10832E+01, 0.10863E+01, 0.10894E+01,
     & 0.10924E+01, 0.10955E+01, 0.10986E+01, 0.11016E+01, 0.11047E+01,
     & 0.11078E+01, 0.11108E+01, 0.11138E+01, 0.11169E+01, 0.11199E+01,
     & 0.11229E+01, 0.11260E+01, 0.11290E+01, 0.11320E+01, 0.11350E+01,
     & 0.11380E+01, 0.11410E+01, 0.11440E+01, 0.11470E+01, 0.11500E+01,
     & 0.11530E+01, 0.11559E+01, 0.11589E+01, 0.11619E+01, 0.11648E+01,
     & 0.11678E+01, 0.11708E+01, 0.11737E+01, 0.11766E+01, 0.11796E+01,
     & 0.11825E+01, 0.11854E+01, 0.11884E+01, 0.11913E+01, 0.11942E+01,
     & 0.11971E+01, 0.12000E+01, 0.12029E+01, 0.12058E+01, 0.12087E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.12116E+01, 0.12145E+01, 0.12174E+01, 0.12202E+01, 0.12231E+01,
     & 0.12260E+01, 0.12289E+01, 0.12317E+01, 0.12346E+01, 0.12374E+01,
     & 0.12403E+01, 0.12431E+01, 0.12459E+01, 0.12488E+01, 0.12516E+01,
     & 0.12544E+01, 0.12572E+01, 0.12601E+01, 0.12629E+01, 0.12657E+01,
     & 0.12685E+01, 0.12713E+01, 0.12741E+01, 0.12769E+01, 0.12797E+01,
     & 0.12824E+01, 0.12852E+01, 0.12880E+01, 0.12908E+01, 0.12935E+01,
     & 0.12963E+01, 0.12990E+01, 0.13018E+01, 0.13046E+01, 0.13073E+01,
     & 0.13100E+01, 0.13128E+01, 0.13155E+01, 0.13182E+01, 0.13210E+01,
     & 0.13237E+01, 0.13264E+01, 0.13291E+01, 0.13318E+01, 0.13345E+01,
     & 0.13372E+01, 0.13399E+01, 0.13426E+01, 0.13453E+01, 0.13480E+01,
     & 0.13507E+01, 0.13534E+01, 0.13561E+01, 0.13587E+01, 0.13614E+01,
     & 0.13641E+01, 0.13667E+01, 0.13694E+01, 0.13720E+01, 0.13747E+01,
     & 0.13773E+01, 0.13800E+01, 0.13826E+01, 0.13852E+01, 0.13879E+01,
     & 0.13905E+01, 0.13931E+01, 0.13957E+01, 0.13984E+01, 0.14010E+01,
     & 0.14036E+01, 0.14062E+01, 0.14088E+01, 0.14114E+01, 0.14140E+01,
     & 0.14166E+01, 0.14191E+01, 0.14217E+01, 0.14243E+01, 0.14269E+01,
     & 0.14295E+01, 0.14320E+01, 0.14346E+01, 0.14372E+01, 0.14397E+01,
     & 0.14423E+01, 0.14448E+01, 0.14474E+01, 0.14499E+01, 0.14525E+01,
     & 0.14550E+01, 0.14575E+01, 0.14601E+01, 0.14626E+01, 0.14651E+01,
     & 0.14676E+01, 0.14702E+01, 0.14727E+01, 0.14752E+01, 0.14777E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.14802E+01, 0.14827E+01, 0.14852E+01, 0.14877E+01, 0.14902E+01,
     & 0.14927E+01, 0.14951E+01, 0.14976E+01, 0.15001E+01, 0.15026E+01,
     & 0.15051E+01, 0.15075E+01, 0.15100E+01, 0.15124E+01, 0.15149E+01,
     & 0.15174E+01, 0.15198E+01, 0.15223E+01, 0.15247E+01, 0.15271E+01,
     & 0.15296E+01, 0.15320E+01, 0.15345E+01, 0.15369E+01, 0.15393E+01,
     & 0.15417E+01, 0.15442E+01, 0.15466E+01, 0.15490E+01, 0.15514E+01,
     & 0.15538E+01, 0.15562E+01, 0.15586E+01, 0.15610E+01, 0.15634E+01,
     & 0.15658E+01, 0.15682E+01, 0.15706E+01, 0.15729E+01, 0.15753E+01,
     & 0.15777E+01, 0.15801E+01, 0.15824E+01, 0.15848E+01, 0.15872E+01,
     & 0.15895E+01, 0.15919E+01, 0.15943E+01, 0.15966E+01, 0.15990E+01,
     & 0.16013E+01, 0.16037E+01, 0.16060E+01, 0.16083E+01, 0.16107E+01,
     & 0.16130E+01, 0.16153E+01, 0.16177E+01, 0.16200E+01, 0.16223E+01,
     & 0.16246E+01, 0.16269E+01, 0.16292E+01, 0.16316E+01, 0.16339E+01,
     & 0.16362E+01, 0.16385E+01, 0.16408E+01, 0.16431E+01, 0.16453E+01,
     & 0.16476E+01, 0.16499E+01, 0.16522E+01, 0.16545E+01, 0.16568E+01,
     & 0.16590E+01, 0.16613E+01, 0.16636E+01, 0.16658E+01, 0.16681E+01,
     & 0.16704E+01, 0.16726E+01, 0.16749E+01, 0.16771E+01, 0.16794E+01,
     & 0.16816E+01, 0.16839E+01, 0.16861E+01, 0.16884E+01, 0.16906E+01,
     & 0.16928E+01, 0.16951E+01, 0.16973E+01, 0.16995E+01, 0.17018E+01,
     & 0.17040E+01, 0.17062E+01, 0.17084E+01, 0.17106E+01, 0.17128E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.17365E+01, 0.17582E+01, 0.17797E+01, 0.18009E+01, 0.18218E+01,
     & 0.18425E+01, 0.18630E+01, 0.18833E+01, 0.19033E+01, 0.19232E+01,
     & 0.19428E+01, 0.19622E+01, 0.19814E+01, 0.20004E+01, 0.20193E+01,
     & 0.20379E+01, 0.20563E+01, 0.20746E+01, 0.20927E+01, 0.21106E+01,
     & 0.21283E+01, 0.21459E+01, 0.21633E+01, 0.21806E+01, 0.21976E+01,
     & 0.22146E+01, 0.22313E+01, 0.22480E+01, 0.22644E+01, 0.22808E+01,
     & 0.22969E+01, 0.23130E+01, 0.23289E+01, 0.23447E+01, 0.23603E+01,
     & 0.23758E+01, 0.23912E+01, 0.24064E+01, 0.24215E+01, 0.24365E+01,
     & 0.24514E+01, 0.24662E+01, 0.24808E+01, 0.24953E+01, 0.25097E+01,
     & 0.25240E+01, 0.25382E+01, 0.25523E+01, 0.25663E+01, 0.25802E+01,
     & 0.25939E+01, 0.26076E+01, 0.26211E+01, 0.26346E+01, 0.26480E+01,
     & 0.26612E+01, 0.26744E+01, 0.26875E+01, 0.27005E+01, 0.27134E+01,
     & 0.27262E+01, 0.27389E+01, 0.27515E+01, 0.27640E+01, 0.27765E+01,
     & 0.27889E+01, 0.28012E+01, 0.28134E+01, 0.28255E+01, 0.28375E+01,
     & 0.28495E+01, 0.28614E+01, 0.28732E+01, 0.28849E+01, 0.28966E+01,
     & 0.29082E+01, 0.29197E+01, 0.29312E+01, 0.29425E+01, 0.29538E+01,
     & 0.29651E+01, 0.29762E+01, 0.29873E+01, 0.29984E+01, 0.30093E+01,
     & 0.30202E+01, 0.30311E+01, 0.30419E+01, 0.30526E+01, 0.30632E+01,
     & 0.30738E+01, 0.30843E+01, 0.30948E+01, 0.31052E+01, 0.31155E+01,
     & 0.31258E+01, 0.31361E+01, 0.31462E+01, 0.31564E+01, 0.31664E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.31764E+01, 0.31864E+01, 0.31963E+01, 0.32061E+01, 0.32159E+01,
     & 0.32257E+01, 0.32354E+01, 0.32450E+01, 0.32546E+01, 0.32641E+01,
     & 0.32736E+01, 0.32831E+01, 0.32925E+01, 0.33018E+01, 0.33111E+01,
     & 0.33203E+01, 0.33295E+01, 0.33387E+01, 0.33478E+01, 0.33569E+01,
     & 0.33659E+01, 0.33749E+01, 0.33838E+01, 0.33927E+01, 0.34016E+01,
     & 0.34104E+01, 0.34191E+01, 0.34278E+01, 0.34365E+01, 0.34452E+01,
     & 0.34538E+01, 0.34623E+01, 0.34708E+01, 0.34793E+01, 0.34878E+01,
     & 0.34962E+01, 0.35045E+01, 0.35128E+01, 0.35211E+01, 0.35294E+01,
     & 0.35376E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.74160E-01,-0.16032E+00,-0.20289E+00,-0.23274E+00,-0.25599E+00,
     &-0.27511E+00,-0.29137E+00,-0.30552E+00,-0.31804E+00,-0.32926E+00,
     &-0.33941E+00,-0.34868E+00,-0.35720E+00,-0.36507E+00,-0.37238E+00,
     &-0.37919E+00,-0.38556E+00,-0.39153E+00,-0.39715E+00,-0.40245E+00,
     &-0.40745E+00,-0.41219E+00,-0.41668E+00,-0.42094E+00,-0.42499E+00,
     &-0.42884E+00,-0.43251E+00,-0.43600E+00,-0.43934E+00,-0.44252E+00,
     &-0.44556E+00,-0.44847E+00,-0.45124E+00,-0.45390E+00,-0.45644E+00,
     &-0.45887E+00,-0.46119E+00,-0.46342E+00,-0.46555E+00,-0.46759E+00,
     &-0.46955E+00,-0.47142E+00,-0.47321E+00,-0.47492E+00,-0.47656E+00,
     &-0.47813E+00,-0.47963E+00,-0.48107E+00,-0.48244E+00,-0.48376E+00,
     &-0.48501E+00,-0.48621E+00,-0.48736E+00,-0.48845E+00,-0.48950E+00,
     &-0.49049E+00,-0.49144E+00,-0.49234E+00,-0.49320E+00,-0.49402E+00,
     &-0.49480E+00,-0.49554E+00,-0.49623E+00,-0.49690E+00,-0.49752E+00,
     &-0.49812E+00,-0.49867E+00,-0.49920E+00,-0.49969E+00,-0.50015E+00,
     &-0.50059E+00,-0.50099E+00,-0.50136E+00,-0.50171E+00,-0.50203E+00,
     &-0.50232E+00,-0.50259E+00,-0.50283E+00,-0.50305E+00,-0.50324E+00,
     &-0.50341E+00,-0.50356E+00,-0.50369E+00,-0.50379E+00,-0.50387E+00,
     &-0.50393E+00,-0.50397E+00,-0.50399E+00,-0.50399E+00,-0.50397E+00,
     &-0.50393E+00,-0.50387E+00,-0.50380E+00,-0.50371E+00,-0.50360E+00,
     &-0.50347E+00,-0.50333E+00,-0.50317E+00,-0.50300E+00,-0.50281E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.50260E+00,-0.50239E+00,-0.50216E+00,-0.50191E+00,-0.50165E+00,
     &-0.50138E+00,-0.50110E+00,-0.50081E+00,-0.50050E+00,-0.50019E+00,
     &-0.49986E+00,-0.49953E+00,-0.49918E+00,-0.49882E+00,-0.49846E+00,
     &-0.49809E+00,-0.49771E+00,-0.49732E+00,-0.49692E+00,-0.49652E+00,
     &-0.49610E+00,-0.49569E+00,-0.49526E+00,-0.49483E+00,-0.49439E+00,
     &-0.49395E+00,-0.49350E+00,-0.49305E+00,-0.49259E+00,-0.49213E+00,
     &-0.49166E+00,-0.49119E+00,-0.49071E+00,-0.49023E+00,-0.48975E+00,
     &-0.48926E+00,-0.48877E+00,-0.48827E+00,-0.48777E+00,-0.48727E+00,
     &-0.48677E+00,-0.48626E+00,-0.48575E+00,-0.48523E+00,-0.48472E+00,
     &-0.48420E+00,-0.48368E+00,-0.48315E+00,-0.48263E+00,-0.48210E+00,
     &-0.48157E+00,-0.48104E+00,-0.48050E+00,-0.47997E+00,-0.47943E+00,
     &-0.47889E+00,-0.47835E+00,-0.47781E+00,-0.47727E+00,-0.47672E+00,
     &-0.47617E+00,-0.47563E+00,-0.47508E+00,-0.47453E+00,-0.47398E+00,
     &-0.47342E+00,-0.47287E+00,-0.47232E+00,-0.47176E+00,-0.47121E+00,
     &-0.47065E+00,-0.47009E+00,-0.46953E+00,-0.46897E+00,-0.46841E+00,
     &-0.46785E+00,-0.46729E+00,-0.46673E+00,-0.46617E+00,-0.46560E+00,
     &-0.46504E+00,-0.46448E+00,-0.46391E+00,-0.46335E+00,-0.46278E+00,
     &-0.46222E+00,-0.46165E+00,-0.46108E+00,-0.46052E+00,-0.45995E+00,
     &-0.45938E+00,-0.45882E+00,-0.45825E+00,-0.45768E+00,-0.45711E+00,
     &-0.45655E+00,-0.45598E+00,-0.45541E+00,-0.45484E+00,-0.45428E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.45371E+00,-0.45314E+00,-0.45257E+00,-0.45200E+00,-0.45144E+00,
     &-0.45087E+00,-0.45030E+00,-0.44973E+00,-0.44917E+00,-0.44860E+00,
     &-0.44803E+00,-0.44746E+00,-0.44690E+00,-0.44633E+00,-0.44576E+00,
     &-0.44520E+00,-0.44463E+00,-0.44406E+00,-0.44350E+00,-0.44293E+00,
     &-0.44237E+00,-0.44180E+00,-0.44124E+00,-0.44067E+00,-0.44011E+00,
     &-0.43954E+00,-0.43898E+00,-0.43842E+00,-0.43785E+00,-0.43729E+00,
     &-0.43673E+00,-0.43617E+00,-0.43560E+00,-0.43504E+00,-0.43448E+00,
     &-0.43392E+00,-0.43336E+00,-0.43280E+00,-0.43224E+00,-0.43168E+00,
     &-0.43112E+00,-0.43056E+00,-0.43000E+00,-0.42945E+00,-0.42889E+00,
     &-0.42833E+00,-0.42778E+00,-0.42722E+00,-0.42666E+00,-0.42611E+00,
     &-0.42555E+00,-0.42500E+00,-0.42445E+00,-0.42389E+00,-0.42334E+00,
     &-0.42279E+00,-0.42223E+00,-0.42168E+00,-0.42113E+00,-0.42058E+00,
     &-0.42003E+00,-0.41948E+00,-0.41893E+00,-0.41838E+00,-0.41783E+00,
     &-0.41729E+00,-0.41674E+00,-0.41619E+00,-0.41564E+00,-0.41510E+00,
     &-0.41455E+00,-0.41401E+00,-0.41346E+00,-0.41292E+00,-0.41238E+00,
     &-0.41183E+00,-0.41129E+00,-0.41075E+00,-0.41021E+00,-0.40967E+00,
     &-0.40913E+00,-0.40859E+00,-0.40805E+00,-0.40751E+00,-0.40697E+00,
     &-0.40643E+00,-0.40589E+00,-0.40536E+00,-0.40482E+00,-0.40429E+00,
     &-0.40375E+00,-0.40322E+00,-0.40268E+00,-0.40215E+00,-0.40161E+00,
     &-0.40108E+00,-0.40055E+00,-0.40002E+00,-0.39949E+00,-0.39896E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.39843E+00,-0.39790E+00,-0.39737E+00,-0.39684E+00,-0.39631E+00,
     &-0.39579E+00,-0.39526E+00,-0.39473E+00,-0.39421E+00,-0.39368E+00,
     &-0.39316E+00,-0.39263E+00,-0.39211E+00,-0.39159E+00,-0.39106E+00,
     &-0.39054E+00,-0.39002E+00,-0.38950E+00,-0.38898E+00,-0.38846E+00,
     &-0.38794E+00,-0.38742E+00,-0.38691E+00,-0.38639E+00,-0.38587E+00,
     &-0.38535E+00,-0.38484E+00,-0.38432E+00,-0.38381E+00,-0.38329E+00,
     &-0.38278E+00,-0.38227E+00,-0.38175E+00,-0.38124E+00,-0.38073E+00,
     &-0.38022E+00,-0.37971E+00,-0.37920E+00,-0.37869E+00,-0.37818E+00,
     &-0.37767E+00,-0.37717E+00,-0.37666E+00,-0.37615E+00,-0.37565E+00,
     &-0.37514E+00,-0.37464E+00,-0.37413E+00,-0.37363E+00,-0.37312E+00,
     &-0.37262E+00,-0.37212E+00,-0.37162E+00,-0.37112E+00,-0.37061E+00,
     &-0.37011E+00,-0.36961E+00,-0.36912E+00,-0.36862E+00,-0.36812E+00,
     &-0.36762E+00,-0.36712E+00,-0.36663E+00,-0.36613E+00,-0.36564E+00,
     &-0.36514E+00,-0.36465E+00,-0.36415E+00,-0.36366E+00,-0.36317E+00,
     &-0.36268E+00,-0.36218E+00,-0.36169E+00,-0.36120E+00,-0.36071E+00,
     &-0.36022E+00,-0.35973E+00,-0.35925E+00,-0.35876E+00,-0.35827E+00,
     &-0.35778E+00,-0.35730E+00,-0.35681E+00,-0.35633E+00,-0.35584E+00,
     &-0.35536E+00,-0.35487E+00,-0.35439E+00,-0.35391E+00,-0.35343E+00,
     &-0.35294E+00,-0.35246E+00,-0.35198E+00,-0.35150E+00,-0.35102E+00,
     &-0.35054E+00,-0.35006E+00,-0.34959E+00,-0.34911E+00,-0.34863E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.34816E+00,-0.34768E+00,-0.34720E+00,-0.34673E+00,-0.34625E+00,
     &-0.34578E+00,-0.34531E+00,-0.34483E+00,-0.34436E+00,-0.34389E+00,
     &-0.34342E+00,-0.34295E+00,-0.34248E+00,-0.34201E+00,-0.34154E+00,
     &-0.34107E+00,-0.34060E+00,-0.34013E+00,-0.33967E+00,-0.33920E+00,
     &-0.33873E+00,-0.33827E+00,-0.33780E+00,-0.33734E+00,-0.33687E+00,
     &-0.33641E+00,-0.33594E+00,-0.33548E+00,-0.33502E+00,-0.33456E+00,
     &-0.33410E+00,-0.33363E+00,-0.33317E+00,-0.33271E+00,-0.33225E+00,
     &-0.33180E+00,-0.33134E+00,-0.33088E+00,-0.33042E+00,-0.32996E+00,
     &-0.32951E+00,-0.32905E+00,-0.32860E+00,-0.32814E+00,-0.32769E+00,
     &-0.32723E+00,-0.32678E+00,-0.32633E+00,-0.32587E+00,-0.32542E+00,
     &-0.32497E+00,-0.32452E+00,-0.32407E+00,-0.32362E+00,-0.32317E+00,
     &-0.32272E+00,-0.32227E+00,-0.32182E+00,-0.32137E+00,-0.32092E+00,
     &-0.32048E+00,-0.32003E+00,-0.31958E+00,-0.31914E+00,-0.31869E+00,
     &-0.31825E+00,-0.31781E+00,-0.31736E+00,-0.31692E+00,-0.31648E+00,
     &-0.31603E+00,-0.31559E+00,-0.31515E+00,-0.31471E+00,-0.31427E+00,
     &-0.31383E+00,-0.31339E+00,-0.31295E+00,-0.31251E+00,-0.31207E+00,
     &-0.31163E+00,-0.31120E+00,-0.31076E+00,-0.31032E+00,-0.30989E+00,
     &-0.30945E+00,-0.30902E+00,-0.30858E+00,-0.30815E+00,-0.30771E+00,
     &-0.30728E+00,-0.30685E+00,-0.30642E+00,-0.30598E+00,-0.30555E+00,
     &-0.30512E+00,-0.30469E+00,-0.30426E+00,-0.30383E+00,-0.30340E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.30297E+00,-0.30254E+00,-0.30212E+00,-0.30169E+00,-0.30126E+00,
     &-0.30084E+00,-0.30041E+00,-0.29998E+00,-0.29956E+00,-0.29913E+00,
     &-0.29871E+00,-0.29828E+00,-0.29786E+00,-0.29744E+00,-0.29701E+00,
     &-0.29659E+00,-0.29617E+00,-0.29575E+00,-0.29533E+00,-0.29491E+00,
     &-0.29449E+00,-0.29407E+00,-0.29365E+00,-0.29323E+00,-0.29281E+00,
     &-0.29239E+00,-0.29197E+00,-0.29156E+00,-0.29114E+00,-0.29072E+00,
     &-0.29031E+00,-0.28989E+00,-0.28948E+00,-0.28906E+00,-0.28865E+00,
     &-0.28823E+00,-0.28782E+00,-0.28741E+00,-0.28700E+00,-0.28658E+00,
     &-0.28617E+00,-0.28576E+00,-0.28535E+00,-0.28494E+00,-0.28453E+00,
     &-0.28412E+00,-0.28371E+00,-0.28330E+00,-0.28289E+00,-0.28248E+00,
     &-0.28208E+00,-0.28167E+00,-0.28126E+00,-0.28085E+00,-0.28045E+00,
     &-0.28004E+00,-0.27964E+00,-0.27923E+00,-0.27883E+00,-0.27842E+00,
     &-0.27802E+00,-0.27762E+00,-0.27721E+00,-0.27681E+00,-0.27641E+00,
     &-0.27601E+00,-0.27561E+00,-0.27520E+00,-0.27480E+00,-0.27440E+00,
     &-0.27400E+00,-0.27360E+00,-0.27320E+00,-0.27281E+00,-0.27241E+00,
     &-0.27201E+00,-0.27161E+00,-0.27122E+00,-0.27082E+00,-0.27042E+00,
     &-0.27003E+00,-0.26963E+00,-0.26924E+00,-0.26884E+00,-0.26845E+00,
     &-0.26805E+00,-0.26766E+00,-0.26727E+00,-0.26687E+00,-0.26648E+00,
     &-0.26609E+00,-0.26570E+00,-0.26530E+00,-0.26491E+00,-0.26452E+00,
     &-0.26413E+00,-0.26374E+00,-0.26335E+00,-0.26296E+00,-0.26257E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.25840E+00,-0.25457E+00,-0.25078E+00,-0.24702E+00,-0.24330E+00,
     &-0.23961E+00,-0.23595E+00,-0.23233E+00,-0.22874E+00,-0.22518E+00,
     &-0.22165E+00,-0.21815E+00,-0.21469E+00,-0.21125E+00,-0.20784E+00,
     &-0.20447E+00,-0.20112E+00,-0.19779E+00,-0.19450E+00,-0.19123E+00,
     &-0.18799E+00,-0.18477E+00,-0.18158E+00,-0.17842E+00,-0.17528E+00,
     &-0.17217E+00,-0.16907E+00,-0.16601E+00,-0.16297E+00,-0.15995E+00,
     &-0.15695E+00,-0.15397E+00,-0.15102E+00,-0.14809E+00,-0.14518E+00,
     &-0.14229E+00,-0.13943E+00,-0.13658E+00,-0.13375E+00,-0.13095E+00,
     &-0.12816E+00,-0.12539E+00,-0.12265E+00,-0.11992E+00,-0.11721E+00,
     &-0.11452E+00,-0.11184E+00,-0.10919E+00,-0.10655E+00,-0.10393E+00,
     &-0.10133E+00,-0.98739E-01,-0.96170E-01,-0.93617E-01,-0.91081E-01,
     &-0.88561E-01,-0.86056E-01,-0.83568E-01,-0.81095E-01,-0.78637E-01,
     &-0.76195E-01,-0.73767E-01,-0.71355E-01,-0.68957E-01,-0.66573E-01,
     &-0.64204E-01,-0.61849E-01,-0.59507E-01,-0.57180E-01,-0.54866E-01,
     &-0.52566E-01,-0.50278E-01,-0.48004E-01,-0.45744E-01,-0.43495E-01,
     &-0.41260E-01,-0.39037E-01,-0.36827E-01,-0.34628E-01,-0.32442E-01,
     &-0.30268E-01,-0.28106E-01,-0.25955E-01,-0.23817E-01,-0.21689E-01,
     &-0.19573E-01,-0.17469E-01,-0.15375E-01,-0.13292E-01,-0.11221E-01,
     &-0.91596E-02,-0.71093E-02,-0.50696E-02,-0.30403E-02,-0.10211E-02,
     & 0.98760E-03, 0.29861E-02, 0.49747E-02, 0.69537E-02, 0.89224E-02/

      DATA (BNC13M(I),I=701,741)/
     & 0.10882E-01, 0.12831E-01, 0.14771E-01, 0.16702E-01, 0.18623E-01,
     & 0.20535E-01, 0.22438E-01, 0.24331E-01, 0.26216E-01, 0.28092E-01,
     & 0.29959E-01, 0.31817E-01, 0.33667E-01, 0.35508E-01, 0.37341E-01,
     & 0.39165E-01, 0.40981E-01, 0.42788E-01, 0.44588E-01, 0.46379E-01,
     & 0.48162E-01, 0.49937E-01, 0.51705E-01, 0.53465E-01, 0.55216E-01,
     & 0.56960E-01, 0.58697E-01, 0.60426E-01, 0.62147E-01, 0.63862E-01,
     & 0.65569E-01, 0.67268E-01, 0.68960E-01, 0.70645E-01, 0.72323E-01,
     & 0.73994E-01, 0.75658E-01, 0.77315E-01, 0.78966E-01, 0.80609E-01,
     & 0.82246E-01
     & /
       END



C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA KMCF323A
C *** CONTAINS THE DATA FOR KUSSIK-MEISNER BINARY COEFFICIENT ARRAYS 
C     NEEDED IN SUBROUTINE KM
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA KMCF323A
C
C *** Common block definition
C
      COMMON /KMC323/
     &BNC01M(  741),BNC02M(  741),BNC03M(  741),BNC04M(  741),
     &BNC05M(  741),BNC06M(  741),BNC07M(  741),BNC08M(  741),
     &BNC09M(  741),BNC10M(  741),BNC11M(  741),BNC12M(  741),
     &BNC13M(  741)
C
C *** NaCl         
C
      DATA (BNC01M(I),I=1,100)/
     &-0.44252E-01,-0.92492E-01,-0.11448E+00,-0.12890E+00,-0.13941E+00,
     &-0.14750E+00,-0.15391E+00,-0.15911E+00,-0.16337E+00,-0.16690E+00,
     &-0.16982E+00,-0.17224E+00,-0.17425E+00,-0.17590E+00,-0.17724E+00,
     &-0.17831E+00,-0.17916E+00,-0.17979E+00,-0.18025E+00,-0.18054E+00,
     &-0.18068E+00,-0.18069E+00,-0.18058E+00,-0.18037E+00,-0.18006E+00,
     &-0.17965E+00,-0.17917E+00,-0.17861E+00,-0.17799E+00,-0.17730E+00,
     &-0.17656E+00,-0.17576E+00,-0.17492E+00,-0.17403E+00,-0.17310E+00,
     &-0.17213E+00,-0.17113E+00,-0.17010E+00,-0.16904E+00,-0.16795E+00,
     &-0.16684E+00,-0.16571E+00,-0.16455E+00,-0.16338E+00,-0.16219E+00,
     &-0.16098E+00,-0.15976E+00,-0.15852E+00,-0.15727E+00,-0.15601E+00,
     &-0.15473E+00,-0.15345E+00,-0.15216E+00,-0.15085E+00,-0.14954E+00,
     &-0.14822E+00,-0.14689E+00,-0.14556E+00,-0.14421E+00,-0.14286E+00,
     &-0.14150E+00,-0.14014E+00,-0.13876E+00,-0.13738E+00,-0.13599E+00,
     &-0.13459E+00,-0.13319E+00,-0.13177E+00,-0.13035E+00,-0.12892E+00,
     &-0.12748E+00,-0.12603E+00,-0.12457E+00,-0.12310E+00,-0.12162E+00,
     &-0.12013E+00,-0.11863E+00,-0.11712E+00,-0.11560E+00,-0.11407E+00,
     &-0.11253E+00,-0.11097E+00,-0.10940E+00,-0.10782E+00,-0.10623E+00,
     &-0.10463E+00,-0.10301E+00,-0.10138E+00,-0.99742E-01,-0.98090E-01,
     &-0.96425E-01,-0.94749E-01,-0.93061E-01,-0.91361E-01,-0.89649E-01,
     &-0.87927E-01,-0.86193E-01,-0.84448E-01,-0.82693E-01,-0.80928E-01/

      DATA (BNC01M(I),I=101,200)/
     &-0.79153E-01,-0.77368E-01,-0.75574E-01,-0.73772E-01,-0.71960E-01,
     &-0.70141E-01,-0.68314E-01,-0.66479E-01,-0.64638E-01,-0.62790E-01,
     &-0.60935E-01,-0.59075E-01,-0.57209E-01,-0.55338E-01,-0.53462E-01,
     &-0.51582E-01,-0.49697E-01,-0.47809E-01,-0.45917E-01,-0.44022E-01,
     &-0.42125E-01,-0.40224E-01,-0.38321E-01,-0.36416E-01,-0.34510E-01,
     &-0.32601E-01,-0.30691E-01,-0.28780E-01,-0.26868E-01,-0.24955E-01,
     &-0.23041E-01,-0.21127E-01,-0.19212E-01,-0.17298E-01,-0.15383E-01,
     &-0.13468E-01,-0.11553E-01,-0.96383E-02,-0.77240E-02,-0.58102E-02,
     &-0.38968E-02,-0.19841E-02,-0.72052E-04, 0.18393E-02, 0.37498E-02,
     & 0.56595E-02, 0.75683E-02, 0.94761E-02, 0.11383E-01, 0.13289E-01,
     & 0.15193E-01, 0.17097E-01, 0.18999E-01, 0.20900E-01, 0.22800E-01,
     & 0.24698E-01, 0.26595E-01, 0.28491E-01, 0.30386E-01, 0.32279E-01,
     & 0.34170E-01, 0.36060E-01, 0.37949E-01, 0.39836E-01, 0.41721E-01,
     & 0.43605E-01, 0.45488E-01, 0.47369E-01, 0.49248E-01, 0.51125E-01,
     & 0.53001E-01, 0.54875E-01, 0.56748E-01, 0.58618E-01, 0.60487E-01,
     & 0.62355E-01, 0.64220E-01, 0.66084E-01, 0.67946E-01, 0.69806E-01,
     & 0.71664E-01, 0.73521E-01, 0.75376E-01, 0.77228E-01, 0.79079E-01,
     & 0.80928E-01, 0.82775E-01, 0.84621E-01, 0.86464E-01, 0.88305E-01,
     & 0.90145E-01, 0.91982E-01, 0.93818E-01, 0.95651E-01, 0.97483E-01,
     & 0.99313E-01, 0.10114E+00, 0.10297E+00, 0.10479E+00, 0.10661E+00/

      DATA (BNC01M(I),I=201,300)/
     & 0.10843E+00, 0.11025E+00, 0.11206E+00, 0.11388E+00, 0.11569E+00,
     & 0.11750E+00, 0.11931E+00, 0.12111E+00, 0.12291E+00, 0.12472E+00,
     & 0.12651E+00, 0.12831E+00, 0.13011E+00, 0.13190E+00, 0.13369E+00,
     & 0.13548E+00, 0.13726E+00, 0.13905E+00, 0.14083E+00, 0.14261E+00,
     & 0.14439E+00, 0.14617E+00, 0.14794E+00, 0.14971E+00, 0.15148E+00,
     & 0.15325E+00, 0.15501E+00, 0.15678E+00, 0.15854E+00, 0.16030E+00,
     & 0.16205E+00, 0.16381E+00, 0.16556E+00, 0.16731E+00, 0.16906E+00,
     & 0.17080E+00, 0.17255E+00, 0.17429E+00, 0.17603E+00, 0.17777E+00,
     & 0.17950E+00, 0.18124E+00, 0.18297E+00, 0.18470E+00, 0.18642E+00,
     & 0.18815E+00, 0.18987E+00, 0.19159E+00, 0.19331E+00, 0.19502E+00,
     & 0.19674E+00, 0.19845E+00, 0.20016E+00, 0.20187E+00, 0.20357E+00,
     & 0.20527E+00, 0.20697E+00, 0.20867E+00, 0.21037E+00, 0.21206E+00,
     & 0.21376E+00, 0.21545E+00, 0.21714E+00, 0.21882E+00, 0.22050E+00,
     & 0.22219E+00, 0.22387E+00, 0.22554E+00, 0.22722E+00, 0.22889E+00,
     & 0.23056E+00, 0.23223E+00, 0.23390E+00, 0.23556E+00, 0.23723E+00,
     & 0.23889E+00, 0.24054E+00, 0.24220E+00, 0.24386E+00, 0.24551E+00,
     & 0.24716E+00, 0.24880E+00, 0.25045E+00, 0.25209E+00, 0.25374E+00,
     & 0.25538E+00, 0.25701E+00, 0.25865E+00, 0.26028E+00, 0.26191E+00,
     & 0.26354E+00, 0.26517E+00, 0.26679E+00, 0.26842E+00, 0.27004E+00,
     & 0.27166E+00, 0.27327E+00, 0.27489E+00, 0.27650E+00, 0.27811E+00/

      DATA (BNC01M(I),I=301,400)/
     & 0.27972E+00, 0.28132E+00, 0.28293E+00, 0.28453E+00, 0.28613E+00,
     & 0.28773E+00, 0.28933E+00, 0.29092E+00, 0.29251E+00, 0.29410E+00,
     & 0.29569E+00, 0.29728E+00, 0.29886E+00, 0.30044E+00, 0.30202E+00,
     & 0.30360E+00, 0.30518E+00, 0.30675E+00, 0.30832E+00, 0.30989E+00,
     & 0.31146E+00, 0.31303E+00, 0.31459E+00, 0.31615E+00, 0.31771E+00,
     & 0.31927E+00, 0.32083E+00, 0.32238E+00, 0.32393E+00, 0.32549E+00,
     & 0.32703E+00, 0.32858E+00, 0.33012E+00, 0.33167E+00, 0.33321E+00,
     & 0.33475E+00, 0.33628E+00, 0.33782E+00, 0.33935E+00, 0.34088E+00,
     & 0.34241E+00, 0.34394E+00, 0.34546E+00, 0.34699E+00, 0.34851E+00,
     & 0.35003E+00, 0.35155E+00, 0.35306E+00, 0.35458E+00, 0.35609E+00,
     & 0.35760E+00, 0.35911E+00, 0.36061E+00, 0.36212E+00, 0.36362E+00,
     & 0.36512E+00, 0.36662E+00, 0.36812E+00, 0.36961E+00, 0.37111E+00,
     & 0.37260E+00, 0.37409E+00, 0.37558E+00, 0.37706E+00, 0.37855E+00,
     & 0.38003E+00, 0.38151E+00, 0.38299E+00, 0.38447E+00, 0.38594E+00,
     & 0.38741E+00, 0.38889E+00, 0.39036E+00, 0.39182E+00, 0.39329E+00,
     & 0.39476E+00, 0.39622E+00, 0.39768E+00, 0.39914E+00, 0.40060E+00,
     & 0.40205E+00, 0.40350E+00, 0.40496E+00, 0.40641E+00, 0.40786E+00,
     & 0.40930E+00, 0.41075E+00, 0.41219E+00, 0.41363E+00, 0.41507E+00,
     & 0.41651E+00, 0.41795E+00, 0.41938E+00, 0.42081E+00, 0.42225E+00,
     & 0.42367E+00, 0.42510E+00, 0.42653E+00, 0.42795E+00, 0.42938E+00/

      DATA (BNC01M(I),I=401,500)/
     & 0.43080E+00, 0.43222E+00, 0.43363E+00, 0.43505E+00, 0.43646E+00,
     & 0.43788E+00, 0.43929E+00, 0.44070E+00, 0.44210E+00, 0.44351E+00,
     & 0.44491E+00, 0.44632E+00, 0.44772E+00, 0.44912E+00, 0.45051E+00,
     & 0.45191E+00, 0.45331E+00, 0.45470E+00, 0.45609E+00, 0.45748E+00,
     & 0.45887E+00, 0.46025E+00, 0.46164E+00, 0.46302E+00, 0.46440E+00,
     & 0.46578E+00, 0.46716E+00, 0.46854E+00, 0.46991E+00, 0.47128E+00,
     & 0.47266E+00, 0.47403E+00, 0.47539E+00, 0.47676E+00, 0.47813E+00,
     & 0.47949E+00, 0.48085E+00, 0.48221E+00, 0.48357E+00, 0.48493E+00,
     & 0.48629E+00, 0.48764E+00, 0.48899E+00, 0.49035E+00, 0.49170E+00,
     & 0.49304E+00, 0.49439E+00, 0.49574E+00, 0.49708E+00, 0.49842E+00,
     & 0.49976E+00, 0.50110E+00, 0.50244E+00, 0.50378E+00, 0.50511E+00,
     & 0.50644E+00, 0.50778E+00, 0.50911E+00, 0.51043E+00, 0.51176E+00,
     & 0.51309E+00, 0.51441E+00, 0.51573E+00, 0.51706E+00, 0.51838E+00,
     & 0.51969E+00, 0.52101E+00, 0.52233E+00, 0.52364E+00, 0.52495E+00,
     & 0.52626E+00, 0.52757E+00, 0.52888E+00, 0.53019E+00, 0.53149E+00,
     & 0.53280E+00, 0.53410E+00, 0.53540E+00, 0.53670E+00, 0.53800E+00,
     & 0.53929E+00, 0.54059E+00, 0.54188E+00, 0.54317E+00, 0.54446E+00,
     & 0.54575E+00, 0.54704E+00, 0.54833E+00, 0.54961E+00, 0.55090E+00,
     & 0.55218E+00, 0.55346E+00, 0.55474E+00, 0.55602E+00, 0.55730E+00,
     & 0.55857E+00, 0.55985E+00, 0.56112E+00, 0.56239E+00, 0.56366E+00/

      DATA (BNC01M(I),I=501,600)/
     & 0.56493E+00, 0.56620E+00, 0.56746E+00, 0.56873E+00, 0.56999E+00,
     & 0.57125E+00, 0.57251E+00, 0.57377E+00, 0.57503E+00, 0.57629E+00,
     & 0.57754E+00, 0.57879E+00, 0.58005E+00, 0.58130E+00, 0.58255E+00,
     & 0.58380E+00, 0.58504E+00, 0.58629E+00, 0.58753E+00, 0.58878E+00,
     & 0.59002E+00, 0.59126E+00, 0.59250E+00, 0.59374E+00, 0.59497E+00,
     & 0.59621E+00, 0.59744E+00, 0.59868E+00, 0.59991E+00, 0.60114E+00,
     & 0.60237E+00, 0.60359E+00, 0.60482E+00, 0.60605E+00, 0.60727E+00,
     & 0.60849E+00, 0.60971E+00, 0.61093E+00, 0.61215E+00, 0.61337E+00,
     & 0.61459E+00, 0.61580E+00, 0.61702E+00, 0.61823E+00, 0.61944E+00,
     & 0.62065E+00, 0.62186E+00, 0.62307E+00, 0.62427E+00, 0.62548E+00,
     & 0.62668E+00, 0.62788E+00, 0.62909E+00, 0.63029E+00, 0.63149E+00,
     & 0.63268E+00, 0.63388E+00, 0.63508E+00, 0.63627E+00, 0.63746E+00,
     & 0.63865E+00, 0.63985E+00, 0.64103E+00, 0.64222E+00, 0.64341E+00,
     & 0.64460E+00, 0.64578E+00, 0.64696E+00, 0.64815E+00, 0.64933E+00,
     & 0.65051E+00, 0.65169E+00, 0.65286E+00, 0.65404E+00, 0.65521E+00,
     & 0.65639E+00, 0.65756E+00, 0.65873E+00, 0.65990E+00, 0.66107E+00,
     & 0.66224E+00, 0.66341E+00, 0.66457E+00, 0.66574E+00, 0.66690E+00,
     & 0.66807E+00, 0.66923E+00, 0.67039E+00, 0.67155E+00, 0.67270E+00,
     & 0.67386E+00, 0.67502E+00, 0.67617E+00, 0.67733E+00, 0.67848E+00,
     & 0.67963E+00, 0.68078E+00, 0.68193E+00, 0.68308E+00, 0.68422E+00/

      DATA (BNC01M(I),I=601,700)/
     & 0.69654E+00, 0.70783E+00, 0.71902E+00, 0.73010E+00, 0.74108E+00,
     & 0.75196E+00, 0.76274E+00, 0.77342E+00, 0.78402E+00, 0.79451E+00,
     & 0.80492E+00, 0.81524E+00, 0.82547E+00, 0.83562E+00, 0.84568E+00,
     & 0.85566E+00, 0.86556E+00, 0.87538E+00, 0.88512E+00, 0.89479E+00,
     & 0.90437E+00, 0.91389E+00, 0.92333E+00, 0.93270E+00, 0.94200E+00,
     & 0.95123E+00, 0.96040E+00, 0.96949E+00, 0.97852E+00, 0.98749E+00,
     & 0.99639E+00, 0.10052E+01, 0.10140E+01, 0.10227E+01, 0.10314E+01,
     & 0.10400E+01, 0.10485E+01, 0.10570E+01, 0.10654E+01, 0.10738E+01,
     & 0.10821E+01, 0.10904E+01, 0.10986E+01, 0.11068E+01, 0.11149E+01,
     & 0.11230E+01, 0.11310E+01, 0.11389E+01, 0.11469E+01, 0.11547E+01,
     & 0.11625E+01, 0.11703E+01, 0.11780E+01, 0.11857E+01, 0.11934E+01,
     & 0.12010E+01, 0.12085E+01, 0.12160E+01, 0.12235E+01, 0.12309E+01,
     & 0.12383E+01, 0.12457E+01, 0.12530E+01, 0.12602E+01, 0.12675E+01,
     & 0.12747E+01, 0.12818E+01, 0.12889E+01, 0.12960E+01, 0.13031E+01,
     & 0.13101E+01, 0.13170E+01, 0.13240E+01, 0.13309E+01, 0.13378E+01,
     & 0.13446E+01, 0.13514E+01, 0.13582E+01, 0.13649E+01, 0.13716E+01,
     & 0.13783E+01, 0.13849E+01, 0.13915E+01, 0.13981E+01, 0.14047E+01,
     & 0.14112E+01, 0.14177E+01, 0.14241E+01, 0.14306E+01, 0.14370E+01,
     & 0.14434E+01, 0.14497E+01, 0.14560E+01, 0.14623E+01, 0.14686E+01,
     & 0.14748E+01, 0.14810E+01, 0.14872E+01, 0.14934E+01, 0.14995E+01/

      DATA (BNC01M(I),I=701,741)/
     & 0.15056E+01, 0.15117E+01, 0.15178E+01, 0.15238E+01, 0.15298E+01,
     & 0.15358E+01, 0.15418E+01, 0.15477E+01, 0.15536E+01, 0.15595E+01,
     & 0.15654E+01, 0.15713E+01, 0.15771E+01, 0.15829E+01, 0.15887E+01,
     & 0.15944E+01, 0.16002E+01, 0.16059E+01, 0.16116E+01, 0.16173E+01,
     & 0.16229E+01, 0.16286E+01, 0.16342E+01, 0.16398E+01, 0.16453E+01,
     & 0.16509E+01, 0.16564E+01, 0.16620E+01, 0.16675E+01, 0.16729E+01,
     & 0.16784E+01, 0.16838E+01, 0.16893E+01, 0.16947E+01, 0.17000E+01,
     & 0.17054E+01, 0.17108E+01, 0.17161E+01, 0.17214E+01, 0.17267E+01,
     & 0.17320E+01
     & /
C
C *** Na2SO4       
C
      DATA (BNC02M(I),I=1,100)/
     &-0.90612E-01,-0.19630E+00,-0.24862E+00,-0.28534E+00,-0.31396E+00,
     &-0.33750E+00,-0.35753E+00,-0.37498E+00,-0.39044E+00,-0.40431E+00,
     &-0.41689E+00,-0.42839E+00,-0.43899E+00,-0.44880E+00,-0.45794E+00,
     &-0.46648E+00,-0.47450E+00,-0.48206E+00,-0.48920E+00,-0.49596E+00,
     &-0.50238E+00,-0.50850E+00,-0.51433E+00,-0.51990E+00,-0.52523E+00,
     &-0.53034E+00,-0.53524E+00,-0.53996E+00,-0.54449E+00,-0.54886E+00,
     &-0.55307E+00,-0.55714E+00,-0.56107E+00,-0.56487E+00,-0.56855E+00,
     &-0.57212E+00,-0.57557E+00,-0.57892E+00,-0.58217E+00,-0.58533E+00,
     &-0.58840E+00,-0.59139E+00,-0.59429E+00,-0.59711E+00,-0.59986E+00,
     &-0.60254E+00,-0.60516E+00,-0.60770E+00,-0.61019E+00,-0.61261E+00,
     &-0.61498E+00,-0.61729E+00,-0.61954E+00,-0.62175E+00,-0.62391E+00,
     &-0.62602E+00,-0.62808E+00,-0.63010E+00,-0.63207E+00,-0.63401E+00,
     &-0.63590E+00,-0.63776E+00,-0.63958E+00,-0.64136E+00,-0.64311E+00,
     &-0.64483E+00,-0.64651E+00,-0.64816E+00,-0.64979E+00,-0.65138E+00,
     &-0.65294E+00,-0.65448E+00,-0.65599E+00,-0.65748E+00,-0.65894E+00,
     &-0.66037E+00,-0.66179E+00,-0.66318E+00,-0.66454E+00,-0.66589E+00,
     &-0.66722E+00,-0.66852E+00,-0.66981E+00,-0.67108E+00,-0.67233E+00,
     &-0.67356E+00,-0.67478E+00,-0.67598E+00,-0.67716E+00,-0.67832E+00,
     &-0.67947E+00,-0.68061E+00,-0.68173E+00,-0.68283E+00,-0.68392E+00,
     &-0.68500E+00,-0.68606E+00,-0.68711E+00,-0.68815E+00,-0.68918E+00/

      DATA (BNC02M(I),I=101,200)/
     &-0.69019E+00,-0.69119E+00,-0.69218E+00,-0.69315E+00,-0.69412E+00,
     &-0.69507E+00,-0.69601E+00,-0.69694E+00,-0.69786E+00,-0.69877E+00,
     &-0.69967E+00,-0.70056E+00,-0.70144E+00,-0.70230E+00,-0.70316E+00,
     &-0.70401E+00,-0.70485E+00,-0.70568E+00,-0.70650E+00,-0.70731E+00,
     &-0.70811E+00,-0.70890E+00,-0.70968E+00,-0.71046E+00,-0.71122E+00,
     &-0.71198E+00,-0.71273E+00,-0.71347E+00,-0.71420E+00,-0.71493E+00,
     &-0.71564E+00,-0.71635E+00,-0.71705E+00,-0.71775E+00,-0.71843E+00,
     &-0.71911E+00,-0.71978E+00,-0.72044E+00,-0.72110E+00,-0.72175E+00,
     &-0.72239E+00,-0.72302E+00,-0.72365E+00,-0.72427E+00,-0.72489E+00,
     &-0.72550E+00,-0.72610E+00,-0.72670E+00,-0.72729E+00,-0.72787E+00,
     &-0.72845E+00,-0.72902E+00,-0.72958E+00,-0.73014E+00,-0.73069E+00,
     &-0.73124E+00,-0.73178E+00,-0.73232E+00,-0.73285E+00,-0.73338E+00,
     &-0.73390E+00,-0.73441E+00,-0.73492E+00,-0.73542E+00,-0.73592E+00,
     &-0.73642E+00,-0.73691E+00,-0.73739E+00,-0.73787E+00,-0.73835E+00,
     &-0.73881E+00,-0.73928E+00,-0.73974E+00,-0.74020E+00,-0.74065E+00,
     &-0.74109E+00,-0.74154E+00,-0.74197E+00,-0.74241E+00,-0.74284E+00,
     &-0.74326E+00,-0.74368E+00,-0.74410E+00,-0.74451E+00,-0.74492E+00,
     &-0.74532E+00,-0.74572E+00,-0.74612E+00,-0.74651E+00,-0.74690E+00,
     &-0.74729E+00,-0.74767E+00,-0.74805E+00,-0.74842E+00,-0.74879E+00,
     &-0.74916E+00,-0.74952E+00,-0.74988E+00,-0.75023E+00,-0.75059E+00/

      DATA (BNC02M(I),I=201,300)/
     &-0.75094E+00,-0.75128E+00,-0.75162E+00,-0.75196E+00,-0.75230E+00,
     &-0.75263E+00,-0.75296E+00,-0.75328E+00,-0.75361E+00,-0.75393E+00,
     &-0.75424E+00,-0.75456E+00,-0.75487E+00,-0.75517E+00,-0.75548E+00,
     &-0.75578E+00,-0.75608E+00,-0.75637E+00,-0.75666E+00,-0.75695E+00,
     &-0.75724E+00,-0.75752E+00,-0.75781E+00,-0.75808E+00,-0.75836E+00,
     &-0.75863E+00,-0.75890E+00,-0.75917E+00,-0.75943E+00,-0.75970E+00,
     &-0.75996E+00,-0.76021E+00,-0.76047E+00,-0.76072E+00,-0.76097E+00,
     &-0.76122E+00,-0.76146E+00,-0.76170E+00,-0.76194E+00,-0.76218E+00,
     &-0.76241E+00,-0.76265E+00,-0.76288E+00,-0.76310E+00,-0.76333E+00,
     &-0.76355E+00,-0.76377E+00,-0.76399E+00,-0.76421E+00,-0.76442E+00,
     &-0.76463E+00,-0.76484E+00,-0.76505E+00,-0.76526E+00,-0.76546E+00,
     &-0.76566E+00,-0.76586E+00,-0.76606E+00,-0.76625E+00,-0.76645E+00,
     &-0.76664E+00,-0.76683E+00,-0.76701E+00,-0.76720E+00,-0.76738E+00,
     &-0.76756E+00,-0.76774E+00,-0.76792E+00,-0.76809E+00,-0.76827E+00,
     &-0.76844E+00,-0.76861E+00,-0.76877E+00,-0.76894E+00,-0.76910E+00,
     &-0.76927E+00,-0.76943E+00,-0.76959E+00,-0.76974E+00,-0.76990E+00,
     &-0.77005E+00,-0.77020E+00,-0.77035E+00,-0.77050E+00,-0.77065E+00,
     &-0.77079E+00,-0.77094E+00,-0.77108E+00,-0.77122E+00,-0.77135E+00,
     &-0.77149E+00,-0.77163E+00,-0.77176E+00,-0.77189E+00,-0.77202E+00,
     &-0.77215E+00,-0.77228E+00,-0.77240E+00,-0.77253E+00,-0.77265E+00/

      DATA (BNC02M(I),I=301,400)/
     &-0.77277E+00,-0.77289E+00,-0.77301E+00,-0.77312E+00,-0.77324E+00,
     &-0.77335E+00,-0.77346E+00,-0.77357E+00,-0.77368E+00,-0.77379E+00,
     &-0.77389E+00,-0.77400E+00,-0.77410E+00,-0.77420E+00,-0.77430E+00,
     &-0.77440E+00,-0.77450E+00,-0.77460E+00,-0.77469E+00,-0.77479E+00,
     &-0.77488E+00,-0.77497E+00,-0.77506E+00,-0.77515E+00,-0.77524E+00,
     &-0.77532E+00,-0.77541E+00,-0.77549E+00,-0.77557E+00,-0.77565E+00,
     &-0.77573E+00,-0.77581E+00,-0.77589E+00,-0.77596E+00,-0.77604E+00,
     &-0.77611E+00,-0.77618E+00,-0.77625E+00,-0.77632E+00,-0.77639E+00,
     &-0.77646E+00,-0.77652E+00,-0.77659E+00,-0.77665E+00,-0.77671E+00,
     &-0.77678E+00,-0.77684E+00,-0.77689E+00,-0.77695E+00,-0.77701E+00,
     &-0.77707E+00,-0.77712E+00,-0.77717E+00,-0.77723E+00,-0.77728E+00,
     &-0.77733E+00,-0.77738E+00,-0.77743E+00,-0.77747E+00,-0.77752E+00,
     &-0.77756E+00,-0.77761E+00,-0.77765E+00,-0.77769E+00,-0.77773E+00,
     &-0.77777E+00,-0.77781E+00,-0.77785E+00,-0.77789E+00,-0.77792E+00,
     &-0.77796E+00,-0.77799E+00,-0.77802E+00,-0.77805E+00,-0.77809E+00,
     &-0.77812E+00,-0.77814E+00,-0.77817E+00,-0.77820E+00,-0.77823E+00,
     &-0.77825E+00,-0.77827E+00,-0.77830E+00,-0.77832E+00,-0.77834E+00,
     &-0.77836E+00,-0.77838E+00,-0.77840E+00,-0.77842E+00,-0.77843E+00,
     &-0.77845E+00,-0.77846E+00,-0.77848E+00,-0.77849E+00,-0.77850E+00,
     &-0.77852E+00,-0.77853E+00,-0.77854E+00,-0.77855E+00,-0.77855E+00/

      DATA (BNC02M(I),I=401,500)/
     &-0.77856E+00,-0.77857E+00,-0.77857E+00,-0.77858E+00,-0.77858E+00,
     &-0.77858E+00,-0.77859E+00,-0.77859E+00,-0.77859E+00,-0.77859E+00,
     &-0.77859E+00,-0.77859E+00,-0.77858E+00,-0.77858E+00,-0.77857E+00,
     &-0.77857E+00,-0.77856E+00,-0.77856E+00,-0.77855E+00,-0.77854E+00,
     &-0.77853E+00,-0.77852E+00,-0.77851E+00,-0.77850E+00,-0.77849E+00,
     &-0.77848E+00,-0.77846E+00,-0.77845E+00,-0.77843E+00,-0.77842E+00,
     &-0.77840E+00,-0.77838E+00,-0.77837E+00,-0.77835E+00,-0.77833E+00,
     &-0.77831E+00,-0.77829E+00,-0.77827E+00,-0.77824E+00,-0.77822E+00,
     &-0.77820E+00,-0.77817E+00,-0.77815E+00,-0.77812E+00,-0.77810E+00,
     &-0.77807E+00,-0.77804E+00,-0.77801E+00,-0.77798E+00,-0.77795E+00,
     &-0.77792E+00,-0.77789E+00,-0.77786E+00,-0.77783E+00,-0.77780E+00,
     &-0.77776E+00,-0.77773E+00,-0.77769E+00,-0.77766E+00,-0.77762E+00,
     &-0.77758E+00,-0.77755E+00,-0.77751E+00,-0.77747E+00,-0.77743E+00,
     &-0.77739E+00,-0.77735E+00,-0.77731E+00,-0.77727E+00,-0.77722E+00,
     &-0.77718E+00,-0.77714E+00,-0.77709E+00,-0.77705E+00,-0.77700E+00,
     &-0.77695E+00,-0.77691E+00,-0.77686E+00,-0.77681E+00,-0.77676E+00,
     &-0.77672E+00,-0.77667E+00,-0.77662E+00,-0.77656E+00,-0.77651E+00,
     &-0.77646E+00,-0.77641E+00,-0.77636E+00,-0.77630E+00,-0.77625E+00,
     &-0.77619E+00,-0.77614E+00,-0.77608E+00,-0.77602E+00,-0.77597E+00,
     &-0.77591E+00,-0.77585E+00,-0.77579E+00,-0.77573E+00,-0.77567E+00/

      DATA (BNC02M(I),I=501,600)/
     &-0.77561E+00,-0.77555E+00,-0.77549E+00,-0.77543E+00,-0.77537E+00,
     &-0.77530E+00,-0.77524E+00,-0.77518E+00,-0.77511E+00,-0.77505E+00,
     &-0.77498E+00,-0.77491E+00,-0.77485E+00,-0.77478E+00,-0.77471E+00,
     &-0.77465E+00,-0.77458E+00,-0.77451E+00,-0.77444E+00,-0.77437E+00,
     &-0.77430E+00,-0.77423E+00,-0.77415E+00,-0.77408E+00,-0.77401E+00,
     &-0.77394E+00,-0.77386E+00,-0.77379E+00,-0.77371E+00,-0.77364E+00,
     &-0.77356E+00,-0.77349E+00,-0.77341E+00,-0.77333E+00,-0.77326E+00,
     &-0.77318E+00,-0.77310E+00,-0.77302E+00,-0.77294E+00,-0.77286E+00,
     &-0.77278E+00,-0.77270E+00,-0.77262E+00,-0.77254E+00,-0.77246E+00,
     &-0.77238E+00,-0.77229E+00,-0.77221E+00,-0.77213E+00,-0.77204E+00,
     &-0.77196E+00,-0.77187E+00,-0.77179E+00,-0.77170E+00,-0.77161E+00,
     &-0.77153E+00,-0.77144E+00,-0.77135E+00,-0.77126E+00,-0.77118E+00,
     &-0.77109E+00,-0.77100E+00,-0.77091E+00,-0.77082E+00,-0.77073E+00,
     &-0.77064E+00,-0.77054E+00,-0.77045E+00,-0.77036E+00,-0.77027E+00,
     &-0.77017E+00,-0.77008E+00,-0.76999E+00,-0.76989E+00,-0.76980E+00,
     &-0.76970E+00,-0.76961E+00,-0.76951E+00,-0.76941E+00,-0.76932E+00,
     &-0.76922E+00,-0.76912E+00,-0.76903E+00,-0.76893E+00,-0.76883E+00,
     &-0.76873E+00,-0.76863E+00,-0.76853E+00,-0.76843E+00,-0.76833E+00,
     &-0.76823E+00,-0.76813E+00,-0.76802E+00,-0.76792E+00,-0.76782E+00,
     &-0.76772E+00,-0.76761E+00,-0.76751E+00,-0.76741E+00,-0.76730E+00/

      DATA (BNC02M(I),I=601,700)/
     &-0.76615E+00,-0.76505E+00,-0.76391E+00,-0.76274E+00,-0.76154E+00,
     &-0.76031E+00,-0.75905E+00,-0.75775E+00,-0.75643E+00,-0.75509E+00,
     &-0.75371E+00,-0.75231E+00,-0.75089E+00,-0.74944E+00,-0.74797E+00,
     &-0.74647E+00,-0.74496E+00,-0.74342E+00,-0.74186E+00,-0.74028E+00,
     &-0.73868E+00,-0.73706E+00,-0.73542E+00,-0.73376E+00,-0.73209E+00,
     &-0.73040E+00,-0.72869E+00,-0.72696E+00,-0.72522E+00,-0.72346E+00,
     &-0.72169E+00,-0.71990E+00,-0.71810E+00,-0.71628E+00,-0.71445E+00,
     &-0.71261E+00,-0.71075E+00,-0.70888E+00,-0.70699E+00,-0.70510E+00,
     &-0.70319E+00,-0.70127E+00,-0.69934E+00,-0.69739E+00,-0.69544E+00,
     &-0.69347E+00,-0.69150E+00,-0.68951E+00,-0.68751E+00,-0.68551E+00,
     &-0.68349E+00,-0.68146E+00,-0.67943E+00,-0.67738E+00,-0.67533E+00,
     &-0.67326E+00,-0.67119E+00,-0.66911E+00,-0.66702E+00,-0.66493E+00,
     &-0.66282E+00,-0.66071E+00,-0.65859E+00,-0.65646E+00,-0.65432E+00,
     &-0.65218E+00,-0.65003E+00,-0.64787E+00,-0.64571E+00,-0.64354E+00,
     &-0.64136E+00,-0.63917E+00,-0.63698E+00,-0.63478E+00,-0.63258E+00,
     &-0.63037E+00,-0.62815E+00,-0.62593E+00,-0.62370E+00,-0.62147E+00,
     &-0.61923E+00,-0.61699E+00,-0.61474E+00,-0.61248E+00,-0.61022E+00,
     &-0.60795E+00,-0.60568E+00,-0.60341E+00,-0.60113E+00,-0.59884E+00,
     &-0.59655E+00,-0.59425E+00,-0.59195E+00,-0.58965E+00,-0.58734E+00,
     &-0.58503E+00,-0.58271E+00,-0.58039E+00,-0.57806E+00,-0.57573E+00/

      DATA (BNC02M(I),I=701,741)/
     &-0.57339E+00,-0.57106E+00,-0.56871E+00,-0.56637E+00,-0.56402E+00,
     &-0.56166E+00,-0.55930E+00,-0.55694E+00,-0.55458E+00,-0.55221E+00,
     &-0.54984E+00,-0.54746E+00,-0.54508E+00,-0.54270E+00,-0.54031E+00,
     &-0.53792E+00,-0.53553E+00,-0.53313E+00,-0.53073E+00,-0.52833E+00,
     &-0.52593E+00,-0.52352E+00,-0.52111E+00,-0.51869E+00,-0.51628E+00,
     &-0.51386E+00,-0.51144E+00,-0.50901E+00,-0.50658E+00,-0.50415E+00,
     &-0.50172E+00,-0.49928E+00,-0.49684E+00,-0.49440E+00,-0.49196E+00,
     &-0.48951E+00,-0.48706E+00,-0.48461E+00,-0.48216E+00,-0.47970E+00,
     &-0.47725E+00
     & /
C
C *** NaNO3        
C
      DATA (BNC03M(I),I=1,100)/
     &-0.45407E-01,-0.98699E-01,-0.12527E+00,-0.14402E+00,-0.15871E+00,
     &-0.17085E+00,-0.18123E+00,-0.19030E+00,-0.19838E+00,-0.20565E+00,
     &-0.21228E+00,-0.21836E+00,-0.22398E+00,-0.22921E+00,-0.23409E+00,
     &-0.23868E+00,-0.24300E+00,-0.24708E+00,-0.25095E+00,-0.25463E+00,
     &-0.25814E+00,-0.26149E+00,-0.26470E+00,-0.26777E+00,-0.27072E+00,
     &-0.27355E+00,-0.27628E+00,-0.27892E+00,-0.28146E+00,-0.28391E+00,
     &-0.28628E+00,-0.28858E+00,-0.29081E+00,-0.29297E+00,-0.29506E+00,
     &-0.29710E+00,-0.29907E+00,-0.30100E+00,-0.30287E+00,-0.30469E+00,
     &-0.30646E+00,-0.30819E+00,-0.30987E+00,-0.31152E+00,-0.31312E+00,
     &-0.31469E+00,-0.31622E+00,-0.31771E+00,-0.31918E+00,-0.32060E+00,
     &-0.32200E+00,-0.32337E+00,-0.32471E+00,-0.32602E+00,-0.32731E+00,
     &-0.32856E+00,-0.32980E+00,-0.33101E+00,-0.33220E+00,-0.33336E+00,
     &-0.33450E+00,-0.33563E+00,-0.33673E+00,-0.33781E+00,-0.33888E+00,
     &-0.33992E+00,-0.34095E+00,-0.34197E+00,-0.34296E+00,-0.34394E+00,
     &-0.34491E+00,-0.34586E+00,-0.34680E+00,-0.34772E+00,-0.34863E+00,
     &-0.34953E+00,-0.35042E+00,-0.35129E+00,-0.35216E+00,-0.35301E+00,
     &-0.35385E+00,-0.35468E+00,-0.35550E+00,-0.35632E+00,-0.35712E+00,
     &-0.35791E+00,-0.35870E+00,-0.35947E+00,-0.36024E+00,-0.36100E+00,
     &-0.36176E+00,-0.36250E+00,-0.36324E+00,-0.36397E+00,-0.36469E+00,
     &-0.36541E+00,-0.36612E+00,-0.36682E+00,-0.36752E+00,-0.36821E+00/

      DATA (BNC03M(I),I=101,200)/
     &-0.36889E+00,-0.36957E+00,-0.37024E+00,-0.37091E+00,-0.37157E+00,
     &-0.37222E+00,-0.37287E+00,-0.37351E+00,-0.37415E+00,-0.37478E+00,
     &-0.37540E+00,-0.37603E+00,-0.37664E+00,-0.37725E+00,-0.37785E+00,
     &-0.37845E+00,-0.37905E+00,-0.37964E+00,-0.38022E+00,-0.38080E+00,
     &-0.38137E+00,-0.38194E+00,-0.38251E+00,-0.38307E+00,-0.38362E+00,
     &-0.38417E+00,-0.38472E+00,-0.38526E+00,-0.38579E+00,-0.38632E+00,
     &-0.38685E+00,-0.38737E+00,-0.38789E+00,-0.38840E+00,-0.38891E+00,
     &-0.38942E+00,-0.38992E+00,-0.39042E+00,-0.39091E+00,-0.39140E+00,
     &-0.39188E+00,-0.39236E+00,-0.39284E+00,-0.39331E+00,-0.39378E+00,
     &-0.39424E+00,-0.39471E+00,-0.39516E+00,-0.39562E+00,-0.39607E+00,
     &-0.39651E+00,-0.39696E+00,-0.39740E+00,-0.39783E+00,-0.39827E+00,
     &-0.39870E+00,-0.39912E+00,-0.39954E+00,-0.39996E+00,-0.40038E+00,
     &-0.40079E+00,-0.40120E+00,-0.40161E+00,-0.40201E+00,-0.40241E+00,
     &-0.40281E+00,-0.40321E+00,-0.40360E+00,-0.40399E+00,-0.40437E+00,
     &-0.40476E+00,-0.40514E+00,-0.40552E+00,-0.40589E+00,-0.40626E+00,
     &-0.40663E+00,-0.40700E+00,-0.40736E+00,-0.40772E+00,-0.40808E+00,
     &-0.40844E+00,-0.40879E+00,-0.40914E+00,-0.40949E+00,-0.40984E+00,
     &-0.41018E+00,-0.41053E+00,-0.41086E+00,-0.41120E+00,-0.41154E+00,
     &-0.41187E+00,-0.41220E+00,-0.41252E+00,-0.41285E+00,-0.41317E+00,
     &-0.41349E+00,-0.41381E+00,-0.41413E+00,-0.41444E+00,-0.41475E+00/

      DATA (BNC03M(I),I=201,300)/
     &-0.41506E+00,-0.41537E+00,-0.41568E+00,-0.41598E+00,-0.41628E+00,
     &-0.41658E+00,-0.41688E+00,-0.41718E+00,-0.41747E+00,-0.41776E+00,
     &-0.41805E+00,-0.41834E+00,-0.41863E+00,-0.41891E+00,-0.41919E+00,
     &-0.41947E+00,-0.41975E+00,-0.42003E+00,-0.42030E+00,-0.42058E+00,
     &-0.42085E+00,-0.42112E+00,-0.42138E+00,-0.42165E+00,-0.42191E+00,
     &-0.42218E+00,-0.42244E+00,-0.42270E+00,-0.42296E+00,-0.42321E+00,
     &-0.42347E+00,-0.42372E+00,-0.42397E+00,-0.42422E+00,-0.42447E+00,
     &-0.42471E+00,-0.42496E+00,-0.42520E+00,-0.42544E+00,-0.42568E+00,
     &-0.42592E+00,-0.42616E+00,-0.42639E+00,-0.42663E+00,-0.42686E+00,
     &-0.42709E+00,-0.42732E+00,-0.42755E+00,-0.42778E+00,-0.42800E+00,
     &-0.42823E+00,-0.42845E+00,-0.42867E+00,-0.42889E+00,-0.42911E+00,
     &-0.42933E+00,-0.42954E+00,-0.42976E+00,-0.42997E+00,-0.43018E+00,
     &-0.43039E+00,-0.43060E+00,-0.43081E+00,-0.43102E+00,-0.43122E+00,
     &-0.43143E+00,-0.43163E+00,-0.43183E+00,-0.43203E+00,-0.43223E+00,
     &-0.43243E+00,-0.43262E+00,-0.43282E+00,-0.43301E+00,-0.43321E+00,
     &-0.43340E+00,-0.43359E+00,-0.43378E+00,-0.43397E+00,-0.43416E+00,
     &-0.43434E+00,-0.43453E+00,-0.43471E+00,-0.43489E+00,-0.43508E+00,
     &-0.43526E+00,-0.43544E+00,-0.43561E+00,-0.43579E+00,-0.43597E+00,
     &-0.43614E+00,-0.43632E+00,-0.43649E+00,-0.43666E+00,-0.43683E+00,
     &-0.43700E+00,-0.43717E+00,-0.43734E+00,-0.43751E+00,-0.43767E+00/

      DATA (BNC03M(I),I=301,400)/
     &-0.43784E+00,-0.43800E+00,-0.43816E+00,-0.43832E+00,-0.43848E+00,
     &-0.43864E+00,-0.43880E+00,-0.43896E+00,-0.43912E+00,-0.43927E+00,
     &-0.43943E+00,-0.43958E+00,-0.43974E+00,-0.43989E+00,-0.44004E+00,
     &-0.44019E+00,-0.44034E+00,-0.44049E+00,-0.44063E+00,-0.44078E+00,
     &-0.44093E+00,-0.44107E+00,-0.44122E+00,-0.44136E+00,-0.44150E+00,
     &-0.44164E+00,-0.44178E+00,-0.44192E+00,-0.44206E+00,-0.44220E+00,
     &-0.44234E+00,-0.44247E+00,-0.44261E+00,-0.44274E+00,-0.44288E+00,
     &-0.44301E+00,-0.44314E+00,-0.44327E+00,-0.44340E+00,-0.44353E+00,
     &-0.44366E+00,-0.44379E+00,-0.44392E+00,-0.44404E+00,-0.44417E+00,
     &-0.44430E+00,-0.44442E+00,-0.44454E+00,-0.44467E+00,-0.44479E+00,
     &-0.44491E+00,-0.44503E+00,-0.44515E+00,-0.44527E+00,-0.44539E+00,
     &-0.44550E+00,-0.44562E+00,-0.44574E+00,-0.44585E+00,-0.44596E+00,
     &-0.44608E+00,-0.44619E+00,-0.44630E+00,-0.44642E+00,-0.44653E+00,
     &-0.44664E+00,-0.44675E+00,-0.44686E+00,-0.44696E+00,-0.44707E+00,
     &-0.44718E+00,-0.44728E+00,-0.44739E+00,-0.44749E+00,-0.44760E+00,
     &-0.44770E+00,-0.44780E+00,-0.44791E+00,-0.44801E+00,-0.44811E+00,
     &-0.44821E+00,-0.44831E+00,-0.44841E+00,-0.44850E+00,-0.44860E+00,
     &-0.44870E+00,-0.44879E+00,-0.44889E+00,-0.44899E+00,-0.44908E+00,
     &-0.44917E+00,-0.44927E+00,-0.44936E+00,-0.44945E+00,-0.44954E+00,
     &-0.44963E+00,-0.44972E+00,-0.44981E+00,-0.44990E+00,-0.44999E+00/

      DATA (BNC03M(I),I=401,500)/
     &-0.45008E+00,-0.45016E+00,-0.45025E+00,-0.45034E+00,-0.45042E+00,
     &-0.45051E+00,-0.45059E+00,-0.45067E+00,-0.45076E+00,-0.45084E+00,
     &-0.45092E+00,-0.45100E+00,-0.45108E+00,-0.45116E+00,-0.45124E+00,
     &-0.45132E+00,-0.45140E+00,-0.45148E+00,-0.45156E+00,-0.45163E+00,
     &-0.45171E+00,-0.45179E+00,-0.45186E+00,-0.45194E+00,-0.45201E+00,
     &-0.45208E+00,-0.45216E+00,-0.45223E+00,-0.45230E+00,-0.45237E+00,
     &-0.45244E+00,-0.45252E+00,-0.45259E+00,-0.45266E+00,-0.45272E+00,
     &-0.45279E+00,-0.45286E+00,-0.45293E+00,-0.45300E+00,-0.45306E+00,
     &-0.45313E+00,-0.45319E+00,-0.45326E+00,-0.45332E+00,-0.45339E+00,
     &-0.45345E+00,-0.45351E+00,-0.45358E+00,-0.45364E+00,-0.45370E+00,
     &-0.45376E+00,-0.45382E+00,-0.45388E+00,-0.45394E+00,-0.45400E+00,
     &-0.45406E+00,-0.45412E+00,-0.45418E+00,-0.45423E+00,-0.45429E+00,
     &-0.45435E+00,-0.45440E+00,-0.45446E+00,-0.45452E+00,-0.45457E+00,
     &-0.45462E+00,-0.45468E+00,-0.45473E+00,-0.45478E+00,-0.45484E+00,
     &-0.45489E+00,-0.45494E+00,-0.45499E+00,-0.45504E+00,-0.45509E+00,
     &-0.45514E+00,-0.45519E+00,-0.45524E+00,-0.45529E+00,-0.45534E+00,
     &-0.45539E+00,-0.45543E+00,-0.45548E+00,-0.45553E+00,-0.45557E+00,
     &-0.45562E+00,-0.45567E+00,-0.45571E+00,-0.45575E+00,-0.45580E+00,
     &-0.45584E+00,-0.45589E+00,-0.45593E+00,-0.45597E+00,-0.45601E+00,
     &-0.45605E+00,-0.45610E+00,-0.45614E+00,-0.45618E+00,-0.45622E+00/

      DATA (BNC03M(I),I=501,600)/
     &-0.45626E+00,-0.45630E+00,-0.45634E+00,-0.45637E+00,-0.45641E+00,
     &-0.45645E+00,-0.45649E+00,-0.45653E+00,-0.45656E+00,-0.45660E+00,
     &-0.45663E+00,-0.45667E+00,-0.45670E+00,-0.45674E+00,-0.45677E+00,
     &-0.45681E+00,-0.45684E+00,-0.45688E+00,-0.45691E+00,-0.45694E+00,
     &-0.45697E+00,-0.45701E+00,-0.45704E+00,-0.45707E+00,-0.45710E+00,
     &-0.45713E+00,-0.45716E+00,-0.45719E+00,-0.45722E+00,-0.45725E+00,
     &-0.45728E+00,-0.45731E+00,-0.45733E+00,-0.45736E+00,-0.45739E+00,
     &-0.45742E+00,-0.45744E+00,-0.45747E+00,-0.45749E+00,-0.45752E+00,
     &-0.45755E+00,-0.45757E+00,-0.45759E+00,-0.45762E+00,-0.45764E+00,
     &-0.45767E+00,-0.45769E+00,-0.45771E+00,-0.45774E+00,-0.45776E+00,
     &-0.45778E+00,-0.45780E+00,-0.45782E+00,-0.45784E+00,-0.45787E+00,
     &-0.45789E+00,-0.45791E+00,-0.45793E+00,-0.45795E+00,-0.45796E+00,
     &-0.45798E+00,-0.45800E+00,-0.45802E+00,-0.45804E+00,-0.45806E+00,
     &-0.45807E+00,-0.45809E+00,-0.45811E+00,-0.45812E+00,-0.45814E+00,
     &-0.45815E+00,-0.45817E+00,-0.45819E+00,-0.45820E+00,-0.45821E+00,
     &-0.45823E+00,-0.45824E+00,-0.45826E+00,-0.45827E+00,-0.45828E+00,
     &-0.45830E+00,-0.45831E+00,-0.45832E+00,-0.45833E+00,-0.45834E+00,
     &-0.45836E+00,-0.45837E+00,-0.45838E+00,-0.45839E+00,-0.45840E+00,
     &-0.45841E+00,-0.45842E+00,-0.45843E+00,-0.45844E+00,-0.45845E+00,
     &-0.45845E+00,-0.45846E+00,-0.45847E+00,-0.45848E+00,-0.45849E+00/

      DATA (BNC03M(I),I=601,700)/
     &-0.45855E+00,-0.45858E+00,-0.45858E+00,-0.45856E+00,-0.45852E+00,
     &-0.45845E+00,-0.45836E+00,-0.45825E+00,-0.45812E+00,-0.45796E+00,
     &-0.45779E+00,-0.45759E+00,-0.45738E+00,-0.45715E+00,-0.45690E+00,
     &-0.45663E+00,-0.45634E+00,-0.45604E+00,-0.45573E+00,-0.45539E+00,
     &-0.45504E+00,-0.45468E+00,-0.45430E+00,-0.45390E+00,-0.45350E+00,
     &-0.45307E+00,-0.45264E+00,-0.45219E+00,-0.45173E+00,-0.45126E+00,
     &-0.45077E+00,-0.45027E+00,-0.44976E+00,-0.44924E+00,-0.44871E+00,
     &-0.44817E+00,-0.44762E+00,-0.44705E+00,-0.44648E+00,-0.44589E+00,
     &-0.44530E+00,-0.44470E+00,-0.44408E+00,-0.44346E+00,-0.44283E+00,
     &-0.44219E+00,-0.44154E+00,-0.44088E+00,-0.44022E+00,-0.43954E+00,
     &-0.43886E+00,-0.43817E+00,-0.43747E+00,-0.43676E+00,-0.43605E+00,
     &-0.43533E+00,-0.43460E+00,-0.43387E+00,-0.43313E+00,-0.43238E+00,
     &-0.43162E+00,-0.43086E+00,-0.43009E+00,-0.42932E+00,-0.42854E+00,
     &-0.42775E+00,-0.42695E+00,-0.42616E+00,-0.42535E+00,-0.42454E+00,
     &-0.42372E+00,-0.42290E+00,-0.42207E+00,-0.42124E+00,-0.42040E+00,
     &-0.41956E+00,-0.41871E+00,-0.41786E+00,-0.41700E+00,-0.41613E+00,
     &-0.41527E+00,-0.41439E+00,-0.41351E+00,-0.41263E+00,-0.41175E+00,
     &-0.41085E+00,-0.40996E+00,-0.40906E+00,-0.40815E+00,-0.40724E+00,
     &-0.40633E+00,-0.40542E+00,-0.40449E+00,-0.40357E+00,-0.40264E+00,
     &-0.40171E+00,-0.40077E+00,-0.39983E+00,-0.39889E+00,-0.39794E+00/

      DATA (BNC03M(I),I=701,741)/
     &-0.39699E+00,-0.39604E+00,-0.39508E+00,-0.39412E+00,-0.39315E+00,
     &-0.39218E+00,-0.39121E+00,-0.39024E+00,-0.38926E+00,-0.38828E+00,
     &-0.38729E+00,-0.38630E+00,-0.38531E+00,-0.38432E+00,-0.38332E+00,
     &-0.38232E+00,-0.38132E+00,-0.38032E+00,-0.37931E+00,-0.37830E+00,
     &-0.37729E+00,-0.37627E+00,-0.37525E+00,-0.37423E+00,-0.37320E+00,
     &-0.37218E+00,-0.37115E+00,-0.37012E+00,-0.36908E+00,-0.36804E+00,
     &-0.36701E+00,-0.36596E+00,-0.36492E+00,-0.36387E+00,-0.36282E+00,
     &-0.36177E+00,-0.36072E+00,-0.35967E+00,-0.35861E+00,-0.35755E+00,
     &-0.35649E+00
     & /
C
C *** (NH4)2SO4    
C
      DATA (BNC04M(I),I=1,100)/
     &-0.90672E-01,-0.19663E+00,-0.24920E+00,-0.28615E+00,-0.31499E+00,
     &-0.33875E+00,-0.35900E+00,-0.37665E+00,-0.39232E+00,-0.40639E+00,
     &-0.41917E+00,-0.43087E+00,-0.44165E+00,-0.45166E+00,-0.46098E+00,
     &-0.46971E+00,-0.47792E+00,-0.48566E+00,-0.49298E+00,-0.49992E+00,
     &-0.50652E+00,-0.51280E+00,-0.51881E+00,-0.52455E+00,-0.53005E+00,
     &-0.53532E+00,-0.54039E+00,-0.54527E+00,-0.54997E+00,-0.55450E+00,
     &-0.55887E+00,-0.56310E+00,-0.56718E+00,-0.57114E+00,-0.57497E+00,
     &-0.57868E+00,-0.58229E+00,-0.58578E+00,-0.58918E+00,-0.59248E+00,
     &-0.59570E+00,-0.59882E+00,-0.60186E+00,-0.60483E+00,-0.60771E+00,
     &-0.61053E+00,-0.61327E+00,-0.61595E+00,-0.61857E+00,-0.62112E+00,
     &-0.62362E+00,-0.62605E+00,-0.62844E+00,-0.63077E+00,-0.63305E+00,
     &-0.63528E+00,-0.63746E+00,-0.63960E+00,-0.64170E+00,-0.64375E+00,
     &-0.64576E+00,-0.64773E+00,-0.64967E+00,-0.65157E+00,-0.65343E+00,
     &-0.65526E+00,-0.65705E+00,-0.65882E+00,-0.66055E+00,-0.66225E+00,
     &-0.66393E+00,-0.66557E+00,-0.66719E+00,-0.66879E+00,-0.67036E+00,
     &-0.67190E+00,-0.67342E+00,-0.67492E+00,-0.67640E+00,-0.67785E+00,
     &-0.67928E+00,-0.68070E+00,-0.68209E+00,-0.68347E+00,-0.68482E+00,
     &-0.68616E+00,-0.68748E+00,-0.68879E+00,-0.69008E+00,-0.69135E+00,
     &-0.69260E+00,-0.69385E+00,-0.69507E+00,-0.69628E+00,-0.69748E+00,
     &-0.69867E+00,-0.69984E+00,-0.70099E+00,-0.70214E+00,-0.70327E+00/

      DATA (BNC04M(I),I=101,200)/
     &-0.70439E+00,-0.70549E+00,-0.70659E+00,-0.70767E+00,-0.70874E+00,
     &-0.70980E+00,-0.71085E+00,-0.71189E+00,-0.71291E+00,-0.71393E+00,
     &-0.71493E+00,-0.71593E+00,-0.71691E+00,-0.71789E+00,-0.71885E+00,
     &-0.71980E+00,-0.72075E+00,-0.72168E+00,-0.72261E+00,-0.72352E+00,
     &-0.72443E+00,-0.72532E+00,-0.72621E+00,-0.72709E+00,-0.72796E+00,
     &-0.72882E+00,-0.72967E+00,-0.73051E+00,-0.73135E+00,-0.73217E+00,
     &-0.73299E+00,-0.73380E+00,-0.73460E+00,-0.73540E+00,-0.73618E+00,
     &-0.73696E+00,-0.73773E+00,-0.73849E+00,-0.73925E+00,-0.74000E+00,
     &-0.74074E+00,-0.74147E+00,-0.74220E+00,-0.74292E+00,-0.74363E+00,
     &-0.74433E+00,-0.74503E+00,-0.74572E+00,-0.74641E+00,-0.74709E+00,
     &-0.74776E+00,-0.74843E+00,-0.74909E+00,-0.74974E+00,-0.75039E+00,
     &-0.75103E+00,-0.75167E+00,-0.75230E+00,-0.75292E+00,-0.75354E+00,
     &-0.75415E+00,-0.75476E+00,-0.75536E+00,-0.75596E+00,-0.75655E+00,
     &-0.75713E+00,-0.75771E+00,-0.75829E+00,-0.75886E+00,-0.75942E+00,
     &-0.75998E+00,-0.76054E+00,-0.76109E+00,-0.76163E+00,-0.76217E+00,
     &-0.76271E+00,-0.76324E+00,-0.76376E+00,-0.76428E+00,-0.76480E+00,
     &-0.76531E+00,-0.76582E+00,-0.76632E+00,-0.76682E+00,-0.76732E+00,
     &-0.76781E+00,-0.76829E+00,-0.76877E+00,-0.76925E+00,-0.76973E+00,
     &-0.77020E+00,-0.77066E+00,-0.77112E+00,-0.77158E+00,-0.77203E+00,
     &-0.77248E+00,-0.77293E+00,-0.77337E+00,-0.77381E+00,-0.77425E+00/

      DATA (BNC04M(I),I=201,300)/
     &-0.77468E+00,-0.77510E+00,-0.77553E+00,-0.77595E+00,-0.77637E+00,
     &-0.77678E+00,-0.77719E+00,-0.77760E+00,-0.77800E+00,-0.77840E+00,
     &-0.77880E+00,-0.77919E+00,-0.77958E+00,-0.77996E+00,-0.78035E+00,
     &-0.78073E+00,-0.78110E+00,-0.78148E+00,-0.78185E+00,-0.78222E+00,
     &-0.78258E+00,-0.78294E+00,-0.78330E+00,-0.78366E+00,-0.78401E+00,
     &-0.78436E+00,-0.78471E+00,-0.78505E+00,-0.78539E+00,-0.78573E+00,
     &-0.78607E+00,-0.78640E+00,-0.78673E+00,-0.78706E+00,-0.78738E+00,
     &-0.78770E+00,-0.78802E+00,-0.78834E+00,-0.78865E+00,-0.78896E+00,
     &-0.78927E+00,-0.78958E+00,-0.78988E+00,-0.79018E+00,-0.79048E+00,
     &-0.79078E+00,-0.79107E+00,-0.79136E+00,-0.79165E+00,-0.79194E+00,
     &-0.79222E+00,-0.79250E+00,-0.79278E+00,-0.79306E+00,-0.79333E+00,
     &-0.79361E+00,-0.79388E+00,-0.79415E+00,-0.79441E+00,-0.79467E+00,
     &-0.79494E+00,-0.79519E+00,-0.79545E+00,-0.79571E+00,-0.79596E+00,
     &-0.79621E+00,-0.79646E+00,-0.79670E+00,-0.79695E+00,-0.79719E+00,
     &-0.79743E+00,-0.79767E+00,-0.79790E+00,-0.79814E+00,-0.79837E+00,
     &-0.79860E+00,-0.79883E+00,-0.79906E+00,-0.79928E+00,-0.79950E+00,
     &-0.79972E+00,-0.79994E+00,-0.80016E+00,-0.80037E+00,-0.80058E+00,
     &-0.80080E+00,-0.80101E+00,-0.80121E+00,-0.80142E+00,-0.80162E+00,
     &-0.80182E+00,-0.80202E+00,-0.80222E+00,-0.80242E+00,-0.80261E+00,
     &-0.80281E+00,-0.80300E+00,-0.80319E+00,-0.80338E+00,-0.80356E+00/

      DATA (BNC04M(I),I=301,400)/
     &-0.80375E+00,-0.80393E+00,-0.80411E+00,-0.80429E+00,-0.80447E+00,
     &-0.80465E+00,-0.80482E+00,-0.80500E+00,-0.80517E+00,-0.80534E+00,
     &-0.80551E+00,-0.80567E+00,-0.80584E+00,-0.80600E+00,-0.80617E+00,
     &-0.80633E+00,-0.80649E+00,-0.80665E+00,-0.80680E+00,-0.80696E+00,
     &-0.80711E+00,-0.80726E+00,-0.80741E+00,-0.80756E+00,-0.80771E+00,
     &-0.80786E+00,-0.80800E+00,-0.80815E+00,-0.80829E+00,-0.80843E+00,
     &-0.80857E+00,-0.80871E+00,-0.80884E+00,-0.80898E+00,-0.80911E+00,
     &-0.80924E+00,-0.80937E+00,-0.80950E+00,-0.80963E+00,-0.80976E+00,
     &-0.80989E+00,-0.81001E+00,-0.81013E+00,-0.81026E+00,-0.81038E+00,
     &-0.81050E+00,-0.81062E+00,-0.81073E+00,-0.81085E+00,-0.81096E+00,
     &-0.81108E+00,-0.81119E+00,-0.81130E+00,-0.81141E+00,-0.81152E+00,
     &-0.81162E+00,-0.81173E+00,-0.81183E+00,-0.81194E+00,-0.81204E+00,
     &-0.81214E+00,-0.81224E+00,-0.81234E+00,-0.81244E+00,-0.81254E+00,
     &-0.81263E+00,-0.81273E+00,-0.81282E+00,-0.81291E+00,-0.81300E+00,
     &-0.81309E+00,-0.81318E+00,-0.81327E+00,-0.81336E+00,-0.81344E+00,
     &-0.81353E+00,-0.81361E+00,-0.81369E+00,-0.81377E+00,-0.81385E+00,
     &-0.81393E+00,-0.81401E+00,-0.81409E+00,-0.81416E+00,-0.81424E+00,
     &-0.81431E+00,-0.81439E+00,-0.81446E+00,-0.81453E+00,-0.81460E+00,
     &-0.81467E+00,-0.81474E+00,-0.81480E+00,-0.81487E+00,-0.81493E+00,
     &-0.81500E+00,-0.81506E+00,-0.81512E+00,-0.81518E+00,-0.81524E+00/

      DATA (BNC04M(I),I=401,500)/
     &-0.81530E+00,-0.81536E+00,-0.81542E+00,-0.81548E+00,-0.81553E+00,
     &-0.81559E+00,-0.81564E+00,-0.81569E+00,-0.81574E+00,-0.81580E+00,
     &-0.81585E+00,-0.81590E+00,-0.81594E+00,-0.81599E+00,-0.81604E+00,
     &-0.81608E+00,-0.81613E+00,-0.81617E+00,-0.81621E+00,-0.81626E+00,
     &-0.81630E+00,-0.81634E+00,-0.81638E+00,-0.81642E+00,-0.81645E+00,
     &-0.81649E+00,-0.81653E+00,-0.81656E+00,-0.81660E+00,-0.81663E+00,
     &-0.81666E+00,-0.81670E+00,-0.81673E+00,-0.81676E+00,-0.81679E+00,
     &-0.81682E+00,-0.81684E+00,-0.81687E+00,-0.81690E+00,-0.81692E+00,
     &-0.81695E+00,-0.81697E+00,-0.81700E+00,-0.81702E+00,-0.81704E+00,
     &-0.81706E+00,-0.81708E+00,-0.81710E+00,-0.81712E+00,-0.81714E+00,
     &-0.81715E+00,-0.81717E+00,-0.81719E+00,-0.81720E+00,-0.81721E+00,
     &-0.81723E+00,-0.81724E+00,-0.81725E+00,-0.81726E+00,-0.81727E+00,
     &-0.81728E+00,-0.81729E+00,-0.81730E+00,-0.81731E+00,-0.81732E+00,
     &-0.81732E+00,-0.81733E+00,-0.81733E+00,-0.81734E+00,-0.81734E+00,
     &-0.81734E+00,-0.81735E+00,-0.81735E+00,-0.81735E+00,-0.81735E+00,
     &-0.81735E+00,-0.81735E+00,-0.81734E+00,-0.81734E+00,-0.81734E+00,
     &-0.81733E+00,-0.81733E+00,-0.81732E+00,-0.81732E+00,-0.81731E+00,
     &-0.81730E+00,-0.81730E+00,-0.81729E+00,-0.81728E+00,-0.81727E+00,
     &-0.81726E+00,-0.81725E+00,-0.81724E+00,-0.81722E+00,-0.81721E+00,
     &-0.81720E+00,-0.81718E+00,-0.81717E+00,-0.81715E+00,-0.81714E+00/

      DATA (BNC04M(I),I=501,600)/
     &-0.81712E+00,-0.81710E+00,-0.81709E+00,-0.81707E+00,-0.81705E+00,
     &-0.81703E+00,-0.81701E+00,-0.81699E+00,-0.81697E+00,-0.81694E+00,
     &-0.81692E+00,-0.81690E+00,-0.81688E+00,-0.81685E+00,-0.81683E+00,
     &-0.81680E+00,-0.81677E+00,-0.81675E+00,-0.81672E+00,-0.81669E+00,
     &-0.81666E+00,-0.81664E+00,-0.81661E+00,-0.81658E+00,-0.81655E+00,
     &-0.81651E+00,-0.81648E+00,-0.81645E+00,-0.81642E+00,-0.81638E+00,
     &-0.81635E+00,-0.81632E+00,-0.81628E+00,-0.81625E+00,-0.81621E+00,
     &-0.81617E+00,-0.81614E+00,-0.81610E+00,-0.81606E+00,-0.81602E+00,
     &-0.81598E+00,-0.81594E+00,-0.81590E+00,-0.81586E+00,-0.81582E+00,
     &-0.81578E+00,-0.81574E+00,-0.81570E+00,-0.81565E+00,-0.81561E+00,
     &-0.81556E+00,-0.81552E+00,-0.81547E+00,-0.81543E+00,-0.81538E+00,
     &-0.81534E+00,-0.81529E+00,-0.81524E+00,-0.81519E+00,-0.81514E+00,
     &-0.81509E+00,-0.81504E+00,-0.81499E+00,-0.81494E+00,-0.81489E+00,
     &-0.81484E+00,-0.81479E+00,-0.81474E+00,-0.81468E+00,-0.81463E+00,
     &-0.81458E+00,-0.81452E+00,-0.81447E+00,-0.81441E+00,-0.81436E+00,
     &-0.81430E+00,-0.81424E+00,-0.81419E+00,-0.81413E+00,-0.81407E+00,
     &-0.81401E+00,-0.81395E+00,-0.81389E+00,-0.81383E+00,-0.81377E+00,
     &-0.81371E+00,-0.81365E+00,-0.81359E+00,-0.81353E+00,-0.81347E+00,
     &-0.81340E+00,-0.81334E+00,-0.81328E+00,-0.81321E+00,-0.81315E+00,
     &-0.81308E+00,-0.81302E+00,-0.81295E+00,-0.81288E+00,-0.81282E+00/

      DATA (BNC04M(I),I=601,700)/
     &-0.81207E+00,-0.81133E+00,-0.81056E+00,-0.80975E+00,-0.80890E+00,
     &-0.80801E+00,-0.80709E+00,-0.80614E+00,-0.80515E+00,-0.80414E+00,
     &-0.80309E+00,-0.80201E+00,-0.80090E+00,-0.79977E+00,-0.79861E+00,
     &-0.79742E+00,-0.79620E+00,-0.79496E+00,-0.79370E+00,-0.79241E+00,
     &-0.79110E+00,-0.78976E+00,-0.78841E+00,-0.78703E+00,-0.78563E+00,
     &-0.78421E+00,-0.78277E+00,-0.78131E+00,-0.77984E+00,-0.77834E+00,
     &-0.77683E+00,-0.77529E+00,-0.77374E+00,-0.77218E+00,-0.77060E+00,
     &-0.76900E+00,-0.76738E+00,-0.76575E+00,-0.76411E+00,-0.76245E+00,
     &-0.76077E+00,-0.75908E+00,-0.75738E+00,-0.75566E+00,-0.75393E+00,
     &-0.75219E+00,-0.75044E+00,-0.74867E+00,-0.74689E+00,-0.74510E+00,
     &-0.74329E+00,-0.74148E+00,-0.73965E+00,-0.73781E+00,-0.73596E+00,
     &-0.73411E+00,-0.73224E+00,-0.73036E+00,-0.72847E+00,-0.72657E+00,
     &-0.72466E+00,-0.72274E+00,-0.72081E+00,-0.71887E+00,-0.71692E+00,
     &-0.71497E+00,-0.71300E+00,-0.71103E+00,-0.70905E+00,-0.70706E+00,
     &-0.70506E+00,-0.70306E+00,-0.70104E+00,-0.69902E+00,-0.69699E+00,
     &-0.69496E+00,-0.69291E+00,-0.69086E+00,-0.68880E+00,-0.68674E+00,
     &-0.68467E+00,-0.68259E+00,-0.68050E+00,-0.67841E+00,-0.67631E+00,
     &-0.67421E+00,-0.67210E+00,-0.66998E+00,-0.66786E+00,-0.66573E+00,
     &-0.66359E+00,-0.66145E+00,-0.65930E+00,-0.65715E+00,-0.65499E+00,
     &-0.65283E+00,-0.65066E+00,-0.64849E+00,-0.64631E+00,-0.64413E+00/

      DATA (BNC04M(I),I=701,741)/
     &-0.64194E+00,-0.63974E+00,-0.63755E+00,-0.63534E+00,-0.63313E+00,
     &-0.63092E+00,-0.62870E+00,-0.62648E+00,-0.62425E+00,-0.62202E+00,
     &-0.61979E+00,-0.61755E+00,-0.61530E+00,-0.61305E+00,-0.61080E+00,
     &-0.60854E+00,-0.60628E+00,-0.60402E+00,-0.60175E+00,-0.59948E+00,
     &-0.59720E+00,-0.59492E+00,-0.59264E+00,-0.59035E+00,-0.58806E+00,
     &-0.58576E+00,-0.58346E+00,-0.58116E+00,-0.57886E+00,-0.57655E+00,
     &-0.57424E+00,-0.57192E+00,-0.56960E+00,-0.56728E+00,-0.56496E+00,
     &-0.56263E+00,-0.56030E+00,-0.55796E+00,-0.55562E+00,-0.55328E+00,
     &-0.55094E+00
     & /
C
C *** NH4NO3       
C
      DATA (BNC05M(I),I=1,100)/
     &-0.45808E-01,-0.10090E+00,-0.12914E+00,-0.14948E+00,-0.16569E+00,
     &-0.17932E+00,-0.19115E+00,-0.20166E+00,-0.21113E+00,-0.21978E+00,
     &-0.22776E+00,-0.23518E+00,-0.24211E+00,-0.24864E+00,-0.25481E+00,
     &-0.26066E+00,-0.26624E+00,-0.27156E+00,-0.27665E+00,-0.28154E+00,
     &-0.28625E+00,-0.29078E+00,-0.29515E+00,-0.29938E+00,-0.30348E+00,
     &-0.30744E+00,-0.31129E+00,-0.31503E+00,-0.31867E+00,-0.32220E+00,
     &-0.32565E+00,-0.32900E+00,-0.33228E+00,-0.33547E+00,-0.33859E+00,
     &-0.34163E+00,-0.34461E+00,-0.34752E+00,-0.35037E+00,-0.35315E+00,
     &-0.35588E+00,-0.35855E+00,-0.36117E+00,-0.36373E+00,-0.36625E+00,
     &-0.36872E+00,-0.37113E+00,-0.37351E+00,-0.37584E+00,-0.37813E+00,
     &-0.38038E+00,-0.38259E+00,-0.38476E+00,-0.38689E+00,-0.38899E+00,
     &-0.39106E+00,-0.39309E+00,-0.39509E+00,-0.39706E+00,-0.39900E+00,
     &-0.40091E+00,-0.40280E+00,-0.40466E+00,-0.40649E+00,-0.40830E+00,
     &-0.41009E+00,-0.41185E+00,-0.41359E+00,-0.41531E+00,-0.41701E+00,
     &-0.41870E+00,-0.42036E+00,-0.42201E+00,-0.42364E+00,-0.42526E+00,
     &-0.42686E+00,-0.42844E+00,-0.43001E+00,-0.43157E+00,-0.43312E+00,
     &-0.43465E+00,-0.43617E+00,-0.43768E+00,-0.43918E+00,-0.44067E+00,
     &-0.44215E+00,-0.44362E+00,-0.44508E+00,-0.44653E+00,-0.44798E+00,
     &-0.44941E+00,-0.45084E+00,-0.45226E+00,-0.45367E+00,-0.45508E+00,
     &-0.45647E+00,-0.45786E+00,-0.45924E+00,-0.46062E+00,-0.46199E+00/

      DATA (BNC05M(I),I=101,200)/
     &-0.46335E+00,-0.46470E+00,-0.46605E+00,-0.46739E+00,-0.46872E+00,
     &-0.47005E+00,-0.47137E+00,-0.47269E+00,-0.47399E+00,-0.47529E+00,
     &-0.47658E+00,-0.47787E+00,-0.47915E+00,-0.48042E+00,-0.48169E+00,
     &-0.48295E+00,-0.48420E+00,-0.48544E+00,-0.48668E+00,-0.48791E+00,
     &-0.48913E+00,-0.49035E+00,-0.49156E+00,-0.49276E+00,-0.49396E+00,
     &-0.49515E+00,-0.49633E+00,-0.49750E+00,-0.49867E+00,-0.49983E+00,
     &-0.50099E+00,-0.50213E+00,-0.50327E+00,-0.50441E+00,-0.50553E+00,
     &-0.50665E+00,-0.50777E+00,-0.50887E+00,-0.50997E+00,-0.51107E+00,
     &-0.51215E+00,-0.51323E+00,-0.51431E+00,-0.51538E+00,-0.51644E+00,
     &-0.51749E+00,-0.51854E+00,-0.51959E+00,-0.52062E+00,-0.52166E+00,
     &-0.52268E+00,-0.52370E+00,-0.52471E+00,-0.52572E+00,-0.52672E+00,
     &-0.52772E+00,-0.52871E+00,-0.52969E+00,-0.53067E+00,-0.53165E+00,
     &-0.53261E+00,-0.53358E+00,-0.53454E+00,-0.53549E+00,-0.53643E+00,
     &-0.53738E+00,-0.53831E+00,-0.53924E+00,-0.54017E+00,-0.54109E+00,
     &-0.54201E+00,-0.54292E+00,-0.54383E+00,-0.54473E+00,-0.54563E+00,
     &-0.54652E+00,-0.54741E+00,-0.54829E+00,-0.54917E+00,-0.55004E+00,
     &-0.55091E+00,-0.55177E+00,-0.55263E+00,-0.55349E+00,-0.55434E+00,
     &-0.55518E+00,-0.55603E+00,-0.55686E+00,-0.55770E+00,-0.55853E+00,
     &-0.55935E+00,-0.56017E+00,-0.56099E+00,-0.56180E+00,-0.56261E+00,
     &-0.56341E+00,-0.56422E+00,-0.56501E+00,-0.56580E+00,-0.56659E+00/

      DATA (BNC05M(I),I=201,300)/
     &-0.56738E+00,-0.56816E+00,-0.56893E+00,-0.56971E+00,-0.57048E+00,
     &-0.57124E+00,-0.57200E+00,-0.57276E+00,-0.57351E+00,-0.57426E+00,
     &-0.57501E+00,-0.57575E+00,-0.57649E+00,-0.57723E+00,-0.57796E+00,
     &-0.57869E+00,-0.57942E+00,-0.58014E+00,-0.58086E+00,-0.58157E+00,
     &-0.58228E+00,-0.58299E+00,-0.58370E+00,-0.58440E+00,-0.58510E+00,
     &-0.58579E+00,-0.58648E+00,-0.58717E+00,-0.58786E+00,-0.58854E+00,
     &-0.58922E+00,-0.58989E+00,-0.59057E+00,-0.59124E+00,-0.59190E+00,
     &-0.59257E+00,-0.59323E+00,-0.59388E+00,-0.59454E+00,-0.59519E+00,
     &-0.59584E+00,-0.59648E+00,-0.59712E+00,-0.59776E+00,-0.59840E+00,
     &-0.59903E+00,-0.59966E+00,-0.60029E+00,-0.60092E+00,-0.60154E+00,
     &-0.60216E+00,-0.60278E+00,-0.60339E+00,-0.60400E+00,-0.60461E+00,
     &-0.60522E+00,-0.60582E+00,-0.60642E+00,-0.60702E+00,-0.60761E+00,
     &-0.60820E+00,-0.60879E+00,-0.60938E+00,-0.60997E+00,-0.61055E+00,
     &-0.61113E+00,-0.61170E+00,-0.61228E+00,-0.61285E+00,-0.61342E+00,
     &-0.61399E+00,-0.61455E+00,-0.61511E+00,-0.61567E+00,-0.61623E+00,
     &-0.61678E+00,-0.61734E+00,-0.61789E+00,-0.61843E+00,-0.61898E+00,
     &-0.61952E+00,-0.62006E+00,-0.62060E+00,-0.62114E+00,-0.62167E+00,
     &-0.62220E+00,-0.62273E+00,-0.62326E+00,-0.62378E+00,-0.62430E+00,
     &-0.62482E+00,-0.62534E+00,-0.62586E+00,-0.62637E+00,-0.62688E+00,
     &-0.62739E+00,-0.62790E+00,-0.62840E+00,-0.62891E+00,-0.62941E+00/

      DATA (BNC05M(I),I=301,400)/
     &-0.62990E+00,-0.63040E+00,-0.63090E+00,-0.63139E+00,-0.63188E+00,
     &-0.63237E+00,-0.63285E+00,-0.63334E+00,-0.63382E+00,-0.63430E+00,
     &-0.63478E+00,-0.63525E+00,-0.63573E+00,-0.63620E+00,-0.63667E+00,
     &-0.63714E+00,-0.63760E+00,-0.63807E+00,-0.63853E+00,-0.63899E+00,
     &-0.63945E+00,-0.63991E+00,-0.64036E+00,-0.64081E+00,-0.64127E+00,
     &-0.64171E+00,-0.64216E+00,-0.64261E+00,-0.64305E+00,-0.64349E+00,
     &-0.64393E+00,-0.64437E+00,-0.64481E+00,-0.64524E+00,-0.64568E+00,
     &-0.64611E+00,-0.64654E+00,-0.64696E+00,-0.64739E+00,-0.64782E+00,
     &-0.64824E+00,-0.64866E+00,-0.64908E+00,-0.64950E+00,-0.64991E+00,
     &-0.65033E+00,-0.65074E+00,-0.65115E+00,-0.65156E+00,-0.65197E+00,
     &-0.65237E+00,-0.65278E+00,-0.65318E+00,-0.65358E+00,-0.65398E+00,
     &-0.65438E+00,-0.65477E+00,-0.65517E+00,-0.65556E+00,-0.65595E+00,
     &-0.65634E+00,-0.65673E+00,-0.65712E+00,-0.65750E+00,-0.65788E+00,
     &-0.65827E+00,-0.65865E+00,-0.65903E+00,-0.65940E+00,-0.65978E+00,
     &-0.66016E+00,-0.66053E+00,-0.66090E+00,-0.66127E+00,-0.66164E+00,
     &-0.66201E+00,-0.66237E+00,-0.66274E+00,-0.66310E+00,-0.66346E+00,
     &-0.66382E+00,-0.66418E+00,-0.66454E+00,-0.66489E+00,-0.66525E+00,
     &-0.66560E+00,-0.66595E+00,-0.66630E+00,-0.66665E+00,-0.66700E+00,
     &-0.66734E+00,-0.66769E+00,-0.66803E+00,-0.66837E+00,-0.66871E+00,
     &-0.66905E+00,-0.66939E+00,-0.66973E+00,-0.67006E+00,-0.67040E+00/

      DATA (BNC05M(I),I=401,500)/
     &-0.67073E+00,-0.67106E+00,-0.67139E+00,-0.67172E+00,-0.67205E+00,
     &-0.67237E+00,-0.67270E+00,-0.67302E+00,-0.67335E+00,-0.67367E+00,
     &-0.67399E+00,-0.67431E+00,-0.67462E+00,-0.67494E+00,-0.67525E+00,
     &-0.67557E+00,-0.67588E+00,-0.67619E+00,-0.67650E+00,-0.67681E+00,
     &-0.67712E+00,-0.67742E+00,-0.67773E+00,-0.67803E+00,-0.67834E+00,
     &-0.67864E+00,-0.67894E+00,-0.67924E+00,-0.67954E+00,-0.67983E+00,
     &-0.68013E+00,-0.68042E+00,-0.68072E+00,-0.68101E+00,-0.68130E+00,
     &-0.68159E+00,-0.68188E+00,-0.68217E+00,-0.68246E+00,-0.68274E+00,
     &-0.68303E+00,-0.68331E+00,-0.68359E+00,-0.68387E+00,-0.68415E+00,
     &-0.68443E+00,-0.68471E+00,-0.68499E+00,-0.68526E+00,-0.68554E+00,
     &-0.68581E+00,-0.68608E+00,-0.68635E+00,-0.68663E+00,-0.68689E+00,
     &-0.68716E+00,-0.68743E+00,-0.68770E+00,-0.68796E+00,-0.68823E+00,
     &-0.68849E+00,-0.68875E+00,-0.68901E+00,-0.68927E+00,-0.68953E+00,
     &-0.68979E+00,-0.69005E+00,-0.69030E+00,-0.69056E+00,-0.69081E+00,
     &-0.69107E+00,-0.69132E+00,-0.69157E+00,-0.69182E+00,-0.69207E+00,
     &-0.69232E+00,-0.69256E+00,-0.69281E+00,-0.69306E+00,-0.69330E+00,
     &-0.69354E+00,-0.69379E+00,-0.69403E+00,-0.69427E+00,-0.69451E+00,
     &-0.69475E+00,-0.69498E+00,-0.69522E+00,-0.69546E+00,-0.69569E+00,
     &-0.69593E+00,-0.69616E+00,-0.69639E+00,-0.69662E+00,-0.69685E+00,
     &-0.69708E+00,-0.69731E+00,-0.69754E+00,-0.69777E+00,-0.69799E+00/

      DATA (BNC05M(I),I=501,600)/
     &-0.69822E+00,-0.69844E+00,-0.69867E+00,-0.69889E+00,-0.69911E+00,
     &-0.69933E+00,-0.69955E+00,-0.69977E+00,-0.69999E+00,-0.70020E+00,
     &-0.70042E+00,-0.70064E+00,-0.70085E+00,-0.70107E+00,-0.70128E+00,
     &-0.70149E+00,-0.70170E+00,-0.70191E+00,-0.70212E+00,-0.70233E+00,
     &-0.70254E+00,-0.70275E+00,-0.70295E+00,-0.70316E+00,-0.70336E+00,
     &-0.70357E+00,-0.70377E+00,-0.70397E+00,-0.70417E+00,-0.70438E+00,
     &-0.70458E+00,-0.70477E+00,-0.70497E+00,-0.70517E+00,-0.70537E+00,
     &-0.70556E+00,-0.70576E+00,-0.70595E+00,-0.70615E+00,-0.70634E+00,
     &-0.70653E+00,-0.70672E+00,-0.70691E+00,-0.70710E+00,-0.70729E+00,
     &-0.70748E+00,-0.70767E+00,-0.70786E+00,-0.70804E+00,-0.70823E+00,
     &-0.70841E+00,-0.70860E+00,-0.70878E+00,-0.70896E+00,-0.70914E+00,
     &-0.70932E+00,-0.70950E+00,-0.70968E+00,-0.70986E+00,-0.71004E+00,
     &-0.71022E+00,-0.71040E+00,-0.71057E+00,-0.71075E+00,-0.71092E+00,
     &-0.71109E+00,-0.71127E+00,-0.71144E+00,-0.71161E+00,-0.71178E+00,
     &-0.71195E+00,-0.71212E+00,-0.71229E+00,-0.71246E+00,-0.71263E+00,
     &-0.71279E+00,-0.71296E+00,-0.71313E+00,-0.71329E+00,-0.71346E+00,
     &-0.71362E+00,-0.71378E+00,-0.71394E+00,-0.71411E+00,-0.71427E+00,
     &-0.71443E+00,-0.71459E+00,-0.71474E+00,-0.71490E+00,-0.71506E+00,
     &-0.71522E+00,-0.71537E+00,-0.71553E+00,-0.71568E+00,-0.71584E+00,
     &-0.71599E+00,-0.71614E+00,-0.71630E+00,-0.71645E+00,-0.71660E+00/

      DATA (BNC05M(I),I=601,700)/
     &-0.71819E+00,-0.71960E+00,-0.72096E+00,-0.72225E+00,-0.72349E+00,
     &-0.72467E+00,-0.72581E+00,-0.72689E+00,-0.72792E+00,-0.72891E+00,
     &-0.72985E+00,-0.73074E+00,-0.73159E+00,-0.73240E+00,-0.73316E+00,
     &-0.73389E+00,-0.73457E+00,-0.73522E+00,-0.73583E+00,-0.73640E+00,
     &-0.73694E+00,-0.73744E+00,-0.73791E+00,-0.73835E+00,-0.73875E+00,
     &-0.73912E+00,-0.73947E+00,-0.73978E+00,-0.74006E+00,-0.74032E+00,
     &-0.74055E+00,-0.74075E+00,-0.74093E+00,-0.74107E+00,-0.74120E+00,
     &-0.74130E+00,-0.74137E+00,-0.74142E+00,-0.74145E+00,-0.74146E+00,
     &-0.74144E+00,-0.74141E+00,-0.74135E+00,-0.74127E+00,-0.74117E+00,
     &-0.74105E+00,-0.74092E+00,-0.74076E+00,-0.74058E+00,-0.74039E+00,
     &-0.74018E+00,-0.73995E+00,-0.73971E+00,-0.73945E+00,-0.73917E+00,
     &-0.73887E+00,-0.73856E+00,-0.73824E+00,-0.73790E+00,-0.73754E+00,
     &-0.73717E+00,-0.73679E+00,-0.73639E+00,-0.73598E+00,-0.73555E+00,
     &-0.73512E+00,-0.73466E+00,-0.73420E+00,-0.73372E+00,-0.73324E+00,
     &-0.73273E+00,-0.73222E+00,-0.73170E+00,-0.73116E+00,-0.73062E+00,
     &-0.73006E+00,-0.72949E+00,-0.72891E+00,-0.72832E+00,-0.72772E+00,
     &-0.72711E+00,-0.72649E+00,-0.72587E+00,-0.72523E+00,-0.72458E+00,
     &-0.72392E+00,-0.72325E+00,-0.72258E+00,-0.72190E+00,-0.72120E+00,
     &-0.72050E+00,-0.71979E+00,-0.71907E+00,-0.71835E+00,-0.71762E+00,
     &-0.71687E+00,-0.71613E+00,-0.71537E+00,-0.71460E+00,-0.71383E+00/

      DATA (BNC05M(I),I=701,741)/
     &-0.71305E+00,-0.71227E+00,-0.71148E+00,-0.71068E+00,-0.70987E+00,
     &-0.70906E+00,-0.70824E+00,-0.70741E+00,-0.70658E+00,-0.70574E+00,
     &-0.70490E+00,-0.70404E+00,-0.70319E+00,-0.70232E+00,-0.70146E+00,
     &-0.70058E+00,-0.69970E+00,-0.69882E+00,-0.69792E+00,-0.69703E+00,
     &-0.69613E+00,-0.69522E+00,-0.69431E+00,-0.69339E+00,-0.69247E+00,
     &-0.69154E+00,-0.69061E+00,-0.68967E+00,-0.68873E+00,-0.68778E+00,
     &-0.68683E+00,-0.68588E+00,-0.68492E+00,-0.68395E+00,-0.68298E+00,
     &-0.68201E+00,-0.68103E+00,-0.68005E+00,-0.67907E+00,-0.67808E+00,
     &-0.67708E+00
     & /
C
C *** NH4Cl        
C
      DATA (BNC06M(I),I=1,100)/
     &-0.44830E-01,-0.95566E-01,-0.11981E+00,-0.13634E+00,-0.14889E+00,
     &-0.15895E+00,-0.16730E+00,-0.17439E+00,-0.18052E+00,-0.18588E+00,
     &-0.19061E+00,-0.19483E+00,-0.19861E+00,-0.20202E+00,-0.20511E+00,
     &-0.20792E+00,-0.21049E+00,-0.21283E+00,-0.21498E+00,-0.21696E+00,
     &-0.21878E+00,-0.22046E+00,-0.22201E+00,-0.22344E+00,-0.22477E+00,
     &-0.22600E+00,-0.22714E+00,-0.22819E+00,-0.22917E+00,-0.23007E+00,
     &-0.23091E+00,-0.23169E+00,-0.23242E+00,-0.23308E+00,-0.23370E+00,
     &-0.23428E+00,-0.23481E+00,-0.23530E+00,-0.23575E+00,-0.23616E+00,
     &-0.23655E+00,-0.23690E+00,-0.23722E+00,-0.23751E+00,-0.23778E+00,
     &-0.23802E+00,-0.23824E+00,-0.23843E+00,-0.23861E+00,-0.23876E+00,
     &-0.23890E+00,-0.23901E+00,-0.23911E+00,-0.23919E+00,-0.23926E+00,
     &-0.23931E+00,-0.23934E+00,-0.23936E+00,-0.23937E+00,-0.23936E+00,
     &-0.23934E+00,-0.23931E+00,-0.23926E+00,-0.23921E+00,-0.23914E+00,
     &-0.23906E+00,-0.23896E+00,-0.23886E+00,-0.23874E+00,-0.23862E+00,
     &-0.23848E+00,-0.23833E+00,-0.23818E+00,-0.23801E+00,-0.23783E+00,
     &-0.23764E+00,-0.23744E+00,-0.23723E+00,-0.23701E+00,-0.23678E+00,
     &-0.23654E+00,-0.23630E+00,-0.23604E+00,-0.23577E+00,-0.23549E+00,
     &-0.23520E+00,-0.23491E+00,-0.23460E+00,-0.23429E+00,-0.23397E+00,
     &-0.23363E+00,-0.23329E+00,-0.23295E+00,-0.23259E+00,-0.23222E+00,
     &-0.23185E+00,-0.23147E+00,-0.23108E+00,-0.23069E+00,-0.23028E+00/

      DATA (BNC06M(I),I=101,200)/
     &-0.22987E+00,-0.22946E+00,-0.22903E+00,-0.22860E+00,-0.22817E+00,
     &-0.22773E+00,-0.22728E+00,-0.22682E+00,-0.22637E+00,-0.22590E+00,
     &-0.22543E+00,-0.22496E+00,-0.22448E+00,-0.22400E+00,-0.22351E+00,
     &-0.22302E+00,-0.22253E+00,-0.22203E+00,-0.22153E+00,-0.22102E+00,
     &-0.22051E+00,-0.22000E+00,-0.21949E+00,-0.21897E+00,-0.21845E+00,
     &-0.21792E+00,-0.21740E+00,-0.21687E+00,-0.21634E+00,-0.21581E+00,
     &-0.21527E+00,-0.21474E+00,-0.21420E+00,-0.21366E+00,-0.21311E+00,
     &-0.21257E+00,-0.21202E+00,-0.21148E+00,-0.21093E+00,-0.21038E+00,
     &-0.20983E+00,-0.20927E+00,-0.20872E+00,-0.20816E+00,-0.20761E+00,
     &-0.20705E+00,-0.20649E+00,-0.20593E+00,-0.20537E+00,-0.20480E+00,
     &-0.20424E+00,-0.20368E+00,-0.20311E+00,-0.20254E+00,-0.20198E+00,
     &-0.20141E+00,-0.20084E+00,-0.20027E+00,-0.19970E+00,-0.19913E+00,
     &-0.19856E+00,-0.19799E+00,-0.19742E+00,-0.19684E+00,-0.19627E+00,
     &-0.19569E+00,-0.19512E+00,-0.19454E+00,-0.19397E+00,-0.19339E+00,
     &-0.19281E+00,-0.19224E+00,-0.19166E+00,-0.19108E+00,-0.19050E+00,
     &-0.18992E+00,-0.18934E+00,-0.18876E+00,-0.18818E+00,-0.18760E+00,
     &-0.18702E+00,-0.18644E+00,-0.18586E+00,-0.18528E+00,-0.18469E+00,
     &-0.18411E+00,-0.18353E+00,-0.18295E+00,-0.18236E+00,-0.18178E+00,
     &-0.18120E+00,-0.18061E+00,-0.18003E+00,-0.17945E+00,-0.17886E+00,
     &-0.17828E+00,-0.17770E+00,-0.17711E+00,-0.17653E+00,-0.17594E+00/

      DATA (BNC06M(I),I=201,300)/
     &-0.17536E+00,-0.17478E+00,-0.17419E+00,-0.17361E+00,-0.17302E+00,
     &-0.17244E+00,-0.17186E+00,-0.17127E+00,-0.17069E+00,-0.17010E+00,
     &-0.16952E+00,-0.16893E+00,-0.16835E+00,-0.16777E+00,-0.16718E+00,
     &-0.16660E+00,-0.16601E+00,-0.16543E+00,-0.16485E+00,-0.16426E+00,
     &-0.16368E+00,-0.16310E+00,-0.16251E+00,-0.16193E+00,-0.16135E+00,
     &-0.16076E+00,-0.16018E+00,-0.15960E+00,-0.15902E+00,-0.15843E+00,
     &-0.15785E+00,-0.15727E+00,-0.15669E+00,-0.15611E+00,-0.15553E+00,
     &-0.15494E+00,-0.15436E+00,-0.15378E+00,-0.15320E+00,-0.15262E+00,
     &-0.15204E+00,-0.15146E+00,-0.15088E+00,-0.15030E+00,-0.14972E+00,
     &-0.14914E+00,-0.14857E+00,-0.14799E+00,-0.14741E+00,-0.14683E+00,
     &-0.14625E+00,-0.14567E+00,-0.14510E+00,-0.14452E+00,-0.14394E+00,
     &-0.14337E+00,-0.14279E+00,-0.14221E+00,-0.14164E+00,-0.14106E+00,
     &-0.14049E+00,-0.13991E+00,-0.13934E+00,-0.13876E+00,-0.13819E+00,
     &-0.13762E+00,-0.13704E+00,-0.13647E+00,-0.13590E+00,-0.13532E+00,
     &-0.13475E+00,-0.13418E+00,-0.13361E+00,-0.13304E+00,-0.13247E+00,
     &-0.13190E+00,-0.13132E+00,-0.13075E+00,-0.13019E+00,-0.12962E+00,
     &-0.12905E+00,-0.12848E+00,-0.12791E+00,-0.12734E+00,-0.12677E+00,
     &-0.12621E+00,-0.12564E+00,-0.12507E+00,-0.12451E+00,-0.12394E+00,
     &-0.12337E+00,-0.12281E+00,-0.12224E+00,-0.12168E+00,-0.12111E+00,
     &-0.12055E+00,-0.11999E+00,-0.11942E+00,-0.11886E+00,-0.11830E+00/

      DATA (BNC06M(I),I=301,400)/
     &-0.11773E+00,-0.11717E+00,-0.11661E+00,-0.11605E+00,-0.11549E+00,
     &-0.11493E+00,-0.11437E+00,-0.11381E+00,-0.11325E+00,-0.11269E+00,
     &-0.11213E+00,-0.11157E+00,-0.11101E+00,-0.11046E+00,-0.10990E+00,
     &-0.10934E+00,-0.10879E+00,-0.10823E+00,-0.10767E+00,-0.10712E+00,
     &-0.10656E+00,-0.10601E+00,-0.10546E+00,-0.10490E+00,-0.10435E+00,
     &-0.10379E+00,-0.10324E+00,-0.10269E+00,-0.10214E+00,-0.10159E+00,
     &-0.10103E+00,-0.10048E+00,-0.99932E-01,-0.99382E-01,-0.98832E-01,
     &-0.98283E-01,-0.97734E-01,-0.97185E-01,-0.96637E-01,-0.96089E-01,
     &-0.95541E-01,-0.94994E-01,-0.94447E-01,-0.93901E-01,-0.93355E-01,
     &-0.92809E-01,-0.92264E-01,-0.91719E-01,-0.91174E-01,-0.90630E-01,
     &-0.90086E-01,-0.89543E-01,-0.89000E-01,-0.88457E-01,-0.87915E-01,
     &-0.87373E-01,-0.86831E-01,-0.86290E-01,-0.85750E-01,-0.85209E-01,
     &-0.84669E-01,-0.84130E-01,-0.83590E-01,-0.83051E-01,-0.82513E-01,
     &-0.81975E-01,-0.81437E-01,-0.80900E-01,-0.80363E-01,-0.79826E-01,
     &-0.79290E-01,-0.78754E-01,-0.78219E-01,-0.77684E-01,-0.77149E-01,
     &-0.76615E-01,-0.76081E-01,-0.75547E-01,-0.75014E-01,-0.74481E-01,
     &-0.73949E-01,-0.73417E-01,-0.72885E-01,-0.72354E-01,-0.71823E-01,
     &-0.71293E-01,-0.70763E-01,-0.70233E-01,-0.69703E-01,-0.69174E-01,
     &-0.68646E-01,-0.68117E-01,-0.67590E-01,-0.67062E-01,-0.66535E-01,
     &-0.66008E-01,-0.65482E-01,-0.64956E-01,-0.64430E-01,-0.63905E-01/

      DATA (BNC06M(I),I=401,500)/
     &-0.63380E-01,-0.62856E-01,-0.62331E-01,-0.61808E-01,-0.61284E-01,
     &-0.60761E-01,-0.60239E-01,-0.59716E-01,-0.59194E-01,-0.58673E-01,
     &-0.58152E-01,-0.57631E-01,-0.57111E-01,-0.56591E-01,-0.56071E-01,
     &-0.55552E-01,-0.55033E-01,-0.54514E-01,-0.53996E-01,-0.53478E-01,
     &-0.52961E-01,-0.52444E-01,-0.51927E-01,-0.51410E-01,-0.50894E-01,
     &-0.50379E-01,-0.49864E-01,-0.49349E-01,-0.48834E-01,-0.48320E-01,
     &-0.47806E-01,-0.47293E-01,-0.46780E-01,-0.46267E-01,-0.45755E-01,
     &-0.45243E-01,-0.44731E-01,-0.44220E-01,-0.43709E-01,-0.43198E-01,
     &-0.42688E-01,-0.42178E-01,-0.41669E-01,-0.41160E-01,-0.40651E-01,
     &-0.40143E-01,-0.39635E-01,-0.39127E-01,-0.38620E-01,-0.38113E-01,
     &-0.37606E-01,-0.37100E-01,-0.36594E-01,-0.36088E-01,-0.35583E-01,
     &-0.35078E-01,-0.34574E-01,-0.34070E-01,-0.33566E-01,-0.33063E-01,
     &-0.32560E-01,-0.32057E-01,-0.31554E-01,-0.31052E-01,-0.30551E-01,
     &-0.30049E-01,-0.29548E-01,-0.29048E-01,-0.28548E-01,-0.28048E-01,
     &-0.27548E-01,-0.27049E-01,-0.26550E-01,-0.26051E-01,-0.25553E-01,
     &-0.25055E-01,-0.24558E-01,-0.24061E-01,-0.23564E-01,-0.23067E-01,
     &-0.22571E-01,-0.22075E-01,-0.21580E-01,-0.21085E-01,-0.20590E-01,
     &-0.20096E-01,-0.19602E-01,-0.19108E-01,-0.18614E-01,-0.18121E-01,
     &-0.17629E-01,-0.17136E-01,-0.16644E-01,-0.16153E-01,-0.15661E-01,
     &-0.15170E-01,-0.14679E-01,-0.14189E-01,-0.13699E-01,-0.13209E-01/

      DATA (BNC06M(I),I=501,600)/
     &-0.12720E-01,-0.12231E-01,-0.11742E-01,-0.11254E-01,-0.10766E-01,
     &-0.10278E-01,-0.97905E-02,-0.93034E-02,-0.88167E-02,-0.83303E-02,
     &-0.78442E-02,-0.73585E-02,-0.68730E-02,-0.63879E-02,-0.59032E-02,
     &-0.54187E-02,-0.49346E-02,-0.44508E-02,-0.39674E-02,-0.34842E-02,
     &-0.30014E-02,-0.25189E-02,-0.20367E-02,-0.15548E-02,-0.10733E-02,
     &-0.59210E-03,-0.11121E-03, 0.36935E-03, 0.84963E-03, 0.13296E-02,
     & 0.18092E-02, 0.22885E-02, 0.27674E-02, 0.32461E-02, 0.37244E-02,
     & 0.42024E-02, 0.46802E-02, 0.51576E-02, 0.56347E-02, 0.61114E-02,
     & 0.65879E-02, 0.70641E-02, 0.75399E-02, 0.80154E-02, 0.84906E-02,
     & 0.89655E-02, 0.94401E-02, 0.99143E-02, 0.10388E-01, 0.10862E-01,
     & 0.11335E-01, 0.11808E-01, 0.12281E-01, 0.12753E-01, 0.13226E-01,
     & 0.13697E-01, 0.14169E-01, 0.14640E-01, 0.15111E-01, 0.15582E-01,
     & 0.16052E-01, 0.16522E-01, 0.16991E-01, 0.17461E-01, 0.17930E-01,
     & 0.18399E-01, 0.18867E-01, 0.19335E-01, 0.19803E-01, 0.20271E-01,
     & 0.20738E-01, 0.21205E-01, 0.21672E-01, 0.22138E-01, 0.22604E-01,
     & 0.23070E-01, 0.23535E-01, 0.24000E-01, 0.24465E-01, 0.24930E-01,
     & 0.25394E-01, 0.25858E-01, 0.26322E-01, 0.26785E-01, 0.27248E-01,
     & 0.27711E-01, 0.28174E-01, 0.28636E-01, 0.29098E-01, 0.29560E-01,
     & 0.30021E-01, 0.30482E-01, 0.30943E-01, 0.31403E-01, 0.31864E-01,
     & 0.32324E-01, 0.32783E-01, 0.33242E-01, 0.33702E-01, 0.34160E-01/

      DATA (BNC06M(I),I=601,700)/
     & 0.39099E-01, 0.43640E-01, 0.48155E-01, 0.52641E-01, 0.57101E-01,
     & 0.61535E-01, 0.65943E-01, 0.70325E-01, 0.74682E-01, 0.79014E-01,
     & 0.83322E-01, 0.87606E-01, 0.91866E-01, 0.96103E-01, 0.10032E+00,
     & 0.10451E+00, 0.10868E+00, 0.11282E+00, 0.11695E+00, 0.12105E+00,
     & 0.12514E+00, 0.12920E+00, 0.13324E+00, 0.13726E+00, 0.14127E+00,
     & 0.14525E+00, 0.14921E+00, 0.15316E+00, 0.15709E+00, 0.16099E+00,
     & 0.16488E+00, 0.16876E+00, 0.17261E+00, 0.17645E+00, 0.18027E+00,
     & 0.18407E+00, 0.18786E+00, 0.19163E+00, 0.19538E+00, 0.19912E+00,
     & 0.20284E+00, 0.20655E+00, 0.21024E+00, 0.21391E+00, 0.21757E+00,
     & 0.22122E+00, 0.22485E+00, 0.22847E+00, 0.23207E+00, 0.23566E+00,
     & 0.23924E+00, 0.24280E+00, 0.24634E+00, 0.24988E+00, 0.25340E+00,
     & 0.25691E+00, 0.26040E+00, 0.26389E+00, 0.26735E+00, 0.27081E+00,
     & 0.27426E+00, 0.27769E+00, 0.28111E+00, 0.28452E+00, 0.28792E+00,
     & 0.29130E+00, 0.29468E+00, 0.29804E+00, 0.30139E+00, 0.30473E+00,
     & 0.30806E+00, 0.31138E+00, 0.31469E+00, 0.31799E+00, 0.32127E+00,
     & 0.32455E+00, 0.32782E+00, 0.33107E+00, 0.33432E+00, 0.33756E+00,
     & 0.34078E+00, 0.34400E+00, 0.34721E+00, 0.35040E+00, 0.35359E+00,
     & 0.35677E+00, 0.35994E+00, 0.36310E+00, 0.36625E+00, 0.36940E+00,
     & 0.37253E+00, 0.37565E+00, 0.37877E+00, 0.38188E+00, 0.38498E+00,
     & 0.38807E+00, 0.39115E+00, 0.39423E+00, 0.39729E+00, 0.40035E+00/

      DATA (BNC06M(I),I=701,741)/
     & 0.40340E+00, 0.40644E+00, 0.40947E+00, 0.41250E+00, 0.41552E+00,
     & 0.41853E+00, 0.42153E+00, 0.42453E+00, 0.42752E+00, 0.43050E+00,
     & 0.43347E+00, 0.43644E+00, 0.43939E+00, 0.44235E+00, 0.44529E+00,
     & 0.44823E+00, 0.45116E+00, 0.45408E+00, 0.45700E+00, 0.45991E+00,
     & 0.46281E+00, 0.46571E+00, 0.46860E+00, 0.47149E+00, 0.47436E+00,
     & 0.47723E+00, 0.48010E+00, 0.48296E+00, 0.48581E+00, 0.48865E+00,
     & 0.49149E+00, 0.49433E+00, 0.49715E+00, 0.49998E+00, 0.50279E+00,
     & 0.50560E+00, 0.50840E+00, 0.51120E+00, 0.51399E+00, 0.51678E+00,
     & 0.51956E+00
     & /
C
C *** (2H,SO4)     
C
      DATA (BNC07M(I),I=1,100)/
     &-0.90523E-01,-0.19581E+00,-0.24777E+00,-0.28414E+00,-0.31242E+00,
     &-0.33564E+00,-0.35536E+00,-0.37250E+00,-0.38765E+00,-0.40122E+00,
     &-0.41351E+00,-0.42472E+00,-0.43503E+00,-0.44456E+00,-0.45341E+00,
     &-0.46168E+00,-0.46943E+00,-0.47672E+00,-0.48359E+00,-0.49009E+00,
     &-0.49625E+00,-0.50210E+00,-0.50767E+00,-0.51299E+00,-0.51807E+00,
     &-0.52293E+00,-0.52759E+00,-0.53206E+00,-0.53635E+00,-0.54048E+00,
     &-0.54446E+00,-0.54830E+00,-0.55200E+00,-0.55557E+00,-0.55902E+00,
     &-0.56236E+00,-0.56559E+00,-0.56872E+00,-0.57176E+00,-0.57470E+00,
     &-0.57756E+00,-0.58033E+00,-0.58303E+00,-0.58564E+00,-0.58819E+00,
     &-0.59067E+00,-0.59308E+00,-0.59543E+00,-0.59772E+00,-0.59995E+00,
     &-0.60213E+00,-0.60425E+00,-0.60632E+00,-0.60834E+00,-0.61031E+00,
     &-0.61224E+00,-0.61412E+00,-0.61596E+00,-0.61775E+00,-0.61951E+00,
     &-0.62123E+00,-0.62292E+00,-0.62456E+00,-0.62618E+00,-0.62776E+00,
     &-0.62930E+00,-0.63082E+00,-0.63230E+00,-0.63376E+00,-0.63519E+00,
     &-0.63659E+00,-0.63796E+00,-0.63931E+00,-0.64063E+00,-0.64193E+00,
     &-0.64320E+00,-0.64445E+00,-0.64568E+00,-0.64689E+00,-0.64808E+00,
     &-0.64924E+00,-0.65039E+00,-0.65152E+00,-0.65263E+00,-0.65372E+00,
     &-0.65479E+00,-0.65584E+00,-0.65688E+00,-0.65790E+00,-0.65891E+00,
     &-0.65990E+00,-0.66087E+00,-0.66183E+00,-0.66278E+00,-0.66371E+00,
     &-0.66463E+00,-0.66553E+00,-0.66642E+00,-0.66730E+00,-0.66816E+00/

      DATA (BNC07M(I),I=101,200)/
     &-0.66901E+00,-0.66985E+00,-0.67068E+00,-0.67149E+00,-0.67230E+00,
     &-0.67309E+00,-0.67387E+00,-0.67464E+00,-0.67540E+00,-0.67615E+00,
     &-0.67689E+00,-0.67762E+00,-0.67834E+00,-0.67905E+00,-0.67975E+00,
     &-0.68044E+00,-0.68112E+00,-0.68179E+00,-0.68245E+00,-0.68310E+00,
     &-0.68375E+00,-0.68438E+00,-0.68501E+00,-0.68563E+00,-0.68624E+00,
     &-0.68684E+00,-0.68743E+00,-0.68802E+00,-0.68860E+00,-0.68917E+00,
     &-0.68973E+00,-0.69029E+00,-0.69083E+00,-0.69137E+00,-0.69191E+00,
     &-0.69243E+00,-0.69295E+00,-0.69347E+00,-0.69397E+00,-0.69447E+00,
     &-0.69497E+00,-0.69545E+00,-0.69593E+00,-0.69641E+00,-0.69688E+00,
     &-0.69734E+00,-0.69779E+00,-0.69824E+00,-0.69869E+00,-0.69912E+00,
     &-0.69956E+00,-0.69998E+00,-0.70041E+00,-0.70082E+00,-0.70123E+00,
     &-0.70164E+00,-0.70204E+00,-0.70243E+00,-0.70282E+00,-0.70321E+00,
     &-0.70359E+00,-0.70396E+00,-0.70433E+00,-0.70470E+00,-0.70506E+00,
     &-0.70541E+00,-0.70577E+00,-0.70611E+00,-0.70646E+00,-0.70679E+00,
     &-0.70713E+00,-0.70746E+00,-0.70778E+00,-0.70810E+00,-0.70842E+00,
     &-0.70873E+00,-0.70904E+00,-0.70934E+00,-0.70964E+00,-0.70994E+00,
     &-0.71023E+00,-0.71052E+00,-0.71081E+00,-0.71109E+00,-0.71137E+00,
     &-0.71164E+00,-0.71191E+00,-0.71218E+00,-0.71244E+00,-0.71270E+00,
     &-0.71296E+00,-0.71321E+00,-0.71346E+00,-0.71371E+00,-0.71395E+00,
     &-0.71419E+00,-0.71443E+00,-0.71466E+00,-0.71489E+00,-0.71512E+00/

      DATA (BNC07M(I),I=201,300)/
     &-0.71534E+00,-0.71556E+00,-0.71578E+00,-0.71599E+00,-0.71620E+00,
     &-0.71641E+00,-0.71662E+00,-0.71682E+00,-0.71702E+00,-0.71722E+00,
     &-0.71741E+00,-0.71760E+00,-0.71779E+00,-0.71798E+00,-0.71816E+00,
     &-0.71834E+00,-0.71852E+00,-0.71870E+00,-0.71887E+00,-0.71904E+00,
     &-0.71921E+00,-0.71937E+00,-0.71954E+00,-0.71970E+00,-0.71986E+00,
     &-0.72001E+00,-0.72016E+00,-0.72032E+00,-0.72046E+00,-0.72061E+00,
     &-0.72075E+00,-0.72090E+00,-0.72103E+00,-0.72117E+00,-0.72131E+00,
     &-0.72144E+00,-0.72157E+00,-0.72170E+00,-0.72182E+00,-0.72195E+00,
     &-0.72207E+00,-0.72219E+00,-0.72231E+00,-0.72242E+00,-0.72254E+00,
     &-0.72265E+00,-0.72276E+00,-0.72287E+00,-0.72297E+00,-0.72307E+00,
     &-0.72318E+00,-0.72328E+00,-0.72337E+00,-0.72347E+00,-0.72356E+00,
     &-0.72366E+00,-0.72375E+00,-0.72384E+00,-0.72392E+00,-0.72401E+00,
     &-0.72409E+00,-0.72417E+00,-0.72425E+00,-0.72433E+00,-0.72441E+00,
     &-0.72448E+00,-0.72455E+00,-0.72463E+00,-0.72470E+00,-0.72476E+00,
     &-0.72483E+00,-0.72489E+00,-0.72496E+00,-0.72502E+00,-0.72508E+00,
     &-0.72514E+00,-0.72519E+00,-0.72525E+00,-0.72530E+00,-0.72535E+00,
     &-0.72540E+00,-0.72545E+00,-0.72550E+00,-0.72555E+00,-0.72559E+00,
     &-0.72563E+00,-0.72568E+00,-0.72572E+00,-0.72576E+00,-0.72579E+00,
     &-0.72583E+00,-0.72586E+00,-0.72590E+00,-0.72593E+00,-0.72596E+00,
     &-0.72599E+00,-0.72602E+00,-0.72604E+00,-0.72607E+00,-0.72609E+00/

      DATA (BNC07M(I),I=301,400)/
     &-0.72611E+00,-0.72613E+00,-0.72615E+00,-0.72617E+00,-0.72619E+00,
     &-0.72621E+00,-0.72622E+00,-0.72623E+00,-0.72625E+00,-0.72626E+00,
     &-0.72627E+00,-0.72628E+00,-0.72628E+00,-0.72629E+00,-0.72629E+00,
     &-0.72630E+00,-0.72630E+00,-0.72630E+00,-0.72630E+00,-0.72630E+00,
     &-0.72630E+00,-0.72630E+00,-0.72629E+00,-0.72629E+00,-0.72628E+00,
     &-0.72627E+00,-0.72627E+00,-0.72626E+00,-0.72624E+00,-0.72623E+00,
     &-0.72622E+00,-0.72621E+00,-0.72619E+00,-0.72617E+00,-0.72616E+00,
     &-0.72614E+00,-0.72612E+00,-0.72610E+00,-0.72608E+00,-0.72606E+00,
     &-0.72603E+00,-0.72601E+00,-0.72598E+00,-0.72596E+00,-0.72593E+00,
     &-0.72590E+00,-0.72587E+00,-0.72584E+00,-0.72581E+00,-0.72578E+00,
     &-0.72575E+00,-0.72571E+00,-0.72568E+00,-0.72564E+00,-0.72561E+00,
     &-0.72557E+00,-0.72553E+00,-0.72549E+00,-0.72545E+00,-0.72541E+00,
     &-0.72537E+00,-0.72532E+00,-0.72528E+00,-0.72524E+00,-0.72519E+00,
     &-0.72514E+00,-0.72510E+00,-0.72505E+00,-0.72500E+00,-0.72495E+00,
     &-0.72490E+00,-0.72485E+00,-0.72480E+00,-0.72474E+00,-0.72469E+00,
     &-0.72463E+00,-0.72458E+00,-0.72452E+00,-0.72446E+00,-0.72441E+00,
     &-0.72435E+00,-0.72429E+00,-0.72423E+00,-0.72417E+00,-0.72411E+00,
     &-0.72404E+00,-0.72398E+00,-0.72392E+00,-0.72385E+00,-0.72379E+00,
     &-0.72372E+00,-0.72365E+00,-0.72358E+00,-0.72352E+00,-0.72345E+00,
     &-0.72338E+00,-0.72331E+00,-0.72323E+00,-0.72316E+00,-0.72309E+00/

      DATA (BNC07M(I),I=401,500)/
     &-0.72302E+00,-0.72294E+00,-0.72287E+00,-0.72279E+00,-0.72271E+00,
     &-0.72264E+00,-0.72256E+00,-0.72248E+00,-0.72240E+00,-0.72232E+00,
     &-0.72224E+00,-0.72216E+00,-0.72208E+00,-0.72200E+00,-0.72191E+00,
     &-0.72183E+00,-0.72175E+00,-0.72166E+00,-0.72158E+00,-0.72149E+00,
     &-0.72140E+00,-0.72131E+00,-0.72123E+00,-0.72114E+00,-0.72105E+00,
     &-0.72096E+00,-0.72087E+00,-0.72078E+00,-0.72069E+00,-0.72059E+00,
     &-0.72050E+00,-0.72041E+00,-0.72031E+00,-0.72022E+00,-0.72012E+00,
     &-0.72003E+00,-0.71993E+00,-0.71983E+00,-0.71973E+00,-0.71964E+00,
     &-0.71954E+00,-0.71944E+00,-0.71934E+00,-0.71924E+00,-0.71914E+00,
     &-0.71903E+00,-0.71893E+00,-0.71883E+00,-0.71873E+00,-0.71862E+00,
     &-0.71852E+00,-0.71841E+00,-0.71831E+00,-0.71820E+00,-0.71809E+00,
     &-0.71799E+00,-0.71788E+00,-0.71777E+00,-0.71766E+00,-0.71755E+00,
     &-0.71744E+00,-0.71733E+00,-0.71722E+00,-0.71711E+00,-0.71700E+00,
     &-0.71689E+00,-0.71677E+00,-0.71666E+00,-0.71655E+00,-0.71643E+00,
     &-0.71632E+00,-0.71620E+00,-0.71609E+00,-0.71597E+00,-0.71585E+00,
     &-0.71574E+00,-0.71562E+00,-0.71550E+00,-0.71538E+00,-0.71526E+00,
     &-0.71514E+00,-0.71502E+00,-0.71490E+00,-0.71478E+00,-0.71466E+00,
     &-0.71454E+00,-0.71442E+00,-0.71429E+00,-0.71417E+00,-0.71405E+00,
     &-0.71392E+00,-0.71380E+00,-0.71367E+00,-0.71355E+00,-0.71342E+00,
     &-0.71329E+00,-0.71317E+00,-0.71304E+00,-0.71291E+00,-0.71278E+00/

      DATA (BNC07M(I),I=501,600)/
     &-0.71266E+00,-0.71253E+00,-0.71240E+00,-0.71227E+00,-0.71214E+00,
     &-0.71201E+00,-0.71187E+00,-0.71174E+00,-0.71161E+00,-0.71148E+00,
     &-0.71135E+00,-0.71121E+00,-0.71108E+00,-0.71094E+00,-0.71081E+00,
     &-0.71068E+00,-0.71054E+00,-0.71040E+00,-0.71027E+00,-0.71013E+00,
     &-0.70999E+00,-0.70986E+00,-0.70972E+00,-0.70958E+00,-0.70944E+00,
     &-0.70930E+00,-0.70917E+00,-0.70903E+00,-0.70889E+00,-0.70875E+00,
     &-0.70860E+00,-0.70846E+00,-0.70832E+00,-0.70818E+00,-0.70804E+00,
     &-0.70790E+00,-0.70775E+00,-0.70761E+00,-0.70747E+00,-0.70732E+00,
     &-0.70718E+00,-0.70703E+00,-0.70689E+00,-0.70674E+00,-0.70660E+00,
     &-0.70645E+00,-0.70630E+00,-0.70616E+00,-0.70601E+00,-0.70586E+00,
     &-0.70571E+00,-0.70556E+00,-0.70542E+00,-0.70527E+00,-0.70512E+00,
     &-0.70497E+00,-0.70482E+00,-0.70467E+00,-0.70452E+00,-0.70436E+00,
     &-0.70421E+00,-0.70406E+00,-0.70391E+00,-0.70376E+00,-0.70360E+00,
     &-0.70345E+00,-0.70330E+00,-0.70314E+00,-0.70299E+00,-0.70284E+00,
     &-0.70268E+00,-0.70253E+00,-0.70237E+00,-0.70222E+00,-0.70206E+00,
     &-0.70190E+00,-0.70175E+00,-0.70159E+00,-0.70143E+00,-0.70127E+00,
     &-0.70112E+00,-0.70096E+00,-0.70080E+00,-0.70064E+00,-0.70048E+00,
     &-0.70032E+00,-0.70016E+00,-0.70000E+00,-0.69984E+00,-0.69968E+00,
     &-0.69952E+00,-0.69936E+00,-0.69920E+00,-0.69904E+00,-0.69887E+00,
     &-0.69871E+00,-0.69855E+00,-0.69839E+00,-0.69822E+00,-0.69806E+00/

      DATA (BNC07M(I),I=601,700)/
     &-0.69627E+00,-0.69459E+00,-0.69289E+00,-0.69115E+00,-0.68939E+00,
     &-0.68761E+00,-0.68581E+00,-0.68398E+00,-0.68213E+00,-0.68026E+00,
     &-0.67837E+00,-0.67646E+00,-0.67453E+00,-0.67259E+00,-0.67062E+00,
     &-0.66864E+00,-0.66664E+00,-0.66463E+00,-0.66260E+00,-0.66055E+00,
     &-0.65849E+00,-0.65641E+00,-0.65433E+00,-0.65222E+00,-0.65011E+00,
     &-0.64798E+00,-0.64584E+00,-0.64368E+00,-0.64152E+00,-0.63934E+00,
     &-0.63715E+00,-0.63495E+00,-0.63274E+00,-0.63052E+00,-0.62829E+00,
     &-0.62605E+00,-0.62380E+00,-0.62154E+00,-0.61927E+00,-0.61700E+00,
     &-0.61471E+00,-0.61242E+00,-0.61011E+00,-0.60780E+00,-0.60548E+00,
     &-0.60316E+00,-0.60082E+00,-0.59848E+00,-0.59613E+00,-0.59378E+00,
     &-0.59141E+00,-0.58904E+00,-0.58667E+00,-0.58428E+00,-0.58189E+00,
     &-0.57950E+00,-0.57710E+00,-0.57469E+00,-0.57228E+00,-0.56986E+00,
     &-0.56743E+00,-0.56500E+00,-0.56257E+00,-0.56013E+00,-0.55768E+00,
     &-0.55523E+00,-0.55278E+00,-0.55032E+00,-0.54785E+00,-0.54538E+00,
     &-0.54291E+00,-0.54043E+00,-0.53795E+00,-0.53546E+00,-0.53297E+00,
     &-0.53047E+00,-0.52797E+00,-0.52547E+00,-0.52296E+00,-0.52045E+00,
     &-0.51793E+00,-0.51541E+00,-0.51289E+00,-0.51037E+00,-0.50784E+00,
     &-0.50530E+00,-0.50277E+00,-0.50023E+00,-0.49769E+00,-0.49514E+00,
     &-0.49259E+00,-0.49004E+00,-0.48749E+00,-0.48493E+00,-0.48237E+00,
     &-0.47980E+00,-0.47724E+00,-0.47467E+00,-0.47210E+00,-0.46952E+00/

      DATA (BNC07M(I),I=701,741)/
     &-0.46694E+00,-0.46436E+00,-0.46178E+00,-0.45920E+00,-0.45661E+00,
     &-0.45402E+00,-0.45143E+00,-0.44884E+00,-0.44624E+00,-0.44364E+00,
     &-0.44104E+00,-0.43844E+00,-0.43583E+00,-0.43322E+00,-0.43062E+00,
     &-0.42800E+00,-0.42539E+00,-0.42278E+00,-0.42016E+00,-0.41754E+00,
     &-0.41492E+00,-0.41230E+00,-0.40967E+00,-0.40704E+00,-0.40442E+00,
     &-0.40179E+00,-0.39916E+00,-0.39652E+00,-0.39389E+00,-0.39125E+00,
     &-0.38861E+00,-0.38597E+00,-0.38333E+00,-0.38069E+00,-0.37805E+00,
     &-0.37540E+00,-0.37275E+00,-0.37010E+00,-0.36746E+00,-0.36480E+00,
     &-0.36215E+00
     & /
C
C *** (H,HSO4)     
C
      DATA (BNC08M(I),I=1,100)/
     &-0.42949E-01,-0.86374E-01,-0.10452E+00,-0.11550E+00,-0.12283E+00,
     &-0.12787E+00,-0.13133E+00,-0.13362E+00,-0.13499E+00,-0.13563E+00,
     &-0.13565E+00,-0.13515E+00,-0.13421E+00,-0.13286E+00,-0.13117E+00,
     &-0.12916E+00,-0.12686E+00,-0.12431E+00,-0.12150E+00,-0.11848E+00,
     &-0.11525E+00,-0.11182E+00,-0.10821E+00,-0.10442E+00,-0.10047E+00,
     &-0.96373E-01,-0.92125E-01,-0.87738E-01,-0.83218E-01,-0.78573E-01,
     &-0.73807E-01,-0.68926E-01,-0.63935E-01,-0.58840E-01,-0.53644E-01,
     &-0.48353E-01,-0.42970E-01,-0.37499E-01,-0.31944E-01,-0.26309E-01,
     &-0.20597E-01,-0.14812E-01,-0.89563E-02,-0.30333E-02, 0.29541E-02,
     & 0.90031E-02, 0.15111E-01, 0.21276E-01, 0.27495E-01, 0.33766E-01,
     & 0.40087E-01, 0.46455E-01, 0.52870E-01, 0.59330E-01, 0.65832E-01,
     & 0.72376E-01, 0.78960E-01, 0.85584E-01, 0.92245E-01, 0.98943E-01,
     & 0.10568E+00, 0.11245E+00, 0.11925E+00, 0.12610E+00, 0.13297E+00,
     & 0.13988E+00, 0.14682E+00, 0.15380E+00, 0.16082E+00, 0.16786E+00,
     & 0.17494E+00, 0.18206E+00, 0.18921E+00, 0.19640E+00, 0.20363E+00,
     & 0.21089E+00, 0.21819E+00, 0.22552E+00, 0.23290E+00, 0.24031E+00,
     & 0.24776E+00, 0.25525E+00, 0.26278E+00, 0.27035E+00, 0.27796E+00,
     & 0.28561E+00, 0.29330E+00, 0.30102E+00, 0.30879E+00, 0.31660E+00,
     & 0.32445E+00, 0.33233E+00, 0.34025E+00, 0.34821E+00, 0.35621E+00,
     & 0.36424E+00, 0.37231E+00, 0.38041E+00, 0.38854E+00, 0.39670E+00/

      DATA (BNC08M(I),I=101,200)/
     & 0.40490E+00, 0.41312E+00, 0.42137E+00, 0.42964E+00, 0.43795E+00,
     & 0.44627E+00, 0.45461E+00, 0.46298E+00, 0.47136E+00, 0.47976E+00,
     & 0.48818E+00, 0.49661E+00, 0.50506E+00, 0.51351E+00, 0.52198E+00,
     & 0.53045E+00, 0.53893E+00, 0.54742E+00, 0.55591E+00, 0.56440E+00,
     & 0.57290E+00, 0.58139E+00, 0.58989E+00, 0.59839E+00, 0.60688E+00,
     & 0.61537E+00, 0.62386E+00, 0.63234E+00, 0.64082E+00, 0.64929E+00,
     & 0.65776E+00, 0.66621E+00, 0.67466E+00, 0.68310E+00, 0.69153E+00,
     & 0.69995E+00, 0.70837E+00, 0.71677E+00, 0.72516E+00, 0.73353E+00,
     & 0.74190E+00, 0.75026E+00, 0.75860E+00, 0.76693E+00, 0.77525E+00,
     & 0.78355E+00, 0.79184E+00, 0.80012E+00, 0.80838E+00, 0.81663E+00,
     & 0.82487E+00, 0.83309E+00, 0.84130E+00, 0.84949E+00, 0.85767E+00,
     & 0.86583E+00, 0.87398E+00, 0.88212E+00, 0.89024E+00, 0.89834E+00,
     & 0.90643E+00, 0.91451E+00, 0.92256E+00, 0.93061E+00, 0.93864E+00,
     & 0.94665E+00, 0.95465E+00, 0.96263E+00, 0.97060E+00, 0.97855E+00,
     & 0.98648E+00, 0.99440E+00, 0.10023E+01, 0.10102E+01, 0.10181E+01,
     & 0.10259E+01, 0.10338E+01, 0.10416E+01, 0.10494E+01, 0.10572E+01,
     & 0.10650E+01, 0.10727E+01, 0.10805E+01, 0.10882E+01, 0.10959E+01,
     & 0.11036E+01, 0.11113E+01, 0.11190E+01, 0.11266E+01, 0.11343E+01,
     & 0.11419E+01, 0.11495E+01, 0.11571E+01, 0.11647E+01, 0.11722E+01,
     & 0.11798E+01, 0.11873E+01, 0.11948E+01, 0.12023E+01, 0.12098E+01/

      DATA (BNC08M(I),I=201,300)/
     & 0.12173E+01, 0.12247E+01, 0.12321E+01, 0.12396E+01, 0.12470E+01,
     & 0.12544E+01, 0.12617E+01, 0.12691E+01, 0.12764E+01, 0.12838E+01,
     & 0.12911E+01, 0.12984E+01, 0.13057E+01, 0.13129E+01, 0.13202E+01,
     & 0.13274E+01, 0.13346E+01, 0.13418E+01, 0.13490E+01, 0.13562E+01,
     & 0.13634E+01, 0.13705E+01, 0.13777E+01, 0.13848E+01, 0.13919E+01,
     & 0.13990E+01, 0.14060E+01, 0.14131E+01, 0.14202E+01, 0.14272E+01,
     & 0.14342E+01, 0.14412E+01, 0.14482E+01, 0.14552E+01, 0.14621E+01,
     & 0.14691E+01, 0.14760E+01, 0.14829E+01, 0.14898E+01, 0.14967E+01,
     & 0.15036E+01, 0.15104E+01, 0.15173E+01, 0.15241E+01, 0.15309E+01,
     & 0.15377E+01, 0.15445E+01, 0.15513E+01, 0.15581E+01, 0.15648E+01,
     & 0.15716E+01, 0.15783E+01, 0.15850E+01, 0.15917E+01, 0.15984E+01,
     & 0.16051E+01, 0.16117E+01, 0.16184E+01, 0.16250E+01, 0.16316E+01,
     & 0.16382E+01, 0.16448E+01, 0.16514E+01, 0.16579E+01, 0.16645E+01,
     & 0.16710E+01, 0.16776E+01, 0.16841E+01, 0.16906E+01, 0.16971E+01,
     & 0.17035E+01, 0.17100E+01, 0.17165E+01, 0.17229E+01, 0.17293E+01,
     & 0.17357E+01, 0.17421E+01, 0.17485E+01, 0.17549E+01, 0.17613E+01,
     & 0.17676E+01, 0.17740E+01, 0.17803E+01, 0.17866E+01, 0.17929E+01,
     & 0.17992E+01, 0.18055E+01, 0.18117E+01, 0.18180E+01, 0.18242E+01,
     & 0.18305E+01, 0.18367E+01, 0.18429E+01, 0.18491E+01, 0.18553E+01,
     & 0.18614E+01, 0.18676E+01, 0.18738E+01, 0.18799E+01, 0.18860E+01/

      DATA (BNC08M(I),I=301,400)/
     & 0.18921E+01, 0.18982E+01, 0.19043E+01, 0.19104E+01, 0.19165E+01,
     & 0.19225E+01, 0.19286E+01, 0.19346E+01, 0.19407E+01, 0.19467E+01,
     & 0.19527E+01, 0.19587E+01, 0.19646E+01, 0.19706E+01, 0.19766E+01,
     & 0.19825E+01, 0.19885E+01, 0.19944E+01, 0.20003E+01, 0.20062E+01,
     & 0.20121E+01, 0.20180E+01, 0.20239E+01, 0.20297E+01, 0.20356E+01,
     & 0.20414E+01, 0.20473E+01, 0.20531E+01, 0.20589E+01, 0.20647E+01,
     & 0.20705E+01, 0.20763E+01, 0.20820E+01, 0.20878E+01, 0.20935E+01,
     & 0.20993E+01, 0.21050E+01, 0.21107E+01, 0.21164E+01, 0.21221E+01,
     & 0.21278E+01, 0.21335E+01, 0.21392E+01, 0.21448E+01, 0.21505E+01,
     & 0.21561E+01, 0.21618E+01, 0.21674E+01, 0.21730E+01, 0.21786E+01,
     & 0.21842E+01, 0.21898E+01, 0.21953E+01, 0.22009E+01, 0.22065E+01,
     & 0.22120E+01, 0.22175E+01, 0.22231E+01, 0.22286E+01, 0.22341E+01,
     & 0.22396E+01, 0.22451E+01, 0.22506E+01, 0.22560E+01, 0.22615E+01,
     & 0.22669E+01, 0.22724E+01, 0.22778E+01, 0.22832E+01, 0.22887E+01,
     & 0.22941E+01, 0.22995E+01, 0.23048E+01, 0.23102E+01, 0.23156E+01,
     & 0.23210E+01, 0.23263E+01, 0.23317E+01, 0.23370E+01, 0.23423E+01,
     & 0.23476E+01, 0.23529E+01, 0.23582E+01, 0.23635E+01, 0.23688E+01,
     & 0.23741E+01, 0.23794E+01, 0.23846E+01, 0.23899E+01, 0.23951E+01,
     & 0.24003E+01, 0.24056E+01, 0.24108E+01, 0.24160E+01, 0.24212E+01,
     & 0.24264E+01, 0.24316E+01, 0.24367E+01, 0.24419E+01, 0.24471E+01/

      DATA (BNC08M(I),I=401,500)/
     & 0.24522E+01, 0.24574E+01, 0.24625E+01, 0.24676E+01, 0.24727E+01,
     & 0.24778E+01, 0.24829E+01, 0.24880E+01, 0.24931E+01, 0.24982E+01,
     & 0.25033E+01, 0.25083E+01, 0.25134E+01, 0.25184E+01, 0.25235E+01,
     & 0.25285E+01, 0.25335E+01, 0.25385E+01, 0.25436E+01, 0.25486E+01,
     & 0.25535E+01, 0.25585E+01, 0.25635E+01, 0.25685E+01, 0.25734E+01,
     & 0.25784E+01, 0.25833E+01, 0.25883E+01, 0.25932E+01, 0.25981E+01,
     & 0.26031E+01, 0.26080E+01, 0.26129E+01, 0.26178E+01, 0.26226E+01,
     & 0.26275E+01, 0.26324E+01, 0.26373E+01, 0.26421E+01, 0.26470E+01,
     & 0.26518E+01, 0.26567E+01, 0.26615E+01, 0.26663E+01, 0.26711E+01,
     & 0.26759E+01, 0.26807E+01, 0.26855E+01, 0.26903E+01, 0.26951E+01,
     & 0.26999E+01, 0.27046E+01, 0.27094E+01, 0.27141E+01, 0.27189E+01,
     & 0.27236E+01, 0.27284E+01, 0.27331E+01, 0.27378E+01, 0.27425E+01,
     & 0.27472E+01, 0.27519E+01, 0.27566E+01, 0.27613E+01, 0.27660E+01,
     & 0.27706E+01, 0.27753E+01, 0.27800E+01, 0.27846E+01, 0.27893E+01,
     & 0.27939E+01, 0.27985E+01, 0.28031E+01, 0.28078E+01, 0.28124E+01,
     & 0.28170E+01, 0.28216E+01, 0.28262E+01, 0.28308E+01, 0.28353E+01,
     & 0.28399E+01, 0.28445E+01, 0.28490E+01, 0.28536E+01, 0.28581E+01,
     & 0.28627E+01, 0.28672E+01, 0.28717E+01, 0.28763E+01, 0.28808E+01,
     & 0.28853E+01, 0.28898E+01, 0.28943E+01, 0.28988E+01, 0.29033E+01,
     & 0.29077E+01, 0.29122E+01, 0.29167E+01, 0.29211E+01, 0.29256E+01/

      DATA (BNC08M(I),I=501,600)/
     & 0.29300E+01, 0.29345E+01, 0.29389E+01, 0.29433E+01, 0.29478E+01,
     & 0.29522E+01, 0.29566E+01, 0.29610E+01, 0.29654E+01, 0.29698E+01,
     & 0.29742E+01, 0.29786E+01, 0.29830E+01, 0.29873E+01, 0.29917E+01,
     & 0.29960E+01, 0.30004E+01, 0.30047E+01, 0.30091E+01, 0.30134E+01,
     & 0.30178E+01, 0.30221E+01, 0.30264E+01, 0.30307E+01, 0.30350E+01,
     & 0.30393E+01, 0.30436E+01, 0.30479E+01, 0.30522E+01, 0.30565E+01,
     & 0.30608E+01, 0.30650E+01, 0.30693E+01, 0.30735E+01, 0.30778E+01,
     & 0.30820E+01, 0.30863E+01, 0.30905E+01, 0.30947E+01, 0.30990E+01,
     & 0.31032E+01, 0.31074E+01, 0.31116E+01, 0.31158E+01, 0.31200E+01,
     & 0.31242E+01, 0.31284E+01, 0.31326E+01, 0.31367E+01, 0.31409E+01,
     & 0.31451E+01, 0.31492E+01, 0.31534E+01, 0.31576E+01, 0.31617E+01,
     & 0.31658E+01, 0.31700E+01, 0.31741E+01, 0.31782E+01, 0.31823E+01,
     & 0.31865E+01, 0.31906E+01, 0.31947E+01, 0.31988E+01, 0.32029E+01,
     & 0.32069E+01, 0.32110E+01, 0.32151E+01, 0.32192E+01, 0.32232E+01,
     & 0.32273E+01, 0.32314E+01, 0.32354E+01, 0.32395E+01, 0.32435E+01,
     & 0.32476E+01, 0.32516E+01, 0.32556E+01, 0.32596E+01, 0.32637E+01,
     & 0.32677E+01, 0.32717E+01, 0.32757E+01, 0.32797E+01, 0.32837E+01,
     & 0.32877E+01, 0.32916E+01, 0.32956E+01, 0.32996E+01, 0.33036E+01,
     & 0.33075E+01, 0.33115E+01, 0.33154E+01, 0.33194E+01, 0.33233E+01,
     & 0.33273E+01, 0.33312E+01, 0.33352E+01, 0.33391E+01, 0.33430E+01/

      DATA (BNC08M(I),I=601,700)/
     & 0.33851E+01, 0.34236E+01, 0.34616E+01, 0.34992E+01, 0.35364E+01,
     & 0.35732E+01, 0.36095E+01, 0.36455E+01, 0.36811E+01, 0.37163E+01,
     & 0.37511E+01, 0.37856E+01, 0.38198E+01, 0.38535E+01, 0.38870E+01,
     & 0.39201E+01, 0.39529E+01, 0.39854E+01, 0.40175E+01, 0.40494E+01,
     & 0.40809E+01, 0.41122E+01, 0.41432E+01, 0.41738E+01, 0.42043E+01,
     & 0.42344E+01, 0.42643E+01, 0.42939E+01, 0.43232E+01, 0.43523E+01,
     & 0.43812E+01, 0.44098E+01, 0.44382E+01, 0.44663E+01, 0.44942E+01,
     & 0.45219E+01, 0.45494E+01, 0.45766E+01, 0.46036E+01, 0.46304E+01,
     & 0.46570E+01, 0.46834E+01, 0.47096E+01, 0.47356E+01, 0.47614E+01,
     & 0.47870E+01, 0.48124E+01, 0.48377E+01, 0.48627E+01, 0.48876E+01,
     & 0.49123E+01, 0.49368E+01, 0.49611E+01, 0.49853E+01, 0.50093E+01,
     & 0.50331E+01, 0.50568E+01, 0.50803E+01, 0.51036E+01, 0.51268E+01,
     & 0.51499E+01, 0.51728E+01, 0.51955E+01, 0.52181E+01, 0.52406E+01,
     & 0.52629E+01, 0.52850E+01, 0.53071E+01, 0.53290E+01, 0.53507E+01,
     & 0.53723E+01, 0.53938E+01, 0.54152E+01, 0.54364E+01, 0.54575E+01,
     & 0.54785E+01, 0.54993E+01, 0.55200E+01, 0.55406E+01, 0.55611E+01,
     & 0.55815E+01, 0.56018E+01, 0.56219E+01, 0.56419E+01, 0.56618E+01,
     & 0.56816E+01, 0.57013E+01, 0.57209E+01, 0.57404E+01, 0.57597E+01,
     & 0.57790E+01, 0.57982E+01, 0.58172E+01, 0.58362E+01, 0.58550E+01,
     & 0.58738E+01, 0.58925E+01, 0.59110E+01, 0.59295E+01, 0.59479E+01/

      DATA (BNC08M(I),I=701,741)/
     & 0.59661E+01, 0.59843E+01, 0.60024E+01, 0.60204E+01, 0.60384E+01,
     & 0.60562E+01, 0.60739E+01, 0.60916E+01, 0.61092E+01, 0.61266E+01,
     & 0.61440E+01, 0.61614E+01, 0.61786E+01, 0.61957E+01, 0.62128E+01,
     & 0.62298E+01, 0.62467E+01, 0.62636E+01, 0.62803E+01, 0.62970E+01,
     & 0.63136E+01, 0.63301E+01, 0.63466E+01, 0.63630E+01, 0.63793E+01,
     & 0.63955E+01, 0.64117E+01, 0.64278E+01, 0.64438E+01, 0.64598E+01,
     & 0.64757E+01, 0.64915E+01, 0.65073E+01, 0.65229E+01, 0.65386E+01,
     & 0.65541E+01, 0.65696E+01, 0.65850E+01, 0.66004E+01, 0.66157E+01,
     & 0.66309E+01
     & /
C
C *** NH4HSO4      
C
      DATA (BNC09M(I),I=1,100)/
     &-0.44571E-01,-0.94628E-01,-0.11852E+00,-0.13486E+00,-0.14728E+00,
     &-0.15726E+00,-0.16555E+00,-0.17260E+00,-0.17867E+00,-0.18397E+00,
     &-0.18863E+00,-0.19275E+00,-0.19642E+00,-0.19969E+00,-0.20262E+00,
     &-0.20523E+00,-0.20757E+00,-0.20965E+00,-0.21150E+00,-0.21315E+00,
     &-0.21460E+00,-0.21587E+00,-0.21698E+00,-0.21793E+00,-0.21873E+00,
     &-0.21940E+00,-0.21993E+00,-0.22034E+00,-0.22063E+00,-0.22082E+00,
     &-0.22089E+00,-0.22087E+00,-0.22075E+00,-0.22053E+00,-0.22023E+00,
     &-0.21984E+00,-0.21937E+00,-0.21882E+00,-0.21820E+00,-0.21750E+00,
     &-0.21673E+00,-0.21590E+00,-0.21500E+00,-0.21404E+00,-0.21302E+00,
     &-0.21194E+00,-0.21081E+00,-0.20962E+00,-0.20838E+00,-0.20709E+00,
     &-0.20575E+00,-0.20436E+00,-0.20293E+00,-0.20146E+00,-0.19994E+00,
     &-0.19838E+00,-0.19679E+00,-0.19515E+00,-0.19348E+00,-0.19177E+00,
     &-0.19003E+00,-0.18826E+00,-0.18645E+00,-0.18461E+00,-0.18274E+00,
     &-0.18084E+00,-0.17890E+00,-0.17694E+00,-0.17496E+00,-0.17294E+00,
     &-0.17090E+00,-0.16882E+00,-0.16673E+00,-0.16460E+00,-0.16246E+00,
     &-0.16028E+00,-0.15808E+00,-0.15586E+00,-0.15361E+00,-0.15134E+00,
     &-0.14905E+00,-0.14673E+00,-0.14439E+00,-0.14202E+00,-0.13963E+00,
     &-0.13722E+00,-0.13479E+00,-0.13234E+00,-0.12987E+00,-0.12737E+00,
     &-0.12485E+00,-0.12232E+00,-0.11976E+00,-0.11719E+00,-0.11460E+00,
     &-0.11199E+00,-0.10936E+00,-0.10671E+00,-0.10405E+00,-0.10137E+00/

      DATA (BNC09M(I),I=101,200)/
     &-0.98681E-01,-0.95974E-01,-0.93253E-01,-0.90519E-01,-0.87773E-01,
     &-0.85014E-01,-0.82245E-01,-0.79465E-01,-0.76675E-01,-0.73876E-01,
     &-0.71069E-01,-0.68253E-01,-0.65431E-01,-0.62602E-01,-0.59767E-01,
     &-0.56926E-01,-0.54081E-01,-0.51232E-01,-0.48379E-01,-0.45522E-01,
     &-0.42663E-01,-0.39802E-01,-0.36939E-01,-0.34075E-01,-0.31210E-01,
     &-0.28345E-01,-0.25479E-01,-0.22614E-01,-0.19749E-01,-0.16885E-01,
     &-0.14023E-01,-0.11162E-01,-0.83023E-02,-0.54449E-02,-0.25898E-02,
     & 0.26281E-03, 0.31128E-02, 0.59598E-02, 0.88039E-02, 0.11645E-01,
     & 0.14482E-01, 0.17316E-01, 0.20147E-01, 0.22973E-01, 0.25796E-01,
     & 0.28615E-01, 0.31429E-01, 0.34240E-01, 0.37046E-01, 0.39848E-01,
     & 0.42645E-01, 0.45438E-01, 0.48226E-01, 0.51010E-01, 0.53789E-01,
     & 0.56563E-01, 0.59333E-01, 0.62097E-01, 0.64857E-01, 0.67612E-01,
     & 0.70362E-01, 0.73107E-01, 0.75847E-01, 0.78581E-01, 0.81311E-01,
     & 0.84036E-01, 0.86755E-01, 0.89470E-01, 0.92179E-01, 0.94883E-01,
     & 0.97582E-01, 0.10028E+00, 0.10296E+00, 0.10565E+00, 0.10832E+00,
     & 0.11100E+00, 0.11367E+00, 0.11633E+00, 0.11898E+00, 0.12164E+00,
     & 0.12428E+00, 0.12692E+00, 0.12956E+00, 0.13219E+00, 0.13482E+00,
     & 0.13744E+00, 0.14005E+00, 0.14266E+00, 0.14526E+00, 0.14786E+00,
     & 0.15046E+00, 0.15304E+00, 0.15563E+00, 0.15821E+00, 0.16078E+00,
     & 0.16335E+00, 0.16591E+00, 0.16847E+00, 0.17102E+00, 0.17356E+00/

      DATA (BNC09M(I),I=201,300)/
     & 0.17611E+00, 0.17864E+00, 0.18117E+00, 0.18370E+00, 0.18622E+00,
     & 0.18874E+00, 0.19125E+00, 0.19375E+00, 0.19626E+00, 0.19875E+00,
     & 0.20124E+00, 0.20373E+00, 0.20621E+00, 0.20868E+00, 0.21115E+00,
     & 0.21362E+00, 0.21608E+00, 0.21854E+00, 0.22099E+00, 0.22343E+00,
     & 0.22588E+00, 0.22831E+00, 0.23074E+00, 0.23317E+00, 0.23559E+00,
     & 0.23801E+00, 0.24042E+00, 0.24283E+00, 0.24523E+00, 0.24763E+00,
     & 0.25002E+00, 0.25241E+00, 0.25479E+00, 0.25717E+00, 0.25955E+00,
     & 0.26192E+00, 0.26428E+00, 0.26664E+00, 0.26900E+00, 0.27135E+00,
     & 0.27370E+00, 0.27604E+00, 0.27838E+00, 0.28071E+00, 0.28304E+00,
     & 0.28536E+00, 0.28768E+00, 0.29000E+00, 0.29231E+00, 0.29462E+00,
     & 0.29692E+00, 0.29922E+00, 0.30151E+00, 0.30380E+00, 0.30609E+00,
     & 0.30837E+00, 0.31064E+00, 0.31291E+00, 0.31518E+00, 0.31745E+00,
     & 0.31971E+00, 0.32196E+00, 0.32421E+00, 0.32646E+00, 0.32870E+00,
     & 0.33094E+00, 0.33317E+00, 0.33540E+00, 0.33763E+00, 0.33985E+00,
     & 0.34207E+00, 0.34428E+00, 0.34649E+00, 0.34870E+00, 0.35090E+00,
     & 0.35310E+00, 0.35529E+00, 0.35748E+00, 0.35967E+00, 0.36185E+00,
     & 0.36403E+00, 0.36620E+00, 0.36837E+00, 0.37054E+00, 0.37270E+00,
     & 0.37486E+00, 0.37702E+00, 0.37917E+00, 0.38131E+00, 0.38346E+00,
     & 0.38560E+00, 0.38773E+00, 0.38987E+00, 0.39199E+00, 0.39412E+00,
     & 0.39624E+00, 0.39836E+00, 0.40047E+00, 0.40258E+00, 0.40469E+00/

      DATA (BNC09M(I),I=301,400)/
     & 0.40679E+00, 0.40889E+00, 0.41099E+00, 0.41308E+00, 0.41517E+00,
     & 0.41725E+00, 0.41933E+00, 0.42141E+00, 0.42349E+00, 0.42556E+00,
     & 0.42763E+00, 0.42969E+00, 0.43175E+00, 0.43381E+00, 0.43586E+00,
     & 0.43791E+00, 0.43996E+00, 0.44200E+00, 0.44404E+00, 0.44608E+00,
     & 0.44811E+00, 0.45014E+00, 0.45217E+00, 0.45419E+00, 0.45621E+00,
     & 0.45823E+00, 0.46024E+00, 0.46225E+00, 0.46426E+00, 0.46626E+00,
     & 0.46826E+00, 0.47026E+00, 0.47225E+00, 0.47424E+00, 0.47623E+00,
     & 0.47821E+00, 0.48020E+00, 0.48217E+00, 0.48415E+00, 0.48612E+00,
     & 0.48809E+00, 0.49005E+00, 0.49202E+00, 0.49397E+00, 0.49593E+00,
     & 0.49788E+00, 0.49983E+00, 0.50178E+00, 0.50372E+00, 0.50566E+00,
     & 0.50760E+00, 0.50954E+00, 0.51147E+00, 0.51340E+00, 0.51532E+00,
     & 0.51725E+00, 0.51917E+00, 0.52108E+00, 0.52300E+00, 0.52491E+00,
     & 0.52682E+00, 0.52872E+00, 0.53062E+00, 0.53252E+00, 0.53442E+00,
     & 0.53631E+00, 0.53820E+00, 0.54009E+00, 0.54198E+00, 0.54386E+00,
     & 0.54574E+00, 0.54761E+00, 0.54949E+00, 0.55136E+00, 0.55323E+00,
     & 0.55509E+00, 0.55696E+00, 0.55882E+00, 0.56067E+00, 0.56253E+00,
     & 0.56438E+00, 0.56623E+00, 0.56807E+00, 0.56992E+00, 0.57176E+00,
     & 0.57360E+00, 0.57543E+00, 0.57726E+00, 0.57909E+00, 0.58092E+00,
     & 0.58275E+00, 0.58457E+00, 0.58639E+00, 0.58821E+00, 0.59002E+00,
     & 0.59183E+00, 0.59364E+00, 0.59545E+00, 0.59725E+00, 0.59905E+00/

      DATA (BNC09M(I),I=401,500)/
     & 0.60085E+00, 0.60265E+00, 0.60444E+00, 0.60623E+00, 0.60802E+00,
     & 0.60981E+00, 0.61159E+00, 0.61337E+00, 0.61515E+00, 0.61693E+00,
     & 0.61870E+00, 0.62047E+00, 0.62224E+00, 0.62401E+00, 0.62577E+00,
     & 0.62753E+00, 0.62929E+00, 0.63105E+00, 0.63280E+00, 0.63455E+00,
     & 0.63630E+00, 0.63805E+00, 0.63980E+00, 0.64154E+00, 0.64328E+00,
     & 0.64501E+00, 0.64675E+00, 0.64848E+00, 0.65021E+00, 0.65194E+00,
     & 0.65367E+00, 0.65539E+00, 0.65711E+00, 0.65883E+00, 0.66055E+00,
     & 0.66226E+00, 0.66397E+00, 0.66568E+00, 0.66739E+00, 0.66909E+00,
     & 0.67080E+00, 0.67250E+00, 0.67419E+00, 0.67589E+00, 0.67758E+00,
     & 0.67928E+00, 0.68096E+00, 0.68265E+00, 0.68434E+00, 0.68602E+00,
     & 0.68770E+00, 0.68938E+00, 0.69105E+00, 0.69273E+00, 0.69440E+00,
     & 0.69607E+00, 0.69774E+00, 0.69940E+00, 0.70106E+00, 0.70273E+00,
     & 0.70438E+00, 0.70604E+00, 0.70770E+00, 0.70935E+00, 0.71100E+00,
     & 0.71265E+00, 0.71429E+00, 0.71594E+00, 0.71758E+00, 0.71922E+00,
     & 0.72086E+00, 0.72249E+00, 0.72413E+00, 0.72576E+00, 0.72739E+00,
     & 0.72901E+00, 0.73064E+00, 0.73226E+00, 0.73388E+00, 0.73550E+00,
     & 0.73712E+00, 0.73874E+00, 0.74035E+00, 0.74196E+00, 0.74357E+00,
     & 0.74518E+00, 0.74678E+00, 0.74839E+00, 0.74999E+00, 0.75159E+00,
     & 0.75318E+00, 0.75478E+00, 0.75637E+00, 0.75796E+00, 0.75955E+00,
     & 0.76114E+00, 0.76273E+00, 0.76431E+00, 0.76589E+00, 0.76747E+00/

      DATA (BNC09M(I),I=501,600)/
     & 0.76905E+00, 0.77063E+00, 0.77220E+00, 0.77377E+00, 0.77534E+00,
     & 0.77691E+00, 0.77848E+00, 0.78004E+00, 0.78161E+00, 0.78317E+00,
     & 0.78473E+00, 0.78629E+00, 0.78784E+00, 0.78939E+00, 0.79095E+00,
     & 0.79250E+00, 0.79404E+00, 0.79559E+00, 0.79714E+00, 0.79868E+00,
     & 0.80022E+00, 0.80176E+00, 0.80330E+00, 0.80483E+00, 0.80636E+00,
     & 0.80790E+00, 0.80943E+00, 0.81096E+00, 0.81248E+00, 0.81401E+00,
     & 0.81553E+00, 0.81705E+00, 0.81857E+00, 0.82009E+00, 0.82160E+00,
     & 0.82312E+00, 0.82463E+00, 0.82614E+00, 0.82765E+00, 0.82916E+00,
     & 0.83066E+00, 0.83217E+00, 0.83367E+00, 0.83517E+00, 0.83667E+00,
     & 0.83817E+00, 0.83966E+00, 0.84116E+00, 0.84265E+00, 0.84414E+00,
     & 0.84563E+00, 0.84711E+00, 0.84860E+00, 0.85008E+00, 0.85157E+00,
     & 0.85305E+00, 0.85452E+00, 0.85600E+00, 0.85748E+00, 0.85895E+00,
     & 0.86042E+00, 0.86189E+00, 0.86336E+00, 0.86483E+00, 0.86629E+00,
     & 0.86776E+00, 0.86922E+00, 0.87068E+00, 0.87214E+00, 0.87360E+00,
     & 0.87506E+00, 0.87651E+00, 0.87796E+00, 0.87941E+00, 0.88086E+00,
     & 0.88231E+00, 0.88376E+00, 0.88520E+00, 0.88665E+00, 0.88809E+00,
     & 0.88953E+00, 0.89097E+00, 0.89240E+00, 0.89384E+00, 0.89527E+00,
     & 0.89671E+00, 0.89814E+00, 0.89957E+00, 0.90099E+00, 0.90242E+00,
     & 0.90385E+00, 0.90527E+00, 0.90669E+00, 0.90811E+00, 0.90953E+00,
     & 0.91095E+00, 0.91236E+00, 0.91378E+00, 0.91519E+00, 0.91660E+00/

      DATA (BNC09M(I),I=601,700)/
     & 0.93176E+00, 0.94563E+00, 0.95937E+00, 0.97297E+00, 0.98643E+00,
     & 0.99976E+00, 0.10130E+01, 0.10260E+01, 0.10390E+01, 0.10518E+01,
     & 0.10645E+01, 0.10771E+01, 0.10896E+01, 0.11020E+01, 0.11142E+01,
     & 0.11264E+01, 0.11384E+01, 0.11504E+01, 0.11622E+01, 0.11740E+01,
     & 0.11856E+01, 0.11972E+01, 0.12086E+01, 0.12200E+01, 0.12313E+01,
     & 0.12425E+01, 0.12536E+01, 0.12646E+01, 0.12755E+01, 0.12864E+01,
     & 0.12971E+01, 0.13078E+01, 0.13184E+01, 0.13290E+01, 0.13394E+01,
     & 0.13498E+01, 0.13601E+01, 0.13704E+01, 0.13805E+01, 0.13906E+01,
     & 0.14006E+01, 0.14106E+01, 0.14205E+01, 0.14303E+01, 0.14401E+01,
     & 0.14498E+01, 0.14594E+01, 0.14690E+01, 0.14785E+01, 0.14880E+01,
     & 0.14974E+01, 0.15067E+01, 0.15160E+01, 0.15252E+01, 0.15344E+01,
     & 0.15435E+01, 0.15526E+01, 0.15616E+01, 0.15706E+01, 0.15795E+01,
     & 0.15883E+01, 0.15971E+01, 0.16059E+01, 0.16146E+01, 0.16233E+01,
     & 0.16319E+01, 0.16404E+01, 0.16489E+01, 0.16574E+01, 0.16658E+01,
     & 0.16742E+01, 0.16825E+01, 0.16908E+01, 0.16991E+01, 0.17073E+01,
     & 0.17155E+01, 0.17236E+01, 0.17317E+01, 0.17397E+01, 0.17477E+01,
     & 0.17557E+01, 0.17636E+01, 0.17715E+01, 0.17793E+01, 0.17871E+01,
     & 0.17949E+01, 0.18026E+01, 0.18103E+01, 0.18180E+01, 0.18256E+01,
     & 0.18332E+01, 0.18408E+01, 0.18483E+01, 0.18558E+01, 0.18632E+01,
     & 0.18707E+01, 0.18780E+01, 0.18854E+01, 0.18927E+01, 0.19000E+01/

      DATA (BNC09M(I),I=701,741)/
     & 0.19073E+01, 0.19145E+01, 0.19217E+01, 0.19289E+01, 0.19360E+01,
     & 0.19431E+01, 0.19502E+01, 0.19572E+01, 0.19643E+01, 0.19713E+01,
     & 0.19782E+01, 0.19852E+01, 0.19921E+01, 0.19989E+01, 0.20058E+01,
     & 0.20126E+01, 0.20194E+01, 0.20262E+01, 0.20329E+01, 0.20397E+01,
     & 0.20464E+01, 0.20530E+01, 0.20597E+01, 0.20663E+01, 0.20729E+01,
     & 0.20794E+01, 0.20860E+01, 0.20925E+01, 0.20990E+01, 0.21055E+01,
     & 0.21119E+01, 0.21184E+01, 0.21248E+01, 0.21311E+01, 0.21375E+01,
     & 0.21438E+01, 0.21502E+01, 0.21564E+01, 0.21627E+01, 0.21690E+01,
     & 0.21752E+01
     & /
C
C *** (H,NO3)      
C
      DATA (BNC10M(I),I=1,100)/
     &-0.44117E-01,-0.91788E-01,-0.11327E+00,-0.12721E+00,-0.13727E+00,
     &-0.14491E+00,-0.15090E+00,-0.15568E+00,-0.15952E+00,-0.16263E+00,
     &-0.16514E+00,-0.16716E+00,-0.16876E+00,-0.17001E+00,-0.17095E+00,
     &-0.17163E+00,-0.17207E+00,-0.17231E+00,-0.17237E+00,-0.17227E+00,
     &-0.17202E+00,-0.17164E+00,-0.17114E+00,-0.17054E+00,-0.16983E+00,
     &-0.16904E+00,-0.16817E+00,-0.16723E+00,-0.16622E+00,-0.16515E+00,
     &-0.16402E+00,-0.16284E+00,-0.16161E+00,-0.16034E+00,-0.15903E+00,
     &-0.15768E+00,-0.15630E+00,-0.15489E+00,-0.15346E+00,-0.15199E+00,
     &-0.15051E+00,-0.14900E+00,-0.14747E+00,-0.14592E+00,-0.14436E+00,
     &-0.14278E+00,-0.14119E+00,-0.13959E+00,-0.13797E+00,-0.13635E+00,
     &-0.13471E+00,-0.13306E+00,-0.13141E+00,-0.12974E+00,-0.12807E+00,
     &-0.12639E+00,-0.12471E+00,-0.12301E+00,-0.12131E+00,-0.11961E+00,
     &-0.11789E+00,-0.11617E+00,-0.11445E+00,-0.11271E+00,-0.11097E+00,
     &-0.10922E+00,-0.10746E+00,-0.10570E+00,-0.10392E+00,-0.10214E+00,
     &-0.10035E+00,-0.98551E-01,-0.96741E-01,-0.94920E-01,-0.93090E-01,
     &-0.91248E-01,-0.89396E-01,-0.87532E-01,-0.85656E-01,-0.83769E-01,
     &-0.81869E-01,-0.79957E-01,-0.78032E-01,-0.76093E-01,-0.74142E-01,
     &-0.72178E-01,-0.70201E-01,-0.68210E-01,-0.66206E-01,-0.64188E-01,
     &-0.62157E-01,-0.60114E-01,-0.58057E-01,-0.55987E-01,-0.53905E-01,
     &-0.51810E-01,-0.49703E-01,-0.47584E-01,-0.45454E-01,-0.43312E-01/

      DATA (BNC10M(I),I=101,200)/
     &-0.41159E-01,-0.38996E-01,-0.36823E-01,-0.34640E-01,-0.32448E-01,
     &-0.30246E-01,-0.28037E-01,-0.25819E-01,-0.23593E-01,-0.21360E-01,
     &-0.19121E-01,-0.16875E-01,-0.14623E-01,-0.12366E-01,-0.10103E-01,
     &-0.78360E-02,-0.55644E-02,-0.32889E-02,-0.10096E-02, 0.12729E-02,
     & 0.35583E-02, 0.58464E-02, 0.81368E-02, 0.10429E-01, 0.12723E-01,
     & 0.15019E-01, 0.17316E-01, 0.19614E-01, 0.21913E-01, 0.24212E-01,
     & 0.26512E-01, 0.28812E-01, 0.31112E-01, 0.33412E-01, 0.35712E-01,
     & 0.38011E-01, 0.40310E-01, 0.42608E-01, 0.44905E-01, 0.47202E-01,
     & 0.49497E-01, 0.51792E-01, 0.54085E-01, 0.56377E-01, 0.58668E-01,
     & 0.60958E-01, 0.63246E-01, 0.65533E-01, 0.67818E-01, 0.70101E-01,
     & 0.72383E-01, 0.74663E-01, 0.76942E-01, 0.79218E-01, 0.81493E-01,
     & 0.83766E-01, 0.86037E-01, 0.88307E-01, 0.90574E-01, 0.92839E-01,
     & 0.95102E-01, 0.97364E-01, 0.99623E-01, 0.10188E+00, 0.10413E+00,
     & 0.10639E+00, 0.10864E+00, 0.11089E+00, 0.11313E+00, 0.11538E+00,
     & 0.11762E+00, 0.11986E+00, 0.12210E+00, 0.12433E+00, 0.12656E+00,
     & 0.12879E+00, 0.13102E+00, 0.13325E+00, 0.13547E+00, 0.13769E+00,
     & 0.13991E+00, 0.14213E+00, 0.14434E+00, 0.14655E+00, 0.14876E+00,
     & 0.15096E+00, 0.15317E+00, 0.15537E+00, 0.15757E+00, 0.15976E+00,
     & 0.16196E+00, 0.16415E+00, 0.16634E+00, 0.16852E+00, 0.17071E+00,
     & 0.17289E+00, 0.17506E+00, 0.17724E+00, 0.17941E+00, 0.18158E+00/

      DATA (BNC10M(I),I=201,300)/
     & 0.18375E+00, 0.18591E+00, 0.18808E+00, 0.19024E+00, 0.19239E+00,
     & 0.19455E+00, 0.19670E+00, 0.19885E+00, 0.20099E+00, 0.20314E+00,
     & 0.20528E+00, 0.20742E+00, 0.20955E+00, 0.21168E+00, 0.21381E+00,
     & 0.21594E+00, 0.21807E+00, 0.22019E+00, 0.22231E+00, 0.22442E+00,
     & 0.22654E+00, 0.22865E+00, 0.23076E+00, 0.23286E+00, 0.23497E+00,
     & 0.23707E+00, 0.23916E+00, 0.24126E+00, 0.24335E+00, 0.24544E+00,
     & 0.24753E+00, 0.24961E+00, 0.25169E+00, 0.25377E+00, 0.25584E+00,
     & 0.25792E+00, 0.25999E+00, 0.26205E+00, 0.26412E+00, 0.26618E+00,
     & 0.26824E+00, 0.27029E+00, 0.27235E+00, 0.27440E+00, 0.27645E+00,
     & 0.27849E+00, 0.28054E+00, 0.28257E+00, 0.28461E+00, 0.28665E+00,
     & 0.28868E+00, 0.29071E+00, 0.29273E+00, 0.29476E+00, 0.29678E+00,
     & 0.29880E+00, 0.30081E+00, 0.30282E+00, 0.30483E+00, 0.30684E+00,
     & 0.30884E+00, 0.31085E+00, 0.31285E+00, 0.31484E+00, 0.31684E+00,
     & 0.31883E+00, 0.32081E+00, 0.32280E+00, 0.32478E+00, 0.32676E+00,
     & 0.32874E+00, 0.33072E+00, 0.33269E+00, 0.33466E+00, 0.33662E+00,
     & 0.33859E+00, 0.34055E+00, 0.34251E+00, 0.34446E+00, 0.34642E+00,
     & 0.34837E+00, 0.35032E+00, 0.35226E+00, 0.35420E+00, 0.35615E+00,
     & 0.35808E+00, 0.36002E+00, 0.36195E+00, 0.36388E+00, 0.36581E+00,
     & 0.36773E+00, 0.36965E+00, 0.37157E+00, 0.37349E+00, 0.37540E+00,
     & 0.37731E+00, 0.37922E+00, 0.38113E+00, 0.38303E+00, 0.38494E+00/

      DATA (BNC10M(I),I=301,400)/
     & 0.38683E+00, 0.38873E+00, 0.39062E+00, 0.39251E+00, 0.39440E+00,
     & 0.39629E+00, 0.39817E+00, 0.40005E+00, 0.40193E+00, 0.40381E+00,
     & 0.40568E+00, 0.40755E+00, 0.40942E+00, 0.41128E+00, 0.41315E+00,
     & 0.41501E+00, 0.41687E+00, 0.41872E+00, 0.42057E+00, 0.42242E+00,
     & 0.42427E+00, 0.42612E+00, 0.42796E+00, 0.42980E+00, 0.43164E+00,
     & 0.43348E+00, 0.43531E+00, 0.43714E+00, 0.43897E+00, 0.44079E+00,
     & 0.44262E+00, 0.44444E+00, 0.44626E+00, 0.44807E+00, 0.44989E+00,
     & 0.45170E+00, 0.45351E+00, 0.45532E+00, 0.45712E+00, 0.45892E+00,
     & 0.46072E+00, 0.46252E+00, 0.46431E+00, 0.46611E+00, 0.46790E+00,
     & 0.46968E+00, 0.47147E+00, 0.47325E+00, 0.47503E+00, 0.47681E+00,
     & 0.47859E+00, 0.48036E+00, 0.48213E+00, 0.48390E+00, 0.48567E+00,
     & 0.48743E+00, 0.48919E+00, 0.49095E+00, 0.49271E+00, 0.49446E+00,
     & 0.49622E+00, 0.49797E+00, 0.49972E+00, 0.50146E+00, 0.50321E+00,
     & 0.50495E+00, 0.50669E+00, 0.50842E+00, 0.51016E+00, 0.51189E+00,
     & 0.51362E+00, 0.51535E+00, 0.51707E+00, 0.51880E+00, 0.52052E+00,
     & 0.52224E+00, 0.52395E+00, 0.52567E+00, 0.52738E+00, 0.52909E+00,
     & 0.53080E+00, 0.53251E+00, 0.53421E+00, 0.53591E+00, 0.53761E+00,
     & 0.53931E+00, 0.54100E+00, 0.54269E+00, 0.54439E+00, 0.54607E+00,
     & 0.54776E+00, 0.54944E+00, 0.55113E+00, 0.55281E+00, 0.55448E+00,
     & 0.55616E+00, 0.55783E+00, 0.55951E+00, 0.56118E+00, 0.56284E+00/

      DATA (BNC10M(I),I=401,500)/
     & 0.56451E+00, 0.56617E+00, 0.56783E+00, 0.56949E+00, 0.57115E+00,
     & 0.57280E+00, 0.57446E+00, 0.57611E+00, 0.57775E+00, 0.57940E+00,
     & 0.58105E+00, 0.58269E+00, 0.58433E+00, 0.58597E+00, 0.58760E+00,
     & 0.58924E+00, 0.59087E+00, 0.59250E+00, 0.59413E+00, 0.59576E+00,
     & 0.59738E+00, 0.59900E+00, 0.60062E+00, 0.60224E+00, 0.60386E+00,
     & 0.60547E+00, 0.60708E+00, 0.60869E+00, 0.61030E+00, 0.61191E+00,
     & 0.61351E+00, 0.61512E+00, 0.61672E+00, 0.61832E+00, 0.61991E+00,
     & 0.62151E+00, 0.62310E+00, 0.62469E+00, 0.62628E+00, 0.62787E+00,
     & 0.62945E+00, 0.63104E+00, 0.63262E+00, 0.63420E+00, 0.63577E+00,
     & 0.63735E+00, 0.63892E+00, 0.64050E+00, 0.64207E+00, 0.64363E+00,
     & 0.64520E+00, 0.64676E+00, 0.64833E+00, 0.64989E+00, 0.65145E+00,
     & 0.65300E+00, 0.65456E+00, 0.65611E+00, 0.65766E+00, 0.65921E+00,
     & 0.66076E+00, 0.66231E+00, 0.66385E+00, 0.66539E+00, 0.66693E+00,
     & 0.66847E+00, 0.67001E+00, 0.67154E+00, 0.67308E+00, 0.67461E+00,
     & 0.67614E+00, 0.67767E+00, 0.67919E+00, 0.68072E+00, 0.68224E+00,
     & 0.68376E+00, 0.68528E+00, 0.68680E+00, 0.68831E+00, 0.68983E+00,
     & 0.69134E+00, 0.69285E+00, 0.69436E+00, 0.69587E+00, 0.69737E+00,
     & 0.69887E+00, 0.70038E+00, 0.70188E+00, 0.70337E+00, 0.70487E+00,
     & 0.70636E+00, 0.70786E+00, 0.70935E+00, 0.71084E+00, 0.71233E+00,
     & 0.71381E+00, 0.71530E+00, 0.71678E+00, 0.71826E+00, 0.71974E+00/

      DATA (BNC10M(I),I=501,600)/
     & 0.72122E+00, 0.72270E+00, 0.72417E+00, 0.72564E+00, 0.72712E+00,
     & 0.72859E+00, 0.73005E+00, 0.73152E+00, 0.73298E+00, 0.73445E+00,
     & 0.73591E+00, 0.73737E+00, 0.73883E+00, 0.74028E+00, 0.74174E+00,
     & 0.74319E+00, 0.74464E+00, 0.74609E+00, 0.74754E+00, 0.74899E+00,
     & 0.75043E+00, 0.75188E+00, 0.75332E+00, 0.75476E+00, 0.75620E+00,
     & 0.75764E+00, 0.75907E+00, 0.76051E+00, 0.76194E+00, 0.76337E+00,
     & 0.76480E+00, 0.76623E+00, 0.76766E+00, 0.76908E+00, 0.77050E+00,
     & 0.77193E+00, 0.77335E+00, 0.77476E+00, 0.77618E+00, 0.77760E+00,
     & 0.77901E+00, 0.78042E+00, 0.78183E+00, 0.78324E+00, 0.78465E+00,
     & 0.78606E+00, 0.78746E+00, 0.78887E+00, 0.79027E+00, 0.79167E+00,
     & 0.79307E+00, 0.79447E+00, 0.79586E+00, 0.79726E+00, 0.79865E+00,
     & 0.80004E+00, 0.80143E+00, 0.80282E+00, 0.80421E+00, 0.80559E+00,
     & 0.80698E+00, 0.80836E+00, 0.80974E+00, 0.81112E+00, 0.81250E+00,
     & 0.81388E+00, 0.81525E+00, 0.81663E+00, 0.81800E+00, 0.81937E+00,
     & 0.82074E+00, 0.82211E+00, 0.82348E+00, 0.82484E+00, 0.82621E+00,
     & 0.82757E+00, 0.82893E+00, 0.83029E+00, 0.83165E+00, 0.83301E+00,
     & 0.83436E+00, 0.83572E+00, 0.83707E+00, 0.83842E+00, 0.83977E+00,
     & 0.84112E+00, 0.84247E+00, 0.84381E+00, 0.84516E+00, 0.84650E+00,
     & 0.84784E+00, 0.84918E+00, 0.85052E+00, 0.85186E+00, 0.85320E+00,
     & 0.85453E+00, 0.85586E+00, 0.85720E+00, 0.85853E+00, 0.85986E+00/

      DATA (BNC10M(I),I=601,700)/
     & 0.87414E+00, 0.88722E+00, 0.90017E+00, 0.91300E+00, 0.92570E+00,
     & 0.93828E+00, 0.95075E+00, 0.96310E+00, 0.97533E+00, 0.98746E+00,
     & 0.99947E+00, 0.10114E+01, 0.10232E+01, 0.10349E+01, 0.10465E+01,
     & 0.10580E+01, 0.10694E+01, 0.10807E+01, 0.10919E+01, 0.11030E+01,
     & 0.11141E+01, 0.11250E+01, 0.11358E+01, 0.11466E+01, 0.11573E+01,
     & 0.11679E+01, 0.11784E+01, 0.11889E+01, 0.11992E+01, 0.12095E+01,
     & 0.12197E+01, 0.12299E+01, 0.12399E+01, 0.12499E+01, 0.12598E+01,
     & 0.12697E+01, 0.12795E+01, 0.12892E+01, 0.12988E+01, 0.13084E+01,
     & 0.13179E+01, 0.13274E+01, 0.13367E+01, 0.13461E+01, 0.13553E+01,
     & 0.13645E+01, 0.13737E+01, 0.13828E+01, 0.13918E+01, 0.14008E+01,
     & 0.14097E+01, 0.14186E+01, 0.14274E+01, 0.14362E+01, 0.14449E+01,
     & 0.14535E+01, 0.14621E+01, 0.14707E+01, 0.14792E+01, 0.14876E+01,
     & 0.14960E+01, 0.15044E+01, 0.15127E+01, 0.15210E+01, 0.15292E+01,
     & 0.15374E+01, 0.15455E+01, 0.15536E+01, 0.15616E+01, 0.15696E+01,
     & 0.15776E+01, 0.15855E+01, 0.15934E+01, 0.16012E+01, 0.16090E+01,
     & 0.16168E+01, 0.16245E+01, 0.16322E+01, 0.16398E+01, 0.16474E+01,
     & 0.16550E+01, 0.16625E+01, 0.16700E+01, 0.16774E+01, 0.16849E+01,
     & 0.16922E+01, 0.16996E+01, 0.17069E+01, 0.17142E+01, 0.17214E+01,
     & 0.17286E+01, 0.17358E+01, 0.17430E+01, 0.17501E+01, 0.17572E+01,
     & 0.17642E+01, 0.17712E+01, 0.17782E+01, 0.17852E+01, 0.17921E+01/

      DATA (BNC10M(I),I=701,741)/
     & 0.17990E+01, 0.18059E+01, 0.18127E+01, 0.18195E+01, 0.18263E+01,
     & 0.18331E+01, 0.18398E+01, 0.18465E+01, 0.18532E+01, 0.18598E+01,
     & 0.18664E+01, 0.18730E+01, 0.18796E+01, 0.18861E+01, 0.18926E+01,
     & 0.18991E+01, 0.19056E+01, 0.19120E+01, 0.19184E+01, 0.19248E+01,
     & 0.19312E+01, 0.19375E+01, 0.19439E+01, 0.19502E+01, 0.19564E+01,
     & 0.19627E+01, 0.19689E+01, 0.19751E+01, 0.19813E+01, 0.19874E+01,
     & 0.19936E+01, 0.19997E+01, 0.20058E+01, 0.20118E+01, 0.20179E+01,
     & 0.20239E+01, 0.20299E+01, 0.20359E+01, 0.20419E+01, 0.20478E+01,
     & 0.20537E+01
     & /
C
C *** (H,Cl)       
C
      DATA (BNC11M(I),I=1,100)/
     &-0.43207E-01,-0.87312E-01,-0.10580E+00,-0.11699E+00,-0.12443E+00,
     &-0.12956E+00,-0.13308E+00,-0.13541E+00,-0.13684E+00,-0.13754E+00,
     &-0.13763E+00,-0.13723E+00,-0.13639E+00,-0.13519E+00,-0.13367E+00,
     &-0.13185E+00,-0.12978E+00,-0.12749E+00,-0.12498E+00,-0.12229E+00,
     &-0.11943E+00,-0.11640E+00,-0.11324E+00,-0.10994E+00,-0.10651E+00,
     &-0.10298E+00,-0.99331E-01,-0.95587E-01,-0.91752E-01,-0.87830E-01,
     &-0.83828E-01,-0.79751E-01,-0.75605E-01,-0.71393E-01,-0.67121E-01,
     &-0.62791E-01,-0.58408E-01,-0.53976E-01,-0.49497E-01,-0.44974E-01,
     &-0.40412E-01,-0.35811E-01,-0.31175E-01,-0.26506E-01,-0.21806E-01,
     &-0.17077E-01,-0.12322E-01,-0.75410E-02,-0.27366E-02, 0.20900E-02,
     & 0.69376E-02, 0.11805E-01, 0.16691E-01, 0.21595E-01, 0.26516E-01,
     & 0.31453E-01, 0.36406E-01, 0.41374E-01, 0.46358E-01, 0.51356E-01,
     & 0.56370E-01, 0.61398E-01, 0.66441E-01, 0.71499E-01, 0.76572E-01,
     & 0.81661E-01, 0.86766E-01, 0.91888E-01, 0.97027E-01, 0.10218E+00,
     & 0.10736E+00, 0.11255E+00, 0.11777E+00, 0.12300E+00, 0.12825E+00,
     & 0.13353E+00, 0.13883E+00, 0.14415E+00, 0.14950E+00, 0.15487E+00,
     & 0.16026E+00, 0.16568E+00, 0.17113E+00, 0.17660E+00, 0.18210E+00,
     & 0.18763E+00, 0.19318E+00, 0.19876E+00, 0.20437E+00, 0.21000E+00,
     & 0.21567E+00, 0.22136E+00, 0.22707E+00, 0.23281E+00, 0.23858E+00,
     & 0.24438E+00, 0.25020E+00, 0.25604E+00, 0.26190E+00, 0.26779E+00/

      DATA (BNC11M(I),I=101,200)/
     & 0.27370E+00, 0.27964E+00, 0.28559E+00, 0.29156E+00, 0.29755E+00,
     & 0.30356E+00, 0.30958E+00, 0.31562E+00, 0.32167E+00, 0.32774E+00,
     & 0.33382E+00, 0.33991E+00, 0.34601E+00, 0.35211E+00, 0.35823E+00,
     & 0.36435E+00, 0.37048E+00, 0.37662E+00, 0.38276E+00, 0.38890E+00,
     & 0.39505E+00, 0.40120E+00, 0.40734E+00, 0.41349E+00, 0.41964E+00,
     & 0.42579E+00, 0.43194E+00, 0.43809E+00, 0.44423E+00, 0.45037E+00,
     & 0.45651E+00, 0.46264E+00, 0.46877E+00, 0.47489E+00, 0.48101E+00,
     & 0.48712E+00, 0.49323E+00, 0.49933E+00, 0.50543E+00, 0.51151E+00,
     & 0.51759E+00, 0.52367E+00, 0.52973E+00, 0.53579E+00, 0.54185E+00,
     & 0.54789E+00, 0.55392E+00, 0.55995E+00, 0.56597E+00, 0.57198E+00,
     & 0.57798E+00, 0.58398E+00, 0.58996E+00, 0.59594E+00, 0.60190E+00,
     & 0.60786E+00, 0.61381E+00, 0.61975E+00, 0.62568E+00, 0.63160E+00,
     & 0.63751E+00, 0.64341E+00, 0.64930E+00, 0.65519E+00, 0.66106E+00,
     & 0.66692E+00, 0.67277E+00, 0.67862E+00, 0.68445E+00, 0.69028E+00,
     & 0.69609E+00, 0.70189E+00, 0.70769E+00, 0.71347E+00, 0.71925E+00,
     & 0.72501E+00, 0.73076E+00, 0.73651E+00, 0.74224E+00, 0.74796E+00,
     & 0.75368E+00, 0.75938E+00, 0.76507E+00, 0.77076E+00, 0.77643E+00,
     & 0.78209E+00, 0.78774E+00, 0.79339E+00, 0.79902E+00, 0.80464E+00,
     & 0.81025E+00, 0.81585E+00, 0.82144E+00, 0.82702E+00, 0.83259E+00,
     & 0.83815E+00, 0.84370E+00, 0.84924E+00, 0.85477E+00, 0.86029E+00/

      DATA (BNC11M(I),I=201,300)/
     & 0.86580E+00, 0.87129E+00, 0.87678E+00, 0.88226E+00, 0.88773E+00,
     & 0.89318E+00, 0.89863E+00, 0.90407E+00, 0.90949E+00, 0.91491E+00,
     & 0.92032E+00, 0.92571E+00, 0.93110E+00, 0.93647E+00, 0.94184E+00,
     & 0.94720E+00, 0.95254E+00, 0.95788E+00, 0.96320E+00, 0.96852E+00,
     & 0.97382E+00, 0.97912E+00, 0.98440E+00, 0.98968E+00, 0.99494E+00,
     & 0.10002E+01, 0.10054E+01, 0.10107E+01, 0.10159E+01, 0.10211E+01,
     & 0.10263E+01, 0.10315E+01, 0.10367E+01, 0.10419E+01, 0.10470E+01,
     & 0.10522E+01, 0.10574E+01, 0.10625E+01, 0.10676E+01, 0.10727E+01,
     & 0.10778E+01, 0.10829E+01, 0.10880E+01, 0.10931E+01, 0.10982E+01,
     & 0.11032E+01, 0.11083E+01, 0.11133E+01, 0.11184E+01, 0.11234E+01,
     & 0.11284E+01, 0.11334E+01, 0.11384E+01, 0.11434E+01, 0.11484E+01,
     & 0.11533E+01, 0.11583E+01, 0.11632E+01, 0.11682E+01, 0.11731E+01,
     & 0.11780E+01, 0.11829E+01, 0.11878E+01, 0.11927E+01, 0.11976E+01,
     & 0.12025E+01, 0.12074E+01, 0.12122E+01, 0.12171E+01, 0.12219E+01,
     & 0.12267E+01, 0.12315E+01, 0.12364E+01, 0.12412E+01, 0.12460E+01,
     & 0.12507E+01, 0.12555E+01, 0.12603E+01, 0.12650E+01, 0.12698E+01,
     & 0.12745E+01, 0.12793E+01, 0.12840E+01, 0.12887E+01, 0.12934E+01,
     & 0.12981E+01, 0.13028E+01, 0.13075E+01, 0.13122E+01, 0.13168E+01,
     & 0.13215E+01, 0.13261E+01, 0.13308E+01, 0.13354E+01, 0.13400E+01,
     & 0.13447E+01, 0.13493E+01, 0.13539E+01, 0.13585E+01, 0.13630E+01/

      DATA (BNC11M(I),I=301,400)/
     & 0.13676E+01, 0.13722E+01, 0.13767E+01, 0.13813E+01, 0.13858E+01,
     & 0.13904E+01, 0.13949E+01, 0.13994E+01, 0.14039E+01, 0.14084E+01,
     & 0.14129E+01, 0.14174E+01, 0.14219E+01, 0.14264E+01, 0.14308E+01,
     & 0.14353E+01, 0.14397E+01, 0.14442E+01, 0.14486E+01, 0.14530E+01,
     & 0.14574E+01, 0.14618E+01, 0.14662E+01, 0.14706E+01, 0.14750E+01,
     & 0.14794E+01, 0.14838E+01, 0.14881E+01, 0.14925E+01, 0.14968E+01,
     & 0.15012E+01, 0.15055E+01, 0.15099E+01, 0.15142E+01, 0.15185E+01,
     & 0.15228E+01, 0.15271E+01, 0.15314E+01, 0.15357E+01, 0.15399E+01,
     & 0.15442E+01, 0.15485E+01, 0.15527E+01, 0.15570E+01, 0.15612E+01,
     & 0.15654E+01, 0.15697E+01, 0.15739E+01, 0.15781E+01, 0.15823E+01,
     & 0.15865E+01, 0.15907E+01, 0.15949E+01, 0.15991E+01, 0.16032E+01,
     & 0.16074E+01, 0.16115E+01, 0.16157E+01, 0.16198E+01, 0.16240E+01,
     & 0.16281E+01, 0.16322E+01, 0.16363E+01, 0.16405E+01, 0.16446E+01,
     & 0.16487E+01, 0.16527E+01, 0.16568E+01, 0.16609E+01, 0.16650E+01,
     & 0.16690E+01, 0.16731E+01, 0.16771E+01, 0.16812E+01, 0.16852E+01,
     & 0.16893E+01, 0.16933E+01, 0.16973E+01, 0.17013E+01, 0.17053E+01,
     & 0.17093E+01, 0.17133E+01, 0.17173E+01, 0.17213E+01, 0.17252E+01,
     & 0.17292E+01, 0.17332E+01, 0.17371E+01, 0.17411E+01, 0.17450E+01,
     & 0.17490E+01, 0.17529E+01, 0.17568E+01, 0.17607E+01, 0.17646E+01,
     & 0.17685E+01, 0.17724E+01, 0.17763E+01, 0.17802E+01, 0.17841E+01/

      DATA (BNC11M(I),I=401,500)/
     & 0.17880E+01, 0.17919E+01, 0.17957E+01, 0.17996E+01, 0.18034E+01,
     & 0.18073E+01, 0.18111E+01, 0.18150E+01, 0.18188E+01, 0.18226E+01,
     & 0.18264E+01, 0.18302E+01, 0.18340E+01, 0.18378E+01, 0.18416E+01,
     & 0.18454E+01, 0.18492E+01, 0.18530E+01, 0.18568E+01, 0.18605E+01,
     & 0.18643E+01, 0.18680E+01, 0.18718E+01, 0.18755E+01, 0.18793E+01,
     & 0.18830E+01, 0.18867E+01, 0.18904E+01, 0.18942E+01, 0.18979E+01,
     & 0.19016E+01, 0.19053E+01, 0.19090E+01, 0.19127E+01, 0.19163E+01,
     & 0.19200E+01, 0.19237E+01, 0.19274E+01, 0.19310E+01, 0.19347E+01,
     & 0.19383E+01, 0.19420E+01, 0.19456E+01, 0.19493E+01, 0.19529E+01,
     & 0.19565E+01, 0.19601E+01, 0.19637E+01, 0.19674E+01, 0.19710E+01,
     & 0.19746E+01, 0.19782E+01, 0.19817E+01, 0.19853E+01, 0.19889E+01,
     & 0.19925E+01, 0.19960E+01, 0.19996E+01, 0.20032E+01, 0.20067E+01,
     & 0.20103E+01, 0.20138E+01, 0.20174E+01, 0.20209E+01, 0.20244E+01,
     & 0.20279E+01, 0.20315E+01, 0.20350E+01, 0.20385E+01, 0.20420E+01,
     & 0.20455E+01, 0.20490E+01, 0.20525E+01, 0.20560E+01, 0.20594E+01,
     & 0.20629E+01, 0.20664E+01, 0.20698E+01, 0.20733E+01, 0.20768E+01,
     & 0.20802E+01, 0.20837E+01, 0.20871E+01, 0.20905E+01, 0.20940E+01,
     & 0.20974E+01, 0.21008E+01, 0.21042E+01, 0.21077E+01, 0.21111E+01,
     & 0.21145E+01, 0.21179E+01, 0.21213E+01, 0.21247E+01, 0.21280E+01,
     & 0.21314E+01, 0.21348E+01, 0.21382E+01, 0.21415E+01, 0.21449E+01/

      DATA (BNC11M(I),I=501,600)/
     & 0.21483E+01, 0.21516E+01, 0.21550E+01, 0.21583E+01, 0.21617E+01,
     & 0.21650E+01, 0.21683E+01, 0.21717E+01, 0.21750E+01, 0.21783E+01,
     & 0.21816E+01, 0.21849E+01, 0.21882E+01, 0.21915E+01, 0.21948E+01,
     & 0.21981E+01, 0.22014E+01, 0.22047E+01, 0.22080E+01, 0.22113E+01,
     & 0.22145E+01, 0.22178E+01, 0.22211E+01, 0.22243E+01, 0.22276E+01,
     & 0.22308E+01, 0.22341E+01, 0.22373E+01, 0.22406E+01, 0.22438E+01,
     & 0.22470E+01, 0.22503E+01, 0.22535E+01, 0.22567E+01, 0.22599E+01,
     & 0.22631E+01, 0.22663E+01, 0.22695E+01, 0.22727E+01, 0.22759E+01,
     & 0.22791E+01, 0.22823E+01, 0.22855E+01, 0.22887E+01, 0.22918E+01,
     & 0.22950E+01, 0.22982E+01, 0.23013E+01, 0.23045E+01, 0.23076E+01,
     & 0.23108E+01, 0.23139E+01, 0.23171E+01, 0.23202E+01, 0.23234E+01,
     & 0.23265E+01, 0.23296E+01, 0.23327E+01, 0.23359E+01, 0.23390E+01,
     & 0.23421E+01, 0.23452E+01, 0.23483E+01, 0.23514E+01, 0.23545E+01,
     & 0.23576E+01, 0.23607E+01, 0.23638E+01, 0.23668E+01, 0.23699E+01,
     & 0.23730E+01, 0.23761E+01, 0.23791E+01, 0.23822E+01, 0.23853E+01,
     & 0.23883E+01, 0.23914E+01, 0.23944E+01, 0.23975E+01, 0.24005E+01,
     & 0.24035E+01, 0.24066E+01, 0.24096E+01, 0.24126E+01, 0.24156E+01,
     & 0.24187E+01, 0.24217E+01, 0.24247E+01, 0.24277E+01, 0.24307E+01,
     & 0.24337E+01, 0.24367E+01, 0.24397E+01, 0.24427E+01, 0.24457E+01,
     & 0.24487E+01, 0.24516E+01, 0.24546E+01, 0.24576E+01, 0.24606E+01/

      DATA (BNC11M(I),I=601,700)/
     & 0.24924E+01, 0.25216E+01, 0.25504E+01, 0.25789E+01, 0.26071E+01,
     & 0.26349E+01, 0.26625E+01, 0.26898E+01, 0.27168E+01, 0.27435E+01,
     & 0.27699E+01, 0.27961E+01, 0.28220E+01, 0.28477E+01, 0.28731E+01,
     & 0.28982E+01, 0.29231E+01, 0.29478E+01, 0.29722E+01, 0.29965E+01,
     & 0.30205E+01, 0.30442E+01, 0.30678E+01, 0.30911E+01, 0.31143E+01,
     & 0.31372E+01, 0.31599E+01, 0.31825E+01, 0.32048E+01, 0.32270E+01,
     & 0.32490E+01, 0.32708E+01, 0.32924E+01, 0.33138E+01, 0.33351E+01,
     & 0.33562E+01, 0.33771E+01, 0.33979E+01, 0.34185E+01, 0.34389E+01,
     & 0.34592E+01, 0.34794E+01, 0.34993E+01, 0.35192E+01, 0.35389E+01,
     & 0.35584E+01, 0.35778E+01, 0.35971E+01, 0.36162E+01, 0.36352E+01,
     & 0.36541E+01, 0.36728E+01, 0.36914E+01, 0.37099E+01, 0.37282E+01,
     & 0.37465E+01, 0.37646E+01, 0.37826E+01, 0.38004E+01, 0.38182E+01,
     & 0.38358E+01, 0.38533E+01, 0.38707E+01, 0.38880E+01, 0.39052E+01,
     & 0.39223E+01, 0.39393E+01, 0.39562E+01, 0.39729E+01, 0.39896E+01,
     & 0.40062E+01, 0.40226E+01, 0.40390E+01, 0.40553E+01, 0.40715E+01,
     & 0.40876E+01, 0.41035E+01, 0.41194E+01, 0.41353E+01, 0.41510E+01,
     & 0.41666E+01, 0.41822E+01, 0.41976E+01, 0.42130E+01, 0.42283E+01,
     & 0.42435E+01, 0.42586E+01, 0.42737E+01, 0.42886E+01, 0.43035E+01,
     & 0.43183E+01, 0.43330E+01, 0.43477E+01, 0.43623E+01, 0.43768E+01,
     & 0.43912E+01, 0.44056E+01, 0.44198E+01, 0.44341E+01, 0.44482E+01/

      DATA (BNC11M(I),I=701,741)/
     & 0.44623E+01, 0.44763E+01, 0.44902E+01, 0.45041E+01, 0.45179E+01,
     & 0.45316E+01, 0.45453E+01, 0.45589E+01, 0.45724E+01, 0.45859E+01,
     & 0.45993E+01, 0.46126E+01, 0.46259E+01, 0.46391E+01, 0.46523E+01,
     & 0.46654E+01, 0.46785E+01, 0.46915E+01, 0.47044E+01, 0.47173E+01,
     & 0.47301E+01, 0.47428E+01, 0.47555E+01, 0.47682E+01, 0.47808E+01,
     & 0.47933E+01, 0.48058E+01, 0.48183E+01, 0.48306E+01, 0.48430E+01,
     & 0.48552E+01, 0.48675E+01, 0.48797E+01, 0.48918E+01, 0.49039E+01,
     & 0.49159E+01, 0.49279E+01, 0.49398E+01, 0.49517E+01, 0.49635E+01,
     & 0.49753E+01
     & /
C
C *** NaHSO4       
C
      DATA (BNC12M(I),I=1,100)/
     &-0.43994E-01,-0.91554E-01,-0.11320E+00,-0.12742E+00,-0.13780E+00,
     &-0.14581E+00,-0.15217E+00,-0.15732E+00,-0.16153E+00,-0.16499E+00,
     &-0.16783E+00,-0.17017E+00,-0.17206E+00,-0.17357E+00,-0.17474E+00,
     &-0.17562E+00,-0.17624E+00,-0.17661E+00,-0.17677E+00,-0.17673E+00,
     &-0.17650E+00,-0.17610E+00,-0.17555E+00,-0.17485E+00,-0.17402E+00,
     &-0.17305E+00,-0.17197E+00,-0.17076E+00,-0.16946E+00,-0.16805E+00,
     &-0.16654E+00,-0.16494E+00,-0.16325E+00,-0.16148E+00,-0.15962E+00,
     &-0.15770E+00,-0.15570E+00,-0.15363E+00,-0.15149E+00,-0.14929E+00,
     &-0.14703E+00,-0.14471E+00,-0.14234E+00,-0.13991E+00,-0.13743E+00,
     &-0.13490E+00,-0.13232E+00,-0.12970E+00,-0.12704E+00,-0.12433E+00,
     &-0.12159E+00,-0.11880E+00,-0.11598E+00,-0.11312E+00,-0.11023E+00,
     &-0.10730E+00,-0.10434E+00,-0.10135E+00,-0.98326E-01,-0.95275E-01,
     &-0.92194E-01,-0.89084E-01,-0.85947E-01,-0.82782E-01,-0.79591E-01,
     &-0.76372E-01,-0.73128E-01,-0.69857E-01,-0.66561E-01,-0.63239E-01,
     &-0.59892E-01,-0.56519E-01,-0.53121E-01,-0.49698E-01,-0.46250E-01,
     &-0.42776E-01,-0.39277E-01,-0.35752E-01,-0.32203E-01,-0.28628E-01,
     &-0.25027E-01,-0.21401E-01,-0.17750E-01,-0.14074E-01,-0.10373E-01,
     &-0.66461E-02,-0.28948E-02, 0.88116E-03, 0.46816E-02, 0.85060E-02,
     & 0.12354E-01, 0.16226E-01, 0.20121E-01, 0.24038E-01, 0.27977E-01,
     & 0.31937E-01, 0.35918E-01, 0.39920E-01, 0.43941E-01, 0.47981E-01/

      DATA (BNC12M(I),I=101,200)/
     & 0.52038E-01, 0.56113E-01, 0.60205E-01, 0.64312E-01, 0.68434E-01,
     & 0.72570E-01, 0.76719E-01, 0.80880E-01, 0.85053E-01, 0.89237E-01,
     & 0.93430E-01, 0.97632E-01, 0.10184E+00, 0.10606E+00, 0.11028E+00,
     & 0.11451E+00, 0.11875E+00, 0.12299E+00, 0.12723E+00, 0.13148E+00,
     & 0.13572E+00, 0.13997E+00, 0.14422E+00, 0.14848E+00, 0.15273E+00,
     & 0.15698E+00, 0.16123E+00, 0.16548E+00, 0.16972E+00, 0.17397E+00,
     & 0.17821E+00, 0.18245E+00, 0.18668E+00, 0.19091E+00, 0.19514E+00,
     & 0.19936E+00, 0.20358E+00, 0.20780E+00, 0.21201E+00, 0.21621E+00,
     & 0.22041E+00, 0.22460E+00, 0.22879E+00, 0.23297E+00, 0.23715E+00,
     & 0.24132E+00, 0.24549E+00, 0.24964E+00, 0.25379E+00, 0.25794E+00,
     & 0.26208E+00, 0.26621E+00, 0.27034E+00, 0.27445E+00, 0.27857E+00,
     & 0.28267E+00, 0.28677E+00, 0.29086E+00, 0.29494E+00, 0.29902E+00,
     & 0.30309E+00, 0.30715E+00, 0.31121E+00, 0.31526E+00, 0.31930E+00,
     & 0.32333E+00, 0.32736E+00, 0.33138E+00, 0.33539E+00, 0.33940E+00,
     & 0.34340E+00, 0.34739E+00, 0.35137E+00, 0.35534E+00, 0.35931E+00,
     & 0.36327E+00, 0.36723E+00, 0.37117E+00, 0.37511E+00, 0.37904E+00,
     & 0.38297E+00, 0.38688E+00, 0.39079E+00, 0.39469E+00, 0.39859E+00,
     & 0.40248E+00, 0.40635E+00, 0.41023E+00, 0.41409E+00, 0.41795E+00,
     & 0.42180E+00, 0.42564E+00, 0.42948E+00, 0.43331E+00, 0.43713E+00,
     & 0.44094E+00, 0.44475E+00, 0.44854E+00, 0.45234E+00, 0.45612E+00/

      DATA (BNC12M(I),I=201,300)/
     & 0.45990E+00, 0.46367E+00, 0.46743E+00, 0.47119E+00, 0.47493E+00,
     & 0.47868E+00, 0.48241E+00, 0.48614E+00, 0.48986E+00, 0.49357E+00,
     & 0.49727E+00, 0.50097E+00, 0.50466E+00, 0.50835E+00, 0.51203E+00,
     & 0.51570E+00, 0.51936E+00, 0.52302E+00, 0.52667E+00, 0.53031E+00,
     & 0.53395E+00, 0.53757E+00, 0.54120E+00, 0.54481E+00, 0.54842E+00,
     & 0.55202E+00, 0.55562E+00, 0.55920E+00, 0.56279E+00, 0.56636E+00,
     & 0.56993E+00, 0.57349E+00, 0.57704E+00, 0.58059E+00, 0.58413E+00,
     & 0.58767E+00, 0.59120E+00, 0.59472E+00, 0.59823E+00, 0.60174E+00,
     & 0.60524E+00, 0.60874E+00, 0.61223E+00, 0.61571E+00, 0.61919E+00,
     & 0.62266E+00, 0.62612E+00, 0.62958E+00, 0.63303E+00, 0.63647E+00,
     & 0.63991E+00, 0.64334E+00, 0.64677E+00, 0.65019E+00, 0.65360E+00,
     & 0.65701E+00, 0.66041E+00, 0.66380E+00, 0.66719E+00, 0.67057E+00,
     & 0.67395E+00, 0.67732E+00, 0.68068E+00, 0.68404E+00, 0.68740E+00,
     & 0.69074E+00, 0.69408E+00, 0.69742E+00, 0.70074E+00, 0.70407E+00,
     & 0.70738E+00, 0.71069E+00, 0.71400E+00, 0.71730E+00, 0.72059E+00,
     & 0.72388E+00, 0.72716E+00, 0.73044E+00, 0.73371E+00, 0.73697E+00,
     & 0.74023E+00, 0.74349E+00, 0.74673E+00, 0.74997E+00, 0.75321E+00,
     & 0.75644E+00, 0.75967E+00, 0.76289E+00, 0.76610E+00, 0.76931E+00,
     & 0.77251E+00, 0.77571E+00, 0.77890E+00, 0.78209E+00, 0.78527E+00,
     & 0.78845E+00, 0.79162E+00, 0.79478E+00, 0.79794E+00, 0.80110E+00/

      DATA (BNC12M(I),I=301,400)/
     & 0.80424E+00, 0.80739E+00, 0.81053E+00, 0.81366E+00, 0.81679E+00,
     & 0.81991E+00, 0.82303E+00, 0.82614E+00, 0.82925E+00, 0.83235E+00,
     & 0.83545E+00, 0.83854E+00, 0.84163E+00, 0.84471E+00, 0.84778E+00,
     & 0.85085E+00, 0.85392E+00, 0.85698E+00, 0.86004E+00, 0.86309E+00,
     & 0.86614E+00, 0.86918E+00, 0.87221E+00, 0.87525E+00, 0.87827E+00,
     & 0.88129E+00, 0.88431E+00, 0.88732E+00, 0.89033E+00, 0.89333E+00,
     & 0.89633E+00, 0.89932E+00, 0.90231E+00, 0.90529E+00, 0.90827E+00,
     & 0.91124E+00, 0.91421E+00, 0.91718E+00, 0.92014E+00, 0.92309E+00,
     & 0.92604E+00, 0.92899E+00, 0.93193E+00, 0.93486E+00, 0.93779E+00,
     & 0.94072E+00, 0.94364E+00, 0.94656E+00, 0.94947E+00, 0.95238E+00,
     & 0.95529E+00, 0.95819E+00, 0.96108E+00, 0.96397E+00, 0.96686E+00,
     & 0.96974E+00, 0.97262E+00, 0.97549E+00, 0.97836E+00, 0.98122E+00,
     & 0.98408E+00, 0.98694E+00, 0.98979E+00, 0.99264E+00, 0.99548E+00,
     & 0.99832E+00, 0.10012E+01, 0.10040E+01, 0.10068E+01, 0.10096E+01,
     & 0.10124E+01, 0.10153E+01, 0.10181E+01, 0.10209E+01, 0.10237E+01,
     & 0.10265E+01, 0.10293E+01, 0.10320E+01, 0.10348E+01, 0.10376E+01,
     & 0.10404E+01, 0.10431E+01, 0.10459E+01, 0.10487E+01, 0.10514E+01,
     & 0.10542E+01, 0.10569E+01, 0.10597E+01, 0.10624E+01, 0.10652E+01,
     & 0.10679E+01, 0.10706E+01, 0.10734E+01, 0.10761E+01, 0.10788E+01,
     & 0.10815E+01, 0.10842E+01, 0.10869E+01, 0.10896E+01, 0.10923E+01/

      DATA (BNC12M(I),I=401,500)/
     & 0.10950E+01, 0.10977E+01, 0.11004E+01, 0.11031E+01, 0.11058E+01,
     & 0.11084E+01, 0.11111E+01, 0.11138E+01, 0.11165E+01, 0.11191E+01,
     & 0.11218E+01, 0.11244E+01, 0.11271E+01, 0.11297E+01, 0.11324E+01,
     & 0.11350E+01, 0.11376E+01, 0.11403E+01, 0.11429E+01, 0.11455E+01,
     & 0.11481E+01, 0.11507E+01, 0.11534E+01, 0.11560E+01, 0.11586E+01,
     & 0.11612E+01, 0.11638E+01, 0.11664E+01, 0.11690E+01, 0.11715E+01,
     & 0.11741E+01, 0.11767E+01, 0.11793E+01, 0.11819E+01, 0.11844E+01,
     & 0.11870E+01, 0.11896E+01, 0.11921E+01, 0.11947E+01, 0.11972E+01,
     & 0.11998E+01, 0.12023E+01, 0.12049E+01, 0.12074E+01, 0.12099E+01,
     & 0.12125E+01, 0.12150E+01, 0.12175E+01, 0.12200E+01, 0.12226E+01,
     & 0.12251E+01, 0.12276E+01, 0.12301E+01, 0.12326E+01, 0.12351E+01,
     & 0.12376E+01, 0.12401E+01, 0.12426E+01, 0.12451E+01, 0.12475E+01,
     & 0.12500E+01, 0.12525E+01, 0.12550E+01, 0.12575E+01, 0.12599E+01,
     & 0.12624E+01, 0.12649E+01, 0.12673E+01, 0.12698E+01, 0.12722E+01,
     & 0.12747E+01, 0.12771E+01, 0.12796E+01, 0.12820E+01, 0.12844E+01,
     & 0.12869E+01, 0.12893E+01, 0.12917E+01, 0.12941E+01, 0.12966E+01,
     & 0.12990E+01, 0.13014E+01, 0.13038E+01, 0.13062E+01, 0.13086E+01,
     & 0.13110E+01, 0.13134E+01, 0.13158E+01, 0.13182E+01, 0.13206E+01,
     & 0.13230E+01, 0.13254E+01, 0.13278E+01, 0.13301E+01, 0.13325E+01,
     & 0.13349E+01, 0.13373E+01, 0.13396E+01, 0.13420E+01, 0.13443E+01/

      DATA (BNC12M(I),I=501,600)/
     & 0.13467E+01, 0.13491E+01, 0.13514E+01, 0.13538E+01, 0.13561E+01,
     & 0.13584E+01, 0.13608E+01, 0.13631E+01, 0.13655E+01, 0.13678E+01,
     & 0.13701E+01, 0.13724E+01, 0.13748E+01, 0.13771E+01, 0.13794E+01,
     & 0.13817E+01, 0.13840E+01, 0.13863E+01, 0.13886E+01, 0.13909E+01,
     & 0.13932E+01, 0.13955E+01, 0.13978E+01, 0.14001E+01, 0.14024E+01,
     & 0.14047E+01, 0.14070E+01, 0.14093E+01, 0.14115E+01, 0.14138E+01,
     & 0.14161E+01, 0.14184E+01, 0.14206E+01, 0.14229E+01, 0.14252E+01,
     & 0.14274E+01, 0.14297E+01, 0.14319E+01, 0.14342E+01, 0.14364E+01,
     & 0.14387E+01, 0.14409E+01, 0.14431E+01, 0.14454E+01, 0.14476E+01,
     & 0.14499E+01, 0.14521E+01, 0.14543E+01, 0.14565E+01, 0.14588E+01,
     & 0.14610E+01, 0.14632E+01, 0.14654E+01, 0.14676E+01, 0.14698E+01,
     & 0.14720E+01, 0.14742E+01, 0.14764E+01, 0.14786E+01, 0.14808E+01,
     & 0.14830E+01, 0.14852E+01, 0.14874E+01, 0.14896E+01, 0.14918E+01,
     & 0.14940E+01, 0.14961E+01, 0.14983E+01, 0.15005E+01, 0.15027E+01,
     & 0.15048E+01, 0.15070E+01, 0.15092E+01, 0.15113E+01, 0.15135E+01,
     & 0.15156E+01, 0.15178E+01, 0.15199E+01, 0.15221E+01, 0.15242E+01,
     & 0.15264E+01, 0.15285E+01, 0.15307E+01, 0.15328E+01, 0.15349E+01,
     & 0.15371E+01, 0.15392E+01, 0.15413E+01, 0.15434E+01, 0.15456E+01,
     & 0.15477E+01, 0.15498E+01, 0.15519E+01, 0.15540E+01, 0.15561E+01,
     & 0.15583E+01, 0.15604E+01, 0.15625E+01, 0.15646E+01, 0.15667E+01/

      DATA (BNC12M(I),I=601,700)/
     & 0.15892E+01, 0.16098E+01, 0.16302E+01, 0.16504E+01, 0.16704E+01,
     & 0.16902E+01, 0.17098E+01, 0.17291E+01, 0.17483E+01, 0.17673E+01,
     & 0.17861E+01, 0.18047E+01, 0.18232E+01, 0.18415E+01, 0.18596E+01,
     & 0.18775E+01, 0.18953E+01, 0.19129E+01, 0.19304E+01, 0.19477E+01,
     & 0.19648E+01, 0.19818E+01, 0.19987E+01, 0.20154E+01, 0.20320E+01,
     & 0.20484E+01, 0.20647E+01, 0.20809E+01, 0.20969E+01, 0.21129E+01,
     & 0.21286E+01, 0.21443E+01, 0.21598E+01, 0.21752E+01, 0.21905E+01,
     & 0.22057E+01, 0.22208E+01, 0.22357E+01, 0.22506E+01, 0.22653E+01,
     & 0.22799E+01, 0.22945E+01, 0.23089E+01, 0.23232E+01, 0.23374E+01,
     & 0.23515E+01, 0.23656E+01, 0.23795E+01, 0.23933E+01, 0.24071E+01,
     & 0.24207E+01, 0.24343E+01, 0.24477E+01, 0.24611E+01, 0.24744E+01,
     & 0.24876E+01, 0.25007E+01, 0.25138E+01, 0.25267E+01, 0.25396E+01,
     & 0.25524E+01, 0.25651E+01, 0.25778E+01, 0.25903E+01, 0.26028E+01,
     & 0.26152E+01, 0.26276E+01, 0.26398E+01, 0.26520E+01, 0.26642E+01,
     & 0.26762E+01, 0.26882E+01, 0.27001E+01, 0.27120E+01, 0.27238E+01,
     & 0.27355E+01, 0.27472E+01, 0.27588E+01, 0.27703E+01, 0.27818E+01,
     & 0.27932E+01, 0.28045E+01, 0.28158E+01, 0.28270E+01, 0.28382E+01,
     & 0.28493E+01, 0.28604E+01, 0.28714E+01, 0.28823E+01, 0.28932E+01,
     & 0.29040E+01, 0.29148E+01, 0.29256E+01, 0.29362E+01, 0.29468E+01,
     & 0.29574E+01, 0.29679E+01, 0.29784E+01, 0.29888E+01, 0.29992E+01/

      DATA (BNC12M(I),I=701,741)/
     & 0.30095E+01, 0.30198E+01, 0.30300E+01, 0.30402E+01, 0.30503E+01,
     & 0.30604E+01, 0.30705E+01, 0.30805E+01, 0.30904E+01, 0.31003E+01,
     & 0.31102E+01, 0.31200E+01, 0.31298E+01, 0.31395E+01, 0.31492E+01,
     & 0.31588E+01, 0.31684E+01, 0.31780E+01, 0.31875E+01, 0.31970E+01,
     & 0.32065E+01, 0.32159E+01, 0.32252E+01, 0.32346E+01, 0.32439E+01,
     & 0.32531E+01, 0.32623E+01, 0.32715E+01, 0.32807E+01, 0.32898E+01,
     & 0.32988E+01, 0.33079E+01, 0.33169E+01, 0.33258E+01, 0.33348E+01,
     & 0.33436E+01, 0.33525E+01, 0.33613E+01, 0.33701E+01, 0.33789E+01,
     & 0.33876E+01
     & /
C
C *** (NH4)3H(SO4)2
C
      DATA (BNC13M(I),I=1,100)/
     &-0.72232E-01,-0.15583E+00,-0.19693E+00,-0.22563E+00,-0.24790E+00,
     &-0.26615E+00,-0.28162E+00,-0.29503E+00,-0.30686E+00,-0.31742E+00,
     &-0.32695E+00,-0.33562E+00,-0.34356E+00,-0.35087E+00,-0.35764E+00,
     &-0.36392E+00,-0.36978E+00,-0.37525E+00,-0.38039E+00,-0.38521E+00,
     &-0.38975E+00,-0.39403E+00,-0.39808E+00,-0.40190E+00,-0.40552E+00,
     &-0.40895E+00,-0.41221E+00,-0.41530E+00,-0.41823E+00,-0.42103E+00,
     &-0.42368E+00,-0.42620E+00,-0.42861E+00,-0.43090E+00,-0.43307E+00,
     &-0.43515E+00,-0.43712E+00,-0.43900E+00,-0.44079E+00,-0.44249E+00,
     &-0.44411E+00,-0.44565E+00,-0.44712E+00,-0.44851E+00,-0.44984E+00,
     &-0.45109E+00,-0.45229E+00,-0.45342E+00,-0.45449E+00,-0.45551E+00,
     &-0.45647E+00,-0.45738E+00,-0.45823E+00,-0.45904E+00,-0.45980E+00,
     &-0.46052E+00,-0.46119E+00,-0.46182E+00,-0.46241E+00,-0.46296E+00,
     &-0.46347E+00,-0.46394E+00,-0.46438E+00,-0.46478E+00,-0.46515E+00,
     &-0.46549E+00,-0.46579E+00,-0.46607E+00,-0.46631E+00,-0.46653E+00,
     &-0.46671E+00,-0.46687E+00,-0.46701E+00,-0.46711E+00,-0.46720E+00,
     &-0.46725E+00,-0.46729E+00,-0.46730E+00,-0.46728E+00,-0.46725E+00,
     &-0.46719E+00,-0.46711E+00,-0.46701E+00,-0.46689E+00,-0.46675E+00,
     &-0.46659E+00,-0.46641E+00,-0.46621E+00,-0.46599E+00,-0.46576E+00,
     &-0.46550E+00,-0.46523E+00,-0.46495E+00,-0.46465E+00,-0.46433E+00,
     &-0.46399E+00,-0.46364E+00,-0.46328E+00,-0.46290E+00,-0.46251E+00/

      DATA (BNC13M(I),I=101,200)/
     &-0.46211E+00,-0.46169E+00,-0.46125E+00,-0.46081E+00,-0.46035E+00,
     &-0.45989E+00,-0.45941E+00,-0.45892E+00,-0.45842E+00,-0.45791E+00,
     &-0.45739E+00,-0.45686E+00,-0.45632E+00,-0.45577E+00,-0.45522E+00,
     &-0.45465E+00,-0.45408E+00,-0.45350E+00,-0.45291E+00,-0.45232E+00,
     &-0.45172E+00,-0.45111E+00,-0.45050E+00,-0.44988E+00,-0.44926E+00,
     &-0.44863E+00,-0.44799E+00,-0.44735E+00,-0.44671E+00,-0.44606E+00,
     &-0.44540E+00,-0.44474E+00,-0.44408E+00,-0.44342E+00,-0.44274E+00,
     &-0.44207E+00,-0.44139E+00,-0.44071E+00,-0.44003E+00,-0.43934E+00,
     &-0.43865E+00,-0.43796E+00,-0.43726E+00,-0.43656E+00,-0.43586E+00,
     &-0.43515E+00,-0.43445E+00,-0.43374E+00,-0.43303E+00,-0.43231E+00,
     &-0.43160E+00,-0.43088E+00,-0.43016E+00,-0.42944E+00,-0.42872E+00,
     &-0.42799E+00,-0.42727E+00,-0.42654E+00,-0.42581E+00,-0.42508E+00,
     &-0.42435E+00,-0.42361E+00,-0.42288E+00,-0.42214E+00,-0.42140E+00,
     &-0.42067E+00,-0.41993E+00,-0.41918E+00,-0.41844E+00,-0.41770E+00,
     &-0.41696E+00,-0.41621E+00,-0.41547E+00,-0.41472E+00,-0.41397E+00,
     &-0.41322E+00,-0.41248E+00,-0.41173E+00,-0.41098E+00,-0.41023E+00,
     &-0.40947E+00,-0.40872E+00,-0.40797E+00,-0.40722E+00,-0.40646E+00,
     &-0.40571E+00,-0.40496E+00,-0.40420E+00,-0.40345E+00,-0.40269E+00,
     &-0.40193E+00,-0.40118E+00,-0.40042E+00,-0.39967E+00,-0.39891E+00,
     &-0.39815E+00,-0.39739E+00,-0.39664E+00,-0.39588E+00,-0.39512E+00/

      DATA (BNC13M(I),I=201,300)/
     &-0.39436E+00,-0.39361E+00,-0.39285E+00,-0.39209E+00,-0.39133E+00,
     &-0.39057E+00,-0.38981E+00,-0.38906E+00,-0.38830E+00,-0.38754E+00,
     &-0.38678E+00,-0.38602E+00,-0.38526E+00,-0.38451E+00,-0.38375E+00,
     &-0.38299E+00,-0.38223E+00,-0.38147E+00,-0.38071E+00,-0.37996E+00,
     &-0.37920E+00,-0.37844E+00,-0.37768E+00,-0.37693E+00,-0.37617E+00,
     &-0.37541E+00,-0.37466E+00,-0.37390E+00,-0.37314E+00,-0.37239E+00,
     &-0.37163E+00,-0.37087E+00,-0.37012E+00,-0.36936E+00,-0.36861E+00,
     &-0.36785E+00,-0.36710E+00,-0.36634E+00,-0.36559E+00,-0.36484E+00,
     &-0.36408E+00,-0.36333E+00,-0.36258E+00,-0.36182E+00,-0.36107E+00,
     &-0.36032E+00,-0.35957E+00,-0.35882E+00,-0.35807E+00,-0.35732E+00,
     &-0.35656E+00,-0.35581E+00,-0.35507E+00,-0.35432E+00,-0.35357E+00,
     &-0.35282E+00,-0.35207E+00,-0.35132E+00,-0.35057E+00,-0.34983E+00,
     &-0.34908E+00,-0.34833E+00,-0.34759E+00,-0.34684E+00,-0.34610E+00,
     &-0.34535E+00,-0.34461E+00,-0.34386E+00,-0.34312E+00,-0.34237E+00,
     &-0.34163E+00,-0.34089E+00,-0.34015E+00,-0.33940E+00,-0.33866E+00,
     &-0.33792E+00,-0.33718E+00,-0.33644E+00,-0.33570E+00,-0.33496E+00,
     &-0.33422E+00,-0.33348E+00,-0.33275E+00,-0.33201E+00,-0.33127E+00,
     &-0.33053E+00,-0.32980E+00,-0.32906E+00,-0.32833E+00,-0.32759E+00,
     &-0.32686E+00,-0.32612E+00,-0.32539E+00,-0.32465E+00,-0.32392E+00,
     &-0.32319E+00,-0.32246E+00,-0.32172E+00,-0.32099E+00,-0.32026E+00/

      DATA (BNC13M(I),I=301,400)/
     &-0.31953E+00,-0.31880E+00,-0.31807E+00,-0.31734E+00,-0.31662E+00,
     &-0.31589E+00,-0.31516E+00,-0.31443E+00,-0.31371E+00,-0.31298E+00,
     &-0.31225E+00,-0.31153E+00,-0.31080E+00,-0.31008E+00,-0.30936E+00,
     &-0.30863E+00,-0.30791E+00,-0.30719E+00,-0.30646E+00,-0.30574E+00,
     &-0.30502E+00,-0.30430E+00,-0.30358E+00,-0.30286E+00,-0.30214E+00,
     &-0.30142E+00,-0.30070E+00,-0.29999E+00,-0.29927E+00,-0.29855E+00,
     &-0.29784E+00,-0.29712E+00,-0.29640E+00,-0.29569E+00,-0.29497E+00,
     &-0.29426E+00,-0.29355E+00,-0.29283E+00,-0.29212E+00,-0.29141E+00,
     &-0.29070E+00,-0.28999E+00,-0.28927E+00,-0.28856E+00,-0.28785E+00,
     &-0.28714E+00,-0.28644E+00,-0.28573E+00,-0.28502E+00,-0.28431E+00,
     &-0.28360E+00,-0.28290E+00,-0.28219E+00,-0.28149E+00,-0.28078E+00,
     &-0.28008E+00,-0.27937E+00,-0.27867E+00,-0.27796E+00,-0.27726E+00,
     &-0.27656E+00,-0.27586E+00,-0.27516E+00,-0.27445E+00,-0.27375E+00,
     &-0.27305E+00,-0.27235E+00,-0.27165E+00,-0.27096E+00,-0.27026E+00,
     &-0.26956E+00,-0.26886E+00,-0.26817E+00,-0.26747E+00,-0.26677E+00,
     &-0.26608E+00,-0.26538E+00,-0.26469E+00,-0.26399E+00,-0.26330E+00,
     &-0.26261E+00,-0.26192E+00,-0.26122E+00,-0.26053E+00,-0.25984E+00,
     &-0.25915E+00,-0.25846E+00,-0.25777E+00,-0.25708E+00,-0.25639E+00,
     &-0.25570E+00,-0.25501E+00,-0.25433E+00,-0.25364E+00,-0.25295E+00,
     &-0.25227E+00,-0.25158E+00,-0.25090E+00,-0.25021E+00,-0.24953E+00/

      DATA (BNC13M(I),I=401,500)/
     &-0.24884E+00,-0.24816E+00,-0.24748E+00,-0.24679E+00,-0.24611E+00,
     &-0.24543E+00,-0.24475E+00,-0.24407E+00,-0.24339E+00,-0.24271E+00,
     &-0.24203E+00,-0.24135E+00,-0.24067E+00,-0.23999E+00,-0.23931E+00,
     &-0.23864E+00,-0.23796E+00,-0.23728E+00,-0.23661E+00,-0.23593E+00,
     &-0.23526E+00,-0.23458E+00,-0.23391E+00,-0.23323E+00,-0.23256E+00,
     &-0.23189E+00,-0.23122E+00,-0.23054E+00,-0.22987E+00,-0.22920E+00,
     &-0.22853E+00,-0.22786E+00,-0.22719E+00,-0.22652E+00,-0.22585E+00,
     &-0.22519E+00,-0.22452E+00,-0.22385E+00,-0.22318E+00,-0.22252E+00,
     &-0.22185E+00,-0.22118E+00,-0.22052E+00,-0.21985E+00,-0.21919E+00,
     &-0.21853E+00,-0.21786E+00,-0.21720E+00,-0.21654E+00,-0.21587E+00,
     &-0.21521E+00,-0.21455E+00,-0.21389E+00,-0.21323E+00,-0.21257E+00,
     &-0.21191E+00,-0.21125E+00,-0.21059E+00,-0.20993E+00,-0.20927E+00,
     &-0.20862E+00,-0.20796E+00,-0.20730E+00,-0.20665E+00,-0.20599E+00,
     &-0.20534E+00,-0.20468E+00,-0.20403E+00,-0.20337E+00,-0.20272E+00,
     &-0.20206E+00,-0.20141E+00,-0.20076E+00,-0.20011E+00,-0.19945E+00,
     &-0.19880E+00,-0.19815E+00,-0.19750E+00,-0.19685E+00,-0.19620E+00,
     &-0.19555E+00,-0.19490E+00,-0.19426E+00,-0.19361E+00,-0.19296E+00,
     &-0.19231E+00,-0.19167E+00,-0.19102E+00,-0.19037E+00,-0.18973E+00,
     &-0.18908E+00,-0.18844E+00,-0.18779E+00,-0.18715E+00,-0.18651E+00,
     &-0.18586E+00,-0.18522E+00,-0.18458E+00,-0.18393E+00,-0.18329E+00/

      DATA (BNC13M(I),I=501,600)/
     &-0.18265E+00,-0.18201E+00,-0.18137E+00,-0.18073E+00,-0.18009E+00,
     &-0.17945E+00,-0.17881E+00,-0.17817E+00,-0.17754E+00,-0.17690E+00,
     &-0.17626E+00,-0.17563E+00,-0.17499E+00,-0.17435E+00,-0.17372E+00,
     &-0.17308E+00,-0.17245E+00,-0.17181E+00,-0.17118E+00,-0.17054E+00,
     &-0.16991E+00,-0.16928E+00,-0.16865E+00,-0.16801E+00,-0.16738E+00,
     &-0.16675E+00,-0.16612E+00,-0.16549E+00,-0.16486E+00,-0.16423E+00,
     &-0.16360E+00,-0.16297E+00,-0.16234E+00,-0.16171E+00,-0.16108E+00,
     &-0.16046E+00,-0.15983E+00,-0.15920E+00,-0.15858E+00,-0.15795E+00,
     &-0.15732E+00,-0.15670E+00,-0.15607E+00,-0.15545E+00,-0.15483E+00,
     &-0.15420E+00,-0.15358E+00,-0.15295E+00,-0.15233E+00,-0.15171E+00,
     &-0.15109E+00,-0.15047E+00,-0.14984E+00,-0.14922E+00,-0.14860E+00,
     &-0.14798E+00,-0.14736E+00,-0.14674E+00,-0.14612E+00,-0.14551E+00,
     &-0.14489E+00,-0.14427E+00,-0.14365E+00,-0.14303E+00,-0.14242E+00,
     &-0.14180E+00,-0.14119E+00,-0.14057E+00,-0.13995E+00,-0.13934E+00,
     &-0.13872E+00,-0.13811E+00,-0.13750E+00,-0.13688E+00,-0.13627E+00,
     &-0.13566E+00,-0.13504E+00,-0.13443E+00,-0.13382E+00,-0.13321E+00,
     &-0.13260E+00,-0.13198E+00,-0.13137E+00,-0.13076E+00,-0.13015E+00,
     &-0.12954E+00,-0.12894E+00,-0.12833E+00,-0.12772E+00,-0.12711E+00,
     &-0.12650E+00,-0.12590E+00,-0.12529E+00,-0.12468E+00,-0.12408E+00,
     &-0.12347E+00,-0.12286E+00,-0.12226E+00,-0.12165E+00,-0.12105E+00/

      DATA (BNC13M(I),I=601,700)/
     &-0.11454E+00,-0.10855E+00,-0.10259E+00,-0.96662E-01,-0.90767E-01,
     &-0.84905E-01,-0.79073E-01,-0.73271E-01,-0.67500E-01,-0.61758E-01,
     &-0.56045E-01,-0.50360E-01,-0.44704E-01,-0.39075E-01,-0.33474E-01,
     &-0.27900E-01,-0.22352E-01,-0.16830E-01,-0.11333E-01,-0.58624E-02,
     &-0.41621E-03, 0.50053E-02, 0.10403E-01, 0.15776E-01, 0.21127E-01,
     & 0.26454E-01, 0.31758E-01, 0.37040E-01, 0.42299E-01, 0.47537E-01,
     & 0.52753E-01, 0.57949E-01, 0.63123E-01, 0.68276E-01, 0.73409E-01,
     & 0.78522E-01, 0.83616E-01, 0.88690E-01, 0.93744E-01, 0.98779E-01,
     & 0.10380E+00, 0.10879E+00, 0.11377E+00, 0.11874E+00, 0.12368E+00,
     & 0.12861E+00, 0.13352E+00, 0.13841E+00, 0.14328E+00, 0.14814E+00,
     & 0.15298E+00, 0.15781E+00, 0.16262E+00, 0.16741E+00, 0.17219E+00,
     & 0.17695E+00, 0.18170E+00, 0.18643E+00, 0.19115E+00, 0.19585E+00,
     & 0.20054E+00, 0.20521E+00, 0.20987E+00, 0.21452E+00, 0.21915E+00,
     & 0.22376E+00, 0.22837E+00, 0.23296E+00, 0.23753E+00, 0.24210E+00,
     & 0.24665E+00, 0.25119E+00, 0.25571E+00, 0.26022E+00, 0.26472E+00,
     & 0.26921E+00, 0.27369E+00, 0.27815E+00, 0.28260E+00, 0.28704E+00,
     & 0.29147E+00, 0.29588E+00, 0.30029E+00, 0.30468E+00, 0.30907E+00,
     & 0.31344E+00, 0.31780E+00, 0.32215E+00, 0.32649E+00, 0.33081E+00,
     & 0.33513E+00, 0.33944E+00, 0.34374E+00, 0.34802E+00, 0.35230E+00,
     & 0.35656E+00, 0.36082E+00, 0.36507E+00, 0.36930E+00, 0.37353E+00/

      DATA (BNC13M(I),I=701,741)/
     & 0.37775E+00, 0.38196E+00, 0.38616E+00, 0.39035E+00, 0.39453E+00,
     & 0.39870E+00, 0.40286E+00, 0.40701E+00, 0.41116E+00, 0.41529E+00,
     & 0.41942E+00, 0.42354E+00, 0.42764E+00, 0.43175E+00, 0.43584E+00,
     & 0.43992E+00, 0.44400E+00, 0.44807E+00, 0.45213E+00, 0.45618E+00,
     & 0.46022E+00, 0.46426E+00, 0.46828E+00, 0.47230E+00, 0.47632E+00,
     & 0.48032E+00, 0.48432E+00, 0.48831E+00, 0.49229E+00, 0.49626E+00,
     & 0.50023E+00, 0.50419E+00, 0.50814E+00, 0.51209E+00, 0.51603E+00,
     & 0.51996E+00, 0.52388E+00, 0.52780E+00, 0.53171E+00, 0.53562E+00,
     & 0.53951E+00
     & /
      END
CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE CHRBLN
CC  Purpose        : Position of last non-blank character in a string
CC  Author         : Athanasios Nenes
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  STR        is the CHARACTER variable containing the string examined
CC  IBLK       is a INTEGER variable containing the position of last non 
CC             blank character. If string is all spaces (ie '   '), then
CC             the value returned is 1.
CC
CC  EXAMPLE:
CC             STR = 'TEST1.DAT     '
CC             CALL CHRBLN (STR, IBLK)
CC          
CC  after execution of this code segment, "IBLK" has the value "9", which
CC  is the position of the last non-blank character of "STR".
CC
CC***********************************************************************
CC
      SUBROUTINE CHRBLN (STR, IBLK)
CC
CC***********************************************************************
      CHARACTER*(*) STR
C
      IBLK = 1                       ! Substring pointer (default=1)
      ILEN = LEN(STR)                ! Length of string 
      DO 10 i=ILEN,1,-1              
         IF (STR(i:i).NE.' ' .AND. STR(i:i).NE.CHAR(0)) THEN
            IBLK = i
            RETURN
         ENDIF
10    CONTINUE
      RETURN
C
      END




CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE SHFTRGHT
CC  Purpose        : RIGHT-JUSTIFICATION FUNCTION ON A STRING
CC  Author         : Athanasios Nenes
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  STRING     is the CHARACTER variable with the string to be justified
CC
CC  EXAMPLE:
CC             STRING    = 'AAAA    '
CC             CALL SHFTRGHT (STRING)
CC          
CC  after execution of this code segment, STRING contains the value
CC  '    AAAA'.
CC
CC*************************************************************************
CC
      SUBROUTINE SHFTRGHT (CHR)
CC
CC***********************************************************************
      CHARACTER CHR*(*)
C
      I1  = LEN(CHR)             ! Total length of string
      CALL CHRBLN(CHR,I2)        ! Position of last non-blank character
      IF (I2.EQ.I1) RETURN
C
      DO 10 I=I2,1,-1            ! Shift characters
         CHR(I1+I-I2:I1+I-I2) = CHR(I:I)
         CHR(I:I) = ' '
10    CONTINUE
      RETURN
C
      END




CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE RPLSTR
CC  Purpose        : REPLACE CHARACTERS OCCURING IN A STRING
CC  Author         : Athanasios Nenes
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  STRING     is the CHARACTER variable with the string to be edited
CC  OLD        is the old character which is to be replaced
CC  NEW        is the new character which OLD is to be replaced with
CC  IERR       is 0 if everything went well, is 1 if 'NEW' contains 'OLD'.
CC             In this case, this is invalid, and no change is done.
CC
CC  EXAMPLE:
CC             STRING    = 'AAAA'
CC             OLD       = 'A'
CC             NEW       = 'B' 
CC             CALL RPLSTR (STRING, OLD, NEW)
CC          
CC  after execution of this code segment, STRING contains the value
CC  'BBBB'.
CC
CC*************************************************************************
CC
      SUBROUTINE RPLSTR (STRING, OLD, NEW, IERR)
CC
CC***********************************************************************
      CHARACTER STRING*(*), OLD*(*), NEW*(*)
C
C *** INITIALIZE ********************************************************
C
      ILO = LEN(OLD)
C
C *** CHECK AND SEE IF 'NEW' CONTAINS 'OLD', WHICH CANNOT ***************
C      
      IP = INDEX(NEW,OLD)
      IF (IP.NE.0) THEN
         IERR = 1
         RETURN
      ELSE
         IERR = 0
      ENDIF
C
C *** PROCEED WITH REPLACING *******************************************
C      
10    IP = INDEX(STRING,OLD)      ! SEE IF 'OLD' EXISTS IN 'STRING'
      IF (IP.EQ.0) RETURN         ! 'OLD' DOES NOT EXIST ; RETURN
      STRING(IP:IP+ILO-1) = NEW   ! REPLACE SUBSTRING 'OLD' WITH 'NEW'
      GOTO 10                     ! GO FOR NEW OCCURANCE OF 'OLD'
C
      END
        

CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE INPTD
CC  Purpose        : Prompts user for a value (DOUBLE). A default value
CC                   is provided, so if user presses <Enter>, the default
CC                   is used. 
CC  Author         : Athanasios Nenes
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  VAR        is the DOUBLE PRECISION variable which value is to be saved 
CC  DEF        is a DOUBLE PRECISION variable, with the default value of VAR.        
CC  PROMPT     is a CHARACTER varible containing the prompt string.     
CC  PRFMT      is a CHARACTER variable containing the FORMAT specifier
CC             for the default value DEF.
CC  IERR       is an INTEGER error flag, and has the values:
CC             0 - No error detected.
CC             1 - Invalid FORMAT and/or Invalid default value.
CC             2 - Bad value specified by user
CC
CC  EXAMPLE:
CC             CALL INPTD (VAR, 1.0D0, 'Give value for A ', '*', Ierr)
CC          
CC  after execution of this code segment, the user is prompted for the
CC  value of variable VAR. If <Enter> is pressed (ie no value is specified)
CC  then 1.0 is assigned to VAR. The default value is displayed in free-
CC  format. The error status is specified by variable Ierr
CC
CC***********************************************************************
CC
      SUBROUTINE INPTD (VAR, DEF, PROMPT, PRFMT, IERR)
CC
CC***********************************************************************
      CHARACTER PROMPT*(*), PRFMT*(*), BUFFER*128
      DOUBLE PRECISION DEF, VAR
      INTEGER IERR
C
      IERR = 0
C
C *** WRITE DEFAULT VALUE TO WORK BUFFER *******************************
C
      WRITE (BUFFER, FMT=PRFMT, ERR=10) DEF
      CALL CHRBLN (BUFFER, IEND)
C
C *** PROMPT USER FOR INPUT AND READ IT ********************************
C
      WRITE (*,*) PROMPT,' [',BUFFER(1:IEND),']: '
      READ  (*, '(A)', ERR=20, END=20) BUFFER
      CALL CHRBLN (BUFFER,IEND)
C
C *** READ DATA OR SET DEFAULT ? ****************************************
C
      IF (IEND.EQ.1 .AND. BUFFER(1:1).EQ.' ') THEN
         VAR = DEF
      ELSE
         READ (BUFFER, *, ERR=20, END=20) VAR
      ENDIF
C
C *** RETURN POINT ******************************************************
C
30    RETURN
C
C *** ERROR HANDLER *****************************************************
C
10    IERR = 1       ! Bad FORMAT and/or bad default value
      GOTO 30
C
20    IERR = 2       ! Bad number given by user
      GOTO 30
C
      END


CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE Pushend 
CC  Purpose        : Positions the pointer of a sequential file at its end
CC                   Simulates the ACCESS='APPEND' clause of a F77L OPEN
CC                   statement with Standard Fortran commands.
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  Iunit      is a INTEGER variable, the file unit which the file is 
CC             connected to.
CC
CC  EXAMPLE:
CC             CALL PUSHEND (10)
CC          
CC  after execution of this code segment, the pointer of unit 10 is 
CC  pushed to its end.
CC
CC***********************************************************************
CC
      SUBROUTINE Pushend (Iunit)
CC
CC***********************************************************************
C
      LOGICAL OPNED
C
C *** INQUIRE IF Iunit CONNECTED TO FILE ********************************
C
      INQUIRE (UNIT=Iunit, OPENED=OPNED)
      IF (.NOT.OPNED) GOTO 25
C
C *** Iunit CONNECTED, PUSH POINTER TO END ******************************
C
10    READ (Iunit,'()', ERR=20, END=20)
      GOTO 10
C
C *** RETURN POINT ******************************************************
C
20    BACKSPACE (Iunit)
25    RETURN
      END



CC*************************************************************************
CC
CC  TOOLBOX LIBRARY v.1.0 (May 1995)
CC
CC  Program unit   : SUBROUTINE APPENDEXT
CC  Purpose        : Fix extension in file name string
CC
CC  ======================= ARGUMENTS / USAGE =============================
CC
CC  Filename   is the CHARACTER variable with the file name
CC  Defext     is the CHARACTER variable with extension (including '.',
CC             ex. '.DAT')
CC  Overwrite  is a LOGICAL value, .TRUE. overwrites any existing extension
CC             in "Filename" with "Defext", .FALSE. puts "Defext" only if 
CC             there is no extension in "Filename".
CC
CC  EXAMPLE:
CC             FILENAME1 = 'TEST.DAT'
CC             FILENAME2 = 'TEST.DAT'
CC             CALL APPENDEXT (FILENAME1, '.TXT', .FALSE.)
CC             CALL APPENDEXT (FILENAME2, '.TXT', .TRUE. )
CC          
CC  after execution of this code segment, "FILENAME1" has the value 
CC  'TEST.DAT', while "FILENAME2" has the value 'TEST.TXT'
CC
CC***********************************************************************
CC
      SUBROUTINE Appendext (Filename, Defext, Overwrite)
CC
CC***********************************************************************
      CHARACTER*(*) Filename, Defext
      LOGICAL       Overwrite
C
      CALL CHRBLN (Filename, Iend)
      IF (Filename(1:1).EQ.' ' .AND. Iend.EQ.1) RETURN  ! Filename empty
      Idot = INDEX (Filename, '.')                      ! Append extension ?
      IF (Idot.EQ.0) Filename = Filename(1:Iend)//Defext
      IF (Overwrite .AND. Idot.NE.0)
     &              Filename = Filename(:Idot-1)//Defext
      RETURN
      END





C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE POLY3
C *** FINDS THE REAL ROOTS OF THE THIRD ORDER ALGEBRAIC EQUATION:
C     X**3 + A1*X**2 + A2*X + A3 = 0.0
C     THE EQUATION IS SOLVED ANALYTICALLY.
C
C     PARAMETERS A1, A2, A3 ARE SPECIFIED BY THE USER. THE MINIMUM
C     NONEGATIVE ROOT IS RETURNED IN VARIABLE 'ROOT'. IF NO ROOT IS 
C     FOUND (WHICH IS GREATER THAN ZERO), ROOT HAS THE VALUE 1D30.
C     AND THE FLAG ISLV HAS A VALUE GREATER THAN ZERO.
C
C     SOLUTION FORMULA IS FOUND IN PAGE 32 OF:
C     MATHEMATICAL HANDBOOK OF FORMULAS AND TABLES
C     SCHAUM'S OUTLINE SERIES
C     MURRAY SPIEGER, McGRAW-HILL, NEW YORK, 1968
C     (GREEK TRANSLATION: BY SOTIRIOS PERSIDES, ESPI, ATHENS, 1976)
C
C     A SPECIAL CASE IS CONSIDERED SEPERATELY ; WHEN A3 = 0, THEN
C     ONE ROOT IS X=0.0, AND THE OTHER TWO FROM THE SOLUTION OF THE
C     QUADRATIC EQUATION X**2 + A1*X + A2 = 0.0
C     THIS SPECIAL CASE IS CONSIDERED BECAUSE THE ANALYTICAL FORMULA 
C     DOES NOT YIELD ACCURATE RESULTS (DUE TO NUMERICAL ROUNDOFF ERRORS)
C
C *** COPYRIGHT 1996-98, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      SUBROUTINE POLY3 (A1, A2, A3, ROOT, ISLV)
C
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      PARAMETER (EXPON=1.D0/3.D0,     ZERO=0.D0, THET1=120.D0/180.D0, 
     &           THET2=240.D0/180.D0, PI=3.14159265358932, EPS=1D-50)
      DOUBLE PRECISION  X(3)
C
C *** SPECIAL CASE : QUADRATIC*X EQUATION *****************************
C
      IF (ABS(A3).LE.EPS) THEN 
         ISLV = 1
         IX   = 1
         X(1) = ZERO
         D    = A1*A1-4.D0*A2
         IF (D.GE.ZERO) THEN
            IX   = 3
            SQD  = SQRT(D)
            X(2) = 0.5*(-A1+SQD)
            X(3) = 0.5*(-A1-SQD)
         ENDIF
      ELSE
C
C *** NORMAL CASE : CUBIC EQUATION ************************************
C
C DEFINE PARAMETERS Q, R, S, T, D 
C
         ISLV= 1
         Q   = (3.D0*A2 - A1*A1)/9.D0
         R   = (9.D0*A1*A2 - 27.D0*A3 - 2.D0*A1*A1*A1)/54.D0
         D   = Q*Q*Q + R*R
C
C *** CALCULATE ROOTS *************************************************
C
C  D < 0, THREE REAL ROOTS
C
         IF (D.LT.-EPS) THEN        ! D < -EPS  : D < ZERO
            IX   = 3
            THET = EXPON*ACOS(R/SQRT(-Q*Q*Q))
            COEF = 2.D0*SQRT(-Q)
            X(1) = COEF*COS(THET)            - EXPON*A1
            X(2) = COEF*COS(THET + THET1*PI) - EXPON*A1
            X(3) = COEF*COS(THET + THET2*PI) - EXPON*A1
C
C  D = 0, THREE REAL (ONE DOUBLE) ROOTS
C
         ELSE IF (D.LE.EPS) THEN    ! -EPS <= D <= EPS  : D = ZERO
            IX   = 2
            SSIG = SIGN (1.D0, R)
            S    = SSIG*(ABS(R))**EXPON
            X(1) = 2.D0*S  - EXPON*A1
            X(2) =     -S  - EXPON*A1
C
C  D > 0, ONE REAL ROOT
C
         ELSE                       ! D > EPS  : D > ZERO
            IX   = 1
            SQD  = SQRT(D)
            SSIG = SIGN (1.D0, R+SQD)       ! TRANSFER SIGN TO SSIG
            TSIG = SIGN (1.D0, R-SQD)
            S    = SSIG*(ABS(R+SQD))**EXPON ! EXPONENTIATE ABS() 
            T    = TSIG*(ABS(R-SQD))**EXPON
            X(1) = S + T - EXPON*A1
         ENDIF
      ENDIF
C
C *** SELECT APPROPRIATE ROOT *****************************************
C
      ROOT = 1.D30
      DO 10 I=1,IX
         IF (X(I).GT.ZERO) THEN
            ROOT = MIN (ROOT, X(I))
            ISLV = 0
         ENDIF
10    CONTINUE
C
C *** END OF SUBROUTINE POLY3 *****************************************
C
      RETURN
      END




C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE POLY3B
C *** FINDS A REAL ROOT OF THE THIRD ORDER ALGEBRAIC EQUATION:
C     X**3 + A1*X**2 + A2*X + A3 = 0.0
C     THE EQUATION IS SOLVED NUMERICALLY (BISECTION).
C
C     PARAMETERS A1, A2, A3 ARE SPECIFIED BY THE USER. THE MINIMUM
C     NONEGATIVE ROOT IS RETURNED IN VARIABLE 'ROOT'. IF NO ROOT IS 
C     FOUND (WHICH IS GREATER THAN ZERO), ROOT HAS THE VALUE 1D30.
C     AND THE FLAG ISLV HAS A VALUE GREATER THAN ZERO.
C
C     RTLW, RTHI DEFINE THE INTERVAL WHICH THE ROOT IS LOOKED FOR.
C
C=======================================================================
C
      SUBROUTINE POLY3B (A1, A2, A3, RTLW, RTHI, ROOT, ISLV)
C
      IMPLICIT DOUBLE PRECISION (A-H, O-Z)
      PARAMETER (ZERO=0.D0, EPS=1D-15, MAXIT=100, NDIV=5)
C
      FUNC(X) = X**3.d0 + A1*X**2.0 + A2*X + A3
C
C *** INITIAL VALUES FOR BISECTION *************************************
C
      X1   = RTLW
      Y1   = FUNC(X1)
      IF (ABS(Y1).LE.EPS) THEN     ! Is low a root?
         ROOT = RTLW
         GOTO 50
      ENDIF
C
C *** ROOT TRACKING ; FOR THE RANGE OF HI AND LO ***********************
C
      DX = (RTHI-RTLW)/FLOAT(NDIV)
      DO 10 I=1,NDIV
         X2 = X1+DX
         Y2 = FUNC (X2)
         IF (SIGN(1.d0,Y1)*SIGN(1.d0,Y2) .LT. ZERO) GOTO 20 ! (Y1*Y2.LT.ZERO)
         X1 = X2
         Y1 = Y2
10    CONTINUE
C
C *** NO SUBDIVISION WITH SOLUTION FOUND 
C
      IF (ABS(Y2) .LT. EPS) THEN   ! X2 is a root
         ROOT = X2
      ELSE
         ROOT = 1.d30
         ISLV = 1
      ENDIF
      GOTO 50
C
C *** BISECTION *******************************************************
C
20    DO 30 I=1,MAXIT
         X3 = 0.5*(X1+X2)
         Y3 = FUNC (X3)
         IF (SIGN(1.d0,Y1)*SIGN(1.d0,Y3) .LE. ZERO) THEN  ! (Y1*Y3 .LE. ZERO)
            Y2    = Y3
            X2    = X3
         ELSE
            Y1    = Y3
            X1    = X3
         ENDIF
         IF (ABS(X2-X1) .LE. EPS*X1) GOTO 40
30    CONTINUE
C
C *** CONVERGED ; RETURN ***********************************************
C
40    X3   = 0.5*(X1+X2)
      Y3   = FUNC (X3)
      ROOT = X3
      ISLV = 0
C
50    RETURN
C
C *** END OF SUBROUTINE POLY3B *****************************************
C
      END
      


ccc      PROGRAM DRIVER
ccc      DOUBLE PRECISION ROOT
cccC
ccc      CALL POLY3 (-1.d0, 1.d0, -1.d0, ROOT, ISLV)
ccc      IF (ISLV.NE.0) STOP 'Error in POLY3'
ccc      WRITE (*,*) 'Root=', ROOT
cccC
ccc      CALL POLY3B (-1.d0, 1.d0, -1.d0, -10.d0, 10.d0, ROOT, ISLV)
ccc      IF (ISLV.NE.0) STOP 'Error in POLY3B'
ccc      WRITE (*,*) 'Root=', ROOT
cccC
ccc      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** FUNCTION EX10
C *** 10^X FUNCTION ; ALTERNATE OF LIBRARY ROUTINE ; USED BECAUSE IT IS
C     MUCH FASTER BUT WITHOUT GREAT LOSS IN ACCURACY. , 
C     MAXIMUM ERROR IS 2%, EXECUTION TIME IS 42% OF THE LIBRARY ROUTINE 
C     (ON A 80286/80287 MACHINE, using Lahey FORTRAN 77 v.3.0).
C
C     EXPONENT RANGE IS BETWEEN -K AND K (K IS THE REAL ARGUMENT 'K')
C     MAX VALUE FOR K: 9.999
C     IF X < -K, X IS SET TO -K, IF X > K, X IS SET TO K
C
C     THE EXPONENT IS CALCULATED BY THE PRODUCT ADEC*AINT, WHERE ADEC
C     IS THE MANTISSA AND AINT IS THE MAGNITUDE (EXPONENT). BOTH 
C     MANTISSA AND MAGNITUDE ARE PRE-CALCULATED AND STORED IN LOOKUP
C     TABLES ; THIS LEADS TO THE INCREASED SPEED.
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      FUNCTION EX10(X,K)
      REAL    X, EX10, Y, AINT10, ADEC10, K
      INTEGER K1, K2
      COMMON /EXPNC/ AINT10(20), ADEC10(200)
C
C *** LIMIT X TO [-K, K] RANGE *****************************************
C
      Y    = MAX(-K, MIN(X,K))   ! MIN: -9.999, MAX: 9.999
C
C *** GET INTEGER AND DECIMAL PART *************************************
C
      K1   = INT(Y)
      K2   = INT(100*(Y-K1))
C
C *** CALCULATE EXP FUNCTION *******************************************
C
      EX10 = AINT10(K1+10)*ADEC10(K2+100)
C
C *** END OF EXP FUNCTION **********************************************
C
      RETURN
      END


C=======================================================================
C
C *** ISORROPIA CODE
C *** BLOCK DATA EXPON
C *** CONTAINS DATA FOR EXPONENT ARRAYS NEEDED IN FUNCTION EXP10
C
C *** COPYRIGHT 1996-2000, UNIVERSITY OF MIAMI, CARNEGIE MELLON UNIVERSITY
C *** WRITTEN BY ATHANASIOS NENES
C
C=======================================================================
C
      BLOCK DATA EXPON
C
C *** Common block definition
C
      REAL AINT10, ADEC10
      COMMON /EXPNC/ AINT10(20), ADEC10(200)
C
C *** Integer part        
C
      DATA AINT10/
     & 0.1000E-08, 0.1000E-07, 0.1000E-06, 0.1000E-05, 0.1000E-04,
     & 0.1000E-03, 0.1000E-02, 0.1000E-01, 0.1000E+00, 0.1000E+01,
     & 0.1000E+02, 0.1000E+03, 0.1000E+04, 0.1000E+05, 0.1000E+06,
     & 0.1000E+07, 0.1000E+08, 0.1000E+09, 0.1000E+10, 0.1000E+11
     & /
C
C *** decimal part        
C
      DATA (ADEC10(I),I=1,100)/
     & 0.1023E+00, 0.1047E+00, 0.1072E+00, 0.1096E+00, 0.1122E+00,
     & 0.1148E+00, 0.1175E+00, 0.1202E+00, 0.1230E+00, 0.1259E+00,
     & 0.1288E+00, 0.1318E+00, 0.1349E+00, 0.1380E+00, 0.1413E+00,
     & 0.1445E+00, 0.1479E+00, 0.1514E+00, 0.1549E+00, 0.1585E+00,
     & 0.1622E+00, 0.1660E+00, 0.1698E+00, 0.1738E+00, 0.1778E+00,
     & 0.1820E+00, 0.1862E+00, 0.1905E+00, 0.1950E+00, 0.1995E+00,
     & 0.2042E+00, 0.2089E+00, 0.2138E+00, 0.2188E+00, 0.2239E+00,
     & 0.2291E+00, 0.2344E+00, 0.2399E+00, 0.2455E+00, 0.2512E+00,
     & 0.2570E+00, 0.2630E+00, 0.2692E+00, 0.2754E+00, 0.2818E+00,
     & 0.2884E+00, 0.2951E+00, 0.3020E+00, 0.3090E+00, 0.3162E+00,
     & 0.3236E+00, 0.3311E+00, 0.3388E+00, 0.3467E+00, 0.3548E+00,
     & 0.3631E+00, 0.3715E+00, 0.3802E+00, 0.3890E+00, 0.3981E+00,
     & 0.4074E+00, 0.4169E+00, 0.4266E+00, 0.4365E+00, 0.4467E+00,
     & 0.4571E+00, 0.4677E+00, 0.4786E+00, 0.4898E+00, 0.5012E+00,
     & 0.5129E+00, 0.5248E+00, 0.5370E+00, 0.5495E+00, 0.5623E+00,
     & 0.5754E+00, 0.5888E+00, 0.6026E+00, 0.6166E+00, 0.6310E+00,
     & 0.6457E+00, 0.6607E+00, 0.6761E+00, 0.6918E+00, 0.7079E+00,
     & 0.7244E+00, 0.7413E+00, 0.7586E+00, 0.7762E+00, 0.7943E+00,
     & 0.8128E+00, 0.8318E+00, 0.8511E+00, 0.8710E+00, 0.8913E+00,
     & 0.9120E+00, 0.9333E+00, 0.9550E+00, 0.9772E+00, 0.1000E+01/

      DATA (ADEC10(I),I=101,200)/
     & 0.1023E+01, 0.1047E+01, 0.1072E+01, 0.1096E+01, 0.1122E+01,
     & 0.1148E+01, 0.1175E+01, 0.1202E+01, 0.1230E+01, 0.1259E+01,
     & 0.1288E+01, 0.1318E+01, 0.1349E+01, 0.1380E+01, 0.1413E+01,
     & 0.1445E+01, 0.1479E+01, 0.1514E+01, 0.1549E+01, 0.1585E+01,
     & 0.1622E+01, 0.1660E+01, 0.1698E+01, 0.1738E+01, 0.1778E+01,
     & 0.1820E+01, 0.1862E+01, 0.1905E+01, 0.1950E+01, 0.1995E+01,
     & 0.2042E+01, 0.2089E+01, 0.2138E+01, 0.2188E+01, 0.2239E+01,
     & 0.2291E+01, 0.2344E+01, 0.2399E+01, 0.2455E+01, 0.2512E+01,
     & 0.2570E+01, 0.2630E+01, 0.2692E+01, 0.2754E+01, 0.2818E+01,
     & 0.2884E+01, 0.2951E+01, 0.3020E+01, 0.3090E+01, 0.3162E+01,
     & 0.3236E+01, 0.3311E+01, 0.3388E+01, 0.3467E+01, 0.3548E+01,
     & 0.3631E+01, 0.3715E+01, 0.3802E+01, 0.3890E+01, 0.3981E+01,
     & 0.4074E+01, 0.4169E+01, 0.4266E+01, 0.4365E+01, 0.4467E+01,
     & 0.4571E+01, 0.4677E+01, 0.4786E+01, 0.4898E+01, 0.5012E+01,
     & 0.5129E+01, 0.5248E+01, 0.5370E+01, 0.5495E+01, 0.5623E+01,
     & 0.5754E+01, 0.5888E+01, 0.6026E+01, 0.6166E+01, 0.6310E+01,
     & 0.6457E+01, 0.6607E+01, 0.6761E+01, 0.6918E+01, 0.7079E+01,
     & 0.7244E+01, 0.7413E+01, 0.7586E+01, 0.7762E+01, 0.7943E+01,
     & 0.8128E+01, 0.8318E+01, 0.8511E+01, 0.8710E+01, 0.8913E+01,
     & 0.9120E+01, 0.9333E+01, 0.9550E+01, 0.9772E+01, 0.1000E+02
     & /
C
C *** END OF BLOCK DATA EXPON ******************************************
C
      END
C=======================================================================
C
C *** ISORROPIA CODE
C *** SUBROUTINE ISOPLUS
C *** THIS SUBROUTINE IS THE MASTER ROUTINE FOR THE ISORROPIA-PLUS
C     THERMODYNAMIC EQUILIBRIUM AEROSOL MODEL (VERSION 1.0)
C    
C *** NOTE: THIS SUBROUTINE IS INCLUDED FOR BACKWARD COMPATABILITY ONLY.
C     A PROGRAMMER SHOULD USE THE MORE COMPLETE SUBROUTINE ISOROPIA INSTEAD
C
C ======================== ARGUMENTS / USAGE ===========================
C
C  INPUT:
C  1. [WI] 
C     DOUBLE PRECISION array of length [5].
C     Concentrations, expressed in moles/m3. Depending on the type of
C     problem solved, WI contains either GAS+AEROSOL or AEROSOL only 
C     concentratios.
C     WI(1) - sodium
C     WI(2) - sulfate
C     WI(3) - ammonium
C     WI(4) - nitrate
C     WI(5) - chloride
C
C  2. [RHI] 
C     DOUBLE PRECISION variable.  
C     Ambient relative humidity expressed in a (0,1) scale.
C
C  3. [TEMPI]
C     DOUBLE PRECISION variable. 
C     Ambient temperature expressed in Kelvins. 
C
C  4. [IPROB]
C     INTEGER variable.
C     The type of problem solved.
C     IPROB = 0  - Forward problem is solved. In this case, array WI
C                  contains GAS and AEROSOL concentrations together.
C     IPROB = 1  - Reverse problem is solved. In this case, array WI
C                  contains AEROSOL concentrations only.
C
C  OUTPUT:
C  1. [GAS]
C     DOUBLE PRECISION array of length [03]. 
C     Gaseous species concentrations, expressed in moles/m3. 
C     GAS(1) - NH3
C     GAS(2) - HNO3
C     GAS(3) - HCl 
C
C  2. [AERLIQ]
C     DOUBLE PRECISION array of length [11]. 
C     Liquid aerosol species concentrations, expressed in moles/m3. 
C     AERLIQ(01) - H+(aq)          
C     AERLIQ(02) - Na+(aq)         
C     AERLIQ(03) - NH4+(aq)
C     AERLIQ(04) - Cl-(aq)         
C     AERLIQ(05) - SO4--(aq)       
C     AERLIQ(06) - HSO4-(aq)       
C     AERLIQ(07) - NO3-(aq)        
C     AERLIQ(08) - H2O             
C     AERLIQ(09) - NH3(aq) (undissociated)
C     AERLIQ(10) - HNCl(aq) (undissociated)
C     AERLIQ(11) - HNO3(aq) (undissociated)
C
C  3. [AERSLD]
C     DOUBLE PRECISION array of length [09]. 
C     Solid aerosol species concentrations, expressed in moles/m3. 
C     AERSLD(01) - NaNO3(s)
C     AERSLD(02) - NH4NO3(s)
C     AERSLD(03) - NaCl(s)         
C     AERSLD(04) - NH4Cl(s)
C     AERSLD(05) - Na2SO4(s)       
C     AERSLD(06) - (NH4)2SO4(s)
C     AERSLD(07) - NaHSO4(s)
C     AERSLD(08) - NH4HSO4(s)
C     AERSLD(09) - (NH4)4H(SO4)2(s)
C
C  4. [DRY]
C     LOGICAL v