using System;
using System.Drawing;
using System.Windows.Forms;
using SMAT_CE;
using System.Collections.Generic;

namespace WinControls
{
    //Define a delegate
    public delegate void ValueRangeChangedHanler(SetValueRange frm);

    public partial class SetValueRange : FormBase
    {
        //Define an event
        public event ValueRangeChangedHanler ValueRangeChanged;

        //Define the handling method of the event
        protected void OnValueRangeChanged(SetValueRange frm)
        {
            if (ValueRangeChanged != null)
            {
                ValueRangeChanged(frm);
            }
        }

        private double _minValue;

        /// <summary>
        ///Minimum value
        /// </summary>
        public double MinValue
        {
            get { return _minValue; }
            set
            {
                _minValue = value;
                txtMin.Text = Math.Round(_minValue,4).ToString();
            }
        }

        private double _maxValue;

        /// <summary>
        ///Maximum value
        /// </summary>
        public double MaxValue
        {
            get { return _maxValue; }
            set
            {
                _maxValue = value;
                txtMax.Text = Math.Round(_maxValue,4).ToString();
            }
        }

        //private string _unit;
        /// <summary>
        ///Unit
        /// </summary>
        public string Unit
        {
            //get { return _unit; }
            set
            {
                //_unit = value;
                lblUnit1.Text = value;
                lblUnit2.Text = value;
            }
        }

        //Define the number of boundary points
        private int count = 7;

        //Legend's color block
        Color[] _colorArray = new Color[7];

        //Users can define color patches according to their own needs
        public Color[] ColorArray
        {
            get { return _colorArray; }
            set
            {
                _colorArray = value;
            }
        }

        //Start value of each color block in legend
        private double[] _valueArray = new double[7];

        public double[] ValueArray
        {
            get { return _valueArray; }
            set { _valueArray = value; }
        }

        public SetValueRange()
        {
            InitializeComponent();

            #region added by Edwin

            if (LanguageOld.CurrentLanguage() == "Ch")
            {
                //Language.GetLanguagePath();
                Dictionary<string, string> dic = new Dictionary<string, string>();
                LanguageOld.SetDicFromXml(this, dic);
                LanguageOld.ChangeLanguage(dic, this);
            }

            #endregion

        }

        private void btnOK_Click(object sender, EventArgs e)
        {
            double min = 0;
            double max = 0;
            bool ok = false;
            string tmp = "";
            string msg = "";
            try
            {
                ok = GetMinMaxValues(ref min, ref max);
                if (!ok) { return; }
                _minValue = min;
                _maxValue = max;

                for (int i = 0; i < count; i++)
                {
                    _colorArray[i] = dgvSetColor.Rows[0].Cells[i].Style.BackColor;
                    // _valueArray[i] = double.Parse(dgvSetColor.Rows[1].Cells[i].Value.ToString());
                    tmp = dgvSetColor.Rows[1].Cells[i].Value.ToString();
                    ok = double.TryParse(tmp, out _valueArray[i]);
                    if (!ok)
                    {
                        msg = string.Format("'{0}' is an invalid boundary value.", tmp);
                        MessageBox.Show(msg, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return;
                    }
                }
                //Verify whether the value of the boundary point group is in the order from small to large.If not, it will prompt the setting error
                ok = VerifyBoundaryValue();
                if (!ok) { return; }
                this.DialogResult = DialogResult.OK;
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        //Verify the effectiveness of boundary points
        private bool VerifyBoundaryValue()
        {
            double tmpPre = 0;
            double tmpCurrent = 0;
            bool ok = false;
            string msg = "";
            try
            {
                for (int i = 1; i < count; i++)
                {
                    tmpPre = _valueArray[i - 1];
                    tmpCurrent = _valueArray[i];
                    if (tmpPre > tmpCurrent)
                    {
                        msg = "Boundary Values are invalid.";
                        MessageBox.Show(msg, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return ok;
                    }
                }
                ok = true;
                return ok;
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
                return ok;
            }
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
        }

        private void btnReset_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Ignore;
        }

        private void SetValueRange_Load(object sender, EventArgs e)
        {
            try
            {
                //InitStartValue();
                // DefaultValueArray(_minValue, _maxValue);
                InitdgColorSet();
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        //Initialize the value of legend
        private void SetDefaultValueArray(double minValue, double maxValue)
        {
            try
            {
                //Divide the color block into 4 equal parts, with 5 boundary values
                //int count = _startValue.Length;

                //Begin this is different from the calculation in colorblendcontrol.At present, colorblendcontrol prevails
                double var = (maxValue - minValue) / (count-1) ;
                //double var = (maxValue - minValue) / ( count - 1 ) ;
                //End            modified by Edwin 20131219
 
                for (int i = 0; i < count; i++)
                {
                    _valueArray[i] = _minValue + i * var;
                    _valueArray[i] = Math.Round(_valueArray[i], 4);
                }
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        //Initialize setting color and boundary values
        private void InitdgColorSet()
        {
            try
            {
                //Add columns for DataGridView
                DataGridViewTextBoxColumn newColColor =
                            new DataGridViewTextBoxColumn();
                // newColColor.Width = 60;
                for (int i = 0; i < count; i++)
                {
                    newColColor = new DataGridViewTextBoxColumn();
                    newColColor.Width = 55;
                    dgvSetColor.Columns.Add(newColColor);
                }
                //dgvSetColor.ColumnCount = 6;
                DataGridViewRow newRowColor = new DataGridViewRow();
                //Set the prompt when the mouse moves to the line mark
                newRowColor.HeaderCell.ToolTipText = "Color";
                //Set row title
                newRowColor.HeaderCell.Value = "Color";
                //Set row Title width
                // newRowColor.HeaderCell.Size.Width = 100;
                //Set row height
                newRowColor.Height = 38;
                //newRowColor.HeaderCell.Style.ApplyStyle(newRowColor.HeaderCell.Style.Alignment);

                DataGridViewRow newRowValueArray = new DataGridViewRow();
                //Set the prompt when the mouse moves to the line mark
                newRowValueArray.HeaderCell.ToolTipText = "Boundary Values";
                //Set row title
                newRowValueArray.HeaderCell.Value = "Boundary Values";
                //Set row height
                newRowValueArray.Height = 39;
                //newRowStartValue
                dgvSetColor.AutoSize = false;
                // Or this:
                DataGridViewTextBoxCell newCell = new DataGridViewTextBoxCell();
                for (int i = 0; i < count; i++)
                {
                    newCell = new DataGridViewTextBoxCell();
                    newRowColor.Cells.Add(newCell);
                    //The Newcell must be added before it can be set
                    newCell.Style.BackColor = _colorArray[i];
                    newCell.ToolTipText = "Double click to customize the color.";

                    //Cells cannot be set to read-only property before being added
                    newCell.ReadOnly = true;

                    newCell = new DataGridViewTextBoxCell();
                    newRowValueArray.Cells.Add(newCell);
                    //newRowColor.Cells[i].Style.BackColor =
                    //newCell.Resizable = true;
                    newCell.Value = _valueArray[i];
                    //newCell.Size.Width = 25;
                    //newCell.Style.BackColor = colorVarray[i];
                    newCell.ToolTipText = "Boundary Values";
                    //Cells cannot be set to read-only property before being added
                    //newCell.ReadOnly = true;
                }
                //newRowColor.ReadOnly = true;
                newRowColor.ReadOnly = false;
                dgvSetColor.Rows.Add(newRowColor);
                dgvSetColor.Rows.Add(newRowValueArray);
                dgvSetColor.Rows[0].Cells[0].Selected = false;
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        private void dgvSetColor_CellDoubleClick(object sender, DataGridViewCellEventArgs e)
        {
            try
            {
                if (e.RowIndex == 0)
                {
                    ColorDialog colorDialog = new ColorDialog();
                    DialogResult rtn = colorDialog.ShowDialog();
                    if (rtn == DialogResult.OK)
                    {
                        foreach (DataGridViewCell cell in dgvSetColor.SelectedCells)
                        {
                            //Change the color of the selected box
                            if (cell.Selected == true)
                            {
                                cell.Style.BackColor = colorDialog.Color;
                                cell.Selected = false;
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        /// <summary>
        ///Set the boundary value according to the minimum and maximum values entered in the interface
        /// </summary>
        private void btnRebuild_Click(object sender, EventArgs e)
        {
            bool ok = false;
            try
            {
                //Verify whether the input maximum and minimum values meet the requirements
                double min = 0;
                double max = 0;
                ok = GetMinMaxValues(ref  min, ref max);
                if (!ok) { return; }
                //Set according to the outgoing minimum and maximum values
                _minValue = min;
                _maxValue = max;
                SetDefaultValueArray(_minValue, _maxValue);
                for (int i = 0; i < count; i++)
                {
                    _colorArray[i] = dgvSetColor.Rows[0].Cells[i].Style.BackColor;
                    dgvSetColor.Rows[1].Cells[i].Value = _valueArray[i];
                }

                //Event trigger
                OnValueRangeChanged(this);
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
            }
        }

        /// <summary>
        ///Get maximum and minimum values
        /// </summary>
        ///<param name="min">latest value</param>
        ///<param name="max">maximum</param>
        private bool GetMinMaxValues(ref double min, ref double max)
        {
            bool ok = false;
            try
            {
                ok = double.TryParse(txtMin.Text, out min);
                if (!ok)
                {
                    MessageBox.Show("The minimum value must be a number.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    return ok;
                }
                ok = double.TryParse(txtMax.Text, out max);
                if (!ok)
                {
                    MessageBox.Show("The maximum value must be number.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    return ok;
                }
                if (max <= min)
                {
                    MessageBox.Show("The maximum value must be greater than the minimum value.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    ok = false;
                    return ok;
                }
                return ok;
            }
            catch (Exception ex)
            {
                CommonClass.LogError (ex);
                return ok;
            }
        }
    }//class
}//namespace
