#!/usr/local/bin/python2.6
# Allocates the gridded reports to annual and monthly county and state reports.
#
# 05/17/2011 James Beidler <beidler.james@epa.gov>

from datetime import date, timedelta
from optparse import OptionParser
import os, csv, sys

# Possible run types for the SMOKE MOVES reports
runTypes = ['RPD', 'RPP', 'RPV']

# Dictionary that translates digit month string to three character month name 
monDict = {'01': 'jan', '02': 'feb', '03': 'mar', '04': 'apr', '05': 'may', '06': 'jun', '07': 'jul', '08': 'aug', '09': 'sep', '10': 'oct', '11': 'nov', '12': 'dec'}  
monList = monDict.keys()
monList.sort()

def openFile(fileName, accessType = 'r'):
	"""
	Tests to see if a file is available for access.  If not it returns an error and exits.
	If it is available it returns an open file object.
	"""
	try: file = open(fileName, accessType)
	except:
		print "ERROR: %s not available for access." %fileName
		sys.exit(1)
	else: return file

def checkEV(evName):
	"""
	Checks if an environment variable is set.  If not, exits.  If it is, returns the variable.
	Takes the name of the environment variable.
	"""
	try: var = os.environ[evName]
	except:
		print "ERROR: Environment variable '%s' is not defined." %evName
		sys.exit(1)
	else: return var

class movesReport(object):
	"""
	Create a class for each month.
	"""
	def __init__(self, Mon = "", runType = "", gridFlag = False):
		self.mon = int(Mon)
		self.countyDescDict = {}
		self.rowList = [] 
		self.colList = [] 
		self.speciesList = []
		self.repDict = {}
		self.runType = runType
	
		# If the file contains grid information, then include that in the expected header line	
		if gridFlag == True:
			self.headLine = ['date', 'col', 'row', 'fips', 'state', 'county', 'scc']	
		else:
			self.headLine = ['date', 'fips', 'state', 'county', 'scc']


	def __nameRepFile(self):
		"""
		Set the infile name for the REP file based on the SMOKE conventions.
		"""
		inFileName = 'rep_mole_%s_%s_%s_%s_%s_%s.txt' %(self.runType, sector, self.esdate, grid, spec, case) 
		inPath = os.path.join(home, 'reports/smkmerge/%s/%s' %(sector, self.runType))
		return os.path.join(inPath, inFileName)

	def __readRepFile(self):
		"""
		Read the rep file and put into a list.
		"""
		repFileName = self.__nameRepFile()
		print repFileName
		for line in csv.reader(openFile(repFileName, 'r'), delimiter=';'):
			if self.headLine[0] in line[0].lower(): 
				self.__createSpeciesList(line[len(self.headLine):])
			if line[0][0] == "#": 
				continue
			# Read the line and gather the info based on the column names in the header line
			infoDict = dict([ (self.headLine[x], line[x].strip()) for x in range(len(self.headLine)) ])
			date = infoDict['date']
			fips = infoDict['fips']
			scc = infoDict['scc']
			state = infoDict['state']
			county = infoDict['county']

			if 'col' in infoDict:
				cell = '%s+%s' %(infoDict['col'], infoDict['row'])
			else:
				cell = '0'

			# Loop over the species in the file and add them to the month dictionary
			for x in range(len(self.speciesList)):	
				speciesName = self.speciesList[x]
				value = float(line[len(self.headLine)+x].strip())

				if fips not in self.repDict: 
					self.repDict[fips] = { cell: { scc: { speciesName: value } } }
				elif cell not in self.repDict[fips]: 
					self.repDict[fips][cell] = { scc: { speciesName: value } } 
				elif scc not in self.repDict[fips][cell]: 
					self.repDict[fips][cell][scc] = { speciesName: value } 
				elif speciesName not in self.repDict[fips][cell][scc]:
					self.repDict[fips][cell][scc][speciesName] = value
				elif speciesName in self.repDict[fips][cell][scc]:
					self.repDict[fips][cell][scc][speciesName] = value + self.repDict[fips][cell][scc][speciesName]
	
			if fips not in self.countyDescDict: 
				self.countyDescDict[fips] = {'st': state, 'cty': county}

	def __createSpeciesList(self, speciesLine):
		"""
		Create a list of species from the report file
		"""
		self.speciesList = []
		for speciesName in speciesLine:
			if speciesName != '':
				self.speciesList.append(speciesName.strip())

	def readMonth(self):
		"""
		Reads the month for each run type and puts into the dictionary
		"""
		today = date(int(Year), self.mon, 1)

		while today.month == self.mon:
			self.esdate = '%s%s%s' %(today.year, str(today)[5:7], str(today)[8:10]) 
			self.__readRepFile()	
 
			today = today + timedelta(1)

		return self.repDict

def openOutFile(speciesList, areaType = 'state', dateType = 'annual', bySCC = False, runName = ''):
	"""
	Set the outfile name.
	"""
	if dateType == 'monthly':
		duration = monDict[Mon]
	else:
		duration = dateType 

	if bySCC == True:
		outFileName = 'rep_scc_%s_%s_%s_%s%s_%s.txt' %(areaType, duration, sector, runName, case, grid)
		byType = 'emissions by SCC'
	if bySCC == False:
		outFileName = 'rep_%s_%s_%s_%s%s_%s.txt' %(areaType, duration, sector, runName, case, grid)
		byType = 'emissions'

	outPath = os.path.join(home, 'reports/smkmerge/%s' %sector)
	outFileName = os.path.join(outPath, outFileName)
	outFile = openFile(outFileName, 'w')

	print 'Writing %s %s to: %s' %(areaType, duration, outFileName)

	if dateType == 'monthly': 
		headerLine = 'Month,FIPS,State Name,'
	else: 
		headerLine = 'FIPS,State Name,'

	if areaType == 'county': 
		headerLine = headerLine + 'County Name,'
	if bySCC == True: 
		headerLine = headerLine + 'SCC7,'

	outFile.write(headerLine + ','.join(speciesList) + '\n')

	return outFile 

def combRuns(inDict, outDict):
	# Add together runs and return an output dictionary
	for fips in inDict:
		if fips not in outDict:
			outDict[fips] = {}
			
		for cell in inDict[fips]:
			if cell not in outDict[fips]:
				outDict[fips][cell] = {}
						
			for scc in inDict[fips][cell]:
				if scc not in outDict[fips][cell]:
					outDict[fips][cell][scc] = {}

				for speciesName in inDict[fips][cell][scc]:
					if speciesName not in outDict[fips][cell][scc]:
						outDict[fips][cell][scc][speciesName] = inDict[fips][cell][scc][speciesName] 
					else:
						outDict[fips][cell][scc][speciesName] = inDict[fips][cell][scc][speciesName] + outDict[fips][cell][scc][speciesName]

	return outDict

def writeRep(outDict, speciesList, areaType = 'state', dateType = 'annual', bySCC = False, runName = ''):
	# Write to the output file

	outFile = openOutFile(speciesList, areaType, dateType, bySCC, runName)

	lineDict = { 'stfips': '', 'scc': {} }

	fipsList = outDict.keys()
	fipsList.sort()
	for fips in fipsList:
		stFips = fips[:3]

		# Write the county values on every new fips if areaType is county
		if lineDict['stfips'] != '' and bySCC != True and areaType == 'county':
			if dateType == 'monthly':
				outFile.write('%s,' %Mon)

			sccSum = {}
			for scc in lineDict['scc']:
				for species in speciesList:
					if species not in sccSum:
						sccSum[species] = lineDict['scc'][scc][species]
					else:
						sccSum[species] = str( float(sccSum[species]) + float(lineDict['scc'][scc][species]) ) 
			outLine = '%s,%s,%s' %(lineDict['fips'], lineDict['st'], lineDict['cty'])
			outLine = '%s,%s\n' %( outLine, ','.join( [ sccSum[sp] for sp in speciesList ] ) ) 
			outFile.write(outLine)

		# Reset the line information for a new county fips after writing any county not by SCC file or if this is the first fips
		if areaType == 'county' or lineDict['stfips'] == '':
			lineDict = { 'fips': fips, 'stfips': stFips, 'st': countyDescDict[fips]['st'], 'cty': countyDescDict[fips]['cty'], 'scc': {} }

		# Write the state values on every new state if areaType is state
		if stFips != lineDict['stfips'] and areaType == 'state':
		
			# Write by SCC	
			if bySCC == True:
				sccList = lineDict['scc'].keys()
				sccList.sort()
				for scc in sccList:
					if dateType == 'monthly':
						outFile.write('%s,' %Mon)
					outLine = '%s,%s,%s' %(lineDict['stfips'], lineDict['st'], scc) 
					outLine = '%s,%s\n' %( outLine, ','.join( [ lineDict['scc'][scc][sp] for sp in speciesList ] ) )
					outFile.write(outLine)
			else:
				# Write by just state
				sccSum = {}
				for scc in lineDict['scc']:
					for species in speciesList:
						if species not in sccSum:
							sccSum[species] = lineDict['scc'][scc][species]
						else:
							sccSum[species] = str( float(sccSum[species]) + float(lineDict['scc'][scc][species]) )
				if dateType == 'monthly':
					outFile.write('%s,' %Mon)
				outLine = '%s,%s' %(lineDict['stfips'], lineDict['st'])
				outLine = '%s,%s\n' %( outLine, ','.join( [ sccSum[sp] for sp in speciesList ] ) )
				outFile.write(outLine)

			# Reset the line information for a new state fips
			lineDict = { 'fips': fips, 'stfips': stFips, 'st': countyDescDict[fips]['st'], 'cty': countyDescDict[fips]['cty'], 'scc': {} }

		# Loop through the output dictionary by all dimensions
		for cell in outDict[fips]:
			#Included for future subsetting
			lineDict['cell'] = cell

			sccList = outDict[fips][cell].keys()
			sccList.sort()

			for scc in sccList:
				if scc not in lineDict['scc']:
					lineDict['scc'][scc] = {}

				for species in speciesList:
				
					# Handle the odd case where the species might be in the specieslist, but not the dict
					if species not in outDict[fips][cell][scc]:
						val = 0
					else:
						val = outDict[fips][cell][scc][species]

						# If the tons conversion is turned on and the species is not a mode species then convert to tons
						if tonsConv == True and species[3:5] != '__':  #Mode species have two underscores at 3 and 4
							val = val * (0.00000110231131) 

					# Add together values if the species is already in the scc
					if species not in lineDict['scc'][scc]:
						lineDict['scc'][scc][species] = str(val)
					else:
						lineDict['scc'][scc][species] = str( val + float(lineDict['scc'][scc][species]) )

				# Write county by SCC 
				if bySCC == True and areaType == 'county':
					if dateType == 'monthly':
						outFile.write('%s,' %Mon)
					outLine = '%s,%s,%s,%s' %(lineDict['fips'], lineDict['st'], lineDict['cty'], scc)
					outLine = '%s,%s\n' %( outLine, ','.join( [ lineDict['scc'][scc][sp] for sp in speciesList ] ) )
					outFile.write(outLine)
					# Reset the SCC values after writing county by SCC
					lineDict['scc'] = {}



# Set up the options parser
parser = OptionParser(usage = 'usage: %prog [options]')
parser.add_option('-m', '--month', dest='runMonth', help='Run the one specified month.  Use a two character month code (ie. 02 or 10)', metavar='MONTH', default='')
parser.add_option('-t', '--tons', action='store_false', dest='tonsConv', help='Turn OFF the conversion of non-mode species from grams to tons.', default=True)
parser.add_option('-s', '--typesplit', action='store_true', dest='typeSplit', help='Split the output reports by run type rather than combing into a single output report file.', default=False)
parser.add_option('-r', '--runtype', dest='runType', help='Run single run type of either RPP, RPD, or RPV.  Automatically turns on the type splitting.', default='')
(options, args) = parser.parse_args()

# Set the global variables based on EMF set environment variables
Year = checkEV('BASE_YEAR')
grid = checkEV('GRID')
sector = checkEV('SECTOR')
spec = checkEV('EMF_SPC')
case = checkEV('CASE')
home = checkEV('PROJECT_ROOT')
gridFlag = False  # Environment variable to specify if the input file is gridded

# Process the parser options
tonsConv = options.tonsConv
typeSplit = options.typeSplit
if options.runMonth != '': 
	monList = [options.runMonth,]
if options.runType != '':
	runTypes = [options.runType,]
	typeSplit = 'Y' 

#### Main loop

annualDict = {}
speciesList = []
countyDescDict = {}

# Loop for each month
for Mon in monList:
	outDict = {}

	for runType in runTypes:
		typeReport = movesReport(Mon, runType, gridFlag)
		typeDict = typeReport.readMonth()
	
		# Keep the specieslist complete 
		if len(speciesList) < 1:	
			speciesList = typeReport.speciesList
		else:
			for species in typeReport.speciesList:

				if species not in speciesList:
					speciesList.append(species)
		speciesList.sort()

		# Keep the state/county descriptions complete
		if len(countyDescDict) < 1:
			countyDescDict = typeReport.countyDescDict
		else:
			for fips in countyDescDict:

				if fips not in countyDescDict:
					countyDescDict[fips] = typeReport.countyDescDict[fips]

		# Put the name of the runtype in the file name if you want to split by type
		if typeSplit == 'Y':
			runName = '%s_' %runType

			for areaType in ['county', 'state']:
				writeRep(typeDict, speciesList, areaType, 'monthly', False, runName)
				writeRep(typeDict, speciesList, areaType, 'monthly', True, runName)
		else:
		# Otherwise add them together
			outDict = combRuns(typeDict, outDict)

		# Add together monthly runs for annual reports
		if runType not in annualDict:
			annualDict[runType] = typeDict 
		else:
			annualDict[runType] = combRuns(typeDict, annualDict[runType])

	if typeSplit != 'Y':
		for areaType in ['county', 'state']:
			writeRep(outDict, speciesList, areaType, 'monthly', False)
			writeRep(outDict, speciesList, areaType, 'monthly', True)

# Write the annual reports if the month list has all twelve months
if len(monList) == 12:
	outDict = {}
	for runType in runTypes:

		if typeSplit == 'Y':
			runName = '%s_' %runType

			for areaType in ['county', 'state']:
				writeRep(annualDict[runType], speciesList, areaType, 'annual', False, runName)
				writeRep(annualDict[runType], speciesList, areaType, 'annual', True, runName)
		else:
			outDict = combRuns(annualDict[runType], outDict)

	if typeSplit != 'Y':
		for areaType in ['county', 'state']:
			writeRep(outDict, speciesList, areaType, 'annual', False)
			writeRep(outDict, speciesList, areaType, 'annual', True)

