#!/usr/bin/env python
# Splits up 12 km surrogates into 4 km surrogates
# Assumes an even distribution of 4 km grid cells within a 12 km cell-fips combination

import sys

inFile = open(sys.argv[1])
outFile = open(sys.argv[2], 'w')

colOffset = (-2736 - -2736) / 4 # (12 km x-orig - 4 km x-orig) / 4 km
rowOffset = (-2088 - -2088) / 4 # (12 km y-orig - 4 km y-orig) / 4 km
colMax = 1332 # last column in 4 km
rowMax = 1008 # last row in 4 km

for line in inFile.readlines():
	if line.startswith('#GRID') or line.startswith('#GRID\t'):
		outFile.write("#GRID	CONUS4k	-2736000.000000	-2088000.000000	4000.000000	4000.000000	1332	1008	1	LAMBERT	meters	33.000000	45.000000	-97.000000	-97.000000	40.000000\n")
		continue
	elif line.startswith('#'):
		outFile.write(line)
		continue

	if '\t' in line:
		line = [cell.strip() for cell in line.strip().split('\t')]
	else:
		line = [cell.strip() for cell in line.strip().split()]

	code = line[0]
	fips = line[1]
	iCol = int(line[2])  # First column and row name is 1, even though it originates at 0
	iRow = int(line[3])
	iRatio = float(line[4])

	if len(line) > 5:
		delim = line[5]
		cellarea = float(line[6]) # Total area of cell in FIPS
		fipsarea = line[7] # Total area in FIPS
		oCA = cellarea / (3 * 3)

	col = (iCol * (12 / 4)) + colOffset - (3 - 1) # (12 km col * (12 grid size / 4 grid size)) + 4 km col offset - (3 4km grid cells back to origin plus one forward to get name of first column)
	row = (iRow * (12 / 4)) + rowOffset - (3 - 1) # (12 km row * (12 grid size / 4 grid size)) + 4 km row offset - (3 grid cells back to origin plus one forward to get name of first row)

	oRatio = iRatio / (3 * 3)  # Divide the ratio of cell to land area by the number of 4 km grid cells in 12 km grid cells

	for x in range(3):
		oCol = col + x
		if oCol > colMax:
			continue

		for y in range(3):
			oRow = row + y
			if oRow > rowMax:
				continue

			if len(line) > 5:
				outLine = [code, fips, str(oCol), str(oRow), str(oRatio), delim, str(oCA), fipsarea, str(oRatio)]
			else:
				outLine = [code, fips, str(oCol), str(oRow), str(oRatio)]

			outFile.write('%s\n' %' '.join(outLine))
