﻿# This script assumes access to the movesdb20151028 database.
# 10/10/2014 djb
# Updated 11/13/2014 djb
# Updated 12/02/2014 djb
#      Eliminate divide by zero errors. 
# Updated 12/04/2014 djb
#      Conserve APU hours and report problems.
# Updated 12/09/2014 djb
#      Provide range to check for differences in auditlog insert.
#      Use all road type VMT to calculate distributions.
#      Apply the hourly distribution as an inverse to the VMT distribution.
# Update 12/11/2014 djb
#      Replace user supplied dayvmtfractions and default hotelling hourly temporalization
#      with hard coded 2011 NEI values (profile 51000).

# Calculate the travel fraction for source type 62 (long haul combination trucks).
# All long haul combination trucks are diesel and only these vehicles
# are assumed to have hotelling emissions.

# ** This script truncates the existing HotellingActivityDistribution table contents
#    and replaces them with the MOVES default values.  This assumes that none of the
#    existing CDBs have HotellingActivityDistribution information.  This truncation
#    can be made optional if information might exist, but the script will need updating.

use CDBPLACEHOLDER;

drop table if exists travelfraction;
create table travelfraction
 select a.yearid,a.sourcetypeid,a.ageid,a.agefraction,b.relativemar,
        (a.agefraction*b.relativemar) as product
 from sourcetypeagedistribution as a,
      movesdb20151028.sourcetypeage as b
 where a.yearid=2014
   and a.sourcetypeid=62
   and a.sourcetypeid=b.sourcetypeid
   and a.ageid=b.ageid;

drop table if exists test.junk;
create table test.junk
 select yearid,sourcetypeid,sum(product) as sumproduct
 from travelfraction
 group by yearid,sourcetypeid;

alter table travelfraction
 add sumproduct double,
 add travelfraction double default 0.0,
 add modelyearid integer;

update travelfraction as a, test.junk as b
 set a.sumproduct=b.sumproduct,
     a.travelfraction=a.product/b.sumproduct,
     a.modelyearid=a.yearid-a.ageid
 where a.yearid=b.yearid
   and a.sourcetypeid=b.sourcetypeid
   and b.sumproduct>0.0;

# Calculate the fraction of APU usage by model year.
# Assumes that only 2010 and later model years use APUs
# and that the same fraction of APU usage occurs in each model year.
# This will not work if the calendar year is before 2010
# or if the APU usage is greater than hotelling hours in those model years.
drop table if exists apuusage;
create table apuusage
 select yearid,sourcetypeid,ageid,modelyearid,travelfraction
 from travelfraction
 where modelyearid>=2010;

drop table if exists test.junk;
create table test.junk
 select yearid,sourcetypeid,sum(travelfraction) as fracsum
 from apuusage
 group by yearid,sourcetypeid;

alter table apuusage
 add fracsum double,
 add apufrac double default 0.0;

update apuusage as a, test.junk as b
 set a.fracsum=b.fracsum,
     a.apufrac=a.travelfraction/b.fracsum
 where a.yearid=b.yearid
   and a.sourcetypeid=b.sourcetypeid
   and b.fracsum>0.0;

# Create a table with the total annual hours from the FF10 file
# and calculate the annual hours by model year.
drop table if exists annual;
create table annual
 select region_cd as countyid,a.yearid,a.sourcetypeid,a.ageid,a.modelyearid,a.travelfraction,
        b.ann_value as totidlehours
 from travelfraction as a,
      ff10data.hotff10 as b,
      county as c
 where b.region_cd=c.countyid
   and b.scc="2202620153"  -- extended idle hours.
   and a.yearid=b.calc_year;

alter table annual
 add totapuhours double default 0.0,
 add hothours double default 0.0,
 add idlehours double default 0.0,
 add apuhours double default 0.0;

update annual as a, ff10data.hotff10 as b
 set a.totapuhours=b.ann_value
 where a.countyid=b.region_cd
   and b.scc="2202620191"  -- APU operation hours.
   and a.yearid=b.calc_year;

update annual set hothours=travelfraction*(totidlehours+totapuhours);
update annual as a, apuusage as b
 set a.apuhours=a.totapuhours*b.apufrac
 where a.yearid=b.yearid
   and a.sourcetypeid=b.sourcetypeid
   and a.modelyearid=b.modelyearid;
update annual set idlehours=hothours-apuhours;

# Provide a warning if the number of APU hours in the FF10 is not equal to 
# the number of APU hours in the table.
drop table if exists test.junk;
create table test.junk
 select countyid,totapuhours,sum(apuhours) as outapuhours,
                 sum(hothours) as hotellinghours,sum(idlehours) as idlehours
 from annual group by countyid;
#select "Warning: APU hours do not match input hours.!",outapuhours,totapuhours
# from test.junk
# where abs(outapuhours-totapuhours)>0.01;
#select "         Extended idle hours                 ",idlehours,hotellinghours
# from test.junk
# where outapuhours<>totapuhours;
truncate auditlog;
insert into auditlog (whenhappened,importername,briefdescription,fulldescription)
 select curdate(),
        "HotellingHours",
        "APU hours do not match input hours.",
        "The APU hours in the HotellingHours table do not match the user provided APU hours."
 from test.junk
 where abs(outapuhours-totapuhours)>0.01;
 
# Allocate the annual hours to months of the year.
drop table if exists monthly;
create table monthly
 select a.countyid,a.yearid,a.sourcetypeid,a.ageid,a.modelyearid,b.monthid,b.monthvmtfraction,
        (a.idlehours*b.monthvmtfraction) as idlemonthhours,
        (a.apuhours*b.monthvmtfraction) as apumonthhours
 from annual as a,
      monthvmtfraction as b
 where a.sourcetypeid=b.sourcetypeid;

# Calculate the fraction of total VMT in each month that occur in each hour of the day
# in each day type in each month.  
drop table if exists monthVMT;
create table monthVMT
 select a.monthid,a.noofdays,b.hourid
 from movesdb20151028.monthofanyyear as a,
      movesdb20151028.hourofanyday as b;
alter table monthvmt add vmtdist double;
update monthvmt set vmtdist=0.065509124 where hourid=1;
update monthvmt set vmtdist=0.067149639 where hourid=2;
update monthvmt set vmtdist=0.067095096 where hourid=3;
update monthvmt set vmtdist=0.064839431 where hourid=4;
update monthvmt set vmtdist=0.059723244 where hourid=5;
update monthvmt set vmtdist=0.051756998 where hourid=6;
update monthvmt set vmtdist=0.041633391 where hourid=7;
update monthvmt set vmtdist=0.034717772 where hourid=8;
update monthvmt set vmtdist=0.028459737 where hourid=9;
update monthvmt set vmtdist=0.023532083 where hourid=10;
update monthvmt set vmtdist=0.020464079 where hourid=11;
update monthvmt set vmtdist=0.019226993 where hourid=12;
update monthvmt set vmtdist=0.01982685  where hourid=13;
update monthvmt set vmtdist=0.020754895 where hourid=14;
update monthvmt set vmtdist=0.02161809  where hourid=15;
update monthvmt set vmtdist=0.024134336 where hourid=16;
update monthvmt set vmtdist=0.028141909 where hourid=17;
update monthvmt set vmtdist=0.033628096 where hourid=18;
update monthvmt set vmtdist=0.039637667 where hourid=19;
update monthvmt set vmtdist=0.045185647 where hourid=20;
update monthvmt set vmtdist=0.050018004 where hourid=21;
update monthvmt set vmtdist=0.053721351 where hourid=22;
update monthvmt set vmtdist=0.057533198 where hourid=23;
update monthvmt set vmtdist=0.06169237  where hourid=24;

# Apply the monthly hourly vmt distribution to the extended idle and apu hours.
# All ages get the same monthly distribution.
drop table if exists drafthours;
create table drafthours
 select a.countyid,a.sourcetypeid,b.hourid,a.monthid,a.yearid,a.ageid,
        ((a.idlemonthhours+a.apumonthhours)*b.vmtdist) as drafthours,b.noofdays
 from monthly as a, 
      monthvmt as b
 where a.monthid=b.monthid;

# Apply the NEI day type distribution to all hours.
drop table if exists drafthotellinghours;
create table drafthotellinghours
 select a.countyid,a.sourcetypeid,a.monthid,a.yearid,a.ageid,
        a.drafthours,a.noofdays,b.hourdayid,b.hourid,b.dayid
 from drafthours as a,
      movesdb20151028.hourday as b
 where a.hourid=b.hourid;

alter table drafthotellinghours
 add hours double default 0.0,
 add weekhours double default 0.0;

# Day type profile:
# profile_id,monday,tuesday,wednesday,thursday,friday,saturday,sunday
# 51000,0.148699906,0.170282823,0.177767116,0.171727395,0.153245692,0.094445667,0.083831401
update drafthotellinghours
 set hours=drafthours*0.178277068 where dayid=2;
update drafthotellinghours
 set hours=drafthours*0.821722932 where dayid=5;

update drafthotellinghours
 set weekhours=hours/(noofdays/7.0)
 where noofdays>0;

# Place the calculated hours in the MOVES formatted table.
drop table if exists hotellinghours;
create table hotellinghours like movesdb20151028.hotellinghours;
insert into hotellinghours
 (sourcetypeid,hourdayid,monthid,yearid,ageid,zoneid,hotellinghours,isuserinput)
 select
  sourcetypeid,hourdayid,monthid,yearid,ageid,
  (countyid*10) as zoneid,
  weekhours as hotellinghours,
  "Y" as isuserinput
 from drafthotellinghours;

# Adjust the HotellingActivityDistribution table to reflect the user supplied
# APU fractions.
truncate hotellingactivitydistribution;
insert into hotellingactivitydistribution
 select * from movesdb20151028.hotellingactivitydistribution;
update hotellingactivitydistribution as a,
       annual as b 
 set opmodefraction=(apuhours/(apuhours+idlehours))
 where a.beginmodelyearid=2010
   and a.opmodeid=201
   and b.modelyearid=2010
   and (apuhours+idlehours)>0.0;
update hotellingactivitydistribution as a,
       annual as b 
 set opmodefraction=1.0-(apuhours/(apuhours+idlehours))
 where a.beginmodelyearid=2010
   and a.opmodeid=200
   and b.modelyearid=2010
   and (apuhours+idlehours)>0.0;

# Remove any intermediate calculation tables.
#drop table if exists annual;
#drop table if exists apuusage;
#drop table if exists monthly;
#drop table if exists drafthotellinghours;
#drop table if exists monthvmt;
#drop table if exists travelfraction;
#drop table if exists test.junk;
#drop table if exists drafthours;
