#--------------------------------------------------------------------------------------
#
# crplot.R - make concentration-response plots using the current ToxCast data
#
# March 2014
# Richard Judson
#
# US EPA
# Questions, comments to: judson.richard@epa.gov
#
#--------------------------------------------------------------------------------------
library(grDevices)
library(RColorBrewer)
library(stringr)
#--------------------------------------------------------------------------------------
#
# Driver for the plots
#
#--------------------------------------------------------------------------------------
driver <- function() {




}
#--------------------------------------------------------------------------------------
#
# plot an assays for a set of chemicals
#
#--------------------------------------------------------------------------------------
plot.cr <- function(assay="ATG_ERa_TRANS",source="Attagene",mode="agonist",to.file=F,do.load.assay=T,do.load.general=T,chem.set="REF",plot.flare=F) {
	cat("==========================================================================\n")
	cat("Process ",assay,"\n")
	cat("==========================================================================\n")

	cat("chem.set: ALL,EDSP21,MED400,REF,BG1,BLA, REF88\n")
	plot.rc <- 2
	plot.test <- 2

	DHOME <- "Data_2012_10_24/"
	CFILE <- "ToxMiner_Chemicals_All_2012_10_24.txt"
	REFILE <- "EDSP21_ER_RefChems_Agonists_15_Nov_2012.txt"
	EDSP21FILE <- "ToxCast_EDSP21_CASRN.txt"
	MED400FILE <- "MED_400_CASRN.txt"
	REF88FILE <- "REF_88_CASRN.txt"
	BLAFILE <- "BLA_CASRN.txt"
	BG1FILE <- "BG1_CASRN.txt"
	QCFILE <- "ToxCast_sample_qc_2012_10_24.txt"
	ATGFILTERFILE <- "ATG_Filter_2012_08_03.csv"
	BG1PLATEQCFILE <- "Tox21_ERa_LUC_BG1_Agonist_plate_qc.txt"
	BLAPLATEQCFILE <- "Tox21_ERa_BLA_Agonist_ratio_plate_qc.txt"

	file.refsd <- paste(assay,"_reference_SD_norm.csv",sep="")
	s <- "Assay,CASRN,NAME,MEAN_SD\n"
	cat(s,file=file.refsd,sep="",append=F)

	QC <- read.table(file=paste(DHOME,QCFILE,sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
	RC <- read.table(file=paste(DHOME,REFILE,sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")

	if(source=="Odyssey Thera") date.string <- "2012_10_24"
	if(source=="Attagene")      date.string <- "2012_10_24"
	if(source=="NCGC")          date.string <- "2012_10_24"
	if(source=="Novascreen")    date.string <- "2012_10_24"

	if(assay=="Tox21_ERa_LUC_BG1_Antagonist")   {refchem <- "4-Hydroxy Tamoxifen";viability <- "Tox21_ERa_LUC_BG1_Antagonist_viability";}
	if(assay=="Tox21_ERa_BLA_Antagonist_ratio") {refchem <- "4-hydroxy Tamoxifen";viability <- "Tox21_ERa_BLA_Antagonist_viability";}
	if(assay=="Tox21_ERa_LUC_BG1_Agonist")      {refchem <- "Beta-estradiol";     viability <- "Tox21_ERa_LUC_BG1_Antagonist_viability";}
	if(assay=="Tox21_ERa_BLA_Agonist_ratio")    {refchem <- "Beta-Estradiol";     viability <- "Tox21_ERa_BLA_Antagonist_viability";}

	if(assay=="ATG_ERa_TRANS") {refchem <- "Estradiol";viability <- "ATG_XTT_Cytotoxicity";}
	if(assay=="ATG_ERE_CIS")   {refchem <- "Estradiol";viability <- "ATG_XTT_Cytotoxicity";}

	#if(assay=="OT_ERa_ERE_LUC_Agonist_1440") {refchem <- "17b-Estradiol";viability <- NA;}
	#if(assay=="OT_ERa_ERE_LUC_Agonist_1440") {refchem <- "17b-Estradiol";viability <- NA;}
	if(source=="Odyssey Thera") {refchem <- "17b-Estradiol";viability <- NA;}
	
	if(assay=="NVS_NR_hER")  {refchem <- "17-B-Estradiol";viability <- NA;}
	if(assay=="NVS_NR_bER")  {refchem <- "17-B-Estradiol";viability <- NA;}
	if(assay=="NVS_NR_mERa") {refchem <- "17-B-Estradiol";viability <- NA;}

	do.plate.qc <- F
	if(assay=="Tox21_ERa_LUC_BG1_Agonist") {
		plate.qc <- read.table(file=paste(DHOME,BG1PLATEQCFILE,sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
		do.plate.qc <- T
	}
	if(assay=="Tox21_ERa_BLA_Agonist_ratio") {
		plate.qc <- read.table(file=paste(DHOME,BLAPLATEQCFILE,sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
		do.plate.qc <- T
	}

	if(source=="Attagene") filter <- read.csv(paste(DHOME,ATGFILTERFILE,sep=""),header=T,row.names=1)
	if(do.load.assay) {
		load.assay(assay,date.string)
		if(!is.na(viability)) load.viability(viability,date.string)
	}
	if(do.load.general) {
		load.fits(source,date.string)
		CHEMS <<- read.table(file=paste(DHOME,CFILE,sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,comment.char="",quote="")
		cat("loaded chemicals\n");
		flush.console()
    }
	if(assay=="Tox21_ERa_LUC_BG1_Agonist") {
		all.conc <- CR[,"conc"]
		all.resp <- CR[,"resp"]

		mad.z.conc <- c(92,42,19,8.7,3.8,1.74,0.77,0.35,0.15,0.069,0.031,0.0135,0.0062,0.00275,0.0012)
		mad.z.resp <- mad.z.conc
		mad.z.resp[] <- 0
		for(i in 1:length(mad.z.conc)) {
			minc <- mad.z.conc[i]/1.2
			maxc <- mad.z.conc[i]*1.2
			flag <- all.conc
			flag[] <- 1
			flag[all.conc>maxc] <- 0
			flag[all.conc<minc] <- 0
			mad.z.resp[i] <- 3*mad(all.resp[flag==1])
		}
		mad.z.conc <- mad.z.conc/1000000
	}
	EDSP21_CHEMS <<- read.table(file=paste(DHOME,EDSP21FILE,sep=""),sep="\t",header=F,check.names=F,stringsAsFactors=F,comment.char="",quote="")
	EDSP21_CHEMS <<- as.character(EDSP21_CHEMS[,1])

	MED400_CHEMS <<- read.table(file=paste(DHOME,MED400FILE,sep=""),sep="\t",header=F,check.names=F,stringsAsFactors=F,comment.char="",quote="")
	MED400_CHEMS <<- as.character(MED400_CHEMS[,1])

	REF88_CHEMS <<- read.table(file=paste(DHOME,REF88FILE,sep=""),sep="\t",header=F,check.names=F,stringsAsFactors=F,comment.char="",quote="")
	REF88_CHEMS <<- as.character(REF88_CHEMS[,1])

	BLA_CHEMS <<- read.table(file=paste(DHOME,BLAFILE,sep=""),sep="\t",header=F,check.names=F,stringsAsFactors=F,comment.char="",quote="")
	BLA_CHEMS <<- as.character(BLA_CHEMS[,1])

	BG1_CHEMS <<- read.table(file=paste(DHOME,BG1FILE,sep=""),sep="\t",header=F,check.names=F,stringsAsFactors=F,comment.char="",quote="")
	BG1_CHEMS <<- as.character(BG1_CHEMS[,1])

	ref.table <- as.data.frame(matrix(data=NA,nrow=3*dim(RC)[1],ncol=16))
	names(ref.table) <- c("Assay","CASRN","Name","sample_id","Expected","Order","AC50","T","B","W","Emax","ActivityCall","ActivityCallNote","SigPoints","Analytical_QC_Grade","COA_Purity")

	score.table <- as.data.frame(matrix(data=NA,nrow=3*length(EDSP21_CHEMS),ncol=14))
	names(score.table) <- c("Assay","CASRN","Name","sample_id","AC50","T","B","W","Emax","ActivityCall","ActivityCallNote","SigPoints","Analytical_QC_Grade","COA_Purity")

	fits <- FITS[FITS[,"assay_name"]==assay,]
	fits.via <- FITS[FITS[,"assay_name"]==viability,]
	cr <- CR

	conc.list <- CR[,"conc"]
	conc.list <- sort(uniquify(conc.list))

	resp.cutoff <- 1
#	if(assay=="Tox21_ERa_LUC_BG1_Antagonist")   resp.cutoff <- 1
#	if(assay=="Tox21_ERa_BLA_Antagonist_ratio") resp.cutoff <- 1
	if(assay=="Tox21_ERa_LUC_BG1_Agonist")      resp.cutoff <- 35
	if(assay=="Tox21_ERa_BLA_Agonist_ratio")   resp.cutoff <- 15

	if(assay=="ATG_ERa_TRANS") resp.cutoff <- 25
	if(assay=="ATG_ERE_CIS")   resp.cutoff <- 25
	atg.cutoff.T <- 40
	atg.cutoff.T <- 25

	if(assay=="OT_ERa_ERE_LUC_Agonist_1440") resp.cutoff <- 15
	if(source=="Odyssey Thera") resp.cutoff <- 25
#	if(resp.cutoff<25) resp.cutoff <- 25

	if(assay=="NVS_NR_hER")  resp.cutoff <- 60
	if(assay=="NVS_NR_bER") resp.cutoff <- 60
	if(assay=="NVS_NR_mERa") resp.cutoff <- 55

	if(!is.na(viability)) cr.via <- CR.VIA
	sample.col.name <- "sample_id"
	if(source=="NCGC") sample.col.name <- "sample_id"
	refcurve <- CR[CR[,sample.col.name]==refchem,]

	fit.params.rc <- fits[fits[,"sample_id"]==refchem,]
	AC50.rc <- fit.params.rc[1,"AC50"]/1000000
	CI_L.rc <- fit.params.rc[1,"AC50_2se_lwr"]/1000000
	CI_U.rc <- fit.params.rc[1,"AC50_2se_upr"]/1000000
	line.x.rc <- as.character(fit.params.rc[1,"line_x"])
	line.y.rc <- as.character(fit.params.rc[1,"line_y"])
	hill.x.rc <- as.numeric(strsplit(line.x.rc,"\\|",perl=F)[[1]])/1000000
	hill.y.rc <- as.numeric(strsplit(line.y.rc,"\\|",perl=F)[[1]])
	B.rc <- fit.params.rc[1,"B"]
	T.rc <- fit.params.rc[1,"T"]
	if(mode=="antagonist") {
		B.rc <- 100-B.rc
		T.rc <- 100-T.rc
	}
	if(mode=="antagonist") hill.y.rc <- 100-hill.y.rc

	conc.list.rc <- as.numeric(refcurve[,"conc"])
	conc.list.rc <- uniquify(conc.list.rc)
	conc.list.rc <- conc.list.rc[1:(length(conc.list.rc)-1)]
	mean.list.rc <- conc.list.rc
	sd.list.rc <- conc.list.rc
	for(i in 1:length(conc.list.rc)) {
		cc <- conc.list.rc[i]
		vals <- refcurve[refcurve[,"conc"]==cc,"resp"]
		vals <- vals[vals<1000000]
		mean.list.rc[i] <- mean(vals)
		sd.list.rc[i] <- sd(vals)
	}
	if(mode=="antagonist") mean.list.rc <- 100-mean.list.rc
	conc.list.rc <- conc.list.rc / 1000000
	if(assay=="NVS_NR_bER") {
		conc.list.rc <- conc.list.rc/10
		hill.x.rc <- hill.x.rc / 10
	}
	xmin <- 0.00001/1000000
	xmax <- 1000/1000000
	ymin <- -10
	ymax.0 <- 150
	nrc <- dim(RC)[1]
	par(mfrow=c(3,2),mar=c(4,4,2,0.1))
	if(to.file) {
		fname <- paste("plots/",assay,"_",chem.set,"_",date.string,".pdf",sep="")
		pdf(file=fname,width=7,height=10,pointsize=12,bg="white",paper="letter",pagecentre=T)
		par(mfrow=c(3,2),mar=c(4,4,2,0.1))
	}
   	nrow <- nrc

	#---------------------------------------------------------------
	#
	# select the subset of chemicals to be processed
	#
	#---------------------------------------------------------------
    casrn.list <- fits[,"chemical_casrn"]
    casrn.list <- sort(uniquify(casrn.list))
   	nrow <- length(casrn.list)
    if(chem.set=="EDSP21") {
    	casrn.list <- casrn.list[is.element(casrn.list,EDSP21_CHEMS)]
    	nrow <- length(casrn.list)
    }
    if(chem.set=="MED400") {
    	casrn.list <- casrn.list[is.element(casrn.list,MED400_CHEMS)]
    	nrow <- length(casrn.list)
    }
    if(chem.set=="REF88") {
    	casrn.list <- casrn.list[is.element(casrn.list,REF88_CHEMS)]
    	nrow <- length(casrn.list)
    }
    if(chem.set=="BLA") {
    	casrn.list <- casrn.list[is.element(casrn.list,BLA_CHEMS)]
    	nrow <- length(casrn.list)
    }
    if(chem.set=="BG1") {
    	casrn.list <- casrn.list[is.element(casrn.list,BG1_CHEMS)]
    	nrow <- length(casrn.list)
    }
    if(chem.set=="REF") {
    	nrow <- dim(RC)[1]
    }
    if(chem.set!="REF") {
    	fits <- fits[is.element(fits[,"chemical_casrn"],casrn.list),]
    }
	table_row <- 1

	if(chem.set=="ALL" || chem.set=="EDSP21" || chem.set=="MED400") fits <- fits[is.element(fits[,"chemical_casrn"],casrn.list),]
	if(chem.set=="REF") fits <- fits[is.element(fits[,"chemical_casrn"],RC[,"CASRN"]),]

	ncol <- dim(fits)[2]
	fits <- cbind(fits,fits[,ncol])
	fits <- cbind(fits,fits[,ncol])
	fits <- cbind(fits,fits[,ncol])
	fits <- cbind(fits,fits[,ncol])
	names(fits)[ncol+1] <- "AnalyticalQCGrade"
	names(fits)[ncol+2] <- "COA_Purity"
	names(fits)[ncol+3] <- "ActivityCall"
	names(fits)[ncol+4] <- "ActivityCallNote"
	fits[,ncol+1] <- NA
	fits[,ncol+2] <- NA
	fits[,ncol+3] <- NA
	fits[,ncol+4] <- NA

	#---------------------------------------------------------------
	#
	# loop over all chemicals
	#
	#---------------------------------------------------------------
	for(i in 1:nrow) {
      	if(chem.set=="REF") {
			casrn <- as.character(RC[i,"CASRN"])
			cname <- as.character(RC[i,"Name"])
      	}
        else {
			casrn <- casrn.list[i]
        }
		ref.sd.list <- c()
		
		sig.points <- 0
		ctemp <- fits[fits[,"chemical_casrn"]==casrn,]
        if(chem.set!="REF") cname <- ctemp[1,"chemical_name"]
		potency <- ""
		modifier <- ""
		rc.order <- ""
		if(chem.set=="REF") {
			potency <- as.character(RC[i,"Potency"])
			modifier <- as.character(RC[i,"Modifier"])
			rc.order <- as.character(RC[i,"Order"])
		}
		#---------------------------------------------------------------
		#
		# test for data availability
		#
		#---------------------------------------------------------------
		test.cr <- cr[cr[,"chemical_casrn"]==casrn,]
		if(!is.na(viability)) test.cr.via <- cr.via[cr.via[,"chemical_casrn"]==casrn,]
		cat("Try: [",casrn,"] [",cname,"] ",dim(test.cr)[1],"\n",sep="")
		if(dim(test.cr)[1]>0) {
			fit.params <- fits[fits[,"chemical_casrn"]==casrn,]
			if(!is.na(viability)) fit.params.via <- fits.via[fits.via[,"chemical_casrn"]==casrn,]
			if(source=="Odyssey Thera") fit.params <- fit.params[1,]
			
			#---------------------------------------------------------------
			#
			# pull out the fitting parameters
			#
			#---------------------------------------------------------------
			if(dim(fit.params)[1]>0) {
				for(j in 1:dim(fit.params)[1]) {
					fit.params.one <- fit.params[j,]
					AC50 <- fit.params.one[1,"AC50"]/1000000
					rAC50 <- fit.params.one[1,"rAC50"]/1000000
					AC50.mod <- fit.params.one[1,"AC50_mod"]
					B <- fit.params.one[1,"B"]
					T <- fit.params.one[1,"T"]
					W <- fit.params.one[1,"W"]
					rsq <- fit.params.one[1,"rsq"]
					logAC50_pval <- fit.params.one[1,"logAC50_pval"]
					converged <- fit.params.one[1,"converged"]
					CI_L <- fit.params.one[1,"AC50_2se_lwr"]/1000000
					CI_U <- fit.params.one[1,"AC50_2se_upr"]/1000000

					CI_L.W <- fit.params.one[1,"W_std_error"]
					CI_U.W <- fit.params.one[1,"W_std_error"]
					CI_L.B <- fit.params.one[1,"B_std_error"]
					CI_U.B <- fit.params.one[1,"B_std_error"]
					CI_L.T <- fit.params.one[1,"T_std_error"]
					CI_U.T <- fit.params.one[1,"T_std_error"]
					Emax <- fit.params.one[1,"Emax"]
					LEC <- fit.params.one[1,"LEC"]/1000000
					overall.test <- as.character(fit.params.one[1,"OVERALL_TEST"])
					line.x <- as.character(fit.params.one[1,"line_x"])
					line.y <- as.character(fit.params.one[1,"line_y"])
					sample_id <- as.character(fit.params.one[1,sample.col.name])

					temp <- strsplit(sample_id,";",perl=F)[[1]]
					sid.list <- temp
					print(sid.list)
					#---------------------------------------------------------------
					#
					# get the analytical purity information
					#
					#---------------------------------------------------------------
					grade <- "NA"
					purity <- -1
					for(g in 1:length(sid.list)) {
						gsid <- sid.list[g]
						if(is.element(gsid,QC[,"sample_id"])) {
							grade <- QC[QC[,1]==gsid,"tox21_grade"]
							grade <- grade[1]
							if(is.na(grade)) grade <- "NA"
							if(nchar(grade)==0) grade <- "NA"
							purity <- QC[QC[,1]==gsid,"coa_purity"]
							purity <- purity[1]
							if(is.na(purity)) purity <- "NA"
							cat("QC Grade ",gsid,"\t",grade,"\t",purity,"\n",sep="")
						}
					}

					#---------------------------------------------------------------
					#
					# process the fitting parameters
					#
					#---------------------------------------------------------------
					converged <- as.character(fit.params.one[,"converged"])
					if(is.na(converged)) converged <- "FALSE"
					if(logAC50_pval>=1000000) logAC50_pval <- 1
					if(rsq>=1000000) rsq <- 0
					if(T>=1000000) T <- 0
					if(B>=1000000) B <- 0
					if(W>=1000000) W <- 0

					if(mode=="antagonist") {
						if(B!=0) B <- 100-B
						if(T!=0) T <- 100-T
					}

					if(CI_L.B>=1000000) CI_L.B <- 0
					if(CI_U.B>=1000000) CI_U.B <- 0
					if(CI_L.T>=1000000) CI_L.T <- 0
					if(CI_U.T>=1000000) CI_U.T <- 0
					if(CI_L.W>=1000000) CI_L.W <- 0
					if(CI_U.W>=1000000) CI_U.W <- 0

					test.cr <- cr[is.element(cr[,sample.col.name],sid.list),]
					conc.list <- test.cr[,"conc"]/1000000
					resp.list <- test.cr[,"resp"]
					if(mode=="antagonist") resp.list <- 100-resp.list
					mask.list <- test.cr[,"mask"]
					conc.list.test <- conc.list
					conc.list.test <- uniquify(conc.list.test)
					conc.list.test <- sort(conc.list.test)
					mean.list.test <- conc.list.test
					n.list.test <- conc.list.test
					n.list.test[] <- 0
					sd.list.test <- conc.list.test
					sd.list.test[] <- 0
					nsample <- 0
					for(k in 1:length(conc.list.test)) {
						cc <- conc.list.test[k]
						resp.list.filtered <- resp.list[mask.list==0]
						conc.list.filtered <- conc.list[mask.list==0]
						vals <- resp.list.filtered[conc.list.filtered==cc]
						if(length(vals)>nsample) nsample <- length(vals)
						mean.list.test[k] <- mean(vals)
						sd.list.test[k] <- sd(vals)
						n.list.test[k] <- length(vals)
					}

					conc.list.test <- conc.list.test[n.list.test>0]
					mean.list.test <- mean.list.test[n.list.test>0]
					sd.list.test <- sd.list.test[n.list.test>0]
					ref.sd.list <- c(ref.sd.list,sd.list.test)
					ymax <- ymax.0
					if(T>ymax) ymax <- T * 1.2
					#---------------------------------------------------------------
					#
					# make the main plot
					#
					#---------------------------------------------------------------
					if(plot.test==1) {
						plot(resp.list~conc.list,xlab="Concentration (M)",cex.axis=1,cex.lab=1,ylab="% E2",log="x",xlim=c(xmin,xmax),ylim=c(ymin,ymax),main=paste(cname,":",casrn,"\n",assay),pch=21,cex.main=1)
						points(resp.list~conc.list,cex=1,bg="cyan",pch=21)
						lines(c(xmin,xmax),c(resp.cutoff,resp.cutoff),col="gray")
						lines(c(LEC,LEC),c(0,100),col="black",lwd=1)

					}
					if(plot.test==2) {
						plot(mean.list.test~conc.list.test,xlab="Concentration (M)",cex.axis=1.1,cex.lab=1.0,ylab="% E2",log="x",xlim=c(xmin,xmax),ylim=c(ymin,ymax),main=paste(cname,":",casrn,"\n",assay),pch=21,cex.main=1)
						#if(assay=="Tox21_ERa_LUC_BG1_Agonist") lines(mad.z.resp~mad.z.conc,col="gray")

						for(k in 1:length(conc.list.test) ){
							x <- conc.list.test[k]
							y <- mean.list.test[k]
							if(length(conc.list.test)>0) {
								if(mode=="agonist" && y>resp.cutoff) sig.points <- sig.points+1
								if(mode=="antagonist" && y<resp.cutoff) sig.points <- sig.points+1
								ysd <- sd.list.test[k]
								lines(c(x,x),c(y-ysd,y+ysd),col="black",lwd=1)
								lines(c(x/1.2,x*1.2),c(y-ysd,y-ysd),col="black",lwd=1)
								lines(c(x/1.2,x*1.2),c(y+ysd,y+ysd),col="black",lwd=1)
								for(m in 1:4) {
									resp.list.filtered <- resp.list[mask.list==m]
									conc.list.filtered <- conc.list[mask.list==m]
									if(m==1) color <- "gray";
									if(m==2) color <- "orange";
									if(m==3) color <- "skyblue1";
									if(m==4) color <- "pink";
									if(m==5) color <- "yellow2";
									if(m==1 && length(resp.list.filtered)>0) {
										points(resp.list.filtered~conc.list.filtered,cex=1.8,col=color,pch=4,lwd=1)
									}
								}
							}
						}
						lines(c(LEC,LEC),c(-10,10),col="green",lwd=2)
						lines(c(xmin,xmax),c(resp.cutoff,resp.cutoff),col="gray",lwd=2,lty=2)
						#if(mode=="agonist") lines(c(xmin,xmax),c(resp.cutoff/2,resp.cutoff/2),col="gray",lwd=1,lty=2)
						lines(c(xmin,xmax),c(0,0),col="gray",lwd=1)
						lines(c(xmin,xmax),c(100,100),col="gray",lwd=1)
						#---------------------------------------------------------------
						#
						# Add the Attagene OS line
						#
						#---------------------------------------------------------------
						atg.os.ac50 <- 1000000
						if(source=="Attagene") {
							sfilter <- filter[sample_id,]
							if(dim(sfilter)[1]==1) {
								atg.os.ac50 <- sfilter[1,1] # need to divide by 1000000?????????????????????????
								print(sfilter)
								lines(c(-50,2000)~c(sfilter[1,1]/1000000,sfilter[1,1]/1000000),lwd=4,col="orange")
								points(sfilter[1,2]/1000000,5,pch=25,bg="red",cex=2)
								points(sfilter[1,3]/1000000,10,pch=25,bg="orange",cex=2)
								points(sfilter[1,4]/1000000,15,pch=25,bg="green",cex=2)
							}
						}
						#---------------------------------------------------------------
						#
						# detect flare
						#
						#---------------------------------------------------------------
						flare <- F
						if(do.plate.qc) {
							qctemp <- plate.qc[is.element(plate.qc[,"sample_id"],sample_id),]
							if(dim(qctemp)[1]>0) {
								print("FLARE")
								for(q in 1:dim(qctemp)[1]) {
									conc <- qctemp[q,"conc"]/1000000
									if(conc>=(AC50/2)) flare <- T
									if(plot.flare) points(conc,qctemp[q,"norm_value"],pch=8,col="red",cex=2)
								}
								flare <- T
							}
						}
					}
					#---------------------------------------------------------------
					#
					# Plot the hill curve for the samples
					#
					#---------------------------------------------------------------
					hill.x <- as.numeric(strsplit(line.x,"\\|",perl=F)[[1]])/1000000
					hill.y <- as.numeric(strsplit(line.y,"\\|",perl=F)[[1]])
					if(mode=="antagonist") hill.y <- 100-hill.y
					lines(hill.y~hill.x,lwd=2)

					#---------------------------------------------------------------
					#
					# Plot the reference chemical curve
					#
					#---------------------------------------------------------------
					if(plot.rc==1) {
						conc.list.ref <- refcurve[,"Conc"]/1000000
						resp.list.ref <- refcurve[,"Resp"]
						points(resp.list.ref~conc.list.ref,pch=22,cex=0.8,bg="red",fg="red")
					}
					if(plot.rc==2) {
						for(k in 1:length(conc.list.rc) ){
							x <- conc.list.rc[k]
							y <- mean.list.rc[k]
							ysd <- sd.list.rc[k]
							lines(c(x,x),c(y-ysd,y+ysd),col="red",lwd=1)
							lines(c(x/1.2,x*1.2),c(y-ysd,y-ysd),col="red",lwd=1)
							lines(c(x/1.2,x*1.2),c(y+ysd,y+ysd),col="red",lwd=1)
							points(x,y,pch=22,cex=1,bg="red",fg="red",col="red")
						}
						lines(hill.y.rc~hill.x.rc,lwd=1,col="red")
						if(assay!="NVS_NR_bER") {
							lines(c(CI_L.rc,CI_U.rc),c(0.5*(T.rc+B.rc),0.5*(T.rc+B.rc)),lwd=2,col="red")
							lines(c(CI_L.rc,CI_L.rc),c(0.5*(T.rc+B.rc)-0.03*T.rc,0.5*(T.rc+B.rc)+0.03*T.rc),lwd=2,col="red")
							lines(c(CI_U.rc,CI_U.rc),c(0.5*(T.rc+B.rc)-0.03*T.rc,0.5*(T.rc+B.rc)+0.03*T.rc),lwd=2,col="red")
						}
					}

					#---------------------------------------------------------------
					#
					# Plot the viability data
					#
					#---------------------------------------------------------------
					do.via.filter <- F
					if(!is.na(viability)) {
						fit.params.one.via <- fit.params.via
						if(dim(fit.params.one.via)[1]>0) {
							AC50.via <- fit.params.one.via[1,"AC50"]
							B.via <- 100-fit.params.one.via[1,"B"]
							T.via <- 100-fit.params.one.via[1,"T"]
							CI_L.via <- fit.params.one.via[1,"AC50_2se_lwr"]/1000000
							CI_U.via <- fit.params.one.via[1,"AC50_2se_upr"]/1000000
							line.x.via <- as.character(fit.params.one.via[1,"line_x"])
							line.y.via <- as.character(fit.params.one.via[1,"line_y"])
							hill.x.via <- as.numeric(strsplit(line.x.via,"\\|",perl=F)[[1]])/1000000
							hill.y.via <- as.numeric(strsplit(line.y.via,"\\|",perl=F)[[1]])
							hill.y.via <- 100-hill.y.via

							test.cr.via <- cr.via[cr.via[,sample.col.name]==sample_id,]
							conc.list.via <- test.cr.via[,"conc"]/1000000
							resp.list.via <- test.cr.via[,"resp"]
							mask.list.via <- test.cr.via[,"mask"]
							conc.list.via <- conc.list.via[mask.list.via==0]
							resp.list.via <- resp.list.via[mask.list.via==0]
							via.med <- median(resp.list.via)
							if(!is.na(via.med)) {
								if(via.med>50) do.via.filter <- T
							}
							conc.list.test.via <- conc.list.via
							conc.list.test.via <- uniquify(conc.list.test.via)
							mean.list.test.via <- conc.list.test.via
							n.list.test.via <- conc.list.test.via
							n.list.test.via[] <- 0
							sd.list.test.via <- conc.list.test.via
							sd.list.test.via[] <- 0
							nsample.via <- 0
							for(k in 1:length(conc.list.test.via)) {
								cc <- conc.list.test.via[k]
								vals <- resp.list.via[conc.list.via==cc]
								if(length(vals)>nsample.via) nsample.via <- length(vals)
#								mean.list.test.via[k] <- mean(vals[vals<90])
#								sd.list.test.via[k] <- sd(vals[vals<90])
#								n.list.test.via[k] <- length(vals[vals<90])

								mean.list.test.via[k] <- mean(vals)
								sd.list.test.via[k] <- sd(vals)
								n.list.test.via[k] <- length(vals)
							}
							conc.list.test.via <- conc.list.test.via[n.list.test.via>0]
							mean.list.test.via <- mean.list.test.via[n.list.test.via>0]
							sd.list.test.via <- sd.list.test.via[n.list.test.via>0]

							via.color <- "gold"
							for(k in 1:length(conc.list.test.via) ){
								x <- conc.list.test.via[k]
								if(source=="Attagene") y <- mean.list.test.via[k]
								else y <- 100-mean.list.test.via[k]
								ysd <- sd.list.test.via[k]
								lines(c(x,x),c(y-ysd,y+ysd),col=via.color,lwd=1)
								lines(c(x/1.2,x*1.2),c(y-ysd,y-ysd),col=via.color,lwd=1)
								lines(c(x/1.2,x*1.2),c(y+ysd,y+ysd),col=via.color,lwd=1)
								points(x,y,pch=22,cex=1,bg=via.color,fg=via.color,col=via.color)
							}
						}
					}
					#---------------------------------------------------------------
					#
					# Derive values for the table
					#
					#---------------------------------------------------------------
					AC50.s <- format(AC50,digits=2)
					AC50.s <- paste(AC50.mod,AC50.s)
					rAC50.s <- format(rAC50,digits=2)
					if(overall.test=="FALSE") AC50.s <- "INACTIVE"

					rAC50.s <- format(rAC50,digits=2)
					rAC50.s <- paste(AC50.mod,rAC50.s)
					if(overall.test=="FALSE") rAC50.s <- "INACTIVE"

					LEC.s <- format(LEC,digits=2)
					LEC.s <- paste(AC50.mod,LEC.s)
					if(overall.test=="FALSE") LEC.s <- "INACTIVE"

					label <- paste("AC50:",AC50.s," (M)\n B:",format(B,digits=2)," T:",format(T,digits=2)," W:",format(W,digits=2),"\nWells per concentration: ",nsample,sep="")
					text(xmin,0.9*ymax,label,pos=4,cex=1)
					potency.s <- potency
					if(chem.set=="REF") {
						if(!is.na(modifier)) potency.s <- paste(potency," ",modifier)
						text(xmin,-10,potency.s,pos=4,cex=1)
					}
					if(AC50<1000000) {
						lines(c(CI_L,CI_U),c(0.5*(T+B),0.5*(T+B)),lwd=2)
						lines(c(CI_L,CI_L),c(0.5*(T+B)-0.03*T,0.5*(T+B)+0.03*T),lwd=2)
						lines(c(CI_U,CI_U),c(0.5*(T+B)-0.03*T,0.5*(T+B)+0.03*T),lwd=2)
					}
					#---------------------------------------------------------------
					#
					# make the overall call
					#
					#---------------------------------------------------------------
					activity.call <- "ACTIVE"
					activity.call.note <- ""
					if(overall.test=="FALSE") {
						activity.call <- "INACTIVE"
						activity.call.note <- "No significant points"
					}
					if(assay=="Tox21_ERa_LUC_BG1_Agonist") {
						temp <- FITS[is.element(FITS[,"sample_id"],sample_id),]
						alist <- c("Tox21_ERa_LUC_BG1_Antagonist_viability","Tox21_AutoFluor_HEK293_Cell_blue","Tox21_AutoFluor_HEK293_Cell_green","Tox21_AutoFluor_HEK293_Cell_red","Tox21_AutoFluor_HEK293_Media_blue","Tox21_AutoFluor_HEK293_Media_green","Tox21_AutoFluor_HEK293_Media_red","Tox21_AutoFluor_HEPG2_Cell_blue","Tox21_AutoFluor_HEPG2_Cell_green","Tox21_AutoFluor_HEPG2_Cell_red","Tox21_AutoFluor_HEPG2_Media_blue","Tox21_AutoFluor_HEPG2_Media_green","Tox21_AutoFluor_HEPG2_Media_red")
						clist <- c("gold","blue","green","red","blue","green","red","blue","green","red","blue","green","red")

						#
						# cytotoxicity
						#
						aa <- alist[1]
						temp.1 <- temp[is.element(temp[,"assay_name"],aa),]
						if(dim(temp.1)[1]==1) {
							aa.AC50 <- temp.1[,"AC50"]/1000000
							if(!is.na(aa.AC50)) {
								points(aa.AC50,-10,pch=22,cex=2,bg=clist[1],fg="black",col="black")
								do.via.filter <- T
								if(aa.AC50<1.0E-8) do.via.filter <- F
								cat("Cytotox check (AC50,cytotox limit): ",AC50,aa.AC50/5,"\n")
								if((activity.call=="ACTIVE") && (AC50>(aa.AC50/5)) && do.via.filter) {
									activity.call <- "SETASIDE"
									activity.call.note <- "Cytotoxicity<AC50"
								}
							}
						}
						#
						# autofluorescence
						#
						for(a1 in 2:length(alist)) {
							aa <- alist[a1]
							temp.1 <- temp[is.element(temp[,"assay_name"],aa),]
							if(dim(temp.1)[1]==1) {
								aa.AC50 <- temp.1[,"AC50"]/1000000
								if(!is.na(aa.AC50)) {
									points(aa.AC50,-10,pch=22,cex=2,bg=clist[a1],fg="black",col="black")
									if((activity.call=="ACTIVE") && ((aa.AC50/5)<AC50)) {
										activity.call <- "ACTIVE"
										activity.call.note <- "Possible Autofluorescence"
									}
								}
							}
						}
						#
						# T filter
						#
						if((activity.call=="ACTIVE") && (T<=resp.cutoff)) {
							activity.call <- "INACTIVE"
							activity.call.note <- "T < Response Cutoff"
							cat("======> call for T\n")
						}
						#
						# Flare reversing slope
						#
						if((activity.call=="ACTIVE")) {
							res <- lm(mean.list.test[1:10]~log10(conc.list.test[1:10]))
							slope <- summary(res)[[4]][2,1]
							pvalue <- summary(res)[[4]][2,4]
							if(T<80 && slope<0 && pvalue<0.001) {
								cat("reverse slope: ",T,slope,pvalue,"\n")
								activity.call <- "SETASIDE"
								activity.call.note <- "Reverse Slope"
							}
						}
						if((activity.call=="ACTIVE") && flare) {
							if(T<60) {
								activity.call <- "SETASIDE"
								activity.call.note <- "Activity may be due to Flare"
							}
							else activity.call.note <- "Flare is present"
						}
					}
					if(assay=="Tox21_ERa_BLA_Agonist_ratio" ) {
						temp <- FITS[is.element(FITS[,"sample_id"],sample_id),]
						alist <- c("Tox21_ERa_BLA_Antagonist_viability","Tox21_AutoFluor_HEK293_Cell_blue","Tox21_AutoFluor_HEK293_Cell_green","Tox21_AutoFluor_HEK293_Cell_red","Tox21_AutoFluor_HEK293_Media_blue","Tox21_AutoFluor_HEK293_Media_green","Tox21_AutoFluor_HEK293_Media_red","Tox21_AutoFluor_HEPG2_Cell_blue","Tox21_AutoFluor_HEPG2_Cell_green","Tox21_AutoFluor_HEPG2_Cell_red","Tox21_AutoFluor_HEPG2_Media_blue","Tox21_AutoFluor_HEPG2_Media_green","Tox21_AutoFluor_HEPG2_Media_red")
						clist <- c("gold","blue","green","red","blue","green","red","blue","green","red","blue","green","red")

						#
						# cytotoxicity
						#
						aa <- alist[1]
						temp.1 <- temp[is.element(temp[,"assay_name"],aa),]
						if(dim(temp.1)[1]==1) {
							aa.AC50 <- temp.1[,"AC50"]/1000000
							if(!is.na(aa.AC50)) {
								points(aa.AC50,-10,pch=22,cex=2,bg=clist[1],fg="black",col="black")
								do.via.filter <- T
								if(aa.AC50<1.0E-8) do.via.filter <- F
								cat("Cytotox check (AC50,cytotox limit): ",AC50,aa.AC50/5,"\n")
								if((activity.call=="ACTIVE") && (AC50>(aa.AC50/5)) && do.via.filter) {
									activity.call <- "SETASIDE"
									activity.call.note <- "Cytotoxicity<AC50"
								}
							}
						}
						#
						# autofluorescence
						#
						for(a1 in 2:length(alist)) {
							aa <- alist[a1]
							temp.1 <- temp[is.element(temp[,"assay_name"],aa),]
							if(dim(temp.1)[1]==1) {
								aa.AC50 <- temp.1[,"AC50"]/1000000
								if(!is.na(aa.AC50)) {
									points(aa.AC50,-10,pch=22,cex=2,bg=clist[a1],fg="black",col="black")
									if((activity.call=="ACTIVE") && ((aa.AC50/5)<AC50)) {
										activity.call <- "SETASIDE"
										activity.call.note <- "Possible Autofluorescence"
									}
								}
							}
						}
						#
						# T filter
						#
						if((activity.call=="ACTIVE") && (T<=resp.cutoff)) {
							activity.call <- "INACTIVE"
							activity.call.note <- "T < Response Cutoff"
						}
						if((activity.call=="ACTIVE") && flare) {
							activity.call <- "ACTIVE"
							activity.call.note <- "Flare is present"
						}
					}
					if(assay=="ATG_ERa_TRANS" || assay=="ATG_ERE_CIS") {
						if(!is.na(atg.os.ac50)) {
							if((activity.call=="ACTIVE") && (AC50>0.75*(atg.os.ac50/1000000))) {
								activity.call <- "INACTIVE"
								activity.call.note <- "Attagene OS Filter"
							}
						}
						cat("T: ",T," : ",resp.cutoff,"\n")
						#if((activity.call=="ACTIVE") && (T<=atg.cutoff.T)) {
						#	activity.call <- "INACTIVE"
						#	activity.call.note <- "T < Response Cutoff"
						#	cat("======> call for T\n")
						#}

          			}
					if((activity.call=="ACTIVE") && (AC50.mod=="<>")) {
						activity.call <- "SETASIDE"
						activity.call.note <- "Bad Fit"
					}
					if(grade=="F" || grade=="W" || grade=="X") {
						activity.call <- "DISCARD"
						activity.call.note <- "Unacceptable Sample Quality"
					}
					if((activity.call=="ACTIVE") && (grade=="B" || grade=="C" || grade=="D")) {
						activity.call <- "SETASIDE"
						activity.call.note <- "Note: Low Sample Purity"
					}
					#---------------------------------------------------------------
					#
					# Build the table
					#
					#---------------------------------------------------------------
					srow <- which.max(fits[,"sample_id"]==sample_id)
					fits[srow,"AnalyticalQCGrade"] <- grade
					fits[srow,"COA_Purity"] <- purity
					fits[srow,"ActivityCall"] <- activity.call
					fits[srow,"ActivityCallNote"] <- activity.call.note

					plot(1~1,xlim=c(0,100),ylim=c(0,100),xlab="",ylab="",lab=c(1,1,0),tck=0.001,xaxt="n",yaxt="n",pch=".")
					cex.val=1
					xoff <- -5
					ytop <- 100
					yd <- 6
					text(xoff,ytop,"Assay",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"CASRN",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Name",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,sample.col.name,cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					if(chem.set=="REF") text(xoff,ytop,"Expected Activity",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"AC50 (M)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"B (%E2)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"T (%E2)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"W",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Relative AC50 (M)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"LEC (M)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Emax (%E2)",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Min Conc Tested",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Max Conc Tested",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"R-squared",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Fit p-value",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Analytical QC",cex=cex.val,pos=4,font=2); ytop <- ytop-yd
					text(xoff,ytop,"Activity Call",cex=cex.val,pos=4,font=2); ytop <- ytop-yd

					if(AC50.mod!="==") rsq <- 0
					if(AC50.mod!="==") logAC50_pval <- 1
					xoff <- 33
					ytop <- 100
					lines(c(xoff,xoff),c(-10,110))
					text(xoff,ytop,assay,cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,casrn,cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,cname,cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,sample_id,cex=cex.val,pos=4); ytop <- ytop-yd
					if(chem.set=="REF") text(xoff,ytop,potency.s,cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(AC50.s," [",format(CI_L,digits=2),",",format(CI_U,digits=2),"]",sep=""),cex=cex.val,pos=4); ytop <- ytop-yd

					text(xoff,ytop,paste(format(B,digits=2)," [",format(B-CI_L.B,digits=2),",",format(B+CI_U.B,digits=2),"]",sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(format(T,digits=2)," [",format(T-CI_L.T,digits=2),",",format(T+CI_U.T,digits=2),"]",sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(format(W,digits=2)," [",format(W-CI_L.W,digits=2),",",format(W+CI_U.W,digits=2),"]",sep=""),cex=cex.val,pos=4); ytop <- ytop-yd

					text(xoff,ytop,paste(rAC50.s,sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(LEC.s,sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(format(Emax,digits=2),sep=""),cex=cex.val,pos=4); ytop <- ytop-yd

					text(xoff,ytop,paste(format(min(conc.list),digits=2),"M"),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(format(max(conc.list),digits=2),"M"),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,format(rsq,digits=2),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,format(logAC50_pval,digits=2),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(grade," : ",purity,"%",sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					text(xoff,ytop,paste(activity.call," ",activity.call.note,sep=""),cex=cex.val,pos=4); ytop <- ytop-yd
					if(chem.set=="REF") {
						ref.table[table_row,"Assay"] <- assay
						ref.table[table_row,"CASRN"] <- casrn
						ref.table[table_row,"Name"] <- cname
						ref.table[table_row,"sample_id"] <- sample_id
						ref.table[table_row,"Expected"] <- potency.s
						ref.table[table_row,"Order"] <- rc.order
						ref.table[table_row,"ActivityCall"] <- activity.call
						ref.table[table_row,"ActivityCallNote"] <- activity.call.note
						ref.table[table_row,"AC50"] <- paste(AC50.s," [",format(CI_L,digits=2),",",format(CI_U,digits=2),"]",sep="")
						ref.table[table_row,"B"] <- paste(format(B,digits=2)," [",format(B-CI_L.B,digits=2),",",format(B+CI_U.B,digits=2),"]",sep="")
						ref.table[table_row,"T"] <- paste(format(T,digits=2)," [",format(T-CI_L.T,digits=2),",",format(T+CI_U.T,digits=2),"]",sep="")
						ref.table[table_row,"W"] <- paste(format(W,digits=2)," [",format(W-CI_L.W,digits=2),",",format(W+CI_U.W,digits=2),"]",sep="")
						ref.table[table_row,"Emax"] <- paste(format(Emax,digits=2))
						ref.table[table_row,"SigPoints"] <- sig.points
						ref.table[table_row,"Analytical_QC_Grade"] <- grade
						ref.table[table_row,"COA_Purity"] <- purity
					}
					else {
						score.table[table_row,"Assay"] <- assay
						score.table[table_row,"CASRN"] <- casrn
						score.table[table_row,"Name"] <- cname
						score.table[table_row,"sample_id"] <- sample_id
						score.table[table_row,"ActivityCall"] <- activity.call
						score.table[table_row,"ActivityCallNote"] <- activity.call.note
						score.table[table_row,"AC50"] <- paste(AC50.s," [",format(CI_L,digits=2),",",format(CI_U,digits=2),"]",sep="")
						score.table[table_row,"B"] <- paste(format(B,digits=2)," [",format(B-CI_L.B,digits=2),",",format(B+CI_U.B,digits=2),"]",sep="")
						score.table[table_row,"T"] <- paste(format(T,digits=2)," [",format(T-CI_L.T,digits=2),",",format(T+CI_U.T,digits=2),"]",sep="")
						score.table[table_row,"W"] <- paste(format(W,digits=2)," [",format(W-CI_L.W,digits=2),",",format(W+CI_U.W,digits=2),"]",sep="")
						score.table[table_row,"Emax"] <- paste(format(Emax,digits=2))
						score.table[table_row,"SigPoints"] <- sig.points
						score.table[table_row,"Analytical_QC_Grade"] <- grade
						score.table[table_row,"COA_Purity"] <- purity
					}
					table_row <- table_row+1
					ytop <- 100-yd/2
					for(k in 1:20) {
						lines(c(-10,110),c(ytop,ytop))
						ytop <- ytop - yd
					}
					mean.ref.sd.list <- mean(ref.sd.list[!is.na(ref.sd.list)])
					cat("\n+++++ ",casrn," : ",cname," : SD MEAN: ",format(mean.ref.sd.list,digits=3),"\n\n",sep="")
					s <- paste(assay,",",casrn,",",cname,",",format(mean.ref.sd.list,digits=3),"\n",sep="")
					cat(s,file=file.refsd,sep="",append=T)

					if(!to.file) browser()
					

				}
			}
		}
	}

	#---------------------------------------------------------------
	#
	# write hte summary tables
	#
	#---------------------------------------------------------------
	if(chem.set=="REF") {
		fname.table <- paste("tables/",assay,"_",chem.set,"_SCORE_",date.string,".csv",sep="")
		write.csv(ref.table[1:(table_row-1),],fname.table,row.names=F)
	}
	fname.table <- paste("tables/",assay,"_",chem.set,"_fit_params_",date.string,".csv",sep="")
	write.csv(fits,fname.table,row.names=F)
    if(to.file) dev.off()
}
#--------------------------------------------------------------------------------------
#
# load data required for the plotting hte reference chemicals
#
#--------------------------------------------------------------------------------------
load.assay <- function(assay="Tox21_ERa_BLA_Agonist_ratio",date.string="2012_07_27") {
	cat("==========================================================================\n")
	cat("load.assay ",assay,"\n")
	cat("==========================================================================\n")
	CR <<- read.table(file=paste("Data_",date.string,"/ToxCast_",assay,"_conc_response_",date.string,".txt",sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
	cat("loaded Conc-response data\n");
	flush.console()
}
#--------------------------------------------------------------------------------------
#
# load data required for the plotting hte reference chemicals
#
#--------------------------------------------------------------------------------------
load.viability <- function(viability,date.string) {
	cat("==========================================================================\n")
	cat("load.viability ",viability,"\n")
	cat("==========================================================================\n")
	file <- paste("Data_",date.string,"/ToxCast_",viability,"_conc_response_",date.string,".txt",sep="")
	CR.VIA <<- read.table(file=file,sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
	cat("loaded viability Conc-response data\n")
	print(dim(CR.VIA))
	flush.console()
}
#--------------------------------------------------------------------------------------
#
# load data required for the plotting the reference chemicals
#
#--------------------------------------------------------------------------------------
load.fits <- function(source="NCGC",date.string) {
	cat("==========================================================================\n")
	cat("load.fits\n")
	cat("==========================================================================\n")
	FITS <<- read.table(file=paste("Data_",date.string,"/ToxCast_assay_fits_",source,"_",date.string,".txt",sep=""),sep="\t",header=TRUE,check.names=F,stringsAsFactors=F,quote="",comment.char="")
	cat("loaded fits\n");
	flush.console()
}
#--------------------------------------------------------------------------------------
#
# cycle through the reference chemicals
#
#--------------------------------------------------------------------------------------
HillFitLine <- function(xmin,xmax,AC50,T,B,W) {
        min.conc <- xmin
        max.conc <- xmax
        log3.min.conc <- log(min.conc,base=3)
        log3.max.conc <- log(max.conc,base=3)
        log3.x <- pretty(c(log3.min.conc,log3.max.conc),n=1000)
        x <- 3^log3.x

        y <- rep(0,length(x))
        for(i in 1:length(x))
        {
          val <- T - (T - B)/(1 + (x[i]/AC50)^(W))
          y[i] <- val
        }
        line <- list(x=x,y=y)
        return(line)
}
