/*********************************************************************/
/*Program: PFDA_BirthWeight.sas                                      */
/*Purpose: BMD modeling for BirthWeight                              */
/*Author: H. Ru                                                      */
/*Last Updated: 04/17/2024                                           */
/*********************************************************************/



/********************************************Macros************************************************/

proc format; 
  	value  betaf 	1="g/ng/ml"
					2="g/IQR (ng/ml)"
					3="g/SD (ng/ml)"
					4="g/2SD (ng/ml)"
					11="g/ln(ng/ml)"
					12="g/IQR ln(ng/ml)"
					13="g/SD ln(ng/ml)"
					14="g/2SD ln(ng/ml)"
					21="g/log10(ng/ml)"
					22="g/IQR log10(ng/ml)"
					23="g/SD log10(ng/ml)"
					24="g/2SD log10(ng/ml)"
					31="g/log2(ng/ml)"
					32="g/IQR log2(ng/ml)"
					33="g/SD log2(ng/ml)"
					34="g/2SD log2(ng/ml)"
					43="g/SD ln(1+ng/ml)"
					44="g/SD ln(1+ng/ml)"
					;
	value exposurespreadf 2="IQR ng/ml"     
					3="SD ng/ml"
					4="2SD ng/ml"
					5="GSD ng/ml"
					6="33-67QR ng/ml"
					7="20-80QR ng/ml"
					12="IQR ln(ng/ml)"
					13="SD ln(ng/ml)"
					14="2SD ln(ng/ml)"
					22="IQR log10(ng/ml)"
					23="SD log10(ng/ml)"
					24="2SD log10(ng/ml)"
					32="IQR log2(ng/ml)"
					33="SD log2(ng/ml)"
					34="2SD log2(ng/ml)"
					
					;
run;



/*%pfda_bw1: macro for bmdl with exact percentage*/
%macro pfda_bw1(Study=, expo_m=, expo_spread1=, expo_spread2=, expo_unit=, 
				beta=, beta_lcl=, beta_ucl=, beta_unit=, 
				controlMean=, controlSd=, extraRisk=, Ntotal=, LWtotal=, outbmddata=);

%if &expo_unit=2 %then %do; /*Median and IQR*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/1.3489795; 
%end;

%if &expo_unit=3 %then %do; /*Mean and SD*/
%let omega=1+(&expo_spread1/&expo_m)**2;
%let expo_mu=log(&expo_m/sqrt(&omega));
%let expo_sigma=log(exp(sqrt(log(&omega)))); 
%end;

%if &expo_unit=4 %then %do; /*Mean and 2SD*/
%let omega=1+(&expo_spread1/(2*&expo_m))**2;
%let expo_mu=log(&expo_m/sqrt(&omega));
%let expo_sigma=log(exp(sqrt(log(&omega)))); 
%end;

%if &expo_unit=5 %then %do; /*GM and GSD*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread1);
%end;

%if &expo_unit=6 %then %do; /*Median and q33-q67*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/0.879826331; 
%end;

%if &expo_unit=7 %then %do; /*Median and q20-q80*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/1.683242467; 
%end;

%if &expo_unit=13 %then %do; /*GM and SD ln(ng/ml)*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(exp(&expo_spread1)); 
%end;

%if &expo_unit=14 %then %do; /*GM and 2SD ln(ng/ml)*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(exp(&expo_spread1/2)); 
%end;



%if &beta_unit=1 or &beta_unit=11 or &beta_unit=21 or &beta_unit=31 %then %do;
%let beta0=&beta;
%let beta0_lcl=&beta_lcl;
%let beta0_ucl=&beta_ucl;
%end;

%if &beta_unit=2 or &beta_unit=12 or &beta_unit=22 or &beta_unit=32 %then %do;
%let IQR=&expo_spread2-&expo_spread1;
%let beta0=&beta/(&IQR);
%let beta0_lcl=&beta_lcl/(&IQR);
%let beta0_ucl=&beta_ucl/(&IQR);
%end;

%if &beta_unit=13 %then %do;
%let beta0=&beta/(&expo_spread1);
%let beta0_lcl=&beta_lcl/(&expo_spread1);
%let beta0_ucl=&beta_ucl/(&expo_spread1);
%end;

%if &beta_unit=14 %then %do;
%let beta0=&beta/(&expo_spread1);
%let beta0_lcl=&beta_lcl/(&expo_spread1);
%let beta0_ucl=&beta_ucl/(&expo_spread1);
%end;

%if &beta_unit=43 %then %do; 
%let logsd=log((1+&expo_spread2)/(1+&expo_spread1))/1.3489795; 
%let beta0=&beta/(&logsd);
%let beta0_lcl=&beta_lcl/(&logsd);
%let beta0_ucl=&beta_ucl/(&logsd);
%put &logsd;
%end;

%if &beta_unit=44 %then %do; 
%let logsd=(log(&expo_spread2/&expo_spread1))/1.3489795; 
%let beta0=&beta/(&logsd);
%let beta0_lcl=&beta_lcl/(&logsd);
%let beta0_ucl=&beta_ucl/(&logsd);
%put &logsd;
%end;


data temp;
    do i = 1 to 6;  
		Study="&study";
		q=(i*10+15)/100;
        x=quantile('LOGNORMAL', q, &expo_mu, &expo_sigma);  
		%if &beta_unit<10 %then %do;
		y2=x*&beta0;
		ylcl2=x*&beta0_lcl;
		yucl2=x*&beta0_ucl;
		lnx=log(x);
		y3=y2; 
		ylcl3=ylcl2;
		yucl3=yucl2;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<20 %then %do;
		y0=log(x)*&beta0;
		ylcl=log(x)*&beta0_lcl;
		yucl=log(x)*&beta0_ucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<30 %then %do;
		y0=log10(x)*&beta0;
		ylcl=log10(x)*&beta0_lcl;
		yucl=log10(x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<40 %then %do;
		y0=log2(x)*&beta0;
		ylcl=log2(x)*&beta0_lcl;
		yucl=log2(x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %do;
		y0=log(1+x)*&beta0;
		ylcl=log(1+x)*&beta0_lcl;
		yucl=log(1+x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);

		output;
		%end;
    end;
	
run;

ods output ParameterEstimates=estimate;
proc reg data = temp;
%if &beta_unit<10 %then %do;
   model y2 = lnx;
   model ylcl2=lnx;
   model yucl2=lnx;

%end;
%else %do;
   model y0 = x;
   model ylcl=x;
   model yucl=x;
   %if &beta_unit>20 %then %do; 
	   model y2 = lnx;  		
	   model ylcl2=lnx;			
	   model yucl2=lnx;
   %end;

%end;

   model y3 = lnxp1;
   model ylcl3=lnxp1;
   model yucl3=lnxp1;

run;
quit;

data est1; 
	Study="&study";
	set estimate;
	drop Model;
	if Variable="Intercept" then delete;
	run;

proc transpose data=est1 out=est2 ;
    by Study;
    id Dependent;
    var Estimate;
run;

data est3; 
	Study="&study";
	set estimate;
	drop Model;
	if Variable="Intercept";
	run;

proc transpose data=est3 out=est4 ;
    by Study;
    id Dependent;
    var Estimate;
run;

data est4; set est4; 
rename 
	%if  &beta_unit<10 %then %do;
	y2=y2int ylcl2=ylcl2int yucl2=yucl2int
	%end;
	%else %if &beta_unit<20 %then %do;
	y0=y0int ylcl=ylclint yucl=yuclint 
	%end;
	%else %do;
	y2=y2int ylcl2=ylcl2int yucl2=yucl2int y0=y0int ylcl=ylclint yucl=yuclint 
	%end;
	y3=y3int ylcl3=ylcl3int yucl3=yucl3int
;
run;

data est6; set est2;
	rename 
	%if  &beta_unit<10 %then %do;
	y2=y2beta ylcl2=ylcl2beta yucl2=yucl2beta
	%end;
	%else %if &beta_unit<20 %then %do;
	y0=y0beta ylcl=ylclbeta yucl=yuclbeta 
	%end;
	%else %do; 
	y2=y2beta ylcl2=ylcl2beta yucl2=yucl2beta y0=y0beta ylcl=ylclbeta yucl=yuclbeta 
	%end;
	y3=y3beta ylcl3=ylcl3beta yucl3=yucl3beta
;
run;

data est5; merge est6 est4; by study; run;

data temp2; merge temp est5; by study; 
run;

data outreexdata;
	length Study $60;
	retain Study expo_m expo_spread1 expo_spread2 expo_unit beta beta_lcl beta_ucl beta_unit
			beta0 beta0_lcl beta0_ucl expo_mu expo_sigma y0 y0_se ylcl yucl y2 y2_se ylcl2 yucl2;
	set est2;
	drop _name_ _label_ y3 ylcl3 yucl3;

	expo_m=&expo_m;
	expo_spread1=&expo_spread1;
	expo_spread2=&expo_spread2; 
	expo_unit=&expo_unit;

	beta=&beta;
	beta_lcl=&beta_lcl;
	beta_ucl=&beta_ucl;
	beta_unit=&beta_unit;

	expo_mu=&expo_mu;
	expo_sigma=&expo_sigma;

	beta0=&beta0;
	beta0_lcl=&beta0_lcl;
	beta0_ucl=&beta0_ucl;

	if y0=. then do; 
		y0=beta0;
		ylcl=beta0_lcl;
		yucl=beta0_ucl;
	end;
	if y2=. then do; 
		y2=beta0;
		ylcl2=beta0_lcl;
		yucl2=beta0_ucl;
	end;

	y0_se=(yucl-ylcl)/(probit(0.975)-probit(0.025)); 
	y2_se=(yucl2-ylcl2)/(probit(0.975)-probit(0.025));

	rename expo_m=ExpoMMGM expo_spread1=ExpoSpread1 expo_spread2=ExpoSpread2 expo_unit=ExpoUnit 
		expo_mu=ExpoDistMu expo_sigma=ExpoDistSigma 
		beta=BetaOrig beta_lcl=BetaOrigLCL beta_ucl=BetaOrigUCL beta_unit=BetaUnit
		beta0=BetaResc beta0_lcl=BetaRescLCL beta0_ucl=BetaRescUCL
		y0=BetaNgml y0_se=BetaNgmlSE ylcl=BetaNgmlLCL yucl=BetaNgmlUCL 
		y2=BetaLnNgml y2_se=BetaLnNgmlSE ylcl2=BetaLnNgmlLCL yucl2=BetaLnNgmlUCL
		;
	format expo_unit exposurespreadf. beta_unit betaf. expo_m expo_spread1 expo_spread2 beta beta_lcl beta_ucl 
			beta0 beta0_lcl beta0_ucl expo_mu expo_sigma y0 y0_se ylcl yucl y2 y2_se ylcl2 yucl2  6.2;
	run;

data &outbmddata;
	retain Study ExpoMMGM ExpoSpread1 ExpoSpread2 ExpoUnit 
		ExpoDistMu ExpoDistSigma 
		BetaOrig BetaOrigLCL BetaOrigUCL BetaUnit
		BetaResc BetaRescLCL BetaRescUCL
		BetaNgml BetaNgmlSE BetaNgmlLCL BetaNgmlUCL 
		BetaLnNgml BetaLnNgmlSE BetaLnNgmlLCL BetaLnNgmlUCL
		NTotal LWtotal ControlMean ControlSD ProbC ProbD TargetMean BetaCI BetaLclOneside Background 
		ControlMean1 Bmd Bmdl BmdlCdf;
	set outreexdata;

	do BackGround=0.19;

		NTotal=&Ntotal;
		LWtotal=&LWtotal;
		ControlMean=&controlMean;
		ControlSD=&controlSd;
		ProbC=&LWtotal/&Ntotal;  
		ProbD=&extraRisk *(1-ProbC)+ProbC; 

		TargetMean=2500-&controlSd *probit(ProbD); 

		BetaLclOneside=BetaNgml-probit(0.95)*BetaNgmlSE;
		ControlMean1=&ControlMean-BackGround*BetaNgml;

		Bmd=((TargetMean-controlMean1 )/(BetaNgml));
		Bmdl=((TargetMean-controlMean1 )/(BetaLclOneside));

		BmdlCdf = cdf('LOGNORMAL', bmdl, ExpoDistMu, ExpoDistSigma); 

		BetaCI=strip(round(BetaNgml, 0.1))||" ("||strip(round(BetaNgmlLCL, 0.1))||", "||strip(round(BetaNgmlUCL, 0.1))||") ";
	
	output;
	end;

	format NTotal LWtotal 10.0 ControlMean ControlSD ControlMean1  6.1 ProbC percent8.2 ProbD percent8.2 
			TargetMean 6.1 Background BetaLclOneside  Bmd Bmdl  6.2 BmdlCdf  percent8.2;


	run;

data &outbmddata; set &outbmddata;
	expo=strip(ExpoMMGM)||" ("||strip(ExpoSpread1)||"-"||strip(ExpoSpread2)||")";
	dist="("||strip(round(ExpoDistMu, .01))||", "||strip(round(ExpoDistSigma, .01))||")";
	beta=strip(BetaOrig)||" ("||strip(BetaOrigLCL)||", "||strip(BetaOrigUCL)||")";
	run;


%mend pfda_bw1;



/*%pfda_bw2: macro for bmdl with alternative tail probability*/
%macro pfda_bw2(Study=, expo_m=, expo_spread1=, expo_spread2=, expo_unit=, 
				beta=, beta_lcl=, beta_ucl=, beta_unit=, 
				controlMean=, controlSd=, extraRisk=, Ntotal=, LWtotal=, outbmddata=);

%if &expo_unit=2 %then %do; /*Median and IQR*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/1.3489795; 
%end;

%if &expo_unit=3 %then %do; /*Mean and SD*/
%let omega=1+(&expo_spread1/&expo_m)**2;
%let expo_mu=log(&expo_m/sqrt(&omega));
%let expo_sigma=log(exp(sqrt(log(&omega)))); 
%end;

%if &expo_unit=4 %then %do; /*Mean and 2SD*/
%let omega=1+(&expo_spread1/(2*&expo_m))**2;
%let expo_mu=log(&expo_m/sqrt(&omega));
%let expo_sigma=log(exp(sqrt(log(&omega)))); 
%end;

%if &expo_unit=5 %then %do; /*GM and GSD*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread1);
%end;

%if &expo_unit=6 %then %do; /*Median and q33-q67*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/0.879826331; 
%end;

%if &expo_unit=7 %then %do; /*Median and q20-q80*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(&expo_spread2/&expo_spread1)/1.683242467; 
%end;

%if &expo_unit=13 %then %do; /*GM and SD ln(ng/ml)*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(exp(&expo_spread1)); 
%end;

%if &expo_unit=14 %then %do; /*GM and 2SD ln(ng/ml)*/
%let expo_mu=log(&expo_m);
%let expo_sigma=log(exp(&expo_spread1/2)); 
%end;



%if &beta_unit=1 or &beta_unit=11 or &beta_unit=21 or &beta_unit=31 %then %do;
%let beta0=&beta;
%let beta0_lcl=&beta_lcl;
%let beta0_ucl=&beta_ucl;
%end;

%if &beta_unit=2 or &beta_unit=12 or &beta_unit=22 or &beta_unit=32 %then %do;
%let IQR=&expo_spread2-&expo_spread1;
%let beta0=&beta/(&IQR);
%let beta0_lcl=&beta_lcl/(&IQR);
%let beta0_ucl=&beta_ucl/(&IQR);
%end;

%if &beta_unit=13 %then %do;
%let beta0=&beta/(&expo_spread1);
%let beta0_lcl=&beta_lcl/(&expo_spread1);
%let beta0_ucl=&beta_ucl/(&expo_spread1);
%end;

%if &beta_unit=14 %then %do;
%let beta0=&beta/(&expo_spread1);
%let beta0_lcl=&beta_lcl/(&expo_spread1);
%let beta0_ucl=&beta_ucl/(&expo_spread1);
%end;

%if &beta_unit=43 %then %do; 
%let logsd=log((1+&expo_spread2)/(1+&expo_spread1))/1.3489795; 
%let beta0=&beta/(&logsd);
%let beta0_lcl=&beta_lcl/(&logsd);
%let beta0_ucl=&beta_ucl/(&logsd);
%put &logsd;
%end;

%if &beta_unit=44 %then %do; 
%let logsd=(log(&expo_spread2/&expo_spread1))/1.3489795; 
%let beta0=&beta/(&logsd);
%let beta0_lcl=&beta_lcl/(&logsd);
%let beta0_ucl=&beta_ucl/(&logsd);
%put &logsd;
%end;


data temp;
    do i = 1 to 6;  
		Study="&study";
		q=(i*10+15)/100;
        x=quantile('LOGNORMAL', q, &expo_mu, &expo_sigma);  
		%if &beta_unit<10 %then %do;
		y2=x*&beta0;
		ylcl2=x*&beta0_lcl;
		yucl2=x*&beta0_ucl;
		lnx=log(x);
		y3=y2; 
		ylcl3=ylcl2;
		yucl3=yucl2;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<20 %then %do;
		y0=log(x)*&beta0;
		ylcl=log(x)*&beta0_lcl;
		yucl=log(x)*&beta0_ucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<30 %then %do;
		y0=log10(x)*&beta0;
		ylcl=log10(x)*&beta0_lcl;
		yucl=log10(x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %if &beta_unit<40 %then %do;
		y0=log2(x)*&beta0;
		ylcl=log2(x)*&beta0_lcl;
		yucl=log2(x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);
		output;
		%end;
		%else %do;
		y0=log(1+x)*&beta0;
		ylcl=log(1+x)*&beta0_lcl;
		yucl=log(1+x)*&beta0_ucl;
		lnx=log(x);
		y2=y0;
		ylcl2=ylcl;
		yucl2=yucl;
		y3=y0;
		ylcl3=ylcl;
		yucl3=yucl;
		lnxp1=log(1+x);

		output;
		%end;
    end;
	
run;

ods output ParameterEstimates=estimate;
proc reg data = temp;
%if &beta_unit<10 %then %do;
   model y2 = lnx;
   model ylcl2=lnx;
   model yucl2=lnx;

%end;
%else %do;
   model y0 = x;
   model ylcl=x;
   model yucl=x;
   %if &beta_unit>20 %then %do; 
	   model y2 = lnx;  		
	   model ylcl2=lnx;			
	   model yucl2=lnx;
   %end;

%end;

   model y3 = lnxp1;
   model ylcl3=lnxp1;
   model yucl3=lnxp1;

run;
quit;

data est1; 
	Study="&study";
	set estimate;
	drop Model;
	if Variable="Intercept" then delete;
	run;

proc transpose data=est1 out=est2 ;
    by Study;
    id Dependent;
    var Estimate;
run;

data est3; 
	Study="&study";
	set estimate;
	drop Model;
	if Variable="Intercept";
	run;

proc transpose data=est3 out=est4 ;
    by Study;
    id Dependent;
    var Estimate;
run;

data est4; set est4; 
rename 
	%if  &beta_unit<10 %then %do;
	y2=y2int ylcl2=ylcl2int yucl2=yucl2int
	%end;
	%else %if &beta_unit<20 %then %do;
	y0=y0int ylcl=ylclint yucl=yuclint 
	%end;
	%else %do;
	y2=y2int ylcl2=ylcl2int yucl2=yucl2int y0=y0int ylcl=ylclint yucl=yuclint 
	%end;
	y3=y3int ylcl3=ylcl3int yucl3=yucl3int
;
run;

data est6; set est2;
	rename 
	%if  &beta_unit<10 %then %do;
	y2=y2beta ylcl2=ylcl2beta yucl2=yucl2beta
	%end;
	%else %if &beta_unit<20 %then %do;
	y0=y0beta ylcl=ylclbeta yucl=yuclbeta 
	%end;
	%else %do; 
	y2=y2beta ylcl2=ylcl2beta yucl2=yucl2beta y0=y0beta ylcl=ylclbeta yucl=yuclbeta 
	%end;
	y3=y3beta ylcl3=ylcl3beta yucl3=yucl3beta
;
run;

data est5; merge est6 est4; by study; run;

data temp2; merge temp est5; by study; 
run;

data outreexdata;
	length Study $60;
	retain Study expo_m expo_spread1 expo_spread2 expo_unit beta beta_lcl beta_ucl beta_unit
			beta0 beta0_lcl beta0_ucl expo_mu expo_sigma y0 y0_se ylcl yucl y2 y2_se ylcl2 yucl2;
	set est2;
	drop _name_ _label_ y3 ylcl3 yucl3;

	expo_m=&expo_m;
	expo_spread1=&expo_spread1;
	expo_spread2=&expo_spread2; 
	expo_unit=&expo_unit;

	beta=&beta;
	beta_lcl=&beta_lcl;
	beta_ucl=&beta_ucl;
	beta_unit=&beta_unit;

	expo_mu=&expo_mu;
	expo_sigma=&expo_sigma;

	beta0=&beta0;
	beta0_lcl=&beta0_lcl;
	beta0_ucl=&beta0_ucl;

	if y0=. then do; 
		y0=beta0;
		ylcl=beta0_lcl;
		yucl=beta0_ucl;
	end;
	if y2=. then do; 
		y2=beta0;
		ylcl2=beta0_lcl;
		yucl2=beta0_ucl;
	end;

	y0_se=(yucl-ylcl)/(probit(0.975)-probit(0.025)); 
	y2_se=(yucl2-ylcl2)/(probit(0.975)-probit(0.025));

	rename expo_m=ExpoMMGM expo_spread1=ExpoSpread1 expo_spread2=ExpoSpread2 expo_unit=ExpoUnit 
		expo_mu=ExpoDistMu expo_sigma=ExpoDistSigma 
		beta=BetaOrig beta_lcl=BetaOrigLCL beta_ucl=BetaOrigUCL beta_unit=BetaUnit
		beta0=BetaResc beta0_lcl=BetaRescLCL beta0_ucl=BetaRescUCL
		y0=BetaNgml y0_se=BetaNgmlSE ylcl=BetaNgmlLCL yucl=BetaNgmlUCL 
		y2=BetaLnNgml y2_se=BetaLnNgmlSE ylcl2=BetaLnNgmlLCL yucl2=BetaLnNgmlUCL
		;
	format expo_unit exposurespreadf. beta_unit betaf. expo_m expo_spread1 expo_spread2 beta beta_lcl beta_ucl 
			beta0 beta0_lcl beta0_ucl expo_mu expo_sigma y0 y0_se ylcl yucl y2 y2_se ylcl2 yucl2  6.2;
	run;

data &outbmddata;
	retain Study ExpoMMGM ExpoSpread1 ExpoSpread2 ExpoUnit 
		ExpoDistMu ExpoDistSigma 
		BetaOrig BetaOrigLCL BetaOrigUCL BetaUnit
		BetaResc BetaRescLCL BetaRescUCL
		BetaNgml BetaNgmlSE BetaNgmlLCL BetaNgmlUCL 
		BetaLnNgml BetaLnNgmlSE BetaLnNgmlLCL BetaLnNgmlUCL
		NTotal LWtotal ControlMean ControlSD ProbC1 ProbD1 TargetMean1 BetaCI BetaLclOneside Background 
		ControlMean1 Bmd Bmdl BmdlCdf;
	set outreexdata;

	do BackGround=0.19;

		NTotal=&Ntotal;
		LWtotal=&LWtotal;
		ControlMean=&controlMean;
		ControlSD=&controlSd;

		BetaLclOneside=BetaNgml-probit(0.95)*BetaNgmlSE;
		ControlMean1=&ControlMean-BackGround*BetaNgml;

		ProbC1=cdf('NORMAL', 2500, ControlMean1, &controlSd);
		ProbD1=&extraRisk *(1-ProbC1)+ProbC1;
		TargetMean1=2500-&controlSd *probit(ProbD1); 

		Bmd=((TargetMean1-controlMean1 )/(BetaNgml));
		Bmdl=((TargetMean1-controlMean1 )/(BetaLclOneside));

		BmdlCdf = cdf('LOGNORMAL', bmdl, ExpoDistMu, ExpoDistSigma); 

		BetaCI=strip(round(BetaNgml, 0.1))||" ("||strip(round(BetaNgmlLCL, 0.1))||", "||strip(round(BetaNgmlUCL, 0.1))||") ";
		
	output;
	end;

	format NTotal LWtotal 10.0 ControlMean ControlSD ControlMean1  6.1 ProbC1 percent8.2 ProbD1 percent8.2 
			TargetMean1 6.1 Background BetaLclOneside  Bmd Bmdl  6.2 BmdlCdf  percent8.2;


	run;

data &outbmddata; set &outbmddata;
	expo=strip(ExpoMMGM)||" ("||strip(ExpoSpread1)||"-"||strip(ExpoSpread2)||")";
	dist="("||strip(round(ExpoDistMu, .01))||", "||strip(round(ExpoDistSigma, .01))||")";
	beta=strip(BetaOrig)||" ("||strip(BetaOrigLCL)||", "||strip(BetaOrigUCL)||")";
	run;


%mend pfda_bw2;



/*******************bmdl with exact percentage****************/
%pfda_bw1(Study=Valvi 2017, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-41, beta_lcl=-102, beta_ucl=18, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd1);

%pfda_bw1(Study=Valvi 2017 Boy, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-44, beta_lcl=-133, beta_ucl=44, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd2);

%pfda_bw1(Study=Valvi 2017 Girl, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-28, beta_lcl=-110, beta_ucl=54, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd3);

%pfda_bw1(Study=Lenters 2016, expo_m=0.245, expo_spread1=1.397, expo_spread2=., expo_unit=14, 
	beta=-43.93, beta_lcl=-104.83, beta_ucl=16.97, beta_unit=14, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd4);

%pfda_bw1(Study=Luo 2021, expo_m=0.48, expo_spread1=0.34, expo_spread2=0.70, expo_unit=2, 
	beta=-96.76, beta_lcl=-178.01, beta_ucl=-15.5, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd5);

%pfda_bw1(Study=Wikstrom 2020, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-58, beta_lcl=-103, beta_ucl=-13, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd6);

%pfda_bw1(Study=Wikstrom 2020 Boy, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-47, beta_lcl=-112, beta_ucl=17, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd7);

%pfda_bw1(Study=Wikstrom 2020 Girl, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-69, beta_lcl=-133, beta_ucl=-6, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd8);

%pfda_bw1(Study=Yao 2021, expo_m=0.55, expo_spread1=0.37, expo_spread2=0.74, expo_unit=2, 
	beta=-46.34, beta_lcl=-131.14, beta_ucl=38.46, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd9);

data bmd;
	set bmd1 bmd2 bmd3 bmd4 bmd5 bmd6 bmd7 bmd8 bmd9 ;
	where background=0.19; 
	run;

/*table for appendix*/
proc report data=bmd HEADLINE HEADSKIP CENTER STYLE(REPORT)={JUST=CENTER} SPLIT='~' nowd 
          SPANROWS LS=256;
COLUMNS study expo dist beta BetaUnit betaci ControlMean1 BetaNgmlSE BetaLclOneside BMD BMDL ;

DEFINE study/ DISPLAY   "Study"   STYLE(COLUMN) = {JUST = L CellWidth=6%} order=data;

DEFINE expo/ DISPLAY   "Exposure Median (IQR or 33-37QR)"   STYLE(COLUMN) = {JUST = L CellWidth=8%} ;
DEFINE dist/ DISPLAY   "Exposure Distribution"   STYLE(COLUMN) = {JUST = L CellWidth=8%} ;

DEFINE Beta/ DISPLAY   "Reported Beta (95%CI)"   STYLE(COLUMN) = {JUST = L CellWidth=10%} ;
DEFINE BetaUnit/ DISPLAY   "Unit"   STYLE(COLUMN) = {JUST = L CellWidth=7%} F=betaf.;
DEFINE BetaCI / DISPLAY   "Re-expressed Beta (95%CI)"   STYLE(COLUMN) = {JUST = L CellWidth=10%} ;
DEFINE ControlMean1/ DISPLAY   "Intercept"   STYLE(COLUMN) = {JUST = L CellWidth=5%} F=6.1;
DEFINE BetaNgmlSE/ DISPLAY   "Beta SE"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2 ;
DEFINE BetaLclOneside/ DISPLAY   "Beta 95% onesided LC"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2 ;

DEFINE Bmd/ DISPLAY   "BMD"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2;
DEFINE Bmdl/ DISPLAY   "BMDL"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2;

run;


/*******************bmdl with alternative tail probability****************/

%pfda_bw2(Study=Valvi 2017, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-41, beta_lcl=-102, beta_ucl=18, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd11);

%pfda_bw2(Study=Valvi 2017 Boy, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-44, beta_lcl=-133, beta_ucl=44, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd12);

%pfda_bw2(Study=Valvi 2017 Girl, expo_m=0.28, expo_spread1=0.22, expo_spread2=0.38, expo_unit=2, 
	beta=-28, beta_lcl=-110, beta_ucl=54, beta_unit=31, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd13);

%pfda_bw2(Study=Lenters 2016, expo_m=0.245, expo_spread1=1.397, expo_spread2=., expo_unit=14, 
	beta=-43.93, beta_lcl=-104.83, beta_ucl=16.97, beta_unit=14, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd14);

%pfda_bw2(Study=Luo 2021, expo_m=0.48, expo_spread1=0.34, expo_spread2=0.70, expo_unit=2, 
	beta=-96.76, beta_lcl=-178.01, beta_ucl=-15.5, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd15);

%pfda_bw2(Study=Wikstrom 2020, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-58, beta_lcl=-103, beta_ucl=-13, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd16);

%pfda_bw2(Study=Wikstrom 2020 Boy, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-47, beta_lcl=-112, beta_ucl=17, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd17);

%pfda_bw2(Study=Wikstrom 2020 Girl, expo_m=0.26, expo_spread1=0.19, expo_spread2=0.34, expo_unit=2, 
	beta=-69, beta_lcl=-133, beta_ucl=-6, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd18);

%pfda_bw2(Study=Yao 2021, expo_m=0.55, expo_spread1=0.37, expo_spread2=0.74, expo_unit=2, 
	beta=-46.34, beta_lcl=-131.14, beta_ucl=38.46, beta_unit=11, 
	Ntotal=3791712, LWtotal=313752, controlMean=3261.64, controlSd=590.66, extraRisk=0.05, outbmddata=bmd19);


data bmd_al;
	set bmd11 bmd12 bmd13 bmd14 bmd15 bmd16 bmd17 bmd18 bmd19 ;
	where background=0.19; 
	run;

/*table for appendix*/
proc report data=bmd_al HEADLINE HEADSKIP CENTER STYLE(REPORT)={JUST=CENTER} SPLIT='~' nowd 
          SPANROWS LS=256;
COLUMNS study expo dist beta BetaUnit betaci ControlMean1 BetaNgmlSE BetaLclOneside ProbC1 BMD BMDL ;

DEFINE study/ DISPLAY   "Study"   STYLE(COLUMN) = {JUST = L CellWidth=6%} order=data;

DEFINE expo/ DISPLAY   "Exposure Median (IQR or 33-37QR)"   STYLE(COLUMN) = {JUST = L CellWidth=8%} ;
DEFINE dist/ DISPLAY   "Exposure Distribution"   STYLE(COLUMN) = {JUST = L CellWidth=8%} ;

DEFINE Beta/ DISPLAY   "Reported Beta (95%CI)"   STYLE(COLUMN) = {JUST = L CellWidth=10%} ;
DEFINE BetaUnit/ DISPLAY   "Unit"   STYLE(COLUMN) = {JUST = L CellWidth=7%} F=betaf.;
DEFINE BetaCI / DISPLAY   "Re-expressed Beta (95%CI)"   STYLE(COLUMN) = {JUST = L CellWidth=10%} ;
DEFINE ControlMean1/ DISPLAY   "Intercept"   STYLE(COLUMN) = {JUST = L CellWidth=5%} F=6.1;
DEFINE BetaNgmlSE/ DISPLAY   "Beta SE"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2 ;
DEFINE BetaLclOneside/ DISPLAY   "Beta 95% onesided LC"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2 ;
define ProbC1/DISPLAY "P0" STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2 F=percent8.2;
DEFINE Bmd/ DISPLAY   "BMD"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2;
DEFINE Bmdl/ DISPLAY   "BMDL"   STYLE(COLUMN) = {JUST = L CellWidth=4%} F=6.2;

run;

