import arcpy

class Toolbox(object):
    def __init__(self):
        """Define the toolbox (the name of the toolbox is the name of the
        .pyt file)."""
        self.label = "Toolbox"
        self.alias = ""
        self.tools = [Tool]

class Tool(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Geospatial Hotspot Analyses"
        self.description = ""
        self.canRunInBackground = False

    def getParameterInfo(self):
        """Define parameter definitions"""
        params = [
            arcpy.Parameter(
                displayName="Workspace",
                name="workspace",
                datatype="DEWorkspace",
                parameterType="Required",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Result Name",
                name="result_name",
                datatype="GPString",
                parameterType="Required",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Input Feature",
                name="input_shapefile",
                datatype="DEFeatureClass",
                parameterType="Required",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Value Field",
                name="value_field",
                datatype="Field",
                parameterType="Required",
                direction="Input"
            ),
        ]
        return params

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""
        workspace = parameters[0].valueAsText
        result_name = parameters[1].valueAsText
        input_shapefile = parameters[2].valueAsText
        value_field = parameters[3].valueAsText

        # Set the workspace path. The code will create results in this geodatabase 
        arcpy.env.workspace = workspace
        arcpy.env.overwriteOutput = True #lowercase for python3

        # HotSpots_stats tool
        arcpy.HotSpots_stats(
            Input_Feature_Class=input_shapefile,
            Input_Field=value_field,
            Output_Feature_Class=workspace + r"/" + result_name,
            Conceptualization_of_Spatial_Relationships="CONTIGUITY_EDGES_CORNERS",
            Distance_Method="EUCLIDEAN_DISTANCE",
            Standardization="NONE",
            Distance_Band_or_Threshold_Distance="",
            Self_Potential_Field="",
            Weights_Matrix_File="",
            Apply_False_Discovery_Rate__FDR__Correction="NO_FDR"
        )

        # Select the census tracts that have greater or equal than 95 for the Getis-Ord Gi* statistical testing significance level
        arcpy.AddField_management(workspace + r"/" + result_name, "CI95", "TEXT")
        query_expression = '"Y" if !Gi_Bin! > 1 else "N"'
        arcpy.CalculateField_management(workspace + r"/" + result_name, "CI95", query_expression, "PYTHON_9.3")

        # Add fields
        arcpy.AddField_management(workspace + r"/" + result_name, "Top20", "TEXT")
        arcpy.AddField_management(workspace + r"/" + result_name, "PercentileValue_20", "TEXT")

        # Get the values of the value field
        values = [row[0] for row in arcpy.da.SearchCursor(workspace + r"/" + result_name, value_field)]

        # Calculate the 80th percentile value
        percentile_value = sorted(values)[int(len(values) * 0.8)]

        # Calculate the "Top20" field based on the percentile
        query_expression = """'Y' if !{}! >= {} else 'N'""".format(value_field, percentile_value)
        arcpy.CalculateField_management(workspace + r"/" + result_name, "Top20", query_expression, "PYTHON_9.3")

        # Set the percentile value
        arcpy.CalculateField_management(workspace + r"/" + result_name, "PercentileValue_20", str(percentile_value), "PYTHON_9.3")