#####################################################################################################
## Palate Fusion Model                                                                             ##
##                                                                                                 ##
## National Center for Computational Toxicology                                                    ##
## United States Environmental Protection Agency                                                   ##
## http://www.epa.gov/ncct/v-embryo/                                                               ##
##                                                                                                 ##
## Author:  M. Shane Hutson                                                                        ##
## Initial Date:  09-17-2012                                                                       ##
## Revision Date: 08-15-2015                                                                       ##
##                                                                                                 ##
## CC3D Version: 3.7.3                                                                             ##
#####################################################################################################
#****************************************************************************************************
# Define constants for referencing Cell Types
MEDIUM = 0
WALL = 1
MESENCH_N = 2
MESENCH_O = 3
MESENCH_EMT = 4
MATRIX = 5
EPI_N = 6
EPI_O = 7
EPI_ME = 8
BM = 9
PERI_N = 10
PERI_O = 11
PERI_ME = 12
PERI_Na = 13
PERI_Oa = 14
PERI_MEa = 15
TJ = 16
TJm = 17
# and for referring to lattice dimensions
XX = 200
YY = 400
ZZ = 1
#*****************************************************************************************************
def retrieveToxCastData(assayDict, chemicalName, casNumber, concentration):
    pass

#*****************************************************************************************************
def configureSimulation(sim,usePI,useHL,useBM):
   global XX, YY, ZZ
   import CompuCellSetup
   from XMLUtils import ElementCC3D

   cc3d=ElementCC3D("CompuCell3D")
   potts=cc3d.ElementCC3D("Potts")
   potts.ElementCC3D("Dimensions",{"x":XX,"y":YY,"z":ZZ})
   potts.ElementCC3D("Anneal",{},0)
   potts.ElementCC3D("Steps",{},6001)
   #potts.ElementCC3D("Temperature",{},4)
   potts.ElementCC3D("FluctuationAmplitude",{},4)
   potts.ElementCC3D("Flip2DimRatio",{},1)
   potts.ElementCC3D("NeighborOrder",{},1.75)
   potts.ElementCC3D("Boundary_x",{},"NoFlux")
   potts.ElementCC3D("Boundary_y",{},"NoFlux")
   potts.ElementCC3D("Boundary_z",{},"NoFlux")
   if useHL: 
       potts.ElementCC3D("LatticeType",{},"Hexagonal")
   potts.ElementCC3D("RandomSeed",{},2001)
   
   
   player=cc3d.ElementCC3D("Plugin",{"Name":"PlayerSettings"})
   player.ElementCC3D("Project2D", {"XYProj":"0"})
   player.ElementCC3D("InitialProjection", {"Projection":"xy"})
   player.ElementCC3D("VisualControl", {"ScreenshotFrequency":"100","ScreenUpdateFrequency":"10"})
   player.ElementCC3D("Cell",{"Type":"0","Color":"#000000"})    #Medium
   player.ElementCC3D("Cell",{"Type":"1","Color":"#FFFFFF"})    #Wall
   player.ElementCC3D("Cell",{"Type":"2","Color":"#99FFFF"})    #Mesench_N
   player.ElementCC3D("Cell",{"Type":"3","Color":"#99FF99"})    #Mesench_O
   player.ElementCC3D("Cell",{"Type":"4","Color":"#FF0000"})    #Mesench_EMT
   player.ElementCC3D("Cell",{"Type":"5","Color":"#FFCC66"})    #Matrix
   player.ElementCC3D("Cell",{"Type":"6","Color":"#3399FF"})    #Epi_N
   player.ElementCC3D("Cell",{"Type":"7","Color":"#3399CC"})    #Epi_O
   player.ElementCC3D("Cell",{"Type":"8","Color":"#3333FF"})    #Epi_ME
   player.ElementCC3D("Cell",{"Type":"9","Color":"#990000"})    #BM
   player.ElementCC3D("Cell",{"Type":"10","Color":"#9966FF"})    #Peri_N
   player.ElementCC3D("Cell",{"Type":"11","Color":"#9966CC"})    #Peri_O
   player.ElementCC3D("Cell",{"Type":"12","Color":"#9900FF"})    #Peri_ME
   player.ElementCC3D("Cell",{"Type":"13","Color":"#FF66FF"})    #Peri_Na
   player.ElementCC3D("Cell",{"Type":"14","Color":"#FF66CC"})    #Peri_Oa
   player.ElementCC3D("Cell",{"Type":"15","Color":"#FF00FF"})    #Peri_MEa
   player.ElementCC3D("Cell",{"Type":"16","Color":"#990099"})    #TJ
   player.ElementCC3D("Cell",{"Type":"17","Color":"#990066"})    #TJm
   player.ElementCC3D("Border",{"BorderColor":"FFCC66","BorderOn":"true"})  #I'm not sure this works for setting color
   
   cellType=cc3d.ElementCC3D("Plugin",{"Name":"CellType"})
   cellType.ElementCC3D("CellType", {"TypeName":"Medium", "TypeId":"0"})
   cellType.ElementCC3D("CellType", {"TypeName":"Wall", "TypeId":"1", "Freeze":""})
   cellType.ElementCC3D("CellType", {"TypeName":"Mesench_N", "TypeId":"2"})
   cellType.ElementCC3D("CellType", {"TypeName":"Mesench_O", "TypeId":"3"})
   cellType.ElementCC3D("CellType", {"TypeName":"Mesench_EMT", "TypeId":"4"})
   cellType.ElementCC3D("CellType", {"TypeName":"Matrix", "TypeId":"5"})
   cellType.ElementCC3D("CellType", {"TypeName":"Epi_N", "TypeId":"6"})
   cellType.ElementCC3D("CellType", {"TypeName":"Epi_O", "TypeId":"7"})
   cellType.ElementCC3D("CellType", {"TypeName":"Epi_ME", "TypeId":"8"})
   cellType.ElementCC3D("CellType", {"TypeName":"BaseMemb", "TypeId":"9"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_N", "TypeId":"10"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_O", "TypeId":"11"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_ME", "TypeId":"12"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_Na", "TypeId":"13"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_Oa", "TypeId":"14"})
   cellType.ElementCC3D("CellType", {"TypeName":"Peri_MEa", "TypeId":"15"})
   cellType.ElementCC3D("CellType", {"TypeName":"TJ", "TypeId":"16"})
   cellType.ElementCC3D("CellType", {"TypeName":"TJm", "TypeId":"17"})

   volumeLocalFlex=cc3d.ElementCC3D("Plugin",{"Name":"VolumeLocalFlex"})

   surfaceLocalFlex=cc3d.ElementCC3D("Plugin",{"Name":"SurfaceLocalFlex"})

   jMeEp = 6
   jMeBM = 6
   if useBM:
       jMeEp = 16
       jMeBM = 6
   contact=cc3d.ElementCC3D("Plugin",{"Name":"ContactCompartment"})
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Medium"},0)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Wall"},0)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Mesench_N"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Mesench_O"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Mesench_EMT"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Matrix"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Epi_N"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Epi_O"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Epi_ME"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"BaseMemb"},100)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_N"},30)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_O"},30)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_ME"},30)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_Na"},6)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_Oa"},6)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"Peri_MEa"},6)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"TJ"},12)
   contact.ElementCC3D("Energy", {"Type1":"Medium", "Type2":"TJm"},30)

   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Wall"},0)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Mesench_N"},3)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Mesench_O"},3)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Mesench_EMT"},3)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Matrix"},1)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Epi_N"},4)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Epi_O"},4)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Epi_ME"},4)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"BaseMemb"},6) # was 4
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_N"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_Na"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_Oa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"Peri_MEa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"TJ"},2)
   contact.ElementCC3D("Energy", {"Type1":"Wall", "Type2":"TJm"},20)

   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Mesench_N"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Mesench_O"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Mesench_EMT"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Matrix"},-1) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Epi_N"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Epi_O"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Epi_ME"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"BaseMemb"},jMeBM) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_N"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_O"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_ME"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"TJ"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_N", "Type2":"TJm"},30)

   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Mesench_O"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Mesench_EMT"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Matrix"},-1) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Epi_N"},jMeEp)  # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Epi_O"},jMeEp)  # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Epi_ME"},jMeEp)  # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"BaseMemb"},jMeBM) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_N"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_O"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_ME"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"TJ"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_O", "Type2":"TJm"},30)

   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Mesench_EMT"},3) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Matrix"},-1) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Epi_N"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Epi_O"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Epi_ME"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"BaseMemb"},jMeBM) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_N"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_O"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_ME"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"TJ"},30)
   contact.ElementCC3D("Energy", {"Type1":"Mesench_EMT", "Type2":"TJm"},30)
   
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Matrix"},0) # was 2
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Epi_N"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Epi_O"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Epi_ME"},jMeEp) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"BaseMemb"},jMeBM) # was 6
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_N"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_O"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_ME"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"TJ"},30)
   contact.ElementCC3D("Energy", {"Type1":"Matrix", "Type2":"TJm"},30)

   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Epi_N"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Epi_O"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Epi_ME"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"BaseMemb"},6) # was 4
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_N"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"TJ"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_N", "Type2":"TJm"},10)

   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Epi_O"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Epi_ME"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"BaseMemb"},6) # was 4
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_N"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"TJ"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_O", "Type2":"TJm"},10)
   
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Epi_ME"},4)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"BaseMemb"},6) # was 4
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_N"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"TJ"},10)
   contact.ElementCC3D("Energy", {"Type1":"Epi_ME", "Type2":"TJm"},10)

   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"BaseMemb"},8) # was 4
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_N"},20) # was 10
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_O"},20) # was 10
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_ME"},20) # was 10
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"TJ"},10)
   contact.ElementCC3D("Energy", {"Type1":"BaseMemb", "Type2":"TJm"},10)

   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_N"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"TJ"},6)
   contact.ElementCC3D("Energy", {"Type1":"Peri_N", "Type2":"TJm"},6) ## was 6

   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"Peri_O"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"TJ"},6)
   contact.ElementCC3D("Energy", {"Type1":"Peri_O", "Type2":"TJm"},6) ## was 6

   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"Peri_ME"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"Peri_Na"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"Peri_Oa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"Peri_MEa"},30)
   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"TJ"},6)
   contact.ElementCC3D("Energy", {"Type1":"Peri_ME", "Type2":"TJm"},6)  ## was 6

   contact.ElementCC3D("Energy", {"Type1":"Peri_Na", "Type2":"Peri_Na"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Na", "Type2":"Peri_Oa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Na", "Type2":"Peri_MEa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Na", "Type2":"TJ"},20)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Na", "Type2":"TJm"},20)  ## was 6

   contact.ElementCC3D("Energy", {"Type1":"Peri_Oa", "Type2":"Peri_Oa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Oa", "Type2":"Peri_MEa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Oa", "Type2":"TJ"},20)
   contact.ElementCC3D("Energy", {"Type1":"Peri_Oa", "Type2":"TJm"},20)  ## was 6

   contact.ElementCC3D("Energy", {"Type1":"Peri_MEa", "Type2":"Peri_MEa"},10)
   contact.ElementCC3D("Energy", {"Type1":"Peri_MEa", "Type2":"TJ"},20)
   contact.ElementCC3D("Energy", {"Type1":"Peri_MEa", "Type2":"TJm"},20)  ## was 6

   contact.ElementCC3D("Energy", {"Type1":"TJ", "Type2":"TJ"},2)
   contact.ElementCC3D("Energy", {"Type1":"TJ", "Type2":"TJm"},2)   ## was 2

   contact.ElementCC3D("Energy", {"Type1":"TJm", "Type2":"TJm"},2)   ## was 2

   # only includes those elements that could be in same cell (cluster)
   contact.ElementCC3D("InternalEnergy", {"Type1":"Peri_N", "Type2":"Peri_Na"},2)
   contact.ElementCC3D("InternalEnergy", {"Type1":"Peri_O", "Type2":"Peri_Oa"},2)
   contact.ElementCC3D("InternalEnergy", {"Type1":"Peri_ME", "Type2":"Peri_MEa"},2)

   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_N"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_O"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_ME"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_Na"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_Oa"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"Peri_MEa"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_N"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_O"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_ME"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_Na"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_Oa"},4)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"Peri_MEa"},4)

   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"TJ"},50)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJ", "Type2":"TJm"},50)
   contact.ElementCC3D("InternalEnergy", {"Type1":"TJm", "Type2":"TJm"},50)

   contact.ElementCC3D("InternalEnergy", {"Type1":"Matrix", "Type2":"Matrix"},1)
   contact.ElementCC3D("InternalEnergy", {"Type1":"BaseMemb", "Type2":"BaseMemb"},6) # was 4

   contact.ElementCC3D("NeighborOrder",{},1.75)
   
   centerOfMass=cc3d.ElementCC3D("Plugin",{"Name":"CenterOfMass"})
   neighborTracker=cc3d.ElementCC3D("Plugin",{"Name":"NeighborTracker"})
   momentOfInertia=cc3d.ElementCC3D("Plugin",{"Name":"MomentOfInertia"})
   lengthConstraint = cc3d.ElementCC3D("Plugin",{"Name":"LengthConstraint"})
   boundaryPixelTracker=cc3d.ElementCC3D("Plugin",{"Name":"BoundaryPixelTracker"})
    
   # setup focalPointPlasticity for polarized elements of periderm cells (can add to this later) 
   focalPointPlasticity=cc3d.ElementCC3D("Plugin",{"Name":"FocalPointPlasticity"})
   focalPointPlasticity.ElementCC3D("Local",{})
   # TJ-TJ links INTERNAL TO A CLUSTER (i.e. INTRACELLULAR)
    #fppTjTjINT=focalPointPlasticity.ElementCC3D("InternalParameters", {"Type1":"TJ", "Type2":"TJ"})
    #fppTjTjINT.ElementCC3D("Lambda",{},0.001)
    #fppTjTjINT.ElementCC3D("ActivationEnergy",{},-50.0)
    #fppTjTjINT.ElementCC3D("TargetDistance",{},3.0)
    #fppTjTjINT.ElementCC3D("MaxDistance",{},20.0)
    #fppTjTjINT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)
    #focalPointPlasticity.ElementCC3D("NeighborOrder",{},1)
   # PERI_N-TJ links INTERNAL TO A CLUSTER (i.e. INTRACELLULAR)
    #fppP1TjINT=focalPointPlasticity.ElementCC3D("InternalParameters", {"Type1":"Peri_N", "Type2":"TJ"})
    #fppP1TjINT.ElementCC3D("Lambda",{},0.001)
    #fppP1TjINT.ElementCC3D("ActivationEnergy",{},-50.0)
    #fppP1TjINT.ElementCC3D("TargetDistance",{},3.0)
    #fppP1TjINT.ElementCC3D("MaxDistance",{},20.0)
    #fppP1TjINT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},2)
    #focalPointPlasticity.ElementCC3D("NeighborOrder",{},1)
   # PERI_ME-TJ links INTERNAL TO A CLUSTER (i.e. INTRACELLULAR)
    #fppP2TjINT=focalPointPlasticity.ElementCC3D("InternalParameters", {"Type1":"Peri_ME", "Type2":"TJ"})
    #fppP2TjINT.ElementCC3D("Lambda",{},0.001)
    #fppP2TjINT.ElementCC3D("ActivationEnergy",{},-50.0)
    #fppP2TjINT.ElementCC3D("TargetDistance",{},3.0)
    #fppP2TjINT.ElementCC3D("MaxDistance",{},20.0)
    #fppP2TjINT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},2)
    #focalPointPlasticity.ElementCC3D("NeighborOrder",{},1)
   # TJ-TJ links BETWEEN CLUSTERS (i.e. EXTRACELLULAR)
   fppTjTjEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"TJ", "Type2":"TJ"})
   fppTjTjEXT.ElementCC3D("Lambda",{},10)
   fppTjTjEXT.ElementCC3D("ActivationEnergy",{},-50.0)
   fppTjTjEXT.ElementCC3D("TargetDistance",{},3.0)
   fppTjTjEXT.ElementCC3D("MaxDistance",{},6.0)
   fppTjTjEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)
   fppTjmTjmEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"TJm", "Type2":"TJm"})
   fppTjmTjmEXT.ElementCC3D("Lambda",{},10)
   fppTjmTjmEXT.ElementCC3D("ActivationEnergy",{},-50.0)
   fppTjmTjmEXT.ElementCC3D("TargetDistance",{},3.0)
   fppTjmTjmEXT.ElementCC3D("MaxDistance",{},6.0)
   fppTjmTjmEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)
   # MESENCH-MESENCH links BETWEEN CLUSTERS
   if False:
       fppMoMoEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_O", "Type2":"Mesench_O"})
       fppMoMoEXT.ElementCC3D("Lambda",{},0)
       fppMoMoEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMoMoEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMoMoEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMoMoEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
       fppMoMnEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_O", "Type2":"Mesench_N"})
       fppMoMnEXT.ElementCC3D("Lambda",{},0)
       fppMoMnEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMoMnEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMoMnEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMoMnEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
       fppMoMeEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_O", "Type2":"Mesench_EMT"})
       fppMoMeEXT.ElementCC3D("Lambda",{},0)
       fppMoMeEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMoMeEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMoMeEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMoMeEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
       fppMnMnEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_N", "Type2":"Mesench_N"})
       fppMnMnEXT.ElementCC3D("Lambda",{},0)
       fppMnMnEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMnMnEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMnMnEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMnMnEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
       fppMnMeEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_N", "Type2":"Mesench_EMT"})
       fppMnMeEXT.ElementCC3D("Lambda",{},0)
       fppMnMeEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMnMeEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMnMeEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMnMeEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
       fppMeMeEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"Mesench_EMT", "Type2":"Mesench_EMT"})
       fppMeMeEXT.ElementCC3D("Lambda",{},0)
       fppMeMeEXT.ElementCC3D("ActivationEnergy",{},-50.0)
       fppMeMeEXT.ElementCC3D("TargetDistance",{},0.0)
       fppMeMeEXT.ElementCC3D("MaxDistance",{},1000.0)
       fppMeMeEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},6)
   # BM-BM links BETWEEN CLUSTERS (i.e. INTER-"CELLULAR")
   fppBmBmEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"BaseMemb", "Type2":"BaseMemb"})
   fppBmBmEXT.ElementCC3D("Lambda",{},40.0)
   fppBmBmEXT.ElementCC3D("ActivationEnergy",{},-100.0)  # was -50
   fppBmBmEXT.ElementCC3D("TargetDistance",{},2.0)
   fppBmBmEXT.ElementCC3D("MaxDistance",{},4.0)
   fppBmBmEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)
   # BM-BM links BETWEEN CLUSTERS (i.e. INTRA-"CELLULAR")
   fppBmBmINT=focalPointPlasticity.ElementCC3D("InternalParameters", {"Type1":"BaseMemb", "Type2":"BaseMemb"})
   fppBmBmINT.ElementCC3D("Lambda",{},40.0)
   fppBmBmINT.ElementCC3D("ActivationEnergy",{},-10.0)  # was -50
   fppBmBmINT.ElementCC3D("TargetDistance",{},2.0)
   fppBmBmINT.ElementCC3D("MaxDistance",{},4.0)
   fppBmBmINT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)
#fppP2TjINT=focalPointPlasticity.ElementCC3D("InternalParameters", {"Type1":"Peri_ME", "Type2":"TJ"})
#fppP2TjINT.ElementCC3D("Lambda",{},0.001)
#fppP2TjINT.ElementCC3D("ActivationEnergy",{},-50.0)
#fppP2TjINT.ElementCC3D("TargetDistance",{},3.0)
#fppP2TjINT.ElementCC3D("MaxDistance",{},20.0)
#fppP2TjINT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},2)
#focalPointPlasticity.ElementCC3D("NeighborOrder",{},1)

#fppTjTjmEXT=focalPointPlasticity.ElementCC3D("Parameters", {"Type1":"TJ", "Type2":"TJm"})
#  fppTjTjmEXT.ElementCC3D("Lambda",{},10)
#  fppTjTjmEXT.ElementCC3D("ActivationEnergy",{},-50.0)
#  fppTjTjmEXT.ElementCC3D("TargetDistance",{},3.0)
#  fppTjTjmEXT.ElementCC3D("MaxDistance",{},6.0)
#  fppTjTjmEXT.ElementCC3D("MaxNumberOfJunctions",{"NeighborOrder":"1"},1)


   flexDiffSolver=cc3d.ElementCC3D("Steppable",{"Type":"DiffusionSolverFE"}) # add "_OpenCL" to use GPU (if it works on your system)
   fieldList=["SHH","FGF10","FGF7","BMP2","BMP4","NOGGIN","TGFb3","EGF"]
   if useBM: fieldList.append("MMP")
   for field in fieldList:
       diffusionField=flexDiffSolver.ElementCC3D("DiffusionField")
       diffusionData=diffusionField.ElementCC3D("DiffusionData")
       diffusionData.ElementCC3D("FieldName",{},field)
       if field=="SHH":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           # SHH "diffusion" is actually reiterated uptake and secretion,
           # so no "diffusion" through Medium or Matrix
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Medium"},0.0)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Matrix"},0.0)
       elif field=="FGF10":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Matrix"},0.1)
       elif field=="FGF7":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.001)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Matrix"},0.1)
       elif field=="BMP2":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
       elif field=="BMP4":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Matrix"},0.2)
       elif field=="NOGGIN":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
       elif field=="TGFb3":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.1)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
       elif field=="EGF":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.1)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)
       elif useBM and field=="MMP":
           diffusionData.ElementCC3D("GlobalDiffusionConstant",{},0.5)
           diffusionData.ElementCC3D("GlobalDecayConstant",{},0.005)
           diffusionData.ElementCC3D("DiffusionCoefficient",{"CellType":"Wall"},0.0)

      
   # use a separate Secretion Plugin so secretion properties can be changed on a cell-by-cell basis
   secretion=cc3d.ElementCC3D("Plugin",{"Name":"Secretion"})
   for field in fieldList:
       secretionField=secretion.ElementCC3D("Field",{"Name":field,"ExtraTimesPerMC":"0"})

   if usePI:
    pifInitializer=cc3d.ElementCC3D("Steppable",{"Type":"PIFInitializer"})
    #pifInitializer.ElementCC3D("PIFName",{},"/disk2/home/shutson/CC3D/Palate/palateShelf_200x300two.piff")
    #pifInitializer.ElementCC3D("PIFName",{},"/Users/hutsonms/CompuCell3D/Palate/palateShelf_200x300one.piff")
    #pifInitializer.ElementCC3D("PIFName",{},"/Users/hutsonms/CompuCell3D/Palate/palateShelf_200x300two.piff")
    pifInitializer.ElementCC3D("PIFName",{},"/Users/hutsonms/CompuCell3D/Palate/palateShelf_200x400MNOtwoB.piff")
    #pifInitializer.ElementCC3D("PIFName",{},"/Users/hutsonms/CompuCell3D/Palate/palateShelf_200x400two.piff")
    #pifInitializer.ElementCC3D("PIFName",{},"/Users/hutsonms/CompuCell3D/Palate/palateShelf_200x300three.piff")

   CompuCellSetup.setSimulationXMLDescription(cc3d)
   
#####################################################################################################################

import sys
from os import environ
from os import getcwd
import string
from datetime import *
import time
import inspect
import math


sys.path.append(environ["PYTHON_MODULE_PATH"])
# define some options here
usePIFInitializer = True
useHexLattice = True
useExplicitBM = True
useTUNLField = False
useBrdUField = False

import CompuCellSetup

sim,simthread = CompuCellSetup.getCoreSimulationObjects()
configureSimulation(sim,usePIFInitializer,useHexLattice,useExplicitBM)
print "Finished configureSimulation call"
CompuCellSetup.initializeSimulationObjects(sim,simthread)
print "Finished initializeSimulationObjects"

import CompuCell
print "Finished import CompuCell"

#ADD ATTRIBUTES AND CREATE EXTRA PLAYER FIELDS HERE
#pyAttributeAdder,listAdder=CompuCellSetup.attachListToCells(sim)
#print "Finished attaching (unpopulated) list to all cells"
pyAttributeAdderDict,dictAdder=CompuCellSetup.attachDictionaryToCells(sim)
print "Finished attaching (unpopulated) dictionary to all cells"
dim=sim.getPotts().getCellFieldG().getDim()
if useTUNLField:
   TUNLField=simthread.createScalarFieldCellLevelPy("TUNL") #initializing TUNL field for tracking apoptosis
if useBrdUField:
   BrdUField=simthread.createScalarFieldCellLevelPy("BrdU") #initializing BrdU field for tracking mitoses
   
#ADD PYTHON STEPPABLES HERE (INCLUDING THOSE THAT UPDATE EXTRA PLAYER FIELDS)
steppableRegistry=CompuCellSetup.getSteppableRegistry()
print "Finished getSteppableRegistry"

if not usePIFInitializer:
    from palate_steppables import SetInitial2DGeometry
    setInitial2DGeometry=SetInitial2DGeometry(_simulator=sim,_frequency=10000)
    setInitial2DGeometry.useHexLattice = useHexLattice
    # LEFT palate shelf
    setInitial2DGeometry.specifyTissueButton(_x0=-0.0625*0.75*YY, _y0=0.5*YY, 
        _rMesench=0.125*0.75*YY, _drEpi=5, _drPeri=4, _polDirection=0, _nCellsList=[35,20,10], 
        _cellTypeList=[[MESENCH_N,MESENCH_O],[EPI_N,EPI_O,EPI_ME],[PERI_N,PERI_O,PERI_ME]])
    # RIGHT palate shelf (reverse N and O designations so N is on same side for both buttons
    setInitial2DGeometry.specifyTissueButton(_x0=XX+0.0625*0.75*YY, _y0=0.5*YY, 
        _rMesench=0.125*0.75*YY, _drEpi=5, _drPeri=4, _polDirection=1.0*math.pi, _nCellsList=[35,20,10], 
        _cellTypeList=[[MESENCH_O,MESENCH_N],[EPI_O,EPI_N,EPI_ME],[PERI_O,PERI_N,PERI_ME]])
    # tongue (old format of this function)
    #setInitial2DGeometry.specifyTissueButton(_x0=0.5*XX, _y0=-0.8*XX, _rMesench=XX, _drEpi=8, _drPeri=4, _polDirection=0.5*math.pi, _nCellsList=[280,60,20], _cellTypeList=[MESENCH_EMT,EPI_N,EPI_N,PERI_N,PERI_N])
    # test
    #setInitial2DGeometry.specifyTissueButton(_x0=0.5*XX, _y0=0.5*YY, _rMesench=0.125*YY, _drEpi=8, _drPeri=4, _polDirection=-0.5*math.pi, _nCellsList=[240,120,40])
    
    
    setInitial2DGeometry.SetSpecialMedialEdge(True)
    setInitial2DGeometry.IncludeMirrorShelf(True)
    steppableRegistry.registerSteppable(setInitial2DGeometry)
    print "Finished importing SetInitial2DGeometry"
else:
    print "Initializing geometry from .pif file"

from palate_steppables import PalateModelCellAttributes
palateModelCellAttributes=PalateModelCellAttributes(_simulator=sim,_frequency=10)
steppableRegistry.registerSteppable(palateModelCellAttributes)
print "Finished importing PalateModelCellAttributes"

from palate_steppables import PalateModelCellBehaviors
palateModelCellBehaviors=PalateModelCellBehaviors(_simulator=sim,_frequency=10)
palateModelCellBehaviors.useHexLattice = useHexLattice
palateModelCellBehaviors.initTargetVolumeToAvgForEachCellType(False)
palateModelCellBehaviors.initTargetSurfaceBasedOnVolume(False)
palateModelCellBehaviors.useRandomVolumeForMesenchCells = True
palateModelCellBehaviors.reduceMotilityOfPeriCells = True
palateModelCellBehaviors.mcsAtWhichToDecreasePeriCellMotility = 300  # was 1800
if useExplicitBM:
    palateModelCellBehaviors.mcsAtWhichToIncreasePeriCellMotility = 5010  #was 1810
else:
    palateModelCellBehaviors.mcsAtWhichToIncreasePeriCellMotility = 1810  #was 1810
palateModelCellBehaviors.allowApoptosisTime = 200  #was 500
palateModelCellBehaviors.allowEmtTime = 200 #was 500
palateModelCellBehaviors.timeToStopGrowth = 3000 # set very large (>10000) to essentially never stop unless signals fade)
palateModelCellBehaviors.apoptosisTimeConstant = 600/5
palateModelCellBehaviors.useAspectRatioForMitosis = False
palateModelCellBehaviors.useBM = useExplicitBM
#palateModelCellBehaviors.doNotDivideTypeList = [MEDIUM,WALL,MATRIX,BM,TJ,TJm,EPI_ME,PERI_ME,PERI_MEa]  #try turning off mitosis in the MEE
#palateModelCellBehaviors.TGFbetaFlag = True   # if False, will turn off increased PERI_ME motility, apoptosis in seam and EMT
palateModelCellBehaviors.geneStatusDict['EGF']=   True
palateModelCellBehaviors.geneStatusDict['TGFb3']= True
palateModelCellBehaviors.geneStatusDict['FGF10']= True
palateModelCellBehaviors.geneStatusDict['FGF7']=  True
palateModelCellBehaviors.geneStatusDict['BMP2']=  True
palateModelCellBehaviors.geneStatusDict['BMP4']=  True
palateModelCellBehaviors.geneStatusDict['NOGGIN']=True
palateModelCellBehaviors.geneStatusDict['SHH']=   True
palateModelCellBehaviors.geneStatusDict['MMP']=   True
#palateModelCellBehaviors.ahrActivity = 0.00  # OLD WAY TO PUT IN AHR ACTIVATION
#palateModelCellBehaviors.ahrDrivenEGFRFoldChange = 1.0  # ANOTHER OLD WAY TO PUT IN CONSTANT AHR ACTIVATION
palateModelCellBehaviors.EGFR_0  = 130  # initial level of EGF Receptor
palateModelCellBehaviors.TGFBR_0 = 150  # initial level of TGFB3 Receptor
palateModelCellBehaviors.ahrDrivenEGFRMaxFoldChange = 1.0  # 1.0 is no change in EGFR levels
palateModelCellBehaviors.ahrDrivenEGFRStartTime = 0    # use 0 to mimic constant levels
palateModelCellBehaviors.ahrDrivenEGFRTauUp = 1         # use a very small number (but not zero) to mimic constant levels
palateModelCellBehaviors.ahrDrivenEGFRTauDown = 6000000      # use a very large number to mimic constant levels

steppableRegistry.registerSteppable(palateModelCellBehaviors)
print "Finished importing PalateModelCellBehaviors"

#from palate_steppables import PalateModelMitosis
#palateModelMitosis=PalateModelMitosis(_simulator=sim,_frequency=10)
#steppableRegistry.registerSteppable(palateModelMitosis)
#print "Finished importing PalateModelMitosis"

#from palate_steppables import PalateModelClusterMitosis
#palateModelClusterMitosis=PalateModelClusterMitosis(_simulator=sim,_frequency=10)
#steppableRegistry.registerSteppable(palateModelClusterMitosis)
#palateModelClusterMitosis.useDivideClusterFunctions = False
#print "Finished importing PalateModelClusterMitosis"

from palate_steppables import ExecuteClusterMitosis
executeClusterMitosis=ExecuteClusterMitosis(_simulator=sim,_frequency=10)
executeClusterMitosis.useDivideClusterFunctions = False # True is not currently working; needs new build of CC3D to work on hex lattice
steppableRegistry.registerSteppable(executeClusterMitosis)
print "Finished importing ExecuteClusterMitosis"

from palate_steppables import ManageEpithelialPolarization
manageEpithelialPolarization=ManageEpithelialPolarization(_simulator=sim, _frequency=10)
manageEpithelialPolarization.setStartMCS(_startMCS=50)
manageEpithelialPolarization.useSpecialJunctionDomain=False  # none of the TJ settings matter if this is false
manageEpithelialPolarization.setNucleateTJFromCellTypes([PERI_N,PERI_O,PERI_ME])
manageEpithelialPolarization.setNucleateTJWhenContactingCellTypes([PERI_N, PERI_O, PERI_ME, TJ, WALL])
manageEpithelialPolarization.setRevertTJWhenNotContactingCellTypes([TJ, TJm, WALL])
manageEpithelialPolarization.setTJNucleationParameters(_TJTargetVolume=9, _TJLambdaVolume=0.5, _TJLambdaSurface=0.1, _transientNucleationMultiplier=2, _maxTJperCell=2)
manageEpithelialPolarization.setTJProbabilities(_probNucleate = 1, _probRevert= 0.5, _probDestabilize= 0.5)
manageEpithelialPolarization.useSpecialApicalDomain=True
manageEpithelialPolarization.addPolarizedCompartmentSet(_apical=PERI_Na, _basal=PERI_N)
manageEpithelialPolarization.addPolarizedCompartmentSet(_apical=PERI_Oa, _basal=PERI_O)
manageEpithelialPolarization.addPolarizedCompartmentSet(_apical=PERI_MEa, _basal=PERI_ME)
steppableRegistry.registerSteppable(manageEpithelialPolarization)
print "Finished importing ManageEpithelialPolarization"

from palate_steppables import SimulatePullByKillingOffCellsOnBorder
simulatePullByKillingOffCellsOnBorder=SimulatePullByKillingOffCellsOnBorder(_simulator=sim,_frequency=10)
simulatePullByKillingOffCellsOnBorder.startPullingMCS = 10000
steppableRegistry.registerSteppable(simulatePullByKillingOffCellsOnBorder)
#print "Finished importing SimulatePullByKillingOffCellsOnBorder"

if useTUNLField:
   from visualization_steppables import TUNLFieldVisualizationSteppable
   tunlFieldVisualizationSteppable=TUNLFieldVisualizationSteppable(_simulator=sim,_frequency=10)
   tunlFieldVisualizationSteppable.setScalarField(TUNLField)
   steppableRegistry.registerSteppable(tunlFieldVisualizationSteppable)

if useBrdUField:
   from visualization_steppables import BrdUFieldVisualizationSteppable
   brduFieldVisualizationSteppable=BrdUFieldVisualizationSteppable(_simulator=sim,_frequency=10)
   brduFieldVisualizationSteppable.setScalarField(BrdUField)
   brduFieldVisualizationSteppable.setBrdUPulseDuration(200)
   brduFieldVisualizationSteppable.setCellTypesToExclude([MEDIUM,WALL,MATRIX,BM])
   steppableRegistry.registerSteppable(brduFieldVisualizationSteppable)



run_time = time.strftime("%m_%d_%Y_%H_%M_%S", time.localtime())
simulationPythonScriptName = CompuCellSetup.simulationPythonScriptName

CompuCellSetup.mainLoop(sim,simthread,steppableRegistry)

#########################################################################################################
# if True, then save a copy of this .py file with a datetime string attached for record keeping
# has not been tested on situations using more than 1 steppable file (that might be in different directories)
if True:
    pythonFileName = simulationPythonScriptName
    #alternative method to get pythonFileName
    #pythonFileName = inspect.stack()[0][1]
    #print "PYTHON SCRIPT NAME USING INSPECT STACK: "+pythonFileName
    #os.system('echo $PATH') - something in CC3D player is changing $PATH variable so os cannot find commands like cp
    #fix required putting in explicit /bin/cp call
    if pythonFileName.endswith('.py'):
        copyPythonFileName = pythonFileName.rpartition('.')[0] + '_' + run_time + '.py'
        print "WILL ATTEMPT TO COPY PYTHON SCRIPT FILE TO: "+copyPythonFileName
        os.system('/bin/cp ' + pythonFileName + ' ' + copyPythonFileName)
        #macostools.copy(pythonFileName, copyPythonFileName)
        
        pythonDir = pythonFileName.rpartition(os.sep)[0]
        #print "pythonDir = "+ pythonDir
        print "Get a list of all the steppable files used for this simulation"
        steppableFileList=[]
        print dir(steppableRegistry)
        for steppableEntry in steppableRegistry.steppableList:
            print steppableEntry
            currentFile = pythonDir+os.sep+str(steppableEntry).strip(' <>').partition('.')[0] + '.py'
            if currentFile not in steppableFileList:
                steppableFileList.append(currentFile)
        #print "adding "+currentFile+" to steppableFileBaseList"
        for steppableFileName in steppableFileList:
            copySteppableFileName = steppableFileName.rpartition('.')[0] + '_' + run_time + '.py'
            print "WILL ATTEMPT TO COPY STEPPABLE FILE TO: "+copySteppableFileName
            os.system('/bin/cp ' + steppableFileName + ' ' + copySteppableFileName)



   

