
  CREATE OR REPLACE PACKAGE "STORETW"."DW_COMP"
AS
   --lv_script_path   VARCHAR2 (1000)
   --   DEFAULT DW_GLOB_VAR.lv_script_path;
   lv_script_path   VARCHAR2 (1000)
      DEFAULT 'http://www.epa.gov/storet/modern/scripts/dw_storet_enhanced.js';

   PROCEDURE bottom (help_counter NUMBER DEFAULT 9999);

   PROCEDURE top (page_title     IN     VARCHAR2,
                  program_path      OUT VARCHAR2,
                  image_path        OUT VARCHAR2,
                  body_on_load   IN     VARCHAR2 DEFAULT NULL);

   PROCEDURE style;

   PROCEDURE script (script_path IN VARCHAR2);

   PROCEDURE geo_select;

   PROCEDURE org_select (tabstart IN NUMBER, org_flag CHAR DEFAULT 'OFF');

   PROCEDURE date_select (tabstart IN NUMBER);

   PROCEDURE medium_select (tabstart IN NUMBER);

   PROCEDURE intent_community_select (tabstart IN NUMBER);

   PROCEDURE char_select (tabstart       IN NUMBER,
                          result_type       CHAR DEFAULT 'REGULAR',
                          station_flag      CHAR DEFAULT 'OFF');

   PROCEDURE huc_popup;

   PROCEDURE org_alias_popup (v_orgstring CHAR);

   PROCEDURE char_alias_popup (v_charstring          CHAR,
                               taxon_filter          CHAR,
                               as_char_alias_type    NUMBER);

   PROCEDURE extref_popup (v_org NUMBER);

   PROCEDURE proj_popup (v_org NUMBER, v_org_name VARCHAR2 DEFAULT NULL);

   PROCEDURE station_popup (v_stationstring    CHAR,
                            v_org              NUMBER,
                            v_searchtype       CHAR,
                            v_extref           CHAR);

   PROCEDURE county_popup (v_statecode NUMBER);

   PROCEDURE display_calendar (date_name   IN VARCHAR2,
                               range_num   IN NUMBER DEFAULT 1,
                               start_tab   IN NUMBER DEFAULT 10,
                               from_year   IN NUMBER DEFAULT 1900,
                               to_year     IN NUMBER DEFAULT 2050);

   PROCEDURE station_report_form (v_stationcount   IN NUMBER,
                                  v_fromstmt       IN VARCHAR2,
                                  v_wherestmt      IN VARCHAR2,
                                  v_filters        IN VARCHAR2,
                                  v_bottom_form    IN BOOLEAN);

   PROCEDURE result_report_form (v_resultcount         IN NUMBER,
                                 v_fromstmt            IN VARCHAR2,
                                 v_wherestmt           IN VARCHAR2,
                                 vb_resultcount        IN NUMBER,
                                 vb_fromstmt           IN VARCHAR2,
                                 vb_wherestmt          IN VARCHAR2,
                                 vh_resultcount        IN NUMBER,
                                 vh_fromstmt           IN VARCHAR2,
                                 vh_wherestmt          IN VARCHAR2,
                                 vr_resultcount        IN NUMBER,
                                 vr_fromstmt           IN VARCHAR2,
                                 vr_wherestmt          IN VARCHAR2,
--CCB --2012-06-22: metrics
                                 vm_resultcount        IN NUMBER,
                                 vm_fromstmt           IN VARCHAR2,
                                 vm_wherestmt          IN VARCHAR2,
                                 v_filters             IN VARCHAR2,
                                 v_rule_optimization   IN VARCHAR2,
                                 result_type           IN VARCHAR2,
                                 v_bottom_form         IN BOOLEAN);
END dw_comp;
/
CREATE OR REPLACE PACKAGE BODY "STORETW"."DW_COMP" AS
   PROCEDURE bottom (help_counter NUMBER DEFAULT 9999)
   AS
      lv_last_update   VARCHAR2 (256) := NULL;
      script_path      VARCHAR2 (2000);
   BEGIN
      script_path := dw_comp.lv_script_path;

      --fetch last update from mt_wh_config table
      BEGIN
         Sp_Fetch_Config_Value ('LAST_UPDATE', lv_last_update);
      EXCEPTION
         WHEN OTHERS
         THEN
            NULL;
      END;

      --EPA Standard Footer
      HTP.PRINT (
         '
<!--googleoff: index-->
      <p id="areafooter">Comments? If you have a question or comments on this website or on the STORET system,
please call STORET User Assistance at 1-800-424-9067,
or send us an email at STORET@epa.gov. </p>
    </div><!-- /main -->
    <div id="areanav">
      <h4 class="skip">Area Navigation</h4>
      <ul>
        <li class="separator"><a href="#">Area Home</a><hr /></li>
        <li><a href="#">Subarea Home</a></li>
        <li><a href="#">Basic Information</a></li>
        <li><a href="#">Where You Live</a></li>
        <li><a href="#">Recent Additions</a></li>
        <li><a href="#">Area sidebar links go here</a></li>
        <li><a href="#">Link #7</a></li>
        <li><a href="#">Link #8</a></li>
      </ul>
    </div><!-- /area-nav -->
  </div><!-- /page -->
</div><!-- /content -->
<div id="footer">
  <ul id="goodies">
    <li id="rss-news"><a href="http://www.epa.gov/newsroom/rssfeeds.htm">News Feeds</a></li>
    <li id="podcasts"><a href="http://www.epa.gov/epahome/podcasts.htm">Podcasts</a></li>
    <li id="mobile"><a href="http://m.epa.gov/">EPA Mobile</a></li>
    <li id="email-news"><a href="http://www.epa.gov/newsroom/email_signups.htm">News by Email</a></li>
    <li id="widgets"><a href="http://www.epa.gov/widgets/">Widgets</a></li>
  </ul>
  <ul id="globalfooter">
    <li><a href="http://www.epa.gov/">EPA Home</a></li>
    <li><a href="http://www.epa.gov/epahome/usenotice.htm">Privacy and Security Notice</a></li>
    <li class="last"><a href="#">Contact Us</a></li>
  </ul>
  <p id="seal">The Seal of the United States Environmental Protection Agency</p>
</div><!-- /footer -->
<p class="skip"><a href="#content" title="Jump to main content.">Jump to main content.</a></p>');

      script (script_path);

      HTP.PRINT ('
</body>
</html>
        ');
   END bottom;

   PROCEDURE top (page_title     IN     VARCHAR2,
                  program_path      OUT VARCHAR2,
                  image_path        OUT VARCHAR2,
                  body_on_load   IN     VARCHAR2 DEFAULT NULL)
   AS
   BEGIN
      HTP.PRINT (
            '

<!DOCTYPE html>
<html lang="en">
<!-- EPA Template, OneEPA Web, 17 November 2010 -->
<head>
<meta charset="utf-8" />
<meta http-equiv="X-UA-Compatible" content="IE=8" />
<title>EPA &gt; STORET &gt; '
         || page_title
         || '</title>
<meta name="DC.description" content="" />
<meta name="keywords" content="" />
<meta name="DC.title" content="" />
<meta name="DC.type" content="" />
<meta name="DC.Subject.epachannel" content="" />
<meta name="DC.Subject.epabrm" content="" />
<meta name="DC.Subject.epacat" content="" />
<meta name="DC.Subject.epaect" content="" />
<meta name="DC.Subject.epaemt" content="" />
<meta name="DC.Subject.epahealth" content="" />
<meta name="DC.Subject.epaindustry" content="" />
<meta name="DC.Subject.epaopt" content="" />
<meta name="DC.Subject.epappt" content="" />
<meta name="DC.Subject.eparat" content="" />
<meta name="DC.Subject.eparegulation" content="" />
<meta name="DC.Subject.eparit" content="" />
<meta name="DC.Subject.epasubstance" content="" />
<meta name="DC.audience" content="" />
<meta name="DC.coverage" content="" />
<meta name="DC.date.created" content="" />
<meta name="DC.date.modified" content="" />
<meta name="DC.date.archivedate" content="" />
<meta name="DC.creator" content="US EPA" />
<meta name="DC.language" content="en" />
<meta name="viewport" content="width=device-width; initial-scale=1.0"/>

<link rel="schema.DC" href="http://purl.org/dc/elements/1.1/" />
<!--RSS Feeds -->
<link rel="meta" href="http://www.epa.gov/labels.rdf" type="application/rdf+xml" title="ICRA labels" />
<link rel="alternate" type="application/atom+xml" title="EPA.gov News" href="http://yosemite.epa.gov/opa/admpress.nsf/RSSRecentNews" />
<link rel="alternate" type="application/atom+xml" title="EPA.gov Headquarters Press Releases" href="http://yosemite.epa.gov/opa/admpress.nsf/RSSByLocation?open&location=Headquarters" />
<link rel="alternate" type="application/atom+xml" title="Greenversations, EPA''s Blog" href="http://blog.epa.gov/blog/feed/" />
');

      style;

      HTP.PRINT (
            '</head>
<body class="basic wide"
        onload='''
         || body_on_load
         || '''>
<!--googleoff: index-->
<p id="sitewidec"></p>
<p class="skip">Jump to <a href="#main" title="Jump to main content">main content</a> or <a href="#areanav" title="Jump to area navigation.">area navigation</a>.</p>
<div id="header">
  <p><a href="http://www.epa.gov/" title="US EPA Home Page"><img src="http://www.epa.gov/epafiles/css/i/print-epa-logo.gif" alt="US Environmental Protection Agency logo" /></a></p>
  <form id="EPAsearch" method="get" action="http://nlquery.epa.gov/epasearch/epasearch">
    <fieldset>
      <ol>
        <li id="azindex"><a href="http://www.epa.gov/epahome/quickfinder.htm" title="Alphabetical list of top EPA topics">A-Z Index</a></li>
        <li><a href="http://www.epa.gov/search.html" title="Advanced search with additional filters">Advanced Search</a></li>
        <li>
          <fieldset id="search">
            <legend>What are you looking for?</legend>
            <input name="querytext" id="searchbox" value=""/>
            <button id="searchbutton" type="submit" title="Search">Search</button>
          </fieldset>
        </li>
      </ol>
      <input type="hidden" name="fld" value="" />
      <input type="hidden" name="areaname" value="" />
      <input type="hidden" name="areacontacts" value="" />
      <input type="hidden" name="areasearchurl" value="" />
      <input type="hidden" name="typeofsearch" value="epa" />
      <input type="hidden" name="result_template" value="epafiles_default.xsl" />
      <input type="hidden" name="filter" value="sample4filt.hts" />
    </fieldset>
  </form><!-- /search -->
  <ul>
    <li id="learn"><a href="http://www.epa.gov/gateway/learn/" title="Learn the Issues">Learn the Issues</a></li>
    <li id="scitech"><a href="http://www.epa.gov/gateway/science/" title="Science and Technology">Science &amp; Technology</a></li>
    <li id="laws"><a href="http://www.epa.gov/lawsregs/" title="Laws and Regulations">Laws &amp; Regulations</a></li>
    <li id="about"><a href="http://www.epa.gov/aboutepa/" title="About EPA">About EPA</a></li>
  </ul><!-- /IA Buckets -->
</div><!-- /header -->
<div id="content">
  <div id="area">
    <p><a href="#">Contact Us</a></p>
    <p>STORET</p>
  </div><!-- /area -->
  <div id="page">
    <div id="main">
      <ul id="breadcrumbs">
        <li class="first"><b>You are here:</b> <a href="http://www.epa.gov/">EPA Home</a></li>
        <li><a href="http://www.epa.gov/water">Water</a></li>
        <li><a href="http://www.epa.gov/owow">Wetlands, Oceans, &amp; Watersheds</a></li>
        <li><a href="http://www.epa.gov/owow/monitoring">Monitoring and Assessing Water Quality</a></li>
        <li><a href="http://www.epa.gov/storet">STORET</a></li>
        <li>Data Warehouse</li>
      </ul>
<!--googleon: index-->
          <h1>STORET Central Warehouse</h1>
        ');
   END top;

   PROCEDURE style
   AS
   BEGIN
      HTP.PRINT ('<!-- ALL PAGE STYLES BELOW THIS LINE -->');
      HTP.PRINT ('<link rel="stylesheet" href="' || dw_glob_var.lv_download_path || '../dw.css" />');
      HTP.PRINT (
         '<link rel="stylesheet" href="http://www.epa.gov/epafiles/templates/webcms/css/epa.css" />');
      HTP.PRINT (
         '<!--[if IE 7]><link rel="stylesheet" href="http://www.epa.gov/epafiles/templates/webcms/css/ie7.css"/><![endif]-->');
      HTP.PRINT (
         '<!--[if lt IE 7]><link rel="stylesheet" href="http://www.epa.gov/epafiles/templates/webcms/css/ie6.css"/><![endif]-->');
   END style;

   PROCEDURE script (script_path IN VARCHAR2)
   AS
   BEGIN
      HTP.PRINT (
         '<script src="http://www.epa.gov/epafiles/js/third-party/jquery.js"></script>');
      HTP.PRINT (
         '<script src="http://www.epa.gov/epafiles/js/epa-core-v4.js"></script>');
      HTP.PRINT ('<!-- ALL SCRIPTS BELOW THIS LINE -->');
      HTP.PRINT (
            '<script src="'
         || script_path
         || '" language="javascript"></script>');
   END script;

   PROCEDURE geo_select
   AS
      --used to generate state list
      CURSOR cstate
      IS
           SELECT pk_isn, state_name
             FROM DI_GEO_STATE
         ORDER BY country_code DESC, state_name;
   BEGIN
      HTP.PRINT (
         '<h4>Geographic Location</h4>
        <center>Select a single type of location search that you wish to perform (state/county, latitude/longitude, or HUC).  Then enter the corresponding search criteria.</center>
        <table COLS=2 WIDTH="100%" class="tablebord zebra">
        <tr><td width="20%"><input checked type=radio name="geographic" value="StateCounty" tabindex=1 onclick="javascript:document.forms[1].geographic[0].checked=true;">&nbsp;State/County<br>(Option A)</td>
        <td align=center><table COLS=2 WIDTH="85%">
        <tr><td style="border:0;border-width:0 0 0 0"><b>State Name</b></td><td style="border:0;border-width:0 0 0 0"><b>County Name</b></td></tr>
        <tr><td valign="top" style="border:0;border-width:0 0 0 0"><font face="Courier"><p>
         <select name="as_state" size="1" tabindex=2 style="font: 10pt/12pt COURIER" onChange="javascript:document.forms[1].geographic[0].checked=true;document.forms[1].as_county.value=''ALL'';document.forms[1].v_county_code.value=''ALL'';">
        <option value="0">ALL');

      FOR state IN cstate
      LOOP
         HTP.formSelectOption (state.state_name,
                               cattributes   => 'VALUE=' || state.pk_isn);
      END LOOP;

      HTP.PRINT (
         '</SELECT></font></td>
        <td style="border:0;border-width:0 0 0 0"><font face="Courier"><p>
        <textarea readonly name="as_county" wrap="virtual" row=2 cols=20 style="font: 10pt/12pt COURIER" onFocus="javascript:this.blur()">ALL</textarea>
        </font>
        <input type="button" value="Look Up" tabindex=3 onclick="openWin(''state'')";" title="Opens a popup window for county selection">
        <input type="hidden" name="v_county_code" value="ALL">
        </td></tr></table>
        </td></tr>
        <tr><td valign="top"><input type=radio name="geographic" value="multipleStates" onClick="javascript:document.forms[1].geographic[1].checked=true;">&nbsp;Select one or more state(s)<br>(Option B)</td>
        <td align=center><font face="Courier">
         <select name="as_multiple_state" size="10" tabindex=22 style="font: 10pt/12pt COURIER" multiple onClick="javascript:document.forms[1].geographic[1].checked=true;">
        <option value="0">ALL');

      FOR state IN cstate
      LOOP
         HTP.formSelectOption (state.state_name,
                               cattributes   => 'VALUE=' || state.pk_isn);
      END LOOP;

      HTP.PRINT (
         '</SELECT></font></td></tr>
        <tr><td><input type=radio name="geographic" value="LatLong" tabindex=4 onClick="javascript:document.forms[1].geographic[2].checked=true;">&nbsp;Latitude/Longitude<br>(in decimal degrees)<br>(Option C)</td>
        <td align=center><B>North Limit</B>
        <BR>&nbsp
        <font face="Courier">
        <INPUT type="TEXT" NAME="MAX_LAT_DD" SIZE="10" value="90" tabindex=5 onChange="javascript:document.forms[1].geographic[2].checked=true;">
        </font>
        <font face="Courier">
        <SELECT NAME="dir_maxlat" style="font: 10pt/12pt COURIER" tabindex=6  onChange="javascript:document.forms[1].geographic[2].checked=true;">
        <OPTION VALUE= SOUTH>S
        <OPTION SELECTED VALUE= NORTH>N
        </SELECT>
        </font>
        <BR>
        <B>West Limit</B>
        &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp
        &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp
        <B>East Limit</B>
        &nbsp
        <BR>
        <font face="Courier">
        <INPUT TYPE="TEXT" NAME="MIN_LONG_DD" SIZE="10" value="180" tabindex=7 onChange="javascript:document.forms[1].geographic[2].checked=true;">
        </font>
        <font face="Courier">
        <SELECT NAME="dir_minlong" style="font: 10pt/12pt COURIER" tabindex=8  onChange="javascript:document.forms[1].geographic[2].checked=true;">
        <OPTION VALUE= EAST>E
        <OPTION SELECTED VALUE= WEST>W
        </SELECT>
        </font>
        &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp
        &nbsp &nbsp &nbsp &nbsp &nbsp
        <font face="Courier">
        <INPUT TYPE="TEXT" NAME="MAX_LONG_DD" SIZE="10" value="0" tabindex=9 onChange="javascript:document.forms[1].geographic[2].checked=true;">
        </font>
        <font face="Courier">
        <SELECT NAME="dir_maxlong" style="font: 10pt/12pt COURIER" tabindex=10  onChange="javascript:document.forms[1].geographic[2].checked=true;">
        <OPTION SELECTED VALUE= EAST>E
        <OPTION  VALUE= WEST>W
        </SELECT>
        </font>
        <BR>
        &nbsp
        <B>South Limit</B>
        <BR>
        &nbsp
        <font face="Courier">
        <INPUT TYPE="TEXT" NAME="MIN_LAT_DD" SIZE="10" value="0" tabindex=11 onChange="javascript:document.forms[1].geographic[2].checked=true;">
        </font>
        <font face="Courier">
        <SELECT NAME="dir_minlat" style="font: 10pt/12pt COURIER" tabindex=12  onChange="javascript:document.forms[1].geographic[2].checked=true;">
        <OPTION SELECTED VALUE= SOUTH>S
        <OPTION  VALUE= NORTH>N
        </SELECT>
        </font>
        </td></tr>
        <tr><td><input type=radio name="geographic" value="HUC" tabindex=13 onClick="javascript:document.forms[1].geographic[3].checked=true;">&nbsp;Select one or more<br>Drainage Basin/HUC<br>(Option D)
        </td>
        <td align=center>Use the Look Up button to create a list of Cataloging Unit(s) <br>
        <font face= "Courier">
         <select name="as_huc" size="10" tabindex=26 style="width:500px; font: 10pt/12pt COURIER" multiple onClick="javascript:document.forms[1].geographic[3].checked=true;">
        <option>HUC CODE &nbsp; HUC NAME</option>
        <option value="dummy">______________________________________________________________</option>
         </select>
        <br>
        <input type="button" value="Look Up" tabindex=14 onclick="openWin(''huc'')" title="Opens a popup window for HUC selection">
        <input type="button" title="Remove highlighted HUCs from selected list" onclick="javascript:deleteSelectedFromHucList();" value="Clear Selected">
        <input type="button" title="Remove all HUCs from the selected list" onclick="javascript:deleteAllFromHucList();" value="Clear All">
        </td></tr></table>
        ');
   END geo_select;

   PROCEDURE org_select (tabstart IN NUMBER, --result_type char default 'REGULAR',
                                             --station_flag char default 'OFF'
    org_flag CHAR DEFAULT 'OFF')
   AS
      --used to generate char alias type list
      CURSOR orgalias
      IS
           SELECT pk_isn,
                  LTRIM (RTRIM (organization_name)) org_name,
                  organization_id
             FROM di_org
         ORDER BY organization_id, org_name;
   BEGIN
      IF org_flag = 'ON'
      THEN
         HTP.PRINT (
            '<center>Find Organizations with at least one reported value of the selected Organizations</center>');
      END IF;

      HTP.PRINT (
            '
        <tr><td width="25%"><input type="radio" name="org_station_project_button" value="Organization Name" tabindex=53 onclick="javascript:document.forms[1].org_station_project_button[1].checked=true;">&nbsp;Select one or more<br>Organization(s)<br>(Option 2)</td>
        <td align="center"><center>Use the Organization Search to create a list of up to 50 Organization(s)
        </center>
        <br>
        Organization Search
        <BR>
        <font face= "Courier, monospace">
        <input type=hidden name="v_org_names">
        <input type="text" name="as_org" tabindex='
         || TO_CHAR (tabstart)
         || ' size="20" maxlength="20" style="font-size: 10pt; font-family: Courier, monospace" onclick="javascript:document.forms[1].org_station_project_button[1].checked=true;">
        </font>
        <BR>
        <font face="Arial">
        <input type="button" tabindex='
         || TO_CHAR (tabstart + 2)
         || ' value="Search"  onclick="openWin(''org'');" title="Opens a popup window for Organization selection">
        <BR><BR><font face= "Courier, monospace">
        <select multiple size="10" name="org_list" style="width: 650px; font-size: 10pt; font-family: Courier, monospace" onclick="javascript:document.forms[1].org_station_project_button[1].checked=true;" >
        <option>ORG ID &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; ORGANIZATION NAME</option>
        <option value="dummy">_________________________________________________________________________________</option>
        </select></font>
        <BR>
        <center><input type="button" tabindex='
         || TO_CHAR (tabstart + 4)
         || ' value="Clear Selected" onclick="javascript:deleteSelectedFromOrgList();document.forms[1].org_station_project_button[1].checked=true;" title="Remove highlighted Organizations from Selected List">
        <input type="button" tabindex='
         || TO_CHAR (tabstart + 5)
         || ' value="Clear All" onclick="javascript:deleteAllFromOrgList();document.forms[1].org_station_project_button[1].checked=true;" title="Remove all Organizations from the Selected List">
        </center>
        </font>
        </td>
        </tr>
        ');
   END org_select;


   PROCEDURE date_select (tabstart IN NUMBER)
   AS
   BEGIN
      HTP.PRINT (
         '
        <h4>Date</h4>
        <table BORDER=1 COLS=1 WIDTH="100%" class="tablebord zebra" >
        <tr>
        <td align="center">
        <center>Specify Activity Start Date range(s)
        <BR>');

      FOR i IN 1 .. 4
      LOOP
         HTP.PRINT (
               '<BR><font face="Arial"><B> Date Range '
            || TO_CHAR (i)
            || ': </B></font>&nbsp &nbsp &nbsp &nbsp');
         DW_COMP.DISPLAY_CALENDAR ('FROM', i, tabstart);
         HTP.PRINT (
            '<font face="Arial">&nbsp &nbsp &nbsp <B>To</B></font>&nbsp &nbsp &nbsp &nbsp');
         DW_COMP.DISPLAY_CALENDAR ('END', i, tabstart);
         HTP.PRINT ('<BR>');
      END LOOP;

      HTP.PRINT ('</td>');
      HTP.tablerowclose;
      HTP.tableclose;
   END date_select;


   PROCEDURE medium_select (tabstart IN NUMBER)
   AS
      --used to generate Activity Medium list
      CURSOR cmed
      IS
           SELECT pk_isn, activity_medium medium
             FROM di_activity_medium
            WHERE     activity_medium <> 'Biological'
                  AND activity_medium <> 'Tissue'
         ORDER BY pk_isn;
   BEGIN
      -- Activity Medium selection
      HTP.PRINT (
            '
        <h4>Activity Medium</h4>
        <table BORDER WIDTH="100%" class="tablebord zebra">
        <tr>
        <td align="center" width="50%">
        Select one or more Activity Medium
        <BR>
        <INPUT TYPE="TEXT" READONLY style="font: 10pt/12pt COURIER; font-weight: bold"  VALUE="        ACTIVITY MEDIUM" size="33">
        <FONT face="Courier"><BR>
        <SELECT NAME="D_MEDIUM_LIST" SIZE="7" tabindex='
         || TO_CHAR (tabstart)
         || ' MULTIPLE style="font: 10pt/12pt COURIER" >
        <OPTION SELECTED  value="ALL">Select All_____________________');

      FOR cmed_var IN cmed
      LOOP
         HTP.formSelectOption (cmed_var.medium,
                               cattributes   => 'value= ' || cmed_var.PK_ISN);
      END LOOP;

      HTP.PRINT ('</SELECT>
        <BR>
        </TD>
        </TR>
        </TABLE>');
   END medium_select;


   PROCEDURE intent_community_select (tabstart IN NUMBER)
   AS
      --used to generate intent list
      CURSOR cintent
      IS
           SELECT pk_isn, activity_intent intent
             FROM di_activity_intent
         ORDER BY pk_isn;

      --used to generate community sampled list
      CURSOR ccomm
      IS
           SELECT pk_isn, activity_community community
             FROM di_community_sampled
         ORDER BY pk_isn;
   BEGIN
      -- Intent and Community Sampled selection
      HTP.PRINT (
            '
        <h4>Activity Intent and Community Sampled</h4>
        <table BORDER COLS=2 WIDTH="100%" class="tablebord zebra">
        <tr>
        <td align="center" width="50%">
        Select one or more Activity Intent
        <BR>
        <INPUT TYPE="TEXT" READONLY style="font: 10pt/12pt COURIER; font-weight: bold"  VALUE="        ACTIVITY INTENT" size="33"  onFocus="javascript:this.blur()">
        <FONT face="Courier"><BR>
        <SELECT NAME="D_INTENT_LIST" SIZE="7" tabindex='
         || TO_CHAR (tabstart)
         || ' MULTIPLE style="font: 10pt/12pt COURIER" >
        <OPTION SELECTED  value="ALL">Select All_____________________');

      FOR cintent_var IN cintent
      LOOP
         HTP.formSelectOption (
            cintent_var.intent,
            cattributes   => 'value= ' || cintent_var.PK_ISN);
      END LOOP;

      HTP.PRINT (
            '</SELECT>
        <BR>
        </TD>
        <td align="center" width="50%">
        Select one or more Community Sampled
        <BR>
        <INPUT TYPE="TEXT" READONLY style="font: 10pt/12pt COURIER; font-weight: bold"  VALUE="       COMMUNITY SAMPLED" size="33"  onFocus="javascript:this.blur()">
        <FONT face="Courier"><BR>
        <SELECT NAME="D_COMMUNITY_LIST" SIZE="7" tabindex='
         || TO_CHAR (tabstart + 1)
         || ' MULTIPLE style="font: 10pt/12pt COURIER" >
        <OPTION SELECTED  value="ALL">Select All_____________________');

      FOR ccomm_var IN ccomm
      LOOP
         HTP.formSelectOption (ccomm_var.community,
                               cattributes   => 'value= ' || ccomm_var.PK_ISN);
      END LOOP;

      HTP.PRINT ('</SELECT>
        <BR>
        </TD>
        </TR>
        </TABLE>');
   END intent_community_select;


   PROCEDURE char_select (tabstart       IN NUMBER,
                          result_type       CHAR DEFAULT 'REGULAR',
                          station_flag      CHAR DEFAULT 'OFF')
   AS
      --used to generate char alias type list
      CURSOR calias
      IS
           SELECT pk_isn, char_alias_type_name
             FROM LU_CHAR_ALIAS_TYPE
            WHERE char_alias_type_category = 'NT'
         ORDER BY pk_isn;

      --used to generate char alias type list for Bio results
      CURSOR calias_bio
      IS
           SELECT pk_isn, char_alias_type_name
             FROM LU_CHAR_ALIAS_TYPE
            WHERE char_alias_type_category IS NOT NULL
         ORDER BY pk_isn;
   BEGIN
      HTP.PRINT ('
        <h4>Characteristic</h4>');

      IF station_flag = 'ON'
      THEN
         HTP.PRINT (
            '<center>Find Stations with at least one reported value of the selected Characteristics</center>');
      END IF;

      HTP.PRINT (
            '
        <table BORDER COLS=1 WIDTH="100%" class="tablebord zebra" >
        <tr>
        <td align="center">
        <center>Use the Characteristic Search to create a list of up to 50 Characteristics</center>
        <br>
        <b>Characteristic Search &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp;
        &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; Search By &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp;</b>
        <BR>
        <font face= "Courier, monospace">
        <input type=hidden name="v_char_names">
        <input type="text" name="as_char" tabindex='
         || TO_CHAR (tabstart)
         || ' size="20" maxlength="20" style="font-size: 10pt; font-family: Courier, monospace">
        &nbsp &nbsp &nbsp &nbsp &nbsp
        <select name="as_char_alias_type" tabindex='
         || TO_CHAR (tabstart + 1)
         || ' size="1" style="font-size: 10pt; font-family: Courier, monospace" >
        <option value=0>CHARACTERISTIC NAME</option>');

      IF result_type = 'BIOLOGICAL'
      THEN
         FOR alias IN calias_bio
         LOOP
            HTP.formSelectOption (alias.char_alias_type_name,
                                  cattributes   => 'VALUE=' || alias.pk_isn);
         END LOOP;
      ELSE
         FOR alias IN calias
         LOOP
            HTP.formSelectOption (alias.char_alias_type_name,
                                  cattributes   => 'VALUE=' || alias.pk_isn);
         END LOOP;
      END IF;

--CCB --2012-06-22: char_select advance --2012-08-27: SELECT SAMPLE
--         <input type="radio" name="char_filter" value="sample" > Sample (whole)
      HTP.PRINT (
            '</font></SELECT>
        <BR>
        <font face="Arial">
        &nbsp;&nbsp;&nbsp;
        <input type="button" tabindex='
         || TO_CHAR (tabstart + 2)
         || ' value="Search"  onclick="openWin(''char_alias'')" title="Opens a popup window for characteristic selection">
        &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;
        <input checked type="checkbox" name="taxon_filter" tabindex='
         || TO_CHAR (tabstart + 3)
         || ' value="ON">&nbsp;Hide Taxonomic Names&nbsp;&nbsp;
        <BR><BR><font face= "Courier, monospace">
        <select multiple size="10" name="char_list" style="font-size: 10pt; font-family: Courier, monospace" >
        <option>&nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp CHARACTERISTIC NAME</option>
        <option value="dummy">___________________________________________________________</option>
        </select></font>
        <BR>
        <input type="button" tabindex='
         || TO_CHAR (tabstart + 4)
         || ' value="Clear Selected" onclick="javascript:deleteSelectedFromCharList();" title="Remove highlighted Characteristics from Selected List">
        <input type="button" tabindex='
         || TO_CHAR (tabstart + 5)
         || ' value="Clear All" onclick="javascript:deleteAllFromCharList();" title="Remove all Characteristics from the Selected List">
         <b>*Include Only:</b>
         <input type="radio" name="char_filter" value="selected" checked> Selected
         <input type="radio" name="char_filter" value="sample" > Sample (whole)
        </font>
        </td>
        </tr>
        </table>');
   END char_select;


   PROCEDURE huc_popup
   AS
      script_path   VARCHAR2 (1000);

      CURSOR c_HucList
      IS
           SELECT TRIM (HYDROLOGIC_UNIT_CODE) AS huc_code,
                  TRIM (HYDROLOGIC_UNIT_NAME) AS huc_name,
                  pk_isn
             FROM DI_DB_CAT
         ORDER BY 1, 2;
   BEGIN
      script_path := dw_comp.lv_script_path;
      HTP.htmlOpen;

      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');
      style;
      HTP.title ('STORET: Select a Hydrologic Unit Code (HUC)');
      HTP.headClose;

      HTP.bodyOpen (
         '',
         'class="basic wide" style="background:#f7f2f2;" onload=adjust_width_n(document.forms[0].as_huc,"~","10~50","None","N");');
      HTP.PRINT ('<br><center>Select a Hydrologic Unit Code (HUC)<br>');

      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');

      HTP.PRINT ('<form name="select_huc">');
      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');

      HTP.formSelectOpen (
         cname         => 'as_huc',
         nsize         => '10',
         cattributes   => 'MULTIPLE tabindex=25 style="font-size: 8pt; font-family: Courier New, Courier, monospace"');

      HTP.formSelectOption ('HUC CODE~HUC NAME', cattributes => 'VALUE=0');
      HTP.formSelectOption (
         '____________________________________________________________',
         cattributes   => 'VALUE=0');

      FOR cv_huc IN c_HucList
      LOOP
         HTP.FormSelectOption (cv_huc.huc_code || '~' || cv_huc.huc_name,
                               cattributes   => 'value=' || cv_huc.huc_code);
      END LOOP;

      HTP.PRINT (
         '
        </SELECT><br><br>
        <input type="hidden" name="v_huc_code" value="ALL">
        <INPUT TYPE="button" VALUE="Select" onclick="passback(''HUC'')">
        <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END huc_popup;

   PROCEDURE char_alias_popup (v_CharString          CHAR,
                               taxon_filter          CHAR,
                               as_char_alias_type    NUMBER)
   AS
      v_CharStringCap   VARCHAR2 (20) := UPPER (v_CharString);
      script_path       VARCHAR2 (1000);
--CCB --2012-09-12: DISABLE Taxon Filter for select Char Alias Type
      v_char_alias_type_name   VARCHAR2 (30) ;

      -- query by search name and retrieve display name
      CURSOR c_CharList
      IS
           SELECT display_name, min(pk_isn) pk_isn
             FROM DI_CHARACTERISTIC
            WHERE     search_name LIKE v_CharStringCap || '%'
                  AND type_code <> 'T'
                  AND type_code <> 'H'
         GROUP BY display_name
         ORDER BY display_name;

      CURSOR c_CharAliasList
      IS
           SELECT display_name, char_alias, ch.pk_isn
             FROM DI_CHARACTERISTIC ch, LU_CHAR_ALIAS ca
            WHERE     ch.pk_isn = fk_characteristic
                  AND UPPER (char_alias) LIKE v_CharStringCap || '%'
                  AND fk_char_alias_type = as_char_alias_type
                  AND type_code <> 'T'
                  AND type_code <> 'H'
         ORDER BY char_alias;

      CURSOR c_CharListTaxon
      IS
           SELECT display_name, min(pk_isn) pk_isn
             FROM DI_CHARACTERISTIC
            WHERE search_name LIKE v_CharStringCap || '%' --uncomment for production:
                                                         AND type_code <> 'H'
         GROUP BY display_name
         ORDER BY display_name;

      CURSOR c_CharAliasListTaxon
      IS
           SELECT display_name, char_alias, ch.pk_isn
             FROM DI_CHARACTERISTIC ch, LU_CHAR_ALIAS
            WHERE     ch.pk_isn = fk_characteristic
                  AND UPPER (char_alias) LIKE v_CharStringCap || '%'
                  AND fk_char_alias_type = as_char_alias_type
                  AND type_code <> 'H'
         ORDER BY char_alias;
   BEGIN
--CCB --2012-09-12: DISABLE Taxon Filter for select Char Alias Type
-- Char Alias Type name = RETIRED NAME (TAXON)
         IF as_char_alias_type = 10
         THEN
           SELECT TRIM(char_alias_type_name) into v_char_alias_type_name
             FROM LU_CHAR_ALIAS_TYPE lcat
            WHERE     lcat.pk_isn = as_char_alias_type;
--            v_char_alias_type_name := 'RETIRED NAME (TAXON)';
         ELSE
--            NULL;
            v_char_alias_type_name := 'OTHER';
--       TAXON COMMON NAME for include Taxonomic Names
--       TAXON COMMON NAME for include Taxonomic Names
            IF as_char_alias_type = 3
            THEN
               v_char_alias_type_name := 'RETIRED NAME (TAXON)';
            ELSE
               v_char_alias_type_name := 'OTHER';
            END IF;
        END IF;
--
      script_path := DW_COMP.lv_script_path;
      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');
      style;
      HTP.title ('STORET: Characteristic Search Results');
      HTP.headClose;

      -- use char cursor if default STORET names is chosen, else use char alias cursors
      IF as_char_alias_type = 0
      THEN
         HTP.bodyOpen (
            'class="basic wide" style="background:#f7f2f2;" onload="document.forms[0].as_county.selectedIndex = -1"');
         HTP.PRINT ('<br><center>Select One or More Characteristics<br>');
         HTP.PRINT (
            '<table width="95%" class="tablebord zebra"><tr><td align="center">');
         HTP.PRINT ('<form name="select_char">');

         HTP.PRINT (
            '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
         HTP.formSelectOpen (
            'as_char',
            nsize         => '10',
            cattributes   => 'tabindex=1 multiple style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
         HTP.formSelectOption ('CHARACTERISTIC NAME',
                               cattributes   => 'value=0');
         HTP.FormSelectOption (
            '________________________________________________________',
            cattributes   => 'value=0');

--CCB --2012-09-12: DISABLE Taxon Filter for select Char Alias Type
--       IF taxon_filter = 'ON'
         IF taxon_filter = 'ON' AND v_char_alias_type_name <> 'RETIRED NAME (TAXON)'
         THEN
            FOR cv_char IN c_CharList
            LOOP
               HTP.FormSelectOption (
                  cv_char.display_name,
                  cattributes   => 'value =' || cv_char.pk_isn);
            END LOOP;
         ELSE
            FOR cv_char IN c_CharListTaxon
            LOOP
               HTP.FormSelectOption (
                  cv_char.display_name,
                  cattributes   => 'value =' || cv_char.pk_isn);
            END LOOP;
         END IF;
      ELSE
         -- for char alias searches, display options in two columns.  the first column contains the
         -- default STORET display char name.  The second contains the alias name.
         HTP.PRINT (
            '<BODY class="basic wide" style="background:#f7f2f2;" onload=adjust_width_n(document.forms[0].as_char,"~","60~60","None","N");>');
         HTP.PRINT ('<br><center>Select One or More Characteristics<br>');
         HTP.PRINT (
            '<table width="95%" class="tablebord zebra"><tr><td align="center">');
         HTP.PRINT ('<form name="select_char">');
         HTP.PRINT (
            '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');

         HTP.formSelectOpen (
            'as_char',
            nsize         => '10',
            cattributes   => 'tabindex=1 multiple style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
         HTP.formSelectOption (
            'CHARACTERISTIC NAME~CHARACTERISTIC ALIAS NAME',
            cattributes   => 'value=0');
         HTP.FormSelectOption (
            '________________________________________________________________________________________________________________________',
            cattributes   => 'value=0');


--CCB --2012-09-12: DISABLE Taxon Filter for select Char Alias Type
--       IF taxon_filter = 'ON'
         IF taxon_filter = 'ON' AND v_char_alias_type_name <> 'RETIRED NAME (TAXON)'
         THEN
            FOR cv_char IN c_CharAliasList
            LOOP
               HTP.FormSelectOption (
                     cv_char.display_name
                  || '~'
                  || SUBSTR (cv_char.char_alias, 0, 60),
                  cattributes   => 'value =' || cv_char.pk_isn);
            END LOOP;
         ELSE
            FOR cv_char IN c_CharAliasListTaxon
            LOOP
               HTP.FormSelectOption (
                     cv_char.display_name
                  || '~'
                  || SUBSTR (cv_char.char_alias, 0, 60),
                  cattributes   => 'value =' || cv_char.pk_isn);
            END LOOP;
         END IF;
      END IF;

      HTP.FormSelectClose;
      HTP.PRINT (
         '<br><br><INPUT TYPE="button" VALUE="Select" onclick="passback(''CHAR_ALIAS'')"> <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;
      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);
      HTP.bodyClose;
      HTP.htmlClose;
   END char_alias_popup;

   PROCEDURE extref_popup (v_org NUMBER)
   AS
      v_OrgName     VARCHAR2 (256);
      script_path   VARCHAR2 (1000);

      CURSOR cextref
      IS
           SELECT pk_isn, ref_scheme_acronym, ref_scheme_name
             FROM LU_EXTERNAL_REF_SCHEME
            WHERE organization_id IN (SELECT organization_id
                                        FROM DI_ORG
                                       WHERE pk_isn = v_org)
         ORDER BY ref_scheme_acronym;
   BEGIN
      script_path := DW_COMP.lv_script_path;

      SELECT organization_name
        INTO v_OrgName
        FROM di_org
       WHERE pk_isn = v_org;

      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');
      style;
      HTP.title ('STORET: Select a Station Alias type');
      HTP.headClose;

      HTP.PRINT (
         '<BODY class="basic wide" style="background:#f7f2f2;" onload=adjust_width_n(document.forms[0].D_EXTREF_LIST,"~","20~60","None","N");>');
      HTP.PRINT (
            '<br><center>Select a Station Alias Type for the Organization: '
         || v_OrgName
         || '<br>');

      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');

      HTP.PRINT ('<form name="select_extref">');
      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
      HTP.formSelectOpen (
         'D_EXTREF_LIST',
         nsize         => '10',
         cattributes   => 'tabindex=25  style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
      HTP.formSelectOption ('STATION ALIAS TYPE~DESCRIPTION',
                            cattributes   => 'VALUE=0');
      HTP.formSelectOption (
         '____________________________________________________________________________________',
         cattributes   => 'VALUE=0');
      HTP.formSelectOption ('STANDARD', cattributes => 'VALUE=0');

      FOR extref IN cextref
      LOOP
         HTP.formSelectOption (
            extref.ref_scheme_acronym || '~' || extref.ref_scheme_name,
            cattributes   => 'VALUE=' || extref.pk_isn);
      END LOOP;

      HTP.FormSelectClose;
      HTP.PRINT (
         '<br><br><INPUT TYPE="button" VALUE="Select" onclick="passback(''EXTREF'')"> <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END extref_popup;

   PROCEDURE proj_popup (v_org NUMBER, v_org_name VARCHAR2 DEFAULT NULL)
   AS
      v_OrgName     VARCHAR2 (256);
      script_path   VARCHAR2 (1000);
      nproj_name    di_project.project_name%TYPE;

      CURSOR cproj
      IS
           SELECT pk_isn, project_cd, project_name
             FROM DI_PROJECT
            WHERE fk_org = v_org
         ORDER BY project_cd;

      CURSOR cproj_nat
      IS
           SELECT COUNT (project_cd) IDC, project_cd, MIN (pk_isn) min_pk_isn
             FROM DI_PROJECT
         GROUP BY project_cd
         ORDER BY project_cd;
   BEGIN
      script_path := DW_COMP.lv_script_path;

      IF v_org = -1
      THEN
         v_OrgName := v_org_name;
      ELSE
         SELECT organization_name
           INTO v_OrgName
           FROM di_org
          WHERE pk_isn = v_org;
      END IF;

      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');
      style;
      HTP.title ('STORET: Select a Project');
      HTP.headClose;

      HTP.PRINT (
         '<BODY class="basic wide" style="background:#f7f2f2;" onload=adjust_width_n(document.forms[0].D_PROJ_LIST,"~","15~60","None","N");>');
      HTP.PRINT (
            '<br><center>Select a Project for the Organization: '
         || v_OrgName
         || '<br>');

      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');

      HTP.PRINT ('<form name="select_proj">');
      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
      HTP.formSelectOpen (
         'D_PROJ_LIST',
         nsize         => '10',
         cattributes   => 'tabindex=25 style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
      HTP.formSelectOption ('PROJECT ID~PROJECT NAME',
                            cattributes   => 'VALUE=0');
      HTP.formSelectOption (
         '_______________________________________________________________________________',
         cattributes   => 'VALUE=0');

      IF v_org = -1
      THEN
         FOR proj IN cproj_nat
         LOOP
            IF proj.IDC >= DW_glob_var.national_project_criteria
            THEN
               SELECT project_name
                 INTO nproj_name
                 FROM DI_PROJECT
                WHERE pk_isn = proj.min_pk_isn;

               IF LENGTH (nproj_name) < 35
               THEN
                  nproj_name := nproj_name || ')';
               ELSE
                  nproj_name := SUBSTR (nproj_name, 1, 32) || '...';
               END IF;

               HTP.formSelectOption (
                     proj.project_cd
                  || '~'
                  || 'National Project (e.g., '
                  || nproj_name,
                  cattributes   => 'VALUE=' || proj.min_pk_isn);
            END IF;
         END LOOP;
      ELSE
         FOR proj IN cproj
         LOOP
            HTP.formSelectOption (
               proj.project_cd || '~' || proj.project_name,
               cattributes   => 'VALUE=' || proj.pk_isn);
         END LOOP;
      END IF;


      HTP.FormSelectClose;
      HTP.PRINT (
         '<br><br><INPUT TYPE="button" VALUE="Select" onclick="passback(''PROJ'')"> <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END proj_popup;

   PROCEDURE org_alias_popup (v_OrgString CHAR)
   AS
      v_OrgStringCap   VARCHAR2 (20) := UPPER (v_OrgString);
      script_path      VARCHAR2 (1000);

      -- query by search name and retrieve display name
      CURSOR c_OrgList
      IS
           SELECT ORGANIZATION_ID, ORGANIZATION_NAME, pk_isn
             FROM DI_ORG
            WHERE    UPPER (ORGANIZATION_NAME) LIKE
                        '%' || v_OrgStringCap || '%'
                  OR UPPER (ORGANIZATION_ID) LIKE '%' || v_OrgStringCap || '%'
                  OR UPPER (PARENT_ORG) LIKE '%' || v_OrgStringCap || '%'
         ORDER BY ORGANIZATION_name;
   BEGIN
      script_path := dw_comp.lv_script_path;
      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');

      style;

      HTP.title ('STORET: Organization Search Results');

      HTP.headClose;

      -- use char cursor if default STORET names is chosen, else use char alias cursors
      HTP.bodyOpen (
         '',
         'class="basic wide" style="background:#f7f2f2;" onload=''adjust_width_n(document.forms[0].as_org,"~","30~70","None","N");''');
      HTP.PRINT ('<br><center>Select One or More Organizations<br>');

      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');

      HTP.PRINT ('<form name="select_org">');

      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
      HTP.formSelectOpen (
         'as_org',
         nsize         => '10',
         cattributes   => 'tabindex=1 multiple style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
      HTP.formSelectOption ('ORG ID~ORGANIZATION NAME',
                            cattributes   => 'value=0');
      HTP.FormSelectOption (
         '_________________________________________________________________________________________________________',
         cattributes   => 'value=0');

      FOR cv_org IN c_OrgList
      LOOP
         HTP.FormSelectOption (
               cv_org.organization_id
            || '~'
            || SUBSTR (cv_org.organization_name, 0, 70),
            cattributes   => 'value=' || cv_org.pk_isn);
      END LOOP;

      HTP.PRINT (
         '</select><br><br><INPUT TYPE="button" VALUE="Select" onclick="passback(''ORGANIZATION_NAME'')">
        <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');

      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END org_alias_popup;

   PROCEDURE station_popup (v_StationString    CHAR,
                            v_org              NUMBER,
                            v_searchtype       CHAR,
                            v_extref           CHAR)
   AS
      v_OrgName            VARCHAR2 (256);
      v_StationStringCap   VARCHAR2 (20) := UPPER (v_StationString);
      script_path          VARCHAR2 (1000);

      -- Based on the search type that was selected on the Results by Station page
      -- and passed in, one of the following queries are used to retrieve data for the
      -- station list.

      --   if search type is "Name",
      --   select data where the Station Name matches the search string
      CURSOR c_SearchNameList
      IS
           SELECT organization_id,
                  station_id,
                  station_name,
                  pk_isn
             FROM FA_STATION
            WHERE     UPPER (station_name) LIKE v_StationStringCap || '%'
                  AND location_point_type = '*POINT OF RECORD'
                  AND fk_org = v_org
         ORDER BY station_name;

      --   if search type is "ID",
      --   select data where the Station ID matches the search string
      CURSOR c_SearchIDList
      IS
           SELECT organization_id,
                  station_id,
                  station_name,
                  pk_isn
             FROM FA_STATION
            WHERE     station_ID LIKE v_StationStringCap || '%'
                  AND location_point_type = '*POINT OF RECORD'
                  AND fk_org = v_org
         ORDER BY station_id;

      --   if search type is "Label",
      --   select data where the Station Label under the selected External Ref Scheme
      --   matches the search string
      CURSOR c_SearchLabelList
      IS
           SELECT b.organization_id,
                  b.station_id,
                  a.station_label,
                  b.station_name,
                  b.pk_isn
             FROM LU_STATION_ALIAS a, FA_STATION b
            WHERE     UPPER (a.station_label) LIKE v_StationStringCap || '%'
                  AND a.fk_station = b.pk_isn
                  AND b.location_point_type = '*POINT OF RECORD'
                  AND b.fk_org = v_org
                  AND EXISTS
                         (SELECT 'x'
                            FROM LU_EXTERNAL_REF_SCHEME c
                           WHERE     c.ref_scheme_acronym =
                                        RPAD (v_extref, 8, ' ')
                                 AND c.pk_isn = a.fk_ref_scheme
                                 AND c.organization_id = b.organization_id)
         ORDER BY a.station_label;
   BEGIN
      script_path := DW_COMP.lv_script_path;

      -- get the org name to be displayed on the web page
      SELECT organization_name
        INTO v_OrgName
        FROM di_org
       WHERE pk_isn = v_org;

      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');
      style;
      HTP.title ('STORET: Station Search Results');
      HTP.headClose;

      HTP.PRINT (
         '<BODY class="basic wide" style="background:#f7f2f2;" onload=adjust_width_n(document.forms[0].d_station_list,"~","20~16~13~21~30","None","N");>');
      HTP.PRINT (
            '<br><center>Select One or More Stations from the Organization: '
         || v_OrgName
         || '<br>');
      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');
      HTP.PRINT ('<form name="select_stations">');
      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
      HTP.formSelectOpen (
         'd_station_list',
         nsize         => '10',
         cattributes   => 'tabindex=1 multiple style="font-size: 8pt; font-family: Courier New, Courier, monospace"');

      HTP.formSelectOption (
         'ORG ID~STATION ID~ALIAS TYPE~STATION ALIAS~STATION NAME',
         cattributes   => 'value=0');
      HTP.FormSelectOption (
         '____________________________________________________________________________________________________________________________',
         cattributes   => 'value=0');

      IF v_searchtype = 'Name'
      THEN
         FOR cv_station IN c_SearchNameList
         LOOP
            HTP.FormSelectOption (
                  cv_station.organization_id
               || '~'
               || cv_station.station_id
               || '~N/A~N/A~'
               || SUBSTR (cv_station.station_name, 1, 30),
               cattributes   => 'value =' || cv_station.pk_isn);
         END LOOP;
      ELSIF v_searchtype = 'ID' OR v_extref = 'STANDARD'
      THEN
         FOR cv_station IN c_SearchIDList
         LOOP
            HTP.FormSelectOption (
                  cv_station.organization_id
               || '~'
               || cv_station.station_id
               || '~N/A~N/A~'
               || SUBSTR (cv_station.station_name, 1, 30),
               cattributes   => 'value =' || cv_station.pk_isn);
         END LOOP;
      ELSE
         FOR cv_station IN c_SearchLabelList
         LOOP
            HTP.FormSelectOption (
                  cv_station.organization_id
               || '~'
               || cv_station.station_id
               || '~'
               || v_extref
               || '~'
               || cv_station.station_label
               || '~'
               || SUBSTR (cv_station.station_name, 1, 30),
               cattributes   => 'value =' || cv_station.pk_isn);
         END LOOP;
      END IF;

      HTP.FormSelectClose;
      HTP.PRINT (
         '<br><INPUT TYPE="button" size="2" VALUE="Select" onclick="passback(''STATION'')"> <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');
      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END station_popup;

   PROCEDURE county_popup (v_StateCode NUMBER)
   AS
      v_StateName   VARCHAR2 (40);
      script_path   VARCHAR2 (1000);

      CURSOR c_CountyList
      IS
         SELECT county_name, pk_isn
           FROM DI_GEO_COUNTY
          WHERE fk_geo_state = v_StateCode;
   BEGIN
      script_path := DW_COMP.lv_script_path;

      SELECT state_name
        INTO v_StateName
        FROM di_geo_state
       WHERE pk_isn = v_StateCode;

      HTP.htmlOpen;
      HTP.headOpen;
      HTP.PRINT (
         '<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">');

      style;

      HTP.title ('STORET: Select Counties');

      HTP.headClose;
      HTP.bodyOpen (
         NULL,
         'class="basic wide" style="background:#f7f2f2;" onload="document.forms[0].as_county.selectedIndex = -1"');

      HTP.PRINT (
            '<br><center>Select One or More Counties in '
         || v_StateName
         || '<br>');

      HTP.PRINT (
         '<table width="95%" class="tablebord zebra"><tr><td align="center">');

      HTP.PRINT ('<form name="select_county">');

      HTP.PRINT (
         '<FONT style="font-size: 10pt; font-family: Courier New, Courier, monospace">');
      HTP.formSelectOpen (
         cname         => 'as_county',
         nsize         => '10',
         cattributes   => 'multiple style="font-size: 8pt; font-family: Courier New, Courier, monospace"');
      HTP.FormSelectOption ('Select All', cattributes => 'value = ALL');

      FOR cv_state IN c_CountyList
      LOOP
         HTP.FormSelectOption (cv_state.county_name,
                               cattributes   => 'value =' || cv_state.pk_isn);
      END LOOP;

      HTP.FormSelectClose;
      HTP.PRINT (
         '<br><br><INPUT TYPE="button" VALUE="Select" onclick="passback(''COUNTY'')"> <INPUT TYPE="button" VALUE="Cancel" onclick="Close_Onclick()"></font>');
      HTP.formclose;

      HTP.PRINT ('</td></tr></table></center>');

      script (script_path);

      HTP.bodyClose;
      HTP.htmlClose;
   END county_popup;

   PROCEDURE display_calendar (DATE_NAME   IN VARCHAR2,
                               RANGE_NUM   IN NUMBER DEFAULT 1,
                               START_TAB   IN NUMBER DEFAULT 10,
                               FROM_YEAR   IN NUMBER DEFAULT 1900,
                               TO_YEAR     IN NUMBER DEFAULT 2050)
   AS
      is_current_month   VARCHAR2 (4) := NULL;
      current_month      VARCHAR2 (3);
      select_day         NUMBER DEFAULT 0;
      select_year        NUMBER DEFAULT TO_NUMBER (TO_CHAR (SYSDATE, 'YYYY'));
      date_range         VARCHAR2 (1);
      tabi               VARCHAR2 (3);
   BEGIN
      date_range := TO_CHAR (range_num);

      --MONTH
      SELECT TO_CHAR (SYSDATE, 'MON') INTO current_month FROM DUAL;

      --If populating the date from, then default start to Jan 1, 1900
      IF DATE_NAME = 'FROM'
      THEN
         tabi := TO_CHAR (START_TAB + ( (range_num - 1) * 6));

         IF DATE_RANGE = '1'
         THEN
            current_month := 'JAN';
         ELSE
            current_month := '---';
         END IF;
      ELSIF DATE_NAME = 'END'
      THEN
         tabi := TO_CHAR (START_TAB + 3 + ( (range_num - 1) * 6));

         IF DATE_RANGE <> '1'
         THEN
            current_month := '---';
         END IF;
      END IF;

      HTP.PRINT ('<font face="Courier">');

      HTP.PRINT (
            '<SELECT NAME="'
         || DATE_NAME
         || '_MON'
         || DATE_RANGE
         || '" SIZE="1" style="font: 10pt/12pt COURIER" tabindex='
         || tabi
         || ' onChange="ChangeOptionDays('''
         || DATE_NAME
         || ''','
         || DATE_RANGE
         || ')">');

      IF current_month = '---'
      THEN
         HTP.formSelectOption ('---',
                               cattributes   => 'VALUE= DUMMY',
                               cselected     => 'TRUE');
      END IF;

      is_current_month := NULL;

      IF current_month = 'JAN'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('JAN',
                            cattributes   => 'VALUE= JAN',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'FEB'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('FEB',
                            cattributes   => 'VALUE= FEB',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'MAR'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('MAR',
                            cattributes   => 'VALUE= MAR',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'APR'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('APR',
                            cattributes   => 'VALUE= APR',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'MAY'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('MAY',
                            cattributes   => 'VALUE= MAY',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'JUN'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('JUN',
                            cattributes   => 'VALUE= JUN',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'JUL'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('JUL',
                            cattributes   => 'VALUE= JUL',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'AUG'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('AUG',
                            cattributes   => 'VALUE= AUG',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'SEP'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('SEP',
                            cattributes   => 'VALUE= SEP',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'OCT'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('OCT',
                            cattributes   => 'VALUE= OCT',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'NOV'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('NOV',
                            cattributes   => 'VALUE= NOV',
                            CSELECTED     => is_current_month);
      is_current_month := NULL;

      IF current_month = 'DEC'
      THEN
         is_current_month := 'TRUE';
      END IF;

      HTP.formSelectOption ('DEC',
                            cattributes   => 'VALUE= DEC',
                            CSELECTED     => is_current_month);
      HTP.formSelectClose;
      HTP.PRINT ('</font>');

      --DAY
      --If populating the date from, then default start to Jan 1, 1900
      IF DATE_NAME = 'FROM'
      THEN
         tabi := TO_CHAR (START_TAB + 1 + ( (range_num - 1) * 6));

         IF DATE_RANGE = '1'
         THEN
            select_day := 1;
         ELSE
            select_day := 0;
         END IF;
      ELSIF DATE_NAME = 'END'
      THEN
         tabi := TO_CHAR (START_TAB + 4 + ( (range_num - 1) * 6));

         IF DATE_RANGE = '1'
         THEN
            select_day := TO_CHAR (SYSDATE, 'DD');
         ELSE
            select_day := 0;
         END IF;
      END IF;

      HTP.PRINT ('<font face="Courier">');

      HTP.PRINT (
            '<SELECT NAME="'
         || DATE_NAME
         || '_DD'
         || DATE_RANGE
         || '" SIZE="1" style="font: 10pt/12pt COURIER" tabindex='
         || tabi
         || '>');

      IF select_day = 0
      THEN
         HTP.formSelectOption ('--',
                               cattributes   => 'VALUE= 0',
                               cselected     => 'TRUE');
      END IF;

      FOR i IN 1 .. 31
      LOOP
         IF i = select_day
         THEN
            HTP.formSelectOption (i,
                                  cattributes   => 'VALUE=' || i,
                                  cselected     => 'TRUE');
         ELSE
            HTP.formSelectOption (i, cattributes => 'VALUE=' || i);
         END IF;
      END LOOP;

      HTP.formSelectClose;
      HTP.PRINT ('</font>');

      --YEAR
      IF DATE_NAME = 'FROM'
      THEN
         tabi := TO_CHAR (START_TAB + 2 + ( (range_num - 1) * 6));

         IF DATE_RANGE = '1'
         THEN
            select_year := 1900;
         ELSE
            select_year := 0;
         END IF;
      ELSIF DATE_NAME = 'END'
      THEN
         tabi := TO_CHAR (START_TAB + 5 + ( (range_num - 1) * 6));

         IF DATE_RANGE = '1'
         THEN
            select_year := TO_NUMBER (TO_CHAR (SYSDATE, 'YYYY'));
         ELSE
            select_year := 0;
         END IF;
      END IF;

      HTP.PRINT ('<font face="Courier">');

      HTP.PRINT (
            '<SELECT NAME="'
         || DATE_NAME
         || '_YYYY'
         || DATE_RANGE
         || '" SIZE="1" style="font: 10pt/12pt COURIER" tabindex='
         || tabi
         || ' onChange="ChangeOptionDays('''
         || DATE_NAME
         || ''','
         || DATE_RANGE
         || ')">');

      IF select_year = 0
      THEN
         HTP.formSelectOption ('----',
                               cattributes   => 'VALUE= 0',
                               cselected     => 'TRUE');
      END IF;

      FOR i IN FROM_YEAR .. TO_YEAR
      LOOP
         IF i = select_year
         THEN
            HTP.formSelectOption (i,
                                  cattributes   => 'VALUE=' || i,
                                  cselected     => 'TRUE');
         ELSE
            HTP.formSelectOption (i, cattributes => 'VALUE=' || i);
         END IF;
      --
      END LOOP;

      HTP.formSelectClose;
      HTP.PRINT ('</font>');
   END display_calendar;

   PROCEDURE station_report_form (v_stationcount   IN NUMBER,
                                  v_fromstmt       IN VARCHAR2,
                                  v_wherestmt      IN VARCHAR2,
                                  v_filters        IN VARCHAR2,
                                  v_bottom_form    IN BOOLEAN)
   AS
   BEGIN
      IF v_StationCount = 0
      THEN
         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         HTP.PRINT (
            'No Stations match your search criteria. Please select ''Back'' to modify your search.<br><br>');
         HTP.PRINT (
            '<form><input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();></form>');

         HTP.PRINT ('</div>');
      ELSIF v_StationCount > dw_glob_var.gv_stn_Overnight_lmt
      THEN
         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         HTP.PRINT (
            'The number of Stations that match your search criteria has exceeded the <BR> allowable');
         HTP.PRINT (
               'Stations Report size limit of '
            || TO_CHAR (dw_glob_var.gv_stn_Overnight_lmt, '999,999,999')
            || '.');
         HTP.PRINT ('Please select ''Back'' to modify your search.<br><br>');
         HTP.PRINT (
            '<form><input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();></form>');

         HTP.PRINT ('</div>');
      ELSIF (    v_StationCount > 0            --dw_glob_var.gv_stn_online_lmt
             AND v_StationCount <= dw_glob_var.gv_stn_Immediate_lmt)
      THEN
         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<form name=formz action="dw_pages.station_hub" method=POST>');

            HTP.PRINT (
                  '<input type="hidden" name="v_FromStmt" value="'
               || dw_util.encrypt_text (v_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_WhereStmt" value="'
               || dw_util.encrypt_text (v_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_StationCount" value="'
               || v_StationCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_filters" value="'
               || v_filters
               || '">');
            HTP.PRINT (
               '<input type="hidden" name="v_mode" value="Immediate">');
         END IF;

         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         -- provide Immediate and Overnight options
         HTP.PRINT ('<ul>');
         HTP.PRINT (
            '<li>Select ''Back'' to modify search parameters and refine your query.</li>');
         HTP.PRINT (
            '<li>Select Batch Processing to generate the report offline based on your current selections.</li>');

         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<li>You may customize the content of your report by selecting Data Elements below.</li>');
         END IF;

         HTP.PRINT ('</ul>');

         HTP.PRINT (
            '<input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();><br>');

         HTP.PRINT ('<center><h4>Batch Processing</h4>');

         IF NOT v_bottom_form
         THEN
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
-- /** --Begin Remark
--
            HTP.PRINT ('
                Please select the appropiate user profile:&nbsp;
		<select name="v_userprofile">
		<option value=8>Public</option>
		<option value=6>University/College</option>
		<option value=9>Tribal</option>
		<option value=2>State/US Government</option>
		<option value=5>Private Non-Industrial</option>
		<option value=7>Private Industry</option>
		<option value=4>Mexican Government/State</option>
		<option value=3>Local/US Government</option>
		<option value=10>Interstate Comsn/US Government</option>
		<option value=1>Federal/US Government</option>
		<!-- <option value=101>CERCLA Superfund</option>
		-->
		<!-- <option value=100>BEACH Program</option>
		-->
		<!-- <option value=111>Izaak Walton League</option>
		-->
		<!-- <option value=110>River Rally River Network</option>
		-->
		</select>
	    <br>
	    ');
--
-- **/ --End Remark
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
            HTP.PRINT (
               'Please enter your email address:&nbsp;<input type="text" size=50 maxlength=50 name=''v_email''><br>');
            HTP.PRINT (
               'Please specify three characters to prefix your report name:&nbsp;<input type="text" size=3 maxlength=3 name=''v_prefix''><br><br>');
         END IF;

         HTP.PRINT (
            '<input type="button" value="Immediate" width=40 onClick="document.formz.v_mode.value=''Immediate'';valiDeferInput();">');
         HTP.PRINT (
            '<input type="button" value="Overnight"  width=40 onClick="document.formz.v_mode.value=''Overnight'';valiDeferInput();"></center><br>');
         HTP.PRINT (
            'Note: immediate batch processing is limited to data requests with results up to ');
         HTP.PRINT (dw_glob_var.gv_stn_Immediate_lmt || '.');
         HTP.PRINT (
            'Any immediate batch request over the limit will be automatically scheduled as Overnight Batch.<br><br>');
         HTP.PRINT (
            'Note: Download files will exceed maximum limits for spread sheet applications (ie. EXCEL - Worksheet size: 65,536 Rows.)');

         HTP.PRINT ('</div>');

         IF v_bottom_form
         THEN
            HTP.PRINT ('</form>');
         END IF;
      ELSIF     v_StationCount > dw_glob_var.gv_stn_Immediate_lmt
            AND v_StationCount <= dw_glob_var.gv_stn_Overnight_lmt
      THEN
         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<form name=formz action="dw_pages.station_hub" method=POST>');

            HTP.PRINT (
                  '<input type="hidden" name="v_FromStmt" value="'
               || dw_util.encrypt_text (v_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_WhereStmt" value="'
               || dw_util.encrypt_text (v_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_StationCount" value="'
               || v_StationCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_filters" value="'
               || v_filters
               || '">');
            HTP.PRINT (
               '<input type="hidden" name="v_mode" value="Overnight">');
         END IF;

         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         -- provide Overnight option only
         HTP.PRINT ('<ul>');
         HTP.PRINT (
            '<li>Select ''Back'' to modify search parameters and refine your query.</li>');
         HTP.PRINT (
            '<li>Select Batch Processing to generate the report offline based on your current selections.</li>');

         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<li>You may customize the content of your report by selecting Data Elements below.</li>');
         END IF;

         HTP.PRINT ('</ul>');

         HTP.PRINT (
            '<input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();><br>');

         HTP.PRINT ('<center><h4>Batch Processing</h4>');

         IF NOT v_bottom_form
         THEN
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
-- /** --Begin Remark
--
            HTP.PRINT ('
                Please select the appropiate user profile:&nbsp;
		<select name="v_userprofile">
		<option value=8>Public</option>
		<option value=6>University/College</option>
		<option value=9>Tribal</option>
		<option value=2>State/US Government</option>
		<option value=5>Private Non-Industrial</option>
		<option value=7>Private Industry</option>
		<option value=4>Mexican Government/State</option>
		<option value=3>Local/US Government</option>
		<option value=10>Interstate Comsn/US Government</option>
		<option value=1>Federal/US Government</option>
		<!-- <option value=101>CERCLA Superfund</option>
		-->
		<!-- <option value=100>BEACH Program</option>
		-->
		<!-- <option value=111>Izaak Walton League</option>
		-->
		<!-- <option value=110>River Rally River Network</option>
		-->
		</select>
	    <br>
	    ');
--
-- **/ --End Remark
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
            HTP.PRINT (
               'Please enter your email address:&nbsp;<input type="text" size=50 maxlength=50 name=''v_email''><br>');
            HTP.PRINT (
               'Please specify three characters to prefix your report name:&nbsp;<input type="text" size=3 maxlength=3 name=''v_prefix''><br><br>');
         END IF;

         HTP.PRINT (
            '<input type="button" value="Overnight"  width=40 onClick="document.formz.v_mode.value=''Overnight'';valiDeferInput();"></center><br>');
         HTP.PRINT (
            'Note: Download files will exceed maximum limits for spread sheet applications (ie. EXCEL - Worksheet size: 65,536 Rows.)');

         HTP.PRINT ('</div>');

         IF v_bottom_form
         THEN
            HTP.PRINT ('</form>');
         END IF;
      END IF;
   END station_report_form;

   PROCEDURE result_report_form (v_resultcount         IN NUMBER,
                                 v_fromstmt            IN VARCHAR2,
                                 v_wherestmt           IN VARCHAR2,
                                 vb_resultcount        IN NUMBER,
                                 vb_fromstmt           IN VARCHAR2,
                                 vb_wherestmt          IN VARCHAR2,
                                 vh_resultcount        IN NUMBER,
                                 vh_fromstmt           IN VARCHAR2,
                                 vh_wherestmt          IN VARCHAR2,
                                 vr_resultcount        IN NUMBER,
                                 vr_fromstmt           IN VARCHAR2,
                                 vr_wherestmt          IN VARCHAR2,
--CCB --2012-06-22: metrics
                                 vm_resultcount        IN NUMBER,
                                 vm_fromstmt           IN VARCHAR2,
                                 vm_wherestmt          IN VARCHAR2,
--
                                 v_filters             IN VARCHAR2,
                                 v_rule_optimization   IN VARCHAR2,
                                 result_type           IN VARCHAR2,
                                 v_bottom_form         IN BOOLEAN)
   AS
   BEGIN
      IF v_ResultCount = 0
      THEN
         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         HTP.PRINT (
            'No Results match your search criteria. Please select ''Back'' to modify your search.<br><br>');
         HTP.PRINT (
            '<form><input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();></form>');

         HTP.PRINT ('</div>');
      ELSIF v_ResultCount > dw_glob_var.gv_rslt_Overnight_lmt
      THEN
         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         HTP.PRINT (
            'The number of Results that match your search criteria has exceeded the <BR> allowable');
         HTP.PRINT (
               'Result Report size limit of '
            || TO_CHAR (dw_glob_var.gv_rslt_Overnight_lmt, '999,999,999')
            || '.');
         HTP.PRINT ('Please select ''Back'' to modify your search.<br><br>');
         HTP.PRINT (
            '<form><input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();></form>');

         HTP.PRINT ('</div>');
      ELSIF (    v_ResultCount > 0           -- dw_glob_var.gv_rslt_online_lmt
             AND v_ResultCount <= dw_glob_var.gv_rslt_Immediate_lmt)
      THEN
         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<form name=formz action="dw_pages.result_hub" method=POST>');
            HTP.PRINT ('<div class="reportTypes">');
            HTP.PRINT ('<h5>Select Report Type(s):</h5><center>');

            IF (vR_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-regular"><input type="checkbox" name="r_Type" value="REGULAR" CHECKED>&nbsp;REGULAR&nbsp;('
                  || vR_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-regular"><input type="checkbox" name="r_Type" value="REGULAR" disabled="disabled">&nbsp;REGULAR&nbsp;('
                  || vR_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

            IF (vB_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-biological"><input type="checkbox" name="r_Type" value="BIOLOGICAL" CHECKED>&nbsp;BIOLOGICAL&nbsp;('
                  || vB_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-biological"><input type="checkbox" name="r_Type" value="BIOLOGICAL" disabled="disabled">&nbsp;BIOLOGICAL&nbsp;('
                  || vB_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

            IF (vH_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-habitat"><input type="checkbox" name="r_Type" value="HABITAT" CHECKED>&nbsp;HABITAT&nbsp;('
                  || vH_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-habitat"><input type="checkbox" name="r_Type" value="HABITAT" disabled="disabled">&nbsp;HABITAT&nbsp;('
                  || vH_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

--CCB --2012-06-22: metrics
            IF (vM_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-metric"><input type="checkbox" name="r_Type" value="METRIC" CHECKED>&nbsp;METRIC&nbsp;('
                  || vM_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-metric"><input type="checkbox" name="r_Type" value="METRIC" disabled="disabled">&nbsp;METRIC&nbsp;('
                  || vM_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;
--

            HTP.PRINT ('</center>');
            HTP.PRINT ('</div>');

            HTP.PRINT (
                  '<input type="hidden" name="v_FromStmt" value="'
               || dw_util.encrypt_text (v_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_WhereStmt" value="'
               || dw_util.encrypt_text (v_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_FromStmt" value="'
               || dw_util.encrypt_text (vR_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_WhereStmt" value="'
               || dw_util.encrypt_text (vR_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_FromStmt" value="'
               || dw_util.encrypt_text (vB_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_WhereStmt" value="'
               || dw_util.encrypt_text (vB_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_FromStmt" value="'
               || dw_util.encrypt_text (vH_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_WhereStmt" value="'
               || dw_util.encrypt_text (vH_WhereStmt)
               || '">');
--CCB --2012-06-22: metrics
            HTP.PRINT (
                  '<input type="hidden" name="vM_FromStmt" value="'
               || dw_util.encrypt_text (vM_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vM_WhereStmt" value="'
               || dw_util.encrypt_text (vM_WhereStmt)
               || '">');
--
            HTP.PRINT (
                  '<input type="hidden" name="v_ResultCount" value="'
               || v_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_ResultCount" value="'
               || vR_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_ResultCount" value="'
               || vB_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_ResultCount" value="'
               || vH_ResultCount
               || '">');
--CCB --2012-06-22: metrics
            HTP.PRINT (
                  '<input type="hidden" name="vM_ResultCount" value="'
               || vM_ResultCount
               || '">');
--
            HTP.PRINT (
                  '<input type="hidden" name="v_filters" value="'
               || v_filters
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_rule_optimization" value="'
               || v_rule_optimization
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_report_type" value="'
               || result_type
               || '">');
            HTP.PRINT (
               '<input type="hidden" name="v_mode" value="Immediate">');
         END IF;

         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         -- provide Immediate and Overnight options
         HTP.PRINT ('<ul>');
         HTP.PRINT (
            '<li>Select ''Back'' to modify search parameters and refine your query.</li>');
         HTP.PRINT (
            '<li>Select Batch Processing to generate the report offline based on your current selections.</li>');

         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<li>You may customize the content of your report by selecting Data Elements below.</li>');
         END IF;

         HTP.PRINT ('</ul>');


         HTP.PRINT (
            '<input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();><br>');

         HTP.PRINT ('<center><h4>Batch Processing</h4>');

         IF NOT v_bottom_form
         THEN
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
-- /** --Begin Remark
--
            HTP.PRINT ('
                Please select the appropiate user profile:&nbsp;
		<select name="v_userprofile">
		<option value=8>Public</option>
		<option value=6>University/College</option>
		<option value=9>Tribal</option>
		<option value=2>State/US Government</option>
		<option value=5>Private Non-Industrial</option>
		<option value=7>Private Industry</option>
		<option value=4>Mexican Government/State</option>
		<option value=3>Local/US Government</option>
		<option value=10>Interstate Comsn/US Government</option>
		<option value=1>Federal/US Government</option>
		<!-- <option value=101>CERCLA Superfund</option>
		-->
		<!-- <option value=100>BEACH Program</option>
		-->
		<!-- <option value=111>Izaak Walton League</option>
		-->
		<!-- <option value=110>River Rally River Network</option>
		-->
		</select>
	    <br>
	    ');
--
-- **/ --End Remark
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
            HTP.PRINT (
               'Please enter your email address:&nbsp;<input type="text" size=50 maxlength=50 name=''v_email''></label><br>');
            HTP.PRINT (
               'Please specify three characters to prefix your report name:&nbsp;<input type="text" size=3 maxlength=3 name=''v_prefix''><br><br>');
         END IF;

         HTP.PRINT (
            '<input type="button" value="Immediate" width=40 onClick=document.formz.v_mode.value=''Immediate'';valiDeferInput();>');
         HTP.PRINT (
            '<input type="button" value="Overnight"  width=40 onClick=document.formz.v_mode.value=''Overnight'';valiDeferInput();></center><br>');
         HTP.PRINT (
            'Note: immediate batch processing is limited to data requests with results up to ');
         HTP.PRINT (dw_glob_var.gv_rslt_Immediate_lmt || '.');
         HTP.PRINT (
            'Any immediate batch request over the limit will be automatically scheduled as Overnight Batch.<br><br>');
         HTP.PRINT (
            'Note: Download files will exceed maximum limits for spread sheet applications (ie. EXCEL - Worksheet size: 65,536 Rows.)');

         HTP.PRINT ('</div>');

         IF v_bottom_form
         THEN
            HTP.PRINT ('</form>');
         END IF;
      ELSIF     v_ResultCount > dw_glob_var.gv_rslt_Immediate_lmt
            AND v_ResultCount <= dw_glob_var.gv_rslt_Overnight_lmt
      THEN
         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<form name=formz action="dw_pages.result_hub" method=POST>');
            HTP.PRINT ('<div class="reportTypes">');
            HTP.PRINT ('<h5>Select Report Type(s):</h5><center>');

            IF (vR_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-regular"><input type="checkbox" name="r_Type" value="REGULAR" CHECKED>&nbsp;REGULAR&nbsp;('
                  || vR_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-regular"><input type="checkbox" name="r_Type" value="REGULAR" disabled="disabled">&nbsp;REGULAR&nbsp;('
                  || vR_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

            IF (vB_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-biological"><input type="checkbox" name="r_Type" value="BIOLOGICAL" CHECKED>&nbsp;BIOLOGICAL&nbsp;('
                  || vB_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-biological"><input type="checkbox" name="r_Type" value="BIOLOGICAL" disabled="disabled">&nbsp;BIOLOGICAL&nbsp;('
                  || vB_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

            IF (vH_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-habitat"><input type="checkbox" name="r_Type" value="HABITAT" CHECKED>&nbsp;HABITAT&nbsp;('
                  || vH_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-habitat"><input type="checkbox" name="r_Type" value="HABITAT" disabled="disabled">&nbsp;HABITAT&nbsp;('
                  || vH_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;

--CCB --2012-06-22: metrics
            IF (vM_ResultCount > 0)
            THEN
               HTP.PRINT (
                     '<div class="reportType-metric"><input type="checkbox" name="r_Type" value="METRIC" CHECKED>&nbsp;METRIC&nbsp;('
                  || vM_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            ELSE
               HTP.PRINT (
                     '<div class="reportType-metric"><input type="checkbox" name="r_Type" value="METRIC" disabled="disabled">&nbsp;METRIC&nbsp;('
                  || vM_ResultCount
                  || '&nbsp;record(s))&nbsp;</div>');
            END IF;
--

            HTP.PRINT ('</center>');
            HTP.PRINT ('</div>');

            HTP.PRINT (
                  '<input type="hidden" name="v_FromStmt" value="'
               || dw_util.encrypt_text (v_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_WhereStmt" value="'
               || dw_util.encrypt_text (v_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_FromStmt" value="'
               || dw_util.encrypt_text (vR_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_WhereStmt" value="'
               || dw_util.encrypt_text (vR_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_FromStmt" value="'
               || dw_util.encrypt_text (vB_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_WhereStmt" value="'
               || dw_util.encrypt_text (vB_WhereStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_FromStmt" value="'
               || dw_util.encrypt_text (vH_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_WhereStmt" value="'
               || dw_util.encrypt_text (vH_WhereStmt)
               || '">');
--CCB --2012-06-22: metrics
            HTP.PRINT (
                  '<input type="hidden" name="vM_FromStmt" value="'
               || dw_util.encrypt_text (vM_FromStmt)
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vM_WhereStmt" value="'
               || dw_util.encrypt_text (vM_WhereStmt)
               || '">');
--
            HTP.PRINT (
                  '<input type="hidden" name="v_ResultCount" value="'
               || v_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vR_ResultCount" value="'
               || vR_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vB_ResultCount" value="'
               || vB_ResultCount
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="vH_ResultCount" value="'
               || vH_ResultCount
               || '">');
--CCB --2012-06-22: metrics
            HTP.PRINT (
                  '<input type="hidden" name="vM_ResultCount" value="'
               || vM_ResultCount
               || '">');
--
            HTP.PRINT (
                  '<input type="hidden" name="v_filters" value="'
               || v_filters
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_rule_optimization" value="'
               || v_rule_optimization
               || '">');
            HTP.PRINT (
                  '<input type="hidden" name="v_report_type" value="'
               || result_type
               || '">');
            HTP.PRINT (
               '<input type="hidden" name="v_mode" value="Immediate">');
         END IF;

         IF v_bottom_form
         THEN
            HTP.PRINT ('<div class="navControls lowerNavControls">');
         ELSE
            HTP.PRINT ('<div class="navControls upperNavControls">');
         END IF;

         -- provide Overnight option only
         HTP.PRINT ('<ul>');
         HTP.PRINT (
            '<li>Select ''Back'' to modify search parameters and refine your query.</li>');
         HTP.PRINT (
            '<li>Select Batch Processing to generate the report offline based on your current selections.</li>');

         IF NOT v_bottom_form
         THEN
            HTP.PRINT (
               '<li>You may customize the content of your report by selecting Data Elements below.</li>');
         END IF;

         HTP.PRINT ('</ul>');

         HTP.PRINT (
            '<input class="backButton" type="button" value="  <<Back  " width=40 onClick=Close_Onclick();><br>');

         HTP.PRINT ('<center><h4>Batch Processing</h4>');

         IF NOT v_bottom_form
         THEN
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
-- /** --Begin Remark
--
            HTP.PRINT ('
                Please select the appropiate user profile:&nbsp;
		<select name="v_userprofile">
		<option value=8>Public</option>
		<option value=6>University/College</option>
		<option value=9>Tribal</option>
		<option value=2>State/US Government</option>
		<option value=5>Private Non-Industrial</option>
		<option value=7>Private Industry</option>
		<option value=4>Mexican Government/State</option>
		<option value=3>Local/US Government</option>
		<option value=10>Interstate Comsn/US Government</option>
		<option value=1>Federal/US Government</option>
		<!-- <option value=101>CERCLA Superfund</option>
		-->
		<!-- <option value=100>BEACH Program</option>
		-->
		<!-- <option value=111>Izaak Walton League</option>
		-->
		<!-- <option value=110>River Rally River Network</option>
		-->
		</select>
	    <br>
	    ');
--
-- **/ --End Remark
--CCB --2012-05-22: User Profile (v_userprofile) added by Kevin Christian
            HTP.PRINT (
               'Please enter your email address:&nbsp;<input type="text" size=50 maxlength=50 name=''v_email''><br>');
            HTP.PRINT (
               'Please specify three characters to prefix your report name:&nbsp;<input type="text" size=3 maxlength=3 name=''v_prefix''><br><br>');
         END IF;

         HTP.PRINT (
            '<input type="button" value="Overnight"  width=40 onClick=document.formz.v_mode.value=''Overnight'';valiDeferInput();></center><br>');
         HTP.PRINT (
            'Note: Download files will exceed maximum limits for spread sheet applications (ie. EXCEL - Worksheet size: 65,536 Rows.)');

         HTP.PRINT ('</div>');

         IF v_bottom_form
         THEN
            HTP.PRINT ('</form>');
         END IF;
      END IF;
   END result_report_form;
END dw_comp;
/

