
  CREATE OR REPLACE PROCEDURE "STORETW"."SP_REFRESH_STATION_CHAR_PROJ" (
   p_record_no   OUT NUMBER,
   p_comments    OUT VARCHAR2)
AS
   l_res        NUMBER := 0;
   l_reg        NUMBER := 0;
   l_hab        NUMBER := 0;
   l_bio        NUMBER := 0;
   l_comments   VARCHAR2 (4000) := '';
BEGIN

   --truncate station_char_project and drop all indexes and FK constraints on station_char_project
   EXECUTE IMMEDIATE 'TRUNCATE TABLE storetw.station_char_project';

   FOR c IN (SELECT index_name
               FROM all_indexes
              WHERE owner = 'STORETW'
                AND table_name = 'STATION_CHAR_PROJECT')
   LOOP
      BEGIN
         EXECUTE IMMEDIATE 'DROP INDEX ' || c.index_name;
      EXCEPTION
         WHEN OTHERS
         THEN
            l_comments :=
                  l_comments
               || 'Drop index '
               || c.index_name
               || ' failed: '
               || SQLCODE
               || '. ';
      END;
   END LOOP;


   -----------------------------------------------------------------------------
   -- Load fa_regular_result into tmp_station_char_project
   -----------------------------------------------------------------------------
   INSERT /*+ APPEND */
         INTO  tmp_station_char_project (fk_station,
                                         fk_char,
                                         fk_project,
                                         result_count,
                                         regular_count,
                                         biological_count,
                                         habitat_count)
          SELECT /*+PARALLEL(a 4) PARALLEL(b 4)*/
                 fk_station,
                 fk_char,
                 fk_project,
                 COUNT(1) result_count,
                 COUNT(1) regular_count,
                 null as biological_count,
                 null as habitat_count
            FROM fa_regular_result a, regular_result_project b
           WHERE a.pk_isn = b.fk_result
             AND a.fk_char IS NOT NULL
             AND a.fk_station IS NOT NULL
          GROUP BY fk_project, fk_station, fk_char;

   l_reg := SQL%ROWCOUNT;

   COMMIT;

   -----------------------------------------------------------------------------
   -- Load fa_habitat_result into tmp_station_char_project
   -----------------------------------------------------------------------------
   INSERT /*+ APPEND */
         INTO  tmp_station_char_project (fk_station,
                                         fk_char,
                                         fk_project,
                                         result_count,
                                         regular_count,
                                         biological_count,
                                         habitat_count)
          SELECT /*+PARALLEL(a 4) PARALLEL(b 4)*/
                 fk_station,
                 fk_char,
                 fk_project,
                 COUNT(1) result_count,
                 null as regular_count,
                 null as biological_count,
                 COUNT(1) habitat_count
            FROM fa_habitat_result a, habitat_result_project b
           WHERE a.pk_isn = b.fk_result
             AND a.fk_char IS NOT NULL
             AND a.fk_station IS NOT NULL
          GROUP BY fk_project, fk_station, fk_char;

   l_hab := SQL%ROWCOUNT;

   COMMIT;

   -----------------------------------------------------------------------------
   -- Load fa_biological_result into tmp_station_char_project
   -----------------------------------------------------------------------------
   INSERT /*+ APPEND */
         INTO  tmp_station_char_project (fk_station,
                                         fk_char,
                                         fk_project,
                                         result_count,
                                         regular_count,
                                         biological_count,
                                         habitat_count)
          SELECT /*+PARALLEL(a 4) PARALLEL(b 4)*/
                 fk_station,
                 fk_char,
                 fk_project,
                 COUNT(1) result_count,
                 null as regular_count,
                 COUNT(1) biological_count,
                 null as habitat_count
            FROM fa_biological_result a, biological_result_project b
           WHERE a.pk_isn = b.fk_result
             AND a.fk_char IS NOT NULL
             AND a.fk_station IS NOT NULL
          GROUP BY fk_project, fk_station, fk_char;

   l_bio := SQL%ROWCOUNT;

   COMMIT;


   -----------------------------------------------------------------------------
   -- Load fa_regular_result into station_char_project
   -----------------------------------------------------------------------------
   INSERT /*+ APPEND */
         INTO  station_char_project (fk_station,
                                     fk_char,
                                     fk_project,
                                     fk_org,
                                     fk_gen_db_cat,
                                     result_count,
                                     regular_count,
                                     biological_count,
                                     habitat_count,
                                     station_id,
                                     station_name,
                                     stn_std_latitude,
                                     stn_std_longitude,
                                     chartype,
                                     characteristic_name,
                                     project_cd,
                                     project_name,
                                     project_por_start,
                                     project_por_end,
                                     organization_id,
                                     organization_name,
                                     hydrologic_unit_code)
      SELECT z.fk_station,
             z.fk_char,
             z.fk_project,
             fs.fk_org,
             fs.fk_gen_db_cat,
             z.result_count,
             z.regular_count,
             z.biological_count,
             z.habitat_count,
             trim(fs.station_id),
             fs.station_name,
             fs.std_latitude,
             fs.std_longitude,
             dc.chartype,
             dc.display_name,
             trim(dp.project_cd),
             dp.project_name,
             dp.project_por_start,
             dp.project_por_end,
             trim(do.organization_id),
             do.organization_name,
             trim(dd.hydrologic_unit_code)
        FROM (SELECT /*+DYNAMIC_SAMPLING(a 5) PARALLEL(a 4) NO_MERGE*/
                     fk_station,
                     fk_char,
                     fk_project,
                     sum(result_count) as result_count,
                     sum(regular_count) as regular_count,
                     sum(biological_count) as biological_count,
                     sum(habitat_count) as habitat_count
                FROM tmp_station_char_project a
               GROUP BY fk_project, fk_station, fk_char) z,
             fa_station fs, di_characteristic dc, di_project dp, di_org do, di_db_cat dd
       WHERE z.fk_station = fs.pk_isn
         AND z.fk_char = dc.pk_isn
         AND z.fk_project = dp.pk_isn
         AND fs.fk_org = do.pk_isn
         AND fs.fk_gen_db_cat = dd.pk_isn (+);

   l_res := SQL%ROWCOUNT;

   COMMIT;



   BEGIN
      EXECUTE IMMEDIATE
         'CREATE INDEX iscp_lat_long ON station_char_project (stn_std_latitude, stn_std_longitude) TABLESPACE storetindx PCTFREE 10 NOPARALLEL NOLOGGING';
   EXCEPTION
      WHEN OTHERS
      THEN
         l_comments :=
            l_comments || 'Create index iscp_lat_long failed: ' || SQLCODE || '. ';
   END;


   BEGIN
      EXECUTE IMMEDIATE
         'CREATE BITMAP INDEX iscp_station_id ON station_char_project(lower(station_id)) TABLESPACE storetindx PCTFREE 10 NOPARALLEL NOLOGGING';
   EXCEPTION
      WHEN OTHERS
      THEN
         l_comments :=
            l_comments || 'Create index iscp_station_id failed: ' || SQLCODE || '. ';
   END;


   BEGIN
      EXECUTE IMMEDIATE
         'CREATE BITMAP INDEX iscp_organization_id ON station_char_project(lower(organization_id)) TABLESPACE storetindx PCTFREE 10 NOPARALLEL NOLOGGING';
   EXCEPTION
      WHEN OTHERS
      THEN
         l_comments :=
            l_comments || 'Create index iscp_organization_id failed: ' || SQLCODE || '. ';
   END;


   BEGIN
      EXECUTE IMMEDIATE
         'CREATE BITMAP INDEX iscp_project_cd ON station_char_project(lower(project_cd)) TABLESPACE storetindx PCTFREE 10 NOPARALLEL NOLOGGING';
   EXCEPTION
      WHEN OTHERS
      THEN
         l_comments :=
            l_comments || 'Create index iscp_project_cd failed: ' || SQLCODE || '. ';
   END;

   BEGIN
      EXECUTE IMMEDIATE
         'CREATE BITMAP INDEX iscp_hydrologic_unit_code ON station_char_project(hydrologic_unit_code) TABLESPACE storetindx PCTFREE 10 NOPARALLEL NOLOGGING';
   EXCEPTION
      WHEN OTHERS
      THEN
         l_comments :=
            l_comments || 'Create index iscp_hydrologic_unit_code failed: ' || SQLCODE || '. ';
   END;


   DBMS_STATS.gather_table_stats (
      ownname            => 'STORETW',
      tabname            => 'STATION_CHAR_PROJECT',
      cascade            => TRUE,
      estimate_percent   => DBMS_STATS.auto_sample_size);


   p_record_no := l_reg + l_bio + l_hab;
   p_comments :=
         l_comments
      || 'INS: '
      || l_res || '. RR: '
      || l_reg
      || '. BR: '
      || l_bio
      || '. HR: '
      || l_hab
      || '.';
END sp_refresh_station_char_proj;
/

