
  CREATE OR REPLACE PROCEDURE "STORETW"."SP_UPDATE_FA_STATION_HUC" (
   p_load_date       DATE,
   p_limiter         NUMBER DEFAULT NULL,
   p_record_no   OUT NUMBER,
   p_comments    OUT VARCHAR2)
AS
   my_cursor     SYS_REFCURSOR;
   str_sql       VARCHAR2 (4000);
   str_sql2      VARCHAR2 (4000);
   num_records   NUMBER;
   num_limiter   NUMBER;

   TYPE tbl_rids  IS TABLE OF ROWID;
   TYPE tbl_pts   IS TABLE OF SDO_GEOMETRY;
   TYPE tbl_huc8  IS TABLE OF VARCHAR2(8);
   TYPE tbl_huc12 IS TABLE OF VARCHAR2(12);
   TYPE tbl_fks   IS TABLE OF NUMBER(12);

   ary_rids      tbl_rids;
   ary_sdo       tbl_pts;
   ary_huc8      tbl_huc8;
   ary_huc12     tbl_huc12;
   ary_fks       tbl_fks;

   l_count         NUMBER := 0;
   l_cnt_upd_st    NUMBER := 0;
   l_cnt_upd_cnt   NUMBER := 0;
   l_cnt_upd_huc   NUMBER := 0;
   l_cnt_upd_huc12 NUMBER := 0;
   l_cnt_gis_st    NUMBER := 0;
   l_cnt_gis_cnt   NUMBER := 0;
   l_cnt_gis_huc   NUMBER := 0;
BEGIN
   -----------------------------------------------------------------------------
   -- Step 10
   -- Apply the limiter
   -----------------------------------------------------------------------------
   IF p_limiter IS NULL
   THEN
      num_limiter := 10000;
   ELSE
      num_limiter := p_limiter;
   END IF;

   -----------------------------------------------------------------------------
   -- Step 20
   -- Update generated columns from user supplied values
   -----------------------------------------------------------------------------

   -- State
   update fa_station
      set fk_gen_geo_state = fk_geo_state
    where nvl(fk_gen_geo_state,-1) <> nvl(fk_geo_state,-1)
      and last_change_date >= p_load_date;

   l_cnt_upd_st := sql%rowcount;

   commit;


   -- County
   update fa_station
      set fk_gen_geo_county = fk_geo_county
    where nvl(fk_gen_geo_county,-1) <> nvl(fk_geo_county,-1)
      and last_change_date >= p_load_date;

   l_cnt_upd_cnt := sql%rowcount;

   commit;


   -- HUC8
   update fa_station
      set fk_gen_db_cat = fk_db_cat
    where nvl(fk_gen_db_cat,-1) <> nvl(fk_db_cat,-1)
      and last_change_date >= p_load_date;

   l_cnt_upd_huc := sql%rowcount;

   commit;


   -- HUC12
   update fa_station
      set gen_huctwelvedigitcode = huctwelvedigitcode
    where nvl(gen_huctwelvedigitcode,-1) <> nvl(huctwelvedigitcode,-1)
      and last_change_date >= p_load_date;

   l_cnt_upd_huc12 := sql%rowcount;

   commit;


   -----------------------------------------------------------------------------
   -- Step 30
   -- Update generated columns from geoemtries when no value provided by user
   -----------------------------------------------------------------------------

   -- State
   str_sql :=
         'select a.rowid, '
      || '       ( select max(b.pk_isn) '
      || '           from di_geo_state b '
      || '          where sdo_relate(b.geom, a.geom, ''MASK=ANYINTERACT'') = ''TRUE'' ) as pk_isn '
      || '  from fa_station a '
      || ' where a.fk_gen_geo_state is null '
      || '   and a.last_change_date >= :p01 '
      || ' order by 1, 2 ';

   OPEN my_cursor FOR str_sql USING p_load_date;

   LOOP
      FETCH my_cursor
      BULK COLLECT INTO ary_rids, ary_fks
      LIMIT num_limiter;

      EXIT WHEN ary_rids.COUNT = 0;

      FORALL i IN 1 .. ary_rids.COUNT
         UPDATE fa_station a
            SET a.fk_gen_geo_state = ary_fks(i)
          WHERE a.rowid = ary_rids(i);

      COMMIT;

      l_cnt_gis_st := l_cnt_gis_st + ary_rids.COUNT;
   END LOOP;

   CLOSE my_cursor;


   -- County
   str_sql :=
         'select a.rowid, '
      || '       ( select max(b.pk_isn) '
      || '           from di_geo_county b '
      || '          where sdo_relate(b.geom, a.geom, ''MASK=ANYINTERACT'') = ''TRUE'' ) as pk_isn '
      || '  from fa_station a '
      || ' where a.fk_gen_geo_county is null '
      || '   and a.last_change_date >= :p01 '
      || ' order by 1, 2 ';

   OPEN my_cursor FOR str_sql USING p_load_date;

   LOOP
      FETCH my_cursor
      BULK COLLECT INTO ary_rids, ary_fks
      LIMIT num_limiter;

      EXIT WHEN ary_rids.COUNT = 0;

      FORALL i IN 1 .. ary_rids.COUNT
         UPDATE fa_station a
            SET a.fk_gen_geo_county = ary_fks(i)
          WHERE a.rowid = ary_rids(i);

      COMMIT;

      l_cnt_gis_cnt := l_cnt_gis_cnt + ary_rids.COUNT;
   END LOOP;

   CLOSE my_cursor;


   -- HUC8
   str_sql :=
         'select z.rowid, z.pk_isn, c.hydrologic_unit_code '
      || '  from di_db_cat c, '
      || '       ( select a.rowid, '
      || '                ( select max(b.pk_isn) '
      || '                    from di_db_cat b '
      || '                   where sdo_relate(b.geom, a.geom, ''MASK=ANYINTERACT'') = ''TRUE'' ) as pk_isn '
      || '           from fa_station a '
      || '          where a.fk_gen_db_cat is null '
      || '            and a.last_change_date >= :p01 ) z '
      || ' where c.pk_isn = z.pk_isn '
      || ' order by 1, 2 ';

   OPEN my_cursor FOR str_sql USING p_load_date;

   LOOP
      FETCH my_cursor
      BULK COLLECT INTO ary_rids, ary_fks, ary_huc8
      LIMIT num_limiter;

      EXIT WHEN ary_rids.COUNT = 0;

      FORALL i IN 1 .. ary_rids.COUNT
         UPDATE fa_station a
            SET a.fk_gen_db_cat = ary_fks(i),
                a.generated_huc = ary_huc8(i)
          WHERE a.rowid = ary_rids(i);

      COMMIT;

      l_cnt_gis_huc := l_cnt_gis_huc + ary_rids.COUNT;
   END LOOP;

   CLOSE my_cursor;


   -----------------------------------------------------------------------------
   -- Step 40
   -- Update HUC12 from web service
   -----------------------------------------------------------------------------
   str_sql :=
         'SELECT '
      || 'a.rowid, '
      || 'a.geom '
      || 'FROM '
      || 'storetw.fa_station a '
      || 'WHERE '
      || 'a.gen_huctwelvedigitcode IS NULL and '
      || 'a.last_change_date >= :p01 ';

   OPEN my_cursor FOR str_sql USING p_load_date;

   LOOP
      FETCH my_cursor
      BULK COLLECT INTO ary_rids, ary_sdo
      LIMIT num_limiter;

      EXIT WHEN ary_rids.COUNT = 0;

      ary_huc12 := tbl_huc12 ();
      ary_huc12.EXTEND (ary_rids.COUNT);

      FOR i IN 1 .. ary_rids.COUNT
      LOOP
         ary_huc12 (i) := SP_fetch_wbd_huc12 (ary_sdo (i));
      END LOOP;

      FORALL i IN 1 .. ary_rids.COUNT
         UPDATE storetw.fa_station a
            SET a.gen_huctwelvedigitcode = ary_huc12 (i)
          WHERE a.rowid = ary_rids (i);

      COMMIT;

      l_count := l_count + ary_rids.COUNT;
   END LOOP;

   CLOSE my_cursor;


   -----------------------------------------------------------------------------
   -- Step 50
   -- Run statistics on updated table
   -----------------------------------------------------------------------------
   dbms_stats.gather_table_stats(
      ownname          => 'STORETW',
      tabname          => 'FA_STATION',
      cascade          => TRUE,
      estimate_percent => DBMS_STATS.auto_sample_size
   );

   p_record_no := l_cnt_upd_st + l_cnt_upd_cnt + l_cnt_upd_huc + l_cnt_gis_st + l_cnt_gis_cnt + l_cnt_gis_huc + l_count;
   p_comments := 'STUPD: '
        || l_cnt_upd_st  || '. CNTUPD: '
        || l_cnt_upd_cnt || '. HUCUPD: '
        || l_cnt_upd_huc || '. HUC12UPD: '
        || l_cnt_upd_huc12 || '. STGEO: '
        || l_cnt_gis_st  || '. CNTGEO: '
        || l_cnt_gis_cnt || '. HUCGEO: '
        || l_cnt_gis_huc || '. HUC12WS: '
        || l_count || '.';

END SP_update_fa_station_huc;
/

