#!/bin/csh -f
#SBATCH -A emis
#SBATCH --gid=emis-hpc
#SBATCH --mem-per-cpu=8g
#SBATCH -n 1
#SBATCH -p singlepe
#SBATCH -t 24:00:00
#SBATCH -o logs/camx_convert_2028fg_12US1_onroad_ca_adj.%j.log
# Convert from 2D emis to tag-ready ptsr

unset verbose

if ( $?SLURM_JOB_ID ) then
 cd $SLURM_SUBMIT_DIR
else
 set SLURM_JOB_ID = $?
endif

#-----------------------------------------------------------------#

set CTM_2DEMIS = Y   ## Run the 2D / low-level emissions CMAQ to CAMx conversion

# Input emissions root path for sector-specific emissions from SMOKE
set EMIS_ROOT = /work/EMIS/em_v7.2/regionalhaze
# Input emissions case name
setenv CASE 2028fg_16j
# Speciation name used in file names
setenv SPEC cmaq_cb6
# Sector name
set SECTOR = onroad_ca_adj
set SECTOR2 = onroad_ca_adj
# Path to sectorlist for representative date approach 
set SECTORLIST = /work/EMIS/cmaq2camx/scripts/2028fg_secsa_cb6/sectorlist

# Output case name of converted emissions
set OUT_CASE = 2028fg_16j 
# Output platform path for CAMx sector files
set OUT_PLAT = /work/ROMO/regionalhaze

set start_date = 2015356          #first Julian day
set end_date = 2016366          #last Julian day

set YEAR = 2016
set datefile = /work/EMIS/cmaq2camx/dates/mergedates_${YEAR}.txt
set execroot = /work/EMIS/cmaq2camx/bin-21oct2013 # Root to cmaq2camx executables

## Grid definition: xorig (km from origin), yorig (km from origin), cols, rows, and cell size (km)
setenv GRID 12US1
setenv XORG -2556.
setenv YORG -1728.
setenv NX 459
setenv NY 299
setenv DX 12.
# Number of layers to output
set LAYERS = 35

## Define the CMAQ->CAMx species mapping table, this is mechanism specific
set SPECIES_MAPPING = /work/EMIS/cmaq2camx/Species_Mapping_Tables/MAPTBL.CMAQ_CB6+AERO6_to_CAMXv7_CB6+CF_May6_2019.txt

######
setenv OUTPATH ${OUT_PLAT}/smoke_out/${OUT_CASE}/${GRID}/camx_v7/sector/$SECTOR

set QADIR = $cwd/qatemp #location for QA files - domain totals and domain total 2D plots of point emissions
if (! -e $QADIR) then
	mkdir -p $QADIR
endif

set tempdir = $cwd/work  ## Path for the temporary work directory
if (! -e $tempdir) then
	mkdir -p $tempdir
endif

setenv SCRIPTLOG cmaq2camx_${CASE}_${GRID}_${SECTOR}_${SLURM_JOB_ID}.log  ## Name of script log 
if ( -e $cwd/$SCRIPTLOG) then
	rm $cwd/$SCRIPTLOG
endif

#-----------------------------------------------------------------#

set today = $start_date

## Run the CMAQ to CAMx conversion by loop over the specified date range 
while ($today <= $end_date )

	echo $today >> $cwd/$SCRIPTLOG
	#-----------------------------------------------------------------#
	## Set up date values from Julian date
	set STDATE = $today

	set GDAY = (`$execroot/j2g $today`)
	set YYYY = `echo $GDAY | cut -c1-4`
	set YY = `echo $YYYY | cut -c3-4`
	set MM = `echo $GDAY | cut -c5-6`
	set DD = `echo $GDAY | cut -c7-8`
	set G_DAY = $YYYY$MM$DD
	set GS_DAY = $YY$MM$DD
	set YYDDD = `echo $today | cut -c3-8`

	set iday = `grep ^$STDATE $datefile`
	set dateheader = `grep ^Date $datefile`
	#---------------------------------------------------------------------------#

        # Find the representative date based on the date approach from the sectorlist
        set dateappr = `grep ^\"$SECTOR2\" $SECTORLIST | cut -d, -f4 | sed 's/"//g'`

        set n = 1
        while ($n <= 9)
            set column = $dateheader[$n]
            if ($dateappr == $column) then
                    set repday = $iday[$n]
            endif
            @ n++
        end

	## Run the 2D / low-level cmaq to camx conversion
	if ($CTM_2DEMIS == Y) then ## 2D emissions preparation
		#need to combine these files, adjust species names, and add duplicate species

		set PATHA = ${EMIS_ROOT}/${CASE}/premerged/${SECTOR}
		set FILEA = emis_mole_${SECTOR}_${repday}_${GRID}_${SPEC}_${CASE}.ncf

		if (! -e $PATHA/$FILEA) then
			gunzip -c $PATHA/${FILEA}.gz > ${tempdir}/temp_${SLURM_JOB_ID}.ncf
		else
			cp $PATHA/$FILEA ${tempdir}/temp_${SLURM_JOB_ID}.ncf
		endif

		echo ${PATHA}/${FILEA} >> $cwd/$SCRIPTLOG
		set OUTFILE = ptsr.${SECTOR}.$G_DAY.$GRID.bin

                if (! -e $OUTPATH/$OUTFILE) then
                    if (! -e $OUTPATH) then
                            mkdir -p $OUTPATH
                    endif

                    rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}
                    ## Map the CMAQ species names to CAMx as specified in the MAPTBL
                    setenv INFILE  ${tempdir}/temp_${SLURM_JOB_ID}.ncf
                    setenv OUTFILE ${tempdir}/tmp_emis_${SLURM_JOB_ID}
                    setenv MAPTBL  $SPECIES_MAPPING
                    $execroot/spcmap

                    setenv CMAQEMIS ${tempdir}/tmp_emis_${SLURM_JOB_ID}
                    rm -f ${tempdir}/tmp_low_${SLURM_JOB_ID} $OUTPATH/$OUTFILE ${tempdir}/tmp_pnt_${SLURM_JOB_ID}

## Convert from low-level cmaq model ready IOAPI 2D file to camx
$execroot/cmaq2uam-conv2dsat << EOF
File Type          |EMIS
OutFile1 (IC,BC,AR)|${tempdir}/tmp_low_${SLURM_JOB_ID}
OutFile2 (TopC,PT) |${OUTPATH}/$OUTFILE
Output Timezone    |0
EOF

		rm -f ${tempdir}/tmp_low_${SLURM_JOB_ID} 
		rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}

		rm -f ${tempdir}/qaemis.ptsr.${SLURM_JOB_ID}.in
cat << ieof > ${tempdir}/qaemis.ptsr.${SLURM_JOB_ID}.in
$OUTPATH/$OUTFILE
$QADIR/ptsr.${SECTOR}.$G_DAY.$GRID.$CASE.camx.tot.txt
$QADIR/ptsr.${SECTOR}.$G_DAY.$GRID.$CASE.camx.hrs.txt
$QADIR/ptsr.${SECTOR}.$G_DAY.$GRID.$CASE.camx.pave.bin
ieof

                    ## Run low-level / 2D QA on camx file
                    #$execroot/qaelev < ${tempdir}/qaemis.ptsr.${SLURM_JOB_ID}.in
                    rm -f ${tempdir}/qaemis.ptsr.${SLURM_JOB_ID}.in
                    rm -f ${tempdir}/temp_${SLURM_JOB_ID}.ncf
                else
                    echo "WARNING $OUTPATH/$OUTFILE already exists. Will not overwrite."
                endif  # run if the sector file is not present
	endif ## end 2D emissions preparation

	@ today = $today + 1

	if ($today == 2015366) then
		set today = 2016001
	endif

end #loop back to next day

exit
