#!/bin/csh -f
#SBATCH -A emis
#SBATCH --gid=emis-hpc
#SBATCH --mem-per-cpu=16g
##SBATCH -p singlepe
#SBATCH -n 1
#SBATCH -t 7-00:00:00
#SBATCH -o logs/camx_convert_2028fg_secsa_36US3.3D.P2_allinln_allmrgpt_test.%j.log


unset verbose

if ( $?SLURM_JOB_ID ) then
 cd $SLURM_SUBMIT_DIR
else
 set SLURM_JOB_ID = $?
endif



#### User defined run settings ####
#-----------------------------------------------------------------#
  ### GLOBAL SETTINGS ###
    ## Path the CMAQ model ready emissions for SMOKE out platform root
    set EMIS_ROOT = /work/ROMO/regionalhaze/smoke_out

    ## Variable definitions for file names
    # Grid name used in file names
    setenv GRID 36US3
    # Case name used in file names
    setenv CASE 2028fg_16j
		setenv CASE2 2028fg_secsa_16j
    # Speciation name used in file names
    setenv SPEC cmaq_cb6

    ## Initial and last date to run
    set start_date = 2016286 #2016163        #first Julian day
    set end_date   = 2016366          #last Julian day

    # 4-character year of run YYYY
    set YEAR = 2016
    # Year-specific dates and representative dates for CAMx conversion
    set datefile = /work/EMIS/cmaq2camx/dates/mergedates_${YEAR}.txt
    # Root path of cmaq2camx executables
		set execroot = /work/EMIS/cmaq2camx/bin-21oct2013
    #set execroot = /work/EMIS/cmaq2camx/bin-27dec2013
	  # Root path of oceanic (seasalt) executables
		set execroot_ss = /work/EMIS/cmaq2camx/oceanic_v4.1.1/src

    ## Grid definition: xorig (km from origin), yorig (km from origin), cols, rows, and cell size (km)
    setenv XORG -2952.
    setenv YORG -2772.
    setenv NX 172
    setenv NY 148
    setenv DX 36.
    # Number of layers to output
    set LAYERS = 35

    ## Define the CMAQ->CAMx species mapping table, this is mechanism specific
		set SPECIES_MAPPING  = /work/EMIS/cmaq2camx/Species_Mapping_Tables/MAPTBL.CMAQ_CB6+AERO6_to_CAMXv7_CB6+CF_May6_2019.txt

  ### RUN TYPE SPECIFIC SETTINGS ###
    ## See below for settings for the 4 run types: 2D emissions conversion, seasalt, point emissions conversion and point emissions merging

  ### 2D Emissions CMAQ->CAMx conversion ###
    # 2D/low-level gridded Emissions conversion flag
    set CTM_2DEMIS = N

  ### 2D Seasalt settings ###
    # 2D Seasalt and 2D merge run flag. Run "CTM_2DEMIS" first.
    set CTM_SEASALT = N
    # CAMx ready WRF met for seasalt
    set MET = /work/EMIS/met/wrfcamx/WRFv3.8_36US3_2016_35aL/v4.6
    # Grid specific land use for seasalt surf zones
    set LUPATH = /work/EMIS/cmaq2camx/oceanmask


  ### Point Emissions CMAQ->CAMx conversion ###
    # Point/elevated Emissions conversion flag
    set CTM_PT3DEMIS = N
    ## Specify the names of the inline sectors to convert from the source apportionment as appears in the sectorlist  
		set point_sectors_convert = (pt_oilgas_allinln ptnonipm_allinln)
    ## Path to SMOKE-style sectorlist for case. Used to define the representative days for sectors.
    set SECTORLIST = $cwd/sectorlist
    # Base path to CMAQ model ready inline point files
    set INLN_ROOT = /work/ROMO/regionalhaze/smoke_out

  ###  CAMx Point Emissions Merge ### 
    # Point/elevated emissions merge flag. Run "CTM_PT3DEMIS" first.
    set CTM_PTSMRG = Y
    # Set the names of the point sectors to be merged and their respective group numbers
		set mrg_sectors = (ptegu cmv_c3 othpt pt_oilgas_allinln ptnonipm_allinln \
		                   afdust_adj afdust_ak_adj ag beis beis_masked cmv_c1c2 cmv_c1c2_masked nonpt nonpt_masked_AKonly nonroad nonroad_masked_AKonly np_oilgas \
		                   onroad onroad_ca_adj onroad_nonconus onroad_can onroad_can_masked onroad_mex onroad_mex_masked \
											 othafdust_adj othafdust_adj_masked othptdust_adj othptdust_adj_masked othar19 othar19_masked othar21 othar21_masked \
											 ptagfire ptfire_14 ptfire_15 ptfire_othna_18 ptfire_othna_18_masked ptfire_othna_20 ptfire_othna_20_masked \
											 rail rwc rwc_masked_AKonly seasalt seasalt_masked)
		set groupnums =   (-1  -1  -1  11  12 \
		                   16  16  10   1   1   5   5  17  17  4   4  11 \
		                    3   3   3  19  19  21  21 \
											 21  19  21  19  19  19  21  21 \
		  								  9  14  15  18  18  20  20 \
												8  13  13  22  22)
		set mrg_grids =   (12US1  36US3  36US3  36US3  36US3 \
		                   12US1  36US3  12US1  12US2  36US3  12US2  36US3  12US1  36US3  12US1  36US3  12US1 \
											 12US1  12US1  36US3  12US2  36US3  12US2  36US3 \
											 12US2  36US3  12US2  36US3  12US2  36US3  12US2  36US3 \
											 36US3  36US3  36US3  36US3  36US3  36US3  36US3 \
											 12US1  12US1  36US3  12US2  36US3)
    ## Pre-converted fire emissions
    # Path to CAMx pre-converted fire sector files
    set ptfire_path = /work/EMIS/cmaq2camx/fires/ptfire_2028fg_secsa_12US2_camx_v7
		set ptagfire_path = /work/EMIS/cmaq2camx/fires/ptagfire_2016ff_12US2_camx_v7
		set ptfire_othna_path = /work/EMIS/cmaq2camx/fires/ptfire_othna_2028fg_secsa_12US2_camx_v7
		set ptfire_othna_masked_path = /work/EMIS/cmaq2camx/fires/ptfire_othna_masked_2028fg_secsa_36US3_camx_v7
    set ptfire_nonconus_path = /work/EMIS/cmaq2camx/fires/ptfire_nonconus_2016fg_36US3_camx_v7
    # List of fire sectors must match order of list of fire paths
		set firesectors = (ptagfire ptfire_14 ptfire_15 ptfire_othna_18 ptfire_othna_18_masked ptfire_othna_20 ptfire_othna_20_masked)
		set firepaths = ($ptagfire_path $ptfire_path $ptfire_path $ptfire_othna_path $ptfire_othna_masked_path $ptfire_othna_path $ptfire_othna_masked_path)
		set firegrids = (12US2 12US2 12US2 12US2 36US3 12US2 36US3)


#-----------------------------------------------------------------#
#### End user defined run settings ####


#### Begin run script loops
set today = $start_date

set QADIR = $cwd/qatemp #location for QA files - domain totals and domain total 2D plots of point emissions
if (! -e $QADIR) then
	mkdir -p $QADIR
endif

set tempdir = $cwd/work  ## Path for the temporary work directory
if (! -e $tempdir) then
	mkdir -p $tempdir
endif

setenv SCRIPTLOG cmaq2camx_${CASE}_${GRID}_${SLURM_JOB_ID}.log  ## Name of script log 
if ( -e $cwd/$SCRIPTLOG) then
	rm $cwd/$SCRIPTLOG
endif

setenv OUTPUT_TIMEZONE 0 ## Timezone for output, typically should be left at 0
setenv OUTPUT_N_LAYERS $LAYERS

## Run the CMAQ to CAMx conversion by loop over the specified date range 
while ($today <= $end_date )

	echo $today >> $cwd/$SCRIPTLOG
	#-----------------------------------------------------------------#
	## Set up date values from Julian date
	set STDATE = $today

	set GDAY = (`$execroot/j2g $today`)
	set YYYY = `echo $GDAY | cut -c1-4`
	set YY = `echo $YYYY | cut -c3-4`
	set MM = `echo $GDAY | cut -c5-6`
	set DD = `echo $GDAY | cut -c7-8`
	set G_DAY = $YYYY$MM$DD
	set GS_DAY = $YY$MM$DD
	set YYDDD = `echo $today | cut -c3-8`

	set iday = `grep ^$STDATE $datefile`
	set dateheader = `grep ^Date $datefile`
	#---------------------------------------------------------------------------#

	## Run the 2D / low-level cmaq to camx conversion ##
	if ($CTM_2DEMIS == 'Y') then ## 2D emissions preparation
		#need to combine these files, adjust species names, and add duplicate species

		set PATHA = ${EMIS_ROOT}/${CASE}/${GRID}/${SPEC}
		set FILEA = emis_mole_all_${G_DAY}_${GRID}_withbeis_${CASE}.ncf

                # Extract emissions files to a temporary path
		if (! -e $PATHA/$FILEA) then
			gunzip -c $PATHA/${FILEA}.gz > ${tempdir}/temp_${SLURM_JOB_ID}.ncf
		else
			cp $PATHA/$FILEA ${tempdir}/temp_${SLURM_JOB_ID}.ncf
		endif

                # Define the output paths for the CAMx ready 2D files
		echo ${PATHA}/${FILEA} >> $cwd/$SCRIPTLOG
		set OUTPATH = ${EMIS_ROOT}/${CASE}/${GRID}/camx_v7
		set OUTFILE = emis2d.$G_DAY.$GRID.$CASE.camx

		if (! -e $OUTPATH) then
			mkdir -p $OUTPATH
		endif

		rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}
		## Map the CMAQ species names to CAMx as specified in the MAPTBL
		setenv INFILE  ${tempdir}/temp_${SLURM_JOB_ID}.ncf
		setenv OUTFILE ${tempdir}/tmp_emis_${SLURM_JOB_ID}
		setenv MAPTBL  $SPECIES_MAPPING
		$execroot/spcmap

		setenv CMAQEMIS ${tempdir}/tmp_emis_${SLURM_JOB_ID}
		rm -f ${tempdir}/tmp_pnt_${SLURM_JOB_ID} $OUTPATH/$OUTFILE 

## Convert from low-level cmaq model ready IOAPI 2D file to CAMx model ready UAM
$execroot/cmaq2uam << EOF
File Type          |EMIS
OutFile1 (IC,BC,AR)|$OUTPATH/$OUTFILE
OutFile2 (TopC,PT) |${tempdir}/tmp_pnt_${SLURM_JOB_ID}
Output Timezone    |$OUTPUT_TIMEZONE
EOF
		rm -f ${tempdir}/tmp_pnt_${SLURM_JOB_ID} ${tempdir}/tmp_emis_${SLURM_JOB_ID}

		rm -f ${tempdir}/qaemis.low.${SLURM_JOB_ID}.in
cat << ieof > ${tempdir}/qaemis.low.${SLURM_JOB_ID}.in
$OUTPATH/$OUTFILE
$QADIR/emis2d.$G_DAY.$GRID.$CASE.camx.tot.txt
$QADIR/emis2d.$G_DAY.$GRID.$CASE.camx.hrs.txt
$QADIR/emis2d.$G_DAY.$GRID.$CASE.camx.pave.bininln
ieof

		## Run low-level / 2D QA on camx file
		$execroot/qalowemis < ${tempdir}/qaemis.low.${SLURM_JOB_ID}.in
		rm -f ${tempdir}/qaemis.low.${SLURM_JOB_ID}.in
		rm -f ${tempdir}/temp_${SLURM_JOB_ID}.ncf

		unset OUTFILE
		unsetenv OUTFILE
	endif ## end 2D emissions preparation

        ## CAMx 2D seasalt surf emissions: PCL, NA, SS, PSO4
        ##  Automatically merges emissions with existing 2D CAMx emissions
	if ($CTM_SEASALT == 'Y') then ## run 2D seasalt merge
		echo "Running seasalt conversion"
		set EMISPATH = ${EMIS_ROOT}/${CASE}/${GRID}/camx_v7
		
setenv BIN_LWMASK

# Generate the 2D seasalt emissions for CAMx
$execroot_ss/oceanic << IEOF
SS scaling factor  :1.0
Ocean salinity     :35.0
Coastline length   :NONE
Output emiss file  :$OUTPATH/seasalt.$today.$GRID.$LAYERS.wrf_GONG.camx
File identifier    :Oceanic Emissions
Number of species  :11
Species name # 1   :NA
Species name # 2   :PCL
Species name # 3   :PSO4
Species name # 4   :SSCL
Species name # 5   :SSBR
Species name # 15  :DMS
Species name # 16  :WIND
Species name # 17  :CHLA
Species name # 18  :RELH
Species name # 19  :DMSWC
Species name # 20  :SRFT
DMS output as SO2  :.false.
CAMx 3D file       :$MET/met3d.$GS_DAY.$GRID.$LAYERS.wrf.camx
CAMx 2D file       :$MET/met2d.$GS_DAY.$GRID.$LAYERS.wrf.camx
CAMx Land Use file :$MET/landuse.$GS_DAY.$GRID.zhang.camx
Saltwater mask     :$LUPATH/oceanmask.$GRID.txt
DMS concentrations :/work/EMIS/cmaq2camx/oceanic_v4.1.1/dms_conc/DMSclim.$MM.csv
IEOF

		setenv OUTFILE $EMISPATH/emis2d.$G_DAY.$GRID.${CASE}+ss.camx
		rm -rf $OUTFILE

## Merge the seasalt emissions with existing 2D CAMx emissions (emis2d)
$execroot_ss/mrgspc << EOF
Input File         |$EMISPATH/emis2d.$G_DAY.$GRID.$CASE.camx
Input File         |$EMISPATH/seasalt.$today.$GRID.${LAYERS}.wrf_GONG.camx
Input File         |END
Output File        |$OUTFILE
EOF
		rm -f $EMISPATH/seasalt.$today.$GRID.${LAYERS}.wrf_GONG.camx 
		rm -f $EMISPATH/emis2d.$G_DAY.$GRID.$CASE.camx
	endif ## 2D seasalt merge


	## Run the elevated point cmaq inline to camx conversion ##
	if ($CTM_PT3DEMIS == 'Y') then ## elevated point file preparation

		foreach SECTOR (${point_sectors_convert})

			# Find the representative date based on the date approach from the sectorlist
			set dateappr = `grep ^\"$SECTOR\" $SECTORLIST | cut -d, -f4 | sed 's/"//g'`

			set n = 1
			while ($n <= 9)
				set column = $dateheader[$n]

				if ($dateappr == $column) then
					set repday = $iday[$n]
				endif
				@ n++
			end
		
			# Get the sector-specific case	
			set scase = `grep ^\"$SECTOR\" $SECTORLIST | cut -d, -f2 | sed 's/"//g'`

			set INPATH = ${INLN_ROOT}/${CASE}/$GRID/${SPEC}
			set inln = $INPATH/$SECTOR/inln_mole_${SECTOR}_${repday}_${GRID}_${SPEC}_${scase}.ncf
			setenv STKGROUP $INPATH/$SECTOR/stack_groups_${SECTOR}_${GRID}_${scase}.ncf
			#setenv STKGROUP $INPATH/$SECTOR/stack_groups_out_${SECTOR}_${repday}_${GRID}_${scase}.ncf

			#convert to camx point format and merge into a single point source file and then run qaemis on camx input file
			ls $inln $STKGROUP

			echo $inln >> $cwd/$SCRIPTLOG

			rm -f ${tempdir}/temp_${SLURM_JOB_ID}.ncf

			if (! -e $inln) then
				gunzip -c $inln.gz > ${tempdir}/temp_${SLURM_JOB_ID}.ncf
			else
				cp $inln ${tempdir}/temp_${SLURM_JOB_ID}.ncf
			endif

			rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}

			## Map the elevated species
			setenv INFILE  ${tempdir}/temp_${SLURM_JOB_ID}.ncf
			setenv OUTFILE ${tempdir}/tmp_emis_${SLURM_JOB_ID}
			setenv MAPTBL  $SPECIES_MAPPING
			$execroot/spcmap

			rm -f ${tempdir}/temp_${SLURM_JOB_ID}.ncf 

			setenv CMAQEMIS ${tempdir}/tmp_emis_${SLURM_JOB_ID}
			set OUTPATH = ${tempdir}
			set OUTPATH = ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7/sector/$SECTOR
			set OUTFILE = ptsr.${SECTOR}.${G_DAY}.${GRID}.bin
			
			mkdir -p $OUTPATH
			rm -f $OUTPATH/$OUTFILE

## Run cmaq inline to camx converter for the inline sector day
$execroot/cmaq2uam << EOF
File Type          |IPNT
OutFile1 (IC,BC,AR)|
OutFile2 (TopC,PT) |$OUTPATH/$OUTFILE
Output Timezone    |$OUTPUT_TIMEZONE
EOF

				rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}

				## QA the elevated sector
				rm -f ${tempdir}/qaemis.${SECTOR}.${SLURM_JOB_ID}.in
cat << ieof > ${tempdir}/qaemis.$SECTOR.${SLURM_JOB_ID}.in
$OUTPATH/$OUTFILE
$QADIR/$SECTOR.$G_DAY.$GRID.$CASE.camx.tot.txt
$QADIR/$SECTOR.$G_DAY.$GRID.$CASE.camx.hrs.txt
$QADIR/$SECTOR.$G_DAY.$GRID.$CASE.camx.pave.bin
ieof

			# $execroot/qaelev < ${tempdir}/qaemis.$SECTOR.${SLURM_JOB_ID}.in
			# rm -f ${tempdir}/qaemis.$SECTOR.${SLURM_JOB_ID}.in

		end ## end loop for conversion of this sector

	endif  ## end point source file processing

	## Run the point sector UAM merge ##
	if ($CTM_PTSMRG == 'Y') then
		set OUTPATH = ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7
		set OUTFILE = mrgpt.allinln_allmrgpt_test.$G_DAY.$GRID.$CASE2.camx
		set INPATH = $OUTPATH/sector

		rm -f $OUTPATH/$OUTFILE ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt

		# Link in the pre-converted CAMx fire sector emissions
		set numsect = `echo $firesectors | wc | awk '{print $2}'`
		set n = 1
		while ($n <= $numsect)
			set firesector = $firesectors[$n]
			set fire_path = $firepaths[$n]
			set firegrid = $firegrids[$n]
			if (! -d ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7/sector/${firesector}) then
				mkdir -p ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7/sector/${firesector}
			endif
			rm -f ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7/sector/${firesector}/ptsr.${firesector}.${G_DAY}.${GRID}.bin
			ln -s $fire_path/ptsr.${firesector}.${G_DAY}.${firegrid}.bin ${EMIS_ROOT}/${CASE2}/${GRID}/camx_v7/sector/${firesector}/ptsr.${firesector}.${G_DAY}.${GRID}.bin
			@ n++
		end

		echo "jdate    |$YYDDD" >> ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt
		echo "emissions|$OUTPATH/$OUTFILE" >> ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt

		# Total number of sectors including all subsectors
		set numsect = `echo ${mrg_sectors} | wc | awk '{print $2}'`

		echo "number pt| $numsect" >> ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt

		set n = 1
		while ($n <= $numsect)
			set SECTOR = $mrg_sectors[$n]
			set MRGGRID = $mrg_grids[$n]
			echo "sat ptsr |${INPATH}/${SECTOR}/ptsr.${SECTOR}.${G_DAY}.${MRGGRID}.bin" >> ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt
			echo "group num| $groupnums[$n]" >> ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt

			@ n++
		end

		## Run point source merge for converted inline data for this day
		$execroot/ptsmrg < ${tempdir}/ptsmrg.${SLURM_JOB_ID}.txt

		rm -rf ${tempdir}/qaemis.elev.${SLURM_JOB_ID}.in
cat << ieof > ${tempdir}/qaemis.elev.${SLURM_JOB_ID}.in
$OUTPATH/$OUTFILE
$QADIR/mrgpt.$G_DAY.$GRID.$CASE2.camx.tot.txt
$QADIR/mrgpt.$G_DAY.$GRID.$CASE2.camx.hrs.txt
$QADIR/mrgpt.$G_DAY.$GRID.$CASE2.camx.pave.bin
ieof

		## Run the elevated emissions QA program
		 #$execroot/qaelev < ${tempdir}/qaemis.elev.${SLURM_JOB_ID}.in
		 #rm -f ${tempdir}/qaemis.elev.${SLURM_JOB_ID}.in
		 #rm -f ${tempdir}/ptsr.*.${SLURM_JOB_ID}.bin 

	endif  ## end point source merge 

	@ today = $today + 1

	if ($today == 2015366) then
		set today = 2016001
	endif

end #loop back to next day

exit
