#!/bin/csh -f
#SBATCH -A emis
#SBATCH --gid=emis-hpc
#SBATCH --mem-per-cpu=8g
#SBATCH -n 1
#SBATCH -p singlepe
#SBATCH -t 24:00:00
#SBATCH -o logs/seasalt2ptsr_36US3_masked.%j.log

# Convert 2D seasalt emissions from the oceanic preprocessor to tag-ready CAMx ptsr.
# 		1.	Run oceanic to generate CAMx format sea salt emissions.
# 		2.	Run camx2ioapi_v6 program to convert the CAMx format emissions 
# 				to CMAQ formation emissions. This program also will convert the 
#         emissions units to CMAQ which is important and needed.
# 		3.	Take the netCDF/IOAPI format sea salt emissions generated by step 2 and 
#         include that in the cmaq2camx emissions processing tools that convert 
#         the gridded 2D files to point source format files and then into the rest 
#         of the normal process. This program expects the emissions units to be CMAQ 
#         so it is good that the step 2 program includes that functionality. 

unset verbose

if ( $?SLURM_JOB_ID ) then
 cd $SLURM_SUBMIT_DIR
else
 set SLURM_JOB_ID = $?
endif

#-----------------------------------------------------------------#
#### User defined run settings ####
#-----------------------------------------------------------------#

# Case name used in file names
set CASE = 2028fg_secsa_16j

set start_date = 2015356	# first Julian day
set end_date = 2016366		# last Julian day

## Grid definition: xorig (km from origin), yorig (km from origin), cols, rows, and cell size (km)
setenv GRID 36US3
setenv XORG -2952.
setenv YORG -2772.
setenv NX 172
setenv NY 148
setenv DX 36.
# Number of layers to output
set LAYERS = 35

# Path to the CMAQ model ready emissions for SMOKE out platform root
set EMIS_ROOT = /work/ROMO/regionalhaze/smoke_out

# CAMx ready WRF met for seasalt
set MET = /work/EMIS/met/wrfcamx/WRFv3.8_36US3_2016_35aL/v4.6
# Grid specific land use for seasalt surf zones
set LUPATH = /work/EMIS/cmaq2camx/oceanmask/36US3_masked

## Define the CMAQ->CAMx species mapping table, this is mechanism specific
set SPECIES_MAPPING = /work/EMIS/cmaq2camx/Species_Mapping_Tables/MAPTBL.CMAQ_CB6+AERO6_to_CAMXv7_CB6+CF_May6_2019.txt

# Root path of executables
set execroot_oceanic = /work/EMIS/cmaq2camx/oceanic_v4.1.1/src
set execroot_camx2ioapi = /work/ROMO/users/kpc/util/camx2ioapi_v6/src
set execroot_cmaq2camx = /work/EMIS/cmaq2camx/bin

#-----------------------------------------------------------------#
#### End user defined run settings ####
#-----------------------------------------------------------------#

set OUTPATH = $EMIS_ROOT/$CASE/$GRID/camx_v7/sector/seasalt_masked
if (! -e $OUTPATH) mkdir -p $OUTPATH

set today = $start_date

## Loop over the specified date range
while ($today <= $end_date )

#-----------------------------------------------------------------#

set STDATE = $today

set GDAY = (`$execroot_cmaq2camx/j2g $today`)

set YYYY = `echo $GDAY | cut -c1-4`
set YY = `echo $GDAY | cut -c3-4`
set MM = `echo $GDAY | cut -c5-6`
set DD = `echo $GDAY | cut -c7-8`

set G_DAY = $YYYY$MM$DD
set GS_DAY = $YY$MM$DD

#-----------------------------------------------------------------#
# Generate the 2D seasalt emissions for CAMx using oceanic
#-----------------------------------------------------------------#

setenv BIN_LWMASK

$execroot_oceanic/oceanic << IEOF
SS scaling factor  :1.0
Ocean salinity     :35.0
Coastline length   :NONE
Output emiss file  :$OUTPATH/seasalt.$today.$GRID.${LAYERS}.wrf_GONG.camx
File identifier    :Oceanic Emissions
Number of species  :11
Species name # 1   :NA
Species name # 2   :PCL
Species name # 3   :PSO4
Species name # 4   :SSCL
Species name # 5   :SSBR
Species name # 15  :DMS
Species name # 16  :WIND
Species name # 17  :CHLA
Species name # 18  :RELH
Species name # 19  :DMSWC
Species name # 20  :SRFT
DMS output as SO2  :.false.
CAMx 3D file       :$MET/met3d.$GS_DAY.$GRID.35.wrf.camx
CAMx 2D file       :$MET/met2d.$GS_DAY.$GRID.35.wrf.camx
CAMx Land Use file :$MET/landuse.$GS_DAY.$GRID.zhang.camx
Saltwater mask     :$LUPATH/oceanmask.36US3_masked.txt
DMS concentrations :/work/EMIS/cmaq2camx/oceanic_v4.1.1/dms_conc/DMSclim.$MM.csv
IEOF

#-----------------------------------------------------------------#
#
#   Convert CAMx AVRG/DEPN outputs or gridded emissions inputs into I/O API format.
#   This program keeps CAMx species names and units (except
#   for emissions for which the unit is converted to moles/s or g/s).
#
#   1. set the environmental variable IOAPI_OUT to the I/O API format output filename
#   2. Input CAMx filename - CAMx avrg, depn, or gridded emissions filename
#   3. Data Type           - AVRG for CAMx average outputs
#                            DDEP for dry deposited mass field of CAMx deposition outputs
#                            WDEP for wet deposited mass field of CAMx deposition outputs
#                            EMIS for CAMx low-level emission inputs
#   4. Sigma Levels        - Sigma coordinates defining upper boundary of each layer
#
#-----------------------------------------------------------------#

setenv IOAPI_OUT $OUTPATH/seasalt.${today}.${GRID}.${LAYERS}.wrf_GONG.ncf
rm -f $IOAPI_OUT

$execroot_camx2ioapi/camx2ioapi << EOF
Input CAMx filename|$OUTPATH/seasalt.${today}.${GRID}.${LAYERS}.wrf_GONG.camx
Data Type          |EMIS
Sigma Levels       |0.0
EOF

#-----------------------------------------------------------------#
# Run the 2D / low-level cmaq to camx conversion
#-----------------------------------------------------------------#

set tempdir = $cwd/work  ## Path for the temporary work directory
if (! -e $tempdir) then
        mkdir -p $tempdir
endif

set PATHA = $OUTPATH
set FILEA = seasalt.${today}.${GRID}.${LAYERS}.wrf_GONG.ncf

cp $PATHA/$FILEA ${tempdir}/temp_${SLURM_JOB_ID}.ncf

set OUTFILE = ptsr.seasalt.$G_DAY.$GRID.bin

if (! -e $OUTPATH/$OUTFILE) then	# run if the sector file is not present
  
	if (! -e $OUTPATH) then
		mkdir -p $OUTPATH
	endif
	
#	rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}
#  ## Map the CMAQ species names to CAMx as specified in the MAPTBL
#  setenv INFILE  ${tempdir}/temp_${SLURM_JOB_ID}.ncf
#  setenv OUTFILE ${tempdir}/tmp_emis_${SLURM_JOB_ID}
#  setenv MAPTBL  $SPECIES_MAPPING
#  $execroot_cmaq2camx/spcmap

  #setenv CMAQEMIS ${tempdir}/tmp_emis_${SLURM_JOB_ID}
	setenv CMAQEMIS ${tempdir}/temp_${SLURM_JOB_ID}.ncf
  rm -f ${tempdir}/tmp_low_${SLURM_JOB_ID} $OUTPATH/$OUTFILE ${tempdir}/tmp_pnt_${SLURM_JOB_ID}
	
## Convert from low-level cmaq model ready IOAPI 2D file to camx
$execroot_cmaq2camx/cmaq2uam_nointerp << EOF
OutFile1 (IC,BC,AR)|${tempdir}/tmp_low_${SLURM_JOB_ID}
OutFile2 (TopC,PT) |${OUTPATH}/$OUTFILE
Output Timezone    |0
EOF

  rm -f ${tempdir}/tmp_low_${SLURM_JOB_ID} 
  rm -f ${tempdir}/tmp_emis_${SLURM_JOB_ID}
	rm -f ${tempdir}/temp_${SLURM_JOB_ID}.ncf

else
	
	echo "WARNING: $OUTPATH/$OUTFILE already exists. Will not overwrite."
	
endif # end 2D emissions preparation

#-----------------------------------------------------------------#

@ today = $today + 1

if ($today == 2015366) then
        set today = 2016001
endif

end  # end loop over days

