      SUBROUTINE LTCALC
C***********************************************************************
C                 LTCALCulation Module of ISC Model - Long Term
C
C        PURPOSE: Controls Flow and Processing of LTCALCulation Modules
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To include check for FREQ = 0.0 for POINT and
C                    VOLUME sources only - 7/7/93
C
C*       MODIFIED BY J. Hardikar, PES, to handle OPENPIT source - 7/20/94
C                 
C        INPUTS:  Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Frequency Element
C
C        OUTPUTS: Array of CONC or DEPOS Values for Each Source/Receptor
C
C        CALLED FROM:   LTMAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'LTCALC'
      PATH = 'CN'

C     Begin Source LOOP
      DO 20 ISRC = 1, NUMSRC
         IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
C           Calculate Point Source Values                   ---   CALL PCALCL
            IF (FREQ(IWS,ISEC,IKST) .NE. 0.0) THEN
               CALL PCALCL
            END IF
         ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
C           Calculate Volume Source Values                  ---   CALL VCALCL
            IF (FREQ(IWS,ISEC,IKST) .NE. 0.0) THEN
               CALL VCALCL
            END IF
         ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
C           Calculate Area Source Values                    ---   CALL ACALCL
            CALL ACALCL
         ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
C*          Calculate OpenPit Source Values                 ---   CALL OCALCL
            CALL OCALCL
         END IF
 20   CONTINUE
C     End Source LOOP

      RETURN
      END

      SUBROUTINE PCALCL
C***********************************************************************
C                 PCALCL Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for POINT sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        MODIFIED:   To check for upwind receptors before call to
C                    LTXYP to prevent runtime math error - 9/29/92
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   LTCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'PCALCL'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFTLT
      CALL EMFTLT(QS)

C     Calculate Deposition Velocities for this Source       ---   CALL VDP
      IF (LDEP) CALL VDP

      IF (QTK .NE. 0.0) THEN
C        Calculate Buoyancy and Momentum Fluxes
         IF (TS .LT. TA)  TS = TA
         FB = (0.25/TS)*(VS*DS*DS)*G*(TS-TA)
         FM = (0.25/TS)*(VS*DS*DS)*VS*TA
C        Adjust Wind Speed to Stack Height                  ---   CALL WSADJ
         CALL WSADJ
C        Calculate Distance to Final Rise                   ---   CALL DISTF
         CALL DISTF
C        Set Wake and Building Type Switches                ---   CALL WAKFLG
         CALL WAKFLG
C        Initialize FSTREC Logical Switch
         FSTREC = .TRUE.
         IF (LDEP) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF

C        Begin Receptor LOOP
         DO 20 IREC = 1, NUMREC
C           Calculate Down and Crosswind Distances          ---   CALL XYDIST
            CALL XYDIST
            IF (X .LE. 0.0) THEN
C              Receptor Upwind of Source
               GO TO 20
            ELSE
C              Calculate Lateral Virtual Distance           ---   CALL LTXYP
               CALL LTXYP(X,XY)
            END IF
            IF (ABS(Y) .GE. (X+XY)*TAN(DELTHP)) THEN
C              Receptor Outside Sector
               GO TO 20
            ELSE IF (DISTR .LT. 1.0 .OR. DISTR .LT. 3.*ZLB ) THEN
C              Receptor Too Close to Source for Calculation
               GO TO 20
            ELSE IF ((X-XRAD) .LT. 0.0) THEN
C              Receptor Upwind of Downwind Edge of Effective Radius
               GO TO 20
            ELSE
C              Determine Effective Plume Height             ---   CALL PHEFF
               CALL PHEFF(X,DHP,HEFLAT)
C              Apply Terrain Adjustment                     ---   CALL STERAD
               CALL STERAD(HEFLAT,ZELEV,HE)
C              Determine Dispersion Parameters              ---   CALL PDISLT
               CALL PDISLT(X,SZ,XZ,SBID)
               IF (CONC) THEN
C                 Calculate Concentration                   ---   CALL PCHILT
                  CALL PCHILT
               ELSE IF (DEPOS) THEN
C                 Calculate Deposition                      ---   CALL PDEPLT
                  CALL PDEPLT
               END IF
            END IF
 20      CONTINUE
C        End Receptor LOOP
      END IF

      RETURN
      END

      SUBROUTINE VCALCL
C***********************************************************************
C                 VCALCL Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for VOLUME sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        MODIFIED BY R. Brode, PES, to initialize SBID - 5/26/94
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   LTCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'VCALCL'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFTLT
      CALL EMFTLT(QS)

C     Calculate Deposition Velocity for this Source         ---   CALL VDP
      IF (LDEP) CALL VDP

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI))) THEN
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
C        Calculate Effective Radius
         XRAD = 2.15*SYINIT
C        Calculate Lateral Virtual Distance
         XY = XRAD/TAN(0.5*DELTHP)
         IF (LDEP) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF
C        Initialize SBID to 0.0 for call to DEPCOR
         SBID = 0.0

C        Begin Receptor LOOP
         DO 20 IREC = 1, NUMREC
C           Calculate Down and Crosswind Distances          ---   CALL XYDIST
            CALL XYDIST
            IF (X.LT.0.0 .OR. (ABS(Y).GE.(X+XY)*TAN(DELTHP))) THEN
C              Receptor Upwind OR Outside Sector
               GO TO 20
            ELSE IF (DISTR .LT. (XRAD+1.0)) THEN
C              Receptor Too Close to Source for Calculation
               GO TO 20
            ELSE IF ((X-XRAD) .LT. 0.0) THEN
C              Receptor Upwind of Downwind Edge
               GO TO 20
            ELSE
C              Determine Effective Plume Height             ---   CALL VHEFF
               CALL VHEFF(ZELEV,HEFLAT,HE)
C              Determine Dispersion Parameters              ---   CALL VDISLT
               CALL VDISLT(X,SZ,XZ)
               IF (CONC) THEN
C                 Calculate Conc. for Virtual Point Source  ---   CALL PCHILT
                  CALL PCHILT
               ELSE IF (DEPOS) THEN
C                 Calculate Depos. for Virtual Point Source ---   CALL PDEPLT
                  CALL PDEPLT
               END IF
            END IF
 20      CONTINUE
C        End Receptor LOOP
      END IF

      RETURN
      END

      SUBROUTINE ACALCL
C***********************************************************************
C                 ACALCL Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for AREA sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        MODIFIED:   To use numerical integration algorithm for
C                    AREA sources - 7/7/93
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        MODIFIED BY R. Brode, PES, to initialize XZ, XY, and SBID - 7/15/94
C
C*       MODIFIED BY J. Hardikar, PES, to make consistent with the new
C*                   OPENPIT Source Methodology - 7/20/94
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   LTCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
          
C     Variable Initializations
      MODNAM = 'ACALCL'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFTLT
      CALL EMFTLT(QS)

C     Calculate Deposition Velocity for this Source         ---   CALL VDP
      IF (LDEP) CALL VDP

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI))) THEN
      
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
      
         IF (LDEP) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF
C*       Initialize XY and XZ to 0.0 (XZ is used in
C*       call to DEPCOR from PLUMEF)
         XY = 0.0
         XZ = 0.0

C        Initialize SBID to 0.0 (for call to DEPCOR from PLUMEF)
         SBID = 0.0

C        Begin Receptor LOOP
         DO 660 IREC = 1, NUMREC
C           Initialize The Variables
            XOLD = 0.0

C           Calculate Concentration for Central Wind Direction
C           Set Wind Direction Variable for Central WD      ---   CALL SETWD
            CALL SETWD(WDRAD)
            IF (FREQ(IWS,ISEC,IKST) .EQ. 0.0) THEN
C              Because of HRVAL = 0.0
               XOLD =0.0
            ELSE
               CALL ACALCT
               XOLD = FREQ(IWS,ISEC,IKST)*HRVAL
            END IF

C           Calculate Concentration for Left Sector Edge Wind Direction
            THETA1 = WDRAD - DELTHP/2.
C           Set Wind Direction Variable for Left Edge WD    ---   CALL SETWD
            CALL SETWD(THETA1)
C           Calculate the Smoothing Frequency Function, SM
            SM = SMFREQ(ISEC,-DELTHP/2.)
            IF (SM .EQ. 0.0 ) THEN
               HRVAL =0.0
            ELSE
               CALL ACALCT
            END IF
C           Save Half The Concentration Value for Trapezoidal Integration
            XOLD = XOLD + 0.5*SM*HRVAL

C           Calculate Concentration for Right Sector Edge Wind Direction
            THETA2 = WDRAD + DELTHP/2.
C           Set Wind Direction Variable for Right Edge WD   ---   CALL SETWD
            CALL SETWD(THETA2)
C           Calculate the Smoothing Frequency Function, SM
            SM = SMFREQ(ISEC,DELTHP/2.)
            IF (SM .EQ. 0.0 ) THEN
               HRVAL =0.0
            ELSE
               CALL ACALCT
            END IF
C           Save Half The Concentration Value for Trapezoidal Integration
            XOLD = XOLD + 0.5*SM*HRVAL

C           Check for No Concentration for This Sector
            IF (XOLD .EQ. 0.0) GO TO 320

C           Set Initial Variable Xold = Total Conc.; Nold = Total #;
C           And CROLD = Total Average Conc.
            NOLD = 2
            CROLD = XOLD/NOLD
            NITER = 2

C           Perform Trapeziodal Integration Across Sector.  The Loop Index, I
C           is the "Halving Interval", up to a Maximum of IMAX1 (set = 10 in
C           a PARAMETER statement in MAIN1LT.INC).
            DO 300 I = 2, IMAX1
               NITER = 2*NITER
               NNEW = 0
               XNEW = 0.0

C              Loop Across Sector for This "Halving Interval"
               DO 310 J = 1, NITER-1, 2
                  NNEW = NNEW + 1
C                 Calculate Current Wind Direction, THETA
                  THETA = ((NITER-J)*THETA1 + J*THETA2)/FLOAT(NITER)
C                 Set Wind Direction And Calculate The Conc.
                  CALL SETWD(THETA)
C                 Calculate the Smoothing Frequency Function, SM
                  SM = SMFREQ(ISEC,THETA-WDRAD)
                  IF (SM .EQ. 0.0 ) THEN
                     HRVAL =0.0
                  ELSE
                     CALL ACALCT
                  END IF
                  XNEW = XNEW + SM*HRVAL
 310           CONTINUE

C              Update The Average Concentration Variable And
C              Check The Convergence Criteria:
C                 Convergence to Within Two Percent on Successive Iterations
C              OR Value of New Concentration Less Than EPS2 (set = 1.0E-10
C                 in a PARAMETER Statement in MAIN1LT.INC)
               CRNEW = (XOLD+XNEW)/(NOLD+NNEW)
               IF (ABS((CRNEW-CROLD)/CRNEW) .LE. 0.02 .OR.
     &             ABS(CRNEW) .LT. EPS2) THEN
                  HRVAL = CRNEW
                  GO TO 320
               END IF

C              Update The Variable Xold = Total Conc.; Nold = Total #;
C              And CROLD = Total Average Conc.
               XOLD  = XOLD + XNEW
               NOLD  = NOLD + NNEW
               CROLD = CRNEW
 300        CONTINUE

C           Set HRVAL Variable for This Receptor
            HRVAL = CRNEW

 320        CONTINUE

C           Sum HRVAL to SRCVAL Array
            SRCVAL(IREC,ISRC) = SRCVAL(IREC,ISRC) + HRVAL

 660     CONTINUE
C        End Receptor LOOP

999      CONTINUE
      END IF
C
      RETURN
      END

      SUBROUTINE OCALCL
C***********************************************************************
C                 OCALCL Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for OPENPIT sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        ADAPTED FROM: Subroutine ACALCL
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   LTCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL XVM(5), YVM(5)
          
C     Variable Initializations
      MODNAM = 'OCALCL'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

      CALL EMFTLT(QS)

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI))) THEN
      
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
      
C        Begin Receptor LOOP
         DO 660 IREC = 1, NUMREC

C           Check for receptor located inside boundary of open pit source
            XVM(1) = XVERT(1) * 1000.
            XVM(2) = XVERT(2) * 1000.
            XVM(3) = XVERT(3) * 1000.
            XVM(4) = XVERT(4) * 1000.
            XVM(5) = XVERT(5) * 1000.
            YVM(1) = YVERT(1) * 1000.
            YVM(2) = YVERT(2) * 1000.
            YVM(3) = YVERT(3) * 1000.
            YVM(4) = YVERT(4) * 1000.
            YVM(5) = YVERT(5) * 1000.
            XR = AXR(IREC)
            YR = AYR(IREC)
            CALL PNPOLY(XR,YR,XVM,YVM,5,INOUT)
            IF (INOUT .GT. 0) THEN
C              Receptor is within boundary - skip to next receptor
               GO TO 660
            END IF

C           Initialize The Variables
            XOLD = 0.0

C           Calculate Concentration for Central Wind Direction
C           Set Wind Direction Variable for Central WD      ---   CALL SETWD
            CALL SETWD(WDRAD)
            IF (FREQ(IWS,ISEC,IKST) .EQ. 0.0) THEN
C              Because of HRVAL = 0.0
               XOLD =0.0
            ELSE
               CALL OCALCT
               XOLD = FREQ(IWS,ISEC,IKST)*HRVAL
            END IF

C           Calculate Concentration for Left Sector Edge Wind Direction
            THETA1 = WDRAD - DELTHP/2.
C           Set Wind Direction Variable for Left Edge WD    ---   CALL SETWD
            CALL SETWD(THETA1)
C           Calculate the Smoothing Frequency Function, SM
            SM = SMFREQ(ISEC,-DELTHP/2.)
            IF (SM .EQ. 0.0 ) THEN
               HRVAL =0.0
            ELSE
               CALL OCALCT
            END IF
C           Save Half The Concentration Value for Trapezoidal Integration
            XOLD = XOLD + 0.5*SM*HRVAL

C           Calculate Concentration for Right Sector Edge Wind Direction
            THETA2 = WDRAD + DELTHP/2.
C           Set Wind Direction Variable for Right Edge WD   ---   CALL SETWD
            CALL SETWD(THETA2)
C           Calculate the Smoothing Frequency Function, SM
            SM = SMFREQ(ISEC,DELTHP/2.)
            IF (SM .EQ. 0.0 ) THEN
               HRVAL =0.0
            ELSE
               CALL OCALCT
            END IF
C           Save Half The Concentration Value for Trapezoidal Integration
            XOLD = XOLD + 0.5*SM*HRVAL

C           Check for No Concentration for This Sector
            IF (XOLD .EQ. 0.0) GO TO 320

C           Set Initial Variable Xold = Total Conc.; Nold = Total #;
C           And CROLD = Total Average Conc.
            NOLD = 2
            CROLD = XOLD/NOLD
            NITER = 2

C           Perform Trapeziodal Integration Across Sector.  The Loop Index, I
C           is the "Halving Interval", up to a Maximum of IMAX1 (set = 10 in
C           a PARAMETER statement in MAIN1LT.INC).
            DO 300 I = 2, IMAX1
               NITER = 2*NITER
               NNEW = 0
               XNEW = 0.0

C              Loop Across Sector for This "Halving Interval"
               DO 310 J = 1, NITER-1, 2
                  NNEW = NNEW + 1
C                 Calculate Current Wind Direction, THETA
                  THETA = ((NITER-J)*THETA1 + J*THETA2)/FLOAT(NITER)
C                 Set Wind Direction And Calculate The Conc.
                  CALL SETWD(THETA)
C                 Calculate the Smoothing Frequency Function, SM
                  SM = SMFREQ(ISEC,THETA-WDRAD)
                  IF (SM .EQ. 0.0 ) THEN
                     HRVAL =0.0
                  ELSE
                     CALL OCALCT
                  END IF
                  XNEW = XNEW + SM*HRVAL
 310           CONTINUE

C              Update The Average Concentration Variable And
C              Check The Convergence Criteria:
C                 Convergence to Within Two Percent on Successive Iterations
C              OR Value of New Concentration Less Than EPS2 (set = 1.0E-10
C                 in a PARAMETER Statement in MAIN1LT.INC)
               CRNEW = (XOLD+XNEW)/(NOLD+NNEW)
               IF (ABS((CRNEW-CROLD)/CRNEW) .LE. 0.02 .OR.
     &             ABS(CRNEW) .LT. EPS2) THEN
                  HRVAL = CRNEW
                  GO TO 320
               END IF

C              Update The Variable Xold = Total Conc.; Nold = Total #;
C              And CROLD = Total Average Conc.
               XOLD  = XOLD + XNEW
               NOLD  = NOLD + NNEW
               CROLD = CRNEW
 300        CONTINUE

C           Set HRVAL Variable for This Receptor
            HRVAL = CRNEW

 320        CONTINUE

C           Sum HRVAL to SRCVAL Array
            SRCVAL(IREC,ISRC) = SRCVAL(IREC,ISRC) + HRVAL

 660     CONTINUE
C        End Receptor LOOP
      END IF
C
      RETURN
      END

      SUBROUTINE SETSRC
C***********************************************************************
C                 SETSRC Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Sets the Source Parameters for a Particular Source
C                 for POINT Sources
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED:   To incorporate inputs for numerical integration
C                    algorithm for AREA source - 7/7/93
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Source Parameters Arrays
C                 Source Index
C
C        OUTPUTS: Source Parameters for a Particular Source
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SETSRC'

C     Assign The Values From Array Elements To Variables
      IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         DS = ADS(ISRC)
         VS = AVS(ISRC)
         TS = ATS(ISRC)

C        Check for Negative Stack Temperature, Used to Indicate Constant TS-TA
         IF (TS .LT. 0.0) THEN
            TS = TA + ABS(TS)
         END IF

         IF (IFVSEC .LE. NSEC) THEN
            DSBH = ADSBH(IFVSEC,ISRC)
            DSBW = ADSBW(IFVSEC,ISRC)
            IF (IDSWAK(IFVSEC,ISRC) .EQ. 0) THEN
               WAKLOW = .FALSE.
            ELSE IF (IDSWAK(IFVSEC,ISRC) .EQ. 1) THEN
               WAKLOW = .TRUE.
            END IF
         END IF

      ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         SYINIT = ASYINI(ISRC)
         SZINIT = ASZINI(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         XINIT = AXINIT(ISRC)
         YINIT = AYINIT(ISRC)
         ANGLE = AANGLE(ISRC)

         SZINIT = ASZINI(ISRC)         

         DO 15 IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
15       CONTINUE  
            

      ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
C        Set Emission Height of Effective Area, HS = 0.0
         HS = 0.0
C        Set Height of Emissions Above Base of Pit, EMIHGT
         EMIHGT = AHS(ISRC)

         XINIT = AXINIT(ISRC)
         YINIT = AYINIT(ISRC)
         ANGLE = AANGLE(ISRC)
         PALPHA = AALPHA(ISRC)
         PDEFF  = APDEFF(ISRC)
         SZINIT = ASZINI(ISRC)
         PITLEN = AMAX1(XINIT,YINIT)
         PITWID = AMIN1(XINIT,YINIT)


         DO 20 IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
20       CONTINUE  
         

      END IF

      NPD = INPD(ISRC)
      IF (NPD .GT. 0) THEN
         DO 25 J = 1, NPD
            PDIAM(J) = APDIAM(J,ISRC)
            PHI(J) = APHI(J,ISRC)
            PDENS(J) = APDENS(J,ISRC)
            VGRAV(J) = AVGRAV(J,ISRC)
            TSTOP(J) = ATSTOP(J,ISRC)
            SC(J) = ASC(J,ISRC)
 25      CONTINUE
      END IF

      RETURN
      END

      SUBROUTINE PHEFF(XARG,DHPOUT,HEOUT)
C***********************************************************************
C                 PHEFF Module of the ISC Model - Long Term
C
C        PURPOSE: Calculates Effective Plume Height for POINT Sources (m)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To remove terrain adjustment to separate subroutine,
C                    and to use calling arguments
C                    R.W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Terrain Elevation of Receptor
C
C        OUTPUTS: Plume Height (HEOUT) without Terrain Adjustment
C
C        CALLED FROM:   PCALC2
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'PHEFF'

C     Calculate Plume Height Without Terrain Adjustment
      IF ((.NOT. WAKE) .AND. (.NOT. GRDRIS)) THEN
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
            HSP = HSPRIM(US,VS,HS,DS)
C           Calculate Final Rise, DHF                       ---   CALL DELH
            CALL DELH(DHF)
         END IF
         IF (NOSTD) THEN
            HEOUT = HS + DHF
         ELSE
            HEOUT = HSP + DHF
         END IF
         IF (.NOT. NOBID) THEN
C           Calculate Gradual Plume Rise for Use in BID Calculation
            IF (XARG .LT. XF) THEN
C              Calculate Gradual Rise, DHPOUT               ---   CALL DHPHS
               CALL DHPHS(XARG,DHF,DHPOUT)
            ELSE
               DHPOUT = DHF
            END IF
         ELSE
            DHPOUT = DHF
         END IF
      ELSE IF (WAKE .AND. WAKESS) THEN
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
C           Calculate Final Rise (at X=XF), DHF             ---   CALL DHPSS
            CALL DHPSS(XF,DHPOUT)
            DHF    = DHPOUT
         END IF
         IF (XARG .LT. XF) THEN
C           Calculate Gradual Rise, DHP                     ---   CALL DHPSS
            CALL DHPSS(XARG,DHPOUT)
         ELSE
            DHPOUT = DHF
         END IF
         HEOUT = HS + DHPOUT
      ELSE
CRWB       if ((WAKE .AND. (.NOT. WAKESS)) .OR.
CRWB          ((.NOT. WAKE) .AND. GRDRIS)) then
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
            HSP = HSPRIM(US,VS,HS,DS)
C           Calculate Final Rise, DHF                       ---   CALL DELH
            CALL DELH(DHF)
         END IF
         IF (XARG .LT. XF) THEN
C           Calculate Gradual Rise, DHP                     ---   CALL DHPHS
            CALL DHPHS(XARG,DHF,DHPOUT)
         ELSE
            DHPOUT = DHF
         END IF
         IF (NOSTD) THEN
            HEOUT = HS + DHPOUT
         ELSE
            HEOUT = HSP + DHPOUT
         END IF
      END IF

      RETURN
      END

      SUBROUTINE STERAD(HEARG,ZARG,HEOUT)
C***********************************************************************
C                 STERAD Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Adjusts Effective Plume Height for Simple Terrain Effects
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    September 30, 1994
C
C        INPUTS:  HEARG = Flat terrain plume height
C                 ZARG  = Elevation of terrain
C
C        OUTPUTS: HEOUT = Effective plume height with terrain adjustment
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'STERAD'

C     Adjust Plume Height for Elevated Terrain, Save Flat Terrain Value (HEFLAT)
C     For Later Comparison With Mixing Height
      IF (FLAT) THEN
         HEOUT  = HEARG
      ELSE IF (ELEV) THEN
C        Calculate Terrain Hgt Above Plant Grade (Chopped-off at Release Height)
         HTERCHOP = AMIN1( HS, (ZARG - ZS))
         HEOUT = HEARG - HTERCHOP
      END IF

C     Don't Allow Effective Plume Height to be < 0.0
      HEOUT = AMAX1( 0.0, HEOUT)

      RETURN
      END

      SUBROUTINE VHEFF(ZARG,HEFOUT,HEOUT)
C***********************************************************************
C                 VHEFF Module of the ISC Model - Long Term
C
C        PURPOSE: Calculates Effective Plume Height for VOLUME Sources (m)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Terrain Elevation of Receptor
C
C        OUTPUTS: Effective Plume Height (HE)
C
C        CALLED FROM:   VCALC2
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'VHEFF'

C     Calculate Terrain Height Above Plant Grade (Chopped-off at Release Height)
      IF (FLAT) THEN
         HTERCHOP = 0.0
      ELSE IF (ELEV) THEN
         HTERCHOP = AMIN1( HS, (ZARG - ZS))
      END IF

C     Calculate Effective Plume Height (No Rise) Adjusted for Terrain Height
      HEOUT = HS - HTERCHOP

C     Save Plume Height for Flat Terrain for Later Comparison to Mixing Height
      HEFOUT = HS

      RETURN
      END

      SUBROUTINE PDISLT(XARG,SZOUT,XZOUT,SBOUT)
C***********************************************************************
C                 PDISLT Module of the ISC Model - Long Term
C
C        PURPOSE: Calculates Dispersion Parameters for POINT Sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        MODIFIED BY D. Strimaitis, SRC (initialize SBID to 0.0)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One STAR Element
C                 Downwind Distance
C
C        OUTPUTS: Vertical Dispersion Coefficients, SZ
C
C        CALLED FROM:   PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'PDISLT'

      IF (.NOT. WAKE) THEN
C        Calculate Sigma-z from Curves                   ---   CALL SIGZ
         CALL SIGZ(XARG,SZARG)
         IF (.NOT. NOBID) THEN
C           Apply BIDLT                                  ---   CALL BIDLT
            CALL BIDLT(DHP,SZARG,SZOUT,SBOUT)
         ELSE
            SBOUT = 0.0
            SZOUT = SZARG
         END IF
         XZOUT = 0.0
      ELSE IF (WAKE) THEN
C        Calculate Building Enhanced Sigma-z             ---   CALL SZENH
         CALL SZENH(XARG,SZARG,XZOUT)
         IF ((.NOT. NOBID) .AND. (.NOT. WAKESS)) THEN
C           Apply BIDLT                                  ---   CALL BIDLT
            CALL BIDLT(DHP,SZARG,SZOUT,SBOUT)
         ELSE
            SBOUT = 0.0
            SZOUT = SZARG
         END IF
      END IF

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE VDISLT(XARG,SZOUT,XZOUT)
C***********************************************************************
C                 VDISLT Module of the ISC Model - Long Term
C
C        PURPOSE: Calculates Dispersion Parameters for VOLUME Sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Meteorological Variables for One STAR Element
C                 Downwind Distance
C
C        OUTPUTS: Vertical Dispersion Coefficient, SZ
C
C        CALLED FROM:   VCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'VDISLT'

C     Calculate Vertical Virtual Distance                ---   CALL XVZ
      CALL XVZ(XARG,XZOUT)
C     Calculate Sigma-z from Curves for X+XZ             ---   CALL SIGZ
      CALL SIGZ(XARG+XZOUT,SZOUT)

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE ADISLT(XARG,SYOUT,SZOUT,XYOUT,XZOUT)
C***********************************************************************
C                 ADISLT Module of the ISC Model - Long Term
C
C        PURPOSE: Calculates Dispersion Parameters for AREA Sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Meteorological Variables for One STAR Element
C                 Downwind Distance
C
C        OUTPUTS: Vertical Dispersion Coefficient, SZ
C
C        CALLED FROM:   ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'ADISLT'

      X1 = XARG * 0.001
      SZOUT = 1.0
      SYOUT = 1.0
      IF (RURAL .AND. X1 .LE. 0.0005) THEN
         CALL SZCOEF(X1, AFAC, BFAC, X1MAX, X1MIN)
         SZOUT = AFAC
      ELSE
C        Calculate Sigma-y from Curves for X                ---   CALL SIGY
         CALL SIGY(XARG,SYOUT)
C        Calculate Sigma-z from Curves for X                ---   CALL SIGZ
         CALL SIGZ(XARG,SZOUT)
      END IF
      SYOUT = MAX(SYOUT,0.0001)
      SZOUT = MAX(SZOUT,0.0001)
      XYOUT = 0.0
      XZOUT = 0.0

C     Add Initial Dispersion for OPENPIT Sources
      IF (SZINIT .GT. 0.0) THEN
         SZOUT = SQRT (SZOUT*SZOUT + SZINIT*SZINIT)
      END IF

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE PCHILT
C***********************************************************************
C                 PCHILT Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Concentration for POINT Sources and VOLUME
C                 Sources (Virtual Points) Using Gaussian Plume Equation
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C
C        CALLED FROM:   PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      LOGICAL LTERR

C     Variable Initializations
      MODNAM = 'PCHILT'
C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR  = .FALSE.

      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         GO TO 99
      ELSE
C        Determine deposition correction factors   ---   CALL DEPCOR
         IF (LDEP) THEN
C           Loop over particle sizes
            DO 150 I=1,NPD
               DQCOR(I)  = 1.0
               PCORZR(I) = 1.0
               PCORZD(I) = 1.0
               SZCOR(I)  = 1.0
               IF (DDPLETE) THEN
                  CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &              X,XZ,HEFLAT,ZI,US,XS,YS,XR,YR,
     &              RURAL,URBAN,KST,SZ,SBID,
     &              SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &              SRCTYP(ISRC),LTGRID,KURDAT,
     &              DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I))
               ENDIF
150         CONTINUE
         ENDIF
         IF (NPD .EQ. 0) THEN
            V = 0.
C           Calculate Concentration Form of V         ---   CALL VERT
            A0 = -0.5/(SZ*SZ)
            CALL VERT(HE,SZ,A0,ZFLAG,V)
C           Include SZ in the denomenator of V
            V = V/SZ
         ELSE
C           Calculate the Vertical Term, V for particles
            V = 0.0
            DO J = 1, NPD
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (X/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = AMAX1(0.0,HESETL)
C              For Dry Flux, Complete Vertical Term is Needed for
C              Each Particle Size                        ---   CALL VERT
               CALL VERT(HESETL,SZADJ,A0,ZFLAG,VJ)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               V = V + (PHI(J) * DQCOR(J) * PCORZR(J) * VJ)/SZADJ
            ENDDO
         END IF
C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (X)
C        Calculate the Smoothing Term, SM                   ---   CALL LTSMTH
         CALL LTSMTH
C        Calculate Contribution for this STAR freq (SZ in denominator of V)
         SRCVAL(IREC,ISRC)= SRCVAL(IREC,ISRC) +
     &         FREQ(IWS,ISEC,IKST)*(QTK*SM*D*V)
     &         /(SRT2PI*RPV*DELTHP*US)
      END IF

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
         CALL DEBOUT
      END IF

 99   CONTINUE

      RETURN
      END

      SUBROUTINE PDEPLT
C***********************************************************************
C                 PDEPLT Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Hourly Deposition for POINT Sources and VOLUME
C                 Sources (Virtual Points) Using Gaussian Plume Equation
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Deposition for Particular Source/Receptor Combination
C
C        CALLED FROM:   PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      LOGICAL LTERR

C     Variable Initializations
      MODNAM = 'PDEPLT'
C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR  = .FALSE.

      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         GO TO 99
      ELSE
C        Determine deposition correction factors   ---   CALL DEPCOR
         IF (LDEP) THEN
C           Loop over particle sizes
            DO 150 I=1,NPD
               DQCOR(I)  = 1.0
               PCORZR(I) = 1.0
               PCORZD(I) = 1.0
               SZCOR(I)  = 1.0
               IF (DDPLETE) THEN
                  CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &              X,XZ,HEFLAT,ZI,US,XS,YS,XR,YR,
     &              RURAL,URBAN,KST,SZ,SBID,
     &              SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &              SRCTYP(ISRC),LTGRID,KURDAT,
     &              DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I))
               ENDIF
150         CONTINUE
         ENDIF
         IF (NPD .EQ. 0) THEN
            V = 0.
C           Calculate Concentration Form of V         ---   CALL VERT
            A0 = -0.5/(SZ*SZ)
            CALL VERT(HE,SZ,A0,ZRDEP,V)
C           Include SZ in the denomenator of V
            V = V/SZ
         ELSE
C           Calculate the Vertical Term, V for particles
            V = 0.0
            DO J = 1, NPD
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (X/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = AMAX1(0.0,HESETL)
C              For Dry Flux, Complete Vertical Term is Needed for
C              Each Particle Size                           ---   CALL VERT
               CALL VERT(HESETL,SZADJ,A0,ZRDEP,VJ)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               V = V + (VDEP(J)*PHI(J)*DQCOR(J)*PCORZR(J)*VJ)/SZADJ
            ENDDO
         END IF
C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (X)
C        Calculate the Smoothing Term, SM                   ---   CALL LTSMTH
         CALL LTSMTH
C        Calculate Contribution for this STAR freq (SZ in denominator of V)
         SRCVAL(IREC,ISRC)= SRCVAL(IREC,ISRC) +
     &         FREQ(IWS,ISEC,IKST)*(QTK*SM*D*V)/
     &         (SRT2PI*RPV*DELTHP*US)
      END IF

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
         CALL DEBOUT
      END IF

 99   CONTINUE

      RETURN
      END

      SUBROUTINE DEBOUT
C***********************************************************************
C                 DEBOUT Module of ISC Model - PASS ONE
C
C        PURPOSE: Outputs Debugging Information: Sigmas, Plume Heights,
C                 etc., for Each Calculation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Debug Outputs
C
C        CALLED FROM:   PCHIL, PDEPL, ACHIL, ADEPL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DEBOUT'

      WRITE(IOUNIT,119) IAVE,IKST,ISEC,IWS,ISRC,IREC
 119  FORMAT(/1X,'IAVE=',I3,'  IKST=',I3,'  ISEC=',I3,'  IWS=',I3,
     &          '  ISRC=',I3,'  IREC=',I3)
      IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
         WRITE(IOUNIT,209) QS,HS,TS,VS,DS,DSBH,DSBW,US
 209     FORMAT(1X,'QS= ',F8.2,' HS= ',F8.2,' TS= ',F8.2,' VS= ',
     &       F8.2,' DS= ',F8.2,' DSBH= ',F8.2,' DSBW= ',F8.2,
     &       ' US= ',F8.5)
         WRITE(IOUNIT,219) FB, FM
 219     FORMAT(1X,' FB= ',F11.5,' FM= ',F11.5)
      ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
         WRITE(IOUNIT,229) QS,HS,SYINIT,SZINIT,US
 229     FORMAT(1X,'QS= ',F8.2,'  HS= ',F8.2,'  SYINIT= ',F8.2,
     &          '  SZINIT= ',F8.2,'  US= ',F8.5)
      ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
         WRITE(IOUNIT,239) QS,HS,XINIT,US
 239     FORMAT(1X,'QS= ',F8.2,'  HS= ',F8.2,'  XINIT= ',F8.2,
     &          '  US= ',F8.5)
      END IF
      WRITE(IOUNIT,319) X,Y,XY,XZ,SZ,RPV,DELTHP,SM
 319  FORMAT(1X,'X= ',F11.5,'  Y= ',F11.5,'  XY= ',F11.5,'  XZ= ',
     &          F11.5,'  SZ= ',F11.5,'  RPV= ',F11.5,'  DELTHP= ',F8.5,
     &          '  SM= ',F8.5)
      IF (CONC) THEN
         WRITE(IOUNIT,409) HE,HEMWAK,HEFLAT,KST,TA,ZI,V,D
 409     FORMAT(1X,'HE= ',F11.5,'  HEMWAK= ',F11.5,'  HEFLAT= ',F11.5,
     &          '  KST= ',I2,'  TA= ',F6.1,'  ZI= ',F8.2,'  V= ',E12.6,
     &          '  D= ',E12.6)
      ELSE
         WRITE(IOUNIT,419) HE,HEMWAK,HEFLAT,KST,TA,ZI,VD,D
 419     FORMAT(1X,'HE= ',F11.5,'  HEMWAK= ',F11.5,'  HEFLAT= ',F11.5,
     &          '  KST= ',I2,'  TA= ',F6.1,'  ZI= ',F8.2,'  VD= ',E12.6,
     &          '  D= ',E12.6)
      END IF
      WRITE(IOUNIT,519) ZLB,RINIT,ZLY,DA,WAKE,WAKESS
 519  FORMAT(1X,'ZLB=',F11.5,'  RINIT= ',F9.4,'  ZLY= ',
     &       F9.4,'  DA= ',F8.6,'  WAKE= ',L3,'  WAKESS=',L3)
      WRITE(IOUNIT,619) FREQ(IWS,ISEC,IKST),QTK,XF,XFB,XFM,DHF,DHP
 619  FORMAT(1X,'FREQ= ',F11.5,' QTK= ',E12.5,'  XF= ',F9.2,
     &       '  XFB= ',F9.2,'  XFM= ',F9.2,'  DHF= ',F9.2,
     &       '  DHP= ',F9.2)
      WRITE(IOUNIT,719) SRCVAL(IREC,ISRC)
 719  FORMAT(1X,'*** SRCVAL(IREC,ISRC)= ',G16.8,' ***')

      RETURN
      END

C***  Begin new code for area source numerical integration algorithm - 7/7/93

      FUNCTION SMFREQ(IDS,WINDD)
C***********************************************************************
C                 SMFREQ Module of ISC2 Long Term Model - ISCLT2
C
C        PURPOSE: Calculates Smoothing Frequency Function, SM
C                 for AREA sources
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Sector Number, IDS
C                 Current WD Deviation from Sector Centerline, WINDD
C
C        OUTPUTS: Smoothing Frequency Function, SM
C
C        CALLED FROM:   ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SMFREQ'

      IF ( WINDD .GT. 0.0 ) THEN
         IDSS = IDS + 1
         IF (IDSS .GT. 16 ) IDSS = IDSS - 16
         SLOPE = (FREQ(IWS,IDSS,IKST) - FREQ(IWS,IDS,IKST))/DELTHP
      ELSE
         IDSS = IDS - 1
         IF (IDSS .LT. 1 ) IDSS = IDSS + 16
         SLOPE = (FREQ(IWS,IDS,IKST) - FREQ(IWS,IDSS,IKST))/DELTHP
      END IF

      SMFREQ = FREQ(IWS,ISEC,IKST) + SLOPE*WINDD

      RETURN
      END

      SUBROUTINE ACALCT
C***********************************************************************
C                 ACALCT Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for AREA sources
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C*       MODIFIED BY J. Hardikar, PES, to make consistent with the new
C*                   OPENPIT Source Methodology - 7/20/94
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

      REAL XSPA(NVMAX), YSPA(NVMAX)
      
C     Variable Initializations
      MODNAM = 'ACALCT'

C     Determine Coordinates of Vertices for AREA Source
C     in Wind Direction Coordinate System             ---   CALL AVERTS
      CALL AVERTS(XVERT,YVERT,XSPA,YSPA,NVERT+1)

C*    Store Coordinates of the Effective Area in COMMON Variables
      DO 40 IVERT = 1,NVERT+1
         SPA(IVERT,1) = XSPA(IVERT)
         SPA(IVERT,2) = YSPA(IVERT)
40    CONTINUE
       
C     Calculate Down and Crosswind Distances          ---   CALL XYDIST
      CALL XYDIST
      HE = HS
      HEFLAT = HE
      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         HRVAL = 0.0
      ELSE
C        Calculate Area Source Integral               ---   CALL AREAIN
         CALL AREAIN
      END IF

      RETURN
      END

      SUBROUTINE OCALCT
C***********************************************************************
C                 OCALCT Module of ISC Model - Long Term
C
C        PURPOSE: Calculates concentration or deposition values
C                 for OPENPIT sources
C
C        PROGRAMMER: JAYANT HARDIKAR, ROGER BRODE
C
C        DATE:    Aug 22, 1994
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One STAR Element
C
C        OUTPUTS: CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   OCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

    
C     Variable Initializations
      MODNAM = 'OCALCT'

C*    Initialize the Total Adjusted Emission Rate from
C*    All Particles
      QPTOT = 0.0          


C*    Loop over Particle Size Categories
      DO 20 ICAT = 1,NPD

C*       Initialize PHI Array
         PHI(ICAT) = APHI(ICAT,ISRC)

C*       Calculate the Escape Fraction for Each Category    ---   CALL ESCAPE
         CALL ESCAPE(ICAT)

C*       Adjust the Emission Rate for Each Category         ---   CALL ADJEMI
         CALL ADJEMI(ICAT,QPTOT)

C*    End Loop Over Particle Size Categories
20    CONTINUE

C*    Skip Calculations if QPTOT = 0.0
      IF (QPTOT .EQ. 0.0)  GO TO 999

C*    Adjust the Mass Fractions for All the Particle 
C*    Size Categories                                       ---   CALL AMFRAC
      CALL AMFRAC(QPTOT)
      
C*    Determine the AlongWind Length of the OPENPIT Source  ---   CALL LWIND
      CALL LWIND

C*    Calculate the Relative Depth of the OPENPIT Source    ---   CALL PDEPTH            
      CALL PDEPTH
      
C*    Calculate the Fractional Size of the 
C*    Effective Pit Area                                    ---   CALL PTFRAC
      CALL PTFRAC


C*    WRITE DEBUG INFORMATION
      IF (DEBUG) THEN
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) 'DETAIL INFORMATION ON THE OPENPIT SOURCE:'
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*)   
      ENDIF
     
C*    Determine the Coordinates of the Effective Pit Area
C*    in Wind Direction Coordinate System                   ---   CALL PITEFF
      CALL PITEFF

C*    Calculate the Emission Rate for the Effective
C*    Pit Area                                              ---   CALL PITEMI
      CALL PITEMI(QPTOT)

C*    WRITE DEBUG INFORMATION
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'OPENPIT PARTICLE CHARACTERISTICS:'
         WRITE (IOUNIT,*) '-------------------------------'
         WRITE (IOUNIT,*) 
         WRITE (IOUNIT,8000) (EFRAC(II),II = 1, NPD)
8000     FORMAT (1X,'ESCAPE FRACTIONS= ',10(F8.3,2X))
         WRITE (IOUNIT,8200) (QPART(II),II = 1, NPD)
8200     FORMAT (1X,'ADJUSTED EMISSION RATES= ',10(F8.3,2X))
         WRITE (IOUNIT,8400) (PHI(II),II = 1, NPD)
8400     FORMAT (1X,'ADJUSTED MASS FRACTIONS= ',10(F8.3,2X))
         WRITE (IOUNIT,*) 'EMISSION RATE OF EFFECTIVE PIT= ',QEFF
         WRITE (IOUNIT,*) 
      ENDIF                  


C     Set Particle Deposition Variables for this Source
      IF (LDEP) THEN
C        Calculate Deposition Velocities for this Source    ---   CALL VDP
         CALL VDP
      END IF

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFACT
      CALL EMFTLT(QEFF)
      
      IF (LDEP) THEN
C        Calculate Min Sigma-z for Settled Plume @ Surface  --- CALL SETSZMN
         CALL SETSZMN
      END IF

C*    Initialize XY,XZ to 0.0 (XZ is used in
C*    call to DEPCOR from PLUMEF)
      XY = 0.0
      XZ = 0.0

C     Initialize SBID to 0.0 (for call to DEPCOR from PLUMEF)
      SBID = 0.0

C     Calculate Down and Crosswind Distances                ---   CALL XYDIST
      CALL XYDIST
      HE = HS
      HEFLAT = HE
      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         HRVAL = 0.0
      ELSE
C        Calculate Area Source Integral                     ---   CALL AREAIN
         CALL AREAIN
      END IF

 999  RETURN
      END


      SUBROUTINE SETWD(WDIN)
C***********************************************************************
C                 SETWD Module of ISC2 Long Term Model - ISCLT2
C
C        PURPOSE: Sets Wind Direction Variables for Current WDIR
C                 for AREA Sources
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Current Wind Direction, WDIN
C
C        OUTPUTS: Wind Direction Variables, WDSIN and WDCOS
C
C        CALLED FROM:   ACALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SETWD'

      WDIR = WDIN
      IF (WDIR .LT. 0.0) THEN
         WDIR = WDIR + 2.*PI
      ELSE IF (WDIR .GT. 2.*PI) THEN
         WDIR = WDIR - 2.*PI
      END IF

C     Determine SIN and COS of WDIR for Later Use
      WDSIN = SIN(WDIR)
      WDCOS = COS(WDIR)

C*    Set the Flow Vector for this Direction for use by OPENPIT Routines
      AFV = WDIR*RTODEG + 180.
      IF (AFV .GT. 360.0)  AFV = AFV - 360.0

      RETURN
      END

      SUBROUTINE ACHI(X1,RCZ)
C***********************************************************************
C                 ACHI Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration for AREA Sources
C                 Using Gaussian Plume Equation for Finite Line Source
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Concentration for A Unit Of Source/Receptor Combination
C
C        CALLED FROM:   PLUMEF
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'ACHI'

C     Convert distance X1 from km to meters
      XARG = X1 * 1000.

      RCZ = 0.0
      IF (XARG .GE. 1.0) THEN
         IF (NPD .EQ. 0) THEN
            V = 0.
C           Calculate Concentration Form of V         ---   CALL VERT
            A0 = -0.5/(SZ*SZ)
            CALL VERT(HE,SZ,A0,ZFLAG,V)
C           Include SZ in the denomenator of V
            V = V/SZ
         ELSE
C           Calculate the Vertical Term, V for particles
            V = 0.0
            DO J = 1, NPD
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (XARG/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = AMAX1(0.0,HESETL)
C              For Dry Flux, Complete Vertical Term is Needed for
C              Each Particle Size                        ---   CALL VERT
               CALL VERT(HESETL,SZADJ,A0,ZFLAG,VJ)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               V = V + (PHI(J) * DQCOR(J) * PCORZR(J) * VJ)/SZADJ
            ENDDO
         END IF
C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (XARG)
C        Calculate Contribution (SZ in denominator of V)
         RCZ = (D*V)/(SRT2PI)
      END IF

      RETURN
      END

      SUBROUTINE ADEP(X1,RCZ)
C***********************************************************************
C                 ADEP Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Deposition for AREA Sources
C                 Using Gaussian Plume Equation for Finite Line Source
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    July 7, 1993
C
C        MODIFIED by Yicheng Zhuang, SRC to combine version 93188 with
C                 version 93046 - 9/28/93
C
C        DATE:    September 28, 1993
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION) - 2/15/93
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Deposition for A Unit Of Source /Receptor Combination
C
C        CALLED FROM:   PLUMEF
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'ADEP'

C     Convert distance X1 from km to meters
      XARG = X1 * 1000.

      RCZ = 0.0
      IF (XARG .GE. 1.0) THEN
         IF (NPD .EQ. 0) THEN
            V = 0.
C           Calculate Concentration Form of V         ---   CALL VERT
            A0 = -0.5/(SZ*SZ)
            CALL VERT(HE,SZ,A0,ZRDEP,V)
C           Include SZ in the denomenator of V
            V = V/SZ
         ELSE
C           Calculate the Vertical Term, V for particles
            V = 0.0
            DO J = 1, NPD
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (XARG/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = AMAX1(0.0,HESETL)
C              For Dry Flux, Complete Vertical Term is Needed for
C              Each Particle Size                        ---   CALL VERT
               CALL VERT(HESETL,SZADJ,A0,ZRDEP,VJ)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               V = V + (VDEP(J)*PHI(J)*DQCOR(J)*PCORZR(J)*VJ)/SZADJ
            ENDDO
         END IF
C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (XARG)
C        Calculate Contribution (SZ in denominator of V)
         RCZ = (D*V)/(SRT2PI)
      END IF

      RETURN
      END

c----------------------------------------------------------------------
      SUBROUTINE VDP
c----------------------------------------------------------------------
c
c --- ISC2LT     Version:  1.0     Level:  930215                   VDP
c                J. Scire, SRC
c
c --- PURPOSE:  Compute particle deposition velocities for each size
c               category of a size distribution.
c
c --- MODIFIED   May 26, 1995
c                Modified atmospheric resistance term, ra, based on
c                D. Byun and R. Dennis, Atmos. Environ., Vol. 29, No. 1
c                R. W. Brode, PES, Inc.
c
c --- MODIFIED: Revised March 8, 1994 to use ADOM1 formulation for
c ---           deposition layer resistance.  D. Bailey  OAQPS
c --- INPUTS:
c     Common block /METVAR/ variables:
c               Z0M - real       - Surface roughness length (m)
c
c             USTAR - real       - Friction velocity (m/s)
c                EL - real       - Monin-Obukhov length (m)
c     Common block /CALCS3/ variables:
c               NPD - integer    - Number of particle size categories
c             PDIAM - real array - Mean diameter (microns) of each
c                                  particle size category
c               PHI - real array - Mass fraction in each size category
c             PDENS - real       - Particle density (g/cm**3)
c                SC - real array - Schmidt number
c             VGRAV - real array - Gravitational settling velocity (m/s)
c             TSTOP - real array - Stopping time (s)
c     Common block /SOURC4/ variables:
c            VAIRMS - real       - Viscosity of air (m**2/s)
c             ZRDEP - real       - Reference height (m)
c            VDPHOR - real       - Phoretic effects term (m/s)
c
c --- OUTPUT:
c     Common block /CALCS3/ variables:
c              VDEP - real array - Deposition velocity (m/s) for each
c                                  particle size category
c
c --- VDP called by:  PCALC, VCALC, ACALC
c --- VDP calls:      none
c----------------------------------------------------------------------
c
      INCLUDE 'MAIN1LT.INC'
c
      real rd(npdmax)
c
      io6=iounit

C     Calculate Deposition Reference Height, ZRDEP.
      ZRDEP = AMAX1( 1.0, 20.*Z0M)
c
c ***
      if(DEBUG)then
         write(io6,*)
         write(io6,*)'SUBR. VDP -- Inputs'
         write(io6,*)'USTAR (m/s)     = ',ustar
         write(io6,*)'MONIN-EL (m)    = ',el
         write(io6,*)'Z0M (m)         = ',z0m
         write(io6,*)'VDPHOR (m/s)    = ',vdphor
         write(io6,*)'NPD             = ',npd
         write(io6,*)'PDIAM (um)      = ',(pdiam(n),n=1,npd)
         write(io6,*)'FRACT           = ',(phi(n),n=1,npd)
         write(io6,*)'PDENS (g/cm**3) = ',(pdens(n),n=1,npd)
         write(io6,*)'SC              = ',(sc(n),n=1,npd)
         write(io6,*)'VGRAV (m/s)     = ',(vgrav(n),n=1,npd)
         write(io6,*)'TSTOP (s)       = ',(tstop(n),n=1,npd)
         write(io6,*)'VAIRMS (m**2/s) = ',vairms
         write(io6,*)'ZRDEP (m)       = ',zrdep
         write(io6,*)'VDPHOR (m/s)    = ',vdphor
      endif
c ***
c
c --- Use minimum value of USTAR to avoid numerical problems
c --- when USTAR near zero
      ustarr=AMAX1(ustar,1.e-9)
c
c --- Minimum absolute value of Monin-Obukhov length is 1.0 m
      if(el.GE.0.0)then
c ---    stable
         ell=AMAX1(el,1.0)
      else
c ---    unstable
         ell=AMIN1(el,-1.0)
      endif
c
c --- Calculate atmospheric resistance (s/m)
      elabs=ABS(ell)
      if (ell .gt. 0.0) then
c ---    Stable
c ---    VK is the von Karman constant, set as parameter in MAIN1LT.INC
         psih = 4.7*zrdep/ell
         ra = (1.0/(vk*ustarr)) * (ALOG(zrdep/z0m) + psih)

      else
c ---    Unstable
         a1 = 16.*zrdep/elabs
         b1 = 16.*z0m/elabs
         ra = (1.0/(vk*ustarr)) * (1.0*ALOG(
     &        ((2.+a1)-2.*SQRT(1.+a1)) * ((2.+b1)+2.*SQRT(1.+b1)) /
     &        (a1*b1) ))
      endif
c
      t1=ustarr*ustarr/vairms
c
c ***
      if(DEBUG)then
         write(io6,*)
         write(io6,*)'USTARR (m/s)    = ',ustarr
         write(io6,*)'ELL (m)         = ',ell
         write(io6,*)'PSIH            = ',psih
      endif
c ***

c --- LOOP OVER SIZE INTERVALS
      do 10 i=1,npd

C     compute the traditional Stokes Number  St = (Vg/g) * (Ustar ^2/Nu)
         st=tstop(i)*t1

c ---    Compute inertial impaction term using ADOM1 formulation        DTB94067
                                                                        DTB94067
         xinert = 10**(-3/st)                                           DTB94067

c ---    Adjust the Schmidt Number                                      DTB94067
         Schmidt = sc(i)**(-0.667)

c ---    Deposition layer resistance (s/m)
         rd(i) = 1.0 / (ustarr * (Schmidt + xinert) )                   DTB94067

c ---    Deposition velocity for this current interval
         vdep(i)=1.0/(ra+rd(i)+ra*rd(i)*vgrav(i))+vgrav(i)+vdphor
10    continue

      if(DEBUG)then
         write(io6,*)
         write(io6,*)'RA (s/m)    = ',ra
         write(io6,*)'RD (s/m)    = ',(rd(n),n=1,npd)
         write(io6,*)'VDEP (m/s)  = ',(vdep(n),n=1,npd)
      endif
c ***
c
      return
      end


c-----------------------------------------------------------------------
      subroutine setszmn
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SETSZMN
c               D. Strimaitis, SRC
c
c PURPOSE:     SETSZMN determines the value of sigma-z at which the rate
c              of growth in sigma-z equals the rate at which the settling
c              velocity acts to reduce the height of the center-of-mass.
c              A default minimum of 2*zd, where zd is the near-surface
c              height at which the deposition flux is evaluated, is
c              returned if there is no balance-point.
c
c ARGUMENTS:  (MAIN1LT.INC)
c    PASSED:  kst       stability class (A=1, F=6)                   [i]
c             zrdep     reference height for deposition flux  (m)    [r]
c             vs        settling velocity  (m/s)                     [r]
c             us        plume advection wind speed (m/s)             [r]
c             urban     logical for URBAN/RURAL dispersion params    [l]
c             npd       number of particle size categories           [i]
c
c  RETURNED:  szmin     Minimum value of sigma-z (m)                 [r]
c
c CALLING ROUTINES:   PCALC, VCALC, ACALC
c
c EXTERNAL ROUTINES:  GCUBIC
c-----------------------------------------------------------------------
      include 'MAIN1LT.INC'

      real root(3),car(6),cau(6),cbr(6),cbu(6)

      data car/.2,.12,.08,.06,.03,.016/
      data cbr/0.,0.,.0002,.0015,.0003,.0003/
      data cau/.24,.24,.2,.14,.08,.08/
      data cbu/.001,.001,0.,.0003,.0015,.0015/


c --- Loop over particle sizes
      do i=1,npd
         xmin=0.0
         szmin(i)=2.*zrdep
         c=rtpiby2*vgrav(i)/us

c ---    Urban section
         if(URBAN) then
            a=cau(kst)
            b=cbu(kst)
            if(kst .GE. 4) then
               if(a .GT. 20.*c) then
                  szmin(i)=a*a/(2.*b*c)
               elseif(a .GT. c) then
c ---             Solve cubic for y=bx, then report x      ---  call GCUBIC
                  aby2csq=(a/(2.*c))**2
                  a1=(3.-aby2csq)
                  a2=(3.-4.*aby2csq)
                  a3=(1.-4.*aby2csq)
                  call GCUBIC(a1,a2,a3,root)
c ---             There should be ONE real root
                  if(root(2) .NE. 0. .OR. root(3) .NE. 0.) then
                     write(*,*) 'SETSZMN: Potential error!!! '
                     write(*,*) 'More than 1 root ----'
                     write(*,*) 'xb= ',(root(j),j=1,3)
                  endif
                  xmin=root(1)/b
                  szmin(i)=a*xmin/SQRT(1.+b*xmin)
               endif
            endif

c ---    Rural section
         else
            a=car(kst)
            b=cbr(kst)
            if(kst .EQ. 3 .OR. kst .EQ. 4) then
               if(a .GT. 20.*c) then
                  szmin(i)=a*a/(2.*b*c)
               elseif(a .GT. c) then
c ---             Solve cubic for y=bx, then report x      ---  call GCUBIC
                  aby2csq=(a/(2.*c))**2
                  a1=(3.-aby2csq)
                  a2=(3.-4.*aby2csq)
                  a3=(1.-4.*aby2csq)
                  call GCUBIC(a1,a2,a3,root)
c ---             There should be ONE real root
                  if(root(2) .NE. 0. .OR. root(3) .NE. 0.) then
                     write(*,*) 'Potential error!!! More than 1 root'
                     write(*,*) 'xb= ',(root(j),j=1,3)
                  endif
                  xmin=root(1)/b
                  szmin(i)=a*xmin/SQRT(1.+b*xmin)
               endif
            elseif(kst .GT. 4) then
               if(a .GT. c) then
                  xmin=(SQRT(a/c)-1.)/b
                  szmin(i)=a*xmin/(1+b*xmin)
               endif
            endif
         endif

      enddo

      return
      end

c-----------------------------------------------------------------------
      subroutine gcubic(a1,a2,a3,root)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           GCUBIC
c               D. Strimaitis, SRC
c
c PURPOSE:     Program solves the general cubic equation of the form:
c                  0 = x**3 + (a1)x**2 + (a2)x + (a3)
c              for the real roots
c              (Numerical Recipes, Press et al., 1986)
c
c ARGUMENTS:
c    PASSED:  a1,a2,a3  constants for terms as described above       [r]
c
c  RETURNED:  root      root(s) of equation                          [r]
c
c CALLING ROUTINES:   (utility routine)
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      real root(3)

      data twopi/6.2831853/,fourpi/12.566371/

      third=1./3.
      a1sq=a1*a1
      a1cube=a1*a1sq
      a1by3=a1*third

      q=(a1sq-3.*a2)/9.
      r=(2.*a1cube-9.*a1*a2+27.*a3)/54.

      qcube=q*q*q
      rsq=r*r

      if(qcube .GE. rsq) then
c ---    THREE real roots
         sqrtq2=SQRT(q)*2.
         theta=ACOS(r/SQRT(qcube))
         root(1)=-sqrtq2*COS(theta/3.)-a1by3
         root(2)=-sqrtq2*COS((theta+twopi)/3.)-a1by3
         root(3)=-sqrtq2*COS((theta+fourpi)/3.)-a1by3
      else
c ---    ONE real root
         arg=(SQRT(rsq-qcube)+ABS(r))**third
         root(1)=-SIGN(1.0,r)*(arg+q/arg)-a1by3
         root(2)=0.
         root(3)=0.
      endif


      return
      end
      
