c-----------------------------------------------------------------------
      subroutine depcor(vdi,vsi,zdi,zri,xri,xvi,hi,hmixi,ui,
     &                  xsrci,ysrci,xreci,yreci,
     &                  rurali,urbani,ksti,sgzi,sgz0i,szmni,
     &                  erin,epin,lterri,debugi,iouniti,
     &                  srctypi,ltgridi,kurdati,
     &                  qcor,pcor,pxrzd,szcor)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           DEPCOR
c               D. Strimaitis, SRC
c
c MODIFIED:    Remove comment erroneously placed on a valid ELSEIF line.
c              R. W. Brode, PES, Inc. - 4/19/96
c
c MODIFIED:    Includes elevation data (MSL) needed to simulate
c              COMPLEX I terrain treatment, and new settling treatment.
c
c MODIFIED:    Uses proper distance-dependent plume rise and sigmas in
c              the integration in QATR2.  Also modified to use terrain
c              adjustment during the integration for simple terrain.
c              R. W. Brode, PES, Inc. - 9/30/94
c
c MODIFIED by R. Brode, PES, added initialization for urban 'cp' - 8/2/94
c
c PURPOSE:     DEPCOR returns correction factors for the emission rate
c              and the vertical distribution of plume material to
c              account for deposition between the source and the current
c              receptor.
c
c ARGUMENTS:
c    PASSED:  vdi,vsi   total deposition and gravitational settling  [r]
c                       velocities (m/s)
c             zdi       height for evaluating deposition (m)         [r]
c             zri       receptor height above sfc (m)                [r]
c             xri       receptor distance (m)                        [r]
c             xvi       virtual source distance (m)                  [r]
c             hi        plume height (m)                             [r]
c             hmixi     mixing height (m)                            [r]
c             ui        wind speed (m/s)                             [r]
c             xsrci     source position in x  (m)                    [r]
c             ysrci     source position in y  (m)                    [r]
c             xreci     receptor position in x  (m)                  [r]
c             yreci     receptor position in y  (m)                  [r]
c             rurali    logical for rural dispersion curves          [l]
c             urbani    logical for urban dispersion curves          [l]
c             ksti      stability class indicator                    [i]
c             sgzi      sigma-z at current receptor (m)              [r]
c             sgz0i     initial sigma-z (e.g. for BID) (m)           [r]
c             szmni     minimum sigma-z for settling   (m)           [r]
c             erin      elevation (MSL) at receptor    (m)           [r]
c             epin      elevation (MSL) at source      (m)           [r]
c             lterri    logical controlling terrain adjustments      [l]
c             debugi    logical controlling DEBUG output             [l]
c             iouniti   unit number for DEBUG output                 [i]
c             ltgridi   logical indicating gridded terrain data      [l]
c  RETURNED:  qcor      ratio of depleted emission rate to original  [r]
c             pcorzr    profile correction factor at receptor height [r]
c             pcorzd    profile correction factor at deposition ht   [r]
c             szcor     sigma-z correction factor (settling)         [r]
c
c CALLING ROUTINES:   PCALC, VCALC, ACALC
c
c EXTERNAL ROUTINES:  SZSETL, RESIST, PROFD, PROFD1, PROFD2,
c                     DEPLETE
c-----------------------------------------------------------------------
c  NOTE:  all inputs ending with "i" are passed to subsequent routines
c         through common /DEPVAR/.

      include 'DEPVAR.INC'
      logical rurali,urbani,lterri,debugi,ltgridi
      character*8 srctypi
c  Arrays RUR and URB contain coefficients derived from Briggs sigma-z
c  coefficients, for use in Horst's resistance and profile functions.
      real rur(3,6),urb(3,6)
      data rur/3.989,0.,0.,
     2         6.649,0.,0.,
     3         9.974,0.03125,0.,
     4         13.298,0.4167,0.,
     5         26.596,0.6667,0.005,
     6         49.868,2.344,0.03296/
      data urb/3.325,0.,0.,
     2         3.325,0.,0.,
     3         3.989,0.,0.,
     4         5.699,0.01531,0.,
     5         9.974,0.2344,0.,
     6         9.974,0.2344,0./

c  Initialize deposition factors to 1, and return if edge of plume is
c  well above the ground at the receptor (h > 5 sigz)
CRWB      Note: These are initialized in SUBs. PCHILT & PDEPLT in CALC1LT.FOR
CRWB      pxrzd=1.0
CRWB      pcor=1.0
CRWB      qcor=1.0
CRWB      szcor=1.0
CRWB      Since 'hi' is plume height without terrain adjustment, this does
CRWB      not apply.
CRWB      if(hi .GT. 5.*sgzi) return

c  Set constants
      rtpiby2=1.2533141
      rt2=1.4142136
      rtpi=1.7724539

c  Assign input variables to working variables
      vd=vdi
      vs=vsi
      zd=zdi
      zr=zri
      xr=xri
      xv=xvi
      h=hi
      hmix=hmixi
      onebyu=1./ui
      xsrc=xsrci
      ysrc=ysrci
      xrec=xreci
      yrec=yreci
      rural=rurali
      urban=urbani
      kst=ksti
      sgz=sgzi
      sgz0=sgz0i
      szmn=szmni
      er=erin
      ep=epin
      lterr=lterri
      debug=debugi
      iounit=iouniti
      ltgrid=ltgridi
      srctyp=srctypi
      kurdat=kurdati

c  Obtain coefficients for resistance and profile functions
c             [a,b,c]p coefficients for profile functions
c             [a,b,c]r coefficients for resistance functions
      if(rural) then
         ar=onebyu*rur(1,kst)
         br=onebyu*rur(2,kst)
         cr=onebyu*rur(3,kst)
         ap=ar
         bp=br/rtpiby2
         cp=cr*rtpiby2
      else
         ar=onebyu*urb(1,kst)
         br=onebyu*urb(2,kst)
         cr=onebyu*urb(3,kst)
         ap=ar
         bp=br/rtpiby2
         cp=0.0
      endif

c  Flush other variables in DEPVAR common
      igrav=0


c  Set the distance at which the plume centerline reaches the surface
c  -- the touchdown distance due to gravitational settling --
c  and calculate the sigma-z at that point (sztd)
      xtd=h/(vs*onebyu)

C     Obtain Sigma-z for This Value of X = XTD
      IF (SRCTYP .EQ. 'POINT' .AND. .NOT.LTERR) THEN
C        Determine Simple Terrain Sigmas
         CALL PDISLT(XTD,SZTD,XGZ,SBID)
CLT      ELSE IF (SRCTYP .EQ. 'POINT' .AND. LTERR) THEN
CLTC        Determine Complex Terrain Sigmas
CLT         CALL PDISC(XTD,SZTD,XZCMP1,SBCMP1)
      ELSE IF (SRCTYP .EQ. 'VOLUME') THEN
         CALL VDISLT(XTD,SZTD,XZ)
      ELSE IF (SRCTYP .EQ. 'AREA') THEN
C        Calculate dispersion coefficients, SY and SZ
         CALL ADISLT(XTD,SGY,SZTD,XY,XZ)
      ELSE IF (SRCTYP .EQ. 'OPENPIT') THEN
C        Calculate dispersion coefficients, SY and SZ
         CALL ADISLT(XTD,SGY,SZTD,XY,XZ)
      END IF


c  Calculate the correction factor for sigma-z at this receptor
c  if x > xtd, to simulate the effect of settling on a surface-based
c  plume.
      if(x .GT. xtd) then
         call SZSETL(x,szgrav)
         szcor=szgrav/sgz
         sgz=szgrav
      endif


c  Obtain profile factor at height zd for current receptor (x=xr).
c  First, check importance of gravitational settling velocity by
c  computing vs*RESIST at the minimum of z=hmix or z=3*sgz.
c  Note: profile function for URBAN class A & B is of a different
c  form than all other classes, and is contained in PROFD2.
      zcheck=3.*sgz
      if(hmix .GT. h) zcheck=AMIN1(hmix,zcheck)
      if(vs*RESIST(zcheck) .GT. 0.1) then
         igrav=1
c  --  gravitational settling is "large", use numerical integration.
         call PROFD(pxrzd)
      elseif(urban .AND. kst .LT. 3) then
         igrav=0
c  --  gravitational settling is "small", use analytic function as
c  --  approximation for URBAN class A & B.
         call PROFD2(pxrzd)
      else
         igrav=0
c  --  gravitational settling is "small", use analytic function for all
c  --  other classes.
         call PROFD1(pxrzd)
      endif

c  Now compute factor at receptor height zr.
      if(zr .LE. zd) then
         pcor=pxrzd
      elseif(igrav .EQ. 0) then
         pcor=pxrzd*(1.+(vd-vs)*RESIST(zr))
      else
         pcor=pxrzd*(1.+((vd-vs)/vs)*(1.-EXP(-vs*RESIST(zr))))
      endif

c  Compute ratio of depleted source strength to initial source strength.
      call DEPLETE(qcor)

      if(debug) then
         write(iounit,*) '-------------------------------------------'
         write(iounit,*) '  DEPCOR Module:'
         write(iounit,*) '    x,y (source) = ',xsrc,ysrc
         write(iounit,*) '    x,y (recept) = ',xrec,yrec
         write(iounit,*) '   LTERR, LTGRID = ',lterr,ltgrid
         write(iounit,*) '  ZTERR(src), ep = ',ZTERR(0.),ep
         write(iounit,*) '  ZTERR(rec), ep = ',ZTERR(xr),er
         write(iounit,*) '  ZTERR(1/2)     = ',ZTERR(xr/2.)
         write(iounit,*) '     igrav,pxrzd = ',igrav,pxrzd
         write(iounit,*) '       pcor,qcor = ',pcor,qcor
         write(iounit,*) '-------------------------------------------'
      endif

      return
      end
c-----------------------------------------------------------------------
      function resist(z)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           RESIST
c               D. Strimaitis, SRC
c
c PURPOSE:     Function RESIST provides the resistance factor for a
c              particular height above the surface (z), relative to a
c              reference height (zd). Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:  z         height above the surface                     [r]
c
c  RETURNED:  resist    resistance factor (via /DEPVAR/)             [r]
c
c CALLING ROUTINES:   DEPCOR, FINT
c
c EXTERNAL ROUTINES:  ROOT
c-----------------------------------------------------------------------


c  AR, BR, and CR are the coefficients of the 3 F(z) forms given for
c  the various forms of the Briggs representation of sigma-z (rural &
c  urban)

c  Common DEPVAR contains AR, BR, CR, and zd
      include 'DEPVAR.INC'

      if(z .GT. zd) then
c  Special Case:  URBAN/Stability Class=A,B
c  Resistance function requires the root of a implicit expression.
c  Because sigma-z functions are the same for URBAN/A,B the equation to
c  solve for x(z) is ax(1+bx)^.5=z*(pi/2)^.5, where a=.24, b=.001
         if(urban .AND. kst .LT. 3) then
c --       cz=SQRT(pi/2) * z/a = 5.222142 * z, where a=.24
           cz=5.222142*z
           call ROOT(cz,xz)
           argz=cz/xz
c  Approximate functional dependence on zd using binomial expansion
c  --      c=2*b*SQRT(pi/2)/a = 0.0104443
c  --      8./(c*zd)=765.96804/zd
           argzd=765.96804/zd
           resist=AR*ALOG((argz-1.)*(argzd+1.)/(argz+1.))
         else
           resist=AR*ALOG(z/zd) + BR*(z-zd) + CR*(z*z-zd*zd)
         endif
      else
         resist=0.0
      endif

      return
      end
c-----------------------------------------------------------------------
      subroutine root(c,x)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           ROOT
c               D. Strimaitis, SRC
c
c PURPOSE:     Program solves an expression of the form:
c                  x*(1+b*x)**.5=c
c              using a simple iteration on:
c                  x=c/(1+b*x0)**.5
c
c              ! WARNING !     This is a special solver for current
c                              application.....it may not converge for
c                              other applications.
c
c ARGUMENTS:
c    PASSED:  c         constant for RHS of equation to solve        [r]
c
c  RETURNED:  x         root of equation                             [r]
c
c CALLING ROUTINES:   RESIST
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      data b/.001/,onebyb/1000./,twob/.002/
      data eby2/.005/,oneby3/.3333333/
c  "e" is a fractional error criterion for convergence, so eby2=e/2

c  First guess
      twobc=twob*c
      if(twobc .LT. 6.) then
         x0=(SQRT(1.+twobc)-1.)*onebyb
      else
         x0=(c*c*onebyb)**oneby3
      endif
10    x=c/SQRT(1.+b*x0)
      errby2=ABS(x-x0)/(x+x0)
      if(errby2 .LE. eby2) goto 100
      x0=x
      goto 10

100   continue
      return
      end
c-----------------------------------------------------------------------
      subroutine profd(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD provides the base profile factor at
c              height zd for the given deposition and settling velocities,
c              and sigma-z.  Here, the settling velocity and diffusion
c              resistance are not "small" so that a numerical integration
c              is used to obtain Ip:
c                    P(x,zd) = 1. / (1.+ Ip*(vd-vs)/vs) .... "pxzd"
c              Based on Horst (1983).
c
c MODIFIED:    To set ndim = 12 instead of 16 for faster convergence.
c              R. W. Brode, PES, Inc. - 09/30/94
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  QATR, FINT
c-----------------------------------------------------------------------

c  Set up call to integration routine QATR(xl,xu,eps,ndim,fct,y,ier,num,aux)
c  Declare parameter to fix the size of the aux array
      parameter(ndim=12)
      real aux(ndim)
      external FINT
      INTEGER*2 NUM
      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LE. 2.*zd) return

c  Evaluate integral Ip:
c  Upper limit of integral reset from infinity to MIN(5*sigma-z,hmix)
      eps=.10
      top=AMIN1(5.*sgz,hmix)
      call QATR(zd,top,eps,ndim,FINT,value,ier,num,aux)
crwb      Comment out warning messge
crwb      if(ier .EQ. 2) then
crwb         write(*,*) 'WARNING from PROFD -  integration failed to'
crwb         write(*,*) 'converge to fractional error of ',eps
crwb      endif
      pxzd=1./(1.+value*(vd-vs)/vs)

      return
      end
c-----------------------------------------------------------------------
      function fint(z)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           FINT
c               D. Strimaitis, SRC
c
c PURPOSE:     Function is the integrand of integral over height to
c              calculate the profile parameter P(x,zd).  The resistance
c              value is returned from the function RESIST.  Common
c              /DEPVAR/ is used to pass data that are constant during
c              the integration, so QATR (the integrator) only needs to
c              pass values of height (z).
c               -VCOUP calculates the vertical coupling factor:
c                       (1/(sgz*SQRT(2pi)))*EXP...
c                 (this includes multiple reflections!)
c
c ARGUMENTS:
c    PASSED:  z         height above surface                         [r]
c
c  RETURNED:  fint      value of integrand                           [r]
c
c CALLING ROUTINES:   QATR
c
c EXTERNAL ROUTINES:  RESIST, VCOUP
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

      arg=vs*RESIST(z)
      a0 = -0.5/(sgz*sgz)
      call vert(0.,sgz,a0,z,vcoup)
      fint=(1.-EXP(-arg))*VCOUP/(SGZ*2.5066283)

      return
      end
c-----------------------------------------------------------------------
      subroutine profd1(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD1
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD1 provides the base profile factor at
c              height zd for the given deposition and settling
c              velocities, and sigma-z.  Here, the settling velocity
c              and diffusion resistance are "small" so that the analytic
c              results are used.
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------


c  AP, BP, and CP are the coefficients of the 3 F(z) forms given for
c  the various forms of the Briggs representation of sigma-z (rural &
c  urban)

c  Approximate the results for a mixing lid by "clamping" the
c  calculation at the limit of a well-mixed plume in the vertical.
c       .7071=SQRT(1/2)
c       .6267=SQRT(pi/8)
c       .5157=SQRT( [SQRT(2/pi)]/3 )
c       .8932=SQRT( SQRT(2/pi) )

      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LT. 2.*zd) return

      if(hmix .GT. h) then
         za=AMIN1(sgz,.7071*hmix)
         if(BP .GT. 0.) zb=AMIN1(sgz,.6267*hmix)
         if(CP .GT. 0.) zc=AMIN1(sgz,.5157*hmix)
      else
         za=sgz
         zb=sgz
         zc=sgz
      endif

      pxzd=1./(1.+(vd-vs)*(AP*(ALOG(rt2*za/zd) -1.) + BP*(zb-rtpiby2*zd)
     1         + CP*(zc*zc-zd*zd)))

      return
      end
c-----------------------------------------------------------------------
      subroutine profd2(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD2
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD2 provides the base profile factor at
c              height zd for the given deposition and settling
c              velocities, and sigma-z.  Here, the settling velocity
c              and diffusion resistance are "small" so that the analytic
c              results are used.
c              --------------- URBAN Class A & B !!! ---------------------
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

c  Approximate the results for a mixing lid by switching to the
c  calculation for the limit of a well-mixed plume in the vertical
c  when sigmaz = .7071 H, where  .7071=SQRT(1/2).

      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LE. 2.*zd) return

c  AP is the coefficient (SQRT(2/pi) / aU)
c     ck = 2*b*SQRT(pi/2)/a = 0.0104443
      ck=0.0104443

      za=sgz
      if(hmix .GT. h) za=AMIN1(sgz,.7071*hmix)
      sgz1=za
      if(za .LT. 300.) then
         sgz1=za*(1.-za*.0006)**2
      else
         sgz1=za*(1.-300.*.0006)**2
      endif
      sgz2=sgz1
      if(sgz1 .GT. 1000.) sgz2=SQRT(1000.*sgz1)
      approx=-1.+ALOG(rt2*sgz1/zd)+ALOG(1.+ck*zd/8.)-
     &       ck*rt2*sgz2/(8.*rtpi)
      pxzd=1./(1.+(vd-vs)*AP*approx)

      return
      end

c-----------------------------------------------------------------------
      subroutine deplete(qcor)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           DEPLETE
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine DEPLETE provides the value of the integral of
c              the product of the vertical distribution function and the
c              profile correction factor over the travel of the plume
c              from the source to the receptor.  Because the integrating
c              subroutine may be used within this integration, the
c              routine QATR has been duplicated as QATR2.
c              Based on Horst (1983).
c
c MODIFIED:    To prevent potential underflow and overflow conditions
c              in calculation of qcor.
c              R. W. Brode, PES, Inc. - 11/08/94
c
c MODIFIED:    To set ndim2 = 12 instead of 16 for faster convergence.
c              This corresponds to 2,049 values in the integral, and
c              gives best overall peformance based on sensitivity tests.
c              R. W. Brode, PES, Inc. - 09/30/94
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  qcor      ratio of depleted emission rate to original  [r]
c
c CALLING ROUTINES:   DEPCOR
c
c EXTERNAL ROUTINES:  QATR2, F2INT
c-----------------------------------------------------------------------

c     Set up call to QATR2(xl,xu,eps,ndim2,fct,y,ier,num,aux2)
c     Declare parameter to fix the size of the aux2 array
      parameter(ndim2=12)
      real aux2(ndim2)
      external F2INT
      INTEGER*2 NUM
      include 'DEPVAR.INC'

c     Evaluate integral:
      eps=.05
c     Do not let integral try to evaluate sigma-z at x=0! -- start at 1m
      call QATR2(1.,xr,eps,ndim2,F2INT,value,ier,num,aux2)

crwb     comment out warning message
crwb     if(ier .EQ. 2) then
crwb         write(*,*) 'WARNING from DEPLETE -  integration failed to'
crwb         write(*,*) 'converge to fractional error of ',eps
crwb         write(iounit,*) 'WARNING from DEPLETE -  integration failed to'
crwb         write(iounit,*) 'converge to fractional error of ',eps
crwb         write(iounit,*) 'on ',kurdat,' at: ',xrec,yrec
crwb      endif

      if (vd*value .gt. 50.0) then
c        Potential underflow, limit product to 50.0
         value = 50.0/vd
      else if (vd*value .lt. -50.0) then
c        Potential overflow, limit product to 50.0
         value = -50.0/vd
      end if

      qcor=EXP(-vd*value)

      if(debug) then
         write(iounit,*) '  DEPLETE: eps, QATR2 iterations = ',eps,num
      endif

      return
      end
c-----------------------------------------------------------------------
      function f2int(x)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           F2INT
c               D. Strimaitis, SRC
c
c MODIFIED:    Call to ZTERR bypassed for AREA and OPENPIT sources,
c              which assume flat terrain.
c              R. W. Brode, PES, Inc. - 4/14/95
c
c MODIFIED:    Includes use of proper distance-dependent plume rise
c              and sigmas.  Also includes terrain adjustment for
c              simple terrain cases, and uses modified SUB. VERT in
c              place of FUNCTION VCOUP.
c              R. W. Brode, PES, Inc. - 9/30/94
c
c MODIFIED:    Includes call to new plume height adjustment subr.
c              to simulate COMPLEX I terrain treatment, and also
c              includes new settling treatment.
c
c PURPOSE:     Function is the integrand of integral over the travel
c              distance to obtain the fraction of material removed from
c              the plume. Common /DEPVAR/ is used to pass data that are
c              constant during the integration, so QATR (the integrator)
c              only needs to pass values of distance.
c
c ARGUMENTS:
c    PASSED:  x         distance from source                         [r]
c
c  RETURNED:  fint      value of integrand                           [r]
c
c CALLING ROUTINES:   QATR2
c
c EXTERNAL ROUTINES:  SZSETL, PROFD, PROFD1, PROD2, VERT,
c                     STERAD, CTERAD, ZTERR
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

c     Fetch terrain elevation (m MSL) for this location
crwb  Modified to bypass call to ZTERR for AREA and OPENPIT sources
      IF (SRCTYP .EQ. 'AREA' .OR. SRCTYP .EQ. 'OPENPIT') THEN
         terr = ep
      ELSE
         terr=ZTERR(x)
      END IF

C     Obtain Plume Height and Sigmas for This Value of X
      IF (SRCTYP .EQ. 'POINT' .AND. .NOT.LTERR) THEN
         CALL PHEFF(X,DHP,HEFLAT)
         CALL STERAD(HEFLAT,TERR,HX)
c        If x > xtd, simulate the effect of settling on a surface-based
c        plume by calculating a modified sigma-z.
         if(x .GT. xtd) then
            call SZSETL(x,sgz)
         else
c           Compute sigma-z in the usual way.
C           Determine Simple Terrain Sigmas
            CALL PDISLT(X,SGZ,XGZ,SBID)
         end if
         COR400 = 1.0
CLT      ELSE IF (SRCTYP .EQ. 'POINT' .AND. LTERR) THEN
CLT         CALL PHEFFC(X,DHPCMP,HECOMP)
CLT         CALL CTERAD(HECOMP,TERR,HX,COR400)
CLTc        If x > xtd, simulate the effect of settling on a surface-based
CLTc        plume by calculating a modified sigma-z.
CLT         if(x .GT. xtd) then
CLT            call SZSETL(x,sgz)
CLT         else
CLTc           Compute sigma-z in the usual way.
CLTC           Determine Complex Terrain Sigmas
CLT            CALL PDISC(X,SGZ,XZCMP1,SBCMP1)
CLT         end if
      ELSE IF (SRCTYP .EQ. 'VOLUME') THEN
         CALL VHEFF(TERR,HEFLAT,HX)
c        If x > xtd, simulate the effect of settling on a surface-based
c        plume by calculating a modified sigma-z.
         if(x .GT. xtd) then
            call SZSETL(x,sgz)
         else
c           Compute sigma-z in the usual way.
C           Determine Simple Terrain Sigmas
C           Determine Dispersion Parameters
            CALL VDISLT(X,SGZ,XZ)
         end if
         COR400 = 1.0
      ELSE IF (SRCTYP .EQ. 'AREA') THEN
         HX = H
c        If x > xtd, simulate the effect of settling on a surface-based
c        plume by calculating a modified sigma-z.
         if(x .GT. xtd) then
            call SZSETL(x,sgz)
         else
c           Compute sigma-z in the usual way.
C           Determine Simple Terrain Sigmas
C           Calculate dispersion coefficients, SY and SZ
            CALL ADISLT(X,SGY,SGZ,XY,XZ)
         end if
         COR400 = 1.0
      ELSE IF (SRCTYP .EQ. 'OPENPIT') THEN
         HX = H
c        If x > xtd, simulate the effect of settling on a surface-based
c        plume by calculating a modified sigma-z.
         if(x .GT. xtd) then
            call SZSETL(x,sgz)
         else
c           Compute sigma-z in the usual way.
C           Determine Simple Terrain Sigmas
C           Calculate dispersion coefficients, SY and SZ
            CALL ADISLT(X,SGY,SGZ,XY,XZ)
         end if
         COR400 = 1.0
      END IF

c  Adjust plume height for gravitational settling
      hh=AMAX1(0.,hx-vs*x*onebyu)


c  -VCOUP calculates the vertical coupling factor:
c    (1/(sgz*SQRT(2pi)))*EXP...  (this includes multiple reflections!)
c  -PROFD1 or PROFD2 calculates the profile correction factor if
c  gravitational settling is weak (analytic representations are used);
c  -PROFD calculates the profile correction factor if gravitational
c  settling is strong (numerical integration is used).
c  -Apply "cor400" to F2INT to simulate correction that is applied
c  by COMPLEX I.
      if (cor400 .eq. 0.0) then
         f2int = 0.0
      else
         a0 = -0.5/(sgz*sgz)
         call vert(hh,sgz,a0,zd,vcoup)
         f2int=cor400*onebyu*VCOUP/(SGZ*2.5066283)
      end if
      if(f2int .GT. 0.0) then
         if(igrav .EQ. 0) then
            if(urban .AND. kst .LT. 3) then
               call PROFD2(pxzd)
            else
               call PROFD1(pxzd)
            endif
         else
            call PROFD(pxzd)
         endif
         f2int=f2int*pxzd
      endif

      return
      end

c-----------------------------------------------------------------------
      subroutine qatr(xl,xu,eps,ndim,fct,y,ier,i,aux)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           QATR
c
c PURPOSE:      Integration routine adapted from the IBM SSP program
c               DQATR.  Modified for single precision.
c
c ARGUMENTS:
c    PASSED:    xl,xu   lower and upper limits of integration        [r]
c               eps     fractional error used to define convergence  [r]
c               ndim    dimension of array aux (parameter)           [p]
c               fct     external function (integrand)
c               aux     working array, passed to allow variable dim. [r]
c  RETURNED:    y       value of integral                            [r]
c               ier     status flag at termination                   [i]
c               i       number of subdivision steps                  [i]
c
c CALLING ROUTINES:     PROFD
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

c  NOTES: status flags denote the following --
c               ier=0   value of integral converged to within eps
c               ier=1   value of integral is diverging
c               ier=2   value of integral did not converge to within
c                       eps before ndim limit was reached

c  NDIM Note:  The aux(ndim) array keeps track of the average value of
c              the integrand for each of the steps in subdividing the
c              interval.  For example, when i=4 in the "do 7 i=2,ndim"
c              loop, aux(4) contains the mean value as obtained from
c              the trapezoidal rule, while aux(1 through 3) contain
c              a set of current Romberg extrapolations.  At each new
c              value of i, the interval is subdivided again, and the
c              integrand is evaluated at jj=2**(i-2) new points.
c              Therefore, at i=5, there will be jj=8 new points added
c              to the 9 points already used in the interval.  When i=17
c              there will be jj=32,768 new points added to the 32,769
c              already used.  This is the maximum number of new points
c              that are allowed as jj is an INTEGER*2 variable, with
c              a maximum value of 2**15.  Therefore, i should not exceed
c              17, and probably should be no larger than 16.  This means
c              that NDIM should be set at 16.  Larger values of NDIM
c              could be accepted if the INTEGER*2 variables were changed
c              to INTEGER*4, but for most applications, 30000 to 60000
c              points ought to be sufficient for evaluating an integral.

      EXTERNAL fct
      dimension aux(ndim)
      integer*2 i,ii,ji,j,jj
      half=0.5

c  Preparations for Romberg loop
      aux(1)=half*(fct(xl)+fct(xu))
      h=xu-xl
      y=h*aux(1)
      if(ndim .LE. 1) then
         ier=2
         return
      elseif(h .EQ. 0.) then
         ier=0
         return
      endif

      hh=h
      delt2=0.
      p=1.
      jj=1

c  Initialize flag for integer*2 limit: jj cannot exceed 32,000
c  This limit should not be reached if NDIM .LE. 16
      lstop=0

      do 7 i=2,ndim
         y=aux(1)
         delt1=delt2
         hd=hh
         hh=half*hh
         p=half*p
         x=xl+hh
         sm=0.

c  Integer*2 limit: jj cannot exceed 32,000
         if(lstop .EQ. 1) then
            write(6,1010)
1010        format(2X,'ERROR FROM QATR - VARIABLE jj EXCEEDED 32,000')
            stop
         endif
         if(jj .GT. 16000) lstop=1

         do 3 j=1,jj
            sm=sm+fct(x)
            x=x+hd
3        continue

c  A new approximation to the integral is computed by means
c  of the trapezoidal rule
         aux(i)=half*aux(i-1)+p*sm

c  Start of Rombergs extrapolation method

         q=1.
         ji=i-1
         do 4 j=1,ji
            ii=i-j
            q=q+q
            q=q+q
            aux(ii)=aux(ii+1)+(aux(ii+1)-aux(ii))/(q-1.)
4        continue

c  End of Romberg step

         delt2=ABS(y-aux(1))
         if(i .GE. 3) then
c  Modification for cases in which function = 0 over interval
            if(y .EQ. 0.) then
               ier=0
               return
            elseif(delt2/y .LE. eps) then
               ier=0
               y=h*aux(1)
               return
c           elseif(delt2 .GE. delt1)then
c              ier=1
c              y=h*y
c              return
            endif
         endif
7     jj=jj+jj
      ier=2
      y=h*aux(1)

      return
      end

c-----------------------------------------------------------------------
      subroutine qatr2(xl,xu,eps,ndim,fct,y,ier,i,aux)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           QATR2
c
c PURPOSE:      Integration routine adapted from the IBM SSP program
c               DQATR.  Modified for single precision.  This is a COPY
c               of QATR for use in double integrations.
c
c MODIFIED:     To use new convergence criteria, including a lower
c               threshold in the value of the integral (1.0E-10), and
c               to check for "delta-x" < 1.0 meters (delta-x = hh).
c               R. W. Brode, PES, Inc. - 9/30/94
c
c ARGUMENTS:
c    PASSED:    xl,xu   lower and upper limits of integration        [r]
c               eps     fractional error used to define convergence  [r]
c               ndim    dimension of array aux (parameter)           [p]
c               fct     external function (integrand)
c               aux     working array, passed to allow variable dim. [r]
c  RETURNED:    y       value of integral                            [r]
c               ier     status flag at terminatio                    [i]
c               i       number of subdivision steps                  [i]
c
c CALLING ROUTINES:     DEPLETE
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

c  NOTES: status flags denote the following --
c               ier=0   value of integral converged to within eps
c               ier=1   value of integral is diverging
c               ier=2   value of integral did not converge to within
c                       eps before ndim limit was reached

c  NDIM Note:  The aux(ndim) array keeps track of the average value of
c              the integrand for each of the steps in subdividing the
c              interval.  For example, when i=4 in the "do 7 i=2,ndim"
c              loop, aux(4) contains the mean value as obtained from
c              the trapezoidal rule, while aux(1 through 3) contain
c              a set of current Romberg extrapolations.  At each new
c              value of i, the interval is subdivided again, and the
c              integrand is evaluated at jj=2**(i-2) new points.
c              Therefore, at i=5, there will be jj=8 new points added
c              to the 9 points already used in the interval.  When i=17
c              there will be jj=32,768 new points added to the 32,769
c              already used.  This is the maximum number of new points
c              that are allowed as jj is an INTEGER*2 variable, with
c              a maximum value of 2**15.  Therefore, i should not exceed
c              17, and probably should be no larger than 16.  This means
c              that NDIM should be set at 16.  Larger values of NDIM
c              could be accepted if the INTEGER*2 variables were changed
c              to INTEGER*4, but for most applications, 30000 to 60000
c              points ought to be sufficient for evaluating an integral.

      EXTERNAL fct
      dimension aux(ndim)
      integer*2 i,ii,ji,j,jj
      half=0.5

c  Preparations for Romberg loop
      aux(1)=half*(fct(xl)+fct(xu))
      h=xu-xl
      y=h*aux(1)
      if(ndim .LE. 1) then
         ier=2
         return
      elseif(h .EQ. 0.) then
         ier=0
         return
      endif

      hh=h
      delt2=0.
      p=1.
      jj=1

c  Initialize flag for integer*2 limit: jj cannot exceed 32,000
c  This limit should not be reached if NDIM .LE. 16
      lstop=0

      do 7 i=2,ndim
         y=aux(1)
         delt1=delt2
         hd=hh
         hh=half*hh
         p=half*p
         x=xl+hh
         sm=0.

c  Integer*2 limit: jj cannot exceed 32,000
         if(lstop .EQ. 1) then
            write(6,1010)
1010        format(2X,'ERROR FROM QATR2- VARIABLE jj EXCEEDED 32,000')
            stop
         endif
         if(jj .GT. 16000) lstop=1

         do 3 j=1,jj
            sm=sm+fct(x)
            x=x+hd
3        continue

c  A new approximation to the integral is computed by means
c  of the trapezoidal rule
         aux(i)=half*aux(i-1)+p*sm

c  Start of Rombergs extrapolation method

         q=1.
         ji=i-1
         do 4 j=1,ji
            ii=i-j
            q=q+q
            q=q+q
            aux(ii)=aux(ii+1)+(aux(ii+1)-aux(ii))/(q-1.)
4        continue

c  End of Romberg step

         delt2=ABS(y-aux(1))
         if(i .GE. 3) then
c  Modification for cases in which function = 0 over interval
            if(y .EQ. 0.) then
               ier=0
               return
crwb        add lower threshold convergence test
            elseif(h*aux(1) .LT. 1.0e-10) then
               ier=0
               y=h*aux(1)
               return
            elseif(delt2/y .LE. eps) then
               ier=0
               y=h*aux(1)
               return
crwb        add lower limit on "delta-x"
            elseif(hh .LT. 1.0) then
               ier=0
               y=h*aux(1)
               return
c           elseif(delt2 .GE. delt1)then
c              ier=1
c              y=h*y
c              return
            endif
         endif
7     jj=jj+jj
      ier=2
      y=h*aux(1)

      return
      end


c-----------------------------------------------------------------------
      function zterr(x)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           ZTERR
c               D. Strimaitis, SRC
c
c PURPOSE:     Function computes the elevation (m MSL) at the position
c              "x", which is the distance downwind of the source, by
c              interpolating within field of gridded terrain elevations.
c
c ARGUMENTS:
c    PASSED:  x      distance from source to interpolation point (m) [r]
c
c  RETURNED:  zterr  value interpolated at x (m MSL)                 [r]
c
c CALLING  ROUTINES:   F2INT
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

      fract=x/xr

      if(LTGRID) then
c ---    Interpolate within gridded terrain ---

c ---    Convert the distance from source to point into (xp,yp) location
         xp=xsrc+(xrec-xsrc)*fract
         yp=ysrc+(yrec-ysrc)*fract

c ---    Set inverse of the size of a grid-cell
         di=1./sizem

c     - ll  denotes lower left corner of a grid-cell
c     - llm denotes lower left corner of grid-cell (1,1) -- this is the
c           lower left corner of the master terrain grid
c
c  Full development of the algorithm to obtain value at point xp,yp
c -- array index of lower left corner of cell that contains point
c     ixll=(xp-xllm)*di+1
c     iyll=(yp-yllm)*di+1
c -- position of lower left value
c     xll=xllm+sizem*(ixll-1)
c     yll=yllm+sizem*(iyll-1)
c -- fractional position of point within cell wrt lower left corner
c     tt=(xp-xll)*di
c     uu=(yp-yll)*di
c -- interpolated value
c     zi=(1.-tt)*(1.-uu)*zarray(ixll,iyll)
c    1     +tt*(1.-uu)*zarray(ixll+1,iyll)
c    2     +tt*uu*zarray(ixll+1,iyll+1)
c    3     +uu*(1.-tt)*zarray(ixll,iyll+1)

c ---    Compact representation:
         xpos=(xp-xllm)*di
         ixll=INT(xpos)+1
         tt=xpos-(ixll-1)
         onemt=1.-tt
         ixllp1=ixll+1
         ypos=(yp-yllm)*di
         iyll=INT(ypos)+1
         uu=ypos-(iyll-1)
         onemu=1.-uu
         iyllp1=iyll+1
         zterr=onemt*onemu*FLOAT(izarray(ixll,iyll))
     1         +tt*onemu*FLOAT(izarray(ixllp1,iyll))
     2         +tt*uu*FLOAT(izarray(ixllp1,iyllp1))
     3         +uu*onemt*FLOAT(izarray(ixll,iyllp1))

      else
c ---    Interpolate height between source and receptor

c ---    Height difference from source to receptor
         delz=er-ep

         zterr=ep+delz*fract

      endif

      return
      end

c-----------------------------------------------------------------------
      subroutine szsetl(x,sz)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SZSETL
c               D. Strimaitis, SRC
c
c PURPOSE:     SZSETL determines the value of sigma-z at a particular
c              distance after the point at which gravitational settling
c              causes the centerline of the plume to reach the ground.
c              A default minimum of 2*zd, where zd is the near-surface
c              height at which the deposition flux is evaluated, is
c              returned if sigma-z would otherwise become LE zero.
c
c ARGUMENTS:
c    PASSED:  x         distance from source   (m)                   [r]
c             /DEPVAR/  --
c             kst       stability class                              [i]
c             zd        reference height for deposition flux (m)     [r]
c             vs        settling vel.          (m/s)                 [r]
c             onebyu    1/wind speed           (s/m)                 [r]
c             urban     logical for URBAN/RURAL dispersion params    [l]
c             xtd       distance from source at which centerline     [r]
c                       reaches the ground     (m)
c             sztd      value of sigma-z @ xtd (m)                   [r]
c             szmn      minimum allowed for "settling" sigma-z       [r]
c
c
c  RETURNED:  sz        revised value of sigma-z at x  (m)           [r]
c
c CALLING ROUTINES:     DEPCOR, F2INT
c
c EXTERNAL ROUTINES:    SZFORM1, SZFORM2, SZFORM3
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

      real car(6),cau(6),cbr(6),cbu(6)

c --- Coefficients for the Briggs Rural and Urban dispersion curves
      data car/.2,.12,.08,.06,.03,.016/
      data cbr/0.,0.,.0002,.0015,.0003,.0003/
      data cau/.24,.24,.2,.14,.08,.08/
      data cbu/.001,.001,0.,.0003,.0015,.0015/

      c=rtpiby2*vs*onebyu

c --- Urban section
      if(URBAN) then
         a=cau(kst)
         b=cbu(kst)
         if(kst .EQ. 3) then
c ---       Stability Class C
            call SZFORM1(a,b,c,xtd,sztd,x,sz)
         elseif(kst .GT. 3) then
c ---       Stability Classes D,E,F
            call SZFORM2(a,b,c,xtd,sztd,szmn,x,sz)
         elseif(kst .LT. 3) then
c ---       Stability Classes A,B
            call SZFORM4(c,xtd,sgz,x,sz)
         endif

c --- Rural section
      else
         a=car(kst)
         b=cbr(kst)
         if(kst .LT. 3) then
c ---       Stability Classes A,B
            call SZFORM1(a,b,c,xtd,sztd,x,sz)
         elseif(kst .GT. 4) then
c ---       Stability Classes E,F
            call SZFORM3(a,b,c,xtd,sztd,x,sz)
         else
c ---       Stability Classes C,D
            call SZFORM2(a,b,c,xtd,sztd,szmn,x,sz)
         endif
      endif

c --- Set minimum accepted value at 2*zd
      sz=AMAX1(sz,2.*zd)

      return
      end

c-----------------------------------------------------------------------
      subroutine szform1(a,b,c,x0,sig0,x,sz)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SZFORM1
c               D. Strimaitis, SRC
c
c PURPOSE:     SZFORM1 determines the value of sigma-z at a particular
c              distance after the point at which gravitational settling
c              causes the centerline of the plume to reach the ground,
c              using the Briggs dispersion function of the form:
c
c              sz = a*x  (RURAL stabilities A,B ; URBAN stability C)
c
c
c ARGUMENTS:
c    PASSED:  a,b       coefficients in sz equation                  [r]
c             c         gravitational settling slope SQRT(pi/2)Vs/U  [r]
c             x0        distance from source at which centerline     [r]
c                       reaches the ground     (m)
c             sig0      value of sigma-z at x0 (m)                   [r]
c             x         distance from source   (m)                   [r]
c
c  RETURNED:  sz        revised value of sigma-z at x  (m)           [r]
c
c CALLING ROUTINES:   SZSETL
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      sz=sig0+(a-c)*(x-x0)
      sz=AMAX1(0.0,sz)

      return
      end

c-----------------------------------------------------------------------
      subroutine szform2(a,b,c,x0,sig0,szmn,x,sz)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SZFORM2
c               D. Strimaitis, SRC
c
c PURPOSE:     SZFORM2 determines the value of sigma-z at a particular
c              distance after the point at which gravitational settling
c              causes the centerline of the plume to reach the ground,
c              using the Briggs dispersion function of the form:
c
c              sz = a*x/SQRT(1+bx)  (RURAL stabilities C,D
c                                    URBAN stabilities D,E,F)
c
c
c ARGUMENTS:
c    PASSED:  a,b       coefficients in sz equation                  [r]
c             c         gravitational settling slope SQRT(pi/2)Vs/U  [r]
c             x0        distance from source at which centerline     [r]
c                       reaches the ground     (m)
c             sig0      value of sigma-z at x0 (m)                   [r]
c             szmn      minimum allowed for "settling" sigma-z       [r]
c             x         distance from source   (m)                   [r]
c
c  RETURNED:  sz        revised value of sigma-z at x  (m)           [r]
c
c CALLING ROUTINES:   SZSETL
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

c --- This algorithm provides an estimate of sigma-z, not the exact
c --- solution.  It uses the results for the limits sz >> a/b and
c --- sz << a/b, and patches these together at sz=2a/b.  Therefore, the
c --- equations used must differentiate between these two regimes.

      data feps/.01/, aeps/.1/, small/1.0e-10/, itermx/300/
c     -- feps    convergence criterion for fractional error
c     -- aeps    convergence criterion for absolute error
c     -- small   tolerance in detecting no-growth situation
c     -- itermx  max number of iterations allowed for inverting solution

      rt2=SQRT(2.)
      abyb=a/b
      abyk=a/c
      twoabyb=2.*abyb
      asqbybk=abyb*abyk
      rtabyk=SQRT(abyk)
      rt4kbya=2./rtabyk


c --- Does modified sigma change, or does d/dx(sigma)-c=0 ?
      if(sig0 .LE. twoabyb) then
         test=1.-2.*sig0/asqbybk
         if(ABS(test) .LT. small) then
c ---       Sigma does not change
            sz=sig0
            return
         endif
      else
         test=1.+sig0/twoabyb-rtabyk
         if(ABS(test) .LT. small) then
c ---       Sigma does not change
            sz=sig0
            return
         endif
      endif

c --- Set matching constants for the Large and Small sz forms, at x0:
      delsig=0.5*asqbybk*((1.-rt4kbya)*ALOG(ABS(1.-rt4kbya))
     &                +(1.+rt4kbya)*ALOG(1.+rt4kbya))
      if(sig0 .LT. twoabyb) then
         fac=1.+sig0/twoabyb
         sigs=abyb*rtabyk*ALOG(ABS((fac-rtabyk)/(fac+rtabyk)))
         sigl=sigs+delsig
      elseif(sig0 .GE. twoabyb) then
         sigl=0.5*asqbybk*ALOG(ABS(1.-2.*sig0/asqbybk))
         sigs=sigl-delsig
      endif

c     write(*,*) 'sigl, sigs   =  ',sigl,sigs

c --- Iterate to find sz for current value of x
      sz=szmn
      if(szmn .GT. 0.) sz=sig0
      szlast=sz
      icount=0

1     if(sz .GE. twoabyb) then
c ---    (Large sz form)
         if(sz .GE. asqbybk) then
c ---       (LOG form of iteration)
            sz=sig0+sigl-c*(x-x0)
     &         -0.5*asqbybk*ALOG(ABS(1.-2.*sz/asqbybk))
         else
c ---       (EXP form of iteration)
            expterm=EXP(-2.*(sz-sig0-sigl+c*(x-x0))/asqbybk)
            test=2.*sz/asqbybk
            if(test .LT. 1.) then
               sz=0.5*asqbybk*(1.-expterm)
            else
               sz=0.5*asqbybk*(1.+expterm)
            endif
         endif
      else
c ---    (Small sz form)
         fac=1.+sz/twoabyb
         if(fac .GE. rt2*rtabyk) then
c ---       (LOG form of iteration)
            sz=sig0+sigs-c*(x-x0)
     &         -abyb*rtabyk*ALOG(ABS(fac-rtabyk)/(fac+rtabyk))
         else
c ---       (EXP form of iteration)
            expterm=EXP(-(sz-sig0-sigs+c*(x-x0))/(abyb*rtabyk))
            test=fac-rtabyk
            if(test .LE. 1.) then
               sz=twoabyb*(rtabyk*(1.-expterm)/(1.+expterm)-1.)
            else
               sz=twoabyb*(rtabyk*(1.+expterm)/(1.-expterm)-1.)
            endif
         endif
      endif

      icount=icount+1
      if(icount .LE. itermx) then
         err=ABS(sz-szlast)
         if(szlast .NE. 0.0) then
            ferr=ABS(err/szlast)
         else
            ferr=ABS(err/sz)
         endif
         if(ferr .GT. feps .AND. err .GT. aeps) then
            szlast=sz
            goto 1
         endif
      else
crwb         Comment out warning message
crwb         write(*,*)
crwb         write(*,*) 'SZFORM2 FAILED -- FATAL'
crwb         write(*,*) 'iterations, sz   =  ',icount,sz
crwb         write(*,*) 'a, b, c          =  ',a,b,c
crwb         write(*,*) 'xtd, sztd        =  ',x0,sig0
      endif

c     write(*,*) 'iterations, sz     =  ',icount,sz

      sz=AMAX1(0.0,sz)

      return
      end

c-----------------------------------------------------------------------
      subroutine szform3(a,b,c,x0,sig0,x,sz)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SZFORM3
c               D. Strimaitis, SRC
c
c PURPOSE:     SZFORM3 determines the value of sigma-z at a particular
c              distance after the point at which gravitational settling
c              causes the centerline of the plume to reach the ground,
c              using the Briggs dispersion function of the form:
c
c              sz = a*x/(1+bx)  (RURAL stabilities E,F)
c
c
c ARGUMENTS:
c    PASSED:  a,b       coefficients in sz equation                  [r]
c             c         gravitational settling slope SQRT(pi/2)Vs/U  [r]
c             x0        distance from source at which centerline     [r]
c                       reaches the ground     (m)
c             sig0      value of sigma-z at x0 (m)                   [r]
c             x         distance from source   (m)                   [r]
c
c  RETURNED:  sz        revised value of sigma-z at x  (m)           [r]
c
c CALLING ROUTINES:   SZSETL
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      rtak=SQRT(a*c)
      ambs0=a-b*sig0
      gamx=((ambs0-rtak)/(ambs0+rtak))*EXP(-2.*b*rtak*(x-x0)/a)
      sz=(a-rtak*(1.+gamx)/(1.-gamx))/b
      sz=AMAX1(0.0,sz)

      return
      end

c-----------------------------------------------------------------------
      subroutine szform4(c,x0,sgz,x,sz)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SZFORM4
c               D. Strimaitis, SRC
c
c PURPOSE:     SZFORM4 determines the value of sigma-z at a particular
c              distance after the point at which gravitational settling
c              causes the centerline of the plume to reach the ground,
c              using the Briggs dispersion function of the form:
c
c              sz = a*x*SQRT(1+bx)  (URBAN stabilities A,B)
c
c    NOTE:     This is an interim treatment that merely subtracts the
c              "fall" due to gravitational settling from the sigma-z
c              at the receptor computed in the absence of settling.
c
c ARGUMENTS:
c    PASSED:  c         gravitational settling slope SQRT(pi/2)Vs/U  [r]
c             x0        distance from source at which centerline     [r]
c                       reaches the ground     (m)
c             sgz       current value of sigma-z at x (m)            [r]
c             x         distance from source   (m)                   [r]
c
c  RETURNED:  sz        revised value of sigma-z at x  (m)           [r]
c
c CALLING ROUTINES:   SZSETL
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------
      sz=sgz-(x-x0)*c
      sz=AMAX1(0.0,sz)

      return
      end
