      SUBROUTINE INPSUM
C***********************************************************************
C                 INPSUM Module of ISC2 Long Term Model - ISCLT2
C
C        PURPOSE: Print Out The Input Data Summary
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Arrays of Model Results
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   MAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'INPSUM'

C     Print Out The Model Options
      CALL PRTOPT

C     Print Out The Input Source Data
      CALL PRTSRC

C     Print Out The Input Receptor Coordinates.
      CALL PRTREC

C     Check For Receptors Too Close To Sources (< 1m or < 3Lb)
      CALL CHKREC

C     Print Out The Input Met Data Summary
      CALL PRTMET

      RETURN
      END

      SUBROUTINE PRTOPT
C***********************************************************************
C                 PRTOPT Module of ISC Long Term Model
C
C        PURPOSE: Print Out The Model Options and Keyword Summary
C
C        PROGRAMMER: Jeff Wang
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        MODIFIED by Roger Brode, PES, Inc., To Remove Summary of
C                    Keywords Table - 11/08/94
C
C        MODIFIED:   To add pathway 'TG' to process input file of Gridded
C                    Terrain data.
C                    Roger Brode - 11/08/94
C
C        MODIFIED by Roger Brode, PES, Inc., To Exclude AVEMOLEN and
C                    AVEUSTAR options - 8/1/94
C
C        MODIFIED:  To Include TOXXFILE Option - 9/29/92
C
C        INPUTS:  Model Options and Keyword Summarys
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   INPSUM
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'PRTOPT'

C     Summarize The Model Options
      CALL HEADER
      WRITE(IOUNIT,9041)
      IF (CONC) THEN
         WRITE(IOUNIT,*) '**Model Is Setup For Calculation of ',
     &        'Average CONCentration Values.'
      ELSE IF (DEPOS) THEN
         WRITE(IOUNIT,*) '**Model Is Setup For Calculation of ',
     &        'Total DEPOSition Values.'
      END IF
      IF (LTGRID) THEN
         WRITE(IOUNIT,*) '**Model Uses GRIDDED TERRAIN Data for ',
     &                   'Depletion Calculations'
      ELSE
         WRITE(IOUNIT,*) '**Model Does NOT Use GRIDDED TERRAIN Data ',
     &                   'for Depletion Calculations'
      END IF

      WRITE(IOUNIT,9099)
      IF (DDPLETE) THEN
         WRITE(IOUNIT,*) '**Model Uses plume DEPLETION.'
      ELSE
         WRITE(IOUNIT,*) '**Model Uses NO plume DEPLETION.'
      END IF

      WRITE(IOUNIT,9099)
      IF (RURAL) THEN
         WRITE(IOUNIT,*) '**Model Uses RURAL Dispersion.'
      ELSE IF (URBAN) THEN
         WRITE(IOUNIT,*) '**Model Uses URBAN Dispersion.'
      END IF

      WRITE(IOUNIT,9099)
      IF (DFAULT) THEN
         WRITE(IOUNIT,*) '**Model Uses Regulatory DEFAULT Options:'
         WRITE(IOUNIT,*) '           1. Final Plume Rise.'
         WRITE(IOUNIT,*) '           2. Stack-tip Downwash.'
         WRITE(IOUNIT,*) '           3. Buoyancy-induced ',
     &           'Dispersion.'
         WRITE(IOUNIT,*) '           4. Default Wind Profile Exponents.'
         WRITE(IOUNIT,*) '           5. Default Vertical Potential',
     &           ' Temperature Gradients.'
         WRITE(IOUNIT,*) '           6. "Upper Bound" Values ',
     &           'For Supersquat Buildings.'
         IF (URBAN .AND. POLLUT .EQ. 'SO2') THEN
            WRITE(IOUNIT,*) '           7. Half-life of 4 hrs for',
     &              ' Urban SO2.'
         ELSE IF (URBAN .AND. POLLUT .NE. 'SO2') THEN
            WRITE(IOUNIT,*) '           7. No Exponential Decay for',
     &              ' URBAN/Non-SO2'
         ELSE
            WRITE(IOUNIT,*) '           7. No Exponential Decay for',
     &              ' RURAL Mode'
         END IF
      ELSE
         WRITE(IOUNIT,*) '**Model Uses User-Specified Options:'
         IF (GRDRIS) THEN
            WRITE(IOUNIT,*) '           1. Gradual Plume Rise.'
         ELSE
            WRITE(IOUNIT,*) '           1. Final Plume Rise.'
         END IF
         IF (NOSTD) THEN
            WRITE(IOUNIT,*) '           2. Not Use Stack-tip ',
     &           'Downwash.'
         ELSE
            WRITE(IOUNIT,*) '           2. Stack-tip Downwash.'
         END IF
         IF (NOBID) THEN
            WRITE(IOUNIT,*) '           3. Not Use Buoyancy-induced ',
     &           'Dispersion.'
         ELSE
            WRITE(IOUNIT,*) '           3. Buoyancy-induced ',
     &           'Dispersion.'
         END IF
         IF (USERP) THEN
            WRITE(IOUNIT,*) '           4. User-Specified Wind Profile',
     &           ' Exponents.'
         ELSE
            WRITE(IOUNIT,*) '           4. Default Wind Profile',
     &           ' Exponents.'
         END IF
         IF (USERDT) THEN
            WRITE(IOUNIT,*) '           5. User-Specified Vertical ',
     &           'Potential Temperature Gradients.'
         ELSE
            WRITE(IOUNIT,*) '           5. Default Vertical Potential',
     &           ' Temperature Gradients.'
         END IF
      END IF

      WRITE(IOUNIT,9099)
      IF (FLAT) THEN
         WRITE(IOUNIT,*) '**Model Assumes Receptors on FLAT Terrain.'
      ELSE IF (ELEV) THEN
         WRITE(IOUNIT,*) '**Model Accepts Receptors on ELEV Terrain.'
      END IF

      WRITE(IOUNIT,9099)
      IF (FLGPOL) THEN
         WRITE(IOUNIT,*) '**Model Accepts FLAGPOLE Receptor Heights.'
      ELSE
         WRITE(IOUNIT,*) '**Model Assumes No FLAGPOLE Receptor Heights.'
      END IF

C     Model Averaging Periods
      WRITE(IOUNIT,9099)
      IF (PERIOD) THEN
         IF (NUMAVE .GT. 0) THEN
            WRITE(IOUNIT,9042) NUMAVE,(KAVE(I),I=1,NAVE)
            WRITE(IOUNIT,9043)
         ELSE
            WRITE(IOUNIT,9045)
         END IF
      ELSE
         WRITE(IOUNIT,9042) NUMAVE,(KAVE(I),I=1,NAVE)
      END IF

C     Meteorological STAR Summaries in Data File
      WRITE(IOUNIT,9099)
      IF (IMSTAT(6) .EQ. 0) THEN
C        No ME STARDATA Keyword, Assume STAR Summaries Match AVERTIME Card
         WRITE(IOUNIT,9046) NUMAVE
      ELSE
         WRITE(IOUNIT,9047) NUMSTR,(KSTAR(I),I=1,NAVE)
      END IF

C     Write Out Numbers of Sources, Groups, and Receptors for This Run
      WRITE(IOUNIT,9099)
      WRITE(IOUNIT,9044) NUMSRC, NUMGRP, NUMREC

C     Write Out Pollutant Type
      WRITE(IOUNIT,9099)
      WRITE(IOUNIT,9048) POLLUT

C     Model Run OR Not Options
      WRITE(IOUNIT,9099)
      IF (RUN) THEN
         WRITE(IOUNIT,*) '**Model Set To Continue RUNning After the ',
     &         'Setup Testing.'
      ELSE
         WRITE(IOUNIT,*) '**Model Will NOT Run After the ',
     &         'Setup Testing.'
      END IF

C     Model Output Options Setting Summary
      WRITE(IOUNIT,9099)
      WRITE(IOUNIT,9070)
      IF (IOSTAT(2) .GT. 0) THEN
C        RECTABLE Keyword Used
         WRITE(IOUNIT,9072)
      END IF
      IF (IOSTAT(3) .GT. 0) THEN
C        MAXTABLE Keyword Used
         WRITE(IOUNIT,9073)
      END IF
      IF (IOSTAT(4) .GT. 0) THEN
C        PLOTFILE Keyword Used
         WRITE(IOUNIT,9074)
      END IF
      IF (IOSTAT(5) .GT. 0) THEN
C        TOXXFILE Keyword Used
         WRITE(IOUNIT,9075)
      END IF

C     Model Misc. Information
      WRITE(IOUNIT,9099)
      WRITE(IOUNIT,9050) ZREF, DECOEF, ROTANG
      WRITE(IOUNIT,9055) EMILBL, EMIFAC, OUTLBL

C     Model I/O Setting Summary
      WRITE(IOUNIT,9099)
      IF (INPFIL .NE. ' ' .OR. OUTFIL .NE. ' ') THEN
         WRITE(IOUNIT,9080) INPFIL, OUTFIL
      END IF
      IF (ERRLST) WRITE(IOUNIT,*) '**Error Message File: ',MSGFIL

 9041 FORMAT(44X,'***     MODEL SETUP OPTIONS SUMMARY       ***'/
     &       63(' -')/)
 9042 FORMAT(1X,'**Model Calculates  ',I2,' STAR Average(s) for the',
     &       ' Following Months: ',12(I2,1X),
     &   /48X,'Seasons/Quarters: ',4(I2,1X),
     &         /54X,'and Annual: ',I2)
 9043 FORMAT(1X,'**Model Also Calculates PERIOD Averages.')
 9045 FORMAT(1X,'**Model Calculates PERIOD Averages Only.')
 9044 FORMAT(1X,'**This Run Includes: ',I4,' Source(s);   ',I4,
     &       ' Source Group(s); and  ',I5,' Receptor(s)')
 9046 FORMAT(1X,'**Model Assumes ',I2,' STAR Summaries In Data File ',
     &       'for the Averaging Periods Identified Above')
 9047 FORMAT(1X,'**Data File Includes ',I2,' STAR Summaries for the',
     &       ' Following Months: ',12(I2,1X),
     &   /48X,'Seasons/Quarters: ',4(I2,1X),
     &         /54X,'and Annual: ',I2)
 9048 FORMAT(1X,'**The Model Assumes A Pollutant Type of:  ',A8)
 9050 FORMAT(1X,'**Misc. Inputs:  Anem. Hgt. (m) = ',F8.2,
     &       ' ;    Decay Coef. = ',G12.4,' ;    Rot. Angle = ',F7.1)
 9055 FORMAT(18X,'Emission Units = ',A40,' ;  Emission Rate Unit ',
     &       'Factor = ',G13.5,
     &      /18X,'Output Units   = ',A40)
 9070 FORMAT(1X,'**Output Options Selected:')
 9072 FORMAT(10X,'Model Outputs Tables of Long Term Values by',
     &       ' Receptor (RECTABLE Keyword)')
 9073 FORMAT(10X,'Model Outputs Tables of Maximum Long Term',
     &       ' Values (MAXTABLE Keyword)')
 9074 FORMAT(10X,'Model Outputs External File(s) of Long Term Values',
     &       ' for Plotting (PLOTFILE Keyword)')
 9075 FORMAT(10X,'Model Outputs External File(s) of Values for Input',
     &       ' to TOXX Model (TOXXFILE Keyword)')
 9080 FORMAT(1X,'**Input Runstream File: ',A40,';  **Output Print ',
     &       'File: ',A40)
 9099 FORMAT(1X,' ')

      RETURN
      END

      SUBROUTINE PRTSRC
C***********************************************************************
C                 PRTSRC Module of ISC Model
C
C        PURPOSE: Print Out The Input Source Data Summary
C
C        PROGRAMMER: JEFF WANG, ROGER BRODE
C
C        MODIFIED:   To summarize additional inputs for new area source
C                    algorithm - 7/7/93
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C*       MODIFIED BY PES (for OPENPIT Source) - 7/22/94
C
C        INPUTS:  Model Options and Keyword Summarys
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   INPSUM
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER BLDING*3, IQUN*12
      CHARACTER ATHRUF(6)*1, SEACHR(4)*6

C     Variable Initializations
      DATA ATHRUF / 'A','B','C','D','E','F' /
      DATA SEACHR /'WINTER','SPRING','SUMMER',' FALL '/
      MODNAM = 'PRTSRC'

      IF (QUARTR) THEN
         SEACHR(1) = 'QUART1'
         SEACHR(2) = 'QUART2'
         SEACHR(3) = 'QUART3'
         SEACHR(4) = 'QUART4'
      END IF

      IF (ISSTAT(8) .EQ. 0) THEN
C        Write Default Emission Rate Units
         IQUN = ' (GRAMS/SEC)'
      ELSE
         IQUN = '(USER UNITS)'
      END IF

C     Write Out The Point Source Data, If Any
      INDC = 0
      DO 600 I = 1, NUMSRC
         IF (SRCTYP(I) .EQ. 'POINT') THEN
            INDC = INDC + 1
            BLDING = 'NO'
            DO 500 J = 1, NSEC
               IF (ADSBH(J,I).NE.0.0 .AND. ADSBW(J,I).NE.0.0) THEN
                  BLDING = 'YES'
               END IF
 500        CONTINUE
            IF (MOD(INDC-1,40) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9046) IQUN
            END IF
            WRITE(IOUNIT,9047) SRCID(I),INPD(I),AQS(I),
     &              AXS(I),AYS(I),AZS(I),AHS(I),ATS(I),AVS(I),ADS(I),
     &              BLDING,QFLAG(I)
         END IF
 600  CONTINUE

C     Write Out The Volume Source Data, If Any
      INDC = 0
      DO 610 I = 1, NUMSRC
         IF (SRCTYP(I) .EQ. 'VOLUME') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,40) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9074) IQUN
            END IF
            WRITE(IOUNIT,9075) SRCID(I),INPD(I),AQS(I),
     &              AXS(I),AYS(I),AZS(I),AHS(I),ASYINI(I),ASZINI(I),
     &              QFLAG(I)
         END IF
 610  CONTINUE

C     Write Out The Area Source Data, If Any
      INDC = 0
      DO 620 I = 1, NUMSRC
         IF (SRCTYP(I) .EQ. 'AREA') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,40) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9076) IQUN
            END IF
            WRITE(IOUNIT,9077) SRCID(I),INPD(I),AQS(I),
     &              AXS(I),AYS(I),AZS(I),AHS(I),AXINIT(I),AYINIT(I),
     &              AANGLE(I), ASZINI(I), QFLAG(I)
C*----     
         END IF

 620  CONTINUE

C*    Write Out The OpenPit Source Data, If Any
      INDC = 0
      DO 625 I = 1, NUMSRC
         IF (SRCTYP(I) .EQ. 'OPENPIT') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,40) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9078) IQUN
            END IF
            WRITE(IOUNIT,9079) SRCID(I),INPD(I),AQS(I),
     &              AXS(I),AYS(I),AZS(I),AHS(I),AXINIT(I),AYINIT(I),
     &              AANGLE(I), AVOLUM(I), QFLAG(I)
         END IF
 625  CONTINUE

C     Print The Source Group IDs with Source IDs
      ICNT = 12
      DO 200 J = 1, NUMGRP
         INGRP = 0
         DO 210 K = 1, NUMSRC
            IF (IGROUP(K,J) .EQ. 1) THEN
               INGRP = INGRP + 1
               WORKID(INGRP) = SRCID(K)
            END IF
  210    CONTINUE
C        Determine Number of Lines @ 12/Line
         NL = 1 + INT((INGRP-1)/12)
         ICNT = ICNT + 2*NL
         IF (J .EQ. 1 .OR. ICNT .GT. 55) THEN
            CALL HEADER
            WRITE(IOUNIT,9058)
            IF (J .NE. 1) ICNT = 12 + 2*NL
         END IF
         DO 202 K = 1, NL
            IF (K .EQ. 1 .AND. K .EQ. NL) THEN
               WRITE(IOUNIT,9068) GRPID(J), (WORKID(I),I=1,INGRP)
            ELSE IF (K .EQ. 1 .AND. K .NE. NL) THEN
               WRITE(IOUNIT,9068) GRPID(J), (WORKID(I),I=1,12*K)
            ELSE IF (K .EQ. NL) THEN
               WRITE(IOUNIT,9067) (WORKID(I),I=1+12*(K-1),INGRP)
            ELSE
               WRITE(IOUNIT,9067) (WORKID(I),I=1+12*(K-1),12*K)
            END IF
  202    CONTINUE
  200 CONTINUE

      INDC = 0
C     Print Out Particle Category Information.
      DO 720 I = 1, NUMSRC
         NPD = INPD(I)
         IF (NPD .NE. 0) THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,3) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9049)
            END IF
            WRITE(IOUNIT,9050) SRCID(I), SRCTYP(I)
            WRITE(IOUNIT,9051) (APHI(J,I),J=1,NPD)
            WRITE(IOUNIT,9052) (APDIAM(J,I),J=1,NPD)
            WRITE(IOUNIT,9053) (APDENS(J,I),J=1,NPD)
         END IF
 720  CONTINUE

      INDC = 0
C     Write Out Direction Specific Bldg. Dimensions, If Present
      DO 630 I = 1, NUMSRC
         BLDING = 'NO'
         DO 650 J = 1, NSEC
            IF (ADSBH(J,I).NE.0.0 .AND. ADSBW(J,I).NE.0.0) THEN
               BLDING = 'YES'
            END IF
 650     CONTINUE
         IF (BLDING .EQ. 'YES') THEN
            INDC = INDC + 1
C           Print Out Direction Specific Bldg. Dimensions
            IF (MOD(INDC-1,4) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9064)
            END IF
            WRITE(IOUNIT,9062) SRCID(I),
     &           (J,ABS(ADSBH(J,I)),ADSBW(J,I),IDSWAK(J,I), J=1,NSEC)
         END IF
 630  CONTINUE

C     Print Source Emission Rate Scalars.
      INDC = 0
      DO 1050 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'SEASON' .OR. QFLAG(I) .EQ. 'QUARTR') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,6) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9002)
               WRITE(IOUNIT,9004) (SEACHR(I1),I1=1,4)
            END IF
            WRITE(IOUNIT,9005) SRCID(I),SRCTYP(I)
            WRITE(IOUNIT,9006) (QFACT(I1,I),I1=1,4)
         END IF
 1050 CONTINUE

      INDC = 0
      DO 1055 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'MONTH') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,6) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9007)
               WRITE(IOUNIT,9008)
               WRITE(IOUNIT,9013)
            END IF
            WRITE(IOUNIT,9009) SRCID(I),SRCTYP(I)
            WRITE(IOUNIT,9010) (QFACT(I1,I),I1=1,12)
         END IF
 1055 CONTINUE

      INDC = 0
      DO 1060 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'SSTAB') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,4) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9021)
               WRITE(IOUNIT,9013)
            END IF
            WRITE(IOUNIT,9009) SRCID(I),SRCTYP(I)
            WRITE(IOUNIT,9022) (J, J=1,6)
            DO 1058 K = 1, 4
               I2 = 1 + (K-1)*6
               WRITE(IOUNIT,9014) SEACHR(K),(QFACT(I1,I),I1=I2,I2+5)
 1058       CONTINUE
         END IF
 1060 CONTINUE

      INDC = 0
      DO 1070 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'SSPEED') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,4) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9011)
               WRITE(IOUNIT,9013)
            END IF
            WRITE(IOUNIT,9009) SRCID(I),SRCTYP(I)
            WRITE(IOUNIT,9012) (J, J=1,6)
            DO 1068 K = 1, 4
               I2 = 1 + (K-1)*6
               WRITE(IOUNIT,9014) SEACHR(K),(QFACT(I1,I),I1=I2,I2+5)
 1068       CONTINUE
         END IF
 1070 CONTINUE

      INDC = 0
      DO 1080 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'STAR') THEN
            INDC = INDC + 1
            IF (MOD(INDC-1,3) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9015)
               WRITE(IOUNIT,9013)
            END IF
            WRITE(IOUNIT,9009) SRCID(I),SRCTYP(I)
            WRITE(IOUNIT,9025) (J, J=1,6)
            DO 760  I1 = 1,6
               IFR = (I1-1)*6 + 1
               ITO = IFR + 5
               WRITE(IOUNIT,9024) ATHRUF(I1),
     &               (QFACT(I2,I),I2=IFR,ITO)
 760        CONTINUE
         END IF
 1080 CONTINUE

      INDC = 0
      DO 1090 I = 1, NUMSRC
         IF (QFLAG(I) .EQ. 'SSTAR') THEN
            INDC = INDC + 1
C            IF (MOD(INDC-1,1) .EQ. 0) THEN
               CALL HEADER
               WRITE(IOUNIT,9018)
               WRITE(IOUNIT,9013)
C            END IF
            WRITE(IOUNIT,9009) SRCID(I),SRCTYP(I)
            DO 790  I1 = 1, 4
               IFR = (I1-1)*36
               WRITE(IOUNIT,9023) SEACHR(I1), (J, J=1,6)
               DO 780 K = 1, 6
                  WRITE(IOUNIT,9024) ATHRUF(K),
     &                               (QFACT(IFR+(K-1)*6+I2,I),I2=1,6)
 780           CONTINUE
 790        CONTINUE
         END IF
 1090 CONTINUE

 820  CONTINUE

 9002 FORMAT(37X,'* SOURCE EMISSION RATE SCALARS WHICH VARY SEASONALLY',
     &       ' *'//)
 9003 FORMAT(56X,'* FOR ALL SOURCES *'//)
 9004 FORMAT(40X,4(A6,9X)/20X,40('- ')/)
 9005 FORMAT(/10X,' SOURCE ID = ',A8,' ;  SOURCE TYPE = ',A8,' :')
 9006 FORMAT(38X,4(E10.5,5X))
 9007 FORMAT(41X,'* SOURCE EMISSION RATE SCALARS WHICH VARY MONTHLY *',
     &       //)
 9008 FORMAT(7X,'JANUARY  FEBRUARY   MARCH     APRIL      MAY       ',
     &  'JUNE      JULY     AUGUST   SEPTEMBER  OCTOBER  NOVEMBER  ',
     &  'DECEMBER'/)
 9009 FORMAT(/' SOURCE ID = ',A8,' ;  SOURCE TYPE = ',A8,' :')
 9010 FORMAT(5X,12E10.4)
 9011 FORMAT(22X,'* SOURCE EMISSION RATE SCALARS WHICH VARY BY SEASON',
     &       ' AND WIND SPEED CATEGORY (SSPEED) *'//)
 9012 FORMAT(/29X,6('    WIND SPEED')/29X,6('    CATEGORY',I2))
 9021 FORMAT(22X,'* SOURCE EMISSION RATE SCALARS WHICH VARY BY SEASON',
     &       ' AND STABILITY CATEGORY (SSTAB) *'//)
 9022 FORMAT(/29X,6('     STABILITY')/29X,6('    CATEGORY',I2))
 9013 FORMAT(1X,65('- ')/)
 9014 FORMAT(21X,A6,2X,6(2X,E12.5))
 9015 FORMAT(22X,'* SOURCE EMISSION RATE SCALARS WHICH VARY WITH',
     &       ' STABILITY AND WIND SPEED (STAR) *'//)
 9017 FORMAT(19X,A1,5X,6(5X,E10.5))
 9018 FORMAT(22X,'* SOURCE EMISSION RATE SCALARS WHICH VARY',
     &       ' BY SEASON, STABILITY AND WIND SPEED (SSTAR) *'//)
 9023 FORMAT(/56X,'SEASON = ',A6/26X,6('   WIND SPEED')/26X,
     &       6('   CATEGORY',I2))
 9024 FORMAT(6X,'STABILITY CATEGORY ',A1,6(1X,E12.5))
 9025 FORMAT(/26X,6('   WIND SPEED')/26X,6('   CATEGORY',I2))
 9046 FORMAT(//50X,'*** POINT SOURCE DATA ***'///14X,
     & 'NUMBER EMISSION RATE',20X,'BASE     STACK   STACK',4X,
     & 'STACK     STACK    BUILDING EMISSION RATE',/4X,
     & 'SOURCE',5X,'PART. ',A12,5X,'X',8X,'Y',6X,'ELEV.    ',
     & 'HEIGHT  TEMP.   EXIT VEL. DIAMETER',3X,'EXISTS   SCALAR VARY',
     & /4X,'  ID       CATS.              ',
     & 1X,2('(METERS) (METERS) '),'(DEG.K) ',' (M/SEC) ',1X,'(METERS)',
     & 16X,'BY'/61(' -')/)
 9047 FORMAT(3X,A8,2X,I5,3X,E11.5,2F10.1,F8.1,4F9.2,
     &       6X,A3,4X,A6)
 9049 FORMAT(50X,'*** SOURCE PARTICULATE DATA ***'//)
 9050 FORMAT(//10X,'*** SOURCE ID = ',A8,'; SOURCE TYPE = ',A8,' ***')
 9051 FORMAT(/10X,'MASS FRACTION ='/2(10X,10(F8.5,',')/))
 9052 FORMAT(/10X,'PARTICLE DIAMETER(MICRONS) ='/2(10X,10(F9.5,',')
     &       /))
 9053 FORMAT(/10X,'PARTICLE DENSITY (G/CM**3) ='/2(10X,10(F8.5,','
     &       )/))
 9058 FORMAT(//43X,'*** SOURCE IDs DEFINING SOURCE GROUPS ***'//
     &       1X,'GROUP ID',49X,'SOURCE IDs'/)
 9068 FORMAT(//2X,A8,1X,12(1X,A8,','))
 9067 FORMAT(/11X,12(1X,A8,','))
 9062 FORMAT(/' SOURCE ID: ',A8,
     &       /,4('    IFV    BH      BW   WAK'),/,
     &       4(4(4X,I3,F7.1,',',F7.1,',',I3,1X)/)/)
 9064 FORMAT(42X,'*** DIRECTION SPECIFIC BUILDING DIMENSIONS ***'/)
 9074 FORMAT(//50X,'*** VOLUME SOURCE DATA ***'//14X,
     & 'NUMBER EMISSION RATE',20X,'BASE    RELEASE    INIT.',4X,
     & 'INIT.   EMISSION RATE',/4X,
     & 'SOURCE',5X,'PART. ',A12,5X,'X',8X,'Y',6X,'ELEV.   ',
     & 'HEIGHT      SY       SZ      SCALAR VARY',
     & /4X,'  ID       CATS.              ',
     & 1X,3('(METERS) (METERS) '),5X,'  BY'/61(' -')/)
 9075 FORMAT(3X,A8,2X,I5,3X,E11.5,2F10.1,F8.1,F9.2,1X,F8.2,1X,
     &       F8.2,3X,A6)
 9076 FORMAT(//50X,'*** AREA SOURCE DATA ***'//14X,
     & 'NUMBER EMISSION RATE',2X,'COORD (SW CORNER)',2X,
     & 'BASE     RELEASE  X-DIM     Y-DIM    ORIENT.',4X,
     & 'INIT.',2X,
     & 'EMISSION RATE',
     & /4X,'SOURCE',5X,'PART. ',A11,7X,'X',8X,'Y',6X,'ELEV.    ',
     & 'HEIGHT  OF AREA   OF AREA   OF AREA     SZ     SCALAR VARY',
     & /4X,'  ID       CATS.   /METER**2)  ',
     & 1X,2('(METERS) (METERS) '),2('(METERS)',2X),' (DEG.)  (METERS)',
     & 6X,'BY'/63(' -')/)
 9077 FORMAT(3X,A8,2X,I5,3X,E11.5,2F10.1,F8.1,F9.2,3(1X,F9.2),1X,F8.2,
     &       5X,A6)
 9078 FORMAT(//50X,'*** OPENPIT SOURCE DATA ***'//14X,
     & 'NUMBER EMISSION RATE',2X,'COORD (SW CORNER)',2X,
     & 'BASE     RELEASE  X-DIM     Y-DIM    ORIENT.',4X,
     & 'VOLUME',3X,'EMISSION RATE',
     & /4X,'SOURCE',5X,'PART. ',A11,7X,'X',8X,'Y',6X,'ELEV.    ',
     & 'HEIGHT  OF PIT    OF PIT    OF PIT     OF PIT    SCALAR VARY',
     & /4X,'  ID       CATS.   /METER**2)  ',
     & 1X,2('(METERS) (METERS) '),2('(METERS)',2X),' (DEG.) ',3X,
     & '(M**3)        BY'
     & /63(' -')/)
 9079 FORMAT(3X,A8,2X,I5,3X,E11.5,2F10.1,F8.1,F9.2,3(1X,F9.2),
     &       3X,E10.5,3X,A6)

      RETURN
      END


      SUBROUTINE PRTREC
C***********************************************************************
C                 PRTREC Module of ISC2 Long Term Model - ISCLT2
C
C        PURPOSE: Print Out The Receptor Network Values
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To Adjust Format Statement 9082 for Boundary
C                    Receptors - 9/29/92
C
C        INPUTS:  Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Arrays of Model Results
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   INPSUM
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER BUF132*132

C     Variable Initializations
      MODNAM = 'PRTREC'

      DO 100 I = 1, INNET
         CALL HEADER
         WRITE(IOUNIT,9034)
         WRITE(IOUNIT,9037) NTID(I), NTTYP(I)
         IF (NTTYP(I) .EQ. 'GRIDCART') THEN
            WRITE(IOUNIT,9038)
         ELSE
            WRITE(IOUNIT,9036) XORIG(I), YORIG(I)
            WRITE(IOUNIT,9039)
         END IF
         WRITE(IOUNIT,9040) (XCOORD(J,I),J=1,NUMXPT(I))
         IF (NTTYP(I) .EQ. 'GRIDCART') THEN
            WRITE(IOUNIT,9041)
         ELSE
            WRITE(IOUNIT,9042)
         END IF
         WRITE(IOUNIT,9040) (YCOORD(J,I),J=1,NUMYPT(I))
         IF (ELEV) THEN
C           Print Terrain Heights for Network
C           Set Number of Columns Per Page, NCPP
            NCPP = 9
C           Set Number of Rows Per Page, NRPP
            NRPP = 40
C           Begin LOOP Through Networks
C           Calculate Number of Pages Per X-Group, NPPX, & Per Y-Group, NPPY
            NPPX = 1 + INT((NUMXPT(I)-1)/NCPP)
            NPPY = 1 + INT((NUMYPT(I)-1)/NRPP)
            DO 40 NX = 1, NPPX
               DO 30 NY = 1, NPPY
                  CALL HEADER
                  WRITE(IOUNIT,9037) NTID(I), NTTYP(I)
                  WRITE(IOUNIT,9011)
                  IF (NX .EQ. NPPX) THEN
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     END IF
                  ELSE
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     END IF
                  END IF
                  WRITE(IOUNIT,9010)
                  IF (NY .EQ. NPPY) THEN
                     DO 20 K = 1+NRPP*(NY-1), NUMYPT(I)
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZELEV(INDZ+J-1),J=1+NCPP*(NX-1),NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZELEV(INDZ+J-1),J=1+NCPP*(NX-1),NCPP*NX)
                        END IF
 20                  CONTINUE
                  ELSE
                     DO 25 K = 1+NRPP*(NY-1), NRPP*NY
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZELEV(INDZ+J-1),J=1+NCPP*(NX-1),NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZELEV(INDZ+J-1),J=1+NCPP*(NX-1),NCPP*NX)
                        END IF
 25                  CONTINUE
                  END IF
 30            CONTINUE
 40         CONTINUE
         END IF
         IF (FLGPOL) THEN
C           Print The Receptor Heights Above Ground for This Network
C           Set Number of Columns Per Page, NCPP
            NCPP = 9
C           Set Number of Rows Per Page, NRPP
            NRPP = 40
C           Begin LOOP Through Networks
C           Calculate Number of Pages Per X-Group, NPPX, & Per Y-Group, NPPY
            NPPX = 1 + INT((NUMXPT(I)-1)/NCPP)
            NPPY = 1 + INT((NUMYPT(I)-1)/NRPP)
            DO 80 NX = 1, NPPX
               DO 70 NY = 1, NPPY
                  CALL HEADER
                  WRITE(IOUNIT,9037) NTID(I), NTTYP(I)
                  WRITE(IOUNIT,9035)
                  IF (NX .EQ. NPPX) THEN
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     END IF
                  ELSE
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     END IF
                  END IF
                  WRITE(IOUNIT,9010)
                  IF (NY .EQ. NPPY) THEN
                     DO 50 K = 1+NRPP*(NY-1), NUMYPT(I)
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZFLAG(INDZ+J-1),J=1+NCPP*(NX-1),NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZFLAG(INDZ+J-1),J=1+NCPP*(NX-1),NCPP*NX)
                        END IF
 50                  CONTINUE
                  ELSE
                     DO 60 K = 1+NRPP*(NY-1), NRPP*NY
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZFLAG(INDZ+J-1),J=1+NCPP*(NX-1),NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                   (AZFLAG(INDZ+J-1),J=1+NCPP*(NX-1),NCPP*NX)
                        END IF
 60                  CONTINUE
                  END IF
 70            CONTINUE
 80         CONTINUE
         END IF
 100  CONTINUE

      IF (IRSTAT(4) .NE. 0) THEN
C        Print Out The Coordinates, Height & Flags For Discrete Cart Receptors
         INDC = 0
         DO 1030 I = 1, NUMREC
            IF (RECTYP(I) .EQ. 'DC') THEN
               INDC = INDC + 1
               IF (MOD(INDC-1,90) .EQ. 0) THEN
                  CALL HEADER
                  WRITE(IOUNIT,9043)
               END IF
               IF (MOD(INDC,2) .NE. 0) THEN
                  WRITE(BUF132(1:55),9045) AXR(I),AYR(I),AZELEV(I),
     &                                     AZFLAG(I)
               ELSE
                  WRITE(BUF132(56:110),9045) AXR(I),AYR(I),AZELEV(I),
     &                                       AZFLAG(I)
                  WRITE(IOUNIT,9090) BUF132
                  WRITE(BUF132,9095)
               END IF
            END IF
 1030    CONTINUE
         IF (MOD(INDC,2) .NE. 0) THEN
            WRITE(IOUNIT,9090) BUF132
            WRITE(BUF132,9095)
         END IF
      END IF

      IF (IRSTAT(5) .NE. 0) THEN
C        Print Out The Coordinates, Height & Flags For Discrete Polar Receptors
         INDC = 0
         DO 1040 I = 1, NUMREC
            IF (RECTYP(I) .EQ. 'DP') THEN
               INDC = INDC + 1
               XRMS = AXR(I) - AXS(IREF(I))
               YRMS = AYR(I) - AYS(IREF(I))
               RANGE  = SQRT(XRMS*XRMS + YRMS*YRMS)
               RADIAL = ATAN2(XRMS, YRMS) * RTODEG
               IF(RADIAL .LE. 0.0) RADIAL = RADIAL + 360.
               IF (MOD(INDC-1,90) .EQ. 0) THEN
                  CALL HEADER
                  WRITE(IOUNIT,9044)
               END IF
               IF (MOD(INDC,2) .NE. 0) THEN
                  WRITE(BUF132(1:65),9047) SRCID(IREF(I)),RANGE,RADIAL,
     &                                     AZELEV(I),AZFLAG(I)
               ELSE
                  WRITE(BUF132(66:130),9047) SRCID(IREF(I)),RANGE,
     &                                       RADIAL,AZELEV(I),AZFLAG(I)
                  WRITE(IOUNIT,9090) BUF132
                  WRITE(BUF132,9095)
               END IF
            END IF
 1040    CONTINUE
         IF (MOD(INDC,2) .NE. 0) THEN
            WRITE(IOUNIT,9090) BUF132
            WRITE(BUF132,9095)
         END IF
      END IF

      IF (IRSTAT(6) .NE. 0) THEN
C        Write Out The Boundary Receptors For The Sources
         INDC = 0
         I = 1
         DO WHILE (I .LE. NUMREC)
            IF (RECTYP(I) .EQ. 'BD') THEN
               INDC = INDC + 1
               ISRF = IREF(I)
               IF (MOD(INDC-1,3) .EQ. 0) THEN
                  CALL HEADER
                  WRITE(IOUNIT,9084)
               END IF
               WRITE(IOUNIT,9082) SRCID(ISRF),SRCTYP(ISRF),
     &             AXS(ISRF),AYS(ISRF),AZS(ISRF),
     &             (J,AXR(I+J-1),AYR(I+J-1),AZELEV(I+J-1),
     &              AZFLAG(I+J-1),J=1,36)
               I = I + 36
            ELSE
               I = I + 1
            END IF
         END DO
      END IF

 9011 FORMAT(/48X,'* ELEVATION HEIGHTS IN METERS *'/)
 9035 FORMAT(/44X,'* RECEPTOR FLAGPOLE HEIGHTS IN METERS *'/)
 9034 FORMAT(/40X,'*** GRIDDED RECEPTOR NETWORK SUMMARY ***')
 9037 FORMAT(/34X,'*** NETWORK ID: ',A8,' ;  NETWORK TYPE: ',
     &       A8,' ***')
 9038 FORMAT(/42X,'*** X-COORDINATES OF GRID ***'/
     &       52X,'(METERS)'/)
 9039 FORMAT(/42X,'*** DISTANCE RANGES OF NETWORK ***'/
     &       52X,'(METERS)'/)
 9036 FORMAT(/42X,'*** ORIGIN FOR POLAR NETWORK ***'/,
     &      32X,'X-ORIG =',F10.2,' ;   Y-ORIG = ',F10.2,'  (METERS)')
 9040 FORMAT(100(5X,10(F10.1,',')/))
 9041 FORMAT(/42X,'*** Y-COORDINATES OF GRID *** ',
     &       /52X,'(METERS)'/)
 9042 FORMAT(/42X,'*** DIRECTION RADIALS OF NETWORK *** ',
     &       /52X,'(DEGREES)'/)
 9010 FORMAT(66(' -')/)
 9013 FORMAT(2X,F10.2,1X,'|',1X,9(1X,F12.2,:))
 9016 FORMAT(3X,' Y-COORD  |',48X,'X-COORD (METERS)')
 9017 FORMAT(3X,' (METERS) |',1X,9(1X,F12.2,:))
 9018 FORMAT(3X,'DIRECTION |',48X,'DISTANCE (METERS)')
 9019 FORMAT(3X,'(DEGREES) |',1X,9(1X,F12.2,:))
 9043 FORMAT(/45X,'*** DISCRETE CARTESIAN RECEPTORS ***',
     &       /45X,'  (X-COORD, Y-COORD, ZELEV, ZFLAG)',
     &       /45X,'              (METERS)'/)
 9044 FORMAT(/45X,' *** DISCRETE POLAR RECEPTORS ***',
     &       /45X,' ORIGIN: (DIST, DIR, ZELEV, ZFLAG)',
     &       /45X,' SRCID: (METERS,DEG,METERS,METERS)'/)
 9045 FORMAT(4X,' (',3(F9.1,', '),F9.1,'); ')
 9047 FORMAT(4X,A8,': (',3(F9.1,', '),F9.1,'); ')
 9082 FORMAT(' BOUNDARY RECEPTORS FOR SOURCE ID: ',A8,/,5X,
     &       ' OF SOURCE TYPE: ',A8,'; WITH ORIGIN AT (',2(F10.2,', '),
     &     F10.2,')'/3(' SEC.    XCOORD      YCOORD   ZELEV  ZFLAG',3X),
     &       /,12(3(I4,1X,F10.2,', ',F10.2,',',F7.2,',',F6.1,3X),/),/)
 9084 FORMAT(/50X,'*** BOUNDARY RECEPTOR LOCATIONS ***',
     &    /47X,'(DISCRETE RECEPTORS AT 10 DEGREE SECTORS)'//)
 9090 FORMAT(A132)
 9095 FORMAT(132(' '))

      RETURN
      END

      SUBROUTINE CHKREC
C***********************************************************************
C                 CHKREC Module of ISC2 Long Term Model - ISCLT2
C
C        PURPOSE: Print Out The Input Met Data Summary and Source Groups
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To account for new area source algorithm, which
C                    allows for receptors located within the area - 7/7/93
C
C        MODIFIED:   To Correct Sector Width/Index for Checking
C                    Receptors < 3*ZLB - 9/29/92
C
C        MODIFIED:   To account for OpenPit Source - PES - 7/22/94
C
C        INPUTS:  Source and Receptor Inputs
C
C        OUTPUTS: Listing of Receptors Too Close To Sources
C
C        CALLED FROM:   INPSUM
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      REAL XVM(5), YVM(5)

C     Variable Initializations
      MODNAM = 'CHKREC'
      INC = 0

C     Begin Source LOOP
      DO 200 ISRC = 1, NUMSRC

C        Set Effective Source Radius Based on Source Type
         IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
            XRAD = 0.0
         ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
            XRAD = 2.15 * ASYINI(ISRC)
         ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
C           Skip to End of Source LOOP for AREA Sources - No Restrictions on
C           Receptor Placement for New Algorithm
            GO TO 200
         ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
C*           Skip to End of Source LOOP for OPENPIT Sources - No Restrictions on
C*            GO TO 200
            XRAD = -1.0
            XVM(1) = AXVERT(1,ISRC) * 1000.
            XVM(2) = AXVERT(2,ISRC) * 1000.
            XVM(3) = AXVERT(3,ISRC) * 1000.
            XVM(4) = AXVERT(4,ISRC) * 1000.
            XVM(5) = AXVERT(5,ISRC) * 1000.
            YVM(1) = AYVERT(1,ISRC) * 1000.
            YVM(2) = AYVERT(2,ISRC) * 1000.
            YVM(3) = AYVERT(3,ISRC) * 1000.
            YVM(4) = AYVERT(4,ISRC) * 1000.
            YVM(5) = AYVERT(5,ISRC) * 1000.
         END IF

C        Begin Receptor LOOP
         DO 100 IREC = 1, NUMREC

C           Calculate DIST From Edge of Source to Receptor
            X = AXR(IREC) - AXS(ISRC)
            Y = AYR(IREC) - AYS(ISRC)
            DIST = SQRT (X*X + Y*Y) - XRAD

            IF (DIST .LT. 0.99) THEN
C              Receptor Is Too Close To Source
               INC = INC + 1
               IF (MOD((INC-1), 40) .EQ. 0) THEN
                  CALL HEADER
                  WRITE(IOUNIT,9002)
               END IF
               WRITE(IOUNIT,9003) SRCID(ISRC), AXR(IREC),
     &                            AYR(IREC), DIST
            ELSE IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
C              Check For Receptors Less Than 3*ZLB For POINT Sources
               ANG = ATAN2(X,Y) * RTODEG
               IF (ANG .LT. 0.0) ANG = ANG + 360.0
               ISEC = INT(ANG/22.5 + 1.4999)
               IF (ISEC .GT. 16) ISEC = 1
               IF (ISEC .LE. NSEC) THEN
                  DSBH = ADSBH(ISEC,ISRC)
                  DSBW = ADSBW(ISEC,ISRC)
                  XMIN = 3.*AMIN1(DSBH,DSBW)
                  IF (DIST .LT. XMIN) THEN
C                    Receptor Is Too Close To Source
                     INC = INC + 1
                     IF (MOD((INC-1), 40) .EQ. 0) THEN
                        CALL HEADER
                        WRITE(IOUNIT,9002)
                     END IF
                     WRITE(IOUNIT,9003) SRCID(ISRC), AXR(IREC),
     &                                  AYR(IREC), DIST
                  END IF
               END IF
            ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
C              Check for receptors within boundary of an open pit source
               XR = AXR(IREC)
               YR = AYR(IREC)
               CALL PNPOLY(XR,YR,XVM,YVM,5,INOUT)
               IF (INOUT .GT. 0) THEN
C                 Receptor is within boundary
                  INC = INC + 1
                  IF (MOD((INC-1), 40) .EQ. 0) THEN
                     CALL HEADER
                     WRITE(IOUNIT,9002)
                  END IF
                  WRITE(IOUNIT,9004) SRCID(ISRC), AXR(IREC),
     &                               AYR(IREC)
               END IF
            END IF

 100     CONTINUE
C        End Receptor LOOP

 200  CONTINUE
C     End Source LOOP

 9002 FORMAT(22X,'* SOURCE-RECEPTOR COMBINATIONS FOR WHICH ',
     & 'CALCULATIONS MAY NOT BE PERFORMED *'/27X,'LESS THAN 1.0 METER',
     & ' OR 3*ZLB IN DISTANCE, OR WITHIN OPEN PIT SOURCE',//
     & /31X,'SOURCE',9X,'- - RECEPTOR LOCATION - -',9X,'DISTANCE',
     & /31X,'  ID  ',9X,'XR (METERS)   YR (METERS)',9X,'(METERS)',
     & /30X,30('- ')/)
 9003 FORMAT(31X,A8,5X,F13.1,1X,F13.1,7X,F10.2)
 9004 FORMAT(31X,A8,5X,F13.1,1X,F13.1,7X,'   OPENPIT')

      RETURN
      END

      SUBROUTINE PRTMET
C***********************************************************************
C                 PRTMET Module of ISC Model
C
C        PURPOSE: Print Out The Input Met Data Summary and Source Groups
C
C        PROGRAMMER: JEFF WANG
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To Print Out Average Temperatures and Mixing Heights
C                    When Only PERIOD Average Is Used - 9/29/92
C
C        MODIFIED:   To Print Out DEPOSITION-related meteorological data
C                    D. STRIMAITIS, SRC - 2/15/93
C
C        INPUTS:  Model Options and Keyword Summaries
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   INPSUM
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER ATHRUF(6)*1

C     Variable Initializations
      DATA ATHRUF / 'A','B','C','D','E','F' /
      MODNAM = 'PRTMET'

C     Start New Page and Print The Titles
      CALL HEADER

C     Print Wind Speed Categories
      WRITE(IOUNIT,9001) (AVESP(I),I=1,NWS)

C     Print Wind Profile Exponents
      IF (DFAULT .OR. .NOT.USERP) THEN
         IF (URBAN) THEN
            DO 111 I = 1, 6
               DO 110 J = 1, 6
                  PUSER(I,J) = PURB(I)
  110          CONTINUE
  111       CONTINUE
         ELSE IF (RURAL) THEN
            DO 222 I = 1, 6
               DO 220 J = 1, 6
                  PUSER(I,J) = PRUR(I)
  220          CONTINUE
  222       CONTINUE
         END IF
      END IF
      WRITE(IOUNIT,9059)
      WRITE(IOUNIT,9016) (K,K=1,6)
      DO 490  I = 1, 6
         WRITE(IOUNIT,9017) ATHRUF(I), (PUSER(I,J),J=1,6)
  490 CONTINUE

C     Print Vertical Potential Temperature Gradients
      IF (DFAULT .OR. .NOT.USERDT) THEN
         IF (URBAN) THEN
            DO 333 I = 1, 6
               DO 330 J = 1, 6
                  DTUSER(I,J) = DTURB(I)
  330          CONTINUE
  333       CONTINUE
         ELSE IF (RURAL) THEN
            DO 444 I = 1, 6
               DO 440 J = 1, 6
                  DTUSER(I,J) = DTRUR(I)
  440          CONTINUE
  444       CONTINUE
         END IF
      END IF
      WRITE(IOUNIT,9060)
      WRITE(IOUNIT,9016) (K, K=1,6)
      DO 520  I = 1, 6
         WRITE(IOUNIT,9017) ATHRUF(I), (DTUSER(I,J),J=1,6)
  520 CONTINUE

C     Print Out Average Ambient Temperature Data
      WRITE(IOUNIT,9010) (ATHRUF(I), I=1,6)
      DO 550 J = 1, NAVE
C        Add check for PERIOD average and KSTAR = 1, 9/29/92
         IF (KAVE(J) .EQ. 1 .OR. (PERIOD .AND. KSTAR(J).EQ.1)) THEN
            WRITE (IOUNIT,9011) AVEPER(J),(AVETA(J,K),K=1,NKST)
         END IF
 550  CONTINUE

C     Print Out Average Mixing Height Data
      CALL HEADER
      N = 0
      WRITE (IOUNIT,9012)
      DO 570 I = 1, NAVE
C        Add check for PERIOD average and KSTAR = 1, 9/29/92
         IF (KAVE(I) .EQ. 1 .OR. (PERIOD .AND. KSTAR(I).EQ.1)) THEN
            N = N + 1
            IF (N .NE. 1 .AND. MOD(N-1,5).EQ.0) THEN
               CALL HEADER
            END IF
            WRITE (IOUNIT,9013) AVEPER(I), (L, L=1,6)
            DO 560 J = 1, NKST
               WRITE (IOUNIT,9014) ATHRUF(J), (AVEZI(I,J,K),K=1,NWS)
 560        CONTINUE
         END IF
 570  CONTINUE

C     Print out additional meteorological information related to        DTB93334
C     particle settling and removal ---
      IF (LDEP) THEN                                                    DTB93334


C     Print Out Average Roughness Lengths                               DTB93334
         CALL HEADER
         N = 0
         WRITE (IOUNIT,9022)
         DO 660 I = 1, NAVE
            IF (KAVE(I) .EQ. 1 .OR. (PERIOD .AND. KSTAR(I).EQ.1)) THEN
               N = N + 1
               IF (N .NE. 1 .AND. MOD(N-1,5).EQ.0) THEN
                  CALL HEADER
               END IF
               WRITE (IOUNIT,9023) AVEPER(I), AVEZ0M(I)
            END IF
 660     CONTINUE

      ENDIF                                                             DTB93334


 9001 FORMAT(//40X,'*** AVERAGE SPEED FOR EACH WIND SPEED CATEGORY',
     &       ' ***'/60X,'(METERS/SEC)'//42X,6(F7.2,','))
 9010 FORMAT(//40X,'*** AVERAGE AMBIENT AIR TEMPERATURE (KELVIN) ***'/
     &       /30X,6('  STABILITY ')/30X,6(' CATEGORY ',A1,1X)/)
 9011 FORMAT(21X,A6,2X,6F12.4)
 9012 FORMAT(/37X,'*** AVERAGE MIXING LAYER HEIGHT (METERS) ***')
 9013 FORMAT(/56X,A6/26X,6('   WIND SPEED')/26X,6('   CATEGORY',
     &       I2))
 9014 FORMAT(6X,'STABILITY CATEGORY ',A1,6(1X,F12.4))
 9016 FORMAT(16X,'STABILITY',30X,'WIND SPEED CATEGORY'/16X,'CATEGORY',
     &       9X,6(I1,14X))
 9017 FORMAT(19X,A1,5X,6(5X,E10.5))

 9022 FORMAT(/37X,'*** AVERAGE SURFACE ROUGHNESS LENGTH (METERS) ***')  DTB93334

 9023 FORMAT(/6X,A6,15X,F12.4)
 9059 FORMAT(//51X,'*** WIND PROFILE EXPONENTS ***'//)
 9060 FORMAT(//42X,'*** VERTICAL POTENTIAL TEMPERATURE GRADIENTS ***'/
     &       53X,'(DEGREES KELVIN PER METER)'//)

      RETURN
      END
