      SUBROUTINE MECARD
C***********************************************************************
C                 MECARD Module of ISC Model
C
C        PURPOSE: To process MEteorology Pathway card images
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Pathway (ME) and Keyword
C
C        OUTPUTS: Meteorology Option Switches
C                 Meteorology Setup Status Switches
C
C        CALLED FROM:   SETUP
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'MECARD'

      IF (KEYWRD .EQ. 'STARTING') THEN
C        Set Status Switch
         IMSTAT(1) = IMSTAT(1) + 1
         IF (IMSTAT(1) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         END IF
      ELSE IF (KEYWRD .EQ. 'INPUTFIL') THEN
C        Set Status Switch
         IMSTAT(2) = IMSTAT(2) + 1
         IF (IMSTAT(2) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Meteorology File Information            ---   CALL METFIL
            CALL METFIL
         END IF
      ELSE IF (KEYWRD .EQ. 'ANEMHGHT') THEN
C        Set Status Switch
         IMSTAT(3) = IMSTAT(3) + 1
         IF (IMSTAT(3) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Anemometer Height Information           ---   CALL ANEMHT
            CALL ANEMHT
         END IF
      ELSE IF (KEYWRD .EQ. 'SURFDATA') THEN
C        Set Status Switch
         IMSTAT(4) = IMSTAT(4) + 1
         IF (IMSTAT(4) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Surface Data Information                ---   CALL SFDATA
            CALL SFDATA
         END IF
      ELSE IF (KEYWRD .EQ. 'UAIRDATA') THEN
C        Set Status Switch
         IMSTAT(5) = IMSTAT(5) + 1
         IF (IMSTAT(5) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Upper Air Data Information              ---   CALL UADATA
            CALL UADATA
         END IF
      ELSE IF (KEYWRD .EQ. 'STARDATA') THEN
C        Set Status Switch
         IMSTAT(6) = IMSTAT(6) + 1
         IF (IMSTAT(6) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process STAR Data Information                   ---   CALL STDATA
            CALL STDATA
         END IF
      ELSE IF (KEYWRD .EQ. 'WDROTATE') THEN
C        Set Status Switch
         IMSTAT(7) = IMSTAT(7) + 1
         IF (IMSTAT(7) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Wind Direction Correction Option        ---   CALL WDROTA
            CALL WDROTA
         END IF
      ELSE IF (KEYWRD .EQ. 'WINDPROF') THEN
C        Set Status Switch
         IMSTAT(8) = IMSTAT(8) + 1
         IF (DFAULT) THEN
C           WRITE Warning Message and Ignore Inputs
            CALL ERRHDL(PATH,MODNAM,'W','206',KEYWRD)
         ELSE
C           Process Wind Speed Profile Exponents            ---   CALL WSPROF
            CALL WSPROF
C           Set Logical Flag Indicating User-specified P-Exponents
            USERP = .TRUE.
         END IF
      ELSE IF (KEYWRD .EQ. 'DTHETADZ') THEN
C        Set Status Switch
         IMSTAT(9) = IMSTAT(9) + 1
         IF (DFAULT) THEN
C           WRITE Warning Message and Ignore Inputs
            CALL ERRHDL(PATH,MODNAM,'W','206',KEYWRD)
         ELSE
C           Process Vertical Pot. Temperature Gradients     ---   CALL DTHETA
            CALL DTHETA
C           Set Logical Flag Indicating User-specified DThetaDZ
            USERDT = .TRUE.
         END IF
      ELSE IF (KEYWRD .EQ. 'AVESPEED') THEN
C        Set Status Switch
         IMSTAT(10) = IMSTAT(10) + 1
         IF (IMSTAT(10) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Wind Speed Categories                   ---   CALL AVESPD
            CALL AVESPD
         END IF
      ELSE IF (KEYWRD .EQ. 'AVETEMPS') THEN
C        Set Status Switch
         IMSTAT(11) = IMSTAT(11) + 1
C        Process Average Air Temperatures                   ---   CALL AVETMP
         CALL AVETMP
      ELSE IF (KEYWRD .EQ. 'AVEMIXHT') THEN
C        Set Status Switch
         IMSTAT(12) = IMSTAT(12) + 1
C        Process Average Mixing Heights                     ---   CALL AVEMHT
         CALL AVEMHT


      ELSE IF (KEYWRD .EQ. 'AVEROUGH') THEN                             DTB93334
C        Set Status Switch                                              DTB93334
         IMSTAT(15) = IMSTAT(15) + 1                                    DTB93334
C        Process Average Roughness Lengths         ---   CALL AVERGH    DTB93334
         CALL AVERGH                                                    DTB93334

C     Calls to AVEMOL, and AVEUST removed  11-4-93                      DTB93308

      ELSE IF (KEYWRD .EQ. 'FINISHED') THEN
C        Set Status Switch
         IMSTAT(20) = IMSTAT(20) + 1
         IF (IMSTAT(20) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
            GO TO 999
         END IF
C        Write Error Messages for Missing Mandatory Keyword(s)
         IF (IMSTAT(1) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','STARTING')
         END IF
         IF (IMSTAT(2) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','INPUTFIL')
         END IF
         IF (IMSTAT(3) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','ANEMHGHT')
         END IF
         IF (IMSTAT(4) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','SURFDATA')
         END IF
         IF (IMSTAT(5) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','UAIRDATA')
         END IF
         IF (IMSTAT(11) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','AVETEMPS')
         END IF
         IF (IMSTAT(12) .EQ. 0) THEN
            CALL ERRHDL(PATH,MODNAM,'E','130','AVEMIXHT')
         END IF

C     Calls to error routines for AVEMOL and AVEUST removed             DTB93308

         IF (LDEP) THEN                                                 DTB93334
            IF (IMSTAT(15) .EQ. 0) THEN                                 DTB93334
               CALL ERRHDL(PATH,MODNAM,'E','130','AVEROUGH')            DTB93334
            END IF                                                      DTB93334
         END IF                                                         DTB93334

C        Check for Lack of STARDATA Input Card
         IF (IMSTAT(6) .EQ. 0) THEN
            IF (PERIOD .AND. .NOT.MONTH .AND. .NOT.SEASON .AND.
     &                                        .NOT.QUARTR) THEN
C              WRITE Error Message:  PERIOD Average Only With No STARDATA Card
               CALL ERRHDL(PATH,MODNAM,'E','368','STARDATA')
            ELSE
C              Set KSTAR Array = KAVE Array
               DO 100 I = 1, NAVE
                  KSTAR(I) = KAVE(I)
 100           CONTINUE
            END IF
         ELSE IF (IMSTAT(6) .EQ. 1) THEN
C           Check for Availability of STAR Data for All KAVE Requested
            DO 200 I = 1, NAVE
               IF (KAVE(I) .EQ. 1 .AND. KSTAR(I) .NE. 1) THEN
C                 WRITE Error Message:  STAR Data Not Available for This KAVE
                  CALL ERRHDL(PATH,MODNAM,'E','398',AVEPER(I))
               END IF
 200        CONTINUE
         END IF

C        Check for Missing AVETEMPS and/or AVEMIXHT Cards For Requested
C        Averaging Periods. --  Do same for
C        AVEROUGH cards if particles are modeled (LDEP=T).
         DO 300 I = 1, NAVE
            IF (KAVE(I) .EQ. 1) THEN
               IF (IAVETA(I) .NE. 1) THEN
C                 WRITE Error Message:  No AVETEMPS For Specified AVEPER
                  CALL ERRHDL(PATH,MODNAM,'E','142',AVEPER(I))
               END IF
               IF (IAVEZI(I) .NE. 1) THEN
C                 WRITE Error Message:  No AVEMIXHT For Specified AVEPER
                  CALL ERRHDL(PATH,MODNAM,'E','144',AVEPER(I))
               END IF

               IF (LDEP) THEN                                           DTB93334
                  IF (IAVEZ0M(I) .NE. 1) THEN                           DTB93334
C                    WRITE Error Message:  No AVEROUGH For Specified AVEPER
                     CALL ERRHDL(PATH,MODNAM,'E','146',AVEPER(I))       DTB93334
                  END IF                                                DTB93334
               END IF                                                   DTB93334

C     Calls to error routines for AVEMOL and AVEUST removed             DTB93308

            END IF
 300     CONTINUE

C        Initialize The NUM of HRS For Each Averaging Period, Used To
C        Adjust The Emission Rate Factors for DEPOSition Calculations
         NUMHRS(1) = 744
         NUMHRS(2) = 672
         NUMHRS(3) = 744
         NUMHRS(4) = 720
         NUMHRS(5) = 744
         NUMHRS(6) = 720
         NUMHRS(7) = 744
         NUMHRS(8) = 744
         NUMHRS(9) = 720
         NUMHRS(10) = 744
         NUMHRS(11) = 720
         NUMHRS(12) = 744
         IF (SEASON .OR. SEADAT) THEN
C           Initialize for Seasons
            NUMHRS(13) = 2160
            NUMHRS(14) = 2208
            NUMHRS(15) = 2208
            NUMHRS(16) = 2184
         ELSE
C           Initialize for Quarters
            NUMHRS(13) = 2160
            NUMHRS(14) = 2184
            NUMHRS(15) = 2208
            NUMHRS(16) = 2208
         END IF
         NUMHRS(17) = 8760

         ISYI = ISYEAR
C        Convert 2-Digit Year to 4-Digit Value And Check for Leap Year
         IF (ISYI .LE. 50) THEN
            ISYI = 2000 + ISYI
         ELSE IF (ISYI .LE. 99) THEN
            ISYI = 1900 + ISYI
         END IF
         IF (MOD(ISYI,4) .EQ. 0 .AND. MOD(ISYI,400) .EQ. 0) THEN
C           Adjust NUMHRS for Leap Year
            NUMHRS(2) = 696
            NUMHRS(13) = 2184
            NUMHRS(17) = 8784
         END IF

C        OPEN Met Data File                                 ---   CALL MEOPEN
         IF (IMSTAT(2) .NE. 0) THEN
            CALL MEOPEN
         END IF

      ELSE
C        Write Error Message: Invalid Keyword for This Pathway
         CALL ERRHDL(PATH,MODNAM,'E','110',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE METFIL
C***********************************************************************
C                 METFIL Module of ISC Model
C
C        PURPOSE: Process Meteorology Input File Options
C                 From Runstream Input Image
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Meteorological Data Filename and Format
C
C        ERROR HANDLING:   Checks for No Parameters;
C                          Checks for No Format (uses default);
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   MECARD
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'METFIL'

      IF (IFC .EQ. 3) THEN
C        Retrieve Met Data Filename as Character Substring to Maintain Case
         METINP = RUNST1(LOCB(3):LOCE(3))
C        Use Default Met Data Format: Initialized in SUBROUTINE VARINI to
CC        METFRM = '(6F10.0)'
      ELSE IF (IFC .EQ. 4) THEN
C        Retrieve Met Data Filename as Character Substring to Maintain Case
         METINP = RUNST1(LOCB(3):LOCE(3))
C        Check for Format String > 60 (Limit for METFRM Variable)
         IF ((LOCE(4)-LOCB(4)) .LE. 59) THEN
            IF ((LOCE(4)-LOCB(4)) .GT. 39) THEN
C              Retrieve Met Format as Char. Substring to Bypass Field Limit
               METFRM = RUNST1(LOCB(4):LOCE(4))
            ELSE
C              Retrieve Met Format From FIELD(4)
               METFRM = FIELD(4)
            END IF
         ELSE
C           WRITE Error Message:  METFRM Field is Too Long
            CALL ERRHDL(PATH,MODNAM,'E','203',' METFRM ')
         END IF
      ELSE IF (IFC .GT. 4) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Warning Message         ! No Parameters Specified
         CALL ERRHDL(PATH,MODNAM,'W','200',KEYWRD)
      END IF

      RETURN
      END

      SUBROUTINE ANEMHT
C***********************************************************************
C                 ANEMHT Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Process Anemometer Height Options
C                 From Runstream Input Image
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Anemometer Height, ZREF (m)
C
C        ERROR HANDLING:   Checks for No Parameters;
C                          Checks for No Units (uses default of m);
C                          Checks for Invalid or Suspicious Values of ZREF;
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   MECARD
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'ANEMHT'

      IF (IFC .EQ. 3 .OR. IFC .EQ. 4) THEN
         CALL STONUM(FIELD(3),40,ANHT,IDUM)
C        Check The Numerical Field
         IF (IDUM.EQ.-1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
            GO TO 999
         END IF
         IF (IFC .EQ. 4 .AND. FIELD(4) .EQ. 'FEET') THEN
            ANHT = 0.3048 * ANHT
         ELSE IF (IFC .EQ. 4 .AND. FIELD(4) .NE. 'METERS') THEN
C           WRITE Warning Message - Invalid ZRUNIT Parameter
            CALL ERRHDL(PATH,MODNAM,'W','203','ZRUNIT')
         END IF
         IF (ANHT .GT. 100.0 .AND. IDUM .EQ. 1) THEN
C           WRITE Warning Message - Possible Error In ANHT
            WRITE(DUMMY,'(F8.3)') ANHT
            CALL ERRHDL(PATH,MODNAM,'W','340',DUMMY)
            ZREF = ANHT
         ELSE IF (ANHT .GE. 2.0 .AND. IDUM .EQ. 1) THEN
            ZREF = ANHT
         ELSE IF (ANHT .GT. 0.0 .AND. IDUM .EQ. 1) THEN
C           WRITE Warning Message - Possible Error In ANHT
            CALL ERRHDL(PATH,MODNAM,'W','340',KEYWRD)
            ZREF = ANHT
         ELSE IF (ANHT .LE. 0.0 .AND. IDUM .EQ. 1) THEN
C           WRITE Error Message - Invalid Anemometer Height
            CALL ERRHDL(PATH,MODNAM,'E','203','Anem Hgt')
         ELSE IF (IDUM .NE. 1) THEN
C           WRITE Error Message - Invalid Numeric Field
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         END IF
      ELSE IF (IFC .GT. 4) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE SFDATA
C***********************************************************************
C                 SFDATA Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Process Meteorology Surface Data Station Options
C                 From Runstream Input Image
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Meteorological Surface Data Station Identification
C
C        ERROR HANDLING:   Checks for Too Few Parameters;
C                          Checks for Invalid Numeric Fields;
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   MECARD
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'SFDATA'

      IF (IFC .EQ. 2) THEN
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
         GO TO 999
      ELSE IF (IFC .LT. 4) THEN
C        WRITE Error Message           ! Not Enough Parameters
         CALL ERRHDL(PATH,MODNAM,'E','201',KEYWRD)
         GO TO 999
      ELSE IF (IFC .GT. 7) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
         GO TO 999
      END IF

      CALL STONUM(FIELD(3),40,FNUM,IDUM)
C     Check The Numerical Field
      IF (IDUM .NE. 1) THEN
         CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         GO TO 199
      END IF
      IDSURF = INT(FNUM)

 199  CALL STONUM(FIELD(4),40,FNUM,IDUM)
C     Check The Numerical Field
      IF (IDUM .NE. 1) THEN
         CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         GO TO 299
      END IF
      ISYEAR = INT(FNUM)

 299  IF (IFC .GE. 5) THEN
C        Retrieve Surface Data Station Name (Optional)
         SFNAME = FIELD(5)
      ELSE
         SFNAME = 'UNKNOWN'
      END IF

      IF (IFC .EQ. 7) THEN
C        Retrieve Coordinates for Surface Data Location (Optional)
         CALL STONUM(FIELD(6),40,SFX,IDUM)
         IF (IDUM .NE. 1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         END IF
         CALL STONUM(FIELD(7),40,SFY,IDUM)
         IF (IDUM .NE. 1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE UADATA
C***********************************************************************
C                 UADATA Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Process Meteorology Upper Air Data Station Options
C                 From Runstream Input Image
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Meteorological Upper Air Data Station Identification
C
C        ERROR HANDLING:   Checks for Too Few Parameters;
C                          Checks for Invalid Numeric Fields;
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   MECARD
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'UADATA'

      IF (IFC .EQ. 2) THEN
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
         GO TO 999
      ELSE IF (IFC .LT. 4) THEN
C        WRITE Error Message           ! Not Enough Parameters
         CALL ERRHDL(PATH,MODNAM,'E','201',KEYWRD)
         GO TO 999
      ELSE IF (IFC .GT. 7) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
         GO TO 999
      END IF

      CALL STONUM(FIELD(3),40,FNUM,IDUM)
C     Check The Numerical Field
      IF (IDUM .NE. 1) THEN
         CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         GO TO 199
      END IF
      IDUAIR = INT(FNUM)

 199  CALL STONUM(FIELD(4),40,FNUM,IDUM)
C     Check The Numerical Field
      IF (IDUM .NE. 1) THEN
         CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         GO TO 299
      END IF
      IUYEAR = INT(FNUM)

 299  IF (IFC .GE. 5) THEN
C        Retrieve Surface Data Station Name (Optional)
         UANAME = FIELD(5)
      ELSE
         UANAME = 'UNKNOWN'
      END IF

      IF (IFC .EQ. 7) THEN
C        Retrieve Coordinates for Surface Data Location (Optional)
         CALL STONUM(FIELD(6),40,UAX,IDUM)
         IF (IDUM .NE. 1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         END IF
         CALL STONUM(FIELD(7),40,UAY,IDUM)
         IF (IDUM .NE. 1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE STDATA
C***********************************************************************
C                 STDATA Module of ISC Model
C
C        PURPOSE: Identify STAR Data From Runstream Input Image
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: STAR Data Array
C
C        ERROR HANDLING:   Checks for Too Many Long Term Averages (>4);
C                          Checks for Invalid Averaging Periods, MOD(24,X) NE 0;
C                          Checks for Duplicate Long Term Averaging Periods
C
C        CALLED FROM:   MECARD
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER*8 KOPT

C     Variable Initializations
      MODNAM = 'STDATA'
      NUMSTR = 0

C     Check for No Parameters
      IF (IFC .LT. 3) THEN
C        WRITE Error Message     ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF

C     Loop Through Fields Setting Logical Switches and Filling KSTAR Array
      DO 50 I = 3, IFC
         KOPT = FIELD(I)
         IF (KOPT .EQ. 'MONTH') THEN
            MONDAT = .TRUE.
            DO 20 J = 1, 12
               KSTAR(J) = 1
 20         CONTINUE
            NUMSTR = NUMSTR + 12
         ELSE IF (KOPT .EQ. 'SEASON') THEN
            SEADAT = .TRUE.
            DO 30 J = 13, 16
               KSTAR(J) = 1
 30         CONTINUE
            NUMSTR = NUMSTR + 4
         ELSE IF (KOPT .EQ. 'ANNUAL') THEN
            ANNDAT = .TRUE.
            KSTAR(17) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'QUARTR') THEN
            QUADAT = .TRUE.
            DO 40 J = 13, 16
               KSTAR(J) = 1
 40         CONTINUE
            NUMSTR = NUMSTR + 4
         ELSE IF (KOPT .EQ. 'JAN') THEN
            MONDAT = .TRUE.
            KSTAR(1) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'FEB') THEN
            MONDAT = .TRUE.
            KSTAR(2) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'MAR') THEN
            MONDAT = .TRUE.
            KSTAR(3) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'APR') THEN
            MONDAT = .TRUE.
            KSTAR(4) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'MAY') THEN
            MONDAT = .TRUE.
            KSTAR(5) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'JUN') THEN
            MONDAT = .TRUE.
            KSTAR(6) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'JUL') THEN
            MONDAT = .TRUE.
            KSTAR(7) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'AUG') THEN
            MONDAT = .TRUE.
            KSTAR(8) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'SEP') THEN
            MONDAT = .TRUE.
            KSTAR(9) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'OCT') THEN
            MONDAT = .TRUE.
            KSTAR(10) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'NOV') THEN
            MONDAT = .TRUE.
            KSTAR(11) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'DEC') THEN
            MONDAT = .TRUE.
            KSTAR(12) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'WINTER') THEN
            SEADAT = .TRUE.
            KSTAR(13) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'SPRING') THEN
            SEADAT = .TRUE.
            KSTAR(14) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'SUMMER') THEN
            SEADAT = .TRUE.
            KSTAR(15) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'FALL') THEN
            SEADAT = .TRUE.
            KSTAR(16) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'QUART1') THEN
            QUADAT = .TRUE.
            KSTAR(13) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'QUART2') THEN
            QUADAT = .TRUE.
            KSTAR(14) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'QUART3') THEN
            QUADAT = .TRUE.
            KSTAR(15) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .EQ. 'QUART4') THEN
            QUADAT = .TRUE.
            KSTAR(16) = 1
            NUMSTR = NUMSTR + 1
         ELSE IF (KOPT .NE. 'PERIOD') THEN
C           WRITE Error Message      ! Invalid Averaging Period
            CALL ERRHDL(PATH,MODNAM,'E','203','AVEPER')
         END IF
 50   CONTINUE

C     Check for Too Many Averaging Periods
      IF (NUMSTR .GT. NAVE) THEN
C        WRITE Error Message:  Too Many Averaging Periods Selected
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      END IF

C     Check for Inconsistent Use of Seasonal and Quarterly Data
      IF (SEADAT .AND. QUADAT) THEN
C        WRITE Error Message:  Can't Process Both Seasons and Quarters
         CALL ERRHDL(PATH,MODNAM,'E','369',KEYWRD)
      END IF
C     Check for Inconsistent Use of Season Averages with Quarterly Data,
C     And Vice Versa
      IF ((SEASON .AND. QUADAT .AND. .NOT.SEADAT) .OR.
     &    (QUARTR .AND. SEADAT .AND. .NOT.QUADAT)) THEN
         DO 100 I = 13, 16
            IF (KAVE(I) .EQ. 1) THEN
C              WRITE Error Message:  STAR Data Not Avail for Specified AVEPER
               CALL ERRHDL(PATH,MODNAM,'E','398',AVEPER(I))
            END IF
 100     CONTINUE
      END IF

C     Check for Possible Inconsistent Use of PERIOD Average
      IF (PERIOD .AND. .NOT.MONTH .AND. .NOT.SEASON .AND.
     &                                  .NOT.QUARTR) THEN
         IF (ANNDAT) THEN
C           WRITE Error Message: PERIOD Average With ANNUAL Data
            CALL ERRHDL(PATH,MODNAM,'E','362',KEYWRD)
         END IF
         IF (MONDAT .AND. (SEADAT .OR. QUADAT)) THEN
C           WRITE Error Message: Too Many Period Or Time Fields
            CALL ERRHDL(PATH,MODNAM,'E','364',KEYWRD)
         END IF
         IF (.NOT.MONDAT .AND. .NOT.SEADAT .AND. .NOT.QUADAT) THEN
C           WRITE Warning Message: Period Average Only With No STARDATA
            CALL ERRHDL(PATH,MODNAM,'E','368',KEYWRD)
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE WDROTA
C***********************************************************************
C                 WDROTA Module of ISC Short Term Model - Version 2
C
C     PURPOSE:    PROCESSES INPUT FOR ROTATING WIND DIRECTION DATA
C
C     PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Wind Direction Rotation Angle
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Field
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      ROTANG = 0.0
      MODNAM = 'WDROTA'

      IF (IFC .EQ. 3) THEN
         CALL STONUM(FIELD(3),40,ROTANG,IDUM)
         IF (IDUM .NE. 1) THEN
C            WRITE Error Message  ! Invalid Numeric Field Encountered
             CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
         ELSE IF (ABS(ROTANG) .GT. 180.0) THEN
C            WRITE Error Message       ! ROTANG Out of Range
             CALL ERRHDL(PATH,MODNAM,'E','380','ROTANG')
         END IF
      ELSE IF (IFC .GT. 3) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE WSPROF
C***********************************************************************
C                 WSPROF Module of ISC Short Term Model - Version 2
C
C     PURPOSE:    PROCESSES INPUT FOR WIND SPEED PROFILE EXPONENTS
C
C     PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of Wind Speed Profile Exponents for Each Stability
C                 and Wind Speed Category
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C                       Checks for Invalid Stability Class Indicator
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'WSPROF'

      IF (IFC .GE. 4 .AND. IFC .LE. 9) THEN
         IWS = 0
C        Determine Stability Category Index from FIELD(3); Accepts Either
C        Character Inputs (A-F) or Numeric Inputs (1-6).
         IF (FIELD(3) .EQ. 'A' .OR. FIELD(3) .EQ. '1') THEN
            IST = 1
         ELSE IF (FIELD(3) .EQ. 'B' .OR. FIELD(3) .EQ. '2') THEN
            IST = 2
         ELSE IF (FIELD(3) .EQ. 'C' .OR. FIELD(3) .EQ. '3') THEN
            IST = 3
         ELSE IF (FIELD(3) .EQ. 'D' .OR. FIELD(3) .EQ. '4') THEN
            IST = 4
         ELSE IF (FIELD(3) .EQ. 'E' .OR. FIELD(3) .EQ. '5') THEN
            IST = 5
         ELSE IF (FIELD(3) .EQ. 'F' .OR. FIELD(3) .EQ. '6') THEN
            IST = 6
         ELSE
C           WRITE Error Message           ! Invalid Stability Class Indicator
            CALL ERRHDL(PATH,MODNAM,'E','203','INDKST')
C           Exit to END
            GO TO 999
         END IF

C        Fill PUSER Array
         DO 100 I = 4, IFC
            CALL STONUM(FIELD(I),40,ZNUM,IDUM)
C           Check The Numerical Field
            IF (IDUM .EQ. -1) THEN
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               GO TO 100
            END IF
            IF (ZNUM .LT. 0.0 .OR. ZNUM .GT. 1.0) THEN
C              WRITE Error Message          ! PUSER Out of Range
               CALL ERRHDL(PATH,MODNAM,'E','380','PUSER')
            ELSE
               DO 90 J = 1,IDUM
                  IWS = IWS + 1
                  IF (IWS .LE. NWS) THEN
                     PUSER(IST,IWS) = ZNUM
                  ELSE
C                    WRITE Error Message    ! Too Many Values
                     CALL ERRHDL(PATH,MODNAM,'E','231',KEYWRD)
                  END IF
 90            CONTINUE
            END IF
 100     CONTINUE

      ELSE IF (IFC .GT. 9) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF

 999  CONTINUE

      RETURN
      END

      SUBROUTINE DTHETA
C***********************************************************************
C                 DTHETA Module of ISC Short Term Model - Version 2
C
C     PURPOSE:    PROCESSES INPUT FOR VERTICAL POTENTIAL TEMPERATURE
C                 GRADIENTS
C
C     PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of Vertical Potential Temperature Gradients for
C                 Each Stability and Wind Speed Category
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C                       Checks for Invalid Stability Class Indicator
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'DTHETA'

      IF (IFC .GE. 4 .AND. IFC .LE. 9) THEN
         IWS = 0

C        Determine Stability Category Index from FIELD(3); Accepts Either
C        Character Inputs (A-F) or Numeric Inputs (1-6).
         IF (FIELD(3) .EQ. 'A' .OR. FIELD(3) .EQ. '1') THEN
            IST = 1
         ELSE IF (FIELD(3) .EQ. 'B' .OR. FIELD(3) .EQ. '2') THEN
            IST = 2
         ELSE IF (FIELD(3) .EQ. 'C' .OR. FIELD(3) .EQ. '3') THEN
            IST = 3
         ELSE IF (FIELD(3) .EQ. 'D' .OR. FIELD(3) .EQ. '4') THEN
            IST = 4
         ELSE IF (FIELD(3) .EQ. 'E' .OR. FIELD(3) .EQ. '5') THEN
            IST = 5
         ELSE IF (FIELD(3) .EQ. 'F' .OR. FIELD(3) .EQ. '6') THEN
            IST = 6
         ELSE
C           WRITE Error Message           ! Invalid Stability Class Indicator
            CALL ERRHDL(PATH,MODNAM,'E','203','INDKST')
C           Exit to END
            GO TO 1999
         END IF

C        Fill DTUSER Array
         DO 100 I = 4, IFC
            CALL STONUM(FIELD(I),40,ZNUM,IDUM)
C           Check The Numerical Field
            IF (IDUM .EQ. -1) THEN
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               GO TO 100
            END IF
            IF (ZNUM .LT. 0.0 .OR. ZNUM .GT. 0.2) THEN
C               WRITE Error Message       ! DTUSER Out of Range
                CALL ERRHDL(PATH,MODNAM,'E','380','DTUSER')
            ELSE
               DO 90 J = 1,IDUM
                  IWS = IWS + 1
                  IF (IWS .LE. NWS) THEN
                     DTUSER(IST,IWS) = ZNUM
                  ELSE
C                    WRITE Error Message    ! Too Many Values
                     CALL ERRHDL(PATH,MODNAM,'E','231',KEYWRD)
                  END IF
 90            CONTINUE
            END IF
 100     CONTINUE

      ELSE IF (IFC .GT. 9) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF
 1999 CONTINUE

      RETURN
      END

      SUBROUTINE AVESPD
C***********************************************************************
C
C                 AVESPD Module of ISC Model
C
C     PURPOSE:    Processes Input Average Wind Speed Categories
C
C     PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of Wind Speed Category Limits (5)
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C                       Checks for Wind Speed Category Decreasing
C
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'AVESPD'

      IF (IFC .EQ. NWS+2) THEN
C        Fill AVESP Array
         DO 100 I = 3, IFC
            CALL STONUM(FIELD(I),40,ZNUM,IDUM)
            IF (IDUM .NE. 1) THEN
C              WRITE Error Message  ! Invalid Numeric Field Encountered
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
            ELSE IF (ZNUM .LT. 1.0 .OR. ZNUM .GT. 30.0) THEN
C              WRITE Error Message       ! AVESP Out of Range
               CALL ERRHDL(PATH,MODNAM,'E','380',KEYWRD)
            ELSE
               IWS = I - 2
               AVESP(IWS) = ZNUM
               IF (IWS.GT.1 .AND. AVESP(IWS).LE.AVESP(IWS-1)) THEN
C                 WRITE Error Message    ! Invalid AVESP Value, LE Previous
                  CALL ERRHDL(PATH,MODNAM,'E','203','AVESP')
               END IF
            END IF
 100     CONTINUE
      ELSE IF (IFC .GT. NWS+2) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','217',KEYWRD)
      END IF

      RETURN
      END

      SUBROUTINE AVETMP
C***********************************************************************
C                 AVETMP Module of ISC Model
C
C     PURPOSE:    Processes Input For Average Potential Temperature
C                 Profiles
C
C     PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C     MODIFIED:   To avoid TA = 0.0 due to zero-divide in SUB U_STAR.
C                 Roger Brode, PES, Inc. - 11/7/94
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of Average Potential Temperature for Each Average
C                 period
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C                       Checks for Invalid Stability Class Indicator
C
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER KOPT*8

C     Variable Initializations
      MODNAM = 'AVETMP'

      IF (IFC .GT. 9) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE IF (IFC .LT. 4) THEN
C        WRITE Error Message           ! Not Enough Parameters
         CALL ERRHDL(PATH,MODNAM,'E','201',KEYWRD)
      ELSE IF (IFC .GE. 4 .AND. IFC .LE. 9) THEN
         IST = 0
C        Determine Average Period from FIELD(3)
         KOPT = FIELD(3)
         IF (KOPT .EQ. 'JAN') THEN
            IOPT = 1
         ELSE IF (KOPT .EQ. 'FEB') THEN
            IOPT = 2
         ELSE IF (KOPT .EQ. 'MAR') THEN
            IOPT = 3
         ELSE IF (KOPT .EQ. 'APR') THEN
            IOPT = 4
         ELSE IF (KOPT .EQ. 'MAY') THEN
            IOPT = 5
         ELSE IF (KOPT .EQ. 'JUN') THEN
            IOPT = 6
         ELSE IF (KOPT .EQ. 'JUL') THEN
            IOPT = 7
         ELSE IF (KOPT .EQ. 'AUG') THEN
            IOPT = 8
         ELSE IF (KOPT .EQ. 'SEP') THEN
            IOPT = 9
         ELSE IF (KOPT .EQ. 'OCT') THEN
            IOPT = 10
         ELSE IF (KOPT .EQ. 'NOV') THEN
            IOPT = 11
         ELSE IF (KOPT .EQ. 'DEC') THEN
            IOPT = 12
         ELSE IF (KOPT .EQ. 'WINTER') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'SPRING') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'SUMMER') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'FALL') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'ANNUAL') THEN
            IOPT = 17
         ELSE IF (KOPT .EQ. 'QUART1') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'QUART2') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'QUART3') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'QUART4') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'AVEPER1') THEN
            IOPT = 1
            AVEPER(IOPT) = 'PER  1'
         ELSE IF (KOPT .EQ. 'AVEPER2') THEN
            IOPT = 2
            AVEPER(IOPT) = 'PER  2'
         ELSE IF (KOPT .EQ. 'AVEPER3') THEN
            IOPT = 3
            AVEPER(IOPT) = 'PER  3'
         ELSE IF (KOPT .EQ. 'AVEPER4') THEN
            IOPT = 4
            AVEPER(IOPT) = 'PER  4'
         ELSE IF (KOPT .EQ. 'AVEPER5') THEN
            IOPT = 5
            AVEPER(IOPT) = 'PER  5'
         ELSE IF (KOPT .EQ. 'AVEPER6') THEN
            IOPT = 6
            AVEPER(IOPT) = 'PER  6'
         ELSE IF (KOPT .EQ. 'AVEPER7') THEN
            IOPT = 7
            AVEPER(IOPT) = 'PER  7'
         ELSE IF (KOPT .EQ. 'AVEPER8') THEN
            IOPT = 8
            AVEPER(IOPT) = 'PER  8'
         ELSE IF (KOPT .EQ. 'AVEPER9') THEN
            IOPT = 9
            AVEPER(IOPT) = 'PER  9'
         ELSE IF (KOPT .EQ. 'AVEPER10') THEN
            IOPT = 10
            AVEPER(IOPT) = 'PER 10'
         ELSE IF (KOPT .EQ. 'AVEPER11') THEN
            IOPT = 11
            AVEPER(IOPT) = 'PER 11'
         ELSE IF (KOPT .EQ. 'AVEPER12') THEN
            IOPT = 12
            AVEPER(IOPT) = 'PER 12'
         ELSE
C           WRITE Error Message           ! Invalid Average Period Field
            CALL ERRHDL(PATH,MODNAM,'E','203','AVEPER')
C           Exit to END
            GO TO 999
         END IF

         IF (KAVE(IOPT) .NE. 1) THEN
C           WRITE Warning Message:  Averaging Period Not Being Calculated
            CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
         ELSE IF (IOPT .GT. 12 .AND. IOPT .LT. 17) THEN
            IF (SEASON .AND. KOPT.NE.'WINTER' .AND. KOPT.NE.'SPRING'
     &                 .AND. KOPT.NE.'SUMMER' .AND. KOPT.NE.'FALL') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE IF (QUARTR .AND.KOPT.NE.'QUART1'.AND.KOPT.NE.'QUART2'
     &                  .AND.KOPT.NE.'QUART3'.AND.KOPT.NE.'QUART4') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE
C              Set Switch To Indicate That Average Temperatures Have Been
C              Input For This Averaging Period.
               IAVETA(IOPT) = 1
            END IF
         ELSE
C           Set Switch To Indicate That Average Temperatures Have Been
C           Input For This Averaging Period.
            IAVETA(IOPT) = 1
         END IF

C        Fill AVETA Array
         DO 100 I = 4, IFC
            CALL STONUM(FIELD(I),40,ZNUM,IDUM)
C           Check The Numerical Field
            IF (IDUM .EQ. -1) THEN
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               GO TO 100
            END IF
            IF (ZNUM.LE.0.0 .OR. ZNUM.GT.400.0) THEN
C              WRITE Error Message          ! AVETA Out of Range
               CALL ERRHDL(PATH,MODNAM,'E','380',KEYWRD)
            ELSE
               DO 90 J = 1, IDUM
                  IST = IST + 1
                  IF (IST .LE. 6) THEN
                     AVETA(IOPT,IST) = ZNUM
                  ELSE
C                    WRITE Error Message    ! Too Many Values
                     CALL ERRHDL(PATH,MODNAM,'E','231',KEYWRD)
                  END IF
 90            CONTINUE
            END IF
 100     CONTINUE
      END IF

 999  CONTINUE

      RETURN
      END

      SUBROUTINE AVEMHT
C***********************************************************************
C                 AVEMHT Module of ISC Model
C
C     PURPOSE:    Processes Input For Mixing Height Data
C
C     PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of Mixing Hight for Each Stability and Average
C                 Period
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C                       Checks for Invalid Stability Class Indicator
C
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER KOPT*8

C     Variable Initializations
      MODNAM = 'AVEMHT'

      IF (IFC .GT. 10) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE IF (IFC .LT. 5) THEN
C        WRITE Error Message           ! Not Enough Parameters
         CALL ERRHDL(PATH,MODNAM,'E','201',KEYWRD)
      ELSE IF (IFC .GE. 4 .AND. IFC .LE. 10) THEN
C        Determine Average Period from FIELD(3)
         KOPT = FIELD(3)
         IF (KOPT .EQ. 'JAN') THEN
            IOPT = 1
         ELSE IF (KOPT .EQ. 'FEB') THEN
            IOPT = 2
         ELSE IF (KOPT .EQ. 'MAR') THEN
            IOPT = 3
         ELSE IF (KOPT .EQ. 'APR') THEN
            IOPT = 4
         ELSE IF (KOPT .EQ. 'MAY') THEN
            IOPT = 5
         ELSE IF (KOPT .EQ. 'JUN') THEN
            IOPT = 6
         ELSE IF (KOPT .EQ. 'JUL') THEN
            IOPT = 7
         ELSE IF (KOPT .EQ. 'AUG') THEN
            IOPT = 8
         ELSE IF (KOPT .EQ. 'SEP') THEN
            IOPT = 9
         ELSE IF (KOPT .EQ. 'OCT') THEN
            IOPT = 10
         ELSE IF (KOPT .EQ. 'NOV') THEN
            IOPT = 11
         ELSE IF (KOPT .EQ. 'DEC') THEN
            IOPT = 12
         ELSE IF (KOPT .EQ. 'WINTER') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'SPRING') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'SUMMER') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'FALL') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'ANNUAL') THEN
            IOPT = 17
         ELSE IF (KOPT .EQ. 'QUART1') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'QUART2') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'QUART3') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'QUART4') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'AVEPER1') THEN
            IOPT = 1
            AVEPER(IOPT) = 'PER  1'
         ELSE IF (KOPT .EQ. 'AVEPER2') THEN
            IOPT = 2
            AVEPER(IOPT) = 'PER  2'
         ELSE IF (KOPT .EQ. 'AVEPER3') THEN
            IOPT = 3
            AVEPER(IOPT) = 'PER  3'
         ELSE IF (KOPT .EQ. 'AVEPER4') THEN
            IOPT = 4
            AVEPER(IOPT) = 'PER  4'
         ELSE IF (KOPT .EQ. 'AVEPER5') THEN
            IOPT = 5
            AVEPER(IOPT) = 'PER  5'
         ELSE IF (KOPT .EQ. 'AVEPER6') THEN
            IOPT = 6
            AVEPER(IOPT) = 'PER  6'
         ELSE IF (KOPT .EQ. 'AVEPER7') THEN
            IOPT = 7
            AVEPER(IOPT) = 'PER  7'
         ELSE IF (KOPT .EQ. 'AVEPER8') THEN
            IOPT = 8
            AVEPER(IOPT) = 'PER  8'
         ELSE IF (KOPT .EQ. 'AVEPER9') THEN
            IOPT = 9
            AVEPER(IOPT) = 'PER  9'
         ELSE IF (KOPT .EQ. 'AVEPER10') THEN
            IOPT = 10
            AVEPER(IOPT) = 'PER 10'
         ELSE IF (KOPT .EQ. 'AVEPER11') THEN
            IOPT = 11
            AVEPER(IOPT) = 'PER 11'
         ELSE IF (KOPT .EQ. 'AVEPER12') THEN
            IOPT = 12
            AVEPER(IOPT) = 'PER 12'
         ELSE
C           WRITE Error Message           ! Invalid Average Period Field
            CALL ERRHDL(PATH,MODNAM,'E','203','AVEPER')
C           Exit to END
            GO TO 999
         END IF

         IF (KAVE(IOPT) .NE. 1) THEN
C           WRITE Warning Message:  Averaging Period Not Being Calculated
            CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
         ELSE IF (IOPT .GT. 12 .AND. IOPT .LT. 17) THEN
            IF (SEASON .AND. KOPT.NE.'WINTER' .AND. KOPT.NE.'SPRING'
     &                 .AND. KOPT.NE.'SUMMER' .AND. KOPT.NE.'FALL') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE IF (QUARTR .AND.KOPT.NE.'QUART1'.AND.KOPT.NE.'QUART2'
     &                  .AND.KOPT.NE.'QUART3'.AND.KOPT.NE.'QUART4') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE
C              Set Switch To Indicate That Average Mixing Heights Have Been
C              Input For This Averaging Period.
               IAVEZI(IOPT) = 1
            END IF
         ELSE
C           Set Switch To Indicate That Average Mixing Heights Have Been
C           Input For This Averaging Period.
            IAVEZI(IOPT) = 1
         END IF

         IWS = 0
C        Determine Stability Category Index from FIELD(4); Accepts Either
C        Character Inputs (A-F) or Numeric Inputs (1-6).
         IF (FIELD(4) .EQ. 'A' .OR. FIELD(4) .EQ. '1') THEN
            IST = 1
         ELSE IF (FIELD(4) .EQ. 'B' .OR. FIELD(4) .EQ. '2') THEN
            IST = 2
         ELSE IF (FIELD(4) .EQ. 'C' .OR. FIELD(4) .EQ. '3') THEN
            IST = 3
         ELSE IF (FIELD(4) .EQ. 'D' .OR. FIELD(4) .EQ. '4') THEN
            IST = 4
         ELSE IF (FIELD(4) .EQ. 'E' .OR. FIELD(4) .EQ. '5') THEN
            IST = 5
         ELSE IF (FIELD(4) .EQ. 'F' .OR. FIELD(4) .EQ. '6') THEN
            IST = 6
         ELSE
C           WRITE Error Message           ! Invalid Stability Class Indicator
            CALL ERRHDL(PATH,MODNAM,'E','203',' STAB ')
C           Exit to END
            GO TO 999
         END IF

C        Fill AVEZI Array
         DO 100 I = 5, IFC
            CALL STONUM(FIELD(I),40,ZNUM,IDUM)
C           Check The Numerical Field
            IF (IDUM .EQ. -1) THEN
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               GO TO 100
            END IF
            IF (ZNUM .LT. 0.0 .OR. ZNUM .GT. 10000.0) THEN
C              WRITE Error Message       ! AVEZI Out of Range
               CALL ERRHDL(PATH,MODNAM,'E','380',KEYWRD)
            ELSE
               DO 90 J = 1,IDUM
                  IWS = IWS + 1
                  IF (IWS .LE. 7) THEN
                     AVEZI(IOPT,IST,IWS) = ZNUM
                  ELSE
C                    WRITE Error Message    ! Too Many Values
                     CALL ERRHDL(PATH,MODNAM,'E','231',KEYWRD)
                  END IF
 90            CONTINUE
            END IF
 100     CONTINUE

      END IF

 999  RETURN
      END

      SUBROUTINE AVERGH                                                 DTB93334
C***********************************************************************
C                 AVERGH Module of ISC Model
C
C     PURPOSE:    Processes Input For Roughness Length Data
C
C     PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    February 15, 1993
C
C     ADAPTED FROM -----------------------------------------------
C                 AVEMHT Module of ISC Model
C
C     PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C     ------------------------------------------------------------
C
C     INPUTS:     Input Runstream Image Parameters
C
C     OUTPUT:     Array of z0 for each Average Period
C
C     CALLED FROM:   MECARD
C
C     ERROR HANDLING:   Checks for No Parameters;
C                       Checks for Too Many Parameters;
C                       Checks for Invalid Numeric Fields;
C
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER KOPT*8

C     Variable Initializations
      MODNAM = 'AVERGH'

      IF (IFC .GT. 4) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE IF (IFC .LT. 4) THEN
C        WRITE Error Message           ! Not Enough Parameters
         CALL ERRHDL(PATH,MODNAM,'E','201',KEYWRD)
      ELSE
C        Determine Average Period from FIELD(3)
         KOPT = FIELD(3)
         IF (KOPT .EQ. 'JAN') THEN
            IOPT = 1
         ELSE IF (KOPT .EQ. 'FEB') THEN
            IOPT = 2
         ELSE IF (KOPT .EQ. 'MAR') THEN
            IOPT = 3
         ELSE IF (KOPT .EQ. 'APR') THEN
            IOPT = 4
         ELSE IF (KOPT .EQ. 'MAY') THEN
            IOPT = 5
         ELSE IF (KOPT .EQ. 'JUN') THEN
            IOPT = 6
         ELSE IF (KOPT .EQ. 'JUL') THEN
            IOPT = 7
         ELSE IF (KOPT .EQ. 'AUG') THEN
            IOPT = 8
         ELSE IF (KOPT .EQ. 'SEP') THEN
            IOPT = 9
         ELSE IF (KOPT .EQ. 'OCT') THEN
            IOPT = 10
         ELSE IF (KOPT .EQ. 'NOV') THEN
            IOPT = 11
         ELSE IF (KOPT .EQ. 'DEC') THEN
            IOPT = 12
         ELSE IF (KOPT .EQ. 'WINTER') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'SPRING') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'SUMMER') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'FALL') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'ANNUAL') THEN
            IOPT = 17
         ELSE IF (KOPT .EQ. 'QUART1') THEN
            IOPT = 13
         ELSE IF (KOPT .EQ. 'QUART2') THEN
            IOPT = 14
         ELSE IF (KOPT .EQ. 'QUART3') THEN
            IOPT = 15
         ELSE IF (KOPT .EQ. 'QUART4') THEN
            IOPT = 16
         ELSE IF (KOPT .EQ. 'AVEPER1') THEN
            IOPT = 1
            AVEPER(IOPT) = 'PER  1'
         ELSE IF (KOPT .EQ. 'AVEPER2') THEN
            IOPT = 2
            AVEPER(IOPT) = 'PER  2'
         ELSE IF (KOPT .EQ. 'AVEPER3') THEN
            IOPT = 3
            AVEPER(IOPT) = 'PER  3'
         ELSE IF (KOPT .EQ. 'AVEPER4') THEN
            IOPT = 4
            AVEPER(IOPT) = 'PER  4'
         ELSE IF (KOPT .EQ. 'AVEPER5') THEN
            IOPT = 5
            AVEPER(IOPT) = 'PER  5'
         ELSE IF (KOPT .EQ. 'AVEPER6') THEN
            IOPT = 6
            AVEPER(IOPT) = 'PER  6'
         ELSE IF (KOPT .EQ. 'AVEPER7') THEN
            IOPT = 7
            AVEPER(IOPT) = 'PER  7'
         ELSE IF (KOPT .EQ. 'AVEPER8') THEN
            IOPT = 8
            AVEPER(IOPT) = 'PER  8'
         ELSE IF (KOPT .EQ. 'AVEPER9') THEN
            IOPT = 9
            AVEPER(IOPT) = 'PER  9'
         ELSE IF (KOPT .EQ. 'AVEPER10') THEN
            IOPT = 10
            AVEPER(IOPT) = 'PER 10'
         ELSE IF (KOPT .EQ. 'AVEPER11') THEN
            IOPT = 11
            AVEPER(IOPT) = 'PER 11'
         ELSE IF (KOPT .EQ. 'AVEPER12') THEN
            IOPT = 12
            AVEPER(IOPT) = 'PER 12'
         ELSE
C           WRITE Error Message           ! Invalid Average Period Field
            CALL ERRHDL(PATH,MODNAM,'E','203','AVEPER')
C           Exit to END
            GO TO 999
         END IF

         IF (KAVE(IOPT) .NE. 1) THEN
C           WRITE Warning Message:  Averaging Period Not Being Calculated
            CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
         ELSE IF (IOPT .GT. 12 .AND. IOPT .LT. 17) THEN
            IF (SEASON .AND. KOPT.NE.'WINTER' .AND. KOPT.NE.'SPRING'
     &                 .AND. KOPT.NE.'SUMMER' .AND. KOPT.NE.'FALL') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE IF (QUARTR .AND.KOPT.NE.'QUART1'.AND.KOPT.NE.'QUART2'
     &                  .AND.KOPT.NE.'QUART3'.AND.KOPT.NE.'QUART4') THEN
C              WRITE Warning Message:  Averaging Period Not Being Calculated
               CALL ERRHDL(PATH,MODNAM,'W','355',KOPT)
            ELSE
C              Set Switch To Indicate That Average z0 Values Have Been
C              Input For This Averaging Period.
               IAVEZ0M(IOPT) = 1
            END IF
         ELSE
C           Set Switch To Indicate That Average z0 Values Have Been
C           Input For This Averaging Period.
            IAVEZ0M(IOPT) = 1
         END IF

C        Fill AVEZ0M Array
            CALL STONUM(FIELD(4),40,ZNUM,IDUM)
C           Check The Numerical Field
            IF (IDUM .EQ. -1) THEN
               CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
               GO TO 100
            END IF
            IF (ZNUM .LT. 0.0 .OR. ZNUM .GT. 10.0) THEN
C              WRITE Error Message       ! AVEZ0M Out of Range
               CALL ERRHDL(PATH,MODNAM,'E','380',KEYWRD)
            ELSE                                                        DTB93308
               AVEZ0M(IOPT) = ZNUM                                      DTB93308
            END IF
 100     CONTINUE

      END IF

 999  RETURN
      END                                                               DTB93334


      SUBROUTINE MEOPEN
C***********************************************************************
C                 MEOPEN Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Open The Input file for Hourly Meteorological Data,
C                 And Check Header Record
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Meteorology File Specifications
C
C        OUTPUTS: File OPEN Error Status
C
C        CALLED FROM:   SETUP
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'MEOPEN'

C     File Unit Initialized in BLOCK DATA INIT
C     File Name and Format Set By Keyword "INPUTFIL" on "ME" pathway
C     OPEN Met Data File

      OPEN(UNIT=MFUNIT,ERR=999,FILE=METINP,FORM='FORMATTED',
     &     IOSTAT=IOERRN,STATUS='OLD')

      GO TO 1000

C     Write Out Error Message for File OPEN Error
 999  CALL ERRHDL(PATH,MODNAM,'E','500',' MET-INP')

 1000 RETURN
      END
