      SUBROUTINE DELH(DHFOUT)
C***********************************************************************
C                 DELH Module of ISC2 Model
C
C        PURPOSE: To Calculate Final Plume Rise
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Source Parameters
C                 Meteorological Variables
C                 Buoyancy and Momentum Fluxes
C
C        OUTPUTS: Final Plume Rise, DHFOUT (m)
C
C        CALLED FROM:   PHEFF
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DELH'

C     Calculate Delta-T
      DELT = TS - TA

      IF (UNSTAB .OR. NEUTRL) THEN
         IF (FB .GE. 55.) THEN
            DTCRIT = 0.00575*TS*((VS*VS)/DS)**0.333333
         ELSE   
            DTCRIT = 0.0297*TS*(VS/(DS*DS))**0.333333
         END IF
         IF (DELT .GE. DTCRIT) THEN
            BUOYNT = .TRUE. 
         ELSE   
            BUOYNT = .FALSE.
         END IF
         IF (BUOYNT) THEN   
            IF (FB .GE. 55.) THEN
               DHFOUT = 38.71*(FB**0.6)/US
            ELSE
               DHFOUT = 21.425*(FB**0.75)/US
            END IF
         ELSE   
            DHFOUT = 3.*DS*VS/US
         END IF

      ELSE IF (STABLE) THEN
         DTCRIT = 0.019582*VS*TA*RTOFS
         IF (DELT .GE. DTCRIT) THEN
            BUOYNT = .TRUE. 
         ELSE   
            BUOYNT = .FALSE.
         END IF
         IF (BUOYNT) THEN   
            DHFOUT = 2.6*(FB/(US*S))**0.333333
C           Compare to Final Plume Rise for Calm Winds, DHCLM
            DHCLM = 4.*FB**0.25/S**0.375
            IF (DHCLM .LT. DHFOUT) DHFOUT = DHCLM
         ELSE   
            DHFOUT = 1.5*(FM/(US*RTOFS))**0.333333
C           Compare to Maximum Momentum Rise for UNSTABLE/NEUTRAL, DHCHK
            DHCHK = 3.*DS*VS/US
            IF (DHCHK .LT. DHFOUT) DHFOUT = DHCHK
         END IF
      END IF

      RETURN
      END

      FUNCTION HSPRIM(US,VS,HS,DS)
C***********************************************************************
C                 HSPRIM Module of the ISC Model - Version 2
C
C        PURPOSE: Calculates Stack Height Adjusted for Stack
C                 Tip Downwash (HS')
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Wind Speed Adjusted to Stack Height
C
C        OUTPUTS: Adjusted Stack Height (m)
C
C        CALLED FROM:   PHEFF
C***********************************************************************

C     Variable Declarations
      CHARACTER MODNAM*6
C     Variable Initializations
      MODNAM = 'HSPRIM'

C     Calculate Adjusted Stack Height (Eqn. 1-7)

      IF (VS .LT. 1.5*US) THEN
         HSPRIM = HS - 2.*DS*(1.5-VS/US)
      ELSE
         HSPRIM = HS
      END IF

      IF (HSPRIM .LT. 0.0)  HSPRIM = 0.0

      RETURN
      END

      SUBROUTINE DHPHS(XARG,DHFARG,DHPOUT)
C***********************************************************************
C                 DHPHS Module of ISC2 Model
C
C        PURPOSE: Calculates Distance-dependent Plume Rise for
C                 Huber-Snyder Downwash Algorithm and for BID
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling arguments
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C
C        OUTPUTS: Distance-dependent Plume Rise, DHPOUT (m)
C
C        CALLED FROM:   PHEFF
C                       BID
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DHPHS'

C     Calculate Distance-dependent Buoyant Plume Rise (Eqn. 1-22)
      XP = AMIN1(XARG,XFB)
      IF (XP .LT. 1.0)  XP = 1.0
      IF (FB .LT. 1.0E-10)  FB = 1.0E-10
      DHPB = 1.60 * (FB*XP*XP)**0.333333 / US

C     Calculate Dist-dependent Momentum Plume Rise          ---   CALL DHPMOM
      CALL DHPMOM(XARG)

C     Select Maximum of Buoyant or Momentum Rise for Gradual Rise
      DHPOUT = AMAX1(DHPB, DHPM)
C     Compare to Final Rise and Select Smaller Value for Gradual Rise
      DHPOUT = AMIN1(DHPOUT, DHFARG)

      RETURN
      END

      SUBROUTINE DHPMOM(XARG)
C***********************************************************************
C                 DHPMOM Module of ISC2 Model
C
C        PURPOSE: Calculates Distance-dependent Momentum Plume Rise
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C
C        OUTPUTS: Distance-dependent Momentum Plume Rise, DHPM (m)
C
C        CALLED FROM:   WAKFLG
C                       DHPHS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DHPMOM'

C     Calculate BETAJ Parameter (Entrainment Coefficient)
      BETAJ = 0.333333 + US/VS

      IF (UNSTAB .OR. NEUTRL) THEN
         XP = AMIN1(XARG,XFM)
         DHPM = (3.*FM*XP/(BETAJ*BETAJ*US*US))**0.333333
      ELSE IF (STABLE) THEN
         XP = AMIN1(XARG,XFM)
         DHPM = 3.*FM*SIN(RTOFS*XP/US) / (BETAJ*BETAJ*US*RTOFS)
C        Set Lower Limit for DHPM to Avoid Negative Arg for Cube Root
         DHPM = AMAX1(1.0E-10, DHPM)
         DHPM = DHPM ** 0.333333
      END IF

C     Do Not Let Gradual Rise Exceed Final Momentum Rise
      DHPM = AMIN1(DHPM, 3.*DS*VS/US)

      RETURN
      END

      SUBROUTINE DHPSS(XARG,DHPOUT)
C***********************************************************************
C                 DHPSS Module of ISC2 Model
C
C        PURPOSE: Calculates Distance-dependent Plume Rise for
C                 Schulman-Scire Downwash Algorithm
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        MODIFIED:   To Change TOL from 1.0E-5 to 1.0E-4 - 9/29/92
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Wake Plume Height, HEMWAK
C
C        OUTPUTS: Distance-dependent BLP Plume Rise, DHPOUT (m)
C
C        CALLED FROM:   PHEFF
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DHPSS'

C     Determine BLP Line Source Parameters, ZLY and RINIT

      X3LB = 3.*ZLB
      IF (HEMWAK .GT. 1.2*DSBH) THEN
         ZLY = 0.0
C        Calculate Building Enhanced Sigma-z at X3LB        ---   CALL SZENH
         CALL SZENH(X3LB,SZ3LB,XZ3LB)
      ELSE
C        Calculate Building Enhanced Sigma-y at X3LB        ---   CALL SYENH
         CALL SYENH(X3LB,SY3LB,XY3LB)
C        Calculate Building Enhanced Sigma-z at X3LB        ---   CALL SZENH
         CALL SZENH(X3LB,SZ3LB,XZ3LB)
         IF (SY3LB .GE. SZ3LB) THEN
C           Note That SRT2PI = SQRT(2.*PI)
            ZLY = SRT2PI*(SY3LB-SZ3LB)
         ELSE
            ZLY = 0.0
         END IF
      END IF
C     Calculate Initial Radius of Plume, RINIT = SQRT(2.)*SZ
      RINIT = 1.414214 * SZ3LB

C     Determine Coefficients A, B and C of Cubic Equation

      A = 3.*ZLY/(PI*BETA) + 3.*RINIT/BETA
      B = 6.*RINIT*ZLY/(PI*BETA*BETA) + 3.*RINIT*RINIT/(BETA*BETA)
C     Compute Coefficient C for Buoyant Rise (CB)           ---   CALL BLPCB
      CALL BLPCB(XARG)
C     Compute Coefficient C for Momentum Rise (CM)          ---   CALL BLPCM
      CALL BLPCM(XARG)

C     Solve Cubic Equation With Buoyant Rise (CB) and Momentum Rise (CM)
C     and Select the Larger of the Two as the Gradual Plume Rise, DHP.
C     Set TOLerance Limit to 1.0E-4, and Initial Guess to Cube Root of C.
      TOL = 1.0E-4

C     First Check For Non-zero CB, To Avoid Zero-divide in CUBIC
      IF (CB .LT. -1.0E-5) THEN
         ZINIT = ABS(CB) ** 0.333333
      ELSE
         CB = -1.0E-5
         ZINIT = 0.01
      END IF
C     Solve Cubic Equation for Buoyant Rise, ZB             ---   CALL CUBIC
      CALL CUBIC(A,B,CB,ZINIT,TOL,ZB)

C     First Check For Non-zero CM, To Avoid Zero-divide in CUBIC
      IF (CM .LT. -1.0E-5) THEN
         ZINIT = ABS(CM) ** 0.333333
      ELSE
         CM = -1.0E-5
         ZINIT = 0.01
      END IF
C     Solve Cubic Equation for Momentum Rise, ZM            ---   CALL CUBIC
      CALL CUBIC(A,B,CM,ZINIT,TOL,ZM)

      DHPOUT = AMAX1(ZB,ZM)

      RETURN
      END

      SUBROUTINE BLPCB(XARG)
C***********************************************************************
C                 BLPCB Module of ISC2 Model
C
C        PURPOSE: Calculates C Coefficient for BLP Buoyant Rise Used in
C                 Schulman-Scire Downwash Algorithm
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C
C        OUTPUTS: Coefficient CB
C
C        CALLED FROM:   DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'BLPCB'

C     Compute Coefficient for Buoyant BLP Rise

      IF (UNSTAB .OR. NEUTRL) THEN
         XP = AMIN1(XARG,XFB)
         CB = -3.*FB*XP*XP/(2.*BETA*BETA*US*US*US)
      ELSE IF (STABLE) THEN
         XP = AMIN1(XARG,XFB)
         CBS = 6.*FB/(BETA*BETA*US*S)
C        Compare Stable Term to Neutral Term
         CBN = 3.*FB*XP*XP/(2.*BETA*BETA*US*US*US)
C        Select Minimum of Stable and Neutral Term
         CB = -1.*AMIN1(CBS,CBN)
      END IF

      RETURN
      END

      SUBROUTINE BLPCM(XARG)
C***********************************************************************
C                 BLPCM Module of ISC2 Model
C
C        PURPOSE: Calculates C Coefficient for BLP Momentum Rise Used in
C                 Schulman-Scire Downwash Algorithm
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C
C        OUTPUTS: Coefficient CM
C
C        CALLED FROM:   DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'BLPCM'

C     Calculate BETAJ Parameter (Entrainment Coefficient)
      BETAJ = 0.333333 + US/VS

C     Compute Coefficient for Momentum BLP Rise

      IF (UNSTAB .OR. NEUTRL) THEN
         XP = AMIN1(XARG,XFM)
         CM = -3.*FM*XP/(BETAJ*BETAJ*US*US)
      ELSE IF (STABLE) THEN
         XP = AMIN1(XARG,XFM)
         CMS = 3.*FM*SIN(RTOFS*XP/US)/(BETAJ*BETAJ*US*RTOFS)
C        Compare Stable Term to Neutral Term
         XFMN = 4.*DS*(VS+3.*US)*(VS+3.*US)/(VS*US)
         XP = AMIN1(XARG,XFMN)
         CMN = 3.*FM*XP/(BETAJ*BETAJ*US*US)
C        Select Minimum of Stable and Neutral Term
         CM = -1.*AMIN1(CMS,CMN)
      END IF

      RETURN
      END

      SUBROUTINE CUBIC(A,B,C,ZINIT,TOL,ZITER)
C***********************************************************************
C                 CUBIC Module of ISC2 Model
C
C        PURPOSE: Solves Cubic Equation Using Newton's Method
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Coefficients (A, B and C) of Cubic Equation
c                 Initial Guess for Variable
C                 Tolerance Level for Iteration
C
C        OUTPUTS: Solution to Cubic Equation;
C                    Z**3 + A*Z**2 + B*Z + C = 0
C
C        CALLED FROM:   DHPSS
C                       XVZ
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL Z(25)

C     Variable Initializations
      MODNAM = 'CUBIC'

C     Assign Initial Guess to Z(1)
      Z(1) = ZINIT

C     Begin Iterative LOOP (24 iterations)
      DO 20 N = 1, 24
C        Calculate Cubic Function and First Derivative With Current Guess
         FZ = Z(N)*Z(N)*Z(N) + A*Z(N)*Z(N) + B*Z(N) + C
         FP = 3.*Z(N)*Z(N) + 2.*A*Z(N) + B
C        Calculate New Guess
         Z(N+1) = Z(N) - FZ/FP
C        Check successive iterations for specified tolerance level
         IF (ABS(Z(N+1) - Z(N)) .LE. TOL) THEN
            ZITER = Z(N+1)
C           Exit Loop
            GO TO 999
         END IF
 20   CONTINUE
C     End Iterative LOOP

C     If No Convergence In Loop, Then Use Average of Last Two Estimates,
C     and Write Information Message
      WRITE(DUMMY,'(4I2.2)') KST, IFVSEC, IWS, IAVE
      CALL ERRHDL(PATH,MODNAM,'I','400',DUMMY)
      ZITER = 0.5 * (Z(24) + Z(25))

 999  RETURN
      END
