      SUBROUTINE LTXYP(XARG,XYOUT)
C***********************************************************************
C                 LTXYP Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Lateral Virtual Distance for POINT Sources
C                 Including Building Enhanced Dispersion
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance
C                 Building Dimensions
C                 Stability Class
C
C        OUTPUTS: Lateral Virtual Distance, XY
C
C        CALLED FROM:  PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'LTXYP'

      IF (WAKE .AND. HEMWAK .LE. 1.2*DSBH) THEN
         XB = AMIN1(ABS(XARG), 10.*ZLB)
C        Calculate Building Enhanced Sigma-y at XB          ---   CALL SYENH
         CALL SYENH(XB,SYOUT,XYOUT)
         XRAD = 2.15*SYOUT
         XYOUT = AMAX1( 0.0, XRAD/TAN(0.5*DELTHP) - 10.*ZLB)
      ELSE
         XYOUT = 0.0
         XRAD = 0.0
      END IF

      RETURN
      END

      SUBROUTINE SIGY(XARG,SYOUT)
C***********************************************************************
C                 SIGY Module of ISC2 Model
C
C        PURPOSE: Calculates Sigma-y Values From Dispersion Curves
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Downwind Distance
C                 Stability Class
C                 Rural or Urban Dispersion Option
C
C        OUTPUTS: Lateral Dispersion Coefficient, SYOUT
C
C        CALLED FROM:   PDIS
C                       VDIS
C                       ADIS
C                       SYENH
C                       DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SIGY'

C     Convert Distance to km
      XKM = XARG * 0.001

C     Determine Sigma-y Based on RURAL/URBAN, Stability Class, and Distance.
C     Stability Classes are Checked in the Order 4, 5, 6, 1, 2, 3
C     For Optimization, Since Neutral and Stable are Generally the Most
C     Frequent Classes.

      IF (RURAL) THEN
         IF (KST .EQ. 4) THEN
            TH = (8.3330 - 0.72382*ALOG(XKM)) * DTORAD
         ELSE IF (KST .EQ. 5) THEN
            TH = (6.25 - 0.54287*ALOG(XKM)) * DTORAD
         ELSE IF (KST .EQ. 6) THEN
            TH = (4.1667 - 0.36191*ALOG(XKM)) * DTORAD
         ELSE IF (KST .EQ. 1) THEN
            TH = (24.1667 - 2.5334*ALOG(XKM)) * DTORAD
         ELSE IF (KST .EQ. 2) THEN
            TH = (18.333 - 1.8096*ALOG(XKM)) * DTORAD
         ELSE IF (KST .EQ. 3) THEN
            TH = (12.5 - 1.0857*ALOG(XKM)) * DTORAD
         END IF
C
C        NOTE THAT 465.11628 = 1000. (m/km) / 2.15
C
         SYOUT = 465.11628 * XKM * TAN(TH)
      ELSE IF (URBAN) THEN
         IF (KST .EQ. 4) THEN
            SYOUT = 160.*XKM/SQRT(1.+0.4*XKM)
         ELSE IF (KST .GE. 5) THEN
            SYOUT = 110.*XKM/SQRT(1.+0.4*XKM)
         ELSE IF (KST .LE. 2) THEN
            SYOUT = 320.*XKM/SQRT(1.+0.4*XKM)
         ELSE IF (KST .EQ. 3) THEN
            SYOUT = 220.*XKM/SQRT(1.+0.4*XKM)
         END IF
      END IF

      RETURN
      END

      SUBROUTINE SIGZ(XARG,SZOUT)
C***********************************************************************
C                 SIGZ Module of ISC2 Model
C
C        PURPOSE: Calculates Sigma-z Values From Dispersion Curves
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Downwind Distance
C                 Stability Class
C                 Rural or Urban Dispersion Option
C
C        OUTPUTS: Vertical Dispersion Coefficient, SZOUT
C
C        CALLED FROM:   PDIS
C                       VDIS
C                       ADIS
C                       SZENH
C                       DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SIGZ'

C     Convert Distance to km
      XKM = XARG * 0.001

C     Determine Sigma-z Based on RURAL/URBAN, Stability Class, and Distance.
C     Stability Classes are Checked in the Order 4, 5, 6, 1, 2, 3
C     For Optimization, Since Neutral and Stable are Generally the Most
C     Frequent Classes.

      IF (RURAL) THEN
C        Retrieve Coefficients, A and B                     ---   CALL SZCOEF
         CALL SZCOEF(XKM,A,B,XMIN,XMAX)
         SZOUT = A*XKM**B
      ELSE IF (URBAN) THEN
         IF (KST .EQ. 4) THEN
            SZOUT = 140.*XKM/SQRT(1.+0.3*XKM)
         ELSE IF (KST .GE. 5) THEN
            SZOUT = 80.*XKM/SQRT(1.+1.5*XKM)
         ELSE IF (KST .LE. 2) THEN
            SZOUT = 240.*XKM*SQRT(1.+XKM)
         ELSE IF (KST .EQ. 3) THEN
            SZOUT = 200.*XKM
         END IF
      END IF

      RETURN
      END

      SUBROUTINE SZCOEF(XKM,A,B,XMIN,XMAX)
C***********************************************************************
C                 SZCOEF Module of ISC2 Model
C
C        PURPOSE: Determines Coefficients and Ranges for Rural Sigma-z
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  KST     Stability Category
C                 XKM     Downwind Distance (km)
C
C        OUTPUTS: Coefficients A and B and Distance Range XMIN and XMAX
C
C        CALLED FROM:   SIGZ
C                       XVZ
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SZCOEF'

      IF (KST .EQ. 4) THEN
         IF (XKM .LE. .30) THEN
            A = 34.459
            B = 0.86974
            XMIN = 0.
            XMAX = 0.30
         ELSE IF (XKM .LE. 1.0) THEN
            A = 32.093
            B = 0.81066
            XMIN = 0.30
            XMAX = 1.
         ELSE IF (XKM .LE. 3.0) THEN
            A = 32.093
            B = 0.64403
            XMIN = 1.
            XMAX = 3.
         ELSE IF (XKM .LE. 10.) THEN
            A = 33.504
            B = 0.60486
            XMIN = 3.
            XMAX = 10.
         ELSE IF (XKM .LE. 30.) THEN
            A = 36.650
            B = 0.56589
            XMIN = 10.
            XMAX = 30.
         ELSE
            A = 44.053
            B = 0.51179
            XMIN = 30.
            XMAX = 100.
         END IF

      ELSE IF (KST .EQ. 5) THEN
         IF (XKM .LE. .10) THEN
            A = 24.26
            B = 0.83660
            XMIN = 0.
            XMAX = .10
         ELSE IF (XKM .LE. .30) THEN
            A = 23.331
            B = 0.81956
            XMIN = 0.10
            XMAX = 0.30
         ELSE IF (XKM .LE. 1.0) THEN
            A = 21.628
            B = 0.75660
            XMIN = 0.30
            XMAX = 1.
         ELSE IF (XKM .LE. 2.0) THEN
            A = 21.628
            B = 0.63077
            XMIN = 1.
            XMAX = 2.
         ELSE IF (XKM .LE. 4.0) THEN
            A = 22.534
            B = 0.57154
            XMIN = 2.
            XMAX = 4.
         ELSE IF (XKM .LE. 10.) THEN
            A = 24.703
            B = 0.50527
            XMIN = 4.
            XMAX = 10.
         ELSE IF (XKM .LE. 20.) THEN
            A = 26.97
            B = 0.46713
            XMIN = 10.
            XMAX = 20.
         ELSE IF (XKM .LE. 40.) THEN
            A = 35.42
            B = 0.37615
            XMIN = 20.
            XMAX = 40.
         ELSE
            A = 47.618
            B = 0.29592
            XMIN = 40.
            XMAX = 100.
         END IF

      ELSE IF (KST .EQ. 6) THEN
         IF (XKM .LE. .20) THEN
            A = 15.209
            B = 0.81558
            XMIN = 0.
            XMAX = 0.20
         ELSE IF (XKM .LE. .70) THEN
            A = 14.457
            B = 0.78407
            XMIN = 0.20
            XMAX = 0.70
         ELSE IF (XKM .LE. 1.0) THEN
            A = 13.953
            B = 0.68465
            XMIN = 0.7
            XMAX = 1.
         ELSE IF (XKM .LE. 2.0) THEN
            A = 13.953
            B = 0.63227
            XMIN = 1.
            XMAX = 2.
         ELSE IF (XKM .LE. 3.0) THEN
            A = 14.823
            B = 0.54503
            XMIN = 2.
            XMAX = 3.
         ELSE IF (XKM .LE. 7.0) THEN
            A = 16.187
            B = 0.46490
            XMIN = 3.
            XMAX = 7.
         ELSE IF (XKM .LE. 15.) THEN
            A = 17.836
            B = 0.41507
            XMIN = 7.
            XMAX = 15.
         ELSE IF (XKM .LE. 30.) THEN
            A = 22.651
            B = 0.32681
            XMIN = 15.
            XMAX = 30.
         ELSE IF (XKM .LE. 60.) THEN
            A = 27.074
            B = 0.27436
            XMIN = 30.
            XMAX = 60.
         ELSE
            A = 34.219
            B = 0.21716
            XMIN = 60.
            XMAX = 100.
         END IF

      ELSE IF (KST .EQ. 1) THEN
         IF (XKM .LE. 0.10) THEN
            A = 122.8
            B = 0.94470
            XMIN = 0.
            XMAX = 0.1
         ELSE IF (XKM .LE. 0.15) THEN
            A = 158.080
            B = 1.05420
            XMIN = 0.1
            XMAX = 0.15
         ELSE IF (XKM .LE. 0.20) THEN
            A = 170.22
            B = 1.09320
            XMIN = 0.15
            XMAX = 0.20
         ELSE IF (XKM .LE. 0.25) THEN
            A = 179.52
            B = 1.12620
            XMIN = 0.20
            XMAX = 0.25
         ELSE IF (XKM .LE. 0.30) THEN
            A = 217.41
            B = 1.2644
            XMIN = 0.25
            XMAX = 0.30
         ELSE IF (XKM .LE. 0.40) THEN
            A = 258.89
            B = 1.4094
            XMIN = 0.30
            XMAX = 0.40
         ELSE IF (XKM .LE. 0.50) THEN
            A = 346.75
            B = 1.72830
            XMIN = 0.40
            XMAX = 0.50
         ELSE
            A = 453.85
            B = 2.11660
            XMIN = 0.50
            XMAX = 100.
         END IF

      ELSE IF (KST .EQ. 2) THEN
         IF (XKM .LE. 0.20) THEN
            A = 90.673
            B = 0.93198
            XMIN = 0.
            XMAX = 0.20
         ELSE IF (XKM .LE. 0.40) THEN
            A = 98.483
            B = 0.98332
            XMIN = 0.20
            XMAX = 0.40
         ELSE
            A = 109.3
            B = 1.0971
            XMIN = 0.40
            XMAX = 100.
         END IF

      ELSE IF (KST .EQ. 3) THEN
            A = 61.141
            B = 0.91465
            XMIN = 0.
            XMAX = 100.
      END IF

      RETURN
      END

      SUBROUTINE SYENH(XARG,SYOUT,XYOUT)
C***********************************************************************
C                 SYENH Module of ISC2 Model
C
C        PURPOSE: Calculates Building Enhanced Sigma-y Values (>= Curves)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Downwind Distance
C                 Building Dimensions
C                 Stability Class
C
C        OUTPUTS: Lateral Dispersion Coefficient, Sigma-y (SYOUT)
C                 Lateral Virtual Distance (XYOUT)
C
C        CALLED FROM:  PDIS
C                      DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SYENH'

      IF (TALL) THEN
         IF (XARG .LT. 10.*ZLB) THEN
C           Calculate Building Enhanced Sigma-y (Eqn. 1-45)
            SY1 = 0.35*ZLB + 0.067*(XARG - 3.*ZLB)
C           Calculate Sigma-y from Dispersion Curves, SY   ---   CALL SIGY
            CALL SIGY(XARG,SYOUT)
            SYOUT = AMAX1(SY1,SYOUT)
         ELSE
C           Calculate Building Enhanced Sigma-y at 10*ZLB
            SYINIT = 0.85*ZLB
CC**********************************************************************
CC         The Following Commented Line Removes Rounding From Estimate
CC         of SYINIT at 10*ZLB.  Rounding in Original ISC Model Causes
CC         Slight Discontinuity at 10*ZLB.
CC
CC            SYINIT = 0.819*ZLB
CC**********************************************************************
C           Calculate Lateral Virtual Distance              ---   CALL XVY
            CALL XVY(XYOUT)
            XYOUT = AMAX1(0.0, (XYOUT-10.*ZLB))
C           Calculate Sigma-y from Curves for X+XY          ---   CALL SIGY
            CALL SIGY(XARG+XYOUT,SYOUT)
         END IF

      ELSE IF (SQUAT) THEN
         IF (XARG .LT. 10.*ZLB) THEN
C           Calculate Buidling Enhanced Sigma-y (Eqn. A-41)
            SY1 = 0.35*DSBW + 0.067*(XARG - 3.*DSBH)
C           Calculate Sigma-y from Dispersion Curves, SY   ---   CALL SIGY
            CALL SIGY(XARG,SYOUT)
            SYOUT = AMAX1(SY1,SYOUT)
         ELSE
C           Calculate Building Enhanced Sigma-y at 10*ZLB
            SYINIT = 0.35*DSBW + 0.5*DSBH
CC**********************************************************************
CC         The Following Commented Line Removes Rounding From Estimate
CC         of SYINIT at 10*ZLB.  Rounding in Original ISC Model Causes
CC         Slight Discontinuity at 10*ZLB.
CC
CC            SYINIT = 0.35*DSBW + 0.469*DSBH
CC**********************************************************************
C           Calculate Lateral Virtual Distance              ---   CALL XVY
            CALL XVY(XYOUT)
            XYOUT = AMAX1(0.0, (XYOUT-10.*ZLB))
C           Calculate Sigma-y from Curves for X+XY          ---   CALL SIGY
            CALL SIGY(XARG+XYOUT,SYOUT)
         END IF

      ELSE IF (SSQUAT) THEN
         IF (XARG .LT. 10.*ZLB) THEN
C           Calculate Building Enhanced Sigma-y
            IF (WAKLOW) THEN
C              Use Eqn. 1-44 for "Lower Bound" Estimate
               SY1 = 1.75*ZLB + 0.067*(XARG - 3.*ZLB)
            ELSE
C              Use Eqn. 1-43 for "Upper Bound" Estimate
               SY1 = 0.35*ZLB + 0.067*(XARG - 3.*ZLB)
            END IF
C           Calculate Sigma-y from Dispersion Curves, SY    ---   CALL SIGY
            CALL SIGY(XARG,SYOUT)
            SYOUT = AMAX1(SY1,SYOUT)
         ELSE
C           Calculate Building Enhanced Sigma-y at 10*ZLB
            IF (WAKLOW) THEN
               SYINIT = 2.25*ZLB
CC**********************************************************************
CC         The Following Commented Line Removes Rounding From Estimate
CC         of SYINIT at 10*ZLB.  Rounding in Original ISC Model Causes
CC         Slight Discontinuity at 10*ZLB.
CC
CC               SYINIT = 2.219*ZLB
CC**********************************************************************
            ELSE
               SYINIT = 0.85*ZLB
CC**********************************************************************
CC         The Following Commented Line Removes Rounding From Estimate
CC         of SYINIT at 10*ZLB.  Rounding in Original ISC Model Causes
CC         Slight Discontinuity at 10*ZLB.
CC
CC               SYINIT = 0.819*ZLB
CC**********************************************************************
            END IF
C           Calculate Lateral Virtual Distance              ---   CALL XVY
            CALL XVY(XYOUT)
            XYOUT = AMAX1(0.0, (XYOUT-10.*ZLB))
C           Calculate Sigma-y from Curves for X+XY          ---   CALL SIGY
            CALL SIGY(XARG+XYOUT,SYOUT)
         END IF
      END IF

      RETURN
      END

      SUBROUTINE SZENH(XARG,SZOUT,XZOUT)
C***********************************************************************
C                 SZENH Module of ISC2 Model
C
C        PURPOSE: Calculates Building Enhanced Sigma-z Values
C                 and Compares to Sigma-z From Dispersion Curves
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Downwind Distance
C                 Stability Class
C                 Rural or Urban Dispersion Option
C                 Wake Plume Height, HEMWAK
C
C        OUTPUTS: Vertical Dispersion Coefficient, Sigma-z (SZOUT)
C                 Vertical Virtual Distance (XZOUT)
C
C        CALLED FROM:   PDIS
C                       DHPSS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SZENH'

C     Calculate Decay Coefficient, DA                       ---   CALL SZDCAY
      CALL SZDCAY

C     Calculate Building Enhanced Sigma-z at 10*ZLB (Eqn. 1-40)
      SZINIT = 1.2 * ZLB * DA
CC**********************************************************************
CC         The Following Commented Line Removes Rounding From Estimate
CC         of SZINIT at 10*ZLB.  Rounding in Original ISC Model Causes
CC         Slight Discontinuity at 10*ZLB.
CC
CC      SZINIT = 1.169 * ZLB * DA
CC**********************************************************************
C     Calculate Vertical Virtual Distance, XZ               ---   CALL XVZ
      CALL XVZ(XARG-10.*ZLB,XZOUT)
      XZOUT = AMAX1(0.0, (XZOUT - 10.*ZLB))

      IF (XARG .LT. 10.*ZLB) THEN
C        Calculate Building Enhanced Sigma-z (Eqn. 1-37 & 1-38)
         SZ1 = (0.7*ZLB + 0.067*(XARG - 3.*ZLB)) * DA
C        Calculate Sigma-z from Curves, SZ2                 ---   CALL SIGZ
         CALL  SIGZ(XARG,SZOUT)
         SZOUT = AMAX1(SZ1,SZOUT)
      ELSE
C        Calculate Sigma-z from Curves using X+XZ           ---   CALL SIGZ
         CALL SIGZ(XARG+XZOUT,SZOUT)
      END IF

      RETURN
      END

      SUBROUTINE XVY(XYOUT)
C***********************************************************************
C                 XVY Module of ISC2 Model
C
C        PURPOSE: Calculates Lateral Virtual Distances
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Initial Dispersion, SYINIT
C                 Stability Class
C                 Rural or Urban Dispersion Option
C
C        OUTPUTS: Lateral Virtual Distance, XYOUT (m)
C
C        CALLED FROM:   VDIS
C                       SYENH
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL A(6),SP(6),SQ(6)

C     Variable Initializations
      DATA A/0.32,0.32,0.22,0.16,0.11,0.11/,
     &     B/0.0004/,
     &     SP/.004781486,.006474168,.009684292,.014649868,.019584802,
     &        0.029481132/,
     &     SQ/1.1235955,1.1086475,1.0905125,1.0881393,1.0857763,
     &        1.0881393/
      MODNAM = 'XVY'

      IF (RURAL) THEN
         XYOUT = (SYINIT*SP(KST))**SQ(KST) * 1000.
      ELSE IF (URBAN) THEN
         A2 = A(KST) * A(KST)
         SY2 = SYINIT * SYINIT
         XYOUT = (B*SY2 + SQRT(B*B*SY2*SY2 + 4.*A2*SY2)) / (2.*A2)
      END IF

      RETURN
      END

      SUBROUTINE XVZ(XARG,XZOUT)
C***********************************************************************
C                 XVZ Module of ISC2 Model
C
C        PURPOSE: Calculates Vertical Virtual Distances
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To use calling argument for output
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        MODIFIED:   To Change TOL from 1.0E-5 to 1.0E-4 - 9/29/92
C
C        INPUTS:  Initial Dispersion, SZINIT
C                 Downwind Distance
C                 Stability Class
C                 Rural or Urban Dispersion Option
C
C        OUTPUTS: Vertical Virtual Distance, XZOUT (m)
C
C        CALLED FROM:   VDIS
C                       SZENH
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL AA(6), BB(6), XXZ(6)

C     Variable Initializations
      DATA AA/0.24,0.24,0.2,0.14,0.08,0.08/
      DATA BB/.001,.001,.0,.0003,.0015,.0015/
      MODNAM = 'XVZ'

      IF (SZINIT .LE. 0.01) THEN
         XZOUT = 0.

      ELSE IF (RURAL) THEN
C        Solve Iteratively
C        Convert Distance to km
         XKM = XARG * 0.001
C        Initial Guess of 10 m
         XXZ(1) = 0.01
         DO 10 N = 1, 5
C           Retrieve Coef. AZ & BZ, Range XMIN & XMAX    ---   CALL SZCOEF
            CALL SZCOEF((XXZ(N)+XKM),AZ,BZ,XMIN,XMAX)
            XXZ(N+1) = (SZINIT/AZ) ** (1./BZ)
C           Check for X+XZ falling within Range of Coefficients
            IF((XXZ(N+1)+XKM).GE.XMIN .AND. (XXZ(N+1)+XKM).LE.XMAX) THEN
               XZOUT = XXZ(N+1) * 1000.
C              EXIT LOOP
               GO TO 999
            END IF
 10      CONTINUE
C        If No Convergence in Loop, Use Smaller of Last Two Estimates,
C        Consistent With Original ISC Model - Version 2
         XZOUT = AMIN1(XXZ(5),XXZ(6)) * 1000.

      ELSE IF (URBAN) THEN
         IF (KST .GE. 4) THEN
            A2  = AA(KST) * AA(KST)
            B   = BB(KST)
            SZ2 = SZINIT * SZINIT
            XZOUT  = (B*SZ2 + SQRT(B*B*SZ2*SZ2 + 4.*A2*SZ2)) / (2.*A2)
         ELSE IF (KST .LE. 2) THEN
C           Set Initial Guess and Tolerance Limit for Cubic Equation
            XZERO = 4. * SZINIT
            TOL = 1.0E-4
C           Set Cubic Coefficients, ACOEF, BCOEF, and CCOEF
            ACOEF = 1./BB(KST)
            BCOEF = 0.0
            CCOEF = -1. * SZINIT*SZINIT/(AA(KST)*AA(KST) * BB(KST))
C           Solve Cubic Equation                          ---   CALL CUBIC
            CALL CUBIC(ACOEF,BCOEF,CCOEF,XZERO,TOL,XZOUT)
         ELSE IF (KST .EQ. 3) THEN
            XZOUT = SZINIT/AA(KST)
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE SZDCAY
C***********************************************************************
C                 SZDCAY Module of ISC2 Model
C
C        PURPOSE: Calculates Linear Decay Coefficient for Sigma-z
C                 Used in Schulman-Scire Building Downwash
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Plume Height (Due to Momentum at X2BH, HEMWAK)
C                 Building Dimensions
C                 Wake Flags
C
C        OUTPUTS: Decay Coefficient
C
C        CALLED FROM:   DHPSS
C                       SZENH
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'SZDCAY'

      IF (WAKESS) THEN
         IF (HEMWAK .LE. DSBH) THEN
            DA = 1.
         ELSE IF (HEMWAK .LE. DSBH+2.*ZLB) THEN
            DA = (DSBH - HEMWAK)/(2.*ZLB) + 1.
         ELSE
            DA = 0.
         END IF
      ELSE
         DA = 1.
      END IF

      RETURN
      END

      SUBROUTINE BIDLT(DHPARG,SZARG,SZOUT,SBOUT)
C***********************************************************************
C                 BID Module of ISC2 Model
C
C        PURPOSE: Applies Bouyancy-Induced Dispersion to
C                 Sigma-y and Sigma-z
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED:   To use calling arguments
C                    R. W. Brode, PES, Inc. - 9/30/94
C
C        MODIFIED BY D. Strimaitis, SRC (add SBID to commons)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Sigma-y
C                 Sigma-z
C                 Downwind Distance
C                 Buoyancy and Momentum Fluxes
C                 Source Parameter Arrays
C
C        OUTPUTS: Sigma-y and Sigma-z Adjusted for BID (SYOUT and SZOUT)
C
C        CALLED FROM:   PDIS
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'BIDLT'

C     Calculate The Coefficients
      SBOUT = DHPARG/3.5
      SBIDSQ = SBOUT*SBOUT
C     Apply BID to Sigma-z
      SZOUT = SQRT(SZARG*SZARG + SBIDSQ)

      RETURN
      END
