      SUBROUTINE AVERTS(XVIN,YVIN,XWD,YWD,NUMV)
C***********************************************************************
C*                AVERTS Module of ISC2 Short Term Model - ISCST3
C*
C*       PURPOSE: Calculates coordinates of vertices for Wind
C*                Direction Coordinate system for OPENPIT sources.
C*
C*       PROGRAMMER: Jeff Wang, Roger Brode
C*       MODIFIED:   Jayant Hardikar, Roger Brode (for OPENPIT sources)
C*
C*       DATE:      July 7, 1993
C*
C*       INPUTS:  Source Coordinates for Specific Source
C*                Number of vertices + 1
C*
C*       OUTPUTS: Array of Vertex Coordinates for Specific Source
C*
C*       CALLED FROM:   PITEFF
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: NUMV, NSP

      REAL :: XVIN(NVMAX), YVIN(NVMAX)
      REAL :: XWD(NVMAX),  YWD(NVMAX)

C*    Variable Initializations
      MODNAM = 'AVERTS'
      
      DO NSP = 1, NUMV
         XWD(NSP) = -(XVIN(NSP)*WDSIN + YVIN(NSP)*WDCOS)
         YWD(NSP) =   XVIN(NSP)*WDCOS - YVIN(NSP)*WDSIN
      END DO

      RETURN
      END

      SUBROUTINE AREAIN
C***********************************************************************
C                 AREAIN Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration for AREA Sources
C                 Using Numerical Integration Algorithm
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY R. Brode, PES, Inc., 4/2/99, to change the lower limit
C                 on (ua-ub) from 1.0 to 0.01 meter.  This prevents potential
C                 anomalous results for very small area sources (about
C                 1.0 meter wide).
C
C        MODIFIED BY R. Brode, PES, Inc. to use -3.9 to +3.9 for width of
C                 the plume for internal consistency with PWIDTH.  Also
C                 added error checks on the number of "sides" exceeding
C                 NVMAX, which could happen with complex AREAPOLY shapes.
C                 12/14/98
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C
C        CALLED FROM:   ACALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: KSIDE, NCP, KSP, KS
      REAL    :: UCRIT, UA, UB, VA, VB, VNMIN, VNMAX, WA, VNORM
      REAL    :: VAL, DVAL, VALD, DVALD
      LOGICAL QGO

C     Variable Initializations
      MODNAM = 'AREAIN'

C     INITIALIZE VARIABLES FOR INTEGRATION PROCEDURE.
      UCRIT = 1.01
      VAL   = 0.0
      VALD  = 0.0
      KSIDE = 0
      DO ncp = 1, NVERT
         ua = SPA(ncp,1)
         ub = SPA(ncp+1,1)
         va = SPA(ncp,2)
         vb = SPA(ncp+1,2)
         IF (ua .ge. ucrit) THEN
            kside = kside + 1
            uvert(kside) = ua
            vvert(kside) = va
         END IF
         IF ((ua .ge. ucrit .AND. ub .lt. ucrit) .OR.
     &       (ua .lt. ucrit .AND. ub .ge. ucrit)) THEN
            kside = kside+1
            IF (kside .LE. NVMAX) THEN
               uvert(kside) = ucrit
               vvert(kside) = va+(ucrit-ua)*(vb-va)/(ub-ua)
            ELSE
C              Write Error Message:  Number of "sides" exceeds NVMAX
               CALL ERRHDL(PATH,MODNAM,'E','405',SRCID(ISRC))
               RUNERR = .TRUE.
               RETURN
            END IF
         END IF
      END DO

      QGO = .FALSE.
      IF (kside .ge. 2) THEN
         QGO = .TRUE.
         vnmin=  3.9
         vnmax= -3.9
         DO ncp = 1, kside
            ua = uvert(ncp)
            va = vvert(ncp)
            call pwidth(ua,va,vnorm,wa)
            vnvert(ncp) = vnorm
            wvert(ncp)  = wa
            vnmax = MAX(vnorm,vnmax)
            vnmin = MIN(vnorm,vnmin)
         END DO
         IF (vnmin .ge. 3.9 .or. vnmax .le. -3.9) QGO = .FALSE.
      END IF

C     Integrate Between Vertices u(1),u(2) THEN u(2),u(3); etc.
      IF (QGO) THEN
C        MAKE 1st Point Same as Last
         ksp = kside+1
         IF (ksp .LE. NVMAX) THEN
            uvert(ksp)  = uvert(1)
            vvert(ksp)  = vvert(1)
            vnvert(ksp) = vnvert(1)
            wvert(ksp)  = wvert(1)
         ELSE
C           Write Error Message:  Number of "sides" exceeds NVMAX
            CALL ERRHDL(PATH,MODNAM,'E','405',SRCID(ISRC))
            RUNERR = .TRUE.
            RETURN
         END IF
         nsegs = 0
         LSEG = .FALSE.
         DO ks = 1, kside
            QGO = .TRUE.
            ivert = ks
            ua = uvert(ks)
            ub = uvert(ks+1)
            dval  = 0.0
            dvald = 0.0
            IF (abs(ua-ub) .lt. 0.01) QGO = .FALSE.
            IF (QGO .AND. TOXICS) THEN
               call pside_tox(ua,ub,dval,dvald)
            ELSE IF (QGO) THEN
               call pside(ua,ub,dval,dvald)
            END IF
            val  = val + dval
            vald = vald + dvald
         END DO
         IF (nsegs .gt. 0) THEN
            LSEG = .TRUE.
            IF (TOXICS) THEN
               call pside2_tox(dval,dvald)
            ELSE
               call pside2(dval,dvald)
            END IF
            val  = val + dval
            vald = vald + dvald
         END IF

C        Calculate hourly value, HRVAL
         HRVAL(ITYP) = ABS(VAL)*QTK*EMIFAC(ITYP)/US
         IF (WETSCIM) HRVALD(ITYP) = ABS(VALD)*QTK*EMIFAC(ITYP)/US
      ELSE

         HRVAL(ITYP)  = 0.0
         HRVALD(ITYP) = 0.0
      END IF

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
         CALL DEBOUT
      END IF

      RETURN
      END

      SUBROUTINE PLUMEF(XARG,POUT,POUTD)
C***********************************************************************
C                 PLUMEF Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Driving Routine for Plume Calculations
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY R. Brode, PES, Inc. to call separate ADIS_ routines
C                    for sigma-y and sigma-z for optimization. - 12/10/98
C
C        MODIFIED BY D. Strimaitis and Yicheng Zhuang, SRC (for DEPOSITION)
C
C        MODIFIED BY R. Brode, PES, Inc. to move calculation of dispersion
C                    coefficients to a new ADIS subroutine - 7/21/94
C
C        DATE:    September 28, 1993
C
C        INPUTS:  Downwind Distance (in km !)
C                 Source Parameter Arrays
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C                 For A Certain Downwind Distance
C
C        CALLED FROM:   TRAPZD
C***********************************************************************
C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
C     Declare Local Variables
      INTEGER :: I
      REAL    :: XARG, POUT, POUTD, VAL, VT, VN, RCZ, RCZD
      LOGICAL LTERR

C     Variable Initializations
      MODNAM = 'PLUMEF'

C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

C     MODIFIED to NOT compute sy, vn, val for cases with VAL=1.0;
C     uses LSEG, a logical variable set in PSIDE2, AREAIN to establish case
      IF (LSEG) THEN
C        Calculate vertical dispersion coefficient, SZ      ---   CALL ADISZ
         CALL ADISZ(XARG,SZ,XZ)
         VAL = 1.0
      ELSE
         CALL XWIDTH(XARG,VT)
         CALL PWIDTH(XARG,VT,VN,VAL)
         CALL ADISZ(XARG,SZ,XZ)
      END IF

C     Get Concentration or Deposition
      CALL ASIMPL(XARG,RCZ,RCZD)

C     Now compute the function
      POUT  = VAL*RCZ
      POUTD = VAL*RCZD

      RETURN
      END


      SUBROUTINE PSIDE(U1,U2,DVAL,DVALD)
C***********************************************************************
C                 PSIDE Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: INTEGRATES SIDE K of POLYGON
C                 int f(u)*CNF(v(u)/sig(u))=f(u)*vn(u) from u1 to u2
C                           CNF = cumulative normal distribution
C                 Computes W(1),W(2)--normalized plume width at   u1    u2
C                 Checks for w(i) outside of -3.9,3.9 with i+, i-
C                 L=-3.9  U=3.9  = bounds for testing
C                 Integrates according to case encountered:
C                 situation     CASE    iplus    iminus  integral limits
C                 L<w1,w2<U      1        0        0         u1,u2
C                 w1,w2<L        2        0       1+2      don't compute
C                 w1,w2>U        3       1+2       0         u1,u2
C                 w1<L<w2<U      4        0        1         u-,u2
C                 w2<L<w1<U      5        0        2         u1,u-
C                 L<w1<U<w2      6        2        0       u1,u+  u+,u2
C                 L<w2<U<w1      7        1        0       u1,u+  u+,u2
C                 w1<L<U<w2      8        2        1       u-,u+  u+,u2
C                 w2<L<U<w1      9        1        2       u1,u+  u+,u-
C
C                 u+ = value of u such that w(u)=U=3.9
C                 u- =     "                w(u)=L=-3.9
C                 u+,u- computed with Brent's Algorithm
C
C                 IF uplus >0, part of side is outside plume
C                 but is integrated anyway, unless there is
C                 a corresponding part on another side that will
C                 cause cancellation.  This is determined in
C                 PSIDE2;
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED by Roger Brode, PES, Inc. to use QATR routine for
C                    integration, and to pass VN(1) and VN(2) to
C                    ZBRENT.  Also changed to use -3.9 to +3.9 for
C                    width of plume for internal consistency with
C                    SUBROUTINE PWIDTH. - 12/14/98
C
C        MODIFIED by Roger Brode, PES, Inc. to correct lower integration
C                    limit for Case 4, and to remove extraneous calls
C                    to XWIDTH and PWIDTH after calls to ZBRENT. - 7/29/94
C
C        INPUTS:  End Points of The Segments
C
C        OUTPUTS: Integral Value (if any) for Segment
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C---- Set convergence criteria for calls to QATR3:
C         NDIM = maximum number of integration steps
C         IMIN = minimum number of integration steps
C         EPSR = relative error tolerance for integral
C         EPST = minimum value threshold for integral
C----
      INTEGER, PARAMETER :: NDIM = 10, IMIN = 4
      REAL, PARAMETER    :: EPSR = 2.0E-2, EPST = 1.0E-5

      INTEGER :: I, KS, IMINUS, IPLUS, NOUT, ICON
      REAL    :: DVAL, DVALD, U1, U2, UMINUS, UPLUS, AUX(NDIM),
     &           u(2), v1(2), vn(2), w(2)

C     Variable Initializations
      MODNAM = 'PSIDE'

C     NSEG = number of segments; set to 0 in AREAIN
C     for each source/rcvr/time step
      dval  = 0.0
      dvald = 0.0

      DO i =  1,2
         ks    = ivert + i-1
         u(i)  = uvert(ks)
         v1(i) = vvert(ks)
         vn(i) = vnvert(ks)
         w(i)  = wvert(ks)
      END DO

      iminus = 0
      iplus  = 0
      uminus = -1.0
      uplus  = -1.0
      DO i = 1,2
         IF (vn(i) .lt. -3.9) iminus = i + iminus
         IF (vn(i) .gt.  3.9) iplus  = i + iplus
      END DO

      IF (iplus.EQ.1 .or. iplus.EQ.2) THEN
         call zbrent( 1,u(1),u(2),vn(1),vn(2),1.0,uplus)
      END IF
      IF (iminus.EQ.1 .or. iminus.EQ.2) THEN
         call zbrent(-1,u(1),u(2),vn(1),vn(2),1.0,uminus)
      END IF

c---- CASE DEPENDs on iplus, iminus
      IF (iplus .EQ. 0) THEN
         IF (iminus .EQ. 0) THEN
c                                             iplus  iminus  case
c                                               0     0       1
            call qatr3(u1,u2,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         ELSE IF (iminus .EQ. 3) THEN
c                                               0     3       2
            dval = 0
         ELSE IF (iminus .EQ. 1) THEN
c                                               0     1       4
            call qatr3(uminus,u2,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         ELSE
c                                               0     2       5
            call qatr3(u1,uminus,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         END IF

      ELSE IF (iplus .EQ. 1) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = uplus
         IF (iminus .EQ. 0) THEN
c                                               1     0       7
            call qatr3(uplus,u2,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         ELSE
c                                               1     2       9
            call qatr3(uplus,uminus,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         END IF

      ELSE IF (iplus .EQ. 2) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = uplus
         ubsegs(nsegs) = u2
         IF (iminus .EQ. 0) THEN
c                                               2     0       6
            call qatr3(u1,uplus,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         ELSE
c                                               2     1       8
            call qatr3(uminus,uplus,epsr,epst,ndim,imin,dval,dvald,
     &                 icon,nout,aux)
         END IF

      ELSE
c                                               3     0       3
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = u2
      END IF

      RETURN
      END

      SUBROUTINE XWIDTH(U,XOUT)
C***********************************************************************
C                 XWIDTH Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Given Any Y Coordinate of A Vertex of an Area
C                 Source, Calculate the X Coordinate
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  The Y Coordinate
C
C        OUTPUTS: The X Coordinate Value
C
C        CALLED FROM:   ZBRENT
C                       PSIDE
C                       PLUMEF
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XOUT, U, U1, U2, V1, V2

C     Variable Initializations
      MODNAM = 'XWIDTH'

      U1 = UVERT(IVERT)
      U2 = UVERT(IVERT+1)
      V1 = VVERT(IVERT)
      V2 = VVERT(IVERT+1)
      XOUT = V1+(U-U1)*(V2-V1)/(U2-U1)

      RETURN
      END

      SUBROUTINE PWIDTH(X1,V1,VN,WIDTH)
C***********************************************************************
C                 PWIDTH Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates The Effective Area of The Plume for A
C                 Certain Downwind Distance
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY R. Brode, PES, Inc. to move calculation of dispersion
C                    coefficients to a new ADIS subroutine - 7/21/94
C
C        MODIFIED BY R. Brode, PES, Inc. to correct table of GA values
C                    and extend GA to 79 values - 7/29/94
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Receptor Height Above Ground
C                 Source Parameter Arrays
C
C        OUTPUTS: The Effective Width
C
C        CALLED FROM:   ZBRENT
C                       PSIDE
C                       PLUMEF
C                       AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: ITEMP
      REAL    :: WIDTH, VN, V1, X1, TEMP, GA(79)

C     Variable Initializations
C     GA ARE VALUES OF THE CUMULATIVE NORMAL DISTRIBUTION IN
C     INCREMENTS OF 0.1 S.
      DATA GA/0.0,.0001,.0001,.0002,.0002,.0003,.0005,.0007,.0010,.0013,
     1.0019,.0026,.0035,.0047,.0062,.0082,.0107,.0139,.0179,.0227,.0287,
     2.0359,.0446,.0548,.0668,.0808,.0968,.1151,.1357,.1587,.1841,.2119,
     3.2420,.2742,.3085,.3445,.3821,.4207,.4602,.5000,.5398,.5793,.6179,
     4.6555,.6915,.7258,.7580,.7881,.8159,.8413,.8643,.8849,.9032,.9192,
     5.9332,.9452,.9554,.9641,.9713,.9773,.9821,.9861,.9893,.9918,.9938,
     6.9953,.9965,.9974,.9981,.9987,.9990,.9993,.9995,.9997,.9998,.9998,
     7.9999,.9999,1.000/
      MODNAM = 'PWIDTH'

      IF (X1 .EQ. 0.0) THEN
         SY = 1.0
         VN = V1
         WIDTH = VN
C        Exit Routine
         GO TO 999
      END IF

C     Calculate lateral dispersion coefficient, SY          ---   CALL ADISY
      CALL ADISY(X1,SY,XY)

      VN = V1/SY
      TEMP = 10*VN + 40
      ITEMP = INT(TEMP)
      WIDTH = 0.0

      IF (ITEMP. GT. 78) THEN
         WIDTH = 1.0000
      ELSE
         IF (ITEMP .GE. 1) THEN
            WIDTH = GA(ITEMP)+(TEMP-FLOAT(ITEMP))*
     &              (GA(ITEMP+1)-GA(ITEMP))
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE ZBRENT(IFD,X1,X2,VN1,VN2,TOL,OUTVAL)
C***********************************************************************
C                 ZBRENT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: To Divide The Segments According to The Plume Split
C                 And Edge Effects
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Source Parameter Arrays
C
C        OUTPUTS: The Effective Integration Segments
C
C        CALLED FROM:   PSIDE
C***********************************************************************

C     Variable Declarations
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER, PARAMETER :: ITMAX = 10
      REAL, PARAMETER    :: EPSZ  = 1.0E-3
      INTEGER :: IFD, ITER
      REAL    :: OUTVAL, TOL, X1, X2, A1, B1, V1, W1, VN, FA, FB, FC,
     &           C1, D1, E1, TOL1, XM, P1, Q1, R1, S1, VN1, VN2

C     Variable Initializations
      MODNAM = 'ZBRENT'

      a1 = x1
      b1 = x2
      fa = vn1-ifd*3.9
      fb = vn2-ifd*3.9

      IF (fb*fa .LE. 0.0) THEN
         fc = fb
         DO iter = 1, itmax
            IF (fb*fc .gt. 0.0) THEN
               c1 = a1
               fc = fa
               d1 = b1-a1
               e1 = d1
            END IF
            IF (abs(fc) .lt. abs(fb)) THEN
               a1 = b1
               b1 = c1
               c1 = a1
               fa = fb
               fb = fc
               fc = fa
            END IF
            tol1 = 2.0*epsz*abs(b1)+0.5*tol
            xm = 0.5*(c1-b1)
            IF (abs(xm).le.tol1  .or. fb .EQ. 0.0) THEN
               outval = b1
               RETURN
            END IF
            IF (abs(e1).ge.tol1 .AND. abs(fa).gt.abs(fb)) THEN
               s1 = fb/fa
               IF (a1 .EQ. c1) THEN
                  p1 = 2.0*xm*s1
                  q1 = 1.0-s1
               ELSE
                  q1 = fa/fc
                  r1 = fb/fc
                  p1 = s1*(2.0*xm*q1*(q1-r1)-(b1-a1)*(r1-1.0))
                  q1 = (q1-1.0)*(r1-1.0)*(s1-1.0)
               END IF
               IF(p1. gt. 0.0) q1 = -q1
               p1 = abs(p1)
               IF (2.0*p1.lt.min(3.0*xm*q1-
     &             abs(tol1*q1),abs(e1-q1))) THEN
                  e1 = d1
                  d1 = p1/q1
               ELSE
                  d1 = xm
                  e1 = d1
               END IF
            ELSE
               d1 = xm
               e1 = d1
            END IF
            a1 = b1
            fa = fb
            IF (abs(d1).gt. tol1) THEN
               b1 = b1+d1
            ELSE
               b1 = b1+sign(tol1,xm)
            END IF
            call xwidth(b1,v1)
            call pwidth(b1,v1,vn,w1)
            fb = vn-ifd*3.9
         END DO
         outval = b1
      END IF

      RETURN
      END

      SUBROUTINE PSIDE2(DVAL,DVALD)
C***********************************************************************
C                 PSIDE2 Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Integrates Over Segments For Which ABS(VN) > VNTEST
C                 Eliminates Overlap of Segments And Useless Integration
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:   Number of The Original Segments
C                  End Points Array of The Segments
C
C        OUTPUT:   The Correction of The Results From PSIDE
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C---- Set convergence criteria for call to QATR3:
C         NDIM = maximum number of integration steps
C         IMIN = minimum number of integration steps
C         EPSR = relative error tolerance for integral
C         EPST = minimum value threshold for integral
C----
      INTEGER, PARAMETER :: NDIM = 10, IMIN = 4
      REAL, PARAMETER    :: EPSR = 2.0E-2, EPST = 1.0E-5

      INTEGER :: I, J, ISEG, NPTS, NOUT, ICON
      REAL    :: DVAL, DVALD, TEMP, U1, U2, UAV, UBV, TMPVAL, TMPVALD,
     &           AUX(NDIM)
      real ulist(nvmax2), useg(nvmax,2)
      integer usign(nvmax), ufac, usegf(nvmax)
      LOGICAL Ltest1,Ltest2

C     Variable Initializations
      MODNAM = 'PSIDE2'

      j = 1
      DO i = 1, nsegs
         ulist(j) = uasegs(i)
         j = j+1
         ulist(j) = ubsegs(i)
         j = j+1
      END DO
      npts = 2*nsegs

      call hpsort(npts,ulist,nvmax2)

      DO i = 1, nsegs
         usign(i) = 1
         IF (uasegs(i) .GT. ubsegs(i)) THEN
            usign(i) = -1
            temp = uasegs(i)
            uasegs(i) = ubsegs(i)
            ubsegs(i) = temp
         END IF
         IF (uasegs(i) .EQ. ubsegs(i)) usign(i) = 0
      END DO
      iseg = 0

      DO i = 2,npts
         u1 = ulist(i-1)
         u2 = ulist(i)
         ufac = 0
c*****
c           compare segment [u1,u2] against each ua,ub
c*****
         IF (u1 .ne. u2) THEN
            DO j = 1, nsegs
               IF (u1.ge.uasegs(j) .AND. u2 .le. ubsegs(j)) THEN
                  ufac = ufac + usign(j)
               END IF
            END DO
c*****
c              make table of segments and factors
c*****
            IF (ufac .ne. 0) THEN
               iseg = iseg+1
               useg(iseg,1) = u1
               useg(iseg,2) = u2
               usegf(iseg) = ufac
            END IF
         END IF
      END DO
c*****
c            CONSOLIDATE SEGMENTS IF iseg>1
c*****
      nsegs = iseg
      IF (nsegs .gt. 1) THEN
         DO iseg = 2, nsegs
            Ltest1 = useg(iseg,1) .EQ. useg(iseg-1,2)
            Ltest2 = usegf(iseg)*usegf(iseg-1) .gt. 0
            IF (Ltest1 .AND. Ltest2) THEN
               usegf(iseg-1) = 0
               useg(iseg,1) = useg(iseg-1,1)
            END IF
         END DO
      END IF
      dval  = 0.0
      dvald = 0.0
      IF (nsegs .gt. 0) THEN
         DO iseg = 1, nsegs
            IF (usegf(iseg) .ne. 0) THEN
               uav = useg(iseg,1)
               ubv = useg(iseg,2)
               ufac = usegf(iseg)
               call qatr3(uav,ubv,epsr,epst,ndim,imin,tmpval,tmpvald,
     &                    icon,nout,aux)
               dval  = dval + ufac*tmpval
               dvald = dvald + ufac*tmpvald
            END IF
         END DO
      END IF

      RETURN
      END

      SUBROUTINE HPSORT(NVAR,UVAR,IDIM)
C***********************************************************************
C                 HPSORT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: A General Program For Heap Sort of An Array
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY R. Brode, PES, Inc. to include a single RETURN,
C                    avoided "no path to statement" messages for
C                    some compilers. - 12/1/97
C
C        INPUTS:  The Array To Be Sorted
C
C        OUTPUTS: The Array Sorted
C
C        CALLED FROM:   PSIDE2
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, J, IDIM, NVAR, ILMID, IR
      REAL    :: RU, UVAR(IDIM)

C     Variable Initializations
      MODNAM = 'HPSORT'

      ILMID = NVAR/2 + 1
      IR = NVAR

10    CONTINUE

      IF (ilmid.gt.1) THEN
         ilmid = ilmid-1
         ru = uvar(ilmid)
      ELSE
         ru = uvar(ir)
         uvar(ir) = uvar(1)
         ir = ir-1
         IF (ir .EQ. 1)THEN
            uvar(1) = ru
C           Processing is done
            GO TO 999
         END IF
      END IF

      i = ilmid
      j = ilmid+ilmid

      DO WHILE (j. le. ir)
         IF (j. lt. ir) THEN
            IF (uvar(j).lt.uvar(j+1) ) j = j+1
         END IF
         IF (ru.lt.uvar(j)) THEN
            uvar(i) = uvar(j)
            i = j
            j = 2*j
         ELSE
            j = ir+1
         END IF
      END DO

      uvar(i) = ru

      GO TO 10

 999  RETURN
      END

C***  End new code for area source numerical integration algorithm - 7/7/93



C***  Subroutines for OPENPIT Source algorithms - 7/19/94


      SUBROUTINE ESCAPE(ICAT)
C***********************************************************************
C*                ESCAPE Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate Escape Fractions for a Particle
C*                Size Category
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Index for Particle Size Category Being Processed
C*                Gravitational Settling Velocity for Current 
C*                     Particle Size Category & Current Source
C*                10-meter Wind Speed for the Current Hour
C*                Constant ALPHA (= 0.029)
C*                
C*
C*       OUTPUTS: The Escape Fraction for the current Size Category
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: ICAT
C*    Variable Initializations
      MODNAM = 'ESCAPE'

      EFRAC(ICAT) = 1.0/(1.0 + VGRAV(ICAT) / (ALPHA * UREF10) )
      
      RETURN
      END


      SUBROUTINE ADJEMI(ICAT,QPTOT)
C***********************************************************************
C*                ADJEMI Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Adjust Emission Rate for Current Particle
C*                Size Category Being Processed
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Index for Particle Size Category Being Processed
C*                Escape Fraction for the Current Size Category
C*                Mass Fraction of the Current Size Category
C*                Total Emission Rate Per Unit Area
C*                
C*
C*       OUTPUTS: Adjusted Emission Rate for the Current Size Category
C*                Cumulative Adjusted Emission Rate Over All Categories
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: ICAT
      REAL    :: QPTOT
C*    Variable Initializations
      MODNAM = 'ADJEMI'

      QPART(ICAT) = EFRAC(ICAT) * PHI(ICAT) * QS
      QPTOT = QPTOT + QPART(ICAT)
          
      RETURN
      END


      SUBROUTINE AMFRAC(QPTOT)
C***********************************************************************
C*                AMFRAC Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Adjust the Mass Fractions for each Particle
C*                Size Category
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Array of Adjusted Emission Rates
C*                Cumulative Adjusted Emission Rate Over All Categories
C*
C*       OUTPUTS: Array of Adjusted Mass Fractions
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: ICAT
      REAL    :: QPTOT
C*    Variable Initializations
      MODNAM = 'AMFRAC'

      DO ICAT = 1,NPD
         PHI(ICAT) = QPART(ICAT)/QPTOT
      END DO
          
      RETURN
      END


      SUBROUTINE LWIND
C***********************************************************************
C*                LWIND Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Along-Wind Length of the OPENPIT
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Wind Flow Vector for the Current Hour
C*                Angle of the Long OPENPIT dimension from the North
C*                Length of the OPENPIT
C*                Width of the OPENPIT
C*
C*       OUTPUTS: Along-Wind Length of the OPENPIT
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
C*    Variable Initializations
      MODNAM = 'LWIND'
      
C*    Determine the Wind Direction Angle Relative to the Long
C*    Axis of the OpenPit
      CALL CTHETA(AFV,PALPHA,THETA)
      
C*    Determine the Along-Wind Length of the OPENPIT
      PITL = PITLEN * (1 - THETA/90.) + PITWID * (THETA / 90.)
      
      RETURN
      END


      SUBROUTINE PDEPTH
C***********************************************************************
C*                PDEPTH Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Relative Depth of the OPENPIT Source
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Effective Depth of the OPENPIT
C*                Release Height Above 
C*                Along Wind Length of the OPENPIT
C*                
C*       OUTPUTS: Relative Depth of the OPENPIT
C*                
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
C*    Variable Initializations
      MODNAM = 'PDEPTH'

      PDREL = (PDEFF-EMIHGT) / PITL

      RETURN
      END
      

      SUBROUTINE PTFRAC 
C***********************************************************************
C*                PTFRAC Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Fractional Size of the Effective
C*                Area of the OPENPIT Source
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Relative Pit Depth
C*
C*       OUTPUTS: Fractional Size of the Effective Area of the OPENPIT
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
C*    Variable Initializations
      MODNAM = 'PTFRAC'
      
      IF (PDREL .GE. 0.2) THEN
         PITFRA = 0.08
      ELSE
         PITFRA = SQRT (1.0 - 1.7*(PDREL**(0.333333)) )
      ENDIF
      
      RETURN
      END
      

      SUBROUTINE PITEFF
C***********************************************************************
C*                PITEFF Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Coordinates of the OPENPIT Source
C*                in Wind Direction Coordinate System
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  
C*
C*       OUTPUTS: Coordinates of the OPENPIT Source in Wind 
C*                Direction Coordinate System
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, II, IUPWND
      REAL    :: SPAMIN, EFFANG, EFFWID, EFFLEN
      REAL    :: XTEMP(NVMAX), YTEMP(NVMAX)
      REAL, PARAMETER :: EPSLON = 0.00001
      
C*    Variable Initializations
      MODNAM = 'PITEFF'
      
C*    Get Vertices of Actual Pit in WD-Coordinate System    ---   CALL AVERTS
      CALL AVERTS(XVERT,YVERT,XTEMP,YTEMP,NVERT+1)
              
C*    Find the Upwind Vertex of the Pit (one with minimum X)
      SPAMIN = 1.0E+20
      IUPWND = 0
      DO IVERT = 1,NVERT
         IF (XTEMP(IVERT) .LT. SPAMIN) THEN
            IUPWND = IVERT
            SPAMIN = XTEMP(IVERT)-EPSLON
         ENDIF
      END DO

C*    If DEBUG Requested, Write Out Pit Info            
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'ACTUAL PIT COORDINATES:'
         WRITE (IOUNIT,*) '----------------'
         WRITE (IOUNIT,*) 'SYSTEM   X1       Y1       X2       Y2',
     &                    '       X3       Y3       X4       Y4'
         WRITE (IOUNIT,*) '-------- -------- -------- -------- ',
     &                    '-------- -------- -------- -------- ',
     &                    '--------'
         WRITE (IOUNIT,8000) (XVERT(II), YVERT(II), II=1,NVERT)
         WRITE (IOUNIT,8100) (XTEMP(II), YTEMP(II), II=1,NVERT)
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) ' UPWIND VERTEX OF THE PIT= ', IUPWND
         WRITE (IOUNIT,*) ' WIND DIRECTION W.R.T. PIT LONG AXIS= ',THETA
         WRITE (IOUNIT,*) ' ALONGWIND LENGTH OF THE PIT= ',PITL
         WRITE (IOUNIT,*) ' RELATIVE DEPTH OF THE PIT= ',PDREL
         WRITE (IOUNIT,*)
8000     FORMAT (1X,'User     ',8(f11.3,1x))
8100     FORMAT (1X,'Wind-Dir ',8(f11.3,1x))
      ENDIF
      
C*    Determine the Angle of the Effective Pit Relative to North
      EFFANG = ANGLE + (90.*(IUPWND - 1))
      
C*    Determine Length and Width Dimensions of the
C*    Effective Pit Area
      EFFWID = PITFRA**(1.0 - (COS(THETA*DTORAD))**2)*PITWID
      EFFLEN = PITFRA**((COS(THETA*DTORAD))**2)*PITLEN

C*    Calculate the Coordinates of the Vertices of the Effective Pit Area
C*    Set Coordinates of Vertices for Rectangular Area (in Kilometers).
C*    Vertices Start with the "Southwest" Corner and Are Defined
C*    Clockwise.  The First Vertex is Repeated as the Last Vertex.


C*    First determine proper 'x-dim' and 'y-dim' for effective area,
C*    taking into account angle of orientation and relation to actual pit.

      IF (XINIT .LE. YINIT .AND. (IUPWND.EQ.1 .OR. IUPWND.EQ.3)) THEN
         XEFF = EFFWID
         YEFF = EFFLEN
      ELSE IF (XINIT.LE.YINIT .AND. (IUPWND.EQ.2 .OR. IUPWND.EQ.4)) THEN
         XEFF = EFFLEN
         YEFF = EFFWID
      ELSE IF (XINIT.GT.YINIT .AND. (IUPWND.EQ.1 .OR. IUPWND.EQ.3)) THEN
         XEFF = EFFLEN
         YEFF = EFFWID
      ELSE IF (XINIT.GT.YINIT .AND. (IUPWND.EQ.2 .OR. IUPWND.EQ.4)) THEN
         XEFF = EFFWID
         YEFF = EFFLEN
      END IF

      XTEMP(1) = XVERT(IUPWND)
      YTEMP(1) = YVERT(IUPWND)

      XTEMP(2) = XTEMP(1) + (YEFF*SIN(EFFANG*DTORAD))
      YTEMP(2) = YTEMP(1) + (YEFF*COS(EFFANG*DTORAD))

      XTEMP(3) = XTEMP(2) + (XEFF*COS(EFFANG*DTORAD))
      YTEMP(3) = YTEMP(2) - (XEFF*SIN(EFFANG*DTORAD))

      XTEMP(4) = XTEMP(3) - (YEFF*SIN(EFFANG*DTORAD))
      YTEMP(4) = YTEMP(3) - (YEFF*COS(EFFANG*DTORAD))

      XTEMP(5) = XVERT(IUPWND)
      YTEMP(5) = YVERT(IUPWND)


C*    Calculate Coordinates of the Effective Pit Area in
C*    Wind Direction Coordinate System                      ---   CALL AVERTS
      CALL AVERTS(XTEMP,YTEMP,XVERT,YVERT,NVERT+1)

C*    If DEBUG Requested, Write Out Pit Info            
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'EFFECTIVE PIT COORDINATES:'
         WRITE (IOUNIT,*) '----------------'
         WRITE (IOUNIT,*) 'SYSTEM   X1       Y1       X2       Y2',
     &                    '       X3       Y3       X4       Y4'
         WRITE (IOUNIT,*) '-------- -------- -------- -------- ',
     &                    '-------- -------- -------- -------- ',
     &                    '--------'
         WRITE (IOUNIT,8000) (XTEMP(II), YTEMP(II), II=1,NVERT)
         WRITE (IOUNIT,8100) (XVERT(II), YVERT(II), II=1,NVERT)
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) 'EFFECTIVE PIT LENGTH = ', EFFLEN
         WRITE (IOUNIT,*) 'EFFECTIVE PIT WIDTH  = ', EFFWID
         WRITE (IOUNIT,*) 'EFFECTIVE PIT ORIENTATION RELATIVE',
     &                    ' TO NORTH= ', EFFANG
         WRITE (IOUNIT,*) 'FRACTIONAL SIZE OF THE EFFECTIVE PIT AREA= ',
     &                     PITFRA
      ENDIF

C     Reassign Effective Area Coordinates to Global Arrays for Subsequent Calcs.
      DO I = 1, 5
         XVERT(I) = XTEMP(I)
         YVERT(I) = YTEMP(I)
      END DO
      
      RETURN
      END


      SUBROUTINE PITEMI(QPTOT)
C***********************************************************************
C*                PITEMI Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Emission Rate for the Effective 
C*                Pit Area
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Fractional Area of the Pit
C*                Total Adjusted Emission Rate
C*
C*       OUTPUTS: Emission Rate for the Effective Area of the Current
C*                OPENPIT Source
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: QPTOT
C*    Variable Initializations
      MODNAM = 'PITEMI'
      
      QEFF = QPTOT / PITFRA
      
      RETURN
      END


      SUBROUTINE CTHETA(AFVIN,ALFIN,THOUT)
C***********************************************************************
C*                CTHETA Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Wind Direction Angle Relative to 
C*                the Pit Long Axis
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 26, 1994
C*
C*       INPUTS:  Flow Vector 
C*                Angle of Pit Long Axis from North
C*
C*       OUTPUTS: THETA - Wind Direction Angle Relative to 
C*                the Pit Long Axis
C*                
C*
C*       CALLED FROM:   LWIND
C***********************************************************************

      IMPLICIT NONE

      REAL :: THOUT, ALFIN, AFVIN, THETA

      if (abs(AFVIN-ALFIN) .le. 90.) then
         THOUT = abs(AFVIN-ALFIN)
      else if (AFVIN .gt. ALFIN) then
         theta = AFVIN - ALFIN
         if (theta .gt. 90.) then
            theta = AFVIN-180. - ALFIN
         endif
         if (theta .gt. 90.) then
            theta = AFVIN-360. - ALFIN
         endif
         if (theta .gt. 90.) then
            theta = AFVIN-540. - ALFIN
         endif
         THOUT = abs(theta)
      else if (AFVIN .lt. ALFIN) then
         theta = AFVIN - ALFIN
         if (theta .lt. -90.) then
            theta = AFVIN + 180. - ALFIN
         endif
         if (theta .lt. -90.) then
            theta = AFVIN + 360. - ALFIN
         endif
         if (theta .lt. -90.) then
            theta = AFVIN + 540. - ALFIN
         endif
         THOUT = abs(theta)
      endif
      RETURN
      end
      
c-----------------------------------------------------------------------
      subroutine qatr3(xl,xu,eps,epst,ndim,imin,y,yd,ier,i,aux)
c-----------------------------------------------------------------------
c
c --- ISCST3                        QATR3
c
c PURPOSE:      Integration routine adapted from the IBM SSP program
c               DQATR.  Modified for single precision.  This is a COPY
c               of QATR for use in area source integrations.
c
c ARGUMENTS:
c    PASSED:    xl,xu   lower and upper limits of integration        [r]
c               eps     fractional error used to define convergence  [r]
c               epst    lower theshold check for value of integral   [r]
c               ndim    dimension of array aux (max no. of steps)    [i]
c               imin    minimum number of "steps" for integral       [i]
c               fct     external function (integrand)                [r]
c               aux     working array, passed to allow variable dim. [r]
c  RETURNED:    y       value of integral                            [r]
c               yd      value of integral without wet depletion      [r]
c               ier     status flag at termination                   [i]
c               i       number of subdivision steps                  [i]
c
c CALLING ROUTINES:     PSIDE, PSIDE2
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

c  NOTES: status flags denote the following --
c               ier=0   value of integral converged to within eps
c               ier=1   value of integral is diverging (not used)
c               ier=2   value of integral did not converge to within
c                       eps before ndim limit was reached

      IMPLICIT NONE

      INTEGER :: NDIM, IMIN, IER
      REAL    :: Y, YD, EPS, EPST, XU, XL, H, HH, DELT2, P,
     &           DELT1, HD, X, SM, SMD, Q, AUX(NDIM),
     &           AUXD(NDIM), P1, P2, P1D, P2D
      integer :: i, ii, ji, j, jj

      DO I = 1, NDIM
         AUX(I)  = 0.0
         AUXD(I) = 0.0
      END DO

c---  Preparations for Romberg loop
      CALL PLUMEF(XL,P1,P1D)
      CALL PLUMEF(XU,P2,P2D)
      aux(1) = 0.5 * (P1+P2)
      auxd(1)= 0.5 * (P1D+P2D)
      h = xu - xl

      if(h .EQ. 0.0 .OR. aux(1) .EQ. 0.0) then
         ier=0
         y  = 0.0
         yd = 0.0
         return
      endif

      hh = h
      delt2 = 0.
      p  = 1.
      jj = 1

      do i = 2, ndim
         y = aux(1)
         delt1 = delt2
         hd = hh
         hh = 0.5 * hh
         p  = 0.5 * p
         x  = xl + hh
         sm  = 0.
         smd = 0.

         do j = 1, jj
            CALL PLUMEF(X,P1,P1D)
            sm  = sm + P1
            smd = smd + P1D
            x   = x + hd
         end do

c----    A new approximation to the integral is computed by means
c        of the trapezoidal rule
         aux(i)  = 0.5*aux(i-1) + p*sm
         auxd(i) = 0.5*auxd(i-1) + p*smd

c----    Start of Rombergs extrapolation method

         q  = 1.
         ji = i-1
         do j = 1, ji
            ii = i-j
            q  = q+q
            q  = q+q
            aux(ii)  = aux(ii+1) + (aux(ii+1)-aux(ii))/(q-1.)
            auxd(ii) = auxd(ii+1) + (auxd(ii+1)-auxd(ii))/(q-1.)
         end do

c----    End of Romberg step

c        Compute absolute error, delt2
         delt2 = ABS(y-aux(1))

         if (i .GE. imin) then
c           Check for covergence of algorithm
            if (ABS(aux(1)) .LT. epst) then
c              Lower threshold convergence test
               ier = 0
               y  = h*aux(1)
               yd = h*auxd(1)
               return
            else if (delt2 .LE. eps*ABS(aux(1)) ) then
c              Relative error convergence test
               ier = 0
               y  = h*aux(1)
               yd = h*auxd(1)
               return
            else if (abs(hh) .LT. 1.0) then
c              Minimum "delta-x" convergence test; < 1.0m
               ier = 0
               y  = h*aux(1)
               yd = h*auxd(1)
               return
            end if
         end if

         jj = jj+jj
      end do

c     Convergence not reached within maximum number of steps
      ier = 2
      y  = h*aux(1)
      yd = h*auxd(1)

      return
      end

      SUBROUTINE PSIDE_TOX(U1,U2,DVAL,DVALD)
C***********************************************************************
C                 PSIDE_TOX Module of ISC2 Short Term Model - ISCST2
C
C        Special version of PSIDE optimized for TOXICS applications.
C        Utilizes Romberg Integration (QATR3) or Gaussian Quadrature (QG2)
C        depending on the source receptor geometry.
C
C        PURPOSE: INTEGRATES SIDE K of POLYGON
C                 int f(u)*CNF(v(u)/sig(u))=f(u)*vn(u) from u1 to u2
C                           CNF = cumulative normal distribution
C                 Computes W(1),W(2)--normalized plume width at   u1    u2
C                 Checks for w(i) outside of -3.9,3.9 with i+, i-
C                 L=-3.9  U=3.9  = bounds for testing
C                 Integrates according to case encountered:
C                 situation     CASE    iplus    iminus  integral limits
C                 L<w1,w2<U      1        0        0         u1,u2
C                 w1,w2<L        2        0       1+2      don't compute
C                 w1,w2>U        3       1+2       0         u1,u2
C                 w1<L<w2<U      4        0        1         u-,u2
C                 w2<L<w1<U      5        0        2         u1,u-
C                 L<w1<U<w2      6        2        0       u1,u+  u+,u2
C                 L<w2<U<w1      7        1        0       u1,u+  u+,u2
C                 w1<L<U<w2      8        2        1       u-,u+  u+,u2
C                 w2<L<U<w1      9        1        2       u1,u+  u+,u-
C
C                 u+ = value of u such that w(u)=U=3.9
C                 u- =     "                w(u)=L=-3.9
C                 u+,u- computed with Brent's Algorithm
C
C                 IF uplus >0, part of side is outside plume
C                 but is integrated anyway, unless there is
C                 a corresponding part on another side that will
C                 cause cancellation.  This is determined in
C                 PSIDE2;
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED by Roger Brode, PES, Inc. to use QATR routine for
C                    integration, and to pass VN(1) and VN(2) to
C                    ZBRENT.  Also changed to use -3.9 to +3.9 for
C                    width of plume for internal consistency with
C                    SUBROUTINE PWIDTH. - 12/14/98
C
C        MODIFIED by Roger Brode, PES, Inc. to correct lower integration
C                    limit for Case 4, and to remove extraneous calls
C                    to XWIDTH and PWIDTH after calls to ZBRENT. - 7/29/94
C
C        INPUTS:  End Points of The Segments
C
C        OUTPUTS: Integral Value (if any) for Segment
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C---- Set convergence criteria for calls to QATR3:
C         NDIM = maximum number of integration steps
C         IMIN = minimum number of integration steps
C         EPSR = relative error tolerance for integral
C         EPST = minimum value threshold for integral
C----
      INTEGER, PARAMETER :: NDIM = 10, IMIN = 4
      REAL, PARAMETER    :: EPSR = 2.0E-2, EPST = 1.0E-5

C---- Set distance factor for switching to Gaussian Quadrature, QG_FACT
C     If Xmax - Xmin is .LT. QG_FACT*Xmin, then use QG2, where
C     Xmax and Xmin are the distances to the endpoints of the side.
      REAL, PARAMETER    :: QG_FACT = 5.0

      INTEGER :: I, KS, IMINUS, IPLUS, NOUT, ICON
      REAL    :: DVAL, DVALD, U1, U2, UMINUS, UPLUS, AUX(NDIM),
     &           u(2), v1(2), vn(2), w(2)

C     Variable Initializations
      MODNAM = 'PSIDE_TOX'

C     NSEG = number of segments; set to 0 in AREAIN
C     for each source/rcvr/time step
      dval  = 0.0
      dvald = 0.0

      DO i =  1,2
         ks    = ivert + i-1
         u(i)  = uvert(ks)
         v1(i) = vvert(ks)
         vn(i) = vnvert(ks)
         w(i)  = wvert(ks)
      END DO

      iminus = 0
      iplus  = 0
      uminus = -1.0
      uplus  = -1.0
      DO i = 1,2
         IF (vn(i) .lt. -3.9) iminus = i + iminus
         IF (vn(i) .gt.  3.9) iplus  = i + iplus
      END DO

      IF (iplus.EQ.1 .or. iplus.EQ.2) THEN
         call zbrent( 1,u(1),u(2),vn(1),vn(2),1.0,uplus)
      END IF
      IF (iminus.EQ.1 .or. iminus.EQ.2) THEN
         call zbrent(-1,u(1),u(2),vn(1),vn(2),1.0,uminus)
      END IF

c---- CASE DEPENDs on iplus, iminus
      IF (iplus .EQ. 0) THEN
         IF (iminus .EQ. 0) THEN
c                                             iplus  iminus  case
c                                               0     0       1
            if (abs(u2-u1) .lt. QG_FACT*min(u1,u2)) then
               call qg2(u1,u2,dval,dvald)
            else
               call qatr3(u1,u2,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         ELSE IF (iminus .EQ. 3) THEN
c                                               0     3       2
            dval = 0
         ELSE IF (iminus .EQ. 1) THEN
c                                               0     1       4
            if (abs(u2-uminus) .lt. QG_FACT*min(uminus,u2)) then
               call qg2(uminus,u2,dval,dvald)
            else
               call qatr3(uminus,u2,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         ELSE
c                                               0     2       5
            if (abs(uminus-u1) .lt. QG_FACT*min(u1,uminus)) then
               call qg2(u1,uminus,dval,dvald)
            else
               call qatr3(u1,uminus,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         END IF

      ELSE IF (iplus .EQ. 1) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = uplus
         IF (iminus .EQ. 0) THEN
c                                               1     0       7
            if (abs(u2-uplus) .lt. QG_FACT*min(uplus,u2)) then
               call qg2(uplus,u2,dval,dvald)
            else
               call qatr3(uplus,u2,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         ELSE
c                                               1     2       9
            if (abs(uminus-uplus) .lt. QG_FACT*min(uplus,uminus)) then
               call qg2(uplus,uminus,dval,dvald)
            else
               call qatr3(uplus,uminus,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         END IF

      ELSE IF (iplus .EQ. 2) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = uplus
         ubsegs(nsegs) = u2
         IF (iminus .EQ. 0) THEN
c                                               2     0       6
            if (abs(uplus-u1) .lt. QG_FACT*min(u1,uplus)) then
               call qg2(u1,uplus,dval,dvald)
            else
               call qatr3(u1,uplus,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         ELSE
c                                               2     1       8
            if (abs(uplus-uminus) .lt. QG_FACT*min(uminus,uplus)) then
               call qg2(uminus,uplus,dval,dvald)
            else
               call qatr3(uminus,uplus,epsr,epst,ndim,imin,dval,dvald,
     &                    icon,nout,aux)
            end if
         END IF

      ELSE
c                                               3     0       3
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = u2
      END IF

      RETURN
      END

      SUBROUTINE PSIDE2_TOX(DVAL,DVALD)
C***********************************************************************
C                 PSIDE2_TOX Module of ISC2 Short Term Model - ISCST2
C
C        Special version of PSIDE2_TOX optimized for TOXICS applications.
C        Utilizes Romberg Integration (QATR3) or Gaussian Quadrature (QG2)
C        depending on the source receptor geometry.
C
C        PURPOSE: Integrates Over Segments For Which ABS(VN) > VNTEST
C                 Eliminates Overlap of Segments And Useless Integration
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:   Number of The Original Segments
C                  End Points Array of The Segments
C
C        OUTPUT:   The Correction of The Results From PSIDE
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C---- Set convergence criteria for call to QATR3:
C         NDIM = maximum number of integration steps
C         IMIN = minimum number of integration steps
C         EPSR = relative error tolerance for integral
C         EPST = minimum value threshold for integral
C----
      INTEGER, PARAMETER :: NDIM = 10, IMIN = 4
      REAL, PARAMETER    :: EPSR = 2.0E-2, EPST = 1.0E-5

C---- Set distance factor for switching to Gaussian Quadrature, QG_FACT
C     If Xmax - Xmin is .LT. QG_FACT*Xmin, then use QG2, where
C     Xmax and Xmin are the distances to the endpoints of the side.
      REAL, PARAMETER    :: QG_FACT = 5.0

      INTEGER :: I, J, ISEG, NPTS, NOUT, ICON
      REAL    :: DVAL, DVALD, TEMP, U1, U2, UAV, UBV, TMPVAL, TMPVALD,
     &           AUX(NDIM)
      real ulist(nvmax2), useg(nvmax,2)
      integer usign(nvmax), ufac, usegf(nvmax)
      LOGICAL Ltest1,Ltest2

C     Variable Initializations
      MODNAM = 'PSIDE2_TOX'

      j = 1
      DO i = 1, nsegs
         ulist(j) = uasegs(i)
         j = j+1
         ulist(j) = ubsegs(i)
         j = j+1
      END DO
      npts = 2*nsegs

      call hpsort(npts,ulist,nvmax2)

      DO i = 1, nsegs
         usign(i) = 1
         IF (uasegs(i) .GT. ubsegs(i)) THEN
            usign(i) = -1
            temp = uasegs(i)
            uasegs(i) = ubsegs(i)
            ubsegs(i) = temp
         END IF
         IF (uasegs(i) .EQ. ubsegs(i)) usign(i) = 0
      END DO
      iseg = 0

      DO i = 2,npts
         u1 = ulist(i-1)
         u2 = ulist(i)
         ufac = 0
c*****
c           compare segment [u1,u2] against each ua,ub
c*****
         IF (u1 .ne. u2) THEN
            DO j = 1, nsegs
               IF (u1.ge.uasegs(j) .AND. u2 .le. ubsegs(j)) THEN
                  ufac = ufac + usign(j)
               END IF
            END DO
c*****
c              make table of segments and factors
c*****
            IF (ufac .ne. 0) THEN
               iseg = iseg+1
               useg(iseg,1) = u1
               useg(iseg,2) = u2
               usegf(iseg) = ufac
            END IF
         END IF
      END DO
c*****
c            CONSOLIDATE SEGMENTS IF iseg>1
c*****
      nsegs = iseg
      IF (nsegs .gt. 1) THEN
         DO iseg = 2, nsegs
            Ltest1 = useg(iseg,1) .EQ. useg(iseg-1,2)
            Ltest2 = usegf(iseg)*usegf(iseg-1) .gt. 0
            IF (Ltest1 .AND. Ltest2) THEN
               usegf(iseg-1) = 0
               useg(iseg,1) = useg(iseg-1,1)
            END IF
         END DO
      END IF
      dval  = 0.0
      dvald = 0.0
      IF (nsegs .gt. 0) THEN
         DO iseg = 1, nsegs
            IF (usegf(iseg) .ne. 0) THEN
               uav = useg(iseg,1)
               ubv = useg(iseg,2)
               ufac = usegf(iseg)
               if (abs(ubv-uav) .lt. QG_FACT*min(uav,ubv)) then
                  call qg2(uav,ubv,tmpval,tmpvald)
               else
                  call qatr3(uav,ubv,epsr,epst,ndim,imin,tmpval,tmpvald,
     &                       icon,nout,aux)
               end if
               dval  = dval + ufac*tmpval
               dvald = dvald + ufac*tmpvald
            END IF
         END DO
      END IF

      RETURN
      END

C
C     ..................................................................
C
C        SUBROUTINE QG2
C
C        PURPOSE
C           TO COMPUTE INTEGRAL(FCT(X), SUMMED OVER X FROM XL TO XU)
C
C        USAGE
C           CALL QG3 (XL,XU,FCT,Y)
C           PARAMETER FCT REQUIRES AN EXTERNAL STATEMENT
C
C        DESCRIPTION OF PARAMETERS
C           XL     - THE LOWER BOUND OF THE INTERVAL.
C           XU     - THE UPPER BOUND OF THE INTERVAL.
C           FCT    - THE NAME OF AN EXTERNAL FUNCTION SUBPROGRAM USED.
C           Y      - THE RESULTING INTEGRAL VALUE.
C
C        REMARKS
C           NONE
C
C        SUBROUTINES AND FUNCTION SUBPROGRAMS REQUIRED
C           THE EXTERNAL FUNCTION SUBPROGRAM FCT(X) MUST BE FURNISHED
C           BY THE USER.
C
C        METHOD
C           EVALUATION IS DONE BY MEANS OF 2-POINT GAUSS QUADRATURE
C           FORMULA, WHICH INTEGRATES POLYNOMIALS UP TO DEGREE 3
C           EXACTLY.
C           FOR REFERENCE, SEE
C           V.I.KRYLOV, APPROXIMATE CALCULATION OF INTEGRALS,
C           MACMILLAN, NEW YORK/LONDON, 1962, PP.100-111 AND 337-338.
C
C     ..................................................................
C
      SUBROUTINE QG2(XL,XU,Y,YD)
C
C
      IMPLICIT NONE

      REAL :: A, B, Y, YD, XL, XU, P1, P2, P3, P1D, P2D, P3D

      A = .5*(XU+XL)
      B = XU-XL
      Y = .2886751*B
      CALL PLUMEF(A+Y,P1,P1D)
      CALL PLUMEF(A-Y,P2,P2D)
      Y = .5*B*(P1+P2)
      YD= .5*B*(P1D+P2D)

      RETURN
      END
