        SUBROUTINE SEQMOD
C-----------------------------------------------------------------------
C PURPOSE: THIS IS THE MAIN SUBROUTINE OF CTDMPLUS. PROGRAM LOOPS IN TIME,
C               IN STACKS, IN HILLS AND IN RECEPTORS OCCUR HERE.
C               CONCENTRATIONS ARE CALCULATED IN THIS ROUTINE.
C
C ASSUMPTIONS: FOR HOURS WITH MISSING METEOROLOGICAL DATA,
C               CONCENTRATIONS ARE SET TO -999 FOR ALL RECEPTORS FOR THE
C               HOUR. ALSO, IF ANY PLUME IS IN AN UNSTABLE LAYER, THE
C               CONCENTRATIONS ARE SET TO -999 FOR THE HOUR.
C
C I/O:
C  INPUT:
C       UNIT=INSFC              FILE=SURFACE
C       UNIT=INPROF             FILE=PROFILE
C       UNIT=INEMIS             FILE=EMISSIONS
C       UNIT=INREC              FILE=RECEPTOR
C       UNIT=INRAW              FILE=RAWIN
C
C  OUTPUT:
C       UNIT=IOUT               FILE=LIST
C       UNIT=IOCONC             FILE=CONCENTRATIONS
C
C       BINARY CONCENTRATION FILE FORMAT IS:
C       4 INTEGER*2 (8 BYTES) + NRECPT REAL*4 CONCENTRATIONS PER RECORD
C       THE VALUES OF THE INTEGER HEADER VARIABLES ARE:
C       (1) = 2-DIGIT YEAR
C       (2) = JULIAN CALENDER DAY
C       (3) = HOUR (1-24)
C       (4) = NUMBER OF THE RECEPTOR WITH THE HIGHEST HOURLY CONC.
C
C
C COMMON BLOCKS: (ALL COMMON BLOCKS FOUND ANYWHERE IN CTDMPLUS ARE USED
C                 HERE, WITH DESCRIPTIONS OF VARIABLES)
C
C
C       CONST   HEAD    HILL    IO      PARAMS  PASL    PASW    PASVAL
C       PROFIL  RECEPT  SFCMET  STACK   STACKS  TIME    TOWER   VARS
C
C
C CALLING ROUTINES:
C       CTDMPLUS (MAIN)
C
C-----------------------------------------------------------------------
C
        PARAMETER(CHIPRT=0.005)
C
      INCLUDE 'PARAMS.INC'
      INCLUDE 'CONST.CMN'
      INCLUDE 'HEAD.CMN'
      INCLUDE 'HILL.CMN'
      INCLUDE 'IO.CMN'
      INCLUDE 'PARAMS.CMN'
      INCLUDE 'PASL.CMN'
      INCLUDE 'PASW.CMN'
      INCLUDE 'PASVAL.CMN'
      INCLUDE 'PROFIL.CMN'
      INCLUDE 'RECEPT.CMN'
      INCLUDE 'SFCMET.CMN'
      INCLUDE 'STACK.CMN'
      INCLUDE 'STACKS.CMN'
      INCLUDE 'TIME.CMN'
      INCLUDE 'TOWER.CMN'
      INCLUDE 'VARS.CMN'
C
C       DEFINE LOCAL VARIABLES (THESE ARE EXPLAINED AS THEY APPEAR
C       IN THE CODE)
C
        REAL    BASEHL, BASEHW, BETEST, BSTKTP, CHIMAX,
     $          COSELW, COSFLO, CSD,
     $          DELH, DELT, DELU, DELZ, DIA, DTH,
     $          DUMY, DUMZ, DX, DY, EPS, FRACT, FRHILL(MAXREC),
     $          HCHILL(MAXREC), HTOPS, HTWRAP, 
     $          PHIR, QS, RMU, RNU, ROTELW, ROTFLO,
     $          RSHL, RSHW, S, SIGRAD, SIGTH(MAXLEV), SINELW, SINFLO,
     $          SMUW, SND, SNUW, SPREAD, SQGAMA, SVMIN, SYS, TLIFT,
     $          TMUW, TNUW, TSZS, TWRAP, U1, U2, UCGAMA, USRAT, WSRT, 
     $          X, XHILLL, XHILLW, XRMAJ,
     $          XSEL, XSEW, XSMAJW, XTEW, XTMAJW, Y, YHILLL,
     $          YHILLW, YRMAJ, YSEL, YSEW, YSMAJW,
     $          YTEW, YTMAJW, YTSTK, Z1, Z2, ZG, ZTSTK
        INTEGER EOF, I, IEND, IFLAG, IFLOW, IHROUT, INDEX, IRISE, 
     $          IUPW(MAXREC), JDY, JHR, JMO, JYR, KHILL(MAXREC), 
     $          KLOW, NH, NHL, NRMAX, NS, WDGOOD, WSGOOD, 
     $          SVGOOD, SWGOOD,OLDAY
        INTEGER*2       MET(5)
        CHARACTER*1     GSGS(2)

        DATA GSGS/'G', 'S'/

        NO = 0
        YES = 1
        DTOR = 0.01745329
        PI = 3.1415926
        PIBY2 = 0.5 * PI
        TWOPI = PI + PI
        SQRPI = SQRT(PI)
        SQR2PI = SQRT(TWOPI)
        SQR2 = SQRT(2.0)
        ALPHA = 1.155
        UCGAMA = 0.36
        SQGAMA = 0.27
        SMALL = 0.00001
        SVMIN = 0.20
        OLDAY = 999

        IHROUT=0
100     CONTINUE
        
C       READ A LINE FROM THE 'SURFACE' FILE
        CALL RDSFC(EOF)

C       CHECK FOR END-OF-FILE
        IF(EOF .EQ. YES) GO TO 999

C -- WRITE OUT DAY NUMBER TO CONSOLE -----
C        IHROUT=IHROUT+1
C        IF(IHROUT/24*24 .EQ. IHROUT) WRITE(0,9500) IHROUT/24
C -- WRITE DATE EACH HOUR ---- DJB 2/6/91
         WRITE(0,9502) KYR,KMO,KDY,KHR
C
C       COMPUTE DATE/TIME DEPENDENT VARIABLES
        CALL SUN(KJCD,TZONE,ALAT,ALONG,TSR,TSS)
C
        IF (ICASE .GT. 0) CALL PAGE(YES)
        TA = -999.
C
C       READ FILE 'PROFILE'
C       INITIALIZE FLAGS FOR AVAILABILITY OF MET VARIABLES FOR THIS HOUR
C
        WDGOOD = NO
        WSGOOD = NO
        SVGOOD = NO
        SWGOOD = NO
C
C       SECTION FOR READING AND PROCESSING "PROFILE"
C
        DO 120 NHT = 1,MAXLEV
                             
C
C        SEE TABLE 3-7 IN CTDMPLUS USER'S GUIDE
C
            READ(INPROF, *,END = 999) JYR,JMO,JDY,JHR,HT(NHT),IEND,
     1                          WDHR(NHT),WSHR(NHT),TAHR(NHT),
     2                          SIGTH(NHT),SWHR(NHT),UVHR(NHT)
C
C        CHECK FOR ERRORS IN INPUT: DATE/TIME INCONSISTENCIES,
C        NEGATIVE HEIGHTS OR HEIGHTS NOT MONOTONICALLY INCREASING
C
             IF(JYR.NE.KYR .OR. JMO.NE.KMO .OR. JDY.NE.KDY .OR.
     1                                          JHR.NE.KHR) THEN
                WRITE(IOUT,6105) JMO,JDY,JYR,JHR,KMO,KDY,KYR,KHR
                STOP
            ENDIF
            IF(HT(NHT) .LT. 0.0) THEN
                WRITE(IOUT,6107) JMO,JDY,JYR,JHR
                WRITE(IOUT,6108) NHT, (HT(NHT)-BASEHT)
                STOP
            ENDIF
            IF(NHT .GT. 1) THEN
                IF( HT(NHT) .LT. (HT(NHT-1) - BASEHT)) THEN
                    WRITE(IOUT,6107) JMO,JDY,JYR,JHR
                    WRITE(IOUT,6109) NHT, (HT(NHT-1)-BASEHT),
     1                                    (HT(NHT)-BASEHT)
                    STOP
                ENDIF
            ENDIF

C           HEIGHT NOW REFERENCED TO STACK BASE ELEVATION NOT TOWER
            HT(NHT) = HT(NHT) + BASEHT
            IF(TA.LT.0.0 .AND. TAHR(NHT).GT.0.0) TA = TAHR(NHT)
C           CHECK FOR ALL MISSING WIND SPEED AND DIRECTION
            IF(WDHR(NHT) .GE. 0.0) WDGOOD = YES
            IF(WSHR(NHT) .GT. 0.0) WSGOOD = YES
            IF(SWHR(NHT) .GE. 0.0) SWGOOD = YES
C
C  SET ZERO WIND SPEEDS TO 0.01.  THIS IS DONE TO AVOID THAT RARE CASE
C  WHERE THE PLUME IS ABOVE THE HIGHEST REPORTED METEOROLOGY LAYER, AND
C  THEREFORE THAT WIND SPEED IS USED IN GETWS AND ULTIMATELY URISE, WHICH
C  USES GETWS AS A DIVISOR WHICH RESULTS IN A DIVIDE-BY-ZERO ERROR UNLESS
C  THE WIND IS MADE NON-ZERO.   -CSC
C
            IF(WSHR(NHT) .EQ. 0.0) WSHR(NHT) = 0.01
C
C       USE YAMARTINO (1984) TO OBTAIN SIGV, UV (IF NECESSARY)
C
            IF(ISIGV .EQ. 0) THEN
C
C       HERE, SIGMA-THETA IS PROVIDED (ISIGV = 0), MAX SIGTH IS 103.9
C
                IF(SIGTH(NHT).LE.0.0 .OR. WSHR(NHT).LE.0.0) THEN
                    SVHR(NHT) = -999.9
                    IF(UVHR(NHT).LE.0.0) UVHR(NHT) = -999.9
                 ELSE
                    SIGTH(NHT) = AMIN1(103.9,SIGTH(NHT))
                    SIGRAD = SIGTH(NHT) * DTOR                    
                    EPS = SIN(SIGRAD * (1.0 - 0.073864*SIGRAD))
                    IF(UVHR(NHT).LE.0.0) UVHR(NHT) = WSHR(NHT) *
     1                                          SQRT(1.0 - EPS*EPS)
                    SVHR(NHT) = SIGRAD * UVHR(NHT)
                    SVGOOD = YES
                ENDIF
            ELSE
C
C       HERE, SIGMA-V IS PROVIDED (ISIGV = 1)
C
                IF(SIGTH(NHT) .LT. 0.0) SIGTH(NHT) = -999.9
                SVHR(NHT) = SIGTH(NHT)
                IF(UVHR(NHT) .LE. 0.0  .AND. WSHR(NHT) .GT. 0.0
     &                                  .AND. SVHR(NHT) .GE. 0.0) THEN
                    SIGRAD = SVHR(NHT) / WSHR(NHT)
                                                  
C
C                   STORE SIGMA-THETA VALUES IN DEGREES (SIGMA-THETA
C                   CANNOT EXCEED 103.9 DEGREES)
C
                    SIGRAD = AMIN1(103.9*DTOR,SIGRAD)
                    SIGTH(NHT) = SIGRAD/DTOR
                    EPS = SIN(SIGRAD * (1.0 - 0.073864*SIGRAD))
                    UVHR(NHT) = WSHR(NHT) * SQRT(1.0-EPS*EPS)
                 ELSE
                    IF(UVHR(NHT) .LE. 0.0) UVHR(NHT) = -999.9
                ENDIF
                IF(SVHR(NHT) .GT. 0.0) SVGOOD = YES
            ENDIF
C
C       CHECK FOR SCALAR WIND SPEED LESS THAN 1 M/SEC, RATIO SCALAR,
C       VECTOR SPEEDS AND SIGW, SIGV UPWARD IF NECESSARY
C
            IF(WSHR(NHT) .GT. 0.0 .AND. UVHR(NHT).GT. WSHR(NHT))
     1          UVHR(NHT) = WSHR(NHT)
            IF( IWS1 .EQ. YES ) THEN
                IF( WSHR(NHT) .LT. 1.0 ) THEN
                    IF( WSHR(NHT) .GT. 0.0 ) THEN
                        WSRT = 1.0/WSHR(NHT)
                        WSHR(NHT) = 1.0
                        IF(UVHR(NHT) .GT. 0.0) UVHR(NHT)=UVHR(NHT)*WSRT
                        IF(SVHR(NHT) .GT. 0.0) SVHR(NHT)=SVHR(NHT)*WSRT
                        IF(SWHR(NHT) .GT. 0.0) SWHR(NHT)=SWHR(NHT)*WSRT
                    ENDIF
                ENDIF
            ENDIF
C
C       IN STABLE CONDITIONS, A VALUE OF SIGMA-V BELOW 0.2 M/S IS
C       SET TO 0.2 M/S, WHETHER OBSERVED OR CALCULATED FROM SIGMA-THETA
C       AND WIND SPEED
C
            IF(SVHR(NHT) .GE. 0.0) THEN
                IF(EL .GT. 0.0 .OR. HT(NHT) .GT. XMH) SVHR(NHT) =
     1             AMAX1(SVHR(NHT), SVMIN)
            ENDIF

C           CHECK FOR LAST PROFILE RECORD
            IF( IEND .EQ. YES ) GO TO 130
120     CONTINUE
C
C       END SECTION FOR PROCESSING "PROFILE" DATA
C
130     IF(TA .LT. 0.0) THEN
            TA = 293.
            TAHR(1) = 293.
        ENDIF
C
C       TEST FOR VARIABLE EMISSION RATES
        IF(IEMIS .EQ. YES) CALL INPEMS
C
C       CHECK FOR MISSING MET DATA; IF MISSING, SKIP CALCULATIONS
C       FOR THIS HOUR. FOR SIGMA-V AND SIGMA-W CHECK AT STABLE/UNSTABLE
C       SPLIT
C
        IF( USTAR0 .LT. 0.0 ) GO TO 330
        IF( Z0 .LT. 0.0 ) GO TO 340
        IF( WSGOOD.EQ.NO  .OR. WDGOOD.EQ.NO ) GO TO 350
C
        IF(ISIGV .EQ. NO) SIGV = -9.9
C
C       DETERMINE A STABILITY CATEGORY (FOR MISCELLANEOUS APPLICATIONS),
C       USING THE GOLDER (1972) CURVES (GIVEN L, U*)
C
        KST = LSTAB(EL,Z0)
C
C       WRITE CASE STUDY PRINTOUT OF METEOROLOGY
C
        IF(ICASE .GT. 0) THEN
            WRITE(IOUT,6115) KYR,KMO,KDY,KHR,XMH,TA,USTAR0,EL,Z0
            DO 150 I = 1,NHT
                WRITE(IOUT,6120) HT(I),WDHR(I),WSHR(I),UVHR(I),TAHR(I),
     1          SIGTH(I),SVHR(I),SWHR(I)
150         CONTINUE
            IF(IWS1 .EQ. 1) THEN
                WRITE(IOUT,6122) BASEHT
             ELSE
                WRITE(IOUT,6123) BASEHT
            ENDIF
        ENDIF
C
C       INITIALIZE & ZERO HOURLY CONCENTRATIONS
          DO  160 NR=1,NRECPT
            CONC(NR) = 0.0
            DO 155 NS = 1,NSTACK
              SCONC(NR,NS) = 0.0
155         CONTINUE
160       CONTINUE
        NRMAX = 0
C

C       PRELIMINARY LOOP ON HILLS: COMPUTE HCRIT AND FROUDE NUMBER
C       AND STORE FOR USE BELOW
C
        DO 180 NH = 1,NHILLS
            HCHILL(NH) = HCRIT(THS(NH),XMH)
            FRHILL(NH) = BULKFR(THS(NH),HCHILL(NH))
            IF(HCHILL(NH) .LT. 0.0) HCHILL(NH) = 0.0
C           LARGE FROUDE NUMBER ESSENTIALLY NEUTRAL CONDITIONS
            IF(FRHILL(NH) .GE. 99.99) FRHILL(NH) = 99.99
180     CONTINUE

C       START LOOP ON STACKS ------------------------------------------

        DO  300  NS=1,NSTACK

C       SET UP STACK COMMON VARIABLES; SEE STACKS.CMN FOR DEFINITIONS

        QS = SOURCE(8,NS)
        IF (ICHIQ .EQ. YES .AND. QS .GT. 0.0) THEN
          QS = 1.0
        ELSE IF (ICHIQ .EQ. YES .AND. QS .LT. 0.0) THEN
          QS = -1.0
        ENDIF
        XS = SOURCE(1,NS)
        YS = SOURCE(2,NS)
C------ ZS = SOURCE(3,NS)
        HS = SOURCE(4,NS)
        DS = SOURCE(5,NS)
        TS = SOURCE(6,NS)
        VS = SOURCE(7,NS)
C       HB (BUILDING HEIGHT) IS NOT USED IN CURRENT MODEL
        HB = SOURCE(9,NS)
        IF(TS .LT. TA) TS = TA
        DELT = (TS-TA) / TS

C        IF NO EMISSIONS, SKIP CALCULATIONS FOR THIS STACK

C        IF(QS .LE. 0.0) THEN
        IF(QS .EQ. 0.0) THEN
            IF(ICASE .GT. 0) WRITE(IOUT,6125) NS
            GO TO 300
        ENDIF
        FB = SOURCE(10,NS) * DELT
C       FM COMPUTED FOR INFORMATIONAL PURPOSES ONLY
        FM = SOURCE(11,NS) * TA / TS
        USTKTP = GETWS(HS)

C       FOR UNSTABLE HOURS, SET WSTAR/US TO A MIN OF 0.167
        IF (EL .LT. 0 .AND. EL .GT. -100) THEN
          WSTAR = AMAX1(0.167*USTKTP,WSTAR0)
        ENDIF

C   CALCULATE THE FOLLOWING FOR PLUME RISE SUBROUTINES
        IF (FB .LE. 55) THEN
          XSTAR = 14.0 * FB**0.625
        ELSE
          XSTAR = 34.0 * FB**0.40
        ENDIF
        XFIN = 3.5 * XSTAR

C       CALCULATE FINAL PLUME RISE

C           STABLE SECTION
        IF(EL .GT. 0.0  .OR.  HS .GT. XMH) THEN
            IFLAG = NO
            CALL SRISE(IFLAG, DELH, IRISE)
C
C       NOTE: XSZS IS DISTANCE TO WHERE TURBULENCE DOMINATES SOURCE-
C       INDUCED EFFECTS.  FOR STABLE CONDITIONS, THIS IS ASSUMED TO BE
C       THE DISTANCE TO FINAL RISE; FOR NONSTABLE CONDITIONS, THIS
C       DISTANCE IS ASSUMED TO BE NEARLY ZERO.
C
            IF(IRISE .LE. 9) THEN
                XSZS = 0.0
              ELSE
                BSTKTP = SQRT((9.8/TA) * GETDTH(HS,XMH))
                XSZS = 2.07 * USTKTP/BSTKTP
            ENDIF
          ELSE
C
C           UNSTABLE SECTION
            CALL URISE(DELH, IRISE, USTKTP, US)
            XSZS = 0.0
        ENDIF

C       COMPUTE PLUME HEIGHT
        HPL = DELH + HS

C
C   DETERMINE WHETHER THIS SOURCE SHOULD BE MODELED AS STABLE OR.
C   UNSTABLE. AFTER DAYTIME CALCULATION GOTO 300 TO GO TO THE NEXT 
C   SOURCE

        IF (HS.LE.XMH .AND. ((EL.LT.0.0 .AND. EL.GT. -100) .OR.
     $                                -XMH/EL .GT. 10)) THEN

C   ZERO DTH BECAUSE IT IS NOT USED IN UNSTABLE CALCULATIONS
         DTH = 0.0

C   IF IUNSTA = NO, SKIP THIS HOUR AND WRITE -999 TO CONC FILE
         IF (IUNSTA .EQ. NO) GO TO 390

C   WRITE WARNING MESSAGE IF UNSTABLE DURING NIGHTTIME
         IF (KHR .GT. TSS  .OR. KHR .LT. TSR) 
     $      WRITE(IOUT,6104) NS,KYR,KMO,KDY,KHR

C   IF BOTH MIXING HEIGHTS ARE MISSING IN DAYTIME, SKIP THIS HOUR

          IF (XMH .EQ. 99999)  GO TO 380
 
          CALL DAYCALC(NS,OLDAY,DELH,QS)
          GOTO 300
        ENDIF


C   IF IUNSTA = 2, SKIP THIS STABLE HOUR AND WRITE -999 TO CONC FILE
         IF (IUNSTA .EQ. 2) GO TO 390

C    CHECK IF WE HAVE AT LEAST ONE VALUE OF SIGMA-V AND SIGMA-W
C    FOR STABLE/NEUTRAL HOURS
        IF (SVGOOD.EQ.NO  .OR. SWGOOD .EQ. NO ) GO TO 350

C       CALCULATE INITIAL SIGMAS DUE TO BUOYANCY

        CALL SIGB(DELH, SYS, SZS)                     

C       DETERMINE PLUME HEIGHT METEOROLOGY

C       GET WIND SPEED (US), WIND DIRECTION (THTA), AND VERTICAL POT.
C       TEMPERATURE LAPSE RATE (DTH) AT PLUME HEIGHT
        CALL PLAVG(HPL, US, THTA, DTH,XMH)

C       CHECK IF WIND SPEED LESS THAN 1.0 M/S AT PLUME HEIGHT

        USRAT = 1.0
        IF( US .LT. 1.0 ) THEN
                IF( IWS1 .EQ. NO ) GO TO 370
C               SET RATIO TO INCREASE UV, SIGW AND SIGV
                USRAT = 1.0 / US
                US = 1.0
        ENDIF
        IF(EL .LT. 0.0) DTH = 0.0
        SIGW = GETSW(HPL)*USRAT
        SIGV = GETSV(HPL)*USRAT
        UV = GETUV(HPL,US,SIGV,XMH)*USRAT
        IF(UV .LT. USTAR0) UV = USTAR0
        IF(UV .GT. US) UV = US
C
C       SET MINIMA FOR SIGW, SIGV: 1% OF US
C
        SIGW = AMAX1(SIGW,0.01*US)
        SIGV = AMAX1(SIGV,0.01*US)
        BRUNT = SQRT(9.8/TA * DTH)
        SND = SIN(THTA*DTOR)
        CSD = COS(THTA*DTOR)
C
C  CONVERT MEAN WIND DIRECTION FROM DEG CW FROM N TO DEG CCW FROM N
C          AND CHANGE TO FLOW DIR IN RADIANS
C
      PHIM = (180.-THTA)*DTOR
      IF(PHIM .LT. 0.0) PHIM = TWOPI + PHIM
C
C  ROTATE COORD. SYS. TO ALIGN ORIGINAL X-AXIS WITH THE MEAN FLOW DIR.
C
      ROTFLO = PIBY2 + PHIM
      SINFLO = SIN(ROTFLO)
      COSFLO = COS(ROTFLO)
C
C  CALCULATE VIRTUAL SOURCE TIME INCREMENT
C  TNEUT, TSTRAT, TTLZ ARE GIVEN BY EQNS 22, 23, AND 24 IN USER'S GUIDE
C
      TNEUT = SIGW/(UCGAMA*HPL)
      TSTRAT = BRUNT/SQGAMA
      TTLZ = 1./(TNEUT+TSTRAT)
C
C     COMPUTE MINIMUM VIRTUAL SOURCE TIME INCREMENT: PLUME GROWTH TO
C     STACK DIAMETER (OR SIGMA-Y,Z GROWTH TO STACK RADIUS).  IF NO
C     "STACK" IS USED (ZERO EXIT VELOCITY, AS WITH A TRACER), ASSUME
C     A DEFAULT DIAMETER OF 1 METER.
C
      DIA = DS
      IF(VS .LT. SMALL) DIA = 1.0
C
C     SEE EQN 29 OF USER'S GUIDE
C     ZTSTK IS VIRTUAL TIME FOR SIGMA-Z GROWTH DUE TO SOURCE EFFECTS
C
      DUMZ = (2.0 * SIGW/DIA)**2
      ZTSTK = (1.+SQRT(1.+16.*DUMZ*TTLZ**2))/(4.*DUMZ*TTLZ)
C
C     COMPUTE VIRTUAL SOURCE TIME INCREMENT, ZTV: PLUME GROWTH DUE TO
C     BUOYANCY (EQN 29)
      IF(SZS .LE. 0.0) THEN
          ZTV = 0.0
        ELSE
          DUMZ = (SIGW/SZS)**2
          ZTV = (1.+SQRT(1.+16.*DUMZ*TTLZ**2))/(4.*DUMZ*TTLZ)
      ENDIF
C
C     ASSUME THAT TTLY IS 10000 / VECTOR WIND SPEED (AT PLUME HT).
C     APPLY EQN 29 FOR SIGMA-Y GROWTH SIMILAR TO THAT DONE FOR SIGMA-Z.
C
      TTLY = 10000.0 / UV
      DUMY = (2.0 * SIGV/DIA)**2
      YTSTK = (1.+SQRT(1.+16.*DUMY*TTLY**2))/(4.*DUMY*TTLY)
      IF(SZS .LE. 0.0) THEN
          YTV = 0.0
        ELSE
          DUMY = (SIGV/SZS)**2
          YTV = (1.+SQRT(1.+16.*DUMY*TTLY**2))/(4.*DUMY*TTLY)
      ENDIF
C
C     FINAL CALCULATION FOR VIRTUAL TIME OF TRAVEL:
C     A) AT A MINIMUM, IT IS THE TIME FOR PLUME GROWTH TO THE STACK
C        DIAMETER SIZE;
C     B) THE TIME FOR PLUME GROWTH DUE TO AMBIENT TURBULENCE TO THE
C       SIZE RESULTING FROM PLUME BUOYANCY, MINUS THE TIME FOR
C       TURBULENT GROWTH TO EXCEED SOURCE-INDUCE EFFECTS, IS USED IF
C       GREATER THAN A).
C
      TSZS = XSZS/UV
C
C     SEE EQN 30 IN USER'S GUIDE
C
      ZTV = AMAX1(ZTSTK,ZTV-TSZS)
      YTV = AMAX1(YTSTK,YTV-TSZS)
C
C       CHECK TO SEE WHICH HILLS ARE DOWNWIND
C
        DO 220 NH=1,NHILLS
220         KHILL(NH) = NO
C
C       IF ANY RECEPTOR ON A HILL IS DOWNWIND OF THE SOURCE, THEN THE
C       ENTIRE HILL IS CONSIDERED TO BE DOWNWIND OF THE SOURCE.
C
        DO 230 NR = 1,NRECPT
            INDEX = NRHILL(NR)
            IF(INDEX .GT. 0) THEN
                IF(KHILL(INDEX) .EQ. YES) GO TO 230
            ENDIF
            XR = RECPT(1,NR)
            YR = RECPT(2,NR)
            CALL PSRCE(SND,CSD,IUPW(NR),Y,X,XS,YS,XR,YR)
            IF(IUPW(NR) .EQ. NO .AND. INDEX .GT. 0) KHILL(INDEX) = YES
230     CONTINUE

C
C****************************************************
C
C
C       CASE STUDY SOURCE PRINTOUT
C
        IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) THEN
            WRITE(IOUT,6130)
            WRITE(IOUT,6135) NS,QS,TS,VS,FB,FM,DELH
            WRITE(IOUT,6140) HPL,THTA,US,UV,SIGV,SIGW,DTH
        ENDIF
C
C       START LOOP OVER HILLS AND RECEPTORS
C
C  ------------ START LOOP ON HILLS -----------------------------------
C
      DO  270  NHL = 0, NHILLS
C
C         IF NO RECEPTORS IN FLAT TERRAIN SKIP TERRAIN SECTION
          IF(NHL .EQ. 0) THEN
C               CHECK FOR NO FLAT TERRAIN (HILL 0) RECEPTORS
                IF( NRFLAT .EQ. NO ) GO TO 270
                GO TO 240
          ENDIF
C
C         IF ALL RECEPTORS ON HILL ARE UPWIND THEN SKIP ENTIRE LOOP
C
          IF( KHILL(NHL) .EQ. NO ) THEN
              IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $                WRITE(IOUT,6265) NHL, NS, THTA
              GO TO 270
          ENDIF
C
          HC = HCHILL(NHL)
          FR = FRHILL(NHL)
          HTOPS = THS(NHL)
          Z0HILL = Z0H(NHL)
C
C  SECTION FOR DEFINING GEOMETRY FOR WRAP COMPUTATIONS ----------------
C         CRITICAL HEIGHT FOR WRAP (HPL - PLUME HT ABOVE STACK BASE)
C
          HTWRAP = AMIN1( HPL, HC )
C
C         GET HILL COORDS, ANGLE AND MAJOR, MINOR AXIS FOR THE WRAP HILL
C         W AT END OF VARIABLE INDICATE WRAP HILL
C         KLOW IS THE ARRAY INDEX TO THE HEIGHT CLOSEST TO (BUT LESS
C         THAN) HTWRAP
          KLOW = KLOSE( ZHS(1,NHL), NZH(NHL), HTWRAP )
          IF( KLOW .EQ. 0 ) KLOW = 1
          BASEHW = ZHS(KLOW,NHL)
          IF( KLOW .EQ. NZH(NHL) ) THEN
C             THE CRITICAL HT IS ABOVE LAST CONTOUR VALUE
C             DO NOT INTERPOLATE, BUT USE INVERSE POLYNOMIAL FORMULA
C
C             GET X, Y, ORIENTATION OF THE ELLIPSE FOR WRAP FROM THE
C             LOOK-UP TABLE.
C
              XHILLW = XHW(KLOW,NHL)
              YHILLW = YHW(KLOW,NHL)
              THTAH = MAJORW(KLOW,NHL)
C             DISTANCE SOURCE TO  HILL CENTER
              DX = XS - XHILLW
              DY = YS - YHILLW
              RSHW = SQRT( DX*DX + DY*DY )
C
C             COMPUTE MAJOR & MINOR AXIS LENGTHS: INTERPOLATE BETWEEN
C             MAJAXW, MINAXW AND 0.0 IF ABOVE THE LAST CONTOUR
C
              FRACT =  1.0 - (HTWRAP - BASEHW)/(HTOPS - BASEHW)
              AAXW = MAJAXW(KLOW,NHL) * FRACT
              BAXW = MINAXW(KLOW,NHL) * FRACT
           ELSE
C             LINEARLY INTERPOLATE BETWEEN TWO VALUES
              XHILLW = XINTRP( ZHS(KLOW,NHL), ZHS(KLOW+1,NHL),  HTWRAP,
     *                         XHW(KLOW,NHL), XHW(KLOW+1,NHL))
              YHILLW = XINTRP( ZHS(KLOW,NHL), ZHS(KLOW+1,NHL),  HTWRAP,
     *                         YHW(KLOW,NHL), YHW(KLOW+1,NHL))
              AAXW = XINTRP( ZHS(KLOW,NHL), ZHS(KLOW+1,NHL),  HTWRAP,
     *                       MAJAXW(KLOW,NHL), MAJAXW(KLOW+1,NHL))
              BAXW = XINTRP( ZHS(KLOW,NHL), ZHS(KLOW+1,NHL),  HTWRAP,
     *                       MINAXW(KLOW,NHL), MINAXW(KLOW+1,NHL))
              THTAH = ANGINT( MAJAXW(KLOW,NHL),MINAXW(KLOW,NHL),
     *                        MAJORW(KLOW,NHL), ZHS(KLOW,NHL),
     *                        MAJAXW(KLOW+1,NHL),MINAXW(KLOW+1,NHL),
     *                        MAJORW(KLOW+1,NHL), ZHS(KLOW+1,NHL),
     *                        HTWRAP )
C
C               CHECK THAT ELLIPSE AXES ARE LESS THAN DIST FROM SOURCE
C               TO HILL CENTER   AAX <= BAX < DIST SOURCE/HILL CENTER
                DX = XS - XHILLW
                DY = YS - YHILLW
                RSHW = SQRT( DX*DX + DY*DY )
C                IF( AAXW .GT. RSHW ) AAXW = 0.99 * RSHW
C                IF( BAXW .GT. AAXW ) BAXW = AAXW
          ENDIF
C
C         SHIFT ORIGIN TO CENTER OF WRAP HILL (E DENOTES ELLIPSE)
          XSEW = XS - XHILLW
          YSEW = YS - YHILLW
          XTEW = XT - XHILLW
          YTEW = YT - YHILLW
C
C         FIND TOWER AND SOURCE LOCATIONS RELATIVE TO MAJOR AXIS OF
C         WRAP HILL ELLIPSE. NOTE: X-AXIS LIES ALONG MAJOR AXIS
          ROTELW = (90.0 - THTAH) * DTOR
          COSELW = COS(ROTELW)
          SINELW = SIN(ROTELW)
          XTMAJW =  XTEW*COSELW + YTEW*SINELW
          YTMAJW = -XTEW*SINELW + YTEW*COSELW
          XSMAJW =  XSEW*COSELW + YSEW*SINELW
          YSMAJW = -XSEW*SINELW + YSEW*COSELW
C
C         COMPUTE ELLIPTICAL COORDS OF SOURCE AND TOWER
          CALL MUNU( XSMAJW, YSMAJW, AAXW, BAXW, SMUW, SNUW )
          CALL MUNU( XTMAJW, YTMAJW, AAXW, BAXW, TMUW, TNUW )
C
C         SET UP DATA FOR FLOW BELOW HC (WRAP)
          CALL WRAPIN( TMUW, TNUW, SMUW, SNUW, XSMAJW, 
     *                 YSMAJW )
C
C         CHECK THE WRAP/LIFT TIME RATIO (TFAC). NOTE THAT THE
C         TIME TO LIFT USES 'SO' WHICH IS COMPUTED BY WRAPIN USING
C         THE WRAP HILL ELLIPSE VARIABLES
C
          TFAC = 0.0
          TWRAP = SOBETA / UVBETA + ZTV
          TLIFT = SO / UV + ZTV
          IF(TLIFT .LT. 0.0) THEN
              IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $            WRITE(IOUT,6270) NHL, NS, THTA
              GO TO 270
          ENDIF
          IF( TLIFT .NE. 0.0 ) TFAC = TWRAP/TLIFT
C         COMPUTE SIGMA-Z AT THE IMPINGEMENT POINT FOR LIFT TIME
          SZTEST = SIGW * TLIFT/SQRT( 1.0 + 0.5*TLIFT/TTLZ )
C
C         WRITE OUT HILL INFO AND WRAP VARIABLES
C
          IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3) THEN
                CALL PAGE(YES)
                WRITE(IOUT,6150) NHL, (HILNAM(I,NHL),I=1,10), HC, FR
                WRITE(IOUT,6155) RSHW, HTWRAP, AAXW, BAXW, THTAH,
     1            ABS(SOBETA)
          ENDIF
C
C  SECTION FOR DEFINING THE GEOMETRY FOR LIFT COMPUTATIONS-------------
C         GET ARRAY INDEX TO CONTOUR BELOW CUT-OFF HILL
C
          KLOW = KLOSE( ZHS(1,NHL), NZH(NHL), HC )
          IF( KLOW .EQ. 0 ) KLOW = 1
C
C         OBTAIN X, Y, ORIENTATION OF CUT-OFF HILL FROM LOOK-UP TABLES
C         FOR LIFT
C
          XHILLL = XHL(KLOW,NHL)
          YHILLL = YHL(KLOW,NHL)
          THTAH = MAJORL(KLOW,NHL)
          DX = XS - XHILLL
          DY = YS - YHILLL
          RSHL = SQRT( DX*DX + DY*DY )
C         CUT-OFF HILL BASE AND HEIGHT FOR LIFT
          BASEHL = ZHS(KLOW,NHL)
          HH = HTOPS - HC
          IF(KLOW .EQ. NZH(NHL)) THEN
C
C             COMPUTE MAJOR & MINOR AXIS LENGTHS: INTERPOLATE BETWEEN
C             MAJAXW, MINAXW AND 0.0 IF ABOVE THE LAST CONTOUR
C
              FRACT =  1.0 - (HC+0.5*HH - BASEHW)/(HTOPS - BASEHW)
              AAXL = MAJAXW(KLOW,NHL) * FRACT
              BAXL = MINAXW(KLOW,NHL) * FRACT
            ELSE
              CALL TERAX( SCALMA(KLOW,NHL), SCALMI(KLOW,NHL),
     *                    EXPOMA(KLOW,NHL), EXPOMI(KLOW,NHL),
     *                    (HTOPS-BASEHL), (HC+0.5*HH), BASEHL, RSHL,
     *                    AAXL, BAXL )
          ENDIF
C
C         CONVERT THTAH TO RADIANS CCW FROM N
          PHIHL=-THTAH*DTOR+TWOPI
C         SHIFT ORIGIN TO CENTER OF LIFT HILL (E DENOTES ELLIPSE)
          XSEL = XS - XHILLL
          YSEL = YS - YHILLL
C
C         ROTATE COORD SYS TO ALIGN ORIGINAL X-AXIS W/ MEAN FLOW DIR
          XSEPL =  XSEL*COSFLO + YSEL*SINFLO
          YSEPL = -XSEL*SINFLO + YSEL*COSFLO
C
C         DETERMINE SPEED SHEAR FOR FLOW; SET UP LIFT VARIABLES
C
          Z1 = HPL
          U1 = US
          Z2 = HC
          IF(ABS(Z2-Z1) .LT. 0.1 * HH) Z2 = HTOPS
          U2 = GETWS(Z2)
          DELU = U2 - U1
          IF(ABS(DELU) .LT. 0.001) DELU = 0.0
          DELZ = Z2 - Z1
          IF(DELZ .EQ. 0.0) THEN
              ALF = 0.0
            ELSE
              ALF = AMAX1(0.0,DELU/DELZ)
          ENDIF
C
          CALL LIFTIN(IFLOW)
          IF(IFLOW .GT. 0) GO TO 360
          IF(FR .LT. 0.8 .AND. (ICASE .EQ. 1 .OR. ICASE .EQ. 3))
     $        WRITE(IOUT, 6158)
C
C         WRITE LIFT INFORMATION
C
          IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) THEN
              WRITE(IOUT,6160) RSHL, HC + 0.5*HH, AAXL, BAXL, THTAH,
     1          ABS(XSEPL), YSEPL, SO
          ENDIF
C
C  SET UP COLUMN TITLES FOR RECEPTOR CASE-STUDY OUTPUT
C
240   IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3 ) THEN
        WRITE(IOUT,6170) GSGS(ICHIQ+1)
        NLINES = NLINES + 5
        IF(NHL .GT. 0) NLINES = NLINES + 22
      ENDIF

C  --------------- START LOOP ON RECEPTORS -----------------------------

          DO  260  NR = 1, NRECPT
C
C             CHECK IF THIS RECEPTOR IS ON CURRENT HILL
              IF( NRHILL(NR) .NE. NHL ) GO TO 260
C
C             SET UP RECEPTOR VARIABLES
C             ZELEV IS RECEPTOR HEIGHT ABOVE GROUND SURFACE
C
              XR = RECPT(1,NR)
              YR = RECPT(2,NR)
              ZELEV = RECPT(3,NR)
              ZG = RECPT(4,NR)
              Z = ZELEV + ZG
C             DO NOT ALLOW RECEPTOR TO BE BELOW THE MODEL ZERO PLANE
              IF( Z .LT. 0.0 ) Z = 0.0
C
C             INITIALIZE CONCENTRATION
              C = 0.0

C             IF OPTION SELECTED, SET CONC FOR RECEPTORS BELOW STACK 
C             TOP TO ZERO
              IF (ISTKTP .EQ. YES .AND. ZG .LT. HS) GO TO 260
C
C             IF ESSENTIALLY FLAT TERRAIN (HILL = 0), CALL FLAT AND EXIT
C
              IF(NHL .EQ. 0) THEN
                  IF(IUPW(NR) .EQ. YES) GO TO 260
                  CALL FLAT(QS,Z)
                  NLINES = NLINES + 2
                  GO TO 250
              ENDIF
C
C             START RECEPTOR COORD SETUP FOR CALL TO LIFT
C             SHIFT ORIGIN TO CENTER OF HILL
              XEL= XR - XHILLL
              YEL= YR - YHILLL
C
C             ALIGN X-AXIS WITH MEAN FLOW
              XEPL =  XEL* COSFLO + YEL* SINFLO
              YEPL = -XEL* SINFLO + YEL* COSFLO
C
C             CALC DIRECTION FROM SOURCE TO RECEPTOR (CCW FROM N)
              PHIR = ATAN2( (XSEL-XEL),(YEL-YSEL) )
              IF( PHIR .LT. 0.0 ) PHIR = PHIR + TWOPI
C
C             CHECK IF TRAJECTORY IS TOWARDS RECEPTOR FOR LIFT CALC.
              SPREAD = ABS( PHIR - PHIM )
              IF( SPREAD .GT. PI ) SPREAD = -SPREAD + TWOPI
C
              IF( SPREAD .GT. PIBY2) THEN
C                 RECEPTOR IS NOT DOWNWIND
                  IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3 ) 
     $               WRITE(IOUT,8010) NR
               ELSE
                  S = XEPL - XSEPL
C                 USE LIFT FOR ALL RECEPTORS AT OR ABOVE HC
                  IF( Z .GE. HC ) THEN
                      CALL LIFT(QS)
                    ELSE
                      IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3 ) 
     $                   WRITE(IOUT,6175) NR
                  ENDIF
              ENDIF
C
C             COMPUTE RECEPTOR COORDS RELATIVE TO WRAP HILL
C
C             FIND RECEPTOR LOCATION RELATIVE TO MAJOR AXIS OF ELLIPSE
C             SHIFT ORIGIN TO CENTER OF WRAP HILL
              XEW= XR - XHILLW
              YEW= YR - YHILLW
C
              XRMAJ =  XEW* COSELW + YEW* SINELW
              YRMAJ = -XEW* SINELW + YEW* COSELW
C
C             IS RECEPTOR ON SOURCE SIDE OF STAGNATION STREAMLINE
              CALL MUNU( XRMAJ, YRMAJ, AAXW, BAXW, RMU, RNU )
              IF( ABS(PSIHAT) .LE. SMALL ) THEN
                  SIGNYE = 1.0
               ELSE
                  SIGNYE = SIGN(1.0, -SIN(RNU+ALPHAW)/PSIHAT)
              ENDIF
C
C             CALCULATE POSITION ALONG BETA
              XRBETA = XRMAJ*COS(BETA) + YRMAJ*SIN(BETA)

C             CHECK IF TRAJECTORY IS TOWARDS RECEPTOR FOR WRAP CALC.
C             USE THE UNMODIFIED SOURCE POSITION
C              BETEST = XRBETA / XSBETA
              BETEST = XRBETA / XSBSAV
              IF( BETEST .GT. 1.0 ) THEN
C                 RECEPTOR IS NOT DOWNWIND
                  IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3 ) 
     $                      WRITE(IOUT,8020) NR
                ELSE IF( (HC/HH) .GT. SMALL ) THEN

C             FOR TRAVEL TIME OF FLOW AROUND THE ELLIPSE, IF THE RECEPTOR
C             IS OUTSIDE OF THE ELLIPSE (RMU > 0), FORCE IT TO BE ON THE
C             ELLIPSE BY RESETTING RMU AND COMPUTING A NEW X,Y

                  IF (RMU .GT. 0) THEN
                    AR = AAXW/BAXW
                    ARSQ = AR * AR
                    IF (ARSQ .LT. 1.0) ARSQ = 1.0
                    DUM1 = SQRT(ARSQ - 1.)

C             TEST FOR A CIRCLE (DUM1 = 0)
                    IF (DUM1 .NE. 0.0) THEN
                      RMU0 = ALOG((AR+1.)/DUM1)
                      SINNU = SIN(RNU)
                      COSNU = COS(RNU)
                      SINHMU = SINH(RMU0)
                      COSHMU = COSH(RMU0)
                      XRELL = (DUM1 * COSHMU * COSNU) * BAXW
                      YRELL = (DUM1 * SINHMU * SINNU) * BAXW
                    ELSE

C             FOR A CIRCLE CENTERED AT (0,0), DECREASE XRMAJ,YRMAJ BY
C             THE RATIO OF THE RADIAL DISTANCE TO THE ELLIPSE DIVIDED 
C             BY THE RADICAL DISTANCE TO (XRMAJ,YRMAJ)

                      RAD = SQRT(XRMAJ*XRMAJ + YRMAJ*YRMAJ)
                      FRAC = AAXW/RAD
                      XRELL = XRMAJ * FRAC
                      YRELL = YRMAJ * FRAC
                    ENDIF

C             CALCULATE POSITION ALONG BETA
                    XRBETA = XRELL*COS(BETA) + YRELL*SIN(BETA)
                  ENDIF
C          COMPUTE AND ADD CONCENTRATION DUE TO PLUME MATERIAL BELOW HC
                  CALL WRAP(QS)
                ELSE IF( ICASE .EQ. 1 .OR. ICASE .EQ. 3) THEN
                  WRITE(IOUT,6178) NR
              ENDIF
C
                NLINES = NLINES + 4

C             STORE HOURLY CONCENTRATION AND SCALE BY EMISSION RATE

250           CONC(NR) = CONC(NR) + C
              SCONC(NR,NS) = SCONC(NR,NS) + C

C               CHECK IF PAGE IS FULL
                IF((ICASE.EQ.1 .OR. ICASE .EQ. 3) .AND. 
     $                        (NLINES .GE. MAXLIN-3)) THEN
                        CALL PAGE(YES)
                        WRITE(IOUT,6170) GSGS(ICHIQ+1)
                        NLINES = NLINES + 5
                ENDIF
C
C         END RECEPTOR LOOP -------------------------------------------
C
260       CONTINUE
C
C       END HILL LOOP -------------------------------------------------
C
270     CONTINUE
C
C       END OF SOURCE LOOP --------------------------------------------
C
300     CONTINUE
                                                  
C  FIND THE MAXIMUM PREDICTED CONCENTRATION

        CHIMAX = 0.0
          DO 305 NR = 1,NRECPT
            IF(CONC(NR) .GT. CHIMAX) THEN
                CHIMAX = CONC(NR)
                NRMAX = NR
            ENDIF
305       CONTINUE

C       PRINT HOURLY MAX CONCENTRATION
        IF(ICASE.EQ.1 .OR. ICASE .EQ. 3) THEN
            IF(CHIMAX .GT. CHIPRT) THEN
                WRITE(IOUT,6185) CHIMAX, GSGS(ICHIQ+1), NRMAX
              ELSE
                WRITE(IOUT,6190) CHIMAX, GSGS(ICHIQ+1), NRMAX
            ENDIF
        ENDIF


C       FILL TOP N ARRAYS
        IF(ITOPN .EQ. YES) CALL TOPN( CONC, NRECPT, 0, GSGS(ICHIQ+1))

C
        GO TO 400

C       HOURLY ERROR SECTION FOR CASES WHEN THE MODEL IS NOT APPROPRIATE

C       USTAR MISSING
330     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $        WRITE(IOUT,9335) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       Z0 MISSING
340     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $     WRITE(IOUT,9345) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       MISSING MET DATA: EITHER WD, WS, SIGV, OR SIGW
350     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $         WRITE(IOUT,9355) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       FLOW ALGORITHM BEYOND DESIGN CRITERIA OR ENDLESS LOOP IN PATH
360     IF (IFLOW .EQ. 1) WRITE(IOUT,9365) JYR, JMO, JDY, KJCD, JHR
        IF (IFLOW .EQ. 2) THEN
          WRITE(IOUT,9366) NS,NHL,JYR,JMO,JDY,KJCD,JHR
          WRITE(0,9366) NS,NHL,JYR,JMO,JDY,KJCD,JHR
        ENDIF
        GO TO 390

C       WIND SPEED AT PLUME HEIGHT LESS THAN 1.0 M/S
370     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $      WRITE(IOUT,9375) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       MISSING MIXING HEIGHTS DURING UNSTABLE HOUR
380     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $      WRITE(IOUT,9385) JYR, JMO, JDY, KJCD, JHR

C       SET CONCENTRATIONS TO MISSING FOR THIS HOUR
390     DO 395 NR=1,NRECPT
          CONC(NR) = -999.
          DO 396 NS=1,NSTACK
            SCONC(NR,NS) = -999.
396       CONTINUE
395     CONTINUE

C       WRITE CONCENTRATIONS TO BINARY OUTPUT FILE

400    IF(ICONC .NE. NO) THEN
            NR = NRECPT
            IF( CONC(1) .EQ. -999. ) NRMAX = 0
            MET(1) = KYR
            MET(2) = KMO
            MET(3) = KDY
            MET(4) = KHR
            MET(5) = NRMAX
            CALL WRITIT(MET,CONC,NRECPT,IOCONC)
        ENDIF

C       PRINT HOURLY SOURCE CONTRIBUTION TABLE (IF NECESSARY)
        IF (ISOR .GT. 0) CALL SOURCES

        GO TO 100

 999    CONTINUE


C       PRINT TOP N TABLE (IF NECESSARY)
        IF(ITOPN .EQ. YES) CALL TOPN( CONC, NRECPT, 1, GSGS(ICHIQ+1))

         RETURN

6104    FORMAT(/,3X,'WARNING!! SOURCE ',I3,' IS IN UNSTABLE LAYER',
     1     ' IN NIGHTTIME ON ',I2,'/',I2,'/',I2,' HR ',I2,'.',/,
     2    3X,'INTERPRET RESULTS WITH CAUTION.')
6105    FORMAT(/,1X,'DISAGREEMENT IN PROFILE VS. SURFACE DATA: ',/,10X,
     1   'PROFILE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2,/,10X,
     2   'SURFACE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2)
6107    FORMAT(/,1X,'PROFILE HEIGHT VALUE INCORRECT: ',/,10X,
     1   'PROFILE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2)
6108    FORMAT(/'   LEVEL NUMBER:',I2,'      HEIGHT:',F8.2/)
6109    FORMAT(/'   LEVEL NUMBER:',I2,'      PREVIOUS HEIGHT:',F8.2,
     1          '    CURRENT HEIGHT:',F8.2)
6115    FORMAT(  1X,80('-'),//,'  INPUT MET DATA FROM  SURFACE  AND ',
     1    ' PROFILE  (NOTE: ****** = MISSING DATA):',//,
     2    T40,'MONIN-     SFC',/,T17,
     3      'MIXING   SFC    SFC    OBUKHOV   ROUGH.',/,T17,
     4      'HEIGHT   TEMP    U*    LENGTH    LENGTH',/,'  YR MO DA HR',
     5   '    (M)      (K)  (M/S)     (M)       (M)',//,
     6    1X,4I3,3X,F6.1,2X,F5.1,1X,F5.3,3X,F7.1,2X,F7.4,///,
     7    T9,'ADJUSTED  WIND    <-WIND  SPEED->  AMB.   SIGMA-',/,
     *    T10,'HEIGHT   DIR.    SCALAR  VECTOR   TEMP   THETA    ',
     *         'SIGMA-V ',
     8    '  SIGMA-W',/,
     9    T10,'  (M)   (DEG)     (M/S)   (M/S)   (K)     (DEG)   ',
     *    ' (M/S)',
     A    '     (M/S)',/)
6120    FORMAT(T10,F6.1,2X,F6.1,2X,F7.2,2X,F7.2,1X,F7.2,1X,F6.1,3X,F7.2,
     1   4X,F7.2)
6122    FORMAT(/,10X,'NOTE: SCALAR WIND SPEEDS ARE',
     1    ' SET TO A MINIMUM OF 1 M/S'/
     2  10X,'NOTE: HEIGHTS ARE REFERENCED TO THE COMMON STACK BASE',
     3  ' ELEVATION',/,10X,
     4  '      THE ADJUSTMENT TO THE INPUT HEIGHT IS ',F5.1,' METERS.'/)
6123    FORMAT(10X,'NOTE: HEIGHTS ARE REFERENCED TO THE COMMON STACK ',
     1  'BASE ELEVATION',/,10X,
     2  '      THE ADJUSTMENT TO THE INPUT HEIGHT IS ',F5.1,' METERS.'/)
6125    FORMAT(/,10X,'NO EMISSIONS FROM SOURCE # ',I2,/)
6130    FORMAT(/,'          <------- SOURCE INFORMATION ------->      ',
     1    'FINAL PLUME',/,
     2    ' SOURCE   QS    TS    VS   BUOY FLUX  MOM FLUX         RISE',
     3    /,'   #    (G/S)   (K)  (M/S)  (M4/S3)    (M4/S2)         ',
     4    '(M)',/)
6135    FORMAT(I4,F9.1,F7.1,F6.2,F9.1,F10.1,5X,F10.2,
     1    //,2X,'VARIABLES AT ',T21,'HEIGHT   WDIR   ',
     2    'USCAL  UVECT  SIGV   SIGW   DTHDZ',/,2X,'PLUME HEIGHT:',
     3    T21,'  (M)    (DEG)  (M/S)  (M/S)  (M/S)  (M/S) (DEG/M)',/)
6140    FORMAT('                   ',F7.1,F7.0,F7.2,2F7.2,2F8.4,/)
6150    FORMAT(//,2X,'INFORMATION FOR HILL ',I2,': ',10A4,//,
     1  7X,'HCRIT = ',F7.1,' M;  FROUDE # ABOVE HCRIT = ',F5.2)
6155    FORMAT(//,5X,'WRAP INFORMATION:',/,7X,'DISTANCE FROM SOURCE TO',
     1  ' HILL CENTER = ',F6.1,' M;  WRAP HT = ',F6.1,' M',/,7X,
     2  'ELLIPSE AXIS LENGTHS: MAJOR = ',F7.1,' M;  MINOR = ',F7.1,
     3  ' M',/,7X,'MAJOR AXIS AZIMUTH FROM NORTH = ',F5.1,' DEG',/,
     4  7X,'DISTANCE TO PRIMARY IMPINGEMENT POINT = ',F7.1,' M',/)
6158    FORMAT(/,5X,'WARNING: FROUDE NUMBER USED IN LIFT MODEL IS ',
     1  'SIGNIFICANTLY BELOW',/,5X,'THE INTENDED RANGE OF ',
     2  'APPLICABILITY.  USE CONCENTRATIONS WITH CAUTION.',/)
6160    FORMAT(/,5X,'LIFT INFORMATION:',/,7X,'DISTANCE FROM SOURCE TO',
     1  ' HILL CENTER = ',F6.1,' M; LIFT MIDPOINT HT = ',F6.1,/,7X,
     2  'ELLIPSE AXIS LENGTHS:  MAJOR = ',F7.1,' M;MINOR = ',F7.1,' M',
     3  /,7X,'MAJOR AXIS AZIMUTH FROM NORTH = ',F5.1,' DEG',/,7X,
     4  'DISTANCE ALONG FLOW FROM SOURCE TO HILL CENTER = ',F7.1,' M',/,
     5  7X,'CROSSFLOW DISTANCE FROM SOURCE TO HILL CENTER = ',F7.1,' M',
     6  /,7X,'DISTANCE TO PRIMARY IMPINGEMENT POINT = ',F7.1,' M',/)
6170    FORMAT(/,T9,'SRC-RECP   SRC-RECP  RECEPTOR   EFF.      FLAT  ',
     1  '    HILL-INDUCED',/,
     2  '     L  DISTANCE   DISTANCE  HT ABOVE SRC-RECP   TERRAIN    ',
     3  ' EFFECTIVE     TOTAL',/,' REC / ALONG FLOW CROSS FLOW STK ',
     4  'BASE HT DIFF  SIG-Y SIG-Z  SIG-Y SIG-Z    CONC',/,
     5  '  #  W     (M)        (M)       (M)      (M)    (M)   (M)',
     6  '    (M)   (M)  (U',A1,'/M**3)')
6175    FORMAT(/,1X,I3,' LIFT N/A (RECEPTOR BELOW HC)')
6178    FORMAT(1X,I3,' WRAP N/A (PLUME ABOVE HC OR UPWIND OF HILL)')
6185    FORMAT(/,1X,'MAXIMUM CONCENTRATION FOR THIS HOUR IS ',
     1  F10.2,' U',A1,'/M**3 AT RECEPTOR # ',I3,/)
6190    FORMAT(/,1X,'MAXIMUM CONCENTRATION FOR THIS HOUR IS ',
     1  1PE10.4,' U',A1,'/M**3 AT RECEPTOR # ',I3,/)
6265    FORMAT(/' ALL RECEPTORS ON HILL ',I2,' ARE UPWIND OF SOURCE ',
     *          I2,' THIS HOUR. WIND DIR. IS ',F5.1,/)
6270    FORMAT(/' POSITION OF HILL ',I2,' IS UPWIND OR FAR TO THE SIDE',
     *          ' OF THE PLUME FROM SOURCE ',I2,/,' THIS HOUR. ',
     *          ' WIND DIRECTION IS ',F5.1,/)
8010    FORMAT(/,1X,I3,' LIFT N/A (PLUME MISSES HILL)')
8020    FORMAT(1X,I3,' WRAP N/A (PLUME MISSES HILL)')
9335    FORMAT(/' USTAR MISSING, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9345    FORMAT(/' Z0 MISSING, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9355    FORMAT(/' MISSING MET. INPUT, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9365    FORMAT(/' FLOW FIELD ALGORITHM SUBJECTED TO INPUT DATA BEYOND ',
     *  'ITS DESIGN CRITERIA;',/,' NO PREDICTIONS: YEAR = ',
     *  I2,'  MONTH = ',I2,'  DAY = ',I2,'  JCD = ',I3,'  HOUR = ',I2/)
9366    FORMAT(/' ENDLESS LOOP IN PATH; STACK ',I2,' HILL ',I2,
     *  /' NO PREDICTIONS: YEAR = ',
     *  I2,'  MONTH = ',I2,'  DAY = ',I2,'  JCD = ',I3,'  HOUR = ',I2/)
9375    FORMAT(/' WIND SPEED LT 1 M/S, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9385    FORMAT(/' MISSING MIXING HEIGHT, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9500    FORMAT(' DAY = ',I4)
9502    FORMAT(' CALCULATING HOUR ',4I3)

        END
