        SUBROUTINE SRISE(IFLAG, DELH, IRISE)
C-----------------------------------------------------------------------
C PURPOSE: THIS ROUTINE CALCULATES FINAL PLUME RISE FOR THE STABLE
C               CASE ( L > 0 )
C
C METHODS:
C
C       IF THE BUOYANCY FLUX (F) IS ZERO THEN THE MOMENTUM RISE IS
C       THE MINIMUM VALUE FROM THE FOLLOWING EQUATIONS:
C               EQN (6)  3*D*VS/U
C               EQN (7)  1.5*(VS*VS*D*D*TA/(4*TS*U))**0.333*S**-0.1667
C       OTHERWISE THE PLUME RISE IS THE MINIMUM VALUE FROM THE FOLLOWING
C       EQUATIONS:
C               EQN (8)  1.3*FB/(U*USTAR**2) * (1+HS/DELH)**0.6667 OR
C               EQN (9)  1.6*(FB*X*X)**0.3333/U,
C                      X =119*F**0.4 IF FB>55; X = 49*F**0.625 IF FB<=55
C               EQN(10)  2.6*(FB/(U*S))**0.3333
C               EQN(11)  4*FB**0.25*S**-0.375
C       IF EQN (7) IS USED, THEN TEST THE NEUTRAL HIGH WIND MODEL:
C               EQN (12)  1.54*(FB/(U*USTAR**2))**0.6667 * HS**0.3333
C       AND IF THE RESULT OF EQN(12) IS LESS THAN EQN(8-11), USE EQN(12)
C
C       IN THE CTDM USER'S GUIDE, THE ABOVE EQUATIONS (6-12) CORRESPOND
C       TO EQUATIONS 10a, 10b, 14, 11, 16a, and 16b, RESPECTIVELY.
C
C ASSUMPTIONS:
C       - CONVERGENCE IS REACHED WHEN TWO CONSECUTIVE ESTIMATES ARE
C         WITHIN 1%.  AT THE END OF ITERATION # MAXITR, THE AVERAGE OF
C         THE LAST TWO ESTIMATES IS USED.  HOWEVER, IF THE SECOND
C         ITERATION PRODUCES A PLUME RISE HIGHER THAN THE FIRST, THEN
C         THE SCHEME IS DIVERGING, AND THE FIRST ITERATION RESULT IS
C         USED.
C       - MINIMUM WIND SPEED IS 0.1 M/S
C
C ARGUMENTS:
C   PASSED:
C       IFLAG   INT     1 IF PARTIAL PENETRATION CALCULATION,
C                       0 OTHERWISE
C
C   RETURNED:
C       DELH    REAL    FINAL PLUME RISE ABOVE STACK TOP (M)
C       IRISE   INT     FLAG INDICATING WHICH PLUME RISE EQUATION WAS
C                               USED TO CALCULATE DELH
C
C CALLING ROUTINES: SEQMOD
C
C EXTERNAL ROUTINES:
C       GETWS - FUNCTION WHICH RETURNS THE WIND SPEED AT HEIGHT Z
C       GETDTH - FUNCTION WHICH RETURNS DTHETA/DZ AT HEIGHT Z
C       PICK4 - SUBROUTINE WHICH RETURNS THE MINIMUM VALUE OF FOUR
C               VARIABLES AND A FLAG INDICATING WHICH VARIABLE WAS
C               PICKED
C
C INTRINSIC FUNCTIONS: ABS  AMAX1
C
C COMMON BLOCKS: STACK  SFCMET  VARS
C-----------------------------------------------------------------------
C
      INCLUDE   'STACK.CMN'
      INCLUDE   'SFCMET.CMN'
      INCLUDE   'VARS.CMN'

C  DEFINE ARGUMENTS
        REAL            DELH
        INTEGER         IRISE,  IFLAG

C  DEFINE LOCAL VARIABLES
        REAL            S16,    UP,     USTAR2, DH1,    DH2,
     1          DH3,    DHG,    G,      ERROR,  TEMP,   GTA,    HS13,
     2          DH4,    DH5,    SDTHDZ, DHO,    DTHDZ
        INTEGER         ITR,    MAXITR
C
        DATA            MAXITR/ 5/
        DATA            G/ 9.80616/
        DATA            ERROR/ 0.01/
C
        US = AMAX1(USTKTP,0.1)
        GTA = G/TA
        USTAR2 = USTAR0*USTAR0
        IF(IFLAG .EQ. 0) THEN
            SDTHDZ=GETDTH(HS,XMH)
          ELSE
            SDTHDZ = 0.005
        ENDIF
C
C       TEST FOR MOMENTUM OR BUOYANCY RISE
C
        IF(FB .GT. 0.0) GO TO 110
C       MOMENTUM RISE
C------------EQUATION (6)
        DH1 = 3.0*DS*VS/US
C------------EQUATION (7)
        IF(SDTHDZ.LE.0.0) THEN
            IRISE = 6
            DELH = DH1
            RETURN
        ENDIF
        S16 = 1.0/(GTA*SDTHDZ)**0.16667
        DH2 = (1.5*(VS*VS*DS*DS*TA/(4.0*TS*US))**0.33333)*S16
C       PICK MINIMUM RISE FROM EQUATIONS (6) AND (7)
        DH3 = 999999.
        DH4 = 999999.
        CALL PICK4(DH1, DH2, DH3, DH4, DELH, IRISE)
        IRISE = IRISE+5
        RETURN
C
C       NEUTRAL/STABLE BUOYANCY RISE
C
C       FOR BUOYANT PLUME RISE, THE WIND SPEED USED IN THE CALCULATION
C       IS THAT HALFWAY BETWEEN THE STACK-TOP HEIGHT AND THE LATEST
C       PLUME HEIGHT ESTIMATE.  SINCE FINAL PLUME RISE (DHG) IS A
C       FUNCTION OF THE WIND SPEED USED IN THE COMPUTATION, AN ITERATIVE
C       SCHEME MUST BE USED.  FOR THE FIRST GUESS, THE WIND SPEED USED
C       IS THAT AT STACK-TOP HEIGHT (US).  A MAXIMUM OF 5 ITERATIONS IS
C       ATTEMPTED.  FOR EACH ITERATION, THE PLUME RISE (FOR GETTING
C       PLUME RISE MIDPOINT METEOROLOGICAL VARIABLES) IS SET TO THE
C       AVERAGE OF THE PREVIOUS 2 ESTIMATES.  DH0 IS THE SECOND OLDEST
C       GUESS, WHILE DH<1,2,3,4> IS THE MOST RECENT GUESS.
C
110     UP = US
C------------EQUATION (8)
        DHG = 1.3*FB/(UP*USTAR2)
        IF(DHG.GT.XMH) THEN
            DH1 = 999999.
            GO TO 130
        ENDIF
        DHO = DHG
        DO 120 ITR=1,MAXITR
            DH1 = 1.3*FB/(UP*USTAR2)*(1.0+HS/DHG)**0.66667
            IF(ABS(DH1-DHG) .LT. ERROR*DH1) GO TO 130
            IF(ITR .EQ. 2 .AND. DH1 .GT. DHG) THEN
                DH1 = DHG
                GO TO 130
            ENDIF
            UP = GETWS(HS+0.5*DH1)
            DHG = 0.5*(DHO+DH1)
            DHO = DH1
            IF(UP .LE. 0.0) GO TO 125
120     CONTINUE
125     DH1 = DHG
C------------EQUATION (9)
130     CONTINUE
        UP = US
        DHG = 1.6*(FB*XFIN*XFIN)**0.33333/UP
        DHO = DHG
        UP=GETWS(HS+0.5*DHG)
        DO 140 ITR=1,MAXITR
            DH2 = 1.6*(FB*XFIN*XFIN)**0.33333/UP
            IF(ABS(DH2-DHG) .LT. ERROR*DH2) GO TO 150
            IF(ITR .EQ. 1 .AND. DH2 .GT. DHG) THEN
                DH2 = DHG
                GO TO 150
            ENDIF
            UP = GETWS(HS+0.5*DH2)
            DHG = 0.5*(DHO+DH2)
            DHO = DH2
            IF(UP .LE. 0.0) GO TO 145
140     CONTINUE
145     DH2=DHG
C------------EQUATION (10)
150     IF(SDTHDZ.LE.0.0) THEN
            DELH = AMIN1(DH1,DH2)
            IF(DH1.LT.DH2) THEN
                IRISE =  8
              ELSE
                IRISE = 9
            ENDIF
            RETURN
        ENDIF
        UP = US
        DTHDZ = SDTHDZ
        DHG = 2.6*(FB/(UP*GTA*DTHDZ))**0.33333
        DHO = DHG
        UP = GETWS(HS+0.5*DHG)
        IF(IFLAG.EQ.0) DTHDZ = GETDTH(HS+0.5*DHG,XMH)
        IF(UP .LE. 0.0 .OR. DTHDZ .LE. 0.0) GO TO 165
        DO 160 ITR = 1,MAXITR
            DH3 = 2.6*(FB/(UP*GTA*DTHDZ))**0.33333
            IF(ABS(DH3-DHG) .LT. ERROR*DH3) GO TO 170
            IF(ITR .EQ. 1 .AND. DH3 .GT. DHG) THEN
                DH3 = DHG
                GO TO 170
            ENDIF
            UP = GETWS(HS+0.5*DH3)
            IF(IFLAG.EQ.0) DTHDZ = GETDTH(HS+0.5*DH3,XMH)
            DHG = 0.5*(DHO+DH3)
            DHO = DH3
            IF(UP .LE. 0.0 .OR. DTHDZ .LE. 0.0) GO TO 165
160     CONTINUE
165     DH3 = DHG
C------------EQUATION (11)
170     TEMP = 4.0*FB**0.25
        DTHDZ = SDTHDZ
        DHG = TEMP/(GTA*DTHDZ)**0.375
        DHO = DHG
        IF(IFLAG.EQ.1) GO TO 185
        DTHDZ = GETDTH(HS+0.5*DHG,XMH)
        IF(DTHDZ .LE. 0.0) GO TO 185
        DO 180 ITR = 1,MAXITR
            DH4 = TEMP/(GTA*DTHDZ)**0.375
            IF(IFLAG.EQ.1) GO TO 190
            IF(ABS(DH4-DHG) .LT. ERROR*DH4) GO TO 190
            IF(ITR .EQ. 1 .AND. DH4 .GT. DHG) THEN
                DH4 = DHG
                GO TO 190
            ENDIF
            DTHDZ = GETDTH(HS+0.5*DH4,XMH)
            DHG = 0.5*(DHO+DH4)
            DHO = DH4
            IF(DTHDZ .LE. 0.0) GO TO 185
180     CONTINUE
185     DH4 = DHG
C       PICK MINIMUM RISE FROM EQUATION'S (8), (9), (10), (11)
190     CALL PICK4(DH1, DH2, DH3, DH4, DELH, IRISE)
        IRISE = IRISE+7
C TEST FOR NEUTRAL HIGH WIND MODEL
        IF(IRISE .NE. 7 .OR. IFLAG .EQ.1) RETURN
C------------EQUATION (12)
        UP = US
        HS13 = HS**0.33333
        TEMP = FB/USTAR2
        DHG = 1.54*(TEMP/UP)**0.66667*HS13
        UP = GETWS(HS+0.5*DHG)
        DHO = DHG
        DO 200 ITR = 1,MAXITR
            DH5 = 1.54*(TEMP/UP)**0.66667*HS13
            IF(ABS(DH5-DHG) .LT. ERROR*DH5) GO TO 210
            UP = GETWS(HS+0.5*DH5)
            DHG = 0.5*(DHO+DH5)
            DHO = DH5
200     CONTINUE
        DH5 = DHG
210     IF(DH5 .LT. DELH) THEN
            DELH = DH5
            IRISE = 12
        ENDIF
C
        RETURN
        END
