      SUBROUTINE SEQMOD
C-----------------------------------------------------------------------
C PURPOSE: THIS IS THE MAIN SUBROUTINE OF CTDMPLUS. PROGRAM LOOPS IN TIME,
C               IN STACKS, IN HILLS AND IN RECEPTORS OCCUR HERE.
C               CONCENTRATIONS ARE CALCULATED IN THIS ROUTINE.
C
C ASSUMPTIONS: FOR HOURS WITH MISSING METEOROLOGICAL DATA,
C               CONCENTRATIONS ARE SET TO -999 FOR ALL RECEPTORS FOR THE
C               HOUR. ALSO, IF ANY PLUME IS IN AN UNSTABLE LAYER, THE
C               CONCENTRATIONS ARE SET TO -999 FOR THE HOUR.
C
C I/O:
C  INPUT:
C       UNIT=INSFC              FILE=SURFACE
C       UNIT=INPROF             FILE=PROFILE
C       UNIT=INEMIS             FILE=EMISSIONS
C       UNIT=INREC              FILE=RECEPTOR
C       UNIT=INRAW              FILE=RAWIN
C
C  OUTPUT:
C       UNIT=IOUT               FILE=LIST
C       UNIT=IOCONC             FILE=CONCENTRATIONS
C
C       BINARY CONCENTRATION FILE FORMAT IS:
C       4 INTEGER*2 (8 BYTES) + NRECPT REAL*4 CONCENTRATIONS PER RECORD
C       THE VALUES OF THE INTEGER HEADER VARIABLES ARE:
C       (1) = 2-DIGIT YEAR
C       (2) = JULIAN CALENDER DAY
C       (3) = HOUR (1-24)
C       (4) = NUMBER OF THE RECEPTOR WITH THE HIGHEST HOURLY CONC.
C
C
C COMMON BLOCKS: (ALL COMMON BLOCKS FOUND ANYWHERE IN CTDMPLUS ARE USED
C                 HERE, WITH DESCRIPTIONS OF VARIABLES)
C
C
C       CONST   HEAD    HILL    IO      PARAMS  PASL    PASW    PASVAL
C       PROFIL  RECEPT  SFCMET  STACK   STACKS  TIME    TOWER   VARS
C
C
C CALLING ROUTINES:
C       CTDMPLUS (MAIN)
C
C-----------------------------------------------------------------------
C

      PARAMETER(CHIPRT=0.005)

      INCLUDE 'PARAMS.INC'
      INCLUDE 'CONST.CMN'
C      INCLUDE 'HEAD.CMN'
      INCLUDE 'HILL.CMN'
      INCLUDE 'IO.CMN'
      INCLUDE 'PARAMS.CMN'
      INCLUDE 'PASL.CMN'
      INCLUDE 'PASW.CMN'
      INCLUDE 'PASVAL.CMN'
      INCLUDE 'PROFIL.CMN'
      INCLUDE 'RECEPT.CMN'
      INCLUDE 'SCREN.CMN'
      INCLUDE 'SFCMET.CMN'
      INCLUDE 'STACK.CMN'
      INCLUDE 'STACKS.CMN'
      INCLUDE 'TIME.CMN'
      INCLUDE 'TOWER.CMN'
      INCLUDE 'VARS.CMN'

C       DEFINE LOCAL VARIABLES (THESE ARE EXPLAINED AS THEY APPEAR
C       IN THE CODE)

        REAL    ADDWD(3), ADDWD2(8), ANGINT, BASEHW, BSTKTP, BULKFR, 
     $          COSELW, DELH, DELT, DMIN, DUMMY, DX, 
     $          DXH, DYH, DY, EPS, FRACT, GETWS, 
     $          GETDTH, GETSV, GETSW, GETUV, HCRIT,  
     $          HPLW, HTOPS, HTWRAP, PSI, ROTELW, 
     $          RSHW, SIGRAD, SIGTH(MAXLEV), SINELW,
     $          SQGAMA, SVMIN, THTA1, THTAMN, TLIFT,
     $          UCGAMA, USRAT, 
     $          WSRT, XHILLL, XHILLW, XINTRP, XSEW, XTEW, 
     $          XTMAJW, YHILLL, YHILLW, YSEW, YTEW, YTMAJW,
     $          WDALL(MAXSOR*MAXHIL), DISCWD(50), ZIFACT(3),
     $          ZIHILL(MAXHIL,3)

        INTEGER EOF, I, IEND, IFLAG, IHROUT, INC, IRISE,
     $          JDY, JHR, JMO, JYR, KLOSE, KLOW, LSTAB, 
     $          MFAC, NH, NHLW, NS, NSW, WDGOOD, 
     $          WSGOOD, SVGOOD, SWGOOD, OLDAY

        INTEGER*2       MET(5)
        CHARACTER*1     GSGS(2)

        DATA GSGS/'G', 'S'/
        DATA ADDWD/0.0,1.0,-1.0/
        DATA ADDWD2/1.0,-1.0,2.0,-2.0,3.0,-3.0,4.0,-4.0/
        DATA ZIFACT/0.5,1.0,1.5/


C  INITIALIZE VALUES FOR COMMON BLOCK
        NO = 0
        YES = 1
        DTOR = 0.01745329
        PI = 3.1415926
        PIBY2 = 0.5 * PI
        TWOPI = PI + PI
        SQRPI = SQRT(PI)
        SQR2PI = SQRT(TWOPI)
        SQR2 = SQRT(2.0)
        ALPHA = 1.155
        SMALL = 0.00001

        UCGAMA = 0.36
        SQGAMA = 0.27
        SVMIN = 0.20
        OLDAY = 999
        IHROUT = 0
        ISIM = 0

C  READ ADDITIONAL PARAMS FROM CTDM.IN IF NEEDED FOR SCREENING MODE
        IF (IRANGE .EQ. 1) THEN 
          READ(IN,*) WDLOW,WDUP,WDINC
          WDUP2 = WDUP + 360.0
        ENDIF
        IF (IDISCR .EQ. 1) READ(IN,*) NUMWD,(DISCWD(J),J=1,NUMWD)


C  FOR SCREEN CASES, FILL THE ZIHLL ARRAY
        IF (ISCRN .GT. 0) THEN

C  FIRST LOOP OVER PRIMARY SOURCES TO FIND THE ZI FOR WHICH THE PENETRATION
C  FACTOR IS LESS THAN OR EQUAL TO 0.1 FOR ALL SOURCES. USE 2 M/S WINDS
C  POTENTIAL TEMP = 293.0 AND POT. TEMP. GRADIENT ABOVE ZI = 0.03

          ZIMAX = 0.
          DO 25 NS = 1,NSTACK
            IF (ISEC(NS) .EQ. 1) GO TO 25
            HS = SOURCE(4,NS)
            TS = SOURCE(6,NS)
            IF (TS .LT. 293.0) TS = 293.0
            FB = SOURCE(10,NS) * (TS-293.0)/TS
C            S  = (9.80616/293.0) * 0.03
            ZI = HS + 3.64 * (FB/(0.002))**0.333333
            IF (ZI .GT. ZIMAX) ZIMAX = ZI
25        CONTINUE

C  NOW LOOP OVER HILLS.  CALCULATE THREE MIXING HEIGHTS BASED ON THE HILL
C  HEIGHT ABOVE THE BASE ELEVATION (0.5H, 1.0H, 1.5H).  IF ZIMAX IS LESS
C  THAN 0.9 * HILL HT. USE ZIMAX IN PLACE OF THE 0.5H VALUE.  IF ZIMAX
C  IS GREATER THAN 1.1 * HILL HT, USE ZIMAX IN PLACE OF THE 1.5H VALUE.
          NUMZI = 0
          DO 33 NHL =1,NHILLS
            DO 37 NZI = 1,3
              ZIHILL(NHL,NZI) = THS(NHL) * ZIFACT(NZI)
37          CONTINUE
            IF (ZIMAX .LT. 0.9*THS(NHL)) THEN
              ZIHILL(NHL,1) = ZIMAX
            ELSE IF (ZIMAX .GT. 1.1*THS(NHL)) THEN
              ZIHILL(NHL,3) = ZIMAX
            ENDIF
33        CONTINUE

C  LOOP OVER HILLS, ZI VALUES, AND SOURCES.  IF ALL P FACTORS ARE GREATER
C  THAN 0.9 FOR A PARTICULAR ZI, SET THE ZI TO -999. AND DON'T USE IT.
          DO 38 NHL = 1,NHILLS
            DO 39 NZI = 1,3
              NUMPEN = 0
              DO 41 NS = 1,NSTACK
                ZB = ZIHILL(NHL,NZI) - SOURCE(4,NS)
                TS = SOURCE(6,NS)
                IF (TS .LT. 293.0) TS = 293.0
                FB = SOURCE(10,NS) * (TS-293.0)/TS
                IF (FB .GT. 0) THEN
                  P = (3.9 - ZB/(FB/0.002)**0.333333)/2.6
                ELSE
                  P = 0.0
                ENDIF
                IF (P .GE. 0.9) NUMPEN = NUMPEN + 1
41            CONTINUE
              IF (NUMPEN .EQ. NSTACK) ZIHILL(NHL,NZI) = -999.
              IF (ZIHILL(NHL,NZI) .GT. 0.) NUMZI = NUMZI + 1
39          CONTINUE
38        CONTINUE

C  INITIALIZE VALUES IN WRITSCR
          CALL WRITSCR(CONC,SCONC,0,NUMZI)

        ENDIF

100     CONTINUE

C       READ A LINE FROM THE 'SURFACE' FILE
        CALL RDSFC(EOF)

C       CHECK FOR END-OF-FILE
        IF(EOF .EQ. YES) GO TO 999

        IF (ISCRN .EQ. 0) THEN
C -- WRITE OUT DAY NUMBER TO CONSOLE -----
         IHROUT=IHROUT+1
C        IF(IHROUT/24*24 .EQ. IHROUT) WRITE(0,9500) IHROUT/24
C -- WRITE DATE EACH HOUR ---- DJB 2/6/91
         WRITE(0,9502) KYR,KMO,KDY,KHR
        ENDIF

C       COMPUTE DATE/TIME DEPENDENT VARIABLES
        CALL SUN(KJCD,TZONE,ALAT,ALONG,TSR,TSS)

        IF (ICASE .GT. 0) CALL PAGE(YES)
        TA = -999.

C       READ FILE 'PROFILE'
C       INITIALIZE FLAGS FOR AVAILABILITY OF MET VARIABLES FOR THIS HOUR

        WDGOOD = NO
        WSGOOD = NO
        SVGOOD = NO
        SWGOOD = NO

C       SECTION FOR READING AND PROCESSING "PROFILE"

        DO 120 NHT = 1,MAXLEV
                             

C        SEE TABLE 3-7 IN CTDMPLUS USER'S GUIDE

            READ(INPROF, *,END = 999) JYR,JMO,JDY,JHR,HT(NHT),IEND,
     1                          WDHR(NHT),WSHR(NHT),TAHR(NHT),
     2                          SIGTH(NHT),SWHR(NHT),UVHR(NHT)



C        CHECK FOR ERRORS IN INPUT: DATE/TIME INCONSISTENCIES,
C        NEGATIVE HEIGHTS OR HEIGHTS NOT MONOTONICALLY INCREASING

             IF(JYR.NE.KYR .OR. JMO.NE.KMO .OR. JDY.NE.KDY .OR.
     1                                          JHR.NE.KHR) THEN
                WRITE(IOUT,6105) JMO,JDY,JYR,JHR,KMO,KDY,KYR,KHR
                STOP
            ENDIF
            IF(HT(NHT) .LT. 0.0) THEN
                WRITE(IOUT,6107) JMO,JDY,JYR,JHR
                WRITE(IOUT,6108) NHT, (HT(NHT)-BASEHT)
                STOP
            ENDIF
            IF(NHT .GT. 1) THEN
                IF( HT(NHT) .LT. (HT(NHT-1) - BASEHT)) THEN
                    WRITE(IOUT,6107) JMO,JDY,JYR,JHR
                    WRITE(IOUT,6109) NHT, (HT(NHT-1)-BASEHT),
     1                                    (HT(NHT)-BASEHT)
                    STOP
                ENDIF
            ENDIF

C           HEIGHT NOW REFERENCED TO STACK BASE ELEVATION NOT TOWER
            HT(NHT) = HT(NHT) + BASEHT
            IF(TA.LT.0.0 .AND. TAHR(NHT).GT.0.0) TA = TAHR(NHT)
C           CHECK FOR ALL MISSING WIND SPEED AND DIRECTION
            IF(WDHR(NHT) .GE. 0.0) WDGOOD = YES
            IF(WSHR(NHT) .GT. 0.0) WSGOOD = YES
            IF(SWHR(NHT) .GE. 0.0) SWGOOD = YES

C  SET ZERO WIND SPEEDS TO 0.01.  THIS IS DONE TO AVOID THAT RARE CASE
C  WHERE THE PLUME IS ABOVE THE HIGHEST REPORTED METEOROLOGY LAYER, AND
C  THEREFORE THAT WIND SPEED IS USED IN GETWS AND ULTIMATELY URISE, WHICH
C  USES GETWS AS A DIVISOR WHICH RESULTS IN A DIVIDE-BY-ZERO ERROR UNLESS
C  THE WIND IS MADE NON-ZERO.   -CSC

            IF(WSHR(NHT) .EQ. 0.0) WSHR(NHT) = 0.01

C       USE YAMARTINO (1984) TO OBTAIN SIGV, UV (IF NECESSARY)

            IF(ISIGV .EQ. 0) THEN

C       HERE, SIGMA-THETA IS PROVIDED (ISIGV = 0), MAX SIGTH IS 103.9

                IF(SIGTH(NHT).LE.0.0 .OR. WSHR(NHT).LE.0.0) THEN
                    SVHR(NHT) = -999.9
                    IF(UVHR(NHT).LE.0.0) UVHR(NHT) = -999.9
                 ELSE
                    SIGTH(NHT) = AMIN1(103.9,SIGTH(NHT))
                    SIGRAD = SIGTH(NHT) * DTOR                    
                    EPS = SIN(SIGRAD * (1.0 - 0.073864*SIGRAD))
                    IF(UVHR(NHT).LE.0.0) UVHR(NHT) = WSHR(NHT) *
     1                                          SQRT(1.0 - EPS*EPS)
                    SVHR(NHT) = SIGRAD * UVHR(NHT)
                    SVGOOD = YES
                ENDIF
            ELSE

C       HERE, SIGMA-V IS PROVIDED (ISIGV = 1)

                IF(SIGTH(NHT) .LT. 0.0) SIGTH(NHT) = -999.9
                SVHR(NHT) = SIGTH(NHT)
                IF(UVHR(NHT) .LE. 0.0  .AND. WSHR(NHT) .GT. 0.0
     &                                  .AND. SVHR(NHT) .GE. 0.0) THEN
                    SIGRAD = SVHR(NHT) / WSHR(NHT)
                                                  

C                   STORE SIGMA-THETA VALUES IN DEGREES (SIGMA-THETA
C                   CANNOT EXCEED 103.9 DEGREES)

                    SIGRAD = AMIN1(103.9*DTOR,SIGRAD)
                    SIGTH(NHT) = SIGRAD/DTOR
                    EPS = SIN(SIGRAD * (1.0 - 0.073864*SIGRAD))
                    UVHR(NHT) = WSHR(NHT) * SQRT(1.0-EPS*EPS)
                 ELSE
                    IF(UVHR(NHT) .LE. 0.0) UVHR(NHT) = -999.9
                ENDIF
                IF(SVHR(NHT) .GT. 0.0) SVGOOD = YES
            ENDIF

C       CHECK FOR SCALAR WIND SPEED LESS THAN 1 M/SEC, RATIO SCALAR,
C       VECTOR SPEEDS AND SIGW, SIGV UPWARD IF NECESSARY

            IF(WSHR(NHT) .GT. 0.0 .AND. UVHR(NHT).GT. WSHR(NHT))
     1          UVHR(NHT) = WSHR(NHT)
            IF( IWS1 .EQ. YES ) THEN
                IF( WSHR(NHT) .LT. 1.0 ) THEN
                    IF( WSHR(NHT) .GT. 0.0 ) THEN
                        WSRT = 1.0/WSHR(NHT)
                        WSHR(NHT) = 1.0
                        IF(UVHR(NHT) .GT. 0.0) UVHR(NHT)=UVHR(NHT)*WSRT
                        IF(SVHR(NHT) .GT. 0.0) SVHR(NHT)=SVHR(NHT)*WSRT
                        IF(SWHR(NHT) .GT. 0.0) SWHR(NHT)=SWHR(NHT)*WSRT
                    ENDIF
                ENDIF
            ENDIF

C       IN STABLE CONDITIONS, A VALUE OF SIGMA-V BELOW 0.2 M/S IS
C       SET TO 0.2 M/S, WHETHER OBSERVED OR CALCULATED FROM SIGMA-THETA
C       AND WIND SPEED

            IF(SVHR(NHT) .GE. 0.0) THEN
                IF(EL .GT. 0.0 .OR. HT(NHT) .GT. XMH) SVHR(NHT) =
     1             AMAX1(SVHR(NHT), SVMIN)
            ENDIF

C           CHECK FOR LAST PROFILE RECORD
            IF( IEND .EQ. YES ) GO TO 130
120     CONTINUE

C       END SECTION FOR PROCESSING "PROFILE" DATA


130     IF(TA .LT. 0.0) THEN
            TA = 293.
            TAHR(1) = 293.
        ENDIF

C       TEST FOR VARIABLE EMISSION RATES
        IF(IEMIS .EQ. YES) CALL INPEMS

C       FOR SCREEN RUNS, CHECK TO SEE IF WE WANT TO MODEL THIS HOUR 
        IF (ISCRN .EQ. 1 .AND. EL .LT. 0) GO TO 100
        IF (ISCRN .EQ. 2 .AND. EL .GT. 0) GO TO 100

C       CHECK FOR MISSING MET DATA; IF MISSING, SKIP CALCULATIONS
C       FOR THIS HOUR. FOR SIGMA-V AND SIGMA-W CHECK AT STABLE/UNSTABLE
C       SPLIT

        IF( USTAR0 .LT. 0.0 ) GO TO 330
        IF( Z0 .LT. 0.0 ) GO TO 340
        IF( WSGOOD.EQ.NO  .OR. WDGOOD.EQ.NO ) GO TO 350

        IF(ISIGV .EQ. NO) SIGV = -9.9

C       DETERMINE A STABILITY CATEGORY (FOR MISCELLANEOUS APPLICATIONS),
C       USING THE GOLDER (1972) CURVES (GIVEN L, U*)

        KST = LSTAB(EL,Z0)

C       WRITE CASE STUDY PRINTOUT OF METEOROLOGY

        IF(ICASE .GT. 0) THEN
            WRITE(IOUT,6115) KYR,KMO,KDY,KHR,XMH,TA,USTAR0,EL,Z0
            DO 150 I = 1,NHT
                WRITE(IOUT,6120) HT(I),WDHR(I),WSHR(I),UVHR(I),TAHR(I),
     1          SIGTH(I),SVHR(I),SWHR(I)
150         CONTINUE
            IF(IWS1 .EQ. 1) THEN
                WRITE(IOUT,6122) BASEHT
             ELSE
                WRITE(IOUT,6123) BASEHT
            ENDIF
        ENDIF

C       INITIALIZE & ZERO HOURLY CONCENTRATIONS
          DO  160 NR=1,NRECPT
            CONC(NR) = 0.0
            DO 155 NS = 1,NSTACK
              SCONC(NR,NS) = 0.0
155         CONTINUE
160       CONTINUE
        NRMAX = 0


C       PRELIMINARY LOOP ON HILLS: COMPUTE HCRIT AND FROUDE NUMBER
C       AND STORE FOR USE BELOW

        DO 180 NH = 1,NHILLS
            HCHILL(NH) = HCRIT(THS(NH),XMH)
            FRHILL(NH) = BULKFR(THS(NH),HCHILL(NH))
            IF(HCHILL(NH) .LT. 0.0) HCHILL(NH) = 0.0
C           LARGE FROUDE NUMBER ESSENTIALLY NEUTRAL CONDITIONS
            IF(FRHILL(NH) .GE. 99.99) FRHILL(NH) = 99.99
180     CONTINUE

          OLDMAX = 0.0
          THTASAVE = 0.0

        IF (ISCRN .EQ. NO) THEN
          ISIGZ = NO
          CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
          GOTO 304
        ENDIF

C ** START WIND DIRECTION DETERMINATION FOR THE SCREEN **

C  IF IAUTO SWITCH = 0 DON'T USE THIS METHOD, IAUTO = 1 JUST DO SOURCE-
C  HILL GEOMETRY WIND DIRECTIONS, IAUTO=2 DO AVERAGE WIND DIRECTIONS, TOO

        IF (IAUTO .EQ. 0) GO TO 300

C       START LOOP OVER STACKS FOR WIND DIRECTION DETERMINATION
        NWD = 0
        DO 185 NSW=1,NSTACK

C  IF THIS IS A SECONDARY SOURCE, SET WIND DIR TO -999 FOR ALL HILLS
C   AND GO TO NEXT SOURCE

        IF (ISEC(NSW) .EQ. 1) GO TO 185

C       SET UP STACK COMMON VARIABLES; SEE STACKS.CMN FOR DEFINITIONS

        HS = SOURCE(4,NSW)
        DS = SOURCE(5,NSW)
        TS = SOURCE(6,NSW)
        VS = SOURCE(7,NSW)
        IF(TS .LT. TA) TS = TA
        DELT = (TS-TA) / TS

C        IF NO EMISSIONS, SKIP CALCULATIONS FOR THIS STACK
 
        IF(SOURCE(8,NSW) .EQ. 0.0) THEN
            IF(ICASE .GT. 0) WRITE(IOUT,6125) NSW
            GO TO 185
        ENDIF
        FB = SOURCE(10,NSW) * DELT
        USTKTP = GETWS(HS)

C   FOR UNSTABLE HOURS, WIND DIRECTION IS SOURCE TO HILL CENTER
         IF (EL. LT. 0) GO TO 574

C   CALCULATE THE FOLLOWING FOR PLUME RISE SUBROUTINES
        IF (FB .LE. 55) THEN
          XSTAR = 14.0 * FB**0.625
        ELSE
          XSTAR = 34.0 * FB**0.40
        ENDIF
        XFIN = 3.5 * XSTAR

C       CALCULATE FINAL PLUME RISE

        IFLAG = 0
        CALL SRISE(IFLAG, DELH, IRISE)

C       NOTE: XSZS IS DISTANCE TO WHERE TURBULENCE DOMINATES SOURCE-
C       INDUCED EFFECTS.  FOR STABLE CONDITIONS, THIS IS ASSUMED TO BE
C       THE DISTANCE TO FINAL RISE; FOR NONSTABLE CONDITIONS, THIS
C       DISTANCE IS ASSUMED TO BE NEARLY ZERO.

            IF(IRISE .LE. 9) THEN
                XSZS = 0.0
              ELSE
                BSTKTP = SQRT((9.8/TA) * GETDTH(HS,XMH))
                XSZS = 2.07 * USTKTP/BSTKTP
            ENDIF

         CALL SIGB(DELH,SYS,SZS)

C       COMPUTE PLUME HEIGHT AND SAVE FOR LATER COMPARISONS
        HPLW = DELH + HS

        CALL PLAVG(HPLW, US, DUMMY, DTH, XMH)

C       CHECK IF WIND SPEED LESS THAN 1.0 M/S AT PLUME HEIGHT

        USRAT = 1.0
        IF( US .LT. 1.0 ) THEN
                IF( IWS1 .EQ. NO ) GO TO 370
C               SET RATIO TO INCREASE UV, SIGW AND SIGV
                USRAT = 1.0 / US
                US = 1.0
        ENDIF
        IF(EL .LT. 0.0) DTH = 0.0
        SIGW = GETSW(HPLW)*USRAT
        SIGV = GETSV(HPLW)*USRAT
        UV = GETUV(HPLW,US,SIGV,XMH)*USRAT
        IF(UV .LT. USTAR0) UV = USTAR0
        IF(UV .GT. US) UV = US

C  CALCULATE VIRTUAL SOURCE TIME INCREMENT
C  TNEUT, TSTRAT, TTLZ ARE GIVEN BY EQNS 22, 23, AND 24 IN USER'S GUIDE

        BRUNT = SQRT(9.8/TA * DTH)
        TNEUT = SIGW/(UCGAMA*HPLW)
        TSTRAT = BRUNT/SQGAMA
        TTLZ = 1./(TNEUT+TSTRAT)
        TTLZW = TTLZ

C     COMPUTE MINIMUM VIRTUAL SOURCE TIME INCREMENT: PLUME GROWTH TO
C     STACK DIAMETER (OR SIGMA-Y,Z GROWTH TO STACK RADIUS).  IF NO
C     "STACK" IS USED (ZERO EXIT VELOCITY, AS WITH A TRACER), ASSUME
C     A DEFAULT DIAMETER OF 1 METER.

      DIA = DS
      IF(VS .LT. SMALL) DIA = 1.0

C     SEE EQN 29 OF USER'S GUIDE
C     ZTSTK IS VIRTUAL TIME FOR SIGMA-Z GROWTH DUE TO SOURCE EFFECTS

      DUMZ = (2.0 * SIGW/DIA)**2
      ZTSTK = (1.+SQRT(1.+16.*DUMZ*TTLZ**2))/(4.*DUMZ*TTLZ)

C     COMPUTE VIRTUAL SOURCE TIME INCREMENT, ZTV: PLUME GROWTH DUE TO
C     BUOYANCY (EQN 29)
      IF(SZS .LE. 0.0) THEN
          ZTV = 0.0
        ELSE
          DUMZ = (SIGW/SZS)**2
          ZTV = (1.+SQRT(1.+16.*DUMZ*TTLZ**2))/(4.*DUMZ*TTLZ)
      ENDIF

      TSZS = XSZS/UV

C     SEE EQN 30 IN USER'S GUIDE
C     SAVE AS ZTVW FOR USE IN CALCULATING SZTEST LATER

      ZTV = AMAX1(ZTSTK,ZTV-TSZS)
      ZTVW = ZTV


574     CONTINUE

C       START LOOP ON HILLS FOR WIND DIRECTION DETERMINATION

        DO 187 NHLW=1,NHILLS

          HC = HCHILL(NHLW)
          HTOPS = THS(NHLW)

C  FOR THE UNSTABLE HOURS IN THE SCREEN WIND DIR IS SOURCE TO HILL CENTER
          IF (EL .LT. 0) THEN
            DXH = SOURCE(1,NSW) - XHL(1,NHLW)
            DYH = SOURCE(2,NSW) - YHL(1,NHLW)
            PSI = ATAN2(DXH,DYH)/DTOR
            IF (PSI .LE. 0.0) PSI = PSI + 360
            THTA = PSI

C   SAVE THIS WIND DIRECTION FOR POSSIBLE USE LATER
            NWD = NWD + 1
            WDALL(NWD) = THTA

            ISIGZ = NO

C   LOOP OVER ZI VALUES
            DO 263 NHLZI = 1,NHILLS
              DO 253 NZI = 1,3
                XMH = ZIHILL(NHLZI,NZI)

C        ZERO THE CONCENTRATION ARRAYS
                DO 255 NR = 1,NRECPT
                  CONC(NR)  = 0.0
                  CONC1(NR) = 0.0
                  DO 254 NS = 1,NSTACK
                    SCONC(NR,NS) = 0.0
                    SCONC1(NR,NS) = 0.0
254               CONTINUE
255             CONTINUE

C   IF XMH = -999., ALL P FACTORS WERE GREATER THAN 0.9, SKIP THIS ZI
                IF (XMH .EQ. -999.) GO TO 253

C   DON'T USE WIND DIRECTION WHERE STACK IS ABOVE ZI FOR UNSTABLE
                IF (HS .GT. XMH) GO TO 253

C   CALCULATE CONCENTRATIONS USING THIS WIND DIRECTION FOR THE SCREEN
                CALL CONCALC(OLDAY,SVGOOD,SWGOOD)

C   WRITE CONCENTRATIONS TO FILE
                CALL WRITSCR(CONC,SCONC,1,NUMZI)
                IF (ISOR .GT. 0) CALL SOURCES

253           CONTINUE
263         CONTINUE
            GO TO 187
          ENDIF


C  SECTION FOR DEFINING GEOMETRY FOR WRAP COMPUTATIONS ----------------
C         CRITICAL HEIGHT FOR WRAP (HPLW - PLUME HT ABOVE STACK BASE)

          HTWRAP = AMIN1(HPLW, HC)

C         GET HILL COORDS, ANGLE AND MAJOR, MINOR AXIS FOR THE WRAP HILL
C         W AT END OF VARIABLE INDICATE WRAP HILL
C         KLOW IS THE ARRAY INDEX TO THE HEIGHT CLOSEST TO (BUT LESS
C         THAN) HTWRAP
          KLOW = KLOSE( ZHS(1,NHLW), NZH(NHLW), HTWRAP )
          IF( KLOW .EQ. 0 ) KLOW = 1
          BASEHW = ZHS(KLOW,NHLW)
          IF( KLOW .EQ. NZH(NHLW) ) THEN
C             THE CRITICAL HT IS ABOVE LAST CONTOUR VALUE
C             DO NOT INTERPOLATE, BUT USE INVERSE POLYNOMIAL FORMULA

C             GET X, Y, ORIENTATION OF THE ELLIPSE FOR WRAP FROM THE
C             LOOK-UP TABLE.

              XHILLW = XHW(KLOW,NHLW)
              YHILLW = YHW(KLOW,NHLW)
              THTAH = MAJORW(KLOW,NHLW)
C             DISTANCE SOURCE TO  HILL CENTER
              DX = XS - XHILLW
              DY = YS - YHILLW
              RSHW = SQRT( DX*DX + DY*DY )

C             COMPUTE MAJOR & MINOR AXIS LENGTHS: INTERPOLATE BETWEEN
C             MAJAXW, MINAXW AND 0.0 IF ABOVE THE LAST CONTOUR

              FRACT =  1.0 - (HTWRAP - BASEHW)/(HTOPS - BASEHW)
              AAXW = MAJAXW(KLOW,NHLW) * FRACT
              BAXW = MINAXW(KLOW,NHLW) * FRACT
           ELSE
C             LINEARLY INTERPOLATE BETWEEN TWO VALUES
              XHILLW = XINTRP( ZHS(KLOW,NHLW), ZHS(KLOW+1,NHLW), HTWRAP,
     *                         XHW(KLOW,NHLW), XHW(KLOW+1,NHLW))
              YHILLW = XINTRP( ZHS(KLOW,NHLW), ZHS(KLOW+1,NHLW), HTWRAP,
     *                         YHW(KLOW,NHLW), YHW(KLOW+1,NHLW))
              AAXW = XINTRP( ZHS(KLOW,NHLW), ZHS(KLOW+1,NHLW),  HTWRAP,
     *                       MAJAXW(KLOW,NHLW), MAJAXW(KLOW+1,NHLW))
              BAXW = XINTRP( ZHS(KLOW,NHLW), ZHS(KLOW+1,NHLW),  HTWRAP,
     *                       MINAXW(KLOW,NHLW), MINAXW(KLOW+1,NHLW))
              THTAH = ANGINT( MAJAXW(KLOW,NHLW),MINAXW(KLOW,NHLW),
     *                        MAJORW(KLOW,NHLW), ZHS(KLOW,NHLW),
     *                        MAJAXW(KLOW+1,NHLW),MINAXW(KLOW+1,NHLW),
     *                        MAJORW(KLOW+1,NHLW), ZHS(KLOW+1,NHLW),
     *                        HTWRAP )

C               CHECK THAT ELLIPSE AXES ARE LESS THAN DIST FROM SOURCE
C               TO HILL CENTER   AAX <= BAX < DIST SOURCE/HILL CENTER
                DX = SOURCE(1,NSW) - XHILLW
                DY = SOURCE(2,NSW) - YHILLW
                RSHW = SQRT( DX*DX + DY*DY )
C                IF( AAXW .GT. RSHW ) AAXW = 0.99 * RSHW
C                IF( BAXW .GT. AAXW ) BAXW = AAXW
          ENDIF

C         SHIFT ORIGIN TO CENTER OF WRAP HILL (E DENOTES ELLIPSE)
          XSEW = SOURCE(1,NSW) - XHILLW
          YSEW = SOURCE(2,NSW) - YHILLW
          XTEW = XT - XHILLW
          YTEW = YT - YHILLW

C         FIND TOWER AND SOURCE LOCATIONS RELATIVE TO MAJOR AXIS OF
C         WRAP HILL ELLIPSE. NOTE: X-AXIS LIES ALONG MAJOR AXIS
          ROTELW = (90.0 - THTAH) * DTOR
          COSELW = COS(ROTELW)
          SINELW = SIN(ROTELW)
          XTMAJW =  XTEW*COSELW + YTEW*SINELW
          YTMAJW = -XTEW*SINELW + YTEW*COSELW
          XSMAJW =  XSEW*COSELW + YSEW*SINELW
          YSMAJW = -XSEW*SINELW + YSEW*COSELW

C         COMPUTE ELLIPTICAL COORDS OF SOURCE AND TOWER
          CALL MUNU( XSMAJW, YSMAJW, AAXW, BAXW, SMUW, SNUW )
          CALL MUNU( XTMAJW, YTMAJW, AAXW, BAXW, TMUW, TNUW )


C       DETERMINE WDBL FOR CASES WHERE HPLW < HC
        PSI = ATAN2(XSEW,YSEW)/DTOR  
        IF (PSI .LE. 0.0) PSI = PSI + 360.0
        THTA1 = PSI

C       INITIALIZE DMIN
        DMIN = 99999

C       CALL WRAPIN FOR DIRECT WIND AND +/- 2 DEG.  IF D LESS THAN 10
C       USE THAT WIND DIRECTION, ELSE FIND THE DIRECTION THAT GIVES 
C       THE MIN D VALUE.

        DO 213 INC = 1,3
          THTA = THTA1 + ADDWD(INC)
          CALL WRAPIN(TMUW,TNUW,SMUW,SNUW,XSMAJW,YSMAJW)
          IF (D .LT. 10.0) THEN
            WDBL = THTA
            GOTO 215
          ELSE
            IF (D .LT. DMIN) THEN
              DMIN = D
              WDBL = THTA
            ENDIF
          ENDIF
213     CONTINUE
        IF (WDBL .EQ. THTA1) GOTO 215

C       NOW GO IN EITHER THE POS OR NEG DIRECTION TO GET A BETTER
C       ESTIMATE OF THE WIND DIRECTION
        IF (WDBL .LT. THTA1) THEN
          MFAC = -1
        ELSE
          MFAC = 1
        ENDIF

        DO 214 INC = 5,30,5
          THTA = THTA1 + MFAC*INC
          CALL WRAPIN(TMUW,TNUW,SMUW,SNUW,XSMAJW,YSMAJW)
          IF (D .LT. 10.0) THEN
            WDBL = THTA
            GOTO 215
          ELSE
            IF (D .LT. DMIN) THEN
              DMIN = D
              WDBL = THTA
            ENDIF
          ENDIF
214     CONTINUE

        THTAMN = WDBL
C       NOW ITERATE AROUND THIS WIND DIRECTION TO FIND THE MIN D.
        DO 216 INC = 1,8
          THTA = WDBL + ADDWD2(INC)
          CALL WRAPIN(TMUW,TNUW,SMUW,SNUW,XSMAJW,YSMAJW)
          IF (D .LT. 10.0) THEN
            WDBL = THTA
            GOTO 215
          ELSE
            IF (D .LT. DMIN) THEN
              DMIN = D
              THTAMN = THTA
            ENDIF
          ENDIF
216     CONTINUE
        WDBL = THTAMN
        D = DMIN

215     IF (WDBL .LT. 0) WDBL = WDBL + 360
        IF (WDBL .GT. 360) WDBL = WDBL - 360

C       DETERMINE WDABVE FOR CASES WHERE HPLW > HC
        KLOW = KLOSE(ZHS(1,NHLW),NZH(NHLW),HC)
        IF (KLOW .EQ. 0) KLOW = 1
        XHILLL = XHL(KLOW,NHLW)
        YHILLL = YHL(KLOW,NHLW)

        DXH = SOURCE(1,NSW) - XHILLL
        DYH = SOURCE(2,NSW) - YHILLL
        PSI = ATAN2(DXH,DYH)/DTOR
        IF (PSI .LE. 0.0) PSI = PSI + 360
        WDABVE = PSI

        IF (HPLW .LT. HC) THEN 
          THTA = WDBL
        ELSE
          THTA = WDABVE
        ENDIF

C       NEED TO CALL WRAPIN ONE MORE TIME FOR SO AND SOBETA
        CALL WRAPIN(TMUW,TNUW,SMUW,SNUW,XSMAJW,YSMAJW)

C       CALCULATE SIGMA-Z AT THE IMPINGEMENT POINT FOR TESTING 
C       THE PLUME HEIGHT AGEINST HCRIT AND SET FLAG

          TLIFT = SO / UV + ZTVW
          IF(TLIFT .LT. 0.0) THEN
              IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $            WRITE(IOUT,6270) NHLW, NS, THTA
              GO TO 187
          ENDIF
          SZTEST = SIGW * TLIFT/SQRT( 1.0 + 0.5*TLIFT/TTLZW )
          WDDIFF = DELWD(WDABVE,WDBL)
          HPLMHC = HPLW - HC
          IF (HPLW .GE. HC .AND. HPLMHC .LT. SZTEST .AND.
     &                          WDDIFF .GT. SMALL) THEN
            ISIGZ = YES
          ELSE
            ISIGZ = NO
          ENDIF

C   SAVE THIS WIND DIRECTION FOR POSSIBLE USE LATER
         NWD = NWD + 1
         WDALL(NWD) = THTA

           DO 355 NR = 1,NRECPT
             CONC(NR)  = 0.0
             CONC1(NR) = 0.0
             DO 354 NS = 1,NSTACK
              SCONC(NR,NS) = 0.0
              SCONC1(NR,NS) = 0.0
354          CONTINUE
355        CONTINUE

           CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
           CALL WRITSCR(CONC,SCONC,1,NUMZI)
           IF (ISOR .GT. 0) CALL SOURCES

C  GO BACK TO DETERMINE A NEW WIND DIRECTION

C      END HILL (WIND DIRECTION) LOOP
187      CONTINUE

C      END SOURCE (WIND DIRECTION) LOOP
185      CONTINUE


C  IF IAUTO = 2, USE THE AVERAGE WIND DIRECTIONS AS WELL
       IF (IAUTO .EQ. 1) GO TO 300
          ISIGZ = 0
          DO 30  J=1,NWD
            DO 20  K=J+1,NWD
              THTA1 = WDALL(J)
              THTA2 = WDALL(K)

C  CALCULATE THE AVERAGE WIND DIRECTION
              THTA1 = DTOR * THTA1
              THTA2 = DTOR * THTA2

              CBAR = 0.5 *(COS(THTA1) + COS(THTA2))
              SBAR = 0.5 * (SIN(THTA1) + SIN(THTA2))
              XPRIME = ATAN(SBAR/CBAR)
              IF (SBAR .GT. 0 .AND. CBAR .GT. 0) THEN
                THTA = XPRIME
              ELSE IF (CBAR .LT. 0) THEN
                THTA = XPRIME + PI
              ELSE
                THTA = XPRIME + 2*PI
              ENDIF
              THTA = THTA/(DTOR)

C      FOR UNSTABLE HOURS, LOOP OVER ZI VALUES
              IF (EL .LT. 0.0) THEN
                DO 457 NHLZI = 1,NHILLS
                  DO 458 NZI = 1,3
                    XMH = ZIHILL(NHLZI,NZI)

C        ZERO THE CONC ARRAY
                    DO 455 NR = 1,NRECPT
                      CONC(NR) = 0.0
                      DO 454 NS = 1, NSTACK
                        SCONC(NR,NS) = 0.0
454                   CONTINUE
455                 CONTINUE
                    
C        IF ZI IS NEGATIVE, SKIP CALCULATIONS AND GO ON TO NEXT ZI
                    IF (XMH .LT. 0.0) GO TO 458

                    CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
                    CALL WRITSCR(CONC,SCONC,1,NUMZI)
                    IF (ISOR .GT. 0) CALL SOURCES

458               CONTINUE
457             CONTINUE
              ELSE
C        ZERO THE CONC ARRAY
                DO 555 NR = 1,NRECPT
                  CONC(NR) = 0.0
                  DO 554 NS = 1,NSTACK
                    SCONC(NR,NS) = 0.0
554               CONTINUE
555             CONTINUE

                CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
                CALL WRITSCR(CONC,SCONC,1,NUMZI)
                IF (ISOR .GT. 0) CALL SOURCES
              ENDIF

20          CONTINUE
30        CONTINUE

C  IF IRANGE = 1 READ IN THE RANGE OF WIND SPEEDS AND THE INCREMENT
300       IF (IRANGE .EQ. 0) GO TO 301
          ISIGZ = 0
          THTA = WDLOW 

C    FOR UNSTABLE HOURS, LOOP OVER ZI VALUES
35        IF (EL .LT. 0) THEN
            DO 657 NHLZI = 1, NHILLS
              DO 658 NZI = 1,3
                XMH = ZIHILL(NHLZI,NZI)

C        ZERO THE CONC ARRAY
                DO 655 NR = 1,NRECPT
                  CONC(NR) = 0.0
                  DO 654 NS = 1,NSTACK
                    SCONC(NR,NS) = 0.0
654               CONTINUE
655             CONTINUE

C        IF ZI IS NEGATIVE, SKIP CALCULATIONS AND GO ON TO NEXT ZI
                IF (XMH .LT. 0.0) GO TO 658
                
                CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
                CALL WRITSCR(CONC,SCONC,1,NUMZI)
                IF (ISOR .GT. 0) CALL SOURCES
658           CONTINUE
657         CONTINUE
          ELSE

C        ZERO THE CONC ARRAY
            DO 755 NR = 1,NRECPT
              CONC(NR) = 0.0
              DO 754 NS = 1,NSTACK
                SCONC(NR,NS) = 0.0
754           CONTINUE
755         CONTINUE

            CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
            CALL WRITSCR(CONC,SCONC,1,NUMZI)
            IF (ISOR .GT. 0) CALL SOURCES
          ENDIF

          THTA = THTA + WDINC
          IF (THTA .GT. 360.0) THTA = THTA - 360.0
          IF (WDUP .LT. WDLOW) THEN
            IF (WDLOW .LT. THTA .AND. THTA .LE. 360.0) THEN
              IF (THTA .GT. WDUP2) GO TO 301
            ELSE
              IF (THTA .GT. WDUP) GO TO 301
            ENDIF
          ELSE
            IF (THTA .GT. WDUP) GO TO 301
          ENDIF
          GO TO 35


C  IF IDISCR = 1 READ IN THE DISCRETE WIND DIRECTIONS TO USE
301       IF (IDISCR .EQ. 0) GO TO 303
          ISIGZ = 0
          DO 36 I = 1,NUMWD
            THTA = DISCWD(I)

C   FOR UNSTABLE HOURS, LOOP OVER ZI VALUES
            IF (EL. LT. 0) THEN
              DO 857 NHLZI = 1,NHILLS
                DO 858 NZI = 1,3
                  XMH = ZIHILL(NHLZI,NZI)

C        ZERO THE CONC ARRAY
                  DO 855 NR = 1,NRECPT
                    CONC(NR) = 0.0
                    DO 854 NS = 1, NSTACK
                      SCONC(NR,NS) = 0.0
854                 CONTINUE
855               CONTINUE

C        IF ZI IS NEGATIVE, SKIP CALCULATIONS AND GO ON TO NEXT ZI
                  IF (XMH .LT. 0.0) GO TO 858

                  CALL CONCALC(OLDAY,SVGOOD,SWGOOD)
                  CALL WRITSCR(CONC,SCONC,1,NUMZI)
                  IF (ISOR .GT. 0) CALL SOURCES
858             CONTINUE
857           CONTINUE
            ELSE
C        ZERO THE CONC ARRAY
              DO 955 NR = 1,NRECPT
                CONC(NR) = 0.0
                DO 954 NS = 1,NSTACK
                  SCONC(NR,NS) = 0.0
954             CONTINUE
955           CONTINUE

              CALL CONCALC(OLDAY,SVGOOD,SWGOOD) 
              CALL WRITSCR(CONC,SCONC,1,NUMZI)
              IF (ISOR .GT. 0) CALL SOURCES
            ENDIF
36        CONTINUE

C  END OF WIND DIRECTION AUTOMATION METHODS
303       CONTINUE

C  GO ON TO NEXT HOUR FOR THE SCREEN
          GO TO 100

C  FIND THE MAXIMUM PREDICTED CONCENTRATION

304       CHIMAX = 0.0
          DO 305 NR = 1,NRECPT
            IF(CONC(NR) .GT. CHIMAX) THEN
                CHIMAX = CONC(NR)
                NRMAX = NR
            ENDIF
305       CONTINUE

C       PRINT HOURLY MAX CONCENTRATION
        IF(ICASE.EQ.1 .OR. ICASE .EQ. 3) THEN
            IF(CHIMAX .GT. CHIPRT) THEN
                WRITE(IOUT,6185) CHIMAX, GSGS(ICHIQ+1), NRMAX
              ELSE
                WRITE(IOUT,6190) CHIMAX, GSGS(ICHIQ+1), NRMAX
            ENDIF
        ENDIF

C       FILL TOP N ARRAYS
        IF(ITOPN .EQ. YES) CALL TOPN( CONC, NRECPT, 0, GSGS(ICHIQ+1))

        GO TO 400

C       HOURLY ERROR SECTION FOR CASES WHEN THE MODEL IS NOT APPROPRIATE

C       USTAR MISSING
330     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $        WRITE(IOUT,9335) JYR, JMO, JDY, KJCD, JHR
        GO TO 390
C
C       Z0 MISSING
340     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $     WRITE(IOUT,9345) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       MISSING MET DATA: EITHER WD, WS, SIGV, OR SIGW
350     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $         WRITE(IOUT,9355) JYR, JMO, JDY, KJCD, JHR
        GO TO 390

C       WIND SPEED AT PLUME HEIGHT LESS THAN 1.0 M/S
370     IF(ICASE .EQ. 1 .OR. ICASE .EQ. 3) 
     $      WRITE(IOUT,9375) JYR, JMO, JDY, KJCD, JHR

C       SET CONCENTRATIONS TO MISSING FOR THIS HOUR
390     DO 395 NR=1,NRECPT
          CONC(NR) = -999.
          DO 396 NS=1,NSTACK
            SCONC(NR,NS) = -999.
396       CONTINUE
395     CONTINUE

C      WRITE CONCENTRATIONS TO OUTPUT FILE
400     IF(ICONC .NE. NO) THEN
            NR = NRECPT
            IF( CONC(1) .EQ. -999. ) NRMAX = 0
            MET(1) = KYR
            MET(2) = KMO
            MET(3) = KDY
            MET(4) = KHR
            MET(5) = NRMAX
            IF (ISCRN .GT. 0) THEN 
              CALL WRITSCR(CONC,SCONC,1,NUMZI)
            ELSE
              CALL WRITIT(MET,CONC,NRECPT,IOCONC)
            ENDIF
        ENDIF

C       PRINT HOURLY SOURCE CONTRIBUTION TABLE (IF NECESSARY)
        IF (ISOR .GT. 0) CALL SOURCES

        GO TO 100

 999    CONTINUE

C       PRINT TOP N TABLE (IF NECESSARY)
        IF(ITOPN .EQ. YES) CALL TOPN( CONC, NRECPT, 1, GSGS(ICHIQ+1))

C       WRITE OUTPUT FILES FOR SCREENING MODE
        IF (ISCRN .GT. 0) CALL WRITSCR(CONC,SCONC,2,NUMZI)

        RETURN

6105    FORMAT(/,1X,'DISAGREEMENT IN PROFILE VS. SURFACE DATA: ',/,10X,
     1   'PROFILE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2,/,10X,
     2   'SURFACE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2)
6107    FORMAT(/,1X,'PROFILE HEIGHT VALUE INCORRECT: ',/,10X,
     1   'PROFILE DATA DATE:HOUR IS ',I2,'/',I2,'/',I2,':',I2)
6108    FORMAT(/'   LEVEL NUMBER:',I2,'      HEIGHT:',F8.2/)
6109    FORMAT(/'   LEVEL NUMBER:',I2,'      PREVIOUS HEIGHT:',F8.2,
     1          '    CURRENT HEIGHT:',F8.2)
6115    FORMAT(  1X,80('-'),//,'  INPUT MET DATA FROM  SURFACE  AND ',
     1    ' PROFILE  (NOTE: ****** = MISSING DATA):',//,
     2    T40,'MONIN-     SFC',/,T17,
     3      'MIXING   SFC    SFC    OBUKHOV   ROUGH.',/,T17,
     4      'HEIGHT   TEMP    U*    LENGTH    LENGTH',/,'  YR MO DA HR',
     5   '    (M)      (K)  (M/S)     (M)       (M)',//,
     6    1X,4I3,3X,F6.1,2X,F5.1,1X,F5.3,3X,F7.1,2X,F7.4,///,
     7    T9,'ADJUSTED  WIND    <-WIND  SPEED->  AMB.   SIGMA-',/,
     *    T10,'HEIGHT   DIR.    SCALAR  VECTOR   TEMP   THETA    ',
     *         'SIGMA-V ',
     8    '  SIGMA-W',/,
     9    T10,'  (M)   (DEG)     (M/S)   (M/S)   (K)     (DEG)   ',
     *    ' (M/S)',
     A    '     (M/S)',/)
6120    FORMAT(T10,F6.1,2X,F6.1,2X,F7.2,2X,F7.2,1X,F7.2,1X,F6.1,3X,F7.2,
     1   4X,F7.2)
6125    FORMAT(/,10X,'NO EMISSIONS FROM SOURCE # ',I2,/)
6122    FORMAT(/,10X,'NOTE: SCALAR WIND SPEEDS ARE',
     1    ' SET TO A MINIMUM OF 1 M/S'/
     2  10X,'NOTE: HEIGHTS ARE REFERENCED TO THE COMMON STACK BASE',
     3  ' ELEVATION',/,10X,
     4  '      THE ADJUSTMENT TO THE INPUT HEIGHT IS ',F5.1,' METERS.'/)
6123    FORMAT(10X,'NOTE: HEIGHTS ARE REFERENCED TO THE COMMON STACK ',
     1  'BASE ELEVATION',/,10X,
     2  '      THE ADJUSTMENT TO THE INPUT HEIGHT IS ',F5.1,' METERS.'/)
6185    FORMAT(/,1X,'MAXIMUM CONCENTRATION FOR THIS HOUR IS ',
     1  F10.2,' U',A1,'/M**3 AT RECEPTOR # ',I3,/)
6190    FORMAT(/,1X,'MAXIMUM CONCENTRATION FOR THIS HOUR IS ',
     1  1PE10.4,' U',A1,'/M**3 AT RECEPTOR # ',I3,/)
6270    FORMAT(/' POSITION OF HILL ',I2,' IS UPWIND OR FAR TO THE SIDE',
     *          ' OF THE PLUME FROM SOURCE ',I2,/,' THIS HOUR. ',
     *          ' WIND DIRECTION IS ',F5.1,/)
9335    FORMAT(/' USTAR MISSING, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9345    FORMAT(/' Z0 MISSING, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9355    FORMAT(/' MISSING MET. INPUT, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)
9500    FORMAT(' DAY = ',I4)
9502    FORMAT(' CALCULATING HOUR ',4I3)
9375    FORMAT(/' WIND SPEED LT 1 M/S, NO PREDICTIONS THIS HOUR',
     *  /'   YEAR=',I2,' MONTH=',I2,' DAY=',I2,' JCD=',I3,' HOUR=',I2/)


        END
