       PROGRAM BMTS

c --> temporary changes with xxx

C-------------------------------------------------------------------
C
C PURPOSE : To implement the Britter-McQuaid model
C
C-------------------------------------------------------------------

c========================================================================
c
c... comments/changes made on 23 July 92 are marked as ts/23jul92
c
c	Attempts to address the two-case approach Britter and McQuaid
c	recommend when dealing with both nonisothermal releases and aerosol
c	releases.
c			! ts/23jul92
c
c	Log-log interpolation is required for reading off the Figures from
c	Britter and McQuaid's original work.  The original code mixed
c	some variables in the log form and some which were not in the
c	log form.
c
c========================================================================

c========================================================================
c
c... Comments/changes made on 10 July 92 are marked as ts/10jul92
c
c	Earlier changes which had been made to estimate the temperature when
c	the liquid (condensed) phase first evaporates in a aerosol/air mixture.
c	
c========================================================================

C >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> DECLARATIONS <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

      INCLUDE 'const.inc'

C Initialize Constants

      zero=0.
      third=1./3.
      one=1.
      g=9.81
      tice=273.2
      vmolstp=22.413
      cpair=1005.9
      mwaird=28.966
      mwh2o=18.016
      cinit=10.**6.

      IPRT = 6
      IERR = 10
      OPEN(IERR,FILE='ERROR.OUT',STATUS='UNKNOWN')

C >>>>>>>>>>>>>>>>>>>>>>>>>>>>> MAIN PROGRAM <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

C Read in the Data

      CALL Read1

C Calculate variables that are derived from the data provided

      CALL Calc1

C Perform Britter-McQuaid Model

      CALL BM

C  Produce output

      CALL BMOut

C End

      CLOSE (IERR, STATUS = 'DELETE')

      END


C >>>>>>>>>>>>>>>>>>>>>>>>>>> SUBPROGRAMS <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

c-----------------------------------------------------------------------
      subroutine READ1
c-----------------------------------------------------------------------
c
c --- READ1     Version: 0.0            Level: 900806           MDA
c ---           D. Strimaitis, SRC
c
c PURPOSE:      Subroutine reads the primary data file (bmts.in) for
c               each experiment.
c
c ARGUMENTS:    none
c
c CALLING ROUTINES: main
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

      INCLUDE 'params.inc'
      INCLUDE 'readin.inc'

      in = 7
      OPEN(in,FILE='BMTS.IN',STATUS='UNKNOWN')

c  Read the chemical properties and conditions of the release
c  (these may change from trial to trial)
      READ(in,101) title
      read(in,*) mw
      read(in,*) tbp
C     Order of variables cpv and latht reversed to match output order
C     and order as passed by TSCREEN - Roger Brode, 5/13/94
      read(in,*) cpv      
      read(in,*) latht
      read(in,*) phase		! ts/23jul92/
      read(in,*) texit
      read(in,*) rate
      read(in,*) dur
      read(in,*) mass
C      read(in,*) cinit
c  Read conditions at the site
      read(in,*) pamb
      read(in,*) rh
      read(in,*) t1
      read(in,*) tav


c  Read attributes of the concentration data
      read(in,*) ndist
      do 20, j=1,ndist
20       read(in,*) xdist(j)

      close(in)
      return

c  Format statements
101   format(79a1)
102   format(a5)

      end


c-----------------------------------------------------------------------
      subroutine CALC1
c-----------------------------------------------------------------------
c
c --- CALC1     Version: 0.0            Level: 901126           MDA
c ---           D. Strimaitis, SRC
c               J. Chang, SRC
c
c PURPOSE:      Subroutine calculates variables that are derived from
c               the data initially provided.
c
c ARGUMENTS:    none
c
c CALLING ROUTINES: main
c
c EXTERNAL ROUTINES:
c-----------------------------------------------------------------------

      INCLUDE 'params.inc'
      INCLUDE 'readin.inc'
      INCLUDE 'const.inc'
      INCLUDE 'calc.inc'

      real mixrat

c  Calculate the molar volume (assume perfect gas) at boiling point
c  ambient, and exit temperature
         vmolamb=vmolstp*t1/(tice*pamb)
         vmolbpt=vmolstp*tbp/(tice*pamb)
         vmolxit=vmolstp*texit/(tice*pamb)

c  Calculate the density of the vapor at exit temperature
         rhoxit=mw/vmolxit

c  Calculate the volume flux of the vapor at exit temperature
c		! vflxxit is really a rate instead of a flux.  ts/23jul92
         vflxxit=rate/rhoxit

c  Calculate the total volume of the vapor cloud at exit temperature
         vtotxit=mass/rhoxit

c
c  In the following, it is assumed that for instantaneous releases,
c  the geometry of the source cloud is a cylinder whose height equals
c  its radius.  Namely, 
c                total volume = pi * radius**3
c  For finite-duration releases, it is assumed that the cross section
c  of the source cloud is a circle with
c               pi * radius**2 * ws(@1m) = volume flux.
c
c  NOTE THAT IT IS DIAMETER WHICH IS ACTUALLY STORED IN THE ARRAYS.
c

c  Calculate the properties of moist air
         psat=.00603*EXP(5417.8*(one/tice-one/t1))
c         mixrat=.01*rh*mwh2o/(mwaird*pamb/psat-one)	! in error; ts
         mixrat=.01*rh*mwh2o/(mwaird*(pamb/psat-one))
         mwair=mwaird*(one+mixrat)/(one+mixrat*mwaird/mwh2o)
         rhoair=mwair/vmolamb

         molfrac=cinit/1000000.

c  Account for initial air in the released material, assuming
c  that the cloud temperature equals the exit temperature
c  (not for aerosol case)
         rhodil=(molfrac*mw+(one-molfrac)*mwair)/
     1             vmolxit
         vflxdil=vflxxit/molfrac
         vtotdil=vtotxit/molfrac

C  Calculate the fraction of a liquid release that may flash to vapor.
C  None flashes if boiling point temperature is greater than ambient,
C  or if the storage phase is not listed as Cryogenic.
C  Note that the exit temperature is used when available.
C  Also calculate the density of an aerosol-vapor mix at the boiling pt
C  Start with vapor state:
c         fvap=one
c         if(phase .EQ. 'L') then
c            fvap=zero
c         elseif(phase .EQ. 'C') then
c            tdiff=texit-tbp
c
c            if(tdiff .GE. zero) then
c               fvap=AMIN1(one,cl*tdiff/latht)
c            endif
c         endif
c
	tdiff=texit-tbp
	fvap = phase
C
C  Section for approximating the effect of aerosols (when present)
C
c         if(phase .EQ. 'C') then
	if(phase .lt. 1.) then		! ts/23jul92/
C
C  Treatment of aerosols (2): dilute vapor/aerosol mixture with enough
C  air to completely evaporate the aerosols
C  (temperature of diluted cloud = boiling pt)
C  This treatment is considered to be a diluted release, i.e., molfrac
C  is possible to be smaller than 1.
	    tdiff=t1-tbp
            molfrac=one

            rhodil=mw/vmolbpt
            vtotdil=mass/rhodil

C            if(type .EQ. 'IR') then
C               vflxdil=zero
C            else
               vflxdil=rate/rhodil
C            endif
            if(tdiff .GT. zero) then
c
c... determine the temperature and mole fraction of the aerosol/air mixture
c	when sufficient air has been added so that the liquid contaminant
c	phase is evaporated.
c
c               molfrac=one/(one+(one-fvap)*(mw/mwair)*
c     1                    (latht/(cpair*tdiff)))
c               rhodil=(molfrac*mw+(one-molfrac)*mwair)/
c     1                   vmolbpt
c               vflxdil=vflxdil/molfrc
c               vtotdil=vtotdil/molfrac
c
c...	start with the first guess for "molfrac"
c
		rgas = 8314./mw
		molfrac = one/(one + latht*(one-fvap)/tdiff/cpv)
c
c...	set up a loop for ten trials...
c.............................................................../ts/10jul92/
c
c	The following procedure estimates the mole fraction when the aerosol
c	is just evaporated.  It takes into account the fact that the
c	air/contaminant mixture will be subcooled at this point.  Rex
c	Britter indicated that I was technically correct on this point,
c	but he indicated that his favored approach was to eschew this
c	complication and calculate the mole fraction as though the
c	temperature when the aerosol evaporates is (approximately) the
c	boiling point temperature.  I believe the reason Britter feels
c	justified in this position is that *the dispersion* calculations
c	will not be significantly effected by this simplification.  Since
c	this is Britter's Workbook method, I am changing the code to reflect
c	his opinion on the matter.
c
c	Britter's simplification is just the first guess I used above.
c	So, the following loop (which produces the refined estimate) is
c	simply commented out.
c
c		do iii = 1,10
c		    ttt1 = t1 - molfrac*(tdiff + latht*(one-fvap)/cpv)
c		    ttt2 = one/(one/tbp - rgas/latht*log(pamb*molfrac))
c		    fff = ttt1 - ttt2
c		    dfff = -(tdiff + latht*(one-fvap)/cpv)
c     .			  - rgas/latht/molfrac/
c     .			    (one/tbp - rgas/latht*log(pamb*molfrac))**2
c		    corr = fff/dfff
c		    if(abs(corr) .le. 0.0001) goto 100
c		    molfrac = molfrac - corr
c		enddo
c
c		write(6,*) 'molfrac failed to converge'
c		molfrac = 1
c
c  100		continue
c
c
c............................................................/ts/10jul92/
c
		temp = t1 - molfrac*(tdiff + latht*(one-fvap)/cpv)
c
c.........................................................../ts/10jul92/
c
c	The second case which Britter and McQuaid recommend for
c	considering aerosol releases essentially assumes that the
c	temperature calculated in the above statement is the ambient
c	temperature.  Or,
c
c	temp = t1
c
c	All of the following statements would still apply equally well
c	with the updated temperature.  I left this statement commented out
c	because it cannot be directly implemented here without
c	some changes.  And, the approach to treating the two cases needs
c	to take into consideration the structure used in TSCREEN.
c	(These changes are implemented below in subroutine BM.)
c
c	Finally, note that similar considerations apply when making
c	predictions for nonisothermal gases.  The present code treats the
c	case when heat transfer leaves the initial temperature unchanged,
c	but the second case when heat transfer changes the initial
c	temperature to ambient temperature (and as a consequence, changes
c	the initial density) is not considered.
c
c.........................................................../ts/10jul92/
c
		wmw = molfrac*mw+(one-molfrac)*mwair
		rhodil= wmw/vmolbpt/temp*tbp

		if(vflxdil .ne. zero) vflxdil=rate/mw/molfrac*wmw/rhodil
		vtotdil=mass/mw/molfrac*wmw/rhodil

            endif
C
C  End aerosol section
        endif

      return
      end


c-----------------------------------------------------------------------
      subroutine BM
c-----------------------------------------------------------------------
c
c --- BM        Version: 0.0            Level: 910124           MDA
c ---           D. Strimaitis, SRC
c               J. Chang, SRC
c
c PURPOSE:      Subroutine performs units conversions needed to
c               use the Britter & McQuaid workbook.
c               It also calls BMMATCH to estimate concentrations from
c               the curves in the workbook.
c
c ARGUMENTS:    none
c
c CALLING ROUTINES: main
c-----------------------------------------------------------------------

      character*1 flagc,flagi
      INCLUDE 'params.inc'
      INCLUDE 'readin.inc'
      INCLUDE 'const.inc'
      INCLUDE 'calc.inc'
      INCLUDE 'bm.inc'
      

c  Select the appropriate properties
c         if(phase .EQ. 'C' .OR. cinit .ne. 1000000.) then
	if(phase .lt. 1. .OR. cinit .ne. 1000000.) then		! ts/23jul92/
c  Use the properties of a source-cloud diluted with air if flashing
c  had occurred, or if air is already mixed in
            rho=rhodil
            vflux=vflxdil
            vol=vtotdil
         else
            rho=rhoxit
            vflux=vflxxit
            vol=vtotxit
         endif

C  Loop over number of distances
      XCOUNT = 0
      DO 200 j=1, ndist

C Display percent complete on the screen
         XCOUNT = XCOUNT + 1
         PERCENT = XCOUNT/ndist*100
         IPERCENT = NINT(PERCENT)
         WRITE(IPRT,101) IPERCENT
101      FORMAT('+',30X,I3,' % Complete')

C Initialize
         u(j) = 0
         conppm(j) = 0
         x=xdist(j)
         
C  Loop over number of windspeeds
         DO 300 i=1, nws10
c  Calculate parameters
            gprime=g*(rho-rhoair)/rhoair
            gprime=AMAX1(zero,gprime)
            dc=SQRT(vflux/ws10(i))
            di=vol**third
            idi=one/di
            idc=9999.
            if(dc.gt.0.) idc=one/dc

c  Is the release dense enough to apply the model?

C            densec='Y'
C            densei='Y'
C            testc=((vflux*gprime*idc)**third)/ws10
C            testi=SQRT(gprime*vol)*idi/ws10
C            if(testc .LT. 0.15) densec='N'
C            if(testi .LT. 0.20) densei='N'

c  As a matter of fact, the curves in correlation figures presented
c  in Britter  and Mcquaid's workbook do converge to passive limit,
c  meaning that the model can be extended to tracer releases.

c  Minimum dist. for instantaneous results
            xinst=ws10(i)*dur/0.6
c  Maximum dist. for continuous results
            xconst=ws10(i)*dur/2.5
c  Model parameters
            pinst=SQRT(gprime/idi)/ws10(i)
	    pconst = 0.
	    if(gprime .ne. 0.) pconst=((gprime**2*vflux)**0.2)/ws10(i)

            qinst=x*idi
            qconst=x*idc
c
c  Determine the attributes of the release
c

C         if (type.eq.'IR') then
C           flagc='N'
C           flagi='Y'
C         else
           if (x.lt.xconst) then
             flagc='Y'
             flagi='N'
           else if (x.gt.xinst) then
             flagc='N'
             flagi='Y'
           else
             flagc='Y'
             flagi='Y'
           end if
C         end if

            call BMMATCH(pconst,qconst,pinst,qinst,
     1                   flagc,flagi,ratio,tav,texit,t1)

c======================================================================
c
c... Additions to handle the two-case approaches for addressing nonisothermal
c	releases and aerosol releases.			! ts/23jul92
c
c...	Treat the nonisothermal (heat transfer) cases first.
c
c	For nonisothermal cases, the cloud density and temperature are
c	modified.
c
	if(texit .ne. t1) then
	    rho_ = rho * (texit/t1)
	    vflux_ = vflux * (t1/texit)
	    vol_ = vol * (t1/texit)
c	    texit = t1			! done in the call statement

            gprime=g*(rho_-rhoair)/rhoair
            gprime=AMAX1(zero,gprime)
            dc=SQRT(vflux_/ws10(i))		! vflux is really a rate.
            di=vol_**third
            idi=one/di
            idc=9999.
            if(dc.gt.0.) idc=one/dc

            pinst=SQRT(gprime/idi)/ws10(i)
	    pconst = 0.
	    if(gprime .ne. 0.) pconst=((gprime**2*vflux_)**0.2)/ws10(i)

            qinst=x*idi
            qconst=x*idc
c
	    case1 = ratio

            call BMMATCH(pconst,qconst,pinst,qinst,
     1                   flagc,flagi,ratio,tav,t1,t1)

	    case2 = ratio
	    ratio = max(case1, case2)
	endif
c
c
c...	Now, treat the aerosol cases.
c
c	For aerosol cases, the cloud density and temperature are
c	modified along with the release rate.
c
c	if(phase .eq. 'C') then
	if(phase .lt. 1.) then		! ts/23jul92/
	    tdiff=t1-tbp
	    rgas = 8314./mw
	    molfrac = one/(one + latht*(one-fvap)/tdiff/cpv)
	    wmw = molfrac*mw + (one-molfrac)*mwair

	    temp_ = t1
	    rho_ = wmw/vmolbpt/temp_*tbp

	    vflux_ = rate/mw/molfrac*wmw/rho_
	    vol_   = mass/mw/molfrac*wmw/rho_
c	    texit = t1			! done in the call statement

            gprime=g*(rho_-rhoair)/rhoair
            gprime=AMAX1(zero,gprime)
            dc=SQRT(vflux_/ws10(i))		! vflux is really a rate.
            di=vol_**third
            idi=one/di
            idc=9999.
            if(dc.gt.0.) idc=one/dc

            pinst=SQRT(gprime/idi)/ws10(i)
	    pconst = 0.
	    if(gprime .ne. 0.) pconst=((gprime**2*vflux_)**0.2)/ws10(i)

            qinst=x*idi
            qconst=x*idc
c
	    case1 = ratio

            call BMMATCH(pconst,qconst,pinst,qinst,
     1                   flagc,flagi,ratio,tav,t1,t1)

	    case2 = ratio
	    ratio = max(case1, case2)
	endif
c
c			! changes to here ts/23jul92/
c======================================================================

C            conc(j)=molfrac*ratio*1.e+6

            conhld=molfrac*ratio*1.e+6
            IF (conhld .GT. conppm(j)) THEN
               conppm(j) = conhld
               u(j) = ws10(i)
            ENDIF

300      CONTINUE
200   continue

C  Convert from (ppm) to (ug/m^3)
      DO j = 1,ndist
c        conc(j) = (conppm(j) * mw) / 0.0245
	conc(j) = (conppm(j) * mw) *1000. * pamb/0.08205/t1	! ts/23jul92
      ENDDO

C Find maximum
      maxidx = 1
      DO j = 2,ndist
         IF (conc(j) .GT. conc(maxidx)) THEN
            maxidx = j
         ENDIF
      ENDDO

C create graph file
      IPLT = 12
C      OPEN(IPLT,FILE='GRAPH.FIL',STATUS='UNKNOWN')
      OPEN(IPLT,FILE='BM.FIL',STATUS='UNKNOWN')
      WRITE(IPLT,70) TITLE
70    FORMAT(79A1)
      WRITE(IPLT,*) ndist
      WRITE(IPLT,500) xdist(1),conc(maxidx),xdist(maxidx),
     &     (conc(i),xdist(i),i = 1,ndist)
500   FORMAT(G12.4)
      CLOSE(IPLT)

      return
      end

c ---------------------------------------------------------------------
      subroutine bmmatch(xc,yc,xi,yi,flagc,flagi,ratio,tav,texit,t1)
c ---------------------------------------------------------------------
c
c --- BMMATCH   Version: 0.0            Level: 901128           MDA
c               J. Chang, SRC
c               C. Haga, SRC
c Purpose:
c     Electronic form of the Britter and McQuaid model
c
c Arguments:
c     XC    real      x-coordinate on the correlation figure for
c                     continuous releases
c     YC    real      y-coordinate on the correlation figure for
c                     continuous releases
c     XI    real      x-coordinate on the correlation figure for
c                     instantaneous releases
c     YI    real      y-coordinate on the correlation figure for
c                     instantaneous releases
c     FLAGC character flag indicating the applicability of conti. rel.
c     FLAGI character flag indicating the applicability of insta. rel.
c     RATIO real      return the value of the ratio of the maximum
c                     concentration at a certain downwind distance and
c                     the initial concentration
c
C     tav   real      averaging time
C     texit real      exit temperature (K)
C     t1    real      ambient temperature #1-lower sensor (K)
c Note:
c     1) For points off the chart, it is assumed that concentration
c     decreases as 1/x**2.  In the near field region, a simple
c     clamping like C/(1+C) is applied so that RATIO will never be
c     greater than 1.				! no longer applies ts/23jul92/
c     2) All digitizations are done in the logarithmic space.
c
c Reference:
c     Britter, R.E. and J. McQuaid, 1988:  Workbook on the Dispersion
c     of Dense Gases.  HSE Contract Research Report No. 17/1988.
c
c ---------------------------------------------------------------------
c
c      INCLUDE 'readin.inc'

	parameter (third=1./3.)

      character*1 flagc,flagi
      real ifcn1,ifcn2,ifcn3,ifcn4,ifcn5,ifcn6,ifcn7
      real ybmc(6),ybmi(7),rbmc(6),rbmi(7),iratio
c
      data rbmc/0.1,0.05,0.02,0.01,0.005,0.002/
      data rbmi/0.1,0.05,0.02,0.01,0.005,0.002,0.001/
c

c
      cratio=0.
      iratio=0.

c --- continuous simulation
c
      if (flagc.eq.'Y' .or. flagc.eq.'y') then
c --- convert x,y to log space
c         rxc=alog10(xc)
	 rxc=alog10(max(xc, 0.01)) ! to catch the passive limit from crashing
c				  ! ts/23jul92
         ryc=alog10(yc)
c
c --- calculate all possible BM y values for the given x
         ybmc(1)=cfcn1(rxc)
         ybmc(2)=cfcn2(rxc)
         ybmc(3)=cfcn3(rxc)
         ybmc(4)=cfcn4(rxc)
         ybmc(5)=cfcn5(rxc)
         ybmc(6)=cfcn6(rxc)
c
c ---- find the position of the given y value on the BM plot
c ---  interpolate or extrapolate to find the correct conc. ratio.
         if (ryc.lt.ybmc(1)) then
c           cratio = (306*yc**(-2.))/(1+306*yc**(-2.))
c           cratio = amax1(cratio,rbmc(1))
           cratio = rbmc(1)                                     ! correction per TOM SPICER'S comments
         else if (ryc.gt.ybmc(6)) then
c           dy=ryc-ybmc(6)
c           cratio = rbmc(6)/(10.**dy)**2.
	    if(xc .ge. 1.) then
		cratio = (22.6/yc)**2/xc ! correction per TOM SPICER'S comments
	    elseif(xc.lt.1. .and. xc.gt.0.2) then
		cratio = (22.6/yc)**2*xc**third ! correction per TOM SPICER'S comments
	    else
		cratio = (17.3/yc)**2 ! correction per TOM SPICER'S comments
	    endif
         else
           do 10 i=1,5
              if (ybmc(i).le.ryc .and. ryc.le.ybmc(i+1)) then
c                   cratio = (rbmc(i+1)-rbmc(i))/(ybmc(i+1)-ybmc(i)) *
c     +                      (ryc-ybmc(i)) + rbmc(i)
		bmci = 10.**(ybmc(i))
		bmcip= 10.**(ybmc(i+1))
                   cratio = rbmc(i) * exp(log(rbmc(i+1)/rbmc(i))/
     .				log(bmcip/bmci) *
     .				log(yc/bmci) )		! ts/23jul92/
              endif
10         continue
         endif
c --- Correct "cratio" for averaging time -- per TOM SPICER comments
         cratio = cratio * (10/tav)**(0.05)
      endif

c --- puff simulation
c
      if (flagi.eq.'Y' .or. flagi.eq.'y') then

c --- convert x,y to log space
c         rxi=alog10(xi)
	 rxi=alog10(max(xi, 0.01)) ! to catch the passive limit from crashing
c				  ! ts/23jul92
         ryi=alog10(yi)
c
c --- calculate all possible BM y values for the given x
         ybmi(1)=ifcn1(rxi)
         ybmi(2)=ifcn2(rxi)
         ybmi(3)=ifcn3(rxi)
         ybmi(4)=ifcn4(rxi)
         ybmi(5)=ifcn5(rxi)
         ybmi(6)=ifcn6(rxi)
         ybmi(7)=ifcn7(rxi)
c
c ---- find the position of the given y value on the BM plot
c ---  interpolate or extrapolate to find the correct conc. ratio.
         if (ryi.lt.ybmi(1)) then
c           iratio = (3.24*yi**(-2.))/(1+3.24*yi**(-2.))
c           iratio = amax1(iratio,rbmi(1))
           iratio = rbmi(1)                   ! correction per TOM SPICER'S comments
         else if (ryi.gt.ybmi(7)) then
c            dy=ryi-ybmi(7)
c            iratio = rbmi(7)/(10.**dy)**2.
	    if(xi .ge. 1.) then
		iratio = (10.2/yi/sqrt(xi))**2.5   ! correction per TOM SPICER'S comments
	    else
		iratio = (10.2/yi)**2.5   ! correction per TOM SPICER'S comments
	    endif
         else
           do 20 i=1,6
              if (ybmi(i).le.ryi .and. ryi.le.ybmi(i+1)) then
c                   iratio = (rbmi(i+1)-rbmi(i))/(ybmi(i+1)-ybmi(i)) *
c     +                      (ryi-ybmi(i)) + rbmi(i)
		bmii = 10.**(ybmi(i))
		bmiip= 10.**(ybmi(i+1))
                   iratio = rbmi(i) * exp(log(rbmi(i+1)/rbmi(i))/
     .				log(bmiip/bmii) *
     .				log(yi/bmii) )	! ts/23jul92/
              endif
20         continue
         endif
      endif

c --- take the minimum concentration ratio if both plume and puff
c --- simulations are applicable
      if ( (flagc.eq.'Y' .or. flagc.eq.'y') .and.
     1     (flagi.eq.'N' .or. flagc.eq.'n') ) then
          ratio=cratio
      elseif ( (flagc.eq.'N' .or. flagc.eq.'n') .and.
     1         (flagi.eq.'Y' .or. flagc.eq.'y') ) then
          ratio=iratio
      elseif ( (flagc.eq.'Y' .or. flagc.eq.'y') .and.
     1         (flagi.eq.'Y' .or. flagc.eq.'y') ) then
          ratio=amin1(cratio,iratio)
C      else
C          print *,'ERROR in BMMATCH: both FLAGC and FLAGI are N'
C          stop
      end if

c --- correct concentration for nonisothermal treatment  correction per TOM SPICER'S comments
      ratio = ratio/(ratio + (1 - ratio) * (texit/t1))

      return
      end

c ---------------------------------------------------------------------
      real function cfcn1(x)
c
c     Corresponds to RATIO = 0.10
c
      if (x.le.-0.55) then
         cfcn1 = 1.75
      else if (-0.55.lt.x .and. x.le.-0.14) then
         cfcn1 = 0.24*x + 1.88
      else if (-0.14.lt.x .and. x.le.1.) then
         cfcn1 = -0.50*x + 1.78
      else
         cfcn1 = -0.50*x + 1.78
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function cfcn2(x)
c
c     Corresponds to RATIO = 0.05
c

      if (x.le.-0.68) then
         cfcn2 = 1.92
      else if (-0.68.lt.x .and. x.le.-0.29) then
         cfcn2 = 0.36*x + 2.16
      else if (-0.29.lt.x .and. x.le.-0.18) then
         cfcn2 = 2.06
      else if (-0.18.lt.x .and. x.le.1.) then
         cfcn2 = -0.56*x + 1.96
      else
         cfcn2 = -0.56*x + 1.96
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function cfcn3(x)
c
c     Corresponds to RATIO = 0.02
c

      if (x.le.-0.69) then
         cfcn3 = 2.08
      else if (-0.69.lt.x .and. x.le.-0.31) then
         cfcn3 = 0.45*x + 2.39
      else if (-0.31.lt.x .and. x.le.-0.16) then
         cfcn3 = 2.25
      else if (-0.16.lt.x .and. x.le.1.) then
         cfcn3 = -0.54*x + 2.16
      else
         cfcn3 = -0.54*x + 2.16
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function cfcn4(x)
c
c     Corresponds to RATIO = 0.01
c

      if (x.le.-0.70) then
         cfcn4 = 2.25
      else if (-0.70.lt.x .and. x.le.-0.29) then
         cfcn4 = 0.49*x + 2.59
      else if (-0.29.lt.x .and. x.le.-0.20) then
         cfcn4 = 2.45
      else if (-0.20.lt.x .and. x.le.1.) then
         cfcn4 = -0.52*x + 2.35
      else
         cfcn4 = -0.52*x + 2.35
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function cfcn5(x)
c
c     Corresponds to RATIO = 0.005
c

      if (x.le.-0.67) then
         cfcn5 = 2.40
      else if (-0.67.lt.x .and. x.le.-0.28) then
         cfcn5 = 0.59*x + 2.80
      else if (-0.28.lt.x .and. x.le.-0.15) then
         cfcn5 = 2.63
      else if (-0.15.lt.x .and. x.le.1.) then
         cfcn5 = -0.49*x + 2.56
      else
         cfcn5 = -0.49*x + 2.56
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function cfcn6(x)
c
c     Corresponds to RATIO = 0.002
c

      if (x.le.-0.69) then
         cfcn6 = 2.60
      else if (-0.69.lt.x .and. x.le.-0.25) then
         cfcn6 = 0.39*x + 2.87
      else if (-0.25.lt.x .and. x.le.-0.13) then
         cfcn6 = 2.77
      else if (-0.13.lt.x .and. x.le.1.) then
         cfcn6 = -0.50*x + 2.71
      else
         cfcn6 = -0.50*x + 2.71
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn1(x)
c
c     Corresponds to RATIO = 0.10
c

      if (x.le.-0.44) then
         ifcn1 = 0.70
      else if (-0.44.lt.x .and. x.le.0.43) then
         ifcn1 = 0.26*x + 0.81
      else if (0.43.lt.x .and. x.le.1.) then
         ifcn1 = 0.93
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn1 = 0.93
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn2(x)
c
c     Corresponds to RATIO = 0.05
c

      if (x.le.-0.56) then
         ifcn2 = 0.85
      else if (-0.56.lt.x .and. x.le.0.31) then
         ifcn2 = 0.26*x + 1.00
      else if (0.31.lt.x .and. x.le.1.) then
         ifcn2 = -0.12*x + 1.12
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn2 = -0.12*x + 1.12
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn3(x)
c
c     Corresponds to RATIO = 0.02
c

      if (x.le.-0.66) then
         ifcn3 = 0.95
      else if (-0.66.lt.x .and. x.le.0.32) then
         ifcn3 = 0.36*x + 1.19
      else if (0.32.lt.x .and. x.le.1.) then
         ifcn3 = -0.26*x + 1.38
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn3 = -0.26*x + 1.38
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn4(x)
c
c     Corresponds to RATIO = 0.01
c

      if (x.le.-0.71) then
         ifcn4 = 1.15
      else if (-0.71.lt.x .and. x.le.0.37) then
         ifcn4 = 0.34*x + 1.39
      else if (0.37.lt.x .and. x.le.1.) then
         ifcn4 = -0.38*x + 1.66
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn4 = -0.38*x + 1.66
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn5(x)
c
c     Corresponds to RATIO = 0.005
c

      if (x.le.-0.52) then
         ifcn5 = 1.48
      else if (-0.52.lt.x .and. x.le.0.24) then
         ifcn5 = 0.26*x + 1.62
      else if (0.24.lt.x .and. x.le.1.) then
         ifcn5 = -0.30*x + 1.75
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn5 = -0.30*x + 1.75
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn6(x)
c
c     Corresponds to RATIO = 0.002
c

      if (x.le.0.27) then
         ifcn6 = 1.83
      else if (0.27.lt.x .and. x.le.1.) then
         ifcn6 = -0.32*x + 1.92
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn6 = -0.32*x + 1.92
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end
c ---------------------------------------------------------------------
      real function ifcn7(x)
c
c     Corresponds to RATIO = 0.001
c

      if (x.le.-0.10) then
         ifcn7 = 2.075
      else if (-0.10.lt.x .and. x.le.1.) then
         ifcn7 = -0.27*x + 2.05
      else
         if (x .gt. 10) then
            x = 10
         endif
         ifcn7 = -0.27*x + 2.05
C         write(*,*)'X Value, ',x,' is out of range !!'
      endif
      return
      end

c-----------------------------------------------------------------------
      subroutine BMOUT1
c-----------------------------------------------------------------------
c
c --- TABBM     Version: 0.0            Level: 910124           MDA
c ---           D. Strimaitis, SRC
c
c PURPOSE:      Subroutine writes file of data needed to run B&M
c               and includes concentration estimates.
c
c ARGUMENTS:    none
c
c CALLING ROUTINES:     main
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

      character*12 fname
      INCLUDE 'params.inc'
      INCLUDE 'readin.inc'
      INCLUDE 'calc.inc'
      INCLUDE 'bm.inc'
      

      io=1
      fname='bmts.out'
      open(io,file=fname)

c  Write information for each of the trials in the group
	write(io,*) 'Vapor fraction after depressurization: ',fvap
	write(io,*) ' Density of depressurized contaminant: ',rhodil
	write(io,*) '               Density of ambient air: ',rhoair
      write(io,4) molfrac
      write(io,5) densei
      write(io,6) densec
      write(io,7) xinst
      write(io,8) pinst
      write(io,9) idi
      write(io,10) xconst
      write(io,11) pconst
      write(io,12) idc
      write(io,13)
      do 101 j=1,ndist
         write(io,14) xdist(j)
         write(io,15) conc(j)
101   continue
      write(io,16)
      write(io,17)
      write(io,18)
      write(io,19)
      write(io,20)
      write(io,21)
      write(io,22)
      write(io,23)
      write(io,24)
      write(io,25)
      write(io,26)
      write(io,27)
      write(io,28)


      close(io)

      return

c  Format Statements (SET UP FOR mx=50)
C1     format(' B&M INPUT DATA FOR  : ',7a10)
C2     format(' CHEMICAL RELEASED   : ',7a10,/)
C3     format(' TRIAL               : ',50a10)
4     format(' MOLE FRACTION       : ',50f10.6)
5     format(' DENSE ENOUGH? (inst): ',50a10)
6     format(' DENSE ENOUGH? (cnst): ',50a10)
7     format(' MIN DIST INST (m)   : ',50f10.0)
8     format(' COR. PARAM. (inst)  : ',50f10.2)
9     format(' 1/Di (1/m)          : ',50g10.3)
10    format(' MAX DIST CNST (m)   : ',50f10.0)
11    format(' COR. PARAM. (cnst)  : ',50f10.2)
12    format(' 1/Dc (1/m)          : ',50g10.3)
13    format(/,' ***** SECTION FOR CONCENTRATION ESTIMATES *****')
14    format(/,' RECEPTOR DIST. (m)  : ',50f10.1)
15    format(' CONCENTRATION (ppm) : ',50f10.2)
16    format(//,' ***** NOTES & DEFINITIONS *****')
17    format(/,'(a) "DENSE ENOUGH" refers to criteria listed in Sectio',
     1       'n 3.5 of the Workbook')
18    format('(b) "inst" refers to an instantaneous release (Section',
     1       ' 3.6)')
19    format('(c) "inst" refers to a continuous release (Section 3.6)')
20    format('(d) "MIN DIST INST" is the minimum distance downwind at',
     1       ' which the release')
21    format('    may be treated as instantaneous')
22    format('(e) "MAX DIST CNST" is the maximum distance downwind at',
     1       ' which the release')
23    format('    may be treated as continuous')
24    format('(f) "Di and Dc" are characteristic source dimensions',
     1       ' for an instantaneous')
25    format('    and continuous release')
26    format('    Di= Qo**1/3   (Figure 11)')
27    format('    Dc= [Qo/U]**.5   (Figure 8)')
28    format('(g) The correlation parameter is the abscissa of',
     1       ' Figures 8 and 11')

      end


c-----------------------------------------------------------------------
      subroutine BMOUT
c-----------------------------------------------------------------------
c
c --- TABBM     Version: 0.0            Level: 910124           MDA
c
c PURPOSE:      Subroutine writes output file from B&M model
c
c ARGUMENTS:    none
c
c CALLING ROUTINES:     main
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------
      character*12 fname
      INCLUDE 'params.inc'
      INCLUDE 'readin.inc'
      INCLUDE 'calc.inc'
      INCLUDE 'bm.inc'


      OPEN(2,FILE='BMMAX',STATUS='UNKNOWN')
      WRITE(2,33) conc(maxidx)
33    FORMAT(G12.4)
      CLOSE(2)

      io=1
      fname='bmts.out'
      open(io,file=fname)

      CALL GETDAT(IY,IM,ID)
      CALL GETTIM(IHR,IMN,ISEC,IHSEC)
      IY = IY - 1900
C
C        WRITE DATE, TIME, AND INPUT VALUES TO OUTPUT FILE
C
      WRITE(io,1) IM,ID,IY,IHR,IMN,ISEC
1     FORMAT(67X,2(I2.2,'-'),I2.2/67X,2(I2.2,':'),I2.2)
      IF (phase .LT. 1.) THEN
         WRITE(io,2) title,pamb,t1,tav,tbp,dur,rate,texit,cpv,latht
     &               ,mass,mw,rh,phase
      ELSE
         WRITE(io,22) title,pamb,t1,tav,tbp,dur,rate,texit
     &               ,mass,mw,rh,phase
      ENDIF
2     FORMAT(' ',1X,'*** B&M MODEL RUN ***',
     &         //,1X,79A1,//,
     &       1X,'INPUTS:',/,
     &       1X,'   AMBIENT PRESSURE (ATM)    = ',G12.4,/,
     &       1X,'   AMBIENT TEMP (K)          = ',G12.4,/,
     &       1X,'   AVERAGING TIME (MIN)      = ',G12.4,/,
     &       1X,'   BOILING PT TEMP (K)       = ',G12.4,/,
     &       1X,'   DURATION (S)              = ',G12.4,/,
     &       1X,'   EMISSION RATE (KG/S)      = ',G12.4,/,
     &       1X,'   EXIT TEMP (K)             = ',G12.4,/,
     &       1X,'   GAS HEAT CAPACITY(J/KG K) = ',G12.4,/,
     &       1X,'   LATENT HEAT (J/KG)        = ',G12.4,/,
     &       1X,'   MASS (KG)                 = ',G12.4,/,
     &       1X,'   MOL. WEIGHT (G/G-MOLE)    = ',G12.4,/,
     &       1X,'   RELATIVE HUMIDITY (%)     = ',G12.4,/,
     &       1X,'   VAPOR FRACTION            = ',G12.4,/)

22    FORMAT(' ',1X,'*** B&M MODEL RUN ***',
     &         //,1X,79A1,//,
     &       1X,'INPUTS:',/,
     &       1X,'   AMBIENT PRESSURE (ATM)    = ',G12.4,/,
     &       1X,'   AMBIENT TEMP (K)          = ',G12.4,/,
     &       1X,'   AVERAGING TIME (MIN)      = ',G12.4,/,
     &       1X,'   BOILING PT TEMP (K)       = ',G12.4,/,
     &       1X,'   DURATION (S)              = ',G12.4,/,
     &       1X,'   EMISSION RATE (KG/S)      = ',G12.4,/,
     &       1X,'   EXIT TEMP (K)             = ',G12.4,/,
     &       1X,'   MASS (KG)                 = ',G12.4,/,
     &       1X,'   MOL. WEIGHT (G/G-MOLE)    = ',G12.4,/,
     &       1X,'   RELATIVE HUMIDITY (%)     = ',G12.4,/,
     &       1X,'   VAPOR FRACTION            = ',G12.4,/)

      WRITE(io,3)
3     FORMAT(/,6X,
     &   '***************************************',/,6X,
     &   '***   SUMMARY OF B&M MODEL RESULTS  ***',/,6X,
     &   '***************************************')
      WRITE(io,4)
4     FORMAT(/4X,'MAX CONC',6X,'MAX CONC',4X,'DIST TO',4X,
     &       'WIND SPEED',
     &      /,4X,'(UG/M**3)',6X,'(PPM)',6X,'MAX (M)',6X,'(M/S)',
     &      /,4X,'---------    ---------    -------    ----------')
      WRITE(io,5) conc(maxidx),conppm(maxidx),xdist(maxidx),u(maxidx)
5     FORMAT(1X,G12.4,1X,G12.4,3X,F7.0,4X,F7.0)
      WRITE(io,6)
6     FORMAT(/1X,'***************************************************',
     &       /1X,'** REMEMBER TO INCLUDE BACKGROUND CONCENTRATIONS **',
     &       /1X,'***************************************************'/)


      WRITE(io,7)
7     FORMAT(/1X,'**********************************',/,
     &        1X,'***        B&M DISTANCES      ***',/,
     &        1X,'**********************************',/)

      write (io,8)
8     FORMAT(/3X,' DIST',8X,'CONC',12X,'CONC',8X,'WIND SPEED',
     &      /,3X,'  (M)',6X,'(UG/M**3)',9X,'(PPM)',9X,'(M/S)',
     &      /,3X,'-------    ---------       ---------     ----------')

      DO j=1, ndist
         WRITE(io,9) xdist(j),conc(j),conppm(j),u(j)
      ENDDO

9     FORMAT(1X,F7.0,3X,G12.4,4X,G12.4,4X,0pF7.0)


       WRITE(io,11) rhodil,rhoair,molfrac,xinst,xconst
11     FORMAT(///,1X,'CALCULATED VALUES:',/,
     &       1X,'   DENSITY OF DEPRESSURIZED CONTAMINANT (KG/M**3)',
     &       ' = ',G12.4,/,
     &       1X,'   DENSITY OF AMBIENT AIR (KG/M**3)              ',
     &       ' = ',G12.4,/,
     &       1X,'   MOLE FRACTION                                 ',
     &        ' = ',G12.4,/,
     &       1X,'   MIN DIST INST (M)                             ',
     &        ' = ',G12.4,/,
     &       1X,'   MAX DIST CNST (M)                             ',
     &       ' = ',G12.4)


C      write(io,*)
C      write(io,*) 'CALCULATED VALUES:'
CC      write(io,*) '  VAPOR FRACTION AFTER DEPRESSURIZATION: ',fvap
C      write(io,*) '   DENSITY OF DEPRESSURIZED CONTAMINANT: ',rhodil
C      write(io,*) '                 DENSITY OF AMBIENT AIR: ',rhoair
C
C
C      write(io,54) molfrac
CC      write(io,55) densei
CC      write(io,56) densec
C      write(io,57) xinst
Cc      write(io,58) pinst
Cc      write(io,59) idi
C      write(io,510) xconst
Cc      write(io,511) pconst
Cc      write(io,512) idc
C
      write(io,516)
C      write(io,517)
      write(io,518)
      write(io,519)
      write(io,520)
      write(io,521)
      write(io,522)
      write(io,523)
C      write(io,524)
C      write(io,525)
C      write(io,526)
C      write(io,527)
C      write(io,528)

54     format(19X,' MOLE FRACTION       : ',50f10.6)
C55     format(19X,' DENSE ENOUGH? (inst): ',50a10)
C56     format(19X,' DENSE ENOUGH? (cnst): ',50a10)
57     format(19X,' MIN DIST INST (m)   : ',50f10.0)
58     format(19X,' COR. PARAM. (inst)  : ',50f10.2)
59     format(19X,' 1/Di (1/m)          : ',50g10.3)
510    format(19X,' MAX DIST CNST (m)   : ',50f10.0)
511    format(19X,' COR. PARAM. (cnst)  : ',50f10.2)
512    format(19X,' 1/Dc (1/m)          : ',50g10.3)

516    format(//,' ***** NOTES & DEFINITIONS *****')
517    format(/,'(a) "DENSE ENOUGH" refers to criteria listed in Sectio',
     1       'n 3.5 of the Workbook')
518    format('(a) "inst" refers to an instantaneous release (Section',
     1       ' 3.6 of B-M Workbook)')
519    format('(b) "cnst" refers to a continuous release (Section',
     1       ' 3.6 of B-M Workbook)')
520    format('(c) "MIN DIST INST" is the minimum distance downwind at',
     1       ' which the release')
521    format('    may be treated as instantaneous')
522    format('(d) "MAX DIST CNST" is the maximum distance downwind at',
     1       ' which the release')
523    format('    may be treated as continuous')
524    format('(e) "Di and Dc" are characteristic source dimensions',
     1       ' for an instantaneous')
525    format('    and continuous release')
526    format('    Di= Qo**1/3   (Figure 11)')
527    format('    Dc= [Qo/U]**.5   (Figure 8)')
528    format('(f) The correlation parameter is the abscissa of',
     1       ' Figures 8 and 11')

      WRITE(io,10)
10    FORMAT(/1X,'**********************************',/,
     &        1X,'***     END OF B&M OUTPUT      ***',/,
     &        1X,'**********************************',/)

      RETURN
      END
