c-----------------------------------------------------------------------
      subroutine depcor(vdi,vsi,zdi,zri,xri,xvi,hi,hmixi,ui,
     &                  rurali,urbani,ksti,sgzi,sgz0i,debugi,iouniti,
     &                  qcor,pcor,pxrzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           DEPCOR
c               D. Strimaitis, SRC
c
c PURPOSE:     DEPCOR returns correction factors for the emission rate
c              and the vertical distribution of plume material to
c              account for deposition between the source and the current
c              receptor.
c
c ARGUMENTS:
c    PASSED:  vdi,vsi   total deposition and gravitational settling  [r]
c                       velocities (m/s)
c             zdi       height for evaluating deposition (m)         [r]
c             zri       receptor height (m)                          [r]
c             xri       receptor distance (m)                        [r]
c             xvi       virtual source distance (m)                  [r]
c             hi        plume height (m)                             [r]
c             hmixi     mixing height (m)                            [r]
c             ui        wind speed (m/s)                             [r]
c             rurali    logical for rural dispersion curves          [l]
c             urbani    logical for urban dispersion curves          [l]
c             ksti      stability class indicator                    [i]
c             sgzi      sigma-z at current receptor (m)              [r]
c             sgz0i     initial sigma-z (e.g. for BID) (m)           [r]
c             debugi    logical controlling DEBUG output             [l]
c             iouniti   unit number for DEBUG output                 [i]
c  RETURNED:  qcor      ratio of depleted emission rate to original  [r]
c             pcorzr    profile correction factor at receptor height [r]
c             pcorzd    profile correction factor at deposition ht   [r]
c
c CALLING ROUTINES:   PCALC, VCALC, ACALC
c
c EXTERNAL ROUTINES:  RESIST, PROFD, PROFD1, PROFD2, DEPLETE
c-----------------------------------------------------------------------
c  NOTE:  all inputs ending with "i" are passed to subsequent routines
c         through common /DEPVAR/.

      include 'DEPVAR.INC'
      logical rurali,urbani,debugi
c  Arrays RUR and URB contain coefficients derived from Briggs sigma-z
c  coefficients, for use in Horst's resistance and profile functions.
      real rur(3,6),urb(3,6)
      data rur/3.989,0.,0.,
     2         6.649,0.,0.,
     3         9.974,0.03125,0.,
     4         13.298,0.4167,0.,
     5         26.596,0.6667,0.005,
     6         49.868,2.344,0.03296/
      data urb/3.325,0.,0.,
     2         3.325,0.,0.,
     3         3.989,0.,0.,
     4         5.699,0.01531,0.,
     5         9.974,0.2344,0.,
     6         9.974,0.2344,0./

c  Initialize deposition factors to 1, and return if edge of plume is
c  well above the ground at the receptor (h > 5 sigz)
      pxrzd=1.0
      pcor=1.0
      qcor=1.0
      if(hi .GT. 5.*sgzi) return

c  Set constants
      rtpiby2=1.2533141
      rt2=1.4142136
      rtpi=1.7724539

c  Assign input variables to working variables
      vd=vdi
      vs=vsi
      zd=zdi
      zr=zri
      xr=xri
      xv=xvi
      h=hi
      hmix=hmixi
      onebyu=1./ui
      rural=rurali
      urban=urbani
      kst=ksti
      sgz=sgzi
      sgz0=sgz0i
      debug=debugi
      iounit=iouniti

c  Obtain coefficients for resistance and profile functions
c             [a,b,c]p coefficients for profile functions
c             [a,b,c]r coefficients for resistance functions
      if(rural) then
         ar=onebyu*rur(1,kst)
         br=onebyu*rur(2,kst)
         cr=onebyu*rur(3,kst)
         ap=ar
         bp=br/rtpiby2
         cp=cr*rtpiby2
      else
         ar=onebyu*urb(1,kst)
         br=onebyu*urb(2,kst)
         cr=onebyu*urb(3,kst)
         ap=ar
         bp=br/rtpiby2
      endif

c  Flush other variables in DEPVAR common
      igrav=0

c  Obtain profile factor at height zd for current receptor (x=xr).
c  First, check importance of gravitational settling velocity by
c  computing vs*RESIST at the minimum of z=hmix or z=3*sgz.
c  Note: profile function for URBAN class A & B is of a different
c  form than all other classes, which is contained in PROFD2.
      zcheck=3.*sgz
      if(hmix .GT. h) zcheck=AMIN1(hmix,zcheck)
      if(vs*RESIST(zcheck) .GT. 0.1) then
         igrav=1
c  --  gravitational settling is "large", use numerical integration.
         call PROFD(pxrzd)
c     elseif(urban .AND. kst .LT. 3) then
         igrav=0
c  --  gravitational settling is "small", use analytic function as
c  --  approximation for URBAN class A & B.
         call PROFD2(pxrzd)
      else
         igrav=0
c  --  gravitational settling is "small", use analytic function for all
c  --  other classes.
         call PROFD1(pxrzd)
      endif

c  Now compute factor at receptor height zr.
      if(zr .LE. zd) then
         pcor=pxrzd
      elseif(igrav .EQ. 0) then
         pcor=pxrzd*(1.+(vd-vs)*RESIST(zr))
      else
         pcor=pxrzd*(1.+((vd-vs)/vs)*(1.-EXP(-vs*RESIST(zr))))
      endif

c  Compute ratio of depleted source strength to initial source strength.
      call DEPLETE(qcor)

      if(debug) then
         write(iounit,*) '  DEPCOR Module:'
         write(iounit,*) '     igrav,pxrzd = ',igrav,pxrzd
         write(iounit,*) '       pcor,qcor = ',pcor,qcor
      endif

      return
      end
c-----------------------------------------------------------------------
      function resist(z)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           RESIST
c               D. Strimaitis, SRC
c
c PURPOSE:     Function RESIST provides the resistance factor for a
c              particular height above the surface (z), relative to a
c              reference height (zd). Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:  z         height above the surface                     [r]
c
c  RETURNED:  resist    resistance factor (via /DEPVAR/)             [r]
c
c CALLING ROUTINES:   DEPCOR, FINT
c
c EXTERNAL ROUTINES:  ROOT
c-----------------------------------------------------------------------


c  AR, BR, and CR are the coefficients of the 3 F(z) forms given for
c  the various forms of the Briggs representation of sigma-z (rural &
c  urban)

c  Common DEPVAR contains AR, BR, CR, and zd
      include 'DEPVAR.INC'

      if(z .GT. zd) then
c  Special Case:  URBAN/Stability Class=A,B
c  Resistance function requires the root of a implicit expression.
c  Because sigma-z functions are the same for URBAN/A,B the equation to
c  solve for x(z) is ax(1+bx)^.5=z*(pi/2)^.5, where a=.24, b=.001
         if(urban .AND. kst .LT. 3) then
c --       cz=SQRT(pi/2) * z/a = 5.222142 * z, where a=.24
           cz=5.222142*z
           call ROOT(cz,xz)
           argz=cz/xz
c  Approximate functional dependence on zd using binomial expansion
c  --      c=2*b*SQRT(pi/2)/a = 0.0104443
c  --      8./(c*zd)=765.96804/zd
           argzd=765.96804/zd
           resist=AR*ALOG((argz-1.)*(argzd+1.)/(argz+1.))
         else
           resist=AR*ALOG(z/zd) + BR*(z-zd) + CR*(z*z-zd*zd)
         endif
      else
         resist=0.0
      endif

      return
      end
c-----------------------------------------------------------------------
      subroutine root(c,x)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           ROOT
c               D. Strimaitis, SRC
c
c PURPOSE:     Program solves an expression of the form:
c                  x*(1+b*x)**.5=c
c              using a simple iteration on:
c                  x=c/(1+b*x0)**.5
c
c              ! WARNING !     This is a special solver for current
c                              application.....it may not converge for
c                              other applications.
c
c ARGUMENTS:
c    PASSED:  c         constant for RHS of equation to solve        [r]
c
c  RETURNED:  x         root of equation                             [r]
c
c CALLING ROUTINES:   RESIST
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      data b/.001/,onebyb/1000./,twob/.002/
      data eby2/.005/,oneby3/.3333333/
c  "e" is a fractional error criterion for convergence, so eby2=e/2

c  First guess
      twobc=twob*c
      if(twobc .LT. 6.) then
         x0=(SQRT(1.+twobc)-1.)*onebyb
      else
         x0=(c*c*onebyb)**oneby3
      endif
10    x=c/SQRT(1.+b*x0)
      errby2=ABS(x-x0)/(x+x0)
      if(errby2 .LE. eby2) goto 100
      x0=x
      goto 10

100   continue
      return
      end
c-----------------------------------------------------------------------
      subroutine profd(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD provides the base profile factor at
c              height zd for the given deposition and settling velocities,
c              and sigma-z.  Here, the settling velocity and diffusion
c              resistance are not "small" so that a numerical integration
c              is used to obtain Ip:
c                    P(x,zd) = 1. / (1.+ Ip*(vd-vs)/vs) .... "pxzd"
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  QATR, FINT
c-----------------------------------------------------------------------

c  Set up call to integration routine QATR(xl,xu,eps,ndim,fct,y,ier,num,aux)
c  Declare parameter to fix the size of the aux array
      parameter(ndim=16)
      real aux(ndim)
      external FINT
      INTEGER*2 NUM
      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LE. 2.*zd) return

c  Evaluate integral Ip:
c  Upper limit of integral reset from infinity to MIN(5*sigma-z,hmix)
      eps=.10
      top=AMIN1(5.*sgz,hmix)
      call QATR(zd,top,eps,ndim,FINT,value,ier,num,aux)
C      if(ier .EQ. 2) then
C         write(*,*) 'WARNING from PROFD -  integration failed to'
C         write(*,*) 'converge to fractional error of ',eps
C      endif
      pxzd=1./(1.+value*(vd-vs)/vs)

      return
      end
c-----------------------------------------------------------------------
      function fint(z)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           FINT
c               D. Strimaitis, SRC
c
c PURPOSE:     Function is the integrand of integral over height to
c              calculate the profile parameter P(x,zd).  The resistance
c              value is returned from the function RESIST.  Common
c              /DEPVAR/ is used to pass data that are constant during
c              the integration, so QATR (the integrator) only needs to
c              pass values of height (z).
c               -VCOUP calculates the vertical coupling factor:
c                       (1/(sgz*SQRT(2pi)))*EXP...
c                 (this includes multiple reflections!)
c
c ARGUMENTS:
c    PASSED:  z         height above surface                         [r]
c
c  RETURNED:  fint      value of integrand                           [r]
c
c CALLING ROUTINES:   QATR
c
c EXTERNAL ROUTINES:  RESIST, VCOUP
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

      arg=vs*RESIST(z)
      fint=(1.-EXP(-arg))*VCOUP(z,0.,sgz,hmix)

      return
      end
c-----------------------------------------------------------------------
      subroutine profd1(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD1
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD1 provides the base profile factor at
c              height zd for the given deposition and settling
c              velocities, and sigma-z.  Here, the settling velocity
c              and diffusion resistance are "small" so that the analytic
c              results are used.
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------


c  AP, BP, and CP are the coefficients of the 3 F(z) forms given for
c  the various forms of the Briggs representation of sigma-z (rural &
c  urban)

c  Approximate the results for a mixing lid by "clamping" the
c  calculation at the limit of a well-mixed plume in the vertical.
c       .7071=SQRT(1/2)
c       .6267=SQRT(pi/8)
c       .5157=SQRT( [SQRT(2/pi)]/3 )
c       .8932=SQRT( SQRT(2/pi) )

      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LE. 2.*zd) return

      if(hmix .GT. h) then
         za=AMIN1(sgz,.7071*hmix)
         if(BP .GT. 0.) zb=AMIN1(sgz,.6267*hmix)
         if(CP .GT. 0.) zc=AMIN1(sgz,.5157*hmix)
      else
         za=sgz
         zb=sgz
         zc=sgz
      endif

      pxzd=1./(1.+(vd-vs)*(AP*(ALOG(rt2*za/zd) -1.) + BP*(zb-rtpiby2*zd)
     1         + CP*(zc*zc-zd*zd)))

      return
      end
c-----------------------------------------------------------------------
      subroutine profd2(pxzd)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           PROFD2
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine PROFD2 provides the base profile factor at
c              height zd for the given deposition and settling
c              velocities, and sigma-z.  Here, the settling velocity
c              and diffusion resistance are "small" so that the analytic
c              results are used.
c              --------------- URBAN Class A & B !!! ---------------------
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  pxzd      profile factor at height zd                  [r]
c
c CALLING ROUTINES:   DEPCOR, F2INT
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

c  Approximate the results for a mixing lid by switching to the
c  calculation for the limit of a well-mixed plume in the vertical
c  when sigmaz = .7071 H, where  .7071=SQRT(1/2).

      include 'DEPVAR.INC'

c  Return a value of 1.0 for pxzd if the sigma-z is less than 2*zd,
c  since the integrals assume that zd is less than the plume spread.
      pxzd=1.0
      if(sgz .LE. 2.*zd) return

c  AP is the coefficient (SQRT(2/pi) / aU)
c     ck = 2*b*SQRT(pi/2)/a = 0.0104443
      ck=0.0104443

      za=sgz
      if(hmix .GT. h) za=AMIN1(sgz,.7071*hmix)
      sgz1=za
      if(za .LT. 300.) then
         sgz1=za*(1.-za*.0006)**2
      else
         sgz1=za*(1.-300.*.0006)**2
      endif
      sgz2=sgz1
      if(sgz1 .GT. 1000.) sgz2=SQRT(1000.*sgz1)
      approx=-1.+ALOG(rt2*sgz1/zd)+ALOG(1.+ck*zd/8.)-
     &       ck*rt2*sgz2/(8.*rtpi)
      pxzd=1./(1.+(vd-vs)*AP*approx)

      return
      end
c-----------------------------------------------------------------------
      subroutine deplete(qcor)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           DEPLETE
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine DEPLETE provides the value of the integral of
c              the product of the vertical distribution function and the
c              profile correction factor over the travel of the plume
c              from the source to the receptor.  Because the integrating
c              subroutine may be used within this integration, the
c              routine QATR has been duplicated as QATR2.
c              Based on Horst (1983).
c
c ARGUMENTS:
c    PASSED:            (see /DEPVAR/)
c
c  RETURNED:  qcor      ratio of depleted emission rate to original  [r]
c
c CALLING ROUTINES:   DEPCOR
c
c EXTERNAL ROUTINES:  QATR2, F2INT
c-----------------------------------------------------------------------

c  Set up call to QATR2(xl,xu,eps,ndim2,fct,y,ier,num,aux2)
c  Declare parameter to fix the size of the aux2 array
      parameter(ndim2=16)
      real aux2(ndim2)
      external F2INT
      INTEGER*2 NUM
      include 'DEPVAR.INC'

      IOUNIT = 13
      iout = 9

c  Evaluate integral:
      eps=.05
c  Do not let integral try to evaluate sigma-z at x=0! -- start at 1m
      call QATR2(1.,xr,eps,ndim2,F2INT,value,ier,num,aux2)
C      if(ier .EQ. 2) then
C         write(*,*) 'WARNING from DEPLETE -  integration failed to'
C         write(*,*) 'converge to fractional error of ',eps
C         write(iout,*) 'WARNING from DEPLETE -  integration failed to'
C         write(iout,*) 'converge to fractional error of ',eps
C      endif
      qcor=EXP(-vd*value)

      if(debug) then
         write(iounit,*) '  DEPLETE: eps, QATR2 iterations = ',eps,num
      endif

      return
      end
c-----------------------------------------------------------------------
      function f2int(x)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930215           F2INT
c               D. Strimaitis, SRC
c
c PURPOSE:     Function is the integrand of integral over the travel
c              distance to obtain the fraction of material removed from
c              the plume. Common /DEPVAR/ is used to pass data that are
c              constant during the integration, so QATR (the integrator)
c              only needs to pass values of distance.
c
c ARGUMENTS:
c    PASSED:  z         height above surface                         [r]
c
c  RETURNED:  fint      value of integrand                           [r]
c
c CALLING ROUTINES:   QATR2
c
c EXTERNAL ROUTINES:  SIGZD, PROFD, PROFD1, PROD2, VCOUP
c-----------------------------------------------------------------------
      include 'DEPVAR.INC'

c  Account for any virtual source distance
      xxv=x+xv

c  Recalculate sigma-z; note that sgz had been for the receptor
c  location -- once the integration in distance begins, this value is
c  not needed, so we replace it with that for the current "x".
      call SIGZD(xxv,sgz)

c  Account for any initial sigma-z (e.g BID)
      sgz=SQRT(sgz**2+sgz0**2)

c  Adjust plume height for gravitational settling
      hh=AMAX1(0.,h-vs*x*onebyu)

c  -VCOUP calculates the vertical coupling factor:
c    (1/(sgz*SQRT(2pi)))*EXP...  (this includes multiple reflections!)
c  -PROFD1 or PROFD2 calculates the profile correction factor if
c  gravitational settling is weak (analytic representations are used);
c  -PROFD calculates the profile correction factor if gravitational
c  settling is strong (numerical integration is used).
      f2int=onebyu*VCOUP(zd,hh,sgz,hmix)
      if(f2int .GT. 0.0) then
         if(igrav .EQ. 0) then
            if(urban .AND. kst .LT. 3) then
               call PROFD2(pxzd)
            else
               call PROFD1(pxzd)
            endif
         else
            call PROFD(pxzd)
         endif
         f2int=f2int*pxzd
      endif

      return
      end
c----------------------------------------------------------------------
      FUNCTION VCOUP(ZR,ZS,SZ,HLID)
c----------------------------------------------------------------------
c
c --- ISCST2     Version: 1.0       Level: 930215                 VCOUP
c                R. Yamartino, SRC
c
c     Adapted from --
c --- CALPUFF    Version: 1.0       Level: 900228                 VCOUP
c                R. Yamartino, SRC
c
c --- PURPOSE:  Computes the vertical coupling coefficient for a source
c               at height ZS, to a receptor at height ZR given a plume
c               with sigma z of SZ and including reflections from the
c               ground and lid at height HLID.
c
c
c --- INPUTS:
c
c                ZR - real    - Z-coordinate of receptor (m)
c                ZS - real    - Z-coordinate of source (m)
c                SZ - real    - Z-sigma at receptor (m)
c              HLID - real    - Mixing depth at receptor (m)
c              Note that these input values must have same units.
c
c
c --- OUTPUTS:
c
c             VCOUP - real    - Vertical coupling coefficient (1/m)
c
c --- VCOUP called by:  FINT, F2INT
c --- VCOUP calls:      none
c----------------------------------------------------------------------
c
c --- All heights have same units.      1/26/89
c
      data small/1.0e-10/,srttpi/2.5066283/,pi/3.1415926/
c
      vcoup = 0.0
c
c !!! Guard against hlid LE plume height:  reset to 10 sigma-z
      if(hlid .LE. zs) hlid=zs+10.*sz
          if((sz/hlid).gt.0.63) go to 15
c !!! Also, make sure that receptors above a real lid have zero conc.
      if(zr .GT. hlid) return
c
c --- Sum the reflection terms
      sz1 = sz + small
      sz2 = sz*sz + small
      x = -0.5*(zr-zs)**2/sz2
      if(x.lt.-20.0) go to 20
      expz = exp(x)
      x = -0.5*(zr+zs)**2/sz2
      if(x.gt.-20.0) expz = expz + exp(x)
c
          do 10 j = -1 , +1 , 2
            zrefl = 2.0*float(j)*hlid
            x = -0.5*(zr+zs+zrefl)**2/sz2
            if(x.gt.-20.0) expz = expz + exp(x)
            x = -0.5*(zr-zs+zrefl)**2/sz2
            if(x.gt.-20.0) expz = expz + exp(x)
 10       continue
c
      vcoup = expz/(srttpi*sz1)
      go to 20
c
c --- Near uniform mixing using approximation of R. Yamartino
c     (JAPCA 27, 5, MAY 1977)
 15   szsb = -0.5*(pi*sz/hlid)**2
      expz = 1.0
      if(szsb.gt.-20.0) then
         beta = exp(szsb)
         beta2 = beta*beta
         expz = (1.0-beta2)*(1.0+beta2+2.0*beta*cos(pi*zs/hlid)*
     *                                       cos(pi*zr/hlid))
      endif
      vcoup = expz/hlid
c
 20   return
      end
c-----------------------------------------------------------------------
      subroutine qatr(xl,xu,eps,ndim,fct,y,ier,i,aux)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           QATR
c
c PURPOSE:      Integration routine adapted from the IBM SSP program
c               DQATR.  Modified for single precision.
c
c ARGUMENTS:
c    PASSED:    xl,xu   lower and upper limits of integration        [r]
c               eps     fractional error used to define convergence  [r]
c               ndim    dimension of array aux (parameter)           [p]
c               fct     external function (integrand)
c               aux     working array, passed to allow variable dim. [r]
c  RETURNED:    y       value of integral                            [r]
c               ier     status flag at termination                   [i]
c               i       number of subdivision steps                  [i]
c
c CALLING ROUTINES:     PROFD
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

c  NOTES: status flags denote the following --
c               ier=0   value of integral converged to within eps
c               ier=1   value of integral is diverging
c               ier=2   value of integral did not converge to within
c                       eps before ndim limit was reached

c  NDIM Note:  The aux(ndim) array keeps track of the average value of
c              the integrand for each of the steps in subdividing the
c              interval.  For example, when i=4 in the "do 7 i=2,ndim"
c              loop, aux(4) contains the mean value as obtained from
c              the trapezoidal rule, while aux(1 through 3) contain
c              a set of current Romberg extrapolations.  At each new
c              value of i, the interval is subdivided again, and the
c              integrand is evaluated at jj=2**(i-2) new points.
c              Therefore, at i=5, there will be jj=8 new points added
c              to the 9 points already used in the interval.  When i=17
c              there will be jj=32,768 new points added to the 32,769
c              already used.  This is the maximum number of new points
c              that are allowed as jj is an INTEGER*2 variable, with
c              a maximum value of 2**15.  Therefore, i should not exceed
c              17, and probably should be no larger than 16.  This means
c              that NDIM should be set at 16.  Larger values of NDIM
c              could be accepted if the INTEGER*2 variables were changed
c              to INTEGER*4, but for most applications, 30000 to 60000
c              points ought to be sufficient for evaluating an integral.

      EXTERNAL fct
      dimension aux(ndim)
      integer*2 i,ii,ji,j,jj
      half=0.5

c  Preparations for Romberg loop
      aux(1)=half*(fct(xl)+fct(xu))
      h=xu-xl
      y=h*aux(1)
      if(ndim .LE. 1) then
         ier=2
         return
      elseif(h .EQ. 0.) then
         ier=0
         return
      endif

      hh=h
      delt2=0.
      p=1.
      jj=1

c  Initialize flag for integer*2 limit: jj cannot exceed 32,000
c  This limit should not be reached if NDIM .LE. 16
      lstop=0

      do 7 i=2,ndim
         y=aux(1)
         delt1=delt2
         hd=hh
         hh=half*hh
         p=half*p
         x=xl+hh
         sm=0.

c  Integer*2 limit: jj cannot exceed 32,000
         if(lstop .EQ. 1) then
            write(6,1010)
1010        format(2X,'ERROR FROM QATR - VARIABLE jj EXCEEDED 32,000')
            stop
         endif
         if(jj .GT. 16000) lstop=1

         do 3 j=1,jj
            sm=sm+fct(x)
            x=x+hd
3        continue

c  A new approximation to the integral is computed by means
c  of the trapezoidal rule
         aux(i)=half*aux(i-1)+p*sm

c  Start of Rombergs extrapolation method

         q=1.
         ji=i-1
         do 4 j=1,ji
            ii=i-j
            q=q+q
            q=q+q
            aux(ii)=aux(ii+1)+(aux(ii+1)-aux(ii))/(q-1.)
4        continue

c  End of Romberg step

         delt2=ABS(y-aux(1))
         if(i .GE. 3) then
c  Modification for cases in which function = 0 over interval
            if(y .EQ. 0.) then
               ier=0
               return
            elseif(delt2/y .LE. eps) then
               ier=0
               y=h*aux(1)
               return
c           elseif(delt2 .GE. delt1)then
c              ier=1
c              y=h*y
c              return
            endif
         endif
7     jj=jj+jj
      ier=2
      y=h*aux(1)

      return
      end
c-----------------------------------------------------------------------
      subroutine qatr2(xl,xu,eps,ndim,fct,y,ier,i,aux)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 930930           QATR2
c
c PURPOSE:      Integration routine adapted from the IBM SSP program
c               DQATR.  Modified for single precision.  This is a COPY
c               of QATR for use in double integrations.
c
c ARGUMENTS:
c    PASSED:    xl,xu   lower and upper limits of integration        [r]
c               eps     fractional error used to define convergence  [r]
c               ndim    dimension of array aux (parameter)           [p]
c               fct     external function (integrand)
c               aux     working array, passed to allow variable dim. [r]
c  RETURNED:    y       value of integral                            [r]
c               ier     status flag at terminatio                    [i]
c               i       number of subdivision steps                  [i]
c
c CALLING ROUTINES:     DEPLETE
c
c EXTERNAL ROUTINES:    none
c-----------------------------------------------------------------------

c  NOTES: status flags denote the following --
c               ier=0   value of integral converged to within eps
c               ier=1   value of integral is diverging
c               ier=2   value of integral did not converge to within
c                       eps before ndim limit was reached

c  NDIM Note:  The aux(ndim) array keeps track of the average value of
c              the integrand for each of the steps in subdividing the
c              interval.  For example, when i=4 in the "do 7 i=2,ndim"
c              loop, aux(4) contains the mean value as obtained from
c              the trapezoidal rule, while aux(1 through 3) contain
c              a set of current Romberg extrapolations.  At each new
c              value of i, the interval is subdivided again, and the
c              integrand is evaluated at jj=2**(i-2) new points.
c              Therefore, at i=5, there will be jj=8 new points added
c              to the 9 points already used in the interval.  When i=17
c              there will be jj=32,768 new points added to the 32,769
c              already used.  This is the maximum number of new points
c              that are allowed as jj is an INTEGER*2 variable, with
c              a maximum value of 2**15.  Therefore, i should not exceed
c              17, and probably should be no larger than 16.  This means
c              that NDIM should be set at 16.  Larger values of NDIM
c              could be accepted if the INTEGER*2 variables were changed
c              to INTEGER*4, but for most applications, 30000 to 60000
c              points ought to be sufficient for evaluating an integral.

      EXTERNAL fct
      dimension aux(ndim)
      integer*2 i,ii,ji,j,jj
      half=0.5

c  Preparations for Romberg loop
      aux(1)=half*(fct(xl)+fct(xu))
      h=xu-xl
      y=h*aux(1)
      if(ndim .LE. 1) then
         ier=2
         return
      elseif(h .EQ. 0.) then
         ier=0
         return
      endif

      hh=h
      delt2=0.
      p=1.
      jj=1

c  Initialize flag for integer*2 limit: jj cannot exceed 32,000
c  This limit should not be reached if NDIM .LE. 16
      lstop=0

      do 7 i=2,ndim
         y=aux(1)
         delt1=delt2
         hd=hh
         hh=half*hh
         p=half*p
         x=xl+hh
         sm=0.

c  Integer*2 limit: jj cannot exceed 32,000
         if(lstop .EQ. 1) then
            write(6,1010)
1010        format(2X,'ERROR FROM QATR2- VARIABLE jj EXCEEDED 32,000')
            stop
         endif
         if(jj .GT. 16000) lstop=1

         do 3 j=1,jj
            sm=sm+fct(x)
            x=x+hd
3        continue

c  A new approximation to the integral is computed by means
c  of the trapezoidal rule
         aux(i)=half*aux(i-1)+p*sm

c  Start of Rombergs extrapolation method

         q=1.
         ji=i-1
         do 4 j=1,ji
            ii=i-j
            q=q+q
            q=q+q
            aux(ii)=aux(ii+1)+(aux(ii+1)-aux(ii))/(q-1.)
4        continue

c  End of Romberg step

         delt2=ABS(y-aux(1))
         if(i .GE. 3) then
c  Modification for cases in which function = 0 over interval
            if(y .EQ. 0.) then
               ier=0
               return
            elseif(delt2/y .LE. eps) then
               ier=0
               y=h*aux(1)
               return
c           elseif(delt2 .GE. delt1)then
c              ier=1
c              y=h*y
c              return
            endif
         endif
7     jj=jj+jj
      ier=2
      y=h*aux(1)

      return
      end
