       MODULE SEAS_STRAT_O3_MIN
       
          IMPLICIT NONE
! Data and routine to get the monthly fraction of ozone column density above the top pressure level
! defined in the GRID_CONF module. The values vary across the model domain based on the latitude of the 
! modeled column.
!
! Data to estimate the fraction use a seasonal climatology for the meridian
! cross-section of ozone concentrations. McPeters and Labow (2012) assembled the climatology based on 
! Aura Microwave Limb Soundings (2004-2010) and balloon sondes (1988-2010). Their data was lasted 
! accessed July 11, 2016 under the URL, ftp://toms.gsfc.nasa.gov/pub/ML_climatology/
!
! References: 
!  1) McPeters, R. D., and G. J. Labow (2012), Climatology 2011: An MLS and sonde derived ozone 
!     climatology for satellite retrieval algorithms, J. Geophys. Res., 117, D10303, 
!     doi:10.1029/2011JD017006.
!
! Revision History: B. Hutzell: 2016, July 11: created

          REAL, PUBLIC :: MONTH_STRAT_03_FRAC ! mimimun fraction O3 fraction in stratos.
            
          PUBLIC       :: SEASONAL_STRAT_O3
          
          PRIVATE
          
          INTEGER,  PARAMETER :: NLAT_PROFILE  = 18
          INTEGER,  PARAMETER :: NLEV_PROFILE  = 65
          INTEGER,  PARAMETER :: NTIME_PROFILE = 12
          
          REAL :: LAT_PROFILE( NLAT_PROFILE ) ! central latitude for range of profile,  deg.
          
          DATA LAT_PROFILE  / -85.00, -75.00, -65.00, -55.00, -45.00, -35.00, -25.00, -15.00, 
     &                         -5.00,   5.00,  15.00,  25.00,  35.00,  45.00,  55.00,  65.00, 
     &                         75.00,  85.00 /
          
          REAL :: LEV_PROFILE( NLEV_PROFILE ) ! pressure level at value of fraction,  mbar     
          DATA LEV_PROFILE /  1000.00, 865.96, 749.89, 649.38, 562.34, 486.97, 421.70, 365.17, 
     &                         316.23, 273.84, 237.14, 205.35, 177.83, 153.99, 133.35, 115.48, 
     &                         100.00,  86.60,  74.99,  64.94,  56.23,  48.70,  42.17,  36.52, 
     &                          31.62,  27.38,  23.71,  20.54,  17.78,  15.40,  13.34,  11.55,
     &                          10.00,   8.66,   7.50,   6.49,   5.62,   4.87,   4.22,   3.65, 
     &                           3.16,   2.74,   2.37,   2.05,   1.78,   1.54,   1.33,   1.15,
     &                           1.00,   0.87,   0.75,   0.65,   0.56,   0.49,   0.42,   0.37, 
     &                           0.32,   0.27,   0.24,   0.21,   0.18,   0.15,   0.13,   0.12,
     &                           0.10 /
          
          
          REAL :: TOP_PROFILE( NLEV_PROFILE ) ! pressure level immediately above value,  mbar     
          DATA TOP_PROFILE /  865.96, 749.89, 649.38, 562.34, 486.97, 421.70, 365.17, 316.23, 
     &                        273.84, 237.14, 205.35, 177.83, 153.99, 133.35, 115.48, 100.00, 
     &                         86.60,  74.99,  64.94,  56.23,  48.70,  42.17,  36.52,  31.62, 
     &                         27.38,  23.71,  20.54,  17.78,  15.40,  13.34,  11.55,  10.00, 
     &                          8.66,   7.50,   6.49,   5.62,   4.87,   4.22,   3.65,   3.16, 
     &                          2.74,   2.37,   2.05,   1.78,   1.54,   1.33,   1.15,   1.00, 
     &                          0.87,   0.75,   0.65,   0.56,   0.49,   0.42,   0.37,   0.32, 
     &                          0.27,   0.24,   0.21,   0.18,   0.15,   0.13,   0.12,   0.10, 
     &                          0.09 /

          REAL    :: TIME_PROFILE( NTIME_PROFILE ) ! calendar date for each meridianal cross-section, fraction of year 
          DATA TIME_PROFILE / 
     &     4.2466E-02, 1.2329E-01, 2.0411E-01, 2.8767E-01, 3.7123E-01,
     &     4.5479E-01, 5.3836E-01, 6.2329E-01, 7.0685E-01, 7.9041E-01,
     &     8.7397E-01, 9.5753E-01/
          
! cumulative downward fraction of ozone column versus level and latitude
        INTEGER :: IILAT 

        REAL :: FRAC_O3_COLUMN( NLAT_PROFILE,NLEV_PROFILE,NTIME_PROFILE )
! Third array index equals  1. Data corresponds to middle of JAN.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9934, 0.9944, 0.9950, 0.9944, 0.9892, 0.9940, 0.9945, 0.9920, 
     &  0.9841, 0.9866, 0.9862, 0.9893, 0.9910, 0.9922, 0.9899, 0.9897, 0.9905/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9956, 0.9879, 0.9895, 0.9891, 0.9875, 0.9777, 0.9851, 0.9881, 0.9823, 
     &  0.9675, 0.9681, 0.9716, 0.9773, 0.9813, 0.9825, 0.9800, 0.9799, 0.9812/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9889, 0.9825, 0.9849, 0.9825, 0.9799, 0.9670, 0.9737, 0.9807, 0.9717, 
     &  0.9531, 0.9513, 0.9575, 0.9655, 0.9717, 0.9729, 0.9706, 0.9707, 0.9725/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9840, 0.9768, 0.9800, 0.9758, 0.9722, 0.9569, 0.9624, 0.9726, 0.9612, 
     &  0.9407, 0.9367, 0.9447, 0.9549, 0.9629, 0.9642, 0.9620, 0.9624, 0.9646/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9796, 0.9710, 0.9748, 0.9690, 0.9648, 0.9473, 0.9514, 0.9642, 0.9510, 
     &  0.9297, 0.9239, 0.9332, 0.9453, 0.9548, 0.9562, 0.9540, 0.9547, 0.9572/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9751, 0.9653, 0.9697, 0.9626, 0.9577, 0.9383, 0.9411, 0.9565, 0.9414, 
     &  0.9198, 0.9128, 0.9230, 0.9366, 0.9475, 0.9488, 0.9466, 0.9475, 0.9504/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9700, 0.9593, 0.9646, 0.9567, 0.9511, 0.9299, 0.9316, 0.9497, 0.9330, 
     &  0.9107, 0.9033, 0.9141, 0.9288, 0.9406, 0.9418, 0.9398, 0.9405, 0.9437/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9625, 0.9511, 0.9576, 0.9497, 0.9444, 0.9216, 0.9229, 0.9437, 0.9261, 
     &  0.9026, 0.8949, 0.9064, 0.9212, 0.9334, 0.9347, 0.9327, 0.9327, 0.9365/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9480, 0.9376, 0.9459, 0.9403, 0.9367, 0.9136, 0.9152, 0.9388, 0.9202, 
     &  0.8956, 0.8877, 0.8994, 0.9132, 0.9244, 0.9255, 0.9234, 0.9223, 0.9267/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9269, 0.9181, 0.9292, 0.9276, 0.9274, 0.9059, 0.9083, 0.9346, 0.9153, 
     &  0.8896, 0.8813, 0.8933, 0.9038, 0.9119, 0.9129, 0.9101, 0.9072, 0.9117/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9037, 0.8961, 0.9096, 0.9121, 0.9165, 0.8974, 0.9020, 0.9309, 0.9109, 
     &  0.8841, 0.8756, 0.8877, 0.8919, 0.8952, 0.8958, 0.8917, 0.8867, 0.8910/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8805, 0.8738, 0.8890, 0.8953, 0.9042, 0.8884, 0.8961, 0.9275, 0.9070, 
     &  0.8791, 0.8704, 0.8828, 0.8775, 0.8745, 0.8738, 0.8679, 0.8610, 0.8649/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8570, 0.8514, 0.8681, 0.8779, 0.8915, 0.8794, 0.8905, 0.9242, 0.9033, 
     &  0.8744, 0.8655, 0.8781, 0.8615, 0.8513, 0.8476, 0.8391, 0.8303, 0.8336/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8316, 0.8274, 0.8458, 0.8594, 0.8783, 0.8701, 0.8849, 0.9204, 0.8997, 
     &  0.8702, 0.8609, 0.8732, 0.8446, 0.8261, 0.8179, 0.8053, 0.7941, 0.7968/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8018, 0.7990, 0.8200, 0.8381, 0.8631, 0.8600, 0.8792, 0.9164, 0.8964, 
     &  0.8665, 0.8563, 0.8677, 0.8269, 0.7984, 0.7841, 0.7661, 0.7517, 0.7535/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7656, 0.7649, 0.7893, 0.8132, 0.8455, 0.8488, 0.8729, 0.9120, 0.8931, 
     &  0.8630, 0.8517, 0.8615, 0.8084, 0.7687, 0.7468, 0.7227, 0.7048, 0.7051/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7217, 0.7233, 0.7521, 0.7829, 0.8242, 0.8348, 0.8645, 0.9055, 0.8882, 
     &  0.8578, 0.8455, 0.8530, 0.7869, 0.7354, 0.7056, 0.6761, 0.6548, 0.6534/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6702, 0.6740, 0.7078, 0.7460, 0.7971, 0.8165, 0.8526, 0.8953, 0.8799, 
     &  0.8494, 0.8359, 0.8400, 0.7593, 0.6966, 0.6600, 0.6260, 0.6015, 0.5985/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6138, 0.6197, 0.6582, 0.7037, 0.7642, 0.7934, 0.8366, 0.8812, 0.8678, 
     &  0.8376, 0.8220, 0.8209, 0.7256, 0.6534, 0.6114, 0.5737, 0.5467, 0.5427/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5548, 0.5624, 0.6047, 0.6564, 0.7245, 0.7634, 0.8136, 0.8606, 0.8493, 
     &  0.8199, 0.8008, 0.7926, 0.6851, 0.6061, 0.5605, 0.5209, 0.4929, 0.4886/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4965, 0.5053, 0.5501, 0.6061, 0.6794, 0.7263, 0.7823, 0.8316, 0.8231, 
     &  0.7948, 0.7715, 0.7541, 0.6380, 0.5553, 0.5082, 0.4686, 0.4411, 0.4370/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4420, 0.4513, 0.4972, 0.5555, 0.6317, 0.6834, 0.7428, 0.7941, 0.7894, 
     &  0.7628, 0.7354, 0.7076, 0.5867, 0.5032, 0.4562, 0.4181, 0.3924, 0.3890/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3927, 0.4019, 0.4473, 0.5059, 0.5825, 0.6363, 0.6967, 0.7495, 0.7493, 
     &  0.7244, 0.6933, 0.6555, 0.5334, 0.4520, 0.4065, 0.3710, 0.3483, 0.3459/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3482, 0.3567, 0.4001, 0.4575, 0.5325, 0.5863, 0.6460, 0.6997, 0.7038, 
     &  0.6805, 0.6463, 0.5995, 0.4790, 0.4019, 0.3591, 0.3272, 0.3083, 0.3069/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3092, 0.3167, 0.3567, 0.4110, 0.4823, 0.5343, 0.5916, 0.6451, 0.6522, 
     &  0.6302, 0.5950, 0.5422, 0.4262, 0.3545, 0.3151, 0.2871, 0.2720, 0.2714/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2754, 0.2815, 0.3168, 0.3664, 0.4317, 0.4801, 0.5339, 0.5858, 0.5945, 
     &  0.5740, 0.5399, 0.4852, 0.3763, 0.3107, 0.2751, 0.2509, 0.2392, 0.2392/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2448, 0.2495, 0.2799, 0.3234, 0.3815, 0.4255, 0.4750, 0.5241, 0.5337, 
     &  0.5147, 0.4828, 0.4295, 0.3297, 0.2705, 0.2389, 0.2184, 0.2096, 0.2100/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2168, 0.2203, 0.2459, 0.2830, 0.3334, 0.3725, 0.4173, 0.4623, 0.4717, 
     &  0.4544, 0.4255, 0.3766, 0.2871, 0.2345, 0.2069, 0.1896, 0.1831, 0.1836/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1914, 0.1937, 0.2148, 0.2456, 0.2883, 0.3223, 0.3618, 0.4017, 0.4102, 
     &  0.3947, 0.3695, 0.3274, 0.2491, 0.2028, 0.1789, 0.1643, 0.1593, 0.1597/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1680, 0.1694, 0.1863, 0.2112, 0.2467, 0.2754, 0.3098, 0.3445, 0.3519, 
     &  0.3381, 0.3166, 0.2825, 0.2154, 0.1753, 0.1546, 0.1420, 0.1380, 0.1383/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1463, 0.1469, 0.1602, 0.1801, 0.2093, 0.2333, 0.2629, 0.2926, 0.2986, 
     &  0.2866, 0.2688, 0.2423, 0.1858, 0.1513, 0.1335, 0.1225, 0.1191, 0.1191/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1263, 0.1263, 0.1365, 0.1522, 0.1759, 0.1957, 0.2209, 0.2460, 0.2506, 
     &  0.2405, 0.2265, 0.2067, 0.1597, 0.1303, 0.1149, 0.1054, 0.1023, 0.1021/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1082, 0.1077, 0.1154, 0.1273, 0.1464, 0.1625, 0.1837, 0.2047, 0.2084, 
     &  0.2003, 0.1897, 0.1753, 0.1365, 0.1117, 0.0986, 0.0902, 0.0875, 0.0871/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0916, 0.0909, 0.0965, 0.1055, 0.1207, 0.1337, 0.1513, 0.1688, 0.1718, 
     &  0.1656, 0.1579, 0.1474, 0.1159, 0.0953, 0.0842, 0.0767, 0.0743, 0.0737/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0767, 0.0758, 0.0798, 0.0864, 0.0985, 0.1088, 0.1233, 0.1375, 0.1400, 
     &  0.1354, 0.1302, 0.1228, 0.0975, 0.0807, 0.0714, 0.0649, 0.0626, 0.0620/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0634, 0.0625, 0.0652, 0.0701, 0.0795, 0.0877, 0.0993, 0.1107, 0.1129, 
     &  0.1098, 0.1064, 0.1012, 0.0812, 0.0677, 0.0602, 0.0545, 0.0524, 0.0518/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0518, 0.0510, 0.0527, 0.0562, 0.0635, 0.0700, 0.0792, 0.0882, 0.0902, 
     &  0.0883, 0.0859, 0.0824, 0.0669, 0.0564, 0.0504, 0.0455, 0.0436, 0.0430/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0418, 0.0411, 0.0420, 0.0445, 0.0502, 0.0553, 0.0625, 0.0695, 0.0715, 
     &  0.0704, 0.0687, 0.0662, 0.0544, 0.0465, 0.0419, 0.0378, 0.0361, 0.0355/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0334, 0.0328, 0.0332, 0.0350, 0.0394, 0.0434, 0.0490, 0.0544, 0.0561, 
     &  0.0557, 0.0543, 0.0526, 0.0439, 0.0381, 0.0346, 0.0312, 0.0298, 0.0292/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0263, 0.0259, 0.0260, 0.0273, 0.0308, 0.0339, 0.0381, 0.0422, 0.0437, 
     &  0.0436, 0.0425, 0.0413, 0.0349, 0.0308, 0.0284, 0.0256, 0.0244, 0.0239/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0206, 0.0203, 0.0202, 0.0213, 0.0240, 0.0264, 0.0296, 0.0327, 0.0338, 
     &  0.0339, 0.0330, 0.0322, 0.0276, 0.0248, 0.0231, 0.0210, 0.0199, 0.0195/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0161, 0.0159, 0.0158, 0.0166, 0.0187, 0.0206, 0.0230, 0.0252, 0.0261, 
     &  0.0262, 0.0256, 0.0251, 0.0217, 0.0198, 0.0188, 0.0171, 0.0162, 0.0159/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0125, 0.0125, 0.0123, 0.0130, 0.0147, 0.0161, 0.0179, 0.0195, 0.0200, 
     &  0.0201, 0.0198, 0.0194, 0.0169, 0.0157, 0.0151, 0.0139, 0.0132, 0.0129/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0097, 0.0098, 0.0096, 0.0102, 0.0115, 0.0126, 0.0139, 0.0150, 0.0154, 
     &  0.0155, 0.0153, 0.0151, 0.0131, 0.0123, 0.0120, 0.0112, 0.0106, 0.0105/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0076, 0.0076, 0.0075, 0.0080, 0.0090, 0.0099, 0.0109, 0.0116, 0.0118, 
     &  0.0119, 0.0118, 0.0117, 0.0102, 0.0096, 0.0095, 0.0090, 0.0085, 0.0084/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0059, 0.0059, 0.0059, 0.0063, 0.0071, 0.0078, 0.0085, 0.0090, 0.0091, 
     &  0.0091, 0.0091, 0.0091, 0.0079, 0.0074, 0.0074, 0.0071, 0.0068, 0.0067/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0045, 0.0046, 0.0046, 0.0049, 0.0056, 0.0061, 0.0066, 0.0069, 0.0069, 
     &  0.0069, 0.0070, 0.0070, 0.0061, 0.0057, 0.0058, 0.0056, 0.0054, 0.0054/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0035, 0.0036, 0.0036, 0.0038, 0.0044, 0.0048, 0.0051, 0.0053, 0.0053, 
     &  0.0053, 0.0054, 0.0054, 0.0047, 0.0044, 0.0044, 0.0044, 0.0043, 0.0043/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0027, 0.0028, 0.0028, 0.0030, 0.0034, 0.0037, 0.0040, 0.0041, 0.0040, 
     &  0.0040, 0.0041, 0.0042, 0.0037, 0.0033, 0.0034, 0.0034, 0.0034, 0.0034/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0021, 0.0022, 0.0022, 0.0023, 0.0027, 0.0029, 0.0031, 0.0032, 0.0031, 
     &  0.0031, 0.0032, 0.0033, 0.0028, 0.0026, 0.0025, 0.0026, 0.0027, 0.0027/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0017, 0.0017, 0.0017, 0.0018, 0.0021, 0.0023, 0.0024, 0.0024, 0.0024, 
     &  0.0024, 0.0025, 0.0025, 0.0022, 0.0020, 0.0019, 0.0020, 0.0022, 0.0022/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0013, 0.0013, 0.0013, 0.0014, 0.0016, 0.0018, 0.0019, 0.0019, 0.0018, 
     &  0.0018, 0.0019, 0.0020, 0.0017, 0.0015, 0.0015, 0.0015, 0.0017, 0.0018/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0010, 0.0010, 0.0011, 0.0013, 0.0014, 0.0015, 0.0015, 0.0014, 
     &  0.0014, 0.0015, 0.0015, 0.0013, 0.0012, 0.0011, 0.0012, 0.0014, 0.0014/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0008, 0.0008, 0.0009, 0.0010, 0.0011, 0.0011, 0.0011, 0.0011, 
     &  0.0011, 0.0011, 0.0012, 0.0010, 0.0009, 0.0008, 0.0009, 0.0011, 0.0012/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0006, 0.0007, 0.0008, 0.0008, 0.0009, 0.0009, 0.0008, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0007, 0.0007, 0.0007, 0.0009, 0.0010/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0006, 0.0007, 0.0007, 0.0007, 0.0006, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0005, 0.0005, 0.0005, 0.0008, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0006, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0004, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0002, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  1 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  2. Data corresponds to middle of FEB.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9933, 0.9948, 0.9944, 0.9942, 0.9886, 0.9940, 0.9951, 0.9918, 
     &  0.9857, 0.9862, 0.9857, 0.9883, 0.9903, 0.9918, 0.9902, 0.9908, 0.9916/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9965, 0.9875, 0.9897, 0.9882, 0.9872, 0.9772, 0.9855, 0.9895, 0.9820, 
     &  0.9709, 0.9680, 0.9704, 0.9756, 0.9802, 0.9822, 0.9806, 0.9819, 0.9833/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9908, 0.9819, 0.9849, 0.9813, 0.9795, 0.9669, 0.9749, 0.9829, 0.9717, 
     &  0.9570, 0.9516, 0.9560, 0.9634, 0.9704, 0.9727, 0.9716, 0.9735, 0.9754/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9861, 0.9763, 0.9797, 0.9743, 0.9716, 0.9571, 0.9641, 0.9755, 0.9616, 
     &  0.9448, 0.9371, 0.9435, 0.9525, 0.9614, 0.9641, 0.9633, 0.9659, 0.9684/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9815, 0.9705, 0.9743, 0.9672, 0.9637, 0.9477, 0.9536, 0.9677, 0.9519, 
     &  0.9338, 0.9242, 0.9323, 0.9427, 0.9531, 0.9561, 0.9556, 0.9588, 0.9618/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9766, 0.9645, 0.9689, 0.9603, 0.9563, 0.9387, 0.9434, 0.9604, 0.9430, 
     &  0.9236, 0.9126, 0.9223, 0.9340, 0.9456, 0.9489, 0.9486, 0.9522, 0.9556/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9714, 0.9584, 0.9636, 0.9537, 0.9495, 0.9300, 0.9336, 0.9538, 0.9352, 
     &  0.9146, 0.9023, 0.9135, 0.9258, 0.9387, 0.9421, 0.9418, 0.9456, 0.9497/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9632, 0.9501, 0.9564, 0.9462, 0.9426, 0.9218, 0.9246, 0.9481, 0.9287, 
     &  0.9067, 0.8931, 0.9056, 0.9180, 0.9315, 0.9349, 0.9343, 0.9379, 0.9427/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9469, 0.9360, 0.9434, 0.9364, 0.9351, 0.9140, 0.9169, 0.9434, 0.9230, 
     &  0.8999, 0.8852, 0.8986, 0.9096, 0.9224, 0.9252, 0.9238, 0.9266, 0.9323/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9248, 0.9168, 0.9252, 0.9242, 0.9264, 0.9063, 0.9102, 0.9395, 0.9182, 
     &  0.8939, 0.8785, 0.8927, 0.8996, 0.9090, 0.9108, 0.9082, 0.9095, 0.9163/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9016, 0.8955, 0.9053, 0.9100, 0.9164, 0.8985, 0.9042, 0.9361, 0.9140, 
     &  0.8885, 0.8725, 0.8875, 0.8865, 0.8903, 0.8910, 0.8869, 0.8864, 0.8940/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8776, 0.8731, 0.8852, 0.8942, 0.9053, 0.8903, 0.8988, 0.9331, 0.9104, 
     &  0.8836, 0.8669, 0.8828, 0.8704, 0.8673, 0.8659, 0.8600, 0.8572, 0.8652/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8519, 0.8494, 0.8644, 0.8777, 0.8939, 0.8822, 0.8938, 0.9300, 0.9070, 
     &  0.8790, 0.8619, 0.8781, 0.8528, 0.8420, 0.8372, 0.8285, 0.8232, 0.8311/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8229, 0.8229, 0.8416, 0.8600, 0.8818, 0.8740, 0.8887, 0.9265, 0.9036, 
     &  0.8747, 0.8571, 0.8734, 0.8353, 0.8156, 0.8058, 0.7937, 0.7854, 0.7928/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7885, 0.7915, 0.8149, 0.8390, 0.8674, 0.8651, 0.8836, 0.9229, 0.9005, 
     &  0.8707, 0.8525, 0.8681, 0.8177, 0.7876, 0.7711, 0.7549, 0.7436, 0.7503/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7479, 0.7542, 0.7831, 0.8140, 0.8506, 0.8548, 0.8782, 0.9190, 0.8975, 
     &  0.8672, 0.8480, 0.8622, 0.7992, 0.7572, 0.7328, 0.7123, 0.6985, 0.7042/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7005, 0.7100, 0.7451, 0.7837, 0.8299, 0.8419, 0.8707, 0.9130, 0.8928, 
     &  0.8623, 0.8420, 0.8541, 0.7774, 0.7227, 0.6904, 0.6666, 0.6511, 0.6558/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6467, 0.6590, 0.7005, 0.7469, 0.8033, 0.8245, 0.8595, 0.9033, 0.8849, 
     &  0.8545, 0.8328, 0.8418, 0.7498, 0.6826, 0.6437, 0.6178, 0.6014, 0.6052/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5895, 0.6044, 0.6516, 0.7050, 0.7710, 0.8019, 0.8442, 0.8898, 0.8732, 
     &  0.8434, 0.8196, 0.8239, 0.7161, 0.6385, 0.5946, 0.5672, 0.5505, 0.5537/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5310, 0.5479, 0.5995, 0.6583, 0.7317, 0.7726, 0.8224, 0.8701, 0.8553, 
     &  0.8265, 0.7991, 0.7963, 0.6754, 0.5906, 0.5440, 0.5162, 0.5002, 0.5032/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4738, 0.4921, 0.5465, 0.6087, 0.6870, 0.7363, 0.7925, 0.8422, 0.8296, 
     &  0.8020, 0.7699, 0.7577, 0.6282, 0.5403, 0.4929, 0.4660, 0.4513, 0.4543/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4207, 0.4398, 0.4954, 0.5590, 0.6395, 0.6942, 0.7544, 0.8056, 0.7961, 
     &  0.7703, 0.7336, 0.7105, 0.5776, 0.4898, 0.4432, 0.4176, 0.4044, 0.4072/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3732, 0.3923, 0.4474, 0.5105, 0.5906, 0.6474, 0.7093, 0.7614, 0.7557, 
     &  0.7320, 0.6919, 0.6582, 0.5262, 0.4411, 0.3963, 0.3724, 0.3607, 0.3632/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3304, 0.3488, 0.4019, 0.4630, 0.5410, 0.5973, 0.6591, 0.7117, 0.7093, 
     &  0.6876, 0.6460, 0.6032, 0.4747, 0.3941, 0.3521, 0.3302, 0.3197, 0.3220/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2925, 0.3098, 0.3596, 0.4174, 0.4909, 0.5445, 0.6046, 0.6568, 0.6566, 
     &  0.6370, 0.5964, 0.5478, 0.4250, 0.3499, 0.3110, 0.2912, 0.2818, 0.2837/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2595, 0.2753, 0.3205, 0.3733, 0.4399, 0.4888, 0.5460, 0.5967, 0.5983, 
     &  0.5810, 0.5430, 0.4928, 0.3779, 0.3090, 0.2735, 0.2555, 0.2471, 0.2486/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2302, 0.2443, 0.2842, 0.3308, 0.3889, 0.4324, 0.4858, 0.5339, 0.5374, 
     &  0.5223, 0.4873, 0.4388, 0.3335, 0.2714, 0.2394, 0.2233, 0.2157, 0.2168/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2036, 0.2159, 0.2507, 0.2906, 0.3400, 0.3778, 0.4263, 0.4705, 0.4754, 
     &  0.4623, 0.4308, 0.3866, 0.2926, 0.2374, 0.2088, 0.1944, 0.1876, 0.1884/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1798, 0.1904, 0.2200, 0.2533, 0.2943, 0.3263, 0.3687, 0.4081, 0.4134, 
     &  0.4022, 0.3747, 0.3369, 0.2554, 0.2070, 0.1817, 0.1688, 0.1627, 0.1632/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1585, 0.1673, 0.1919, 0.2191, 0.2523, 0.2787, 0.3148, 0.3490, 0.3539, 
     &  0.3442, 0.3212, 0.2905, 0.2216, 0.1798, 0.1578, 0.1463, 0.1407, 0.1410/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1391, 0.1463, 0.1664, 0.1880, 0.2147, 0.2361, 0.2663, 0.2953, 0.2991, 
     &  0.2905, 0.2719, 0.2482, 0.1912, 0.1557, 0.1366, 0.1263, 0.1213, 0.1214/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1215, 0.1272, 0.1431, 0.1600, 0.1814, 0.1984, 0.2231, 0.2470, 0.2495, 
     &  0.2420, 0.2276, 0.2102, 0.1639, 0.1342, 0.1179, 0.1088, 0.1042, 0.1043/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1055, 0.1099, 0.1220, 0.1350, 0.1520, 0.1653, 0.1851, 0.2044, 0.2059, 
     &  0.1996, 0.1887, 0.1765, 0.1394, 0.1150, 0.1013, 0.0933, 0.0893, 0.0893/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0909, 0.0941, 0.1031, 0.1128, 0.1262, 0.1365, 0.1523, 0.1676, 0.1685, 
     &  0.1634, 0.1553, 0.1470, 0.1175, 0.0979, 0.0866, 0.0797, 0.0761, 0.0761/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0775, 0.0797, 0.0862, 0.0933, 0.1037, 0.1117, 0.1240, 0.1358, 0.1362, 
     &  0.1324, 0.1265, 0.1210, 0.0980, 0.0826, 0.0737, 0.0678, 0.0647, 0.0647/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0655, 0.0669, 0.0712, 0.0764, 0.0845, 0.0905, 0.1000, 0.1087, 0.1088, 
     &  0.1061, 0.1022, 0.0987, 0.0808, 0.0691, 0.0623, 0.0574, 0.0547, 0.0548/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0547, 0.0554, 0.0581, 0.0618, 0.0681, 0.0727, 0.0799, 0.0862, 0.0860, 
     &  0.0844, 0.0819, 0.0796, 0.0659, 0.0573, 0.0523, 0.0483, 0.0461, 0.0462/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0451, 0.0454, 0.0468, 0.0494, 0.0542, 0.0579, 0.0633, 0.0678, 0.0674, 
     &  0.0666, 0.0649, 0.0635, 0.0532, 0.0470, 0.0436, 0.0405, 0.0386, 0.0388/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0368, 0.0367, 0.0373, 0.0391, 0.0429, 0.0457, 0.0498, 0.0529, 0.0525, 
     &  0.0521, 0.0510, 0.0502, 0.0426, 0.0381, 0.0360, 0.0337, 0.0322, 0.0324/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0296, 0.0292, 0.0293, 0.0306, 0.0336, 0.0359, 0.0390, 0.0411, 0.0407, 
     &  0.0405, 0.0397, 0.0393, 0.0337, 0.0306, 0.0294, 0.0279, 0.0267, 0.0269/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0235, 0.0231, 0.0229, 0.0238, 0.0262, 0.0281, 0.0304, 0.0319, 0.0314, 
     &  0.0314, 0.0308, 0.0307, 0.0265, 0.0243, 0.0238, 0.0229, 0.0221, 0.0222/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0186, 0.0181, 0.0178, 0.0186, 0.0205, 0.0219, 0.0237, 0.0248, 0.0242, 
     &  0.0242, 0.0239, 0.0239, 0.0208, 0.0191, 0.0191, 0.0187, 0.0181, 0.0182/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0145, 0.0141, 0.0139, 0.0145, 0.0160, 0.0172, 0.0185, 0.0192, 0.0187, 
     &  0.0187, 0.0185, 0.0186, 0.0162, 0.0149, 0.0151, 0.0151, 0.0147, 0.0149/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0113, 0.0109, 0.0108, 0.0113, 0.0125, 0.0134, 0.0144, 0.0149, 0.0144, 
     &  0.0144, 0.0143, 0.0145, 0.0126, 0.0115, 0.0118, 0.0121, 0.0119, 0.0120/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0088, 0.0085, 0.0084, 0.0089, 0.0098, 0.0105, 0.0113, 0.0116, 0.0112, 
     &  0.0111, 0.0111, 0.0113, 0.0098, 0.0089, 0.0091, 0.0096, 0.0095, 0.0096/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0068, 0.0066, 0.0066, 0.0069, 0.0077, 0.0083, 0.0088, 0.0090, 0.0086, 
     &  0.0085, 0.0086, 0.0088, 0.0076, 0.0068, 0.0070, 0.0075, 0.0076, 0.0077/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0052, 0.0051, 0.0051, 0.0054, 0.0061, 0.0065, 0.0068, 0.0070, 0.0067, 
     &  0.0066, 0.0067, 0.0068, 0.0059, 0.0052, 0.0053, 0.0058, 0.0060, 0.0061/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0040, 0.0039, 0.0040, 0.0042, 0.0047, 0.0050, 0.0053, 0.0054, 0.0052, 
     &  0.0051, 0.0051, 0.0053, 0.0045, 0.0040, 0.0040, 0.0044, 0.0047, 0.0047/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0031, 0.0030, 0.0031, 0.0033, 0.0037, 0.0039, 0.0041, 0.0041, 0.0040, 
     &  0.0040, 0.0040, 0.0041, 0.0035, 0.0030, 0.0030, 0.0033, 0.0036, 0.0037/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0024, 0.0023, 0.0024, 0.0026, 0.0029, 0.0031, 0.0032, 0.0032, 0.0031, 
     &  0.0031, 0.0031, 0.0032, 0.0027, 0.0023, 0.0023, 0.0025, 0.0028, 0.0028/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0018, 0.0018, 0.0019, 0.0020, 0.0023, 0.0024, 0.0025, 0.0025, 0.0024, 
     &  0.0024, 0.0024, 0.0024, 0.0021, 0.0018, 0.0017, 0.0019, 0.0021, 0.0022/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0014, 0.0014, 0.0015, 0.0016, 0.0018, 0.0019, 0.0019, 0.0019, 0.0019, 
     &  0.0019, 0.0019, 0.0019, 0.0016, 0.0014, 0.0013, 0.0014, 0.0016, 0.0017/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0011, 0.0011, 0.0011, 0.0012, 0.0014, 0.0015, 0.0015, 0.0015, 0.0014, 
     &  0.0015, 0.0014, 0.0015, 0.0013, 0.0011, 0.0010, 0.0011, 0.0012, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0009, 0.0009, 0.0009, 0.0010, 0.0011, 0.0011, 0.0011, 0.0011, 0.0011, 
     &  0.0011, 0.0011, 0.0011, 0.0010, 0.0008, 0.0008, 0.0008, 0.0009, 0.0009/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0007, 0.0008, 0.0009, 0.0009, 0.0009, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0006, 0.0006, 0.0006, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0005, 0.0006, 0.0007, 0.0007, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  2 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  3. Data corresponds to middle of MAR.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9924, 0.9935, 0.9927, 0.9935, 0.9895, 0.9937, 0.9947, 0.9914, 
     &  0.9867, 0.9878, 0.9841, 0.9856, 0.9888, 0.9904, 0.9893, 0.9908, 0.9916/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9970, 0.9851, 0.9869, 0.9851, 0.9858, 0.9791, 0.9848, 0.9885, 0.9822, 
     &  0.9726, 0.9708, 0.9670, 0.9716, 0.9777, 0.9799, 0.9790, 0.9813, 0.9826/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9914, 0.9782, 0.9809, 0.9774, 0.9778, 0.9695, 0.9741, 0.9815, 0.9725, 
     &  0.9596, 0.9544, 0.9503, 0.9585, 0.9670, 0.9696, 0.9693, 0.9725, 0.9740/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9861, 0.9717, 0.9753, 0.9697, 0.9699, 0.9603, 0.9634, 0.9740, 0.9628, 
     &  0.9481, 0.9396, 0.9358, 0.9469, 0.9574, 0.9603, 0.9605, 0.9644, 0.9661/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9810, 0.9653, 0.9699, 0.9624, 0.9623, 0.9516, 0.9528, 0.9666, 0.9532, 
     &  0.9371, 0.9267, 0.9235, 0.9366, 0.9485, 0.9517, 0.9524, 0.9569, 0.9588/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9762, 0.9590, 0.9645, 0.9556, 0.9551, 0.9437, 0.9427, 0.9596, 0.9440, 
     &  0.9268, 0.9152, 0.9128, 0.9275, 0.9406, 0.9438, 0.9449, 0.9498, 0.9521/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9709, 0.9526, 0.9589, 0.9491, 0.9484, 0.9365, 0.9335, 0.9535, 0.9360, 
     &  0.9176, 0.9050, 0.9034, 0.9192, 0.9331, 0.9364, 0.9377, 0.9427, 0.9453/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9631, 0.9446, 0.9516, 0.9418, 0.9421, 0.9294, 0.9249, 0.9483, 0.9293, 
     &  0.9096, 0.8960, 0.8952, 0.9112, 0.9255, 0.9285, 0.9294, 0.9340, 0.9366/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9490, 0.9323, 0.9400, 0.9321, 0.9354, 0.9224, 0.9170, 0.9437, 0.9237, 
     &  0.9027, 0.8881, 0.8879, 0.9025, 0.9158, 0.9177, 0.9173, 0.9209, 0.9225/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9290, 0.9151, 0.9240, 0.9202, 0.9279, 0.9154, 0.9100, 0.9399, 0.9189, 
     &  0.8968, 0.8813, 0.8813, 0.8920, 0.9012, 0.9018, 0.8992, 0.9013, 0.9010/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9070, 0.8949, 0.9055, 0.9067, 0.9191, 0.9080, 0.9037, 0.9364, 0.9147, 
     &  0.8917, 0.8752, 0.8753, 0.8783, 0.8807, 0.8801, 0.8750, 0.8751, 0.8729/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8834, 0.8731, 0.8857, 0.8915, 0.9087, 0.8998, 0.8978, 0.9331, 0.9110, 
     &  0.8871, 0.8695, 0.8696, 0.8608, 0.8557, 0.8534, 0.8456, 0.8430, 0.8394/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8573, 0.8492, 0.8646, 0.8755, 0.8977, 0.8912, 0.8923, 0.9300, 0.9077, 
     &  0.8827, 0.8640, 0.8641, 0.8413, 0.8287, 0.8236, 0.8128, 0.8071, 0.8021/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8272, 0.8218, 0.8406, 0.8574, 0.8856, 0.8824, 0.8870, 0.9266, 0.9045, 
     &  0.8786, 0.8587, 0.8585, 0.8220, 0.8011, 0.7923, 0.7779, 0.7686, 0.7620/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7914, 0.7887, 0.8120, 0.8355, 0.8706, 0.8727, 0.8815, 0.9230, 0.9014, 
     &  0.8747, 0.8537, 0.8525, 0.8029, 0.7724, 0.7592, 0.7404, 0.7267, 0.7191/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7493, 0.7496, 0.7786, 0.8097, 0.8528, 0.8613, 0.8757, 0.9190, 0.8983, 
     &  0.8710, 0.8487, 0.8460, 0.7831, 0.7425, 0.7232, 0.6996, 0.6816, 0.6734/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7006, 0.7040, 0.7395, 0.7788, 0.8308, 0.8471, 0.8680, 0.9129, 0.8935, 
     &  0.8658, 0.8423, 0.8375, 0.7606, 0.7099, 0.6837, 0.6558, 0.6346, 0.6262/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6456, 0.6520, 0.6943, 0.7415, 0.8027, 0.8284, 0.8569, 0.9037, 0.8856, 
     &  0.8579, 0.8329, 0.8250, 0.7336, 0.6725, 0.6401, 0.6093, 0.5861, 0.5777/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5877, 0.5969, 0.6451, 0.6994, 0.7688, 0.8042, 0.8416, 0.8909, 0.8742, 
     &  0.8468, 0.8193, 0.8069, 0.7013, 0.6311, 0.5938, 0.5611, 0.5370, 0.5288/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5296, 0.5410, 0.5934, 0.6526, 0.7282, 0.7729, 0.8194, 0.8718, 0.8567, 
     &  0.8298, 0.7985, 0.7797, 0.6626, 0.5857, 0.5454, 0.5123, 0.4885, 0.4805/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4734, 0.4863, 0.5409, 0.6032, 0.6825, 0.7349, 0.7890, 0.8442, 0.8311, 
     &  0.8051, 0.7691, 0.7421, 0.6181, 0.5375, 0.4962, 0.4640, 0.4413, 0.4335/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4213, 0.4351, 0.4902, 0.5536, 0.6343, 0.6916, 0.7508, 0.8078, 0.7969, 
     &  0.7722, 0.7323, 0.6966, 0.5702, 0.4892, 0.4481, 0.4172, 0.3958, 0.3883/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3749, 0.3888, 0.4426, 0.5050, 0.5848, 0.6442, 0.7057, 0.7635, 0.7549, 
     &  0.7318, 0.6900, 0.6466, 0.5216, 0.4428, 0.4027, 0.3734, 0.3533, 0.3460/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3325, 0.3460, 0.3971, 0.4573, 0.5346, 0.5937, 0.6555, 0.7135, 0.7065, 
     &  0.6853, 0.6434, 0.5942, 0.4729, 0.3983, 0.3599, 0.3324, 0.3135, 0.3064/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2937, 0.3066, 0.3545, 0.4114, 0.4845, 0.5404, 0.6005, 0.6575, 0.6521, 
     &  0.6333, 0.5930, 0.5410, 0.4257, 0.3563, 0.3203, 0.2946, 0.2768, 0.2701/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2588, 0.2712, 0.3151, 0.3675, 0.4342, 0.4844, 0.5410, 0.5956, 0.5926, 
     &  0.5762, 0.5385, 0.4873, 0.3803, 0.3170, 0.2838, 0.2600, 0.2434, 0.2370/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2276, 0.2394, 0.2790, 0.3257, 0.3845, 0.4278, 0.4797, 0.5307, 0.5303, 
     &  0.5164, 0.4814, 0.4335, 0.3370, 0.2804, 0.2502, 0.2285, 0.2131, 0.2071/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1996, 0.2108, 0.2462, 0.2869, 0.3374, 0.3734, 0.4192, 0.4653, 0.4669, 
     &  0.4551, 0.4233, 0.3806, 0.2964, 0.2466, 0.2198, 0.2001, 0.1859, 0.1803/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1749, 0.1853, 0.2167, 0.2516, 0.2938, 0.3227, 0.3613, 0.4012, 0.4035, 
     &  0.3935, 0.3658, 0.3298, 0.2587, 0.2159, 0.1924, 0.1747, 0.1617, 0.1564/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1533, 0.1629, 0.1903, 0.2194, 0.2538, 0.2763, 0.3076, 0.3411, 0.3428, 
     &  0.3343, 0.3113, 0.2823, 0.2239, 0.1881, 0.1679, 0.1522, 0.1402, 0.1354/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1342, 0.1430, 0.1666, 0.1904, 0.2178, 0.2351, 0.2600, 0.2871, 0.2871, 
     &  0.2798, 0.2617, 0.2392, 0.1924, 0.1630, 0.1460, 0.1322, 0.1213, 0.1168/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1174, 0.1252, 0.1451, 0.1642, 0.1859, 0.1989, 0.2181, 0.2389, 0.2370, 
     &  0.2307, 0.2174, 0.2008, 0.1638, 0.1404, 0.1264, 0.1145, 0.1047, 0.1005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1024, 0.1093, 0.1257, 0.1406, 0.1575, 0.1670, 0.1815, 0.1969, 0.1937, 
     &  0.1883, 0.1790, 0.1672, 0.1382, 0.1199, 0.1089, 0.0988, 0.0900, 0.0862/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0890, 0.0948, 0.1081, 0.1194, 0.1323, 0.1392, 0.1500, 0.1609, 0.1570, 
     &  0.1526, 0.1462, 0.1381, 0.1154, 0.1014, 0.0930, 0.0848, 0.0771, 0.0736/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0769, 0.0817, 0.0921, 0.1005, 0.1102, 0.1150, 0.1228, 0.1301, 0.1258, 
     &  0.1223, 0.1183, 0.1130, 0.0954, 0.0848, 0.0787, 0.0722, 0.0657, 0.0625/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0660, 0.0699, 0.0778, 0.0837, 0.0909, 0.0942, 0.0997, 0.1042, 0.0998, 
     &  0.0972, 0.0950, 0.0917, 0.0781, 0.0701, 0.0660, 0.0611, 0.0557, 0.0529/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0562, 0.0593, 0.0650, 0.0689, 0.0741, 0.0764, 0.0802, 0.0828, 0.0786, 
     &  0.0768, 0.0757, 0.0738, 0.0633, 0.0573, 0.0546, 0.0512, 0.0469, 0.0445/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0475, 0.0497, 0.0536, 0.0560, 0.0597, 0.0613, 0.0640, 0.0653, 0.0616, 
     &  0.0604, 0.0599, 0.0589, 0.0507, 0.0462, 0.0446, 0.0425, 0.0392, 0.0371/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0397, 0.0413, 0.0437, 0.0449, 0.0476, 0.0487, 0.0506, 0.0513, 0.0480, 
     &  0.0473, 0.0472, 0.0467, 0.0403, 0.0368, 0.0360, 0.0349, 0.0326, 0.0308/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0328, 0.0338, 0.0350, 0.0356, 0.0375, 0.0383, 0.0397, 0.0400, 0.0374, 
     &  0.0369, 0.0369, 0.0367, 0.0317, 0.0289, 0.0286, 0.0282, 0.0267, 0.0253/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0269, 0.0274, 0.0278, 0.0279, 0.0293, 0.0300, 0.0310, 0.0312, 0.0292, 
     &  0.0287, 0.0288, 0.0287, 0.0248, 0.0226, 0.0225, 0.0226, 0.0218, 0.0206/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0218, 0.0220, 0.0218, 0.0218, 0.0229, 0.0234, 0.0241, 0.0243, 0.0228, 
     &  0.0224, 0.0225, 0.0224, 0.0194, 0.0175, 0.0175, 0.0179, 0.0176, 0.0167/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0174, 0.0175, 0.0170, 0.0169, 0.0178, 0.0182, 0.0187, 0.0190, 0.0178, 
     &  0.0175, 0.0175, 0.0175, 0.0150, 0.0135, 0.0135, 0.0140, 0.0140, 0.0134/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0138, 0.0137, 0.0132, 0.0131, 0.0138, 0.0141, 0.0145, 0.0148, 0.0140, 
     &  0.0137, 0.0137, 0.0136, 0.0117, 0.0104, 0.0103, 0.0108, 0.0111, 0.0106/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0108, 0.0107, 0.0102, 0.0102, 0.0107, 0.0110, 0.0113, 0.0115, 0.0110, 
     &  0.0107, 0.0107, 0.0106, 0.0091, 0.0080, 0.0079, 0.0083, 0.0087, 0.0084/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0084, 0.0082, 0.0078, 0.0079, 0.0084, 0.0085, 0.0088, 0.0090, 0.0086, 
     &  0.0084, 0.0083, 0.0083, 0.0070, 0.0061, 0.0060, 0.0063, 0.0067, 0.0066/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0064, 0.0063, 0.0060, 0.0061, 0.0065, 0.0066, 0.0068, 0.0070, 0.0067, 
     &  0.0066, 0.0065, 0.0064, 0.0055, 0.0047, 0.0045, 0.0047, 0.0051, 0.0051/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0049, 0.0048, 0.0046, 0.0047, 0.0051, 0.0051, 0.0052, 0.0054, 0.0053, 
     &  0.0051, 0.0050, 0.0050, 0.0042, 0.0036, 0.0035, 0.0036, 0.0039, 0.0039/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0037, 0.0036, 0.0035, 0.0037, 0.0039, 0.0040, 0.0041, 0.0042, 0.0041, 
     &  0.0040, 0.0039, 0.0039, 0.0033, 0.0028, 0.0026, 0.0027, 0.0029, 0.0029/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0028, 0.0027, 0.0027, 0.0028, 0.0031, 0.0031, 0.0031, 0.0033, 0.0032, 
     &  0.0031, 0.0030, 0.0030, 0.0025, 0.0022, 0.0020, 0.0020, 0.0022, 0.0022/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0021, 0.0021, 0.0021, 0.0022, 0.0024, 0.0024, 0.0024, 0.0025, 0.0025, 
     &  0.0024, 0.0024, 0.0023, 0.0020, 0.0017, 0.0016, 0.0015, 0.0016, 0.0017/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0016, 0.0016, 0.0016, 0.0017, 0.0019, 0.0019, 0.0019, 0.0020, 0.0019, 
     &  0.0019, 0.0018, 0.0018, 0.0015, 0.0013, 0.0012, 0.0012, 0.0012, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0012, 0.0012, 0.0013, 0.0013, 0.0014, 0.0014, 0.0015, 0.0015, 0.0015, 
     &  0.0015, 0.0014, 0.0014, 0.0012, 0.0010, 0.0009, 0.0009, 0.0009, 0.0009/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0009, 0.0010, 0.0010, 0.0011, 0.0011, 0.0011, 0.0012, 0.0012, 
     &  0.0011, 0.0011, 0.0011, 0.0009, 0.0008, 0.0007, 0.0007, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0008, 0.0008, 0.0009, 0.0009, 0.0009, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0008, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0006, 0.0006, 0.0007, 0.0007, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0005, 0.0004, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0002, 0.0002, 0.0003, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0001, 0.0002, 0.0002, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  3 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  4. Data corresponds to middle of APR.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9904, 0.9924, 0.9915, 0.9929, 0.9896, 0.9921, 0.9932, 0.9917, 
     &  0.9874, 0.9873, 0.9837, 0.9847, 0.9871, 0.9888, 0.9886, 0.9904, 0.9922/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9958, 0.9818, 0.9845, 0.9830, 0.9842, 0.9790, 0.9817, 0.9856, 0.9830, 
     &  0.9745, 0.9703, 0.9668, 0.9695, 0.9744, 0.9771, 0.9771, 0.9802, 0.9825/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9884, 0.9737, 0.9772, 0.9748, 0.9754, 0.9690, 0.9697, 0.9774, 0.9739, 
     &  0.9616, 0.9535, 0.9507, 0.9554, 0.9622, 0.9657, 0.9660, 0.9702, 0.9729/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9818, 0.9661, 0.9706, 0.9673, 0.9671, 0.9597, 0.9583, 0.9688, 0.9648, 
     &  0.9497, 0.9377, 0.9363, 0.9430, 0.9512, 0.9552, 0.9559, 0.9609, 0.9642/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9756, 0.9591, 0.9645, 0.9603, 0.9593, 0.9512, 0.9475, 0.9606, 0.9559, 
     &  0.9387, 0.9236, 0.9234, 0.9319, 0.9411, 0.9455, 0.9467, 0.9523, 0.9559/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9697, 0.9523, 0.9585, 0.9537, 0.9521, 0.9435, 0.9374, 0.9530, 0.9474, 
     &  0.9284, 0.9108, 0.9117, 0.9218, 0.9319, 0.9367, 0.9381, 0.9441, 0.9481/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9638, 0.9456, 0.9527, 0.9473, 0.9455, 0.9365, 0.9284, 0.9464, 0.9400, 
     &  0.9191, 0.8994, 0.9012, 0.9125, 0.9234, 0.9283, 0.9297, 0.9359, 0.9398/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9562, 0.9379, 0.9462, 0.9400, 0.9391, 0.9301, 0.9203, 0.9408, 0.9338, 
     &  0.9109, 0.8894, 0.8919, 0.9036, 0.9149, 0.9195, 0.9200, 0.9255, 0.9283/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9439, 0.9267, 0.9365, 0.9311, 0.9324, 0.9240, 0.9132, 0.9360, 0.9285, 
     &  0.9038, 0.8808, 0.8836, 0.8943, 0.9044, 0.9082, 0.9062, 0.9091, 0.9099/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9264, 0.9108, 0.9228, 0.9199, 0.9249, 0.9181, 0.9068, 0.9319, 0.9240, 
     &  0.8979, 0.8734, 0.8760, 0.8835, 0.8895, 0.8917, 0.8862, 0.8848, 0.8834/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9052, 0.8913, 0.9058, 0.9066, 0.9159, 0.9113, 0.9007, 0.9283, 0.9201, 
     &  0.8927, 0.8666, 0.8688, 0.8698, 0.8694, 0.8688, 0.8595, 0.8543, 0.8507/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8813, 0.8691, 0.8864, 0.8913, 0.9053, 0.9031, 0.8950, 0.9250, 0.9165, 
     &  0.8880, 0.8602, 0.8621, 0.8523, 0.8451, 0.8410, 0.8277, 0.8197, 0.8143/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8547, 0.8444, 0.8650, 0.8746, 0.8934, 0.8940, 0.8897, 0.9217, 0.9131, 
     &  0.8837, 0.8542, 0.8558, 0.8327, 0.8189, 0.8111, 0.7941, 0.7834, 0.7754/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8228, 0.8154, 0.8398, 0.8553, 0.8799, 0.8842, 0.8844, 0.9182, 0.9098, 
     &  0.8796, 0.8485, 0.8493, 0.8132, 0.7927, 0.7811, 0.7600, 0.7462, 0.7354/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7833, 0.7796, 0.8092, 0.8315, 0.8631, 0.8733, 0.8787, 0.9147, 0.9067, 
     &  0.8758, 0.8427, 0.8424, 0.7940, 0.7660, 0.7501, 0.7248, 0.7074, 0.6940/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7367, 0.7373, 0.7735, 0.8035, 0.8435, 0.8606, 0.8725, 0.9107, 0.9036, 
     &  0.8721, 0.8368, 0.8350, 0.7744, 0.7388, 0.7169, 0.6872, 0.6656, 0.6505/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6828, 0.6881, 0.7324, 0.7708, 0.8196, 0.8440, 0.8644, 0.9048, 0.8986, 
     &  0.8665, 0.8294, 0.8257, 0.7530, 0.7092, 0.6806, 0.6467, 0.6209, 0.6048/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6217, 0.6319, 0.6853, 0.7321, 0.7896, 0.8222, 0.8527, 0.8958, 0.8907, 
     &  0.8580, 0.8184, 0.8122, 0.7278, 0.6750, 0.6404, 0.6029, 0.5736, 0.5571/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5579, 0.5729, 0.6347, 0.6888, 0.7539, 0.7949, 0.8363, 0.8831, 0.8797, 
     &  0.8465, 0.8033, 0.7932, 0.6980, 0.6367, 0.5971, 0.5566, 0.5249, 0.5085/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4956, 0.5144, 0.5822, 0.6416, 0.7120, 0.7607, 0.8125, 0.8638, 0.8624, 
     &  0.8289, 0.7813, 0.7660, 0.6621, 0.5942, 0.5509, 0.5087, 0.4757, 0.4596/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4368, 0.4581, 0.5293, 0.5921, 0.6657, 0.7202, 0.7803, 0.8360, 0.8363, 
     &  0.8030, 0.7518, 0.7298, 0.6207, 0.5491, 0.5037, 0.4610, 0.4274, 0.4118/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3830, 0.4054, 0.4778, 0.5423, 0.6172, 0.6751, 0.7407, 0.7997, 0.8007, 
     &  0.7684, 0.7154, 0.6866, 0.5760, 0.5035, 0.4575, 0.4154, 0.3820, 0.3669/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3358, 0.3581, 0.4293, 0.4933, 0.5672, 0.6264, 0.6948, 0.7558, 0.7567, 
     &  0.7259, 0.6732, 0.6386, 0.5298, 0.4590, 0.4138, 0.3729, 0.3403, 0.3260/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2938, 0.3151, 0.3831, 0.4450, 0.5162, 0.5749, 0.6440, 0.7058, 0.7063, 
     &  0.6775, 0.6264, 0.5877, 0.4830, 0.4157, 0.3723, 0.3331, 0.3019, 0.2883/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2561, 0.2763, 0.3401, 0.3984, 0.4653, 0.5213, 0.5886, 0.6493, 0.6503, 
     &  0.6241, 0.5753, 0.5351, 0.4366, 0.3741, 0.3335, 0.2965, 0.2669, 0.2541/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2233, 0.2422, 0.3010, 0.3542, 0.4149, 0.4661, 0.5287, 0.5865, 0.5890, 
     &  0.5656, 0.5201, 0.4810, 0.3907, 0.3343, 0.2973, 0.2631, 0.2354, 0.2235/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1952, 0.2127, 0.2655, 0.3126, 0.3661, 0.4112, 0.4673, 0.5207, 0.5248, 
     &  0.5043, 0.4626, 0.4261, 0.3459, 0.2965, 0.2635, 0.2324, 0.2069, 0.1960/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1708, 0.1867, 0.2336, 0.2747, 0.3209, 0.3592, 0.4074, 0.4547, 0.4594, 
     &  0.4416, 0.4046, 0.3720, 0.3030, 0.2611, 0.2324, 0.2045, 0.1814, 0.1715/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1494, 0.1638, 0.2054, 0.2409, 0.2799, 0.3114, 0.3510, 0.3906, 0.3945, 
     &  0.3793, 0.3478, 0.3204, 0.2628, 0.2283, 0.2039, 0.1794, 0.1586, 0.1496/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1305, 0.1436, 0.1806, 0.2108, 0.2429, 0.2682, 0.2996, 0.3312, 0.3331, 
     &  0.3202, 0.2948, 0.2728, 0.2257, 0.1982, 0.1779, 0.1567, 0.1383, 0.1303/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1140, 0.1259, 0.1587, 0.1841, 0.2102, 0.2300, 0.2543, 0.2781, 0.2776, 
     &  0.2667, 0.2472, 0.2302, 0.1923, 0.1707, 0.1542, 0.1362, 0.1202, 0.1131/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0995, 0.1103, 0.1392, 0.1603, 0.1812, 0.1963, 0.2145, 0.2312, 0.2281, 
     &  0.2191, 0.2051, 0.1928, 0.1623, 0.1456, 0.1326, 0.1177, 0.1039, 0.0977/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0868, 0.0966, 0.1218, 0.1390, 0.1554, 0.1666, 0.1797, 0.1906, 0.1857, 
     &  0.1782, 0.1687, 0.1602, 0.1358, 0.1229, 0.1129, 0.1008, 0.0892, 0.0839/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0756, 0.0843, 0.1061, 0.1199, 0.1324, 0.1404, 0.1495, 0.1561, 0.1502, 
     &  0.1442, 0.1379, 0.1321, 0.1125, 0.1025, 0.0950, 0.0855, 0.0760, 0.0715/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0656, 0.0733, 0.0919, 0.1027, 0.1120, 0.1174, 0.1233, 0.1268, 0.1205, 
     &  0.1156, 0.1116, 0.1079, 0.0922, 0.0844, 0.0790, 0.0718, 0.0641, 0.0604/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0567, 0.0634, 0.0791, 0.0873, 0.0940, 0.0973, 0.1008, 0.1022, 0.0961, 
     &  0.0922, 0.0896, 0.0874, 0.0749, 0.0686, 0.0648, 0.0596, 0.0536, 0.0507/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0489, 0.0546, 0.0674, 0.0735, 0.0781, 0.0798, 0.0816, 0.0817, 0.0764, 
     &  0.0734, 0.0715, 0.0702, 0.0602, 0.0550, 0.0525, 0.0489, 0.0444, 0.0421/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0419, 0.0467, 0.0569, 0.0612, 0.0642, 0.0648, 0.0654, 0.0649, 0.0606, 
     &  0.0581, 0.0566, 0.0559, 0.0479, 0.0437, 0.0420, 0.0396, 0.0363, 0.0346/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0357, 0.0396, 0.0475, 0.0504, 0.0521, 0.0520, 0.0519, 0.0513, 0.0478, 
     &  0.0458, 0.0446, 0.0441, 0.0378, 0.0343, 0.0331, 0.0317, 0.0294, 0.0282/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0303, 0.0333, 0.0392, 0.0408, 0.0418, 0.0411, 0.0408, 0.0402, 0.0375, 
     &  0.0360, 0.0349, 0.0346, 0.0296, 0.0267, 0.0258, 0.0250, 0.0236, 0.0228/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0255, 0.0278, 0.0319, 0.0327, 0.0331, 0.0323, 0.0319, 0.0315, 0.0295, 
     &  0.0282, 0.0273, 0.0270, 0.0231, 0.0207, 0.0200, 0.0195, 0.0187, 0.0183/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0214, 0.0230, 0.0257, 0.0260, 0.0260, 0.0252, 0.0248, 0.0246, 0.0231, 
     &  0.0221, 0.0214, 0.0211, 0.0180, 0.0160, 0.0155, 0.0152, 0.0147, 0.0145/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0178, 0.0189, 0.0204, 0.0203, 0.0202, 0.0195, 0.0192, 0.0192, 0.0181, 
     &  0.0173, 0.0167, 0.0164, 0.0140, 0.0124, 0.0119, 0.0117, 0.0115, 0.0115/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0147, 0.0153, 0.0160, 0.0157, 0.0156, 0.0151, 0.0149, 0.0150, 0.0142, 
     &  0.0136, 0.0130, 0.0128, 0.0109, 0.0096, 0.0092, 0.0090, 0.0089, 0.0090/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0120, 0.0124, 0.0125, 0.0121, 0.0120, 0.0116, 0.0116, 0.0117, 0.0112, 
     &  0.0107, 0.0102, 0.0100, 0.0085, 0.0075, 0.0071, 0.0069, 0.0069, 0.0070/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0098, 0.0099, 0.0096, 0.0093, 0.0092, 0.0090, 0.0090, 0.0091, 0.0088, 
     &  0.0084, 0.0080, 0.0078, 0.0066, 0.0059, 0.0055, 0.0053, 0.0053, 0.0054/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0078, 0.0078, 0.0073, 0.0071, 0.0071, 0.0069, 0.0069, 0.0071, 0.0069, 
     &  0.0066, 0.0062, 0.0061, 0.0052, 0.0046, 0.0042, 0.0041, 0.0041, 0.0041/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0062, 0.0061, 0.0055, 0.0053, 0.0054, 0.0053, 0.0054, 0.0055, 0.0054, 
     &  0.0051, 0.0048, 0.0047, 0.0040, 0.0035, 0.0033, 0.0031, 0.0031, 0.0032/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0049, 0.0047, 0.0041, 0.0041, 0.0041, 0.0041, 0.0042, 0.0043, 0.0042, 
     &  0.0040, 0.0038, 0.0037, 0.0031, 0.0028, 0.0026, 0.0024, 0.0024, 0.0024/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0039, 0.0036, 0.0031, 0.0031, 0.0032, 0.0032, 0.0032, 0.0033, 0.0033, 
     &  0.0031, 0.0029, 0.0029, 0.0024, 0.0022, 0.0020, 0.0019, 0.0018, 0.0018/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0030, 0.0028, 0.0023, 0.0023, 0.0024, 0.0024, 0.0025, 0.0026, 0.0025, 
     &  0.0024, 0.0023, 0.0022, 0.0019, 0.0017, 0.0016, 0.0014, 0.0014, 0.0014/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0024, 0.0022, 0.0017, 0.0018, 0.0019, 0.0019, 0.0019, 0.0020, 0.0020, 
     &  0.0019, 0.0018, 0.0017, 0.0015, 0.0013, 0.0012, 0.0011, 0.0011, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0019, 0.0017, 0.0013, 0.0014, 0.0014, 0.0015, 0.0015, 0.0016, 0.0015, 
     &  0.0015, 0.0014, 0.0013, 0.0011, 0.0010, 0.0010, 0.0009, 0.0008, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0015, 0.0013, 0.0010, 0.0010, 0.0011, 0.0011, 0.0012, 0.0012, 0.0012, 
     &  0.0011, 0.0011, 0.0010, 0.0009, 0.0008, 0.0007, 0.0007, 0.0007, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0012, 0.0010, 0.0008, 0.0008, 0.0009, 0.0009, 0.0009, 0.0009, 0.0009, 
     &  0.0009, 0.0008, 0.0008, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0008, 0.0006, 0.0006, 0.0007, 0.0007, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0006, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0007, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 0.0006, 0.0006, 
     &  0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0005, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  4 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  5. Data corresponds to middle of MAY.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9887, 0.9897, 0.9899, 0.9925, 0.9905, 0.9900, 0.9914, 0.9912, 
     &  0.9892, 0.9895, 0.9848, 0.9826, 0.9871, 0.9879, 0.9885, 0.9904, 0.9929/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9955, 0.9788, 0.9804, 0.9800, 0.9834, 0.9799, 0.9788, 0.9826, 0.9823, 
     &  0.9782, 0.9751, 0.9697, 0.9663, 0.9736, 0.9752, 0.9764, 0.9795, 0.9829/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9870, 0.9700, 0.9720, 0.9708, 0.9745, 0.9697, 0.9676, 0.9734, 0.9730, 
     &  0.9669, 0.9604, 0.9548, 0.9515, 0.9608, 0.9627, 0.9645, 0.9684, 0.9720/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9791, 0.9620, 0.9644, 0.9624, 0.9664, 0.9605, 0.9570, 0.9643, 0.9638, 
     &  0.9560, 0.9458, 0.9411, 0.9383, 0.9490, 0.9512, 0.9532, 0.9578, 0.9615/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9720, 0.9546, 0.9576, 0.9547, 0.9591, 0.9522, 0.9473, 0.9556, 0.9548, 
     &  0.9453, 0.9316, 0.9284, 0.9264, 0.9379, 0.9405, 0.9426, 0.9477, 0.9516/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9656, 0.9477, 0.9513, 0.9477, 0.9524, 0.9448, 0.9387, 0.9477, 0.9465, 
     &  0.9354, 0.9186, 0.9169, 0.9155, 0.9277, 0.9306, 0.9327, 0.9379, 0.9424/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9595, 0.9409, 0.9453, 0.9413, 0.9462, 0.9381, 0.9308, 0.9407, 0.9391, 
     &  0.9264, 0.9071, 0.9064, 0.9057, 0.9181, 0.9216, 0.9230, 0.9278, 0.9328/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9531, 0.9338, 0.9389, 0.9351, 0.9403, 0.9320, 0.9234, 0.9346, 0.9328, 
     &  0.9185, 0.8968, 0.8968, 0.8966, 0.9090, 0.9125, 0.9124, 0.9147, 0.9204/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9441, 0.9244, 0.9305, 0.9281, 0.9339, 0.9263, 0.9164, 0.9294, 0.9274, 
     &  0.9115, 0.8878, 0.8882, 0.8879, 0.8988, 0.9014, 0.8973, 0.8947, 0.9004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9305, 0.9105, 0.9188, 0.9190, 0.9263, 0.9199, 0.9097, 0.9247, 0.9228, 
     &  0.9054, 0.8800, 0.8805, 0.8787, 0.8854, 0.8856, 0.8754, 0.8668, 0.8713/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9122, 0.8924, 0.9032, 0.9071, 0.9165, 0.9117, 0.9032, 0.9206, 0.9187, 
     &  0.9000, 0.8730, 0.8734, 0.8677, 0.8672, 0.8639, 0.8474, 0.8334, 0.8364/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8903, 0.8708, 0.8842, 0.8918, 0.9042, 0.9012, 0.8970, 0.9169, 0.9150, 
     &  0.8952, 0.8663, 0.8666, 0.8540, 0.8444, 0.8370, 0.8156, 0.7978, 0.7998/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8656, 0.8464, 0.8618, 0.8735, 0.8899, 0.8893, 0.8912, 0.9132, 0.9115, 
     &  0.8907, 0.8600, 0.8599, 0.8379, 0.8198, 0.8081, 0.7832, 0.7624, 0.7635/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8360, 0.8172, 0.8351, 0.8523, 0.8734, 0.8763, 0.8852, 0.9093, 0.9080, 
     &  0.8863, 0.8536, 0.8529, 0.8207, 0.7952, 0.7794, 0.7516, 0.7276, 0.7278/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7980, 0.7804, 0.8020, 0.8265, 0.8540, 0.8618, 0.8788, 0.9053, 0.9048, 
     &  0.8823, 0.8472, 0.8455, 0.8033, 0.7705, 0.7505, 0.7194, 0.6919, 0.6910/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7519, 0.7363, 0.7631, 0.7964, 0.8319, 0.8457, 0.8721, 0.9010, 0.9013, 
     &  0.8782, 0.8405, 0.8376, 0.7858, 0.7452, 0.7198, 0.6848, 0.6536, 0.6513/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6976, 0.6852, 0.7188, 0.7614, 0.8053, 0.8264, 0.8633, 0.8952, 0.8960, 
     &  0.8723, 0.8317, 0.8276, 0.7663, 0.7179, 0.6859, 0.6466, 0.6120, 0.6083/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6355, 0.6270, 0.6688, 0.7205, 0.7724, 0.8017, 0.8506, 0.8866, 0.8880, 
     &  0.8635, 0.8194, 0.8138, 0.7427, 0.6864, 0.6478, 0.6046, 0.5674, 0.5624/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5704, 0.5658, 0.6158, 0.6754, 0.7343, 0.7716, 0.8326, 0.8740, 0.8768, 
     &  0.8511, 0.8031, 0.7949, 0.7145, 0.6508, 0.6063, 0.5601, 0.5209, 0.5149/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5072, 0.5056, 0.5614, 0.6269, 0.6906, 0.7350, 0.8063, 0.8542, 0.8590, 
     &  0.8322, 0.7801, 0.7677, 0.6801, 0.6107, 0.5618, 0.5137, 0.4736, 0.4667/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4472, 0.4477, 0.5071, 0.5763, 0.6429, 0.6926, 0.7714, 0.8253, 0.8319, 
     &  0.8045, 0.7497, 0.7318, 0.6401, 0.5673, 0.5159, 0.4672, 0.4269, 0.4197/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3912, 0.3934, 0.4544, 0.5253, 0.5930, 0.6461, 0.7291, 0.7880, 0.7952, 
     &  0.7679, 0.7127, 0.6891, 0.5963, 0.5230, 0.4710, 0.4227, 0.3829, 0.3754/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3416, 0.3449, 0.4050, 0.4750, 0.5418, 0.5962, 0.6810, 0.7433, 0.7505, 
     &  0.7236, 0.6698, 0.6416, 0.5503, 0.4791, 0.4282, 0.3813, 0.3427, 0.3351/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2978, 0.3018, 0.3584, 0.4251, 0.4895, 0.5436, 0.6281, 0.6926, 0.6999, 
     &  0.6739, 0.6222, 0.5910, 0.5030, 0.4357, 0.3872, 0.3425, 0.3055, 0.2979/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2595, 0.2638, 0.3154, 0.3775, 0.4381, 0.4899, 0.5712, 0.6357, 0.6436, 
     &  0.6190, 0.5701, 0.5379, 0.4552, 0.3933, 0.3483, 0.3064, 0.2714, 0.2639/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2262, 0.2306, 0.2768, 0.3331, 0.3886, 0.4363, 0.5110, 0.5729, 0.5816, 
     &  0.5589, 0.5138, 0.4825, 0.4070, 0.3520, 0.3115, 0.2728, 0.2402, 0.2331/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1977, 0.2017, 0.2423, 0.2921, 0.3417, 0.3843, 0.4504, 0.5074, 0.5166, 
     &  0.4960, 0.4554, 0.4264, 0.3592, 0.3120, 0.2766, 0.2416, 0.2118, 0.2053/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1730, 0.1766, 0.2117, 0.2552, 0.2989, 0.3362, 0.3927, 0.4422, 0.4506, 
     &  0.4325, 0.3973, 0.3715, 0.3133, 0.2738, 0.2438, 0.2125, 0.1858, 0.1802/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1512, 0.1546, 0.1851, 0.2229, 0.2609, 0.2927, 0.3393, 0.3796, 0.3858, 
     &  0.3703, 0.3411, 0.3194, 0.2703, 0.2380, 0.2132, 0.1858, 0.1624, 0.1577/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1320, 0.1353, 0.1619, 0.1947, 0.2272, 0.2537, 0.2910, 0.3218, 0.3250, 
     &  0.3122, 0.2890, 0.2715, 0.2309, 0.2050, 0.1849, 0.1614, 0.1413, 0.1375/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1152, 0.1185, 0.1418, 0.1700, 0.1977, 0.2192, 0.2484, 0.2707, 0.2706, 
     &  0.2602, 0.2425, 0.2291, 0.1957, 0.1750, 0.1589, 0.1392, 0.1223, 0.1193/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1007, 0.1038, 0.1242, 0.1486, 0.1718, 0.1887, 0.2109, 0.2259, 0.2227, 
     &  0.2144, 0.2018, 0.1918, 0.1644, 0.1478, 0.1352, 0.1189, 0.1051, 0.1029/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0881, 0.0910, 0.1088, 0.1298, 0.1489, 0.1616, 0.1778, 0.1871, 0.1819, 
     &  0.1753, 0.1665, 0.1592, 0.1369, 0.1235, 0.1137, 0.1006, 0.0896, 0.0881/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0769, 0.0796, 0.0951, 0.1130, 0.1284, 0.1376, 0.1489, 0.1541, 0.1483, 
     &  0.1429, 0.1364, 0.1310, 0.1128, 0.1020, 0.0945, 0.0842, 0.0757, 0.0748/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0670, 0.0694, 0.0827, 0.0980, 0.1102, 0.1162, 0.1237, 0.1259, 0.1204, 
     &  0.1157, 0.1106, 0.1068, 0.0919, 0.0832, 0.0775, 0.0697, 0.0633, 0.0629/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0582, 0.0602, 0.0716, 0.0845, 0.0938, 0.0974, 0.1019, 0.1021, 0.0973, 
     &  0.0933, 0.0889, 0.0862, 0.0742, 0.0670, 0.0628, 0.0570, 0.0525, 0.0524/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0503, 0.0518, 0.0615, 0.0723, 0.0792, 0.0809, 0.0831, 0.0822, 0.0782, 
     &  0.0747, 0.0709, 0.0690, 0.0593, 0.0534, 0.0503, 0.0460, 0.0430, 0.0432/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0431, 0.0443, 0.0524, 0.0613, 0.0663, 0.0664, 0.0671, 0.0658, 0.0625, 
     &  0.0594, 0.0562, 0.0547, 0.0469, 0.0421, 0.0398, 0.0368, 0.0348, 0.0352/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0368, 0.0375, 0.0443, 0.0514, 0.0548, 0.0540, 0.0537, 0.0522, 0.0495, 
     &  0.0470, 0.0442, 0.0431, 0.0369, 0.0330, 0.0312, 0.0291, 0.0280, 0.0284/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0310, 0.0315, 0.0370, 0.0426, 0.0447, 0.0432, 0.0424, 0.0411, 0.0390, 
     &  0.0369, 0.0347, 0.0337, 0.0288, 0.0257, 0.0242, 0.0228, 0.0222, 0.0226/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0260, 0.0262, 0.0306, 0.0349, 0.0361, 0.0343, 0.0332, 0.0322, 0.0307, 
     &  0.0290, 0.0271, 0.0264, 0.0225, 0.0199, 0.0188, 0.0177, 0.0175, 0.0178/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0217, 0.0217, 0.0251, 0.0283, 0.0287, 0.0269, 0.0258, 0.0252, 0.0241, 
     &  0.0227, 0.0212, 0.0206, 0.0175, 0.0155, 0.0146, 0.0138, 0.0137, 0.0139/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0178, 0.0178, 0.0204, 0.0226, 0.0225, 0.0208, 0.0200, 0.0196, 0.0188, 
     &  0.0178, 0.0166, 0.0161, 0.0137, 0.0121, 0.0113, 0.0107, 0.0107, 0.0108/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0146, 0.0145, 0.0164, 0.0178, 0.0175, 0.0160, 0.0154, 0.0153, 0.0148, 
     &  0.0140, 0.0130, 0.0126, 0.0107, 0.0095, 0.0088, 0.0084, 0.0083, 0.0083/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0118, 0.0117, 0.0130, 0.0138, 0.0134, 0.0123, 0.0119, 0.0119, 0.0116, 
     &  0.0110, 0.0102, 0.0098, 0.0084, 0.0074, 0.0069, 0.0065, 0.0065, 0.0064/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0094, 0.0093, 0.0102, 0.0106, 0.0102, 0.0094, 0.0092, 0.0093, 0.0090, 
     &  0.0086, 0.0080, 0.0077, 0.0066, 0.0058, 0.0054, 0.0051, 0.0050, 0.0049/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0074, 0.0073, 0.0079, 0.0080, 0.0077, 0.0072, 0.0071, 0.0072, 0.0071, 
     &  0.0067, 0.0062, 0.0060, 0.0052, 0.0046, 0.0042, 0.0040, 0.0039, 0.0038/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0058, 0.0058, 0.0060, 0.0060, 0.0058, 0.0055, 0.0055, 0.0056, 0.0055, 
     &  0.0052, 0.0048, 0.0047, 0.0040, 0.0036, 0.0033, 0.0031, 0.0030, 0.0029/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0046, 0.0045, 0.0046, 0.0044, 0.0044, 0.0042, 0.0043, 0.0044, 0.0043, 
     &  0.0041, 0.0038, 0.0037, 0.0031, 0.0028, 0.0026, 0.0024, 0.0023, 0.0022/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0036, 0.0036, 0.0035, 0.0033, 0.0033, 0.0032, 0.0033, 0.0034, 0.0033, 
     &  0.0032, 0.0029, 0.0029, 0.0025, 0.0022, 0.0020, 0.0019, 0.0018, 0.0017/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0029, 0.0029, 0.0026, 0.0024, 0.0025, 0.0024, 0.0026, 0.0027, 0.0026, 
     &  0.0025, 0.0023, 0.0022, 0.0019, 0.0017, 0.0016, 0.0015, 0.0014, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0023, 0.0023, 0.0020, 0.0018, 0.0019, 0.0019, 0.0020, 0.0021, 0.0020, 
     &  0.0019, 0.0018, 0.0017, 0.0015, 0.0014, 0.0013, 0.0012, 0.0011, 0.0010/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0019, 0.0019, 0.0015, 0.0014, 0.0014, 0.0014, 0.0015, 0.0016, 0.0016, 
     &  0.0015, 0.0014, 0.0014, 0.0012, 0.0011, 0.0010, 0.0009, 0.0008, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0016, 0.0015, 0.0012, 0.0010, 0.0011, 0.0011, 0.0012, 0.0013, 0.0012, 
     &  0.0012, 0.0011, 0.0011, 0.0009, 0.0008, 0.0008, 0.0007, 0.0006, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0013, 0.0012, 0.0009, 0.0008, 0.0008, 0.0009, 0.0009, 0.0010, 0.0009, 
     &  0.0009, 0.0008, 0.0008, 0.0007, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0011, 0.0010, 0.0007, 0.0006, 0.0006, 0.0007, 0.0007, 0.0008, 0.0007, 
     &  0.0007, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0009, 0.0008, 0.0005, 0.0004, 0.0005, 0.0005, 0.0006, 0.0006, 0.0006, 
     &  0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0004, 0.0003, 0.0004, 0.0004, 0.0004, 0.0005, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0003, 0.0002, 0.0003, 0.0003, 0.0003, 0.0004, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0002, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  5 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  6. Data corresponds to middle of JUN.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9886, 0.9889, 0.9896, 0.9922, 0.9904, 0.9885, 0.9902, 0.9894, 
     &  0.9894, 0.9918, 0.9890, 0.9810, 0.9872, 0.9882, 0.9888, 0.9899, 0.9917/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9950, 0.9780, 0.9787, 0.9794, 0.9828, 0.9798, 0.9770, 0.9801, 0.9789, 
     &  0.9781, 0.9800, 0.9779, 0.9633, 0.9734, 0.9759, 0.9768, 0.9787, 0.9820/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9858, 0.9685, 0.9696, 0.9702, 0.9739, 0.9696, 0.9657, 0.9698, 0.9684, 
     &  0.9667, 0.9679, 0.9666, 0.9476, 0.9601, 0.9636, 0.9644, 0.9672, 0.9718/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9776, 0.9599, 0.9615, 0.9619, 0.9658, 0.9603, 0.9545, 0.9595, 0.9582, 
     &  0.9557, 0.9563, 0.9554, 0.9337, 0.9476, 0.9516, 0.9522, 0.9559, 0.9617/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9704, 0.9521, 0.9541, 0.9544, 0.9587, 0.9518, 0.9437, 0.9497, 0.9483, 
     &  0.9453, 0.9456, 0.9446, 0.9209, 0.9356, 0.9403, 0.9408, 0.9450, 0.9518/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9640, 0.9450, 0.9474, 0.9476, 0.9522, 0.9443, 0.9336, 0.9410, 0.9392, 
     &  0.9355, 0.9357, 0.9345, 0.9094, 0.9245, 0.9297, 0.9303, 0.9349, 0.9424/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9581, 0.9384, 0.9412, 0.9413, 0.9464, 0.9375, 0.9243, 0.9333, 0.9310, 
     &  0.9267, 0.9264, 0.9253, 0.8989, 0.9142, 0.9199, 0.9200, 0.9248, 0.9327/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9527, 0.9319, 0.9355, 0.9351, 0.9410, 0.9312, 0.9158, 0.9262, 0.9239, 
     &  0.9189, 0.9177, 0.9169, 0.8895, 0.9047, 0.9102, 0.9091, 0.9127, 0.9195/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9461, 0.9244, 0.9290, 0.9277, 0.9352, 0.9248, 0.9080, 0.9199, 0.9178, 
     &  0.9120, 0.9099, 0.9095, 0.8806, 0.8949, 0.8992, 0.8954, 0.8945, 0.8979/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9360, 0.9136, 0.9198, 0.9175, 0.9274, 0.9176, 0.9007, 0.9145, 0.9126, 
     &  0.9059, 0.9027, 0.9030, 0.8720, 0.8829, 0.8842, 0.8758, 0.8680, 0.8672/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9210, 0.8985, 0.9064, 0.9031, 0.9160, 0.9077, 0.8938, 0.9097, 0.9080, 
     &  0.9004, 0.8960, 0.8969, 0.8626, 0.8672, 0.8639, 0.8496, 0.8352, 0.8320/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9009, 0.8791, 0.8879, 0.8843, 0.9007, 0.8946, 0.8875, 0.9053, 0.9038, 
     &  0.8954, 0.8893, 0.8910, 0.8514, 0.8479, 0.8396, 0.8196, 0.8011, 0.7967/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8773, 0.8560, 0.8650, 0.8623, 0.8831, 0.8797, 0.8813, 0.9010, 0.8998, 
     &  0.8906, 0.8824, 0.8848, 0.8383, 0.8269, 0.8140, 0.7897, 0.7683, 0.7628/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8489, 0.8279, 0.8367, 0.8373, 0.8634, 0.8642, 0.8750, 0.8965, 0.8957, 
     &  0.8859, 0.8752, 0.8782, 0.8243, 0.8058, 0.7887, 0.7608, 0.7361, 0.7296/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8126, 0.7918, 0.8012, 0.8082, 0.8408, 0.8474, 0.8685, 0.8919, 0.8917, 
     &  0.8812, 0.8680, 0.8710, 0.8102, 0.7842, 0.7626, 0.7308, 0.7024, 0.6950/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7682, 0.7480, 0.7593, 0.7757, 0.8156, 0.8291, 0.8615, 0.8870, 0.8873, 
     &  0.8760, 0.8605, 0.8631, 0.7953, 0.7618, 0.7345, 0.6981, 0.6657, 0.6569/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7153, 0.6966, 0.7120, 0.7386, 0.7861, 0.8074, 0.8524, 0.8806, 0.8813, 
     &  0.8684, 0.8510, 0.8529, 0.7779, 0.7370, 0.7031, 0.6614, 0.6252, 0.6150/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6543, 0.6380, 0.6591, 0.6960, 0.7506, 0.7802, 0.8390, 0.8711, 0.8723, 
     &  0.8573, 0.8380, 0.8385, 0.7561, 0.7078, 0.6671, 0.6204, 0.5809, 0.5694/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5904, 0.5766, 0.6033, 0.6496, 0.7102, 0.7474, 0.8199, 0.8572, 0.8597, 
     &  0.8423, 0.8206, 0.8187, 0.7296, 0.6745, 0.6276, 0.5765, 0.5345, 0.5221/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5280, 0.5165, 0.5468, 0.6002, 0.6650, 0.7082, 0.7917, 0.8354, 0.8402, 
     &  0.8209, 0.7967, 0.7909, 0.6968, 0.6361, 0.5845, 0.5305, 0.4873, 0.4744/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4683, 0.4587, 0.4906, 0.5485, 0.6159, 0.6637, 0.7542, 0.8045, 0.8121, 
     &  0.7915, 0.7655, 0.7547, 0.6580, 0.5937, 0.5396, 0.4841, 0.4409, 0.4280/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4119, 0.4040, 0.4362, 0.4964, 0.5647, 0.6155, 0.7091, 0.7654, 0.7756, 
     &  0.7543, 0.7272, 0.7115, 0.6147, 0.5496, 0.4951, 0.4397, 0.3974, 0.3846/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3618, 0.3554, 0.3859, 0.4455, 0.5127, 0.5644, 0.6582, 0.7194, 0.7318, 
     &  0.7102, 0.6826, 0.6633, 0.5685, 0.5051, 0.4522, 0.3984, 0.3578, 0.3455/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3180, 0.3126, 0.3393, 0.3955, 0.4603, 0.5112, 0.6031, 0.6682, 0.6822, 
     &  0.6606, 0.6332, 0.6115, 0.5204, 0.4604, 0.4103, 0.3592, 0.3210, 0.3095/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2797, 0.2750, 0.2972, 0.3485, 0.4097, 0.4582, 0.5455, 0.6115, 0.6267, 
     &  0.6058, 0.5794, 0.5569, 0.4711, 0.4159, 0.3699, 0.3222, 0.2869, 0.2762/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2459, 0.2419, 0.2598, 0.3056, 0.3620, 0.4069, 0.4868, 0.5498, 0.5653, 
     &  0.5460, 0.5215, 0.4996, 0.4210, 0.3716, 0.3306, 0.2872, 0.2552, 0.2458/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2163, 0.2127, 0.2269, 0.2666, 0.3176, 0.3580, 0.4289, 0.4861, 0.5012, 
     &  0.4838, 0.4618, 0.4416, 0.3710, 0.3281, 0.2927, 0.2540, 0.2260, 0.2181/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1903, 0.1872, 0.1981, 0.2319, 0.2775, 0.3132, 0.3746, 0.4238, 0.4366, 
     &  0.4216, 0.4029, 0.3852, 0.3232, 0.2866, 0.2567, 0.2229, 0.1992, 0.1927/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1675, 0.1648, 0.1734, 0.2020, 0.2422, 0.2730, 0.3246, 0.3645, 0.3737, 
     &  0.3612, 0.3464, 0.3316, 0.2784, 0.2477, 0.2230, 0.1941, 0.1746, 0.1695/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1475, 0.1451, 0.1520, 0.1762, 0.2112, 0.2370, 0.2793, 0.3101, 0.3152, 
     &  0.3051, 0.2940, 0.2821, 0.2374, 0.2121, 0.1920, 0.1678, 0.1521, 0.1483/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1299, 0.1278, 0.1334, 0.1540, 0.1841, 0.2051, 0.2390, 0.2619, 0.2633, 
     &  0.2552, 0.2474, 0.2382, 0.2010, 0.1802, 0.1639, 0.1440, 0.1316, 0.1289/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1144, 0.1126, 0.1172, 0.1349, 0.1604, 0.1769, 0.2033, 0.2196, 0.2182, 
     &  0.2116, 0.2064, 0.1995, 0.1688, 0.1517, 0.1385, 0.1224, 0.1129, 0.1111/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1007, 0.0990, 0.1029, 0.1182, 0.1396, 0.1517, 0.1717, 0.1827, 0.1800, 
     &  0.1745, 0.1707, 0.1656, 0.1404, 0.1265, 0.1159, 0.1030, 0.0959, 0.0948/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0882, 0.0867, 0.0901, 0.1034, 0.1209, 0.1293, 0.1438, 0.1511, 0.1484, 
     &  0.1433, 0.1402, 0.1364, 0.1158, 0.1043, 0.0959, 0.0858, 0.0807, 0.0801/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0767, 0.0754, 0.0785, 0.0900, 0.1042, 0.1093, 0.1195, 0.1242, 0.1217, 
     &  0.1170, 0.1141, 0.1112, 0.0944, 0.0850, 0.0783, 0.0706, 0.0672, 0.0669/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0663, 0.0652, 0.0680, 0.0780, 0.0892, 0.0918, 0.0985, 0.1014, 0.0992, 
     &  0.0948, 0.0921, 0.0898, 0.0761, 0.0685, 0.0633, 0.0575, 0.0552, 0.0552/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0569, 0.0559, 0.0584, 0.0671, 0.0758, 0.0764, 0.0805, 0.0822, 0.0802, 
     &  0.0762, 0.0737, 0.0719, 0.0608, 0.0546, 0.0506, 0.0463, 0.0450, 0.0450/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0485, 0.0476, 0.0497, 0.0573, 0.0638, 0.0629, 0.0652, 0.0661, 0.0642, 
     &  0.0608, 0.0585, 0.0571, 0.0482, 0.0431, 0.0400, 0.0369, 0.0362, 0.0363/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0409, 0.0401, 0.0419, 0.0484, 0.0531, 0.0512, 0.0522, 0.0527, 0.0511, 
     &  0.0481, 0.0462, 0.0450, 0.0379, 0.0338, 0.0314, 0.0292, 0.0289, 0.0290/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0342, 0.0334, 0.0349, 0.0404, 0.0436, 0.0411, 0.0413, 0.0415, 0.0402, 
     &  0.0379, 0.0363, 0.0353, 0.0296, 0.0263, 0.0244, 0.0230, 0.0229, 0.0229/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0283, 0.0276, 0.0288, 0.0333, 0.0353, 0.0326, 0.0324, 0.0326, 0.0316, 
     &  0.0297, 0.0284, 0.0277, 0.0232, 0.0205, 0.0190, 0.0180, 0.0180, 0.0179/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0233, 0.0226, 0.0236, 0.0272, 0.0282, 0.0255, 0.0252, 0.0254, 0.0247, 
     &  0.0233, 0.0222, 0.0216, 0.0181, 0.0160, 0.0148, 0.0142, 0.0141, 0.0139/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0189, 0.0183, 0.0191, 0.0219, 0.0222, 0.0198, 0.0195, 0.0198, 0.0193, 
     &  0.0182, 0.0174, 0.0169, 0.0142, 0.0125, 0.0116, 0.0112, 0.0110, 0.0108/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0152, 0.0147, 0.0153, 0.0173, 0.0172, 0.0152, 0.0151, 0.0154, 0.0151, 
     &  0.0142, 0.0136, 0.0133, 0.0111, 0.0098, 0.0091, 0.0088, 0.0086, 0.0084/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0122, 0.0117, 0.0121, 0.0135, 0.0132, 0.0116, 0.0117, 0.0120, 0.0118, 
     &  0.0111, 0.0106, 0.0104, 0.0088, 0.0077, 0.0071, 0.0069, 0.0067, 0.0065/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0096, 0.0092, 0.0095, 0.0104, 0.0100, 0.0089, 0.0090, 0.0094, 0.0092, 
     &  0.0087, 0.0083, 0.0081, 0.0069, 0.0061, 0.0056, 0.0055, 0.0052, 0.0050/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0075, 0.0071, 0.0074, 0.0078, 0.0075, 0.0068, 0.0070, 0.0073, 0.0072, 
     &  0.0068, 0.0065, 0.0064, 0.0054, 0.0048, 0.0044, 0.0043, 0.0040, 0.0039/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0058, 0.0055, 0.0057, 0.0058, 0.0056, 0.0052, 0.0054, 0.0056, 0.0056, 
     &  0.0052, 0.0050, 0.0050, 0.0042, 0.0038, 0.0035, 0.0034, 0.0031, 0.0030/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0045, 0.0043, 0.0044, 0.0043, 0.0042, 0.0040, 0.0042, 0.0044, 0.0043, 
     &  0.0041, 0.0039, 0.0039, 0.0033, 0.0030, 0.0027, 0.0026, 0.0024, 0.0023/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0035, 0.0034, 0.0034, 0.0032, 0.0031, 0.0030, 0.0033, 0.0034, 0.0034, 
     &  0.0032, 0.0031, 0.0030, 0.0026, 0.0023, 0.0021, 0.0021, 0.0019, 0.0018/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0028, 0.0027, 0.0027, 0.0023, 0.0024, 0.0023, 0.0025, 0.0027, 0.0026, 
     &  0.0025, 0.0024, 0.0024, 0.0020, 0.0018, 0.0017, 0.0016, 0.0015, 0.0014/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0022, 0.0022, 0.0021, 0.0017, 0.0018, 0.0018, 0.0020, 0.0021, 0.0020, 
     &  0.0019, 0.0019, 0.0019, 0.0016, 0.0014, 0.0013, 0.0013, 0.0011, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0018, 0.0017, 0.0016, 0.0012, 0.0013, 0.0014, 0.0015, 0.0016, 0.0016, 
     &  0.0015, 0.0014, 0.0014, 0.0012, 0.0011, 0.0010, 0.0010, 0.0009, 0.0009/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0015, 0.0014, 0.0013, 0.0009, 0.0010, 0.0011, 0.0012, 0.0012, 0.0012, 
     &  0.0011, 0.0011, 0.0011, 0.0010, 0.0009, 0.0008, 0.0008, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0012, 0.0012, 0.0010, 0.0007, 0.0008, 0.0008, 0.0009, 0.0010, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0007, 0.0006, 0.0006, 0.0006, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0010, 0.0008, 0.0005, 0.0006, 0.0007, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0008, 0.0006, 0.0004, 0.0005, 0.0005, 0.0006, 0.0006, 0.0006, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0005, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0004, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0003, 0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0002, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0002, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  6 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  7. Data corresponds to middle of JUL.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9873, 0.9889, 0.9892, 0.9924, 0.9910, 0.9882, 0.9895, 0.9883, 
     &  0.9904, 0.9939, 0.9914, 0.9804, 0.9873, 0.9881, 0.9891, 0.9904, 0.9921/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9950, 0.9762, 0.9784, 0.9790, 0.9832, 0.9806, 0.9765, 0.9791, 0.9768, 
     &  0.9800, 0.9848, 0.9818, 0.9622, 0.9729, 0.9757, 0.9770, 0.9794, 0.9831/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9858, 0.9662, 0.9689, 0.9696, 0.9743, 0.9706, 0.9644, 0.9686, 0.9652, 
     &  0.9693, 0.9746, 0.9721, 0.9465, 0.9589, 0.9634, 0.9643, 0.9678, 0.9732/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9773, 0.9571, 0.9603, 0.9612, 0.9663, 0.9614, 0.9524, 0.9581, 0.9539, 
     &  0.9590, 0.9642, 0.9621, 0.9324, 0.9456, 0.9515, 0.9520, 0.9562, 0.9630/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9696, 0.9489, 0.9525, 0.9537, 0.9591, 0.9531, 0.9409, 0.9483, 0.9432, 
     &  0.9492, 0.9547, 0.9523, 0.9194, 0.9328, 0.9399, 0.9401, 0.9449, 0.9531/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9628, 0.9415, 0.9454, 0.9470, 0.9527, 0.9455, 0.9301, 0.9393, 0.9334, 
     &  0.9400, 0.9465, 0.9428, 0.9073, 0.9208, 0.9290, 0.9289, 0.9339, 0.9434/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9567, 0.9347, 0.9390, 0.9409, 0.9469, 0.9386, 0.9202, 0.9313, 0.9248, 
     &  0.9316, 0.9394, 0.9342, 0.8963, 0.9097, 0.9189, 0.9184, 0.9230, 0.9335/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9511, 0.9283, 0.9330, 0.9348, 0.9414, 0.9319, 0.9113, 0.9243, 0.9175, 
     &  0.9241, 0.9331, 0.9266, 0.8863, 0.8994, 0.9095, 0.9082, 0.9107, 0.9216/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9448, 0.9211, 0.9264, 0.9274, 0.9349, 0.9243, 0.9035, 0.9186, 0.9112, 
     &  0.9175, 0.9275, 0.9199, 0.8774, 0.8893, 0.8989, 0.8962, 0.8932, 0.9039/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9362, 0.9116, 0.9171, 0.9167, 0.9255, 0.9146, 0.8966, 0.9136, 0.9057, 
     &  0.9116, 0.9224, 0.9141, 0.8690, 0.8781, 0.8843, 0.8790, 0.8680, 0.8771/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9236, 0.8982, 0.9036, 0.9014, 0.9118, 0.9014, 0.8901, 0.9093, 0.9010, 
     &  0.9062, 0.9178, 0.9090, 0.8608, 0.8650, 0.8648, 0.8551, 0.8369, 0.8438/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9055, 0.8797, 0.8845, 0.8811, 0.8937, 0.8850, 0.8838, 0.9053, 0.8967, 
     &  0.9012, 0.9134, 0.9044, 0.8526, 0.8501, 0.8424, 0.8277, 0.8048, 0.8102/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8821, 0.8563, 0.8599, 0.8565, 0.8729, 0.8685, 0.8775, 0.9013, 0.8927, 
     &  0.8965, 0.9088, 0.8999, 0.8440, 0.8346, 0.8201, 0.8011, 0.7746, 0.7786/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8530, 0.8271, 0.8299, 0.8286, 0.8504, 0.8520, 0.8710, 0.8970, 0.8887, 
     &  0.8918, 0.9037, 0.8948, 0.8348, 0.8187, 0.7981, 0.7746, 0.7446, 0.7470/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8162, 0.7907, 0.7934, 0.7970, 0.8254, 0.8344, 0.8641, 0.8923, 0.8844, 
     &  0.8867, 0.8979, 0.8889, 0.8244, 0.8012, 0.7740, 0.7459, 0.7121, 0.7125/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7720, 0.7475, 0.7513, 0.7619, 0.7979, 0.8154, 0.8566, 0.8871, 0.8795, 
     &  0.8806, 0.8907, 0.8816, 0.8118, 0.7821, 0.7477, 0.7143, 0.6765, 0.6747/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7202, 0.6971, 0.7041, 0.7226, 0.7664, 0.7930, 0.8468, 0.8801, 0.8725, 
     &  0.8717, 0.8804, 0.8710, 0.7959, 0.7599, 0.7180, 0.6787, 0.6369, 0.6329/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6602, 0.6392, 0.6516, 0.6781, 0.7294, 0.7653, 0.8324, 0.8693, 0.8619, 
     &  0.8587, 0.8656, 0.8555, 0.7756, 0.7328, 0.6835, 0.6382, 0.5925, 0.5865/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5967, 0.5782, 0.5964, 0.6303, 0.6879, 0.7318, 0.8114, 0.8532, 0.8472, 
     &  0.8418, 0.8463, 0.8346, 0.7504, 0.7010, 0.6453, 0.5944, 0.5456, 0.5379/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5347, 0.5185, 0.5408, 0.5797, 0.6416, 0.6916, 0.7806, 0.8285, 0.8259, 
     &  0.8187, 0.8210, 0.8068, 0.7188, 0.6638, 0.6033, 0.5484, 0.4977, 0.4889/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4761, 0.4619, 0.4858, 0.5271, 0.5916, 0.6456, 0.7400, 0.7945, 0.7970, 
     &  0.7884, 0.7887, 0.7712, 0.6809, 0.6222, 0.5592, 0.5021, 0.4511, 0.4415/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4216, 0.4090, 0.4330, 0.4745, 0.5400, 0.5959, 0.6921, 0.7532, 0.7606, 
     &  0.7511, 0.7491, 0.7287, 0.6380, 0.5782, 0.5151, 0.4579, 0.4077, 0.3979/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3740, 0.3625, 0.3844, 0.4238, 0.4884, 0.5440, 0.6392, 0.7059, 0.7172, 
     &  0.7071, 0.7030, 0.6806, 0.5916, 0.5329, 0.4719, 0.4166, 0.3685, 0.3589/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3331, 0.3223, 0.3399, 0.3749, 0.4374, 0.4910, 0.5830, 0.6540, 0.6679, 
     &  0.6574, 0.6519, 0.6285, 0.5428, 0.4866, 0.4291, 0.3771, 0.3323, 0.3231/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2973, 0.2871, 0.3000, 0.3297, 0.3890, 0.4392, 0.5259, 0.5977, 0.6129, 
     &  0.6025, 0.5965, 0.5732, 0.4924, 0.4398, 0.3869, 0.3391, 0.2984, 0.2900/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2651, 0.2555, 0.2647, 0.2893, 0.3440, 0.3898, 0.4691, 0.5374, 0.5527, 
     &  0.5430, 0.5373, 0.5151, 0.4407, 0.3927, 0.3453, 0.3028, 0.2668, 0.2593/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2359, 0.2270, 0.2332, 0.2530, 0.3023, 0.3431, 0.4139, 0.4760, 0.4903, 
     &  0.4816, 0.4764, 0.4562, 0.3891, 0.3463, 0.3050, 0.2682, 0.2372, 0.2311/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2098, 0.2016, 0.2054, 0.2210, 0.2646, 0.3003, 0.3621, 0.4163, 0.4280, 
     &  0.4206, 0.4167, 0.3989, 0.3395, 0.3023, 0.2669, 0.2355, 0.2096, 0.2048/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1860, 0.1785, 0.1810, 0.1934, 0.2311, 0.2616, 0.3145, 0.3595, 0.3675, 
     &  0.3618, 0.3594, 0.3445, 0.2930, 0.2613, 0.2315, 0.2053, 0.1839, 0.1805/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1640, 0.1575, 0.1594, 0.1696, 0.2014, 0.2268, 0.2708, 0.3068, 0.3115, 
     &  0.3071, 0.3062, 0.2940, 0.2503, 0.2238, 0.1992, 0.1776, 0.1604, 0.1582/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1440, 0.1383, 0.1403, 0.1487, 0.1751, 0.1956, 0.2315, 0.2598, 0.2621, 
     &  0.2585, 0.2585, 0.2489, 0.2122, 0.1903, 0.1702, 0.1525, 0.1389, 0.1376/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1257, 0.1210, 0.1232, 0.1305, 0.1520, 0.1680, 0.1966, 0.2184, 0.2191, 
     &  0.2158, 0.2163, 0.2090, 0.1785, 0.1606, 0.1442, 0.1299, 0.1193, 0.1188/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1092, 0.1053, 0.1078, 0.1143, 0.1315, 0.1435, 0.1659, 0.1825, 0.1822, 
     &  0.1790, 0.1795, 0.1739, 0.1490, 0.1343, 0.1209, 0.1095, 0.1016, 0.1016/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0943, 0.0911, 0.0937, 0.0997, 0.1132, 0.1218, 0.1391, 0.1517, 0.1510, 
     &  0.1476, 0.1478, 0.1436, 0.1232, 0.1112, 0.1003, 0.0914, 0.0857, 0.0861/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0807, 0.0781, 0.0809, 0.0864, 0.0968, 0.1026, 0.1158, 0.1254, 0.1242, 
     &  0.1207, 0.1206, 0.1174, 0.1008, 0.0910, 0.0822, 0.0754, 0.0714, 0.0721/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0683, 0.0663, 0.0691, 0.0743, 0.0822, 0.0858, 0.0957, 0.1029, 0.1012, 
     &  0.0978, 0.0976, 0.0951, 0.0817, 0.0737, 0.0667, 0.0615, 0.0589, 0.0597/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0572, 0.0557, 0.0585, 0.0633, 0.0691, 0.0711, 0.0783, 0.0836, 0.0818, 
     &  0.0786, 0.0783, 0.0763, 0.0656, 0.0590, 0.0534, 0.0496, 0.0480, 0.0489/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0475, 0.0464, 0.0489, 0.0533, 0.0575, 0.0582, 0.0634, 0.0673, 0.0655, 
     &  0.0626, 0.0622, 0.0607, 0.0521, 0.0468, 0.0424, 0.0396, 0.0388, 0.0396/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0390, 0.0382, 0.0405, 0.0443, 0.0472, 0.0471, 0.0507, 0.0535, 0.0519, 
     &  0.0494, 0.0491, 0.0480, 0.0412, 0.0368, 0.0333, 0.0314, 0.0311, 0.0318/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0317, 0.0311, 0.0331, 0.0364, 0.0382, 0.0375, 0.0401, 0.0420, 0.0407, 
     &  0.0387, 0.0385, 0.0377, 0.0323, 0.0288, 0.0260, 0.0247, 0.0247, 0.0252/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0256, 0.0252, 0.0269, 0.0295, 0.0305, 0.0296, 0.0314, 0.0328, 0.0317, 
     &  0.0302, 0.0301, 0.0295, 0.0253, 0.0225, 0.0203, 0.0194, 0.0195, 0.0199/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0206, 0.0202, 0.0217, 0.0236, 0.0241, 0.0232, 0.0245, 0.0256, 0.0247, 
     &  0.0235, 0.0235, 0.0231, 0.0198, 0.0175, 0.0158, 0.0152, 0.0154, 0.0156/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0165, 0.0162, 0.0173, 0.0187, 0.0188, 0.0180, 0.0190, 0.0198, 0.0192, 
     &  0.0183, 0.0183, 0.0180, 0.0155, 0.0137, 0.0124, 0.0119, 0.0120, 0.0121/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0131, 0.0129, 0.0137, 0.0146, 0.0145, 0.0139, 0.0147, 0.0154, 0.0149, 
     &  0.0142, 0.0142, 0.0141, 0.0122, 0.0108, 0.0097, 0.0094, 0.0094, 0.0094/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0104, 0.0102, 0.0108, 0.0113, 0.0112, 0.0108, 0.0114, 0.0120, 0.0116, 
     &  0.0111, 0.0111, 0.0110, 0.0095, 0.0085, 0.0076, 0.0074, 0.0074, 0.0073/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0082, 0.0080, 0.0085, 0.0086, 0.0085, 0.0083, 0.0089, 0.0093, 0.0090, 
     &  0.0086, 0.0086, 0.0086, 0.0075, 0.0067, 0.0060, 0.0058, 0.0057, 0.0056/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0064, 0.0063, 0.0066, 0.0065, 0.0065, 0.0064, 0.0069, 0.0072, 0.0070, 
     &  0.0067, 0.0067, 0.0067, 0.0059, 0.0052, 0.0047, 0.0046, 0.0044, 0.0043/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0051, 0.0049, 0.0051, 0.0048, 0.0049, 0.0050, 0.0053, 0.0056, 0.0054, 
     &  0.0052, 0.0052, 0.0052, 0.0046, 0.0041, 0.0037, 0.0036, 0.0034, 0.0034/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0040, 0.0039, 0.0039, 0.0036, 0.0037, 0.0038, 0.0042, 0.0044, 0.0042, 
     &  0.0040, 0.0040, 0.0041, 0.0036, 0.0032, 0.0029, 0.0028, 0.0027, 0.0026/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0032, 0.0031, 0.0030, 0.0027, 0.0028, 0.0030, 0.0032, 0.0034, 0.0033, 
     &  0.0031, 0.0031, 0.0032, 0.0028, 0.0025, 0.0023, 0.0022, 0.0021, 0.0020/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0026, 0.0025, 0.0023, 0.0020, 0.0022, 0.0023, 0.0025, 0.0026, 0.0025, 
     &  0.0024, 0.0024, 0.0025, 0.0022, 0.0020, 0.0018, 0.0017, 0.0016, 0.0016/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0021, 0.0021, 0.0018, 0.0015, 0.0017, 0.0018, 0.0020, 0.0020, 0.0020, 
     &  0.0019, 0.0019, 0.0019, 0.0017, 0.0016, 0.0014, 0.0014, 0.0013, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0017, 0.0017, 0.0014, 0.0011, 0.0013, 0.0014, 0.0015, 0.0016, 0.0015, 
     &  0.0014, 0.0014, 0.0015, 0.0013, 0.0012, 0.0011, 0.0011, 0.0010, 0.0010/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0014, 0.0014, 0.0011, 0.0009, 0.0010, 0.0011, 0.0012, 0.0012, 0.0012, 
     &  0.0011, 0.0011, 0.0011, 0.0011, 0.0010, 0.0009, 0.0008, 0.0008, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0012, 0.0012, 0.0009, 0.0007, 0.0008, 0.0009, 0.0009, 0.0010, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0008, 0.0007, 0.0007, 0.0006, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0010, 0.0007, 0.0005, 0.0006, 0.0007, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0008, 0.0005, 0.0004, 0.0005, 0.0005, 0.0006, 0.0006, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0004, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0003, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0002, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  7 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  8. Data corresponds to middle of AUG.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9868, 0.9878, 0.9897, 0.9927, 0.9908, 0.9881, 0.9897, 0.9885, 
     &  0.9907, 0.9937, 0.9900, 0.9812, 0.9873, 0.9880, 0.9887, 0.9896, 0.9902/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9936, 0.9744, 0.9764, 0.9797, 0.9838, 0.9807, 0.9751, 0.9790, 0.9764, 
     &  0.9809, 0.9846, 0.9802, 0.9631, 0.9727, 0.9752, 0.9765, 0.9780, 0.9794/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9823, 0.9630, 0.9660, 0.9704, 0.9751, 0.9709, 0.9610, 0.9680, 0.9635, 
     &  0.9702, 0.9745, 0.9702, 0.9474, 0.9587, 0.9624, 0.9640, 0.9660, 0.9679/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9724, 0.9528, 0.9567, 0.9620, 0.9671, 0.9618, 0.9472, 0.9569, 0.9513, 
     &  0.9595, 0.9641, 0.9599, 0.9334, 0.9454, 0.9501, 0.9518, 0.9541, 0.9565/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9635, 0.9434, 0.9482, 0.9543, 0.9599, 0.9534, 0.9340, 0.9461, 0.9401, 
     &  0.9499, 0.9548, 0.9498, 0.9207, 0.9327, 0.9383, 0.9400, 0.9427, 0.9457/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9555, 0.9350, 0.9405, 0.9472, 0.9534, 0.9457, 0.9222, 0.9366, 0.9301, 
     &  0.9410, 0.9467, 0.9403, 0.9091, 0.9211, 0.9274, 0.9291, 0.9317, 0.9356/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9482, 0.9273, 0.9335, 0.9407, 0.9474, 0.9384, 0.9114, 0.9284, 0.9213, 
     &  0.9328, 0.9393, 0.9316, 0.8984, 0.9105, 0.9172, 0.9187, 0.9208, 0.9258/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9415, 0.9198, 0.9268, 0.9345, 0.9416, 0.9310, 0.9015, 0.9214, 0.9140, 
     &  0.9256, 0.9329, 0.9238, 0.8887, 0.9009, 0.9080, 0.9088, 0.9090, 0.9147/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9343, 0.9114, 0.9192, 0.9269, 0.9344, 0.9225, 0.8928, 0.9154, 0.9077, 
     &  0.9192, 0.9273, 0.9172, 0.8799, 0.8914, 0.8983, 0.8977, 0.8935, 0.8987/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9250, 0.9002, 0.9088, 0.9156, 0.9235, 0.9114, 0.8850, 0.9103, 0.9023, 
     &  0.9137, 0.9226, 0.9115, 0.8719, 0.8814, 0.8860, 0.8825, 0.8716, 0.8750/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9115, 0.8846, 0.8939, 0.8990, 0.9075, 0.8965, 0.8780, 0.9059, 0.8975, 
     &  0.9087, 0.9184, 0.9064, 0.8643, 0.8700, 0.8701, 0.8621, 0.8444, 0.8459/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8917, 0.8633, 0.8726, 0.8770, 0.8875, 0.8797, 0.8712, 0.9019, 0.8933, 
     &  0.9043, 0.9145, 0.9018, 0.8570, 0.8577, 0.8518, 0.8393, 0.8161, 0.8163/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8658, 0.8367, 0.8457, 0.8516, 0.8660, 0.8631, 0.8648, 0.8979, 0.8894, 
     &  0.9001, 0.9106, 0.8974, 0.8497, 0.8451, 0.8332, 0.8166, 0.7886, 0.7872/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8344, 0.8054, 0.8140, 0.8239, 0.8437, 0.8465, 0.8584, 0.8938, 0.8856, 
     &  0.8958, 0.9065, 0.8928, 0.8420, 0.8317, 0.8133, 0.7924, 0.7596, 0.7562/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7972, 0.7688, 0.7771, 0.7931, 0.8190, 0.8288, 0.8514, 0.8894, 0.8817, 
     &  0.8910, 0.9016, 0.8875, 0.8329, 0.8157, 0.7902, 0.7642, 0.7264, 0.7208/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7551, 0.7277, 0.7362, 0.7592, 0.7917, 0.8091, 0.8436, 0.8842, 0.8770, 
     &  0.8853, 0.8952, 0.8810, 0.8218, 0.7978, 0.7643, 0.7325, 0.6897, 0.6815/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7082, 0.6823, 0.6920, 0.7215, 0.7606, 0.7858, 0.8333, 0.8767, 0.8697, 
     &  0.8764, 0.8855, 0.8710, 0.8072, 0.7766, 0.7350, 0.6968, 0.6488, 0.6381/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6553, 0.6314, 0.6443, 0.6785, 0.7239, 0.7566, 0.8177, 0.8647, 0.8584, 
     &  0.8629, 0.8709, 0.8558, 0.7877, 0.7502, 0.7003, 0.6557, 0.6025, 0.5894/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5994, 0.5775, 0.5946, 0.6317, 0.6823, 0.7214, 0.7948, 0.8469, 0.8429, 
     &  0.8451, 0.8517, 0.8354, 0.7630, 0.7187, 0.6615, 0.6110, 0.5534, 0.5385/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5449, 0.5250, 0.5449, 0.5814, 0.6359, 0.6797, 0.7620, 0.8207, 0.8210, 
     &  0.8214, 0.8263, 0.8086, 0.7321, 0.6818, 0.6192, 0.5643, 0.5038, 0.4875/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4939, 0.4758, 0.4961, 0.5290, 0.5860, 0.6327, 0.7198, 0.7860, 0.7917, 
     &  0.7907, 0.7940, 0.7743, 0.6951, 0.6406, 0.5750, 0.5176, 0.4557, 0.4387/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4470, 0.4307, 0.4496, 0.4770, 0.5350, 0.5828, 0.6711, 0.7447, 0.7552, 
     &  0.7532, 0.7545, 0.7331, 0.6529, 0.5967, 0.5309, 0.4729, 0.4110, 0.3937/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4058, 0.3910, 0.4067, 0.4279, 0.4848, 0.5317, 0.6184, 0.6980, 0.7120, 
     &  0.7091, 0.7086, 0.6859, 0.6070, 0.5512, 0.4877, 0.4310, 0.3706, 0.3535/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3697, 0.3562, 0.3673, 0.3812, 0.4357, 0.4804, 0.5635, 0.6469, 0.6637, 
     &  0.6597, 0.6577, 0.6346, 0.5581, 0.5042, 0.4446, 0.3909, 0.3332, 0.3166/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3369, 0.3246, 0.3313, 0.3384, 0.3892, 0.4309, 0.5087, 0.5922, 0.6105, 
     &  0.6057, 0.6025, 0.5797, 0.5071, 0.4561, 0.4014, 0.3524, 0.2984, 0.2827/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3057, 0.2945, 0.2982, 0.3002, 0.3459, 0.3837, 0.4548, 0.5341, 0.5524, 
     &  0.5472, 0.5434, 0.5217, 0.4546, 0.4077, 0.3588, 0.3155, 0.2663, 0.2516/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2756, 0.2655, 0.2674, 0.2657, 0.3055, 0.3388, 0.4025, 0.4749, 0.4918, 
     &  0.4867, 0.4827, 0.4629, 0.4021, 0.3601, 0.3174, 0.2805, 0.2366, 0.2233/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2467, 0.2376, 0.2388, 0.2347, 0.2683, 0.2970, 0.3532, 0.4170, 0.4308, 
     &  0.4262, 0.4230, 0.4055, 0.3515, 0.3150, 0.2785, 0.2476, 0.2093, 0.1974/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2187, 0.2107, 0.2124, 0.2070, 0.2347, 0.2588, 0.3073, 0.3614, 0.3711, 
     &  0.3675, 0.3656, 0.3508, 0.3039, 0.2730, 0.2425, 0.2172, 0.1843, 0.1740/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1921, 0.1850, 0.1880, 0.1824, 0.2044, 0.2239, 0.2650, 0.3092, 0.3152, 
     &  0.3125, 0.3119, 0.2999, 0.2600, 0.2346, 0.2097, 0.1893, 0.1615, 0.1529/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1672, 0.1611, 0.1655, 0.1601, 0.1773, 0.1927, 0.2268, 0.2621, 0.2653, 
     &  0.2632, 0.2635, 0.2542, 0.2208, 0.2002, 0.1802, 0.1640, 0.1407, 0.1337/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1446, 0.1393, 0.1450, 0.1400, 0.1532, 0.1651, 0.1929, 0.2202, 0.2213, 
     &  0.2194, 0.2203, 0.2137, 0.1862, 0.1697, 0.1536, 0.1409, 0.1219, 0.1163/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1244, 0.1199, 0.1262, 0.1218, 0.1318, 0.1408, 0.1630, 0.1837, 0.1832, 
     &  0.1813, 0.1824, 0.1780, 0.1558, 0.1427, 0.1298, 0.1199, 0.1047, 0.1005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1064, 0.1026, 0.1090, 0.1051, 0.1127, 0.1192, 0.1367, 0.1524, 0.1509, 
     &  0.1487, 0.1497, 0.1470, 0.1293, 0.1188, 0.1086, 0.1011, 0.0892, 0.0862/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0905, 0.0872, 0.0933, 0.0897, 0.0956, 0.1002, 0.1138, 0.1254, 0.1231, 
     &  0.1207, 0.1217, 0.1203, 0.1063, 0.0979, 0.0898, 0.0843, 0.0752, 0.0732/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0766, 0.0738, 0.0792, 0.0759, 0.0803, 0.0834, 0.0937, 0.1023, 0.0996, 
     &  0.0970, 0.0981, 0.0976, 0.0865, 0.0799, 0.0734, 0.0694, 0.0627, 0.0616/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0644, 0.0620, 0.0666, 0.0634, 0.0668, 0.0687, 0.0763, 0.0826, 0.0798, 
     &  0.0773, 0.0783, 0.0784, 0.0697, 0.0644, 0.0593, 0.0565, 0.0518, 0.0513/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0538, 0.0518, 0.0555, 0.0523, 0.0548, 0.0558, 0.0613, 0.0659, 0.0634, 
     &  0.0612, 0.0620, 0.0624, 0.0557, 0.0514, 0.0473, 0.0455, 0.0423, 0.0423/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0445, 0.0429, 0.0459, 0.0428, 0.0444, 0.0449, 0.0487, 0.0520, 0.0499, 
     &  0.0481, 0.0488, 0.0493, 0.0441, 0.0406, 0.0374, 0.0362, 0.0342, 0.0345/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0366, 0.0352, 0.0375, 0.0345, 0.0356, 0.0356, 0.0383, 0.0406, 0.0390, 
     &  0.0375, 0.0381, 0.0387, 0.0346, 0.0318, 0.0293, 0.0285, 0.0274, 0.0279/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0299, 0.0288, 0.0304, 0.0276, 0.0282, 0.0280, 0.0299, 0.0316, 0.0303, 
     &  0.0292, 0.0297, 0.0302, 0.0271, 0.0249, 0.0229, 0.0223, 0.0217, 0.0223/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0243, 0.0234, 0.0245, 0.0219, 0.0222, 0.0219, 0.0233, 0.0245, 0.0236, 
     &  0.0227, 0.0231, 0.0235, 0.0212, 0.0194, 0.0178, 0.0174, 0.0172, 0.0177/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0196, 0.0188, 0.0196, 0.0172, 0.0173, 0.0170, 0.0181, 0.0191, 0.0183, 
     &  0.0177, 0.0180, 0.0183, 0.0165, 0.0152, 0.0139, 0.0136, 0.0135, 0.0140/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0156, 0.0151, 0.0154, 0.0134, 0.0135, 0.0132, 0.0140, 0.0148, 0.0143, 
     &  0.0138, 0.0140, 0.0142, 0.0129, 0.0119, 0.0109, 0.0106, 0.0106, 0.0110/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0124, 0.0120, 0.0121, 0.0104, 0.0104, 0.0103, 0.0109, 0.0116, 0.0111, 
     &  0.0107, 0.0109, 0.0111, 0.0101, 0.0093, 0.0085, 0.0082, 0.0083, 0.0085/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0098, 0.0094, 0.0093, 0.0080, 0.0080, 0.0080, 0.0085, 0.0090, 0.0087, 
     &  0.0084, 0.0085, 0.0086, 0.0079, 0.0073, 0.0066, 0.0064, 0.0065, 0.0066/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0076, 0.0073, 0.0071, 0.0060, 0.0061, 0.0062, 0.0066, 0.0070, 0.0068, 
     &  0.0065, 0.0066, 0.0067, 0.0061, 0.0057, 0.0052, 0.0050, 0.0050, 0.0051/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0059, 0.0057, 0.0054, 0.0046, 0.0047, 0.0048, 0.0051, 0.0055, 0.0053, 
     &  0.0051, 0.0051, 0.0052, 0.0048, 0.0045, 0.0041, 0.0039, 0.0039, 0.0039/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0045, 0.0044, 0.0040, 0.0034, 0.0036, 0.0037, 0.0040, 0.0043, 0.0041, 
     &  0.0039, 0.0039, 0.0040, 0.0037, 0.0035, 0.0032, 0.0030, 0.0030, 0.0030/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0035, 0.0034, 0.0030, 0.0026, 0.0027, 0.0029, 0.0031, 0.0033, 0.0032, 
     &  0.0030, 0.0030, 0.0031, 0.0029, 0.0027, 0.0025, 0.0024, 0.0024, 0.0023/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0028, 0.0027, 0.0023, 0.0020, 0.0021, 0.0022, 0.0024, 0.0026, 0.0025, 
     &  0.0024, 0.0024, 0.0024, 0.0022, 0.0021, 0.0020, 0.0019, 0.0018, 0.0018/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0022, 0.0021, 0.0018, 0.0015, 0.0016, 0.0017, 0.0019, 0.0020, 0.0019, 
     &  0.0018, 0.0018, 0.0018, 0.0017, 0.0017, 0.0015, 0.0015, 0.0014, 0.0014/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0017, 0.0017, 0.0013, 0.0011, 0.0012, 0.0013, 0.0014, 0.0016, 0.0015, 
     &  0.0014, 0.0014, 0.0014, 0.0014, 0.0013, 0.0012, 0.0012, 0.0011, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0014, 0.0013, 0.0010, 0.0009, 0.0010, 0.0010, 0.0011, 0.0012, 0.0012, 
     &  0.0011, 0.0011, 0.0011, 0.0011, 0.0010, 0.0010, 0.0009, 0.0009, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0011, 0.0010, 0.0008, 0.0007, 0.0007, 0.0008, 0.0009, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0008, 0.0007, 0.0007, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0009, 0.0008, 0.0006, 0.0005, 0.0006, 0.0006, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0006, 0.0006, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0007, 0.0005, 0.0004, 0.0004, 0.0005, 0.0005, 0.0006, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0004, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0003, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0002, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  8 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals  9. Data corresponds to middle of SEP.
        DATA ( FRAC_O3_COLUMN( IILAT,  1,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9830, 0.9883, 0.9904, 0.9929, 0.9908, 0.9883, 0.9898, 0.9878, 
     &  0.9907, 0.9933, 0.9877, 0.9823, 0.9878, 0.9893, 0.9879, 0.9880, 0.9889/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9915, 0.9667, 0.9763, 0.9810, 0.9840, 0.9805, 0.9745, 0.9788, 0.9747, 
     &  0.9807, 0.9830, 0.9760, 0.9657, 0.9742, 0.9771, 0.9758, 0.9757, 0.9773/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9755, 0.9516, 0.9654, 0.9721, 0.9752, 0.9701, 0.9590, 0.9670, 0.9605, 
     &  0.9709, 0.9718, 0.9650, 0.9508, 0.9608, 0.9648, 0.9635, 0.9636, 0.9659/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9602, 0.9379, 0.9556, 0.9639, 0.9669, 0.9600, 0.9445, 0.9552, 0.9473, 
     &  0.9615, 0.9611, 0.9543, 0.9374, 0.9482, 0.9529, 0.9516, 0.9521, 0.9549/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9462, 0.9254, 0.9466, 0.9565, 0.9591, 0.9505, 0.9309, 0.9438, 0.9352, 
     &  0.9521, 0.9511, 0.9443, 0.9255, 0.9365, 0.9417, 0.9403, 0.9412, 0.9445/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9336, 0.9140, 0.9383, 0.9496, 0.9520, 0.9417, 0.9181, 0.9334, 0.9240, 
     &  0.9430, 0.9417, 0.9352, 0.9149, 0.9258, 0.9315, 0.9300, 0.9309, 0.9344/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9222, 0.9033, 0.9308, 0.9430, 0.9455, 0.9336, 0.9063, 0.9240, 0.9143, 
     &  0.9345, 0.9335, 0.9269, 0.9056, 0.9159, 0.9222, 0.9204, 0.9211, 0.9243/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9114, 0.8929, 0.9235, 0.9362, 0.9390, 0.9257, 0.8957, 0.9159, 0.9060, 
     &  0.9269, 0.9263, 0.9196, 0.8975, 0.9069, 0.9134, 0.9111, 0.9105, 0.9128/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8994, 0.8812, 0.9150, 0.9273, 0.9310, 0.9172, 0.8864, 0.9092, 0.8988, 
     &  0.9201, 0.9202, 0.9133, 0.8901, 0.8979, 0.9039, 0.9002, 0.8967, 0.8970/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8840, 0.8662, 0.9033, 0.9145, 0.9195, 0.9064, 0.8782, 0.9036, 0.8927, 
     &  0.9142, 0.9149, 0.9080, 0.8834, 0.8880, 0.8925, 0.8858, 0.8776, 0.8749/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8626, 0.8459, 0.8870, 0.8972, 0.9040, 0.8926, 0.8709, 0.8986, 0.8873, 
     &  0.9089, 0.9103, 0.9034, 0.8768, 0.8771, 0.8784, 0.8673, 0.8540, 0.8486/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8332, 0.8190, 0.8652, 0.8762, 0.8855, 0.8777, 0.8643, 0.8940, 0.8824, 
     &  0.9042, 0.9062, 0.8993, 0.8699, 0.8655, 0.8626, 0.8468, 0.8288, 0.8214/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7974, 0.7867, 0.8391, 0.8527, 0.8657, 0.8629, 0.8578, 0.8897, 0.8780, 
     &  0.8998, 0.9024, 0.8953, 0.8629, 0.8538, 0.8459, 0.8255, 0.8028, 0.7935/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7608, 0.7529, 0.8112, 0.8274, 0.8449, 0.8477, 0.8511, 0.8854, 0.8739, 
     &  0.8955, 0.8984, 0.8911, 0.8553, 0.8407, 0.8271, 0.8018, 0.7739, 0.7625/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7285, 0.7209, 0.7824, 0.7993, 0.8216, 0.8309, 0.8438, 0.8807, 0.8699, 
     &  0.8911, 0.8938, 0.8864, 0.8463, 0.8245, 0.8041, 0.7734, 0.7400, 0.7264/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7044, 0.6930, 0.7529, 0.7683, 0.7956, 0.8114, 0.8355, 0.8751, 0.8654, 
     &  0.8857, 0.8882, 0.8806, 0.8351, 0.8058, 0.7776, 0.7411, 0.7018, 0.6860/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6891, 0.6696, 0.7212, 0.7330, 0.7653, 0.7876, 0.8242, 0.8669, 0.8584, 
     &  0.8776, 0.8794, 0.8716, 0.8204, 0.7835, 0.7467, 0.7040, 0.6588, 0.6407/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6777, 0.6475, 0.6855, 0.6920, 0.7290, 0.7577, 0.8074, 0.8543, 0.8473, 
     &  0.8652, 0.8659, 0.8575, 0.8004, 0.7555, 0.7098, 0.6610, 0.6101, 0.5899/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6659, 0.6247, 0.6467, 0.6467, 0.6875, 0.7216, 0.7837, 0.8361, 0.8317, 
     &  0.8481, 0.8479, 0.8381, 0.7751, 0.7226, 0.6690, 0.6148, 0.5588, 0.5367/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6527, 0.6013, 0.6057, 0.5981, 0.6412, 0.6789, 0.7510, 0.8103, 0.8097, 
     &  0.8246, 0.8238, 0.8121, 0.7436, 0.6847, 0.6253, 0.5673, 0.5076, 0.4841/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6386, 0.5775, 0.5636, 0.5478, 0.5917, 0.6315, 0.7097, 0.7765, 0.7806, 
     &  0.7938, 0.7925, 0.7788, 0.7063, 0.6430, 0.5802, 0.5201, 0.4580, 0.4333/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6225, 0.5531, 0.5222, 0.4983, 0.5414, 0.5820, 0.6624, 0.7363, 0.7447, 
     &  0.7563, 0.7539, 0.7385, 0.6643, 0.5991, 0.5352, 0.4746, 0.4114, 0.3859/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6033, 0.5280, 0.4826, 0.4516, 0.4921, 0.5319, 0.6114, 0.6907, 0.7027, 
     &  0.7124, 0.7086, 0.6922, 0.6184, 0.5538, 0.4910, 0.4317, 0.3689, 0.3433/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5808, 0.5015, 0.4440, 0.4070, 0.4441, 0.4819, 0.5586, 0.6410, 0.6560, 
     &  0.6636, 0.6583, 0.6413, 0.5692, 0.5069, 0.4469, 0.3905, 0.3296, 0.3043/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5509, 0.4710, 0.4058, 0.3651, 0.3983, 0.4335, 0.5059, 0.5880, 0.6046, 
     &  0.6102, 0.6035, 0.5864, 0.5177, 0.4589, 0.4034, 0.3511, 0.2934, 0.2688/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5099, 0.4346, 0.3681, 0.3263, 0.3549, 0.3869, 0.4537, 0.5315, 0.5482, 
     &  0.5522, 0.5446, 0.5282, 0.4644, 0.4107, 0.3609, 0.3140, 0.2605, 0.2369/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4610, 0.3939, 0.3311, 0.2902, 0.3140, 0.3423, 0.4026, 0.4737, 0.4891, 
     &  0.4918, 0.4839, 0.4688, 0.4112, 0.3634, 0.3202, 0.2794, 0.2309, 0.2085/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4097, 0.3516, 0.2952, 0.2563, 0.2759, 0.3007, 0.3541, 0.4168, 0.4291, 
     &  0.4311, 0.4240, 0.4108, 0.3602, 0.3187, 0.2821, 0.2474, 0.2042, 0.1832/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3608, 0.3105, 0.2610, 0.2250, 0.2409, 0.2623, 0.3086, 0.3615, 0.3698, 
     &  0.3715, 0.3663, 0.3555, 0.3122, 0.2773, 0.2470, 0.2183, 0.1804, 0.1609/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3162, 0.2724, 0.2292, 0.1964, 0.2091, 0.2272, 0.2663, 0.3092, 0.3138, 
     &  0.3154, 0.3121, 0.3041, 0.2679, 0.2395, 0.2152, 0.1919, 0.1590, 0.1411/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2759, 0.2377, 0.1999, 0.1704, 0.1805, 0.1955, 0.2279, 0.2614, 0.2634, 
     &  0.2649, 0.2632, 0.2579, 0.2284, 0.2057, 0.1865, 0.1679, 0.1398, 0.1236/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2395, 0.2063, 0.1730, 0.1467, 0.1550, 0.1672, 0.1935, 0.2187, 0.2187, 
     &  0.2200, 0.2195, 0.2169, 0.1933, 0.1756, 0.1606, 0.1460, 0.1224, 0.1080/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2062, 0.1779, 0.1487, 0.1256, 0.1322, 0.1420, 0.1629, 0.1813, 0.1799, 
     &  0.1810, 0.1813, 0.1808, 0.1624, 0.1488, 0.1373, 0.1261, 0.1067, 0.0941/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1758, 0.1522, 0.1268, 0.1066, 0.1120, 0.1196, 0.1360, 0.1493, 0.1471, 
     &  0.1478, 0.1484, 0.1495, 0.1353, 0.1250, 0.1163, 0.1080, 0.0924, 0.0816/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1483, 0.1288, 0.1069, 0.0896, 0.0940, 0.0998, 0.1123, 0.1218, 0.1192, 
     &  0.1194, 0.1204, 0.1224, 0.1116, 0.1040, 0.0976, 0.0916, 0.0794, 0.0703/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1238, 0.1079, 0.0893, 0.0747, 0.0782, 0.0824, 0.0917, 0.0985, 0.0958, 
     &  0.0955, 0.0968, 0.0992, 0.0912, 0.0856, 0.0810, 0.0769, 0.0677, 0.0602/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1023, 0.0896, 0.0739, 0.0616, 0.0643, 0.0673, 0.0740, 0.0789, 0.0764, 
     &  0.0758, 0.0771, 0.0796, 0.0737, 0.0697, 0.0664, 0.0639, 0.0572, 0.0512/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0840, 0.0738, 0.0606, 0.0502, 0.0523, 0.0542, 0.0589, 0.0626, 0.0605, 
     &  0.0598, 0.0610, 0.0632, 0.0590, 0.0560, 0.0537, 0.0523, 0.0478, 0.0432/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0686, 0.0604, 0.0492, 0.0406, 0.0421, 0.0432, 0.0464, 0.0492, 0.0476, 
     &  0.0470, 0.0480, 0.0498, 0.0467, 0.0446, 0.0430, 0.0424, 0.0396, 0.0361/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0557, 0.0490, 0.0396, 0.0324, 0.0334, 0.0340, 0.0362, 0.0384, 0.0372, 
     &  0.0367, 0.0375, 0.0389, 0.0366, 0.0351, 0.0340, 0.0338, 0.0323, 0.0299/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0451, 0.0395, 0.0316, 0.0257, 0.0263, 0.0265, 0.0281, 0.0299, 0.0290, 
     &  0.0287, 0.0292, 0.0302, 0.0285, 0.0275, 0.0266, 0.0267, 0.0261, 0.0245/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0363, 0.0317, 0.0251, 0.0202, 0.0206, 0.0206, 0.0219, 0.0232, 0.0226, 
     &  0.0224, 0.0227, 0.0235, 0.0222, 0.0214, 0.0208, 0.0210, 0.0209, 0.0199/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0290, 0.0252, 0.0197, 0.0158, 0.0160, 0.0160, 0.0170, 0.0181, 0.0176, 
     &  0.0175, 0.0177, 0.0182, 0.0172, 0.0166, 0.0161, 0.0163, 0.0166, 0.0161/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0230, 0.0199, 0.0154, 0.0122, 0.0124, 0.0124, 0.0132, 0.0141, 0.0138, 
     &  0.0136, 0.0137, 0.0141, 0.0133, 0.0129, 0.0125, 0.0126, 0.0130, 0.0128/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0181, 0.0156, 0.0119, 0.0094, 0.0096, 0.0096, 0.0103, 0.0110, 0.0108, 
     &  0.0107, 0.0107, 0.0109, 0.0103, 0.0100, 0.0097, 0.0097, 0.0102, 0.0102/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0141, 0.0121, 0.0092, 0.0073, 0.0074, 0.0074, 0.0080, 0.0086, 0.0084, 
     &  0.0084, 0.0083, 0.0085, 0.0080, 0.0078, 0.0075, 0.0075, 0.0079, 0.0080/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0109, 0.0093, 0.0070, 0.0055, 0.0057, 0.0058, 0.0062, 0.0067, 0.0066, 
     &  0.0065, 0.0065, 0.0066, 0.0062, 0.0060, 0.0058, 0.0058, 0.0060, 0.0062/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0083, 0.0070, 0.0053, 0.0042, 0.0044, 0.0045, 0.0048, 0.0052, 0.0051, 
     &  0.0051, 0.0050, 0.0051, 0.0048, 0.0047, 0.0045, 0.0044, 0.0046, 0.0048/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0063, 0.0053, 0.0040, 0.0032, 0.0033, 0.0035, 0.0038, 0.0041, 0.0040, 
     &  0.0039, 0.0039, 0.0039, 0.0037, 0.0036, 0.0035, 0.0034, 0.0035, 0.0037/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0048, 0.0041, 0.0031, 0.0025, 0.0026, 0.0027, 0.0029, 0.0032, 0.0031, 
     &  0.0031, 0.0030, 0.0031, 0.0029, 0.0028, 0.0027, 0.0026, 0.0027, 0.0028/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0037, 0.0031, 0.0024, 0.0019, 0.0020, 0.0021, 0.0023, 0.0025, 0.0024, 
     &  0.0024, 0.0023, 0.0024, 0.0022, 0.0022, 0.0021, 0.0020, 0.0021, 0.0021/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0028, 0.0024, 0.0018, 0.0014, 0.0015, 0.0016, 0.0018, 0.0019, 0.0019, 
     &  0.0018, 0.0018, 0.0018, 0.0017, 0.0017, 0.0016, 0.0016, 0.0016, 0.0016/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0022, 0.0018, 0.0014, 0.0011, 0.0012, 0.0012, 0.0014, 0.0015, 0.0015, 
     &  0.0014, 0.0014, 0.0014, 0.0014, 0.0013, 0.0013, 0.0012, 0.0012, 0.0012/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0017, 0.0014, 0.0011, 0.0009, 0.0009, 0.0010, 0.0011, 0.0012, 0.0011, 
     &  0.0011, 0.0011, 0.0011, 0.0011, 0.0010, 0.0010, 0.0010, 0.0009, 0.0010/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0013, 0.0011, 0.0008, 0.0007, 0.0007, 0.0007, 0.0008, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0008, 0.0008, 0.0008, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0008, 0.0007, 0.0005, 0.0005, 0.0006, 0.0006, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0006, 0.0006, 0.0006, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0007, 0.0005, 0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0005, 0.0004, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0004, 0.0003, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0002, 0.0002, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65,  9 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals 10. Data corresponds to middle of OCT.
        DATA ( FRAC_O3_COLUMN( IILAT,  1, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9866, 0.9890, 0.9917, 0.9934, 0.9902, 0.9897, 0.9907, 0.9887, 
     &  0.9896, 0.9909, 0.9844, 0.9846, 0.9888, 0.9901, 0.9883, 0.9872, 0.9876/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9929, 0.9743, 0.9797, 0.9832, 0.9850, 0.9801, 0.9768, 0.9802, 0.9765, 
     &  0.9785, 0.9783, 0.9697, 0.9697, 0.9762, 0.9778, 0.9763, 0.9749, 0.9753/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9802, 0.9627, 0.9710, 0.9750, 0.9765, 0.9698, 0.9613, 0.9678, 0.9631, 
     &  0.9677, 0.9660, 0.9568, 0.9557, 0.9634, 0.9654, 0.9643, 0.9631, 0.9639/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9686, 0.9519, 0.9631, 0.9671, 0.9683, 0.9598, 0.9458, 0.9548, 0.9495, 
     &  0.9578, 0.9543, 0.9451, 0.9432, 0.9517, 0.9540, 0.9532, 0.9519, 0.9532/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9578, 0.9418, 0.9556, 0.9594, 0.9605, 0.9504, 0.9311, 0.9424, 0.9361, 
     &  0.9480, 0.9433, 0.9343, 0.9321, 0.9411, 0.9436, 0.9430, 0.9414, 0.9429/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9479, 0.9325, 0.9486, 0.9521, 0.9532, 0.9415, 0.9176, 0.9313, 0.9239, 
     &  0.9384, 0.9334, 0.9244, 0.9224, 0.9316, 0.9343, 0.9337, 0.9316, 0.9332/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9386, 0.9236, 0.9420, 0.9452, 0.9460, 0.9330, 0.9053, 0.9219, 0.9134, 
     &  0.9296, 0.9249, 0.9155, 0.9138, 0.9231, 0.9257, 0.9250, 0.9221, 0.9235/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9293, 0.9147, 0.9352, 0.9376, 0.9385, 0.9247, 0.8943, 0.9139, 0.9046, 
     &  0.9218, 0.9179, 0.9078, 0.9060, 0.9151, 0.9176, 0.9160, 0.9119, 0.9125/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9184, 0.9039, 0.9266, 0.9273, 0.9290, 0.9161, 0.8846, 0.9072, 0.8970, 
     &  0.9148, 0.9120, 0.9012, 0.8989, 0.9069, 0.9087, 0.9052, 0.8986, 0.8973/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9039, 0.8891, 0.9147, 0.9133, 0.9165, 0.9056, 0.8758, 0.9016, 0.8905, 
     &  0.9085, 0.9069, 0.8956, 0.8920, 0.8976, 0.8977, 0.8914, 0.8812, 0.8770/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8841, 0.8694, 0.8989, 0.8960, 0.9012, 0.8922, 0.8678, 0.8967, 0.8848, 
     &  0.9028, 0.9023, 0.8907, 0.8849, 0.8866, 0.8842, 0.8746, 0.8600, 0.8530/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8588, 0.8452, 0.8794, 0.8765, 0.8840, 0.8771, 0.8603, 0.8922, 0.8797, 
     &  0.8977, 0.8980, 0.8863, 0.8772, 0.8743, 0.8686, 0.8557, 0.8364, 0.8274/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8310, 0.8199, 0.8589, 0.8562, 0.8662, 0.8620, 0.8533, 0.8877, 0.8750, 
     &  0.8931, 0.8939, 0.8822, 0.8691, 0.8615, 0.8516, 0.8351, 0.8111, 0.8000/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8065, 0.7980, 0.8392, 0.8348, 0.8475, 0.8471, 0.8467, 0.8834, 0.8709, 
     &  0.8888, 0.8897, 0.8782, 0.8603, 0.8469, 0.8318, 0.8110, 0.7820, 0.7684/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7891, 0.7813, 0.8195, 0.8100, 0.8259, 0.8308, 0.8397, 0.8789, 0.8671, 
     &  0.8847, 0.8853, 0.8737, 0.8498, 0.8287, 0.8072, 0.7813, 0.7470, 0.7310/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7781, 0.7684, 0.7975, 0.7813, 0.8006, 0.8121, 0.8317, 0.8736, 0.8632, 
     &  0.8803, 0.8805, 0.8685, 0.8374, 0.8075, 0.7784, 0.7465, 0.7070, 0.6887/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7694, 0.7556, 0.7701, 0.7478, 0.7707, 0.7891, 0.8208, 0.8655, 0.8569, 
     &  0.8735, 0.8730, 0.8604, 0.8213, 0.7821, 0.7448, 0.7068, 0.6621, 0.6413/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7604, 0.7405, 0.7366, 0.7083, 0.7346, 0.7599, 0.8048, 0.8529, 0.8465, 
     &  0.8622, 0.8609, 0.8473, 0.7997, 0.7510, 0.7056, 0.6619, 0.6118, 0.5885/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7501, 0.7226, 0.6992, 0.6643, 0.6933, 0.7248, 0.7821, 0.8350, 0.8318, 
     &  0.8464, 0.8442, 0.8287, 0.7729, 0.7155, 0.6631, 0.6144, 0.5595, 0.5336/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7377, 0.7014, 0.6584, 0.6170, 0.6473, 0.6832, 0.7507, 0.8098, 0.8107, 
     &  0.8244, 0.8214, 0.8034, 0.7402, 0.6756, 0.6182, 0.5661, 0.5079, 0.4800/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7211, 0.6762, 0.6153, 0.5681, 0.5985, 0.6370, 0.7111, 0.7772, 0.7824, 
     &  0.7952, 0.7915, 0.7708, 0.7022, 0.6325, 0.5719, 0.5180, 0.4579, 0.4286/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6967, 0.6453, 0.5709, 0.5196, 0.5491, 0.5889, 0.6659, 0.7383, 0.7474, 
     &  0.7588, 0.7543, 0.7316, 0.6598, 0.5875, 0.5255, 0.4708, 0.4101, 0.3801/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6622, 0.6077, 0.5258, 0.4729, 0.5004, 0.5402, 0.6168, 0.6939, 0.7064, 
     &  0.7161, 0.7103, 0.6866, 0.6140, 0.5411, 0.4795, 0.4257, 0.3658, 0.3361/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6182, 0.5637, 0.4793, 0.4272, 0.4526, 0.4912, 0.5658, 0.6454, 0.6608, 
     &  0.6685, 0.6610, 0.6368, 0.5649, 0.4934, 0.4337, 0.3821, 0.3246, 0.2957/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5664, 0.5140, 0.4318, 0.3828, 0.4065, 0.4435, 0.5143, 0.5935, 0.6105, 
     &  0.6162, 0.6071, 0.5828, 0.5135, 0.4453, 0.3890, 0.3408, 0.2867, 0.2591/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5094, 0.4608, 0.3845, 0.3402, 0.3621, 0.3968, 0.4626, 0.5378, 0.5549, 
     &  0.5588, 0.5486, 0.5253, 0.4607, 0.3978, 0.3463, 0.3024, 0.2527, 0.2267/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4519, 0.4079, 0.3391, 0.2998, 0.3198, 0.3517, 0.4113, 0.4802, 0.4960, 
     &  0.4985, 0.4880, 0.4665, 0.4082, 0.3519, 0.3059, 0.2670, 0.2222, 0.1981/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3973, 0.3577, 0.2964, 0.2620, 0.2802, 0.3091, 0.3622, 0.4228, 0.4355, 
     &  0.4371, 0.4277, 0.4091, 0.3580, 0.3090, 0.2688, 0.2349, 0.1952, 0.1729/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3464, 0.3112, 0.2570, 0.2272, 0.2437, 0.2695, 0.3158, 0.3667, 0.3753, 
     &  0.3763, 0.3692, 0.3543, 0.3112, 0.2698, 0.2355, 0.2063, 0.1714, 0.1509/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2992, 0.2685, 0.2215, 0.1959, 0.2105, 0.2330, 0.2723, 0.3134, 0.3182, 
     &  0.3189, 0.3141, 0.3033, 0.2681, 0.2342, 0.2056, 0.1810, 0.1505, 0.1318/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2564, 0.2302, 0.1899, 0.1680, 0.1807, 0.2000, 0.2326, 0.2646, 0.2667, 
     &  0.2673, 0.2643, 0.2574, 0.2295, 0.2024, 0.1789, 0.1583, 0.1321, 0.1151/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2180, 0.1958, 0.1615, 0.1431, 0.1541, 0.1703, 0.1968, 0.2207, 0.2210, 
     &  0.2216, 0.2202, 0.2166, 0.1953, 0.1741, 0.1551, 0.1381, 0.1156, 0.1004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1838, 0.1652, 0.1363, 0.1211, 0.1305, 0.1438, 0.1648, 0.1822, 0.1814, 
     &  0.1821, 0.1818, 0.1807, 0.1651, 0.1488, 0.1338, 0.1199, 0.1010, 0.0875/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1538, 0.1384, 0.1143, 0.1017, 0.1097, 0.1203, 0.1366, 0.1490, 0.1478, 
     &  0.1485, 0.1490, 0.1495, 0.1384, 0.1263, 0.1147, 0.1037, 0.0879, 0.0761/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1280, 0.1151, 0.0951, 0.0847, 0.0912, 0.0995, 0.1118, 0.1207, 0.1194, 
     &  0.1199, 0.1210, 0.1225, 0.1150, 0.1062, 0.0975, 0.0890, 0.0761, 0.0660/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1059, 0.0951, 0.0784, 0.0699, 0.0751, 0.0813, 0.0904, 0.0969, 0.0956, 
     &  0.0960, 0.0974, 0.0995, 0.0946, 0.0886, 0.0823, 0.0759, 0.0655, 0.0570/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0871, 0.0780, 0.0642, 0.0571, 0.0611, 0.0656, 0.0722, 0.0771, 0.0761, 
     &  0.0764, 0.0779, 0.0802, 0.0771, 0.0731, 0.0688, 0.0642, 0.0560, 0.0490/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0713, 0.0637, 0.0521, 0.0462, 0.0491, 0.0522, 0.0570, 0.0609, 0.0602, 
     &  0.0604, 0.0618, 0.0640, 0.0621, 0.0597, 0.0568, 0.0537, 0.0475, 0.0419/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0582, 0.0517, 0.0420, 0.0370, 0.0390, 0.0410, 0.0446, 0.0478, 0.0473, 
     &  0.0475, 0.0488, 0.0506, 0.0495, 0.0482, 0.0464, 0.0445, 0.0400, 0.0356/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0472, 0.0416, 0.0334, 0.0292, 0.0305, 0.0319, 0.0346, 0.0372, 0.0369, 
     &  0.0371, 0.0382, 0.0397, 0.0390, 0.0383, 0.0374, 0.0364, 0.0333, 0.0300/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0380, 0.0332, 0.0264, 0.0229, 0.0237, 0.0246, 0.0268, 0.0290, 0.0288, 
     &  0.0290, 0.0299, 0.0309, 0.0304, 0.0302, 0.0298, 0.0294, 0.0275, 0.0250/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0305, 0.0264, 0.0208, 0.0179, 0.0184, 0.0190, 0.0207, 0.0226, 0.0224, 
     &  0.0226, 0.0233, 0.0241, 0.0237, 0.0236, 0.0236, 0.0236, 0.0225, 0.0208/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0242, 0.0208, 0.0162, 0.0139, 0.0142, 0.0147, 0.0161, 0.0176, 0.0175, 
     &  0.0177, 0.0181, 0.0186, 0.0183, 0.0183, 0.0184, 0.0187, 0.0182, 0.0171/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0190, 0.0163, 0.0126, 0.0107, 0.0109, 0.0113, 0.0125, 0.0137, 0.0137, 
     &  0.0138, 0.0141, 0.0144, 0.0141, 0.0141, 0.0142, 0.0147, 0.0146, 0.0140/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0149, 0.0126, 0.0097, 0.0083, 0.0085, 0.0088, 0.0098, 0.0107, 0.0108, 
     &  0.0109, 0.0110, 0.0112, 0.0109, 0.0108, 0.0110, 0.0114, 0.0117, 0.0114/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0115, 0.0098, 0.0075, 0.0064, 0.0065, 0.0069, 0.0076, 0.0084, 0.0085, 
     &  0.0085, 0.0085, 0.0087, 0.0084, 0.0083, 0.0084, 0.0088, 0.0092, 0.0091/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0089, 0.0075, 0.0058, 0.0049, 0.0051, 0.0053, 0.0060, 0.0065, 0.0066, 
     &  0.0066, 0.0066, 0.0067, 0.0065, 0.0064, 0.0064, 0.0067, 0.0072, 0.0072/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0067, 0.0057, 0.0044, 0.0038, 0.0039, 0.0041, 0.0046, 0.0051, 0.0051, 
     &  0.0052, 0.0052, 0.0052, 0.0050, 0.0049, 0.0048, 0.0051, 0.0055, 0.0056/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0051, 0.0044, 0.0034, 0.0029, 0.0030, 0.0032, 0.0036, 0.0040, 0.0040, 
     &  0.0040, 0.0040, 0.0040, 0.0038, 0.0037, 0.0037, 0.0038, 0.0042, 0.0043/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0039, 0.0033, 0.0026, 0.0023, 0.0024, 0.0025, 0.0028, 0.0031, 0.0031, 
     &  0.0031, 0.0031, 0.0031, 0.0030, 0.0029, 0.0028, 0.0029, 0.0032, 0.0033/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0030, 0.0026, 0.0020, 0.0018, 0.0018, 0.0019, 0.0022, 0.0024, 0.0024, 
     &  0.0024, 0.0024, 0.0024, 0.0023, 0.0022, 0.0021, 0.0022, 0.0024, 0.0025/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0023, 0.0020, 0.0016, 0.0014, 0.0014, 0.0015, 0.0017, 0.0019, 0.0019, 
     &  0.0019, 0.0019, 0.0019, 0.0018, 0.0017, 0.0016, 0.0016, 0.0018, 0.0019/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0018, 0.0015, 0.0012, 0.0011, 0.0011, 0.0012, 0.0013, 0.0015, 0.0015, 
     &  0.0015, 0.0015, 0.0015, 0.0014, 0.0013, 0.0013, 0.0012, 0.0014, 0.0014/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0014, 0.0012, 0.0010, 0.0008, 0.0009, 0.0009, 0.0010, 0.0011, 0.0011, 
     &  0.0011, 0.0011, 0.0011, 0.0011, 0.0010, 0.0010, 0.0009, 0.0010, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0011, 0.0009, 0.0007, 0.0006, 0.0007, 0.0007, 0.0008, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0008, 0.0007, 0.0007, 0.0008, 0.0008/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0007, 0.0006, 0.0005, 0.0005, 0.0005, 0.0006, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0006, 0.0006, 0.0006, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0007, 0.0006, 0.0005, 0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0003, 0.0003, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65, 10 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals 11. Data corresponds to middle of NOV.
        DATA ( FRAC_O3_COLUMN( IILAT,  1, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9894, 0.9920, 0.9932, 0.9939, 0.9899, 0.9910, 0.9926, 0.9906, 
     &  0.9882, 0.9880, 0.9859, 0.9867, 0.9900, 0.9908, 0.9894, 0.9877, 0.9884/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9934, 0.9803, 0.9851, 0.9860, 0.9863, 0.9794, 0.9791, 0.9841, 0.9803, 
     &  0.9753, 0.9717, 0.9721, 0.9731, 0.9787, 0.9795, 0.9780, 0.9762, 0.9771/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9828, 0.9714, 0.9786, 0.9788, 0.9782, 0.9691, 0.9646, 0.9735, 0.9685, 
     &  0.9631, 0.9565, 0.9600, 0.9598, 0.9672, 0.9682, 0.9670, 0.9655, 0.9666/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9745, 0.9629, 0.9723, 0.9716, 0.9703, 0.9591, 0.9505, 0.9622, 0.9566, 
     &  0.9522, 0.9426, 0.9487, 0.9478, 0.9567, 0.9577, 0.9568, 0.9555, 0.9570/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9671, 0.9547, 0.9662, 0.9645, 0.9625, 0.9494, 0.9372, 0.9511, 0.9449, 
     &  0.9417, 0.9303, 0.9382, 0.9371, 0.9472, 0.9482, 0.9474, 0.9462, 0.9481/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9599, 0.9468, 0.9602, 0.9579, 0.9551, 0.9403, 0.9248, 0.9408, 0.9340, 
     &  0.9317, 0.9195, 0.9289, 0.9277, 0.9387, 0.9397, 0.9387, 0.9375, 0.9399/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9529, 0.9393, 0.9544, 0.9515, 0.9481, 0.9317, 0.9133, 0.9317, 0.9243, 
     &  0.9225, 0.9099, 0.9206, 0.9194, 0.9310, 0.9319, 0.9306, 0.9291, 0.9320/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9451, 0.9314, 0.9476, 0.9437, 0.9408, 0.9237, 0.9028, 0.9238, 0.9159, 
     &  0.9145, 0.9011, 0.9135, 0.9116, 0.9237, 0.9241, 0.9221, 0.9200, 0.9236/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9340, 0.9215, 0.9380, 0.9330, 0.9320, 0.9159, 0.8932, 0.9170, 0.9087, 
     &  0.9072, 0.8935, 0.9072, 0.9040, 0.9157, 0.9151, 0.9120, 0.9083, 0.9122/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9180, 0.9074, 0.9243, 0.9187, 0.9210, 0.9075, 0.8844, 0.9111, 0.9024, 
     &  0.9008, 0.8869, 0.9016, 0.8961, 0.9063, 0.9040, 0.8987, 0.8926, 0.8962/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8973, 0.8885, 0.9068, 0.9016, 0.9077, 0.8973, 0.8763, 0.9058, 0.8968, 
     &  0.8949, 0.8810, 0.8966, 0.8876, 0.8949, 0.8901, 0.8822, 0.8733, 0.8760/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8736, 0.8663, 0.8871, 0.8830, 0.8924, 0.8852, 0.8688, 0.9008, 0.8919, 
     &  0.8895, 0.8756, 0.8921, 0.8784, 0.8816, 0.8736, 0.8625, 0.8510, 0.8522/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8500, 0.8444, 0.8676, 0.8643, 0.8766, 0.8720, 0.8615, 0.8959, 0.8873, 
     &  0.8846, 0.8705, 0.8879, 0.8686, 0.8669, 0.8547, 0.8399, 0.8254, 0.8251/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8300, 0.8255, 0.8491, 0.8445, 0.8599, 0.8583, 0.8544, 0.8910, 0.8831, 
     &  0.8800, 0.8657, 0.8837, 0.8577, 0.8499, 0.8324, 0.8132, 0.7946, 0.7928/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8139, 0.8093, 0.8296, 0.8212, 0.8403, 0.8433, 0.8473, 0.8862, 0.8792, 
     &  0.8757, 0.8610, 0.8789, 0.8450, 0.8289, 0.8048, 0.7802, 0.7570, 0.7531/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7997, 0.7937, 0.8071, 0.7937, 0.8172, 0.8263, 0.8397, 0.8810, 0.8755, 
     &  0.8717, 0.8564, 0.8734, 0.8301, 0.8046, 0.7727, 0.7422, 0.7141, 0.7076/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7848, 0.7757, 0.7789, 0.7608, 0.7889, 0.8054, 0.8295, 0.8734, 0.8698, 
     &  0.8657, 0.8497, 0.8653, 0.8112, 0.7758, 0.7362, 0.7002, 0.6674, 0.6583/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7669, 0.7536, 0.7444, 0.7217, 0.7543, 0.7788, 0.8144, 0.8615, 0.8604, 
     &  0.8557, 0.8389, 0.8526, 0.7866, 0.7413, 0.6949, 0.6541, 0.6171, 0.6054/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7450, 0.7271, 0.7050, 0.6780, 0.7141, 0.7462, 0.7934, 0.8444, 0.8466, 
     &  0.8417, 0.8236, 0.8342, 0.7568, 0.7028, 0.6508, 0.6059, 0.5653, 0.5514/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7172, 0.6946, 0.6609, 0.6308, 0.6686, 0.7070, 0.7642, 0.8202, 0.8264, 
     &  0.8214, 0.8019, 0.8079, 0.7213, 0.6605, 0.6047, 0.5570, 0.5140, 0.4986/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6811, 0.6548, 0.6129, 0.5817, 0.6200, 0.6626, 0.7271, 0.7886, 0.7986, 
     &  0.7936, 0.7728, 0.7736, 0.6807, 0.6151, 0.5570, 0.5077, 0.4637, 0.4475/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6357, 0.6075, 0.5623, 0.5323, 0.5707, 0.6157, 0.6840, 0.7502, 0.7635, 
     &  0.7582, 0.7367, 0.7324, 0.6361, 0.5675, 0.5085, 0.4588, 0.4150, 0.3987/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5832, 0.5551, 0.5108, 0.4838, 0.5217, 0.5673, 0.6366, 0.7059, 0.7220, 
     &  0.7164, 0.6941, 0.6854, 0.5882, 0.5185, 0.4604, 0.4119, 0.3698, 0.3540/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5276, 0.5008, 0.4593, 0.4360, 0.4731, 0.5182, 0.5864, 0.6573, 0.6758, 
     &  0.6696, 0.6461, 0.6334, 0.5374, 0.4684, 0.4125, 0.3667, 0.3277, 0.3131/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4725, 0.4473, 0.4092, 0.3897, 0.4259, 0.4695, 0.5347, 0.6045, 0.6244, 
     &  0.6175, 0.5929, 0.5775, 0.4853, 0.4191, 0.3665, 0.3245, 0.2893, 0.2763/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4187, 0.3953, 0.3611, 0.3454, 0.3801, 0.4212, 0.4814, 0.5473, 0.5670, 
     &  0.5595, 0.5352, 0.5186, 0.4332, 0.3715, 0.3233, 0.2858, 0.2549, 0.2437/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3675, 0.3461, 0.3160, 0.3037, 0.3361, 0.3738, 0.4278, 0.4878, 0.5057, 
     &  0.4978, 0.4751, 0.4589, 0.3823, 0.3265, 0.2834, 0.2506, 0.2242, 0.2149/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3201, 0.3007, 0.2746, 0.2650, 0.2948, 0.3286, 0.3758, 0.4285, 0.4427, 
     &  0.4348, 0.4153, 0.4010, 0.3344, 0.2852, 0.2473, 0.2190, 0.1969, 0.1895/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2768, 0.2596, 0.2371, 0.2298, 0.2567, 0.2862, 0.3263, 0.3705, 0.3804, 
     &  0.3729, 0.3573, 0.3465, 0.2905, 0.2481, 0.2154, 0.1911, 0.1728, 0.1670/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2378, 0.2227, 0.2036, 0.1980, 0.2217, 0.2468, 0.2798, 0.3158, 0.3221, 
     &  0.3154, 0.3031, 0.2962, 0.2507, 0.2151, 0.1873, 0.1667, 0.1515, 0.1471/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2032, 0.1901, 0.1741, 0.1698, 0.1901, 0.2108, 0.2375, 0.2661, 0.2701, 
     &  0.2643, 0.2547, 0.2513, 0.2152, 0.1860, 0.1627, 0.1453, 0.1328, 0.1296/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1729, 0.1616, 0.1479, 0.1445, 0.1617, 0.1783, 0.1994, 0.2217, 0.2243, 
     &  0.2196, 0.2124, 0.2117, 0.1838, 0.1604, 0.1410, 0.1264, 0.1163, 0.1140/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1463, 0.1365, 0.1247, 0.1219, 0.1361, 0.1491, 0.1656, 0.1829, 0.1847, 
     &  0.1811, 0.1760, 0.1773, 0.1562, 0.1379, 0.1220, 0.1097, 0.1016, 0.1000/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1229, 0.1145, 0.1044, 0.1019, 0.1132, 0.1231, 0.1360, 0.1495, 0.1510, 
     &  0.1484, 0.1450, 0.1476, 0.1320, 0.1179, 0.1051, 0.0948, 0.0883, 0.0874/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1027, 0.0954, 0.0867, 0.0843, 0.0930, 0.1003, 0.1104, 0.1210, 0.1223, 
     &  0.1205, 0.1187, 0.1220, 0.1107, 0.1003, 0.0900, 0.0816, 0.0764, 0.0759/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0852, 0.0790, 0.0712, 0.0688, 0.0753, 0.0807, 0.0886, 0.0970, 0.0982, 
     &  0.0972, 0.0963, 0.1001, 0.0922, 0.0847, 0.0767, 0.0697, 0.0656, 0.0655/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0701, 0.0647, 0.0578, 0.0554, 0.0602, 0.0641, 0.0704, 0.0771, 0.0783, 
     &  0.0779, 0.0776, 0.0815, 0.0761, 0.0709, 0.0648, 0.0591, 0.0559, 0.0561/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0571, 0.0524, 0.0464, 0.0441, 0.0474, 0.0504, 0.0554, 0.0608, 0.0620, 
     &  0.0619, 0.0621, 0.0657, 0.0622, 0.0587, 0.0542, 0.0497, 0.0472, 0.0475/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0461, 0.0421, 0.0368, 0.0347, 0.0371, 0.0392, 0.0433, 0.0476, 0.0487, 
     &  0.0489, 0.0493, 0.0525, 0.0503, 0.0481, 0.0450, 0.0414, 0.0394, 0.0398/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0367, 0.0335, 0.0289, 0.0270, 0.0287, 0.0304, 0.0336, 0.0371, 0.0381, 
     &  0.0383, 0.0388, 0.0415, 0.0401, 0.0389, 0.0368, 0.0342, 0.0326, 0.0330/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0290, 0.0264, 0.0225, 0.0209, 0.0222, 0.0235, 0.0261, 0.0289, 0.0297, 
     &  0.0299, 0.0304, 0.0325, 0.0316, 0.0311, 0.0299, 0.0279, 0.0267, 0.0271/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0228, 0.0207, 0.0175, 0.0162, 0.0172, 0.0182, 0.0202, 0.0225, 0.0232, 
     &  0.0234, 0.0237, 0.0253, 0.0247, 0.0246, 0.0240, 0.0227, 0.0218, 0.0222/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0177, 0.0161, 0.0135, 0.0125, 0.0133, 0.0142, 0.0158, 0.0175, 0.0181, 
     &  0.0183, 0.0184, 0.0196, 0.0191, 0.0192, 0.0190, 0.0182, 0.0176, 0.0179/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0138, 0.0125, 0.0105, 0.0097, 0.0104, 0.0111, 0.0123, 0.0136, 0.0142, 
     &  0.0143, 0.0143, 0.0151, 0.0147, 0.0148, 0.0149, 0.0145, 0.0140, 0.0144/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0107, 0.0097, 0.0081, 0.0076, 0.0081, 0.0087, 0.0096, 0.0107, 0.0111, 
     &  0.0112, 0.0111, 0.0117, 0.0113, 0.0113, 0.0115, 0.0114, 0.0111, 0.0114/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0082, 0.0075, 0.0063, 0.0059, 0.0064, 0.0068, 0.0075, 0.0083, 0.0087, 
     &  0.0087, 0.0086, 0.0091, 0.0086, 0.0086, 0.0088, 0.0089, 0.0088, 0.0090/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0063, 0.0058, 0.0049, 0.0046, 0.0050, 0.0053, 0.0059, 0.0065, 0.0068, 
     &  0.0068, 0.0067, 0.0070, 0.0066, 0.0065, 0.0067, 0.0069, 0.0068, 0.0070/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0048, 0.0044, 0.0038, 0.0036, 0.0039, 0.0042, 0.0046, 0.0051, 0.0053, 
     &  0.0053, 0.0052, 0.0054, 0.0051, 0.0049, 0.0050, 0.0052, 0.0053, 0.0054/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0037, 0.0034, 0.0029, 0.0028, 0.0030, 0.0033, 0.0036, 0.0039, 0.0041, 
     &  0.0041, 0.0041, 0.0042, 0.0039, 0.0037, 0.0037, 0.0039, 0.0041, 0.0042/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0028, 0.0026, 0.0023, 0.0022, 0.0024, 0.0025, 0.0028, 0.0031, 0.0032, 
     &  0.0032, 0.0032, 0.0032, 0.0030, 0.0028, 0.0028, 0.0029, 0.0032, 0.0033/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0022, 0.0020, 0.0018, 0.0017, 0.0019, 0.0020, 0.0022, 0.0024, 0.0025, 
     &  0.0025, 0.0025, 0.0025, 0.0023, 0.0021, 0.0021, 0.0022, 0.0025, 0.0026/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0017, 0.0016, 0.0014, 0.0013, 0.0015, 0.0016, 0.0017, 0.0019, 0.0019, 
     &  0.0019, 0.0019, 0.0019, 0.0018, 0.0016, 0.0016, 0.0016, 0.0019, 0.0020/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0013, 0.0012, 0.0011, 0.0010, 0.0011, 0.0012, 0.0013, 0.0014, 0.0015, 
     &  0.0015, 0.0015, 0.0015, 0.0014, 0.0013, 0.0012, 0.0012, 0.0015, 0.0016/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0010, 0.0008, 0.0008, 0.0009, 0.0009, 0.0010, 0.0011, 0.0012, 
     &  0.0012, 0.0012, 0.0012, 0.0010, 0.0010, 0.0009, 0.0009, 0.0012, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0007, 0.0007, 0.0006, 0.0007, 0.0007, 0.0008, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0007, 0.0007, 0.0007, 0.0010, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0005, 0.0005, 0.0005, 0.0006, 0.0006, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0005, 0.0005, 0.0008, 0.0009/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0004, 0.0004, 0.0004, 0.0006, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0005, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0004, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0003, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0001, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 
     &  0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65, 11 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 
! Third array index equals 12. Data corresponds to middle of DEC.
        DATA ( FRAC_O3_COLUMN( IILAT,  1, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 
     &  1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  2, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  1.0000, 0.9927, 0.9950, 0.9948, 0.9943, 0.9907, 0.9933, 0.9939, 0.9913, 
     &  0.9863, 0.9878, 0.9862, 0.9891, 0.9912, 0.9917, 0.9897, 0.9885, 0.9890/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  3, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9954, 0.9862, 0.9893, 0.9888, 0.9870, 0.9807, 0.9830, 0.9865, 0.9817, 
     &  0.9704, 0.9712, 0.9716, 0.9765, 0.9813, 0.9813, 0.9790, 0.9777, 0.9782/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  4, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9879, 0.9797, 0.9840, 0.9823, 0.9792, 0.9707, 0.9698, 0.9773, 0.9713, 
     &  0.9553, 0.9561, 0.9581, 0.9641, 0.9712, 0.9710, 0.9690, 0.9677, 0.9681/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  5, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9824, 0.9732, 0.9789, 0.9757, 0.9710, 0.9609, 0.9569, 0.9676, 0.9603, 
     &  0.9420, 0.9422, 0.9457, 0.9530, 0.9619, 0.9616, 0.9598, 0.9587, 0.9589/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  6, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9775, 0.9667, 0.9737, 0.9692, 0.9629, 0.9512, 0.9446, 0.9578, 0.9493, 
     &  0.9299, 0.9295, 0.9345, 0.9430, 0.9534, 0.9531, 0.9513, 0.9503, 0.9504/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  7, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9724, 0.9605, 0.9682, 0.9625, 0.9552, 0.9421, 0.9332, 0.9484, 0.9392, 
     &  0.9190, 0.9183, 0.9244, 0.9342, 0.9457, 0.9454, 0.9435, 0.9425, 0.9424/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  8, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9668, 0.9543, 0.9624, 0.9552, 0.9480, 0.9335, 0.9226, 0.9399, 0.9305, 
     &  0.9089, 0.9084, 0.9155, 0.9264, 0.9387, 0.9381, 0.9361, 0.9349, 0.9345/ 
        DATA ( FRAC_O3_COLUMN( IILAT,  9, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9589, 0.9471, 0.9554, 0.9455, 0.9409, 0.9255, 0.9130, 0.9324, 0.9228, 
     &  0.8997, 0.9000, 0.9076, 0.9191, 0.9315, 0.9307, 0.9283, 0.9266, 0.9257/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 10, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9458, 0.9365, 0.9453, 0.9320, 0.9329, 0.9177, 0.9045, 0.9261, 0.9163, 
     &  0.8915, 0.8927, 0.9007, 0.9117, 0.9231, 0.9221, 0.9184, 0.9159, 0.9142/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 11, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9274, 0.9204, 0.9307, 0.9157, 0.9231, 0.9092, 0.8969, 0.9207, 0.9106, 
     &  0.8845, 0.8862, 0.8946, 0.9035, 0.9124, 0.9110, 0.9050, 0.9011, 0.8984/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 12, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.9062, 0.9001, 0.9121, 0.8980, 0.9113, 0.8990, 0.8898, 0.9160, 0.9055, 
     &  0.8782, 0.8800, 0.8892, 0.8938, 0.8986, 0.8963, 0.8875, 0.8820, 0.8780/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 13, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8845, 0.8788, 0.8921, 0.8794, 0.8974, 0.8872, 0.8832, 0.9117, 0.9008, 
     &  0.8722, 0.8743, 0.8843, 0.8826, 0.8819, 0.8778, 0.8657, 0.8585, 0.8537/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 14, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8638, 0.8587, 0.8727, 0.8607, 0.8824, 0.8747, 0.8768, 0.9074, 0.8965, 
     &  0.8666, 0.8689, 0.8798, 0.8699, 0.8626, 0.8556, 0.8401, 0.8308, 0.8255/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 15, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8446, 0.8396, 0.8534, 0.8413, 0.8668, 0.8623, 0.8702, 0.9027, 0.8925, 
     &  0.8617, 0.8637, 0.8751, 0.8559, 0.8403, 0.8291, 0.8099, 0.7976, 0.7913/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 16, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8248, 0.8193, 0.8317, 0.8192, 0.8490, 0.8494, 0.8633, 0.8979, 0.8887, 
     &  0.8573, 0.8585, 0.8699, 0.8403, 0.8142, 0.7974, 0.7736, 0.7576, 0.7497/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 17, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.8021, 0.7956, 0.8056, 0.7934, 0.8283, 0.8348, 0.8560, 0.8927, 0.8850, 
     &  0.8533, 0.8533, 0.8641, 0.8228, 0.7854, 0.7616, 0.7326, 0.7127, 0.7028/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 18, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7744, 0.7665, 0.7732, 0.7626, 0.8034, 0.8172, 0.8466, 0.8854, 0.8795, 
     &  0.8477, 0.8465, 0.8559, 0.8015, 0.7530, 0.7223, 0.6882, 0.6646, 0.6532/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 19, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.7393, 0.7300, 0.7335, 0.7259, 0.7726, 0.7948, 0.8332, 0.8742, 0.8706, 
     &  0.8391, 0.8363, 0.8433, 0.7745, 0.7156, 0.6787, 0.6403, 0.6131, 0.6004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 20, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6957, 0.6858, 0.6876, 0.6840, 0.7363, 0.7669, 0.8148, 0.8585, 0.8577, 
     &  0.8270, 0.8220, 0.8250, 0.7420, 0.6744, 0.6324, 0.5903, 0.5599, 0.5463/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 21, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.6439, 0.6342, 0.6362, 0.6376, 0.6940, 0.7324, 0.7888, 0.8361, 0.8383, 
     &  0.8088, 0.8010, 0.7981, 0.7034, 0.6294, 0.5837, 0.5394, 0.5070, 0.4930/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 22, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5871, 0.5783, 0.5819, 0.5883, 0.6475, 0.6919, 0.7546, 0.8058, 0.8112, 
     &  0.7829, 0.7721, 0.7618, 0.6591, 0.5810, 0.5331, 0.4882, 0.4552, 0.4412/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 23, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.5291, 0.5217, 0.5276, 0.5386, 0.5994, 0.6475, 0.7136, 0.7681, 0.7768, 
     &  0.7497, 0.7359, 0.7177, 0.6105, 0.5305, 0.4818, 0.4376, 0.4052, 0.3916/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 24, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4723, 0.4663, 0.4747, 0.4896, 0.5508, 0.6004, 0.6674, 0.7240, 0.7359, 
     &  0.7102, 0.6934, 0.6677, 0.5588, 0.4793, 0.4314, 0.3897, 0.3592, 0.3465/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 25, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.4177, 0.4130, 0.4234, 0.4415, 0.5020, 0.5516, 0.6176, 0.6753, 0.6901, 
     &  0.6655, 0.6456, 0.6129, 0.5049, 0.4279, 0.3823, 0.3444, 0.3170, 0.3057/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 26, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3672, 0.3634, 0.3751, 0.3951, 0.4538, 0.5020, 0.5650, 0.6222, 0.6386, 
     &  0.6149, 0.5928, 0.5556, 0.4511, 0.3785, 0.3362, 0.3027, 0.2791, 0.2693/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 27, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.3211, 0.3180, 0.3300, 0.3507, 0.4064, 0.4517, 0.5099, 0.5643, 0.5807, 
     &  0.5581, 0.5356, 0.4971, 0.3991, 0.3321, 0.2937, 0.2651, 0.2454, 0.2372/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 28, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2793, 0.2766, 0.2884, 0.3087, 0.3600, 0.4015, 0.4537, 0.5041, 0.5193, 
     &  0.4980, 0.4764, 0.4391, 0.3497, 0.2891, 0.2551, 0.2312, 0.2155, 0.2090/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 29, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2423, 0.2399, 0.2509, 0.2699, 0.3159, 0.3528, 0.3986, 0.4439, 0.4567, 
     &  0.4371, 0.4174, 0.3837, 0.3043, 0.2505, 0.2206, 0.2011, 0.1891, 0.1842/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 30, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.2100, 0.2077, 0.2176, 0.2344, 0.2746, 0.3064, 0.3457, 0.3850, 0.3952, 
     &  0.3775, 0.3603, 0.3321, 0.2636, 0.2165, 0.1906, 0.1747, 0.1657, 0.1620/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 31, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1814, 0.1791, 0.1876, 0.2020, 0.2362, 0.2629, 0.2960, 0.3293, 0.3374, 
     &  0.3218, 0.3072, 0.2848, 0.2272, 0.1867, 0.1645, 0.1515, 0.1449, 0.1423/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 32, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1559, 0.1537, 0.1608, 0.1728, 0.2012, 0.2231, 0.2508, 0.2788, 0.2854, 
     &  0.2719, 0.2598, 0.2427, 0.1951, 0.1609, 0.1420, 0.1313, 0.1264, 0.1247/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 33, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1335, 0.1313, 0.1368, 0.1464, 0.1695, 0.1872, 0.2103, 0.2336, 0.2389, 
     &  0.2276, 0.2185, 0.2060, 0.1671, 0.1385, 0.1225, 0.1135, 0.1099, 0.1091/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 34, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.1135, 0.1113, 0.1154, 0.1227, 0.1413, 0.1554, 0.1746, 0.1938, 0.1980, 
     &  0.1890, 0.1828, 0.1741, 0.1426, 0.1189, 0.1055, 0.0978, 0.0952, 0.0949/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 35, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0955, 0.0935, 0.0963, 0.1017, 0.1164, 0.1277, 0.1435, 0.1593, 0.1629, 
     &  0.1560, 0.1522, 0.1463, 0.1210, 0.1017, 0.0905, 0.0838, 0.0818, 0.0820/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 36, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0794, 0.0776, 0.0795, 0.0833, 0.0947, 0.1037, 0.1165, 0.1294, 0.1327, 
     &  0.1275, 0.1256, 0.1219, 0.1020, 0.0864, 0.0773, 0.0715, 0.0699, 0.0702/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 37, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0654, 0.0638, 0.0648, 0.0674, 0.0762, 0.0832, 0.0936, 0.1040, 0.1069, 
     &  0.1033, 0.1027, 0.1007, 0.0852, 0.0730, 0.0657, 0.0606, 0.0592, 0.0597/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 38, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0532, 0.0518, 0.0523, 0.0539, 0.0607, 0.0662, 0.0744, 0.0827, 0.0854, 
     &  0.0831, 0.0831, 0.0824, 0.0706, 0.0612, 0.0554, 0.0510, 0.0498, 0.0503/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 39, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0426, 0.0416, 0.0416, 0.0426, 0.0478, 0.0521, 0.0586, 0.0651, 0.0676, 
     &  0.0662, 0.0666, 0.0666, 0.0578, 0.0508, 0.0464, 0.0426, 0.0416, 0.0421/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 40, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0339, 0.0331, 0.0328, 0.0334, 0.0374, 0.0407, 0.0458, 0.0509, 0.0530, 
     &  0.0523, 0.0529, 0.0533, 0.0469, 0.0419, 0.0386, 0.0354, 0.0345, 0.0349/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 41, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0266, 0.0260, 0.0257, 0.0260, 0.0291, 0.0316, 0.0355, 0.0395, 0.0412, 
     &  0.0409, 0.0416, 0.0421, 0.0375, 0.0341, 0.0318, 0.0291, 0.0283, 0.0288/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 42, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0207, 0.0204, 0.0200, 0.0202, 0.0226, 0.0246, 0.0276, 0.0306, 0.0319, 
     &  0.0318, 0.0325, 0.0330, 0.0298, 0.0275, 0.0260, 0.0238, 0.0231, 0.0236/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 43, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0161, 0.0160, 0.0156, 0.0157, 0.0176, 0.0191, 0.0214, 0.0237, 0.0247, 
     &  0.0247, 0.0253, 0.0257, 0.0234, 0.0220, 0.0211, 0.0193, 0.0188, 0.0192/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 44, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0125, 0.0125, 0.0122, 0.0123, 0.0138, 0.0149, 0.0167, 0.0184, 0.0191, 
     &  0.0191, 0.0196, 0.0199, 0.0182, 0.0173, 0.0169, 0.0156, 0.0152, 0.0155/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 45, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0097, 0.0097, 0.0095, 0.0096, 0.0108, 0.0117, 0.0130, 0.0142, 0.0148, 
     &  0.0148, 0.0151, 0.0154, 0.0141, 0.0136, 0.0134, 0.0124, 0.0121, 0.0124/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 46, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0075, 0.0076, 0.0074, 0.0075, 0.0085, 0.0092, 0.0102, 0.0111, 0.0115, 
     &  0.0115, 0.0117, 0.0119, 0.0109, 0.0105, 0.0106, 0.0099, 0.0097, 0.0099/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 47, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0058, 0.0059, 0.0058, 0.0059, 0.0067, 0.0072, 0.0079, 0.0086, 0.0089, 
     &  0.0089, 0.0090, 0.0092, 0.0084, 0.0081, 0.0082, 0.0078, 0.0076, 0.0078/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 48, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0045, 0.0046, 0.0045, 0.0046, 0.0052, 0.0057, 0.0062, 0.0067, 0.0069, 
     &  0.0069, 0.0070, 0.0071, 0.0064, 0.0062, 0.0063, 0.0061, 0.0060, 0.0061/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 49, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0035, 0.0035, 0.0035, 0.0036, 0.0041, 0.0044, 0.0048, 0.0052, 0.0053, 
     &  0.0053, 0.0054, 0.0055, 0.0049, 0.0047, 0.0048, 0.0047, 0.0047, 0.0048/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 50, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0027, 0.0027, 0.0028, 0.0028, 0.0032, 0.0035, 0.0038, 0.0040, 0.0041, 
     &  0.0041, 0.0042, 0.0042, 0.0038, 0.0036, 0.0036, 0.0036, 0.0037, 0.0038/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 51, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0021, 0.0021, 0.0021, 0.0022, 0.0025, 0.0027, 0.0029, 0.0031, 0.0031, 
     &  0.0031, 0.0032, 0.0033, 0.0029, 0.0027, 0.0027, 0.0028, 0.0029, 0.0030/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 52, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0016, 0.0016, 0.0017, 0.0017, 0.0020, 0.0021, 0.0023, 0.0024, 0.0024, 
     &  0.0024, 0.0025, 0.0025, 0.0023, 0.0021, 0.0020, 0.0022, 0.0023, 0.0024/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 53, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0013, 0.0013, 0.0013, 0.0013, 0.0015, 0.0017, 0.0018, 0.0019, 0.0019, 
     &  0.0019, 0.0019, 0.0020, 0.0017, 0.0016, 0.0015, 0.0017, 0.0019, 0.0019/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 54, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0010, 0.0010, 0.0010, 0.0011, 0.0012, 0.0013, 0.0014, 0.0014, 0.0014, 
     &  0.0014, 0.0015, 0.0015, 0.0014, 0.0012, 0.0012, 0.0013, 0.0015, 0.0016/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 55, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0008, 0.0008, 0.0008, 0.0008, 0.0009, 0.0010, 0.0011, 0.0011, 0.0011, 
     &  0.0011, 0.0012, 0.0012, 0.0010, 0.0009, 0.0009, 0.0010, 0.0012, 0.0013/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 56, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0006, 0.0006, 0.0006, 0.0006, 0.0007, 0.0008, 0.0008, 0.0009, 0.0009, 
     &  0.0009, 0.0009, 0.0009, 0.0008, 0.0007, 0.0007, 0.0008, 0.0010, 0.0011/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 57, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0005, 0.0005, 0.0005, 0.0005, 0.0006, 0.0006, 0.0007, 0.0007, 0.0007, 
     &  0.0007, 0.0007, 0.0007, 0.0006, 0.0006, 0.0005, 0.0006, 0.0008, 0.0009/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 58, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0005, 0.0005, 0.0005, 0.0005, 0.0005, 
     &  0.0005, 0.0005, 0.0006, 0.0005, 0.0004, 0.0004, 0.0004, 0.0007, 0.0007/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 59, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0004, 
     &  0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0006, 0.0006/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 60, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 
     &  0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0002, 0.0003, 0.0005, 0.0005/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 61, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0003, 0.0002, 0.0002, 0.0002, 0.0002, 0.0004, 0.0004/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 62, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 
     &  0.0002, 0.0002, 0.0002, 0.0002, 0.0002, 0.0001, 0.0002, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 63, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0002, 0.0002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0003, 0.0003/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 64, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0002, 0.0002/ 
        DATA ( FRAC_O3_COLUMN( IILAT, 65, 12 ), IILAT = 1, NLAT_PROFILE ) / 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 
     &  0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001/ 


        CONTAINS
          
         SUBROUTINE SEASONAL_STRAT_O3( JDATE, JTIME )

           USE GRID_CONF           ! horizontal & vertical domain specifications
           USE UTILIO_DEFN
         
           IMPLICIT NONE
!..........Arguments:
           INTEGER, INTENT( IN ) :: JDATE     ! current model date, coded YYYYDDD
           INTEGER, INTENT( IN ) :: JTIME     ! current model time, coded HHMMSS
       
!...........Local:
           INTEGER         :: COL       ! column loop counter
           INTEGER         :: ROW       ! row loop counter
           INTEGER         :: ITIME     ! time loop counter
           INTEGER         :: ILAT      ! lat loop counter
           INTEGER         :: ILEV      ! level loop counter
           INTEGER         :: ASTAT     ! memory allocation status

           INTEGER         :: ITIME_UPPER ! array index for profile time greater than model 
           INTEGER         :: ITIME_LOWER ! array index for profile time lesser than model 
           INTEGER         :: ILEV_UPPER  ! array index for profile Level greater than PTOP
           INTEGER         :: ILEV_LOWER  ! array index for profile level lesser than PTOP 

           INTEGER         :: YEAR
           INTEGER         :: DAY       ! julian day
           LOGICAL         :: LEAP_YEAR ! is a leap year
           LOGICAL         :: FIRST_CALL  = .TRUE.

           CHARACTER( 32 )     :: PNAME = 'SEASONAL_STRAT_O3' 
           CHARACTER( 120 )    :: XMSG  = ' '   

           REAL, PARAMETER    :: MBAR_PER_PA = 9.86923267E-3 ! mbars per Pascals
            
           REAL               :: TIME_UPPER   ! weighting factor for time interpolation
           REAL               :: TIME_LOWER   ! weighting factor for time interpolation
           REAL               :: LEV_UPPER    ! weighting factor for level interpolation (mbar)
           REAL               :: LEV_LOWER    ! weighting factor for level interpolation (mbar)
           REAL               :: PTOP         ! pressure at top level of  model (mbar)
           REAL               :: TIME         ! fraction of year elapsed
           REAL               :: FREQ_PROFILE ! local frequency of profile data change 
           REAL               :: NORMALIZE    ! Level normaliztion factor, 1/mbar
           REAL               :: QLOWER       ! intermediate interpolation result
           REAL               :: QUPPER       ! intermediate interpolation result
           REAL               :: DTIME_UPPER  ! difference to upper time interpolation point 
           REAL               :: DTIME_LOWER  ! difference to lower time interpolation point 
           REAL               :: DLEV_UPPER   ! difference to upper level interpolation point 
           REAL               :: DLEV_LOWER   ! difference to lower level interpolation point 

           REAL, ALLOCATABLE  :: XCROSS_PROFILE( :  ) ! intepolating fraction Ozone column at VGTOP_GD

            ALLOCATE ( XCROSS_PROFILE( NLAT_PROFILE ), STAT = ASTAT )
            IF ( ASTAT .NE. 0 ) THEN
              XMSG = 'Failure allocating XCROSS_PROFILE'
              CALL M3EXIT ( PNAME, JDATE, JTIME, XMSG, XSTAT1 )
            END IF

!determine model time           
            DAY  = MOD( JDATE, 1000 )
            YEAR = INT( 0.001 * REAL(JDATE) )
            
            LEAP_YEAR = ( ( MOD(YEAR,4) .EQ. 0 )
     &                      .AND. ( MOD(YEAR,100) .NE. 0 )
     &                           .OR. ( MOD(YEAR,400) .EQ. 0  ))

            IF( LEAP_YEAR )THEN
               TIME = REAL( DAY ) / 366.0
            ELSE
               TIME = REAL( DAY ) / 365.0
            ENDIF
! find where time is located in profile data
! interpolation over time considers the profile data to be cyclic
            IF( TIME .LT. TIME_PROFILE( 1 ) )THEN
               ITIME_LOWER = NTIME_PROFILE
               ITIME_UPPER = 1
               TIME_LOWER = TIME_PROFILE( NTIME_PROFILE ) - 1.0
               TIME_UPPER = TIME_PROFILE(  1 )
            ELSE IF( TIME .GT. TIME_PROFILE( NTIME_PROFILE ) )THEN
               ITIME_LOWER = NTIME_PROFILE
               ITIME_UPPER = 1
               TIME_LOWER = TIME_PROFILE( NTIME_PROFILE )
               TIME_UPPER = TIME_PROFILE(  1 ) + 1.0
            ELSE
               FIND_TIME: DO ITIME_LOWER = 1, NTIME_PROFILE - 1
                  ITIME_UPPER = ITIME_LOWER  + 1
                  TIME_LOWER  = TIME_PROFILE( ITIME_LOWER )
                  TIME_UPPER  = TIME_PROFILE( ITIME_UPPER )
                  IF( TIME .GE. TIME_LOWER .AND. TIME .LE. TIME_UPPER )EXIT FIND_TIME
               END DO FIND_TIME
            END IF
            
            FREQ_PROFILE  = 1.0 / ( TIME_UPPER - TIME_LOWER )
            DTIME_UPPER   = ( TIME_UPPER - TIME )
            DTIME_LOWER   = ( TIME - TIME_LOWER )
                         
            PTOP = MBAR_PER_PA * VGTOP_GD
! find where PTOP reside relative pressure level of ozone column profiles
            IF ( PTOP .GT. LEV_PROFILE( 1 ) )  THEN
                ILEV = 0
            ELSE IF ( PTOP .LE. LEV_PROFILE( NLEV_PROFILE ) ) THEN
                ILEV = NLEV_PROFILE
            ELSE
               FIND_ILEV: DO ILEV = 1, NLEV_PROFILE - 1
                  IF ( PTOP .LE. LEV_PROFILE( ILEV ) .AND. PTOP .GT. LEV_PROFILE( ILEV + 1 ) ) THEN
                     EXIT FIND_ILEV
                  END IF
               END DO FIND_ILEV
           END IF
            
! get meridian cross-section of fraction O3 column at PTOP
           IF( ILEV .EQ. 0 .OR. ILEV .GE. NLEV_PROFILE )THEN ! only interpolate over time
              ILEV = MAX( ILEV, 1 )
              DO ILAT = 1, NLAT_PROFILE
                 XCROSS_PROFILE( ILAT ) = FREQ_PROFILE
     &                                   * ( DTIME_UPPER * FRAC_O3_COLUMN( ILAT, ILEV, ITIME_LOWER )
     &                                   +   DTIME_LOWER * FRAC_O3_COLUMN( ILAT, ILEV, ITIME_UPPER ) )
              END DO
           ELSE ! interpolate over time and between pressure levels 
              ILEV_UPPER = ILEV + 1
              ILEV_LOWER = ILEV
              DLEV_UPPER = ( LEV_PROFILE( ILEV_UPPER ) - PTOP )
              DLEV_LOWER = ( PTOP - LEV_PROFILE( ILEV_LOWER ) )
              NORMALIZE  = FREQ_PROFILE / ( LEV_PROFILE( ILEV + 1 ) - LEV_PROFILE( ILEV ) ) 
              DO ILAT = 1, NLAT_PROFILE 
                 QLOWER = DLEV_UPPER 
     &                  * ( DTIME_UPPER * FRAC_O3_COLUMN( ILAT, ILEV_LOWER, ITIME_LOWER )
     &                  +   DTIME_LOWER * FRAC_O3_COLUMN( ILAT, ILEV_LOWER, ITIME_UPPER ) )
                 QUPPER = DLEV_LOWER
     &                  * ( DTIME_UPPER * FRAC_O3_COLUMN( ILAT, ILEV_UPPER, ITIME_LOWER )
     &                  +   DTIME_LOWER * FRAC_O3_COLUMN( ILAT, ILEV_UPPER, ITIME_UPPER ) )
                 XCROSS_PROFILE( ILAT ) = NORMALIZE * ( QLOWER + QUPPER )
              END DO
           END IF
                      
! get minimum column fraction
          MONTH_STRAT_03_FRAC = MINVAL( XCROSS_PROFILE )

          WRITE(LOGDEV,99000)MONTH_STRAT_03_FRAC

          DEALLOCATE( XCROSS_PROFILE )

99000     FORMAT(/5X,
     &           'Minimum ratio acceptable between Ozone column above model top to Total Column: ',
     &            F9.6, /)         

         END SUBROUTINE SEASONAL_STRAT_O3
       END MODULE SEAS_STRAT_O3_MIN
