#!/bin/tcsh -fx

# Version @(#)$Id$
# Path    $Source$
# Date    $Date$

# This script sets up needed environment variables for running SMOKE custom
#    reports.  It is NOT specific to a SMOKE source category.
#
#   Script arguments:
#      argv[1] = GRID
#      argv[2] = IOAPI_GRIDNAME_1
#      argv[3] = List of Sectors  (sector abbreviations, in a space-delimited list)
#      argv[4] = List of SMOKE source types for each sector (A, M, or P)
#
# This script is intended to be used with the EMF
# source emissions in SMOKE for the EPA 2002 modeling platform, and 
# calls the scripts that runs the SMOKE programs. 
#
# Script created by : M. Houyoux, Environmental Protection Agency
# December, 2008
#
#*********************************************************************

## log w/ EMF server that script is running
$EMF_CLIENT -k $EMF_JOBKEY -s "Running" 

## set variables so that intermediate logical file names will be set by Assigns file.
setenv RUN_SMKINVEN  Y       #  run inventory import program
setenv RUN_SPCMAT    Y       #  run speciation matrix program
setenv RUN_GRDMAT    Y       #  run gridding matrix program
setenv RUN_TEMPORAL  Y       #  run temporal allocation program
setenv RUN_ELEVPOINT Y       #  run elevated point assignment program

## quality assurance
setenv RUN_SMKREPORT Y       # Y runs reporting 

setenv REGISTER_REPOUT    N       # Imports Smkreport and Smkmerge reports into EMF
setenv PROMPTFLAG         N       # Y (never set to Y for batch processing)
setenv AUTO_DELETE        Y       # Y deletes SMOKE I/O API output files (recommended)
setenv AUTO_DELETE_LOG    Y       # Y automatically deletes logs without asking
if ( ! $?DEBUGMODE ) then
   setenv DEBUGMODE          N       # Y changes script to use debugger
endif
setenv DEBUG_EXE    totalview     # Sets the debugger to use when DEBUGMODE = Y

##############################################################################

switch ( $#argv )
   case 0:
   case 1:
   case 2:
   case 3:
      echo "SCRIPT ERROR: Script requires 4 arguments as follows:"
      echo " "
      echo "     <grid abbrv> <I/O API gridname> <sector list> <source type list>"
      echo " "
      echo "  In the above list, the arguments are defined as follows:"
      echo "     <grid abbrv>       : Grid abbreviation (e.g., 36US1)"
      echo "     <I/O API gridname> : I/O API gridname that needs to match entry in the"
      echo "                          GRIDDESC input file"
      echo "     <sector list>      : space-delimited list of sector abbreviations (in quotes)"
      echo "     <source type list> : space-delimited list of SMOKE source types (A,M,or P) (in quotes)"
      echo " "
      echo "  Example:"
      echo '     <script name> 36US1 36US1_148X112 "ag afdust" "A A"'
      echo "              This example runs the script for the 36US1 grid"
      echo "              for the ag and afdust sectors (which are both SMOKE source" 
      echo '              type of "A"'
      $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: smoke script did not receive more than 3 arguments" -t "e"
      exit( 1 )
endsw

# Get the first two options for the grid abbreviation and I/O API grid
setenv GRID "$argv[1]"
setenv IOAPI_GRIDNAME_1 "$argv[2]"
setenv QA_TYPE custom               # override any other QA_TYPE settings

# Temporarily set the SMOKE source type so can invoke the Assigns file.
setenv SMK_SOURCE A           # source category to process
setenv MRG_SOURCE $SMK_SOURCE # source category to merge

## source the ASSIGNS file to get REPOUT
source $ASSIGNS_FILE             ! logs, etc. will go to sector-specific logs dir

# Set the sector list from the arguments
set list_of_sectors = ($argv[3])
set source_types    = ($argv[4])

set source_type = ($argv[4])
## List of all the helper scripts that are run in this script
set emf_cleanup  = $SCRIPTS/run/emf_cleanup.csh
set timetracker  = $SCRIPTS/run/timetracker_v2.csh
set combine_data = $SCRIPTS/run/combine_data_v5.csh
set qa_run       = $SCRIPTS/run/qa_run_v8.csh
set log_analyzer = $SCRIPTS/log_analyzer/log_analyzer.py
set msg_list     = $SCRIPTS/log_analyzer/known_messages.txt
#set set_months   = $SCRIPTS/run/set_months_v3.csh
#set set_days     = $SCRIPTS/run/set_days_v4.csh

## If running from EMF, move old EMF-created scripts to "old"
if ( $?EMF_JOBID ) then
   source $emf_cleanup
   if ( $status != 0 ) then
	echo "ERROR: running EMF script/log cleanup script"
	$EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: running EMF script/log cleanup script" -t "e" -x $emf_cleanup
	exit( 1 )
   endif
endif

## Invoke script to interpret calling arguments
#$EMF_CLIENT -k $EMF_JOBKEY -m "Running set_months" -x $set_months  ## log w/ EMF server
#set exitstat = 0
#switch ( $#argv )
#   case 4:
#      source $set_months $argv[3] "$argv[4]"
#      if ( $status != 0 ) set exitstat = 1
#   breaksw
#   case 5: 
#      source $set_months $argv[3] "$argv[4]" $argv[5]
#      if ( $status != 0 ) set exitstat = 1
#   breaksw
#   case 6:
#      source $set_months $argv[3] "$argv[4]" $argv[5]
#      if ( $status != 0 ) set exitstat = 1
#      setenv TLABEL $argv[6]
#   breaksw
#endsw
#if ( $exitstat != 0 ) then
#    echo "ERROR: setting months"
#    $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: setting months" -t "e" -x $set_months  
#    exit (1)
#endif

# Set spinup duration - the set_months will have QA'f the $argv[5] value
#if ( $#argv > 4 ) setenv SPINUP_DURATION $argv[5]

# Save spinup array from set_months
#set spinup_array = ( $SPINUP_LIST )

# Loop through the list of sectors

set nsec = 0
foreach sec ( $list_of_sectors ) 

   @ nsec = $nsec + 1

   setenv SECTOR $sec
   setenv EISECTOR $sec

   ## Set naming label
   set namelabel = _smkreport_custom_${SECTOR}_${CASE}_${GRID}

   ## Record the helper scripts being used
   set suffix = _$namelabel.txt
   echo "# Helper scripts used for $SECTOR" > $LOGS/helper_scripts_list$suffix
   echo $emf_cleanup >> $LOGS/helper_scripts_list$suffix
   #echo $set_months >> $LOGS/helper_scripts_list$suffix
   echo $timetracker >> $LOGS/helper_scripts_list$suffix
   echo $combine_data >> $LOGS/helper_scripts_list$suffix
   echo $qa_run >> $LOGS/helper_scripts_list$suffix
   #echo $set_days >> $LOGS/helper_scripts_list$suffix
   echo $log_analyzer >> $LOGS/helper_scripts_list$suffix
   echo $msg_list >> $LOGS/helper_scripts_list$suffix

   # Set the SMOKE source tye for the current sector and other info
   setenv SMK_SOURCE $source_type[$nsec]
   setenv MRG_SOURCE $SMK_SOURCE # source category to merge
   setenv SUBSECT $SECTOR                   # set variable for input/output names
   setenv SRCABBR $SUBSECT                  # set abbreviation for naming log files

   ## Just in case a TSUP file needed, muck with the dates to always choose the 4th of January
   setenv G_STDATE ${YEAR}001
   setenv G_STDATE_ADVANCE 3

   ## Source the ASSIGNS file
   source $ASSIGNS_FILE             ! logs, etc. will go to sector-specific logs dir

   ## Set Time Log filename and initialize file
   setenv TIMELOG $LOGS/timelog_$namelabel.txt

   # Only initialize TIMELOG if it doesn't already exist, since the timeracker 
   #   can now delete/add entries to prevent duplicates
   if ( ! -e $TIMELOG ) then 
      $EMF_CLIENT -k $EMF_JOBKEY -m "Initializing Time Log" -x $timetracker  ## log w/ EMF server
      $timetracker Y $TIMELOG
      if ( $status != 0 ) then
	   echo "ERROR: running timetracker"
	   $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: running timetracker to initialize time log" -t "e" -x $timetracker
	   exit ( 1 )
      endif
   endif

   # Reset PTSUP file name because this script doesn't know which sectors might have
   # put the month in their SUBSECT when running the main run scripts.  So,
   # just set with an 'ls' command to get it right.
   setenv PTSUP `ls $INTERMED/ptsup_${SECTOR}_*_$G_STDATE.txt`

   # Setup REPCONFIG file names
   # Set the output file name dates using the most current file matching the prefix
   set tmplist = (`env| grep REPCONFIG_CUSTOM | cut -d\= -f2` )
   if ( $#tmplist > 0 ) then
      setenv REPCONFIG $REPOUT/custom_reps/in/repconfig_custom_${SECTOR}.txt
   else
      echo "SCRIPT ERROR: No environment variables with REPCONFIG_CUSTOM_ prefix are found."
      $EMF_CLIENT -k $EMF_JOBKEY -m "SCRIPT ERROR: No environment variables with REPCONFIG_CUSTOM_ prefix are found." -t "e" -x smk_custom_report.csh
      exit ( 1 )
   endif

   if (! -e $REPOUT/custom_reps/in ) mkdir -p $REPOUT/custom_reps/in

   # Initialize the REPCONFIG file with the appropriate header
   echo "SMK_SOURCE "$source_types[$nsec] > $REPCONFIG
   echo "/DELIMITER/ |" >> $REPCONFIG

   # Create the REPCONFIG file from the pieces
   foreach f ( `setenv |  grep REPCONFIG_CUSTOM | sort | cut -d"=" -f2` )
      cat $f >> $REPCONFIG

      if ( $status != 0 ) then
          echo "SCRIPT ERROR: Could not add file $f to REPCONFIG"
          $EMF_CLIENT -k $EMF_JOBKEY -m "SCRIPT ERROR: Could not add file $f to REPCONFIG" -t "e"
          exit ( 1 )
      endif
     
   end

   ## Set the output file names and labels
   set out_ev_list = (`grep "/NEWFILE/" $REPCONFIG | cut -c10- | cut -d"#" -f1`)
   set out_label_list = (`grep "/NEWFILE/" $REPCONFIG | cut -d"#" -f3` )  # because inline comments are set with a double pound sign

   # Check that the number of reports and the number of labels is the same, if not, error
   if ( $#out_ev_list != $#out_label_list ) then
      echo "SCRIPT ERROR: Labels (using ## <label> at end of /REPORT/ line) were not found for all reports in REPCONFIG file."


 echo "out_ev_list ="$out_ev_list, $#out_ev_list
 echo "out_label_list ="$out_label_list, $#out_label_list

      $EMF_CLIENT -k $EMF_JOBKEY -m "Labels (using ## <label> at end of /REPORT/ line) were not found for all reports in REPCONFIG file." -t "e" -x smk_custom_report.csh
      exit ( 1 )
   endif

   echo out_ev_list = $out_ev_list

   set n = 0
   foreach ev ( $out_ev_list )
      @ n = $n + 1
      set label = $out_label_list[$n]
      setenv $ev $REPOUT/custom_reps/rep_${SECTOR}_custom_${CASE}_${label}_$GRID.txt

   setenv | grep $ev

   end

   setenv RUN_PART1 Y
   setenv RUN_PART2 N
   setenv RUN_PART3 N
   setenv RUN_PART4 N

   ## Run QA for part 1, non-gridded reports
   ## Run only if Smkreport still set to Y (may have been turned off by RUNSET)
   ## Run only for primary grid
   if ( $RUN_SMKREPORT == Y ) then
      setenv QA_LABEL custom           # Used to name the report inputs and outputs
      setenv REPLABEL custom           # Used internally by Smkreport
      source $qa_run

      # Check status of QA run to see if it worked. Give error if failed
      if ( $status != 0 ) then
          echo "ERROR: Running qa_run for $QA_TYPE" 
          $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: Running qa_run for $QA_TYPE" -t "e" -x $qa_run -p $EMF_PERIOD 
          set exitstat = 1
          goto end_of_script
      endif  # If qa script failure or not

   endif  # End if primary grid or not

   setenv RUN_PART1 N

end  # End loop over sectors

# Label for the end of the script, used during script abort
end_of_script:

## Register time log
echo "SCRIPT NOTE: Registering time log"
$EMF_CLIENT -k $EMF_JOBKEY -F $TIMELOG -T "SMOKE time log (External)" -N "SMOKE timelog $namelabel" -O "Timelog $namelabel (External)"

## Run log file analyzer
$log_analyzer -k $msg_list --list_unknowns -l 3 -f $REPOUT/log_analyzer/rep_logs_${namelabel}_level3.csv -d $LOGS
if ( $status != 0 ) then
   echo "ERROR: running log_analyzer, level 3"
   $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: running log_analyzer, level3" -t "e" -x $log_analyzer
   set exitstat = 1

### Register log analyzer output
#else
#   echo "SCRIPT NOTE: Registering log summary, level3"
#   $EMF_CLIENT -k $EMF_JOBKEY -F $REPOUT/log_analyzer/rep_logs_${namelabel}_level3.csv \
#        -T "Log summary level 3" -O "Level 3 log summary ${namelabel}"
endif

## If log analyzer returns exit status 10, then the log analyzer ran to completion, but found some errors or unrecognized
#  warnings. In this case, we want this script to exit with "Failed" status, but we still want to register the level 1 report in the EMF.
#  If log analyzer returns exit status 1, it didn't work at all, so bomb out immediately.

$log_analyzer -k $msg_list --list_unknowns -e 1 -f $REPOUT/log_analyzer/rep_logs_${namelabel}_level1.csv -d $LOGS
if ( $status == 10 ) then
   echo "SCRIPT NOTE: Registering log summary, level1"
   $EMF_CLIENT -k $EMF_JOBKEY -F $REPOUT/log_analyzer/rep_logs_${namelabel}_level1.csv \
        -T "Log summary level 1" -O "Level 1 log summary ${namelabel}"
	
   echo "ERROR: Level 1 errors or Level 0 warnings found. Check log files, fix inputs, and rerun."
   $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: Level 1 errors or Level 0 warnings found. Check log files, fix inputs, and rerun." -t "e" -x $log_analyzer
   set exitstat = 1

## Register log analyzer output
else if ( $status != 0 ) then
   echo "ERROR: running log_analyzer, level 1"
   $EMF_CLIENT -k $EMF_JOBKEY -m "ERROR: running log_analyzer, level 1" -t "e" -x $log_analyzer
   set exitstat = 1
else
   echo "SCRIPT NOTE: Registering log summary, level1"
   $EMF_CLIENT -k $EMF_JOBKEY -F $REPOUT/log_analyzer/rep_logs_${namelabel}_level1.csv \
        -T "Log summary level 1" -O "Level 1 log summary ${namelabel}"
endif

## Ending of script
#
exit( $exitstat )
