#!/usr/local/bin/python2.6
# Subsets a gridded emissions file
# 5/11/11 James Beidler <beidler.james@epa.gov>

from numpy import *
from scipy.io.netcdf import *
import sys, os

#if len(sys.argv) != 2:
#        print "You must provide an input file name"
#        print "subset.py <infile.py>"
#        sys.exit()

#inFileName = sys.argv[1]

# Dictionary of attributes for possible subset grids
gridDict = { '12EUS1': { 'name': '12EUS1_279X240', 'xorig': -1008000, 'yorig': -1620000, 'nrows': 240, 'ncols': 279 }, \
'12WUS1': { 'name': '12WUS1_213X192', 'xorig': -2412000, 'yorig': -972000, 'nrows': 192, 'ncols': 213 } }

# Hours in the file - generally 25, but put it in here just in case
hours = 25

class gridAtt(object):
	"""
	Provides the grid information in an object form
	"""

	def __init__(self, gridAbbrev):

		self.abbrev = gridAbbrev
		self.name = gridDict[gridAbbrev]['name']
		self.xcell = 12000  # Change if moving to a different grid
		self.ycell = 12000  # ''
		self.xorig = gridDict[gridAbbrev]['xorig']
		self.yorig = gridDict[gridAbbrev]['yorig']
		self.nrows = gridDict[gridAbbrev]['nrows']
		self.ncols = gridDict[gridAbbrev]['ncols']

def openNCF(fileName, accessType = 'r'):
	"""
	Wrapper that tests to see if a NetCDF file is available for access.  If not it returns an error and exits.
	If it is available it returns an open file object.
	"""
	try: file = netcdf_file(fileName, accessType)
	except:
		print "ERROR: %s not available for access." %fileName
		sys.exit(1)
	else: return file

def checkEV(evName):
        """
        Checks if an environment variable is set.  If not, exits.  If it is, returns the variable.
        Takes the name of the environment variable.
        """
        try: var = os.environ[evName]
        except:
                print "ERROR: Environment variable '%s' is not defined." %evName
                sys.exit(1)
        else: return var

esdate = checkEV('ESDATE')  # Output date as YYYYMMDD
grid = checkEV('GRID')    # Grid
imPath = checkEV('IMD_ROOT') # Intermediate path for the case
spec = checkEV('EMF_SPC')   # Speciation
case = checkEV('CASE')    # Case
sector = checkEV('SECTOR') # sector

inFileName = '%s/%s/emis_mole_%s_%s_%s_%s_%s.ncf' %(imPath, sector, sector, esdate, grid, spec, case)

# Open the input file 
inFile = openNCF(inFileName, 'r')

# Set some initial variables based on the input file 
nlays = inFile.dimensions['LAY']
nvars = inFile.dimensions['VAR']
tstep = inFile.dimensions['TSTEP']
speciesList = [ var.strip() for var in getattr(inFile, 'VAR-LIST').split(' ') if var.strip() != '' ]
inXorig = getattr(inFile, 'XORIG')
inYorig = getattr(inFile, 'YORIG')
inXcell = getattr(inFile, 'XCELL') # Our grids tend to be square, so only one cell size is needed.

# Loop over the grids to subset to
for gridAbbrev in ['12EUS1', '12WUS1']:
	print 'Subsetting %s to %s...' %(inFileName, gridAbbrev)

	outFileName = '%s/%s/emis_mole_%s_%s_%s_%s_%s.ncf' %(imPath, sector, sector, esdate, gridAbbrev, spec, case)
	# Set the output file name for the grid
	outFile = openNCF(outFileName, 'w')

	# Get the grid object for the given gridAbbrevs
	grid = gridAtt(gridAbbrev)

	#Process outfile dimensions
	outDims = { 'VAR': int(nvars), 'TSTEP': tstep, 'DATE-TIME': 2, 'ROW': grid.nrows, 'COL': grid.ncols, 'LAY': int(nlays) }
	for dim, value in outDims.items(): outFile.createDimension(dim, value)

	# Set global attributes
	globalAtts = dir(inFile)
	# Ignore automatically created attributes, plus the grid specific ones and history because that doesn't play well for some reason.
	ignoredAtts = ('close', 'createDimension', 'createVariable', 'flush', 'sync', 'NROWS', 'NCOLS', 'XORIG', 'YORIG', 'GDNAM', 'HISTORY')
	attDict = {}
	for attName in [ globalAtts[attNum] for attNum in range(len(globalAtts)) ]:
		if attName in ignoredAtts: continue
		if attName != attName.upper(): continue  # Checks to see if the attribute is in caps.  All the ones that we create should be.
		attVal = getattr(inFile, attName)
		setattr( outFile, attName, attVal ) 
	setattr(outFile, 'NROWS', grid.nrows)
	setattr(outFile, 'NCOLS', grid.ncols)
	#IOAPI requires float64 datatype for these attributes
	setattr(outFile, 'XORIG', float64(grid.xorig) )
	setattr(outFile, 'YORIG', float64(grid.yorig) )
	setattr(outFile, 'GDNAM', grid.name)
	setattr(outFile, 'HISTORY', ' ')

	# Calculate the grid offsets for origination point of new grid on old grid
	# Grids start in the bottom left/SW corner
	xCells = (grid.xorig - inXorig) / inXcell  # Calculate the raw distance between the two origination points and divide by cell size 
	yCells = (grid.yorig - inYorig) / inXcell
 
	# Loop through and subset each species variable
	for speciesName in speciesList:
		speciesIn = inFile.variables[speciesName]
		dataIn = speciesIn[:]
		dataOut = zeros([hours, nlays, grid.nrows, grid.ncols], 'f')
		speciesOut = outFile.createVariable(speciesName, 'f', ('TSTEP', 'LAY', 'ROW', 'COL'))
		speciesOut.long_name = speciesIn.long_name
		speciesOut.units = speciesIn.units
		speciesOut.var_desc = speciesIn.var_desc

		for col in range(grid.ncols):
			for row in range(grid.nrows):
				iRow = yCells + row
				iCol = xCells + col
				dataOut[:,:,row,col] = dataIn[:,:,iRow,iCol]
		speciesOut[:] = dataOut

	# Write the timestep flag 
	tflagOut = outFile.createVariable('TFLAG', 'i', ('TSTEP', 'VAR', 'DATE-TIME'))
	tflagOut.long_name = 'TFLAG' 
	tflagOut.units = '<YYYYDDD,HHMMSS>'
	tflagOut.var_desc = 'Timestep-valid flags:  (1) YYYYDD or (2) HHMMSS'
	tflagIn = inFile.variables['TFLAG'][:]
	tflagOut[:] = tflagIn

