#!/usr/local/bin/python2.6
# Applies factors by species to California grid cells to make adjustments to onroad data
# 08 June 2012 - J. Beidler CSC
# 11 June 2012 - Modified to use matrix math.  10-20x speed-up.
# 7 Aug 2012 - Added grid description parser and updated species for caps-only.

import numpy as np
from scipy.io.netcdf import *
from datetime import timedelta, date, datetime
import sys, csv, os

# List of model ready species that use the VOC ratio file plus HAPS -- no HAPS in new CA for 2007ee
vocList = ('ALDX', 'CH4', 'ETH', 'ETHA', 'ETOH', 'IOLE', 'ISOP', 'MEOH', 'NVOL', 'OLE', 'PAR', 'TERP', 'TOL', 'UNK', 'UNR', 'VOC_INV', 'XYL', \
'BUTADIENE13', 'ACROLEIN', 'FORM', 'FORM_PRIMARY', 'BENZENE', 'ALD2', 'ALD2_PRIMARY')
# List of model ready species that use the PM2_5 ratio file
pm25List = ('PAL', 'PCA', 'PCL', 'PEC', 'PFE', 'PH2O', 'PK', 'PMFINE', 'PMG', 'PMN', 'PMOTHR', 'PNA', 'PNCOM', 'PNH4', 'PNO3', 'POC', 'PSI', 'PSO4', 'PTI')
# Dictionary x-ref table of model ready pollutants to ratio file name
speciesDict = {'CO': 'CO', 'NH3': 'NH3', 'NH3_FERT': 'NH3', 'HONO': 'NOX', 'NO': 'NOX', 'NO2': 'NOX', \
   'PM2_5': 'PM2_5', 'PMC': 'PMC', 'SO2': 'SO2', 'SULF': 'SO2', 'VOC': 'VOC'}

# End User Config

def checkEV(evName):
	"""
	Checks if an environment variable is set.  If not, exits.  If it is, returns the variable.
	Takes the name of the environment variable.
	"""
	try: 
		var = os.environ[evName]
	except:
		print "ERROR: Environment variable '%s' is not defined." %evName
		sys.exit(1)
	else: 
		return var

def openFile(fileName, accessType = 'r'):
	"""
	Wrapper that tests to see if a file is available for access.  If not it returns an error and exits.
	If it is available it returns an open file object.
	"""
	try: 
		file = open(fileName, accessType)
	except:
		sys.exit("ERROR: %s not available for access." %fileName)
	else: 
		return file

def parseFloat(x):
	"""
	returns a floating point with the correct number of trailing zeros based on the .dx
	"""

	if 'D' in x:	
		num = x.strip().split('.')[0]
		zerofloat = x.strip().split('.')[1][1:]
		numlen = len(num) + int(zerofloat)
		while len(num) < numlen:
			num = num + '0'

		return np.float64(num)
	else:
		return float(x)

def getGrid(grid, gridDesc):
	"""
	Reads the grid description file and loads the grid information for the specified grid named.
	"""
	try: gd = open(gridDesc, 'r')
	except: sys.exit('ERROR: Cannot open grid description file %s.' %gridDesc)
	else: state = 1 

	for line in gd.readlines():
		if state == 1:
			if grid.strip().upper() in line:
				state = 2
			continue
		elif state == 2:
			splitLine = line.split(',')
			gridInfo = {'xorig': parseFloat(splitLine[1]), 'yorig': parseFloat(splitLine[2]), 'xcell': parseFloat(splitLine[3]), 'ycell': parseFloat(splitLine[4]), 'cols': int(splitLine[5]), 'rows': int(splitLine[6])}
			break
	if state == 1: sys.exit('ERROR: Grid %s not found in grid description file.' %grid)

	return gridInfo

def openNCF(fileName, accessType = 'r'):
	"""
	Wrapper that tests to see if a NetCDF file is available for access.  If not it returns an error and exits.
	If it is available it returns an open file object.
	"""
	try: 
		file = netcdf_file(fileName, accessType)
	except:
		print "ERROR: %s not available for access." %fileName
		sys.exit(1)
	else: 
		return file

class ratioFile(object):
	"""
	Adjustment ratio class
	"""

	def __init__(self, ratioCode, ratioPath = '/garnet/work/bte/WO113.1/calif/ratios'):
		self.prefix = checkEV('RATIO_PREFIX')
		self.ratioCode = ratioCode
		self.inFileName = '%s/%s_calif_cell_ratios_%s.csv' %(ratioPath, self.prefix, self.ratioCode)
		self.inFile = openFile(self.inFileName)
		self.factors = self.__parseRatio() 

	def __parseRatio(self):
		"""
		Read in the ratio file and create a 2d array of values
		"""

		ratioArr = np.ones([gridInfo['rows'], gridInfo['cols']])

		for line in self.inFile.readlines():
			line = [cell.strip() for cell in line.strip().split(',')]

			if 'col' in line[0]:
				continue

			col = str(int(line[0]) - 1) # Python starts numbering at 0
			row = str(int(line[1]) - 1)
			val = float(line[3])

			try:
				ratioArr[row,col] = val
			except:
				pass
#				print 'WARNING: Ratio location (%s,%s) outside of grid boundaries (%s,%s)' %(col, row, gridInfo['cols'], gridInfo['rows'])

		return ratioArr

class onroadAdj(object):
	"""
	Onroad Adjustment class.
	"""

	def __init__(self, Year = "", Mon = "", Day = "", grid = "", sector = "", spec = "", case = "", imPath = ""):
		"""
		"""
		self.Year = Year
		self.Mon = Mon
		self.Day = Day
		self.grid = grid
		self.sector = sector
		self.spec = spec
		self.case = case
		self.imPath = imPath

		self.date =  self.Year + self.Mon + self.Day
		self.inFileName = ''
		self.outFileName = ''

		self.inFile = self.__loadInFile()
		self.outFile = self.__loadOutFile()

	def threeDigit(self, x):
		"""
		Returns a three character number string from an integer.
		"""
		if len(str(x)) == 1: return '00' + str(x)
		elif len(str(x)) == 2: return '0' + str(x)
		else: return str(x)

	def conv2jul(self, year, month, day):
		"""
		Returns Julian date from year, month, and day.
		"""
		t = time.mktime((int(year), int(month), int(day), 0, 0, 0, 0, 0, 0))
		return int(str(time.gmtime(t)[0]) + self.threeDigit(time.gmtime(t)[7]))
				
	def __loadInFile(self):
		"""
		Set the infile name based on the SMOKE conventions.
		"""
		inFileName = 'emis_mole_' + self.sector + '_%s_' %self.date + self.grid + '_' + self.spec + '_' + self.case + '.ncf'
		inPath = os.path.join(self.imPath, self.sector)
		self.inFileName = os.path.join(inPath, inFileName)

		print "In File: " + self.inFileName
		return openNCF(self.inFileName)

	def __loadOutFile(self):
		"""
		Set the outfile name based on the SMOKE conventions.
		"""
		outFileName = 'emis_mole_' + self.sector + '_adj_%s_' %self.date + self.grid + '_' + self.spec + '_' + self.case + '.ncf'
		outPath = os.path.join(self.imPath, '%s_adj' %self.sector)

		# Try to create the directory.  Ignore error if it already exists.  Otherwise exit.
		try:
			os.mkdir(outPath)
		except OSError as e:
			if e.errno != 17:
				sys.exit('Unable to create output directory: %s' %outPath)
		self.outFileName = os.path.join(outPath, outFileName)

		print "Out File: " + self.outFileName
		return openNCF(self.outFileName, 'w')
	 
	def outFileSettings(self):
		"""
		Defines dimensional and global attribute settings for the outfile.
		"""
		# Set dimensions
		for dimName in self.inFile.dimensions.keys():
			if dimName == 'TSTEP':
				self.outFile.createDimension(dimName, self.inFile.dimensions[dimName])
			else:		
				self.outFile.createDimension(dimName, int(self.inFile.dimensions[dimName]))

		# Set file attributes based on old file
		outList = ['IOAPI_VERSION', 'EXEC_ID', 'FTYPE', 'CDATE', 'CTIME', 'WDATE', 'WTIME', 'SDATE', 'STIME', 'TSTEP', 'NTHIK', 'NCOLS', 'NROWS', 'NLAYS', 'NVARS', 'GDTYP', \
		'P_ALP', 'P_BET', 'P_GAM', 'XCENT', 'YCENT', 'XORIG', 'YORIG', 'XCELL', 'YCELL', 'VGTYP', 'VGTOP', 'VGLVLS', 'GDNAM', 'UPNAM', 'VAR-LIST', 'FILEDESC', 'HISTORY']

		for att in outList: 
			if att == 'HISTORY':
				attVal = ' '
			else:
				attVal = getattr(self.inFile, att)

			setattr(self.outFile, att, attVal)
	
	def parseVarList(self, variableNames):	
		"""
		Define the species tuple for the output variable list global attribute in the out file.
		"""
		speciesTuple = []
		for species in variableNames:
			if species == 'TFLAG': continue
			species = ''.join(self.convertSpeciesName(species))
			if len(species) > 16: species = species[:16]
			while len(species) < 16: species = species + ' '
			speciesTuple = speciesTuple + [species]
		return speciesTuple

	def processSpecies(self, speciesName, ratioTable):
		"""
		Process the input species and adjust based on the ratio table
		"""
		# Fetch a variable from the in file
		species = self.inFile.variables[speciesName]
		dataIn = species[:]

		speciesOut = self.outFile.createVariable(speciesName, 'f', ('TSTEP','LAY','ROW','COL'))
		speciesOut.long_name = species.long_name
		speciesOut.units = species.units
		speciesOut.var_desc = species.var_desc

		dataOut = np.zeros([dataIn.shape[0],dataIn.shape[1],dataIn.shape[2],dataIn.shape[3]], 'f')

		for hour in range(dataOut.shape[0]):
			for lay in range(dataOut.shape[1]):

				dataOut[hour,lay] = dataIn[hour][lay] * ratioTable

		speciesOut[:] = dataOut

	def passSpecies(self, speciesName):
		"""
		Process the input species and pass them through.
		"""
		# Fetch a variable from the in file
		species = self.inFile.variables[speciesName]
		dataIn = species[:]

		speciesOut = self.outFile.createVariable(speciesName, 'f', ('TSTEP','LAY','ROW','COL'))
		speciesOut.long_name = species.long_name
		speciesOut.units = species.units
		speciesOut.var_desc = species.var_desc

		speciesOut[:] = dataIn

	def createTFLAG(self):
		"""
		Create a new TFLAG and adjust to size for number of variables in the out file.
		"""
		TFLAG = self.inFile.variables['TFLAG']
		TFLAGIn = TFLAG[:]
		speciesOut = self.outFile.createVariable('TFLAG', 'i', ('TSTEP', 'VAR', 'DATE-TIME'))
#		dataOut = np.zeros([TFLAG.shape[0],TFLAG.shape[1],TFLAG.shape[2]], 'i')
		setattr(speciesOut, 'long_name', getattr(TFLAG, 'long_name'))
		setattr(speciesOut, 'units', getattr(TFLAG, 'units'))
		setattr(speciesOut, 'var_desc', getattr(TFLAG, 'var_desc'))

		speciesOut[:] = TFLAGIn



### Main script

baseYear = checkEV('BASE_YEAR') # Modeling year, used in definition of dayPath
ratioPath = checkEV('RATIO_PATH') # Path to California ratio files
esdate = checkEV('ESDATE')  # Output date as YYYYMMDD
grid = checkEV('GRID')    # Grid
imPath = checkEV('IMD_ROOT')  # Intermediate path for the case
sector = checkEV('SECTOR')   # Sector
spec = checkEV('EMF_SPC')   # Speciation
case = checkEV('CASE')    # Case
gridDesc = checkEV('GRIDDESC')   # Grid description

Mon = esdate[4:6]
Day = esdate[6:8]
Year = esdate[:4]

# Get grid information for the specified grid
gridInfo = getGrid(grid, gridDesc)

# Create new class object for the adjustments
adjust = onroadAdj(Year, Mon, Day, grid, sector, spec, case, imPath)

# Create the NCF settings for the outfile based on the infile
adjust.outFileSettings()

# Get the list of variables from the infile
vNames = adjust.inFile.variables.keys()

### Run main loop

ratioDict = {}
for speciesName in vNames:

	# Set the ratio code, which is the name used for loading the ratio file
	if speciesName in vocList:
		ratioCode = 'VOC'
	elif speciesName in pm25List:
		ratioCode = 'PM2_5'
	elif speciesName in speciesDict.keys():
		ratioCode = speciesDict[speciesName]
	elif speciesName == 'TFLAG':
		continue
	else:
		ratioCode = '' 

	if ratioCode == '':
		# Pass through species without ratios
		adjust.passSpecies(speciesName)
	else:
		
		if ratioCode not in ratioDict:
			# Load the ratio factors
			ratioDict[ratioCode] = ratioFile(ratioCode, ratioPath)

		ratioTable = ratioDict[ratioCode].factors
		# Apply the factors to the pollutant
		adjust.processSpecies(speciesName, ratioTable)

adjust.createTFLAG()

adjust.inFile.close()
adjust.outFile.close()
