#!/bin/csh -f
#PBS -j oe
#PBS -r n

unset verbose

limit stacksize unlimited

# Start and end dates (Julian dates)
set start_date = 2011121    
set end_date = 2011273
set YEAR = 2011
set season = summer

if ( $?PBS_JOBID ) then
 cd $PBS_O_WORKDIR
else
 set PBS_JOBID = $?
endif

## Definitions for case name, directory structures, etc, that are used
#  by every sector in the case
#  Anything defined in directory_definitions can be overridden here 
#  if desired
source ../../../2023el_cb6v2_v6_11g/scripts/directory_definitions.csh
setenv OLDCASE $CASE
setenv CASE 2023el_ussa_cb6v2_v6_11g   ## primary case name

setenv GRID $REGION_ABBREV
setenv SPEC $EMF_SPC

if ($REGION_ABBREV == 12US2) then
  setenv XORG -2412.
  setenv YORG -1620.
  setenv NX 396
  setenv NY 246
  setenv DX 12.
  set LAYERS = 25  ## Number of layers to output
else
  echo "If not running 12US2, define grid parameters for your grid at the top of this script."
  exit (1)
endif

set casescripts = $CASESCRIPTS/../$CASE/camx
set camxscripts = $INSTALL_DIR/camx
set SPECIES_MAPPING = $camxscripts/Species_Mapping_Tables/MAPTBL.CMAQ_CB6+AERO6_to_CAMXv6_CB6+CF+RTRACPM_2015feb17

#-----------------------------------------------------------------#

set CTM_PT3DEMIS = Y  ## Run the elevated emissions CMAQ to CAMx conversion
set CTM_PTSMRG = Y    ## Merge elevated emissions 

set EMIS_ROOT = $INSTALL_DIR/$OLDCASE/smoke_out  ## Model ready emissions SMOKE out platform root
set INLN_ROOT = $EMIS_ROOT

# SECTORLIST 
set SECTORLIST = ../../../2023el_cb6v2_v6_11g/scripts/sectorlist_2023el_cb6v2_21oct2016_v0
set ptfire_path = $INSTALL_DIR/2011ek_cb6v2_v6_11g/smoke_out/2011ek_cb6v2_v6_11g/$GRID/$SPEC/ptfire3D  # Path to CAMx converted ptfire files
set ptfire_mxca_path = $INSTALL_DIR/2011el_cb6v2_v6_11g/smoke_out/2011el_cb6v2_v6_11g/$GRID/$SPEC/ptfire_mxca3D  # Path to CAMx converted ptfire files

if (! -e $ptfire_path) then
  echo "Please generate the ptfire3D ptsr files first."
  echo "The script to generate these is: ../../../2011ek_cb6v2_v6_11g/scripts/camx/camx_convert_2011ek_cb6v2_12US2_ptfire.csh"
  exit (1)
endif

if (! -e $ptfire_mxca_path) then
  echo "Please generate the ptfire_mxca3D ptsr files first."
  echo "The script to generate these is: ../../../2011el_cb6v2_v6_11g/scripts/camx/camx_convert_2011el_cb6v2_12US2_ptfire_mxca.csh"
  exit (1)
endif

set datefile = $camxscripts/dates/mergedates_${YEAR}.txt
set execroot = $camxscripts/bin # Root to cmaq2camx executables

set QADIR = $casescripts/qatemp #location for QA files - domain totals and domain total 2D plots of point emissions
if (! -e $QADIR) then
	mkdir -p $QADIR
endif

set tempdir = $casescripts/work  ## Path for the temporary work directory
if (! -e $tempdir) then
	mkdir -p $tempdir
endif

setenv SCRIPTLOG cmaq2camx_${CASE}_${GRID}_${PBS_JOBID}.log  ## Name of script log 
if ( -e $casescripts/$SCRIPTLOG) then
	rm $casescripts/$SCRIPTLOG
endif

#-----------------------------------------------------------------#

set today = $start_date

setenv OUTPUT_TIMEZONE 0 ## Timezone for output, typically should be left at 0
setenv OUTPUT_N_LAYERS $LAYERS

## Run the CMAQ to CAMx conversion by loop over the specified date range 
while ($today <= $end_date )

    echo $today >> $casescripts/$SCRIPTLOG
    #-----------------------------------------------------------------#
    set STDATE = $today

    set GDAY = (`$execroot/j2g $today`)
    set YYYY = `echo $GDAY | cut -c1-4`
    set YY = `echo $YYYY | cut -c3-4`
    set MM = `echo $GDAY | cut -c5-6`
    set DD = `echo $GDAY | cut -c7-8`
    set G_DAY = $YYYY$MM$DD
    set GS_DAY = $YY$MM$DD
    set YYDDD = `echo $today | cut -c3-8`

    set iday = `grep ^$STDATE $datefile`
    set dateheader = `grep ^Date $datefile`
    #---------------------------------------------------------------------------#

    ## Run the elevated point cmaq inline to camx conversion
    if ($CTM_PT3DEMIS == Y) then ## elevated point file preparation
        ## Specify the names of the inline sectors to convert from the source apportionment as appears in the sectorlist 
        set sectors = (ag agfire ptegu_${season} cmv nonpt nonroad np_oilgas othar othon othpt ptnonipm pt_oilgas rail rwc onroad onroad_catx_adj)

        foreach SECTOR ($sectors)
            # Set up subsectors, generally used for onroad sectors	
            if ($SECTOR == "onroad" || $SECTOR == "onroad_ca_adj" || $SECTOR == "onroad_catx_adj") then
                set subsects = (RPD RPV RPP RPH)
            else
                set subsects = (NONE)
            endif

            # Find the representative date based on the date approach from the sectorlist
            set dateappr = `grep ^\"$SECTOR\" $SECTORLIST | cut -d, -f4 | sed 's/"//g'`

            # Set sector case 
            set sector_case = `grep ^\"$SECTOR\" $SECTORLIST | cut -d, -f2 | sed 's/"//g'`
	    
	    if ($SECTOR == agfire) set sector_case = $OLDCASE

            set n = 1
            while ($n <= 9)
                set column = $dateheader[$n]
                if ($dateappr == $column) then
                        set repday = $iday[$n]
                endif
                @ n++
            end
            
            foreach subsect ($subsects)
                # Set the input inline and stack groups files
                if $subsect != "NONE" then
                    set INPATH = ${EMIS_ROOT}/${sector_case}/$GRID/${SPEC}/$SECTOR/$subsect
                    set sectname = ${SECTOR}_${subsect}
                    set inln = $INPATH/sginln_mole_${subsect}_${SECTOR}_${repday}_${GRID}_${SPEC}_${sector_case}.ncf
                    setenv STKGROUP $INPATH/stack_groups_out_${subsect}_${SECTOR}_${repday}_${GRID}_${sector_case}.ncf

                    # For onroad, we don't always get a STACK_GROUPS_OUT for every day.
                    # Grab from a different date. They're all the same, so it shouldn't matter which.
                    # I hard-wired this to use the STACK_GROUPS_OUT from January 1; may need to change
                    # that for future runs
                    if (! -e $STKGROUP) then
                        setenv STKGROUP $INPATH/stack_groups_out_${subsect}_${SECTOR}_${YEAR}0101_${GRID}_${sector_case}.ncf
                    endif
                else
                    set INPATH = ${EMIS_ROOT}/${sector_case}/$GRID/${SPEC}/$SECTOR
		    if ($sector == agfire) then
		        set INPATH = $INSTALL_DIR/${sector_case}/smoke_out/${sector_case}/$GRID/${SPEC}/$SECTOR/$subsect
			if (! -e $INPATH) then
			    echo "ERROR: agfire sector must be run with SMK_SRCGROUP_OUTPUT_YN=Y under the 2011ek case first."
			    exit (1)
			endif
		    endif
                    set sectname = ${SECTOR}
                    set inln = $INPATH/sginln_mole_${sectname}_${repday}_${GRID}_${SPEC}_${sector_case}.ncf
                    setenv STKGROUP $INPATH/stack_groups_out_${sectname}_${repday}_${GRID}_${sector_case}.ncf
                endif


                #convert to camx point format and merge into a single point source file and then run qaemis on camx input file
                ls $inln $STKGROUP

                echo $inln >> $casescripts/$SCRIPTLOG

                rm -f ${tempdir}/temp_${PBS_JOBID}.ncf

                if (! -e $inln) then
                    gunzip -c $inln.gz > ${tempdir}/temp_${PBS_JOBID}.ncf
                else
                    cp $inln ${tempdir}/temp_${PBS_JOBID}.ncf
                endif

                rm -f ${tempdir}/tmp_emis_${PBS_JOBID}

                ## Map the elevated species
                setenv INFILE  ${tempdir}/temp_${PBS_JOBID}.ncf
                setenv OUTFILE ${tempdir}/tmp_emis_${PBS_JOBID}
                setenv MAPTBL  $SPECIES_MAPPING
                $execroot/spcmap

                rm -f ${tempdir}/temp_${PBS_JOBID}.ncf 

                setenv CMAQEMIS ${tempdir}/tmp_emis_${PBS_JOBID}
                set OUTPATH = ${tempdir}
                set OUTPATH = ${EMIS_ROOT}/${CASE}/${GRID}/camx/sector/$sectname
                set OUTFILE = ptsr.${sectname}.${repday}.${GRID}.bin

                if (! -e $OUTPATH/$OUTFILE) then
                    mkdir -p $OUTPATH
                    rm -f $OUTPATH/$OUTFILE

## Run cmaq inline to camx converter for the inline sector day
$execroot/cmaq2uam << EOF
File Type          |IPNT
OutFile1 (IC,BC,AR)|
OutFile2 (TopC,PT) |$OUTPATH/$OUTFILE
Output Timezone    |$OUTPUT_TIMEZONE
EOF
                    rm -f ${tempdir}/tmp_emis_${PBS_JOBID}

                    ## QA the elevated sector
                    rm -f ${tempdir}/qaemis.${sectname}.${PBS_JOBID}.in
cat << ieof > ${tempdir}/qaemis.$sectname.${PBS_JOBID}.in
$OUTPATH/$OUTFILE
$QADIR/$sectname.$repday.$GRID.$CASE.camx.tot.txt
$QADIR/$sectname.$repday.$GRID.$CASE.camx.hrs.txt
$QADIR/$sectname.$repday.$GRID.$CASE.camx.pave.bin
ieof
                    $execroot/qaelev < ${tempdir}/qaemis.$sectname.${PBS_JOBID}.in
                    rm -f ${tempdir}/qaemis.$sectname.${PBS_JOBID}.in
                else
                    echo "WARNING: $OUTPATH/$OUTFILE already exists. Will not overwrite."
                endif  ## end representative day writing

            end ## end loop for conversion of this subsector

        end ## end loop for conversion of this sector

    endif  ## end point source file processing

    ## Run the point sector UAM merge
    if ($CTM_PTSMRG == 'Y') then
        set OUTPATH = ${EMIS_ROOT}/${CASE}/${GRID}/camx
        set OUTFILE = mrgpt.$G_DAY.$GRID.$CASE.camx
        set INPATH = $OUTPATH/sector

        rm -f $OUTPATH/$OUTFILE ${tempdir}/ptsmrg.${PBS_JOBID}.txt

        # Link in ptfire
        set firesectors = (ptfire3D ptfire_mxca3D)
        set firepaths = ($ptfire_path $ptfire_mxca_path)
        # Total number of sectors including all subsectors
        set numsect = `echo $firesectors | wc | awk '{print $2}'`
        set n = 1
        while ($n <= $numsect)
            set firesector = $firesectors[$n]
            set fire_path = $firepaths[$n]
            if (! -d ${EMIS_ROOT}/${CASE}/${GRID}/camx/sector/${firesector}) then
                    mkdir -p ${EMIS_ROOT}/${CASE}/${GRID}/camx/sector/${firesector}
            endif
            rm -f ${EMIS_ROOT}/${CASE}/${GRID}/camx/sector/${firesector}/ptsr.${firesector}.${G_DAY}.${GRID}.bin
            ln -s $fire_path/ptsr.${firesector}.${G_DAY}.${GRID}.bin ${EMIS_ROOT}/${CASE}/${GRID}/camx/sector/${firesector}/ptsr.${firesector}.${G_DAY}.${GRID}.bin
            @ n++
        end

        echo "jdate    |$YYDDD" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt
        echo "emissions|$OUTPATH/$OUTFILE" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt

        # Set the names of the sectors to be merged and their respective group numbers
        # Use -1 if group numbers are already set in the input file
        #
        set sectors =   (ag agfire cmv nonpt nonroad np_oilgas othar othon othpt ptegu_${season} ptnonipm pt_oilgas rail rwc onroad onroad_catx_adj afdust_adj othafdust_adj beis ptfire3D ptfire_mxca3D)
        set groupnums = (-1 -1     -1  -1    -1      -1        -1    -1    -1    -1              -1       -1        -1   -1  -1     -1              1          52            1    54       52)

        # Total number of sectors including all subsectors
        set numsect = `echo $sectors | wc | awk '{print $2}'`

        # Add in the subsectors that will be merged
        foreach onroadsect (onroad onroad_catx_adj)
            if (`echo $sectors | grep -c $onroadsect` > 0) then
                @ numsect = $numsect + 3
            endif
        end

        echo "number pt| $numsect" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt

        set sect = 1
        set loopend = `echo $sectors | wc | awk '{print $2}'`
        while ($sect <= $loopend)
            set SECTOR = $sectors[$sect]

            # Find the representative date based on the date approach from the sectorlist
            set dateappr = `grep ^\"$SECTOR\" $SECTORLIST | cut -d, -f4 | sed 's/"//g'`

            set n = 1
            while ($n <= 9)
                set column = $dateheader[$n]
                if ($dateappr == $column) then
                    set repday = $iday[$n]
                endif
                @ n++
            end

            # Set up subsectors, generally used for onroad sectors	
            if ($SECTOR == "onroad" || $SECTOR == "onroad_catx_adj" || $SECTOR == "onroad_ca_adj") then
                set subsects = (RPD RPV RPP RPH)
            else
                set subsects = (NONE)
            endif

            foreach subsect ($subsects)
                # Output the sector path and group assignment to a temporary merge file
                if $subsect != "NONE" then
                    set infile = ${INPATH}/${SECTOR}_${subsect}/ptsr.${SECTOR}_${subsect}.${repday}.${GRID}.bin
                    echo "sat ptsr |$infile" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt
                else
                    set infile = ${INPATH}/${SECTOR}/ptsr.${SECTOR}.${repday}.${GRID}.bin
                    echo "sat ptsr |$infile" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt
                endif

                if (! -e $infile) then
                    echo "ERROR: Cannot find $infile"
                    exit
                endif

                echo "group num| $groupnums[$sect]" >> ${tempdir}/ptsmrg.${PBS_JOBID}.txt
            end
            @ sect++
        end

        ## Run point source merge for converted inline data for this day
        $execroot/ptsmrg < ${tempdir}/ptsmrg.${PBS_JOBID}.txt

        rm -rf ${tempdir}/qaemis.elev.${PBS_JOBID}.in
cat << ieof > ${tempdir}/qaemis.elev.${PBS_JOBID}.in
$OUTPATH/$OUTFILE
$QADIR/mrgpt.$G_DAY.$GRID.$CASE.camx.tot.txt
$QADIR/mrgpt.$G_DAY.$GRID.$CASE.camx.hrs.txt
$QADIR/mrgpt.$G_DAY.$GRID.$CASE.camx.pave.bin
ieof

        ## Run the elevated emissions QA program
        $execroot/qaelev < ${tempdir}/qaemis.elev.${PBS_JOBID}.in
        rm -f ${tempdir}/qaemis.elev.${PBS_JOBID}.in
        rm -f ${tempdir}/ptsr.*.${PBS_JOBID}.bin 

    endif  ## end point source merge 

    @ today = $today + 1

    if ($today == 2010366) then
            set today = 2011001
    endif

end #loop back to next day

exit


