
        SUBROUTINE RDSREF( FDEV )

C***********************************************************************
C  subroutine body starts at line 
C
C  DESCRIPTION:
C     Reads the speciation cross-reference file for any source category.  It
C     allocates memory (locally) for reading the unsorted x-refs. It sorts the
C     x-refs for processing. It allocates memory for the appropriate x-ref 
C     tables and populates the tables (passed via modules).
C
C  PRECONDITIONS REQUIRED:
C     File unit FDEV already is opened... MORE
C
C  SUBROUTINES AND FUNCTIONS CALLED:
C
C  REVISION  HISTORY:
C     Created 1/99 by M. Houyoux
C
C****************************************************************************/
C
C Project Title: Sparse Matrix Operator Kernel Emissions (SMOKE) Modeling
C                System
C File: @(#)$Id$
C
C COPYRIGHT (C) 2004, Environmental Modeling for Policy Development
C All Rights Reserved
C 
C Carolina Environmental Program
C University of North Carolina at Chapel Hill
C 137 E. Franklin St., CB# 6116
C Chapel Hill, NC 27599-6116
C 
C smoke@unc.edu
C
C Pathname: $Source$
C Last updated: $Date$ 
C
C***************************************************************************

C.........  MODULES for public variables
C.........  This module is for cross reference tables
        USE MODXREF, ONLY: INDXTA, CSRCTA, CSCCTA, CMACTA, CISICA, CSPRNA, ISPTA

C.........  This module contains the information about the source category
        USE MODINFO, ONLY: CATEGORY, NIPPA, EANAM, LSCCEND

        IMPLICIT NONE

C...........   INCLUDES

        INCLUDE 'EMCNST3.EXT'   !  emissions constant parameters

C...........   EXTERNAL FUNCTIONS and their descriptions:
        CHARACTER(2)     CRLF
        LOGICAL         ENVYN, BLKORCMT
        INTEGER         FIND1
        INTEGER         FINDC
        INTEGER         GETFLINE
        INTEGER         INDEX1
        INTEGER         STR2INT

        EXTERNAL  BLKORCMT, CRLF, ENVYN, FIND1, FINDC, GETFLINE, INDEX1,
     &            STR2INT

C...........   SUBROUTINE ARGUMENTS
        INTEGER, INTENT (IN) :: FDEV   ! cross-reference file unit no.
 
C...........   Local parameters
        INTEGER, PARAMETER :: MXCOL    = 12

        CHARACTER(6), PARAMETER :: LOCCATS( 3 ) = 
     &                         ( / 'AREA  ', 'MOBILE', 'POINT ' / )

C...........   Sorted pollutant/emission type names
        INTEGER               INDXP  ( NIPPA ) !  sort index for pols/etypes
        CHARACTER(IOVLEN3) :: SRTINAM( NIPPA ) !  sorted pol/etype names

C...........   Array of point source plant characeristics
        CHARACTER(CHRLEN3) CHARS( 5 )

C...........   Array for parsing list-formatted inputs
        CHARACTER(50)          SEGMENT( MXCOL )

C...........   Other local variables
        INTEGER         I, J, J1, J2, K, L, N    !  counters and indices

        INTEGER         IDIU    !  temporary diurnal profile code     
        INTEGER         IDUM    !  dummy integer
        INTEGER         IMON    !  temporary monthly profile code     
        INTEGER         IOS     !  i/o status
        INTEGER         IWEK    !  temporary weekly profile code
        INTEGER         IREC    !  record counter
        INTEGER      :: JS = 0  !  position of SCC in source chars in x-ref file
        INTEGER         JSPC    !  tmp index to master pollutant/etype list
        INTEGER         LPCK    !  length of point definition packet
        INTEGER      :: NCP = 0 !  input point source header parm
        INTEGER         NLINES  !  number of lines
        INTEGER         NXREF   !  number of valid x-ref entries
        INTEGER         RDT     !  temporary road class code
        INTEGER         VTYPE   !  temporary vehicle type number

        LOGICAL      :: EFLAG = .FALSE.   !  true: error found
        LOGICAL      :: PFLAG = .FALSE.   !  true: tmp pol-spec rec skipped
        LOGICAL      :: SKIPPOL = .FALSE. !  true: pol-spec rec skipped in x-ref
        LOGICAL      :: SKIPREC = .FALSE. !  true: record skipped in x-ref file

        CHARACTER          SCC1     !  1st character of SCC
        CHARACTER(5)       CPOS     !  temporary pol code or position

        CHARACTER(300)     LINE     !  line buffer
        CHARACTER(300)     MESG     !  message buffer
        CHARACTER(IOVLEN3) CPOA     !  temporary pollutant/emis type name
        CHARACTER(MACLEN3) CMCT     !  temporory MACT code
        CHARACTER(SICLEN3) CSIC     !  temporary SIC
        CHARACTER(ALLLEN3) CSRCALL  !  buffer for source char, incl pol
        CHARACTER(FIPLEN3) CFIP     !  buffer for CFIPS code
        CHARACTER(RWTLEN3) CRWT     !  buffer for roadway type code
        CHARACTER(VIDLEN3) CVID     !  buffer for vehicle type ID (no name)
        CHARACTER(FIPLEN3) FIPZERO  !  buffer for zero Cy/St/Co code
        CHARACTER(LNKLEN3) LNKZERO  !  buffer for zero Link ID
        CHARACTER(SCCLEN3) TSCC     !  temporary SCC
        CHARACTER(SCCLEN3) SCCZERO  !  buffer for zero SCC
        CHARACTER(PLTLEN3) PLT      !  tmp plant ID
        CHARACTER(SICLEN3) SICZERO  !  buffer for zero SIC
        CHARACTER(SPNLEN3) SPCODE   !  tmp speciation profile code

        CHARACTER(16) :: PROGNAME = 'RDSREF' ! program name

C***********************************************************************
C   begin body of subroutine RDSREF

C.........  Ensure that the CATEGORY is valid
        I = INDEX1( CATEGORY, 3, LOCCATS )

        IF( I .LE. 0 ) THEN
            L = LEN_TRIM( CATEGORY )
            MESG = 'INTERNAL ERROR: category "' // CATEGORY( 1:L ) // 
     &             '" is not valid in routine ' // PROGNAME
            CALL M3MSG2( MESG ) 
            CALL M3EXIT( PROGNAME, 0, 0, ' ', 2 ) 

        END IF

C.........  Set up zero strings for FIPS code, SCC code, and SIC code
        FIPZERO = REPEAT( '0', FIPLEN3 )
        SCCZERO = REPEAT( '0', SCCLEN3 )
        LNKZERO = REPEAT( '0', LNKLEN3 )
        SICZERO = REPEAT( '0', SICLEN3 )

C.........  Sort the actual list of pollutant/emis type names and store it
        DO I = 1, NIPPA
            INDXP( I ) = I
        END DO

        CALL SORTIC( NIPPA, INDXP, EANAM )

        DO I = 1, NIPPA
            J = INDXP( I )
            SRTINAM( I ) = EANAM( J )
        END DO

C.........  Write status message
        MESG = 'Reading speciation cross-reference file...'
        CALL M3MSG2( MESG )

C.........  Get the number of lines in the file
        NLINES = GETFLINE( FDEV, 'Speciation cross reference file' )

C.........  Allocate memory for unsorted data used in all source categories 
        ALLOCATE( CSPRNA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'CSPRNA', PROGNAME )
        ALLOCATE( ISPTA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'ISPTA', PROGNAME )
        ALLOCATE( CSCCTA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'CSCCTA', PROGNAME )
        ALLOCATE( CSRCTA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'CSRCTA', PROGNAME )
        ALLOCATE( CMACTA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'CMACTA', PROGNAME )
        ALLOCATE( CISICA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'CISICA', PROGNAME )
        ALLOCATE( INDXTA( NLINES ), STAT=IOS )
        CALL CHECKMEM( IOS, 'INDXTA', PROGNAME )

C.........  Set up constants for loop.
C.........  Length of point definition packet, plus one
        LPCK = LEN_TRIM( PDEFPCKT ) + 1 

C.........  Put file read pointer at top of file
        REWIND( FDEV )

C.........  Initialize character strings
        CHARS   = ' ' ! array
        SEGMENT = ' '  ! array

C.........  Read lines and store unsorted data for the source category of 
C           interest
        IREC   = 0
        N      = 0
        NCP    = 6        ! ORL and IDA default (4+2)
        JS     = 6        ! ORL and IDA default (4+2)
        DO I = 1, NLINES

            READ( FDEV, 93000, END=999, IOSTAT=IOS ) LINE
            IREC = IREC + 1

            IF ( IOS .NE. 0 ) THEN
                EFLAG = .TRUE.
                WRITE( MESG,94010 ) 
     &              'I/O error', IOS, 
     &              'reading speciation x-ref file at line', IREC
                CALL M3MESG( MESG )
                CYCLE
            END IF

C.............  Skip blank lines or comments
            IF( BLKORCMT( LINE ) ) CYCLE

            J = INDEX( LINE, PDEFPCKT ) ! can be in middle of file
            L = LEN_TRIM( LINE )

C.............  Read point source header information
            IF( J .GT. 0 ) THEN

                IF( L .GT. LPCK ) THEN
                    READ( LINE( LPCK:L ), * ) NCP, JS

                ELSE
                    EFLAG = .TRUE.
                    WRITE( MESG,94010 ) 'ERROR: Incomplete point '//
     &                     'source definition packet at line', IREC
                    CALL M3MSG2( MESG )

                END IF

C.................  Adjust for FIPS code and Plant ID, which are always there
                NCP = NCP + 2
                IF( JS .GT. 0 ) JS = JS + 2

                CYCLE

C.............  If not a header line, then it's a regular line.  The records
C               that don't apply to this source category or to the current
C               inventory will be filtered out by FLTRXREF
            ELSE
            
C.................  Compare point source definition from header to inventory
                IF( CATEGORY .EQ. 'POINT' ) CALL CHKPTDEF( NCP, JS )

                CALL PARSLINE( LINE, MXCOL, SEGMENT )

                TSCC   = SEGMENT( 1 )
                SPCODE = SEGMENT( 2 )                    
                CPOA   = SEGMENT( 3 )
                CFIP   = SEGMENT( 4 )
                CMCT   = SEGMENT( 5 )
                CSIC   = SEGMENT( 6 )
                PLT    = SEGMENT( 7 )
                CHARS( 1:5 ) = SEGMENT( 8:MXCOL )

C.................  Adjust these for proper sorting and matching with profiles
C                   file.
                SPCODE = ADJUSTR( SPCODE )
                CPOA   = ADJUSTL( CPOA   )

C.................  Skip all point entries for nonpoint sectors
                IF ( CATEGORY /= 'POINT' .AND. 
     &               PLT /= ' '                ) CYCLE

C.................  Post-process x-ref information to scan for '-9', pad
C                   with zeros, compare SCC version master list, and compare
C                   pollutant/emission type name with master list.
                CALL FLTRXREF( CFIP, CSIC, TSCC, CPOA, CMCT, 
     &                         IDUM, IDUM, JSPC, PFLAG, SKIPREC )
     
                SKIPPOL = ( SKIPPOL .OR. PFLAG )

C.................  Filter the case where the pollutant code is not present
                IF( CPOA .EQ. ' ' ) THEN
                    EFLAG = .TRUE.
                    WRITE( MESG, 94010 ) 
     &                     'ERROR: Skipping cross-reference entry ' //
     &                     'at line', IREC, 
     &                     'because of missing pollutant.'
                    CALL M3MESG( MESG )
                    CYCLE

                END IF

                IF( SKIPREC ) CYCLE  ! Skip this record

C.................  Write pollutant position to character string
                WRITE( CPOS, '(I5.5)' ) JSPC  

C.................  Check for bad cross-reference code
                IF( SPCODE .EQ. ' ' ) THEN
                    WRITE( MESG, 94010 ) 
     &                'WARNING: Skipping blank profile code in cross-'//
     &                'reference file at line ', IREC
                    CALL M3MESG( MESG )
                    CYCLE

                ENDIF

C.................  If SIC is defined, make sure SCC is not and fill SCC
C                   with SIC value and special identifier
                IF( CSIC /= SICZERO .AND. TSCC /= SCCZERO ) THEN
                    WRITE( MESG,94010 ) 'WARNING: Both SCC and SIC ' //
     &                     'values are given at line', I, '.'//CRLF() //
     &                     BLANK10 // 'Only the SCC will be used ' //
     &                     'for this cross-reference entry.'
                    CALL M3MSG2( MESG )
                    CSIC = SICZERO

                END IF

C.................  Increment count of valid x-ref entries and check it
                N = N + 1
                IF( N .GT. NLINES ) CYCLE  ! Ensure no overflow

C.................  Store case-specific fields from cross reference
                CSRCALL = ' '
                SELECT CASE( CATEGORY )

                CASE( 'AREA' )
                
                    CALL BLDCSRC( CFIP, TSCC, CHRBLNK3,
     &                            CHRBLNK3, CHRBLNK3, CHRBLNK3,
     &                            CHRBLNK3, POLBLNK3, CSRCALL   )

                    CSRCTA( N ) = CSRCALL( 1:SRCLEN3 ) // CMCT // CSIC // CPOS

                CASE( 'MOBILE' )

C M Houyoux note: TSCC has been put in here instead of road type
C     and link has been removed.  These were breaking the county-SCC specific
C     assignments by setting CNFIP in xreftbl.f to be non-blank and not the SCC.
C     However, this change breaks link-specific profile assignments, which
C     are not likely to be used anyway.  I suggest that we just remove
C     link-specific assignments from the documentation for Spcmat.
                    CALL BLDCSRC( CFIP, TSCC, CHRBLNK3, CHRBLNK3,
     &                            CHRBLNK3, CHRBLNK3, CHRBLNK3, 
     &                            POLBLNK3, CSRCALL )

                    CSRCTA( N ) = CSRCALL( 1:SRCLEN3 ) // CMCT // CSIC // CPOS

                CASE( 'POINT' )

C.....................  Store sorting criteria as right-justified in fields
                    CALL BLDCSRC( CFIP, PLT, CHARS(1),
     &                            CHARS(2), CHARS(3), CHARS(4),
     &                            CHARS(5), POLBLNK3, CSRCALL   )

                    CSRCTA( N ) = CSRCALL( 1:SRCLEN3 ) // TSCC // 
     &                            CMCT // CSIC // CPOS

                END SELECT

C.................  Store case-indpendent fields from cross-reference
                INDXTA( N ) = N
                ISPTA ( N ) = JSPC    ! Save index to EANAM or zero
                CSCCTA( N ) = TSCC
                CMACTA( N ) = CMCT
                CISICA( N ) = CSIC
                CSPRNA( N ) = SPCODE

            END IF  !  This line matches source category of interest

        END DO      ! End of loop on I for reading in speciation x-ref file

C.........  Reset number of cross-reference entries in case some were dropped
        NXREF = N

C.........  Write warning message for pollutants in cross-reference that are
C           not in master list
        IF( SKIPPOL ) THEN
            MESG = 'Pollutant-specific entries in the speciation ' //
     &             'cross-reference file have ' // CRLF() // BLANK10 //
     &             'been skipped.'
            CALL M3WARN( PROGNAME, 0, 0, MESG )
        END IF

        IF( NXREF .EQ. 0 ) THEN
            EFLAG = .TRUE.
            MESG = 'ERROR: No valid speciation cross-reference entries!'
            CALL M3MSG2( MESG )

        ELSE IF( NXREF .GT. NLINES ) THEN
            EFLAG = .TRUE.
            WRITE( MESG,94010 ) 'INTERNAL ERROR: dimension for ' //
     &             'storing speciation cross-reference was', NLINES,
     &             CRLF() // BLANK10 // 'but actually needed', NXREF
            CALL M3MSG2( MESG )

        END IF

C.......  Check for errors reading XREF file, and abort
        IF( EFLAG ) THEN
            MESG = 'Problem reading speciation cross-reference file.'
            CALL M3EXIT( PROGNAME, 0, 0, MESG, 2 )
        END IF

        MESG = 'Processing speciation cross-reference file...'
        CALL M3MSG2( MESG )

C.........  Sort speciation cross-reference entries. Since CPOS was used in 
C           building CSRCTA, and CPOS will equal "0" when the x-ref entry is
C           not pollutant/emistype-specific, the these entries will
C           always appear first.  This is necessary for the table-generating
C           subroutines.
        CALL SORTIC( NXREF, INDXTA, CSRCTA )

        CALL XREFTBL( 'SPECIATION', NXREF )

C.........  Deallocate other temporary unsorted arrays
        DEALLOCATE( CSCCTA, ISPTA, CMACTA, CISICA, CSRCTA, CSPRNA, INDXTA )

C.........  Rewind file
        REWIND( FDEV )

        RETURN

C.........  Error message for reaching the end of file too soon
999     MESG = 'End of file reached unexpectedly. ' //
     &         'Check format of speciation' // CRLF() // BLANK5 //
     &         'cross-reference file.'
        CALL M3EXIT( PROGNAME, 0, 0, MESG, 2 )

C******************  FORMAT  STATEMENTS   ******************************

C...........   Formatted file I/O formats............ 93xxx

93000   FORMAT( A )

C...........   Internal buffering formats............ 94xxx

94010   FORMAT( 10( A, :, I8, :, 1X ) )

        END SUBROUTINE RDSREF
