#!/bin/csh
#SBATCH --export=NONE

set case = 2018gc # label for output filenaming only
set year = 2018 # Meteorological year for which you are generating profiles

setenv PROMPTFLAG N                # Y prompts for input
setenv GRID   12US1 # grid name for filename labeling
setenv IOAPI_GRIDNAME_1   12US1_459X299 # grid name as listed in GRIDDESC

# Profile Methods: MET, AGNH3, RWC. For RWC, leave these alone.
setenv PROFILE_METHOD RWC
setenv TPRO_OUTPUT  DAILY

# Determine a list of surrogates for county averaging. Generally should be left alone unless you want to use a different surrogate for averaging cells to counties.
setenv SRG_LIST  "340"          

# List a list of SCC users would like to process 
# This affects the temporal xref output by Gentpro. Unless you want to change the SCCs that receive temperature-based temporal profiles
# in RWC, we recommend you ignore the temporal xref that is output by SMOKE and instead use the RWC temporal xref that is provided in whichever 
# modeling platform you are using. Note that this SCC_LIST does not include all RWC SCCs; there are a few SCCs for which we intentionally use 
# standard temporal profiles, not temperature-based ones.
setenv SCC_LIST  "2104008100,2104008200,2104008210,2104008220,2104008230,2104008300,2104008310,2104008320,2104008330,2104008400,2104008420,2104008510,2104009000"

# By default, Gentpro calculates a temporal factor for counties where the daily minimum temperature is below 50 degrees F.
# The RWC_COUNTY_TEMP settings allow you to override that value for select counties.
# EPA overrides the default temperature value for southern (warmer) states with a value of 60 degrees F according to the RWC_COUNTY_TEMP file.
# SMOKE v4.X does not support state-level temperature overrides; you must list each individual county.
# If you see this error message in your Gentpro log: "ERROR: No entries in COUNTY_TEMP_RWC file"
# that message is inaccurate and can be ignored.
setenv RWC_COUNTY_TEMP_YN Y
setenv RWC_COUNTY_TEMP RWC_county_threshold_smoke40.csv

setenv RWC_ALT_EQ_YN Y # Use alternate RWC profile formula. EPA recommends using this formula and keeping the parameter set to Y.

setenv TEMP_VAR   "TEMP2"      # Leave alone for RWC.

setenv BASEDIR $cwd

setenv SMK_BIN $cwd

setenv GE_DAT $cwd/ge_dat

# This package contains the land area surrogate (340) for the 12km CONUS domain from the 2017 emissions modeling platform.
# Your domain does not have to match the surrogate domain exactly, as long as your resolution and map projection are the same. 
# Surrogates for other domains and resolutions can also be used, but EPA standard practice is to run Gentpro with 12km resolution data.
# EPA standard practice is also to use 12km-based temporal profiles for finer resolution domains, although Gentpro can also be run
# for finer resolutions.
# If you are using a 2014 or 2016 case, since the FIPS codes have changed over time,
# use the land area surrogate and COSTCY from that platform instead of the ones from 2017 platform.
setenv SRGPRO_PATH $GE_DAT # Location of spatial surrogate (just need the single surrogate listed in SRG_LIST)
setenv SRGDESC $GE_DAT/srgdesc_CONUS12_2017NEI_17dec2019_29apr2020_v1.txt # Surrogate descriptions file which lists the filename for the spatial surrogates

setenv COSTCY   $GE_DAT/costcy_for_2017platform_20aug2020_v1.txt # State and county names and FIPS codes, included in this package
setenv GRIDDESC $GE_DAT/griddesc_lambertonly_29may2019_nf_v9.txt # grid description file, included in this package

# This package contains a sample METLIST file. Edit this file to point to your daily METCRO2D (MCIP) files.
# You should also include the last day of the previous year, and the first two days of the next year, in this file.
# If METCRO2D files are not available for those dates, create them by running m3tshift (I/O API utility, a.k.a. m3tools)
# on the first or last day of met data that you do have.
setenv METLIST  metlist_2018j_12US1.txt

# Temporal xref for input to Gentpro. In general you should not need to change this.
setenv TREF_IN $GE_DAT/atref_rwc_forGentpro_2011v2platform_SMOKE36_07nov2014_v1

# Define the modeling period
setenv STDATE ${year}001               # Starting date in Julian
setenv ENDATE ${year}365               # Ending date in Julian 
if ($year == 2004 || $year == 2008 || $year == 2012 || $year == 2016 || $year == 2020) then
  setenv ENDATE ${year}366               # Ending date in Julian 
endif
##############################################################################
# Define output file name
set SENS = RWC_${case}_${GRID}
setenv TREF_OUT  $BASEDIR/Gentpro_TREF_${TPRO_OUTPUT}_${PROFILE_METHOD}.${SENS}.txt
setenv TPRO_MON  $BASEDIR/Gentpro_TPRO_MON_${TPRO_OUTPUT}_${PROFILE_METHOD}.${SENS}.txt
setenv TPRO_DAY  $BASEDIR/Gentpro_TPRO_DAY_${TPRO_OUTPUT}_${PROFILE_METHOD}.${SENS}.txt
setenv TPRO_HOUR $BASEDIR/Gentpro_TPRO_HOUR_${TPRO_OUTPUT}_${PROFILE_METHOD}.${SENS}.ncf
setenv LOGFILE   $BASEDIR/logs/Gentpro_TREF_${TPRO_OUTPUT}_${PROFILE_METHOD}.${SENS}.log
mkdir -p $BASEDIR/logs

##############################################################################

rm -f $LOGFILE

# The "gentpro" executable included in this package is from SMOKE v4.8.
$SMK_BIN/gentpro
