import sqlalchemy

class NR08aData(object):
    def __init__(self, data, data_type):
        self.data = data
        self.data_type = data_type
        
        if self.data_type.mysql_columns is not None:
            self.columns = self.data_type.mysql_columns
        else:
            self.columns = self.data_type.columns
    
    def write(self, engine, schema=None, append=False):
        table = self.data_type.table
        
        if schema is not None and not self._db_exists(engine, schema):
            self._create_db(engine, schema)
        if not self._table_exists(engine, table, schema) or not append:
            self._create_table(engine, table, schema)
        
        if len(self.data) == 0: return
        
        # Some DataTypes need formatting/pre-processing before they can be inserted
        if self.data_type.format_data is not None:
            self.data = self.data_type.format_data(self.data)
        
        # Turn off strict mode to avoid unnecessary errors
        engine.execute("SET sql_mode='';")
        
        if schema is not None:
            query = sqlalchemy.text('INSERT INTO `{}`.`{}` VALUES ({})'.format(schema, table, ','.join([':' + c.name for c in self.columns])))
        else:
            query = sqlalchemy.text('INSERT INTO `{}` VALUES ({})'.format(table, ','.join([':' + c.name for c in self.columns])))
        
        for row in self.data:
            values = {}
            for k, v in zip(self.columns, row):
                values[k.name] = v
            engine.execute(query, values)
    
    def _db_exists(self, engine, schema):
        dbs = [db[0] for db in engine.execute("SHOW DATABASES").fetchall() if '#' not in db[0]]
        return schema in dbs
    
    def _create_db(self, engine, schema):
        engine.execute('CREATE DATABASE IF NOT EXISTS `{}`'.format(schema))
    
    def _table_exists(self, engine, table, schema=None):
        if schema is not None:
            tables = [t[0] for t in engine.execute('SHOW TABLES FROM `{}`'.format(schema)).fetchall()]
        else:
            tables = [t[0] for t in engine.execute('SHOW TABLES').fetchall()]
        return table in tables
    
    def _create_table(self, engine, table, schema=None):
        cols = ['`{}` {} NOT NULL'.format(col.name, col.mysql_type) for col in self.columns]
        if schema is not None:
            engine.execute('DROP TABLE IF EXISTS `{}`.`{}`'.format(schema, table))
            query = '''
                        CREATE TABLE `{schema}`.`{table}` ({cols}) ENGINE=MyISAM DEFAULT CHARSET=utf8
                    '''.format(schema=schema, table=table, cols=','.join(cols))
        else:
            engine.execute('DROP TABLE IF EXISTS `{}`'.format(table))
            query = '''
                        CREATE TABLE `{table}` ({cols}) ENGINE=MyISAM DEFAULT CHARSET=utf8
                    '''.format(table=table, cols=','.join(cols))
        
        engine.execute(query)
