import argparse
import logging
import os
import warnings
import sys

from ConnectToMySQL import ConnectToMySQL
from NR08a.DataFile import DataFile
from NR08a.DataTypes import DataTypesPicker

parser = argparse.ArgumentParser(prog='ProcessNRTxtFiles',
                                 description='this program imports NONROAD input text files into MySQL')
parser.add_argument('dir', help='directory to scan for NONROAD inputs', default='')
parser.add_argument('db', help='name of MySQL database to receive NONROAD inputs', default='')
parser.add_argument('-a', '--append', dest='append', action='store_true',
                    help='if db exists, do not overwrite it')
parser.add_argument('--debug', dest='debug', action='store_true',
                    help='log debug messages')

if __name__ == '__main__':
    args = parser.parse_args()
    if not os.path.exists(args.dir):
        parser.error('Cannot find ' + args.dir)
    
    logger = logging.getLogger(__name__)
    level = logging.DEBUG if args.debug else logging.INFO
    logging.basicConfig(filename='log.txt',
                        level=level,
                        format='%(asctime)s: %(levelname)s - %(message)s',
                        datefmt='%Y-%m-%d %H:%M:%S'
                        )#, filemode='w')
    logger.info('#### ProcessNRTxtFiles Log Entry ####')
    
    try:
        logger.info('Connecting to MySQL...')
        engine = ConnectToMySQL.create_engine('mysql.ini')
        logger.info('Connection established.')
    except Exception as e:
        logger.exception(str(e))
        sys.exit('Could not connect to MySQL. Please check mysql.ini.')
        
    with warnings.catch_warnings():
        warnings.simplefilter('ignore')
        
        if not args.append:
            engine.execute('DROP DATABASE IF EXISTS {};'.format(args.db))
        engine.execute('CREATE DATABASE IF NOT EXISTS {};'.format(args.db))
        logger.info('Using database {}'.format(args.db))
        
        data_files = []
        for root, directories, filenames in os.walk(args.dir):
            for filename in filenames:
                data_files.append(os.path.join(root,filename))
        logger.debug('Found these files to process: {}'.format(', '.join(data_files)))
        
        errors = 0
        for data_file in data_files:
            for data_type in DataTypesPicker(data_file):
                try:
                    DataFile(data_file).read(data_type).write(engine, schema=args.db, append=True)
                    logger.info('Processed {} ({})'.format(data_file, data_type.section))
                except Exception as e:
                    errors += 1
                    section = data_type.section if data_type is not None else 'unknown data type'
                    logger.error('Failed processing {} ({}): {}'.format(data_file, section, str(e)))
    
    message = 'Completed processing {} file{} with {} error{}.'
    message = message.format(len(data_files), 's' if len(data_files) != 1 else '',
                             errors, 's' if errors != 1 else '')
    logger.info(message)
    print(message)
