DROP DATABASE IF EXISTS speciation_outside_moves_collected;
CREATE DATABASE speciation_outside_moves_collected;

DELIMITER //
DROP PROCEDURE IF EXISTS speciation_outside_moves_collected.speciate //
CREATE PROCEDURE speciation_outside_moves_collected.speciate (
	in outputDB varchar(99),
    in workingDB varchar(99)
    ) BEGIN
    
    /* -------------------------------------- PREP COLLECTOR DATABASE IF NECESSARY ------------------------------------------------ */
    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.SMOKE_MOVES_mapping (
		processID 		SMALLINT(6),
		processName 	VARCHAR(50),
		roadTypeID 		SMALLINT(6),
		rateTable 		VARCHAR(10),
		SMOKE_process 	SMALLINT(6),
		SMOKE_mode 		VARCHAR(20),
        PRIMARY KEY (processID, processName, roadTypeID, rateTable, SMOKE_process, SMOKE_mode)
	);

    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.base_schema (
		monthID 				SMALLINT(6),
        SMOKE_SCC 				VARCHAR(10),
        togSpeciationProfileID  VARCHAR(10),
        pollutantID 			SMALLINT(6),
        pollutantName 			VARCHAR(50),
        SMOKE_mode 				VARCHAR(20),
        countyID 				INT(11),
        ratio 					DOUBLE,
        PRIMARY KEY (monthID, SMOKE_SCC, togSpeciationProfileID, pollutantID, pollutantName, SMOKE_mode,countyID)
	);
    
    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.exh_nhtog LIKE speciation_outside_moves_collected.base_schema;
    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.epm_nhtog LIKE speciation_outside_moves_collected.base_schema;
    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.evp_nhtog LIKE speciation_outside_moves_collected.base_schema;
    CREATE TABLE IF NOT EXISTS speciation_outside_moves_collected.rfl_nhtog LIKE speciation_outside_moves_collected.base_schema;
    
    -- The values in this table are hardcoded into the script, which may seem like an odd choice. However, having
    --  this table is very helpful in converting MOVES SCCs to SMOKE SCCs, and keeping track of a csv file in 
    -- 	addition to this script is probably more work than its worth. If the mapping of MOVES SCCs to SMOKE SCCs
    -- 	changes, then this script will have to change accordingly.
	INSERT IGNORE INTO speciation_outside_moves_collected.SMOKE_MOVES_mapping 
		(processID, processName, roadTypeID, rateTable, SMOKE_process, SMOKE_mode) VALUES 
		(1, "Running Exhaust", 1, "RPHO", 92, "EXH_NHTOG"),
		(1, "Running Exhaust", 2, "RPD", 72, "EXH_NHTOG"),
		(1, "Running Exhaust", 3, "RPD", 72, "EXH_NHTOG"),
		(1, "Running Exhaust", 4, "RPD", 72, "EXH_NHTOG"),
		(1, "Running Exhaust", 5, "RPD", 72, "EXH_NHTOG"),
		
		(15, "Crankcase Running Exhaust", 1, "RPHO", 92, "EXH_NHTOG"),
		(15, "Crankcase Running Exhaust", 2, "RPD", 72, "EXH_NHTOG"),
		(15, "Crankcase Running Exhaust", 3, "RPD", 72, "EXH_NHTOG"),
		(15, "Crankcase Running Exhaust", 4, "RPD", 72, "EXH_NHTOG"),
		(15, "Crankcase Running Exhaust", 5, "RPD", 72, "EXH_NHTOG"),
		
		(2, "Start Exhaust", 1, "RPS", 72, "EXH_NHTOG"),
		(16, "Crankcase Start Exhaust", 1, "RPS", 72, "EXH_NHTOG"),
		
		(11, "Evap Permeation", 1, "RPV", 72, "EPM_NHTOG"),
		(11, "Evap Permeation", 2, "RPD", 72, "EPM_NHTOG"),
		(11, "Evap Permeation", 3, "RPD", 72, "EPM_NHTOG"),
		(11, "Evap Permeation", 4, "RPD", 72, "EPM_NHTOG"),
		(11, "Evap Permeation", 5, "RPD", 72, "EPM_NHTOG"),
		
		(12, "Evap Fuel Vapor Venting", 1, "RPP", 72, "EVP_NHTOG"),
		
		(13, " Running Exhaust", 1, "RPV", 72, "EVP_NHTOG"),
		(13, " Running Exhaust", 2, "RPD", 72, "EVP_NHTOG"),
		(13, " Running Exhaust", 3, "RPD", 72, "EVP_NHTOG"),
		(13, " Running Exhaust", 4, "RPD", 72, "EVP_NHTOG"),
		(13, " Running Exhaust", 5, "RPD", 72, "EVP_NHTOG"),
		
		(18, "Refueling Displacement Vapor Loss", 2, "RPD", 62, "RFL_NHTOG"),
		(18, "Refueling Displacement Vapor Loss", 3, "RPD", 62, "RFL_NHTOG"),
		(18, "Refueling Displacement Vapor Loss", 4, "RPD", 62, "RFL_NHTOG"),
		(18, "Refueling Displacement Vapor Loss", 5, "RPD", 62, "RFL_NHTOG"),
		
		(19, "Refueling Displacement Vapor Loss", 2, "RPD", 62, "RFL_NHTOG"),
		(19, "Refueling Displacement Vapor Loss", 3, "RPD", 62, "RFL_NHTOG"),
		(19, "Refueling Displacement Vapor Loss", 4, "RPD", 62, "RFL_NHTOG"),
		(19, "Refueling Displacement Vapor Loss", 5, "RPD", 62, "RFL_NHTOG"),
		
		(90, " Extended Idle Exhaust", 1, "RPH", 53, "EXH_NHTOG"),
		(91, "Auxiliary Power Exhaust", 1, "RPH", 91, "EXH_NHTOG"),
		(17, "Crankcase Extended Idle Exhaust", 1, "RPH", 53, "EXH_NHTOG");
    
    /* -------------------------------------- PREP WORKING DATABASE ------------------------------------------------ */
    SET @stmt = CONCAT("DROP DATABASE IF EXISTS ", workingDB);
    PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("CREATE DATABASE ", workingDB);
    PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    /* -------------------------------------- GET REQUIRED VARS FROM OUTPUT DB ------------------------------------------------ */
    -- This section gets the default database info, county, and year used in the MOVES run. Note that this assumes 
    --  that there is only one county, year, CDB, and default database used. If there are multiple, the script will only run
    -- 	on the first combination
    SET @defDB="moves3_fuels";

    SET @stmt = CONCAT("SET @calendarYear = (SELECT DISTINCT yearID FROM ", outputDB, ".movesoutput LIMIT 1)");
    PREPARE _sql FROM @stmt; EXECUTE _sql;

    SET @stmt = CONCAT("SET @ctyID = (SELECT DISTINCT domainCountyID FROM ", outputDB, ".movesrun LIMIT 1)");
    PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    /* -------------------------------------- GET REQUIRED DATA FROM DEFAULT DB ------------------------------------------------ */
    SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".fuelsByCounty
		SELECT countyID, rc.fuelYearID as yearID, fuelRegionID, monthGroupID as monthID, 
			fuelTypeID, fst.fuelSubtypeID, ff.fuelFormulationID, marketShare
		FROM ", @defDB, ".fuelsupply fs
		JOIN ", @defDB, ".regioncounty rc on (regionID = fuelRegionID and rc.fuelYearID = fs.fuelYearID)
		JOIN ", @defDB, ".fuelformulation ff on (ff.fuelFormulationID = fs.fuelFormulationID)
		JOIN ", @defDB, ".fuelsubtype fst on (ff.fuelSubtypeID = fst.fuelSubtypeID)
		WHERE rc.countyID = ", @ctyID, " and rc.fuelYearID = ", @calendarYear, " and regionCodeID = 1 and marketShare > 0");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".togspeciation
		SELECT *, 
			SUBSTR(modelYearGroupID, 1, 4) as beginModelYearID, 
            SUBSTR(modelYearGroupID, 5, 8) as endModelYearID
		FROM ", @defDB, ".togspeciation");
    PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    /* -------------------------------------- GET MOVES RUN NONHAPTOG ------------------------------------------------ */
    set @stmt = CONCAT("CREATE TABLE ", workingDB, ".movesoutput 
		SELECT yearID, monthID, stateID, countyID, 
			pollutantID, processID, sourceTypeID, regClassID, fuelTypeID, modelYearID, roadTypeID, SCC, 
            emissionQuant
		FROM ", outputDB, ".movesoutput WHERE pollutantID = 88");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
	
    /* -------------------------------------- SPECIATE THE OUTPUT ------------------------------------------------ */
    -- This step requires running several queries, the results of which are stored in the working database. They
    -- 	will be unnecessary after the procedure finishes
    
    SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".nonhaptog_byrg_bymy 
		SELECT pollutantID, yearID, countyID, monthID, regClassID ,modelYearID, SCC, fuelTypeID, processID, roadTypeID, sourceTypeID, 
			sum(emissionQuant) as monthlyEmissions
		FROM ", workingDB, ".movesoutput
		GROUP BY yearID, countyID, monthID, regClassID, modelYearID, SCC, fuelTypeID, processID, roadTypeID, sourceTypeID");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".nonhaptog_complete
		SELECT *, monthlyEmissions * marketShare as weightedMonthlyEmissions
		FROM ", workingDB, ".nonhaptog_byrg_bymy 
		JOIN ", workingDB, ".fuelsByCounty USING (countyID, yearID, monthID, fuelTypeID)");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
	
    SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".nonhaptog_speciated
		SELECT * FROM ", workingDB, ".togspeciation
		JOIN (SELECT pollutantID, fuelSubtypeID, processID, countyID, yearID, monthID, fuelTypeID, modelYearID, SCC, roadTypeID,
				sourceTypeID,fuelRegionID, fuelFormulationID, marketShare, monthlyEmissions, weightedMonthlyEmissions,
				CASE WHEN processID = 11 THEN 0 ELSE regClassID END as regClassID,
				regClassID as regClassID_orig
			  FROM ", workingDB, ".nonhaptog_complete) t2 USING (fuelSubtypeID, regClassID, processID)
		WHERE modelYearID BETWEEN beginModelYearID and endModelYearID"); 
	PREPARE _sql FROM @stmt; EXECUTE _sql;
	
	SET @stmt = CONCAT("CREATE TABLE ", workingDB, ".nonhaptog_speciated_smoke
		SELECT *, CONCAT(SUBSTR(SCC, 1, 8), SMOKE_process) as SMOKE_SCC
		FROM ", workingDB, ".nonhaptog_speciated
		JOIN speciation_outside_moves_collected.SMOKE_MOVES_mapping USING (processID, roadTypeID)");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    /* -------------------------------------- TRANSFER TO COLLECTED DATABASE ------------------------------------------------ */
    SET @stmt = CONCAT("INSERT INTO speciation_outside_moves_collected.exh_nhtog
		SELECT t1.monthID, t1.SMOKE_SCC, t1.togSpeciationProfileID, t1.pollutantID,
			'NonHAPTOG' as pollutantName, t1.SMOKE_mode, t1.countyID, 	
			COALESCE(SUM(weightedMonthlyEmissions) / NULLIF(total, 0), 0) as ratio
		FROM ", workingDB, ".nonhaptog_speciated_smoke t1
		JOIN (SELECT countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode, 
				sum(weightedMonthlyEmissions) as total 
			  FROM ", workingDB, ".nonhaptog_speciated_smoke
			  GROUP BY countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode) t2
		WHERE t1.processID in (1,15, 2,16, 17, 90,91) AND
			t1.countyID = t2.countyID AND t1.yearID = t2.yearID AND t1.monthID = t2.monthID AND 
            t1.SMOKE_SCC = t2.SMOKE_SCC AND t1.SMOKE_mode = t2.SMOKE_mode
		GROUP BY t1.monthID, t1.countyID, t1.yearID, t1.monthID, t1.SMOKE_SCC, t1.togspeciationprofileID");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("INSERT INTO speciation_outside_moves_collected.epm_nhtog
		SELECT t1.monthID, t1.SMOKE_SCC, t1.togSpeciationProfileID, t1.pollutantID,
			'NonHAPTOG' as pollutantName, t1.SMOKE_mode, t1.countyID, 	
			COALESCE(SUM(weightedMonthlyEmissions) / NULLIF(total, 0), 0) as ratio
		FROM ", workingDB, ".nonhaptog_speciated_smoke t1
		JOIN (SELECT countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode, 
				sum(weightedMonthlyEmissions) as total 
			  FROM ", workingDB, ".nonhaptog_speciated_smoke
			  GROUP BY countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode) t2
		WHERE t1.processID in (11) AND
			t1.countyID = t2.countyID AND t1.yearID = t2.yearID AND t1.monthID = t2.monthID AND 
            t1.SMOKE_SCC = t2.SMOKE_SCC AND t1.SMOKE_mode = t2.SMOKE_mode
		GROUP BY t1.monthID, t1.countyID, t1.yearID, t1.monthID, t1.SMOKE_SCC, t1.togspeciationprofileID");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("INSERT INTO speciation_outside_moves_collected.evp_nhtog
		SELECT t1.monthID, t1.SMOKE_SCC, t1.togSpeciationProfileID, t1.pollutantID,
			'NonHAPTOG' as pollutantName, t1.SMOKE_mode, t1.countyID, 	
			COALESCE(SUM(weightedMonthlyEmissions) / NULLIF(total, 0), 0) as ratio
		FROM ", workingDB, ".nonhaptog_speciated_smoke t1
		JOIN (SELECT countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode, 
				sum(weightedMonthlyEmissions) as total 
			  FROM ", workingDB, ".nonhaptog_speciated_smoke
			  GROUP BY countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode) t2
		WHERE t1.processID in (12,13) AND
			t1.countyID = t2.countyID AND t1.yearID = t2.yearID AND t1.monthID = t2.monthID AND 
            t1.SMOKE_SCC = t2.SMOKE_SCC AND t1.SMOKE_mode = t2.SMOKE_mode
		GROUP BY t1.monthID, t1.countyID, t1.yearID, t1.monthID, t1.SMOKE_SCC, t1.togspeciationprofileID");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    SET @stmt = CONCAT("INSERT INTO speciation_outside_moves_collected.rfl_nhtog
		SELECT t1.monthID, t1.SMOKE_SCC, t1.togSpeciationProfileID, t1.pollutantID,
			'NonHAPTOG' as pollutantName, t1.SMOKE_mode, t1.countyID, 	
			COALESCE(SUM(weightedMonthlyEmissions) / NULLIF(total, 0), 0) as ratio
		FROM ", workingDB, ".nonhaptog_speciated_smoke t1
		JOIN (SELECT countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode, 
				sum(weightedMonthlyEmissions) as total 
			  FROM ", workingDB, ".nonhaptog_speciated_smoke
			  GROUP BY countyID, yearID, monthID, SMOKE_SCC, SMOKE_mode) t2
		WHERE t1.processID in (18,19) AND
			t1.countyID = t2.countyID AND t1.yearID = t2.yearID AND t1.monthID = t2.monthID AND 
            t1.SMOKE_SCC = t2.SMOKE_SCC AND t1.SMOKE_mode = t2.SMOKE_mode
		GROUP BY t1.monthID, t1.countyID, t1.yearID, t1.monthID, t1.SMOKE_SCC, t1.togspeciationprofileID");
	PREPARE _sql FROM @stmt; EXECUTE _sql;
    
    /* -------------------------------------- ADJUSTMENTS IF NECESSARY ------------------------------------------------ */
    -- If the calendar year is before 2010, the starts for diesel source types 51, 61, and 62 in July need to be replaced
    -- 	with January data.
    IF @calendarYear < 2010 THEN
		SET @stmt = CONCAT("REPLACE INTO exh_nhtog 
			SELECT moay.monthID, SMOKE_SCC, togSpeciationProfileID, pollutantID, pollutantName, SMOKE_mode, countyID, ratio
			FROM speciation_outside_moves_collected.exh_nhtog t1
			JOIN ", @defDB, "monthOfAnyYear moay
			WHERE t1.monthID = 1 AND SUBSTR(SMOKE_SCC, 5, 4) = '0172'
			AND SUBSTR(SMOKE_SCC, 3, 2) = 2 AND SUBSTR(SMOKE_SCC, 5, 2) in (51,61,62)");
		PREPARE _sql FROM @stmt; EXECUTE _sql;
	END IF;
    
    -- Also if the calendar year is before 2010, there is no CNG in July, so we use the January profiles
    set @stmt = CONCAT("REPLACE INTO exh_nhtog 
		SELECT moay.monthID, SMOKE_SCC, togSpeciationProfileID, pollutantID, pollutantName, SMOKE_mode, countyID, ratio
		FROM exh_nhtog t1
		JOIN ", @defDB, ".monthofanyyear moay
		WHERE t1.monthID = 1 AND SUBSTR(SMOKE_SCC, 5, 4) = '0172'
		AND SUBSTR(SMOKE_SCC, 3, 2) = 3");
    PREPARE _sql FROM @stmt; EXECUTE _sql;
END;
//
DELIMITER ;

/* -------------------------------------- EXAMPLE CALLS ------------------------------------------------
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_1_other', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_7_other', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_1_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_2_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_3_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_4_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_5_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_6_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_7_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_8_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_9_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_10_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_11_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_12_start', 'speciate_working');
CALL speciation_outside_moves_collected.speciate('db_results_inv_2018_20210412_batch0001_c34035_2018_evp', 'speciate_working');*/
