# ReverseFF10_Script_20230118.plx
# This script truncates the VMT, Population, and Hotelling tables of the CDB 
# and insert values from the database 'epa2020ff10_18jan2023'.
# Notes: epa2020ff10_18jan2023 contains the final FF10 VMT/VPOP/HOT from
# EPA SharePoint site.
#
use warnings;
#system('cls');

$year=2020;          	#<<<------- Target year for data.
$ff10db=epa2020ff10_18jan2023; 	#<<<------- Name of FF10 database that contains ff10vmt, ff10pop, and ff10hot.
$mvsdb=movesdb20220802;		#<<<------- MOVES database used in 2020 NEI.
$slmdb=streetlight_2020_20230106_final;    #<<<------- Database containing the "fallback" StreetLight Data month VMT distributions by state for COMM and PERS vehicles.
$dieselauxfraction=0.072; 		#<<<------ fraction of diesel auxiliary power unit (APU) hotelling hours relative to the "diesel APU + diesel main engine" hotelling hours together.
$weekDayAvgDayRatio=1.0129733403037; 		#<<<------ allocation factor for day type 5 in hotellingHoursPerDay CDB table.  Multiply this value by average day hours  to get the average weekday.
$weekendDayAvgDayRatio=0.967566649240754;		#<<<------ allocation factor for day type 2 in hotellingHoursPerDay CDB table.  Multiply this value by average day  hours to get the average weekend day.

# Path to MySQL data folder.
$mysqlpath='C:\\ProgramData\\MariaDB\\MariaDB 10.4\\data';
print "mysqlpath = $mysqlpath\n";

# Location of the list of CDBs to be analyzed.
# $cdblist='P:\EPA_2020_NEI\cdbs_final_edits\CDBs_allcounties_20230117.csv';
$cdblist='C:\Users\ADenBleyker\Desktop\MySQL_Local_Processing\CDBs_allcounties_20230117.csv'; 

print "cdblist= $cdblist\n";

$sql = "
DROP DATABASE IF EXISTS ${ff10db};
CREATE DATABASE IF NOT EXISTS ${ff10db};

CREATE TABLE IF NOT EXISTS ${ff10db}.ff10POP
  (  country_cd         char(2),   -- US
     region_cd          char(5),   -- county Id
     tribal_code        char(2),
     census_tract_cd    char(2),
     shape_id           char(2),
     scc                char(12),
     CD                 char(1),
     MSR                char(1),
     activity_type      char(8),  -- POP
     ann_value          double,   -- annual value for VMT or POP
     calc_year          smallint, -- calc year
     date_updated       char(12), -- date updated
     data_set_id        char(2),  -- vmt only
     jan_value          varchar(10),   -- does not apply for VPOP
     feb_value          varchar(10),   -- does not apply for VPOP
     mar_value          varchar(10),   -- does not apply for VPOP
     apr_value         varchar(10),   -- does not apply for VPOP
     may_value          varchar(10),   -- does not apply for VPOP
     jun_value          varchar(10),   -- does not apply for VPOP
     jul_value          varchar(10),   -- does not apply for VPOP
     aug_value          varchar(10),   -- does not apply for VPOP
     sep_value          varchar(10),   -- does not apply for VPOP
     oct_value          varchar(10),   -- does not apply for VPOP
     nov_value          varchar(10),   -- does not apply for VPOP
     dec_value          varchar(10),   -- does not apply for VPOP
     comment            char(255)
  )
     ENGINE=MyISAM DEFAULT CHARSET=latin1;

CREATE TABLE IF NOT EXISTS ${ff10db}.ff10VMT
  (  country_cd         char(2),   -- US
     region_cd          char(5),   -- county Id
     tribal_code        char(2),
     census_tract_cd    char(2),
     shape_id           char(2),
     scc                char(12),
     CD                 char(1),
     MSR                char(1),
     activity_type      char(50),  -- VMT
     ann_value          double,   -- annual value for VMT or POP
     calc_year          smallint, -- calc year
     date_updated       char(12), -- date updated
     data_set_id        char(50),  -- vmt only
     jan_value          double,   -- vmt only
     feb_value          double,   -- vmt only
     mar_value          double,   -- vmt only
     apr_value          double,   -- vmt only
     may_value          double,   -- vmt only
     jun_value          double,   -- vmt only
     jul_value          double,   -- vmt only
     aug_value          double,   -- vmt only
     sep_value          double,   -- vmt only
     oct_value          double,   -- vmt only
     nov_value          double,   -- vmt only
     dec_value          double,
     comment            char(255)
  )
     ENGINE=MyISAM DEFAULT CHARSET=latin1;

CREATE TABLE         ${ff10db}.FF10HOT (
country_cd          char(2)
,region_cd          char(5)
,tribal_code        char(2)
,census_tract_cd    char(2)
,shape_id           char(2)
,scc                char(12)
,CD                 char(1)
,MSR                char(1)
,activity_type      char(50)
,ann_value          double
,calc_year          smallint
,date_updated       char(12)
,data_set_id        char(50)
,jan_value          double
,feb_value          double
,mar_value          double
,apr_value          double
,may_value          double
,jun_value          double
,jul_value          double
,aug_value          double
,sep_value          double
,oct_value          double
,nov_value          double
,dec_value          double
,comment            char(255)
)ENGINE=MyISAM DEFAULT CHARSET=latin1;

TRUNCATE ${ff10db}.ff10POP;
LOAD DATA LOCAL INFILE 'C:\\\\a\\\\ff10\\\\VPOP_2020NEI_full_20220729_01aug2022_nf_v1.csv' INTO TABLE ${ff10db}.ff10POP FIELDS TERMINATED BY ',' LINES TERMINATED BY '\r\n' IGNORE 27 LINES;

TRUNCATE ${ff10db}.ff10VMT;
LOAD DATA INFILE 'C:/a/ff10/VMT_2020NEI_full_monthly_run3_09jan2023_v0.csv' INTO TABLE ${ff10db}.ff10VMT FIELDS TERMINATED BY ',' LINES TERMINATED BY '\r\n' IGNORE 16 LINES;

TRUNCATE ${ff10db}.ff10HOT;
LOAD DATA INFILE 'C:/a/ff10/HOTELING_2020NEI_monthly_05oct2022_v0.csv' INTO TABLE ${ff10db}.ff10HOT FIELDS TERMINATED BY ',' LINES TERMINATED BY '\r\n' IGNORE 13 LINES;


-- Compute the roadTypeDistribution from ff10VMT
DROP TABLE IF EXISTS ${ff10db}.moves_roadTypeDistribution;
CREATE TABLE ${ff10db}.moves_roadTypeDistribution (
	countyID INTEGER,
	sourceTypeID INTEGER,
	roadTypeID INTEGER,
	roadTypeVMTFraction DOUBLE,
	PRIMARY KEY(countyID, sourceTypeID, roadTypeID),
	KEY(countyID, sourceTypeID)
	)ENGINE=MyISAM DEFAULT CHARSET=latin1;

-- Make a complete template that covers all categories
INSERT INTO ${ff10db}.moves_roadTypeDistribution SELECT DISTINCT a.region_cd*1 as countyID, b.sourceTypeID, c.roadTypeID, 0 as roadTypeVMTFraction
	FROM ${ff10db}.ff10vmt a, ${mvsdb}.sourceusetype b, ${mvsdb}.roadtype c
	WHERE c.roadTypeID in (2,3,4,5)
	ORDER BY a.region_cd, b.sourceTypeID, c.roadTypeID;

-- Insert VMT totals where they exist
DROP TABLE IF EXISTS ${ff10db}.vmtsums;
CREATE TABLE ${ff10db}.vmtsums (countyID INTEGER, sourceTypeID INTEGER, roadTypeID INTEGER, roadTypeVMTFraction DOUBLE, PRIMARY KEY(countyID, sourceTypeID, roadTypeID));
INSERT INTO ${ff10db}.vmtsums 
	SELECT region_cd*1 as countyID, MID(scc,5,2)*1 as sourceTypeID, MID(scc,7,2)*1 as roadTypeID, SUM(ann_value) as roadTypeVMTFraction
	FROM ${ff10db}.ff10VMT
	GROUP BY region_cd, sourceTypeID, roadTypeID
	ORDER BY region_cd, sourceTypeID, roadTypeID;
ALTER TABLE ${ff10db}.moves_roadTypeDistribution ADD KEY(countyID, sourceTypeID);
ALTER TABLE ${ff10db}.vmtsums ADD KEY(countyID, sourceTypeID);
UPDATE ${ff10db}.moves_roadTypeDistribution a, ${ff10db}.vmtsums b SET a.roadTypeVMTFraction=b.roadTypeVMTFraction where a.countyID=b.countyID AND a.sourceTypeID=b.sourceTypeID AND a.roadTypeID=b.roadTypeID;

-- Identify missing categories and store them in a list
DROP TABLE IF EXISTS ${ff10db}.moves_roadTypeDistribution_sums;
CREATE TABLE ${ff10db}.moves_roadTypeDistribution_sums (countyID INTEGER, sourceTypeID INTEGER, totals DOUBLE, county_source VARCHAR(10), PRIMARY KEY (county_source));
INSERT INTO ${ff10db}.moves_roadTypeDistribution_sums
	SELECT countyID, sourceTypeID, sum(roadTypeVMTFraction) as totals, CONCAT(countyID,'_',sourceTypeID) AS county_source
	FROM ${ff10db}.moves_roadTypeDistribution 
	GROUP BY countyID, sourceTypeID;
DELETE FROM ${ff10db}.moves_roadTypeDistribution_sums WHERE totals != 0;


-- Calculate MOVES default road type distributions by county, and no need to normalize as this happens in a later step
DROP TABLE IF EXISTS ${ff10db}.rtd_def;
CREATE TABLE ${ff10db}.rtd_def (countyID integer, sourceTypeID integer, roadTypeID integer, roadTypeVMTFraction DOUBLE, county_source VARCHAR(10), PRIMARY KEY(countyID, sourceTypeID, roadtypeID), KEY(county_source));
INSERT INTO ${ff10db}.rtd_def
	SELECT truncate(b.zoneID/10,0), a.sourceTypeID, a.roadTypeID, a.roadTypeVMTFraction*b.SHOAllocFactor as roadTypeVMTFraction, CONCAT(truncate(b.zoneID/10,0),'_',a.sourceTypeID)
	FROM ${mvsdb}.roadTypeDistribution a, ${mvsdb}.zoneRoadType b
	WHERE a.roadTypeID IN (2,3,4,5) AND a.roadTypeID=b.roadTypeID;

-- Take action to make the impending gap-filling faster: reduce the size of rtd_def to just those categories needed. 
DROP TABLE IF EXISTS ${ff10db}.rtd_def1;
CREATE TABLE ${ff10db}.rtd_def1 LIKE ${ff10db}.rtd_def;
INSERT INTO ${ff10db}.rtd_def1 
	SELECT a.countyID, a.sourceTypeID, a.roadTypeID, a.roadTypeVMTFraction, b.county_source 
	FROM ${ff10db}.rtd_def a RIGHT JOIN ${ff10db}.moves_roadTypeDistribution_sums b ON a.county_source=b.county_source;

-- Insert MOVES defaults where a category is missing (whole county-source type combination totals zero)
UPDATE ${ff10db}.moves_roadTypeDistribution a, ${ff10db}.rtd_def1 c
	SET a.roadTypeVMTFraction=c.roadTypeVMTFraction
	WHERE a.countyID=c.countyID AND a.sourceTypeID=c.sourceTypeID AND a.roadTypeID=c.roadTypeID;

-- Normalize to create VMT distributions over the road types
DROP TABLE IF EXISTS ${ff10db}.moves_roadTypeVMT;
CREATE TABLE ${ff10db}.moves_roadTypeVMT (countyID INTEGER, sourceTypeID INTEGER, total DOUBLE, PRIMARY KEY(countyID, sourceTypeID));
INSERT INTO ${ff10db}.moves_roadTypeVMT
	SELECT countyID, sourceTypeID, sum(roadTypeVMTFraction) as total
	FROM ${ff10db}.moves_roadTypeDistribution
	GROUP BY countyID, sourceTypeID
	ORDER BY countyID, sourceTypeID;

UPDATE ${ff10db}.moves_roadTypeDistribution a, ${ff10db}.moves_roadTypeVMT b
	SET a.roadTypeVMTFraction=a.roadTypeVMTFraction/b.total
	WHERE a.countyID=b.countyID AND a.sourceTypeID=b.sourceTypeID;

-- Compute the hotellingMonthAdjust from ff10HOT
DROP TABLE IF EXISTS ${ff10db}.moves_prehotellingmonthadjust;
DROP TABLE IF EXISTS ${ff10db}.moves_hotellingmonthadjust;
CREATE TABLE ${ff10db}.moves_hotellingmonthadjust LIKE ${mvsdb}.hotellingmonthadjust;
CREATE TABLE ${ff10db}.moves_prehotellingmonthadjust LIKE ${mvsdb}.hotellingmonthadjust;

-- Insert Hotelling month totals where they exist - this table does not include all possible counties, only the ones that exist in ff10hot
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 1 AS monthID, SUM(jan_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 2 AS monthID, SUM(feb_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 3 AS monthID, SUM(mar_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 4 AS monthID, SUM(apr_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 5 AS monthID, SUM(may_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 6 AS monthID, SUM(jun_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 7 AS monthID, SUM(jul_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 8 AS monthID, SUM(aug_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 9 AS monthID, SUM(sep_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 10 AS monthID, SUM(oct_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 11 AS monthID, SUM(nov_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;
INSERT INTO ${ff10db}.moves_prehotellingmonthadjust SELECT region_cd*10, 12 AS monthID, SUM(dec_value) FROM ${ff10db}.ff10hot GROUP BY region_cd ORDER BY region_cd;

-- Create a complete table including all zoneIDs
INSERT INTO ${ff10db}.moves_hotellingmonthadjust 
	SELECT DISTINCT a.region_cd*10 as zoneID, b.monthID, 0 as monthAdjustment
	FROM ${ff10db}.ff10vmt a, ${mvsdb}.monthofanyyear b;

-- Update the complete table with the hotellingMonthAdjust information, where it exists.
UPDATE ${ff10db}.moves_hotellingmonthadjust a, ${ff10db}.moves_prehotellingmonthadjust b SET a.monthAdjustment=b.monthAdjustment WHERE a.zoneID=b.zoneID AND a.monthID=b.monthID;

-- Identify the zoneIDs without any hoteling data. Place them in a list by themselves
DROP TABLE IF EXISTS ${ff10db}.hot_temp;
CREATE TABLE ${ff10db}.hot_temp (zoneID INTEGER, totals DOUBLE, PRIMARY KEY(zoneID));
INSERT INTO ${ff10db}.hot_temp SELECT zoneID, sum(monthAdjustment) FROM ${ff10db}.moves_hotellingmonthadjust GROUP BY zoneID;
DELETE FROM ${ff10db}.hot_temp WHERE totals != 0;

-- Some counties have zero (0) hotelling.  In these counties, we will set monthly distributions to flat (equal by day) and months account for differences in number of days/mo.
UPDATE ${ff10db}.moves_hotellingmonthadjust a, ${ff10db}.hot_temp b SET a.monthAdjustment=30/366 WHERE a.zoneID=b.zoneID AND a.monthID IN (4,6,9,11);
UPDATE ${ff10db}.moves_hotellingmonthadjust a, ${ff10db}.hot_temp b SET a.monthAdjustment=31/366 WHERE a.zoneID=b.zoneID AND a.monthID IN (1,3,5,7,8,10,12);
UPDATE ${ff10db}.moves_hotellingmonthadjust a, ${ff10db}.hot_temp b SET a.monthAdjustment=29/366 WHERE a.zoneID=b.zoneID AND a.monthID IN (2);

-- Normalize to create Hotelling distributions over the months that sum to 12 (rather than 1).
DROP TABLE IF EXISTS ${ff10db}.moves_hotellingAnnual;
CREATE TABLE ${ff10db}.moves_hotellingAnnual (zoneID INTEGER, annual_value DOUBLE, PRIMARY KEY(zoneID));
INSERT INTO ${ff10db}.moves_hotellingAnnual SELECT zoneID, sum(monthAdjustment) as annual_value FROM ${ff10db}.moves_hotellingmonthadjust GROUP BY zoneID;
UPDATE ${ff10db}.moves_hotellingmonthadjust a, ${ff10db}.moves_hotellingAnnual b SET a.monthAdjustment=(a.monthAdjustment+1e-15)/(b.annual_value+1e-15)*12 WHERE a.zoneID=b.zoneID;

-- Create monthVMTFraction table
DROP TABLE IF EXISTS ${ff10db}.moves_monthvmtfraction_unpadded;
CREATE TABLE ${ff10db}.moves_monthvmtfraction_unpadded (countyID INTEGER, sourceTypeID INTEGER, monthID INTEGER, monthVMTFraction DOUBLE, PRIMARY KEY(countyID,sourceTypeID,monthID));

-- Insert VMT month totals
-- The ff10 table is missing rows for source types without VMT (e.g., source types 41 and 42 in countyID 4009)

INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 1 AS monthID, SUM(jan_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 2 AS monthID, SUM(feb_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 3 AS monthID, SUM(mar_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 4 AS monthID, SUM(apr_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 5 AS monthID, SUM(may_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 6 AS monthID, SUM(jun_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 7 AS monthID, SUM(jul_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 8 AS monthID, SUM(aug_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 9 AS monthID, SUM(sep_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 10 AS monthID, SUM(oct_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 11 AS monthID, SUM(nov_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;
INSERT INTO ${ff10db}.moves_monthvmtfraction_unpadded SELECT region_cd*1 as countyID, mid(SCC,5,2) as sourceTypeID, 12 AS monthID, SUM(dec_value) FROM ${ff10db}.ff10vmt GROUP BY region_cd, sourceTypeID ORDER BY region_cd, sourceTypeID;

DROP TABLE IF EXISTS ${ff10db}.moves_monthvmtfraction;
CREATE TABLE ${ff10db}.moves_monthvmtfraction like ${ff10db}.moves_monthvmtfraction_unpadded;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 1 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 2 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 3 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 4 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 5 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 6 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 7 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 8 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 9 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 10 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 11 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
INSERT INTO ${ff10db}.moves_monthvmtfraction SELECT DISTINCT a.countyID, b.sourceTypeID, 12 as monthID, 0 as monthVMTFraction FROM ${ff10db}.moves_monthvmtfraction_unpadded a, ${mvsdb}.sourceusetype b;
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_monthvmtfraction_unpadded b SET a.monthVMTFraction=b.monthVMTFraction WHERE a.countyID=b.countyID AND a.sourceTypeID=b.sourceTypeID AND a.monthID=b.monthID;

-- Sum to annual
DROP TABLE IF EXISTS ${ff10db}.moves_VMTannual;
CREATE TABLE ${ff10db}.moves_VMTannual (countyID INTEGER, sourceTypeID INTEGER, annual_value DOUBLE, PRIMARY KEY(countyID, sourceTypeID));
INSERT INTO ${ff10db}.moves_VMTannual SELECT countyID, sourceTypeID, sum(monthVMTFraction) AS annual_value FROM ${ff10db}.moves_monthVMTFraction GROUP BY countyID, sourceTypeID ORDER BY countyID, sourceTypeID;

-- Set distributions to statewide averages from StreetLight where the FF10 annual_value of VMT is zero for a source type.  This way we always have a good distribution ready for 2020, in case annual VMT is added to the MOVES CDB later.
-- First, copy StreetLight distributions into ff10db to append a stateID for easier joining with other tables.
DROP TABLE IF EXISTS ${ff10db}.sl_pers;
DROP TABLE IF EXISTS ${ff10db}.sl_comm;
CREATE TABLE ${ff10db}.sl_pers (stateID INTEGER, stateAbbr VARCHAR(5), vehicleType VARCHAR(6), monthID INTEGER, milesFraction DOUBLE, PRIMARY KEY(stateID, vehicleType, monthID));
CREATE TABLE ${ff10db}.sl_comm LIKE ${ff10db}.sl_pers;
INSERT INTO ${ff10db}.sl_pers SELECT b.stateID, a.stateAbbr, a.vehicleType, a.monthID, a.milesFraction FROM ${slmdb}.mvmtfraction_noroads_pers_statesubstitutions a, ${mvsdb}.state b WHERE a.stateAbbr=b.stateAbbr AND a.vehicleType='PERS';
INSERT INTO ${ff10db}.sl_comm SELECT b.stateID, a.stateAbbr, a.vehicleType, a.monthID, a.milesFraction FROM ${slmdb}.mvmtfraction_noroads_comm_statesubstitutions a, ${mvsdb}.state b WHERE a.stateAbbr=b.stateAbbr AND a.vehicleType='COMM';

-- Merge StreetLight backups. Assign flat profile for ST 62 backup, per GDIT.
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b, ${ff10db}.sl_comm c 
	SET a.monthVMTFraction=c.milesFraction 
	WHERE a.sourceTypeID IN (32,41,42,43,51,52,53,54,61) 
	  AND (a.sourceTypeID=b.sourceTypeID AND a.countyID=b.countyID AND b.annual_value=0) AND (c.stateID=TRUNCATE(a.countyID/1000,0) AND a.monthID=c.monthID);
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b, ${ff10db}.sl_pers c 
	SET a.monthVMTFraction=c.milesFraction 
	WHERE a.sourceTypeID IN (11,21,31) 
	  AND (a.sourceTypeID=b.sourceTypeID AND a.countyID=b.countyID AND b.annual_value=0) AND (c.stateID=TRUNCATE(a.countyID/1000,0) AND a.monthID=c.monthID);
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b SET a.monthVMTFraction=30 WHERE a.sourceTypeID=62 AND a.monthID in (4,6,9,11) 	    AND (b.sourceTypeID=62 AND a.countyID=b.countyID AND b.annual_value=0);
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b SET a.monthVMTFraction=31 WHERE a.sourceTypeID=62 AND a.monthID in (1,3,5,7,8,10,12) AND (b.sourceTypeID=62 AND a.countyID=b.countyID AND b.annual_value=0);
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b SET a.monthVMTFraction=29 WHERE a.sourceTypeID=62 AND a.monthID in (2) 		    AND (b.sourceTypeID=62 AND a.countyID=b.countyID AND b.annual_value=0);

-- Recalculate annual totals from the monthly table, now that we have added backup distributions.
TRUNCATE ${ff10db}.moves_VMTannual;
INSERT INTO ${ff10db}.moves_VMTannual select countyID, sourceTypeID, sum(monthVMTFraction) AS annual_value FROM ${ff10db}.moves_monthVMTFraction GROUP BY countyID, sourceTypeID ORDER BY countyID, sourceTypeID;

-- Finally, calculate the monthVMTFractions
UPDATE ${ff10db}.moves_monthvmtfraction a, ${ff10db}.moves_VMTannual b SET a.monthVMTFraction=a.monthVMTFraction/b.annual_value WHERE a.countyID=b.countyID AND a.sourceTypeID=b.sourceTypeID;  

-- Create StartsMonthAdjust and IdleMonthAdjust tables (ONI) from monthVMTFractions
DROP TABLE IF EXISTS ${ff10db}.moves_startsMonthAdjust;
DROP TABLE IF EXISTS ${ff10db}.moves_idleMonthAdjust;
CREATE TABLE ${ff10db}.moves_startsMonthAdjust (countyID INTEGER, monthID INTEGER, sourceTypeID INTEGER, monthAdjustment DOUBLE, PRIMARY KEY(countyID, monthID, sourceTypeID));
CREATE TABLE ${ff10db}.moves_idleMonthAdjust (countyID INTEGER, sourceTypeID INTEGER, monthID INTEGER, idleMonthAdjust DOUBLE, PRIMARY KEY(countyID, monthID, sourceTypeID));
INSERT INTO ${ff10db}.moves_startsMonthAdjust SELECT countyID, monthID, sourceTypeID, monthVMTFraction*12 AS monthAdjustment FROM ${ff10db}.moves_monthVMTFraction ORDER BY countyID, sourceTypeID, monthID;
INSERT INTO ${ff10db}.moves_idleMonthAdjust   SELECT countyID, sourceTypeID, monthID, monthVMTFraction*12 AS idleMonthAdjust FROM ${ff10db}.moves_monthVMTFraction ORDER BY countyID, sourceTypeID, monthID;

-- Create hotellingHoursPerDay and hotellingActivityDistribution from ff10 hotelling.  Use county list from ff10 vmt as it is more complete.
DROP TABLE IF EXISTS ${ff10db}.moves_hotellingHoursPerDay;
CREATE TABLE ${ff10db}.moves_hotellingHoursPerDay LIKE ${mvsdb}.hotellingHoursPerDay;
INSERT INTO ${ff10db}.moves_hotellingHoursPerDay SELECT DISTINCT ${year} as yearID, a.region_cd*10 as zoneID, b.dayID, 0 as hotellingHoursPerDay FROM ${ff10db}.ff10vmt a, ${mvsdb}.dayofanyweek b;

DROP TABLE IF EXISTS ${ff10db}.hotelling_tmp;
CREATE TABLE ${ff10db}.hotelling_tmp (zoneID INTEGER, avgDay DOUBLE, PRIMARY KEY(zoneID));
INSERT INTO ${ff10db}.hotelling_tmp SELECT region_cd*10 as zoneID, sum(ann_value)/365 as avgDay FROM ${ff10db}.ff10hot GROUP BY region_cd;

UPDATE ${ff10db}.moves_hotellingHoursPerDay a, ${ff10db}.hotelling_tmp b SET a.hotellingHoursPerDay=b.avgDay*${weekDayAvgDayRatio} WHERE a.zoneID=b.zoneID AND a.dayID=5;
UPDATE ${ff10db}.moves_hotellingHoursPerDay a, ${ff10db}.hotelling_tmp b SET a.hotellingHoursPerDay=b.avgDay*${weekendDayAvgDayRatio} WHERE a.zoneID=b.zoneID AND a.dayID=2;

DROP TABLE IF EXISTS ${ff10db}.moves_hotellingActivityDistribution;
CREATE TABLE ${ff10db}.moves_hotellingActivityDistribution LIKE ${mvsdb}.hotellingActivityDistribution;
INSERT INTO ${ff10db}.moves_hotellingActivityDistribution SELECT  DISTINCT a.region_cd*10 AS zoneID, b.beginModelYearID, b.endModelYearID, b.opModeID, b.opModeFraction FROM ${ff10db}.ff10vmt a, ${mvsdb}.hotellingActivityDistribution b;
UPDATE ${ff10db}.moves_hotellingActivityDistribution SET opModeFraction=0;
UPDATE ${ff10db}.moves_hotellingActivityDistribution SET opModeFraction=${dieselauxfraction} WHERE opModeID=201;
UPDATE ${ff10db}.moves_hotellingActivityDistribution SET opModeFraction=1-${dieselauxfraction} WHERE opModeID=200;

-- Make SourceTypeYear and SourceTypeYearVMT tables back-filled with zeros
DROP TABLE IF EXISTS ${ff10db}.moves_sourceTypeYear;
CREATE TABLE  ${ff10db}.moves_sourceTypeYear (countyID INTEGER, yearID INTEGER, sourceTypeID INTEGER, salesGrowthFactor DOUBLE DEFAULT NULL, sourceTypePopulation DOUBLE DEFAULT NULL, migrationrate DOUBLE DEFAULT NULL, PRIMARY KEY(countyID, yearID, sourceTypeID));
INSERT INTO  ${ff10db}.moves_sourceTypeYear
	SELECT DISTINCT a.countyID, ${year} as yearID, b.sourceTypeID, NULL as salesGrowthFactor, 0 as sourceTypePopulation, NULL as migrationRate
	FROM ${ff10db}.moves_roadTypeDistribution a, ${mvsdb}.sourceusetype b;

DROP TABLE IF EXISTS ${ff10db}.popsums;
CREATE TABLE ${ff10db}.popsums (countyID INTEGER, yearID INTEGER, sourceTypeID INTEGER, population DOUBLE, PRIMARY KEY(countyID, yearID, sourceTypeID));
INSERT INTO ${ff10db}.popsums SELECT region_cd*1, calc_year, mid(scc,5,2)*1, sum(ann_value) FROM ${ff10db}.ff10pop GROUP BY region_cd, calc_year, mid(scc,5,2);
UPDATE ${ff10db}.moves_sourceTypeYear a, ${ff10db}.popsums b SET a.sourceTypePopulation=b.population WHERE (a.countyID=b.countyID and a.yearID=b.yearID AND a.sourceTypeID=b.sourceTypeID);

DROP TABLE IF EXISTS ${ff10db}.moves_sourceTypeYearVMT;
CREATE TABLE ${ff10db}.moves_sourceTypeYearVMT (countyID INTEGER, yearID INTEGER, sourceTypeID INTEGER, VMT DOUBLE, PRIMARY KEY(countyID, yearID, sourceTypeID));
INSERT INTO ${ff10db}.moves_sourceTypeYearVMT
	SELECT DISTINCT a.countyID, ${year} as yearID, b.sourceTypeID, 0 as VMT 	
	FROM ${ff10db}.moves_sourceTypeYear a, ${mvsdb}.sourceusetype b;
DROP TABLE IF EXISTS ${ff10db}.vmtsums1;
CREATE TABLE ${ff10db}.vmtsums1 (countyID INTEGER, yearID INTEGER, sourceTypeID INTEGER, VMT DOUBLE, PRIMARY KEY(countyID, yearID, sourceTypeID));
INSERT INTO ${ff10db}.vmtsums1 SELECT region_cd*1 as countyID, calc_year as yearID, mid(scc,5,2)*1 as sourceTypeID, sum(ann_value) as VMT from ${ff10db}.ff10vmt GROUP BY region_cd, calc_year, mid(scc,5,2);
UPDATE ${ff10db}.moves_sourceTypeYearVMT a, ${ff10db}.vmtsums1 b SET a.VMT=b.VMT WHERE a.countyID=b.countyID and a.yearID=b.yearID AND a.sourceTypeID=b.sourceTypeID;




-- Table cleanup
DROP TABLE IF EXISTS ${ff10db}.moves_monthvmtfraction_unpadded;
DROP TABLE IF EXISTS ${ff10db}.moves_VMTannual;
DROP TABLE IF EXISTS ${ff10db}.moves_hotellingannual;
DROP TABLE IF EXISTS ${ff10db}.moves_roadTypeVMT;
DROP TABLE IF EXISTS ${ff10db}.sl_pers;
DROP TABLE IF EXISTS ${ff10db}.sl_comm;
DROP TABLE IF EXISTS ${ff10db}.hotelling_tmp;
DROP TABLE IF EXISTS ${ff10db}.vmtsums;
DROP TABLE IF EXISTS ${ff10db}.rtd_def;
DROP TABLE IF EXISTS ${ff10db}.rtd_def_sums;
DROP TABLE IF EXISTS ${ff10db}.moves_roadTypeDistribution_sums;
DROP TABLE IF EXISTS ${ff10db}.popsums;
DROP TABLE IF EXISTS ${ff10db}.vmtsums1;
DROP TABLE IF EXISTS ${ff10db}.rtd_def1;
DROP TABLE IF EXISTS ${ff10db}.moves_prehotellingmonthadjust;
DROP TABLE IF EXISTS ${ff10db}.hot_temp;

";
open(out1,">script.sql");

print out1 $sql;
close(out1);
`mysql -P3307 -u root < script.sql`;

print "VPOP, VMT and Hotelling imported\n";
print "SourceTypeYear calculated from FF10POP\n";
print "RoadTypeDistribution, MonthVMTFraction, and SourceTypeYearVMT calculated from FF10VMT\n";
print "StartsMonthAdjust and IdleMonthAdjust calculated from MonthVMTFraction\n";
print "HotellingHoursPerDay, HotellingActivityDistribution, and HotellingMonthAdjust calculated from FF10 Hotelling\n";


$count=0;

open(cdb1,"$cdblist");
foreach $xxx(<cdb1>)
{
	print "\n";
	$dbid = $xxx; 
	chomp($dbid);
	#$cdb2=substr $dbid, 0, 7;
	$fips=substr $dbid, 1, 5;
	$fipsint = $fips*1;
	#$imcdb = "${cdb2}${year}_IM";
	#$actcdb ="${cdb2}2011_activity";
	$count=$count+1;
	$newdb=$dbid;  
	print "Updating $newdb\n";

$sql = "

DROP TABLE IF EXISTS ${dbid}.hotellingActivityDistribution;
DROP TABLE IF EXISTS ${dbid}.hotellingHoursPerDay;
DROP TABLE IF EXISTS ${dbid}.hotellingMonthAdjust;
DROP TABLE IF EXISTS ${dbid}.idleMonthAdjust;
DROP TABLE IF EXISTS ${dbid}.monthVMTFraction;
DROP TABLE IF EXISTS ${dbid}.roadTypeDistribution;
DROP TABLE IF EXISTS ${dbid}.sourceTypeYear;
DROP TABLE IF EXISTS ${dbid}.sourceTypeYearVMT;
DROP TABLE IF EXISTS ${dbid}.startsMonthAdjust;

CREATE TABLE ${dbid}.hotellingActivityDistribution LIKE ${mvsdb}.hotellingActivityDistribution;
CREATE TABLE ${dbid}.hotellingHoursPerDay LIKE ${mvsdb}.hotellingHoursPerDay;
CREATE TABLE ${dbid}.hotellingMonthAdjust LIKE ${mvsdb}.hotellingMonthAdjust;
CREATE TABLE ${dbid}.idleMonthAdjust LIKE ${mvsdb}.idleMonthAdjust;
CREATE TABLE ${dbid}.monthVMTFraction LIKE ${mvsdb}.monthVMTFraction;
CREATE TABLE ${dbid}.roadTypeDistribution LIKE ${mvsdb}.roadTypeDistribution;
CREATE TABLE ${dbid}.sourceTypeYear LIKE ${mvsdb}.sourceTypeYear;
CREATE TABLE ${dbid}.sourceTypeYearVMT LIKE ${mvsdb}.sourceTypeYearVMT;
CREATE TABLE ${dbid}.startsMonthAdjust LIKE ${mvsdb}.startsMonthAdjust;

-- Clear out the other 3 VMT tables aside from sourceTypeYearVMT
CREATE TABLE IF NOT EXISTS ${dbid}.sourceTypeDayVMT LIKE ${mvsdb}.sourceTypeDayVMT;
CREATE TABLE IF NOT EXISTS ${dbid}.HPMSVtypeYear LIKE ${mvsdb}.HPMSVtypeYear;
CREATE TABLE IF NOT EXISTS ${dbid}.HPMSVtypeDay LIKE ${mvsdb}.HPMSVtypeDay;
TRUNCATE ${dbid}.sourceTypeDayVMT;
TRUNCATE ${dbid}.HPMSVtypeYear;
TRUNCATE ${dbid}.HPMSVtypeDay;

INSERT INTO  ${dbid}.hotellingActivityDistribution SELECT * FROM ${ff10db}.moves_hotellingActivityDistribution WHERE zoneID=${fipsint}*10;
INSERT INTO  ${dbid}.hotellingHoursPerDay SELECT * FROM ${ff10db}.moves_hotellingHoursPerDay WHERE zoneID=${fipsint}*10 ORDER BY dayID;
INSERT INTO  ${dbid}.hotellingMonthAdjust SELECT * FROM ${ff10db}.moves_hotellingMonthAdjust WHERE zoneID=${fipsint}*10 ORDER BY monthID;
INSERT INTO  ${dbid}.idleMonthAdjust SELECT sourceTypeID, monthID, idleMonthAdjust FROM ${ff10db}.moves_idleMonthAdjust WHERE countyID=${fipsint} ORDER BY sourceTypeID, monthID;
INSERT INTO  ${dbid}.monthVMTFraction SELECT sourceTypeID, monthID, monthVMTFraction FROM ${ff10db}.moves_monthVMTFraction WHERE countyID=${fipsint} ORDER BY sourceTypeID, monthID;
INSERT INTO  ${dbid}.roadTypeDistribution SELECT sourceTypeID, roadTypeID, roadTypeVMTFraction FROM ${ff10db}.moves_roadTypeDistribution WHERE countyID=${fipsint} ORDER BY sourceTypeID, roadTypeID;
INSERT INTO  ${dbid}.sourceTypeYear SELECT yearID, sourceTypeID, salesGrowthFactor, sourceTypePopulation, migrationrate FROM ${ff10db}.moves_sourceTypeYear WHERE countyID=${fipsint} ORDER BY sourceTypeID;
INSERT INTO  ${dbid}.sourceTypeYearVMT SELECT yearID, sourceTypeID, VMT FROM ${ff10db}.moves_sourceTypeYearVMT WHERE countyID=${fipsint} ORDER BY sourceTypeID;
INSERT INTO  ${dbid}.startsMonthAdjust SELECT monthID, sourceTypeID, monthAdjustment FROM ${ff10db}.moves_startsMonthAdjust WHERE countyID=${fipsint} ORDER BY sourceTypeID, monthID;

FLUSH TABLES;

";
open(out1,">script.sql");

print out1 $sql;
close(out1);
`mysql -P3307 -u root < script.sql`;


}	# -- $pdb loop.
