#!/usr/bin/perl
#
# Filename   : Speciate_nr_cb6_20250203.plx
# Author     : Allison DenBleyker, ERG
#					Adapted from (1) ExtractActivity_nr.pl by Catherine Seppanen, UNC
#							 and (2) C:\Users\Public\EPA\MOVES\MOVES5.0\database\ProfileWeightScripts\profileWeights_nonroad_cb6.sql.	
# Description: Speciate nonroad residual total organic gases and PM2.5 for MOVES version 3.0.4 and later. 
#
# Usage: Speciate_nr_cb6_20250203.plx <ConfigurationFile>
# where
#   ConfigurationFile - text file containing configuration parameters like database connection information and output file

use strict;
use warnings 'FATAL' => 'all';
use DBI;

#================================================================================================
# Read configuration file

(scalar(@ARGV) == 1) or die <<END;
Usage: $0 <ConfigurationFile>
END

my ($configFile) = @ARGV;

my $configFH;
open($configFH, "<", $configFile) or die "Unable to open configuration file: $configFile\n";

my %config;
my $lineNo = 0;
while (my $line = <$configFH>)
{
  $lineNo++;
  # remove newlines, comments, leading and trailing whitespace
  chomp($line);
  $line =~ s/#.*//;
  $line =~ s/^\s+//;
  $line =~ s/\s+$//;
  next unless length($line);
  
  my ($var, $value) = split(/\s*=\s*/, $line, 2);
  unless (defined $value)
  {
    warn "Skipping invalid line $lineNo in configuration file (missing equals sign)\n";
    next;
  }
  $config{uc($var)} = $value;
}

close($configFH);

#================================================================================================
# Check required configuration parameters

unless (exists $config{'OUTPUT'} && length($config{'OUTPUT'}))
{
  die "Missing output file (OUTPUT) in configuration file\n";
}

unless (exists $config{'DB_NAME'} && length($config{'DB_NAME'}))
{
  die "Missing MySQL database name (DB_NAME) in configuration file\n";
}

#================================================================================================
# Check output files

my $outFile_pm = $config{'OUTPUT_PM'};

my $outFH_pm;
open($outFH_pm, ">", $outFile_pm) or die "Unable to open output file: $outFile_pm\n";

my $outFile_nhtog = $config{'OUTPUT_NHTOG'};

my $outFH_nhtog;
open($outFH_nhtog, ">", $outFile_nhtog) or die "Unable to open output file: $outFile_nhtog\n";

#================================================================================================
# Open database connection

my $dbHost = exists $config{'DB_HOST'} ? $config{'DB_HOST'} : "localhost";
my $dbUser = exists $config{'DB_USER'} ? $config{'DB_USER'} : "";
my $dbPass = exists $config{'DB_PASS'} ? $config{'DB_PASS'} : "";
my $dbName = $config{'DB_NAME'};
my $dbTable = "movesoutput";
my $dbTable_new = "movesoutput_new";
my $dbTable_nhtog = "movesoutput_nhtog";
my $dbTable_pm = "movesoutput_pm";

my $connectionInfo = "dbi:mysql:$dbName;$dbHost";
my $dbh = DBI->connect($connectionInfo, $dbUser, $dbPass) or die "Could not connect to database: $dbName\n";

# check that table exists
my $sth = $dbh->prepare("SELECT 1 FROM $dbTable");
$sth->execute() or die "Could not query database table: $dbTable\n";

#================================================================================================
# Create a runtime table for use to aid joints

my $sql = <<END;
CREATE TABLE ${dbTable_new} (
	yearID					SMALLINT(5),
	monthID 				SMALLINT(5),
	countyID 				INT(10),
	pollutantID 			SMALLINT(5),
	processID	 			SMALLINT(5),
	fuelSubtypeID			SMALLINT(5),
	SCC		 				VARCHAR(10),
	engTechID 				SMALLINT(5),
	tierID	 				SMALLINT(5),
	strokes	 				SMALLINT(5),
	emissionQuant			DOUBLE,
	PRIMARY KEY (yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes)
	)
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

#================================================================================================
# Create one table each for NonHAPTOG and PM

$sql = <<END;
CREATE TABLE ${dbTable_pm} (
	yearID					SMALLINT(5),
	monthID 				SMALLINT(5),
	countyID 				INT(10),
	pollutantID 			SMALLINT(5),
	pollutantName 			VARCHAR(50),
	processID	 			SMALLINT(5),
	fuelSubtypeID			SMALLINT(5),
	SCC		 				VARCHAR(10),
	engTechID 				SMALLINT(5),
	tierID	 				SMALLINT(5),
	strokes	 				SMALLINT(5),
	emissionQuant			DOUBLE,
	pmSpeciationProfileID	VARCHAR(10),
	PRIMARY KEY (yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes)
	)
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

$sql = <<END;
CREATE TABLE ${dbTable_nhtog} (
	yearID					SMALLINT(5),
	monthID 				SMALLINT(5),
	countyID 				INT(10),
	pollutantID 			SMALLINT(5),
	pollutantName 			VARCHAR(50),
	processID	 			SMALLINT(5),
	fuelSubtypeID			SMALLINT(5),
	SCC		 				VARCHAR(10),
	engTechID 				SMALLINT(5),
	tierID	 				SMALLINT(5),
	strokes	 				SMALLINT(5),
	emissionQuant			DOUBLE,
	togSpeciationProfileID	VARCHAR(10),
	PRIMARY KEY (yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes)
	)
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

#================================================================================================
# Process NONHAPTOG

$sth = $dbh->prepare("TRUNCATE $dbTable_new");
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

$sql = <<END;
INSERT INTO ${dbTable_new} 
		  (yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes, emissionQuant)
	SELECT yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes,
		sum(emissionQuant * dayID * noOfDays / 7) as emissionQuant
	FROM ${dbTable}
	JOIN movesdb20241112_nrupdates.monthofanyyear using (monthID)
	JOIN movesdb20241112_nrupdates.enginetech using (engTechID)
	WHERE pollutantID = 88 AND emissionQuant > 0
	GROUP BY yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

# Add the speciation profile and GROC factor

$sql = <<END;
INSERT INTO ${dbTable_nhtog} 
	SELECT yearID, monthID, countyID, pollutantID, CONCAT('NONHAPTOG',togSpeciationProfileID) as pollutantName, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes, emissionQuant, togSpeciationProfileID 
	FROM ${dbTable_new}
	JOIN movesdb20241112_nrupdates.nrrocspeciation USING (fuelSubtypeID, engTechID, tierID, strokes, processID)
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

#================================================================================================
# Process PM

$sth = $dbh->prepare("TRUNCATE $dbTable_new");
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

$sql = <<END;
INSERT INTO ${dbTable_new} 
		  (yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes, emissionQuant)
	SELECT yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes,
		sum(emissionQuant * dayID * noOfDays / 7) as emissionQuant
	FROM ${dbTable}
	JOIN movesdb20241112_nrupdates.monthofanyyear using (monthID)
	JOIN movesdb20241112_nrupdates.enginetech using (engTechID)
	WHERE pollutantID = 110 AND emissionQuant > 0
	GROUP BY yearID, monthID, countyID, pollutantID, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

# Add the speciation profile and GROC factor

$sql = <<END;
INSERT INTO ${dbTable_pm}
	SELECT yearID, monthID, countyID, pollutantID, CONCAT('PM2_5',pmSpeciationProfileID) as pollutantName, processID, fuelSubtypeID, SCC, engTechID, tierID, strokes, emissionQuant, pmSpeciationProfileID
	FROM ${dbTable_new}
	JOIN movesdb20241112_nrupdates.nrrocspeciation USING (fuelSubtypeID, engTechID, tierID, strokes, processID)
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

# cleanup

$sth = $dbh->prepare("DROP TABLE IF EXISTS $dbTable_new");
$sth->execute() or die 'Error executing query: ' . $sth->errstr;


#================================================================================================
# Output speciated data

my @fields = (
		'yearID',
		'monthID',
		'countyID',
		'pollutantID',
		'pollutantName',
		'processID',
		'fuelSubtypeID',
		'SCC',
		'engTechID',
		'tierID',
		'strokes',
		'emissionQuant',
		'togSpeciationProfileID');
		
print $outFH_nhtog join(',', @fields) . "\n";

$sql = <<END;
SELECT 	yearID,
		monthID,
		countyID,
		pollutantID,
		pollutantName,
		processID,
		fuelSubtypeID,
		SCC,
		engTechID,
		tierID,
		strokes,
		emissionQuant,
		togSpeciationProfileID
FROM $dbTable_nhtog
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

while (my @data = $sth->fetchrow_array())
{
  print $outFH_nhtog join(',', map { defined($_) ? $_ : '' } @data) . "\n";
}

close($outFH_nhtog);

@fields = (
		'yearID',
		'monthID',
		'countyID',
		'pollutantID',
		'pollutantName',
		'processID',
		'fuelSubtypeID',
		'SCC',
		'engTechID',
		'tierID',
		'strokes',
		'emissionQuant',
		'pmSpeciationProfileID');
		
print $outFH_pm join(',', @fields) . "\n";

$sql = <<END;
SELECT 	yearID,
		monthID,
		countyID,
		pollutantID,
		pollutantName,
		processID,
		fuelSubtypeID,
		SCC,
		engTechID,
		tierID,
		strokes,
		emissionQuant,
		pmSpeciationProfileID
FROM $dbTable_pm
END
$sth = $dbh->prepare($sql);
$sth->execute() or die 'Error executing query: ' . $sth->errstr;

while (my @data = $sth->fetchrow_array())
{
  print $outFH_pm join(',', map { defined($_) ? $_ : '' } @data) . "\n";
}

close($outFH_pm);

#================================================================================================
# Clean up temporary tables

unless (exists $config{'DEBUG'} && $config{'DEBUG'} eq 'Y')
{
  $sth = $dbh->prepare(<<END);
DROP TABLE IF EXISTS ${dbTable_nhtog}
END

  $sth->execute() or die 'Error executing query: ' . $sth->errstr;
  
    $sth = $dbh->prepare(<<END);
DROP TABLE IF EXISTS ${dbTable_pm}
END

  $sth->execute() or die 'Error executing query: ' . $sth->errstr;
}
