      SUBROUTINE EMFTLT (QARG)
C***********************************************************************
C                 EMFTLT Module of ISC Model
C
C        PURPOSE: Applies Variable Emission Rate and
C                 Unit Conversion Factors
C
C        PROGRAMMER: JEFF WANG, ROGER BRODE
C
C*       MODIFIED BY J. Hardikar, PES, to make consistent with the new
C*                   OPENPIT Source Methodology - 7/20/94
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Date and Hour
C                 Meteorological Variables for One Hour
C                 Variable Emission Rate Flags and Factors
C                 Unit Conversion Rate Factors
C
C        OUTPUTS: Adjusted Emission Rate, QTK
C
C        CALLED FROM:   PCALCL
C                       VCALCL
C                       ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'EMFTLT'

C     Apply Emission Unit Factor (EMIFAC) and Variable Emission Rate
C     Factor, Based on Value of QFLAG
      IF (QFLAG(ISRC) .EQ. ' ') THEN
         QTK = QARG * EMIFAC
      ELSE IF (QFLAG(ISRC) .EQ. 'SEASON') THEN
         QTK = QARG * EMIFAC * QFACT(ISEA,ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'QUARTR') THEN
         QTK = QARG * EMIFAC * QFACT(IQUA,ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'MONTH' .AND. IAVE .LE. 12) THEN
         QTK = QARG * EMIFAC * QFACT(IAVE,ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'SSTAB') THEN
         QTK = QARG * EMIFAC * QFACT((IKST+NKST*(ISEA-1)),ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'SSPEED') THEN
         QTK = QARG * EMIFAC * QFACT((IWS+NWS*(ISEA-1)),ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'STAR') THEN
         QTK = QARG * EMIFAC * QFACT((IWS+(IKST-1)*NWS),ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'SSTAR') THEN
         QTK = QARG * EMIFAC * QFACT((IWS+(IKST-1)*NWS
     &                           + (ISEA-1)*NKST*NWS),ISRC)
      END IF

      IF (DEPOS) THEN
C        Adjust Emission Rate Factor To Give Total Emissions For DEPOSition
         QTK = QTK * NUMHRS(IAVE)
      END IF

      RETURN
      END

      SUBROUTINE WSADJ
C***********************************************************************
C                 WSADJ Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Adjusts Wind Speed from Anemometer Height to Stack Height
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        INPUTS:  Arrays of Source Parameters
C                 Meteorological Variables for One Hour
C                 Wind Speed Profile Exponents (Default or User-defined)
C
C        OUTPUTS: Stack Top Wind Speed, US
C
C        CALLED FROM:   PCALCL
C                       VCALCL
C                       ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'WSADJ'

C     Adjust Wind Speed -- Assume Wind Speed Constant Below 10 meters
      IF (HS .GE. 10.0) THEN
         US = UREF * (HS/ZREF)**P
      ELSE IF (ZREF .GT. 10.0) THEN
         US = UREF * (10.0/ZREF)**P
      ELSE
         US = UREF
      END IF

C     Do Not Allow Stack Height Wind Speed < 1.0 m/s
      IF (US .LT. 1.0) THEN
         US = 1.0
      END IF

      RETURN
      END

      SUBROUTINE DISTF
C***********************************************************************
C                 DISTF Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Distance to Final Plume Rise
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C
C        OUTPUTS: Distance to Final Plume Rise, XF (m), and Distance
C                 to Final Buoyant Rise (XFB) and Final Momentum Rise (XFM)
C
C        CALLED FROM:   PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DISTF'

      IF (UNSTAB .OR. NEUTRL) THEN
         IF (FB .GE. 55.) THEN
            XFB = 119. * FB**0.4
         ELSE IF (FB .GT. 0.) THEN
            XFB = 49. * FB**0.625
         ELSE
            XFB = 4.*DS*(VS+3.*US)*(VS+3.*US)/(VS*US)
         END IF
         XFM = 4.*DS*(VS+3.*US)*(VS+3.*US)/(VS*US)
         XF = AMAX1(XFB,XFM)
      ELSE IF (STABLE) THEN
         XFB = 2.0715*US/RTOFS
         XFM = 0.5*PI*US/RTOFS
         XF = AMAX1(XFB,XFM)
      END IF

      RETURN
      END

      SUBROUTINE WAKFLG
C***********************************************************************
C                 WAKFLG Module of ISC Short Term Model - Version 2
C
C        PURPOSE: To Set Wake Flags for Building Downwash Algorithms
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Building Dimensions
C                 Source Parameters
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Logical Flags for Wake Switches, WAKE and WAKESS;
C                 And Building Types, TALL, SQUAT, and SSQUAT;
C                 And Value of ZLB
C
C        CALLED FROM:   PCALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'WAKFLG'

C     Set Initial Wake Switches Based on Building Dimensions
      IF (DSBH.EQ.0.0 .OR. DSBW.EQ.0.0 .OR.
     &    HS .GT. (DSBH + 1.5*AMIN1(DSBH,DSBW))) THEN
         WAKE   = .FALSE.
         WAKESS = .FALSE.
      ELSE IF (HS .GT. (DSBH + 0.5*AMIN1(DSBH,DSBW))) THEN
         WAKE   = .TRUE.
         WAKESS = .FALSE.
      ELSE
         WAKE   = .TRUE.
         WAKESS = .TRUE.
      END IF

C     Set Final Wake Switches Based on Plume Height
      IF (WAKE) THEN
         X2BH = DSBH + DSBH
C        Calculate Gradual Momentum Rise at X2BH            ---   CALL DHPMOM
         CALL DHPMOM(X2BH)
         HEMWAK = HS + DHPM
         IF (WAKESS) THEN
            IF (HEMWAK .LE. (DSBH + 2.0*AMIN1(DSBH,DSBW))) THEN
               WAKE   = .TRUE.
            ELSE
               WAKE   = .FALSE.
               WAKESS = .FALSE.
            END IF
         ELSE
            IF (HEMWAK .LE. (DSBH + 1.5*AMIN1(DSBH,DSBW))) THEN
               WAKE = .TRUE.
            ELSE
               WAKE = .FALSE.
            END IF
         END IF
      ELSE
         HEMWAK = 0.0
      END IF

C     Set Value of ZLB And Set Logical Flags for Building Type
      IF (WAKE) THEN
         ZLB = AMIN1(DSBH,DSBW)
         IF (DSBW .LT. DSBH) THEN
C           Tall Building
            TALL  = .TRUE.
            SQUAT = .FALSE.
            SSQUAT= .FALSE.
         ELSE IF (DSBW .LE. 5.*DSBH) THEN
C           Squat Building
            TALL  = .FALSE.
            SQUAT = .TRUE.
            SSQUAT= .FALSE.
         ELSE
C           Super-Squat Building
            TALL  = .FALSE.
            SQUAT = .FALSE.
            SSQUAT= .TRUE.
         END IF
      ELSE
         ZLB = 0.0
      END IF

      RETURN
      END

      SUBROUTINE XYDIST
C***********************************************************************
C                 XYDIST Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Sets Receptor Variables and Calculates Downwind (X)
C                 and Crosswind (Y) Distances,
C                 and Radial Distance from Source to Receptor (DISTR)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Source Location
C                 Arrays of Receptor Locations
C                 SIN and COS of Wind Direction FROM Which Wind
C                 is Blowing, WDSIN and WDCOS
C
C        OUTPUTS: Values of X, Y, and DISTR (m)
C
C        CALLED FROM:   PCALCL
C                       VCALCL
C                       ACALCL
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'XYDIST'

C     Set Receptor Coordinates, Terrain Elevation and Flagpole Heights
      XR = AXR(IREC)
      YR = AYR(IREC)
      ZELEV = AZELEV(IREC)
      ZFLAG = AZFLAG(IREC)

C     Calculate Downwind (X) and Crosswind (Y) Distances
      X = -((XR-XS)*WDSIN + (YR-YS)*WDCOS)
      Y =   (XR-XS)*WDCOS - (YR-YS)*WDSIN

C     Calculate Source-Receptor (Radial) Distance, DISTR
      DISTR = SQRT (X*X + Y*Y)

      RETURN
      END

      SUBROUTINE LTSMTH
C***********************************************************************
C                 LTSMTH Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Sector Smoothing Term (SM) for use in
C                 Concentration or Deposition Calculations
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance, X
C                 Crosswind Distance, Y
C                 Radial Distance, DISTR
C                 Lateral Virtual Distance, XY
C
C        OUTPUTS: Smoothing Term, SM
C
C        CALLED FROM:   PCHILT
C                       ACHILT
C                       PDEPLT
C                       ADEPLT
C***********************************************************************
C
C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'LTSMTH'

C     Calculate 'Radial-Plus-Virtual' Distance, RPV
      XPXY = X + XY
      RPV  = SQRT(XPXY*XPXY + Y*Y)

C     Calculate Smoothing Term, SM
      ARG = AMIN1(1.0, XPXY/RPV)
      SM  = ABS(DELTHP - ACOS(ARG))/DELTHP

      RETURN
      END

      SUBROUTINE DECAY (XARG)
C***********************************************************************
C                 DECAY Module of ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Decay Term for Use in Gaussian Plume Equation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance
C                 Stack Top Wind Speed
C                 Decay Coefficient
C
C        OUTPUTS: Decay Term, D
C
C        CALLED FROM:   CHI
C                       DEP
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'DECAY'

      D = 1.0

      IF (DECOEF .GT. 0.0) THEN
         D = EXP (AMAX1 (EXPLIM, -DECOEF*XARG/US))
      END IF

      RETURN
      END

      SUBROUTINE VERT(HEARG,SZARG,A0,ZARG,VOUT)
C***********************************************************************
C                 VERT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Vertical Term for Use in Gaussian Plume Equation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED BY R.W. Brode, PES, Inc. to use calling arguments - 9/30/94
C
C        MODIFIED BY D. Strimaitis, SRC (for Wet REMOVAL of Gases)
C
C        DATE:    November 8, 1993
C
C
C        INPUTS:  Plume Height
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C
C        OUTPUTS: Vertical Term, VOUT
C
C        CALLED FROM:   PCHILT, PDEPLT, ACHI, ADEP
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'VERT'

      VOUT = 0.0

      IF (ZARG .EQ. 0.0) THEN
C        Vertical Term for Case With No Flagpole Receptor
         IF (STABLE .OR. ZI.GE.10000.) THEN
            A1 = A0 * HEARG * HEARG
            IF (A1 .GT. EXPLIM)  VOUT = 2.*EXP(A1)
         ELSE IF ((SZARG/ZI) .GE. 1.6) THEN
            VOUT  = SRT2PI*(SZARG/ZI)
         ELSE
            A1 = A0 * HEARG * HEARG
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            SUM = 0.0
            DO 100 I = 1, 100
               T  = 0.0
               TWOIZI = 2.*I*ZI
               A2 = A0 * (TWOIZI-HEARG) * (TWOIZI-HEARG)
               A3 = A0 * (TWOIZI+HEARG) * (TWOIZI+HEARG)
               IF (A2 .GT. EXPLIM)  T = EXP(A2)
               IF (A3 .GT. EXPLIM)  T = T + EXP(A3)
               SUM = SUM + T
               IF (ABS(T) .LE. 5.0E-9) THEN
C                 Exit Loop
                  GO TO 200
               END IF
 100        CONTINUE
C           Calculate Total Vert. Term - (2.*) was Removed for Optimization
 200        VOUT  = 2.*(VOUT + SUM)
         END IF
      ELSE
C        Vertical Term for Case of ZARG .NE. 0.0
         IF (STABLE .OR. ZI .GE. 10000.) THEN
            A1 = A0 * (ZARG-HEARG) * (ZARG-HEARG)
            A2 = A0 * (ZARG+HEARG) * (ZARG+HEARG)
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            IF (A2 .GT. EXPLIM)  VOUT = VOUT + EXP(A2)
         ELSE IF (SZARG/ZI .GE. 1.6) THEN
            VOUT  = SRT2PI*(SZARG/ZI)
         ELSE
            A1 = A0 * (ZARG-HEARG) * (ZARG-HEARG)
            A2 = A0 * (ZARG+HEARG) * (ZARG+HEARG)
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            IF (A2 .GT. EXPLIM)  VOUT = VOUT + EXP(A2)
            SUM = 0.0
            DO 300 I = 1, 100
               T  = 0.0
               TWOIZI = 2.*I*ZI
               A3 = A0 * (ZARG-(TWOIZI-HEARG)) * (ZARG-(TWOIZI-HEARG))
               A4 = A0 * (ZARG+(TWOIZI-HEARG)) * (ZARG+(TWOIZI-HEARG))
               A5 = A0 * (ZARG-(TWOIZI+HEARG)) * (ZARG-(TWOIZI+HEARG))
               A6 = A0 * (ZARG+(TWOIZI+HEARG)) * (ZARG+(TWOIZI+HEARG))
               IF (A3 .GT. EXPLIM)  T = T + EXP(A3)
               IF (A4 .GT. EXPLIM)  T = T + EXP(A4)
               IF (A5 .GT. EXPLIM)  T = T + EXP(A5)
               IF (A6 .GT. EXPLIM)  T = T + EXP(A6)
               SUM = SUM + T
               IF (ABS(T) .LE. 1.0E-8) THEN
C                 Exit Loop
                  GO TO 400
               END IF
 300        CONTINUE
 400        VOUT  = VOUT + SUM
         END IF
      END IF

      RETURN
      END
