      SUBROUTINE METEXT
C***********************************************************************
C                METEXT Module of ISC Model - Long Term
C
C        PURPOSE: Controls Extraction and Quality Assurance of
C                 Joint Frequency of Occurrences (STAR) Data
C
C        PROGRAMMER: ROGER BRODE, JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Meteorology File Specifications
C
C        OUTPUTS: Frequency of Occurences for Specific Averaging Period
C
C*       7/27/94     J. Hardikar, PES, Inc.
C*                   Added code to calculate reference wind speed at 10m
C*                   to be used for OPENPIT source algorithms
C
C        CALLED FROM:   MAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'METEXT'
      PATH = 'MX'
      FTOTAL = 0.0

C     READ Frequency of Occurrence Data
      IF (METFRM .EQ. 'FREE') THEN
         DO 40 K = 1, NKST
            DO 50 J = 1, NSEC
               READ(MFUNIT,*,END=1000,ERR=99,IOSTAT=IOERRN)
     &             (FREQ(I,J,K), I=1, NWS)
               ILINE = ILINE + 1
C              Accumulate Total Frequency, FTOTAL
               DO 60 I = 1, NWS
                  FTOTAL = FTOTAL + FREQ(I,J,K)
 60            CONTINUE
 50         CONTINUE
 40      CONTINUE
      ELSE
         DO 140 K = 1, NKST
            DO 150 J = 1, NSEC
               READ(MFUNIT,METFRM,END=1000,ERR=99,IOSTAT=IOERRN)
     &             (FREQ(I,J,K), I=1, NWS)
               ILINE = ILINE + 1
C              Accumulate Total Frequency, FTOTAL
               DO 160 I = 1, NWS
                  FTOTAL = FTOTAL + FREQ(I,J,K)
 160           CONTINUE
 150        CONTINUE
 140     CONTINUE
      END IF

C     Check Total Frequency From Array
      IF (FTOTAL .GE. 2.0 .AND. (FTOTAL .EQ. INT(FTOTAL))) THEN
C        Convert Frequency Counts to Percent Frequency
         DO 240 K = 1, NKST
            DO 250 J = 1, NSEC
               DO 260 I = 1, NWS
                  FREQ(I,J,K) = (1./FTOTAL) * FREQ(I,J,K)
 260           CONTINUE
 250        CONTINUE
 240     CONTINUE
      ELSE IF (FTOTAL .LT. 0.98 .OR. FTOTAL .GT. 1.02) THEN
C        WRITE Warning: Sum of Frequencies Doesn't Total To 1.0 (+/ 0.02 %)
         CALL ERRHDL(PATH,MODNAM,'W','480',AVEPER(IAVE))
      END IF

C     Set Season and Quarter Indexes
      IF (IAVE .GT. 12 .AND. IAVE .LT. 17) THEN
         ISEA = IAVE - 12
         IQUA = IAVE - 12
      ELSE IF (IAVE .LE. 2) THEN
         ISEA = 1
         IQUA = 1
      ELSE IF (IAVE .EQ. 3) THEN
         ISEA = 2
         IQUA = 1
      ELSE IF (IAVE .EQ. 4 .OR. IAVE .EQ. 5) THEN
         ISEA = 2
         IQUA = 2
      ELSE IF (IAVE .EQ. 6) THEN
         ISEA = 3
         IQUA = 2
      ELSE IF (IAVE .EQ. 7 .OR. IAVE .EQ. 8) THEN
         ISEA = 3
         IQUA = 3
      ELSE IF (IAVE .EQ. 9) THEN
         ISEA = 4
         IQUA = 3
      ELSE IF (IAVE .EQ. 10 .OR. IAVE .EQ. 11) THEN
         ISEA = 4
         IQUA = 4
      ELSE IF (IAVE .EQ. 12) THEN
         ISEA = 1
         IQUA = 4
      ELSE
         ISEA = 1
         IQUA = 1
      END IF

      GO TO 999

 99   CALL ERRHDL(PATH,MODNAM,'E','510','MET-INP')
      RUNERR = .TRUE.

      GO TO 999

 1000 EOF = .TRUE.
      CALL ERRHDL(PATH,MODNAM,'E','575',AVEPER(IAVE))
      RUNERR = .TRUE.

 999  RETURN
      END

      SUBROUTINE METSET
C***********************************************************************
C                METSET Module of ISC Model - Long Term
C
C        PURPOSE: Sets Meteorological Variables for Calculation Purposes
C
C        PROGRAMMER: JEFF WANG, ROGER BRODE
C
C        MODIFIED:   To avoid potential math error due to negative
C                    ambient temperatures in calculating the square
C                    root of the stability parameter, RTOFS - 4/19/93
C                    (Note: Runtime errors are prevented in ISCLT2 by
C                    error checking in SUB. AVETMP, however, this change
C                    makes the code more consistent with ISCST2.)
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION)
C            Note: IKST>6 adjustment has been moved ahead of all lines of
C                  code that use IKST as an array index, and the assignment
C                  of KST variable.
C
C        MODIFIED BY Roger Brode, PES, Inc. to bypass calls to OBUKHOV and
C                    U_STAR if LDEP = .FALSE. - 7/26/94
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Meteorology File Specifications
C
C        OUTPUTS: Frequency of Occurences for Four Seasons
C
C        CALLED FROM:   MAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'

C     Variable Initializations
      MODNAM = 'METSET'
      PATH = 'MX'

C     Determine Flow Vector, AFV, and Associated Sector, IFVSEC
      AFV = (22.5*(ISEC-1)) + 180.
      IF (AFV .GT. 360.0)  AFV = AFV - 360.0
      IFVSEC = ISEC + 8
      IF (IFVSEC .GT. NSEC) IFVSEC = IFVSEC - NSEC

C     Set Stability Category Logical Flags and Adjust for KST > 6
      UNSTAB = .FALSE.
      NEUTRL = .FALSE.
      STABLE = .FALSE.
      IF (IKST .GT. 6) IKST = 6
      IF (IKST .LT. 4) THEN
         UNSTAB = .TRUE.
      ELSE IF (IKST .EQ. 4) THEN
         NEUTRL = .TRUE.
      ELSE IF (IKST .GT. 4) THEN
         STABLE = .TRUE.
      END IF

C     Set Stability Category Variable, KST
      KST = IKST

C     Select UREF, TA and ZI From Arrays
      UREF = AVESP(IWS)
      TA   = AVETA(IAVE,IKST)
      ZI   = AVEZI(IAVE,IKST,IWS)

      IF (LDEP) THEN
C     Calculate Monin-Obukhov Length and Friction Velocity for Deposition Use
C        First set roughness length                                     DTB93334
         Z0M = AVEZ0M(IAVE)                                             DTB93334

C        Calulate M-O Length and Friction Velocity                      DTB93334
         CALL  OBUKHOV                                                  DTB93334
         CALL  U_STAR                                                   DTB94063
      END IF

C     Apply ROTANG Adjustment to Flow Vector
      IF (ROTANG .NE. 0.0) THEN
         AFV = AFV - ROTANG
         IF (AFV .LE. 0.0) THEN
            AFV = AFV + 360.
         END IF
      END IF

C     Convert Flow Vector in Degrees to Wind Direction in Radians
      WDRAD = (AFV + 180.0) * DTORAD
C     Determine SIN and COS of WDRAD for Later Use
      WDSIN = SIN(WDRAD)
      WDCOS = COS(WDRAD)

C     Select Appropriate Power Law Exponent
      IF (USERP) THEN
         P = PUSER(IKST,IWS)
      ELSE IF (URBAN) THEN
         P = PURB(IKST)
      ELSE IF (RURAL) THEN
         P = PRUR(IKST)
      END IF

C*    Scale the Wind Speed from Anemometer Height to 10 meter

      UREF10 = UREF * (10.0/ZREF)**P

C*    Do Not Allow 10m Wind Speed < 1.0 m/s
      IF (UREF10 .LT. 1.0) THEN
         UREF10 = 1.0
      END IF

C     Select Appropriate delta theta/delta z
      IF (USERDT) THEN
         DTDZ = DTUSER(IKST,IWS)
      ELSE IF (URBAN) THEN
         DTDZ = DTURB(IKST)
      ELSE IF (RURAL) THEN
         DTDZ = DTRUR(IKST)
      END IF
      IF (DTDZ .NE. 0.0 .AND. TA .GT. 0.0) THEN
         S = G*DTDZ/TA
         RTOFS = SQRT(S)
      ELSE
         S = 1.0E-10
         RTOFS = 1.0E-10
      END IF

 999  RETURN
      END

      SUBROUTINE METDAT(SEACHR,K)
C***********************************************************************
C                 METDAT Module of ISC Model
C
C        PURPOSE: Print Out The Summary Of The Meteorology Data
C
C        PROGRAMMER: JEFF WANG
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Meteorology Input Data
C
C        OUTPUTS: Printed Model Outputs
C
C        CALLED FROM:   METEXT
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      CHARACTER SEACHR*6, ATHRUF(6)*1

C     Variable Initializations
      DATA ATHRUF / 'A','B','C','D','E','F' /
      MODNAM = 'METDAT'

C     Meteorology Data Summary
      IF (MOD((K-1),2) .EQ. 0) THEN
         CALL HEADER
         NEWMET = .FALSE.
         WRITE(IOUNIT,9011)
         WRITE(IOUNIT,9016) METINP, METFRM
         WRITE(IOUNIT,9020) IDSURF, IDUAIR, SFNAME, UANAME,
     &                      ISYEAR, IUYEAR
      END IF
      WRITE(IOUNIT,9027) SEACHR, ATHRUF(K)
      WRITE(IOUNIT,9030)
      WRITE(IOUNIT,9031) (I, I=1,NWS)
      WRITE(IOUNIT,9034) (AVESP(I), I=1,NWS)
      WRITE(IOUNIT,9036)
      DEGS = 0.
      DO 20 J = 1, NSEC
         WRITE(IOUNIT,9032) DEGS,(FREQ(I,J,K),I=1,NWS)
         DEGS = DEGS + 360./NSEC
  20  CONTINUE

      IF (K .EQ. 6) THEN
C        Write Out Sum of Frequencies, FTOTAL
         WRITE(IOUNIT,9050) FTOTAL
      END IF

 9011 FORMAT(12X,'*** FREQUENCY OF OCCURRENCE ',
     &       'OF WIND SPEED, DIRECTION AND STABILITY ***'/)
 9016 FORMAT(12X,'FILE: ',A40,' FORMAT: ',A60)
 9020 FORMAT(12X,'SURFACE STATION NO.: ',I6,20X,
     &       'UPPER AIR STATION NO.: ',I6/27X,'NAME: ',A40,3X,
     &       'NAME: ',A40/27X,'YEAR: ',I6,37X,'YEAR: ',I6)
 9027 FORMAT(/38X,A6,': STABILITY CATEGORY  ',A1)
 9030 FORMAT(/12X,6(2X,'WIND SPEED '))
 9031 FORMAT(12X,6(2X,'CATEGORY',I2,1X))
 9034 FORMAT(1X,'DIRECTION',2X,6(' (',F6.3,' M/S)'))
 9036 FORMAT(1X,'(DEGREES)',2X,6('  -----------'))
 9032 FORMAT(1X,F10.3,2X,6(2X,F10.8,1X))
 9050 FORMAT(/17X,'SUM OF FREQUENCIES, FTOTAL = ',F11.5)

      RETURN
      END

      SUBROUTINE OBUKHOV
C************************************************************************
C                OBUKHOV Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Monin-Obukhov Length
C
C        PROGRAMMER: DESMOND BAILEY
C
C        DATE:       March 4, 1994
C
C        MODIFIED BY:   Roger Brode, PES, Inc. to use MAIN1LT.INC to
C                       pass values in COMMON block - 7/22/94
C
C        INPUTS:
C                    Z0M    Surface Roughness Length
C                    KST    P-G Stability Class
C
C        OUTPUTS:    EL     M-O Length (m)
C
C        CALLED FROM:    METSET
C*************************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'

C      PARAMETER (G = 9.80655, VK = 0.4, CP = 1004., PRESMB = 1013.,
C     &           ELMIN = 2., ANEM = 10.)

C      COMMON UREF, TA, KST, USTAR, EL, Z0M, RHO


      DIMENSION A(6), B(6)

      DATA A /-0.0875, -0.03849, -0.00807, 0.0,  0.00807, 0.03849/
      DATA B /-0.1029, -0.17140, -0.30490, 0.0, -0.3049, -0.1714/

      Z0 = Z0M
      IF( Z0 .LT. 0.001 ) Z0 = 0.001

       IF(KST .EQ .4) THEN
          EL = 9000.0
       ELSE

         EL = 1.0/(A(KST)*Z0**B(KST))

       END IF

      RETURN
      END


      SUBROUTINE U_STAR
C************************************************************************
C                U_STAR Module of ISC Model - Long Term
C
C        PURPOSE: Calculates Friction Velocity, USTAR
C
C        PROGRAMMER: DESMOND BAILEY
C
C        DATE:       March 4, 1994
C
C        INPUTS:
C                    Z0M    Surface Roughness Length
C                    KST    P-G Stability Class
C
C        OUTPUTS:    EL     M-O Length (m)
C
C        CALLED FROM:    METSET
C*************************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'

c --- compute nighttime u* using the Weil - Brower method; or
c --- compute daytime u* using the Holtslag - van Ulden method.


c --- compute air density in kg/m**3 -- rho = p/(r * t) --

c     RHO = P/R*T
c     R   = 287 m**2/(deg k * sec**2)
c     1/R = (100. kg/(m*sec**2) per mb)/(287 m**2/(deg k * sec**2))
c     1/R = 0.3484321

      RHO  =  0.3484321 * PRESMB / TA

      IF (KST .GT. 4) THEN

            CALL USLNITE
      ELSE
            CALL USLDAY

      ENDIF

      RETURN
      END

C     ********************************************************************
      SUBROUTINE USLDAY

c     Purpose:   Calculates u* for neutral and unstable conditions
C                using Wang and Chen's technique
c
c     Reference: Wang and Chen, 1980:  Estimations of heat and momentum
c                fluxes near the ground.  Proc. 2nd Joint Conf. on
c                Applications of Air Poll. Meteorol., AMS, 764-769.
c
c     Arguments passed:     none

      INCLUDE 'MAIN1LT.INC'

c
c --- Calculate neutral friction velocity
C *** Beware of possible devide by zero  ***
      USTARN = VK*UREF / ALOG (ANEM/Z0M)
c
      IF ( KST .EQ. 4) THEN
         USTAR = USTARN
      else

C     Calculate friction velocity for unstable conditions.

         CALL WC

      endif
      return
      end

C     ********************************************************************

      SUBROUTINE WC
c
c     Purpose: Implementation of Wang and Chen's technique to
c              parameterize ustar under convective conditions


      INCLUDE 'MAIN1LT.INC'

      DIMENSION SHFF(6)
      DATA SHFF/260., 210., 130., 0.0, -10., -20./

      SHF = SHFF(KST)

c   Arguments passed:
c       ustar  real      calculated friction velocity
c       UREF   real      measured wind speed, m/s
c       anem   real      anemometer height, m
c       Z0M    real      roughness length, m
c       VK     real      von karman constant
c       g      real      gravitational acceleration, m/sec**2
c       shf    real      sensible heat flux, w/m**2
c       cp     real      specific heat at constant pressure, j/k-kg
c       rho    real      density of air, kg/m**3
c       TA     real      air temperature, k
c

        ratio  =  Z0M/anem
        ratln  =  ALOG(ratio)
        d1     =  0.107

        IF (ratio .LE. 0.01) d1 = 0.128 + 0.005*ratln

        d2     =  1.95+32.6*(ratio)**0.45
        x      =  shf/(rho*cp)
        y      =  VK * g * anem / TA
        z      =  (-ratln/(VK*UREF))**3
        ustar  =  VK*UREF*(1.+d1*alog(1.+d2*x*y*z))/(-ratln)

        return
        end

C     *******************************************************************

      SUBROUTINE USLNITE
c
c     Purpose: This routine calculates ustar for the stable cases (l > 0)
c              using the Weil-Brower technique (1983)
c
c     Arguments passed:

c       anem   real      anemometer height
c       elmin  real      the lower limit on L when stable
c
c
      INCLUDE 'MAIN1LT.INC'

c      real ROOTS(3)
c
c --- Assign constants: von karman constant, specific heat, and grav.
c
C     Assume clear sky conditions

      FRCC = 0.0

c --- const = maximum product of ustar and thetastar
c --- bbeta is used for profile relatationships in stable conditions
c
      const = 0.05
      bbeta = 4.7
c
      cdn    =  VK/ALOG(anem/Z0M)
      ths1   =  0.09*(1.-0.5*frcc**2)
      ths2   =  (TA*cdn*UREF**2)/(4.0*bbeta*anem*g)
      thstar =  AMIN1(ths1,ths2)
      unot   =  SQRT((bbeta*anem*g*thstar)/(TA))
c
c --- Since thstar is taken as the smaller of ths1 and ths2,
c --- (2.*unot/(sqrt(cdn)*UREF))**2 can only be as big as 1.0
c
      dum    = 1.-(2.*unot/(SQRT(cdn)*UREF))**2
c
c --- Prevent round-off error
c
      IF (dum.lt.0.) dum = 0.0

      USTAR  =  (cdn*UREF/2.)*(1.+SQRT(dum))
c
c --- Special attention required for the high ustar cases
c
C     if(USTAR*thstar .GT. const) then
C         aa = -cdn*UREF
C         b  = 0.0
C         c  = beta*anem*g*const*cdn/TA
C
C         call cubic(aa,b,c, ROOTS, nroots)
C         call pickus(ROOTS, nroots,USTAR,cdn,UREF)
C
C         thstar = const/USTAR
C     endif
C
C     if(EL .GE. elmin) goto 100
C      EL   = elmin
C     USTAR = SQRT(ELMIN*VK*g*thstar/TA)
c
100   continue

      return
      end
