      SUBROUTINE AVERTS(XVIN,YVIN,XWD,YWD,NUMV)
C***********************************************************************
C*                AVERTS Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: Calculates coordinates of vertices for Wind
C*                Direction Coordinate system for AREA and OPENPIT
C*                sources.
C*
C*       PROGRAMMER: Jeff Wang, Roger Brode
C*       MODIFIED:   Jayant Hardikar, Roger Brode (for OPENPIT sources)
C*
C*       DATE:      July 7, 1993
C*
C*       INPUTS:  Source Coordinates for Specific Source
C*                Number of vertices + 1
C*
C*       OUTPUTS: Array of Vertex Coordinates for Specific Source
C*
C*       CALLED FROM:   ACALC, PITEFF
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

      REAL XVIN(NVMAX),YVIN(NVMAX)
      REAL XWD(NVMAX),YWD(NVMAX)

C*    Variable Initializations
      MODNAM = 'AVERTS'
      

      DO 1670 NSP = 1, NUMV
         XWD(NSP) = -(XVIN(NSP)*WDSIN + YVIN(NSP)*WDCOS)
         YWD(NSP) =   XVIN(NSP)*WDCOS - YVIN(NSP)*WDSIN
1670  CONTINUE

      RETURN
      END

      SUBROUTINE AREAIN
C***********************************************************************
C                 AREAIN Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration for AREA Sources
C                 Using Numerical Integration Algorithm
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C
C        CALLED FROM:   ACALC
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      LOGICAL QGO
      REAL VAL, DVAL

C     Variable Initializations
      MODNAM = 'AREAIN'

C     INITIALIZE VARIABLES FOR INTEGRATION PROCEDURE.
      RP1 = -(XR*WDSIN+YR*WDCOS)/1000.0
      RP2 =  (XR*WDCOS-YR*WDSIN)/1000.0
      UCRIT = 0.00101
      VAL = 0.0
      KSIDE = 0
      do 1658 ncp = 1, NVERT
         ua = RP1-SPA(ncp,1)
         ub = RP1-SPA(ncp+1,1)
         va = RP2-SPA(ncp,2)
         vb = RP2-SPA(ncp+1,2)
         IF (ua .ge. ucrit) THEN
            kside = kside + 1
            uvert(kside) = ua
            vvert(kside) = va
         END IF
         IF ((ua .ge. ucrit .AND. ub .lt. ucrit) .OR.
     1       (ua .lt. ucrit .AND. ub .ge. ucrit)) THEN
            kside = kside+1
            uvert(kside) = ucrit
            vvert(kside) = va+(ucrit-ua)*(vb-va)/(ub-ua)
         END IF
1658  CONTINUE

      QGO = .FALSE.
      IF (kside .ge. 2) THEN
         QGO = .TRUE.
         vnmin=  4.0
         vnmax= -4.0
         do 1659 ncp = 1,kside
            ua = uvert(ncp)
            va = vvert(ncp)
            call pwidth(ua,va,vnorm,wa)
            vNVERT(ncp) = vnorm
            wvert(ncp) = wa
            vnmax = amax1(vnorm,vnmax)
            vnmin = amin1(vnorm,vnmin)
1659     CONTINUE
         IF (vnmin .ge. 4.0 .or. vnmax .le. -4.0) QGO = .FALSE.
      END IF

C     Integrate Between Vertices u(1),u(2) THEN u(2),u(3); etc.
      IF (QGO) THEN
C        MAKE 1st Point Same as Last
         ksp = kside+1
         uvert(ksp) = uvert(1)
         vvert(ksp) = vvert(1)
         vNVERT(ksp) = vNVERT(1)
         wvert(ksp) = wvert(1)
         nsegs = 0
         LSEG = .FALSE.
         do 1660 ks = 1,kside
            QGO = .TRUE.
            ivert = ks
            ua = uvert(ks)
            ub = uvert(ks+1)
            dval = 0.0
            IF (abs(ua-ub) .le. 0.0001) QGO = .FALSE.
            IF (QGO) call pside(ua,ub,dval)
            val = val+dval
1660     CONTINUE
         IF (nsegs .gt. 0) THEN
            LSEG = .TRUE.
            call pside2(dval)
            val = val+dval
         END IF
      END IF

      HRVAL = ABS(VAL)*QTK/US

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
         CALL DEBOUT
      END IF

      RETURN
      END

      SUBROUTINE QROMB(A1,B1,SS1)
C***********************************************************************
C                 QROMB Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Performs Romberg Integration of Function Using
C                 Polynomial Extrapolation for h=0 With h1(i)=h1(i-1)/4
C                 Modifed To Use Variable Order Extrapolation
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED by Roger Brode, PES, Inc. to change lower limit on
C                    J from 3 to 4, and correct lower threshold check
C                    for SS1. - 7/29/94
C
C        INPUTS:  Left Maximum Value of the Integral
C                 Right Maximum Limit of the Integral
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C
C        CALLED FROM:   PSIDE, PSIDE2
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL S1(21), H1(21)
C     Variable Initializations
      MODNAM = 'QROMB'

      H1(1) = 1
      CALL TRAPZD(A1,B1,S1(1),1)
      SS1 = S1(1)

      DO 11 J = 2, JMAX1
         H1(J) = 0.25*H1(J-1)
         CALL TRAPZD(A1,B1,S1(J),J)
         KP = MIN(J,K1)-1
         CALL POLINT(H1(J-KP),S1(J-KP),KP+1,SS1,DSS1)
C***********************************************************************
C        Check The Convergence Criteria:
C        EPS is tolerance level for convergence of the integral,
C          initially set = 1.0E-4 in a PARAMETER statement in MAIN1LT.INC;
C        EPS2 is lower threshold for the integral, initially set = 1.0E-10
C          in a PARAMETER statement in MAIN1LT.INC;
C        J is number of halving intervals and must be at least 4 for
C          convergence criteria to be met (i.e., minimum of 9 data points).
C          Maximum number of intervals is set by JMAX1 (=10).
C***********************************************************************
         IF ((ABS(DSS1) .LE. EPS*ABS(SS1) .OR. ABS(SS1) .LE. EPS2)
     &          .AND. J .GE. 4) GO TO 999
  11  CONTINUE

 999  RETURN
      END

      SUBROUTINE POLINT(XA,YA,N1,Y1,DY1)
C***********************************************************************
C                 POLINT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Computes Y(X) as Interpolation of XA, YA
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  The Edge Pairs
C                 The Dimension of The Edge Pairs
C
C        OUTPUTS: Interpolation of XA and YA
C
C        CALLED FROM:   QROMB
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      DIMENSION XA(N1),YA(N1),C1(JMAX1),D1(JMAX1)

C     Variable Initializations
      MODNAM = 'POLINT'

      ns = n1
      y1 = ya(ns)
      dIFt = abs(xa(n1))
C     Set Up Interpolation/Divided Differences
      do 11 i = 1,n1
         c1(i) = ya(i)
         d1(i) = ya(i)
  11  CONTINUE

C     Compute Table Entries
      ns = ns-1
      do 13 m1 = 1,n1-1
         do 12 i = 1,n1-m1
            ho = xa(i)
            hp = xa(i+m1)
            w = c1(i+1)-d1(i)
            den = w/(ho-hp)
            d1(i) = hp*den
            c1(i) = ho*den
  12     CONTINUE
         IF (2*ns .lt. n1-m1) THEN
            dy1 = c1(ns+1)
         else
            dy1 = d1(ns)
            ns = ns-1
         END IF
         y1 = y1+dy1
  13  CONTINUE

      RETURN
      END

      SUBROUTINE TRAPZD(XTMIN,XTMAX,VAL,N)
C***********************************************************************
C                 TRAPZD Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Standard Trapezoidal Integration Routine for 2 Dimensional
C                 integrals. It Integrates the function plumef(x)*
C                 (erf(y2(x))-erf(y1(x)), where Y2 And Y1 Are Determined from
C                 Geometric Terms Computed in ACALC And Found In PLUMEF
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Lower Limit For The Integration
C                 Upper Limit For The Integration
C
C        OUTPUTS: The Result Produced By The Integration
C
C        CALLED FROM:   QROMB
C***********************************************************************

C     Variable Declarations
      real del, sum, sval
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'TRAPZD'

      IF (n .EQ. 1) THEN
         call plumef(xtmax,sum1)
         call plumef(xtmin,sum2)
         sum = sum1+sum2
         del = xtmax-xtmin
         sval = 0.0
         neval2 = 1
      else
         del = (xtmax-xtmin)/neval2
         x1 = xtmin+del*0.5
         sum = 0.0
         do 2 i = 1,neval2
            call plumef(x1,sumc)
            sum = sum+sumc
            x1 = x1+del
2        CONTINUE
         neval2 = neval2*2
      END IF

      val = 0.5*(sval+del*sum)
      sval = val

      RETURN
      END
 	
      SUBROUTINE PLUMEF(X1,POUT)
C***********************************************************************
C                 PLUMEF Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Driving Program for Plume Calculations
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY D. Strimaitis and Yicheng Zhuang, SRC (for DEPOSITION)
C
C        MODIFIED BY R. Brode, PES, Inc. to move calculation of dispersion
C                    coefficients to a new ADISLT subroutine - 7/21/94
C
C        DATE:    September 28, 1993
C
C        INPUTS:  Downwind Distance (in km !)
C                 Source Parameter Arrays
C
C        OUTPUTS: Concentration for Particular Source/Receptor Combination
C                 For A Certain Downwind Distance
C
C        CALLED FROM:   TRAPZD
C***********************************************************************
C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C     Declare Local Variables
      LOGICAL LTERR

C     Variable Initializations
      MODNAM = 'PLUMEF'

C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

c     Scale x1 from kilometers to meters, and place in variable XARG
      XARG = X1*1000.0

C     MODIFIED to NOT compute vn, val for cases with val=1.0, uses LSEG,
C     a logical variable set in PSIDE2, AREAIN to establish case
      IF (LSEG) THEN
C        Calculate dispersion coefficients, SY and SZ       ---   CALL ADISLT
         CALL ADISLT(XARG,SY,SZ,XY,XZ)
         VAL = 1.0
      ELSE
         CALL XWIDTH(X1,VT)
         CALL PWIDTH(X1,VT,VN,VAL)
      END IF

C     Determine deposition correction factors   ---   CALL DEPCOR
      IF (LDEP) THEN
C        Loop over particle sizes
         DO 150 I=1,NPD
C           Initialize dry source depletion factors, profile
C           correction factors, and sigma-z settling correction
C           factors to unity.
            DQCOR(I) = 1.
            PCORZR(I) = 1.
            PCORZD(I) = 1.
            SZCOR(I) = 1.
            IF (DDPLETE) THEN
               CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &           XARG,XZ,HE,ZI,US,XS,YS,XR,YR,
     &           RURAL,URBAN,KST,SZ,SBID,
     &           SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &           SRCTYP(ISRC),LTGRID,KURDAT,
     &           DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I))
            ENDIF
150      CONTINUE
      ENDIF

C     Get Concentration or Deposition
      IF (CONC) THEN
         CALL ACHI(X1,RCZ)
      ELSE IF (DEPOS) THEN
         CALL ADEP(X1,RCZ)
      END IF

C     Now compute the function
      POUT = VAL*RCZ*1000.0

      RETURN
      END

      SUBROUTINE PSIDE(U1,U2,DVAL)
C***********************************************************************
C                 PSIDE Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: INTEGRATES SIDE K of POLYGON
C                 int f(u)*CNF(v(u)/sig(u))=f(u)*vn(u) from u1 to u2
C                           CNF = cumulative normal distribution
C                 Computes W(1),W(2)--normalized plume width at   u1    u2
C                 Checks for w(i) outside of -4.0,4.0 with i+, i-
C                 L=-4.0  U=4.0  = bounds for testing
C                 Integrates according to case encountered
C                 situation     CASE    iplus    iminus  integral limits
C                 L<w1,w2<U      1        0        0         u1,u2
C                 w1,w2<L        2        0       1+2      don't compute
C                 w1,w2>U        3       1+2       0         u1,u2
C                 w1<L<w2<U      4        0        1         u-,u2
C                 w2<L<w1<U      5        0        2         u1,u-
C                 L<w1<U<w2      6        2        0       u1,u+  u+,u2
C                 L<w2<U<w1      7        1        0       u1,u+  u+,u2
C                 w1<L<U<w2      8        2        1       u-,u+  u+,u2
C                 w2<L<U<w1      9        1        2       u1,u+  u+,u-
C
C                 u+ = value of u such that w(u)=U=4.0
C                 u- =     "                w(u)=L=-4.0
C                 u+,u- computed with Brent's Algorithm
C
C                 IF uplus >0, part of side is outside plume
C                 but is integrated anyway, unless there is
C                 a corresponding part on another side that will
C                 cause cancellation.  This is determined in
C                 PSIDE2;
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED by Roger Brode, PES, Inc. to correct lower integration
C                    limit for Case 4, and to remove extraneous calls
C                    to XWIDTH and PWIDTH after calls to ZBRENT. - 7/29/94
C
C        INPUTS:  End Points of The Segments
C
C        OUTPUTS: The Effective Sides
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      real u(2), v1(2), vn(2), w(2)

C     Variable Initializations
      MODNAM = 'PSIDE'

C     NSEG = number of segments; set to 0 in AREAIN
C     for each source/rcvr/time step
      dval = 0.0
      do 2 i =  1,2
         ks = ivert + i-1
         u(i) = uvert(ks)
         v1(i) = vvert(ks)
         vn(i) = vNVERT(ks)
         w(i) = wvert(ks)
2     CONTINUE

      iminus = 0
      iplus = 0
      uminus = -1.
      uplus =  -1.0
      do 3 i = 1,2
         IF (vn(i) .lt. -4.0) iminus = i + iminus
         IF (vn(i) .gt.  4.0) iplus  = i + iplus
3     CONTINUE

      ua = u(1)
      ub = u(2)
      IF (iplus.EQ.1 .or. iplus.EQ.2) THEN
         call zbrent(1,u(1),u(2),0.0001,uplus)
crwb         call xwidth(uplus,vtemp)
crwb         call pwidth(uplus,vtemp,vnt,wtemp)
      END IF
      IF (iminus.EQ.1 .or. iminus.EQ.2) THEN
         call zbrent(-1,u(1),u(2),0.0001,uminus)
crwb         call xwidth(uminus,vtemp)
crwb         call pwidth(uminus,vtemp,vnt,wtemp)
      END IF

c     CASE DEPENDs on iplus, iminus
      IF (iplus .EQ. 0) THEN
         IF (iminus .EQ. 0) THEN
c                                             iplus  iminus  case
c                                               0     0       1
            call qromb(u1,u2,dval)
         else IF(iminus .EQ. 3) THEN
c                                               0     3       2
            dval = 0
         else IF(iminus .EQ. 1) THEN
c                                               0     1       4
            call qromb(uminus,u2,dval)
         else
c                                               0     2       5
            call qromb(u1,uminus,dval)
c              changed from u1,uminus
         END IF
      else IF(iplus .EQ. 1) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = uplus
         IF (iminus .EQ. 0) THEN
c                                               1     0       7
            call qromb(uplus,u2,dval)
         else
c                                               1     2       9
            call qromb(uplus,uminus,dval)
         END IF
      else IF(iplus .EQ. 2) THEN
         nsegs = nsegs+1
         uasegs(nsegs) = uplus
         ubsegs(nsegs) = u2
         IF (iminus .EQ. 0) THEN
c                                               2     0       6
            call qromb(u1,uplus,dval)
         else
c                                               2     1       8
            call qromb(uminus,uplus,dval)
         END IF
      else
c                                               3     0       3
         nsegs = nsegs+1
         uasegs(nsegs) = u1
         ubsegs(nsegs) = u2
      END IF

      RETURN
      END

      SUBROUTINE XWIDTH(U,XOUT)
C***********************************************************************
C                 XWIDTH Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Given Any Y Coordinate of A Vertex of an Area
C                 Source, Calculate the X Coordinate
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  The Y Coordinate
C
C        OUTPUTS: The X Coordinate Value
C
C        CALLED FROM:   ZBRENT
C                       PSIDE
C                       PLUMEF
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'XWIDTH'

      U1 = UVERT(IVERT)
      U2 = UVERT(IVERT+1)
      V1 = VVERT(IVERT)
      V2 = VVERT(IVERT+1)
      XOUT = V1+(U-U1)*(V2-V1)/(U2-U1)

      RETURN
      END

      SUBROUTINE PWIDTH(X1,V1,VN,WIDTH)
C***********************************************************************
C                 PWIDTH Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates The Effective Area of The Plume for A
C                 Certain Downwind Distance
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        MODIFIED BY R. Brode, PES, Inc. to move calculation of dispersion
C                    coefficients to a new ADISLT subroutine - 7/21/94
C
C        MODIFIED BY R. Brode, PES, Inc. to correct table of GA values
C                    and extend GA to 79 values - 7/29/94
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Receptor Height Above Ground
C                 Source Parameter Arrays
C
C        OUTPUTS: The Effective Width
C
C        CALLED FROM:   ZBRENT
C                       PSIDE
C                       PLUMEF
C                       AREAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      REAL GA(79)

C     Variable Initializations
c     GA ARE VALUES OF THE CUMULATIVE NORMAL DISTRIBUTION IN
C     INCREMENTS OF 0.1 S.
      DATA GA/0.0,.0001,.0001,.0002,.0002,.0003,.0005,.0007,.0010,.0013,
     1.0019,.0026,.0035,.0047,.0062,.0082,.0107,.0139,.0179,.0227,.0287,
     2.0359,.0446,.0548,.0668,.0808,.0968,.1151,.1357,.1587,.1841,.2119,
     3.2420,.2742,.3085,.3445,.3821,.4207,.4602,.5000,.5398,.5793,.6179,
     4.6555,.6915,.7258,.7580,.7881,.8159,.8413,.8643,.8849,.9032,.9192,
     5.9332,.9452,.9554,.9641,.9713,.9773,.9821,.9861,.9893,.9918,.9938,
     6.9953,.9965,.9974,.9981,.9987,.9990,.9993,.9995,.9997,.9998,.9998,
     7.9999,.9999,1.000/
      MODNAM = 'PWIDTH'

      IF (X1 .EQ. 0.0) THEN
         SZ = 1.0
         SY = 1.0
         VN = V1*1000.0
         WIDTH = VN
C        Exit Routine
         GO TO 999
      END IF

      XARG = X1*1000.0
C     Calculate dispersion coefficients, SY and SZ          ---   CALL ADISLT
      CALL ADISLT(XARG,SY,SZ,XY,XZ)

      VN = 1000.0*V1/SY
      TEMP = 10*VN + 40
      ITEMP = INT(TEMP)
      WIDTH = 0.0

      IF (ITEMP. GT. 78) THEN
         WIDTH = 1.0000
      ELSE
         IF (ITEMP .GE. 1) THEN
            WIDTH = GA(ITEMP)+(TEMP-FLOAT(ITEMP))*
     1              (GA(ITEMP+1)-GA(ITEMP))
         END IF
      END IF

 999  RETURN
      END

      SUBROUTINE ZBRENT(IFD,X1,X2,TOL,OUTVAL)
C***********************************************************************
C                 ZBRENT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: To Divide The Segments According to The Plume Split
C                 And Edge Effects
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Source Parameter Arrays
C
C        OUTPUTS: The Effective Integration Segments
C
C        CALLED FROM:   PSIDE
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'

C     Variable Initializations
      MODNAM = 'ZBRENT'

      a1 = x1
      b1 = x2
      call xwidth(a1,v1)
      call pwidth(a1,v1,vn,w1)
      fa = vn-ifd*4.0
      call xwidth(b1,v1)
      call pwidth(b1,v1,vn,w1)
      fb = vn-ifd*4.0
      IF (fb*fa .LE. 0.0) THEN
         fc = fb
         do 11 iter = 1, itmax
            IF (fb*fc .gt. 0.0) THEN
               c1 = a1
               fc = fa
               d1 = b1-a1
               e1 = d1
            END IF
            IF (abs(fc) .lt. abs(fb)) THEN
               a1 = b1
               b1 = c1
               c1 = a1
               fa = fb
               fb = fc
               fc = fa
            END IF
            tol1 = 2.0*eps*abs(b1)+0.5*tol
            xm = 0.5*(c1-b1)
            IF (abs(xm).le.tol1  .or. fb .EQ. 0.0) THEN
               outval = b1
               RETURN
            END IF
            IF (abs(e1).ge.tol1 .AND. abs(fa).gt.abs(fb)) THEN
               s1 = fb/fa
               IF (a1 .EQ. c1)THEN
                  p1 = 2.0*xm*s1
                  q1 = 1.0-s1
               else
                  q1 = fa/fc
                  r1 = fb/fc
                  p1 = s1*(2.0*xm*q1*(q1-r1)-(b1-a1)*(r1-1.0))
                  q1 = (q1-1.0)*(r1-1.0)*(s1-1.0)
               END IF
               IF(p1. gt. 0.0) q1 = -q1
               p1 = abs(p1)
               IF (2.0*p1.lt.min(3.0*xm*q1-
     &             abs(tol1*q1),abs(e1-q1))) THEN
                  e1 = d1
                  d1 = p1/q1
               else
                  d1 = xm
                  e1 = d1
               END IF
            else
               d1 = xm
               e1 = d1
            END IF
            a1 = b1
            fa = fb
            IF (abs(d1).gt. tol1)THEN
               b1 = b1+d1
            else
               b1 = b1+sign(tol1,xm)
            END IF
            call xwidth(b1,v1)
            call pwidth(b1,v1,vn,w1)
            fb = vn-ifd*4.0
  11     CONTINUE
         outval = b1
      END IF

      RETURN
      END
	
      SUBROUTINE PSIDE2(DVAL)
C***********************************************************************
C                 PSIDE2 Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Integrates Over Segments For Which ABS(VN) > VNTEST
C                 Eliminates Overlap of Segments And Useless Integration
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
c        INPUTS:   Number of The Original Segments
c                  End Points Array of The Segments
C
C        OUTPUT:   The Correction of The Results From PSIDE
C
C        CALLED FROM:   AREAIN
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      real ulist(nvmax2), useg(nvmax,2)
      integer usign(nvmax), ufac, usegf(nvmax)
      LOGICAL Ltest1,Ltest2

C     Variable Initializations
      MODNAM = 'PSIDE2'

      j = 1
      do 1 i = 1, nsegs
         ulist(j) = uasegs(i)
         j = j+1
         ulist(j) = ubsegs(i)
         j = j+1
1     CONTINUE
      npts = 2*nsegs
      call hpsort(npts,ulist,nvmax2)
      do 10 i = 1, nsegs
         usign(i) = 1
         IF (uasegs(i) .gt. ubsegs(i)) THEN
            usign(i) = -1
            temp = uasegs(i)
            uasegs(i) = ubsegs(i)
            ubsegs(i) = temp
         END IF
         IF(uasegs(i) .EQ. ubsegs(i)) usign(i) = 0
10    CONTINUE
      iseg = 0

      do 2 i = 2,npts
         u1 = ulist(i-1)
         u2 = ulist(i)
         ufac = 0
c*****
c           compare segment [u1,u2] against each ua,ub
c*****
         IF (u1.ne.u2) THEN
            do 3 j = 1, nsegs
               IF (u1.ge.uasegs(j) .AND. u2 .le. ubsegs(j)) THEN
                  ufac = ufac + usign(j)
               END IF
3           CONTINUE
c*****
c              make table of segments and factors
c*****
            IF (ufac.ne.0) THEN
               iseg = iseg+1
               useg(iseg,1) = u1
               useg(iseg,2) = u2
               usegf(iseg) = ufac
            END IF
         END IF
2     CONTINUE
c*****
c            CONSOLIDATE SEGMENTS IF iseg>1
c*****
      nsegs = iseg
      IF (nsegs .gt. 1) THEN
         do 4 iseg = 2, nsegs
            Ltest1 = useg(iseg,1) .EQ. useg(iseg-1,2)
            Ltest2 = usegf(iseg)*usegf(iseg-1) .gt. 0
            IF (Ltest1 .AND. Ltest2) THEN
               usegf(iseg-1) = 0
               useg(iseg,1) = useg(iseg-1,1)
            END IF
4        CONTINUE
      END IF
      dval = 0.0
      IF (nsegs .gt. 0) THEN
         do 5 iseg = 1, nsegs
            IF (usegf(iseg) .ne. 0) THEN
               uav = useg(iseg,1)
               ubv = useg(iseg,2)
               ufac = usegf(iseg)
               LSEG = .TRUE.
               call qromb(uav,ubv,tmpval)
               dval = dval + ufac*tmpval
            END IF
5        CONTINUE
      END IF

      RETURN
      END

      SUBROUTINE HPSORT(NVAR,UVAR,IDIM)
C***********************************************************************
C                 HPSORT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: A General Program For Heap Sort of An Array
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C                    Adapted From Codes By Richard Strelitz, CSC
C
C        DATE:    July 7, 1993
C
C        INPUTS:  The Array To Be Sorted
C
C        OUTPUTS: The Array Sorted
C
C        CALLED FROM:   PSIDE2
C***********************************************************************

C     Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      DIMENSION UVAR(IDIM)

C     Variable Initializations
      MODNAM = 'HPSORT'

      ILMID = NVAR/2 + 1
      IR = NVAR
10    CONTINUE
      IF (ilmid.gt.1) THEN
         ilmid = ilmid-1
         ru = uvar(ilmid)
      else
         ru = uvar(ir)
         uvar(ir) = uvar(1)
         ir = ir-1
         IF (ir .EQ. 1)THEN
            uvar(1) = ru
            RETURN
         END IF
      END IF
      i = ilmid
      j = ilmid+ilmid
      DO WHILE (j. le. ir)
         IF (j. lt. ir) THEN
            IF (uvar(j).lt.uvar(j+1) ) j = j+1
         END IF
         IF (ru.lt.uvar(j)) THEN
            uvar(i) = uvar(j)
            i = j
            j = 2*j
         else
            j = ir+1
         END IF
      END DO
      uvar(i) = ru
      go to 10

      RETURN
      END

C***  End new code for area source numerical integration algorithm - 7/7/93



C***  Subroutines for OPENPIT Source algorithms - 7/19/94


      SUBROUTINE ESCAPE(ICAT)
C***********************************************************************
C*                ESCAPE Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate Escape Fractions for a Particle
C*                Size Category
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Index for Particle Size Category Being Processed
C*                Gravitational Settling Velocity for Current 
C*                     Particle Size Category & Current Source
C*                10-meter Wind Speed for the Current Hour
C*                Constant ALPHA (= 0.029)
C*                
C*
C*       OUTPUTS: The Escape Fraction for the current Size Category
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'ESCAPE'

      EFRAC(ICAT) = 1.0/(1.0 + VGRAV(ICAT) / (ALPHA * UREF10) )
      
      RETURN
      END



      SUBROUTINE ADJEMI(ICAT,QPTOT)
C***********************************************************************
C*                ADJEMI Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Adjust Emission Rate for Current Particle
C*                Size Category Being Processed
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Index for Particle Size Category Being Processed
C*                Escape Fraction for the Current Size Category
C*                Mass Fraction of the Current Size Category
C*                Total Emission Rate Per Unit Area
C*                
C*
C*       OUTPUTS: Adjusted Emission Rate for the Current Size Category
C*                Cumulative Adjusted Emission Rate Over All Categories
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'ADJEMI'

      QPART(ICAT) = EFRAC(ICAT) * PHI(ICAT) * QS
      QPTOT = QPTOT + QPART(ICAT)
          
      RETURN
      END


      SUBROUTINE AMFRAC(QPTOT)
C***********************************************************************
C*                AMFRAC Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Adjust the Mass Fractions for each Particle
C*                Size Category
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Array of Adjusted Emission Rates
C*                Cumulative Adjusted Emission Rate Over All Categories
C*
C*       OUTPUTS: Array of Adjusted Mass Fractions
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'AMFRAC'

      DO 20 ICAT = 1,NPD
         PHI(ICAT) = QPART(ICAT)/QPTOT
20    CONTINUE         
          
      RETURN
      END


      SUBROUTINE LWIND
C***********************************************************************
C*                LWIND Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Along-Wind Length of the OPENPIT
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Wind Flow Vector for the Current Hour
C*                Angle of the Long OPENPIT dimension from the North
C*                Length of the OPENPIT
C*                Width of the OPENPIT
C*
C*       OUTPUTS: Along-Wind Length of the OPENPIT
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'LWIND'
      

C*    Determine the Wind Direction Angle Relative to the Long
C*    Axis of the OpenPit
      CALL CTHETA(AFV,PALPHA,THETA)
      
C*    Determine the Along-Wind Length of the OPENPIT
      PITL = PITLEN * (1 - THETA/90.) + PITWID * (THETA / 90.)
      
      RETURN
      END


      SUBROUTINE PDEPTH
C***********************************************************************
C*                PDEPTH Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Relative Depth of the OPENPIT Source
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Effective Depth of the OPENPIT
C*                Release Height Above 
C*                Along Wind Length of the OPENPIT
C*                
C*       OUTPUTS: Relative Depth of the OPENPIT
C*                
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'PDEPTH'

      PDREL = (PDEFF-EMIHGT) / PITL

      RETURN
      END
      
            

      SUBROUTINE PTFRAC 
C***********************************************************************
C*                PTFRAC Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Calculate the Fractional Size of the Effective
C*                Area of the OPENPIT Source
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Relative Pit Depth
C*
C*       OUTPUTS: Fractional Size of the Effective Area of the OPENPIT
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'PTFRAC'
      
      IF (PDREL .GE. 0.2) THEN
         PITFRA = 0.08
      ELSE
         PITFRA = SQRT (1.0 - 1.7*(PDREL**(0.333333)) )
      ENDIF
      
      RETURN
      END
      

      SUBROUTINE PITEFF
C***********************************************************************
C*                PITEFF Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Coordinates of the OPENPIT Source
C*                in Wind Direction Coordinate System
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  
C*
C*       OUTPUTS: Coordinates of the OPENPIT Source in Wind 
C*                Direction Coordinate System
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
      
      REAL XTEMP(NVMAX),YTEMP(NVMAX), XSPA(NVMAX), YSPA(NVMAX)
      REAL EPSLON
      
C*    Variable Initializations
      MODNAM = 'PITEFF'
      EPSLON = 0.00001
      
C*    Get Vertices of Actual Pit in WD-Coordinate System    ---   CALL AVERTS
      
      CALL AVERTS(XVERT,YVERT,XSPA,YSPA,NVERT+1)
              
C*    Find the Upwind Vertex of the Pit (one with minimum X)
      SPAMIN = 1.0E+20
      IUPWND = 0
      DO 20 IVERT = 1,NVERT
         IF (XSPA(IVERT) .LT. SPAMIN) THEN
            IUPWND = IVERT
            SPAMIN = XSPA(IVERT)-EPSLON
         ENDIF
20    CONTINUE         

C*    If DEBUG Requested, Write Out Pit Info            
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'ACTUAL PIT COORDINATES:'
         WRITE (IOUNIT,*) '----------------'
         WRITE (IOUNIT,*) 'SYSTEM   X1       Y1       X2       Y2',
     &                    '       X3       Y3       X4       Y4'
         WRITE (IOUNIT,*) '-------- -------- -------- -------- ',
     &                    '-------- -------- -------- -------- ',
     &                    '--------'
         WRITE (IOUNIT,8000) (XVERT(II),YVERT(II),II=1,NVERT)
         WRITE (IOUNIT,8100) (XSPA (II),YSPA (II),II=1,NVERT)         
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) ' UPWIND VERTEX OF THE PIT= ', IUPWND
         WRITE (IOUNIT,*) ' WIND DIRECTION W.R.T. PIT LONG AXIS= ',THETA
         WRITE (IOUNIT,*) ' ALONGWIND LENGTH OF THE PIT= ',PITL
         WRITE (IOUNIT,*) ' RELATIVE DEPTH OF THE PIT= ',PDREL
         WRITE (IOUNIT,*)
8000     FORMAT (1X,'User     ',8(f8.3,1x))
8100     FORMAT (1X,'Wind-Dir ',8(f8.3,1x))
      ENDIF
      
C*    Determine the Angle of the Effective Pit Relative
C*    to North
      EFFANG = ANGLE + (90.*(IUPWND - 1))
      
C*    Determine Length and Width Dimensions of the
C*    Effective Pit Area
      EFFWID = PITFRA**(1.0 - (COS(THETA*DTORAD))**2)*PITWID
      EFFLEN = PITFRA**((COS(THETA*DTORAD))**2)*PITLEN

C*    Calculate the Coordinates of the Vertices of the 
C*    Effective Pit Area
C*    Set Coordinates of Vertices for Rectangular Area (in Kilometers).
C*    Vertices Start with the "Southwest" Corner and Are Defined
C*    Clockwise.  The First Vertex is Repeated as the Last Vertex.


C*    First determine proper 'x-dim' and 'y-dim' for effective area,
C*    taking into account angle of orientation and relation to actual pit.

      IF (XINIT .LE. YINIT .AND. (IUPWND.EQ.1 .OR. IUPWND.EQ.3)) THEN
         XEFF = EFFWID
         YEFF = EFFLEN
      ELSE IF (XINIT.LE.YINIT .AND. (IUPWND.EQ.2 .OR. IUPWND.EQ.4)) THEN
         XEFF = EFFLEN
         YEFF = EFFWID
      ELSE IF (XINIT.GT.YINIT .AND. (IUPWND.EQ.1 .OR. IUPWND.EQ.3)) THEN
         XEFF = EFFLEN
         YEFF = EFFWID
      ELSE IF (XINIT.GT.YINIT .AND. (IUPWND.EQ.2 .OR. IUPWND.EQ.4)) THEN
         XEFF = EFFWID
         YEFF = EFFLEN
      END IF

      XTEMP(1) = XVERT(IUPWND)
      YTEMP(1) = YVERT(IUPWND)

      XTEMP(2) = XTEMP(1) +
     &                (YEFF*SIN(EFFANG*DTORAD)) / 1000.
      YTEMP(2) = YTEMP(1) +
     &                (YEFF*COS(EFFANG*DTORAD)) / 1000.

      XTEMP(3) = XTEMP(2) +
     &                (XEFF*COS(EFFANG*DTORAD)) / 1000.
      YTEMP(3) = YTEMP(2) -
     &                (XEFF*SIN(EFFANG*DTORAD)) / 1000.

      XTEMP(4) = XTEMP(3) -
     &                (YEFF*SIN(EFFANG*DTORAD)) / 1000.
      YTEMP(4) = YTEMP(3) -
     &                (YEFF*COS(EFFANG*DTORAD)) / 1000.

      XTEMP(5) = XVERT(IUPWND)
      YTEMP(5) = YVERT(IUPWND)


C*    Calculate Coordinates of the Effective Pit Area in
C*    Wind Direction Coordinate System                      ---   CALL AVERTS
      CALL AVERTS(XTEMP,YTEMP,XSPA,YSPA,NVERT+1)

C*    If DEBUG Requested, Write Out Pit Info            
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'EFFECTIVE PIT COORDINATES:'
         WRITE (IOUNIT,*) '----------------'
         WRITE (IOUNIT,*) 'SYSTEM   X1       Y1       X2       Y2',
     &                    '       X3       Y3       X4       Y4'
         WRITE (IOUNIT,*) '-------- -------- -------- -------- ',
     &                    '-------- -------- -------- -------- ',
     &                    '--------'
       WRITE (IOUNIT,8000) (XTEMP(II),YTEMP(II),II=1,NVERT)
         WRITE (IOUNIT,8100) (XSPA (II),YSPA (II),II=1,NVERT)         
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) 'EFFECTIVE PIT LENGTH = ', EFFLEN
         WRITE (IOUNIT,*) 'EFFECTIVE PIT WIDTH  = ', EFFWID
         WRITE (IOUNIT,*) 'EFFECTIVE PIT ORIENTATION RELATIVE',
     &                    ' TO NORTH= ', EFFANG
         WRITE (IOUNIT,*) 'FRACTIONAL SIZE OF THE EFFECTIVE PIT AREA= ',
     &                     PITFRA
      ENDIF
      
C*    Store Coordinates of the Effective Area in COMMON Variables
      DO 40 IVERT = 1,NVERT+1
         SPA(IVERT,1) = XSPA(IVERT)
         SPA(IVERT,2) = YSPA(IVERT)
40    CONTINUE
         
         
      RETURN
      END


      SUBROUTINE PITEMI(QPTOT)
C***********************************************************************
C*                PITEMI Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Emission Rate for the Effective 
C*                Pit Area
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 20, 1994
C*
C*       INPUTS:  Fractional Area of the Pit
C*                Total Adjusted Emission Rate
C*
C*       OUTPUTS: Emission Rate for the Effective Area of the Current
C*                OPENPIT Source
C*                
C*
C*       CALLED FROM:   OCALC
C***********************************************************************

C*    Variable Declarations
      INCLUDE 'MAIN1LT.INC'
      INCLUDE 'MAIN2LT.INC'
      INCLUDE 'MAIN3LT.INC'
C*    Variable Initializations
      MODNAM = 'PITEMI'
      
      QEFF = QPTOT / PITFRA
      
      RETURN
      END


      SUBROUTINE CTHETA(AFVIN,ALFIN,THOUT)
C***********************************************************************
C*                CTHETA Module of ISC2 Short Term Model - ISCST2
C*
C*       PURPOSE: To Determine the Wind Direction Angle Relative to 
C*                the Pit Long Axis
C*
C*       PROGRAMMER: Jayant Hardikar, Roger Brode
C*
C*       DATE:    July 26, 1994
C*
C*       INPUTS:  Flow Vector 
C*                Angle of Pit Long Axis from North
C*
C*       OUTPUTS: THETA - Wind Direction Angle Relative to 
C*                the Pit Long Axis
C*                
C*
C*       CALLED FROM:   LWIND
C***********************************************************************

      if (abs(AFVIN-ALFIN) .le. 90.) then
         THOUT = abs(AFVIN-ALFIN)
      else if (AFVIN .gt. ALFIN) then
         theta = AFVIN - ALFIN
         if (theta .gt. 90.) then
            theta = AFVIN-180. - ALFIN
         endif
         if (theta .gt. 90.) then
            theta = AFVIN-360. - ALFIN
         endif
         if (theta .gt. 90.) then
            theta = AFVIN-540. - ALFIN
         endif
         THOUT = abs(theta)
      else if (AFVIN .lt. ALFIN) then
         theta = AFVIN - ALFIN
         if (theta .lt. -90.) then
            theta = AFVIN + 180. - ALFIN
         endif
         if (theta .lt. -90.) then
            theta = AFVIN + 360. - ALFIN
         endif
         if (theta .lt. -90.) then
            theta = AFVIN + 540. - ALFIN
         endif
         THOUT = abs(theta)
      endif
      RETURN
      end
      
