      PROGRAM METLIST
C***********************************************************************
C     PROGRAM METLIST (Version 92062)
C
C     PURPOSE: This is a utility program that is designed to print
C              out a summary of the meteorology data for a user-
C              specified range of days.
C
C     PROGRAMMED BY:  Roger W. Brode, Kevin Stroupe, and JieFu Wang
C                     Pacific Environmental Services, Inc.
C                     3708 Mayfair Street, Suite 202
C                     Durham, North Carolina  27707
C
C     DATE:    March 2, 1992
C
C     INPUTS:  Meteorology Input Data
C
C     OUTPUTS: Meteorological Data
C
C     The ISCST2 model has been developed for the U.S. Environmental
C     Protection Agency under Contract No. 68D00124.  The Project
C     Manager is Roger W. Brode, PES.  The EPA Work Assignment Manager
C     is Russell F. Lee, who also contributed to the development of this
C     program.  For instructions on running this program, refer to Appendix C
C     of the User's Guide for the Industrial Source Complex (ISC2)
C     Dispersion Models, Volume I - User Instructions.
C
C***********************************************************************


C >>>>>>>>>>>>>>>>>>>>>>>>>>>>> DECLARATIONS <<<<<<<<<<<<<<<<<<<<<<<<<<<

      PARAMETER (NHR= 24)

      COMMON /HEADR/ IPAGE,IOUNIT
      REAL AUREF(NHR), ATA(NHR), AAFV(NHR), AAFVR(NHR), AZI(2,NHR)
      REAL APROF(NHR), ADTDZ(NHR)
      INTEGER IKST(NHR)
      INTEGER IMNTH(NHR), INDY(NHR), INHR(NHR)
      CHARACTER*40 METFIL
      CHARACTER*40 OUTFIL
      CHARACTER*65 METFOR
      CHARACTER*65 METFRM
      LOGICAL DONE

      DATA IOUNIT/ 8/, MFUNIT/19/, IPAGE/ 0/


C >>>>>>>>>>>>>>>>>>>>>>>>>>>> MAIN PROGRAM <<<<<<<<<<<<<<<<<<<<<<<<<<
C +++++++++++++++++++++++ Get Input From User ++++++++++++++++++++
      WRITE(*,*) 'Enter Meteorology File Name: '
      READ(*,2222) METFIL
 2222 FORMAT(A40)
      WRITE(*,*) 'Options for File Format Are: '
      WRITE(*,*) '   ASCII'
      WRITE(*,*) '   UNFORM'
      WRITE(*,*) '   FREE'
      WRITE(*,*) '   CARD'
      WRITE(*,*) '   Fortran format specifier'
      WRITE(*,*) ' '
      WRITE(*,*) 'Enter File Format:'
      READ(*,2223) METFOR
      CALL LWRUPR(METFOR)
 2223 FORMAT(A65)
      WRITE(*,*) 'Enter Output File Name: '
      READ(*,2222) OUTFIL

      DONE = .FALSE.
      DO WHILE (.NOT. DONE)
         WRITE(*,*) 'Enter Day Range (e.g. 1,365): '
         READ(*,*) JSDAY, JEDAY
         IF (JSDAY .LT. 1) THEN
            WRITE(*,*)'Start day must be greater than or equal to 1 !!'
         ELSE IF (JEDAY .GT. 366) THEN
            WRITE(*,*) 'End day must be less than or equal to 366 !!'
         ELSE IF (JEDAY .LT. JSDAY) THEN
            WRITE(*,*) 'Start day must be less than end day !!'
         ELSE
            DONE = .TRUE.
         END IF
      END DO

      IF (METFOR .EQ. 'ASCII') THEN
         METFRM = '(4I2,2F9.4,F6.1,I2,2F7.1)'
      ELSE IF (METFOR .NE. 'FREE' .AND. METFOR .NE. 'CARD' .AND.
     &         METFOR .NE. 'UNFORM') THEN
         METFRM = METFOR
      END IF

      IF (METFOR .EQ. 'UNFORM') THEN
         OPEN(MFUNIT,FILE=METFIL,STATUS='UNKNOWN',FORM='UNFORMATTED')
      ELSE
         OPEN(MFUNIT,FILE=METFIL,STATUS='UNKNOWN')
      END IF
      OPEN(IOUNIT,FILE=OUTFIL,STATUS='UNKNOWN')

C ++++++++++++++++++++++++ Read header data +++++++++++++++++++++++
      IF (METFOR .EQ. 'UNFORM') THEN
         READ(MFUNIT,ERR=99,IOSTAT=IOERRN) IDSURF, ISYEAR, IDUAIR,
     &                                     IUYEAR
      ELSE
         READ(MFUNIT,*,ERR=99,IOSTAT=IOERRN) IDSURF, ISYEAR, IDUAIR,
     &                                       IUYEAR
      END IF

C     Begin Loop Through Days
      DO 2000 IDY = 1, 366

C ++++++++++ If day is greater than end day, then go to end +++++++++++
         IF (IDY .GT. JEDAY) THEN
             GO TO 9999
         END IF

C ++++++++++++++++ Read data in UNFORM format +++++++++++++++++++++++++
         IF (METFOR .EQ. 'UNFORM') THEN
            READ(MFUNIT,END=999,ERR=99,IOSTAT=IOERRN) IYEAR,IMONTH,
     &                               DAY,IKST,AUREF,ATA,AAFV,AAFVR,AZI
            IDAY = INT(DAY)

C +++++++++++++++++ Read Data in ASCII, FREE, or Fortran specified format +++
         ELSE IF (METFOR .NE. 'CARD') THEN
C           READ In 24 Hours of Data
            DO 101 I = 1, 24
               IF (METFOR .EQ. 'FREE') THEN
C                 +++ Read Hourly Records from ASCII File Using FREE Format
C                 Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB
                  READ(MFUNIT,*,END=999,ERR=99) IYEAR,
     &                 IMONTH,IDAY,IHOUR,AFV,UREF,TA,KST,ZIRUR,ZIURB
               ELSE
C                 +++ Read Hourly Records from Formatted ASCII File Using METFRM
                  READ(MFUNIT,METFRM,END=999,ERR=99) IYEAR,
     &                 IMONTH,IDAY,IHOUR,AFV,UREF,TA,KST,ZIRUR,ZIURB
               END IF
               IMNTH(I) = IMONTH
               INDY(I)  = IDAY
               INHR(I)  = IHOUR
               AAFVR(I) = AFV
               AUREF(I) = UREF
               ATA(I)   = TA
               IKST(I)  = KST
               AZI(1,I) = ZIRUR
               AZI(2,I) = ZIURB
 101        CONTINUE

C +++++++++++++++++++ Read data in CARD format +++++++++++++++++++++++++
         ELSE IF (METFOR .EQ. 'CARD') THEN
C           READ In First 24 Hours of Data
            DO 200 I = 1, 24
C              Read Hourly Records from ASCII File Using CARD Format
C              Yr, Mn, Dy, Hr, AFV, UREF, TA, KST, ZIRUR, ZIURB, P, DTDZ
               READ(MFUNIT,9009,END=999,ERR=99,IOSTAT=IOERRN) IYEAR,
     &              IMONTH,IDAY,IHOUR,AFV,UREF,TA,KST,ZIRUR,ZIURB,P,DTDZ
 9009          FORMAT(4I2,2F9.4,F6.1,I2,2F7.1,2F8.4)
               IMNTH(I) = IMONTH
               INDY(I)  = IDAY
               INHR(I)  = IHOUR
               AAFVR(I) = AFV
               AUREF(I) = UREF
               ATA(I)   = TA
               IKST(I)  = KST
               AZI(1,I) = ZIRUR
               AZI(2,I) = ZIURB
               APROF(I) = P
               ADTDZ(I) = DTDZ
 200        CONTINUE
         END IF

C ++++++++++++++++++++ LOOP while day is less than start day +++++++
         IF (IDY .LT. JSDAY) THEN
            GO TO 2000
         END IF

C   +++++++++++++++++++++++ Write Output File +++++++++++++++++++++
C        WRITE Out Header Information
         CALL HEADER
         WRITE(IOUNIT,9011) IDY
         WRITE(IOUNIT,9016) METFIL, METFOR
         WRITE(IOUNIT,9020) IDSURF, IDUAIR, ISYEAR, IUYEAR

         IF (METFOR .NE. 'CARD') THEN
            WRITE(IOUNIT,9025)
         ELSE
            WRITE(IOUNIT,9026)
         END IF

C        WRITE Out 24 Hourly Records
         DO 300 I = 1, 24

            IF (METFOR .EQ. 'UNFORM') THEN
               WRITE(IOUNIT,9032) IYEAR, IMONTH, IDAY, I, AAFVR(I),
     &               AUREF(I), ATA(I), IKST(I), AZI(1,I), AZI(2,I)
            ELSE IF (METFOR .NE. 'CARD') THEN
               WRITE(IOUNIT,9032) IYEAR, IMNTH(I), INDY(I), INHR(I),
     &               AAFVR(I),AUREF(I),ATA(I), IKST(I), AZI(1,I),
     &               AZI(2,I)
            ELSE
               WRITE(IOUNIT,9033) IYEAR, IMNTH(I), INDY(I), INHR(I),
     &               AAFVR(I),AUREF(I),ATA(I), IKST(I), AZI(1,I),
     &               AZI(2,I),APROF(I), ADTDZ(I)
            END IF

 300     CONTINUE

C        Write Out Explanatory Message for Stability Class
         WRITE(IOUNIT,9050)
 2000 CONTINUE

      GO TO 9999

C +++++++++++++++++++++++++++ Formats for output +++++++++++++++++++++++
 9011 FORMAT(/25X,'*** METEOROLOGICAL DATA FOR DAY ',I3,' *** '/)
 9016 FORMAT(12X,'FILE: ',A40,' FORMAT: ',A60)
 9020 FORMAT(12X,'SURFACE STATION NO.: ',I6,20X,
     &       'UPPER AIR STATION NO.: ',I6,/27X,'YEAR: ',I6,37X,
     &       'YEAR: ',I6)
 9025 FORMAT(/38X,'FLOW',4X,'SPEED',3X,'TEMP',5X,'STAB',4X,
     &       'MIXING HEIGHT (M)',
     &       /11X,'YEAR',2X,'MONTH',2X,'DAY',2X,'HOUR',4X,'VECTOR',
     &       3X,'(M/S)',4X,'(K)',5X,'CLASS',4X,'RURAL',4X,'URBAN',
     &       /60('- ')/)
 9026 FORMAT(/38X,'FLOW',4X,'SPEED',3X,'TEMP',5X,'STAB',4X,
     &       'MIXING HEIGHT (M)',4X,'WIND',4X,'VERT TEMP',
     &       /11X,'YEAR',2X,'MONTH',2X,'DAY',2X,'HOUR',4X,'VECTOR',
     &       3X,'(M/S)',4X,'(K)',5X,'CLASS',4X,'RURAL',4X,'URBAN',
     &       6X,'PROF',4X,'GRAD (K/M)',
     &       /60('- ')/)
 9032 FORMAT(12X,4(I2,4X),1X,F6.1,2X,F6.2,2X,F6.1,6X,I1,5X,2(F7.1,2X))
 9033 FORMAT(12X,4(I2,4X),1X,F6.1,2X,F6.2,2X,F6.1,6X,I1,5X,2(F7.1,2X),
     &       2(F8.4,2X))
 9050 FORMAT(///' *** NOTES:  STABILITY CLASS 1=A, 2=B, 3=C, 4=D, 5=E',
     &       ' AND 6=F.',
     &         /'             FLOW VECTOR IS DIRECTION TOWARD WHICH ',
     &       'WIND IS BLOWING.')

C     WRITE Error Message:  Error Reading Met Data Input File
  99  CONTINUE
      WRITE(*,*) 'ERROR READING THE METEOROLOGY FILE'

      GO TO 9999

C     WRITE Error Message:  End of File Reached
 999  CONTINUE
      WRITE(*,*) 'END OF FILE REACHED BEFORE COMPLETING RANGE'

 9999 CLOSE (IOUNIT)

      STOP
      END


C >>>>>>>>>>>>>>>>>>>>>>>>>> SUBROUTINES <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

      SUBROUTINE HEADER
C***********************************************************************
C                 Module HEADER of METLIST
C
C        PURPOSE: Control Page Feed and Header Information for
C                 Printed File Output
C
C        PROGRAMMER: Roger Brode, Jeff Wang, Kevin Stroupe
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Page Number from COMMON
C
C        OUTPUTS: Page Feed and Header
C
C        CALLED FROM:  main program
C***********************************************************************
C
C     Variable Declarations
      COMMON/HEADR/IPAGE,IOUNIT
      INTEGER*2 IPTHR, IPTMIN, IPTSEC, IPTHUN, IPTYR, IPTMON, IPTDAY
      CHARACTER RUNDAT*8, RUNTIM*8
      CHARACTER VERSN*5
      COMMON /DATTIM/ RUNDAT, RUNTIM

      DATA VERSN/'92062'/

C     Increment Page Number Counter
      IPAGE = IPAGE + 1

C     Retrieve Date and Time Variables for First Call
      IF (IPAGE .EQ. 1) THEN
         RUNDAT = ' '
         RUNTIM = ' '

C        For Porting Model to Other Systems, Change Date & Time Function Calls
C        In the Next Two Lines, Or Comment Out the Next Five Executable Lines

         CALL GETDAT(IPTYR, IPTMON, IPTDAY)
         CALL GETTIM(IPTHR, IPTMIN, IPTSEC, IPTHUN)

C        Convert Year to Two Digits
         IPTYR = IPTYR - 100*INT(IPTYR/100)
C        Write Date and Time to Character Variables, RUNDAT & RUNTIM
         WRITE(RUNDAT,'(2(I2.2,1H-),I2.2)') IPTMON, IPTDAY, IPTYR
         WRITE(RUNTIM,'(2(I2.2,1H:),I2.2)') IPTHR, IPTMIN, IPTSEC
      END IF

C     Write Header to Printed Output File
      WRITE(IOUNIT,9028) VERSN, RUNDAT
      WRITE(IOUNIT,9029) RUNTIM
      WRITE(IOUNIT,9030) IPAGE

 9028 FORMAT('',29X,'*** METLIST - VERSION ',A5,' ***',58X,A8)
 9029 FORMAT(118X,A8)
 9030 FORMAT(118X,'PAGE ',I3)

      RETURN
      END

      SUBROUTINE LWRUPR(METFOR)
C***********************************************************************
C                 LWRUPR Module of METLIST
C
C        PURPOSE: Transfer All Characters From Lower Case To
C                 Upper Case
C
C        PROGRAMMER: Kevin Stroupe, Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Input Runstream Card Image, Number of Characters
C
C        OUTPUTS: Input Runstream Card Image in Uppercase
C
C        CALLED FROM:   main program
C***********************************************************************
C
C     Variable Declarations
      CHARACTER UPCASE*26
      CHARACTER LWCASE*26
      CHARACTER*65 METFOR

C     Variable Initializations
      DATA UPCASE/'ABCDEFGHIJKLMNOPQRSTUVWXYZ'/
      DATA LWCASE/'abcdefghijklmnopqrstuvwxyz'/

      DO 20 I = 1, 65
        IF (METFOR(I:I) .NE. ' ') THEN
           INDCHK = INDEX(LWCASE,METFOR(I:I))
           IF (INDCHK .NE. 0) THEN
              METFOR(I:I) = UPCASE(INDCHK:INDCHK)
           END IF
         END IF
 20   CONTINUE

      RETURN
      END
