##  Process_AERMOD_test_cases_output.R
##  July 05, 2023
##
##  Read AERMOD POST (.PST) files for all test cases and create rdata files for each case.
##  Resulting .rdata files can be used as inputs for the "compare_AERMOD_test_Cases.R script
##  for statistical analysis and model comparison. 
##
##  Note: Processing will result in deletion of postfiles.
##
##  MODIFIED: 
##  Now outputs a list of “failed testcases” in a file called “FailedTestCases.csv”, which will 
##  list ONLY cases that fail. This file is ALWAYS generated. If no cases fail, the file will have text: 
##  “There were no errors when processing the test cases."
##
##    WSP 8/09/23

## clear Environments and console:
rm(list = ls(all.names = TRUE))

## clear global environments:
rm(list=ls())

##  Load R packages
library(ggplot2)
require(gridExtra)
require(reshape2)

################################
## User Defined Section 
################################
## Define test case identifiers
testcase_versions <- c("aermet_24142_aermod_24142",
                       "aermet_23132_aermod_23132",
                       "aermet_23132_aermod_24142")

## Set working directory, will identify directory where Rscript is located, no user input needed.
## Uncomment the second line if you would like to define your own working directory
working_dir <- normalizePath(getwd(),winslash="/",mustWork=NA) 
#working_dir <- 'D:/23132_Release_Files/aermod/TestCasesDeliverable/TestCases/'

################################
## End of User Defined Section 
################################

for (j in 1:length(testcase_versions)){
  
  testcase_version <- testcase_versions[j]
  print(paste("Now examining", testcase_version))
  
  ## Define postfiles directory for test case.
  post_dir <- paste(working_dir, testcase_version, "postfiles/",sep="/")
  
  ## Define rdata directory for test case.  Create directory if does not exist.
  rdata_dir <- paste(working_dir,testcase_version, "rdata/",sep="/")
  if(!dir.exists(rdata_dir)) {
    dir.create(rdata_dir,showWarnings = FALSE)
  }
  
  ## Source related R functions 
  source(paste(working_dir, "read_POS_TESTDEP.fun.R", sep="/"))
  source(paste(working_dir, "read_POS.fun.R", sep="/"))
  
  ## Define AERMOD test cases
  cases <- c("AERMOD-BALDWIN45_01H","AERMOD-BALDWINHORIZ_01H","AERMOD-BALDWINVERT_01H","AERTEST_01H",
             "ALLSRCS_AREA_01H","ALLSRCS_AREAP_01H","ALLSRCS_BLINE_01H","ALLSRCS_CIRC_01H",
             "ALLSRCS_LINE_01H","ALLSRCS_OPENPIT_01H","ALLSRCS_RLINEB_01H","ALLSRCS_RLINEB2_01H",
             "ALLSRCS_RLINEBA_01H","ALLSRCS_RLINEDE_01H","ALLSRCS_STACK_01H","ALLSRCS_STACKDW_01H",
             "ALLSRCS_VOL_01H","BLP_URBAN_2S26_01H","BLP_URBAN_2S29_01H",
             "CAPPED_NOSTD_STACK2C_01H","CAPPED_NOSTD_STACK2CE_01H","CAPPED_NOSTD_STACK2H_01H",
             "CAPPED_NOSTD_STACK2HE_01H","CAPPED_STACK1_01H","CAPPED_STACK1C_01H",
             "CAPPED_STACK1C0_01H","CAPPED_STACK1H_01H","CAPPED_STACK2_01H","CAPPED_STACK2C_01H",
             "CAPPED_STACK2C0_01H","CAPPED_STACK2CE_01H","CAPPED_STACK2H_01H","FLATELEV_ELEV_STK_01H",
             "FLATELEV_FLAT_STK_01H","HRDOW_STACK1_01H","HRDOW_STACK2_01H","HRDOW_STACK3_01H",
             "HRDOW_STACK4_01H","HRDOW_STACK5_01H","HRDOW_STACK6_01H","IN_URBAN_01H",
             "LOVETT_01H","LOVETT_24H","LVT24_ELEV","LVT24_FLAT","MCR_01H","MCR_03H","MCR_24H",
             "MULTURB_STACK1_01H","MULTURB_STACK2_01H","MULTURB_STACK3_01H","MULTURB_STACK4_01H",
             "NO2_ARM2_PPB_01H","NO2_ARM2_UGM3_01H","NO2_GRSM_PPB_01H","NO2_GRSM_UGM3_01H",
             "NO2_OLM_PPB_01H","NO2_OLM_UGM3_01H","NO2_PVMRM_PPB_01H","NO2_PVMRM_UGM3_01H",
             "NO2_TTRM_PPB_01H","NO2_TTRM_UGM3_01H","NO2_ARM2_PPB_BKG","NO2_ARM2_UGM3_BKG",
             "NO2_GRSM_PPB_BKG","NO2_GRSM_UGM3_BKG","NO2_OLM_PPB_BKG","NO2_OLM_UGM3_BKG",
             "NO2_PVMRM_PPB_BKG","NO2_PVMRM_UGM3_BKG","NO2_TTRM_PPB_BKG","NO2_TTRM_UGM3_BKG",
             "NO2_ARM2_PPB_SRC","NO2_ARM2_UGM3_SRC","NO2_GRSM_PPB_SRC","NO2_GRSM_UGM3_SRC",
             "NO2_OLM_PPB_SRC","NO2_OLM_UGM3_SRC","NO2_PVMRM_PPB_SRC","NO2_PVMRM_UGM3_SRC",
             "NO2_TTRM_PPB_SRC","NO2_TTRM_UGM3_SRC","OLM_01H","OLMGRP_01H","OPENPITS_PITGAS_01H",
             "OPENPITS_PITPRT1_01H","OPENPITS_PITPRT2_01H","PSDCRED_NAAQS_01H","PSDCRED_PSDINC_01H",
             "PVMRM_01H","SURFCOAL_01H", "SURFCOAL_ACTVTY_01H","SURFCOAL_NPIT_01H",
             "SURFCOAL_OTHERS_01H","SURFCOAL_ROADS_01H","SURFCOAL_SPIT_01H",
             "TEST1_BASE_CART_3COND_SNC","TEST3_BASE_CART_3COND_SNC_BAR",
             "TEST4_BASE_CART_3COND_SNC_DEP","TEST20_URBAN_CART_3COND_SNC","TESTGAS_01H",
             "TESTGAS2_01H","TESTPART_01H","TESTPM10_01H","TESTPM10_MULTYR_01H","TESTPRT2_01H","TESTPRT2_MON")
  cases_length <- length(cases)
  cases_versions_length <- length(testcase_versions)
  
  ## Set and create runtime variables and vectors
  is_active_runtime_tests <- TRUE
  number_of_runtime_cycles <- 1
  if (!exists("runtime_stats")) {
    runtime_stats <- cases[1]
    for (i in 2: cases_length) {
      ## adding all of the case names to the runtime_stats tibble.
      runtime_stats <- rbind(runtime_stats, cases[i])
    }
  }
  
  ##  Loop through AERMOD test cases to create rdata files
  for (i in 1:cases_length) {
    print(cases[i])

    runtime_data <- c(number_of_runtime_cycles) #Used to store runtime data
    unprocessed_case_data <- paste(post_dir,cases[i],".PST",sep="")
    if(!file.exists(unprocessed_case_data)) {
      failed_test_case <- c(testcase_versions[j], cases[i])
      ## If the file does not exist, add to the failing test case list.
      if (!exists("failed_test_cases")) {
        failed_test_cases <- failed_test_case
      } else {
        failed_test_cases <- rbind(failed_test_cases, failed_test_case)
      }
      print(paste(unprocessed_case_data, "does not exist."))
      ## set runtime to zero since the case failed.
      average_runtime <- 0
      
    } else if (file.exists(unprocessed_case_data) ) {
      
      tryCatch(
        {
          for (k in 1:runtime_data) {
            start_time <- Sys.time()
            ## Deposition cases are read in differently 
            if( cases[i]=="TESTGAS_01H" || cases[i]=="TESTGAS2_01H" || cases[i]=="TESTPART_01H" || 
                cases[i]=="TESTPRT2_01H"|| cases[i]=="TESTPRT2_MON") {
              case_postdata <- read_POS_TESTDEP.fun(unprocessed_case_data)
            } else {
              case_postdata <- read_POS.fun(unprocessed_case_data)
            }
            ## Compute the time difference before and after running the test case
            runtime_data[k] <- Sys.time() - start_time
          }
          average_runtime <- mean(runtime_data) 
        },
        
        error = function(e) { ## This block is only entered if there is an error when processing the data
          print(paste("There was an error processing", unprocessed_case_data))
          failed_test_case <- c(testcase_versions[j], cases[i])
          if (!exists("failed_test_cases")) {
            failed_test_cases <<- failed_test_case ## the <<- operand assigns the variable on a global scope. This is a problem since we need to modify
            ## the failed_test_cases outside of this current error function
          } else {
            failed_test_cases <<- rbind(failed_test_cases, failed_test_case)
          }
          average_runtime <- 0
        },
        
        ## Sometimes processing will create a warning instead of an error. However, they should still be processed in the
        ## same way since we need the processing to create no errors.
        
        warning = function(w) { ## This block is only entered if there is a warning when processing the data
          print(paste("There was an error processing", unprocessed_case_data))
          failed_test_case <- c(testcase_versions[j], cases[i])
          if (!exists("failed_test_cases")) {
            failed_test_cases <<- failed_test_case ## using <<- to assign to global variable
          } else {
            failed_test_cases <<- rbind(failed_test_cases, failed_test_case)
          }
          average_runtime <<- 0
        })
    }
    
    ##  Save rdata files to rdata_dir and print message  
    if (exists("case_postdata") && length(case_postdata) > 1) {
      save(case_postdata,file=file.path(rdata_dir,paste(cases[i],".rdata",sep="")))
      print(paste("completed creation of ",cases[i],".rdata file",sep=""))
    }
    ## Remove case data if it exists
    if (exists("case_postdata")) {
      rm(case_postdata) 
    }
    
    ## Uncommemnt line below if you would like to delete FORTRAN .PST files from postfiles directory
    #  file.remove(paste(post_dir,cases[i],".PST",sep=""))
    
    ## adding to the runtime vector
    runtime_stats <- rbind(runtime_stats, average_runtime)
  }
  
  print("Processing rdata files complete") 
  
  if(exists("failed_test_cases")) {
    print(paste("Failed cases = ",failed_test_cases,sep=""))
  }else{
    print("No failed test cases")
  }
}

if (exists("failed_test_cases")) {
  finished_failed_test_cases_df <- as.data.frame(failed_test_cases, row.names=FALSE)
  formatted_colnames_df <- c("AERMOD Version", "Test Case")
  write.table(finished_failed_test_cases_df,file=paste(working_dir,"FailedTestCases.csv", sep="/"), sep=",",col.names=formatted_colnames_df,row.names=FALSE)
} else {
  write.table("There were no errors when processing the test cases.",file=paste(working_dir,"FailedTestCases.csv", sep="/"), sep=",",col.names="",row.names=FALSE)
  ## If there are no failed test cases, "FailedTestCases.csv" is still created. 
}

#################################################################################