
UNIT Pcplot87;
{$N+}   { Use a coprocessor }

INTERFACE

USES Graph;

CONST

  matimin   =   1;     { minimum i subscript of matrix type variables }
  matimax   =  20;     { maximum i subscript of matrix type variables }
  matjmin   =   1;     { minimum j subscript of matrix type variables }
  matjmax   =  20;     { maximum j subscript of matrix type variables }
  vecmin    =   1;     { minimum subscript of vector type variables }
  vecmax    = 300;     { maximum subscript of vector type variables }
  maxlevels =  20;     { maximum number of levels used by various routines }
  titlesize =  60;     { maximum number of characters allowed in the title }
  xlabsize  =  15;     { maximum number of characters in abscissa label }
  ylabsize  =  15;     { maximum number of characters in ordinate label }
  lablen    =   5;     { maximum length of numeric labels }

TYPE

  flag    = (normal, warning, error);
  rtype   = Double;
  matrix  = Array[matimin..matimax, matjmin..matjmax] of rtype;
  matvecx = Array[matimin..matimax] of rtype;
  matvecy = Array[matjmin..matjmax] of rtype;
  vector  = Array[vecmin..vecmax] of rtype;
  xltype  = String[xlabsize];
  yltype  = String[ylabsize];
  ttype   = String[titlesize];
  str11   = String[11];
  str80   = String[80];

  stattype = Record
    cond : flag;
    msg  : String[40];
    lvl  : String[20];
  End;

VAR

  matz              : matrix;
  vecx, vecy, vecz  : vector;
  title             : ttype;
  xlabel            : xltype;
  ylabel            : yltype;
  status            : stattype;

{$L Herc.OBJ}
{$L CGA.OBJ}
{$L EVGA.OBJ}



PROCEDURE contour(VAR z : matrix; VAR  xl, yl, xu, yu, xmin, ymin, xmax, ymax,
                              low, up, int : rtype; mt : ttype; fnam : str11);

PROCEDURE barnes(VAR xdata, ydata, zdata : vector; num : Integer;
                 VAR zgrid : matrix; xl, yl, xu, yu, d0 : double);

PROCEDURE xyplot(VAR xdata, ydata : vector;
                     num, ntickx, nticky : Integer;
                     xlow, xhigh  : rtype;
                     ylow, yhigh  : rtype;
                     xtitle       : xltype;
                     ytitle       : yltype;
                     mtitle       : ttype);

IMPLEMENTATION

CONST

  nr = 500; {number of possible starting points}

TYPE

  clist   = Array[1..maxlevels] of rtype;

VAR

  ixstor, iystor : Integer; {current pen position, in common in FORTRAN code}
  nw             : Integer; { number of whole numbers in contour labels }
  nd             : Integer; { number of decimal places in contour labels }
  ir  : Array[1..nr] of Integer; {in common /conre2/ in fortran}
  inx : Array[1..8] of Integer; {possible x step choices}
  iny : Array[1..8] of Integer; {possible y step choices}
  maxplotx, maxploty : Integer; {maximum x,y plotting values}
  minplotx, minploty : Integer; {minimum x,y plotting values}
  maxuserx, maxusery : rtype;   {maximum user coordinates}
  minuserx, minusery : rtype;   {minimum user coordinates}
  np : Integer;                 {stores starting points for contour lines}
  numseg : Integer;             {current line segment}
  graphdriver, graphmode, grapherror : Integer;
  labelthis : Boolean;          {label this line True/False}
  pixsep    : rtype;       { ratio of y pixel distance to x pixel distance }
  segcount  : Integer;     { number of line segments to draw before labeling }
  linelabel : Boolean;     {flag indicating to label major lines or not}
  labelinc  : Integer;     {every "labelinc"th level to be major level}

PROCEDURE  HercDriver; External;

PROCEDURE CGADriver; External;

PROCEDURE EVGADriver; External;

PROCEDURE LeftTrim(Var S : Str80);
Begin
  While S[1] = ' ' do begin
    S := Copy(S,2,Length(S));
    If Length(S) = 0 then Exit;
  end;
End;


PROCEDURE graphcheck;

BEGIN

  {check for error on last graphics call}
  grapherror := GraphResult;
  IF grapherror <> grok THEN
    BEGIN
      CloseGraph;
      WRITELN(' *** Graphics error *** ');
      WRITELN(GraphErrorMsg(grapherror));
      HALT;
    END

END; {graphcheck}


FUNCTION ixypak(ixcoord, iycoord : Integer) : Integer;

BEGIN

  {store x, y coords in single entry of one array}
  ixypak := (iycoord-1)*(matimax-matimin+1) + ixcoord;

END;


FUNCTION xscale(x : rtype) : rtype;

BEGIN

  {currently assumes linear mapping}
  xscale := (x - minuserx) / (maxuserx - minuserx);

END; {xscale}


FUNCTION yscale(y : rtype) : rtype;

BEGIN

  {currently assumes linear mapping}
  yscale := (y - minusery) / (maxusery - minusery);

END; {yscale}


FUNCTION inside(mxa, mxb, mya, myb : Integer) : Boolean;

BEGIN

  {check to see if points are physically on the screen}
  IF (mxa < 0) OR (mya < 0) OR (mxb > GetMaxX) OR (myb > GetMaxY) THEN
    inside := False
  ELSE
    inside := True;

END; {inside}


PROCEDURE setmap(mxa, mxb, mya, myb : Integer; gxc, gxd, gyc, gyd : rtype;
              ltype : Integer);

BEGIN

  CASE ltype OF

    1 : BEGIN {linear both axis}
          IF inside(mxa, mxb, mya, myb) THEN
            BEGIN
              minplotx := mxa;
              maxplotx := mxb;
              minploty := mya;
              maxploty := myb;
              minuserx := gxc;
              maxuserx := gxd;
              minusery := gyc;
              maxusery := gyd;
            END
          ELSE
            BEGIN
              status.cond := error;
              status.msg := 'one or more image coordinates out of range';
              status.lvl := 'setmap';
            END;
        END
  ELSE
    BEGIN
      status.cond := error;
      status.msg := 'nonlinear mapping not supported';
      status.lvl := 'setmap';
    END;
  END; {case, else}

END; {setmap}


PROCEDURE fl2int(x, y : rtype; VAR mx, my : Integer);

BEGIN

  {currently only linear scaling}
  mx := Round(xscale(x)*(maxplotx-minplotx)) + minplotx;

  {note that 0,0 is at top left rather than bottom left}
  {so mapping equation is different for y than x}
  my := maxploty - Round(yscale(y)*(maxploty-minploty));

END; {fl2int}


PROCEDURE matminmax(VAR a : matrix; VAR min, max : rtype);

VAR

  i,j : Integer;

BEGIN

  {find the minimum and maximum of the grid}
  min := a[matimin,matjmin];
  max := a[matimin,matjmin];

    FOR i := matimin TO matimax DO
      FOR j := matjmin TO matjmax DO
        BEGIN
          IF a[i,j] > max THEN max := a[i,j];
          IF a[i,j] < min THEN min := a[i,j];
        END;

END; {matminmax}


PROCEDURE clgen(VAR z : matrix; low, high, incr : rtype; VAR cl : clist;
                VAR nl : Integer; VAR cnstfld : Boolean);
TYPE

  hltestval = (ok, same, bad);
  inctestval = (zero, neg, pos);

VAR

  hltest  : hltestval;  { test for high/low conditions }
  inctest : inctestval; { test for incr conditions }
  i, j    : Integer;    { counters }

BEGIN

  { assume normal completion, if not true will be reset}
  cnstfld := False;
  status.cond := normal;
  status.msg := 'normal completion';
  status.lvl := 'clgen';

  IF high > low THEN      { normal condition}
    hltest := ok
  ELSE
    IF high = low THEN    { only one contour regardless of "incr" }
      hltest := same
    ELSE                  { error low > high, not allowed }
      hltest := bad;


  CASE hltest OF     { handle results of above test }

    ok    : BEGIN    { normal high/low look at incr }

              IF incr = 0.0 THEN
                inctest := zero
              ELSE
                IF incr > 0.0 THEN
                  inctest := pos
                ELSE
                  inctest := neg;

              CASE inctest OF

                zero : BEGIN  { report error and set nl to 0 }

                         status.cond := error;
                         status.msg := 'zero contour interval not allowed';
                         status.lvl := 'clgen';
                         nl := 0;

                       END; {zero case}

                neg :  BEGIN  { set to ten levels between min and max }

                         matminmax(z, low, high); { get min and max }

                         {check for constant field}
                         IF low = high THEN

                           BEGIN
                             nl := 0;
                             cnstfld := True;
                             status.cond := error;
                             status.msg := 'constant field detected';
                             status.lvl := 'clgen';
                           END

                         ELSE

                           BEGIN

                             nl := 10;
                             FOR i := 1 TO 10 DO
                               cl[i] := ((high-low)/9.0)*(i-1) + low;

                           END; {else}

                       END; {neg case}

                pos :  BEGIN  { start at low and step until past high }

                         i := 1;

                         WHILE (i <= maxlevels) AND (low <= high) DO
                           BEGIN
                             cl[i] := low;
                             low := low + incr;
                             Inc(i);
                           END;

                         nl := i - 1;

                         IF low < high THEN
                           BEGIN
                             nl := maxlevels;
                             status.cond := warning;
                             status.msg := 'upper limit not reached, maxlevels expired';
                             status.lvl := 'clgen';
                           END;

                         {check for constant field}
                         matminmax(z, low, high); { get min and max }

                         IF low = high THEN

                           BEGIN
                             nl := 0;
                             cnstfld := True;
                             status.cond := error;
                             status.msg := 'constant field detected';
                             status.lvl := 'clgen';
                           END;

                       END; {pos case}

               END; {inctest}

            END; {ok case}

    same  : BEGIN  { only one contour level }

              nl := 1;
              cl[1] := low;

              {check for constant field}
              matminmax(z, low, high); { get min and max }

              IF low = high THEN

                BEGIN
                  nl := 0;
                  cnstfld := True;
                  status.cond := error;
                  status.msg := 'constant field detected';
                  status.lvl := 'clgen';
                END; {if low = high}

            END; {same case}

    bad   : BEGIN  { low > high indicate error condition }

              nl := 0;
              status.cond := error;
              status.msg := 'Lower contour interval exceeds upper';
              status.lvl := 'clgen';

            END; {bad case}

  END; { case hltest }

END; { clgen }


FUNCTION fx(x, y : rtype) : rtype;

{to apply transformations change this function}
BEGIN
  fx := x;
END; {fx}


FUNCTION fy(x, y: rtype) : rtype;
Begin
  fy := y;
End; {fy}


FUNCTION c(p1, p2, cv : rtype) : rtype;
BEGIN
  c := (p1 - cv) / (p1 - p2);
END; {c}


PROCEDURE frstd(fldx, fldy : rtype);
VAR

  ixpen, iypen : Integer;

BEGIN

  {this is really a needless procedure for the current case it simply  }
  {updates the global current pen position, which could be done in the }
  {calling routine. It is included here to remind that for future cases}
  {it may need to be updated. The FORTRAN version moved the pen to an  }
  {impossible position stored the location and then moved the pen with }
  {the pen up to the wanted location, here I have simply recorded this }
  {position from the start}

  {calculate integer plotting coord's}
  fl2int(fldx, fldy, ixpen, iypen);

  {store pen location}
  ixstor := ixpen;
  iystor := iypen;

END; {frstd}


PROCEDURE labelit(x1, y1, x2, y2 : Integer; cv : rtype);


VAR

  step, index  : Integer; {loop control variables}
  smid, last   : Integer; {midpoint and last index of the label}
  sxpt, sypt   : Integer; {integer plotting coordinates of label}
  xmid, ymid   : Integer; {midpoint coordinates of the line segment}
  xdif, ydif   : Integer; {difference in x, y coordinates}
  delxminus    : Integer; {left x coordinate of label string}
  delxplus     : Integer; {right x coordinate of label string}
  delyminus    : Integer; {bottom y coordinate of vertical label string}
  delyplus     : Integer; {top y coordinate of vertical label string}
  ctrlabel     : String[lablen]; {contour label string}
  testchar     : String[1];      {string to determine character size}
  m, b         : rtype;   {line parameters}
  npix         : Word;

BEGIN

  SetTextJustify(CenterText, CenterText);
  graphcheck;

  testchar := '5';
  npix := TextWidth(testchar);           { width of one label character }

  xmid := (x1+x2) DIV 2;                 { x coord of line midpoint }
  ymid := (y1+y2) DIV 2;                 { y coord of line midpoint }
  xdif := x2 - x1;                       { difference between x end points }
  ydif := y2 - y1;                       { difference between y end points }
  Str(cv:nw:nd, ctrlabel);               { convert label to string }
  last := Length(ctrlabel);              { find length of the label string }
  delxminus := xmid-((npix*last) div 2); { left side of label}
  delxplus  := xmid+((npix*last) div 2); { right side of label}
  delyminus := ymid-((npix*last) div 2); { bottom of label}
  delyplus  := ymid+((npix*last) div 2); { top of label }

  IF Odd(last) THEN
    smid := (last DIV 2) + 1
  ELSE
    smid := last DIV 2;

  {if horizontal or vertical one call to "OutTextXY" will do}
  IF (xdif = 0) OR (ydif = 0) THEN
    BEGIN
      IF (xdif <> 0) OR (ydif <> 0) THEN
        BEGIN
          IF xdif = 0 THEN
            BEGIN
              SetTextStyle(DefaultFont, VertDir, 1);
              IF y2 < y1 THEN
                BEGIN
                  IF (delyminus > y2) AND (delyplus < y1) THEN
                    BEGIN
                      Line(x1, y2, x1, delyminus-1);
                      Line(x1, delyplus+1, x1, y1);
                    END;
                END
              ELSE
                IF (delyminus > y1) AND (delyplus < y2) THEN
                  BEGIN
                    Line(x1, y1, x1, delyminus-1);
                    Line(x1, delyplus+1, x1, y2);
                  END;
              OutTextXY(xmid, ymid, ctrlabel);
            END {xdif = 0}
          ELSE
            BEGIN
              SetTextStyle(DefaultFont, HorizDir, 1);
              IF x2 < x1 THEN
                BEGIN
                  IF (delxminus > x2) AND (delxplus < x1) THEN
                    BEGIN
                      Line(x2, y1, delxminus-1, y1);
                      Line(delxplus+1, y1, x1, y1);
                    END;
                END
              ELSE
                IF (delxminus > x1) AND (delxplus < x2) THEN
                  BEGIN
                    Line(x1, y1, delxminus-1, y1);
                    Line(delxplus+1, y1, x2, y1);
                  END;
              OutTextXY(xmid, ymid, ctrlabel);
            END;
        END
    END {xdif = 0 or ydif = 0}

  {case of a sloped line segment}
  ELSE
    BEGIN
      m := ydif/xdif;
      b := y1 - m*x1;

      IF x1 > x2 THEN     {left to right}
        BEGIN
          step := -1;
          index := last;
          IF (delxminus > x2) AND (delxplus < x1) THEN
            BEGIN
              Line(x2, y2, delxminus-1, Round(m*delxminus + b));
              Line(delxplus+1, Round(m*delxplus + b), x1, y1);
            END;
        END {x1 > x2}
      ELSE
        BEGIN             {right to left}
          step := 1;
          index := 1;
          IF (delxminus > x1) AND (delxplus < x2) THEN
            BEGIN
              Line(x1, y1, delxminus-1, Round(m*delxminus + b));
              Line(delxplus+1, Round(m*delxplus + b), x2, y2);
            END;
        END;

      SetTextStyle(DefaultFont, HorizDir, 1);
      WHILE (index >= 1) AND (index <= last) DO
        BEGIN
          sxpt := (index - smid)*npix + xmid;
          sypt := Round(m*sxpt + b);
          OutTextXY(sxpt, sypt, ctrlabel[index]);
          index := index + step;
        END; {while}
    END;

    numseg := 0;

END; {labelit}


PROCEDURE vectd(fldx, fldy, cv : rtype);

VAR

  ixpen, iypen : Integer;

BEGIN

  {calculate integer pen position}
  fl2int(fldx, fldy, ixpen, iypen);

  IF (labelthis AND (numseg >= segcount)) THEN
    labelit(ixstor, iystor, ixpen, iypen, cv)
  ELSE
    Line(ixstor, iystor, ixpen, iypen);

  graphcheck;


  ixstor := ixpen;
  iystor := iypen;

END; {vectd}


PROCEDURE drline(VAR z : matrix; VAR ix, iy, idx, idy, is : Integer; cv : rtype; open : Boolean);

VAR

  isub, ix0, iy0, is0, ix2, iy2 : Integer; {indexes}
  xold, yold, x, y : rtype;                {user coordinates}
  out : Boolean;                           {loop control variable}

BEGIN

  {two basic contour type exist, open and closed  }
  {open contours start and end on a boundary      }
  {closed contours start and end at the same point}
  numseg := 0;
  ix0 := ix;
  iy0 := iy;
  is0 := is;
  out := False;

  IF idx = 0 THEN

    BEGIN
      x := ix;
      isub := iy + idy;
      y := c(z[ix,iy],z[ix,isub],cv)*idy + iy;
    END

  ELSE

    BEGIN
      y := iy;
      isub := ix + idx;
      x := c(z[ix,iy],z[isub,iy],cv)*idx + ix;
    END;

  frstd( fx(x,y), fy(x,y) ); { set starting point }

  REPEAT

    BEGIN

      Inc(is);

      IF is>8 THEN
        is := is - 8;  { inx/iny are circular }
      idx := inx[is];
      idy := iny[is];
      ix2 := ix + idx;
      iy2 := iy + idy;

      IF open THEN
        IF (ix2 > matimax) OR (iy2 > matjmax) OR (ix2 < matimin) OR (iy2 < matjmin) THEN
          BEGIN
            {ncar called lastd, no need here}
            Exit;  { return from preocedure }
          END; {bounds check}

      IF cv <= z[ix2,iy2] THEN

        BEGIN
          is := is + 4;
          ix := ix2;
          iy := iy2;
        END

      ELSE

        BEGIN

          IF Odd(is) THEN

            BEGIN

              IF idx = 0 THEN
                BEGIN
                  x := ix;
                  isub := iy + idy;
                  y := c(z[ix,iy],z[ix,isub],cv)*idy + iy;
                END
              ELSE
                BEGIN
                  y := iy;
                  isub := ix + idx;
                  x := c(z[ix,iy],z[isub,iy],cv)*idx + ix;
                END;

              vectd( fx(x,y), fy(x,y), cv );
              Inc(numseg);
              xold := x;
              yold := y;

              IF is = 1 THEN

                BEGIN

                  Inc(np);

                  IF np > nr THEN
                    BEGIN
                      { ncar called lastd, no need here}
                      out := True;
                    END
                  ELSE
                    BEGIN
                      ir[np] := ixypak(ix,iy);
                      IF Not open THEN
                        IF (ix = ix0) AND (iy = iy0) AND (is = is0) THEN
                          BEGIN
                            {ncar called lastd, no need here}
                            out := True;
                          END;
                    END;

                END

              ELSE

                IF Not open THEN
                  IF (ix = ix0) AND (iy = iy0) AND (is = is0) THEN
                    BEGIN
                      {ncar called lastd, no need here}
                      out := True;
                    END;

            END; {if Odd(is)}

        END;

    END; {repeat}

  UNTIL out;

END; {drline}


PROCEDURE stline(VAR z : matrix; cv: rtype);

VAR

  i, j, k, ip1, jp1 : Integer;       {counters}
  ix, iy, idx, idy, is, ixy : Integer; {CONREC2 common variables moved to local}
  open : Boolean; {replaces iss, true if open contour}
  noskip : Boolean;

BEGIN

  { CONREC2 common block varaibles in FORTRAN code }
  open := True;
  np := 0;

  {look for contours along the perimeter, i.e. open contours}
  FOR ip1 := matimin+1 TO matimax DO

    BEGIN

      i := ip1 - 1;
      IF (z[i,matjmin] < cv) AND (z[ip1,matjmin] >= cv) THEN
        BEGIN
          ix := ip1;
          iy := matjmin;
          idx := -1;
          idy := 0;
          is := 1;
          drline(z, ix, iy, idx, idy, is, cv, open);
        END;

      IF (z[ip1,matjmax] < cv) AND (z[i,matjmax] >= cv) THEN
        BEGIN
          ix := i;
          iy := matjmax;
          idx := 1;
          idy := 0;
          is := 5;
          drline(z, ix, iy, idx, idy, is, cv, open);
        END;

    END; {for ip1}

  FOR jp1 := matjmin+1 TO matjmax DO

    BEGIN

      j := jp1-1;
      IF (z[matimax,j] < cv) AND (z[matimax,jp1] >= cv) THEN
        BEGIN
          ix := matimax;
          iy := jp1;
          idx := 0;
          idy := -1;
          is := 7;
          drline(z, ix, iy, idx, idy, is, cv, open);
        END;

      IF (z[matimin,jp1] < cv) AND (z[matimin,j] >= cv) THEN
        BEGIN
          ix := matimin;
          iy := j;
          idx := 0;
          idy := 1;
          is := 3;
          drline(z, ix, iy, idx, idy, is, cv, open);
        END

    END; {for jp1}

  open := False;

  {now find closed contours}
  FOR jp1 := matjmin+2 TO matjmax DO

    BEGIN

      j := jp1 - 1;

      FOR ip1 := matimin+1 TO matimax DO

        BEGIN

          i := ip1 - 1;

          IF (z[i,j] < cv) AND (z[ip1,j] >= cv) THEN

            BEGIN

              ixy := ixypak(ip1,j);
              noskip := True;
              k := 1;
              WHILE (k <= np) AND noskip DO
                BEGIN
                  IF ir[k] = ixy THEN
                    noskip := False;
                  Inc(k);
                END;
              IF noskip THEN
                BEGIN
                  Inc(np);
                  IF np > nr THEN
                    Exit;
                  ir[np] := ixy;
                  ix := ip1;
                  iy := j;
                  idx := -1;
                  idy := 0;
                  is := 1;
                  drline(z, ix, iy, idx, idy, is, cv, open);
                END;

            END; {if z[i,j]...}

        END; {for ip1}

    END; {for jp1}

END; {stline}


FUNCTION power(number, expon : double) : double;

BEGIN

  power := Exp(expon*Ln(number));

END; {power}


PROCEDURE barnes;

TYPE

  matvecx = Array[matimin..matimax] of Single;
  matvecy = Array[matjmin..matjmax] of Single;

LABEL

  firstime;

VAR

  ix, iy, k, l, inner : Integer;
  mm, nn, x2, y2      : Integer;
  grdat1              : matrix;
  w, dist, k0, k1     : double;
  weight, dum         : double;
  sumwgt, wgtfun      : double;
  interp, factor      : double;
  z1, z2, z3, z4      : double;
  dx, dy, err         : double;
  avespac, minspac    : double;
  xm1, xm2, gamma     : double;
  d1, df, rc, rcsav   : double;
  temp                : String[6];
  zx                  : matvecx;
  zy                  : matvecy;
  insidex, insidey    : vector;
  insidedat           : vector;
  first               : boolean;

BEGIN

  status.cond := normal;
  status.msg := 'normal completion';
  status.lvl := 'Barnes';

  gamma := 0.3;

  d1 := power(d0,gamma);
  df := d0*( 1.0 + power(d0,gamma-1.0) - power(d0,gamma) );

  IF (df <= 0.0) OR (df >= 1.0) THEN
    BEGIN
      status.cond := error;
      status.msg := 'resolution parameter "d" out of range';
      status.lvl := 'barnes';
      Exit;
    END;

  IF (xl > xu) OR (yl > yu) THEN
    BEGIN
      status.cond := error;
      status.msg := 'one or more lower boundary greater than upper';
      status.lvl := 'Barnes';
      Exit;
    END;

  {calculate x, y coordinates along output grid}
  FOR k := matimin TO matimax DO
    zx[k] := ((xu-xl)/(matimax-matimin))*(k-matimin) + xl;

  FOR k := matjmin TO matjmax DO
    zy[k] := ((yu-yl)/(matjmax-matjmin))*(k-matjmin) + yl;

  {calculate twice average grid spacing}
  avespac := 0.0;
  inner := 0;
  FOR k := vecmin TO vecmin + num - 1 DO

    IF (xdata[k] >= xl) AND (xdata[k] < xu) AND (ydata[k] >= yl) AND (ydata[k] < yu) THEN
      BEGIN

        insidex[inner+vecmin] := xdata[k];
        insidey[inner+vecmin] := ydata[k];
        insidedat[inner+vecmin] := zdata[k];
        Inc(inner);
        minspac := 99.99E25;

        FOR l := vecmin TO vecmin + num - 1 DO
          BEGIN
            IF (xdata[l] >= xl) AND (xdata[l] < xu) AND (ydata[l] >= yl) AND (ydata[l] < yu) THEN
              BEGIN
                IF ( (xdata[k] <> xdata[l]) OR (ydata[k] <> ydata[l]) ) THEN
                  BEGIN
                    dist := Sqr(xdata[l]-xdata[k]) + Sqr(ydata[l]-ydata[k]);
                    IF (dist < minspac) THEN
                      minspac := dist;
                  END;
              END;
          END;

          avespac := avespac + minspac;

      END; {if xdata[k]}

  avespac := 4.0*avespac / inner;
  k0 := (-1.0*Ln(d0)*avespac) / Sqr(Pi);
  k1 := gamma*k0;
  rc := 20.0*k0;
  first := true;

  {calculate spacing on output grid}
  dx := zx[matimin+1] - zx[matimin];
  dx := 1.0 / dx;
  dy := zy[matjmin+1] - zy[matjmin];
  dy := 1.0 / dy;

  factor := -1.0 / k0;

Firstime:
  {first guess pass}
  FOR iy := matjmin TO matjmax DO

    FOR ix := matimin TO matimax DO

      BEGIN

        sumwgt := 0.0;
        wgtfun := 0.0;

        FOR k := vecmin TO vecmin + inner - 1 DO

          BEGIN

              dist := Sqr(zx[ix] - insidex[k]) + Sqr(zy[iy] - insidey[k]);
              IF (dist <= rc) THEN
                BEGIN
                  w := dist*factor;
                  weight := Exp(w);
                  dum := weight*insidedat[k];
                  wgtfun := wgtfun + dum;
                  sumwgt := sumwgt + weight;
                END; {if dist <= rc}

          END; {for k}

          If (First) and (sumwgt = 0.0) then begin
              rcsav := rc;
              rc := 1.0E30;
              first := false;
              goto firstime;
          end
          else begin
            if (sumwgt = 0.0) then
              grdat1[ix,iy] := 0.0
            else
              grdat1[ix,iy] := wgtfun / sumwgt;
          end;

      END; {for ix}

  factor := -1.0 / k1;
  rc := rcsav;

  {correction pass}
  FOR iy := matjmin TO matjmax DO

    FOR ix := matimin TO matimax DO

      BEGIN

        sumwgt := 0.0;
        wgtfun := 0.0;

        FOR k := vecmin TO vecmin + inner - 1 DO

          BEGIN

              dist := Sqr(zx[ix] - insidex[k]) + Sqr(zy[iy] - insidey[k]);
              IF (dist <= rc) THEN
                BEGIN
                  w := dist*factor;
                  weight := Exp(w);
                  mm := matimin + Trunc((insidex[k]-zx[matimin])*dx);
                  nn := matjmin + Trunc((insidey[k]-zy[matjmin])*dy);
                  x2 := mm + 1;
                  y2 := nn + 1;
                  xm1 := insidex[k] - zx[mm];
                  xm2 := insidey[k] - zy[nn];
                  z1 := grdat1[mm,nn];
                  z2 := grdat1[mm,y2];
                  z3 := grdat1[x2,y2];
                  z4 := grdat1[x2,nn];
                  interp := z1 + ((xm1*(z4-z1))*dx + (xm2*(z2-z1))*dy + (((xm1*xm2)*dx)*(z3-z4+z1-z2))*dy);
                  err := insidedat[k] - interp;
                  dum := weight*err;
                  wgtfun := wgtfun + dum;
                  sumwgt := sumwgt + weight;
                END; {if dist <= rc}

          END; {for k}

          If sumwgt = 0.0  then
            zgrid[ix,iy] := grdat1[ix,iy]
          else
            zgrid[ix,iy] := grdat1[ix,iy] + wgtfun/sumwgt;

      END; {for ix}

END; {barnes}


PROCEDURE xyplot;

VAR

  radius     : Word;
  mxa, mxb   : Integer;
  mya, myb   : Integer;
  px, py, i  : Integer;
  xmid, ymid : Integer;
  axislabel  : String[lablen];

BEGIN

  radius := 5;

  InitGraph(graphdriver, graphmode, '');
  graphcheck;
  SetLineStyle(SolidLn, 0, NormWidth);
  graphcheck;

  mxb := GetMaxX - ((TextWidth('h')*lablen) DIV 2) + Round(2*pixsep);
  mxa := (TextWidth('h')*lablen) + Round(pixsep*2);
  mya := TextHeight('h') + 2;
  myb := GetMaxY - (TextHeight('h') + 2);

  setmap(mxa, mxb, mya, myb, xlow, xhigh, ylow, yhigh, 1);

  { draw x axis and y axis}
  Line(minplotx, maxploty, maxplotx, maxploty);
  Line(minplotx, maxploty, minplotx, minploty);

  { place tick marks on x axis}
  FOR i := 0 TO ntickx-1 DO
    BEGIN
      px := (i*(maxplotx-minplotx)) DIV (ntickx-1) + minplotx;
      Line(px, maxploty+2, px, maxploty-2);
    END;

  { place tick marks on y axis}
  FOR i := 0 TO nticky-1 DO
    BEGIN
      py := (i*(maxploty-minploty)) DIV (nticky-1) + minploty;
      Line(Round(minplotx-2*pixsep), py, Round(minplotx+2*pixsep), py);
    END;

  { write "xlow" and "xhigh" on x axis }
  SetTextStyle(DefaultFont, HorizDir, 1);
  SetTextJustify(CenterText, BottomText);
  Str(xlow:nw:nd, axislabel);
  OutTextXY(minplotx, GetMaxY, axislabel);
  Str(xhigh:nw:nd, axislabel);
  OutTextXY(maxplotx, GetMaxY, axislabel);

  { write "ylow" and "yhigh" on y axis}
  SetTextJustify(CenterText, CenterText);
  Str(ylow:nw:nd, axislabel);
  OutTextXY((minplotx DIV 2), maxploty, axislabel);
  Str(yhigh:nw:nd, axislabel);
  OutTextXY((minplotx DIV 2), minploty, axislabel);

  { plot points }
  FOR i := vecmin TO vecmin + num - 1 DO
    BEGIN
      fl2int( fx(xdata[i],ydata[i]), fy(xdata[i],ydata[i]), px, py );
      Circle(px, py, radius);
    END;

  { put title and axis labels on plot }
  xmid := (maxplotx + minplotx) DIV 2;
  ymid := (maxploty + minploty) DIV 2;
  SetTextStyle(DefaultFont, HorizDir, 1);
  SetTextJustify(CenterText, TopText);
  OutTextXY(xmid, 0, mtitle);
  SetTextJustify(CenterText, BottomText);
  OutTextXY(xmid, GetMaxY, xlabel);
  SetTextStyle(DefaultFont, VertDir, 1);
  SetTextJustify(CenterText, CenterText);
  OutTextXY((minplotx DIV 2), ymid, ylabel);

  READLN;
  CloseGraph;

END; {xyplot}

PROCEDURE plothill(filnam : Str11);
Var
  i,NumCtrs,
  NPCSV,NPCSVE,
  j,xconi,yconi   : integer;

  xcon,ycon       : rtype;

  PLTfile         : text;

Begin
  Assign(PLTFile,'\CTDM\TERRAIN\'+filnam);
  Reset(PLTFile);
  For i := 1 to 3 do                              {skip down to coordinates}
    Readln(PLTfile);
 Readln(PLTfile,NumCtrs);
 For i := 1 to NumCtrs do                  { skip through sorted id numbers }
   Readln(PLTFile);
 Readln(PLTFile); Readln(PLTFile);                   { skip over boundaries }
 For i := 1 to NumCtrs do begin
   Readln(PLTfile,NPCSV);                 { get number points on unedited contour }
   Readln(PLTfile,xcon,ycon);                   { get initial contour point }
   fl2int(xcon,ycon,xconi,yconi);
   MoveTo(xconi,yconi);
   PutPixel(xconi,yconi,1);
   For j := 2 to NPCSV do begin                {draw the rest of the contours}
     Readln(PLTfile,xcon,ycon);
     fl2int(xcon,ycon,xconi,yconi);
     LineTo(xconi,yconi);
   end;
   Readln(PLTFile,NPCSVE);                        { skip over edited contours}
   For j := 1 to NPCSVE do
     Readln(PLTFile,xcon,ycon);
 end;
 Close(PLTFile);
end;        {PlotHill}

PROCEDURE PressEnter;
Var
  Middle, TitleWidth,TitleHeight : Integer;
  S : String[24];
Begin
  S := 'Press ENTER to continue.';
  TitleHeight := TextHeight(S);
  SetTextStyle(0,0,0);
  SetTextJustify(CenterText,TopText);
  Middle := GetMaxX div 2;
  OutTextXY(Middle,GetMaxY - TitleHeight,S);
end; { PressEnter }


PROCEDURE contour;

TYPE

  lvltype = (minor, major);
  lvllist = Array[1..maxlevels] of lvltype;

VAR

  cl        : clist;   {list of contour levels returned from clgen}
  nl        : Integer; {number of contour levels returned from clgen}
  index     : Integer; {counter}
  mxa, mxb  : Integer; { min, max x screen coordinates }
  mya, myb  : Integer; { min, max y screen coordinates }
  xli,xui   : Integer; { array screen coordinates }
  yli,yui   : Integer; { array screen coordinates}
  xmid      : Integer; { x midpoint of the screen }
  aaspect   : Real;    {input array aspect ratio}
  saspect   : Real;    {screen aspect ratio}
  cnstfld   : Boolean; {flag to indicate if constant field was found}
  lvlstyle  : lvllist; {array indicating level type (minor/major) }

BEGIN


  { label every "labelinc"th level}
  FOR index := 1 TO maxlevels DO
    lvlstyle[index] := minor;

  IF linelabel THEN

    BEGIN

      index := 1;
      WHILE index <= maxlevels DO
      BEGIN
        lvlstyle[index] := major;
        index := index + labelinc;
      END;

    END; {if linelabel}


  { find contour levels }
  clgen(z, low, up, int, cl, nl, cnstfld);

  {check for constant field and normal completion}
  IF (NOT cnstfld) AND (status.cond <> error) THEN

    BEGIN

      { set up display }
      InitGraph(graphdriver, graphmode, '');
      graphcheck;
      SetLineStyle(DashedLn, 0, NormWidth);   {select dashed lines for hill}
      graphcheck;
      If (GraphDriver = CGA) and (GraphMode = CGAC0) then
        SetColor(3)                              { Select Yellow for contours }
       else
        If GraphDriver in [EGA64,VGA] then
          SetColor(14)
         else
          SetColor(GetMaxColor);

      graphcheck;

      { set up scaling }
      { scale to same aspect ratio as coordinates input }
      aaspect := (ymax - ymin) / (xmax - xmin);
      saspect := (GetMaxY + 1 - 2*(TextHeight('H')+2)) / (GetMaxX + 1);
      IF aaspect >= saspect*pixsep THEN
        { use maximum y screen coordinates and scale x appropriately }
        BEGIN
          mya := 0;
          myb := GetMaxY;
          mxa := (GetMaxX - ROUND( (myb-mya+1)*pixsep/aaspect ) + 1) DIV 2;
          mxb := mxa + ROUND( (myb-mya+1)*pixsep/aaspect) - 1;
        END
      ELSE
        { use maximum x screen coordinates and scale y appropriately }
        BEGIN
          mxa := 0;
          mxb := GetMaxX;
          mya := (GetMaxY - ROUND( ((mxb-mxa+1)*aaspect)/pixsep ) + 1) DIV 2;
          myb := mya + ROUND( ((mxb-mxa+1)*aaspect)/pixsep ) - 1;
        END;
      setmap(mxa, mxb, mya + (TextHeight('H') + 2), myb - (TextHeight('P')+2), xmin, xmax, ymin, ymax, 1);
      If Fnam <> '' then plothill(Fnam);

      SetLineStyle(SolidLn, 0, NormWidth);   {use solid line for contours}

      SetColor(GetMaxColor);

      {draw perimeter in MaxColor }
      Rectangle(minplotx,minploty,maxplotx,maxploty);

      { put title on plot }
      xmid := GetMaxx DIV 2;
      SetTextJustify(CenterText, TopText);
      OutTextXY(xmid, 0, mt);

      If (GraphDriver = CGA) and (GraphMode = CGAC0) then
        SetColor(2);                               { Select HiRed for Elipses }
      If GraphDriver in [EGA,EGA64,VGA] then
        SetColor(LightRed);                        { Select HiRed for Elipses }

      fl2int(xl,yl,xli,yli);
      fl2int(xu,yu,xui,yui);
      setmap(xli, xui, yli, yui, ROUND(matimin), ROUND(matimax), ROUND(matjmin), ROUND(matjmax), 1);
      IF status.cond = error THEN
         BEGIN
           WRITELN('***ERROR*** from routine contour');
           WRITELN(status.msg);
           WRITELN(status.lvl);
           HALT;
         END;

      graphcheck;

      { if labeling scale labels }
      {IF linelabel THEN
        scalelab(cl, nl, lvlstyle);}

      FOR index := 1 TO nl DO

        BEGIN

          CASE lvlstyle[index] OF

            minor : BEGIN
                      labelthis := False;
                      stline(z, cl[index]);
                    END; {minor case}

            major : BEGIN
                      labelthis := True;
                      stline(z, cl[index]);
                    END; {major case}

          END; {case of lvlstyle}

        END; {for index}


      {allow user to view plot before closing plot}
      SetColor(GetMaxColor);

      PressEnter;
      READLN;
      CloseGraph;

    END {if not cnstfld}

  ELSE

    BEGIN

      WRITELN('***ERROR*** from routine contour');
      WRITELN(status.msg);
      WRITELN(status.lvl);

    END;

END; {contour}



BEGIN { PCPLOT start up code }

  { initialize inx, iny }

  inx[1] := -1;       { ix increment array ("idx") }
  inx[2] := -1;            { ---------------- }
  inx[3] := 0;             { | -1 |  0 | +1 | }
  inx[4] := 1;             { |--------------| }
  inx[5] := 1;             { | -1 |  X | +1 | }
  inx[6] := 1;             { |--------------| }
  inx[7] := 0;             { | -1 |  0 | +1 | }
  inx[8] := -1;            { ---------------- }

  iny[1] := 0;       { ix increment array ("idx") }
  iny[2] := 1;            { ---------------- }
  iny[3] := 1;            { | +1 | +1 | +1 | }
  iny[4] := 1;            { |--------------| }
  iny[5] := 0;            { |  0 |  Y |  0 | }
  iny[6] := -1;           { |--------------| }
  iny[7] := -1;           { | -1 | -1 | -1 | }
  iny[8] := -1;           { ---------------- }

  If RegisterBGIdriver(@HercDriver) < 0 then graphcheck;
  If RegisterBGIdriver(@CGADriver) < 0 then graphcheck;
  If RegisterBGIdriver(@EVGADriver) < 0 then graphcheck;

  { try to autodetect graphics device type}
  graphdriver := Detect;

  DetectGraph(graphdriver, graphmode);
  graphcheck;

  CASE graphdriver OF

    CGA      : BEGIN
                 IF graphdriver = CGA THEN
                   IF graphmode = cgahi THEN
                     pixsep := 7.0/3.0
                   ELSE
                     pixsep := 3.5/3.0;
               END; {cga case}

    HercMono : pixsep := 6.25/4.25;

    EGA      : pixsep := 1.46;

    VGA      : pixsep := 1.474;

  END; {case graphdriver}

  segcount := 9;
  labelinc := 2;
  linelabel := True;
  nw := 1;
  nd := 1;

END. {pcplot}
