C=======================================================================
      MODULE SfcChars
      
C     Variables and subroutines used for writing out derived surface
C     used as input to AERMET.

C     Uses the following modules:
C     - StartVars
C     - Constants
C     - UserParams
C     - ProcCtrlFile
C     - FileUnits
C     - AvgParams

C     Contains the following procedures:
C     - subroutine Write_SfcChars

C=======================================================================

      USE StartVars, only: run_start, iYr2
      USE Constants, only: Version, WinNoSnowLbl, WinWSnowLbl, 
     &                     SpringLbl, SummerLbl, AutumnLbl,
     &                     Eps5
  
      USE UserParams, only: TemporalRes, TempResStr, NumSectors,
     &    StartDir, WndSectors, Anem_Height, IBL_Factor, ZoRad, 
     &    AssignMnth, WinNoSnMnths, WinWSnMnths, SprgMnths, SumrMnths, 
     &    AutmMnths, CoordType, CenterLatIn, CenterLonIn,
     &    Datum, CenterUTME, CenterUTMN, CenterUTMZone, 
     &    VaryAP, Airport, AirportChr, SnowChr, 
     &    MoistStr, AridChr, StatnOrdr
      
      use ProcCtrlFile, only: NLCDYear, ZoEffFlg, ZoRadFlg, 
     &    Zo_Method, TITLE1, TITLE2, UseImp, UseCan                              
      
      USE FileUnits, only: SfcUnt, SfcFile, LogUnt, OutUnt
      
      USE AvgParams, only: AvgAlbedo, AvgBowen, AvgZo

      IMPLICIT NONE
      SAVE
      
      contains     
      
C=======================================================================
      SUBROUTINE Write_SfcChars

C     Output averaged surface characteristic values (B, albedo, Z0) to
C     file in format for input into AERMET
C=======================================================================

      use UserParams, only:    WndSectors, CoordType,
     &                         LCFile, ImpFile, 
     &                         CanFile, Airport,
     &                         tmpStr, tmpStr2, StatnOrdr
      use ProcCtrlFile, only: ImpYear, CanYear
      
      implicit none
      
      DOUBLE PRECISION  :: EndDir       ! wind sector end direction 
      INTEGER (kind=4)  :: NumFreq      ! frequency
      
      INTEGER (kind=4)  :: i, j         ! loop counters
         
C --- Initialize frequency
      NumFreq = 1

C --- Determine frequency based on temporal resolution
      IF( TemporalRes == 'A' )THEN
         NumFreq = 1
      ELSEIF( TemporalRes == 'S' )THEN
         NumFreq = 4
      ELSEIF( TemporalRes == 'M' ) THEN
         NumFreq = 12
      ENDIF

C ---------------------------------------------------------------------
C     Write to output file
C ---------------------------------------------------------------------
         
C --- open sfc characteristic values output file
      OPEN(Unit=SfcUnt,File=SfcFile,status='unknown',err=900)
      
CCRT  --------  WRITE HEADER INFO, SUMMARY OF USER INPUTS --------------
     
C --- write AERSURFACE version date and run date & time
      WRITE(SfcUnt,40,err=910) 'Generated by AERSURFACE, Version ',
     &    trim(Version),
     &    run_start(2),run_start(3),iYr2,run_start(5),
     &    run_start(6),run_start(7)
   40 FORMAT('** ',2a,t67,i2.2,'/',i2.2,'/',i2.2,' **'/
     &       '**',t67,i2.2,':',i2.2,':',i2.2,' **'/)

C --- write titles
      WRITE(SfcUnt,42,err=910) trim(TITLE1)
   42 FORMAT('** Title 1: ',a)
   
C     write title 2 only if length is greater than zero
      IF (len_trim(adjustl(TITLE2)) > 0) THEN
         WRITE(SfcUnt,43,err=910) trim(TITLE2)
      END IF
   43 FORMAT('** Title 2: ',a)

C --- Station location
      
         WRITE(SfcUnt,'("** Primary Site (Zo):")',err=910)
         
         IF( CoordType == 'LATLON' )THEN
            WRITE(SfcUnt,45) CenterLatIn, CenterLonIn, Datum
     
         ELSEIF( CoordType == 'UTM' )THEN
            WRITE(SfcUnt,46) CenterUTME, CenterUTMN, 
     &                       CenterUTMZone, Datum
         END IF 

C ---    Include input datafile names in output file headers
         WRITE(SfcUnt,50) NLCDYear, LCFILE(1:LEN_TRIM(LCFILE))
CRLM    O012_DataYearKeyword_WSP (BEGIN)
CRLM    This NLCDYear check is irrelevant when UseImp and UseCan are used.
CRLM     IF( NLCDYear .NE. 1992 )THEN

           IF( UseImp .and. LEN_TRIM(ImpFile) .GT. 3 )THEN
              WRITE(SfcUnt,51) ImpYear, ImpFile(1:LEN_TRIM(ImpFile))
           ENDIF

           IF( UseCan .and. LEN_TRIM(CanFile) .GT. 3 )THEN
              WRITE(SfcUnt,52) CanYear, CanFile(1:LEN_TRIM(CanFile))
           ENDIF

CRLM     ENDIF
CRLM    O012_DataYearKeyword_WSP (END)

     
      
   45 FORMAT(
     &       '**   Center Latitude  (decimal degrees): ',F12.6,/,
     &       '**   Center Longitude (decimal degrees): ',F12.6,/,
     &       '**   Datum: ', A)
   46 FORMAT(
     &       '**   Center UTM Easting  (meters): ', F10.1,/,
     &       '**   Center UTM Northing (meters): ', F10.1,/,
     &       '**   UTM Zone: ', I3,'  Datum: ', A)
c   48 FORMAT(
c     &       '** Center Latitude  (decimal degrees): ',F12.6,/,
c     &       '** Center Longitude (decimal degrees): ',F12.6,/,
c     &       '** Datum: ', A)
c   49 FORMAT(
c     &       '** Center UTM Easting  (meters): ', F10.1,/,
c     &       '** Center UTM Northing (meters): ', F10.1,/,
c     &       '** UTM Zone: ', I3,'  Datum: ', A)

   50 FORMAT('** NLCD Version:  ', I4,/,
     &       '** NLCD DataFile: ',a)
   51 FORMAT('** MPRV Version:  ', I4,/,
     &       '** MPRV DataFile: ',a)
   52 FORMAT('** CNPY Version:  ', I4,/,
     &       '** CNPY DataFile: ',a)

C --- Non-Airport Sectors

      IF (VaryAP) THEN
C ---    Assign Non-Airport sector IDs to string
         tmpStr = ""
         j = 0
         DO i=1,numSectors
            IF (.NOT.WndSectors%sec_ap(i)) THEN
               j = j + 1
               WRITE(tmpStr2,'(i2)') i
               IF (i==1) THEN
                  tmpStr = trim(adjustl(tmpStr2))
               ELSE
                  tmpStr = trim(adjustl(tmpStr))//" "
     &                   //trim(adjustl(tmpStr2))
               END IF
            END IF
         END DO
         IF (LEN_TRIM(tmpStr) .EQ. 0) THEN
C ---       No NonAP sectors found
            tmpStr = "None"
         ELSE IF (j .EQ. numSectors) THEN
C ---       All sectors are NonAP
            tmpStr = "All"
         END IF
      ELSE IF (.NOT.Airport ) THEN
         tmpStr = "All"
      ELSE
         tmpStr = "None"
      END IF 
CRLM O013_AirportFlag_WSP Begin          
      WRITE(SfcUnt,53,err=910) trim(adjustl(tmpStr))
   53 format('** High Z0 (Non-Aiport) Sector IDs: ',a)
CRLM   53 format('** Non-Airport Sector IDs: ',a)  
   
CRLM O013_AirportFlag_WSP End    
   
C ---    Roughness method  
         WRITE(SfcUnt,54,err=910) Zo_Method
   54    format('** Zo Method: ',a)

C --- Anemometer height and IBL Factor or Radius based on Zo option
      IF (ZoEffFlg) THEN   

C ---    Anemometer height to surface characteristics output file
         WRITE(SfcUnt,55,err=910) Anem_Height
   55    format('** Anemometer Height (m): ',F6.2)

C ---    IBL Factor   
         WRITE(SfcUnt,56,err=910) IBL_Factor
   56    format('** IBL Factor: ',F5.1)  

      ELSE IF (ZoRadFlg) THEN
C ---    Zo Radius   
         WRITE(SfcUnt,58,err=910) ZoRad
   58    format('** Zo Radius (m): ',F8.1)
   
      END IF

     
C --- Continuous snow cover
      WRITE(SfcUnt,73) SnowChr 
   73 FORMAT('** Continuous snow cover: ',A) 

C --- sfc moisture, arid region
      WRITE(SfcUnt,75) trim(adjustl(MoistStr)), AridChr
   75 FORMAT('** Surface moisture: ',A,
     &          ';  Arid: ',A)

C --- set month/season assignment output string
      IF (AssignMnth) THEN
         tmpStr = 'User-specified'
      ELSE
         tmpStr = 'Default'
      ENDIF 
     
C --- month/season assignmemts
      WRITE(SfcUnt,80) trim(adjustl(tmpStr))
  80  FORMAT('** Month/Season assignments: ',A)

      WRITE(SfcUnt,82) 
     &    trim(adjustl(WinNoSnowLbl)),
     &    trim(adjustl(WinNoSnMnths)),
     &    trim(adjustl(WinWSnowLbl)),
     &    trim(adjustl(WinWSnMnths)),
     &    trim(adjustl(SpringLbl)),   
     &    trim(adjustl(SprgMnths)),  
     &    trim(adjustl(SummerLbl)),   
     &    trim(adjustl(SumrMnths)),   
     &    trim(adjustl(AutumnLbl)),   
     &    trim(adjustl(AutmMnths))
  82  FORMAT(5('** ',A,': ',A,/))
   
CCRT  --------------  END OF HEADER AND INPUT SUMMARY ------------------


C --- output temporal resolution and number of sectors
      IF (StatnOrdr == 'SECONDARY') THEN
         WRITE(SfcUnt,105,err=910) trim(adjustl(TempResStr)), 
     &      NumSectors
      ELSE ! Default is PRIMARY
         WRITE(SfcUnt,100,err=910) trim(adjustl(TempResStr)), 
     &      NumSectors
      ENDIF
  100 FORMAT(/,'   FREQ_SECT  ',A,I3)
  105 FORMAT(/,'   FREQ_SECT2 ',A,I3)

C --- output the beginning and ending direction for each sector

C --- the end direction is the start direction for the next sector
C --- if the last sector ends at north, direction should be 360.0
      DO i=1,NumSectors
         IF( i == NumSectors .AND. 
     &     abs(StartDir(1) - 0.0D0) >= Eps5) THEN
            EndDir = StartDir(1)
         ELSEIF( i == NumSectors .AND. 
     &     abs(StartDir(1) - 0.0D0) < Eps5) THEN
            EndDir = 360.0
         ELSE
            EndDir = StartDir(i+1)
         ENDIF
         IF (StatnOrdr == 'SECONDARY') THEN
            WRITE(SfcUnt,205,err=910) i, StartDir(i), EndDir
         ELSE ! Default is PRIMARY
            WRITE(SfcUnt,200,err=910) i, StartDir(i), EndDir
         ENDIF
      ENDDO
C  200 FORMAT( '   SECTOR ',I3,2I5)
  200 FORMAT( '   SECTOR ',I3,2F8.2)
  205 FORMAT( '   SECTOR2',I3,2F8.2)
  
C --- Add comment to output file to indicate the order the 
C     the sfc chars. are output
      IF( TemporalRes == 'M' )THEN
         WRITE(SfcUnt,250,err=910) '**           Month    Sect    Alb',
     &      '      Bo        Zo'
      ELSEIF( TemporalRes == 'S' )THEN
         WRITE(SfcUnt,250,err=910) '**           Season   Sect    Alb',
     &      '      Bo        Zo'
      ELSE
         WRITE(SfcUnt,250,err=910) '**                    Sect    Alb',
     &      '      Bo        Zo'
      ENDIF
  250 FORMAT(/2a)  
  
C --- Output the site characteristics for each sector (i) for each
C     frequency index (j, i.e., 1=annual, 4=seasonal, 12=monthly).
C --- For monthly, replicate seasons according to AERMET 
C     (Winter = Dec, Jan, Feb; Spring = Mar, Apr, May, etc.)

      DO j=1,NumFreq
         DO i=1,NumSectors       

C           bowen ratio and albedo - treat as one sector; each sector gets 
C           the same value - stored as sector 1

C ---       ANNUAL and SEASONAL; 

            IF( NumFreq < 12 )THEN
               IF (StatnOrdr == 'SECONDARY') THEN
                  WRITE(SfcUnt,305,err=910) j,i,AvgAlbedo(1,j),
     &                       AvgBowen(1,j),AvgZo(i,j)
               ELSE ! Default is PRIMARY
                  WRITE(SfcUnt,300,err=910) j,i,AvgAlbedo(1,j),
     &                       AvgBowen(1,j),AvgZo(i,j)
               ENDIF
C ---       MONTHLY
            ELSEIF( NumFreq == 12 )THEN
               IF (StatnOrdr == 'SECONDARY') THEN
                  WRITE(SfcUnt,305,err=910) j,i,
     &               AvgAlbedo(1,j),
     &               AvgBowen(1,j), 
     &               AvgZo(i,j)    
               ELSE ! Default is PRIMARY
                  
                  WRITE(SfcUnt,300,err=910) j,i,AvgAlbedo(1,j),
     &                       AvgBowen(1,j),AvgZo(i,j)
               ENDIF             
           
            ENDIF
         ENDDO
      ENDDO
      
  300 FORMAT( '   SITE_CHAR  ',I3,5x,I3,F9.2,F9.2,F10.3)
  305 FORMAT( '   SITE_CHAR2 ',I3,5x,I3,F9.2,F9.2,F10.3)
  
      CLOSE(SfcUnt)

      RETURN
      
  900 WRITE(*,901) trim(SfcFile)
      WRITE(LogUnt,901) trim(SfcFile)
      WRITE(OutUnt,901) trim(SfcFile)
  901 FORMAT(
     &  /,' There was a problem opening the output file:'
     &  /,a,//,' Processing was aborted.',/)
     
  910 WRITE(*,902) trim(SfcFile)
      WRITE(LogUnt,902) trim(SfcFile)
      WRITE(OutUnt,902) trim(SfcFile)
  902 FORMAT(
     &  /,' There was a problem writing to the output file:'
     &  /,a,//,' Processing was aborted.',/)
      STOP
      
      END SUBROUTINE Write_SfcChars
      
      end module SfcChars