	program tank
c
c......................................................................
c
c	program to estimate the release rate from a reservoir with attached
c	piping.
c
c......................................................................
c
	implicit real*8(a-h,o-z), integer*4(i-n)

	external choke, subcrit
c
c... because of the structure of ZBRENT, a common area is required
c
	common /input/ rmw, pa, p1, t1, gamma, rrr, rn, 
     .		rm2, y2, p2, t2, rm3, y3, p3, t3, ggg
c
c... invariant constants
c
	data pi/3.1415926536D0/
	data in/8/
	data rrr/8314.D0/
c
c... Use a simple user interface since this is not the critical issue here.
c
	open(unit=in, file='tank.dat', status='old')
c
	read(in,*) do		! orifice diameter [=] m
	ao = pi*do**2/4.D0
	read(in,*) dp		! pipe diameter [=] m
	read(in,*) rlp		! pipe length [=]
	read(in,*) rmw		! molecular weight [=] kg/kmol
	read(in,*) ne		! nunmber of elbows [=] d'less
	read(in,*) pa		! ambient pressure [=] N/m**2
	read(in,*) p1		! reservoir pressure [=] N/m**2
	read(in,*) t1		! reservoir temperature [=] K
	read(in,*) gamma	! heat capacity ratio [=] d'less
c
c... A.  Pipe Friction Loss.  Estimate the pipe friction loss..................
c
	fff = 0.0045D0
  100	continue
	rn = 4.D0 * fff * rlp/dp  + 0.5D0  +  0.75D0*real(ne)
	if(do .le. 0.2D0*dp) rn = rn + 0.5D0	! entrance losses
	write(6,*) 'Pipe friction loss: ',rn
c
c... B.  Choked flow.  Estimate the discharge rate as if the flow is choked....
c	Choked flow conditions hold when rm3 = 1.
c
	rm3 = 1.D0
	y2l = 1.D0
	y2u = (gamma + 1.D0)/2.D0
	tol = 1.D-6
	call zbrent(y2, choke, y2l, y2u, tol, ierr)	! find y2-pipe inlet
	if(ierr .ne. 0) then
	    write(6,*) 'zbrent failed for choked case'
	    stop
	endif
	rm2 = 2.D0*(y2 - 1.D0)/(gamma - 1.D0)
	rm2 = sqrt(rm2)
	ggg = rmw * gamma /rrr/t1 * y2**((gamma + 1)/(1.D0 - gamma))
	ggg = p1 * rm2 * sqrt(ggg)
	p3 = ggg * sqrt(rrr * t1 / rmw / gamma * 2.D0 /(gamma + 1.D0))

	if(p3 .le. p1 .and. p3 .ge. pa) then	! is this *really* choked?
	    eee = ggg * ao			! choked solution is valid
	    t3 = 2.D0 * t1 / (gamma + 1.D0)
	    write(6,*) '           Choked-flow estimate of Y2: ',y2
	    write(6,*) '           Choked-flow estimate of M2: ',rm2
	    write(6,*) '    Choked-flow estimate of mass rate: ',
     .			ggg*ao,' kg/s'
	    write(6,*) 'Choked-flow estimate of exit pressure: ',p3,
     .			' N/m**2'
	    goto 300				! estimate discharge density
	endif
c
c... C.  Discharge for Subcritical flow........................................
c	Subcritical flow conditions are met when P3 is ambient pressure.
c
	p3 = pa
	rm3l = 0.0D0
	rm3u = 1.0D0
	tol = 1.D-6
	call zbrent(rm3, subcrit, rm3l, rm3u, tol, ierr)  ! find rm3-pipe exit
	if(ierr .ne. 0) then
	    write(6,*) 'zbrent failed for subcritical case, ierr: ',ierr
	    stop
	endif

	if(p1 .le. p2 .or. t1 .le. t2) then	! is this solution valid?
	    write(6,*) '... increasing the value of the friction factor'
	    fff = 1.1D0 * fff
	    if(fff .gt. 0.01) then
		write(6,*) 'fff loop appears to have failed...'
		stop
	    endif
	    goto 100
	endif
	write(6,*) '           Subcritical-flow estimate of Y2: ',y2
	write(6,*) '           Subcritical-flow estimate of M2: ',rm2
	write(6,*) '    Subcritical-flow estimate of mass rate: ',ggg*ao,
     .			' kg/s'
	write(6,*) 'Subcritical-flow estimate of exit pressure: ',p3,
     .			' N/m**2'
c
c... D.  Check the Discharge temperature T3....................................
c
  300	continue
	write(6,*)'The discharge temperature is now ready to be checked!'
	rho3 = pa*rmw/rrr/t3
	write(6,*) 'Discharge density: ',rho3,' kg/m**3'
c
	stop
	end
c
c
	function choke(y2)
c
c.............................................................................
c
c	function for determining choked flow conditions
c
c.............................................................................
c
	implicit real*8(a-h,o-z), integer*4(i-n)

	common /input/ rmw, pa, p1, t1, gamma, rrr, rn, 
     .		rm2, cy2, p2, t2, rm3, y3, p3, t3, ggg
c
c... for the special case that y2=1, handle the solution analytically...
c
	if(y2 .eq. 1.D0) then
	    choke = 1.D0 - gamma
	    return
	endif
c
c... so that this routine can be used for subcritical flow as well,
c	calculate the flow parameter y3 from the passed M3 (rm3).  Of
c	course, rm3=1 for the choked case.  Also, handle the rm3=0 case
c	analytically.
c
	y3 = 1.D0 + (gamma - 1.D0)/2.D0 * rm3**2
	if(rm3 .eq. 0.D0) then
	    choke = 1.D0 + gamma*rn
	    return
	endif
c
c... This is the left-hand-side of equation (2-14)...
c
	choke = (gamma + 1.D0)/2.D0 *
     .		alog(rm3**2/y3 * (gamma - 1.D0)/2.D0 * y2/(y2 - 1.D0))
     .		- ((gamma - 1.D0)/2.D0/(y2 - 1.D0) - 1.D0/rm3**2)
     .		+ gamma * rn

	return
	end
c
c
	function subcrit(rm3)
c
c.............................................................................
c
c	function for determining subcritical-flow conditions
c
c.............................................................................
c
	implicit real*8(a-h,o-z), integer*4(i-n)

	external choke

	common /input/ rmw, pa, p1, t1, gamma, rrr, rn, 
     .		rm2, y2, p2, t2, crm3, y3, p3, t3, ggg
c
c... find y2 for this choice of M3 (rm3).  Note that the passed rm3 sets
c	the value of rm3 in common/input/.
c
	crm3 = rm3
	y2l = 1.D0
	y2u = (gamma + 1.D0)/2.D0
	tol = 1.D-6
	call zbrent2(y2, choke, y2l, y2u, tol, ierr)	! find y2-pipe inlet
	if(ierr .ne. 0) then
	    write(6,*) 'zbrent failed for choked case in subcrit'
	    stop
	endif
c
c... handle the case when y2=1 analytically.
c
	if(y2 .eq. 1.D0) then
	    p2 = p1		! for the sake of this estimate
	    y3 = 1.D0 + (gamma - 1.D0)/2.D0 * rm3**2
	    rm2 = 0.0001D0
	    p2_pa = pa/rm2 * rm3 * sqrt(y3/y2)
	    subcrit = p2_pa - p2
	    return
	endif
c
c... calculate the other parameters... The equations used are: (2-12),
c	(2-13), (2-15), (2-17), (2-19), (2-16), and (2-18).
c
	rm2 = 2.D0*(y2 - 1.D0)/(gamma - 1.D0)
	rm2 = sqrt(rm2)
	y3 = 1.D0 + (gamma - 1.D0)/2.D0 * rm3**2
	ggg = rmw * gamma /rrr/t1 * y2**((gamma + 1.D0)/(1.D0 - gamma))
	ggg = p1 * rm2 * sqrt(ggg)
	t3 = (rm3*pa/ggg)**2 * gamma*rmw/rrr
	t2 = t3 * y3/y2
	p2 = ggg/rm2 * sqrt(rrr*t2/gamma/rmw)
	p2_ = p1 * (t2/t1)**(gamma/(gamma-1.D0))
c
c... check for convergence...
c
	subcrit = p2_ - p2
	return
	end
c
c
	subroutine zbrent(answer, func, x1, x2, tol, ierr)
c
c......................................................................
c
c	Function to determine the root of FUNC which is between X1 and X2
c	from Press et al., pg 253.  Note that the SIGN function has
c	been used here to avoid underflows corrupting the programs
c	logic.
c
c	ANSWER is the returned result.
c	FUNC is the function whose zero is desired.
c	X1 and X2 constitute the search interval for the independent
c		variable.
c	TOL is the absolute tolerance for the INDEPENDENT variable.
c	IERR is an error flag (set nonzero) in case of failure:
c		1 - failed to converge after the specified number of trials
c		2 - initial interval fails to bracket the root to FUNC.
c
c....................................................................
c
	implicit real*8(a-h,o-z), integer*4(i-n)

	parameter (itmax=100, eps=3.D-8)

c
c... check the passed arguments for validity and set up the procedure.
c	Although the SIGN function traps underflows, it may not give
c	a valid comparison if FA or FB or zero.
c
	aaa = x1
	bbb = x2
	ierr = 0
	fa = func(aaa)
	fb = func(bbb)
	if(fa .eq. 0.D0) then
	   answer = aaa
	   return
	endif
	if(fb .eq. 0.D0) then
	   answer = bbb
	   return
	endif
	if(sign(1.D0,fa)*sign(1.D0,fb) .gt. 0.D0) then
	   ierr = 2
	   return
	endif

	fc = fb

	do 11 iter = 1,itmax
c
c... shuffle A,B,C and adjust the bounding interval D
c
	if(sign(1.D0,fb)*sign(1.D0,fc) .gt. 0.D0) then
	    ccc = aaa
	    fc = fa
	    ddd = bbb-aaa
	    eee = ddd
	endif

	if(abs(fc) .lt. abs(fb)) then
	    aaa = bbb
	    bbb = ccc
	    ccc = aaa
	    fa = fb
	    fb = fc
	    fc = fa
	endif
c
c... convergence check
c
	tol1 = 2.D0*eps*abs(bbb) + 0.5D0*tol
	xm = 0.5D0*(ccc-bbb)
	if(abs(xm).le. tol1 .or. fb.eq.0.D0) then
	    answer = bbb
	    return
	endif
c
c... attempt inverse quadratic interpolation
c
	if(abs(eee).ge.tol1 .and. abs(fa).gt.abs(fb)) then
	    sss = fb/fa

	    if(aaa.eq.ccc) then
		ppp = 2.D0*xm*sss
		qqq = 1.D0-sss
	    else
		qqq = fa/fc
		rrr = fb/fc
		ppp = sss*(2.D0*xm*qqq*(qqq-rrr) - (bbb-aaa)*(rrr-1.D0))
				qqq = (qqq-1.D0)*(rrr-1.D0)*(sss-1.D0)
	    endif
c
c.......... check to insure in bounds
c
	    if(ppp .gt. 0.D0) qqq = -qqq
		ppp = abs(ppp)
		if
     .  (2.D0*ppp .lt. min(3.D0*xm*qqq-abs(tol1*qqq),abs(eee*qqq))) then

c.................. interpolation valid
		    eee = ddd
		    ddd = ppp/qqq
		else

c.................. interpolation failed, use bisection
		    ddd = xm
		    eee = ddd
		endif
	    else

c.............. bounds decreasing too slowly, use bisection.
		ddd = xm
		eee = ddd
	    endif
c
c...... move last best guess to A
c
	aaa = bbb
	fa = fb
c
c... evaluate new trial root
c
	if(abs(ddd) .gt. tol1) then
		bbb = bbb+ddd
	else
		bbb = bbb+sign(tol1,xm)
	endif
	fb = func(bbb)
  11	continue
c
c... loop failed to converge the x range
c
	ierr = 1
	return
	end

c
c
	subroutine zbrent2(answer, func, x1, x2, tol, ierr)
c
c......................................................................
c
c	Function to determine the root of FUNC which is between X1 and X2
c	from Press et al., pg 253.  Note that the SIGN function has
c	been used here to avoid underflows corrupting the programs
c	logic.
c
c	ANSWER is the returned result.
c	FUNC is the function whose zero is desired.
c	X1 and X2 constitute the search interval for the independent
c		variable.
c	TOL is the absolute tolerance for the INDEPENDENT variable.
c	IERR is an error flag (set nonzero) in case of failure:
c		1 - failed to converge after the specified number of trials
c		2 - initial interval fails to bracket the root to FUNC.
c
c....................................................................
c
	implicit real*8(a-h,o-z), integer*4(i-n)

	parameter (itmax=100, eps=3.D-8)

c
c... check the passed arguments for validity and set up the procedure.
c	Although the SIGN function traps underflows, it may not give
c	a valid comparison if FA or FB or zero.
c
	aaa = x1
	bbb = x2
	ierr = 0
	fa = func(aaa)
	fb = func(bbb)
	if(fa .eq. 0.D0) then
	   answer = aaa
	   return
	endif
	if(fb .eq. 0.D0) then
	   answer = bbb
	   return
	endif
	if(sign(1.D0,fa)*sign(1.D0,fb) .gt. 0.D0) then
	   ierr = 2
	   return
	endif

	fc = fb

	do 11 iter = 1,itmax
c
c... shuffle A,B,C and adjust the bounding interval D
c
	if(sign(1.D0,fb)*sign(1.D0,fc) .gt. 0.D0) then
	    ccc = aaa
	    fc = fa
	    ddd = bbb-aaa
	    eee = ddd
	endif

	if(abs(fc) .lt. abs(fb)) then
	    aaa = bbb
	    bbb = ccc
	    ccc = aaa
	    fa = fb
	    fb = fc
	    fc = fa
	endif
c
c... convergence check
c
	tol1 = 2.D0*eps*abs(bbb) + 0.5D0*tol
	xm = 0.5D0*(ccc-bbb)
	if(abs(xm).le. tol1 .or. fb.eq.0.D0) then
	    answer = bbb
	    return
	endif
c
c... attempt inverse quadratic interpolation
c
	if(abs(eee).ge.tol1 .and. abs(fa).gt.abs(fb)) then
	    sss = fb/fa

	    if(aaa.eq.ccc) then
		ppp = 2.D0*xm*sss
		qqq = 1.D0-sss
	    else
		qqq = fa/fc
		rrr = fb/fc
		ppp = sss*(2.D0*xm*qqq*(qqq-rrr) - (bbb-aaa)*(rrr-1.D0))
				qqq = (qqq-1.D0)*(rrr-1.D0)*(sss-1.D0)
	    endif
c
c.......... check to insure in bounds
c
	    if(ppp .gt. 0.D0) qqq = -qqq
		ppp = abs(ppp)
		if
     .  (2.D0*ppp .lt. min(3.D0*xm*qqq-abs(tol1*qqq),abs(eee*qqq))) then

c.................. interpolation valid
		    eee = ddd
		    ddd = ppp/qqq
		else

c.................. interpolation failed, use bisection
		    ddd = xm
		    eee = ddd
		endif
	    else

c.............. bounds decreasing too slowly, use bisection.
		ddd = xm
		eee = ddd
	    endif
c
c...... move last best guess to A
c
	aaa = bbb
	fa = fb
c
c... evaluate new trial root
c
	if(abs(ddd) .gt. tol1) then
		bbb = bbb+ddd
	else
		bbb = bbb+sign(tol1,xm)
	endif
	fb = func(bbb)
  11	continue
c
c... loop failed to converge the x range
c
	ierr = 1
	return
	end
