library(leafpop)
library(leaflet)
library(leaflet.extras)
library(maps)
library(mapdata)
library(webshot)
library(lattice)
library(htmlwidgets)
library(pandoc)
library(htmltools)
library(ncdf4)
library(raster)

source("MET_amet.prism-lib.R")

######################################> Base map stuff

leaflet_map <- c("OpenStreetMap.Mapnik","OpenTopoMap","Esri.WorldImagery","Esri.WorldStreetMap","USGS.USTopo","USGS.USImagery","USGS.USImageryTopo")

GetURL <- function(service, host = "basemap.nationalmap.gov") {
  sprintf("https://%s/arcgis/services/%s/MapServer/WmsServer", host, service)
}

att <- paste0("<a href='https://www.usgs.gov/'>",
              "U.S. Geological Survey</a> | ",
              "<a href='https://www.usgs.gov/laws/policies_notices.html'>",
              "Policies</a>")

mapStates <- map("state",fill=T,plot=F)
my.leaf.base <- leaflet(options = leafletOptions(zoomSnap = 0.25, zoomDelta =0.25), data=mapStates) %>%
      addProviderTiles(leaflet_map[1],group="Street Map")  %>%
      addProviderTiles(leaflet_map[2],group="Topo Map") %>%
      addProviderTiles(leaflet_map[3],group="ESRI World Imagery") %>%
      addProviderTiles(leaflet_map[4],group="ESRI Street Map") %>%
      addWMSTiles(GetURL("USGSImageryOnly"),group="USGS Imagery",layers="0",attribution=att) %>%
      addWMSTiles(GetURL("USGSTopo"),group="USGS Topo",layers="0",attribution=att) %>%
      addWMSTiles(GetURL("USGSImageryTopo"),group="USGS Topo Imagery",layers="0",attribution=att)

base_Groups = c("Street Map","Topo Map","ESRI World Imagery","ESRI Street Map")

######################################>  Read in netcdf file with modeled surfaces and regrid to lat/long projection

source.open <- nc_open(filename="model.ncf", write=FALSE, readunlim=FALSE)
  model_precip_mm <- ncvar_get(source.open,"O3")
  prism_precip_mm <- ncvar_get(source.open,"PM25")
  model_lat <- ncvar_get(source.open,"LAT")
  model_lon <- ncvar_get(source.open,"LONG")
  landmask <- ncvar_get(source.open,"LWMASK")
    lm.na   <-ifelse(landmask == 0, 1, 1)
    lm.na   <-ifelse(landmask == 0, NA, 1)
    dv      <-dim(model_lat)
    dx      <- ncatt_get(source.open, varid=0, attname="XCELL" )$value

grid <- list(lat=model_lat, lon=model_lon, landmask=landmask,lm.na=lm.na,nx=dv[1],ny=dv[2],dx=dx)
dxdykm <- 40

  leaffile <- paste("prism.leaf.html",sep="")
  regrid_ll<-regrid2d_to_latlon(model_precip_mm, grid$lat, grid$lon,
                               grid_data2=prism_precip_mm, dxdykm=dxdykm, model="wrf")

  temp1 <- regrid_ll$varout #O3
  temp1[temp1 > 85] <- 84.99

  regrid_ll<-regrid2d_to_latlon(prism_precip_mm, grid$lat, grid$lon,
                               grid_data2=prism_precip_mm, dxdykm=dxdykm, model="wrf")

  temp2 <- regrid_ll$varout #PM25
  temp2[temp2 > 100] <- 99.99

  rlon     <- range(regrid_ll$grid$lon)
  rlat     <- range(regrid_ll$grid$lat)
  xy       <- cbind(as.vector(regrid_ll$grid$lon), as.vector(regrid_ll$grid$lat))

  r <- raster(ncols=regrid_ll$grid$nx, nrows=regrid_ll$grid$ny, xmn=rlon[1], xmx=rlon[2], ymn=rlat[1], ymx=rlat[2])
  # get the (last) indices
  r1 <- rasterize(xy, r)
  r2 <- rasterize(xy, r)
  values(r1) <- as.vector(temp1)
  values(r2) <- as.vector(temp2)

#    cols1 <-c('#ffffe5','#f7fcb9','#d9f0a3','#addd8e','#78c679','#41ab5d','#238443','#006837','#004529')
#    cols1 <-c('#ffffe5','#f7fcb9','#d9f0a3','#addd8e','#78c679','#41ab5d','#238443','#006837','#004529','#e7e1ef','#c994c7','#dd1c77')
#    cols1 <-c('#ffffe5','#f7fcb9','#d9f0a3','#addd8e','#78c679','#41ab5d','#238443','#006837','#004529','#2171b5','#6baed6','#bdd7e7','#eff3ff')
     cols1 <- c(grey(.8),"mediumpurple","darkorchid4", "#002FFF", "green", "yellow", "orange", "red", "brown")
my.pm.colors <- colorRampPalette(c(grey(.8),"mediumpurple","lightblue", "green", "yellow", "orange", "red", "brown"))
#    dcols1<-c('#543005','#8c510a','#bf812d','#dfc27d','#f6e8c3','#f5f5f5','#c7eae5','#80cdc1','#35978f','#01665e','#003c30')

  #obins <-c(0,10,20,30,40,50,60,70,80,90)
  omax <- max(temp1, na.rm=T)
  obins<- seq(20,85,by=5)
#  obins<-round(c(obins,omax))

  pmax  <- max(temp2, na.rm=T)
  pbins <- seq(0,100,by=5)
#  pbins<-round(c(pbins,pmax))

  modgroup   <-paste("O3 MODEL GRID")
  obsgroup   <-paste("PM25 MODEL GRID")
  transp <- 0.6
 
  pal1 <- colorBin(cols1, values(r1), na.color = "transparent",pretty=T,bins=obins)
  pal2 <- colorBin(my.pm.colors(20), values(r2), na.color = "transparent",pretty=T,bins=pbins)

  #####> Save to leafet for QA

  my.leaf<-leaflet() %>% addTiles() %>%
  addRasterImage(flip(r1,"y"), colors = pal1, opacity = transp, group=modgroup) %>%
  addRasterImage(flip(r2,"y"), colors = pal2, opacity = transp, group=obsgroup) %>%
  addLegend(pal = pal1, values = values(r1),title = "MDA8 O3 (ppb)")  %>%
  addLegend(pal = pal2, values = values(r2),title = "Daily Avg PM2.5 (ug/m3)")  %>%
  addLayersControl(overlayGroups = c(modgroup, obsgroup),
                   options = layersControlOptions(collapsed = FALSE))  %>%
  hideGroup(c(modgroup))
  saveWidget(my.leaf, file=leaffile, selfcontained=T)

####################################> Read in emissions data

a <- "emissions.2022.egus.csv"
b <- read.csv(file=a,skip=0,header=T)
head(b)
egus <- subset(b,latitude>min(model_lat) & latitude<max(model_lat) & longitude>min(model_lon) & longitude<max(model_lon) )

a <- "emissions.2022.nonegus.csv"
b <- read.csv(file=a,skip=0,header=T)
head(b)
nonegus <- subset(b,latitude>min(model_lat) & latitude<max(model_lat) & longitude>min(model_lon) & longitude<max(model_lon) )

a <- "emissions.2022.rail.csv"
b <- read.csv(file=a,skip=0,header=T)
head(b)
rail <- subset(b,latitude>min(model_lat) & latitude<max(model_lat) & longitude>min(model_lon) & longitude<max(model_lon) )

####################################> Read in fire location data

a <- "hms.txt"
b <- read.csv(file=a,skip=0,header=T)
head(b)
fires <- subset(b,Lat>min(model_lat) & Lat<max(model_lat) & Lon>min(model_lon) & Lon<max(model_lon) )
head(fires)

####################################> Read in observed/modeled pairs

a <- "dates.txt"
b <- read.csv(file=a,skip=0,header=T)
head(b)
today <- b

a <- "/work/ROMO/lrt/cmaq/36US3/postp/base2023/predobs.2023.36US3.35.base2023.airnow_dailyavg.csv"
b <- read.csv(file=a,skip=0,header=T)
d <- subset(b,SYYYY==today$YEAR & SMM==today$MONTH & SDD==today$DAY & PM25_ob >0 & PM25_mod > 0 ) 
d$diff <- as.numeric(d$PM25_mod)  - as.numeric(d$PM25_ob)
pm25 <- d

a <- "/work/ROMO/lrt/cmaq/36US3/postp/base2023/predobs.2023.36US3.35.base2023.airnow_O3.csv"
b <- read.csv(file=a,skip=5,header=T)
d <- subset(b,SYYYY==today$YEAR & SMM==today$MONTH & SDD==today$DAY & O3_8hrmax_ob >0 & O3_8hrmax_mod > 0 )
d$diff <- as.numeric(d$O3_8hrmax_mod)  - as.numeric(d$O3_8hrmax_ob)
ozone <- d
rm(b)
rm(d)

########################> Overlay information (contents)

 fires_data <-  data.frame(stat_id=fires$Satellite,lat=fires$Lat,lon=fires$Lon,Value=fires$FRP)
 egu_data <-  data.frame(stat_id=egus$facility_id,lat=egus$latitude,lon=egus$longitude,Value=egus$ann_value)
 nonegu_data <-  data.frame(stat_id=nonegus$facility_id,lat=nonegus$latitude,lon=nonegus$longitude,Value=nonegus$ann_value)
 rail_data <-  data.frame(stat_id=rail$facility_id,lat=rail$latitude,lon=rail$longitude,Value=rail$ann_value)

 sinfo_ozone_data<-data.frame(stat_id=ozone$SiteId,lat=ozone$Latitude,lon=ozone$Longitude,Obs_Val=ozone$O3_8hrmax_ob,Mod_Val=ozone$O3_8hrmax_mod,Diff_Val=ozone$diff)
 sinfo_pm_data<-data.frame(stat_id=pm25$SiteId,lat=pm25$Latitude,lon=pm25$Longitude,Obs_Val=pm25$PM25_ob,Mod_Val=pm25$PM25_mod,Diff_Val=pm25$diff) 

 units <- 'ppb'
      contents1_ozone <- paste("Site:", sinfo_ozone_data$stat_id, "<br />",
                  "Obs:", round(sinfo_ozone_data$Obs_Val, 1), units, "<br />",
                  "Model:", round(sinfo_ozone_data$Mod_Val, 1), units,"<br />",
                  "Bias:", round(sinfo_ozone_data$Diff_Val, 1), units, sep=" ")

      contents2_ozone <- paste("Site:", sinfo_ozone_data$stat_id,
                  " | Obs:", round(sinfo_ozone_data$Obs_Val, 1), units,
                  " | Model:", round(sinfo_ozone_data$Mod_Val, 1), units,
                  " | Bias:", round(sinfo_ozone_data$Diff_Val, 1), units, sep=" ")

      contents_egus <- paste("Facility:", egu_data$stat_id,
                  " | Annual total NOX emissions:", round(egu_data$Value, 1), "tpy", sep=" ")
      contents_fires <- paste("Satellite:", fires_data$stat_id,
                  " | FRP:", round(fires_data$Value, 1), "watts", sep=" ")
      contents_nonegus <- paste("Facility:", nonegu_data$stat_id,
                  " | Annual total NOX emissions:", round(nonegu_data$Value, 1), "tpy", sep=" ")
      contents_rail <- paste("Facility:", rail_data$stat_id,
                  " | Annual total NOX emissions:", round(rail_data$Value, 1), "tpy", sep=" ")

 units <- 'ug/m3'
      contents1_pm <- paste("Site:", sinfo_pm_data$stat_id, "<br />",
                  "Obs:", round(sinfo_pm_data$Obs_Val, 1), units, "<br />",
                  "Model:", round(sinfo_pm_data$Mod_Val, 1), units,"<br />",
                  "Bias:", round(sinfo_pm_data$Diff_Val, 1), units, sep=" ")

      contents2_pm <- paste("Site:", sinfo_pm_data$stat_id,
                  " | Obs:", round(sinfo_pm_data$Obs_Val, 1), units,
                  " | Model:", round(sinfo_pm_data$Mod_Val, 1), units,
                  " | Bias:", round(sinfo_pm_data$Diff_Val, 1), units, sep=" ")

#########################> Reset point colors

sinfo_ozone_data$Obs_Val[sinfo_ozone_data$Obs_Val > 85] <- 84.99
sinfo_ozone_data$Mod_Val[sinfo_ozone_data$Mod_Val > 85] <- 84.99

sinfo_pm_data$Mod_Val[sinfo_pm_data$Mod_Val > 100] <- 99.99
sinfo_pm_data$Obs_Val[sinfo_pm_data$Obs_Val > 100] <- 99.99

sinfo_ozone_data$Diff_Val[sinfo_ozone_data$Diff_Val > 30] <- 29.99
sinfo_ozone_data$Diff_Val[sinfo_ozone_data$Diff_Val < -30] <- -29.99

sinfo_pm_data$Diff_Val[sinfo_pm_data$Diff_Val > 30] <- 29.99
sinfo_pm_data$Diff_Val[sinfo_pm_data$Diff_Val < -30] <- -29.99

fires_data$Value[fires_data$Value < 0] <- 1

#########################> Colors

  omax <- max(temp1, na.rm=T)
  obins<- seq(20,85,by=5)
#  obins<-round(c(obins,85))

  pmax  <- max(temp2, na.rm=T)
  pbins <-c(0,1,2,3,4,5,6,7,8,9,10)
#  pbins<-round(c(pbins,60))

my.colors <- colorRampPalette(c(grey(.8),"mediumpurple","darkorchid4", "#002FFF","lightblue", "green", "yellow", "orange", "red", "brown"))
my.pm.colors <- colorRampPalette(c(grey(.8),"mediumpurple","lightblue", "green", "yellow", "orange", "red", "brown"))
my.diff.colors <- colorRampPalette(c("purple", "#002FFF", "deepskyblue", "lightblue","gray", "gray", "yellow", "orange", "red", "brown"))

min.data <- -30
max.data <- 30
n.bins <- 13
binpal3 <- colorBin(my.diff.colors(10), c(min.data,max.data), n.bins-1 , pretty = FALSE)

binpal4 <- colorBin(my.pm.colors(20),c(0,100),20,pretty = FALSE)
binpal5 <- colorBin(my.pm.colors(20),c(0,100),20,pretty=FALSE)
binpal2 <- colorBin(my.colors(length(obins)), c(min(obins),max(obins)), length(obins)-1 , pretty = FALSE)


#######################> Leaflet 

fill_opacity <- 0.8
plot_rad <- 10
plot_names <- c("O3 OBS","O3 MODEL","O3 BIAS","PM25 OBS","PM25 MODEL","PM25 BIAS","O3 MODEL GRID","PM25 MODEL GRID","FIRES","EGU","NONEGU","RAIL")
legend_title <- c("O3 OBS","O3 MODEL","O3 BIAS","PM25 OBS","PM25 MODEL","PM25 BIAS","O3 MODEL GRID","PM25 MODEL GRID","FIRES","EGU","NONEGU","RAIL")
main_title <- paste(today$YYYY,today$MONTH,today$DAY,sep="/")
base_Groups = c("Street Map","Topo Map","ESRI World Imagery","ESRI Street Map")

my.leaf.base <- leaflet(options = leafletOptions(zoomSnap = 0.25, zoomDelta =0.25), data=mapStates)
my.leaf <- my.leaf.base
my.leaf <- my.leaf %>% addProviderTiles("OpenStreetMap.Mapnik",group="Street Map")
my.leaf <- my.leaf %>% addProviderTiles("OpenTopoMap",group="Topo Map")
my.leaf <- my.leaf %>% addProviderTiles("Esri.WorldImagery",group="ESRI World Imagery")
my.leaf <- my.leaf %>% addProviderTiles("Esri.WorldStreetMap",group="ESRI Street Map")

my.leaf <- my.leaf %>% addRasterImage(flip(r1,"y"), colors = pal1, opacity = transp, group="O3 MODEL GRID")
my.leaf <- my.leaf %>% addRasterImage(flip(r2,"y"), colors = pal2, opacity = transp, group="PM25 MODEL GRID")

my.leaf <- my.leaf %>% addCircleMarkers(sinfo_ozone_data$lon,sinfo_ozone_data$lat,color="black",fillColor=~binpal2(sinfo_ozone_data$Obs_Val),group="O3 OBS",radius=plot_rad,data=sinfo_ozone_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_ozone, label=contents2_ozone, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(sinfo_ozone_data$lon,sinfo_ozone_data$lat,color="black",fillColor=~binpal2(sinfo_ozone_data$Mod_Val),group="O3 MODEL",radius=plot_rad,data=sinfo_ozone_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_ozone, label=contents2_ozone, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(sinfo_ozone_data$lon,sinfo_ozone_data$lat,color="black",fillColor=~binpal3(sinfo_ozone_data$Diff_Val),group="O3 BIAS",radius=plot_rad,data=sinfo_ozone_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_ozone, label=contents2_ozone, labelOptions = labelOptions(noHide = F, textsize = "15px"))

my.leaf <- my.leaf %>% addCircleMarkers(sinfo_pm_data$lon,sinfo_pm_data$lat,color="black",fillColor=~binpal5(sinfo_pm_data$Obs_Val),group="PM25 OBS",radius=plot_rad,data=sinfo_pm_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_pm, label=contents2_pm, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(sinfo_pm_data$lon,sinfo_pm_data$lat,color="black",fillColor=~binpal4(sinfo_pm_data$Mod_Val),group="PM25 MODEL",radius=plot_rad,data=sinfo_pm_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_pm, label=contents2_pm, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(sinfo_pm_data$lon,sinfo_pm_data$lat,color="black",fillColor=~binpal3(sinfo_pm_data$Diff_Val),group="PM25 BIAS",radius=plot_rad,data=sinfo_pm_data,opacity=1,fillOpacity=fill_opacity,stroke=TRUE,weight=1,popup=contents1_pm, label=contents2_pm, labelOptions = labelOptions(noHide = F, textsize = "15px"))

my.leaf <- my.leaf %>% addCircleMarkers(egu_data$lon,egu_data$lat,color="black",fillColor="darkgray",group="EGU",radius=(3*(egu_data$Value/1000.))+3,data=egu_data,opacity=1,fillOpacity=0.1,stroke=TRUE,weight=1,popup=contents_egus, label=contents_egus, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(fires_data$lon,fires_data$lat,color="red",fillColor="red",group="FIRES",radius=(fires_data$Value/100.)+3,data=fires_data,opacity=1,fillOpacity=0.1,stroke=TRUE,weight=1,popup=contents_fires, label=contents_fires, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(nonegu_data$lon,nonegu_data$lat,color="darkblue",fillColor="darkblue",group="NONEGU",radius=(3*(nonegu_data$Value/1000.))+3,data=nonegu_data,opacity=1,fillOpacity=0.1,stroke=TRUE,weight=1,popup=contents_nonegus, label=contents_nonegus, labelOptions = labelOptions(noHide = F, textsize = "15px"))
my.leaf <- my.leaf %>% addCircleMarkers(rail_data$lon,rail_data$lat,color="purple",fillColor="purple",group="RAIL",radius=(3*(rail_data$Value/1000.))+3,data=rail_data,opacity=1,fillOpacity=0.1,stroke=TRUE,weight=1,popup=contents_rail, label=contents_rail, labelOptions = labelOptions(noHide = F, textsize = "15px"))

my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal2, values =~sinfo_ozone_data$Obs_Val, group=plot_names[1], layerId=plot_names[1], title = legend_title[1], opacity = 2) 
my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal2, values =~sinfo_ozone_data$Mod_Val, group=plot_names[2], layerId=plot_names[2], title = legend_title[2], opacity = 2)
my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal3, values = c(-30,30), group=plot_names[3], layerId=plot_names[3], title = legend_title[3], opacity = 2)
my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal5, values =~sinfo_pm_data$Obs_Val, group=plot_names[4], layerId=plot_names[4], title = legend_title[4], opacity = 2)
my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal4, values =~sinfo_pm_data$Mod_Val, group=plot_names[5], layerId=plot_names[5], title = legend_title[5], opacity = 2)
my.leaf <- my.leaf %>% addLegend("bottomright", pal = binpal3, values = c(-30,30), group=plot_names[6], layerId=plot_names[6], title = legend_title[6], opacity = 2)

my.leaf <- my.leaf %>% addLegend("bottomright", pal = pal1, values = values(r1), group=plot_names[7], layerId=plot_names[7], title = legend_title[7], opacity = 2)
my.leaf <- my.leaf %>% addLegend("bottomright", pal = pal2, values = values(r2), group=plot_names[8], layerId=plot_names[8], title = legend_title[8], opacity = 2)

my.leaf <- my.leaf %>% addControl(main_title,position="topleft",className="map-title")
my.leaf <- my.leaf %>% addLayersControl(baseGroups = base_Groups,overlayGroups = plot_names, options =  layersControlOptions(collapsed = FALSE,position="topleft"))
my.leaf <- my.leaf %>% hideGroup(c("O3 MODEL","O3 BIAS","PM25 OBS","PM25 MODEL","PM25 BIAS","EGU","O3 MODEL GRID","PM25 MODEL GRID","FIRES","NONEGU","RAIL"))

saveWidget(my.leaf, file='test.html',selfcontained=T)

