      PROGRAM m3conc2aot
       
      !-------------------------------------------------------------------------------------------------
      ! Descriptions: To calculate aerosol optical thickness using cctm output
      !
      ! Preconditions required:
      !
      ! Functions and subroutines called:
      !
      ! Revision history:
      !    Created Mar 27, 2008 D. Yin (ARB/CalEPA)
      !    Revised Jan 29, 2009 D. Yin to add proper values to termination index xstat
      ! 
      ! References:
      !    Malm, W.C., Sisler, J.F., Huffman, D., Eldred, R.A., and Cahill, T.A. 
      !       Spatial and seasonal trends in particle concentration and optical extinction in 
      !          the United States, JGR, 99(D1), 1347-1370, 1994.
      !    Binkowski, F.S., and S.J. Roselle, Models-3 Community Multiscale Air Quality (CMAQ) 
      !       model aerosol component, 1. Model description, JGR, 108(D6), 4183, 2003.
      !    Mebust, M.R., B.K. Eder, F.S. Binkowski, and S.J. Roselle, Models-3 Community Multiscale
      !       Air Quality (CMAQ) model aerosol component, 2. Model evaluation, JGR, 108(D6), 4184, 2003.
      !    Roy, B., R. Mathur, A.B. Gilliland, and S.C. Howard (2007), A comparison of CMAQ-based aerosol
      !       properties with IMPROVE, MODIS, and AERONET data, JGR, 112, D14301.
      !--------------------------------------------------------------------------------------------------

      IMPLICIT none

      !includes 

      INCLUDE 'PARMS3.EXT'      ! I/O API constants
      INCLUDE 'FDESC3.EXT'      ! I/O API file description data structure
      INCLUDE 'IODECL3.EXT'     ! I/O API function declarations

      !parameters 
      CHARACTER(LEN=16), PARAMETER :: met_cro_3d='met_cro_3d' !met data file from mcip
      CHARACTER(LEN=16), PARAMETER :: m3_conc='m3_conc'       !aq data from cctm
      CHARACTER(LEN=16), PARAMETER :: aot_output='aot_output' !aot 
      CHARACTER(LEN=16), PARAMETER :: pname='m3conc2aot'      !caller

      REAL, PARAMETER :: rh_in (98) = (/1.,2.,3.,4.,5.,6.,7.,8.,9.,10.,
     &11.,12.,13.,14.,15.,16.,17.,18.,19.,20.,
     &21.,22.,23.,24.,25.,26.,27.,28.,29.,30.,
     &31.,32.,33.,34.,35.,36.,37.,38.,39.,40.,
     &41.,42.,43.,44.,45.,46.,47.,48.,49.,50.,
     &51.,52.,53.,54.,55.,56.,57.,58.,59.,60.,
     &61.,62.,63.,64.,65.,66.,67.,68.,69.,70.,
     &71.,72.,73.,74.,75.,76.,77.,78.,79.,80.,
     &81.,82.,83.,84.,85.,86.,87.,88.,89.,90.,
     &91.,92.,93.,94.,95.,96.,97.,98./)

      REAL, PARAMETER :: frh_out(98) =(/1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,
     &1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,
     &1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,
     &1.,1.,1.,1.,1.,1.,1.02,1.04,1.06,1.08,
     &1.1,1.13,1.15,1.18,1.2,1.23,1.26,1.28,1.31,1.34,
     &1.37,1.41,1.44,1.47,1.51,1.54,1.58,1.62,1.66,1.7,
     &1.74,1.79,1.83,1.88,1.93,1.98,2.03,2.08,2.14,2.19,
     &2.25,2.31,2.37,2.43,2.5,2.56,2.63,2.7,2.78,2.86,
     &2.94,3.03,3.12,3.22,3.33,3.45,3.58,3.74,3.93,4.16,
     &4.45,4.84,5.37,6.16,7.4,9.59,14.1,26.4/)

      !externals
      INTEGER, EXTERNAL :: ENVINT

      !local variables 
      INTEGER :: jdate          !current model date, YYYYDDD
      INTEGER :: jtime          !current model time, HHMMSS
      INTEGER :: nstep          !total number of steps of data processing

      INTEGER :: strtcol, endcol !start and end column
      INTEGER :: strtrow, endrow !start and end row 
      INTEGER :: strtlvl, endlvl !start and end level

      INTEGER :: xstat 

      INTEGER :: loop_index1, loop_index2, loop_index3
      INTEGER :: loop_master

      INTEGER :: ncols_gd
      INTEGER :: nrows_gd
      INTEGER :: nlays_gd
      INTEGER :: gdtyp_gd
      INTEGER :: tstep_gd         ! time step, HHMMSS


      CHARACTER(LEN=256) :: xmsg 

      REAL :: xcell_gd
      REAL :: ycell_gd
      REAL :: xorig_gd
      REAL :: yorig_gd

      REAL :: frh_coe                                         !return of frh 
      REAL :: deltaz 

      REAL, ALLOCATABLE :: ta(:,:,:)                          !air temperature in K
      REAL, ALLOCATABLE :: qv(:,:,:)                          !water vapor mixing ratio in kg/kg
      REAL, ALLOCATABLE :: pres(:,:,:)                        !prssure in pa
      REAL, ALLOCATABLE :: zf(:,:,:)                          !full-level height above ground in m
      REAL, ALLOCATABLE :: rh(:,:,:)                          !relative humidity in % 
      REAL, ALLOCATABLE :: aso4j(:,:,:)                       !j-mode SO4 in microg/m3 
      REAL, ALLOCATABLE :: aso4i(:,:,:)                       !i-mode SO4 in microg/m3 
      REAL, ALLOCATABLE :: anh4j(:,:,:)                       !j-mode NH4 in microg/m3 
      REAL, ALLOCATABLE :: anh4i(:,:,:)                       !i-mode NH4 in microg/m3 
      REAL, ALLOCATABLE :: ano3j(:,:,:)                       !j-mode NO3 in microg/m3 
      REAL, ALLOCATABLE :: ano3i(:,:,:)                       !i-mode NO3 in microg/m3 
      REAL, ALLOCATABLE :: aorgaj(:,:,:)                      !j-mode ORGA in microg/m3 
      REAL, ALLOCATABLE :: aorgai(:,:,:)                      !i-mode ORGA in microg/m3 
      REAL, ALLOCATABLE :: aorgpaj(:,:,:)                     !j-mode ORGPA in microg/m3 
      REAL, ALLOCATABLE :: aorgpai(:,:,:)                     !i-mode ORGPA in microg/m3 
      REAL, ALLOCATABLE :: aorgbj(:,:,:)                      !j-mode ORGB in microg/m3 
      REAL, ALLOCATABLE :: aorgbi(:,:,:)                      !i-mode ORGB in microg/m3 
      REAL, ALLOCATABLE :: aecj(:,:,:)                        !j-mode EC in microg/m3 
      REAL, ALLOCATABLE :: aeci(:,:,:)                        !i-mode EC in microg/m3 
      REAL, ALLOCATABLE :: a25j(:,:,:)                        !j-mode 25 in microg/m3 
      REAL, ALLOCATABLE :: acors(:,:,:)                       !c-mode other than soil in microg/m3 
      REAL, ALLOCATABLE :: asoil(:,:,:)                       !c-mode soil in microg/m3 
      REAL, ALLOCATABLE :: anaj(:,:,:)                        !j-mode sodium in microg/m3 
      REAL, ALLOCATABLE :: anak(:,:,:)                        !c-mode sodium in microg/m3 
      REAL, ALLOCATABLE :: aclj(:,:,:)                        !j-mode chloride in microg/m3 
      REAL, ALLOCATABLE :: aclk(:,:,:)                        !c-mode chloride in microg/m3 
      REAL, ALLOCATABLE :: aso4k(:,:,:)                       !c-mode SO4 in microg/m3 
      REAL, ALLOCATABLE :: extinction(:,:,:)                  !extinction in 1/m
      REAL, ALLOCATABLE :: aot(:,:)                           !calculated aerosol optical thickness, dimensionless

      !start of the executable code

      jdate=ENVINT('start_date','Calculation start date',9999999,xstat)
      IF ( xstat.NE.0) THEN
         PRINT *, 'Error in getting start date!'
         STOP
      ENDIF

      jtime=ENVINT('start_time','Calculation start time',999999,xstat)
      IF ( xstat.NE.0) THEN
         PRINT *, 'Error in getting start time!'
         STOP
      ENDIF

      nstep=ENVINT('total_step','Calculation total steps',999,xstat)
      IF ( xstat.NE.0) THEN
         PRINT *, 'Error in getting total steps for calculation!'
         STOP
      ENDIF

      !open met file 
      IF ( .NOT. OPEN3( met_cro_3d, fsread3, pname ) ) THEN
         XMSG = 'Could not open ' // met_cro_3d // ' file'
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      IF ( .NOT. DESC3( met_cro_3d ) ) THEN
         xmsg ='Could not get '//met_cro_3d//' file description'
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      xcell_gd=xcell3d
      ycell_gd=ycell3d
      ncols_gd=ncols3d
      nrows_gd=nrows3d
      nlays_gd=nlays3d
      gdtyp_gd=gdtyp3d
      xorig_gd=xorig3d
      yorig_gd=yorig3d
      tstep_gd=tstep3d

      !open conc file
      IF ( .NOT. OPEN3( m3_conc, fsread3, pname ) ) THEN
         XMSG = 'Could not open ' // m3_conc // ' file'
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      IF ( .NOT. DESC3( m3_conc ) ) THEN
         xmsg ='Could not get '//m3_conc//' file description'
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      !check consistency between the above two files 

      IF ( xcell_gd.NE.xcell3d .OR.
     &     ycell_gd.NE.ycell3d .OR.
     &     ncols_gd.NE.ncols3d .OR.
     &     nrows_gd.NE.nrows3d .OR.
     &     nlays_gd.NE.nlays3d .OR.
     &     gdtyp_gd.NE.gdtyp3d .OR.
     &     xorig_gd.NE.xorig3d .OR.
     &     yorig_gd.NE.yorig3d .OR.  
     &     tstep_gd.NE.tstep3d ) THEN
         xmsg ='Find inconsistency between met and conc file'
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      !allocate memory
      ALLOCATE ( ta(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( qv(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( pres(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( zf(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( rh(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aso4j(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aso4i(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( anh4j(ncols_gd,nrows_gd,nlays_gd) )
      ALLOCATE ( anh4i(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( ano3j(ncols_gd,nrows_gd,nlays_gd) )
      ALLOCATE ( ano3i(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aorgaj(ncols_gd,nrows_gd,nlays_gd) )
      ALLOCATE ( aorgai(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aorgpaj(ncols_gd,nrows_gd,nlays_gd) )
      ALLOCATE ( aorgpai(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aorgbj(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aorgbi(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aecj(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aeci(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( a25j(ncols_gd,nrows_gd,nlays_gd) )
      ALLOCATE ( acors(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( asoil(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( anaj(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( anak(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aclj(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aclk(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aso4k(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( extinction(ncols_gd,nrows_gd,nlays_gd) ) 
      ALLOCATE ( aot(ncols_gd,nrows_gd) ) 
     
      strtcol=1
      endcol=ncols_gd
      strtrow=1
      endrow=nrows_gd
      strtlvl=1
      endlvl=nlays_gd

      !open aot output file
      sdate3d=jdate
      stime3d=jtime
      nvars3d=1
      nlays3d=1
      vtype3d(1) = m3real
      vname3d(1)='AOT'
      units3d(1)='DIMENSIONLESS'
      vdesc3d(1)='Aerosol optical thickness'
      fdesc3d=' '
      fdesc3d(1)='m3conc2aot output'

      IF ( .NOT. OPEN3( aot_output, fsunkn3, pname ) ) THEN
         XMSG = 'Could not open ' // aot_output // ' file'
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF

      master_loop: DO loop_master=1, nstep

      !print *,'jdate=',jdate,jtime

      IF ( .NOT. INTERPX ( met_cro_3d, 'TA', pname,
     &                       strtcol,endcol, strtrow, endrow, 
     &                       strtlvl,endlvl,jdate, jtime, ta ) ) THEN
         xmsg = 'Could not read ' //'TA'// ' from ' // met_cro_3d
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( met_cro_3d, 'QV', pname,
     &                       strtcol,endcol, strtrow, endrow, 
     &                       strtlvl,endlvl,jdate, jtime, qv ) ) THEN
         xmsg = 'Could not read ' //'QV'// ' from ' // met_cro_3d
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( met_cro_3d, 'PRES', pname,
     &                       strtcol,endcol, strtrow, endrow, 
     &                       strtlvl,endlvl,jdate, jtime, pres ) ) THEN
         xmsg = 'Could not read ' //'PRES'// ' from ' // met_cro_3d
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( met_cro_3d, 'ZF', pname,
     &                       strtcol,endcol, strtrow, endrow, 
     &                       strtlvl,endlvl,jdate, jtime, zf ) ) THEN
         xmsg = 'Could not read ' //'PRES'// ' from ' // met_cro_3d
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      DO loop_index3=strtlvl,endlvl
         DO loop_index2=strtrow,endrow
            DO loop_index1=strtcol,endcol
               CALL tpr2rh (ta(loop_index1,loop_index2,loop_index3) 
     &            , pres(loop_index1,loop_index2,loop_index3) 
     &            , qv(loop_index1,loop_index2,loop_index3) 
     &            , rh(loop_index1,loop_index2,loop_index3))
            ENDDO
         ENDDO
      ENDDO

      IF ( .NOT. INTERPX ( m3_conc, 'ASO4J', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aso4j ) ) THEN
         xmsg = 'Could not read ' //'ASO4J'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ASO4I', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aso4i ) ) THEN
         xmsg = 'Could not read ' //'ASO4I'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ANH4J', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, anh4j ) ) THEN
         xmsg = 'Could not read ' //'ANH4J'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ANH4I', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, anh4i ) ) THEN
         xmsg = 'Could not read ' //'ANH4I'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ANO3J', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, ano3j ) ) THEN
         xmsg = 'Could not read ' //'ANO3J'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ANO3I', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, ano3i ) ) THEN
         xmsg = 'Could not read ' //'ANO3I'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGAJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgaj ))THEN
         xmsg = 'Could not read ' //'AORGAJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGAI', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgai ))THEN
         xmsg = 'Could not read ' //'AORGAI'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGPAJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgpaj ))THEN
         xmsg = 'Could not read ' //'AORGPAJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGPAI', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgpai ))THEN
         xmsg = 'Could not read ' //'AORGPAI'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGBJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgbj ))THEN
         xmsg = 'Could not read ' //'AORGBJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AORGBI', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aorgbi))THEN
         xmsg = 'Could not read ' //'AORGBI'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AECJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aecj ))THEN
         xmsg = 'Could not read ' //'AECJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'AECI', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aeci ))THEN
         xmsg = 'Could not read ' //'AECI'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'A25J', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, a25j ))THEN
         xmsg = 'Could not read ' //'A25J'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ACORS', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, acors ))THEN
         xmsg = 'Could not read ' //'ACORS'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ASOIL', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, asoil ))THEN
         xmsg = 'Could not read ' //'ASOIL'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF
 
      IF ( .NOT. INTERPX ( m3_conc, 'ANAJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, anaj ))THEN
         xmsg = 'Could not read ' //'ANAJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ANAK', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, anak ))THEN
         xmsg = 'Could not read ' //'ANAK'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ACLJ', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aclj ))THEN
         xmsg = 'Could not read ' //'ACLJ'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ACLK', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aclk ))THEN
         xmsg = 'Could not read ' //'ACLK'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      IF ( .NOT. INTERPX ( m3_conc, 'ASO4K', pname,
     &                       strtcol,endcol, strtrow, endrow,
     &                       strtlvl,endlvl,jdate, jtime, aso4k ))THEN
         xmsg = 'Could not read ' //'ASO4K'// ' from ' // m3_conc
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      END IF

      !print *, ta(10,10,1)
      !print *, qv(10,10,1)
      !print *, pres(10,10,1)
      !print *, zf(10,10,1)
      !print *, rh(10,10,1)
      !print *, aso4j(10,10,1)
      !print *, aso4i(10,10,1)
      !print *, anh4i(10,10,1)
      !print *, anh4j(10,10,1)
      !print *, ano3j(10,10,1)
      !print *, ano3i(10,10,1)
      !print *, aorgaj(10,10,1)
      !print *, aorgai(10,10,1)
      !print *, aorgpai(10,10,1)
      !print *, aorgpaj(10,10,1)
      !print *, aorgbj(10,10,1)
      !print *, aorgbi(10,10,1)
      !print *, aecj(10,10,1)
      !print *, aeci(10,10,1)
      !print *, a25j(10,10,1)
      !print *, acors(10,10,1)
      !print *, asoil(10,10,1)

      DO loop_index3=strtlvl,endlvl
         DO loop_index2=strtrow,endrow
            DO loop_index1=strtcol,endcol
               CALL interp_linear (rh_in, frh_out, 98, 
     & MIN(rh(loop_index1,loop_index2,loop_index3),98.), frh_coe )
         !print *,rh(loop_index1,loop_index2,loop_index3),frh_coe
               extinction(loop_index1,loop_index2,loop_index3)=
     &0.003*frh_coe*(aso4j(loop_index1,loop_index2,loop_index3)/1000.+  !microg/m3 to mg/m3
     &               aso4i(loop_index1,loop_index2,loop_index3)/1000.+
     &               anh4j(loop_index1,loop_index2,loop_index3)/1000.+
     &               anh4i(loop_index1,loop_index2,loop_index3)/1000.+
     &               ano3j(loop_index1,loop_index2,loop_index3)/1000.+
     &               ano3i(loop_index1,loop_index2,loop_index3)/1000.+
     &               anaj(loop_index1,loop_index2,loop_index3)/1000.+
     &               anak(loop_index1,loop_index2,loop_index3)/1000.+
     &               aclj(loop_index1,loop_index2,loop_index3)/1000.+
     &               aclk(loop_index1,loop_index2,loop_index3)/1000.+
     &               aso4k(loop_index1,loop_index2,loop_index3)/1000.)+
     &0.004*        (aorgaj(loop_index1,loop_index2,loop_index3)/1000.+
     &               aorgai(loop_index1,loop_index2,loop_index3)/1000.+
     &               aorgpaj(loop_index1,loop_index2,loop_index3)/1000.+
     &               aorgpai(loop_index1,loop_index2,loop_index3)/1000.+
     &               aorgbj(loop_index1,loop_index2,loop_index3)/1000.+
     &               aorgbi(loop_index1,loop_index2,loop_index3)/1000.)+
     &0.001*        (a25j(loop_index1,loop_index2,loop_index3)/1000.+
     &               asoil(loop_index1,loop_index2,loop_index3)/1000.)+
     &0.0006*       (acors(loop_index1,loop_index2,loop_index3)/1000.)+
     &0.01*         (aecj(loop_index1,loop_index2,loop_index3)/1000.+
     &               aeci(loop_index1,loop_index2,loop_index3)/1000.)
         !if (extinction(loop_index1,loop_index2,loop_index3).lt.0.) then
         !print *,extinction(loop_index1,loop_index2,loop_index3),frh_coe
         !print *,aso4j(loop_index1,loop_index2,loop_index3)
         !print *,      aso4i(loop_index1,loop_index2,loop_index3)
         !print *,           anh4j(loop_index1,loop_index2,loop_index3)
         !print*,           anh4i(loop_index1,loop_index2,loop_index3)
         !print*,           ano3j(loop_index1,loop_index2,loop_index3)
         !print*,           ano3i(loop_index1,loop_index2,loop_index3)
         !print*,    aorgaj(loop_index1,loop_index2,loop_index3)
         !print*,            aorgai(loop_index1,loop_index2,loop_index3)
         !print*,            aorgpaj(loop_index1,loop_index2,loop_index3)
         !print*,            aorgpai(loop_index1,loop_index2,loop_index3)
         !print*,            aorgbj(loop_index1,loop_index2,loop_index3)
         !print*,            aorgbi(loop_index1,loop_index2,loop_index3)
         !print*,     a25j(loop_index1,loop_index2,loop_index3)
         !print*,   acors(loop_index1,loop_index2,loop_index3)
         !print*,           asoil(loop_index1,loop_index2,loop_index3)
         !print*,      aecj(loop_index1,loop_index2,loop_index3)
         !print*,            aeci(loop_index1,loop_index2,loop_index3)
          !      pause
          !endif
            ENDDO
         ENDDO
      ENDDO
 
      aot=0.0 
  
      DO loop_index2=strtrow,endrow
         DO loop_index1=strtcol,endcol
            DO loop_index3=strtlvl,endlvl

              IF (loop_index3.EQ.strtlvl) THEN
                 deltaz=zf(loop_index1,loop_index2,loop_index3)
              ELSE
                 deltaz=zf(loop_index1,loop_index2,loop_index3)-
     &                  zf(loop_index1,loop_index2,loop_index3-1)
              ENDIF

              aot(loop_index1,loop_index2)=aot(loop_index1,loop_index2)+
     &            deltaz*
     &            extinction(loop_index1,loop_index2,loop_index3)

         !print *,'loop_index1=',loop_index1,'loop_index2=',loop_index2
         !print *,extinction(loop_index1,loop_index2,loop_index3)
         !print *,aot(loop_index1,loop_index2)
         !print *,zf(loop_index1,loop_index2,loop_index3)
         !print *,deltaz
            ENDDO
         !print *,'loop_index1=',loop_index1,'loop_index2=',loop_index2
         !print *,aot(loop_index1,loop_index2)
         !pause
         ENDDO
      ENDDO

      print *,'jdate=',jdate,jtime
      !writing
      IF ( .NOT. WRITE3( aot_output, 'AOT', jdate, jtime, aot))THEN
         XMSG = 'Could not write to ' // aot_output // ' file'
         xstat=1  ! I/O related error
         CALL M3EXIT( pname, jdate, jtime, xmsg, xstat )
      ENDIF
      
      print *,maxval(aot),'max'
      print *,minval(aot),'min'

      CALL NEXTIME(jdate,jtime,tstep_gd)

      ENDDO master_loop
      
      STOP
      END 
