      PROGRAM vert_xsect_data

      !-------------------------------------------------------------------------------------------------
      ! Descriptions: To create I/O API data file for a vertical cross section along a series pair of,
      !               lat & lon, or a straight x/y cut
      !
      ! Preconditions required:
      !
      ! Functions and subroutines called:
      !
      ! Revision history:
      !    Created Jun 04, 2008 D. Yin (ARB/CalEPA)
      !
      !--------------------------------------------------------------------------------------------------

      IMPLICIT none

      !includes

      INCLUDE 'PARMS3.EXT'      ! I/O API constants
      INCLUDE 'FDESC3.EXT'      ! I/O API file description data structure
      INCLUDE 'IODECL3.EXT'     ! I/O API function declarations

      !parameters
      CHARACTER(LEN=16), PARAMETER :: m3_conc='m3_conc'       !aq data from cctm
      CHARACTER(LEN=16), PARAMETER :: metcro_3d='metcro_3d'   !met data from mcip
      CHARACTER(LEN=16), PARAMETER :: grdcro_2d='grdcro_2d'   !met data from mcip
      CHARACTER(LEN=16), PARAMETER :: xsect_data='xsect_data' !vertical cross section data
      CHARACTER(LEN=16), PARAMETER :: pname='vert_xsect_data' !caller

      INTEGER, PARAMETER :: num_target_layer=50
      REAL :: target_layer(50)              !output vertical height ABS, in m 

      !externals and their descriptions 
      INTEGER, EXTERNAL :: GETEFILE 
      INTEGER, EXTERNAL :: GETFLINE
      INTEGER, EXTERNAL :: ENVINT  

      !local variables and their descriptions

      CHARACTER(LEN=20) :: gname                              !grid name
      CHARACTER(LEN=120) :: iname                             !input file name

      REAL, ALLOCATABLE :: lat ( : )                          !path lat in degree
      REAL, ALLOCATABLE :: lon ( : )                          !path lon in degree
      REAL, ALLOCATABLE :: x_cord ( : )                       !x_cord 
      REAL, ALLOCATABLE :: y_cord ( : )                       !y_cord 
      REAL, ALLOCATABLE :: conc_dummy ( : , : , : )           !cross section data array 
      REAL, ALLOCATABLE :: zh ( : , : , : )                   !cross section data array 
      REAL, ALLOCATABLE :: dummy ( : , : , : )                !dummy
      REAL, ALLOCATABLE :: target_data ( : , : , :)           !target data
      REAL, ALLOCATABLE :: tr ( : )                           ! terrain height in m
      REAL :: ht

      INTEGER :: xsect_type                                   !vertical cross section type,1:y crosssection
                                                              ! 2: x cross section, 3: random, needs lat-lon pairs
      INTEGER :: y_loc                                        ! row location for y cross section
      INTEGER :: x_loc                                        ! col location for x cross section

      INTEGER :: ncols, nrows                                 !col and row number of the domain
      INTEGER :: dev_in                                       !device of input lat, lon file
      INTEGER :: status 
      INTEGER :: ndim                                         !number of lat and lon pairs
      INTEGER :: loop, loop2
      INTEGER :: step, v, strtcol, endcol, strtrow, endrow 
      INTEGER :: jdate, jtime 
      INTEGER :: nlays

      CHARACTER(LEN=256) :: xmsg                              !message

      !start of the executable code

      xsect_type=ENVINT('xsect_type','cross section type',9999,status)
      IF ( status.NE.0) THEN
         PRINT *, 'Error in getting cross section type!'
         STOP
      ENDIF

      IF ( xsect_type.EQ.1) THEN
         y_loc=ENVINT('y_loc','row location',9999,status)
         IF ( status.NE.0) THEN
            PRINT *, 'Error in getting y_loc!'
            STOP
         ENDIF
      ELSEIF ( xsect_type.EQ.2 ) THEN
         x_loc=ENVINT('x_loc','row location',9999,status)
         IF ( status.NE.0) THEN
            PRINT *, 'Error in getting x_lco!'
            STOP
         ENDIF
      ELSEIF ( xsect_type.EQ.3) THEN
         CALL ENVSTR ( 'gname','name of grid','who knows',gname,status)

         IF ( status.NE.0) THEN
            PRINT *, 'Error in getting grid name!'
            STOP
         ENDIF

         CALL ENVSTR('input_data','input data file',' ',iname, status)
         IF ( status.NE.0) THEN
            PRINT *, 'Error in getting input data file name!'
            STOP
         ENDIF

         dev_in=GETEFILE( iname,.TRUE., .TRUE., pname)
         IF (dev_in.LT.0) THEN
            PRINT *,'ERROR in opening input data file!'
            STOP
         ENDIF

         ndim=GETFLINE ( dev_in, 
     &       'check number of lat-lon pairs of the flight path' )

         !allocate memory
         ALLOCATE ( lon ( ndim ) ) 
         ALLOCATE ( lat ( ndim ) ) 

         DO loop=1,ndim
            READ(dev_in,*) lon(loop),lat(loop)
         ENDDO

         CLOSE (dev_in) 

      ELSE
         PRINT *, xsect_type, ' is a wrong cross section type!'
         STOP
      ENDIF

      !open m3conc file
      IF ( .NOT. OPEN3( m3_conc, fsread3, pname ) ) THEN
         XMSG = 'Could not open ' // m3_conc // ' file'
         CALL M3EXIT( pname, 0, 0, xmsg, status )
      ENDIF

      IF ( .NOT. DESC3( m3_conc ) ) THEN
         xmsg ='Could not get '//m3_conc//' file description'
         CALL M3EXIT( pname, 0, 0, xmsg, status )
      ENDIF

      nlays=NLAYS3D

      IF (xsect_type.EQ.1) THEN
         ndim=NCOLS3D
      ELSEIF (xsect_type.EQ.2) THEN
         ndim=NROWS3D
      ENDIF

      !allocate memory
      ALLOCATE ( conc_dummy ( ndim,1,nlays3d ) )
      ALLOCATE ( zh ( ndim,1,nlays3d ) )
      ALLOCATE ( dummy ( 1,1,nlays3d ) )
      ALLOCATE ( tr ( ndim ) ) 
      ALLOCATE ( target_data ( ndim, 1, num_target_layer ) )
      ALLOCATE ( x_cord ( ndim) ) 
      ALLOCATE ( y_cord ( ndim) ) 

      IF (xsect_type.EQ.1) THEN
         DO loop =1,ndim
            x_cord(loop)=loop
            y_cord(loop)=y_loc
         ENDDO
      ELSEIF (xsect_type.EQ.2) THEN
         DO loop =1,ndim
            x_cord(loop)=x_loc
            y_cord(loop)=loop
         ENDDO
      ELSEIF (xsect_type.EQ.3) THEN
         DO loop=1,ndim
            CALL latlon2xy(lon(loop),lat(loop),gname,ncols,nrows,
     &                 x_cord(loop),y_cord(loop))
            x_cord(loop)=x_cord(loop)-0.5
            y_cord(loop)=y_cord(loop)-0.5       !dot to cross

            !print *,x_cord(loop), y_cord(loop), ncols,nrows
             IF( x_cord(loop).LT.1 .OR. x_cord(loop).GT.ncols .OR.
     &           y_cord(loop).LT.1 .OR. y_cord(loop).GT.nrows ) THEN
                 PRINT *, 'The point of ', loop, 'is out of domain!'
                 STOP
             ENDIF
         ENDDO
      ENDIF

      !open output file 

      NCOLS3D = ndim
      NROWS3D = 1
      NLAYS3D=num_target_layer
      VGTOP3D = BADVAL3  
      DO  loop = 1, NLAYS3D
          VGLVS3D( loop ) = FLOAT(loop)*100.
      ENDDO
      DO  loop = NLAYS3D+1, MXLAYS3
          VGLVS3D( loop ) = BADVAL3
      ENDDO

      fdesc3d=' '

      IF ( .NOT. OPEN3( xsect_data, fsunkn3, pname ) ) THEN
         XMSG = 'Could not open ' // xsect_data // ' file'
         CALL M3EXIT( pname, 0, 0, xmsg, status )
      ENDIF

      JDATE = SDATE3D
      JTIME = STIME3D

      DO loop=1,ndim

         strtcol=NINT(x_cord(loop))
         endcol=strtcol
         strtrow=NINT(y_cord(loop))
         endrow=strtrow

         IF ( .NOT. INTERPX( metcro_3d, 'ZH', pname,
     &      strtcol,endcol, strtrow,endrow, 1,nlays,
     &      jdate, jtime, dummy ) ) THEN
            xmsg = 'Could not read ZH from '
     &               // metcro_3d
            CALL M3EXIT ( pname, jdate, jtime, xmsg, status )
         ENDIF

         IF ( .NOT. INTERPX( grdcro_2d, 'HT', pname,
     &      strtcol,endcol, strtrow,endrow, 1,1,
     &      jdate, jtime, ht ) ) THEN
            xmsg = 'Could not read HT from '
     &              // grdcro_2d
            CALL M3EXIT ( pname, jdate, jtime, xmsg, status )
         ENDIF

         ZH(loop,1,:)=dummy(1,1,:)+ht !height above sea level
         TR(loop)=ht

      ENDDO

      DO loop=1,num_target_layer
         target_layer(loop)=FLOAT(loop)*100.
      ENDDO

      DO STEP = 1, MXREC3D ! NSTEPS
         DO V = 1, NVARS3D
            DO loop = 1, ndim
               strtcol=NINT(x_cord(loop))
               endcol=strtcol
               strtrow=NINT(y_cord(loop))
               endrow=strtrow

               IF ( .NOT. INTERPX( m3_conc, VNAME3D ( V ), pname,
     &            strtcol,endcol, strtrow,endrow, 1,nlays,
     &            jdate, jtime, dummy ) ) THEN
                  xmsg = 'Could not read' //VNAME3D (V) // 'from '
     &                 // m3_conc
                  CALL M3EXIT ( pname, jdate, jtime, xmsg, status )
               ENDIF

               conc_dummy(loop,1,:)=dummy(1,1,:) 
             
               CALL interpv_linear (nlays, conc_dummy(loop,1,:),
     &        zh(loop,1,:), num_target_layer,target_data(loop,1,:),
     &        target_layer)

              DO loop2=1,num_target_layer
                 IF (target_layer(loop2).LE.TR(loop)) THEN
                    target_data(loop,1,loop2)=-999.          !no value underneath terrain height
                 ENDIF
              ENDDO


            ENDDO ! loop

            IF ( .NOT. WRITE3( xsect_data, VNAME3D( V ),
     &                         JDATE, JTIME, target_data ) ) THEN
                 xmsg = 'Could not write "' // TRIM( VNAME3D(V) ) //
     &                  '" to "' // TRIM( xsect_data ) // '"'
                 CALL M3EXIT( pname, JDATE, JTIME, xmsg, status )
            ENDIF  ! if write failed

         ENDDO              !  end loop on variables V for this time step

         CALL NEXTIME( JDATE, JTIME, TSTEP3D )

      ENDDO          !  end loop on output time steps

      !release memory
      IF ( xsect_type .EQ.3 ) THEN
         DEALLOCATE ( lat ) 
         DEALLOCATE ( lon ) 
      ENDIF 
      DEALLOCATE ( conc_dummy )
      DEALLOCATE ( zh )
      DEALLOCATE ( dummy )
      DEALLOCATE ( tr ) 
      DEALLOCATE ( target_data )
      DEALLOCATE ( x_cord )  
      DEALLOCATE ( y_cord )

      END
